#ifdef HAVE_CONFIG_H  
#  include <config.h>
#endif

#include <gnome.h>

#include "callbacks.h"
#include "support.h"
#include "main.h"
#include "trx.h"
#include "prefs.h"
#include "ptt.h"
#include "wf.h"

struct preferences prefs;
struct preferences tmp_prefs;

GdkFont *rxfont, *txfont;

#define	DEFAULT_FONT	"-*-lucidatypewriter-medium-r-*-*-12-*-*-*-*-*-*-*"

/* ---------------------------------------------------------------------- */

#define confcpy(d,s)	{ strncpy((d),(s),sizeof(d)); d[sizeof(d)-1] = 0; }

static int color_to_int(GdkColor *color)
{
	int ret = 0;

	ret |= (color->red   & 0xff00) << 8;
	ret |= (color->green & 0xff00);
	ret |= (color->blue  & 0xff00) >> 8;

	return ret;
}

static void alloc_color(GdkColor *color, int c)
{
	GdkColormap *cmap;

	color->red   = (0xff0000 & c) >> 8;
	color->green = (0x00ff00 & c);
	color->blue  = (0x0000ff & c) << 8;

	cmap = gdk_colormap_get_system();

	if (!gdk_color_alloc(cmap, color))
		g_error("couldn't allocate color");
}

static void entry_changed(GtkEditable *editable, gpointer user_data)
{
	GtkWidget *propertybox;
	gchar *entry = (gchar *) user_data;

	propertybox = gtk_widget_get_toplevel(GTK_WIDGET(editable));
	gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

	strncpy(entry, gtk_entry_get_text(GTK_ENTRY(editable)), 64);
	entry[63] = 0;
}

static void float_entry_changed(GtkEditable *editable, gpointer user_data)
{
	GtkWidget *propertybox;
	gfloat *value = (gfloat *) user_data;

	propertybox = gtk_widget_get_toplevel(GTK_WIDGET(editable));
	gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

	*value = atof(gtk_entry_get_text(GTK_ENTRY(editable)));
}

static void color_set(GnomeColorPicker *gnomecolorpicker,
		guint arg1,
		guint arg2,
		guint arg3,
		guint arg4,
		gpointer user_data)
{
	GtkWidget *propertybox;
	GdkColor *color = (GdkColor *) user_data;
	int value = 0;

	propertybox = gtk_widget_get_toplevel(GTK_WIDGET(gnomecolorpicker));
	gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

	value |= (arg1 & 0xff00) << 8;
	value |= (arg2 & 0xff00);
	value |= (arg3 & 0xff00) >> 8;

	alloc_color(color, value);
}

static void font_set(GnomeFontPicker *gnomefontpicker,
		gchar *arg1,
		gpointer user_data)
{
	GtkWidget *propertybox;
	gchar *font = (gchar *) user_data;

	propertybox = gtk_widget_get_toplevel(GTK_WIDGET(gnomefontpicker));
	gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

	strncpy(font, arg1, 256);
	font[255] = 0;
}

static void button_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
	GtkWidget *propertybox;
	gboolean *flag = (gboolean *) user_data;

	propertybox = gtk_widget_get_toplevel(GTK_WIDGET(togglebutton));
	gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

	if (togglebutton->active)
		*flag = TRUE;
	else
		*flag = FALSE;
}

static void spinbutton_changed(GtkSpinButton *spinbutton, gpointer user_data)
{
	GtkWidget *propertybox;
	gfloat *value = (gfloat *) user_data;

	propertybox = gtk_widget_get_toplevel(GTK_WIDGET(spinbutton));
	gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

	*value = gtk_spin_button_get_value_as_float(spinbutton);
}

static void color_picker_set(GnomeColorPicker *cp, GdkColor *clr)
{
	gnome_color_picker_set_i16(cp, clr->red, clr->green, clr->blue, 0);
}

static void sensitivity_scale_changed(GtkAdjustment *adj, gpointer user_data)
{
	GtkWidget *propertybox = (GtkWidget *) user_data;

	gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

	tmp_prefs.sensitivity = adj->value;
}

static void window_button_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
	GtkWidget *propertybox;
	gint window = GPOINTER_TO_INT(user_data);

	if (togglebutton->active) {
		propertybox = gtk_widget_get_toplevel(GTK_WIDGET(togglebutton));
		gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

		tmp_prefs.window = window;
	}
}

static void rttybits_selected(GtkMenuShell *menushell, gpointer user_data)
{
	GtkWidget *propertybox = GTK_WIDGET(user_data);
	GtkWidget *activeitem;
	gint index;

	gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

	activeitem = gtk_menu_get_active(GTK_MENU(menushell));
	index = g_list_index(menushell->children, activeitem);

	tmp_prefs.rttybits = index;
}

static void rttyparity_selected(GtkMenuShell *menushell, gpointer user_data)
{
	GtkWidget *propertybox = GTK_WIDGET(user_data);
	GtkWidget *activeitem;
	gint index;

	gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

	activeitem = gtk_menu_get_active(GTK_MENU(menushell));
	index = g_list_index(menushell->children, activeitem);

	tmp_prefs.rttyparity = index;
}

static void rttystop_selected(GtkMenuShell *menushell, gpointer user_data)
{
	GtkWidget *propertybox = GTK_WIDGET(user_data);
	GtkWidget *activeitem;
	gint index;

	gnome_property_box_changed(GNOME_PROPERTY_BOX(propertybox));

	activeitem = gtk_menu_get_active(GTK_MENU(menushell));
	index = g_list_index(menushell->children, activeitem);

	tmp_prefs.rttystop = index;
}

static void delimitstr(gchar *str)
{
	gchar *p, *p2, prev = ',';

	/* replace all delimiter chars with a comma */
	p = g_strdelimit(str, ", \t\r\n", ',');

	/* remove multiple commas */
	p2 = str;
	while (*p) {
		if (*p == ',' && prev == ',')
			p++;
		else
			prev = *p2++ = *p++;
	}
	*p2 = 0;
}

static void setqsobands(gchar *bands)
{
	GtkCombo *qsobandcombo;
	GList *list = NULL;
	gchar **p1, **p2;

	delimitstr(bands);

	p1 = g_strsplit(bands, ",", 32);

	for (p2 = p1; *p2; p2++)
		list = g_list_append(list, (gpointer) *p2);

	qsobandcombo = GTK_COMBO(lookup_widget(appwindow, "qsobandcombo"));
	gtk_combo_set_popdown_strings(qsobandcombo, list);

	g_strfreev(p1);
	g_list_free(list);
}

/* ---------------------------------------------------------------------- */

void apply_config(GtkWidget *widget)
{
	GnomePropertyBox *propertybox;
	gint i;

	propertybox = GNOME_PROPERTY_BOX(widget);

	/*
	 * Reconfigure stuff.
	 */
	if (strcmp(prefs.rx_font, tmp_prefs.rx_font))
		rxfont = gdk_font_load(tmp_prefs.rx_font);

	if (strcmp(prefs.tx_font, tmp_prefs.tx_font))
		txfont = gdk_font_load(tmp_prefs.tx_font);

	if (strcmp(prefs.pttdev, tmp_prefs.pttdev) ||
	    prefs.pttinv != tmp_prefs.pttinv)
		init_ptt(tmp_prefs.pttdev, tmp_prefs.pttinv);

	if (prefs.window != tmp_prefs.window)
		wf_setwindow(tmp_prefs.window);

	/*
	 * Copy the temp config back to the real one.
	 */
	prefs = tmp_prefs;

	/*
	 * Restart the modem.
	 */
	i = trx_get_state();
        if (i != TRX_STATE_ABORT && i != TRX_STATE_PAUSE) {
		/* abort the trx */
                trx_set_state_wait(TRX_STATE_ABORT);
                /* clear tx typeahead buffer */
                while (trx_get_tx_char() != -1);
                /* switch to rx */
                trx_set_state_wait(TRX_STATE_RX);
                push_button("rxbutton");
        }

	setqsobands(prefs.qsobands);

	/*
	 * Save it to disk.
	 */
	gnome_config_set_string("/gMFSK/info/mycall",   prefs.mycall);
	gnome_config_set_string("/gMFSK/info/myname",   prefs.myname);
	gnome_config_set_string("/gMFSK/info/myqth",    prefs.myqth);
	gnome_config_set_string("/gMFSK/info/myloc",    prefs.myloc);
	gnome_config_set_string("/gMFSK/info/myemail",  prefs.myemail);

	i = color_to_int(&prefs.hl_color);
	gnome_config_set_int("/gMFSK/colors/hl", i);

	i = color_to_int(&prefs.rx_color);
	gnome_config_set_int("/gMFSK/colors/rx", i);

	i = color_to_int(&prefs.tx_color);
	gnome_config_set_int("/gMFSK/colors/tx", i);

	i = color_to_int(&prefs.tx_win_color);
	gnome_config_set_int("/gMFSK/colors/txwin", i);

	i = color_to_int(&prefs.rx_win_color);
	gnome_config_set_int("/gMFSK/colors/rxwin", i);

	gnome_config_set_string("/gMFSK/fonts/rxfont",  prefs.rx_font);
	gnome_config_set_string("/gMFSK/fonts/txfont",  prefs.tx_font);

	gnome_config_set_float("/gMFSK/misc/txoffset",  prefs.txoffset);
	gnome_config_set_float("/gMFSK/misc/sqval",     prefs.sqval);

	gnome_config_set_bool("/gMFSK/misc/afc",        prefs.afc);
	gnome_config_set_bool("/gMFSK/misc/squelch",    prefs.squelch);

	gnome_config_set_string("/gMFSK/sound/dev",     prefs.sounddev);
	gnome_config_set_bool("/gMFSK/sound/8bit",      prefs.eightbitsound);
	gnome_config_set_bool("/gMFSK/sound/fulldup",   prefs.fullduplexsound);
	gnome_config_set_bool("/gMFSK/sound/testmode",  prefs.testmode);

	gnome_config_set_string("/gMFSK/ptt/dev",       prefs.pttdev);
	gnome_config_set_bool("/gMFSK/ptt/inverted",    prefs.pttinv);

	gnome_config_set_float("/gMFSK/wf/sensitivity", prefs.sensitivity);
	gnome_config_set_int("/gMFSK/wf/window",        prefs.window);

	gnome_config_set_float("/gMFSK/rtty/shift",     prefs.rttyshift);
	gnome_config_set_float("/gMFSK/rtty/baud",      prefs.rttybaud);
	gnome_config_set_int("/gMFSK/rtty/bits",        prefs.rttybits);
	gnome_config_set_int("/gMFSK/rtty/parity",      prefs.rttyparity);
	gnome_config_set_int("/gMFSK/rtty/stop",        prefs.rttystop);
	gnome_config_set_bool("/gMFSK/rtty/reverse",    prefs.rttyreverse);
	gnome_config_set_bool("/gMFSK/rtty/msbfirst",   prefs.rttymsb);

	gnome_config_set_string("/gMFSK/misc/bands",    prefs.qsobands);

	gnome_config_set_string("/gMFSK/misc/timefmt",  prefs.timefmt);
	gnome_config_set_string("/gMFSK/misc/datefmt",  prefs.datefmt);

	gnome_config_sync();
}

void load_config(void)
{
	gchar *p;
	gint i;
	gboolean b;
	gfloat f;

	p = gnome_config_get_string("/gMFSK/info/mycall=");
	confcpy(prefs.mycall, p);
	g_free(p);

	p = gnome_config_get_string("/gMFSK/info/myname=");
	confcpy(prefs.myname, p);
	g_free(p);

	p = gnome_config_get_string("/gMFSK/info/myqth=");
	confcpy(prefs.myqth, p);
	g_free(p);

	p = gnome_config_get_string("/gMFSK/info/myloc=");
	confcpy(prefs.myloc, p);
	g_free(p);

	p = gnome_config_get_string("/gMFSK/info/myemail=");
	confcpy(prefs.myemail, p);
	g_free(p);


	i = gnome_config_get_int("/gMFSK/colors/hl=255");
	alloc_color(&prefs.hl_color, i);

	i = gnome_config_get_int("/gMFSK/colors/rx=0");
	alloc_color(&prefs.rx_color, i);

	i = gnome_config_get_int("/gMFSK/colors/tx=16711680");
	alloc_color(&prefs.tx_color, i);

	i = gnome_config_get_int("/gMFSK/colors/rxwin=16777215");
	alloc_color(&prefs.rx_win_color, i);

	i = gnome_config_get_int("/gMFSK/colors/txwin=16777215");
	alloc_color(&prefs.tx_win_color, i);


	p = gnome_config_get_string("/gMFSK/fonts/rxfont=" DEFAULT_FONT);
	confcpy(prefs.rx_font, p);
	g_free(p);
	rxfont = gdk_font_load(prefs.rx_font);

	p = gnome_config_get_string("/gMFSK/fonts/txfont=" DEFAULT_FONT);
	confcpy(prefs.tx_font, p);
	g_free(p);
	txfont = gdk_font_load(prefs.tx_font);


	f = gnome_config_get_float("/gMFSK/misc/txoffset=0");
	prefs.txoffset = f;

	f = gnome_config_get_float("/gMFSK/misc/sqval=15");
	prefs.sqval = f;


	b = gnome_config_get_bool("/gMFSK/misc/afc=FALSE");
	prefs.afc = b;

	b = gnome_config_get_bool("/gMFSK/misc/squelch=FALSE");
	prefs.squelch = b;


	p = gnome_config_get_string("/gMFSK/sound/dev=/dev/dsp");
	confcpy(prefs.sounddev, p);
	g_free(p);

	b = gnome_config_get_bool("/gMFSK/sound/8bit=FALSE");
	prefs.eightbitsound = b;

	b = gnome_config_get_bool("/gMFSK/sound/fulldup=FALSE");
	prefs.fullduplexsound = b;

	b = gnome_config_get_bool("/gMFSK/sound/testmode=FALSE");
	prefs.testmode = b;


	p = gnome_config_get_string("/gMFSK/ptt/dev=none");
	confcpy(prefs.pttdev, p);
	g_free(p);

	b = gnome_config_get_bool("/gMFSK/ptt/inverted=FALSE");
	prefs.pttinv = b;


	f = gnome_config_get_float("/gMFSK/wf/sensitivity=50");
	prefs.sensitivity = f;

	i = gnome_config_get_int("/gMFSK/wf/window=1");
	prefs.window = i;


	f = gnome_config_get_float("/gMFSK/rtty/shift=170");
	prefs.rttyshift = f;

	f = gnome_config_get_float("/gMFSK/rtty/baud=45.45");
	prefs.rttybaud = f;

	i = gnome_config_get_int("/gMFSK/rtty/bits=0");
	prefs.rttybits = i;

	i = gnome_config_get_int("/gMFSK/rtty/parity=0");
	prefs.rttyparity = i;

	i = gnome_config_get_int("/gMFSK/rtty/stop=1");
	prefs.rttystop = i;

	b = gnome_config_get_bool("/gMFSK/rtty/reverse=TRUE");
	prefs.rttyreverse = b;

	b = gnome_config_get_bool("/gMFSK/rtty/msbfirst=FALSE");
	prefs.rttymsb = b;


	p = gnome_config_get_string("/gMFSK/misc/bands=1.8,3.5,7,10,14,18,21,24,28");
	confcpy(prefs.qsobands, p);
	setqsobands(prefs.qsobands);
	g_free(p);


	p = gnome_config_get_string("/gMFSK/misc/timefmt=%H:%M:%S %Z");
	confcpy(prefs.timefmt, p);
	g_free(p);

	p = gnome_config_get_string("/gMFSK/misc/datefmt=%d.%m.%Y");
	confcpy(prefs.datefmt, p);
	g_free(p);
}

static gchar *float_to_str(gfloat f)
{
	static gchar buf[16];

	snprintf(buf, sizeof(buf), "%g", f);
	buf[sizeof(buf) - 1] = 0;

	return buf;
}

void fill_config(GtkWidget *widget)
{
	GnomePropertyBox *propertybox;
	GtkAdjustment *adj;
	gpointer p;

	propertybox = GNOME_PROPERTY_BOX(widget);

	/*
	 * Copy the real config to a temp struct.
	 */
	tmp_prefs = prefs;

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "callentry");
        gtk_entry_set_text(GTK_ENTRY(p), prefs.mycall);
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(entry_changed),
			   tmp_prefs.mycall);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "nameentry");
        gtk_entry_set_text(GTK_ENTRY(p), prefs.myname);
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(entry_changed),
			   tmp_prefs.myname);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "qthentry");
        gtk_entry_set_text(GTK_ENTRY(p), prefs.myqth);
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(entry_changed),
			   tmp_prefs.myqth);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "locentry");
        gtk_entry_set_text(GTK_ENTRY(p), prefs.myloc);
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(entry_changed),
			   tmp_prefs.myloc);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "emailentry");
        gtk_entry_set_text(GTK_ENTRY(p), prefs.myemail);
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(entry_changed),
			   tmp_prefs.myemail);


	p = gtk_object_get_data(GTK_OBJECT(propertybox), "hlcolorpicker");
	color_picker_set(GNOME_COLOR_PICKER(p), &prefs.hl_color);
	gtk_signal_connect(GTK_OBJECT(p), "color_set",
			   GTK_SIGNAL_FUNC(color_set),
			   &tmp_prefs.hl_color);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "rxcolorpicker");
	color_picker_set(GNOME_COLOR_PICKER(p), &prefs.rx_color);
	gtk_signal_connect(GTK_OBJECT(p), "color_set",
			   GTK_SIGNAL_FUNC(color_set),
			   &tmp_prefs.rx_color);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "txcolorpicker");
	color_picker_set(GNOME_COLOR_PICKER(p), &prefs.tx_color);
	gtk_signal_connect(GTK_OBJECT(p), "color_set",
			   GTK_SIGNAL_FUNC(color_set),
			   &tmp_prefs.tx_color);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "txwincolorpicker");
	color_picker_set(GNOME_COLOR_PICKER(p), &prefs.tx_win_color);
	gtk_signal_connect(GTK_OBJECT(p), "color_set",
			   GTK_SIGNAL_FUNC(color_set),
			   &tmp_prefs.tx_win_color);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "rxwincolorpicker");
	color_picker_set(GNOME_COLOR_PICKER(p), &prefs.rx_win_color);
	gtk_signal_connect(GTK_OBJECT(p), "color_set",
			   GTK_SIGNAL_FUNC(color_set),
			   &tmp_prefs.rx_win_color);


	p = gtk_object_get_data(GTK_OBJECT(propertybox), "rxfontpicker");
	gnome_font_picker_set_font_name(GNOME_FONT_PICKER(p), prefs.rx_font);
	gnome_font_picker_set_mode(GNOME_FONT_PICKER(p),
				   GNOME_FONT_PICKER_MODE_FONT_INFO);
	gtk_signal_connect(GTK_OBJECT(p), "font_set",
			   GTK_SIGNAL_FUNC(font_set),
			   tmp_prefs.rx_font);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "txfontpicker");
	gnome_font_picker_set_font_name(GNOME_FONT_PICKER(p), prefs.tx_font);
	gnome_font_picker_set_mode(GNOME_FONT_PICKER(p),
				   GNOME_FONT_PICKER_MODE_FONT_INFO);
	gtk_signal_connect(GTK_OBJECT(p), "font_set",
			   GTK_SIGNAL_FUNC(font_set),
			   tmp_prefs.tx_font);


	p = gtk_object_get_data(GTK_OBJECT(propertybox), "txoffsetspinbutton");
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(p), prefs.txoffset);
	gtk_signal_connect (GTK_OBJECT(p), "changed",
			    GTK_SIGNAL_FUNC(spinbutton_changed),
			    &tmp_prefs.txoffset);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "squelchspinbutton");
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(p), prefs.sqval);
	gtk_signal_connect (GTK_OBJECT(p), "changed",
			    GTK_SIGNAL_FUNC(spinbutton_changed),
			    &tmp_prefs.sqval);


	p = gtk_object_get_data(GTK_OBJECT(propertybox), "afccheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), prefs.afc);
	gtk_signal_connect (GTK_OBJECT(p), "toggled",
			    GTK_SIGNAL_FUNC(button_toggled),
			    &tmp_prefs.afc);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "squelchcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), prefs.squelch);
	gtk_signal_connect (GTK_OBJECT(p), "toggled",
			    GTK_SIGNAL_FUNC(button_toggled),
			    &tmp_prefs.squelch);


	p = gtk_object_get_data(GTK_OBJECT(propertybox), "soundentry");
        gtk_entry_set_text(GTK_ENTRY(p), prefs.sounddev);
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(entry_changed),
			   tmp_prefs.sounddev);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "eightbitcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), prefs.eightbitsound);
	gtk_signal_connect (GTK_OBJECT(p), "toggled",
			    GTK_SIGNAL_FUNC(button_toggled),
			    &tmp_prefs.eightbitsound);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "rwcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), prefs.fullduplexsound);
	gtk_signal_connect (GTK_OBJECT(p), "toggled",
			    GTK_SIGNAL_FUNC(button_toggled),
			    &tmp_prefs.fullduplexsound);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "testcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), prefs.testmode);
	gtk_signal_connect (GTK_OBJECT(p), "toggled",
			    GTK_SIGNAL_FUNC(button_toggled),
			    &tmp_prefs.testmode);


	p = gtk_object_get_data(GTK_OBJECT(propertybox), "pttentry");
        gtk_entry_set_text(GTK_ENTRY(p), prefs.pttdev);
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(entry_changed),
			   tmp_prefs.pttdev);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "pttinvcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), prefs.pttinv);
	gtk_signal_connect (GTK_OBJECT(p), "toggled",
			    GTK_SIGNAL_FUNC(button_toggled),
			    &tmp_prefs.pttinv);


	p = gtk_object_get_data(GTK_OBJECT(propertybox), "sensitivityscale");
	adj = gtk_range_get_adjustment(GTK_RANGE(p));
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj), prefs.sensitivity);
	gtk_signal_connect(GTK_OBJECT(adj), "value-changed",
			   GTK_SIGNAL_FUNC(sensitivity_scale_changed),
			   propertybox);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "rectradiobutton");
	if (prefs.window == WINDOW_RECT)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), TRUE);
	gtk_signal_connect(GTK_OBJECT(p), "toggled",
			   GTK_SIGNAL_FUNC(window_button_toggled),
			   GINT_TO_POINTER(WINDOW_RECT));

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "triaradiobutton");
	if (prefs.window == WINDOW_TRIA)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), TRUE);
	gtk_signal_connect(GTK_OBJECT(p), "toggled",
			   GTK_SIGNAL_FUNC(window_button_toggled),
			   GINT_TO_POINTER(WINDOW_TRIA));

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "hammradiobutton");
	if (prefs.window == WINDOW_HAMM)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), TRUE);
	gtk_signal_connect(GTK_OBJECT(p), "toggled",
			   GTK_SIGNAL_FUNC(window_button_toggled),
			   GINT_TO_POINTER(WINDOW_HAMM));


	p = gtk_object_get_data(GTK_OBJECT(propertybox), "rttyshiftentry");
        gtk_entry_set_text(GTK_ENTRY(p), float_to_str(prefs.rttyshift));
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(float_entry_changed),
			   &tmp_prefs.rttyshift);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "rttybaudentry");
        gtk_entry_set_text(GTK_ENTRY(p), float_to_str(prefs.rttybaud));
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(float_entry_changed),
			   &tmp_prefs.rttybaud);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "bitsoptionmenu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(p), prefs.rttybits);
	gtk_signal_connect(GTK_OBJECT(GTK_OPTION_MENU(p)->menu), "deactivate",
			   GTK_SIGNAL_FUNC(rttybits_selected),
			   propertybox);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "parityoptionmenu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(p), prefs.rttyparity);
	gtk_signal_connect(GTK_OBJECT(GTK_OPTION_MENU(p)->menu), "deactivate",
			   GTK_SIGNAL_FUNC(rttyparity_selected),
			   propertybox);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "stopoptionmenu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(p), prefs.rttystop);
	gtk_signal_connect(GTK_OBJECT(GTK_OPTION_MENU(p)->menu), "deactivate",
			   GTK_SIGNAL_FUNC(rttystop_selected),
			   propertybox);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "reversecheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), prefs.rttyreverse);
	gtk_signal_connect (GTK_OBJECT(p), "toggled",
			    GTK_SIGNAL_FUNC(button_toggled),
			    &tmp_prefs.rttyreverse);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "msbcheckbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(p), prefs.rttymsb);
	gtk_signal_connect (GTK_OBJECT(p), "toggled",
			    GTK_SIGNAL_FUNC(button_toggled),
			    &tmp_prefs.rttymsb);


	p = gtk_object_get_data(GTK_OBJECT(propertybox), "qsobandsentry");
        gtk_entry_set_text(GTK_ENTRY(p), prefs.qsobands);
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(entry_changed),
			   tmp_prefs.qsobands);


	p = gtk_object_get_data(GTK_OBJECT(propertybox), "timeentry");
        gtk_entry_set_text(GTK_ENTRY(p), prefs.timefmt);
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(entry_changed),
			   tmp_prefs.timefmt);

	p = gtk_object_get_data(GTK_OBJECT(propertybox), "dateentry");
        gtk_entry_set_text(GTK_ENTRY(p), prefs.datefmt);
	gtk_signal_connect(GTK_OBJECT(p), "changed",
			   GTK_SIGNAL_FUNC(entry_changed),
			   tmp_prefs.datefmt);
}

/* ---------------------------------------------------------------------- */
