package kalyptusCxxToC;

use File::Path;
use File::Basename;

use Carp;
use Ast;
use kdocAstUtil;
use kdocUtil; 
use Iter;
use kalyptusDataDict;

use strict;
no strict "subs";

use vars qw/ @clist $host $who $now $gentext %functionId $docTop
	$lib $rootnode $outputdir $opt $debug $typeprefix $eventHandlerCount $constructorCount *CLASS *HEADER *QTCTYPES *KDETYPES /;

BEGIN
{
@clist = ();

	# Page footer

	$who = kdocUtil::userName();
	$host = kdocUtil::hostName();
	$now = localtime;
	$gentext = "$who\@$host on $now, using kalyptus $main::Version.";

	$docTop =<<EOF
                             -------------------
    begin                : $now
    copyright            : (C) 2000-2001 Lost Highway Ltd. All rights reserved.
    email                : Lost_Highway\@tipitina.demon.co.uk
    generated by         : $gentext
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 ***************************************************************************/

EOF

}


sub writeDoc
{
	( $lib, $rootnode, $outputdir, $opt ) = @_;

	$debug = $main::debuggen;

	mkpath( $outputdir ) unless -f $outputdir;

	my $file = "$outputdir/qtc_types.h";
	open( QTCTYPES, ">$file" ) || die "Couldn't create $file\n";
	print QTCTYPES "/***************************************************************************\n";
	print QTCTYPES "                            qtc_types.h -  description\n";
	print QTCTYPES $docTop;
	print QTCTYPES "#ifndef QTC_TYPES_H\n";
	print QTCTYPES "#define QTC_TYPES_H\n";

	$file = "$outputdir/kde_types.h";
	open( KDETYPES, ">$file" ) || die "Couldn't create $file\n";
	print KDETYPES "/***************************************************************************\n";
	print KDETYPES "                            kde_types.h -  description\n";
	print KDETYPES $docTop;
	print KDETYPES "#ifndef KDE_TYPES_H\n";
	print KDETYPES "#define KDE_TYPES_H\n";

	# Document all compound nodes
	Iter::LocalCompounds( $rootnode, sub { writeClassDoc( shift ); } );
	
	print QTCTYPES "typedef struct { int dummy; } qt_Signal;\n";
	print QTCTYPES "typedef struct { int dummy; } qt_Slot;\n";
	print QTCTYPES "typedef void (*qt_intCallback)(int);\n";
	print QTCTYPES "typedef void (*qt_voidCallback)();\n";
	print QTCTYPES "typedef void (*qt_floatCallback)(float);\n";
	print QTCTYPES "typedef void (*qt_QRectCallback)(qt_QRect *);\n";
	print QTCTYPES "typedef void (*qt_QSizeCallback)(qt_QSize *);\n";
	print QTCTYPES "typedef void (*qt_UserDataCallback)(void *, void *);\n";
	print QTCTYPES "typedef int  (*qt_eventFilter)(qt_QObject*,qt_QEvent*);\n";
	print QTCTYPES "typedef int (*qt_EventDelegate)(void *, char *, void *, char *);\n";
	print QTCTYPES "extern qt_EventDelegate Qt_EventDelegate;\n";
	print QTCTYPES "#endif\n";

	print KDETYPES "#endif\n";

	close QTCTYPES;
	close KDETYPES;
}




=head2 writeClassDoc

	Write documentation for one compound node.

=cut

sub writeClassDoc
{
	my( $node ) = @_;

	print "Enter: $node->{astNodeName}\n" if $debug;
	if( exists $node->{ExtSource} ) {
		warn "Trying to write doc for ".$node->{AstNodeName}.
			" from ".$node->{ExtSource}."\n";
		return;
	}

	my $typeName = $node->{astNodeName}."*";

	if ( kalyptusDataDict::ctypemap($typeName) eq "" ) {
		$typeprefix = ($typeName =~ /^Q/ ? "qt_" : "kde_");
		kalyptusDataDict::setctypemap($typeName, $typeprefix.$node->{astNodeName}."*");
		print "'$typeName' => '$typeprefix$typeName',\n";
	} elsif ( kalyptusDataDict::ctypemap($typeName) =~ /^qt_/ ) {
		$typeprefix = "qt_";
	} elsif ( kalyptusDataDict::ctypemap($typeName) =~ /^kde_/ ) {
		$typeprefix = "kde_";
	} else {
		$typeprefix = "kde_";
	}

	my $file = "$outputdir/$typeprefix".join("__", kdocAstUtil::heritage($node)).".h";
	my $docnode = $node->{DocNode};
	my @list = ();
	my $version = undef;
	my $author = undef;

#	if( $#{$node->{Kids}} < 0 || $node->{Access} eq "private" || exists $node->{Tmpl} ) {
	if( $#{$node->{Kids}} < 0 || $node->{Access} eq "private") {
		return;
	}

	open( HEADER, ">$file" ) || die "Couldn't create $file\n";
	$file =~ s/\.h/.cpp/;
	open( CLASS, ">$file" ) || die "Couldn't create $file\n";

	# Header

	my $short = "";
	my $extra = "";

	if ( kalyptusDataDict::ctypemap($typeName) =~ /^kde_/ ) {
 		print KDETYPES "typedef struct { int dummy; } ", $typeprefix, $node->{astNodeName}, ";\n";
	} else {
 		print QTCTYPES "typedef struct { int dummy; } ", $typeprefix, $node->{astNodeName}, ";\n";
    }

	print HEADER "/***************************************************************************\n";
	print HEADER "                            ", $typeprefix, $node->{astNodeName}, ".h -  description\n";
	print HEADER $docTop;

	print CLASS "/***************************************************************************\n";
	print CLASS "                            ", $typeprefix, $node->{astNodeName}, ".cpp -  description\n";
	print CLASS $docTop;
	print CLASS "extern \"C\" {\n#include \"", $typeprefix, $node->{astNodeName}, ".h\"\n}\n\n";

	print HEADER "#ifndef ", uc($typeprefix), uc($node->{astNodeName}), "\n";
	print HEADER "#define ", uc($typeprefix), uc($node->{astNodeName}), "\n\n";
	print HEADER "#include <qtc/qtc_types.h>\n\n";

	if ( kalyptusDataDict::ctypemap($typeName) =~ /^kde_/ ) {
		print HEADER "#include <kdec/kde_types.h>\n\n";
	}

	if ( defined $docnode ) {
		print HEADER "/**\n";
		if ( defined $docnode->{Text} ) {
			my $node;
			foreach $node ( @{$docnode->{Text}} ) {
				next if $node->{NodeType} ne "DocText";
				print HEADER $node->{astNodeName}, "\n";
			}
		}

		exists $docnode->{Author} && print HEADER " \@author ", $docnode->{Author}, "\n";
		exists $docnode->{Version} && print HEADER " \@version ", $docnode->{Version}, "\n";
		exists $docnode->{ClassShort} && print HEADER " \@short ", $docnode->{ClassShort}, "\n";
		print HEADER "*/\n";
	}

	my $sourcename = $node->{Source}->{astNodeName};
	
	if ( $sourcename =~ m!.*(dom|kabc|kdeprint|kdesu|kio|kjs|kparts|ktexteditor|libkmid)/([^/]*$)! ) {
		$sourcename = $1."/".$2;
	} else {
		$sourcename =~ s!.*/([^/]*$)!$1!;
	}

	print CLASS "#include <",$sourcename , ">\n\n";

	$constructorCount = 0;

	Iter::MembersByType ( $node,
		sub { print HEADER "", $_[0], ""; print CLASS "", $_[0], ""; },
		sub {	my ($node, $kid ) = @_;
                 preParseMember( $node, $kid );
               },
		sub { print HEADER ""; print CLASS ""; }
	);

	if ( ! exists $node->{Pure} && $constructorCount > 0 ) {
		print CLASS "class ", $node->{astNodeName}, "Bridge : public ", kalyptusDataDict::addNamespace($node->{astNodeName}), "\n{\npublic:\n";

		Iter::MembersByType ( $node,
			sub { print HEADER "", $_[0], ""; print CLASS "", $_[0], ""; },
			sub {	my ($node, $kid ) = @_;
                    generateBridgeClass( $node, $kid );
                 },
			sub { print HEADER ""; print CLASS ""; }
			);

		generateBridgeEventHandlers($node);

		print CLASS "};\n\n";
	}

	%functionId = ();
	$eventHandlerCount = 0;

	Iter::MembersByType ( $node,
		sub { print HEADER "", $_[0], ""; print CLASS "", $_[0], ""; },
		sub {	my ($node, $kid ) = @_;
                               listMember( $node, $kid );
                         },
		sub { print HEADER ""; print CLASS ""; }
	);

	# ancestors
	my @ancestors = ();
	Iter::Ancestors( $node, $rootnode, undef, undef,
		sub { # print
			my ( $ances, $name, $type, $template ) = @_;

			push @ancestors, $name;

		},
		undef
	);

	if ( $#ancestors > 0 ) {
		# 'type transfer' functions to cast for correct use of multiple inheritance
		foreach my $ancestor (@ancestors) {
			print HEADER "\n/\*\* Casts a '$typeprefix", $node->{astNodeName}, " *' to a '", kalyptusDataDict::ctypemap($ancestor."\*"), "' \*/\n";
			print HEADER kalyptusDataDict::ctypemap($ancestor."\*"), " ", $typeprefix, $node->{astNodeName}, "_", $ancestor;
			print HEADER "(", $typeprefix, $node->{astNodeName}, "* instPointer);\n";

			print CLASS kalyptusDataDict::ctypemap($ancestor."\*"), " ", $typeprefix, $node->{astNodeName}, "_", $ancestor;
			print CLASS "(", $typeprefix, $node->{astNodeName}, "* instPointer){\n";
			print CLASS "\treturn (", kalyptusDataDict::ctypemap($ancestor."\*"), ") (", $ancestor, " *) (", $node->{astNodeName}, " *) instPointer;\n}\n";
		}
	}

	print HEADER "\n#endif\n";

	close HEADER;
	close CLASS;
}


sub preParseMember
{
	my( $class, $m ) = @_;
	my $name = $m->{astNodeName};

	if( $m->{NodeType} eq "method" ) {
		# A JBridge class will only be generated if there is at least one
		# public or protected constructor
		if ( $name eq $class->{astNodeName} && $m->{Access} ne "private" ) {
			$constructorCount++;
		}
    }
}

sub generateBridgeEventHandlers
{
	my( $node ) = @_;
	my %allmem = ();
	my $key;

	my $m;
	my $name;

	kdocAstUtil::allMembers( \%allmem, $node );

	foreach $key (keys (%allmem)) {
		$m = $allmem{$key};
		$name = $m->{astNodeName} ;
		my $type = $m->{NodeType};
		my $docnode = $m->{DocNode};
		my $cparams = $m->{Params};
		my $parent = $m->{Parent};
		my $cplusplusparams;

	if( $type eq "method" && $m->{Access} eq "protected"  && $name =~ /.*Event$/
		&& $name !~ /qwsEvent/ && $name !~ /x11Event/ && $name !~ /winEvent/ && $name !~ /macEvent/ && $name !~ /movableDropEvent/ ) {

			$cparams =~ s/=\s*[-\"\w]*//g;
			$cparams =~ s/\s+/ /g;
			$cparams =~ s/\s*([,\*\&])\s*/$1 /g;
			$cparams =~ s/^\s*void\s*$//;
			my $argId = 0;
			my @cargs = kdocUtil::splitUnnested(",", $cparams);
			$cparams = "";
			foreach my $arg ( @cargs ) {
				my $argType;
				my $cargType;

				$arg =~ s/\s*([^\s].*[^\s])\s*/$1/;
				if ( $arg =~ /(.*)\s+(\w+)$/ ) {
					$argType = $1;
					$arg = $2;
				} else {
					$argType = $arg;
					$argId++;
					$arg = "arg".$argId;
				}
				$cplusplusparams .= $argType." ".$arg.", ";
				$cparams .= $arg.", ";
			}
			$cparams =~ s/, $//;
			$cplusplusparams =~ s/, $//;

			$eventHandlerCount++;
			my $eventType = $cplusplusparams;
			$eventType =~ s/(.*)\*.*$/$1/;
			print CLASS "\tvoid $name(", $cplusplusparams, ") {\n",
			"\t\tif (Qt_EventDelegate == 0L || !(*Qt_EventDelegate)(this, \"", $name, "\", $cparams, \"$eventType\")) {\n",
			"\t\t\t", $parent->{astNodeName}, "::", $name, "($cparams);\n",
			"\t\t}\n",
			"\t\treturn;\n\t}\n";
		}
	}

}

sub generateBridgeClass
{
	my( $class, $m ) = @_;
	my $name;
    my $function;

	$name = $m->{astNodeName} ;
	my $type = $m->{NodeType};
	my $docnode = $m->{DocNode};

	if( $type eq "method" && $m->{Access} ne "private" && $m->{Access} ne "private_slots" && $m->{Access} ne "signals" ) {
		if ( $m->{ReturnType} =~ /[<>]/ || $m->{Params} =~ /[<>]/ || $m->{Params} =~ /Impl/) {
#			print "template based method not converted: ", $m->{ReturnType}, " ", $m->{Params}, "\n";
			return;
		}

		my $returnType = $m->{ReturnType};
		my $cparams = $m->{Params};
		my $cplusplusparams;
		# TODO port to $m->{ParamList}
		$cparams =~ s/=\s*(("[^\"]*")|(\'.\')|(([-\w:.]*)\s*(\|\s*[-\w]*)*(\(\w*\))?))//g;
		$cparams =~ s/\s+/ /g;
		$cparams =~ s/\s*([,\*\&])\s*/$1 /g;
		$cparams =~ s/^\s*void\s*$//;
		$cparams =~ s/^\s*$//;
		my $argId = 0;
		my @cargs = kdocUtil::splitUnnested(",", $cparams);
		$cparams = "";
		foreach my $arg ( @cargs ) {
			my $argType;
			my $cargType;
			$arg =~ s/\s*([^\s].*[^\s])\s*/$1/;
			if ( $arg =~ /(.*)\s+(\w+)$/ ) {
				$argType = $1;
				$arg = $2;
			} else {
				$argType = $arg;
				$argId++;
				$arg = "arg".$argId;
			}
			$cplusplusparams .= $argType." ".$arg.", ";
			$cparams .= $arg.", ";
		}
		$cparams =~ s/, $//;
		$cplusplusparams =~ s/, $//;

		my $flags = $m->{Flags};

		if ( !defined $flags ) {
			warn "Method ".$m->{astNodeName}.  " has no flags\n";
		}

		my $extra = "";
		$extra .= "static " if $flags =~ "s";
		if ( $name =~ /operator/ ) {
			return;
		}


		if ( $name eq $class->{astNodeName} ) {
			if ( $returnType =~ "~" ) {
				print CLASS "\t~", $name, "Bridge() {}\n";
			} else {
				print CLASS $extra,
					"\t", $name, "Bridge(", $cplusplusparams, ") : $name($cparams) {}\n";
			}
		} elsif( $type eq "method" && $m->{Access} eq "protected"  && $name =~ /.*Event$/ ) {
			;
		} elsif( $m->{Access} =~ /^protected/ ){
			if ( $returnType =~ "void" ) {
				print CLASS "\tvoid protected_$name(", $cplusplusparams, ") {\n",
				"\t\t", $class->{astNodeName}, "::$name($cparams);\n",
				"\t\treturn;\n\t}\n";
			} else {
				print CLASS "\t$returnType protected_$name(", $cplusplusparams, ") {\n",
				"\t\treturn ($returnType) ", $class->{astNodeName}, "::$name($cparams);\n\t}\n";
			}
		}
	}

}

sub listMember
{
	my( $class, $m ) = @_;
	my $name;
    my $function;

	$name = $m->{astNodeName} ;
	my $type = $m->{NodeType};
	my $docnode = $m->{DocNode};

	if ( $m->{ReturnType} =~ /~/ ) {
		$name = "~".$name;
	}

	if ( $functionId{$name} eq "" ) {
		$functionId{$name} = 0;
		$function = $name;
	} else {
		$functionId{$name}++;
		$function = $name.$functionId{$name};
	}

	$function =~ s/~//;

	if( $type eq "method" && $m->{Access} ne "private" && $m->{Access} ne "private_slots" && $m->{Access} ne "signals" ) {
		if ( $m->{ReturnType} =~ /[<>]/ || $m->{Params} =~ /[<>]/  || $m->{Params} =~ /\.\.\./ || $m->{Params} =~ /Impl/
				|| $m->{ReturnType} =~ /QAuBucket/ || $m->{Params} =~ /QAuBucket/
				|| $m->{ReturnType} =~ /QMember/ || $m->{Params} =~ /QMember/   ) {
			return;
		}

		my $returnType = $m->{ReturnType};
		$returnType =~ s/const\s*//;
		$returnType =~ s/\s*([,\*\&])\s*/$1/;
		$returnType =~ s/^\s*//;
		$returnType =~ s/\s*$//;
		if ( $returnType ne "" && kalyptusDataDict::ctypemap($returnType) eq "" ) {
			print "'$returnType' => '$typeprefix$returnType',\n";
		} else {
			$returnType = kalyptusDataDict::ctypemap($returnType);
		}
		# TODO port to $m->{ParamList}
		my $cparams = $m->{Params};
		my $cplusplusparams;
		$cparams =~ s/\s+/ /g;
		$cparams =~ s/\s*([,\*\&])\s*/$1 /g;
		$cparams =~ s/^\s*void\s*$//;
		my $argId = 0;
		my @cargs = kdocUtil::splitUnnested(",", $cparams);
		$cparams = "";
		foreach my $arg ( @cargs ) {
			my $argType;
			my $cargType;
			if ( $arg =~ /^\s*$/ ) {
				next;
			}

			# A '<arg> = <value>' default parameter
			$arg =~ s/\s*([^\s].*[^\s])\s*/$1/;
			$arg =~ s/(\w+)\[\]/\* $1/;
			$arg =~ s/=\s*(("[^\"]*")|(\'.\')|(([-\w:.]*)\s*(\|\s*[-\w]*)*(\(\w*\))?))//;

			if ( $arg =~ /^(.*)\s+(\w+)\s*$/ ) {
				$argType = $1;
				$arg = $2;
			} else {
				$argType = $arg;
				$argId++;
				$arg = "arg".$argId;
			}
			$arg =~ s/^id$/identifier/;
			$argType =~ s/\s*([^\s].*[^\s])\s*/$1/;
			$argType =~ s/\s*const//g;
			$argType =~ s/^\s*//;
			$argType =~ s/([\*\&])\s*([\*\&])/$1$2/;
			$cargType = kalyptusDataDict::ctypemap($argType);

			if ( $argType =~ /^[A-Z][^:]*$/ && $cargType eq "int" && kalyptusDataDict::ctypemap($class->{astNodeName}."::".$argType) ne "" ) {
				$cplusplusparams .= "(".$class->{astNodeName}."::".$argType.")";
			} elsif ( $argType =~ /^\s*WFlags\s*$/ ) {
				$cplusplusparams .= "(QWidget::WFlags)";
			} elsif ( $argType =~ /^\s*ArrowType\s*$/ ) {
				$cplusplusparams .= "(Qt::ArrowType)";
			} elsif ( $argType =~ /^\s*Orientation\s*$/ ) {
				$cplusplusparams .= "(Qt::Orientation)";
			} elsif ( $argType =~ /^\s*BrushStyle\s*$/ ) {
				$cplusplusparams .= "(Qt::BrushStyle)";
			} elsif ( $argType =~ /^\s*BGMode\s*$/ ) {
				$cplusplusparams .= "(Qt::BGMode)";
			} elsif ( $argType =~ /^\s*PenCapStyle\s*$/ ) {
				$cplusplusparams .= "(Qt::PenCapStyle)";
			} elsif ( $argType =~ /^\s*PenStyle\s*$/ ) {
				$cplusplusparams .= "(Qt::PenStyle)";
			} elsif ( $argType =~ /^\s*PenJoinStyle\s*$/ ) {
				$cplusplusparams .= "(Qt::PenJoinStyle)";
			} elsif ( $argType =~ /^\s*RasterOp\s*$/ ) {
				$cplusplusparams .= "(Qt::RasterOp)";
			} elsif ( $argType =~ /^\s*TextFormat\s*$/ ) {
				$cplusplusparams .= "(Qt::TextFormat)";
			} elsif ( $argType =~ /^\s*QDragMode\s*$/ ) {
				$cplusplusparams .= "(QDragObject::DragMode)";
			} elsif ( $argType =~ /^\s*GUIStyle\s*$/ ) {
				$cplusplusparams .= "(Qt::GUIStyle)";
			} elsif ( $argType =~ /^\s*Type\s*$/ ) {
				$cplusplusparams .= "(QEvent::Type)";
			} else {
				$cplusplusparams .= "(".kalyptusDataDict::addNamespace($argType).")";
			}

			if ( $cargType eq "" ) {
				print "'$argType' => '$typeprefix$argType',\n";
				$argType =~ s/\&.*$//;
				$cparams .= $argType." ".$arg.", ";
			} else {
				$cparams .= kalyptusDataDict::ctypemap($argType)." ".$arg.", ";
			}

			if ( ( $cargType =~ /^qt_.*\*/ || $cargType =~ /^kde_.*\*/ ) && $argType =~ /^[^\*]*$/ ) {
				$argType =~ s/^(.*)\&.*$/$1/;
				$cplusplusparams .= "* (".kalyptusDataDict::addNamespace($argType)."*)".$arg.", ";
			} else {
				$cplusplusparams .= $arg.", ";
			}


		}
		$cparams =~ s/, $//;
		$cplusplusparams =~ s/, $//;

		my $flags = $m->{Flags};

		if ( !defined $flags ) {
			warn "Method ".$m->{astNodeName}.  " has no flags\n";
		}


		my $extra = "";
		$extra .= "static " if $flags =~ "s";

		if ( $name =~ /operator/  ) {
			return;
		}

		if ( $m->{Access} =~ /protected/ && $name ne $class->{astNodeName}  ) {
			if ( $class->{Pure} ) {
				return;
			}

			$name = "protected_".$name;
		}

		if ( $name eq $class->{astNodeName} && $class->{Pure} ) {
			return;
		}

		if ( defined $docnode ) {
			if ( defined $docnode->{Text} ) {
				print HEADER "\n/** ";
				my $node;
				my $line;
				foreach $node ( @{$docnode->{Text}} ) {
					next if $node->{NodeType} ne "DocText";
					$line = $node->{astNodeName};
					print HEADER $line, "\n";
				}
				print HEADER "*/\n";
			}
		}

		if ( $name eq $class->{astNodeName} ) {
			print HEADER $extra,
				$typeprefix, $name, " * ", $typeprefix, "new_", $function,
				"(", $cparams, ");\n";
			print CLASS $extra,
				$typeprefix, $name, " * ", $typeprefix, "new_", $function,
				"(", $cparams, "){\n",
				"\treturn (", $typeprefix, $name, " *) new ", $name, "Bridge(", $cplusplusparams, ");\n}\n";
		} elsif ( $returnType =~ /~/  ) {
			print HEADER $extra,
				"void ", $typeprefix, "del_", $function,
				"( ", $typeprefix, $class->{astNodeName}, "* p );\n";
			if (exists $class->{Pure} || $constructorCount == 0) {
				print CLASS $extra,
					"void ", $typeprefix, "del_", $function,
					"( ", $typeprefix, $class->{astNodeName}, "* p ){\n\tdelete (", kalyptusDataDict::addNamespace($class->{astNodeName}), "*) p;\n}\n";
			} else {
				print CLASS $extra,
					"void ", $typeprefix, "del_", $function,
					"( ", $typeprefix, $class->{astNodeName}, "* p ){\n\tdelete (", $class->{astNodeName}, "Bridge*) p;\n}\n";
			}
		} else {
			if ( $name =~ /.*Event$/ ) {
				return;
			}

			# Class or instance method
			my $selfstring;
			if ( $extra =~ /static/ ) {
				if ( exists $class->{Pure} || $constructorCount == 0 ) {
					$selfstring = kalyptusDataDict::addNamespace($class->{astNodeName})."::";
				} else {
					$selfstring = $class->{astNodeName}."Bridge::";
				}
				print HEADER $returnType,
					" ", $typeprefix, $class->{astNodeName}, "_", $function,
					"( ", $cparams, ");\n";
				print CLASS $returnType,
					" ", $typeprefix, $class->{astNodeName}, "_", $function,
					"( ", $cparams, "){\n";
			} else {
				if ( exists $class->{Pure} || $constructorCount == 0 ) {
					$selfstring = "((".kalyptusDataDict::addNamespace($class->{astNodeName})."*)instPointer)->";
				} else {
					$selfstring = "((".$class->{astNodeName}."Bridge*)instPointer)->";
				}
				print HEADER $returnType,
					" ", $typeprefix, $class->{astNodeName}, "_", $function,
					"(", $typeprefix, $class->{astNodeName}, "* instPointer", ($cparams eq "" ? "" : ","), $cparams, ");\n";
				print CLASS $returnType,
					" ", $typeprefix, $class->{astNodeName}, "_", $function,
					"( ", $typeprefix, $class->{astNodeName}, "* instPointer", ($cparams eq "" ? "" : ","), $cparams, "){\n";
    		}
			if ( $returnType =~ /^\s*void\s*$/ ) {
				print CLASS "\t", $selfstring, $name, "(", $cplusplusparams, ");\n\treturn;\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QBrush\s*$/ ) {
				print CLASS "\tQBrush _b= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QBrush(_b.color(),_b.style());\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QColorGroup\s*$/ ) {
				print CLASS "\tQColorGroup _c= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QColorGroup(_c.foreground(),_c.background(),_c.light(),_c.dark(),_c.mid(),_c.text(),_c.base());\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QDateTime\s*$/ ) {
				print CLASS "\tQDateTime _dt= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QDateTime (_dt.date(),_dt.time());\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QDate\s*$/ ) {
				print CLASS "\tQDate _d= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QDate(_d.year(),_d.month(),_d.day());\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QPen\s*$/ ) {
				print CLASS "\tQPen _b= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QPen(_b.color(),_b.width(),_b.style());\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QPoint\s*\&?\s*$/ ) {
				print CLASS "\tQPoint _p= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QPoint(_p.x(),_p.y());\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QRect\s*$/ ) {
				print CLASS "\tQRect _r= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QRect(_r.left(),_r.top(),_r.width(),_r.height());\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QSizePolicy\s*$/ ) {
				print CLASS "\tQSizePolicy _s= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QSizePolicy(_s.horData(),_s.verData(),_s.hasHeightForWidth());\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QSize\s*$/ ) {
				print CLASS "\tQSize _s= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QSize(_s.width(),_s.height());\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QStyle\s*$/ ) {
				print CLASS "\tQStyle * _s= \&", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ") _s;\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QTime\s*$/ ) {
				print CLASS "\tQTime _t= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QTime(_t.hour(),_t.minute(),_t.second(),_t.msec());\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QWMatrix\s*$/ ) {
				print CLASS "\tQWMatrix _m= ", $selfstring, $name, "(", $cplusplusparams, ");\n" ;
				print CLASS "\treturn (", $returnType, ")new QWMatrix(_m.m11(),_m.m12(),_m.m21(),_m.m22(),_m.dx(),_m.dy());\n}\n" ;
			} elsif (	($returnType =~ /qt_/ || $returnType =~ /kde_/)
						&& $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?(\w*)\s*$/ )
			{
				my $valueType = kalyptusDataDict::addNamespace($3);
				print CLASS "\treturn (", $returnType, ")new $valueType(", $selfstring, $name, "(", $cplusplusparams, "));\n}\n"; ;
			} elsif (	($returnType =~ /qt_/ || $returnType =~ /kde_/)
						&& $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?(\w*)\s*\&?\s*$/ )
			{
				my $constOpt = $2;
				my $valueType = kalyptusDataDict::addNamespace($3);
				print CLASS "\treturn (", $returnType, ") ($constOpt $valueType *)\&", $selfstring, $name, "(", $cplusplusparams, ");\n}\n"; ;
			} else {
				print CLASS "\treturn (", $returnType, ") ", $selfstring, $name, "(", $cplusplusparams, ");\n}\n" ;
			}
   		}
	} elsif( $type eq "enum" ) {
		# Convert each enum value to '#define <uppercased class name>_<enum name> <enum value>'
		my $enum = $m->{astNodeName};
		my %enumMap = ();
		# Add a C++ to C type mapping for this enum - ie an int in C
		$enum =~ s/\s//g;
		kalyptusDataDict::setctypemap($enum, 'int');
		$enum = $class->{astNodeName}."::".$enum;
		kalyptusDataDict::setctypemap($enum, 'int');
		
		my @enums = split(",", $m->{Params});
		my $enumCount = 0;
		foreach my $enum ( @enums ) {
			$enum =~ s/\s//g;
			if ( $enum =~ /(.*)=(.*)\s*(\||\&|>>|<<)\s*(.*)/ ) {
				# or'd, and'd or shifted pair of values
				print HEADER "#define ", uc($class->{astNodeName}), "_", $1, "\t",
					($enumMap{$2} eq "" ? $2 : $enumMap{$2}), $3, ($enumMap{$4} eq "" ? $4 : $enumMap{$4}), "\n";
				$enumMap{$1} = uc($class->{astNodeName})."_".$1;
			} elsif ( $enum =~ /(.*)=(.*)/ ) {
				print HEADER "#define ", uc($class->{astNodeName}), "_", $1, "\t", ($enumMap{$2} eq "" ? $2 : $enumMap{$2}), "\n",;
				$enumMap{$1} = uc($class->{astNodeName})."_".$1;
			} else {
				print HEADER "#define ", uc $class->{astNodeName}, "_",  $enum, "\t", $enumCount, "\n",;
				$enumMap{$enum} = uc($class->{astNodeName})."_".$enum;
				$enumCount++;
			}
		}
	}

}


1;

