/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <errno.h>
#include <string.h>
#include <signal.h> /* for sigignore(SIGPIPE) on unix */
#include <unistd.h>

#include <glib.h>
#include <gnet/gnet.h>

#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/sockets.h>

/*********************
**********************  Structs
*********************/

struct _PanSocket
{
	/* parent class */
	PanObject parent;

	/* public fields */
	time_t last_action_time;

	/* public fields */
	gboolean error;

	/* private fields */  
	gboolean need_auth;
	char * nntp_group_name;
	char * nntp_username;
	char * nntp_password;
	GString * line_buffer;
	GTcpSocket * gnet_socket;
	GIOChannel * gio_channel;

	/* statistics */
	char * host_address;
	char * host_name;
	unsigned int bytes_read;
	unsigned int bytes_written;
	time_t byte_count_start_time;
};

/*********************
**********************  Variables
*********************/

static gulong total_bytes = 0;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

PanSocket*
pan_socket_new (const char     * server_name,
                const char     * server_address,
                int              port)
{
	PanSocket * sock;

	/* sanity clause */
	g_return_val_if_fail (is_nonempty_string (server_name), NULL);
	g_return_val_if_fail (is_nonempty_string (server_address), NULL);
	g_return_val_if_fail (port>=0, NULL);

	/* needed on Unix */
#	ifdef SIGPIPE
	sigignore (SIGPIPE);
#	endif

	/* create the socket */
	sock = g_new0 (PanSocket, 1);	
	debug1 (DEBUG_PAN_OBJECT, "pan_socket_new: %p", sock);
	pan_socket_constructor (sock, pan_socket_destructor, server_name, server_address, port);
	return sock;
}


void
pan_socket_constructor (PanSocket             * sock,
			PanObjectDestructor     destructor,
			const char            * server_name,
			const char            * server_address,
			gint                    port)
{
	/* sanity clause */
	g_return_if_fail (sock!=NULL);
	g_return_if_fail (is_nonempty_string(server_name));
	g_return_if_fail (is_nonempty_string(server_address));
	g_return_if_fail (port>=0);

	/* constructor */
	pan_object_constructor (PAN_OBJECT(sock), destructor);
	sock->host_address = g_strdup (server_address);
	sock->host_name = g_strdup (server_name);
	sock->bytes_read = 0;
	sock->bytes_written = 0;
	sock->byte_count_start_time = time(0);
	sock->line_buffer = g_string_new (NULL);
	sock->error = FALSE;
	sock->need_auth = FALSE;
	sock->nntp_group_name = NULL;
	sock->nntp_username = NULL;
	sock->nntp_password = NULL;

	/* set up the serv_addr struct */
	errno = 0;
	sock->gnet_socket = gnet_tcp_socket_connect (server_address, port);
	if (sock->gnet_socket==NULL)
		log_add_va (LOG_ERROR, _("Connection to %s, port %d failed: %s"), server_address, port, g_strerror(errno));
	else {
		sock->gio_channel = gnet_tcp_socket_get_iochannel (sock->gnet_socket);
		if (g_io_channel_get_encoding (sock->gio_channel) != NULL)
			g_io_channel_set_encoding (sock->gio_channel, NULL, NULL);
		g_io_channel_set_buffered (sock->gio_channel, TRUE);
		g_io_channel_set_line_term (sock->gio_channel, "\n", 1);
		log_add_va (LOG_INFO, _("New connection %p for %s, port %d"), sock, server_address, port);
	}

        debug1 (DEBUG_PAN_OBJECT, "pan_socket_constructor: %p", sock);
}


void
pan_socket_destructor (PanObject *obj)
{
	PanSocket * sock;

	g_return_if_fail (obj != NULL);
	sock = PAN_SOCKET(obj);
	debug1 (DEBUG_SOCKET, "socket closed: %p", sock);
        debug1 (DEBUG_PAN_OBJECT, "pan_socket_destructor: %p", sock);

	/* free up our pieces */
	g_io_channel_unref (sock->gio_channel);
	gnet_tcp_socket_delete (sock->gnet_socket);
	g_free (sock->nntp_group_name);
	g_free (sock->nntp_username);
	g_free (sock->nntp_password);
	g_free (sock->host_address);
	g_free (sock->host_name);
	g_string_free (sock->line_buffer, TRUE);

	/* pass along to superclass */
	pan_object_destructor (obj);
}

 
void
pan_socket_set_nntp_auth (PanSocket            * socket,
                          gboolean               need_auth,
                          const gchar          * username,
                          const gchar          * password)
{
	g_return_if_fail (socket!=NULL);

	socket->need_auth = need_auth;
	replace_gstr (&socket->nntp_username, g_strdup(username));
	replace_gstr (&socket->nntp_password, g_strdup(password));
}


/*
 * get_server - read from a socket until \n
 */
int
pan_socket_getline (PanSocket    * sock,
                    const char  ** setme)
{
	gboolean done = FALSE;
	GError * err = NULL;
	GIOStatus status;
	debug_enter ("pan_socket_getline");

	/* sanity clause */
	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (!sock->error, -1);
	g_return_val_if_fail (setme!=NULL, -1);

	/* read the next line */
	g_string_truncate (sock->line_buffer, 0);
	while (!done) {
		sock->last_action_time = time(0);
		status = g_io_channel_read_line_string (sock->gio_channel, sock->line_buffer, NULL,  &err);
		switch (status) {
			case G_IO_STATUS_AGAIN:
				if (err != NULL)
					g_error_free (err);
				g_usleep (G_USEC_PER_SEC);
				continue;
			case G_IO_STATUS_ERROR:
			case G_IO_STATUS_EOF:
				if (err != NULL) {
					log_add_va (LOG_ERROR, _("Error reading from socket: %s"), err->message);
					g_error_free (err);
				}
				pan_socket_set_error_flag (sock, TRUE);
				return -1;
			case G_IO_STATUS_NORMAL:
				done = TRUE;
				break;
		}
	}

	sock->bytes_read += sock->line_buffer->len;
	total_bytes += sock->line_buffer->len;

	debug2 (DEBUG_SOCKET_INPUT,
	       "socket [%p] received [%s]", sock,
	       (char*)sock->line_buffer->str);

	*setme = (const char*) sock->line_buffer->str;
	debug_exit ("pan_socket_getline");
	return 0;
}


int 
pan_socket_putline (PanSocket     * sock,
                    const char    * line)
{
	gsize nbytes;
	gsize nleft;

	/* sanity clause */
	g_return_val_if_fail (is_nonempty_string(line), -1);
	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (!sock->error, -1);

	debug3 (DEBUG_SOCKET_OUTPUT,
	        "[%ld][socket %p putline][%s]", time(0), (void*)sock, line);

	/* try to write the string */
	nleft = nbytes = strlen (line);
	while (nleft>0)
	{
		gsize bytes_written = 0;
		GIOStatus status;
		GError * err = NULL;

		sock->last_action_time = time (0);
		status = g_io_channel_write_chars (sock->gio_channel, line, nleft, &bytes_written, &err);
		switch (status)
		{
			case G_IO_STATUS_AGAIN:
				if (err != NULL)
					g_error_free (err);
				g_usleep (G_USEC_PER_SEC);
				/* fall through */
			case G_IO_STATUS_NORMAL:
				nleft -= bytes_written;
				line += bytes_written;
				continue;
			case G_IO_STATUS_EOF:
			case G_IO_STATUS_ERROR:
				if (err != NULL) {
					log_add_va (LOG_ERROR, _("Error writing to socket: %s"), err->message);
					g_error_free (err);
				}
				pan_socket_set_error_flag (sock, TRUE);
				return -1;
		}
	}

	sock->bytes_written += nbytes;
	total_bytes += nbytes;
	g_io_channel_flush (sock->gio_channel, NULL);

	return nleft;
}

gboolean 
pan_socket_putline_va (PanSocket    *sock,
                       const gchar  *format,
                       ...)
{
	va_list args;
	char* line;
	int retval;

	g_return_val_if_fail (format!=NULL, -1);

	va_start(args, format);
	line = g_strdup_vprintf(format, args);
	va_end(args);

	retval = pan_socket_putline (sock, line);

	g_free (line);
	return retval;
}


/* Flush a socket; wait for a closing socket */
void
pan_socket_flush (PanSocket * ps)
{
	char buf[512];
	gsize bytes_read = 0;
	GError * err = NULL;
	GIOStatus status;

	for (;;)
	{
		status = g_io_channel_read_chars (ps->gio_channel, buf, sizeof(buf), &bytes_read, &err);
		switch (status)
		{
			case G_IO_STATUS_EOF:
				return;
			case G_IO_STATUS_AGAIN:
				continue;
			case G_IO_STATUS_NORMAL:
				if (bytes_read == 0)
					return;
				continue;
			case G_IO_STATUS_ERROR:
				log_add_va (LOG_ERROR, _("Error reading from socket: %s"), err->message);
				g_error_free (err);
				return;
		}
	}
}

/**
***
**/

gboolean
pan_socket_needs_auth (const PanSocket * sock)
{
	return sock->need_auth;
}

const char*
pan_socket_get_username (const PanSocket * sock)
{
	return sock->nntp_username;
}

const char*
pan_socket_get_password (const PanSocket * sock)
{
	return sock->nntp_password;
}

const char*
pan_socket_get_current_group (const PanSocket * sock)
{
	return sock->nntp_group_name;
}

void
pan_socket_set_current_group (PanSocket * sock, const char * group_name)
{
	replace_gstr (&sock->nntp_group_name, g_strdup(group_name));
}

const char*
pan_socket_get_server_name (const PanSocket * sock)
{
	return sock->host_name;
}

void
pan_socket_set_error_flag (PanSocket   * sock,
                           gboolean      err_state)
{
	sock->error = err_state;
}

gboolean
pan_socket_get_error_flag (const PanSocket * sock)
{
	return sock->error;
}

time_t
pan_socket_get_last_action_time  (const PanSocket * sock)
{
	return sock->last_action_time;
}

void pan_socket_set_last_action_time  (PanSocket * sock, time_t action_time)
{
	sock->last_action_time = action_time;
}

time_t
pan_socket_get_statistics_start_time (const PanSocket  * sock)
{
	return sock->byte_count_start_time;
}


/**
***
**/

void
pan_socket_reset_statistics (PanSocket* sock)
{
	sock->bytes_read = 0;
	sock->bytes_written = 0;
	sock->byte_count_start_time = time(0);
}

double
pan_socket_get_xfer_rate_KBps (const PanSocket* sock)
{
	const double bytes = sock ? sock->bytes_read + sock->bytes_written : 0;
	const double K = bytes / 1024.0;
	const time_t time_elapsed = sock ? time(0) - sock->byte_count_start_time: 0;
	const double KBps = time_elapsed ? (K / time_elapsed) : 0;
	return KBps;
}

gulong
pan_socket_get_total_xfer_K (void)
{
	return total_bytes / (gulong)1024;
}
