/*
 *  Copyright (C) 2003  Tommi Komulainen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation. 
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */ 

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gul-gui-option.h"

#include <bonobo/bonobo-i18n.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>

#include <string.h>

/**************************************************************************
 * GulGuiOption
 *
 * GulGuiOption is an option item that has a explicitly defined value and a
 * (translated) title for showing to the user.  By linking the title and the
 * value together, instead of relying on the order of the items, we can sort
 * the user visible titles according to user's locale.  The end result is
 * likely to look more sane than just simply translating the titles in place,
 * which can make the order of items look random.
 */
/**
 * gul_gui_option_new:
 * @title: localized title for the option
 * @value: the value
 *
 * Returns: a newly allocated #GulGuiOption item.
 */
GulGuiOption *
gul_gui_option_new (const char *title, const char *value)
{
	GulGuiOption *item;
	char          *tmp;

	item = g_new(GulGuiOption, 1);
	item->title = g_strdup (title);
	item->value = g_strdup (value);

	tmp = g_utf8_casefold (item->title, -1);
	item->key = g_utf8_collate_key (tmp, -1);
	g_free (tmp);

	return item;
}

static void
gul_gui_option_free (GulGuiOption *item)
{
	if (item != NULL)
	{
		g_free (item->title);
		g_free (item->value);
		g_free (item->key);
		g_free (item);
	}
}

/**
 * gul_gui_option_list_new:
 * @options: an array of #GulGuiOption items
 * @n_options: number of items in @options
 *
 * Creates a list of #GulGuiOption items from the array.
 *
 * Returns: a newly allocated list of #GulGuiOption items.
 */
GSList *
gul_gui_option_list_new (const GulGuiOption *options, guint n_options)
{
	GSList *list;
	guint   i;

	list = NULL;
	for (i = 0; i < n_options; i++)
	{
		const GulGuiOption *opt = &options[i];
		GulGuiOption *item;

		item = gul_gui_option_new (_(opt->title), opt->value);
		list = g_slist_prepend (list, item);
	}

	return gul_gui_option_list_sort (list);
}

void
gul_gui_option_list_free (GSList *options)
{
	g_slist_foreach (options, (GFunc)gul_gui_option_free, NULL);
	g_slist_free (options);
}

static int
gul_gui_option_cmp (gconstpointer a, gconstpointer b)
{
	const GulGuiOption *opt1 = a;
	const GulGuiOption *opt2 = b;
	return strcmp (opt1->key, opt2->key);
}

GSList *
gul_gui_option_list_insert_sorted (GSList *options, GulGuiOption *item)
{
	return g_slist_insert_sorted (options, item, gul_gui_option_cmp);
}

/**
 * gul_gui_option_list_sort:
 * @options: a list of #GulGuiOption items
 *
 * Sorts the list items by title in ascending order.
 *
 * Returns: the list sorted in ascending order according to locale.
 */
GSList *
gul_gui_option_list_sort (GSList *options)
{
	return g_slist_sort (options, gul_gui_option_cmp);
}

static int
gul_gui_option_cmp_value (gconstpointer a, gconstpointer b)
{
	const GulGuiOption *opt1  = a;
	const char         *value = b;
	return strcmp (opt1->value, value);
}

/**
 * gul_gui_option_list_find:
 * @options: a list of #GulGuiOption items
 * @value: value of an option to find
 *
 * Returns: the #GulGuiOption item that has value @value, or %NULL
 */
GulGuiOption *
gul_gui_option_list_find (GSList *options, const char *value)
{
	GSList *link;

	g_return_val_if_fail (value != NULL, NULL);

	link = g_slist_find_custom (options, value, gul_gui_option_cmp_value);
	if (link == NULL) return NULL;
	return link->data;
}

/**************************************************************************
 * Helper functions to populate GtkOptionMenus with GulGuiOptions.
 */
static const char *gul_gui_option_key = "GulGui::Option";

static void
set_option_in_menu_item (GtkWidget *item, GulGuiOption *option)
{
	g_return_if_fail (GTK_IS_MENU_ITEM(item));
	g_assert (option != NULL);
	g_object_set_data (G_OBJECT(item), gul_gui_option_key, option);
}

static GulGuiOption *
get_option_from_menu_item (GtkWidget *item)
{
	g_return_val_if_fail (GTK_IS_MENU_ITEM(item), NULL);
	return g_object_get_data (G_OBJECT(item), gul_gui_option_key);
}

/** gul_gui_option_menu_set_options:
 * @optionmenu: a #GtkOptionMenu
 * @options: a list of #GulGuiOption items
 *
 * Fills the @optionmenu with the items from the @options list removing any
 * existing old menu items from the @optionmenu.  Currently selected item can
 * be set with gul_gui_option_menu_set_value() and retrieved with
 * gul_gui_option_menu_get_option() or gul_gui_option_menu_get_value().
 *
 * If the value of a #GulGuiOption item is %NULL, the corresponding menu item
 * is made insensitive.
 *
 * The #GulGuiOption items in the list must persist until the @optionmenu is
 * destroyed.
 */
void
gul_gui_option_menu_set_options (GtkOptionMenu *optionmenu, GSList *options)
{
	GtkWidget  *menu;
	GSList     *l;
	guint       ndx;

	g_return_if_fail (GTK_IS_OPTION_MENU(optionmenu));

	menu = gtk_menu_new ();

	for (l = options, ndx = 0; l != NULL; l = l->next, ++ndx)
	{
		GulGuiOption *option = l->data;

		if (option != NULL && option->title != NULL)
		{
			GtkWidget *item;

			item = gtk_menu_item_new_with_label (option->title);
			set_option_in_menu_item (item, option);
			gtk_widget_show (item);

			gtk_widget_set_sensitive (GTK_WIDGET(item),
					          option->value != NULL);

			gtk_menu_shell_append (GTK_MENU_SHELL(menu), item);
		}
		else if (option == NULL)
		{
			g_warning ("%s: (%d) option == NULL", G_STRLOC, ndx);
		}
		else
		{
			g_warning ("%s: (%d) option->title == NULL", G_STRLOC, ndx);
		}
	}

	gtk_option_menu_set_menu (optionmenu, menu);
}

/**
 * gul_gui_option_menu_get_option:
 * @optionmenu: a #GtkOptionMenu
 *
 * Retrieves the currently selected #GulGuiOption item.
 *
 * Returns: the selected #GulGuiOption item or %NULL
 */
GulGuiOption *
gul_gui_option_menu_get_option (GtkOptionMenu *optionmenu)
{
	GtkWidget *menu;
	GtkWidget *item;

	g_return_val_if_fail (GTK_IS_OPTION_MENU(optionmenu), NULL);

	menu = gtk_option_menu_get_menu (optionmenu);

	item = gtk_menu_get_active (GTK_MENU(menu));
	g_return_val_if_fail (GTK_IS_MENU_ITEM(item), NULL);

	return get_option_from_menu_item (item);
}

/**
 * gul_gui_option_menu_get_value:
 * @optionmenu: a #GtkOptionMenu
 *
 * Retrieves the value of the currently selected #GulGuiOption item.
 *
 * Returns: the value of the selected #GulGuiOption item or %NULL
 */
const char *
gul_gui_option_menu_get_value (GtkOptionMenu *optionmenu)
{
	GulGuiOption *option;

	g_return_val_if_fail (GTK_IS_OPTION_MENU(optionmenu), NULL);

	option = gul_gui_option_menu_get_option (optionmenu);
	g_return_val_if_fail (option != NULL, NULL);

	return option->value;
}

/**
 * gul_gui_option_menu_set_value:
 * @optionmenu: a #GtkOptionMenu
 * @value: value of the item to set selected
 *
 * Select the #GulGuiOption item that has the value @value.
 */
void
gul_gui_option_menu_set_value (GtkOptionMenu *optionmenu, const char *value)
{
	GtkWidget  *menu;
	GList      *l;
	int         ndx;

	g_return_if_fail (GTK_IS_OPTION_MENU(optionmenu));
	g_return_if_fail (value != NULL);

	menu = gtk_option_menu_get_menu (optionmenu);
	l = GTK_MENU_SHELL(menu)->children;

	for (ndx = 0; l != NULL; l = l->next, ++ndx)
	{
		GulGuiOption *option;

		option = get_option_from_menu_item (l->data);
		if (option != NULL && option->value != NULL && !strcmp (option->value, value))
		{
			gtk_option_menu_set_history (optionmenu, ndx);
			return;
		}
		else if (option == NULL)
		{
			g_warning ("%s: (%d) option == NULL", G_STRLOC, ndx);
		}
	}
}

