/*
  Copyright (c) 2000 Caldera Systems

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "batchmode.h"
#include "screenpageimpl.h"
#include "kxdata.h"

#include <kcmdlineargs.h>

bool findMode( uint xResolution, uint yResolution, uint vertRefresh, KXModeData &mode )
{
    KXModeDataList modes = KXData::global()->currentModes();
    modes += KXData::vesaModes();

    KXModeDataList::ConstIterator it = modes.begin();
    KXModeDataList::ConstIterator end = modes.end();
    for (; it != end; ++it )
        if ( ScreenPage::checkCompatibility( *it, KXData::global()->currentMonitors()[ 0 ],
                                             KXData::global()->currentVideoCards()[ 0 ] ) && 
             (*it).xResolution() == xResolution &&
             (*it).yResolution() == yResolution &&
             (*it).vRefresh() == vertRefresh )
        {
            mode = *it;
            return true;
        }

    return false;
}

void processBatchMode( KCmdLineArgs *args )
{
      if ( args->isSet( "pointer" ) )
      {
          QStringList pointerConfig = QStringList::split( ',', QString::fromLocal8Bit( args->getOption( "pointer" ) ), true );

          if ( pointerConfig.count() == 3 )
          {
              KXPointerData pointer = KXData::global()->currentPointers()[ 0 ];

              KXPointerModel model = pointer.pointerModel();

              if ( !pointerConfig[ 0 ].isEmpty() )
                  model.setXFreeName( pointerConfig[ 0 ] );

              if ( !pointerConfig[ 1 ].isEmpty() )
              {
                  bool wheelsOk = false;
                  int wheels = pointerConfig[ 0 ].toInt( &wheelsOk );
                  if ( wheelsOk )
                      model.setWheels( wheels );
              }

              if ( !pointerConfig[ 2 ].isEmpty() )
                  pointer.setPort( pointerConfig[ 2 ] );

              pointer.setPointerModel( model );

              KXData::global()->currentPointers()[ 0 ] = pointer;
          }
          else
              qDebug( "BATCH MODE WARNING: Invalid number of arguments passed for pointer configuration. %i excepted, got %i", 3, pointerConfig.count() );
      }

      if ( args->isSet( "keyboard" ) )
      {
          QStringList keyboardConfig = QStringList::split( ',', QString::fromLocal8Bit( args->getOption( "keyboard" ) ), true );

          if ( keyboardConfig.count() == 2 )
          {
              KXKeyboardData keyboard = KXData::global()->currentKeyboards()[ 0 ];

              if ( !keyboardConfig[ 0 ].isEmpty() )
              {
                  KXKeyboardModel model = keyboard.model();
                  model.setId( keyboardConfig[ 0 ] );
                  keyboard.setModel( model );
              }

              if ( !keyboardConfig[ 1 ].isEmpty() )
              {
                  KXKeyboardLayout layout = keyboard.layout();
                  layout.setId( keyboardConfig[ 1 ] );
                  keyboard.setLayout( layout );
              }

              KXData::global()->currentKeyboards()[ 0 ] = keyboard;
          }
          else
              qDebug( "BATCH MODE WARNING: Invalid number of arguments passed for keyboard configuration. %i excepted, got %i", 2, keyboardConfig.count() );

      }

      if ( args->isSet( "serverflags" ) )
      {
          QString allowZapOption = QString::fromLocal8Bit( args->getOption( "serverflags" ) );
          if ( !allowZapOption.isEmpty() )
          {
              bool allowZapOk = false;
              int allowZap = allowZapOption.toInt( &allowZapOk );
              if ( allowZapOk )
                  KXData::global()->generalServerData().setZapAllowed( allowZap == 1 );
          }
      }

      if( args->isSet( "monitor" ) )
      {
          QStringList monitorConfig = QStringList::split( ',', QString::fromLocal8Bit( args->getOption( "monitor" ) ), true );
          if ( monitorConfig.count() == 4 )
          {
              KXMonitorData monitor = KXData::global()->currentMonitors()[ 0 ];

              float hSyncMin = monitor.hSyncMin();
              float hSyncMax = monitor.hSyncMax();
              float vSyncMin = monitor.vSyncMin();
              float vSyncMax = monitor.vSyncMax();

              bool conversionOk = false;
              float temp = 0.0;

              if ( !monitorConfig[ 0 ].isEmpty() )
              {
                  temp = monitorConfig[ 0 ].toFloat( &conversionOk );
                  if ( conversionOk )
                      hSyncMin = temp;
              }

              if ( !monitorConfig[ 1 ].isEmpty() )
              {
                  temp = monitorConfig[ 1 ].toFloat( &conversionOk );
                  if ( conversionOk )
                      hSyncMax = temp;
              }

              if ( !monitorConfig[ 2 ].isEmpty() )
              {
                  temp = monitorConfig[ 2 ].toFloat( &conversionOk );
                  if ( conversionOk )
                      vSyncMin = temp;
              }

              if ( !monitorConfig[ 3 ].isEmpty() )
              {
                  temp = monitorConfig[ 3 ].toFloat( &conversionOk );
                  if ( conversionOk )
                      vSyncMax = temp;
              }

              monitor.setHSync( hSyncMin, hSyncMax );
              monitor.setVSync( vSyncMin, vSyncMax );
              KXData::global()->currentMonitors()[ 0 ] = monitor;
          }
          else
              qDebug( "BATCH MODE WARNING: Invalid number of arguments passed for monitor configuration. %i excepted, got %i", 4, monitorConfig.count() );
      }

      if ( args->isSet( "screen" ) )
      {
          QStringList screenConfig = QStringList::split( ',', QString::fromLocal8Bit( args->getOption( "screen" ) ), true );

          if ( screenConfig.count() < 1 )
              qDebug( "BATCH MODE WARNING: Invalid number of arguments passed for screen configuration. %s excepted, got %i", ">=1", screenConfig.count() );
          else
          {
              KXScreenData screen = KXData::global()->currentScreens()[ 0 ];

              if ( !screenConfig[ 0 ].isEmpty() )
              {
                  bool conversionOk = false;
                  int defaultDepth = screenConfig[ 0 ].toInt( &conversionOk );
                  if ( conversionOk )
                      screen.setDefaultDepth( defaultDepth );
              }

              screenConfig.remove( screenConfig.begin() );

              if ( screenConfig.count() > 0 && screenConfig.count() % 4 != 0 )
                  qDebug( "BATCH MODE WARNING: Invalid number of arguments passed for in display info of screen configuration." );
              else
              {
                  KXDisplayDataList &displays = screen.displays();

                  QStringList::ConstIterator it = screenConfig.begin();
                  QStringList::ConstIterator end = screenConfig.end();
                  for (; it != end; )
                  {
                      bool conversionOk = false;
                      uint depth = (*it).toInt( &conversionOk );
                      if ( !conversionOk )
                      {
                          ++it; ++it; ++it; ++it;
                          continue;
                      }

                      ++it;

                      uint xResolution = (*it).toInt( &conversionOk );

                      if ( !conversionOk )
                      {
                          ++it; ++it; ++it;
                          continue;
                      }

                      ++it ;

                      uint yResolution = (*it).toInt( &conversionOk );

                      if ( !conversionOk )
                      {
                          ++it; ++it;;
                          continue;
                      }

                      ++it;

                      uint vRefresh = (*it).toInt( &conversionOk );

                      ++it;

                      if ( !conversionOk )
                          continue;

                      KXModeData mode;

                      if ( !findMode( xResolution, yResolution, vRefresh, mode ) )
                          continue;

                      KXDisplayDataList::Iterator it = displays.begin();
                      KXDisplayDataList::Iterator end = displays.end();
                      for (; it != end; ++it )
                      {
                          if ( (*it).depth() == depth )
                          {
                              QStringList modes = (*it).modes();
                              modes[ 0 ] = mode.name();
                              (*it).setModes( modes );
                          }
                      }

                      if ( !mode.node() )
                          KXData::global()->currentModes() += mode;
                  }
              }

              KXData::global()->currentScreens()[ 0 ] = screen;
          }
      }

      if ( args->isSet( "videocard" ) )
      {
          QStringList videoCardConfig = QStringList::split( ',', QString::fromLocal8Bit( args->getOption( "videocard" ) ), true );

          if ( videoCardConfig.count() == 3 )
          {
              KXVideoCardData vcard = KXData::global()->currentVideoCards()[ 0 ];

              if ( !videoCardConfig[ 0 ].isEmpty() )
                  vcard.setDriver( videoCardConfig[ 0 ] );

              if ( !videoCardConfig[ 1 ].isEmpty() )
                  vcard.setVendor( videoCardConfig[ 1 ] );

              if ( !videoCardConfig[ 2 ].isEmpty() )
                  vcard.setModel( videoCardConfig[ 2 ] );

              KXData::global()->currentVideoCards()[ 0 ] = vcard;
          }
          else
              qDebug( "BATCH MODE WARNING: Invalid number of arguments passed for videocard configuration. %i excepted, got %i", 3, videoCardConfig.count() );
      }

}
