/*****************************************************************

Copyright (c) 2000-2001 Matthias Ettrich <ettrich@kde.org>
              2000-2001 Matthias Elter   <elter@kde.org>
       	      2001      Carsten Pfeiffer <pfeiffer@kde.org>
              2001      Martijn Klingens <mklingens@yahoo.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <qpixmap.h>
#include <qtoolbutton.h>
#include <qpainter.h>
#include <qtooltip.h>

#include <qpopupmenu.h>
#include <krun.h>

#include <dcopclient.h>

#include <kglobal.h>
#include <kapplication.h>
#include <kiconloader.h>
#include <kdebug.h>
#include <klocale.h>
#include <kwinmodule.h>

#include "systemtrayapplet.h"
#include "systemtrayapplet.moc"

#include <X11/Xlib.h>

extern "C"
{
    KPanelApplet* init(QWidget *parent, const QString& configFile)
    {
	KGlobal::locale()->insertCatalogue("ksystemtrayapplet");
        return new SystemTrayApplet(configFile, KPanelApplet::Normal,
                                    0, parent, "ksystemtrayapplet");
    }
}

SystemTrayApplet::SystemTrayApplet(const QString& configFile, Type type, int actions,
				   QWidget *parent, const char *name)
  : KPanelApplet(configFile, type, actions, parent, name)
{
    m_Wins.setAutoDelete(true);
    setFrameStyle(Panel | Sunken);
    setBackgroundMode(X11ParentRelative);

    kwin_module = new KWinModule(this);

    // register existing tray windows
    const QValueList<WId> systemTrayWindows = kwin_module->systemTrayWindows();
    bool existing = false;
    for (QValueList<WId>::ConstIterator it = systemTrayWindows.begin(); it!=systemTrayWindows.end(); ++it ) {
        QXEmbed* emb = new QXEmbed(this);
        emb->setAutoDelete(false);
        emb->setBackgroundMode(X11ParentRelative);
        connect(emb, SIGNAL(embeddedWindowDestroyed()), SLOT(updateTrayWindows()));
        m_Wins.append(emb);

        emb->embed(*it);
        emb->resize(24, 24);
        emb->show();
        existing = true;
    }
    if (existing)
        layoutTray();

    // the KWinModule notifies us when tray windows are added or removed
    connect( kwin_module, SIGNAL( systemTrayWindowAdded(WId) ),
             this, SLOT( systemTrayWindowAdded(WId) ) );
    connect( kwin_module, SIGNAL( systemTrayWindowRemoved(WId) ),
             this, SLOT( updateTrayWindows() ) );

    QCString screenstr;
    screenstr.setNum(qt_xscreen());
    QCString trayatom = "_NET_SYSTEM_TRAY_S" + screenstr;

    Display *display = qt_xdisplay();
    
    net_system_tray_selection = XInternAtom( display, trayatom, FALSE );
    net_system_tray_opcode = XInternAtom( display, "_NET_SYSTEM_TRAY_OPCODE", FALSE );

    // Acquire system tray
    XSetSelectionOwner(display,
                       net_system_tray_selection,
                       winId(),
                       CurrentTime);

    WId root = qt_xrootwin();

    if (XGetSelectionOwner (display, net_system_tray_selection) == winId())
    {
        XClientMessageEvent xev;

        xev.type = ClientMessage;
        xev.window = root;

        xev.message_type = XInternAtom (display, "MANAGER", False);
        xev.format = 32;
        xev.data.l[0] = CurrentTime;
        xev.data.l[1] = net_system_tray_selection;
        xev.data.l[2] = winId();
        xev.data.l[3] = 0;        /* manager specific data */
        xev.data.l[4] = 0;        /* manager specific data */

        XSendEvent (display, root, False, StructureNotifyMask, (XEvent *)&xev);

    } 
}

SystemTrayApplet::~SystemTrayApplet()
{
    kdDebug() << "SystemTrayApplet::~SystemTrayApplet" << endl;
    m_Wins.clear();
}

bool SystemTrayApplet::x11Event( XEvent *e ) 
{
#define SYSTEM_TRAY_REQUEST_DOCK    0
#define SYSTEM_TRAY_BEGIN_MESSAGE   1
#define SYSTEM_TRAY_CANCEL_MESSAGE  2
    if ( e->type == ClientMessage ) {
        if ( e->xclient.message_type == net_system_tray_opcode &&
             e->xclient.data.l[1] == SYSTEM_TRAY_REQUEST_DOCK) {
            systemTrayWindowAdded(e->xclient.data.l[2]);
            return true;
        }
    }
    return KPanelApplet::x11Event( e ) ;
}

void SystemTrayApplet::mousePressEvent(QMouseEvent* e)
{
    propagateMouseEvent(e);
}

void SystemTrayApplet::mouseReleaseEvent(QMouseEvent* e)
{
    propagateMouseEvent(e);
}

void SystemTrayApplet::mouseDoubleClickEvent(QMouseEvent* e)
{
    propagateMouseEvent(e);
}

void SystemTrayApplet::mouseMoveEvent(QMouseEvent* e)
{
    propagateMouseEvent(e);
}

void SystemTrayApplet::propagateMouseEvent(QMouseEvent* e)
{
    if ( !isTopLevel()  ) {
	QMouseEvent me(e->type(), mapTo( topLevelWidget(), e->pos() ),
		       e->globalPos(), e->button(), e->state() );
	QApplication::sendEvent( topLevelWidget(), &me );
    }
}

void SystemTrayApplet::systemTrayWindowAdded( WId w )
{
    QXEmbed* emb = new QXEmbed(this);
    emb->setAutoDelete(false);
    emb->setBackgroundMode(X11ParentRelative);
    connect(emb, SIGNAL(embeddedWindowDestroyed()), SLOT(updateTrayWindows()));
    m_Wins.append(emb);

    emb->embed(w);
    emb->resize(24, 24);
    emb->show();

    layoutTray();
    emit updateLayout();
}

void SystemTrayApplet::updateTrayWindows()
{
    QXEmbed* emb = m_Wins.first();
    while ((emb = m_Wins.current()) != 0L) {
        WId wid = emb->embeddedWinId();
        if ((wid == 0) || !kwin_module->systemTrayWindows().contains(wid))
            m_Wins.remove(emb);
        else
            m_Wins.next();
    }
    layoutTray();
    emit updateLayout();
}

int SystemTrayApplet::widthForHeight(int h) const
{
    int ret;

    if (h < 48)
        ret = m_Wins.count() * 24 + 4;
    else
        ret = ( (m_Wins.count()+1) / 2 ) * 24 + 4;

    ret = (ret >= 28) ? ret : 0;
    return ret;
}

int SystemTrayApplet::heightForWidth(int w) const
{
    int ret;

    if (w < 48)
        ret =  m_Wins.count() * 24 + 4;
    else
        ret = ( (m_Wins.count()+1) / 2 ) * 24 + 4;

    ret = (ret >= 28) ? ret : 0;
    return ret;
}

void SystemTrayApplet::resizeEvent( QResizeEvent* )
{
    layoutTray();
}

void SystemTrayApplet::layoutTray()
{
    if (m_Wins.count() == 0)
        return;

    int i;

    QXEmbed* emb;
    int col = 0;

    if (orientation() == Vertical) {
        i = 0;
        for (emb = m_Wins.first(); emb != 0L; emb = m_Wins.next()) {
            if ( (m_Wins.count() == 1) || width() < 48 )
                emb->move(width() / 2 - 12, 2 + i*24);
            else {
                emb->move(((i%2) == 0) ? 2 : width() - 26, 2 + col*24);
                if ( (i%2) != 0 )
                    ++col;
            }
            i++;
        }
    }
    else {
        i = 0;
        for (emb = m_Wins.first(); emb != 0L; emb = m_Wins.next()) {
            if ( (m_Wins.count() == 1) || height() < 48 )
                emb->move(2 + i*24, height() / 2 - 12);
            else {
                emb->move(2 + col*24, ((i%2) == 0) ? 2 : height() - 26);
                if ( (i%2) != 0 )
                    ++col;
            }
            i++;
        }
    }
    updateGeometry();
}
