#include <qradiobutton.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <qbuttongroup.h>
#include <qlistview.h>
#include <qhbox.h>

#include <kurl.h>
#include <kmessagebox.h>
#include <kiconloader.h>

#include "mailsender.h"
#include "kbbprefs.h"
#include "kbbmainwindow.h"
#include "serverconfigdialog.h"

#include "preferencesdialog.h"

PreferencesDialog::PreferencesDialog( QWidget* parent,  const char* name )
    : KDialogBase ( IconList, i18n("Preferences"), Ok|Apply|Cancel, Ok,
                    parent, name, false, true )
{
    setupServerPage();
    setupAdvancedPage();

    readConfig();

    connect( mBugzillaURLLineEdit, SIGNAL( textChanged ( const QString & ) ),
             SLOT( slotBugzillaURLChanged( const QString & ) ) );
    slotBugzillaURLChanged( mBugzillaURLLineEdit->text() );
}

PreferencesDialog::~PreferencesDialog()
{
}

void PreferencesDialog::setupServerPage()
{
  QFrame *topFrame = addPage( i18n("Servers"), 0,
                                DesktopIcon( "gohome", KIcon::SizeMedium ) );

  QBoxLayout *layout = new QVBoxLayout( topFrame );

  mServerList = new QListView( topFrame );
  mServerList->addColumn( i18n("Name") );
  mServerList->addColumn( i18n("Base URL") );
  layout->addWidget( mServerList );

  QHBox *buttonBox = new QHBox( topFrame );
  layout->addWidget( buttonBox );

  QPushButton *addButton = new QPushButton( i18n("Add Server..."), buttonBox );
  connect( addButton, SIGNAL( clicked() ), SLOT( addServer() ) );

  QPushButton *editButton = new QPushButton( i18n("Edit Server..."), buttonBox );
  connect( editButton, SIGNAL( clicked() ), SLOT( editServer() ) );

  QPushButton *removeButton = new QPushButton( i18n("Remove Server"), buttonBox );
  connect( removeButton, SIGNAL( clicked() ), SLOT( removeServer() ) );

  QGroupBox *serverBox = new QGroupBox( 1, Horizontal,
                                        i18n( "Bugzilla Server URL" ),
                                        topFrame );
  layout->addWidget( serverBox );

  mBugzillaURLLineEdit = new QLineEdit( serverBox );
}

void PreferencesDialog::setupAdvancedPage()
{
  QFrame *topFrame = addPage( i18n("Advanced"), 0,
                                DesktopIcon( "misc", KIcon::SizeMedium ) );

  QBoxLayout *layout = new QVBoxLayout( topFrame );

  QButtonGroup *mailGroup = new QButtonGroup( 1, Horizontal,
                                              i18n( "Mail Client" ), topFrame );
  layout->addWidget( mailGroup );

  mKMailButton = new QRadioButton( i18n( "&KMail" ), mailGroup );
  mDirectButton = new QRadioButton( i18n( "D&irect" ), mailGroup );
  mSendmailButton = new QRadioButton( i18n( "&Sendmail" ), mailGroup );

  mShowClosedCheckBox = new QCheckBox( i18n( "Show closed bugs" ), topFrame );
  layout->addWidget( mShowClosedCheckBox );

  mShowWishesCheckBox = new QCheckBox( i18n( "Show wishes" ), topFrame );
  layout->addWidget( mShowWishesCheckBox );

  mSendBccCheckBox = new QCheckBox( i18n( "Send BCC to myself" ), topFrame );
  layout->addWidget( mSendBccCheckBox );
}

void PreferencesDialog::slotBugzillaURLChanged( const QString &text)
{
    enableButtonApply( !text.isEmpty() );
}

void PreferencesDialog::setDefaults()
{
    KBBPrefs::instance()->setDefaults();
    readConfig();
}

void PreferencesDialog::slotOk()
{
    KURL data( mBugzillaURLLineEdit->text() );
    if ( data.isMalformed() )
    {
	KMessageBox::error( this, i18n( "Malformed URL! Please check again." ) );
        return;
    }
    writeConfig();
    accept();
    KBBMainWindow *main=(KBBMainWindow*)parent();
    main->updatePackage();
}

void PreferencesDialog::slotCancel()
{
    hide();
}

void PreferencesDialog::addServer()
{
  ServerConfigDialog *dlg = new ServerConfigDialog( this );
  int result = dlg->exec();
  if ( result == QDialog::Accepted ) {
    new QListViewItem( mServerList, dlg->serverName(), dlg->serverUrl() );
  }
}

void PreferencesDialog::editServer()
{
  QListViewItem *item = mServerList->currentItem();
  if ( !item ) return;

  ServerConfigDialog *dlg = new ServerConfigDialog( this );
  dlg->setServer( item->text( 0 ), item->text( 1 ) );

  int result = dlg->exec();
  if ( result == QDialog::Accepted ) {
    item->setText( 0, dlg->serverName() );
    item->setText( 1, dlg->serverUrl() );
  }
}

void PreferencesDialog::removeServer()
{
  QListViewItem *item = mServerList->currentItem();
  if ( !item ) return;

  delete item;
}

void PreferencesDialog::readConfig()
{
    int client = KBBPrefs::instance()->mMailClient;
    switch(client) {
      default:
      case MailSender::KMail:
        mKMailButton->setChecked(true);
        break;
      case MailSender::Sendmail:
        mSendmailButton->setChecked(true);
        break;
      case MailSender::Direct:
        mDirectButton->setChecked(true);
        break;
    }
    mShowClosedCheckBox->setChecked( KBBPrefs::instance()->mShowClosedBugs );
    mShowWishesCheckBox->setChecked( KBBPrefs::instance()->mShowWishes );
    mSendBccCheckBox->setChecked( KBBPrefs::instance()->mSendBCC );
    mBugzillaURLLineEdit->setText( KBBPrefs::instance()->mBugzillaURL );
}

void PreferencesDialog::writeConfig()
{
    MailSender::MailClient client = MailSender::KMail;

    if (mKMailButton->isChecked()) client = MailSender::KMail;
    if (mSendmailButton->isChecked()) client = MailSender::Sendmail;
    if (mDirectButton->isChecked()) client = MailSender::Direct;

    KBBPrefs::instance()->mMailClient = client;
    KBBPrefs::instance()->mShowClosedBugs = mShowClosedCheckBox->isChecked();
    KBBPrefs::instance()->mShowWishes = mShowWishesCheckBox->isChecked();
    KBBPrefs::instance()->mSendBCC = mSendBccCheckBox->isChecked();
    KBBPrefs::instance()->mBugzillaURL = mBugzillaURLLineEdit->text();
    KBBPrefs::instance()->writeConfig();
}

#include "preferencesdialog.moc"
