# -*- encoding: UTF-8 -*-

import uno
import re
import sys
import os
import traceback
from com.sun.star.text.TextMarkupType import PROOFREADING
from com.sun.star.beans import PropertyValue

pkg = "pt_BR"
lang = "pt"
locales = {'pt-BR': ['pt', 'BR', '']}
version = "0.4.3"
author = "Raimundo Santos Moura"
name = "Lightproof grammar checker (Portuguese Brazilian)"

import lightproof_handler_pt_BR

# loaded rules (check for Update mechanism of the editor)
try:
    langrule
except NameError:
    langrule = {}

# ignored rules
ignore = {}

# cache for morphogical analyses
analyses = {}
stems = {}
suggestions = {}

# assign Calc functions
calcfunc = None

SMGR = None

# check settings
def option(lang, opt):
    return lightproof_handler_pt_BR.get_option(lang.Language + "_" + lang.Country, opt)

# filtering affix fields (ds, is, ts etc.)
def onlymorph(st):
    if st is not None:
        st = re.sub(r"^.*(st:|po:)", r"\\1", st) # keep last word part
        st = re.sub(r"\\b(?=[dit][sp]:)","@", st) # and its affixes
        st = re.sub(r"(?<!@)\\b\w\w:\w+","", st).replace('@','').strip()
    return st

# if the pattern matches all analyses of the input word,
# return the last matched substring
def _morph(rLoc, word, pattern, all, onlyaffix):
    global analyses
    if not word:
        return None
    if word not in analyses:
        x = spellchecker.spell(u"<?xml?><query type='analyze'><word>" + word + "</word></query>", rLoc, ())
        if not x:
            return None
        t = x.getAlternatives()
        if not t:
            t = [""]
        analyses[word] = t[0].split("</a>")[:-1]
    a = analyses[word]
    result = None
    p = re.compile(pattern)
    for i in a:
        if onlyaffix:
            i = onlymorph(i)
        result = p.search(i)
        if result:
            result = result.group(0)
            if not all:
                return result
        elif all:
            return None
    return result

def morph(rLoc, word, pattern, all=True):
    return _morph(rLoc, word, pattern, all, False)

def affix(rLoc, word, pattern, all=True):
    return _morph(rLoc, word, pattern, all, True)

def spell(rLoc, word):
    if not word:
        return None
    return spellchecker.isValid(word, rLoc, ())

# get the tuple of the stem of the word or an empty array
def stem(rLoc, word):
    global stems
    if not word:
        return []
    if word not in stems:
        x = spellchecker.spell(u"<?xml?><query type='stem'><word>" + word + "</word></query>", rLoc, ())
        if not x:
            return []
        t = x.getAlternatives()
        if not t:
            t = []
        stems[word] = list(t)
    return stems[word]

# get the tuple of the morphological generation of a word or an empty array
def generate(rLoc, word, example):
    if not word:
        return []
    x = spellchecker.spell(u"<?xml?><query type='generate'><word>" + word + "</word><word>" + example + "</word></query>", rLoc, ())
    if not x:
        return []
    t = x.getAlternatives()
    if not t:
        t = []
    return list(t)

# get suggestions
def suggest(rLoc, word):
    global suggestions
    if not word:
        return word
    if word not in suggestions:
        x = spellchecker.spell("_" + word, rLoc, ())
        if not x:
            return word
        t = x.getAlternatives()
        suggestions[word] = "\\n".join(t)
    return suggestions[word]

# get the nth word of the input string or None
def word(s, n):
    a = re.match(r"(?u)( [-.\w%%]+){" + str(n-1) + r"}( [-.\w%%]+)", s)
    if not a:
        return ''
    return a.group(2)[1:]

# get the (-)nth word of the input string or None
def wordmin(s, n):
    a = re.search(r"(?u)([-.\w%%]+ )([-.\w%%]+ ){" + str(n-1) + "}$", s)
    if not a:
        return ''
    return a.group(1)[:-1]

def calc(funcname, par):
    global calcfunc
    global SMGR
    if calcfunc is None:
        calcfunc = SMGR.createInstance( "com.sun.star.sheet.FunctionAccess")
        if calcfunc is None:
                return None
    return calcfunc.callFunction(funcname, par)

def proofread( nDocId, TEXT, LOCALE, nStartOfSentencePos, nSuggestedSentenceEndPos, rProperties ):
    global ignore
    aErrs = []
    s = TEXT[nStartOfSentencePos:nSuggestedSentenceEndPos]
    for i in get_rule(LOCALE).dic:
        # 0: regex,  1: replacement,  2: message,  3: condition,  4: ngroup,  (5: oldline),  6: case sensitive ?
        if i[0] and str(i[0]) not in ignore:
            for m in i[0].finditer(s):
                try:
                    if not i[3] or eval(i[3]):
                        aErr = uno.createUnoStruct( "com.sun.star.linguistic2.SingleProofreadingError" )
                        aErr.nErrorStart        = nStartOfSentencePos + m.start(i[4]) # nStartOfSentencePos
                        aErr.nErrorLength       = m.end(i[4]) - m.start(i[4])
                        aErr.nErrorType         = PROOFREADING
                        aErr.aRuleIdentifier    = str(i[0])
                        iscap = (i[-1] and m.group(i[4])[0:1].isupper())
                        if i[1][0:1] == "=":
                            aErr.aSuggestions = tuple(cap(eval(i[1][1:]).replace('|', "\n").split("\n"), iscap, LOCALE))
                        elif i[1] == "_":
                            aErr.aSuggestions = ()
                        else:
                            aErr.aSuggestions = tuple(cap(m.expand(i[1]).replace('|', "\n").split("\n"), iscap, LOCALE))
                        comment = i[2]
                        if comment[0:1] == "=":
                            comment = eval(comment[1:])
                        else:
                            comment = m.expand(comment)
                        aErr.aShortComment      = comment.replace('|', '\n').replace('\\n', '\n').split("\n")[0].strip()
                        aErr.aFullComment       = comment.replace('|', '\n').replace('\\n', '\n').split("\n")[-1].strip()
                        if "://" in aErr.aFullComment:
                            p = PropertyValue()
                            p.Name = "FullCommentURL"
                            p.Value = aErr.aFullComment
                            aErr.aFullComment = aErr.aShortComment
                            aErr.aProperties        = (p,)
                        else:
                            aErr.aProperties        = ()
                        aErrs = aErrs + [aErr]
                except Exception as e:
                    if len(i) == 7:
                        raise Exception(str(e), i[5])
                    raise

    return tuple(aErrs)

def cap(a, iscap, rLoc):
    if iscap:
        for i in range(0, len(a)):
            if a[i][0:1] == "i":
                if rLoc.Language == "tr" or rLoc.Language == "az":
                    a[i] ="\u0130" + a[i][1:]
                elif a[i][1:2] == "j" and rLoc.Language == "nl":
                    a[i] = "IJ" + a[i][2:]
                else:
                    a[i] = "I" + a[i][1:]
            else:
                a[i] = a[i].capitalize()
    return a

def compile_rules(dic):
    # compile regular expressions
    for i in dic:
        try:
            if re.compile(r"[\(][?]iu[)]").match(i[0]):
                i += [True]
                i[0] = re.sub(r"[\(][?]iu[)]", "(?u)", i[0])
            else:
                i += [False]
            i[0] = re.compile(i[0])
        except Exception:
            if 'PYUNO_LOGLEVEL' in os.environ:
                print("Lightproof: bad regular expression: ", traceback.format_exc())
            i[0] = None

def get_rule(loc):
    try:
        return langrule[pkg]
    except Exception:
        langrule[pkg] = __import__("lightproof_" + pkg)
        compile_rules(langrule[pkg].dic)
    return langrule[pkg]

def get_path():
    return os.path.join(os.path.dirname(sys.modules[__name__].__file__), __name__ + ".py")

# [code]

# Abreviacoes em Portugues - Brasil
abbrev = re.compile("(?i)\\b([a-zA-Z]|it|ít|Jr|lb|ll|lr|mk|MM|Mr|ob|Ob|of|Of|op|Op|or|oz|pa|pç|Pe|pf|pg|pl|pm|pp|pr|Pr|ql|Rd|rg|sc|sf|sg|sr|ss|SS|tr|TT|tv|un|vb|vc|vj|Vj|vl|vo|zo|abr|Abr|abs|adj|adm|adv|alt|ant|art|Art|ass|Ass|atm|Att|aux|Aux|bat|bbl|Bel|bíb|bim|biz|bol|bot|btl|bud|cab|caç|cad|caf|câm|can|cap|car|cat|Cat|cau|cav|cel|cên|cet|cfr|cid|cig|cin|cir|cit|civ|cob|cód|col|Col|con|côn|Côn|cop|coq|cos|cot|côv|ctv|cul|cut|cvs|dad|dat|dec|Dec|ded|def|dem|dep|Dep|des|det|dev|dic|Dic|did|dif|Dif|dim|din|dir|doc|dog|dól|dom|dór|dra|Dra|Drs|dur|ecd|ecl|écl|edd|emb|Emb|emp|enc|End|enf|eng|Eng|ens|ent|enx|esc|Esc|esl|esp|Esp|esq|est|Est|etc|etn|étn|euf|eur|Exa|exe|exp|ext|fáb|fac|fam|fed|fem|fen|fer|fev|Fev|fig|fil|fin|fís|fls|fog|fol|fos|fot|fov|fss|fut|gal|Gal|gar|gat|gav|gen|Gen|gên|ger|gin|gír|gót|gov|Gov|grd|guz|hab|hag|her|het|hib|híb|hig|hip|híp|hom|hon|hot|iat|ict|igr|imp|inc|ind|índ|inf|ing|ins|int|inv|iog|irl|irr|isl|isr|iug|jam|jan|jap|jav|jes|joc|jog|jud|jul|jun|jur|lab|lad|lâm|lap|lat|leg|lég|let|lex|lib|lig|lim|lin|líq|lit|Lit|liv|loc|log|lóg|loq|Ltd|lug|lut|luv|maç|mad|mag|mai|Mai|maj|Maj|man|map|maq|máq|mat|Mat|máx|mct|mec|med|Med|méd|mem|mer|met|mex|mín|mit|mob|mod|mon|mov|Mrs|mun|mús|nac|nap|nat|náu|nav|naz|neg|nep|nig|nob|nom|nor|nov|Nov|núm|obr|Obr|obs|Obs|oft|onç|opp|ópt|ord|org|orn|ort|our|out|pag|pág|Pág|paq|pça|Pça|pdl|pec|ped|pej|pel|pen|peq|pér|pes|pet|píl|pol|Pol|pot|pov|Pov|pto|pts|pub|púb|Púb|pug|qua|qui|quí|rac|rád|rec|red|ref|reg|rel|rem|rep|res|ret|rev|rib|rit|rod|rot|rub|rur|sab|Sáb|sac|san|sân|sap|sát|sch|scr|sec|Sec|seg|sel|sen|Sen|sep|seq|sér|set|sex|sib|sid|síl|sin|sir|sír|sit|Snr|soc|Soc|sól|son|Sór|sov|spp|sra|sto|suc|Suc|suf|suj|Súm|sup|Súv|tab|táb|tam|tâm|tan|tát|tec|téc|tel|Tel|ten|Ten|ter|tes|tib|tip|tir|tít|ton|tôn|tóp|tox|tun|tup|tur|ucr|ult|umb|urb|vad|veg|vel|ven|vet|vid|Vid|vig|Vig|vin|vit|Vit|viz|voc|vog|vol|vox|xin|adit|antr|arom|arts|atôm|bact|bált|bárb|basq|beir|beng|berb|bibl|bíbl|Bíbl|bien|biol|biom|bioq|biot|birm|bisp|bitr|boêm|bomb|bord|borg|borr|bras|Bras|bret|brig|Brig|brit|brom|Bros|búlg|burl|calç|cálc|cald|calv|cant|caps|capt|card|carn|carp|carr|cart|cast|caus|celt|célt|cent|cênt|cerv|cest|chap|chil|chin|chul|cicl|cien|cing|cinz|circ|círc|citt|clas|clás|clim|clín|códs|cogn|cols|comb|comp|conc|cond|conf|cong|conj|Conj|conq|cons|cont|corp|corr|corv|Corv|cosm|cost|créd|crim|crôn|cult|curt|cyat|decl|decr|Decr|dens|depr|deps|dept|derm|desc|desp|diác|dial|dioc|dipl|disc|diss|docs|dogm|dpto|Dpto|dram|drav|ecol|econ|Econ|edif|Edif|educ|efem|egíp|embr|emol|empr|epig|erud|escr|esgr|esot|estr|etim|etol|exag|excl|expl|expr|falc|farm|fasc|fauv|feit|feud|figd|finl|fisc|fisl|flam|flum|fluv|folc|folh|fols|form|fórm|fort|frac|frag|freg|freq|frig|frut|fulv|fund|futb|gaél|galv|gasc|gaul|geod|geof|geog|geol|geom|geon|geôn|germ|glót|gnom|graf|gráf|grav|guat|guin|gutt|hast|hebd|hebr|helm|herb|herp|hibr|híbr|hidr|hier|higr|hind|hisp|hist|hond|hort|húng|ibér|ibid|idol|Ilmo|ilum|imit|impr|incs|infl|ingl|Insc|inst|intj|intr|inus|iran|iraq|iron|irôn|ital|itál|jard|jorn|just|lact|lanc|lanç|larg|laud|légs|leit|ling|líng|livr|Livr|lomb|long|ltda|Ltda|lund|lunf|magn|malg|malh|marc|marg|marn|marr|marx|masc|medv|memo|mens|merc|metr|métr|ming|minh|mist|míst|mMin|moag|moed|mont|morf|mult|náua|náut|neur|nórd|norm|nucl|nutr|obed|obst|ocid|ocul|ofid|olig|onom|opos|orat|orig|orog|orôn|ostr|pagg|págg|págs|pals|parl|parn|part|pass|past|patr|pátr|pedr|pent|perf|pern|pers|pérs|pesc|pesq|pess|petr|pgto|pint|pinx|pisc|pizz|plat|pleb|poét|port|poss|prác|prát|prec|pred|pref|Pref|prep|pres|Pres|pret|prev|prim|priv|proc|prod|prof|Prof|prom|pron|prop|prot|prót|prov|prox|psic|psiq|públ|pulv|quím|rall|refl|Resp|RESP|rest|romn|russ|rúst|sagr|sals|sâns|Sarg|scul|secr|sect|segg|segs|sent|seqq|serg|serr|serv|sérv|sign|silv|simb|símb|sing|Sing|sint|sínt|sist|sost|Sras|Sres|srta|stac|subj|suec|suff|supl|surr|tail|tang|tard|taur|teat|tecn|técn|tect|temp|teol|teôn|teos|term|térm|terr|test|têxt|tint|torp|tosc|trab|trad|tráf|trat|Trav|trib|trig|trim|trit|triv|trop|turc|unid|unif|univ|urol|urug|utop|vadm|vasc|vect|vend|verb|vern|vers|vidr|viol|vitr|voll|vols|vulc|vulg|zend|zool|zoot|abrev|Abrev|Admin|anton|apass|aprox|Aprox|autom|Autom|bimen|biogr|bispd|boliv|buroc|calig|canad|capac|capit|capix|catal|categ|Catol|catól|catóp|cenog|centr|cerâm|chanc|cienc|ciênc|cient|cineg|cinem|cinol|citol|cláss|clich|colet|compl|compr|compt|comte|concl|concr|confl|const|Const|contr|coord|Coord|copul|corog|cosec|cresc|cript|crist|crust|culin|cutel|dasim|decor|defin|democ|demog|depto|deriv|descr|desen|desin|desus|didát|dimin|dinam|dinâm|distr|docum|domin|domín|ecles|eletr|elétr|elipt|embal|emigr|empír|endoc|energ|entom|epigr|epíst|equat|equit|equiv|escoc|escul|eslav|eslov|espec|espel|espet|espir|estad|estat|estét|estil|estim|eston|estud|etióp|etnog|etnol|eufem|eufêm|eufon|eufôn|eufor|exérc|exper|fascs|fenom|fenôm|figur|filat|filol|fitog|folcl|fonét|fonol|fotom|fracc|franc|frânc|futeb|futur|galic|genét|genov|geogn|geogr|ginec|gloss|groen|Hares|heort|herál|hipál|hipoc|hipol|hipot|hipót|homof|homog|homon|homôn|idiot|Ilmos|imigr|imper|incóg|indef|indet|indiv|indon|indum|infan|infer|infin|inscr|insep|irreg|islam|joalh|lapid|latit|limit|liter|litog|litol|litur|ludol|lusit|maçon|malab|malac|manuf|march|marin|marít|matad|mater|mecan|mecân|memor|merid|metaf|metáf|metát|meton|mexic|micol|miner|mitol|mitôn|monog|natur|neerl|neutr|nobil|notic|numis|obsol|ocean|ocult|odont|oneol|optat|ordin|orign|ornit|orogr|ortop|otorr|ouriv|parag|parág|Parág|paran|parôn|passm|passr|passt|patol|pedag|pedol|períf|peruv|pirot|políc|polin|polít|popul|posit|potam|presc|prest|princ|probl|propr|prosc|prost|provb|provc|provç|pseud|quant|Quant|quart|quest|quích|quimb|quinz|recip|recíp|regim|relaç|relat|relig|reloj|repúb|restr|retór|Rio-S|sapat|semic|semin|semit|semít|seric|sider|sigil|silog|simpl|sinéd|sinon|sinôn|Srtas|subst|suprf|suprl|suprs|taxid|teatr|tecel|tecgo|tecna|telec|telef|teleg|telev|terap|terat|tibet|tipol|topog|topol|topon|topôn|tunis|ucran|urban|utópi|veloc|venat|venez|veros|veter|vinic|vitic|volat|xenof|xerog|xilog|advers|bacter|balíst|biblog|biblol|biblon|biodim|biofís|biogen|biogên|biogeo|biotip|birrel|bromat|calcog|caligr|calvin|casuís|catacr|catópt|cibern|cirurg|colomb|combin|comdor|comend|comerc|compar|condic|confed|Confed|confls|conhec|conjug|consec|consel|conseq|constr|contab|coreog|corogr|corrup|corrut|cosmog|cosmol|cronol|cronom|cronôn|declin|defect|democr|demogr|deprec|desemb|Desemb|determ|dialet|dialét|diplom|disfem|dogmát|dualid|emprés|encicl|encícl|escand|escult|espect|esport|estrat|estrut|etimol|etnogr|exclam|excurs|explet|explor|explos|export|extens|extrat|farmac|ferrad|ferrov|fisioc|fisiol|fispat|fitogr|floric|flumin|formul|forrag|fotoan|fotogr|frenol|frutíf|geneal|geofís|geomor|ginást|glotol|grafol|groenl|guatem|hagiog|hagiol|haplol|helmin|heráld|herpet|herpét|hibrid|hidrod|hidrog|hidrom|hidrot|hierôn|hipérb|hipiat|hipnot|histol|homeop|homogr|hortic|hotent|iconog|iconol|idolol|ilustr|imperf|impess|import|impres|improp|impror|incoat|indian|indoch|indost|indust|indúst|infant|inform|instit|integr|intens|interj|intern|interr|invenç|invest|island|jardin|jurisp|labort|latoar|legisl|litogr|lituan|liturg|logíst|madeir|magnet|magnit|maiúsc|maomet|maranh|marinh|marroq|martin|mediev|mercad|meteor|metrif|metrol|microg|microl|microm|minúsc|moçamb|monogr|museol|neolog|neozel|normat|norueg|numism|obstet|oftalm|onomat|origin|ortogr|paleob|paleog|paleoz|paraib|parnas|partic|patron|patrôn|pediat)\\.")

# pattern for paragraph checking
paralcap = re.compile(u"(?u)^[a-z].*[.?!] [A-Z].*[.?!][)\u201d]?$")

# Minhas Implementacoes

# Tenta corrigir erros de identificacao da regra de capitalizacao
# erroscap = re.compile("(?i)\\.([A-Z])")
erroscap = re.compile("(?i)\\.([A-ZÀÁÂÃÇÉÊÍÓÔÕÚ])|\\.\\[|\\.\\(|\\.\\)|\\?\\)|\\!\\)|[a-z]\\)")
erroscapa = re.compile("(?i)[.?!]\\, ([a-záâãçéêóôõú]+)")
erroscapb = re.compile("(?i)[a-z]\\)")
erroscapc = re.compile("(?i)[.?!][\"\u2019\u201d]")

# Tenta corrigir erros de identificacao parentese de fechamento
errosprnt = re.compile(u"(?u)^([a-z]|[ivxl]+)\\)")

# Excecao da identificacao da regra Singular-plural - determinado periodo presos
excsingpl = re.compile("(?i)\\b(determinado período [a-záâçéêíóôúã]+s|sua [a-záâçéêíóôúã]+[aeiolmrz] [a-záâçéêíóôúã]+s|[a-záâçéêíóôúã]+[aeiolmrz] outros)\\b")


# Excecao da identificacao da regra dah - dar
excparadah = re.compile("(?i)\\b([Pp]ara da (mesma|minha|sua|tua|nossa))")

# Excecao da identificacao da regra ha
excha = re.compile("(?i)(0 |1 |2 |3 |4 |5 |6 |7 |8 |9 |\\/)ha")

# Excecao da identificacao do para para - repeticao
excpPara = re.compile("(?i)\\b([Pp]ara para (lhe|lhes|lho|lha|me|nos|se|si|te|vos)) ")

# Identificacao do ou seja - entre virgulas
vOusejac = re.compile("(?i)\\b(([a-zA-Z0-9áâçéêíóôúã]+) ou seja ([a-zA-Z0-9áâçéêíóôúã]+)|([a-z0-9A-Záâçéêíóôúã]+) ou seja ([a-zA-Záâçéêíóôúã]+)-([mst]e|lhe|lhes|nos|vos|la|las|lo|los))\\b")
vOusejae = re.compile("(?i)\\b(([a-zA-Z0-9áâçéêíóôúã]+), ou seja ([a-zA-Z0-9áâçéêíóôúã]+)|([a-z0-9A-Záâçéêíóôúã]+), ou seja ([a-zA-Záâçéêíóôúã]+)-([mst]e|lhe|lhes|nos|vos|la|las|lo|los))\\b")
vOusejad = re.compile("(?i)\\b(([a-zA-Z0-9áâçéêíóôúã]+) ou seja, ([a-zA-Z0-9áâçéêíóôúã]+)|([a-z0-9A-Záâçéêíóôúã]+) ou seja, ([a-zA-Záâçéêíóôúã]+)-([mst]e|lhe|lhes|nos|vos|la|las|lo|los))\\b")
excOuseja = re.compile("(?i)\\b([Ss]eja ([a-zA-Z0-9áâçéêíóôúã]+) ou seja|[Ss]eja ([a-z0-9A-Záâçéêíóôúã]+) ([a-z0-9A-Záâçéêíóôúã]+) ou seja|[Qq]uer seja|[Qq]ue seja|[Qq]ue ([a-zA-Záâçéêíóôúã]+) seja|[Cc]aso seja)\\b")

# Excecao da identificacao de todas a -
excTodas = re.compile("(?i)\\b( a todas a | para todas a |todas a [a-zç]+[ae]m | de todas a que | todas a mesma | todas a um| de todas a mais |entre todas a mais|[Tt]odas a dançar|[Tt]odas a pé|[Tt]odas a favor)")

# Excecao da identificacao de ambas a -
excAmbas = re.compile("(?i)\\b( a ambas a | para ambas a |ambas a [a-zç]+[ae]m | ambas a mesma | ambas a um|ambas a [a-zç]+[aeio]r |[Aa]mbas a pé|[Aa]mbas a favor)")

# Excecao da identificacao frases com concordancia verbal
excfrvb = re.compile("(?i)\\b([Ss]e algum dia|[Tt]alvez algum dia)\\b")

# Excecao da identificacao do para para - repeticao
excMelhorq = re.compile("(?i)\\b([a-zA-Zçáéíóúâêô]+)os melhor que (ninguém|todo|os)\\b")

# Excecao da identificacao a maioria - maneira
excMaioria = re.compile("(?i)\\b((([Ss]ua [a-zA-Zçáéíóúâêô]+|grande|absoluta|[Nn]a|[Nn]a [a-zA-Zçáéíóúâêô]+|[Ss]ua) maioria [a-zA-Zçáéíóúâêô]+s)|((alguma|certa|dessa|desta|mesma|outra|qual|qualquer|que|tal) maneira [a-zçáéíóúâêô]+s))\\b")

# Excecao da identificacao a maior parte
excMparte = re.compile("(?i)\\b([Bb]oa|[Dd]essa|[Dd]esta|[Ee]m|[Gg]rande|[Mm]aior|[Nn]essa|[Nn]esta|[Nn]uma|[Pp]equena|[Qq]ualquer|[Tt]oda|[Uu]ma) parte [a-zA-Zçáéíóúâêô]+s\\b")
excFparte = re.compile("(?i)\\b([a-zA-Zçáéíóúâêô]+es) (boa|dessa|desta|grande|maior|nessa|nesta|numa|pequena|qualquer|toda|uma) parte\\b")

# Excecao da identificacao de tal modo
excMmodo = re.compile("(?i)\\b([Aa]lgum|[Cc]erto|[Dd]esse|[Dd]este|[Mm]esmo|[Nn]enhum|[Oo]utro|[Pp]elo|[Tt]al) modo [a-zA-Zçáéíóúâêô]+s\\b")

# Excecao da identificacao da regra divida - divda
excdivida = re.compile("(?i)\\b([Dd]ivida (a|as|o|os|em|por|com|pela|pelas|pelo|pelos|tudo|isto|isso))")

# Excecao da identificacao da regra noticia - notIcia
excnoticia = re.compile("(?i)\\b(noticia (a|as|o|os|tudo|isto|isso))")

# Excecao da identificacao da regra camisa 10
excamisa = re.compile("(?i)\\b(camisa ([0-9]+|Verde|Alvo|Branco))")

# Excecao da identificacao da regra camisa 10
exceapouco = re.compile("(?i)\\b(A pouco mais de ([0-9]+|um|uma|dois|duas|três|quatro|cinco|seis|sete|oito|nove|dez|onze|doze|treze|catorze|quinze|dezesseis|dezessete|dezoito|dezenove|vinte) (ano|anos|mês|meses|semana|semanas|dia|dias|hora|horas|minuto|minutos|segundo|segundos|volta|voltas) (para|de|da|das|do|dos))")

# Excecao da identificacao da regra palavras terminadas em ido - Ex. uma foto sua vestido de...
excfido = re.compile("(?i)\\b(foto|fotos|imagem|imangens|caricatura|caricaturas|fotografia|fotografias) (minha|minhas|sua|suas) (vestido|vestidos)")

# Excecao de plural com anos Ex. em 1984 meu pai - 12/2014 processo
exckano = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) [a-zA-Záâãçéêíóôú]+[aeioulmnrz]")

# Excecao de plural com horas Ex. as 00:00 horas
exchora = re.compile("(?i)\\b(0[2-9]|[2-9]|[1-9][0-9]):[0-5][0-9] \\b")

# Excecao de plural com anos Ex. em 1984 meu pai - 12/2014 processo
exckano = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) [a-zA-Záâãçéêíóôú]+[aeioulmnrz]")
excYmfoa = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)[oa]\\b")
excYmasco = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+o)\\b")
excYmasca = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+a)\\b")
excYnca = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)([aeiou])nça\\b")
excYncia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)([âêíôú])ncia\\b")
excYgrafia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)([âêíôú])grafia\\b")
excYogia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+[gl])ogia\\b")
excYrgia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)rgia\\b")
excYlia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)([áéíóú])lia\\b")
excYapia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)erapia\\b")
excYria = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)([áéíóú])ria\\b")
excYsia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)([áéíú])sia\\b")
excYrsia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)([áéíóú])rsia\\b")
excYssia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)([áéíú])ssia\\b")
excYrnia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)([áéíú])rnia\\b")
excYnia = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)([âêíôú])nia\\b")
excYmfeu = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Zçáéíóúâêô]+)u\\b")
excYmascu = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)u\\b")
excYfemu = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)u\\b")
excYmfei = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Zçáéíóúâêô]+)i\\b")
excYmasci = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)i\\b")
excYfemi = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)i\\b")
excYmfal = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Zçáéíóúâêô]+[a])l\\b")
excYmascal = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+[a])l\\b")
excYfemal = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+[a])l\\b")
excYmfel = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Z]+)el\\b")
excYmascel = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)el\\b")
excYfemel = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)el\\b")
excYamscel = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+[e])l\\b")
excYmasces = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)ês\\b")
excYvel = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Zç]+[áéíóú])vel\\b")
excYmfe = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Zçáéíóúâêô]+)e\\b")
excYmasce = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)e\\b")
excYfeme = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)e\\b")
excYinvaz = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Z]+)z\\b")
excYmascz = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Z]+)z\\b")
excYfemz = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Z]+)z\\b")
excYmfm = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Zçáéíóúâêô]+)m\\b")
excYmascm = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)m\\b")
excYfemm = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-zA-Záâãçéêíóôú]+)m\\b")
excYfema = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+[a])\\b")
excYfdade = re.compile("(?i)\\b([Ee]m 19[0-9][0-9]|[Ee]m 20[0-9][0-9]|[Dd]e 19[0-9][0-9]|[Dd]e 20[0-9][0-9]|[Dd]esde 19[0-9][0-9]|[Dd]esde 20[0-9][0-9]|([0-9]+)[/:;]([0-9]+)) ([a-záâãçéêíóôú]+)dade\\b")

# Excecao de plural - singular
excplSin = re.compile("(?i)\\b([a-zA-Záâãçéêíóôõú]+[s] [a-zA-Záâãçéêíóôõú]+[aeioulmnrxz], [a-zA-Záâãçéêíóôõú]+[aeioulmnrxz] e |[a-zA-Záâãçéêíóôõú]+[s] [a-zA-Záâãçéêíóôõú]+[aeioulmnrxz], [a-zA-Záâãçéêíóôõú]+[aeioulmnrxz] e |[a-zA-Záâãçéêíóôõú]+[s] [a-zA-Záâãçéêíóôõú]+[aeioulmnrxz], [a-zA-Záâãçéêíóôõú]+[aeioulmnrxz], [a-zA-Záâãçéêíóôõú]+[aeioulmnrxz])\\b")

# Excecao de viu ele e viram eles
excviuEle = re.compile("(?i)\\b(viu el(a|e) que|viu el(a|e) (a|as|o|os)|viu el(a|e)[.;,]|o que viu el(a|e)|, viu el(a|e) [a-zA-Záâãçéêíóôõú]+|viram el(a|e)s que|viram el(a|e)s (a|as|o|os)|viram el(a|e)s[.;,]|o que viram el(a|e)s|, viram el(a|e)s [a-zA-Záâãçéêíóôõú]+)\\b")

# OUTROS  *-*-*-*-*-*-*-

# Identificacao de Nao a - Nao ha
vnaoHa = re.compile("(?i)\\b([Nn]ão a ([a-z]+ade|[a-zç]+ão|[a-z]+eir[ao]|[a-zçáéíóú]+ito|[a-z]+iv[ao]|[a-zç]+ões|[a-z]+nte|[a-z]+nto|[a-zêâáíó]+ci[ao]|[a-zç]+ras|[a-zç]+res|[a-zçáéíóú]+ri[ao][a-zçáéíóú]+ri[ao]s|[a-zçáéíóú]+tos|abertura|abrigo|academias|acordo|acúmulo|admirar|aforismos|agenda|agricultura|água|aí|ainda|ajuda|ajuste|alarme|álbum|alcaide|aldeias|alface|algas|algum|alguma|alguns|ali|alma|almoço|alta|alternativas|alvará|amanuense|amargura|ambulantes|amigos|amor|amplas|animais|ânimo|anos|ao|apenas|apetite|aplausos|apoio|apoios|aposta|aquela|aquelas|aquele|aqueles|aqui|área|aristocracias|arma|arte|artista|árvore|arvoredo|as|aspas|assadura|assim|assistentes|atalhos|ataque|ataques|até|ateus|atmosfera|atraso|atrasos|aulas|automóvel|autor|avanço|azar|azuis|bagatelas|baia|baile|balança|balas|baliza|bálsamo|bananas|banco|bando|banheiros|banhos|banquetes|barco|barracas|base|beijo|bela|beleza|belos|bem|bernarda|bicho|birra|boa|bogas|bom|bombas|bonança|bonde|bons|bonzo|borda|bota|botequim|braço|braços|bruxas|buracos|cá|cabeça|cabotino|cabreiros|caça|cachaça|cachorrinho|cagaço|calçado|cálculo|caldeiradas|calor|cama|camarada|camas|camelos|caminho|caminhos|camisas|campo|cantinho|caos|capas|capitães|capitais|capital|capricho|caprichos|carácter|caramujo|carne|carro|carros|carta|cartas|cartilha|casa|casas|casca|caso|casos|castelhanos|castelo|castigo|catarse|caudal|causas|cavalo|cavalos|cego|cegonha|centro|cerimônia|certas|certeza|certezas|céu|chá|chama|chamariz|chance|chaves|chefe|chefes|cheiros|cheques|choque|chuva|cinco|cineasta|cinema|cinemas|cipó|ciúmes|claro-escuro|classe|clientela|clima|clube|coadjutor|cobertura|cobra|cobrança|coiros|coisa|coisa-ruim|com|combate|comida|como|companhia|compromisso|comum|confiança|conforto|conselhos|consenso|contestar|contrabandistas|controle|cousa|cova|crase|criados|crianças|criatura|crime|crimes|criminoso|crise|culpado|cultura|cura|dados|data|de|decerto|dentro|denúncias|desar|descanso|desculpa|desculpas|desejar|desemprego|deserto|desgraçado|desleixo|desordem|despedida|dessas|destino|desvios|detalhes|deus|deuses|dever|dez|dia|diabos|diálogo|dias|dieta|diferença|diferenças|dificuldades|diploma|diretor|disfarce|disso|disto|divergências|dívida|dividendos|divina|divinas|do|doce|doçura|doença|doenças|dois|dolo|domingo|dor|dormimos|dormir|dous|doutrina|doutros|drama|droga|duas|duques|dúvida|duvidar|dúvidas|e|é|economia|egoísmo|elefantes|em|embargo|empadas|empates|empatia|empenho|empenhos|emplastro|emprego|encarregados|encontro|energia|enfermidades|engano|enganos|entre|epidemia|epidemias|episódio|época|épocas|equipe|equipes|ermos|erro|erros|erva|escândalo|escapar|escolas|escolha|escravos|esforços|espaço|espaços|espécies|esperança|essa|esse|esses|esta|estatísticas|este|estes|estômago|estratégias|estúdios|estudos|evidências|exagero|exemplo|falar|falta|família|fato|fé|febre|férias|festa|festas|fidalgos|filósofo|fim|fome|força|forças|forma|formas|fórmula|fórmulas|fortuna|fugir|fumo|futuro|garantia|garantias|gonçalinho|gosto|governo|graça|grades|grande|grandes|grupos|hipótese|hoje|homem|homens|honra|ideologias|igual|imagens|impossíveis|indícios|inimigo|interesse|já|jogos|jornais|justiça|lá|lei|leis|liames|limite|limites|linguagem|livro|livros|lixo|lua|lugar|luta|lutas|luz|mágoa|maior|mais|mal|malandro|males|manhãs|mãos|margem|mas|máscara|médico|médicos|medida|medo|meio|meios|melhor|mendigo|menina|mesas|mesmo|método|métodos|milagre|mister|moça|modelo|modelos|moléstia|molhos|morte|motivos|motorista|mudanças|muita|muitas|mulher|música|na|nada|nas|nascentes|negar|negros|nela|nele|neles|nem|nenhum|nenhuma|nenhuns|nessa|nesta|neste|ninguém|nisso|no|noite|nome|nomes|nos|notas|notícias|novidades|novos|números|o|obstáculos|ofensa|olhos|onde|ordem|ou|ouro|outra|outro|outros|pai|país|palavra|papel|para|partido|partidos|passagem|peixe|penedos|perda|perigo|períodos|perplexidades|perspectivas|pesar|pessoa|pessoal|pessoas|piolhos|pior|pistas|planos|poder|poesia|pois|polícias|política|políticas|por|porém|porque|portugueses|posse|pota|povo|praxe|prazer|prazo|pressa|pressas|princípio|princípios|problema|problemas|prodígio|programa|progresso|protetor|prova|provas|público|quaisquer|qualquer|quase|que|quem|raças|raciocínios|racismo|rapaz|receio|receitas|recuos|recurso|recursos|referências|reforma|refugiados|registo|registro|registros|regra|rei|remédio|remorso|reservas|resposta|respostas|riqueza|risco|riscos|rosas|rua|saída|salas|sangue|segredo|segredos|segunda|seis|sempre|senador|sentido|sentir|sequer|ser|sereia|serviço|simpatia|sinais|sinal|só|sobre|social|sol|som|sono|subsídios|subtilezas|surf|tal|talvez|também|tanta|tarefa|tecnologia|temor|tempo|terapêutica|terra|testemunhas|tipo|todo|tolo|trabalho|tráfico|tragédia|três|trevas|tristeza|tristezas|tropa|um|uma|umas|utopias|vacilar|vacina|vaga|vagas|vaidades|valor|vedetas|ver|verdadeiros|versos|vestígio|véu|vida|vingança|vinho|vislumbrar|vizinhos|volta|voltas|voz))\\b")
vdisNaoa = re.compile("(?i)\\b([Dd]iga|[Dd]igam|[Dd]igo|[Dd]irá|[Dd]irão|[Dd]iremos|[Dd]iria|[Dd]iriam|[Dd]isse|[Dd]issemos|[Dd]isseram|[Dd]izem|[Dd]iz|[Dd]izer|[Dd]izia|[Dd]iziam|[Dd]o|[Ee]|[Nn]o|[Oo]|ou|[Pp]ara|[Pp]ro|[Ss]onoro|[Uu]m) não a\\b")
vMaisNaoa = re.compile("(?i)\\b(é que|é o que|é a que|são os que|são as que|que não a que)\\b")

# Excecao de tem sem acento
exctema = re.compile("(?i)\\b((das|de|dos|dessas|destas|nessas|nestas|desses|destes|nesses|nestes) ([a-zA-Záâãçéêíóôõú]+[s]) ([a-zA-Záâãçéêíóôõú]+[s])|[AaOo] [a-zA-Záâãçéêíóôõú]+[aeioulmrz] entre [a-zA-Záâãçéêíóôõú]+[aeioulmrz] [a-zA-Záâãçéêíóôõú]+[aeioulmrz] e [a-zA-Záâãçéêíóôõú]+[s] [a-zA-Záâãçéêíóôõú]+[s]) tem\\b")
exctemb = re.compile("(?i)\\b(das|de|dos|dessas|destas|nessas|nestas|desses|destes|nesses|nestes) ([a-zA-Záâãçéêíóôõú]+[s]) tem\\b")

# Excecao de obedecer
excobdc = re.compile("(?i)\\b([Dd]eso|[Oo])bede([cç])(\\w+) a ([0-9]+|[a-zA-Záâãçéêíóôõú]+[o]|algum|alguma|algumas|dez|dois|duas|ela|elas|ele|eles|essa|essas|esse|esses|esta|estas|este|estes|minha|minhas|ninguém|nossa|nossas|nove|seis|sete|sua|suas|toda|todas|todos|três|um|uma|umas|uns|você|vossa)\\b")

# Excecao de compostos contra
exccontra = re.compile("(?i)\\b([Aa]berta|[Aa]bertamente|[Aa]bertos|[Aa]brigada|[Aa]brigo|[Aa]buso|[Aa]ção|[Aa]cautelar-se|[Aa]ções|[Aa]cordo|[Aa]cusação|[Aa]cusações|[Aa]dvertiu|[Aa]dvogado|[Aa]éreos|[Aa]final|[Aa]gir|[Aa]gora|[Aa]gressão|[Aa]gressões|[Aa]inda|[Aa]juda|[Aa]lemães|[Aa]lerta|[Aa]lgo|[Aa]lguma|[Aa]liança|[Aa]lma|[Aa]manha|[Aa]manhã|[Aa]meaça|[Aa]meaças|[Aa]mistoso|[Aa]mistosos|[Aa]muleto|[Aa]nátema|[Aa]no|[Aa]nos|[Aa]ntes|[Aa]nticorpos|[Aa]ntídoto|[Aa]penas|[Aa]pertadas|[Aa]pertado|[Aa]pertando|[Aa]pertando-a|[Aa]pertava|[Aa]pertava-a|[Aa]pertou|[Aa]pertou-a|[Aa]pertou-o|[Aa]poiado|[Aa]rgumentar|[Aa]rgumento|[Aa]rgumentos|[Aa]rma|[Aa]rmada|[Aa]rmado|[Aa]rmados|[Aa]rmas|[Aa]rremessavam|[Aa]rremesso|[Aa]rremessou|[Aa]rremete|[Aa]rremetendo|[Aa]rremeter|[Aa]rremeteu|[Aa]rremetia|[Aa]rremetidas|[Aa]rrojou-se|[Aa]rtigo|[Aa]rtigos|[Aa]ssalto|[Aa]ssim|[Aa]taque|[Aa]taques|[Aa]té|[Aa]tenta|[Aa]tentado|[Aa]tentados|[Aa]tentando|[Aa]tentar|[Aa]tentou|[Aa]tirada|[Aa]tiradas|[Aa]tirado|[Aa]tiram|[Aa]tirando|[Aa]tirar|[Aa]tirar-se|[Aa]tiraram|[Aa]tirou|[Aa]tirou-se|[Aa]to|[Aa]utoridade|[Aa]uxílio|[Aa]vançando|[Aa]vançar|[Aa]vançaram|[Aa]vançou|[Bb]aixa|[Bb]aixas|[Bb]andeira|[Bb]arreira|[Bb]arreiras|[Bb]atalha|[Bb]atalhas|[Bb]atendo|[Bb]ater|[Bb]ateram|[Bb]ateu|[Bb]em|[Bb]ilhões|[Bb]lasfemar|[Bb]loco|[Bb]oca|[Bb]oicote|[Bb]ola|[Bb]omba|[Bb]orracha|[Bb]raço|[Bb]radar|[Bb]radava|[Bb]rancos|[Bb]rasil|[Bb]rasileira|[Bb]rigar|[Bb]ritânica|[Cc]abeça|[Cc]abo|[Cc]aiu|[Cc]alúnia|[Cc]ampanha|[Cc]ampanhas|[Cc]ampo|[Cc]andidatura|[Cc]aptura|[Cc]ara|[Cc]asa|[Cc]asar|[Cc]ausa|[Cc]avalaria|[Cc]ensura|[Cc]ensuras|[Cc]ento|[Cc]errada|[Cc]errados|[Cc]hoque|[Cc]idadania|[Cc]idadão|[Cc]idadãos|[Cc]idade|[Cc]igarro|[Cc]ilada|[Cc]ivil|[Cc]lama|[Cc]lamam|[Cc]lamando|[Cc]lamar|[Cc]lássico|[Cc]oisa|[Cc]oisas|[Cc]ólera|[Cc]oligação|[Cc]olocar|[Cc]ombate|[Cc]ombatendo|[Cc]ombater|[Cc]ombateram|[Cc]ombates|[Cc]ombatiam|[Cc]ometida|[Cc]ometidas|[Cc]ometido|[Cc]ometidos|[Cc]omissão|[Cc]omo|[Cc]omplô|[Cc]omprimidos|[Cc]omum|[Cc]oncedidas|[Cc]oncreto|[Cc]onforto|[Cc]onjura|[Cc]onsciência|[Cc]onspira|[Cc]onspiração|[Cc]onspirador|[Cc]onspiram|[Cc]onspirar|[Cc]onspirava|[Cc]onspiravam|[Cc]onstante|[Cc]ontos|[Cc]onvenção|[Cc]oração|[Cc]oroa|[Cc]orpo|[Cc]orre|[Cc]orrer|[Cc]orrida|[Cc]orrupção|[Cc]ostas|CPI|[Cc]rânio|[Cc]riança|[Cc]riatura|[Cc]rime|[Cc]rimes|[Cc]riminal|[Cc]riminosa|[Cc]rítica|[Cc]ríticas|[Cc]ruz|[Cc]ruzada|[Cc]ruzadas|[Dd]anos|[Dd]e|[Dd]ecisão|[Dd]ecisiva|[Dd]eclamam|[Dd]eclamando|[Dd]eclamar|[Dd]eclarações|[Dd]efendê-las|[Dd]efender|[Dd]efender-se|[Dd]efendido|[Dd]efesa|[Dd]efesas|[Dd]ele|[Dd]elito|[Dd]entes|[Dd]entro|[Dd]enúncia|[Dd]enúncias|[Dd]epõe|[Dd]epois|[Dd]epor|[Dd]eputados|[Dd]esconfiança|[Dd]esesperadamente|[Dd]esferir|[Dd]espeito|[Dd]espejar|[Dd]eus|[Dd]ifamação|[Dd]ignidade|[Dd]ireitos|[Dd]iretamente|[Dd]irigia-se|[Dd]irigida|[Dd]irigidas|[Dd]irigido|[Dd]irigidos|[Dd]iscriminação|[Dd]iscurso|[Dd]isparado|[Dd]isparar|[Dd]ispararam|[Dd]isparou|[Dd]isputa|[Dd]ito|[Dd]izer|[Dd]o|[Dd]obrado|[Dd]ocumento|[Dd]ois|[Dd]omingo|[Dd]roga|[Dd]rogas|[Dd]ura|[Dd]uras|[Ee]|[Éé]|[Ee]conômico|[Ee]feito|[Ee]ficaz|[Ee]ficazes|[Ee]l-rei|[Ee]la|[Ee]le|[Ee]leições|[Ee]les|[Ee]m|[Ee]mbargo|[Ee]mbora|[Ee]mpate|[Ee]nérgica|[Ee]nergicamente|[Ee]ntão|[Ee]quipe|[Ee]ra|[Ee]ram|[Ee]rguer|[Ee]rguesse|[Ee]sbarrar|[Ee]sbarrou|[Ee]sbracejar|[Ee]sbravejar|[Ee]scala|[Ee]screveu|[Ee]scudo|[Ee]sforços|[Ee]spada|Espanha|[Ee]special|[Ee]specialmente|[Ee]speculativo|[Ee]spingarda|[Ee]spírito|[Ee]stá|[Ee]stado|[Ee]stamos|[Ee]stão|[Ee]star|[Ee]stava|[Ee]stavam|[Ee]steve|[Ee]stou|[Ee]strangeiro|[Ee]streia|[Ee]stréia|[Ee]streitou|[Ee]terno|[Ee]u|EUA|Europa|[Ee]uropeia|[Ee]uropeu|[Ee]xatamente|[Ee]xercida|[Ee]xército|[Ee]xpedição|[Ee]xpedições|[Ee]xpulso|[Ff]ace|[Ff]ala|[Ff]alam|[Ff]alando|[Ff]alar|[Ff]az|[Ff]azer|[Ff]ederal|[Ff]eita|[Ff]eitas|[Ff]eitiço|[Ff]eito|[Ff]erro|[Ff]ez|[Ff]ilha|[Ff]ilho|[Ff]inal|[Ff]inalmente|[Ff]ogo|[Ff]oi|[Ff]or|[Ff]ora|[Ff]oram|[Ff]orça|[Ff]orças|[Ff]ormais|[Ff]ormal|[Ff]ortaleza|[Ff]orte|[Ff]ortemente|[Ff]ortes|[Ff]osse|[Ff]raco|[Ff]rança|[Ff]rancamente|[Ff]ranceses|[Ff]raude|[Ff]raudes|[Ff]rente|[Ff]rontalmente|[Ff]ui|[Ff]úria|[Ff]uriosa|[Ff]urioso|[Ff]uror|[Gg]arantia|[Gg]arantidos|[Gg]ente|[Gg]eral|[Gg]ol|[Gg]olpe|[Gg]ols|[Gg]overno|[Gg]rêmio|[Gg]reve|[Gg]rito|[Gg]ritos|[Gg]ritou|[Gg]uarda|[Gg]uerra|[Gg]uerras|[Gg]uerrilha|[Gg]uerrilheiros|[Hh]á|[Hh]avia|[Hh]oje|[Hh]omem|[Hh]omens|[Hh]omicídio|[Hh]oras|[Hh]ostil|[Hh]ostilidade|[Hh]ostilidades|[Hh]ostis|[Ii]a|[Ii]am|[Ii]lícito|[Ii]mpério|[Ii]mplacável|[Ii]mportante|[Ii]mprecações|[Ii]mprensa|[Ii]munidade|[Ii]munizada|[Ii]nconstitucionalidade|[Ii]ncorruptíveis|[Ii]ndícios|[Ii]ndignação|[Ii]ndignada|[Ii]ndignado|[Ii]ndignava|[Ii]ndividuais|[Ii]ndo|[Ii]nfrações|[Ii]ngleses|[Ii]niciada|[Ii]njúrias|[Ii]nquérito|[Ii]nquéritos|[Ii]nsulto|[Ii]nsurge|[Ii]nsurge-se|[Ii]nsurgia|[Ii]nsurgia-se|[Ii]nsurgiam|[Ii]nsurgido|[Ii]nsurgindo-se|[Ii]nsurgir|[Ii]nsurgir-se|[Ii]nsurgiram|[Ii]nsurgiram-se|[Ii]nsurgiu|[Ii]nsurgiu-se|[Ii]nsurjo|[Ii]nsurreição|[Ii]ntentada|[Ii]nternacionais|[Ii]nternacional|[Ii]nterpostos|[Ii]nvectivas|[Ii]nveste|[Ii]nvestem|[Ii]nvestia|[Ii]nvestiam|[Ii]nvestidas|[Ii]nvestindo|[Ii]nvestir|[Ii]nvestiram|[Ii]nvestirem|[Ii]nvestiu|[Ii]r|[Ii]ra|[II]raque|[Ii]rritação|[Ii]rritada|[Ii]rritado|[Ii]rritou-se|[Ii]sso|[II]tália|[Jj]á|[Jj]aneiro|[Jj]oga|[Jj]ogadores|[Jj]ogam|[Jj]ogamos|[Jj]ogando|[Jj]ogar|[Jj]ogará|[Jj]ogava|[Jj]ogo|[Jj]ogos|[Jj]ogou|[Jj]ornais|[Jj]udiciais|[Jj]udicial|[Jj]udicialmente|[Jj]uízo|[Jj]ulho|[Jj]ustiça|[Ll]ançada|[Ll]ançadas|[Ll]ançado|[Ll]ançar-se|[Ll]egal|[Ll]egislação|[Ll]egislativo|[Ll]ei|[Ll]eis|[Ll]evanta|[Ll]evantadas|[Ll]evantam|[Ll]evantar|[Ll]evantava|[Ll]evante|[Ll]evantou|[Ll]ibelo|[Ll]iga|[Ll]iminar|[Ll]iminares|[Ll]íngua|[Ll]isboa|[Ll]ivros|[Ll]obby|[Ll]ocal|[Ll]ogo|[Ll]uta|[Ll]utador|[Ll]utam|[Ll]utando|[Ll]utar|[Ll]utará|[Ll]utaram|[Ll]utas|[Ll]utava|[Ll]utavam|[Ll]uto|[Ll]utou|[Mm]ãe|[Mm]ais|[Mm]aldições|[Mm]anifestação|[Mm]anifestações|[Mm]anifestantes|[Mm]anifestar|[Mm]anifestar-se|[Mm]anifestaram|[Mm]anifestaram-se|[Mm]anifesto|[Mm]anifestou|[Mm]anifestou-se|[Mm]antinha|[Mm]ão|[Mm]ãos|[Mm]archar|[Mm]archava|[Mm]as|[Mm]assacres|[Mm]assas|[Mm]aus-tratos|[Mm]e|[Mm]edicamentos|[Mm]édicos|[Mm]edida|[Mm]edidas|[Mm]elhor|[Mm]enina|[Mm]enos|[Mm]esmo|[Mm]ilitar|[Mm]ilitares|[Mm]im|[Mm]inutos|[Mm]ofinas|[Mm]olotov|[Mm]omento|[Mm]onarquia|[Mm]onteiro|[Mm]orte|[Mm]ostrou|[Mm]ove|[Mm]oveu|[Mm]ovida|[Mm]ovido|[Mm]ovimento|[Mm]ovimentos|[Mm]uito|[Mm]ulher|[Mm]ulheres|[Mm]undial|[Mm]undo|[Mm]urmurando|[Mm]urmurar|[Nn]acional|[Nn]acionalista|[Nn]ada|[Nn]ão|[Nn]ariz|[Nn]aturais|[Nn]atureza|[Nn]azi|[Nn]egros|[Nn]em|[Nn]oite|[Nn]omeadamente|[Nn]orte|[Nn]ota|[Nn]ovo|[Nn]unca|[Oo]|[Oo]brar|[Oo]bras|[Óó]dio|[Óó]dios|[Oo]fensas|[Oo]fensiva|[Oo]fensivas|[Oo]ito|[Oo]lhos|[Oo]mbros|OMC|[Oo]ntem|ONU|[Oo]peração|[Oo]perações|[Oo]pinião|[Oo]posição|[Oo]posicionista|[Oo]posições|[Oo]rações|[Oo]rdem|[Oo]rganismo|[Oo]u|[Oo]utros|[Pp]á|[Pp]ai|[Pp]aís|[Pp]aíses|[Pp]alavra|[Pp]arlamentares|[Pp]articular|[Pp]artida|[Pp]artido|[Pp]artiu|[Pp]assado|[Pp]aulista|[Pp]aulo|[Pp]é|[Pp]ecado|[Pp]ecados|[Pp]edra|[Pp]eito|[Pp]ermanente|[Pp]essoal|[Pp]essoas|[Pp]ode|[Pp]odem|[Pp]oder|[Pp]odia|[Pp]olícia|[Pp]olicial|[Pp]olítica|[Pp]ontos|[Pp]opulação|[Pp]opular|[Pp]opulares|[Pp]ortuguesa|[Pp]osição|[Pp]osso|[Pp]ouco|[Pp]ovo|[Pp]raguejando|[Pp]raguejar|[Pp]raticadas|[Pp]raticado|[Pp]raticados|[Pp]recaução|[Pp]recauções|[Pp]recaver|[Pp]recipitou-se|[Pp]reconceito|[Pp]reconceitos|[Pp]regar|[Pp]reservativo|[Pp]ressão|[Pp]revenção|[Pp]revenções|[Pp]revenido|[Pp]revenir|[Pp]revenir-se|[Pp]rincipalmente|[Pp]risão|[Pp]roceder|[Pp]rocesso|[Pp]rocesso-crime|[Pp]rocessos|[Pp]roferida|[Pp]roferidas|[Pp]rograma|[Pp]ronunciado|[Pp]ropaganda|[Pp]roteção|[Pp]rotege|[Pp]rotegê-la|[Pp]rotegê-lo|[Pp]roteger|[Pp]rotegia|[Pp]rotegidas|[Pp]rotesta|[Pp]rotestado|[Pp]rotestam|[Pp]rotestando|[Pp]rotestar|[Pp]rotestaram|[Pp]rotestarem|[Pp]rotestava|[Pp]rotestavam|[Pp]roteste|[Pp]rotestei|[Pp]rotesto|[Pp]rotestos|[Pp]rotestou|[Pp]rotetor|[Pp]rova|[Pp]rovado|[Pp]rovas|[Pp]rovidências|[Pp]ública|[Pp]ublicamente|[Pp]úblico|[Qq]uarta-feira|[Qq]uatro|[Qq]ue|[Qq]ueixa|[Qq]ueixa-crime|[Qq]ueixas|[Qq]uinta-feira|[Rr]acismo|[Rr]adicalmente|[Rr]aiva|[Rr]aivosos|[Rr]azão|[Rr]eação|[Rr]eage|[Rr]eagia|[Rr]eagindo|[Rr]eagir|[Rr]eagiram|[Rr]eagiria|[Rr]eagiu|[Rr]eais|[Rr]eaja|[Rr]ebelada|[Rr]ebelado|[Rr]ebelados|[Rr]ebelando-se|[Rr]ebelar|[Rr]ebelar-se|[Rr]ebelaram|[Rr]ebelaram-se|[Rr]ebeldia|[Rr]ebelião|[Rr]ebeliões|[Rr]ebelou-se|[Rr]eclamar|[Rr]eclamavam|[Rr]eclamou|[Rr]ecorrer|[Rr]ecriminações|[Rr]ecurso|[Rr]ecursos|[Rr]efúgio|[Rr]egressiva|[Rr]egresso|[Rr]emar|[Rr]emédio|[Rr]emédios|[Rr]epresália|[Rr]epresentação|[Rr]epresentar|[Rr]epública|[Rr]eserva|[Rr]eservas|[Rr]esistência|[Rr]esmungando|[Rr]esmungar|[Rr]essentimento|[Rr]etaliação|[Rr]evolta|[Rr]evolta-se|[Rr]evoltada|[Rr]evoltado|[Rr]evoltam|[Rr]evoltam-se|[Rr]evoltando-se|[Rr]evoltar|[Rr]evoltar-se|[Rr]evoltaram|[Rr]evoltaram-se|[Rr]evoltaria|[Rr]evoltasse|[Rr]evoltava|[Rr]evoltava-me|[Rr]evoltava-se|[Rr]evoltei|[Rr]evoltei-me|[Rr]evolto-me|[Rr]evoltou|[Rr]evoltou-se|[Rr]evolução|[Rr]evolucionário|[Rr]osnando|[Rr]osto|[Rr]oupa|[Rr]ussa|[Rr]ussas|Rússia|[Rr]usso|[Rr]ussos|[Ss]abia|[Ss]anções|[Ss]anta|[Ss]ão|[Ss]e|[Ss]ecreto|[Ss]eguinte|[Ss]egurança|[Ss]eguro|[Ss]eguros|[Ss]eio|[Ss]eis|[Ss]eja|[Ss]eleção|[Ss]emana|[Ss]emifinais|[Ss]empre|[Ss]endo|[Ss]enhor|[Ss]entença|[Ss]entenças|[Ss]entia|[Ss]equer|[Ss]er|[Ss]erá|[Ss]erem|[Ss]ermão|[Ss]exual|[Ss]ilêncio|[Ss]im|[Ss]ó|[Ss]oares|[Ss]ocial|[Ss]ociedade|[Ss]ofreu|[Ss]oldados|[Ss]omente|[Ss]omos|[Ss]ou|[Ss]ucessos|[Ss]ul|[Ss]urda|[Ss]uspeitas|[Tt]alvez|[Tt]ambém|[Tt]anto|[Tt]arde|[Tt]em|[Tt]empo|[Tt]enho|[Tt]ens|[Tt]entado|[Tt]entar|[Tt]er|[Tt]erá|[Tt]erapia|[Tt]erra|[Tt]erritório|[Tt]errível|[Tt]erroristas|[Tt]igre|[Tt]inha|[Tt]inham|[Tt]inta|[Tt]iro|[Tt]iros|[Tt]itãs|[Tt]oda|[Tt]odas|[Tt]odo|[Tt]odos|[Tt]otalmente|[Tt]rabalhar|[Tt]rabalho|[Tt]rama|[Tt]ramada|[Tt]ramar|[Tt]ramavam|[Tt]ratado|[Tt]ratamentos|[Tt]ravada|[Tt]ribunal|[Tt]riunfante|[Tt]rovejou|[Tt]udo|[Uu]m|[Uu]ma|[Uu]mas|[Uu]ne|[Uu]nida|[Uu]nidas|[Uu]nidos|[Uu]nir|[Uu]ns|[Uu]sada|[Uu]sadas|[Uu]sado|[Uu]sados|[Uu]tilizados|[Vv]acina|[Vv]acinação|[Vv]acinadas|[Vv]acinado|[Vv]acinar|[Vv]acinas|[Vv]ai|[Vv]aler|[Vv]ão|[Vv]elha|[Vv]erdadeiramente|[Vv]ez|[Vv]ezes|[Vv]ícios|[Vv]ida|[Vv]ingança|[Vv]inganças|[Vv]inha|[Vv]iolência|[Vv]iolências|[Vv]iolenta|[Vv]iolentamente|[Vv]iolentos|[Vv]irar|[Vv]irou-se|[Vv]ista|[Vv]itória|[Vv]itórias|[Vv]itoriosa|[Vv]ociferando|[Vv]ociferar|[Vv]ociferava|[Vv]olta|[Vv]olta-se|[Vv]oltado|[Vv]oltam-se|[Vv]oltando|[Vv]oltar|[Vv]oltar-se|[Vv]oltaram-se|[Vv]oltasse|[Vv]oltava-se|[Vv]oltou-se|[Vv]ontade|[Vv]ota|[Vv]otado|[Vv]otam|[Vv]otando|[Vv]otar|[Vv]otará|[Vv]otaram|[Vv]otarão|[Vv]otava|[Vv]ote|[Vv]oto|[Vv]otos|[Vv]otou|[Vv]ou|[Vv]oz) contra\\b")

# Excecao de Esta em
excestaem = re.compile("(?i)\\b([a-zç]+ndo|[a-zç]+(-se|lhe|lhes|nos)|[Cc]om|[Cc]omo|[Cc]onsiste|[Cc]onsistia|[Cc]onsistindo|[Cc]onvertida|[Dd]isse|[Ee]|[Éé]|[Mm]ais|[Qq]uanto|[Ss]eja|[Ss]enão|[Ss]obre|[Tt]erra|[Tt]ransformada|[Tt]ransitada) esta (de|em|no|na|nos|nas)\\b")

# Paronimos especiais *-*-*-*-*-*-*-


# Excecao de autentico - autentica sem acento
excauttic = re.compile("(?i)\\b([Ee]u|[Mm]e|[Ss]e|[Tt]e|[Ll]he|[Ll]hes) autentico\\b")
excauttix = re.compile("(?i)\\b[Aa]utentic[ao] (o|a|as|os|um|uma|uns|umas|todo|toda|todos|todas|apenas)\\b")

# Excecao de divorcio sem acento
excdivorc = re.compile("(?i)\\b(a|[Ee]u|[Mm]e|[Ss]e|[Tt]e|[Ll]he|[Ll]hes) divorcio\\b")

# Excecao de fotografo sem acento
excfotogra = re.compile("(?i)\\b(a|[Ee]u|[Mm]e|[Ss]e|[Tt]e|[Ll]he|[Ll]hes) fotografo\\b")
excfotogrx = re.compile("(?i)\\b[Ff]otograf[ao] (o|a|as|os|um|uma|uns|umas|todo|toda|todos|todas|tudo|apenas)\\b")

# Identificacao de tipos plural
vtipos = re.compile("(?i)\\b([a-zA-Záâãçéêíóôúõ]+s tipo (de|não|já|tém)|[a-zA-Záâãçéêíóôúõ]+s tipo [a-záâãçéêíóôúõ]+s|[Tt]odos os tipo|[a-zA-Záâãçéêíóôúõ]+s tipo são|[a-zA-Záâãçéêíóôúõ]+s tipo [a-zç]+[ae]m)\\b")

# Identificacao de fabrica sem acento
vfabrica = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]lguma|[Aa]ntiga|[Aa]quela|[Àà]quela|[Aa]utêntica|[Cc]ada|[Cc]hamavam|[Cc]om|[Dd]a|[Dd]aquela|[Dd]e|[Dd]écima|[Dd]essa|[Dd]esta|[Dd]ita|[Dd]uma|[Ee]|[Éé]|[Ee]m|[Ee]ssa|[Ee]sta|[Ee]studa|[Ee]xtensa|[Ff]echar|[Ff]oi|[Ff]utura|[Gg]rande|[Gg]rossa|[Gg]rosseira|[Hh]istórica|[Ii]mensa|[Ii]mportante|[Ii]nternacional|[Mm]aior|[Mm]aravilhosa|[Mm]elhor|[Mm]esma|[Mm]ortífera|[Mm]últipla|[Nn]a|[Nn]aquela|[Nn]esta|[Nn]ona|[Nn]ossa|[Nn]ova|[Nn]uma|[Oo]itava|[Oo]u|[Oo]utra|[Pp]ela|[Pp]equena|[Pp]rimeira|[Pp]rópria|[Qq]ualquer|[Qq]uarta|[Qq]ue|[Qq]uerem|[Qq]uinta|[Rr]eal|[Rr]ecente|[Rr]eferida|[Rr]ica|[Ss]egunda|[Ss]em|[Ss]étima|[Ss]exta|[Ss]ua|[Tt]anta|[Tt]enho|[Tt]erá|[Tt]erceira|[Úú]ltima|[Uu]m|[Uu]ma|[Úú]nica|[Vv]elha|[Vv]erdadeira) fabrica\\b")

# Identificacao de inicio sem acento
vinicio = re.compile("(?i)\\b(a|[Aa]í|[Aa]o|[Aa]trasa|[Aa]trasaram|[Aa]trasou|[Bb]oicotam|[Bb]om|[Cc]ada|[Cc]om|[Cc]omo|[Cc]ujo|[Cc]urioso|[Dd]á|[Dd]ado|[Dd]ando|[Dd]ão|[Dd]ar|[Dd]ará|[Dd]aremos|[Dd]aria|[Dd]ava|[Dd]avam|[Dd]e|[Dd]ê|[Dd]emos|[Dd]era|[Dd]eram|[Dd]esse|[Dd]este|[Dd]eu|[Dd]ifícil|[Dd]isse|[Dd]o|[Ee]|[Ee]m|[Ee]sse|[Ee]ste|[Ii]ndica|[Mm]ais|[Mm]arca|[Mm]arcam|[Mm]arcou|[Mm]au|[Mm]elhor|[Mm]esmo|[Mm]eu|[Nn]aquele|[Nn]esse|[Nn]este|[Nn]o|[Nn]os|[Nn]osso|[Nn]ovo|[Nn]um|[Oo]|[Oo]s|[óó]timo|[Oo]u|[Pp]ara|[Pp]elo|[Pp]ior|[Pp]leno|[Pp]róprio|[Qq]ualquer|[Rr]ecente|[Ss]eu|[Ss]imples|[Tt]ardio|[Tt]em|[Tt]êm|[Tt]endo|[Tt]enha|[Tt]er|[Tt]erá|[Tt]erão|[Tt]eria|[Tt]eve|[Tt]ido|[Tt]inha|[Tt]inham|[Tt]iver|[Tt]ivera|[Tt]iveram|[Tt]odo|[Uu]m) inicio\\b")

# Identificacao de lastima sem acento
vlastima = re.compile("(?i)\\b([Aa]|[Àà]|[Cc]ausa|[Cc]ausava|[Cc]erta|[Cc]om|[Dd]a|[Dd]ava|[Dd]e|[Dd]essa|[Éé]|[Ee]m|[Ee]ra|[Ff]az|[Ff]azer|[Ff]azia|[Ff]oi|[Gg]rande|[Jj]ulgo|[Mm]enor|[Mm]erece|[Mm]ereço|[Mm]esma|[Mm]uita|[Nn]aquela|[Oo]u|[Pp]or|[Pp]rofunda|[Pp]rópria|[Qq]ue|[Ss]eria|[Ss]ó|[Ss]ua|[Tt]alvez|[Tt]anta|[Uu]ma|[Vv]erdadeira) lastima\\b")

# Identificacao de magoa sem acento
vmagoa = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]foga|[Aa]lguma|[Aa]penas|[Aa]quela|[Bb]astante|[Cc]ada|[Cc]ausou|[Cc]erta|[Cc]om|[Cc]uja|[Dd]a|[Dd]aquela|[Dd]e|[Dd]emonstrando|[Dd]essa|[Dd]oe|[Dd]olorosa|[Ee]|[Éé]|[Ee]m|[Ee]ra|[Ee]ssa|[Ee]sta|[Ee]stranha|[Ee]terna|[Ff]ingida|[Ff]ingindo|[Ff]osse|[Ff]unda|[Gg]rande|[Gg]uarda|[Gg]uardo|[Hh]á|[Hh]orrível|[Hh]umana|[Ii]mensa|[Ii]nfinda|[Ii]nfinita|[Ii]nterior|[Íí]ntima|[Jj]usta|[Ll]eve|[Mm]aior|[Mm]esma|[Mm]inha|[Mm]uita|[Nn]a|[Nn]aquela|[Nn]egra|[Nn]em|[Nn]enhuma|[Nn]essa|[Nn]ossa|[Nn]uma|[Oo]bscura|[Oo]culta|[Oo]u|[Pp]ara|[Pp]ela|[Pp]equena|[Pp]esada|[Pp]rimeira|[Pp]rofunda|[Pp]rópria|[Pp]ura|[Qq]ualquer|[Qq]ue|[Ss]em|[Ss]entia|[Ss]obra|[Ss]olene|[Ss]ua|[Ss]úbita|[Tt]al|[Tt]anta|[Tt]em|[Tt]ens|[Tt]inha|[Tt]raz|[Tt]ua|[Uu]ma|[Úú]nica|[Vv]erdadeira|[Vv]isível) magoa\\b")

# Identificacao de mascara sem acento
vmascara = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]lguma|[Aa]quela|[Aa]s|[Bb]ela|[Bb]oa|[Cc]ada|[Cc]élebre|[Cc]om|[Dd]a|[Dd]aquela|[Dd]e|[Dd]essa|[Dd]uma|[Éé]|[Ee]legante|[Ee]m|[Ee]spessa|[Ee]ssa|[Ee]sta|[Ee]stupenda|[Ee]xpressiva|[Ff]eia|[Ff]ria|[Hh]á|[Hh]edionda|[Ii]nteressante|[Mm]eia|[Mm]esma|[Mm]inha|[Mm]isteriosa|[Nn]a|[Nn]aquela|[Nn]em|[Nn]enhuma|[Nn]uma|[Pp]ela|[Pp]equena|[Pp]obre|[Pp]or|[Pp]ôr|[Pp]ôs|[Pp]rimeira|[Pp]unha|[Ss]em|[Ss]ob|[Ss]ólida|[Ss]ua|[Tt]al|[Tt]razem|[Tt]razes|[Tt]raziam|[Tt]ua|[Uu]m|[Uu]ma|[Uu]sa|[Uu]sará|[Uu]sarei|[Uu]sava|[Uu]sei|[Uu]so|[Uu]sou) mascara\\b")

# Identificacao de negocio sem acento
vnegocio = re.compile("(?i)\\b([Aa]|[Aa]lgum|[Aa]o|[Aa]quele|[Bb]elo|[Bb]om|[Cc]ada|[Cc]erto|[Cc]om|[Cc]omo|[Dd]aquele|[Dd]e|[Dd]esse|[Dd]este|[Dd]o|[Dd]os|[Éé]|[Ee]m|[Ee]mbrulhado|[Ee]ra|[Ee]sse|[Ee]ste|[Ee]xcelente|[Ff]az|[Ff]azer|[Ff]azia|[Ff]echado|[Ff]echar|[Ff]ez|[Ff]oi|[Ff]raco|[Ff]uturo|[Gg]igantesco|[Gg]rande|[Gg]rave|[Hh]á|[Hh]averá|[Hh]avia|[Ii]gual|[Ii]menso|[Ii]mportante|[Ll]iquidado|[Mm]aior|[Mm]ais|[Mm]aldito|[Mm]au|[Mm]elhor|[Mm]esmo|[Mm]eu|[Mm]uito|[Nn]aquele|[Nn]enhum|[Nn]esse|[Nn]este|[Nn]o|[Nn]osso|[Nn]ovo|[Nn]um|[Oo]|[Óó]timo|[Oo]utro|[Pp]ara|[Pp]arecer|[Pp]elo|[Pp]equeno|[Pp]or|[Pp]rincipal|[Pp]róprio|[Qq]ualquer|[Qq]ue|[Qq]uer|[Qq]uero|[Ss]emelhante|[Ss]er|[Ss]eu|[Ss]ó|[Tt]al|[Tt]em|[Tt]enho|[Tt]riste|[Tt]udo|[Úú]ltimo|[Uu]m) negocio\\b")

# Identificacao de pratica sem acento
vpratica = re.compile("(?i)\\b([A-Za-z]+dade|[A-Za-z]+mente|[A-Za-zêâ]+ncia|[A-Za-z]+ção|[Aa]|[Àà]|[Aa]cesa|[Aa]dquirir|[Aa]gradável|[Aa]legada|[Aa]lguma|[Aa]lta|[Aa]lternativa|[Aa]nálise|[Aa]ntiga|[Aa]quela|[Àà]quela|[Aa]rte|[Aa]ssim|[Aa]stronomia|[Aa]titude|[Aa]tiva|[Aa]tual|[Aa]ula|[Aa]videz|[Bb]astante|[Bb]oa|[Bb]reve|[Cc]ada|[Cc]erta|[Cc]oisa|[Cc]om|[Cc]omete|[Cc]omo|[Cc]omponente|[Cc]ompreensão|[Cc]onquista|[Cc]onsiderada|[Cc]onsiderado|[Cc]onstante|[Cc]ontra|[Cc]orreta|[Cc]ostumada|[Cc]ousa|[Cc]uja|[Dd]a|[Dd]e|[Dd]eficiente|[Dd]escolorida|[Dd]essa|[Dd]esta|[Dd]eterminada|[Dd]iscussão|[Dd]oce|[Dd]uma|[Ee]|[Éé]|[Ee]m|[Ee]ncetada|[Ee]nquanto|[Ee]nsino|[Ee]ra|[Ee]ram|[Ee]scola|[Ee]ssa|[Ee]sta|[Ee]stava|[Ee]ventual|[Ee]vidente|[Ee]xerceu|[Ee]xige|[Ee]xpressão|[Ee]xtensa|[Ff]az|[Ff]azem|[Ff]ervorosa|[Ff]ilosofia|[Ff]orma|[Ff]osse|[Ff]requente|[Gg]alante|[Gg]anhar|[Gg]ente|[Gg]eral|[Gg]estão|[Gg]lória|[Gg]rande|[Hh]á|[Hh]agiografia|[Hh]aja|[Hh]avido|[Hh]oje|[Ii]deia|[Ii]magem|[Ii]ncansável|[Ii]ncitar|[Ii]ndiferença|[Ii]ndigesta|[Ii]niciou|[Ii]ntolerável|[Jj]ardinagem|[Ll]abuta|[Ll]inguagem|[Ll]inha|[Ll]iturgia|[Ll]ógica|[Ll]onga|[Mm]á|[Mm]ais|[Mm]alta|[Mm]aneira|[Mm]as|[Mm]ecânica|[Mm]edicina|[Mm]edida|[Mm]eio|[Mm]elhor|[Mm]enor|[Mm]esma|[Mm]etade|[Mm]inha|[Mm]isteriosa|[Mm]oral|[Mm]udar|[Mm]uita|[Mm]uito|[Nn]a|[Nn]em|[Nn]enhuma|[Nn]esta|[Nn]ossa|[Nn]ova|[Nn]uma|[Oo]bra|[Oo]rdem|[Oo]u|[Oo]utra|[Pp]ara|[Pp]arte|[Pp]atrocinar|[Pp]ela|[Pp]equena|[Pp]essoa|[Pp]oesia|[Pp]olítica|[Pp]or|[Pp]orém|[Pp]ouca|[Pp]ouco|[Pp]reciso|[Pp]regunta|[Pp]rópria|[Pp]rova|[Pp]rovável|[Qq]ualquer|[Qq]ue|[Qq]uestão|[Rr]apariga|[Rr]azão|[Rr]eforma|[Rr]egra|[Rr]esposta|[Rr]evelava|[Rr]iqueza|[Ss]ádica|[Ss]aída|[Ss]alutar|[Ss]ão|[Ss]ediciosa|[Ss]eja|[Ss]em|[Ss]emelhante|[Ss]er|[Ss]ido|[Ss]imples|[Ss]obre|[Ss]ofisticada|[Ss]ua|[Tt]al|[Tt]anta|[Tt]anto|[Tt]ão|[Tt]em|[Tt]êm|[Tt]enho|[Tt]ens|[Tt]er|[Tt]ermo|[Tt]inha|[Tt]oda|[Tt]ornar|[Tt]ornou|[Uu]ma|[Vv]enerada|[Vv]ertente|[Vv]ida) pratica\\b")

# Identificacao de pratico sem acento
vpratico = re.compile("(?i)\\b([A-Za-z]+mente|[A-Za-z]+mento|[Aa]lcance|[Aa]lgum|[Aa]penas|[Aa]r|[Aa]specto|[Bb]astante|[Bb]em|[Cc]ampo|[Cc]aráter|[Cc]ientífico|[Cc]oeficiente|[Cc]omo|[Cc]ontrole|[Cc]urso|[Dd]e|[Dd]entista|[Dd]esempenho|[Dd]estino|[Dd]evasso|[Dd]o|[Dd]omínio|[Ee]|[Éé]|[Ee]feito|[Ee]m|[Ee]nsino|[Ee]ra|[Ee]spírito|[Ee]ste|[Ee]stivesse|[Ee]studo|[Ee]xame|[Ee]xemplo|[Ee]xercício|[Ff]ica|[Ff]ilósofo|[Gg]ênio|[Gg]rande|[Gg]uia|[Hh]ábito|[Hh]omem|[Ii]nteresse|[Ii]ntuito|[Ii]sso|[Jj]uízo|[Ll]ado|[Mm]ais|[Mm]anejo|[Mm]anual|[Mm]arido|[Mm]eio|[Mm]enos|[Mm]étodo|[Mm]odo|[Mm]uito|[Mm]undo|[Nn]ada|[Nn]avegante|[Nn]egociante|[Nn]ível|[Oo]|[Oo]bjetivo|[Oo]lho|[Oo]utro|[Pp]aís|[Pp]alavreado|[Pp]asso|[Pp]elo|[Pp]ouco|[Pp]rincípio|[Pp]roblema|[Pp]rocesso|[Pp]sicólogo|[Rr]amo|[Rr]ealismo|[Rr]emédio|[Rr]esponsável|[Rr]esultado|[Rr]oteiro|[Ss]aber|[Ss]enso|[Ss]entido|[Ss]er|[Ss]erá|[Ss]eria|[Ss]ignificado|[Ss]obretudo|[Ss]oldado|[Ss]ujeito|[Ss]uporte|[Tt]alento|[Tt]ão|[Tt]erreno|[Tt]este|[Tt]ornar|[Tt]rabalho|[Tt]reino|[Uu]m|[Uu]so|[Vv]alor) pratico\\b")

# Identificacao de publica sem acento
vpublica = re.compile("(?i)\\b([A-Za-z]+dade|[A-Za-z]+ência|[A-Za-z]+[sç]ão|[A-Za-z]+ária|[A-Za-z]+mente|[Àà]|[Aa]dvocacia|[Aa]genda|[Aa]legria|[Aa]lma|[Aa]ntipatia|[Áá]rea|[Aa]rte|[Aa]ula|[Aa]ura|[Bb]astante|[Bb]em|[Bb]iblioteca|briga|[Cc]abine|[Cc]adeia|[Cc]adeira|[Cc]ampanha|[Cc]arreira|[Cc]arta|[Cc]asa|[Cc]ausa|[Cc]ena|[Cc]ensura|[Cc]erimônia|[Cc]iência|[Cc]isterna|[Cc]ivil|[Cc]oisa|[Cc]om|[Cc]onfiança|[Cc]onsulta|[Cc]onversa|[Cc]ousa|[Cc]rítica|[Dd]a|[Dd]e|[Dd]efensora|[Dd]efensoria|[Dd]efesa|[Dd]emanda|[Dd]enúncia|[Dd]esgraça|[Dd]esonra|[Dd]esordem|[Dd]espedida|[Dd]espesa|[Dd]isputa|[Dd]ívida|é|[Ee]conomia|[Ee]m|[Ee]missora|[Ee]mpresa|[Ee]ncomenda|ênfase|[Ee]ntrada|[Ee]ra|[Ee]scola|[Ee]scritura|[Ee]sfera|[Ee]stá|[Ee]státua|[Ee]steira|[Ee]stima|[Ee]vento|[Ee]xpectativa|[Ff]ace|[Ff]azem|[Ff]azenda|[Ff]é|[Ff]eira|[Ff]esta|[Ff]ez|[Ff]igura|[Ff]inanceira|[Ff]ita|[Ff]onte|[Ff]ora|[Ff]oram|[Ff]orça|[Ff]orma|[Ff]ortuna|[Ff]osse|[Ff]otografia|[Gg]arantia|[Gg]astança|[Hh]igiene|[Hh]istória|[Hh]omenagem|[Hh]onra|[Hh]ospitalar|[Ii]magem|[Ii]ncerteza|[Ii]ndiferença|[Ii]nfraestrutura|[Ii]niciativa|[Ii]nsegurança|[Jj]ustiça|[Ll]eitura|[Ll]impeza|[Ll]inguagem|[Ll]ivraria|[Ll]uz|[Mm]ais|[Mm]áquina|[Mm]edicina|[Mm]emória|[Mm]entira|[Mm]esa|[Mm]iséria|[Mm]issa|[Mm]odéstia|[Mm]oral|[Mm]orte|[Mm]ulher|[Mm]useologia|[Nn]amorada|[Nn]atureza|[Nn]otícia|[Oo]bra|[Oo]ferta|[Oo]pinião|[Oo]rdem|[Oo]rgia|[Pp]alavra|[Pp]alestra|[Pp]arceria|[Pp]arte|[Pp]arvalheira|[Pp]asmaceira|[Pp]az|[Pp]ela|[Pp]enal|[Pp]erformance|[Pp]essoa|[Pp]ista|[Pp]olêmica|[Pp]olítica|[Pp]osse|[Pp]oupança|[Pp]raça|[Pp]referissem|[Pp]reguiça|[Pp]restando|[Pp]rofessora|[Pp]rofissional|[Pp]romessa|[Pp]romotora|[Pp]romotoria|[Pp]rova|[Qq]ueixa|[Qq]uestão|[Rr]azão|[Rr]eceita|[Rr]ede|[Rr]elevância|[Rr]epugnância|[Rr]etórica|[Rr]eunião|[Rr]iqueza|[Rr]uptura|[Ss]aúde|[Ss]ecretaria|[Ss]egurança|[Ss]eja|[Ss]er|[Ss]erá|[Ss]erventia|[Ss]ervidora|[Ss]ido|[Ss]impatia|[Ss]ocial|[Ss]olene|[Ss]uspeita|[Tt]anto|[Tt]ão|[Tt]arifa|[Tt]elefônica|[Tt]irania|[Tt]olerância|[Tt]omada|[Tt]orna|[Tt]ornada|[Tt]ornado|[Tt]ornar|[Tt]ornasse|[Tt]ornava|[Tt]ornei|[Tt]ornou|[Tt]rajetória|[Tt]ribuna|[Vv]enda|[Vv]ergonha|[Vv]ia|[Vv]iagem|[Vv]ida|[Vv]igilância|[Vv]irtude|[Vv]isita|[Vv]istoria|[Vv]oz) publica\\b")

# Identificacao de republica sem acento
vrepub = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]ntiga|[Aa]quela|[Aa]tual|[Cc]ada|[Cc]omo|[Cc]omunista|[Cc]uriosa|[Dd]a|[Dd]aquela|[Dd]e|[Dd]emocrática|[Dd]essa|[Dd]esta|[Dd]o|[Dd]uma|[Ee]|[Éé]|[Ee]m|[Ee]scola|[Ee]ssa|[Ee]sta|[Ff]amosa|[Ff]ez|[Ff]ormada|[Ff]ormosa|[Ff]osse|[Ff]utura|[Gg]loriosa|[Gg]rande|[Hh]averá|[Jj]á|[Jj]ovem|[Jj]uvenil|[Mm]aldita|[Mm]inha|[Mm]iserável|[Nn]a|[Nn]ascente|[Nn]em|[Nn]essa|[Nn]esta|[Nn]ossa|[Nn]ova|[Nn]uma|[Oo]u|[Oo]utra|[Pp]alavra|[Pp]ara|[Pp]ela|[Pp]equena|[Pp]obre|[Pp]or|[Pp]rimeira|[Qq]uarta|[Qq]ue|[Qq]uinta|[Ss]egunda|[Ss]ua|[Tt]erceira|[Úú]ltima|[Uu]ma|[Vv]elha|[Vv]erdadeira|[Vv]izinha) republica\\b")

# Identificacao de tabula sem acento
vtabula = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]lguma|[Aa]ntiga|[Aa]quela|[Àà]quela|[Aa]utêntica|[Cc]ada|[Cc]hamavam|[Cc]om|[Dd]a|[Dd]aquela|[Dd]e|[Dd]écima|[Dd]esta|[Dd]uma|[Ee]|[Éé]|[Ee]m|[Ee]ssa|[Ee]sta|[Ff]az|[Ff]ez|[Gg]rande|[Gg]rossa|[Ii]mensa|[Mm]aior|[Mm]elhor|[Mm]esma|[Nn]a|[Nn]aquela|[Nn]esta|[Nn]ossa|[Nn]ova|[Nn]uma|[Oo]u|[Oo]utra|[Pp]ela|[Pp]equena|[Pp]rópria|[Qq]ualquer|[Qq]ue|[Rr]eferida|[Rr]ica|[Ss]em|[Ss]ua|[Tt]anta|[Úú]ltima|[Uu]ma|[Úú]nica|[Vv]elha|[Vv]erdadeira) tabula\\b")

# Identificacao de tabula com acento
vtabulb = re.compile("(?i)\\b([A-Za-z]+or|[A-Za-z]+ora|[A-Za-zá]+tico|[A-Za-zé]+nico|[A-Za-z]+al|[A-Za-z]+dade|[A-Za-z]+ência|[A-Za-z]+[sç]ão|[A-Za-z]+ária|[A-Za-z]+mente|[Aa]lun[ao]) tábula\\b")

# Identificacao de credito sem acento
vcredito = re.compile("(?i)\\b([Aa]|[Aa]bonando|[Aa]bre|[Aa]brir|[Aa]cessar|[Aa]charia|[Aa]dmitem|[Aa]lcançarem|[Aa]lgum|[Aa]ltera|[Aa]mpliar|[Aa]ntigo|[Aa]o|[Aa]presente|[Aa]rranjou|[Bb]om|[Bb]onito|[Bb]uscando|[Bb]uscar|[Cc]olocou|[Cc]om|[Cc]omo|[Cc]oncedam|[Cc]oncede|[Cc]onceder|[Cc]oncedia|[Cc]onsegue|[Cc]onseguiram|[Cc]onsolidado|[Cc]onstarão|[Cc]onstituirá|[Cc]ontratar|[Cc]ronista|[Cc]ujo|[Dd]ado|[Dd]ai|[Dd]ando|[Dd]ar|[Dd]aria|[Dd]aríeis|[Dd]ava|[Dd]e|[Dd]ei|[Dd]era|[Dd]eram|[Dd]ermos|[Dd]esse|[Dd]esses|[Dd]este|[Dd]eterminado|[Dd]eu|[Dd]evido|[Dd]o|[Éé]|[Ee]|[Ee]m|[Ee]mpresas|[Ee]ncontra|[Ee]scasso|[Ee]sse|[Ee]ste|[Ee]studa|[Ee]vento|[Ee]ventual|[Ff]itas|[Ff]ornecer|[Gg]anhara|[Gg]anharão|[Gg]anhasse|[Gg]arante|[Gg]arantir|[Gg]rande|[Hh]á|[Hh]averá|[Ii]menso|[Ii]mplicam|[Ii]mplicará|[Ii]nteiro|[Ll]argo|[Mm]aior|[Mm]ais|[Mm]ediante|[Mm]enor|[Mm]enos|[Mm]erece|[Mm]erecem|[Mm]erecerem|[Mm]esmo|[Mm]eu|[Mm]uda|[Mm]uito|[Mm]unicipal|[Nn]ão|[Nn]egar|[Nn]em|[Nn]enhum|[Nn]o|[Nn]osso|[Nn]ovo|[Oo]|[Oo]btém|[Oo]bter|[Oo]btinham|[Oo]btiveram|[Oo]corre|[Oo]ferecendo|[Oo]u|[Pp]ara|[Pp]ede|[Pp]elo|[Pp]equeno|[Pp]iano|[Pp]leno|[Pp]or|[Pp]ouco|[Pp]oupança|[Pp]residente|[Pp]restar|[Pp]rimeiro|[Pp]róprio|[Qq]ualquer|[Qq]uer|[Qq]ueremos|[Rr]eceber|[Rr]eceberam|[Rr]egateavam|[Rr]espectivo|[Rr]estringem|[Rr]estringir|[Ss]egundo|[Ss]em|[Ss]eu|[Ss]implifica|[Ss]obre|[Ss]olicitar|[Tt]al|[Tt]anto|[Tt]em|[Tt]êm|[Tt]enho|[Tt]er|[Tt]erá|[Tt]eu|[Tt]inha|[Tt]iveram|[Tt]ivessem|[Tt]ransferir|[Úú]ltimo|[Uu]m|[Uu]tilizado|[Vv]olume|[Vv]osso) credito\\b")

# Identificacao de debito sem acento
vdebito = re.compile("(?i)\\b([Aa]|[Aa]lgum|[Aa]lto|[Aa]o|[Aa]presenta|[Aa]presente|[Aa]presentou|[Cc]om|[Cc]omo|[Dd]e|[Dd]esse|[Dd]o|[Éé]|[Ee]|[Ee]m|[Ee]sse|[Hh]aja|[Hh]avia|[Ii]mportante|[Mm]aior|[Mm]ediante|[Mm]enor|[Mm]eu|[Nn]o|[Nn]ovo|[Oo]|[Oo]u|[Pp]agar|[Pp]ara|[Pp]rimeiro|[Qq]ualquer|[Ss]em|[Ss]eu|[Ss]ó|[Ss]omente|[Tt]eu|[Uu]m|[Úú]nico|[Úú]ltimo|[Vv]elho|[Zz]era|[Zz]erou|[Zz]eraram|[Zz]erei) debito\\b")

# Identificacao de exercito sem acento
vexercito = re.compile("(?i)\\b([0-9]+º|[Aa]gora|[Aa]ntigo|[Aa]o|[Aa]poia|[Aa]quele|[Aa]utodenomina|[Bb]om|[Bb]rilhante|[Cc]ada|[Cc]hamado|[Cc]om|[Cc]ujo|[Dd]aquele|[Dd]e|[Dd]écimo|[Dd]enominada|[Dd]enominado|[Dd]esmobilizado|[Dd]esse|[Dd]este|[Dd]iminuto|[Dd]izer|[Dd]o|[Ee]|[Ee]lo|[Ee]m|[Ee]norme|[Ee]ntre|[Ee]quipado|[Ee]sse|[Ee]ste|[Ff]ormidável|[Ff]orte|[Ff]uturo|[Gg]rande|[Gg]rupo|[Hh]umilde|[Ii]menso|[Ii]ra|[Ll]amuriento|[Ll]ento|[Ll]evantar|[Ll]evarão|[Mm]aior|[Mm]elhor|[Mm]esmo|[Mm]obiliza|[Mm]otivado|[Nn]em|[Nn]esse|[Nn]o|[Nn]ono|[Nn]osso|[Nn]ovo|[Nn]um|[Nn]umeroso|[Oo]|[Oo]itavo|[Oo]rdenado|[Oo]u|[Oo]utro|[Pp]acífico|[Pp]elo|[Pp]equeno|[Pp]oderoso|[Pp]otencial|[Pp]reparado|[Pp]rimeiro|[Pp]róprio|[Qq]ualquer|[Qq]uarto|[Qq]uinto|[Rr]ebelde|[Rr]estante|[Ss]egundo|[Ss]em|[Ss]étimo|[Ss]eu|[Ss]exto|[Ss]ó|[Ss]obre|[Tt]erceiro|[Tt]ínhamos|[Uu]m|[Uu]ma|[Vv]asto|[Vv]elho|[Vv]issem) [Ee]xercito\\b")

# Identificacao de interprete sem acento
vinterprete = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]o|[Aa]penas|[Aa]quele|[Bb]rilhante|[Cc]abal|[Cc]ada|[Cc]arvalho|[Cc]omo|[Cc]ompetente|[Cc]ompleto|[Dd]a|[Dd]e|[Dd]epois|[Dd]o|[Ee]|[Éé]|[Ee]m|[Ee]nquanto|[Ee]ste|[Ee]xcelente|[Ff]amosa|[Ff]iel|[Ff]oi|[Gg]igante|[Gg]rande|[Gg]uia|[Ii]mportante|[Jj]ovem|[Ll]egítimo|[Mm]aior|[Mm]elhor|[Mm]inha|[Nn]o|[Oo]|[Pp]ara|[Pp]elo|[Pp]or|[Pp]rimeira|[Pp]rimeiro|[Pp]rincipal|[Pp]róprio|[Rr]espeitoso|[Ss]er|[Ss]eu|[Ss]ou|[Ss]ua|[Tt]em|[Tt]eria|[Tt]radutor|[Tt]ribunal|[Uu]m|[Uu]ma|[Úú]nico|[Vv]erboso|[Vv]irtual) interprete\\b")

# Identificacao de perpetua com acento
vperpetua = re.compile("(?i)\\b([Ee]|[Ee]la|[Ee]le|[Qq]ue|[Ss]e|[Vv]ocê) perpetua\\b")

# Identificacao de replica sem acento
vreplica = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]chou|[Aa]dmite|[Aa]dmitia|[Aa]dmito|[Aa]lguma|[Aa]quela|[Bb]oa|[Bb]reve|[Cc]om|[Cc]omo|[Cc]uja|[Dd]a|[Dd]ar|[Dd]ava|[Dd]e|[Ee]|[Éé]|[Ee]m|[Ee]xcelente|[Ee]xigem|[Hh]ouve|[Mm]inha|[Mm]ínima|[Mm]uita|[Nn]a|[Nn]aquela|[Oo]ferecia|[Pp]ela|[Pp]rimeira|[Pp]rovocava|[Qq]ualquer|[Ss]em|[Ss]ua|[Tt]em|[Tt]ermos|[Úú]ltima|[Uu]m|[Uu]ma|[Vv]erdadeira) replica\\b")

# Identificacao de tremula sem acento
vtremula = re.compile("(?i)\\b([Oo]) ([a-zA-Záâãçéêíóôú]+) tremula\\b")

# Identificacao de auxilio sem acento
vauxilio = re.compile("(?i)\\b([Aa]inda|[Aa]lgum|[Aa]ntes|[Aa]o|[Aa]quele|[Aa]ssegurados|[Bb]elo|[Bb]om|[Bb]uscar|[Cc]éu|[Cc]hamo|[Cc]hamou|[Cc]om|[Cc]omo|[Cc]onceder|[Cc]onsiderável|[Cc]ujo|[Dd]ava|[Dd]e|[Dd]ecisivo|[Dd]esse|[Dd]essem|[Dd]ispensando|[Dd]ispensar|[Dd]o|[Éé]|[Ee]|[Ee]m|[Ee]ncontrar|[Ee]sperando|[Ee]sperar|[Ee]sse|[Ff]orneceu|[Gg]eneroso|[Gg]rande|[Gg]rave|[Ii]mplorando|[Ii]mportante|[Ii]ncômodo|[Ii]nsignificante|[Jj]ustificável|[Mm]aior|[Mm]ais|[Mm]enor|[Mm]eu|[Nn]ecessitando|[Nn]egar|[Nn]enhum|[Nn]o|[Nn]osso|[Oo]|[Oo]bter|[Oo]ferecer|[Oo]u|[Oo]utro|[Pp]ara|[Pp]eça|[Pp]eçamos|[Pp]eças|[Pp]eço|[Pp]ede|[Pp]edido|[Pp]edia|[Pp]ediam|[Pp]edíamos|[Pp]edimos|[Pp]edindo|[Pp]edir|[Pp]edira|[Pp]edirá|[Pp]edirão|[Pp]ediram|[Pp]edirei|[Pp]ediremos|[Pp]ediria|[Pp]ediriam|[Pp]edisse|[Pp]ediu|[Pp]elo|[Pp]equeno|[Pp]ermanente|[Pp]oderoso|[Pp]or|[Pp]otente|[Pp]ouco|[Pp]recioso|[Pp]restam|[Pp]restando|[Pp]restar|[Pp]restaram|[Pp]restarão|[Pp]reste|[Pp]restou|[Pp]rocurando|[Pp]rocurar|[Pp]rocurassem|[Pp]rocurou|[Pp]rometeu|[Pp]rometeram|[Pp]rometiam|[Pp]ronto|[Qq]ualquer|[Rr]eceber|[Rr]eceberam|[Rr]ecebeu|[Rr]ecebido|[Rr]ecusavam|[Rr]equerendo|[Ss]em|[Ss]eu|[Ss]imples|[Ss]ó|[Ss]olicitar|[Ss]olicitou|[Ss]uplicaram|[Ss]uplicou|[Ss]ustentado|[Tt]ambém|[Tt]anto|[Tt]em|[Tt]êm|[Tt]emos|[Tt]enha|[Tt]enho|[Tt]erá|[Tt]erão|[Tt]erei|[Tt]eria|[Tt]eriam|[Tt]eu|[Tt]eve|[Tt]ido|[Tt]inha|[Tt]inham|[Tt]ive|[Tt]ivera|[Tt]iveram|[Tt]ríplice|[Uu]m|[Úú]nico|[Úú]til|[Vv]alioso|[Vv]eio|[Vv]inha|[Vv]irá|[Vv]iria|[Vv]iriam) auxilio\\b")

# Identificacao de distancia sem acento
vdistancia = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]brindo|[Aa]lguma|[Aa]parente|[Aa]quela|[Àà]quela|[Bb]astante|[Bb]oa|[Bb]reve|[Cc]aso|[Cc]erta|[Cc]om|[Cc]onsiderável|[Cc]uja|[Cc]urta|[Dd]a|[Dd]ada|[Dd]aquela|[Dd]e|[Dd]enominada|[Dd]esconsoladora|[Dd]eseja|[Dd]essa|[Dd]esta|[Dd]eterminada|[Dd]uma|[Ee]|[Éé]|[Ee]m|[Ee]ncurtar|[Ee]norme|[Ee]ssa|[Ee]sta|[Ee]xigia|[Ee]xiste|[Ee]xplorada|[Ff]irme|[Gg]anhar|[Gg]anhara|[Gg]élida|[Gg]rande|[Gg]uardando|[Gg]uardou|[Hh]á|[Hh]abitual|[Hh]avia|[Hh]ouvera|[Ii]gual|[Ii]mensa|[Ii]mpercorrível|[Ii]mpõe|[Ii]mpunha|[Ii]naceitável|[Ii]ndicando|[Ii]nsondável|[Ii]ntérmina|[Ii]nterpusesse|[Ii]ntransponível|[Ll]arga|[Ll]onga|[Mm]aior|[Mm]ais|[Mm]antendo|[Mm]antenha|[Mm]anter|[Mm]arcando|[Mm]édia|[Mm]eia|[Mm]enor|[Mm]esma|[Mm]ínima|[Mm]uita|[Nn]a|[Nn]aquela|[Nn]ecessária|[Nn]enhuma|[Nn]essa|[Nn]esta|[Nn]otável|[Nn]uma|[Oo]u|[Pp]ela|[Pp]equena|[Pp]or|[Pp]ouca|[Pp]rimeira|[Pp]rópria|[Qq]ualquer|[Qq]ue|[Qq]uer|[Qq]uerem|[Qq]ueria|[Qq]uero|[Qq]uis|[Rr]azoável|[Rr]elativa|[Rr]emovida|[Rr]epresentar|[Rr]espectiva|[Rr]espeitável|[Rr]espeitosa|[Ss]egura|[Ss]em|[Ss]ua|[Tt]al|[Tt]anta|[Tt]inha|[Tt]omando|[Tt]omou|[Úú]ltima|[Úú]nica|[Uu]ma|[Vv]ence|[Vv]enceu[Aa]|[Àà]|[Aa]brindo|[Aa]lguma|[Aa]parente|[Aa]quela|[Àà]quela|[Bb]astante|[Bb]oa|[Bb]reve|[Cc]aso|[Cc]erta|[Cc]om|[Cc]onsiderável|[Cc]uja|[Cc]urta|[Dd]a|[Dd]ada|[Dd]aquela|[Dd]e|[Dd]enominada|[Dd]esconsoladora|[Dd]eseja|[Dd]essa|[Dd]esta|[Dd]eterminada|[Dd]uma|[Ee]|[Éé]|[Ee]m|[Ee]ncurtar|[Ee]norme|[Ee]ssa|[Ee]sta|[Ee]xigia|[Ee]xiste|[Ee]xplorada|[Ff]irme|[Gg]anhar|[Gg]anhara|[Gg]élida|[Gg]rande|[Gg]uardando|[Gg]uardou|[Hh]á|[Hh]abitual|[Hh]avia|[Hh]ouvera|[Ii]gual|[Ii]mensa|[Ii]mpercorrível|[Ii]mpõe|[Ii]mpunha|[Ii]naceitável|[Ii]ndicando|[Ii]nsondável|[Ii]ntérmina|[Ii]nterpusesse|[Ii]ntransponível|[Ll]arga|[Ll]onga|[Mm]aior|[Mm]ais|[Mm]antendo|[Mm]antenha|[Mm]anter|[Mm]arcando|[Mm]édia|[Mm]eia|[Mm]enor|[Mm]esma|[Mm]ínima|[Mm]uita|[Nn]a|[Nn]aquela|[Nn]ecessária|[Nn]enhuma|[Nn]essa|[Nn]esta|[Nn]otável|[Nn]uma|[Oo]u|[Pp]ela|[Pp]equena|[Pp]or|[Pp]ouca|[Pp]rimeira|[Pp]rópria|[Qq]ualquer|[Qq]ue|[Qq]uer|[Qq]uerem|[Qq]ueria|[Qq]uero|[Qq]uis|[Rr]azoável|[Rr]elativa|[Rr]emovida|[Rr]epresentar|[Rr]espectiva|[Rr]espeitável|[Rr]espeitosa|[Ss]egura|[Ss]em|[Ss]ua|[Tt]al|[Tt]anta|[Tt]inha|[Tt]omando|[Tt]omou|[Úú]ltima|[Úú]nica|[Uu]ma|[Vv]ence|[Vv]elha) distancia\\b")

# Identificacao de circulo sem acento
vcirculo = re.compile("(?i)\\b([Aa]canhado|[Aa]legre|[Aa]mplo|[Aa]o|[Aa]pertado|[Aa]quele|[Aa]ristocrático|[Bb]reve|[Cc]ada|[Cc]hamado|[Cc]omo|[Dd]aquele|[Dd]e|[Dd]enominado|[Dd]eplorável|[Dd]esfecho|[Dd]esse|[Dd]este|[Dd]o|[Dd]um|[Dd]uplo|[Ee]m|[Ee]sse|[Ee]ste|[Ee]streitíssimo|[Ee]streito|[Ee]terno|[Ff]azer|[Ff]azia|[Ff]eito|[Ff]izemos|[Ff]ormando|[Ff]ormavam|[Gg]rande|[Ii]menso|[Ii]mplacável|[Ii]mponderável|[Ii]mportante|[Ll]argo|[Ll]egítimo|[Ll]imitado|[Ll]iterário|[Mm]aior|[Mm]aldito|[Mm]eio|[Mm]esmo|[Mm]esquinho|[Mm]eu|[Mm]onstruoso|[Nn]aquele|[Nn]esse|[Nn]este|[Nn]o|[Nn]osso|[Nn]outro|[Nn]um|[Oo]|[Oo]itavo|[Oo]utro|[Pp]elo|[Pp]equeníssimo|[Pp]equeno|[Pp]or|[Pp]rimeiro|[Pp]rimitivo|[Pp]róprio|[Qq]ue|[Qq]uinto|[Rr]eduzido|[Rr]espeitoso|[Ss]egundo|[Ss]erviram|[Ss]eu|[Ss]uspeito|[Tt]errível|[Tt]inha|[Tt]razia|[Úú]ltimo|[Uu]m|[Uu]ni|[Vv]asto) circulo\\b")

# Identificacao de transito sem acento
vtransito= re.compile("(?i)\\b([Aa]gravam|[Aa]lgum|[Aa]livia|[Aa]o|[Bb]astante|[Bb]om|[Cc]aótico|[Cc]om|[Cc]ontrolam|[Cc]ujo|[Dd]e|[Dd]o|[Ee]|[Ee]m|[Ee]nfrentou|[Ee]sse|[Ee]ste|[Ff]ácil|[Ff]ará|[Gg]rande|[Hh]á|[Hh]aver|[Hh]avia|[Ii]nferniza|[Ii]nfernizam|[Ii]ntermitente|[Ll]argo|[Ll]ento|[Ll]ivre|[Mm]aior|[Mm]ais|[Mm]uda|[Mm]uito|[Nn]este|[Nn]o|[Oo]|[Oo]ferecer|[Oo]rientam|[Pp]ara|[Pp]ára|[Pp]aram|[Pp]elo|[Pp]ior|[Pp]or|[Pp]revê|[Pp]róprio|[Rr]ealizando|[Rr]eformula|[Ss]em|[Ss]eu|[Ss]imples|[Ss]obre|[Ss]ol|[Tt]anto|[Tt]em|[Tt]emeroso|[Tt]er|[Tt]inha|[Tt]iveram|[Tt]ivessem|[Úú]ltimo|[Uu]m) transito\\b")

# Identificacao de tramite sem acento
vtramite= re.compile("(?i)\\b([Aa]lgum|[Aa]o|[Aa]quele|[Bb]om|[Cc]ada|[Cc]om|[Cc]ontrolam|[Cc]ujo|[Dd]e|[Dd]o|[Ee]|[Ee]m|[Ee]sse|[Ee]ste|[Hh]á|[Hh]aver|[Hh]avia|[Ll]ivre|[Mm]aior|[Mm]ais|[Mm]uda|[Mm]uito|[Nn]ecessário|[Nn]esse|[Nn]este|[Nn]o|[Oo]|[Pp]ara|[Pp]ára|[Pp]aram|[Pp]elo|[Pp]or|[Pp]rimeiro|[Rr]eferido|[Ss]em|[Ss]eu|[Ss]imples|[Ss]obre|[Tt]anto|[Tt]em|[Tt]er|[Tt]inha|[Tt]iveram|[Tt]ivessem|[Úú]ltimo|[Uu]m) tramite\\b")

# Identificacao de animo sem acento
vanimo= re.compile("(?i)\\b([Aa]lgum|[Aa]o|[Aa]quele|[Bb]om|[Cc]ada|[Cc]om|[Cc]ujo|[Dd]e|[Dd]o|[Éé]|[Ee]|[Ee]m|[Ee]sse|[Ee]ste|[Hh]á|[Hh]aver|[Hh]avia|[Mm]aior|[Mm]ais|[Mm]elhor|[Mm]eu|[Mm]uda|[Mm]uito|[Nn]esse|[Nn]este|[Nn]o|[Nn]osso|[Oo]|[Pp]elo|[Pp]or|[Ss]em|[Ss]eu|[Ss]obre|[Tt]anto|[Tt]em|[Tt]er|[Tt]eu|[Tt]inha|[Tt]iveram|[Tt]ivessem|[Uu]m) animo\\b")

# Identificacao de parodia sem acento
vparodia = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]lguma|[Aa]quela|[Aa]trevida|[Bb]ela|[Bb]oa|[Cc]ada|[Cc]élebre|[Cc]erta|[Cc]om|[Dd]a|[Dd]aquela|[Dd]e|[Dd]essa|[Dd]esta|[Dd]ivertida|[Dd]uma|[Éé]|[Ee]m|[Ee]ssa|[Ee]sta|[Ff]eia|[Ii]nteressante|[Mm]esma|[Mm]inha|[Nn]a|[Nn]aquela|[Nn]em|[Nn]enhuma|[Nn]uma|[Oo]utra|[Pp]ela|[Pp]equena|[Pp]obre|[Pp]or|[Pp]ôr|[Pp]rimeira|[Ss]em|[Ss]ua|[Tt]al|[Tt]razem|[Tt]raziam|[Tt]ua|[Uu]ma|[Vv]elha) parodia\\b")

# Identificacao de concilio sem acento
vconcilio= re.compile("(?i)\\b([CILMXV]+|[Aa]lgum|[Aa]o|[Aa]quele|[Cc]ada|[Cc]ujo|[Dd]e|[Dd]o|[Ee]m|[Ee]sse|[Ee]ste|[Hh]á|[Hh]aver|[Hh]avia|[Mm]aior|[Mm]ais|[Mm]elhor|[Mm]uda|[Mm]uito|[Nn]esse|[Nn]este|[Nn]o|[Nn]osso|[Nn]um|[Oo]|[Pp]elo|[Pp]or|[Pp]rimeiro|[Qq]uarto|[Qq]uinto|[Ss]agrado|[Ss]anto|[Ss]ecreto|[Ss]egundo|[Ss]em|[Ss]exto|[Ss]étimo|[Ss]eu|[Tt]erceiro|[Uu]m) [Cc]oncilio\\b")

# Identificacao de continua sem acento
vcontinua= re.compile("(?i)\\b([a-z]+[sç]ão|[Aa]|[Àà]|[Aa]nálise|[Aa]prendizagem|[Aa]quela|[Cc]amada|[Cc]onectividade|[Cc]orreia|[Cc]orrente|[Dd]a|[Dd]e|[Dd]escarga|[Dd]essa|[Dd]iscursiva|[Éé]|[Ee]m|[Ee]ra|[Ee]ssa|[Ee]scala|[Ee]spera|[Ee]ssa|[Ee]sta|[Ee]stá|[Ee]strutura|[Ff]ase|[Ff]oi|[Ff]orma|[Ii]nconstância|[Ll]eitura|[Ll]ida|[Ll]inha|[Ll]uta|[Mm]ais|[Mm]alha|[Mm]aneira|[Mm]archa|[Mm]assa|[Mm]atriz|[Mm]edida|[Mm]elhoria|[Mm]oldagem|[Nn]a|[Nn]atureza|[Nn]ecessariamente|[Nn]essa|[Nn]uma|[Pp]arte|[Pp]ela|[Pp]or|[Pp]ública|[Qq]ueda|[Rr]ajada|[Rr]eciclagem|[Ss]ai|[Ss]aída|[Ss]er|[Ss]érie|[Ss]uperfície|[Tt]ão|[Uu]ma|[Vv]igilância) continua\\b")

# Identificacao de continuo sem acento
vcontinuo= re.compile("(?i)\\b([Aa]|[Aa]bandono|[Aa]buso|[Aa]cionamento|[Aa]çoite|[Aa]companhamento|[Aa]fastamento|[Aa]lgo|[Aa]ntigo|[Aa]o|[Aa]rdor|[Aa]taque|[Aa]to|[Aa]trito|[Aa]umento|[Aa]ustero|[Bb]ailado|[Bb]aixo|[Bb]arateamento|[Bb]arulho|[Bb]om|[Bb]rado|[Cc]iclo|[Cc]ircular|[Cc]itoplasma|[Cc]om|[Cc]ombate|[Cc]omércio|[Cc]onjunto|[Cc]rescendo|[Cc]rescer|[Cc]rescimento|[Cc]ultivo|[Dd]aquele|[Dd]e|[Dd]ecair|[Dd]eclive|[Dd]esde|[Dd]esdobramento|[Dd]esejo|[Dd]esencontro|[Dd]esenvolver|[Dd]esenvolvimento|[Dd]esgaste|[Dd]eslumbramento|[Dd]isparo|[Dd]o|[Dd]um|[Éé]|[Ee]m|[Ee]mpobrecimento|[Ee]nfraquecimento|[Ee]ntre|[Ee]ra|[Ee]rosão|[Ee]sforço|[Ee]spectro|[Ee]sse|[Ee]stado|[Ee]ste|[Ff]io|[Ff]lagelo|[Ff]luido|[Ff]luxo|[Ff]oco|[Ff]ogo|[Ff]oi|[Ff]ornecimento|[Ff]orno|[Ff]riso|[Ff]ui|[Gg]emido|[Gg]iro|[Gg]olpe|[Gg]ravitar|[Hh]ábito|[Ii]nstrumento|[Ii]nterdisciplinar|[Ii]nteresse|[Ii]nvestimento|[Jj]ato|[Jj]ogo|[Ll]abor|[Ll]abutar|[Ll]evantamento|[Mm]ais|[Mm]ental|[Mm]odo|[Mm]onitoramento|[Mm]ontanhoso|[Mm]ovimento|[Mm]urmúrio|[Nn]o|[Nn]ovo|[Nn]um|[Nn]umérico|[Oo]|[Oo]rdem|[Pp]adecer|[Pp]aralelo|[Pp]elo|[Pp]ensamento|[Pp]esar|[Pp]igarro|[Pp]or|[Pp]ovoamento|[Pp]rocesso|[Qq]uase|[Qq]uerer|[Rr]econhecimento|[Rr]egime|[Rr]egistador|[Rr]egisto|[Rr]esfriamento|[Rr]evezamento|[Rr]isco|[Rr]itmo|[Rr]odopiar|[Rr]olo|[Rr]uído|[Rr]umor|[Ss]eja|[Ss]ente|[Ss]erá|[Ss]erviço|[Ss]eu|[Ss]imples|[Ss]istema|[Ss]oluçar|[Ss]om|[Ss]onho|[Ss]ubconsciente|[Ss]ubir|[Ss]usto|[Tt]ão|[Tt]empo|[Tt]érmico|[Tt]exto|[Tt]rabalho|[Tt]raçado|[Tt]raço|[Tt]rato|[Tt]rem|[Tt]remor|[Tt]ropear|[Tt]urbilhão|[Uu]m|[Uu]so|[Vv]aivém|[Vv]elho|[Vv]ertia|[Vv]olver|[Vv]ômito|[Vv]oo|[Vv]oz|[Zz]unido) continuo\\b")

# Identificacao de especifica sem acento
vspecifa= re.compile("(?i)\\b([a-z]+nte|[a-z]+tar|[a-z]+o[nsrv]a|[a-z]+[sç]ão|[a-záâéíóôú]+ica|[a-záâéíóôú]+ria|[a-záâéíóôú]+tura|[a-záâéíóôú]+ral|[a-záâéíóôú]+ade|[a-zêâ]+ncia|[a-z]+gia|[a-z]+cial|[a-z]+iva|[a-z]+ional|[a-záâéíóôú]+ica|[Aa]bordagem|[Aa]nálise|[Áá]rea|[Aa]rte|[Aa]ssembleia|[Aa]titude|[Bb]ase|[Bb]em|[Bb]iblioteca|[Cc]artilha|[Cc]ausa|[Cc]autelar|[Cc]lasse|[Cc]láusula|[Cc]oisa|[Cc]om|[Cc]onhecimento|[Cc]or|[Cc]ura|[Dd]a|[Dd]ata|[Dd]e|[Dd]ieta|[Dd]iferença|[Dd]isciplina|[Dd]oença|[Ee]|[Éé]|[Ee]mpresa|[Êê]nfase|[Ee]nzima|[Éé]poca|[Ee]ra|[Ee]scala|[Ee]scolha|[Ee]tnia|[Ff]aixa|[Ff]erramenta|[Ff]obia|[Ff]oi|[Ff]olha|[Ff]orma|[Ff]ranja|[Gg]ama|[Ii]deia|[Ii]nterface|[Ll]egal|[Ll]ei|[Ll]íngua|[Ll]inguagem|[Ll]inha|[Mm]ais|[Mm]aneira|[Mm]arca|[Mm]assa|[Mm]edida|[Mm]ental|[Mm]eta|[Mm]udança|[Mm]uito|[Mm]undo|[Mm]usical|[Nn]a|[Nn]egócios|[Nn]enhuma|[Nn]orma|[Oo]rdem|[Oo]utra|[Pp]aixão|[Pp]arte|[Pp]esquisa|[Pp]essoa|[Pp]reço|[Pp]reços|[Pp]roposta|[Pp]roteína|[Qq]ueixa|[Qq]uestão|[Rr]aça|[Rr]egião|[Rr]egra|[Rr]eligião|[Rr]esposta|[Rr]iqueza|[Ss]aída|[Ss]ala|[Ss]er|[Ss]impatia|[Ss]intaxe|[Ss]ubárea|[Tt]ão|[Tt]arefa|[Tt]ribal|[Tt]utela|[Tt]utelar|[Uu]ma|[Uu]rbana|[Vv]ariável|[Vv]ia|[Vv]isual|[Vv]ocabular) especifica\\b")

# Identificacao de especifico sem acento
vspecifo= re.compile("(?i)\\b([a-z]+nte|[a-z]+nto|[a-z]+sto|[a-z]+i[tv]o|[a-z]+ado|[a-z]+[sç]ão|[a-z]+[dt]or|[a-z]+eto|[a-záâéíóôú]+ico|[a-záâéíóôú]+rio|[a-záâéíóôú]+ulo|[a-záâéíóôú]+ismo|[Aa]cordo|[Aa]lgo|[Aa]lgoritmo|[Aa]lvo|[Ââ]mbito|[Aa]nticorpo|[Aa]poio|[Aa]rranjo|[Aa]rtigo|[Aa]specto|[Aa]to|[Bb]em|[Cc]alor|[Cc]aminho|[Cc]ampo|[Cc]anal|[Cc]arácter|[Cc]aso|[Cc]heiro|[Cc]onteúdo|[Cc]ontexto|[Cc]orpo|[Cc]urso|[Dd]e|[Dd]eficit|[Dd]emanda|[Dd]esejo|[Dd]espacho|[Dd]iploma|[Dd]o|[Dd]olo|[Dd]omínio|[Ee]|[Éé]|[Ee]m|[Ee]mbora|[Ee]ndereço|[Ee]nergia|[Ee]stilo|[Ee]stresse|[Ee]studo|[Ee]xame|[Ee]xemplo|[Ff]enômeno|[Ff]im|[Ff]oi|[Ff]ruto|[Ff]undo|[Gg]ene|[Gg]ênero|[Gg]rupo|[Hh]abitat|[Íí]ndice|[Ii]ndivíduo|[Ii]nimigo|[Ii]nteresse|[Ii]ntervalo|[Jj]argão|[Ll]imite|[Ll]ocal|[Ll]ugar|[Mm]ais|[Mm]andato|[Mm]arketing|[Mm]aterial|[Mm]étodo|[Mm]eu|[Mm]odo|[Nn]ada|[Nn]ão|[Nn]ome|[Nn]úmero|[Oo]|[Oo]nda|[Óó]rgão|[Pp]adrão|[Pp]alco|[Pp]apel|[Pp]erigo|[Pp]eso|[Pp]esquisa|[Pp]razo|[Pp]reparo|[Pp]roblema|[Pp]rocesso|[Pp]roduto|[Pp]rograma|[Qq]uase|[Rr]egistro|[Rr]emédio|[Rr]isco|[Rr]itual|[Rr]oteiro|[Ss]eja|[Ss]ensor|[Ss]entido|[Ss]erviço|[Ss]istema|[Ss]ite|[Ss]ocial|[Ss]om|[Ss]oro|[Ss]ubstrato|[Tt]ão|[Tt]ecidular|[Tt]empo|[Tt]ermo|[Tt]exto|[Tt]ipo|[Tt]rabalho|[Tt]reino|[Úú]ltimo|[Uu]m|[Uu]so|[Vv]alor|[Vv]isual) especifico\\b")

# Identificacao de estimulo sem acento
vestimulo= re.compile("(?i)\\b([Aa]|[Aa]lgum|[Aa]o|[Àà]quele|[Aa]rdente|[Cc]ada|[Cc]laro|[Cc]om|[Cc]omo|[Cc]onfirma|[Dd]ado|[Dd]aquele|[Dd]ar|[Dd]ava|[Dd]avam|[Dd]e|[Dd]ê|[Dd]efiniu|[Dd]esse|[Dd]eram|[Dd]eterminado|[Dd]eu|[Dd]o|[Dd]um|[Ee]|[Éé]|[Ee]m|[Ee]ncontra|[Ee]ncontrava|[Ee]ncontravam|[Ee]ncontraram|[Ee]ncontrasse|[Ee]ncontrou|[Ee]ntre|[Ee]sse|[Ee]ste|[Ee]stranho|[Ee]xtrair|[Ee]xtremo|[Ff]ormar|[Ff]orte|[Gg]rande|[Hh]á|[Hh]aja|[Hh]avia|[Hh]ouve|[Hh]ouvesse|[Hh]ouver|[Ii]mportante|[Jj]á|[Ll]igeiro|[Ll]ubrificante|[Mm]aior|[Mm]ais|[Mm]elhor|[Mm]esmo|[Mm]eu|[Mm]ínimo|[Nn]ão|[Nn]atural|[Nn]em|[Nn]enhum|[Nn]o|[Nn]obre|[Nn]ovo|[Nn]osso|[Nn]um|[Oo]|[Oo]u|[Oo]utro|[Pp]alavra|[Pp]ara|[Pp]elo|[Pp]oderoso|[Pp]or|[Pp]rimeiro|[Pp]rincipal|[Pp]roduz|[Pp]roduza|[Pp]roduziu|[Pp]rover|[Qq]uente|[Rr]ecebe|[Rr]ecebeu|[Rr]ecebia|[Rr]econfortante|[Ss]ão|[Ss]em|[Ss]emelhante|[Ss]eu|[Ss]imples|[Tt]al|[Tt]em|[Tt]ivesse|[Tt]odo|[Uu]m|[Úú]ltimo|[Úú]nico|[Uu]sava|[Vv]ivo) estimulo\\b")

# Identificacao de integra sem acento
vintegra= re.compile("(?i)\\b([Aa]|[Àà]|[Bb]astante|[Ee]|[Ee]mpresa|[Ee]stava|[Ff]amília|[Ff]irma|[Ff]orma|[Gg]arota|[Ii]nterna|[Ll]eia|[Mm]ais|[Mm]elhor|[Mm]enina|[Mm]oça|[Mm]uito|[Mm]ulher|[Nn]a|[Pp]ermanece|[Pp]essoa|[Ss]ua|[Vv]ida) integra\\b")

# Identificacao de integro sem acento
vintegro= re.compile("(?i)\\b([Aa]bsolutamente|[Bb]astante|[Dd]e|[Dd]o|[Ee]|[Hh]omem|[Mm]agistrado|[Mm]ais|[Mm]antenha|[Mm]as|[Mm]enino|[Mm]uito|[Oo]|[Pp]ersonagem|[Rr]apaz|[Tt]udo) integro\\b")

# Identificacao de intimo sem acento
vintimo = re.compile("(?i)\\b([A-Za-z]+mente|[A-Za-z]+mento|[Aa]bandono|[Aa]fetivo|[Aa]gora|[Aa]gravo|[áá]lbum|[Aa]lgum|[Aa]liado|[Aa]lvoroço|[Aa]margor|[Aa]migo|[Aa]mor|[Aa]nelo|[Aa]ntagonismo|[Aa]o|[Aa]quele|[Aa]rrepio|[Bb]aile|[Bb]om|[Cc]aderno|[Cc]amarada|[Cc]aminho|[Cc]ântico|[Cc]anto|[Cc]arácter|[Cc]aráter|[Cc]asulo|[Cc]ausaria|[Cc]ausava|[Cc]hoque|[Cc]írculo|[Cc]lima|[Cc]olóquio|[Cc]om|[Cc]ombate|[Cc]omo|[Cc]ompanheiro|[Cc]ompromisso|[Cc]onselheiro|[Cc]ontato|[Cc]onvívio|[Cc]unho|[Dd]aquele|[Dd]e|[Dd]ebate|[Dd]ecoro|[Dd]esafio|[Dd]esejo|[Dd]esespero|[Dd]esgosto|[Dd]esprezo|[Dd]iálogo|[Dd]iário|[Dd]iscurso|[Dd]o|[Dd]rama|[Ee]|[Éé]|[Ee]lo|[Ee]m|[Ee]ncanto|[Ee]nxoval|[Ee]ra|[Ee]stá|[Ee]ste|[Ff]enômeno|[Ff]icara|[Ff]icou|[Ff]oco|[Ff]oi|[Ff]oro|[Ff]ui|[Ff]uror|[Gg]emido|[Gg]overno|[Gg]ozo|[Gg]rito|[Hh]omem|[Hh]orrível|[Ii]deal|[Ii]mpulso|[Ii]ndicar|[Ii]ntérprete|[Jj]antar|[Jj]úbilo|[Jj]uízo|[Ll]aço|[Ll]irismo|[Ll]ivro|[Ll]uto|[Mm]ais|[Mm]enos|[Mm]eu|[Mm]istério|[Mm]odelo|[Mm]olde|[Mm]onólogo|[Mm]otivo|[Mm]uito|[Mm]undo|[Nn]egócio|[Nn]o|[Nn]osso|[Nn]um|[Oo]|[Oo]lhar|[Oo]ráculo|[Oo]rgulho|[Pp]adecer|[Pp]anorama|[Pp]apel|[Pp]ara|[Pp]arecia|[Pp]arente|[Pp]ede|[Pp]elo|[Pp]iano|[Pp]or|[Pp]ouco|[Pp]razer|[Pp]róprio|[Qq]ualquer|[Qq]uase|[Qq]ue|[Rr]apaz|[Rr]ecolho|[Rr]efeitório|[Rr]elato|[Ss]ão|[Ss]arau|[Ss]ecretário|[Ss]endo|[Ss]enso|[Ss]entido|[Ss]er|[Ss]eria|[Ss]eu|[Ss]ido|[Ss]obressalto|[Ss]ol|[Ss]ossego|[Ss]uspiro|[Tt]ambém|[Tt]ão|[Tt]erreno|[Tt]error|[Tt]eu|[Tt]odo|[Tt]om|[Tt]orna|[Tt]rato|[Uu]m|[Vv]erbo|[Vv]iver) intimo\\b")

# Identificacao de intima sem acento
vintima= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[a-zêâ]+ncia|[a-z]+[dhistxç]ão|[Aa]|[Aa]gonia|[Aa]legria|[Aa]lma|[Aa]miga|[Aa]mizade|[Aa]nsiedade|[Aa]quela|[Aa]titude|[Aa]tividade|[Bb]em|[Bb]ondade|[Cc]alma|[Cc]arta|[Cc]ausa|[Cc]eia|[Cc]ena|[Cc]erteza|[Cc]om|[Cc]omo|[Cc]onfiança|[Cc]onsulta|[Cc]ontextura|[Cc]onversa|[Cc]rônica|[Cc]umplicidade|[Dd]a|[Dd]aquela|[Dd]e|[Dd]elícia|[Dd]escoberta|[Dd]essa|[Dd]esta|[Dd]or|[Ee]|[Éé]|[Ee]m|[Ee]nergia|[Ee]ra|[Ee]sfera|[Ee]sperança|[Ee]ssa|[Ee]sta|[Ee]stante|[Ee]strutura|[Ff]alta|[Ff]azer|[Ff]elicidade|[Ff]esta|[Ff]ez|[Ff]oi|[Ff]orça|[Ff]orma|[Ff]osse|[Ff]raternidade|[Ff]rieza|[Gg]inástica|[Hh]igiene|[Hh]istória|[Ii]nimiga|[Ii]rlanda|[Jj]á|[Ll]inguagem|[Ll]iterária|[Ll]uta|[Ll]uz|[Mm]agoa|[Mm]ágoa|[Mm]ais|[Mm]antém|[Mm]as|[Mm]enos|[Mm]esa|[Mm]esma|[Mm]inha|[Mm]oralidade|[Mm]ostrar|[Mm]uita|[Mm]uito|[Nn]a|[Nn]ão|[Nn]atureza|[Nn]ecessidade|[Nn]ossa|[Nn]ota|[Nn]udez|[Nn]uma|[Oo]rdem|[Oo]u|[Pp]agina|[Pp]ágina|[Pp]alavra|[Pp]alestra|[Pp]alpite|[Pp]arte|[Pp]az|[Pp]ela|[Pp]ergunta|[Pp]erspectiva|[Pp]essoa|[Pp]oesia|[Pp]ossuem|[Pp]rática|[Qq]ue|[Qq]uem|[Rr]ealidade|[Rr]eviravolta|[Rr]evolta|[Rr]evolve|[Rr]oupa|[Ss]egurança|[Ss]enhora|[Ss]er|[Ss]inceridade|[Ss]ociedade|[Ss]ombra|[Ss]ou|[Ss]ua|[Tt]ambém|[Tt]oda|[Tt]ornou|[Tt]ortura|[Tt]risteza|[Tt]única|[Uu]ma|[Vv]erdade|[Vv]ida|[Vv]oz|[Zz]ona) intima\\b")

# Identificacao de renuncia sem acento
vrenuncia = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]nuncia|[Aa]nunciava|[Aa]nunciam|[Aa]nunciou|[Cc]om|[Cc]omo|[Cc]oncedeu|[Cc]uja|[Dd]a|[Dd]e|[Dd]esaprova|[Dd]oce|[Dd]upla|[Ee]|[Éé]|[Ee]m|[Ee]ssa|[Ee]stúpida|[Ee]ventual|[Ff]azer|[Ff]requente|[Gg]enerosa|[Hh]ouve|[Ii]mediata|[Ii]mpliquem|[Ii]mporta|[Ii]mportará|[Mm]inha|[Nn]a|[Nn]ega|[Nn]ossa|[Nn]uma|[Oo]u|[Pp]ela|[Pp]equena|[Pp]osterior|[Pp]resumir|[Pp]romove|[Pp]romover|[Pp]ura|[Rr]epresentaram|[Ss]ignifique|[Ss]ua|[Tt]ua|[Úú]ltima|[Úú]nica|[Uu]ma) renuncia\\b")

# Identificacao de denuncia sem acento
vdenuncia = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]lguma|[Aa]pós|[Aa]presenta|[Aa]presentam|[Aa]presentar|[Aa]presentaram|[Aa]presente|[Aa]presentem|[Aa]presentou|[Aa]pura|[Aa]purando|[Aa]purar|[Aa]quela|[Aa]purou|[Aa]ssunto|[Cc]lara|[Cc]om|[Cc]omo|[Cc]onfirma|[Cc]onfirmam|[Cc]onforme|[Cc]ovarde|[Cc]uja|[Dd]a|[Dd]ar|[Dd]as|[Dd]e|[Dd]esmente|[Dd]esmentem|[Dd]esmentiu|[Dd]essa|[Dd]uma|[Ee]|[Éé]|[Ee]m|[Ee]ncaminha|[Ee]ncaminham|[Ee]ncaminhou|[Ee]ngolir|[Ee]specífica|[Ee]ssa|[Ee]sta|[Ee]xista|[Ee]xistiu|[Ee]xistindo|[Ff]alsa|[Ff]ormaliza|[Ff]ormalizando|[Ff]ormalizaram|[Ff]ormalizou|[Ff]ormulada|[Gg]rande|[Gg]rave|[Hh]á|[Hh]avia|[Hh]ouve|[Hh]ouver|[Ii]nexplicada|[Ii]ngênua|[Ii]nvestiga|[Ii]nvestigam|[Ii]nvestigar|[Ii]nvestigava|[Jj]ulga|[Jj]ulgam|[Jj]ulgou|[Ll]evar|[Mm]as|[Mm]ediante|[Mm]eia|[Mm]uita|[Nn]a|[Nn]ão|[Nn]enhuma|[Nn]ova|[Nn]uma|[Oo]ferece|[Oo]ferecer|[Oo]fereceu|[Oo]u|[Oo]utra|[Pp]ela|[Pp]or|[Pp]ossível|[Pp]reciosa|[Pp]resente|[Pp]rimeira|[Qq]ualquer|[Rr]ecebe|[Rr]eceberam|[Rr]eceberem|[Rr]ecebeu|[Ss]egunda|[Ss]egundo|[Ss]imples|[Ss]obre|[Ss]ua|[Tt]em|[Tt]errível|[Tt]oda|[Tt]remenda|[Uu]ma|[Vv]elada) denuncia\\b")

# Identificacao de libero sem acento
vlibero = re.compile("(?i)\\b([Cc]omo|[Cc]opa|[Dd]e|[Dd]o|[Éé]|[Ee]ducador|[Ff]oi|[Ff]ui|[Gg]rande|[Mm]aior|[Mm]elhor|[Nn]o|[Oo]|[Tt]ermo|[Uu]m) libero\\b")

# Identificacao de liquido sem acento
vliquido = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[Aa]|[Aa]ço|[Aa]dorado|[Aa]lcatrão|[Áá]lcool|[Aa]lgum|[Aa]mortecedor|[Aa]o|[Aa]quele|[Aa]r|[Aa]ro|[Aa]rtificioso|[Aa]tivo|[Aa]umento|[Aa]zoto|[Bb]arro|[Bb]astante|[Bb]atom|[Bb]eber|[Bb]exiga|[Bb]rilho|[Bb]ucho|[Cc]afé|[Cc]arbono|[Cc]élebre|[Cc]erto|[Cc]éu|[Cc]humbo|[Cc]imento|[Cc]intilador|[Cc]irculante|[Cc]lorofórmio|[Cc]om|[Cc]ombustível|[Cc]omo|[Cc]omposto|[Cc]ontêm|[Cc]onteúdo|[Cc]ontribuinte|[Cc]or-de-rosa|[Cc]orretivo|[Cc]ristal|[Dd]aquele|[Dd]e|[Dd]enominado|[Dd]erramava|[Dd]esarranjo|[Dd]espenhadeiro|[Dd]esse|[Dd]este|[Dd]etergente|[Dd]iamante|[Dd]ireito|[Dd]o|[Ee]|[Éé]|[Ee]feito|[Ee]fluente|[Ee]m|[Ee]mpréstimo|[Ee]ndividamento|[Ee]ndosperma|[Ee]nxofre|[Ee]spelho|[Ee]sse|[Ee]stado|[Ee]stados|[Ee]ste|[Ee]sturrinho|[Ee]tano|[Ee]xportador|[Ee]xterno|[Ff]ato|[Ff]aturamento|[Ff]icar|[Ff]io|[Ff]luxo|[Gg]anho|[Gg]ás|[Gg]iz|[Gg]licerol|[Gg]licol|[Hh]élio|[Hh]idrocarboneto|[Hh]idrogênio|[íí]mã|[Ii]nvestimento|[Ii]ogurte|[Jj]uro|[Ll]ençol|[Ll]infa|[Ll]itro|[Ll]ucro|[Mm]ais|[Mm]aterial|[Mm]eio|[Mm]enos|[Mm]icrocosmos|[Mm]iraculoso|[Mm]onstro|[Mm]uito|[Nn]éctar|[Nn]enhum|[Nn]esse|[Nn]itrogênio|[Nn]o|[Nn]úcleo|[Nn]um|[Oo]|[Oo]bjetivo|[Óó]leo|OMO|[Óó]pio|[Oo]rdenado|[Oo]u|[Oo]uro|[Oo]utro|[Oo]val|[Oo]xigênio|[Pp]ara|[Pp]arece|[Pp]atrimônio|[Pp]elo|[Pp]eso|[Pp]or|[Pp]recioso|[Pp]reciso|[Pp]reço|[Pp]rejuízo|[Pp]rêmio|[Pp]resente|[Pp]roduto|[Pp]ropanona|[Pp]us|[Qq]ualquer|[Qq]ue|[Rr]eal|[Rr]egisto|[Rr]endimento|[Rr]esgate|[Rr]esultado|[Rr]etorno|[Rr]ubi|[Ss]abão|[Ss]abonete|[Ss]alário|[Ss]aldo|[Ss]empre|[Ss]er|[Ss]eu|[Ss]ódio|[Ss]olda|[Ss]onho|[Ss]ubsídio|[Tt]al|[Tt]alco|[Tt]ambém|[Tt]ão|[Tt]em|[Tt]eto|[Tt]inta|[Tt]ítulo|[Tt]olueno|[Tt]orna-se|[Tt]otal|[Tt]ronco|[Tt]udo|[Uu]m|[Uu]rina|[Vv]alor|[Vv]encimento|[Vv]eneno|[Vv]enenoso|[Vv]inagre) liquido\\b")

# Identificacao de liquido sem acento
vliquida = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[Aa]|[Àà]|[áá]gua|[Aa]lma|[Aa]presenta-se|[Bb]ala|[Cc]aptação|[Cc]era|[Cc]hama|[Cc]ocaína|[Cc]oisa|[Cc]omo|[Cc]ompletamente|[Cc]onsistência|[Cc]ontribuição|[Cc]orrente|[Cc]riação|[Cc]romatografia|[Dd]a|[Dd]e|[Dd]escarga|[Dd]ieta|[Dd]ívida|[Dd]orsal|[Ee]|[Éé]|[Ee]nergia|[Ee]ntrada|[Ee]xtensão|[Ee]xterna|[Ff]aixa|[Ff]ase|[Ff]orma|[Ff]rações|[Gg]eralmente|[Gg]lucose|[Hh]erança|[Ii]nterna|[Jj]oia|[Ll]âmina|[Ll]igação|[Mm]ais|[Mm]assa|[Mm]ontanha|[Mm]ulta|[Nn]asceu|[Nn]em|[Nn]uma|[Nn]uvem|[Oo]peracional|[Oo]u|[Pp]arecia|[Pp]arte|[Pp]erda|[Pp]lanície|[Pp]oeira|[Pp]olpa|[Pp]rata|[Pp]rodutividade|[Pp]rópria|[Pp]rosa|[Qq]uantidade|[Qq]uase|[Qq]ue|[Rr]eceita|[Rr]ede|[Rr]efulgência|[Rr]enda|[Rr]entabilidade|[Rr]ocha|[Ss]ituação|[Ss]olução|[Ss]oma|[Ss]ubstância|[Ss]uperfície|[Ss]uspensão|[Tt]axa|[Tt]erra|[Tt]omador|[Tt]ributária|[Uu]ma) liquida\\b")

# Identificacao de liquido sem acento
vmalicia = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[Aa]|[Aa]legre|[Aa]lguma|[Aa]quela|[Aa]visada|[Cc]erta|[Cc]om|[Cc]uja|[Dd]a|[Dd]e|[Dd]iscreta|[Dd]uma|[Ee]|[Éé]|[Ee]m|[Ee]nvolve|[Ee]ssa|[Ee]sta|[Ee]terna|[Ff]ina|[Gg]raciosa|[Hh]á|[Ii]nocente|[Ii]nofensiva|[Ll]igeira|[Mm]ais|[Mm]enos|[Mm]inha|[Nn]a|[Nn]enhuma|[Nn]uma|[Oo]u|[Oo]utra|[Pp]ela|[Pp]etulante|[Pp]or|[Pp]ôs|[Pp]rofunda|[Pp]ura|[Qq]ue|[Rr]isonha|[Ss]anta|[Ss]em|[Ss]erá|[Ss]ua|[Tt]anta|[Tt]em|[Tt]erna|[Tt]oda|[Tt]ua|[Uu]ma|[Vv]ê|[Vv]eem|[Vv]i|[Vv]ia|[Vv]iu) malicia\\b")

# Identificacao de matricula sem acento
vmatricula = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]bra|[Aa]bre|[Aa]brir|[Aa]briu|[Cc]ada|[Cc]arro|[Cc]om|[Cc]onhecida|[Cc]uja|[Dd]a|[Dd]e|[Ee]|[Ee]ssa|[Ee]sta|[Ff]echa|[Ff]echou|[Mm]inha|[Nn]a|[Nn]ova|[Pp]ede|[Pp]ediu|[Pp]ela|[Pp]rimeira|[Rr]equereu|[Ss]egunda|[Ss]em|[Ss]ua|[Uu]ma) matricula\\b")

# Identificacao de participe sem acento
vparticipe = re.compile("(?i)\\b([Aa]ção|[Aa]o|[Aa]ssassin[oa] e|[Aa]utor e|[Cc]oautor e|[Cc]om|[Cc]omo|[Cc]omo sujeito|[Dd]a|[Dd]e|[Dd]o|[Éé]|[Ee]ntre|[Ff]oi|[Hh]aja mais de um|[Hh]ouvesse mais de um|[Oo]|[Pp]ara|[Pp]or|[Pp]ossível|[Ss]eja|[Ss]er|[Ss]erá|[Ss]oldado|[Tt]estemunha e|[Vv]ítima ou) participe\\b")

# Identificacao de participes sem acento
vparticips = re.compile("(?i)\\b([Aa]ções|[Aa]os|[Aa]s|[Aa]utor e|[Aa]utores e|[Cc]oautor e|[Cc]oautores e|[Cc]om|[Cc]omo|[Cc]omo sujeitos|[Dd]as|[Dd]e|[Dd]os|[Ee]ntre|[Nn]ovos|[Oo]s|[Pp]ara|[Pp]or|[Pp]ossíveis|[Ss]ão|[Ss]ejam|[Ss]ejam|[Ss]erão|[Ss]erem|[Ss]oldados) participes\\b")

# Identificacao de pericia sem acento
vpericia = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]dmirável|[Aa]rrojada|[Aa]ssombrosa|[Cc]om|[Cc]oncentram|[Cc]uja|[Dd]a|[Dd]e|[Dd]iz|[Ee]|[Éé]|[Ee]stranha|[Ee]xige|[Ee]xigiu|[Ff]ator|[Ff]alsa|[Ff]azer|[Ff]eito|[Ff]izeram|[Gg]rande|[Hh]abitual|[Hh]aver|[Ii]gual|[Ii]nsuficiente|[Mm]uita|[Nn]a|[Oo]u|[Pp]ara|[Pp]ela|[Pp]or|[Pp]rimeira|[Pp]rópria|[Qq]uarta|[Qq]ue|[Rr]equer|[Rr]equerer|[Rr]equeria|[Rr]equisitar|[Ss]em|[Ss]ua|[Ss]uma|[Ss]urpreendente|[Tt]al|[Tt]anta|[Úú]ltima|[Uu]ma|[Vv]erdadeira) pericia\\b")

# Identificacao de prestigio sem acento
vprestigio = re.compile("(?i)\\b([Aa]balado|[Aa]dquirir|[Aa]dquiriu|[Aa]lcançado|[Aa]lgum|[Aa]mericana|[Aa]ntigo|[Aa]o|[Aa]penas|[Aa]rranjar|[Aa]ssim|[Aa]ugusto|[Bb]astante|[Bb]elo|[Bb]rilhante|[Cc]atalítico|[Cc]erto|[Cc]olhe|[Cc]om|[Cc]onferindo|[Cc]onquista|[Cc]onquistar|[Cc]onseguiram|[Cc]onsiderável|[Cc]ujo|[Dd]á|[Dd]ando-lhe|[Dd]ar|[Dd]e|[Dd]ê|[Dd]eem|[Dd]efinitivo|[Dd]esmedido|[Dd]esse|[Dd]iabólico|[Dd]o|[Ee]|[Éé]|[Ee]les|[Ee]levado|[Ee]m|[Ee]norme|[Ee]sse|[Ee]u|[Ff]ascinante|[Ff]raco|[Ff]unesto|[Gg]anha|[Gg]anhar|[Gg]anhou|[Gg]enial|[Gg]rande|[Ii]menso|[Ii]mpressionante|[Ii]ncontestado|[Ii]nédito|[Ii]nsubstituível|[Jj]usto|[Ll]arguíssimo|[Mm]aior|[Mm]ais|[Mm]edíocre|[Mm]enor|[Mm]esmo|[Mm]eu|[Mm]uito|[Nn]em|[Nn]enhum|[Nn]esse|[Nn]o|[Nn]ovo|[Oo]|[Oo]bter|[Pp]ara|[Pp]articular|[Pp]elo|[Pp]erde|[Pp]erdendo|[Pp]erder|[Pp]erdeu|[Pp]erdia|[Pp]erdido|[Pp]or|[Pp]recioso|[Pp]rocurando|[Pp]róprio|[Qq]ue|[Rr]eal|[Rr]ecuperar|[Rr]égio|[Ss]eco|[Ss]em|[Ss]entia|[Ss]eu|[Ss]ignifica|[Ss]imples|[Ss]ingular|[Ss]inistro|[Ss]omente|[Tt]al|[Tt]anto|[Tt]em|[Tt]êm|[Tt]enho|[Tt]er|[Tt]eria|[Tt]errível|[Tt]eu|[Tt]inha|[Tt]inham|[Tt]iveram|[Uu]m|[Uu]ntado|[Vv]ago|[Vv]aleram|[Vv]enerando|[Vv]ulgar) prestigio\\b")

# Identificacao de retifica sem acento
vretifica = re.compile("(?i)\\b([Aa]|[Àà]|[Cc]om|[Cc]uja|[Dd]a|[Dd]e|[Éé]|[Ee]xige|[Ee]xigiu|[Ff]alsa|[Ff]azer|[Ff]eito|[Ff]ez|[Ff]izeram|[Gg]rande|[Hh]aver|[Mm]uita|[Nn]a|[Oo]u|[Pp]ara|[Pp]ela|[Pp]or|[Pp]rimeira|[Pp]rópria|[Qq]uarta|[Qq]ue|[Rr]equer|[Rr]equisitar|[Ss]em|[Ss]ua|[Tt]al|[Tt]anta|[Úú]ltima|[Uu]ma|[Vv]erdadeira) retifica\\b")

# Identificacao de solicito sem acento
vsolicito = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[Aa]diantou|[Aa]guardava|[Aa]o|[Bb]uscar|[Cc]aixeiro|[Cc]omo|[Cc]riado|[Ee]|[Éé]|[Ee]le|[Ee]sponjava|[Ee]studou|[Gg]esto|[Gg]uarda|[Ll]embrava|[Mm]ais|[Mm]esmo|[Mm]eu|[Mm]ostrava|[Mm]uito|[Oo]|[Oo]bservando|[Pp]astor|[Qq]ue|[Ss]empre|[Ss]eu|[Tt]ão|[Tt]odo|[Uu]m|[Vv]ela) solicito\\b")

# Identificacao de vicio sem acento
vvicio = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[Aa]|[Aa]char|[Aa]che|[Aa]chou|[Aa]legre|[Aa]lgum|[Aa]o|[Aa]quele|[Bb]elo|[Cc]omo|[Cc]ujo|[Dd]aquele|[Dd]e|[Dd]esse|[Dd]este|[Dd]o|[Dd]um|[Ee]|[Éé]|[Ee]l|[Ee]m|[Ee]ra|[Ee]sse|[Ee]ste|[Ee]xiste|[Ff]eio|[Gg]rande|[Hh]á|[Hh]aja|[Ii]rresistível|[Mm]aldito|[Mm]au|[Mm]áximo|[Mm]enor|[Mm]esmo|[Mm]eu|[Mm]uito|[Nn]aquele|[Nn]enhum|[Nn]o|[Nn]oturno|[Oo]|[Oo]u|[Oo]utro|[Pp]arece|[Pp]elo|[Pp]éssimo|[Pp]ior|[Pp]leno|[Pp]or|[Pp]róprio|[Qq]uanto|[Qq]ue|[Rr]essudando|[Ss]eja|[Ss]em|[Ss]er|[Ss]eu|[Ss]ó|[Ss]ofre|[Ss]oturno|[Ss]upôs|[Tt]al|[Tt]em|[Tt]eu|[Tt]ornando|[Tt]ornou|[Uu]m|[Úú]nico|[Vv]elho|[Vv]ira|[Vv]irando|[Vv]irou) vicio\\b")

# Identificacao de vinculo sem acento
vvinculo = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[Aa]lgum|[Aa]o|[Aa]pertado|[Aa]quele|[Cc]om|[Cc]ujo|[Dd]e|[Dd]esse|[Dd]este|[Dd]iferente|[Dd]iscute|[Dd]o|[Dd]oce|[Dd]uplo|[Ee]m|[Éé]m|[Ee]sse|[Ee]stabelece|[Ee]stabelecer|[Ee]stabeleceram|[Ee]stabeleceu|[Ee]ste|[Ee]xiste|[Ff]orte|[Gg]anhou|[Gg]era|[Gg]erasse|[Gg]ere|[Mm]ais|[Mm]esmo|[Mm]eu|[Mm]uito|[Nn]ega|[Nn]enhum|[Oo]|[Oo]utro|[Pp]elo|[Pp]or|[Pp]ossuem|[Pp]ossui|[Pp]ossuia|[Pp]rimeiro|[Pp]ouco|[Qq]ualquer|[Ss]em|[Ss]eu|[Tt]al|[Tt]ampouco|[Tt]êm|[Tt]endo|[Tt]enha|[Tt]enho|[Tt]inha|[Tt]ivesse|[Úú]ltimo|[Uu]m|[Úú]nico|[Vv]ital) vinculo\\b")

# Identificacao de vitima sem acento
vvitima = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[Aa]|[ÀàÀà]|[Aa]caba|[Aa]cabasse|[Aa]legadamente|[Aa]lguma|[Aa]luna|[Aa]nos|[Aa]ntiga|[Aa]parente|[Aa]parentemente|[Cc]ada|[Cc]air|[Cc]aiu|[Cc]ertamente|[Cc]om|[Cc]omo|[Cc]onsidera|[Cc]onsiderado|[Cc]onsiderar|[Cc]onsiderava|[Cc]riatura|[Cc]uja|[Dd]a|[Dd]e|[Dd]ela|[Dd]ele|[Dd]epois|[Dd]esapareceu|[Dd]esgraçada|[Dd]esignada|[Dd]ivina|[Dd]iz|[Dd]izes|[Dd]uma|[Ee]|[Éé]|[Ee]la|[Ee]le|[Ee]m|[Ee]ntre|[Ee]ntregava|[Ee]ra|[Éé]s|[Ee]scolhida|[Ee]scravidão|[Ee]ssa|[Ee]sta|[Ee]streada|[Ee]studante|[Ee]u|[Ff]ácil|[Ff]acilmente|[Ff]aleceu|[Ff]atalmente|[Ff]az|[Ff]azer|[Ff]oi|[Ff]or|[Ff]ora|[Ff]oram|[Ff]osse|[Ff]osses|[Ff]oste|[Ff]ui|[Gg]rande|[Hh]á|[Hh]ospitalizado|[Ii]mbele|[Ii]ndefesa|[Ii]nevitavelmente|[Ii]nfeliz|[Ii]nfelizmente|[Ii]ngênua|[Ii]nocente|[Ll]ogo|[Mm]aior|[Mm]ais|[Mm]ansa|[Mm]inha|[Mm]ísera|[Mm]orre|[Mm]orrer|[Mm]orreu|[Mm]orrido|[Mm]ostra|[Mm]ulher|[Mm]usical|[Nn]a|[Nn]em|[Nn]enhuma|[Nn]obre|[Nn]oite|[Nn]ova|[Nn]ovamente|[Nn]uma|[Oo]itava|[Oo]ntem|[Oo]u|[Oo]utra|[Pp]adre|[Pp]ara|[Pp]arecia|[Pp]ela|[Pp]ereceu|[Pp]erpétua|[Pp]ersonagem|[Pp]obre|[Pp]opulação|[Pp]or|[Pp]raga|[Pp]rimeira|[Pp]rincipal|[Pp]rópria|[Pp]rovavelmente|[Pp]róxima|[Qq]ualquer|[Qq]uarta|[Qq]ue|[Qq]uerida|[Rr]ecente|[Ss]e|[Ss]egunda|[Ss]eja|[Ss]em|[Ss]empre|[Ss]endo|[Ss]ente|[Ss]entir|[Ss]entiu|[Ss]er|[Ss]erá|[Ss]ereis|[Ss]eria|[Ss]ido|[Ss]ó|[Ss]ocial|[Ss]omos|[Ss]ou|[Ss]ua|[Ss]ucumbido|[Ss]uposta|[Tt]alvez|[Tt]ambém|[Tt]erceira|[Tt]inham|[Tt]orna|[Tt]ornar|[Tt]ornaria|[Tt]ornou|[Tt]ua|[Úú]ltima|[Uu]m|[Uu]ma|[Úú]nica|[Vv]ez|[Vv]ia|[Vv]isse|[Vv]oluntária) vitima\\b")

# Identificacao de noticia sem acento
vnoticia = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]cabrunhadora|[Aa]gradável|[Aa]inda|[Aa]lguma|[Aa]pareceu|[Aa]quela|[Àà]quela|[Aa]traente|[Bb]ela|[Bb]oa|[Bb]reve|[Cc]ada|[Cc]erta|[Cc]ertamente|[Cc]hamado|[Cc]hegasse|[Cc]hegou|[Cc]ircunstanciada|[Cc]om|[Cc]omprida|[Cc]onforme|[Cc]onsoladora|[Cc]onsta|[Cc]uja|[Cc]uriosa|[Dd]a|[Dd]á|[Dd]ado|[Dd]ando|[Dd]ão|[Dd]aquela|[Dd]ar|[Dd]arão|[Dd]arem|[Dd]aremos|[Dd]armos|[Dd]ava|[Dd]avam|[Dd]e|[Dd]ê|[Dd]ei|[Dd]eixasse|[Dd]ele|[Dd]er|[Dd]era|[Dd]eram|[Dd]esagradável|[Dd]esastrosa|[Dd]esgostosa|[Dd]esmentiu|[Dd]esoladora|[Dd]essa|[Dd]esse|[Dd]essem|[Dd]esta|[Dd]etalhada|[Dd]eu|[Dd]isseram|[Dd]olorosa|[Dd]ou|[Dd]uma|[Ee]|[Éé]|[Ee]le|[Ee]ncontra|[Ee]ra|[Ee]ram|[Ee]ssa|[Ee]sta|[Ee]stupenda|[Ee]xata|[Ee]xistindo|[Ee]xtraordinária|[Ff]alsa|[Ff]antasiosa|[Ff]atal|[Ff]azem|[Ff]eliz|[Ff]oi|[Ff]ora|[Ff]oram|[Ff]ossem|[Ff]raca|[Ff]unesta|[Ff]utura|[Gg]rande|[Gg]rata|[Gg]rave|[Hh]á|[Hh]aja|[Hh]aver|[Hh]avia|[Hh]orrível|[Hh]ouve|[Hh]ouvera|[Hh]ouvesse|[Ii]a|[Ii]mpacta|[Ii]mportante|[Ii]nesperada|[Ii]ntempestiva|[Jj]ornais|[Ll]arga|[Ll]eve|[Ll]igeira|[Ll]isonjeira|[Ll]ogo|[Ll]úgubre|[Mm]á|[Mm]ais|[Mm]andaria|[Mm]ande|[Mm]enor|[Mm]esma|[Mm]inha|[Mm]ínima|[Mm]isteriosa|[Nn]a|[Nn]aquela|[Nn]em|[Nn]enhuma|[Nn]esta|[Nn]uma|[Oo]ntem|[Óó]tima|[Oo]u|[Oo]utra|[Oo]utro|[Oo]uvidos|[Pp]avorosa|[Pp]ediu|[Pp]ela|[Pp]equena|[Pp]esco|[Pp]éssima|[Pp]ior|[Pp]obre|[Pp]or|[Pp]orém|[Pp]ortentosa|[Pp]ossui|[Pp]ouca|[Pp]ouco|[Pp]révia|[Pp]rimeira|[Pp]rincipal|[Pp]ronta|[Pp]rópria|[Pp]róprios|[Pp]róspera|[Qq]ual|[Qq]ualquer|[Qq]uanta|[Qq]ue|[Qq]uerer|[Rr]ara|[Rr]ecebemos|[Rr]eceber|[Rr]ecebera|[Rr]eceberam|[Rr]ecebeste|[Rr]eferida|[Ss]aber|[Ss]abia|[Ss]ão|[Ss]eguinte|[Ss]egunda|[Ss]egundo|[Ss]ejam|[Ss]em|[Ss]emelhante|[Ss]er|[Ss]eriam|[Ss]ido|[Ss]imples|[Ss]ó|[Ss]obretudo|[Ss]omente|[Ss]oube|[Ss]ua|[Ss]urpreendente|[Tt]al|[Tt]anta|[Tt]em|[Tt]êm|[Tt]emos|[Tt]endenciosa|[Tt]endo|[Tt]enhamos|[Tt]enho|[Tt]ens|[Tt]er|[Tt]eriam|[Tt]errível|[Tt]eve|[Tt]ido|[Tt]inha|[Tt]inham|[Tt]ínhamos|[Tt]ive|[Tt]ivemos|[Tt]iver|[Tt]ivera|[Tt]iveram|[Tt]ivesse|[Tt]iveste|[Tt]ornarem|[Tt]ornou|[Tt]rágica|[Tt]raz|[Tt]razer|[Tt]razido|[Tt]riste|[Tt]rouxe|[Úú]ltima|[Uu]ma|[Úú]nica|[Vv]er|[Vv]ez|[Vv]inha|[Vv]irar|[Vv]irou|[Tt]erei|[Tt]raga|[Tt]ragam|[Ee]spalhafatosa|[Ss]ensacional|[Mm]aravilhosa|[Ss]obre) noticia\\b")

# Identificacao de agencia sem acento
vagencia = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[Aa]|[Àà]|[Aa]lguma|[Aa]penas|[Aa]quela|[Aa]s|[Cc]ada|[Cc]om|[Cc]ria|[Cc]uja|[Dd]a|[Dd]aquela|[Dd]e|[Dd]essa|[Dd]esta|[Dd]eterminada|[Dd]uma|[Ee]|[Éé]|[Ee]ficiente|[Ee]m|[Ee]ntre|[Ee]ntrevista|[Ee]ssa|[Ee]sta|[Ee]xistindo|[Ff]utura|[Gg]igantesca|[Gg]rande|[Hh]aja|[Ii]mportante|[Ii]naugura|[Ii]nauguramos|[Ii]naugurar|[Ii]nauguraremos|[Ii]naugurou|[Jj]ovem|[Mm]aior|[Mm]esma|[Mm]inha|[Mm]oderna|[Mm]odesta|[Mm]ontam|[Nn]a|[Nn]enhuma|[Nn]esta|[Nn]ossa|[Nn]otória|[Nn]ova|[Nn]uma|[Oo]u|[Oo]utra|[Pp]ara|[Pp]ela|[Pp]equena|[Pp]ermanente|[Pp]obre|[Pp]or|[Pp]retendida|[Pp]revista|[Pp]rimeira|[Pp]rincipal|[Pp]roeminente|[Pp]rópria|[Pp]róxima|[Qq]ualquer|[Rr]ecente|[Ss]ó|[Ss]obre|[Ss]ofisticada|[Ss]omente|[Ss]oturna|[Ss]ua|[Tt]em|[Tt]erá|[Tt]erão|[Tt]ipo|[Úú]ltima|[Uu]ma|[Úú]nica|[Vv]iagem) agencia\\b")
vagenciap = re.compile("(?i)\\bagencia ([Àà]|[Aa]berta|[Aa]crescenta|[Áá]gua|[Aa]lemã|[Aa]mericana|[Aa]natólia|[Aa]nsa|[Aa]ntara|[Aa]ntidrogas|[Aa]tômica|[Aa]ustríaca|[Aa]utorizada|[Aa]valiar|[Bb]ancária|[Bb]ancárias|[Bb]oca|[Bb]rasil|[Bb]rasileira|[Bb]ritânica|[Cc]ausado|[Cc]entral|[Cc]entro|[Cc]omprar|[Cc]ontinua|[Cc]orrespondente|[Cc]redenciada|[Cc]riou|[Dd]a|[Dd]as|[Dd]e|[Dd]epositária|[Dd]esigner|[Dd]essa|[Dd]esse|[Dd]everá|[Dd]inheiro|[Dd]isse|[Dd]istribuiu|[Dd]iz|[Dd]o|[Dd]os|[Ee]|[Éé]|[Ee]fe|[Ee]m|[Ee]scala|[Ee]spacial|[Ee]spanhola|[Ee]special|[Ee]specializada|[Ee]specífica|[Ee]stá|[Ee]stado|[Ee]stadual|[Ee]stão|[Ee]statal|[Ee]studa|[Ee]uropeia|[Ee]xistente|[Ff]echadas|[Ff]ederal|[Ff]icar|[Ff]inanceira|[Ff]oi|[Ff]olha|[Ff]otográfica|[Ff]otográficas|[Ff]rancesa|[Ff]unerária|[Gg]anhou|[Gg]aúcha|[Gg]overnamental|[Gg]uardaria|[Hh]á|[Hh]umanitária|[Ii]mobiliária|[Ii]ncendiária|[Ii]ndependência|[Ii]ndica|[Ii]nformativa|[Ii]nglesa|[Ii]nstalada|[Ii]nternacionais|[Ii]nternacional|[Ii]raniana|[Jj]á|[Jj]udaica|[Ll]eva|[Ll]isboeta|[Ll]ocal|[Ll]otérica|[Ll]usa|[Mm]agno|[Mm]as|[Mm]atrimonial|[Mm]atriz|[Mm]enino|[Mm]eridional|[Mm]eteorológica|[Mm]undial|[Nn]a|[Nn]acional|[Nn]ada|[Nn]ão|[Nn]o|[Nn]orte-americana|[Nn]os|[Nn]oticiosa|[Nn]ova|[Oo]|[Oo]ferece|[Oo]ficial|[Oo]nde|[Oo]u|[Pp]aga|[Pp]ara|[Pp]aranaense|[Pp]arece|[Pp]arisiense|[Pp]aulista|[Pp]ermanente|[Pp]orque|[Pp]ortuguesa|[Pp]revê|[Pp]roponente|[Pp]ublicitária|[Qq]ue|[Qq]uer|[Rr]ecebeu|[Rr]ecifense|[Rr]eferiu|[Rr]egional|[Rr]eguladora|[Rr]egulatória|[Rr]esponsável|[Rr]odoviária|[Rr]ussa|[Ss]acada|[Ss]afras|[Ss]antana|[Ss]ão|[Ss]aúde|[Ss]e|[Ss]obre|[Ss]omará|[Ss]uperavitária|[Tt]ambém|[Tt]anto|[Tt]elegráfica|[Tt]em|[Tt]inha|[Tt]raria|[Tt]ucana|[Uu]niversitária|[Uu]sa|[Vv]ai|[Vv]enha|[Vv]inculada|[Vv]oltada)\\b")

# Identificacao de buque sem acento
vbuque = re.compile("(?i)\\b([Bb]uqu(e|es)) de ([Ff]lores|[Nn]oiva|[Nn]oivas|[Vv]ioleta|[Vv]ioletas|[Pp]érolas|[Pp]érola|[Rr]osas|[Cc]ravo|[Mm]inirrosas|[Mm]argaridas|[Mm]argarida|[Ss]anto|[Bb]roches|[Rr]oletes)\\b")

# Identificacao de evidencia sem acento
vevidencia = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]bsurda|[Aa]chamos|[Aa]dquiriu|[Aa]lguma|[Aa]parente|[Aa]presenta|[Aa]presentam|[Cc]om|[Cc]omo|[Cc]rua|[Cc]uja|[Dd]a|[Dd]aquela|[Dd]e|[Dd]eclarada|[Dd]emonstrativa|[Dd]era|[Dd]esta|[Ee]|[Éé]|[Ee]m|[Ee]spantosa|[Ee]ssa|[Ee]sta|[Ee]xiste|[Ff]lagrante|[Ff]ornecendo|[Ff]orte|[Ff]ulgurante|[Gg]anhando|[Gg]anharam|[Gg]rande|[Hh]á|[Íí]ntima|[Ii]rrecusável|[Jj]usticeira|[Ll]ampejante|[Mm]aior|[Mm]ais|[Mm]enos|[Mm]inha|[Mm]uita|[Nn]a|[Nn]ão|[Nn]enhuma|[Nn]essa|[Nn]ova|[Nn]uma|[Oo]utra|[Pp]ela|[Pp]lena|[Pp]orque|[Pp]reciosa|[Pp]rimeira|[Pp]rincipal|[Pp]rogressiva|[Pp]rópria|[Qq]ualquer|[Rr]esplandecente|[Ss]e|[Ss]emelhante|[Ss]er|[Ss]imples|[Ss]ólida|[Ss]ua|[Ss]úbita|[Tt]al|[Tt]amanha|[Tt]anta|[Tt]oda|[Tt]odo|[Uu]ma) evidencia\\b")

# Identificacao de gerencia sem acento
vgerencia = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[Aa]|[Àà]|[Aa]lguma|[Aa]lta|[Aa]penas|[Aa]quela|[Aa]s|[Aa]tual|[Cc]ada|[Cc]om|[Cc]ria|[Cc]uja|[Dd]a|[Dd]aquela|[Dd]e|[Dd]essa|[Dd]esta|[Dd]eterminada|[Dd]uma|[Ee]|[Éé]|[Ee]ficiente|[Ee]m|[Ee]ntre|[Ee]ssa|[Ee]sta|[Ee]xistindo|[Ff]utura|[Gg]igantesca|[Gg]rande|[Hh]aja|[Ii]mportante|[Mm]aior|[Mm]esma|[Mm]inha|[Mm]oderna|[Mm]odesta|[Mm]ontam|[Nn]a|[Nn]enhuma|[Nn]essa|[Nn]esta|[Nn]ossa|[Nn]otória|[Nn]ova|[Nn]uma|[Oo]u|[Oo]utra|[Pp]ara|[Pp]ela|[Pp]equena|[Pp]ermanente|[Pp]obre|[Pp]or|[Pp]retendida|[Pp]rimeira|[Pp]rincipal|[Pp]roeminente|[Pp]rópria|[Pp]róxima|[Qq]ualquer|[Rr]ecente|[Ss]obre|[Ss]ofisticada|[Ss]ua|[Tt]em|[Tt]erá|[Tt]erão|[Tt]ipo|[Úú]ltima|[Uu]ma|[Úú]nica) gerencia\\b")

# Identificacao de negligencia sem acento
vnegligencia = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]lega|[Aa]legada|[Aa]legaram|[Aa]legou|[Aa]pura|[Cc]om|[Cc]omo|[Cc]ontra|[Dd]a|[Dd]aquela|[Dd]e|[Dd]izer|[Ee]|[Éé]|[Ee]m|[Ee]ssa|[Ee]ventual|[Ff]oi|[Hh]á|[Hh]avia|[Hh]ouve|[Mm]enor|[Mm]era|[Mm]esma|[Mm]inha|[Nn]a|[Oo]u|[Pp]ela|[Pp]erfeita|[Pp]or|[Pp]ropositada|[Pp]ura|[Qq]ue|[Ss]em|[Ss]ua|[Tt]oda|[Tt]otal|[Tt]remenda|[Uu]ma) negligencia\\b")

# Identificacao de providencia sem acento
vprovidencia = re.compile("(?i)\\b([Aa]|[Àà]|[Aa]bençoada|[Aa]certada|[Aa]dote|[Aa]guarda|[Aa]guardando|[Aa]lguma|[Aa]s|[Bb]ela|[Cc]hamamos|[Cc]hamava|[Cc]omo|[Cc]ruel|[Cc]uja|[Dd]a|[Dd]á|[Dd]aquela|[Dd]ar|[Dd]e|[Dd]esta|[Dd]evida|[Dd]ivina|[Dd]ona|[Dd]uma|[Ee]|[Éé]|[Ee]ntre|[Ee]spera|[Ee]sperando|[Ee]ssa|[Ee]sta|[Ee]terna|[Gg]rande|[Ii]mportante|[Mm]inha|[Nn]a|[Nn]enhuma|[Nn]ossa|[Nn]ova|[Oo]|[Oo]u|[Oo]utra|[Pp]articular|[Pp]ela|[Pp]rimeira|[Pp]rincipal|[Pp]róxima|[Qq]ualquer|[Ss]egunda|[Ss]ua|[Tt]al|[Tt]erceira|[Tt]oda|[Tt]omar|[Tt]omou|[Tt]ua|[Uu]ma|[Úú]nica|[Uu]rgente|[Vv]elha) providencia\\b")

# Identificacao de influencia sem acento
vinfluencia = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-o|-me|-lo|-te)|[Aa]|[Aa]|[Àà]|[Aa]bençoada|[Aa]centuada|[Aa]dmirável|[Aa]dquirido|[Aa]gitadora|[Aa]í|[Aa]lguma|[Aa]li|[Aa]lta|[Aa]lucinadora|[Aa]luguer|[Aa]mericano|[Aa]presenta|[Aa]presentando|[Aa]presente|[Aa]presentou|[Aa]quela|[Àà]quela|[Áá]rea|[Aa]té|[Aa]tuais|[Aa]utêntica|[Bb]astante|[Bb]em|[Bb]enéfica|[Bb]enévola|[Bb]enigna|[Bb]oa|[Bb]randa|[Bb]reve|[Cc]erta|[Cc]hamarei|[Cc]hamava|[Cc]lara|[Cc]om|[Cc]omo|[Cc]onquistassem|[Cc]onsiderável|[Cc]rescente|[Cc]uja|[Dd]a|[Dd]e|[Dd]ebate|[Dd]ecidida|[Dd]ecisiva|[Dd]eclarada|[Dd]emasiada|[Dd]emonstra|[Dd]enotando|[Dd]essa|[Dd]esta|[Dd]izer|[Dd]oce|[Dd]upla|[Dd]uplicada|[Dd]uradoura|[Ee]|[Éé]|[Ee]ficaz|[Ee]m|[Ee]norme|[Ee]ra|[Ee]ssa|[Ee]sta|[Ee]stranha|[Ee]ventual|[Ee]xcelente|[Ee]xcessiva|[Ee]xerce|[Ee]xercem|[Ee]xercendo|[Ee]xercer|[Ee]xercerá|[Ee]xerceram|[Ee]xerceu|[Ee]xercia|[Ee]xercido|[Ee]xtraordinária|[Ff]atal|[Ff]eliz|[Ff]irme|[Ff]orte|[Ff]osse|[Ff]ossem|[Ff]unda|[Ff]unesta|[Gg]anhar|[Gg]anhou|[Gg]eralmente|[Gg]igantesca|[Gg]rande|[Gg]rave|[Gg]rossa|[Hh]á|[Hh]aja|[Hh]avia|[Ii]mportante|[Ii]ncontestável|[Ii]ncrível|[Ii]ndeclinável|[Ii]nexplicável|[Ii]nsistente|[Ii]ntervenha|[Ii]rresistível|[Ii]sso|[Ll]arga|[Ll]egítima|[Ll]egitimamente|[Ll]eve|[Mm]á|[Mm]ágica|[Mm]aior|[Mm]ais|[Mm]aléfica|[Mm]aligna|[Mm]arcada|[Mm]arcante|[Mm]as|[Mm]enor|[Mm]enos|[Mm]esma|[Mm]esquinha|[Mm]im|[Mm]inha|[Mm]ínima|[Mm]isteriosa|[Mm]ística|[Mm]oeda|[Mm]uita|[Mm]utua|[Mm]útua|[Nn]a|[Nn]as|[Nn]atural|[Nn]efasta|[Nn]em|[Nn]enhuma|[Nn]esta|[Nn]isso|[Nn]ítida|[Nn]ossa|[Nn]otável|[Nn]otória|[Nn]ula|[Oo]bteve|[Óó]bvia|[Oo]pressiva|[Oo]u|[Oo]utra|[Pp]articular|[Pp]ela|[Pp]equena|[Pp]erde|[Pp]erdendo|[Pp]erder|[Pp]erderam|[Pp]erniciosa|[Pp]lena|[Pp]oderosa|[Pp]or|[Pp]ositiva|[Pp]ossível|[Pp]ouca|[Pp]rimeira|[Pp]rimitiva|[Pp]rincipal|[Pp]rofunda|[Pp]rópria|[Qq]uaisquer|[Qq]ual|[Qq]ualquer|[Qq]ue|[Rr]ápida|[Rr]eal|[Rr]ealidade|[Rr]ecebe|[Rr]ecebendo|[Rr]eceber|[Rr]eceberam|[Rr]ecebeu|[Rr]ecebia|[Rr]ecebiam|[Rr]ecíproca|[Rr]ecolhe|[Rr]eivindicar|[Rr]etirou|[Rr]evela|[Rr]evelam|[Rr]evelando|[Rr]omântica|[Ss]alutar|[Ss]ecreta|[Ss]em|[Ss]er|[Ss]éria|[Ss]ignificativa|[Ss]imular|[Ss]ob|[Ss]ofre|[Ss]ofrem|[Ss]ofremos|[Ss]ofrendo|[Ss]ofrer|[Ss]ofreram|[Ss]ofreu|[Ss]ofri|[Ss]ofrido|[Ss]olo|[Ss]ou|[Ss]ua|[Ss]uavíssima|[Ss]uperior|[Tt]al|[Tt]alvez|[Tt]amanha|[Tt]ambém|[Tt]anta|[Tt]em|[Tt]êm|[Tt]endo|[Tt]enha|[Tt]enham|[Tt]enho|[Tt]er|[Tt]erá|[Tt]erão|[Tt]erceira|[Tt]eria|[Tt]errível|[Tt]eve|[Tt]ido|[Tt]inha|[Tt]inham|[Tt]ive|[Tt]ivemos|[Tt]iveram|[Tt]otal|[Tt]remenda|[Tt]ua|[Uu]m|[Uu]ma|[Úú]nica|[Vv]asta|[Vv]endido|[Vv]erdadeira|[Vv]iagem) influencia\\b")

# Identificacao de referencia sem acento
vreferencia = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]inda|[Aa]lguma|[Aa]nos|[Aa]ntiga|[Aa]quela|[Aa]té|[Bb]reve|[Cc]ada|[Cc]ena|[Cc]epa|[Cc]ertames|[Cc]idade|[Cc]lara|[Cc]om|[Cc]omo|[Cc]onstará|[Cc]onstitui|[Cc]rua|[Cc]uja|[Dd]a|[Dd]e|[Dd]estacada|[Dd]ireta|[Ee]|[Éé]|[Ee]dição|[Ee]m|[Ee]mpresa|[Ee]ncontramos|[Ee]ncontravam|[Ee]ra|[Ee]special|[Ee]ssa|[Ee]sta|[Ee]stranha|[Ee]videncia|[Ff]aça|[Ff]açam|[Ff]açamos|[Ff]aço|[Ff]ará|[Ff]az|[Ff]az-se|[Ff]azem|[Ff]azendo|[Ff]azer|[Ff]azermos|[Ff]azia|[Ff]aziam|[Ff]ez|[Ff]izemos|[Ff]izeram|[Ff]oi|[Ff]orte|[Ff]ugitiva|[Gg]rande|[Hh]á|[Hh]avia|[Hh]ouve|[Ii]dentificando|[Ii]gualmente|[Ii]mportante|[Ii]nclui|[Ii]ncontornável|[Ll]eve|[Ll]igeira|[Mm]aior|[Mm]ais|[Mm]elhor|[Mm]enor|[Mm]erece|[Mm]erecendo|[Mm]esma|[Mm]inha|[Nn]a|[Nn]ão|[Nn]aquela|[Nn]enhuma|[Nn]ormalmente|[Nn]ova|[Nn]uma|[Oo]mitindo-se|[Oo]u|[Oo]utra|[Pp]ara|[Pp]ela|[Pp]equena|[Pp]eriódica|[Pp]or|[Pp]ossuindo|[Pp]rimeira|[Pp]rincipal|[Pp]rovidencial|[Pp]ura|[Qq]ualquer|[Ss]eguinte|[Ss]egunda|[Ss]em|[Ss]er|[Ss]imples|[Ss]obre|[Ss]ua|[Tt]al|[Tt]em|[Tt]êm|[Tt]enha|[Tt]ímida|[Tt]inha|[Tt]oda|[Tt]ornar|[Tt]ornaram|[Úú]ltima|[Uu]ma|[Úú]nica|[Uu]sei|[Vv]árias|[Vv]ejo|[Vv]erdadeira) referencia\\b")

# Identificacao de reverencia sem acento
vreverencia = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Cc]om|[Dd]a|[Dd]e|[Dd]evida|[Ee]|[Ee]m|[Ee]nvolve|[Ff]az|[Ff]azem|[Ff]ez|[Ff]izeram|[Gg]alante|[Gg]rande|[Ii]lustre|[Ii]nvocando|[Ii]rônica|[Mm]agna|[Mm]áxima|[Mm]enos|[Mm]esma|[Mm]ostre|[Nn]a|[Nn]egar|[Nn]ossa|[Nn]ova|[Nn]uma|[Pp]ensativa|[Pp]rofunda|[Pp]rovocar|[Rr]eligiosa|[Ss]e|[Ss]em|[Ss]ilenciosa|[Ss]ua|[Ss]ubmissa|[Tt]em|[Tt]erna|[Uu]ma|[Uu]niversal|[Vv]irtuosa|[Vv]ossa) reverencia\\b")

# Identificacao de sequencia sem acento
vsequencia = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]bra|[Aa]bram|[Aa]bre|[Aa]brir|[Aa]briram|[Aa]briu|[Aa]dia|[Aa]diou|[Aa]lguma|[Aa]lucinante|[Aa]ntecipa|[Aa]ntecipou|[Aa]tingir|[Aa]quela|[Aa]ssombrosa|[Bb]oa|[Bb]usca|[Bb]uscam|[Cc]ada|[Cc]erta|[Cc]om|[Cc]omo|[Cc]uja|[Cc]urta|[Dd]a|[Dd]ada|[Dd]ado|[Dd]ando|[Dd]ar|[Dd]e|[Dd]ê|[Dd]eram|[Dd]esignada|[Dd]essa|[Dd]esta|[Dd]eterminada|[Dd]eu|[Ee]|[Ee]m|[Ee]ncerra|[Ee]ncerrar|[Ee]ncerraram|[Ee]ncerre|[Ee]ncerrou|[Ee]norme|[Ee]ssa|[Ee]sta|[Ff]antástica|[Hh]avia|[Ll]onga|[Mm]ais|[Mm]ediante|[Mm]elhor|[Mm]esma|[Mm]ira|[Mm]iram|[Mm]ísera|[Nn]a|[Nn]enhuma|[Nn]essa|[Nn]esta|[Nn]ova|[Nn]uma|[Oo]ferece|[Oo]ferecem|[Oo]ferecer|[Oo]fereceu|[Oo]ra|[Óó]tima|[Oo]u|[Oo]utra|[Pp]ela|[Pp]or|[Pp]rimeira|[Pp]rópria|[Qq]ualquer|[Qq]uebra|[Qq]uebram|[Qq]uebrar|[Qq]uebraram|[Qq]uebrou|[Qq]uer|[Rr]ápida|[Ss]eguinte|[Ss]em|[Ss]imples|[Ss]obre|[Ss]ua|[Tt]al|[Tt]em|[Tt]êm|[Tt]erá|[Tt]erão|[Tt]eve|[Tt]iveram|[Tt]radicional|[Úú]ltima|[Uu]ma|[Uu]tilizar|[Vv]aloriza|[Vv]ira) sequencia\\b")

# Identificacao de vivencia sem acento
vvivencia = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]lguma|[Cc]asta|[Cc]om|[Cc]omo|[Dd]a|[Dd]aquela|[Dd]e|[Dd]essa|[Dd]upla|[Ee]|[Ee]m|[Ee]ssa|[Ee]sta|[Gg]rande|[Ii]ncipiente|[Ll]arga|[Ll]onga|[Mm]inha|[Mm]ínima|[Mm]uita|[Nn]a|[Nn]aquela|[Nn]enhuma|[Nn]essa|[Nn]ossa|[Nn]ova|[Oo]u|[Pp]ela|[Pp]ermanente|[Pp]or|[Pp]rópria|[Qq]ue|[Ss]ua|[Tt]em|[Tt]ua|[Uu]ma) vivencia\\b")

# Identificacao de anuncio sem acento
vanuncio = re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]nimador|[Aa]o|[Aa]ntigo|[Aa]quele|[Bb]elo|[Bb]om|[Bb]onito|[Cc]olecionar|[Cc]om|[Cc]onsumo|[Cc]rescente|[Cc]ujo|[Dd]aquele|[Dd]ar|[Dd]e|[Dd]este|[Dd]o|[Ee]|[Éé]|[Ee]m|[Ee]sse|[Ee]ste|[Ee]xcelente|[Ff]az|[Ff]azer|[Ff]ez|[Gg]rande|[Gg]ravar|[Ii]minente|[Ll]indo|[Ll]uz|[Mm]au|[Mm]edonho|[Mm]elhor|[Mm]eu|[Mm]i|[Nn]aquele|[Nn]este|[Nn]o|[Nn]ovo|[Nn]um|[Oo]|[Óó]timo|[Oo]utro|[Pp]ara|[Pp]elo|[Pp]equeno|[Pp]iedoso|[Pp]or|[Pp]ôr|[Pp]ôs|[Pp]rimeiro|[Pp]ublica|[Pp]ublicou|[Pp]us|[Qq]ualquer|[Qq]ue|[Ss]eguinte|[Ss]egundo|[Ss]em|[Ss]eria|[Ss]eu|[Ss]impático|[Ss]imples|[Tt]al|[Tt]errível|[Tt]eu|[Tt]riste|[Uu]m|[Vv]asto|[Vv]elho) anuncio\\b")

# Identificacao de duvida sem acento
vduvida= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]lguma|[Aa]quela|[Bb]oa|[Cc]erta|[Cc]omo|[Dd]a|[Dd]e|[Dd]eixa|[Dd]eixar|[Dd]essa|[Dd]esta|[Ee]|[Éé]|[Ee]m|[Ee]ssa|[Ee]sta|[Ff]az|[Gg]rande|[Hh]a|[Hh]á|[Hh]aja|[Hh]avendo|[Hh]aver|[Hh]averá|[Hh]avia|[Hh]ouve|[Hh]ouver|[Ll]evantar|[Ll]eve|[Ll]igeira|[Mm]ais|[Mm]enor|[Mm]esma|[Mm]inha|[Mm]ínima|[Mm]uita|[Nn]a|[Nn]aquela|[Nn]enhuma|[Nn]essa|[Nn]esta|[Nn]ossa|[Nn]uma|[Oo]correr|[Oo]ferecia|[Oo]u|[Oo]utra|[Pp]adecia|[Pp]ara|[Pp]ela|[Pp]õe|[Pp]onha|[Pp]onho|[Pp]or|[Pp]ôr|[Pp]orá|[Pp]orei|[Pp]ôs|[Pp]rimeira|[Pp]rincipal|[Pp]rópria|[Pp]unha|[Qq]ualquer|[Qq]ue|[Rr]esta|[Rr]estava|[Ss]em|[Ss]imples|[Ss]ofria|[Ss]ua|[Tt]em|[Tt]êm|[Tt]emos|[Tt]enha|[Tt]enham|[Tt]enhas|[Tt]enho|[Tt]er|[Tt]eria|[Tt]eve|[Tt]inha|[Tt]ive|[Tt]iver|[Tt]ivessem|[Tt]oda|[Uu]ma|[Úú]nica) duvida\\b")

# Identificacao de duvidas sem acento
vduvids= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Aa]dmite|[Aa]dmitia|[Aa]dmitindo|[Aa]gora|[Aa]lanceantes|[Aa]lguma|[Aa]lgumas|[Aa]limentar|[Aa]placava|[Aa]presentando|[Aa]quelas|[Aa]s|[Àà]s|[Aa]tiçavam|[Cc]ausam|[Cc]ausando|[Cc]ausar|[Cc]ertas|[Cc]erto|[Cc]olocam|[Cc]olocar|[Cc]om|[Cc]ompartilham|[Cc]ortar|[Cc]riar|[Cc]ultivar|[Dd]as|[Dd]e|[Dd]eixa|[Dd]eixado|[Dd]eixam|[Dd]eixando|[Dd]eixar|[Dd]eixara|[Dd]eixarei|[Dd]eixasse|[Dd]eixava|[Dd]eixavam|[Dd]eixou|[Dd]emais|[Dd]estas|[Dd]irimir|[Dd]iscute|[Dd]issipar|[Dd]iversas|[Dd]uas|[Ee]|[Ee]m|[Ee]ntre|[Ee]ra|[Ee]sclarecendo|[Ee]sclarecer|[Ee]spalhar|[Ee]spírito|[Ee]squecendo|[Ee]ssas|[Ee]stas|[Ee]ventuais|[Ee]vitar|[Ee]xistam|[Ee]xistem|[Ee]xistindo|[Ee]xistirem|[Ee]xplicar|[Ff]icam|[Ff]icaram|[Ff]icassem|[Ff]icavam|[Ff]iquem|[Ff]ortes|[Ff]requentes|[Ff]undadas|[Gg]erar|[Gg]randes|[Hh]á|[Hh]aja|[Hh]avendo|[Hh]aver|[Hh]averá|[Hh]averia|[Hh]avia|[Hh]ouve|[Hh]ouver|[Hh]ouvesse|[Ii]nfindas|[Ii]njustas|[Ii]nsinuando|[Ii]ntermináveis|[Ii]númeras|[Ii]sso|[Jj]á|[Ll]ançam|[Ll]ançar|[Ll]ancinantes|[Ll]ançou|[Ll]evanta|[Ll]evantadas|[Ll]evantam|[Ll]evantando|[Ll]evantar|[Ll]evantaram|[Ll]evantou|[Mm]aiores|[Mm]ais|[Mm]anifestou|[Mm]as|[Mm]enos|[Mm]esmas|[Mm]il|[Mm]inhas|[Mm]ostrando|[Mm]uitas|[Nn]as|[Nn]ossa|[Nn]ossas|[Nn]ovas|[Nn]ovo|[Nn]utria|[Nn]utrir|[Oo]ferece|[Oo]u|[Oo]utras|[Pp]airem|[Pp]ara|[Pp]are|[Pp]elas|[Pp]equenas|[Pp]ermanecem|[Pp]ermitem|[Pp]ermitiam|[Pp]ersistam|[Pp]ersistem|[Pp]erturbantes|[Pp]onho|[Pp]or|[Pp]ôr|[Pp]orque|[Pp]oucas|[Pp]rimeiras|[Pp]rincipais|[Pp]rincípio|[Pp]róprias|[Pp]rovocava|[Qq]uaisquer|[Qq]uantas|[Qq]ue|[Qq]ueria|[Qq]uero|[Rr]einam|[Rr]esolver|[Rr]estam|[Rr]estar|[Rr]estaram|[Rr]estarem|[Rr]estassem|[Rr]estavam|[Rr]estem|[Rr]esultaram|[Rr]evisor|[Ss]anar|[Ss]ão|[Ss]e|[Ss]em|[Ss]emelhantes|[Ss]érias|[Ss]imples|[Ss]ob|[Ss]obraram|[Ss]obre|[Ss]obrevivem|[Ss]ofreu|[Ss]ofria|[Ss]uas|[Ss]ubsistam|[Ss]ubsistem|[Ss]ubsistiam|[Ss]ucessivas|[Ss]ugeriu|[Ss]urgem|[Ss]urgir|[Ss]urgiram|[Ss]urgirem|[Ss]uscita|[Ss]uscitam|[Ss]uscitar|[Ss]uscitou|[Tt]ais|[Tt]ambém|[Tt]antas|[Tt]em|[Tt]êm|[Tt]êm-se|[Tt]emos|[Tt]endo|[Tt]enha|[Tt]enham|[Tt]enhamos|[Tt]enhas|[Tt]enho|[Tt]ens|[Tt]er|[Tt]erei|[Tt]erríveis|[Tt]estassem|[Tt]eve|[Tt]inha|[Tt]inham|[Tt]ira|[Tt]irando|[Tt]irar|[Tt]ire|[Tt]ive|[Tt]iver|[Tt]ivera|[Tt]iveram|[Tt]iverem|[Tt]ivesse|[Tt]ransformou|[Tt]ransparecer|[Tt]uas|[Úú]ltimas|[Vv]árias|[Vv]arrer|[Vv]ez|[Vv]ezes|[Vv]inham) duvidas\\b")

# Identificacao de calunia sem acento
vcalunia= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]lguma|[Aa]pelidei|[Aa]quela|[Aa]troz|[Cc]om|[Dd]a|[Dd]e|[Dd]essa|[Ee]|[Éé]|[Ee]m|[Ee]ra|[Ee]squisita|[Ee]ssa|[Ee]sta|[Ee]stúpida|[Ff]azendo|[Ff]azer|[Ff]ez|[Ff]osse|[Gg]rande|[Gg]randessíssima|[Gg]randíssima|[Hh]orrível|[Ii]nfame|[Mm]ais|[Mm]inha|[Nn]a|[Nn]em|[Nn]essa|[Nn]ova|[Nn]uma|[Oo]u|[Pp]ela|[Pp]erversa|[Pp]or|[Qq]ual|[Qq]ualquer|[Qq]ue|[Ss]emelhante|[Ss]er|[Ss]imples|[Ss]ua|[Tt]errível|[Tt]orpe|[Tt]udo|[Uu]ma|[Vv]elha|[Vv]il) calunia\\b")

# Identificacao de refugio sem acento
vrefugio= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]bençoado|[Aa]cha|[Aa]charam|[Aa]chou|[Aa]o|[Aa]quele|[Àà]quele|[Aa]vistavam|[Bb]usca|[Bb]uscamos|[Bb]uscando|[Bb]uscar|[Bb]uscarem|[Bb]uscasse|[Bb]uscava|[Bb]uscavam|[Bb]usquei|[Cc]eleste|[Cc]laro|[Cc]om|[Cc]omo|[Dd]aquele|[Dd]e|[Dd]erradeiro|[Dd]escobrira|[Dd]esencantado|[Dd]esse|[Dd]o|[Ee]|[Éé]|[Ee]m|[Ee]ncontrar|[Ee]ncontrara|[Ee]ncontraram|[Ee]ncontrarem|[Ee]ncontraria|[Ee]ncontrava|[Ee]ncontrou|[Ee]sse|[Ee]ste|[Ff]oram|[Ff]resco|[Gg]lorioso|[Ii]gnorado|[Ii]ntelectual|[Mm]aior|[Mm]elhor|[Mm]eu|[Mm]odesto|[Mm]ortal|[Mm]urado|[Nn]aquele|[Nn]em|[Nn]esse|[Nn]este|[Nn]o|[Nn]osso|[Nn]um|[Oo]|[Oo]ferecer|[Oo]u|[Oo]utro|[Pp]ara|[Pp]ediu|[Pp]rocura|[Pp]rocuram|[Pp]rocurar|[Pp]rocuraram|[Pp]rocurarem|[Pp]rocurava|[Pp]rocurou|[Ss]eu|[Ss]imples|[Ss]ou|[Tt]ambém|[Úú]ltimo|[Uu]m|[Úú]nico) refugio\\b")

# Identificacao de suplica sem acento
vsuplica= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]flita|[Aa]ntiga|[Àà]quela|[Bb]randa|[Bb]ruta|[Cc]ada|[Cc]om|[Cc]omo|[Dd]a|[Dd]aquela|[Dd]e|[Dd]essa|[Dd]esta|[Ee]|[Ee]m|[Ee]ssa|[Ee]sta|[Ff]azia|[Ff]ervente|[Gg]rande|[Hh]á|[Hh]avia|[Hh]umilde|[Ll]igeira|[Mm]esma|[Mm]inha|[Mm]uda|[Nn]a|[Nn]ossa|[Nn]ova|[Nn]uma|[Oo]u|[Pp]ela|[Qq]uase|[Ss]ua|[Tt]anta|[Tt]riste|[Úú]ltima|[Uu]ma|[Úú]nica) suplica\\b")

# Identificacao de copia sem acento
vcopia= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]chara|[Aa]chará|[Aa]lguma|[Aa]penas|[Aa]presentação|[Aa]quela|[Aa]rgêntea|[Aa]rquivada|[Aa]tiram|[Aa]utos|[Bb]ela|[Bb]oa|[Cc]ada|[Cc]om|[Cc]omo|[Cc]onsiderável|[Cc]ontra|[Cc]uja|[Dd]a|[Dd]ar|[Dd]arei|[Dd]ava|[Dd]e|[Dd]eixará|[Dd]eixei|[Dd]esignavam|[Dd]essa|[Dd]ivulgou|[Dd]ou|[Ee]|[Éé]|[Ee]ducação|[Ee]m|[Ee]ncaminhar|[Ee]ncontrou|[Ee]ntregará|[Ee]ntregue|[Ee]nviado|[Ee]nviará|[Ee]nviávamos|[Ee]nviei|[Ee]nviem|[Ee]nviou|[Ee]ra|[Éé]s|[Ee]sta|[Ee]xata|[Ff]amosa|[Ff]az|[Ff]ia|[Ff]lagrante|[Gg]lobo|[Gg]rande|[Gg]uardo|[Hh]avia|[Hh]umilde|[Ii]mperfeita|[Jj]untamos|[Ll]arga|[Ll]ibreto|[Mm]agnífica|[Mm]aior|[Mm]ais|[Mm]andaram|[Mm]andei|[Mm]anter|[Mm]inha|[Mm]or|[Nn]a|[Nn]ele|[Nn]otável|[Nn]ova|[Oo]u|[Oo]utra|[Pp]aisagem|[Pp]ara|[Pp]ela|[Pp]equena|[Pp]éssima|[Pp]or|[Pp]ossuir|[Pp]reciosa|[Pp]rimeira|[Pp]rovidenciará|[Pp]úblico|[Qq]ue|[Rr]emetida|[Rr]emeto|[Rr]ender|[Ss]ão|[Ss]e|[Ss]egunda|[Ss]em|[Ss]imples|[Ss]incera|[Ss]ó|[Ss]ua|[Tt]amanha|[Tt]anta|[Tt]em|[Tt]irar|[Tt]raga|[Tt]ragam|[Tt]ransbordante|[Tt]riste|[Tt]rouxe|[Tt]rouxeram|[Uu]m|[Uu]ma|[Úú]nica|[Vv]ariada|[Vv]elha|[Vv]elhíssima) copia\\b")

# Identificacao de acordao sem acento
vacordao= re.compile("(?i)\\b([Cc]âmara|[Cc]onselho|[Cc]onstitucional|[Ee]leitoral|[Ee]mentári|[Ii]nconstitucional|[Jj]uiz|[Jj]uíza|[Jj]ulgamento|[Jj]urídic|[Jj]urisprudência|[Jj]ustiça|[Ll]egislação|[Ll]ei|[Ll]eitura|[Mm]inistr|[Pp]arecer|[Pp]lenário|[Pp]ublicado|[Rr]egional|[Ss]ecção|[Ss]entença|[Ss]upremo|[Tt]eor|[Tt]ribunal|MP|STF|STJ|SUPREMO|TCE|TCU|TNU|TRF|TRIBUNAL|TRT)")

# Identificacao de deposito sem acento
vdeposito= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Aa]lgum|[Aa]o|[Aa]quele|[Aa]utêntico|[Cc]m|[Cc]om|[Cc]omo|[Dd]e|[Dd]er|[Dd]o|[Ee]|[Éé]|[Ee]m|[Ee]norme|[Ee]scondido|[Ee]sse|[Ee]ste|[Ff]ez|[Gg]era|[Gg]erava|[Gg]rande|[Hh]avia|[Ii]mprudente|[Ii]nfinito|[Ii]nviolado|[Mm]aior|[Mm]antém|[Mm]ediante|[Nn]enhum|[Nn]este|[Nn]o|[Nn]osso|[Nn]outro|[Nn]ovo|[Nn]um|[Oo]|[Oo]u|[Pp]agar|[Pp]ara|[Pp]elo|[Pp]equeno|[Pp]esado|[Pp]or|[Pp]recioso|[Pp]rimeiro|[Pp]rincipal|[Pp]ro|[Pp]rofundo|[Qq]ue|[Rr]eceber|[Rr]eferido|[Rr]equerer|[Rr]espectivo|[Rr]ígido|[Ss]agrado|[Ss]eu|[Ss]imples|[Ss]ó|[Ss]oturno|[Tt]al|[Uu]m|[Uu]m|[Úú]nico|[Vv]irou) deposito\\b")

# Identificacao de formula sem acento
vformula= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]certaram|[Aa]lguma|[Aa]mpla|[Aa]quela|[Aa]través|[Bb]oletim|[Cc]ampeonato|[Cc]élebre|[Cc]erta|[Cc]om|[Cc]omo|[Cc]omplexa|[Cc]oncisa|[Cc]onforme|[Cc]onhecida|[Cc]uja|[Dd]a|[Dd]e|[Dd]efendem|[Dd]escobre|[Dd]esta|[Dd]eterminada|[Dd]o|[Dd]uma|[Ee]|[Éé]|[Ee]m|[Ee]mpresa|[Ee]ncontrar|[Ee]ssa|[Ee]sta|[Ee]vita|[Ee]xclusivamente|[Hh]á|[Ii]nvariável|[Mm]ediante|[Mm]esma|[Mm]inha|[Nn]a|[Nn]aquela|[Nn]enhuma|[Nn]essa|[Nn]esta|[Nn]ossa|[Nn]ova|[Nn]uma|[Oo]|[Oo]s|[Oo]u|[Oo]utra|[Pp]ela|[Pp]or|[Pp]rópria|[Qq]ualquer|[Qq]uer|[Ss]eguinte|[Ss]egunda|[Ss]imples|[Ss]ua|[Tt]al|[Tt]ornam|[Uu]m|[Uu]ma|[Vv]elha) formula\\b")

# Identificacao de hospede sem acento
vhospede= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]caso|[Aa]nteriormente|[Aa]ntigo|[Aa]o|[Aa]quele|[Bb]ela|[Bb]om|[Cc]aro|[Cc]omo|[Dd]a|[Dd]e|[Dd]esalmado|[Dd]istinto|[Dd]o|[Dd]um|[Ee]|[Éé]|[Ee]grégio|[Ee]ra|[Ee]ras|[Ee]stá|[Ee]stava|[Ee]ste|[Ee]stimável|[Ee]terno|[Ff]oi|[Ff]ragrante|[Gg]rande|[Hh]avia|[Ii]lustre|[Ii]nteressante|[Ii]solo|[Ll]eviano|[Ll]inda|[Mm]aldito|[Mm]as|[Mm]au|[Mm]eu|[Mm]ísero|[Mm]isterioso|[Nn]enhum|[Nn]o|[Nn]obre|[Nn]ossa|[Nn]osso|[Nn]ova|[Nn]ovo|[Oo]|[Oo]dioso|[Oo]u|[Oo]utro|[Pp]arecia|[Pp]elo|[Pp]equeno|[Pp]rezado|[Pp]rimeiro|[Pp]róprio|[Qq]ue|[Qq]uerida|[Qq]uerido|[Ss]endo|[Ss]er|[Ss]erá|[Ss]eu|[Ss]imples|[Ss]ingular|[Ss]ingularíssimo|[Ss]ou|[Ss]ua|[Tt]aciturno|[Tt]al|[Tt]errível|[Tt]eu|[Tt]inha|[Tt]raiçoeira|[Tt]riste|[Úú]ltimo|[Uu]m|[Uu]ma|[Uu]ni|[Úú]nico|[Vv]elho|[Vv]osso) hospede\\b")

# Identificacao de opera sem acento
vopera= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]guardada|[Aa]lguma|[Bb]ela|[Bb]oa|[Cc]antar|[Cc]uja|[Dd]a|[Dd]aquela|[Dd]e|[Dd]essa|[Dd]esta|[Dd]uma|[Ee]|[Éé]|[Ee]ntre|[Ee]ssa|[Ee]sta|[Ff]amosa|[Ff]azer|[Ff]utebol|[Ff]utura|[Gg]raciosa|[Gg]rande|[Ii]nclui|[Nn]a|[Nn]ossa|[Nn]uma|[Pp]ara|[Pp]ela|[Pp]or|[Pp]róxima|[Qq]ualquer|[Ss]em|[Ss]obre|[Ss]umptuosa|[Úú]ltima|[Vv]er) [Oo]pera\\b")

# Identificacao de orbita sem acento
vorbita= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Àà]|[Aa]lta|[Bb]aixa|[Cc]ada|[Cc]uja|[Dd]a|[Dd]e|[Dd]essa|[Dd]esta|[Dd]eterminada|[Ee]m|[Ee]ssa|[Ee]xcêntrica|[Ll]imítrofe|[Mm]esma|[Mm]inha|[Nn]a|[Nn]esta|[Nn]uma|[Oo]utra|[Pp]ara|[Pp]ela|[Pp]equena|[Pp]rimeira|[Qq]uarta|[Ss]ua|[Tt]ranquila|[Uu]m|[Uu]ma|[Vv]astíssima) orbita\\b")

# Identificacao de ovulo sem acento
vovulo= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Aa]o|[Aa]quele|[Cc]ada|[Cc]hamadas|[Dd]e|[Dd]o|[Ee]|[Ee]m|[Ee]sse|[Ee]ste|[Nn]o|[Nn]um|[Oo]|[Oo]u|[Pp]rimeiro|[Pp]róprio|[Qq]ue|[Tt]ornam|[Úú]ltimo|[Uu]m|[Úú]nico|[Úú]ltimo) ovulo\\b")

# Identificacao de prospero sem acento
vprospero= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]inda|[Aa]no|[Aa]r|[Bb]astante|[Cc]olega|[Cc]olono|[Cc]om|[Cc]omércio|[Cc]omo|[Dd]e|[Dd]o|[Ee]|[Éé]|[Ee]m|[Ee]ra|[Ee]stá|[Ee]stado|[Ff]oi|[Ff]uturo|[Gg]rau|[Mm]ais|[Mm]ilionário|[Mm]isterioso|[Mm]uito|[Oo]|[Pp]aís|[Pp]edagógico|[Rr]elativamente|[Rr]ubor|[Ss]eu|[Tt]ão|[Tt]ornavam|[Uu]m|[Vv]i) prospero\\b")

# Identificacao de prospera sem acento
vprospera= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Aa]li|[Aa]tividade|[Bb]arriguinha|[Bb]astante|[Bb]urguesia|[Cc]arreira|[Cc]idade|[Dd]a|[Dd]emocracia|[Ee]|[Ee]m|[Ee]mpresa|[Ee]ra|[Ee]ssa|[Ee]xtraordinariamente|[Ff]ace|[Ff]ortuna|[Mm]ais|[Mm]as|[Mm]enos|[Nn]ação|[Nn]esta|[Pp]rovíncia|[Ss]ociedade|[Ss]ua|[Tt]ão|[Tt]ransição|[Uu]ma|[Vv]iagem|[Vv]ida|[Zz]ona) prospera\\b")

# Identificacao de revolver sem acento
vrevolver= re.compile("(?i)\\b([A-Za-zçáê]+(-se|-lhe|-lhes|-o|-me|-nos|-la|-lo|-te)|[Aa]|[Aa]o|[Aa]té|[Cc]om|[Dd]e|[Dd]o|[Dd]um|[Ee]|[Ee]sse|[Ee]ste|[Gg]rande|[Ii]nseparável|[Mm]elhor|[Mm]encionei|[Mm]eu|[Nn]em|[Nn]o|[Nn]um|[Oo]|[Oo]u|[Pp]ara|[Pp]elo|[Ss]em|[Ss]eu|[Tt]enho|[Tt]eu|[Tt]oma|[Tt]omado|[Uu]m|[Vv]elho|[Vv]erdadeiro) revolver\\b")


# fim Paronimos especiais *-*-*-*-



punct = { "?": "do ponto de interrogação", "!": "do ponto de exclamação",
  ",": "da vírgula", ":": "do dois pontos", ";": "do ponto e vírgula",
  "(": "do parêntese de abertura", ")": "do parêntese de fechamento",
  "[": "do colchete de abertura", "]": "do colchete de fechamento",
 "\u201c": "das aspas de abertura","\u201d": "das aspas de fechamento"}

aaVer = { "quando": "Quando", "Quando": "Quando",
  "que": "Que", "Que": "Que", "se": "Se",
  "Se": "Se"}

aacento = {"abada": "abadá",
"abobora": "abóbora",
"aboboras": "abóboras",
"abundancia": "abundância",
"abundancias": "abundâncias",
"acaro":"ácaro",
"acida":"ácida",
"acidas":"ácidas",
"acido":"ácido",
"adverbio": "advérbio",
"Adverbio": "Advérbio",
"Africa":"África",
"africa":"áfrica",
"Africas":"Áfricas",
"africas":"áfricas",
"africo":"áfrico",
"Agencia": "Agência",
"Agencias": "Agências",
"agencias": "agências",
"agua":"água",
"aguas":"águas",
"Agua":"Água",
"Aguas":"Águas",
"alfandega": "alfândega",
"alfandegas": "alfândegas",
"alivio": "alívio",
"amago":"âmago",
"amilase": "amílase",
"angulo":"ângulo",
"anonima": "anônima",
"anonimas": "anônimas",
"anonimo": "anônimo",
"antidoto": "antídoto",
"antiespirita": "antiespírita",
"antifona": "antífona",
"antifonas": "antífonas",
"antifono": "antífono",
"Anuncio": "Anúncio",
"apostola": "apóstola",
"apostolas": "apóstolas",
"apostolo": "apóstolo",
"apostrofe": "apóstrofe",
"apostrofes": "apóstrofes",
"apostrofo": "apóstrofo",
"apostrofos": "apóstrofos",
"arabia": "Arábia",
"Arabia": "Arábia",
"arabias": "Arábias",
"Arabias": "Arábias",
"arbitro":"árbitro",
"artificio": "artifício",
"assedio": "assédio",
"Assedio": "Assédio",
"assuca": "açúcar",
"assucar": "açúcar",
"atico":"ático",
"atona":"átona",
"atonas":"átonas",
"atono":"átono",
"atoxica":  "atóxica",
"atoxicas":  "atóxicas",
"atoxico":  "atóxico",
"auspicia": "auspícia",
"auspicias": "auspícias",
"auspicio": "auspício",
"Autenticas":  "Autênticas",
"autenticas":  "autênticas",
"Autografo": "Autógrafo",
"autografo": "autógrafo",
"autopsia":  "autópsia",
"autopsias":  "autópsias",
"avicola": "avícola",
"avicolas": "avícolas",
"avicula": "avícola",
"aviculas": "avícolas",
"avida":"ávida",
"avidas":"ávidas",
"avido":"ávido",
"babeis": "babéis",
"Babeis": "babéis",
"balaustre": "balaústre",
"balaustres": "balaústres",
"bau": "baú",
"bibliografa": "bibliógrafa",
"bibliografas": "bibliógrafas",
"bibliografo": "bibliógrafo",
"biopsia": "biópsia",
"biopsias": "biópsias",
"biotipo": "biótipo",
"bordeis": "bordéis",
"brecho": "brechó",
"bugues": "burguês",
"bussola": "bússola",
"bussolas": "bússolas",
"buzio": "búzio",
"cadencia": "cadência",
"cafe": "café",
"cafes": "cafés",
"cafune": "cafuné",
"calorifica": "calorífica",
"calorificas": "caloríficas",
"calorifico": "calorífico",
"calunias": "calúnias",
"cambio": "câmbio",
"canada": "Canadá",
"Canada": "Canadá",
"canfora": "cânfora",
"capitulo": "capítulo",
"Capitulo": "Capítulo",
"CAPITULO": "CAPÍTULO",
"carate": "caratê",
"caricia": "carícia",
"caricias": "carícias",
"carie": "cárie",
"caries": "cáries",
"carreteis": "carretéis",
"carteis": "cartéis",
"catalogo": "catálogo",
"catimbo": "catimbó",
"caustica": "cáustica",
"causticas": "cáusticas",
"caustico": "cáustico",
"centigrada": "centígrada",
"centigradas": "centígradas",
"centigrado": "centígrado",
"centrifuga": "centrífuga",
"centrifugas": "centrífugas",
"centrifugo": "centrífugo",
"chines": "chinês",
"cientifica": "científica",
"cientificas": "científicas",
"cientifico": "científico",
"cinematografo": "cinematógrafo",
"cinzeis": "cinzéis",
"cipo": "cipó",
"circunstancia": "circunstância",
"circunstancias": "circunstâncias",
"ciume": "ciúme",
"ciumes": "ciúmes",
"CLAUSULA": "CLÁUSULA",
"clausula": "cláusula",
"Clausula": "Cláusula",
"CLAUSULAS": "CLÁUSULAS",
"clausulas": "cláusulas",
"Clausulas": "Cláusulas",
"clemencia": "clemência",
"clemencias": "clemências",
"Clinica": "Clínica",
"clinica": "clínica",
"Clinicas": "Clínicas",
"clinicas": "clínicas",
"clinico": "clínico",
"coagulo": "coágulo",
"cocega": "cócega",
"cocegas": "cócegas",
"confidencia": "confidência",
"confidencias": "confidências",
"colonia": "colônia",
"colonias": "colônias",
"coloquio": "colóquio",
"comedia": "comédia",
"Comedia": "Comédia",
"comedias": "comédias",
"Comedias": "Comédias",
"comercio": "comércio",
"Comercio": "Comércio",
"comissaria": "comissária",
"comissarias": "comissárias",
"comissario": "comissário",
"comoda": "cômoda",
"comodas": "cômodas",
"comodo": "cômodo",
"compendio": "compêndio",
"compito": "cômpito",
"concava": "côncava",
"concavas": "côncavas",
"concavo": "côncavo",
"conferencia": "conferência",
"conferencias": "conferências",
"congenere": "congênere",
"congeneres": "congêneres",
"consocio": "consórcio",
"consorcio": "consórcio",
"contingencia": "contingência",
"contingencias": "contingências",
"continuas": "contínuas",
"convenio": "convênio",
"Convenio": "Convênio",
"Coparticipe": "Copartícipe",
"coparticipe": "copartícipe",
"Coparticipes": "Copartícipes",
"coparticipes": "copartícipes",
"Copia": "Cópia",
"Copias": "Cópias",
"copias": "cópias",
"Cranio": "Crânio",
"cranio": "crânio",
"cubica": "cúbica",
"cubicas": "cúbicas",
"cubico": "cúbico",
"cumulo": "cúmulo",
"curia": "cúria",
"Curia": "Cúria",
"custodia": "custódia",
"Custodia": "Custódia",
"custodias": "custódias",
"Custodio": "Custódio",
"dadiva": "dádiva",
"dadivas": "dádivas",
"datilografo": "datilógrafo",
"Debito": "Déebito",
"decima": "décima",
"Decima": "Décima",
"decimas": "décimas",
"Decimo": "Décimo",
"decimo": "décimo",
"denuncias": "denúncias",
"Deposito": "Depósito",
"Dicionario": "Dicionário",
"dicionario": "dicionário",
"diligencia": "diligência",
"diligencias": "diligências",
"distancias": "distâncias",
"domestica": "doméstica",
"domesticas": "domésticas",
"domestico": "doméstico",
"domicilio": "domicílio",
"duble": "dublê",
"dubles": "dublês",
"Duvidas": "Dúvidas",
"duzia": "dúzia",
"duzias": "dúzias",
"eclampsia": "eclâmpsia",
"eclampsias": "eclâmpsias",
"ecografo": "ecógrafo",
"Efigie": "Efígie",
"efigie": "efígie",
"Efigies": "Efígies",
"efigies": "efígies",
"eletrolise": "eletrólise",
"eminencia": "eminência",
"Eminencia": "Eminência",
"eminencias": "eminências",
"empresaria": "empresária",
"empresarias": "empresárias",
"empresario": "empresário",
"enfase":"ênfase",
"enfases":"ênfases",
"EPIGRAFE": "EPÍGRAFE",
"epigrafe": "epígrafe",
"Epigrafe": "Epígrafe",
"EPIGRAFES": "EPÍGRAFES",
"epigrafes": "epígrafes",
"Epigrafes": "Epígrafes",
"epigrafo": "epígrafo",
"epigramatica": "epigramática",
"epigramaticas": "epigramáticas",
"epigramatico": "epigramático",
"episodio": "episódio",
"equivoco": "equívoco",
"escambio": "escâmbio",
"escoria": "escória",
"escorias": "escórias",
"espatula": "espátula",
"espatulas": "espátulas",
"especie": "espécie",
"especies": "espécies",
"especificas": "específicas",
"espectroscopio": "espectroscópio",
"espirita": "espírita",
"espiritas": "espíritas",
"espirito": "espírito",
"esplendida": "esplêndida",
"esplendidas": "esplêndidas",
"esplendido": "esplêndido",
"espolio": "espólio",
"estabulo": "estábulo",
"estadio": "estádio",
"estancia": "estância",
"Estancia": "Estância",
"estancias": "estâncias",
"estatua": "estátua",
"estatuas": "estátuas",
"estomago": "estômago",
"estrategia": "estratégia",
"estrategias": "estratégias",
"estupida": "estúpida",
"estupidas": "estúpidas",
"estupido": "estúpido",
"Exercito": "Exército",
"exito":"êxito",
"experiencia": "experiência",
"experiencias": "experiências",
"explicitas": "explícitas",
"explicito": "explícito",
"extravagancia": "extravagância",
"extravagancias": "extravagâncias",
"evidencias": "evidências",
"Fabrica": "fábrica",
"Fabricas": "fábricas",
"fabricas": "fábricas",
"fabula": "fábula",
"fabulas": "fábulas",
"falacia": "falácia",
"falacias": "falácias",
"ferias": "férias",
"fieis": "fiéis",
"figado": "fígado",
"filosofa": "filósofa",
"filosofas": "filósofas",
"filosofo": "filósofo",
"flamula": "flâmula",
"flamulas": "flâmulas",
"fonografo": "fonógrafo",
"Formica": "Fórmica",
"formica": "fórmica",
"formicas": "fórmicas",
"Formula": "Fórmula",
"Formulas": "Fórmulas",
"formulas": "fórmulas",
"fosforo": "fósforo",
"fotocopia": "fotocópia",
"fotocopias": "fotocópias",
"frigida": "frígida",
"frigidas": "frígidas",
"frigido": "frígido",
"frigorifico": "frigorífico",
"fuba": "fubá",
"fubas": "fubás",
"fulgido": "fúlgido",
"Galaxia": "Galáxia",
"galaxia": "galáxia",
"galaxias": "galáxias",
"gamba": "gambá",
"gambas": "gambás",
"Gambia": "Gâmbia",
"ganancia": "ganância",
"ganancias": "ganâncias",
"geografa": "geógrafa",
"geografas": "geógrafas",
"geografo": "geógrafo",
"georreferencia": "georreferência",
"georreferencias": "georreferências",
"Gerencia": "Gerência",
"Gerencias": "Gerências",
"gerencias": "gerências",
"ginastica": "ginástica",
"ginasticas": "ginásticas",
"ginastico": "ginástico",
"gloria": "glória",
"Gloria": "Glória",
"glorias": "glórias",
"gondola": "gôndola",
"gondolas": "gôndolas",
"gramatica": "gramática",
"gramaticas": "gramáticas",
"gramatico": "gramático",
"gravida": "grávida",
"gravidas": "grávidas",
"gravido": "grávido",
"guara": "guará",
"guaras": "guarás",
"guarana": "guaraná",
"guaranas": "guaranás",
"halito": "hálito",
"hemodialise": "hemodiálise",
"hemodialises": "hemodiálises",
"hemodiastase": "hemodiástase",
"hemodiastases": "hemodiástases",
"heptafono": "heptáfono",
"heptafonos": "heptáfonos",
"hibrida": "híbrida",
"hibridas": "híbridas",
"hibrido": "híbrido",
"hidrogena": "hidrógena",
"hidrogenas": "hidrógenas",
"hidrogeno": "hidrógeno",
"hidrolise": "hidrólise",
"hidrolises": "hidrólises",
"hidrometro": "hidrômetro",
"hilaria": "hilária",
"hilarias": "hilárias",
"hilario": "hilário",
"historia": "história",
"historias": "histórias",
"homofila": "homófila",
"homofilas": "homófilas",
"homofilo": "homófilo",
"homofona": "homófona",
"homofonas": "homófonas",
"homofono": "homófono",
"honorifica": "honorífica",
"honorificas": "honoríficas",
"honorifico": "honorífico",
"Horoscopo": "Horóscopo",
"horoscopo": "horóscopo",
"Hospede": "Hóspede",
"Hospedes": "Hóspedes",
"hospedes": "hóspedes",
"ignominia": "ignomínia",
"ignominias": "ignomínias",
"ilegitima": "ilegítima",
"ilegitimas": "ilegítimas",
"ilegitimo": "ilegítimo",
"impar":"ímpar",
"impares":"ímpares",
"impertinencia": "impertinência",
"impertinencias": "impertinências",
"impeto":"ímpeto",
"impia":"ímpia",
"impias":"ímpias",
"impio":"ímpio",
"indicio": "indício",
"indulgencia": "indulgência",
"indulgencias": "indulgências",
"Industria": "Iindústria",
"industria": "indústria",
"Industrias": "Indústrias",
"industrias": "indústrias",
"Influencia": "Influência",
"influencias": "influências",
"ingenua": "ingênua",
"ingenuas": "ingênuas",
"ingenuo": "ingênuo",
"Inicio": "Início",
"INICIO": "INÍCIO",
"insignia": "insígnia",
"insignias": "insígnias",
"insipida": "insípida",
"insipidas": "insípidas",
"insipido": "insípido",
"insonia": "insônia",
"insonias": "insônias",
"instancia": "instância",
"instancias": "instâncias",
"integras":"íntegras",
"inteligencia": "inteligência",
"inteligencias": "inteligências",
"intimas":"íntimas",
"inventario": "inventário",
"iridio": "irídio",
"iris":"íris",
"irreverencia": "irreverência",
"irreverencias": "irreverências",
"jatancia": "jatância",
"jatancias": "jatâncias",
"jornalistica": "jornalística",
"jornalisticas": "jornalísticas",
"jornalistico": "jornalístico",
"juri": "júri",
"Juri": "Júri",
"lagrima": "lágrima",
"lagrimas": "lágrimas",
"lamina": "lâmina",
"laminas": "lâminas",
"Lampada": "Lâmpada",
"lampada": "lâmpada",
"Lampadas": "Lâmpadas",
"lampadas": "lâmpadas",
"Lastima": "Lástima",
"Lastimas": "Lástimas",
"lastimas": "lástimas",
"legitimo": "legítimo",
"licitas": "lícitas",
"licito": "lícito",
"Liquidas": "Líquidas",
"liquidas": "líquidas",
"Liquido": "Líquido",
"lirica": "lírica",
"liricas": "líricas",
"lirico": "lírico",
"litigio": "litígio",
"logica": "lógica",
"logicas": "lógicas",
"logico": "lógico",
"lucifero": "lucífero",
"ma": "má",
"macunaima": "macunaíma",
"Macunaima": "Macunaíma",
"mafia": "máfia",
"mafias": "máfias",
"mafua": "mafuá",
"mafuas": "mafuás",
"magica": "mágica",
"magicas": "mágicas",
"magico": "mágico",
"magnifica": "magnífica",
"magnificas": "magníficas",
"magnifico": "magnífico",
"Malaria": "Malária",
"malaria": "malária",
"malarias": "malárias",
"maleficio": "malefício",
"malicias": "malícias",
"maluques": "maluquês",
"maniaca": "maníaca",
"maniacas": "maníacas",
"maniaco": "maníaco",
"manigancia": "manigância",
"manigancias": "manigâncias",
"maquina": "máquina",
"maquinas": "máquinas",
"Mascara": "Máscara",
"Mascaras": "Máscaras",
"mascaras": "máscaras",
"Matricula": "Matrícula",
"Matriculas": "Matrículas",
"matriculas": "matrículas",
"matrimonio": "matrimônio",
"Medica": "Médica",
"medica": "médica",
"Medicas": "Médicas",
"medicas": "médicas",
"Medico": "Médico",
"medico": "médico",
"megametro": "megâmetro",
"Memoria": "Memória",
"memoria": "memória",
"memorias": "memórias",
"mercancia": "mercância",
"mercancias": "mercâncias",
"metafisica": "metafísica",
"metafisicas": "metafísicas",
"metafisico": "metafísico",
"microbio": "micróbio",
"micrografo": "micrógrafo",
"milicia": "milícia",
"milicias": "milícias",
"milimetro": "milímetro",
"mimeografo": "mimeógrafo",
"mimica": "mímica",
"mimicas": "mímicas",
"mimico": "mímico",
"minudencia": "minudência",
"minudencias": "minudências",
"minuscula": "minúscula",
"minusculas": "minúsculas",
"minusculo": "minúsculo",
"misera": "mísera",
"miseras": "míseras",
"misero": "mísero",
"monossilaba": "monossílaba",
"monossilabas": "monossílabas",
"monossilabo": "monossílabo",
"murmurio": "murmúrio",
"musica": "música",
"musicas": "músicas",
"musico": "músico",
"naufrago": "náufrago",
"necropsia": "necrópsia",
"necropsias": "necrópsias",
"negligencias": "negligências",
"Negocio": "Negócio",
"nevoa": "névoa",
"nevoas": "névoas",
"Noticia": "Notícia",
"Noticias": "Notícias",
"noticias": "notícias",
"noveis": "novéis",
"obliqua": "oblíqua",
"obliquas": "oblíquas",
"obliquo": "oblíquo",
"obsequio": "obséquio",
"obstaculo": "obstáculo",
"oficio": "ofício",
"omega":"ômega",
"oraculo": "oráculo",
"Orbita":"Órbita",
"Orbitas":"Órbitas",
"orbitas":"órbitas",
"osculo":"ósculo",
"Ovulo":"Óvulo",
"pacifico": "pacífico",
"pagina": "página",
"paginas": "páginas",
"paleofilo": "paleófilo",
"paleografo": "paleógrafo",
"paleto": "paletó",
"palmatoria": "palmatória",
"palmatorias": "palmatórias",
"pandego": "pândego",
"panico": "pânico",
"pantografo": "pantógrafo",
"papeis": "papéis",
"pasteis": "pastéis",
"paralitica": "paralítica",
"paraliticas": "paralíticas",
"paralitico": "paralítico",
"Parana": "Paraná",
"Parodia": "Paródia",
"parodias": "paródias",
"Paroquia": "Paróquia",
"paroquia": "paróquia",
"Paroquias": "Paróquias",
"paroquias": "paróquias",
"pascoa": "páscoa",
"pascoas": "páscoas",
"Patio": "Pátio",
"patio": "pátio",
"Patrimonio": "Patrimônio",
"patrimonio": "patrimônio",
"pavia": "pávia",
"pavias": "pávias",
"pendencia": "pendência",
"pendencias": "pendências",
"pendulo": "pêndulo",
"penes": "pênis",
"penitencia": "penitência",
"penitencias": "penitências",
"perdulario": "perdulário",
"perfeiçoes": "perfeições",
"Pericia": "Perícia",
"Pericias": "Perícias",
"pericias": "perícias",
"perola": "pérola",
"perolas": "pérolas",
"perpetuas": "perpétuas",
"perpetuo": "perpétuo",
"pestilencia": "pestilência",
"pestilencias": "pestilências",
"pilula": "pílula",
"pilulas": "pílulas",
"piramide": "pirâmide",
"piramides": "pirâmides",
"pirografo": "pirógrafo",
"placida": "plácida",
"placidas": "plácidas",
"placido": "plácido",
"plagio": "plágio",
"planteis": "plantéis",
"plastica": "plástica",
"plasticas": "plásticas",
"plastico": "plástico",
"polemica": "polêmica",
"polemicas": "polêmicas",
"polemico": "polêmico",
"poligrafa": "polígrafa",
"poligrafas": "polígrafas",
"poligrafo": "polígrafo",
"porem": "porém",
"porno": "pornô",
"pornos": "pornôs",
"potencia": "potência",
"potencias": "potências",
"preambulo": "preâmbulo",
"preçario": "preçário",
"prefacio": "prefácio",
"preferencia": "preferência",
"preferencias": "preferências",
"premio": "prêmio",
"prenuncio": "prenúncio",
"presagio": "presságio",
"presidio": "presídio",
"pressagio": "presságio",
"principio": "princípio",
"Privilegio": "Privilégio",
"privilegio": "privilégio",
"prodiga": "pródiga",
"prodigas": "pródigas",
"prodigo": "pródigo",
"prologo": "prólogo",
"pronostico": "pronóstico",
"prontuario": "prontuário",
"propicio": "propício",
"proposito": "propósito",
"Prospero": "Próspero",
"Prospera": "Próspera",
"Prosperas": "Prósperas",
"prosperas": "prósperas",
"proverbio": "provérbio",
"Proverbio": "Provérbio",
"Providencia": "Providência",
"providencias": "providências",
"prudencia": "prudência",
"prudencias": "prudências",
"psicanalise": "psicanálise",
"psicanalises": "psicanálises",
"psicoanalise": "psicoanálise",
"psicoanalises": "psicoanálises",
"psicologa": "psicóloga",
"psicologas": "psicólogas",
"psicologo": "psicólogo",
"Publica": "Pública",
"Publicas": "Públicas",
"publicas": "públicas",
"quadrupede": "quadrúpede",
"quadrupedes": "quadrúpedes",
"quadrupla": "quádrupla",
"quadruplas": "quádruplas",
"quadruplo": "quádruplo",
"quarteis": "quartéis",
"querencia": "querência",
"querencias": "querências",
"quilometro": "quilômetro",
"quintessencia": "quintessência",
"quintessencias": "quintessências",
"radio": "rádio",
"reciproca": "recíproca",
"reciprocas": "recíprocas",
"reciproco": "recíproco",
"Referencia": "Referência",
"referencias": "referências",
"reminiscencia": "reminiscência",
"reminiscencias": "reminiscências",
"renuncias": "renúncias",
"Replica": "Réplica",
"Replicas": "Réplicas",
"replicas": "réplicas",
"Republica": "República",
"Republicas": "Repúblicas",
"republicas": "repúblicas",
"residencia": "residência",
"residencias": "residências",
"reticencia": "reticência",
"reticencias": "reticências",
"retorica": "retórica",
"retoricas": "retóricas",
"retorico": "retórico",
"retrograda": "retrógrada",
"retrogradas": "retrógradas",
"retrogrado": "retrógrado",
"reusa": "reúsa",
"reusas": "reúsas",
"reuso": "reúso",
"Reverencia": "Reverência",
"reverencias": "reverências",
"Revolve": "Revólver",
"revolve": "Revólver",
"revolves": "revólveres",
"Revolver": "Revólver",
"Revolveres": "Revólveres",
"revolveres": "revólveres",
"rodizio": "rodízio",
"Romania": "România",
"rufio": "rúfio",
"rustica": "rústica",
"rusticas": "rústicas",
"rustico": "rústico",
"sache": "sachê",
"saches": "sachês",
"salario": "salário",
"saltica": "sáltica",
"salticas": "sálticas",
"saltico": "sáltico",
"Secretario": "Secretário",
"secretario": "secretário",
"seculo": "século",
"Sequencia": "Sequência",
"sequencias": "sequências",
"sequio": "séquio",
"serie": "série",
"series": "séries",
"serigrafo": "serígrafo",
"serio": "sério",
"silaba": "sílaba",
"silabas": "sílabas",
"silabo": "sílabo",
"Silencio": "Silêncio",
"silencio": "silêncio",
"simbolo": "símbolo",
"sincope": "síncope",
"sincopes": "síncopes",
"sindica": "síndica",
"sindicas": "síndicas",
"sindico": "síndico",
"sitio": "sítio",
"solida": "sólida",
"solidas": "sólidas",
"solido": "sólido",
"sonambula": "sonâmbula",
"sonambulas": "sonâmbulas",
"sonambulo": "sonâmbulo",
"soporifero": "soporífero",
"subespecifica": "subespecífica",
"subespecificas": "subespecíficas",
"subespecifico": "subespecífico",
"Subsecretario": "Subsecretário",
"subsecretario": "subsecretário",
"subsidio": "subsídio",
"substancia": "substância",
"substancias": "substâncias",
"sudorifica": "sudorífica",
"sudorificas": "sudoríficas",
"sudorifico": "sudorífico",
"sumaria": "sumária",
"sumarias": "sumárias",
"Sumario": "Sumário",
"sumario": "sumário",
"sumula": "súmula",
"sumulas": "súmulas",
"suplicas": "suúplicas",
"suplicio": "suplício",
"taboa": "tábua",
"taboas": "tábuas",
"tabua": "tábua",
"tabuas": "tábuas",
"Tabula": "Tábula",
"Tabulas": "Tábulas",
"tabulas": "tábulas",
"Telegrafo": "Telégrafo",
"telegrafo": "telégrafo",
"tendencia": "tendência",
"tendencias": "tendências",
"tipica": "típica",
"tipicas": "típicas",
"tipico": "típico",
"tipografo": "tipógrafo",
"titulo": "título",
"Titulo": "Título",
"tomografo": "tomógrafo",
"toneis": "tonéis",
"toner": "tôner",
"topografo": "topógrafo",
"toxica": "tóxica",
"toxicas": "tóxicas",
"toxico": "tóxico",
"trafico": "tráfico",
"tragedia": "tragédia",
"tragedias": "tragédias",
"tramites": "trâmites",
"Transito": "Trânsito",
"translucida": "translúcida",
"translucidas": "translúcidas",
"translucido": "translúcido",
"Tremula": "Trêmula",
"tremulas": "trêmulas",
"Tremulo": "Trêmulo",
"tremulo": "trêmulo",
"triangulo": "triângulo",
"tripe": "tripé",
"tropega": "trôpega",
"tropegas": "trôpegas",
"tropego": "trôpego",
"tropico": "trópico",
"tumulo": "túmulo",
"ulcera":"úlcera",
"ulceras":"úlceras",
"ultima":"última",
"ultimas":"últimas",
"ultimo":"último",
"unanime": "unânime",
"unanimes": "unânimes",
"urticaria": "urticária",
"veiculo": "veículo",
"vermifugo": "vermífugo",
"virgula": "vírgula",
"virgulas": "vírgulas",
"Vitima": "Vítima",
"vitimas": "vítimas",
"vitoria": "vitória",
"vitorias": "vitórias",
"Vivencia": "Vivóncia",
"vivencias": "vivóncias",
"voluntaria": "voluntária",
"voluntarias": "voluntárias",
"voluntario": "voluntário",
"xilografo": "xilógrafo",
"xoco": "xocó",
"zoografo": "zoógrafo"}


acAcento = set(["abada",
"abobora", "aboboras", "abundancia",
"abundancias", "acaro", "acida", "acidas", "acido",
"adverbio", "Adverbio", "Africa", "africa",
"Africas", "africas", "africo", "Agencia", "Agencias", "agencias",
"alfandega","alfandegas", "alivio", "amago", "amilase", "angulo",
"anonima", "anonimas", "anonimo",
"antidoto", "antiespirita", "antifona", "antifonas",
"antifono", "Anuncio", "apostola", "apostolas", "apostolo",
"apostrofe", "apostrofes", "apostrofo", "apostrofos",
"arabia", "Arabia", "arabias", "Arabias"
"arbitro", "artificio", "assedio", "Assedio",
"assuca", "assucar", "atico", "atona", "atonas", "atono",
"atoxica", "atoxicas", "atoxico", "auspicia",
"auspicias", "auspicio", "Autenticas", "autenticas", "Autografo", "autografo", "autopsia", "autopsias",
"avicola", "avicolas", "avicula", "aviculas",
"avida", "avidas", "avido", "babeis", "Babeis", "balaustre", "balaustres", "bau",
"bibliografa", "bibliografas", "bibliografo", "biopsia",
"biopsias", "biotipo", "bordeis", "brecho", "burgues", "bussola",
"bussolas", "buzio", "cadencia", "cafe",
"cafes", "cafune", "calorifica", "calorificas",
"calorifico", "calunias", "cambio", "canada", "Canada",
"canfora", "CAPITULO", "Capitulo", "capitulo", "carate",
"caricia", "caricias", "carie", "caries", "carreteis", "carteis", "catalogo", "catimbo",
"caustica", "causticas", "caustico", "centigrada",
"centigradas", "centigrado", "centrifuga", "centrifugas",
"centrifugo", "chines", "cientifica", "cientificas",
"cientifico", "cinematografo", "cinzeis", "cipo", "circunstancia",
"ciume", "ciumes", "CLAUSULA", "clausula",
"Clausula", "CLAUSULAS", "clausulas", "Clausulas",
"Clinica", "clinica", "Clinicas", "clinicas", "clinico", "coagulo",
"cocega", "cocegas", "colonia", "colonias",
"coloquio", "comedia", "Comedia", "comedias",
"Comedias", "comercio", "Comercio",
"comissaria", "comissarias", "comissario", "comoda",
"comodas", "comodo", "compendio", "compito",
"concava", "concavas", "concavo", "conferencia",
"conferencias", "confidencias", "congenere", "congeneres", "consocio",
"consorcio", "contingencia", "contingencias", "continuas", "convenio",
"Convenio", "Coparticipe", "coparticipe", "Coparticipes", "coparticipes",
"Copia", "Copias", "copias",
"Cranio", "cranio", "cubica", "cubicas", "cubico", "cumulo",
"curia", "Curia", "custodia", "Custodia",
"custodias", "Custodio", "datilografo", "Debito", "Decima",
"decima", "decimas", "Decimo", "decimo", "denuncias", "Deposito", "Dicionario", "dicionario",
"diligencia", "diligencias", "distancias",
"domestica", "domesticas", "domestico", "domicilio",
"duble", "dubles", "Duvidas", "duzia", "duzias", "eclampsia", "eclampsias",
"ecografo", "Efigie", "efigie", "Efigies",
"efigies", "eletrolise", "eminencia", "Eminencia",
"eminencias", "empresaria", "empresarias", "empresario",
"enfase", "enfases", "EPIGRAFE", "epigrafe",
"Epigrafe", "EPIGRAFES", "epigrafes", "Epigrafes",
"epigrafo", "epigramatica", "epigramaticas", "epigramatico",
"episodio", "equivoco", "escambio", "escoria", "escorias",
"espatula", "espatulas", "especie", "especies", "especificas",
"espectroscopio", "espirita", "espiritas", "espirito",
"esplendida", "esplendidas", "esplendido", "espolio",
"estabulo", "estadio", "estancia", "Estancia",
"estancias", "estatua", "estatuas", "estomago",
"estrategia", "estrategias", "extravagancia", "extravagancias",
"estupida", "estupidas", "estupido", "Exercito", "exito", "experiencia",
"experiencias", "explicita", "explicitas", "explicito", "evidencias",
"Fabrica", "Fabricas", "fabricas", "fabula",
"fabulas", "falacia", "falacias", "ferias", "fieis", "figado",
"filosofa", "filosofas", "filosofo", "flamula", "flamulas", "fonografo",
"Formica", "formica", "formicas", "Formula", "Formulas", "formulas", "fosforo",
"fotocopia", "fotocopias", "frigida", "frigidas",
"frigido", "frigorifico", "fuba", "fubas",
"fulgido", "Galaxia", "galaxia", "galaxias", "gamba",
"gambas", "Gambia", "ganancia", "ganancias", "geografa",
"geografas", "geografo", "georreferencia", "georreferencias",
"Gerencia", "Gerencias", "gerencias", "ginastica", "ginasticas", "ginastico", "gloria",
"Gloria", "glorias", "gondola", "gondolas",
"gramatica", "gramaticas", "gramatico", "gravida",
"gravidas", "gravido", "guara", "guaras", "guarana", "guaranas", "halito",
"hemodialise", "hemodialises", "hemodiastase", "hemodiastases",
"heptafono", "heptafonos", "hibrida", "hibridas",
"hibrido", "hidrogena", "hidrogenas", "hidrogeno",
"hidrolise", "hidrolises", "hidrometro", "hilaria",
"hilarias", "hilario", "historia", "historias",
"homofila", "homofilas", "homofilo", "homofona",
"homofonas", "homofono", "honorifica", "honorificas",
"honorifico", "Horoscopo", "horoscopo", "Hospede", "Hospedes", "hospedes", "ignominia",
"ignominias", "ilegitima", "ilegitimas",
"ilegitimo", "impar", "impares", "impeto",
"impia", "impias", "impio", "indicio",
"indulgencia", "indulgencias", "Industria", "industria",
"Industria", "industrias", "Influencia", "influencias",
"ingenua", "ingenuas", "ingenuo", "Inicio", "INICIO", "insignia",
"insignias", "insipida", "insipidas", "insipido",
"insonia", "insonias", "instancia", "instancias",
"inteligencia", "inteligencias", "intimas", "inventario", "iridio",
"iris", "irreverencia", "irreverencias", "jornalistica",
"jornalisticas", "jornalistico", "lagrima", "lagrimas",
"lamina", "laminas", "Lampada", "lampada", "Lampadas", "lampadas", "Lastima", "Lastimas",
"lastimas", "legitimo", "licitas",
"licito", "Liquidas", "liquidas", "Liquido", "lirica", "liricas", "lirico",
"litigio", "logica", "logicas", "logico",
"lucifero", "ma", "macunaima", "Macunaima", "mafia",
"mafias", "mafua", "mafuas", "magica",
"magicas", "magico", "magnifica", "magnificas",
"magnifico", "Malaria", "malaria", "malarias",
"maleficio", "malicias", "maluques", "maniaca",
"maniacas", "maniaco", "manigancia", "manigancias", "maquina",
"maquinas", "Mascara", "Mascaras", "mascaras",
"Matricula", "Matriculas", "matriculas", "matrimonio",
"Medica", "medica", "Medicas", "medicas",
"Medico", "medico", "megametro", "Memoria",
"memoria", "memorias", "metafisica", "metafisicas",
"metafisico", "microbio", "micrografo", "milicia",
"milicias", "milimetro", "mimeografo", "mimica",
"mimicas", "mimico", "minuscula", "minusculas",
"minusculo", "misera", "miseras", "misero",
"monossilaba", "monossilabas", "monossilabo", "murmurio",
"musica", "musicas", "musico", "naufrago",
"necropsia", "necropsias", "negligencias", "Negocio", "nevoa", "nevoas",
"Noticia", "Noticias", "noticias", "noveis",
"obsequio", "obstaculo", "oficio", "omega",
"oraculo", "Orbita", "Orbitas", "orbitas", "osculo", "Ovulo", "pacifico", "pagina",
"paginas", "paleofilo", "paleografo", "paleto",
"palmatoria", "palmatorias",
"pandego", "panico", "pantografo", "papeis", "paralitica",
"paraliticas", "paralitico", "Parana",
"Parodia", "parodias", "Paroquia", "paroquia",
"Paroquias", "paroquias", "pascoa", "pascoas", "pasteis",
"Patio", "patio", "Patrimonio", "patrimonio",
"pavia", "pavias", "pendencia", "pendencias",
"pendulo", "penes", "penitencia", "penitencias",
"perdulario", "perfeiçoes", "Pericia", "Pericias", "pericias",
"perola", "perolas", "perpetuas", "perpetuo", "piramide", "piramides",
"pirografo", "placida", "placidas", "placido",
"plagio", "planteis", "plastica", "plasticas", "plastico",
"polemica", "polemicas", "polemico", "poligrafa",
"poligrafas", "poligrafo", "porem", "porno",
"pornos", "potencia", "potencias", "preambulo", "preçario",
"prefacio", "preferencia", "preferencias", "premio",
"prenuncio", "presagio", "presidio", "pressagio",
"principio", "prodiga", "prodigas", "prodigo",
"prologo", "pronostico", "prontuario", "propicio",
"proposito", "Prospero", "Prospera", "Prosperas", "prosperas",
"proverbio", "Proverbio", "Providencia", "providencias", "prudencia",
"prudencias", "psicanalise", "psicanalises",
"psicoanalise", "psicoanalises", "psicologa",
"psicologas", "psicologo", "Publica", "Publicas", "publicas",
"quadrupla", "quadruplas",
"quadruplo", "quarteis", "quilometro", "radio", "reciproca",
"reciprocas", "reciproco", "Referencia", "referencias",
"renuncias", "Replica", "Replicas", "replicas",
"Republica", "Republicas", "republicas", "residencia", "residencias",
"reticencia", "reticencias", "retorica", "retoricas",
"retorico", "retrograda", "retrogradas",
"retrogrado", "retrogrado", "reusa", "reusas",
"reuso", "Reverencia", "reverencias", "Revolve", "revolve", "revolves", "Revolver", "Revolveres",
"revolveres", "rodizio", "Romania", "rufio",
"rustica", "rusticas", "rustico", "sache",
"saches", "salario", "saltica", "salticas",
"saltico", "Secretario", "secretario", "seculo",
"Sequencia", "sequencias", "sequio", "serie", "series", "serigrafo", "serio",
"silaba", "silabas", "silabo", "Silencio", "silencio", "simbolo",
"sincope", "sincopes", "sindica",
"sindicas", "sindico", "sitio", "solida",
"solidas", "solido", "sonambula", "sonambulas",
"sonambulo", "soporifero", "subespecifica", "subespecificas",
"subespecifico", "Subsecretario", "subsecretario", "subsidio",
"substancia", "substancias",
"sudorifica", "sudorificas", "sudorifico", "sumaria",
"sumarias", "Sumario", "sumario", "sumula",
"sumulas", "suplicio", "taboa", "taboas", "tabua",
"tabuas", "Tabula", "Tabulas", "tabulas", "Telegrafo", "telegrafo",
"tendencia", "tendencias", "tipica", "tipicas",
"tipico", "tipografo", "titulo", "Titulo",
"tomografo", "toneis", "toner", "topografo", "toxica",
"toxicas", "toxico", "trafico", "tragedia",
"tragedias", "tramites", "Transito", "translucida", "translucidas", "translucido",
"Tremula", "tremulas", "Tremulo", "tremulo",
"triangulo", "tripe", "tropega",
"tropegas", "tropego", "tropico", "tumulo",
"ulcera", "ulceras", "ultima", "ultimas",
"ultimo", "unanime", "unanimes", "urticaria",
"veiculo", "vermifugo", "virgula", "virgulas",
"Vitima", "vitimas",
"vitoria", "vitorias", "Vivencia", "vivencias", "voluntaria", "voluntarias",
"voluntario", "xilografo", "xoco", "zoografo"])


aaGrafia = {"abita": "habita",
"abitam": "habitam",
"abitar": "habitar",
"abitaram": "habitaram",
"abitarem": "habitarem",
"abitaria": "habitaria",
"abitariam": "habitariam",
"abitasse": "habitasse",
"abitassem": "habitassem",
"abitava": "habitava",
"abitavam": "habitavam",
"abite": "habite",
"abitei": "habitei",
"abitem": "habitem",
"abito": "habitou",
"abitou": "habitou",
"arroxa": "arrocha",
"arroxam": "arrocham",
"arroxar": "arrochar",
"arroxaram": "arrocharam",
"arroxarem": "arrocharem",
"arroxasse": "arrochasse",
"arroxassem": "arrochassem",
"arroxava": "arrochava",
"arroxavam": "arrochavam",
"arroxe": "arroche",
"arroxei": "arrochei",
"arroxem": "arrochem",
"arroxo": "arrocho",
"arroxou": "arrochou",
"asar": "azar",
"Asar": "Azar",
"asia": "azia",
"Asia":"Ásia",
"asseita": "aceita",
"asseitam": "aceitam",
"asseitar": "aceitar",
"asseitaram": "aceitaram",
"asseitarem": "aceitarem",
"asseitasse": "aceitasse",
"asseitassem": "aceitassem",
"asseitava": "aceitava",
"asseitavam": "aceitavam",
"asseite": "aceite",
"asseitei": "aceitei",
"asseitem": "aceitem",
"asseito": "aceito",
"asseitou": "aceitou",
"asserta": "acerta",
"assertam": "acertam",
"assertar": "acertar",
"assertaram": "acertaram",
"assertarem": "acertarem",
"assertasse": "acertasse",
"assertassem": "acertassem",
"assertava": "acertava",
"assertavam": "acertavam",
"asserte": "acerte",
"assertei": "acertei",
"assertem": "acertem",
"asserto": "acerto",
"assertou": "acertou",
"aza": "asa",
"azas": "asas",
"azinha": "asinha",
"azinhas": "asinhas",
"cerra": "serra",
"cerrar": "serrar",
"cerras": "serras",
"estrema": "extrema",
"estremas": "extremas",
"estremo": "extremo",
"estremos": "extremos",
"facha": "faixa",
"fachas": "faixas",
"jia": "gia",
"jias": "gias",
"liza": "lisa",
"lizas": "lisas",
"lizo": "liso",
"mofina": "morfina",
"maz": "mas",
"níveo": "nível",
"pence": "pense",
"pences": "penses",
"rezina": "resina",
"rezinas": "resinas",
"sega": "cega",
"segas": "cegas",
"sego": "cego",
"sine": "cine",
"treno": "treino",
"trocha": "trouxa",
"trochas": "trouxas",
"Xácara": "Chácara",
"xácara": "chácara",
"xácaras": "Chácaras"}

acGrafia = set(["abita", "abitam", "abitar", "abitaram", "abitarem",
"abitaria", "abitariam", "abitasse", "abitassem", "abitava", "abitavam",
"abite", "abitei", "abitem", "abito", "abitou", "arroxa", "arroxam",
"arroxar", "arroxaram", "arroxarem", "arroxasse", "arroxassem",
"arroxava", "arroxavam", "arroxe", "arroxei", "arroxem", "arroxo",
"arroxou", "asar", "Asar", "asia", "Asia", "asseita", "asseitam",
"asseitar", "asseitaram",
"asseitarem", "asseitasse", "asseitassem", "asseitava", "asseitavam",
"asseite", "asseitei", "asseitem", "asseito", "asseitou", "asserta",
"assertam", "assertar", "assertaram", "assertarem", "assertasse",
"assertassem", "assertava", "assertavam", "asserte", "assertei",
"assertem", "asserto", "assertou", "aza",  "azas", "azinha", "azinhas",
"cerra", "cerrar", "cerras", "estrema", "estremas", "estremo", "estremos",
"facha", "fachas", "jia", "jias", "liza", "lizas", "lizo", "maz", "mofina",
"níveo", "pence", "pences", "rezina", "rezinas", "sega", "segas",
"sego", "treno", "trocha", "trochas", "Xácara",
"xácara", "xácaras"])


# palavras terminadas em s - gas, pais
varpesse = {"antepaís": "antepaíses",
"convés": "conveses",
"deus": "deuses",
"gás": "gases",
"obus": "obuses",
"país": "países",
"satanás": "satanases",
"semideus": "semideuses",
"viés": "vieses"}

# palavras terminadas em s - gas, pais
varsesse = {"antepaíses": "antepaís",
"conveses": "convés",
"deuses": "deus",
"gases": "gás",
"obuses": "obus",
"países": "país",
"satanases": "satanás",
"semideuses": "semideus",
"vieses": "viés"}

# palavras terminadas em s - gas, pais
aapesse = set(["antepaís", "convés", "deus", "gás",
"obus", "país", "satanás", "semideus", "viés"])

# palavras terminadas em s - gas, pais
aasesse = set(["antepaíses", "conveses", "deuses", "gases", "obuses",
"países", "satanases", "semideuses", "vieses"])






comCrase = set(["anterior", "anteriores", "compara", "comparada",
"comparadas", "equivale", "equivalente", "equivalentes",
"iguais", "igual", "inferior", "inferiores", "oposta",
"opostas", "paralela", "paralelas", "perpendicular", "perpendiculares", "posterior",
"posteriores", "semelhante", "semelhantes", "similar", "similares",
"superior", "superiores"])


semCrase = set(["acidente", "acidentes", "acolher","ácool",
"alguém", "algum", "alguma", "algumas", "alguns", "ambas",
"ambiente", "ambientes", "bem", "bordoadas", "braçadas",
 "cabeçadas", "cacetadas", "cada", "celular",
"celulares", "chibatadas", "convite", "convites", "crime", "crimes",
"cuja", "cujas", "Deus", "diesel", "dizem", "dois",
"ela", "elas", "ele", "eletricidade", "eles", "encher", "encolher",
"escâncaras", "escolher", "essa",
"essas", "esse", "esses", "esta", "estas", "este",
"estes", "facadas", "faz", "fez", "filme", "filmes", "fim", "fiz", "galope",
"gasolina", "granel", "homem", "homens", "Jesus", "joelhadas", "jusante",
"lápis", "lenha", "leste", "lufadas", "lume", "mais", "mal",
"males",  "marteladas", "melhor", "mim", "montante", "nome",
"ninguém", "nomes", "noroeste", "nordeste", "norte",
"nós", "oeste", "outrem", "pauladas", "pão", "perfume",
"pé", "pilha", "pior", "pontapés", "porradas", "porretadas",
"preencher", "prestações", "punhaladas", "querosene",
"recolher", "rigor", "rosas", "sangue", "sangues", "sós", "sudeste",
"sudoeste", "sul", "tem", "ti", "time", "times", "toda", "todas", "tolher",
"toque", "transporte", "transportes", "trote", "um", "uma", "vapor",
"volume", "volumes", "vós",
"vossa", "Vossa", "vossas", "Zeus"])


#femeCrase  - Palavras femininas terminadas em -o-
femeCrase = set(["acessão", "Acessão",
"acupressão", "Acupressão", "adesão", "Adesão",
"admissão", "Admissão", "adustão", "Adustão",
"agressão", "Agressão", "alusão", "Alusão",
"amarelidão", "Amarelidão", "amplidão", "Amplidão",
"animadversão", "Animadversão", "antevisão", "Antevisão",
"apreensão", "Apreensão", "aptidão", "Aptidão",
"ascensão", "Ascensão", "aspersão", "Aspersão",
"autogestão", "Autogestão", "aversão", "Aversão",
"bernardão", "Bernardão", "bioimpressão",
"Bioimpressão", "branquidão", "Branquidão",
"brusquidão", "Brusquidão", "certidão", "Certidão",
"cessão", "Cessão", "Cinco", "cinco", "circuncisão", "Circuncisão",
"cisão", "Cisão", "coalizão", "Coalizão",
"coesão", "Coesão", "colisão", "Colisão",
"combustão", "Combustão", "comichão", "Comichão",
"comissão", "Comissão", "compaixão", "Compaixão",
"compreensão", "Compreensão", "compressão", "Compressão",
"compulsão", "Compulsão", "comunhão", "Comunhão",
"concessão", "Concessão", "concisão", "Concisão",
"conclusão", "Conclusão", "concussão", "Concussão",
"conexão", "Conexão", "confissão", "Confissão",
"confusão", "Confusão", "congestão", "Congestão",
"contramão",  "Contarrazão", "contarrazão",
"contusão", "Contusão", "conversão",
"Conversão", "convexão", "Convexão", "convulsão",
"Convulsão", "corrasão", "Corrasão",
"corrosão", "Corrosão", "cosmovisão", "Cosmovisão",
"crucifixão", "Crucifixão", "Cubatão", "dão",
"Dão", "decisão", "Decisão", "defensão",
"Defensão", "deflexão", "Deflexão", "demissão",
"Demissão", "densidão", "Densidão", "depressão",
"Depressão", "derisão", "Derisão", "descensão",
"Descensão", "descompressão", "Descompressão", "desconexão",
"Desconexão", "desilusão", "Desilusão", "Deso", "DESO",
"despretensão", "Despretensão", "desunião",
"Desunião", "devassidão", "Devassidão",
"Dezoito", "dezoito", "difusão", "Difusão", "digestão",
"Digestão", "digressão", "Digressão", "dimensão",
"Dimensão", "discussão", "Discussão", "dispersão",
"Dispersão", "dissensão", "dissuasão", "Dissuasão", "distensão",
"Distensão", "diversão", "Diversão", "divisão",
"Divisão", "do", "Do", "dos",
"Dos", "eclosão", "Eclosão", "efusão",
"Efusão", "egestão", "Egestão", "electrocussão",
"Electrocussão", "eletrofusão", "Eletrofusão", "emersão",
"Emersão", "emissão", "Emissão", "emulsão",
"Emulsão", "então", "entrevisão", "Entrevisão", "erosão",
"Erosão", "escansão", "Escansão",
"escravidão", "Escravidão", "escrupulidão",
"Escrupulidão", "escuridão", "Escuridão", "espessidão",
"Espessidão", "evasão", "Evasão", "Ex-prisão",
"exatidão", "Exatidão", "exaustão", "Exaustão",
"excisão", "Excisão", "exclusão", "Exclusão",
"excomunhão", "Excomunhão", "excursão", "Excursão",
"expansão", "Expansão", "explosão", "Explosão",
"expressão", "Expressão", "expulsão", "Expulsão",
"extensão", "Extensão", "extorsão", "Extorsão",
"extroversão", "Extroversão", "extrusão", "Extrusão",
"fão", "Fão", "fissão", "Fissão",
"flexão", "Flexão", "fonofoto", "Fonofoto", "foto", "Foto",
"fotoemissão", "Fotoemissão", "fotofissão", "Fotofissão",
"fotos", "Fotos", "Frazão", "fresquidão",
"Fresquidão", "frouxidão", "Frouxidão", "fusão",
"Fusão", "gestão", "Gestão", "gratidão",
"Gratidão", "hipertensão", "Hipertensão", "hipotensão",
"Hipotensão", "ilusão", "Ilusão", "imensidão",
"Imensidão", "imersão", "Imersão", "imissão",
"Imissão", "implosão", "Implosão", "imprecisão",
"Imprecisão", "impressão", "Impressão", "impulsão",
"Impulsão", "imunodepressão", "Imunodepressão", "inaptidão",
"Inaptidão", "incisão", "Incisão", "inclusão",
"Inclusão", "incompreensão", "Incompreensão", "incursão",
"Incursão", "indecisão", "Indecisão", "indigestão",
"Indigestão", "indivisão", "Indivisão", "inexatidão",
"Inexatidão", "inexpressão", "Inexpressão", "inflexão",
"Inflexão", "infusão", "Infusão", "ingestão",
"Ingestão", "ingratidão", "Ingratidão", "insubmissão",
"Insubmissão", "intensão", "Intensão", "intercessão",
"Intercessão", "interconexão", "Interconexão", "intromissão",
"Intromissão", "introversão", "Introversão", "intrusão",
"Intrusão", "invasão", "Invasão", "inversão",
"Inversão", "irreflexão", "Irreflexão", "irreligião",
"Irreligião", "irrisão", "Irrisão", "lassidão",
"Lassidão", "legião", "Legião", "lentidão",
"Lentidão", "lesão", "Lesão", "libido",
"Libido", "lipo", "Lipo", "lipos", "Lipos", "macro", "Macro", "macrorregião",
"Macrorregião", "macros", "Macros",
"mansão", "Mansão", "mansidão",
"Mansidão", "mão", "Mão", "mãos",
"Mãos", "micro", "Micro", "microrregião", "Microrregião", "micros", "Micros",
"missão", "Missão", "mornidão",
"Mornidão", "moto", "Moto", "motos", "Motos", "motos",
"multidão", "Multidão", "negridão",
"Negridão", "obsessão", "Obsessão", "ocasião",
"Ocasião", "oclusão", "Oclusão", "oito", "Oito", "omissão",
"Omissão", "opinião", "Opinião", "opressão",
"Opressão", "paixão", "Paixão", "pensão",
"Pensão", "percussão", "Percussão", "perdigão",
"Perdigão", "perfusão", "Perfusão", "permissão",
"Permissão", "perplexão", "Perplexão", "persuasão",
"Persuasão", "perversão", "Perversão", "podridão",
"Podridão", "possessão", "Possessão", "precessão",
"Precessão", "precisão", "Precisão", "preclusão",
"Preclusão", "preensão",
"Preensão", "pressão", "Pressão", "pretensão",
"Pretensão", "pretidão", "Pretidão", "previsão",
"Previsão", "prisão", "Prisão", "procissão",
"Procissão", "profissão", "Profissão", "profusão",
"Profusão", "progressão", "Progressão", "prontidão",
"Prontidão", "propensão", "Propensão", "propulsão",
"Propulsão", "provisão", "Provisão", "pulsão",
"Pulsão", "quatro", "Quatro", "questão", "Questão",
"Rádio", "rádio", "radiodifusão",
"Radiodifusão", "radiotelevisão", "Radiotelevisão", "rasão",
"Rasão", "razão", "Razão", "readmissão",
"Readmissão", "rebelião", "Rebelião", "recensão",
"Recensão", "recessão", "Recessão", "reclusão",
"Reclusão", "reconversão", "Reconversão", "reflexão",
"Reflexão", "região", "Região", "regressão",
"Regressão", "reimpressão", "Reimpressão", "reinversão",
"Reinversão", "religião", "Religião", "remissão",
"Remissão", "repercussão", "Repercussão", "repreensão",
"Repreensão", "repressão", "Repressão", "repulsão",
"Repulsão", "rescisão", "Rescisão", "retidão",
"Retidão", "retransmissão", "Retransmissão", "retroversão",
"Retroversão", "reunião", "Reunião", "reversão",
"Reversão", "revisão", "Revisão", "rezão",
"Rezão", "rouquidão", "Rouquidão", "sazão",
"sansão", "Sazão", "secessão", "Secessão", "sequidão",
"Sequidão", "servidão", "Servidão", "sessão",
"Sessão", "sobrefusão", "Sobrefusão", "sofreguidão",
"Sofreguidão", "soidão", "Soidão", "solidão",
"Solidão", "solifluxão", "Solifluxão", "suão",
"Suão", "suasão", "Suasão", "subcomissão",
"Subcomissão", "subdivisão", "Subdivisão", "submersão",
"Submersão", "submissão", "Submissão", "subpressão", "subversão",
"Subversão", "sucessão", "Sucessão", "sugestão",
"Sugestão", "superpressão", "supervisão",
"Supervisão", "supressão",
"Supressão", "suspensão", "Suspensão", "televisão",
"Televisão", "tensão", "Tensão", "transfusão",
"Transfusão", "transgressão", "Transgressão", "transmissão",
"Transmissão", "tribo", "Tribo", "tribos",
"Tribos", "união", "União", "vaguidão",
"Vaguidão", "vastidão", "Vastidão", "vazão",
"Vazão", "vermelhidão", "Vermelhidão", "versão",
"Versão", "visão", "Visão"])



masAoCrase = set(["artesão", "Artesão",
"blusão", "Blusão",
"brasão", "Brasão",
"cabeção", "Cabeção",
"cação", "Cação",
"calção", "Calção",
"coração", "Coração",
"cortesão", "Cortesão",
"diapasão", "Diapasão",
"faisão", "Faisão",
"garção", "Garção",
"pisão", "Pisão",
"salsão", "Salsão",
"são", "São",
"tição", "Tição",
"travessão", "Travessão"])


malHifen = set(["acondicionada", "acondicionadas", "acondicionado",
"acondicionados", "aconselhada", "aconselhadas", "aconselhado",
"aconselhados", "acostumada", "acostumadas", "acostumado", "acostumados",
"adaptada", "adaptadas", "adaptado", "adaptados", "afamada", "afamadas",
"afamado", "afamados", "afortunada", "afortunadas", "afortunado", "afortunados",
"agourada", "agouradas", "agourado", "agourados", "agradecida", "agradecidas",
"agradecido", "agradecidos", "ajambrada", "ajambradas", "ajambrado",
"ajambrados", "ajeitada", "ajeitadas", "ajeitado", "ajeitados", "amada",
"amadas", "amado", "amados", "apessoada", "apessoadas", "apessoado",
"apessoados", "armada", "armadas", "armado", "armados", "arranjada",
"arranjadas", "arranjado", "arranjados", "assada", "assadas", "assado",
"assados", "assombrada", "assombradas", "assombrado", "assombrados",
"assombramento", "assombramentos", "assombro", "assombros", "aventurada",
"aventuradas", "aventurado", "aventurados", "avisada", "avisadas", "avisado",
"avisados", "educada", "educadas", "educado", "educados", "empregada",
"empregadas", "empregado", "empregado", "encarada", "encaradas", "encarado",
"encarados", "enganado", "ensinado", "entendido", "entendidos", "estar",
"executada", "executadas", "executado", "executados", "habituada",
"habituadas", "habituado", "habituados", "humorada", "humoradas", "humorado",
"humorados", "informada", "informadas", "informado", "informados",
"intencionada", "intencionadas", "intencionado", "intencionados", "limpo",
"limpos"])

malMau = set(["agouro", "aluno", "aspecto", "atendimento", "bocado",
"cheiro", "colesterol", "conselho", "costume", "desempenho", "elemento",
"estado", "exemplo", "gosto", "jeito", "momento", "motorista",
"negociante", "profissional", "resultado", "sinal", "tempo", "uso"])

# AosExcecao = excecao para os prefixos auto, super, inter, sub, pseudo etc

AosExcecao = set(["a", "agora", "ante", "antes", "ao", "aos", "as", "cada",
"censura", "ciber", "citroque", "como", "comporta", "conditione", "da", "das", "de", "depois", "do", "dois",
"dos", "duas", "durante", "e", "eis", "ela", "elas", "em", "entre", "essa", "essas",
"esse", "esses", "esta", "estas", "este", "estes", "estava", "esteve",
"estive", "examine", "extra", "extras", "feita", "feitas", "feito", "feitos", "faz",
"fez", "fica", "foi", "funciona", "giga", "gosta", "gostamos", "gostaria",
"gostava", "goste", "gosto", "gostos", "hasta", "infra",
"inter", "isso", "isto", "judice", "júdice", "junto",
"kb", "kbyte", "kbytes", "kilo", "kilobyte", "lege", "mais", "mas", "me", "mega",
"megas", "meia", "meio", "mesma", "mesmas", "mesmo", "mesmos", "minha",
"minhas", "muita", "muitas", "muito", "muitos", "na", "nas", "no", "nos",
"nossa", "nossas", "nosso", "nossos", "o", "os", "outra", "outras", "outro",
"outros", "para", "pare", "pede", "pode", "poderia", "pois", "posso",
"pouca", "poucas", "pouco", "poucos", "pra", "porque", "pros", "quaisquer",
"qual", "quando", "qualquer", "que", "quem", "quis", "quer", "saiba", "se", "seja",
"sejam", "sente", "sentem", "somente", "sua", "suas", "super", "suporta",
"supra", "tambem", "também", "tem", "ter", "tera", "terreste", "terrestes",
"teus", "teve", "tinha", "tive", "toda", "todas",
"todo", "todos", "tua", "tuas", "tudo", "uma", "umas", "usa", "usam", "uso",
"vamos", "vem", "vez", "vinha", "vossa", "vossas", "vosso",
"vossos"])


meSes = set(["janeiro", "fevereiro", "março", "abril", "maio", "junho",
"julho", "agosto", "setembro", "outubro", "novemvro", "dezembro"])

seMana = set(["segunda", "terça", "quarta", "quinta", "sexta",
"sábado", "segunda-feira", "terça-feira",
"quarta-feira", "quinta-feira", "sexta-feira"])


preAnalise = set(["a", "alguma", "breve", "com", "cuja", "da", "de", "dessa",
"desta", "em", "essa", "esta", "esta", "farei", "faria", "fariam", "faz",
"fazem", "fazemos", "fazia", "faziam", "feita", "fez", "fiz", "fizemos",
"fizeram", "minha", "muita", "na", "nessa", "nesta", "nossa", "nova", "numa",
"outra", "para", "pela", "por", "posterior", "pouca", "primeira", "profunda",
"qualquer", "segunda", "segundo", "simples", "sob", "sua", "tamanha", "tanta",
"tem", "toda", "uma"])


posAnalise = set(["da", "das", "de", "dos", "farei", "faria", "fariam", "faz",
"fazem", "fazemos", "fazia", "faziam", "fez", "fiz", "fizemos", "fizeram",
"tem"])

preEsta = set(["agora", "ainda", "como", "e", "ela", "ele", "hoje",
"já", "mas", "me", "te", "lhe", "não", "que", "quem",
"se", "também", "você"])

posEsta = set(["aberto", "aceito", "afeito", "afeto", "aflito", "agora",
"ali", "alto", "ao", "apronto", "apto", "atento", "aqui", "assim", "barato",
"benquisto", "bem", "bom", "bonito", "cada", "chato", "certo", "cheia",
"cheio", "circunscrito", "claro", "coberto", "com", "como", "completo",
"composto", "convicto", "correto", "curto", "decerto", "dentro", "deposto",
"descoberto", "descrito", "deserto", "desfeito", "disposta", "disposto",
"dito", "doente", "ela", "ele", "eleito", "em", "entre", "envolto", "enxuto",
"escrito", "esquisito", "exato", "exausto", "explícito",
"exposto", "extinto", "extraposto", "faminto", "farto", "feita", "feito",
"fora", "frito", "funcionamento", "gasto", "hoje", "imperfeito",
"implícito", "incerto", "incompleto", "indisposto", "inquieto",
"insatisfeito", "inscrito", "intato", "invicto", "isento", "isso", "isto",
"junto", "justo", "lento", "liberto", "longe", "mais", "mal", "manuscrito", "me",
"melhor", "mesmo", "meu", "morto", "muito", "na", "nas", "nisso", "nisto", "no", "nos",
"num", "numa", "o",  "obscuro", "obsoleto", "para", "perfeito", "perto",
"pior", "ponto", "portanto", "posto", "predisposto", "prescrito", "presente",
"preso", "preto", "prevista", "previsto", "pronto", "proposto", "quieto",
"recoberto", "repleto", "resoluto", "restrito", "revolto", "roto",
"satisfeito", "sendo", "seu", "solto", "sonolento", "sujeito", "tanto",
"teu", "todo", "tonto", "torto", "transposto", "tudo", "um", "visto", "vivo"])

excpreEsta = set(["el", "tod"])


excvelEsta = set(["abominá"
"admirá", "adorá", "agradá",
"amá", "apetecí", "desagradá",
"desejá", "detestá", "formidá",
"horrí", "impagá", "impossí",
"incansá", "incompará", "incompreensí",
"inesgotá", "inexorá", "inexplicá",
"inqualificá", "interminá", "invisí",
"irredutí", "lamentá", "memorá",
"miserá", "notá", "possí",
"prová", "respeitá", "responsá",
"saudá", "terrí", "variá",
"vulnerá"])


proClise = set(["abafa", "abala", "abalei", "abalou", "abana",
"abandonei", "abandonou", "abateu", "abati", "abominei", "abominou",
"aborreceu", "aborreci", "aborrecia", "abracei", "abracou", "abria",
"abroquela", "absolveu", "absolvi", "absorvia", "acabrunha", "acalma",
"acanhava", "acariciei", "acariciou", "aceita", "aceitava",
"aceitei", "aceito", "aceitou", "acendeu", "acendi", "acendia", "acerta",
"acha", "achava", "achei", "acho", "achou", "aclama", "acolheu",
"acolhi", "acolhia", "acometeu", "acometia", "acompanha",
"acompanhava", "acompanhei", "acompanhou", "aconchega", "aconchegava",
"aconselha", "aconselhava", "aconselhei", "aconselhou",
"acorda", "acordava", "acordei", "acordou", "acuda", "acusa",
"adivinha", "adivinhei", "adivinhou", "admira", "admirava", "admita",
"admitia", "adora", "adorava", "adota", "advirta", "afaga",
"afagava", "afasta", "afastava", "afastei", "afastou",
"afligia", "afoga", "afugenta", "agarra", "agarrei",
"agarrou", "agasalha", "agita", "agitava", "aguardava", "aguenta", "ajeita",
"ajuda", "ajudava", "ajudei", "ajudou", "ajusta", "alcancei",
"alegrava", "alimenta", "alimentei", "alimentou", "alucina", "ama",
"amarra", "amava", "ameacei", "amei", "amo", "amou", "amparei",
"amparou", "anima", "animava", "animou", "aniquila", "anula",
"anulou", "apaga", "apagou", "apague", "apalpa", "apalpei", "apalpou",
"apanha", "apanhei", "apanhou", "aperta", "apertava", "apertei",
"apertou", "aplaudia", "apoiei", "apoiou", "apresenta",
"apresentei", "apresentou", "aproximou", "aqueceu", "arranca", "arrancou",
"arranquei", "arrasta", "arrastava", "arrastei", "arrastou",
"arrebata", "arrebatou", "arremessa", "arremessei", "arremessou", "arruma",
"arrumei", "arrumou", "assa", "assalta", "assaltou", "assombra", "assombrou",
"assou", "assusta", "assustava", "assustou", "atacou", "atendeu", "atendi",
"aterra", "atesta", "atira", "atirei", "atirou", "atormenta",
"atormentava", "autorizei", "autorizou", "avistei", "avistou",
"baixa", "bateu", "bati", "bebeu", "bebi", "beija", "beijava",
"beijei", "beijou", "bota", "botava", "botei", "botou", "buscou", "busquei",
"cansou", "caracteriza", "carrega", "carregava", "carregou", "carreguei",
"casei", "casou", "castiga", "castigava", "castigou", "castiguei", "cega",
"censurava", "cerca", "cercou", "chama", "chamava", "chamei",
"chamou", "chegou", "cheguei", "choca", "cobria", "cola",
"colheu", "colhi", "coloca", "colocou", "coloquei", "colou",
"coma", "comeu", "comi", "comia", "comoveu", "comovi", "comovia",
"completa", "comprei", "compro", "comprou", "comunica",
"concebeu", "concebi", "condenei", "condenou", "conduza",
"confirmou", "conheceu", "conheci", "conhecia", "conquista", "consagrei",
"consagrou", "conseguira", "conservei", "conservou",
"considerava", "considerei", "considero", "considerou", "consola",
"consolava", "consolei", "consolou", "constrangia", "contaminei",
"contaminou", "contava", "contempla", "contemplava",
"contemplei", "contemplou", "continha", "contorceu", "contrariava",
"convenceu", "convenci", "converteu", "converti", "convida",
"convidava", "convidei", "convidou", "copia", "correu", "corri",
"corta", "cortou", "cria", "criei", "criou", "cumprimenta",
"cumprimentei", "cumprimentou", "dava", "decorei", "decorou", "defendeu",
"defendi", "defendia", "dei", "deita", "deitei", "deitou", "deitou", "deixa",
"deixava", "deixei", "deixo", "deixou", "derrama", "derramou",
"desconhecia", "descreveu", "descrevi", "desejava", "desejei", "desejou",
"desenhou", "desequilibrou", "despejei", "despejou", "desperta", "despertei",
"despertou", "despreza", "desprezei", "desprezou", "detesta", "detestava",
"detinha", "deu", "devolveu", "devolvi", "devora", "devorava", "devorei",
"devorou", "dilacerou", "dispensou", "ditei", "doma", "domina", "dominava",
"dominei", "dominou", "dou", "edita", "editou", "educa", "educou", "eduquei",
"elegeu", "embala", "embalava", "embalou", "embrulha", "embrulhou", "empurra",
"empurrei", "empurrou", "encaminha", "encanta", "encantava",
"encarava", "encarei", "encarou", "encarrega", "encarregou",
"encarreguei", "enchei", "encheu", "enchi", "enchia", "encontra",
"encontrava", "encontrei", "encontrou", "encoraja", "encostei",
"encostou", "endireita", "enfia", "enfureceu", "enrolei", "enrolou",
"enterrei", "enterrou", "entrega", "entregava", "entregou",
"entreguei", "entretinha", "entristeceu", "entristeci", "entristecia",
"entusiasmava", "envolveu", "envolvi", "envolvia", "enxota",
"enxotou", "ergueu", "ergui", "erguia", "escolheu", "escolhi", "esconde",
"escondeu", "escondi", "escondia", "escorri", "escuta", "escutava",
"escutei", "escutou", "esfrega", "esfregou", "esmaga", "esmagava", "esmagou",
"esmurrei", "esmurrou", "espanca", "espanta", "esperava",
"esperei", "espero", "esperou", "espreita", "esqueceu", "esqueci", "estendeu",
"estendi", "estendia", "estimula", "estimulei", "estimulo", "estimulou",
"estrangula", "estuda", "estudei", "estudo", "estudou", "evita", "evitava",
"examina", "examinava", "examinei", "examinou", "excitava",
"exorta", "expulsa", "expulsou", "expunha", "fabricou", "fabriquei", "fareja",
"fascina", "fascinava", "fecha", "fechei", "fechou", "feriu",
"fita", "fitava", "fitei", "fitou", "fixei", "fixou", "folheei",
"folheou", "forcei", "fortaleceu", "fortaleci", "furei", "furou", "ganha",
"ganhei", "ganhou", "gastei", "gastou", "guarda", "guardava",
"guardei", "guardou", "humilhava", "ignora", "imaginava", "imita",
"imitei", "imitou", "imobiliza", "imortaliza", "impedia",
"impelia", "impugna", "impunha", "incomoda", "incomodava", "indenize",
"inibia", "inquietava", "instala", "instiga", "insulta", "interdizia",
"interessa", "interpela", "interpelei", "interpelou", "interpretei",
"interpretou", "interroga", "interrogava", "interrogou", "interroguei",
"interrompeu", "interrompi", "interrompia", "invadia",
"inveja", "invejava", "invejo", "invejou", "irrita", "irritava", "irritei",
"irritou", "isola", "joga", "jogou", "joguei", "julga", "julgava",
"julgou", "julguei", "junta", "junte", "jurei", "jurou", "lancei", "larga",
"largou", "larguei", "lavei", "lavou", "leu", "leva",
"levanta", "levantei", "levantou", "levava", "levei", "levo", "levou", "li",
"lia", "liberta", "liga", "limpa", "livrei", "livrou", "magoava",
"manda", "mandava", "mandei", "mandou", "maneja", "mata",
"matei", "mato", "matou", "meteu", "meti", "modifica", "modifico",
"modificou", "monta", "mordeu", "mordi", "mordia", "mostra", "mostrava",
"mostrei", "mostrou", "moveu", "movi", "movia", "movo", "nobilita", "nomeei",
"nomeia", "nomeou", "obriga", "obrigava", "obrigou", "obriguei",
"observava", "observei", "obtinha", "odiava", "odiei", "odiou",
"ofendeu", "ofendi", "olha", "olhava", "olhei", "olho", "olhou",
"oprimia", "ordena", "ouvia", "ouvira", "paga", "pagou", "paguei",
"parta", "partilha", "passa", "passava", "passei", "passou", "pega", "pegou",
"peguei", "percebeu", "percebi", "percebia", "percorreu", "percorri",
"percorria", "perdeu", "perdi", "perseguia", "perturba", "perturbava", "pica",
"pinta", "pisa", "planta", "plantei", "plantou", "ponha", "precedeu",
"precedi", "preencheu", "preenchi", "preferia", "prenda",
"prendeu", "prendi", "prendia", "preocupa", "preocupava", "pressentia",
"previa", "procura", "procurava", "procurei", "procurou",
"promoveu", "promovi", "protegeu", "protegi", "proteja", "prova", "provoca",
"publica", "publicou", "publiquei", "pungia", "punha", "puxa",
"puxava", "puxei", "puxou", "queima", "queira", "queria", "quero", "rasga",
"rasgou", "rasguei", "realiza", "receba", "recebeu", "recebi", "recebia",
"recolheu", "recolhi", "recoloca", "recomenda", "reconheceu",
"reconheci", "reconhecia", "recorda", "recordava", "reduzia", "reencontrei",
"reencontrou", "rejeita", "relega", "releu", "reli", "remeteu", "remeti",
"remoeu", "renomeei", "renomeou", "repassa", "repelia",
"repetia", "repito", "repreendeu", "repreendi", "repreendia", "representa",
"reproduzia", "resolveu", "resolvi", "respeita", "respeitei", "respeitou",
"retarda", "retenha", "retinha", "retirei", "retirou", "retrata",
"revela", "risca", "rompeu", "rompi", "roubei", "roubou",
"sacudia", "salvei", "salvou", "saudei", "saudou", "seduzia", "seguia",
"segura", "segurei", "segurou", "senta", "sentei",
"sentia", "sentou", "servia", "siga", "sinto", "solta", "soltei",
"soltou", "submeteu", "submeti", "substitua", "suga",
"sujeita", "suplanta", "suporta", "supunha", "surpreendeu", "surpreendi",
"suspendeu", "suspendi", "sustenta", "tangeu", "tangi", "tapei", "tapou",
"temia", "tenha", "tenho", "testa", "tinha", "tira", "tirava",
"tirei", "tirou", "toca", "tocava", "tolda", "tolheu", "tolhi", "toma",
"tomava", "tomba", "tomei", "tomou", "torna", "tornava", "tornei", "tornou",
"torturava", "trabalha", "traga", "trancou", "tranquei", "tranquilizei",
"tranquilizou", "transforma", "transformei", "transformou", "trata",
"tratava", "tratei", "tratou", "trazia", "trucida", "ultrapassa",
"utiliza", "veda", "veja", "vejo", "venceu", "venci", "vencia", "venda",
"vendeu", "vendi", "verteu", "vestia", "via", "vibrou", "vinga",
"vira", "virei", "virou", "visitava", "volta"])


aaPlural = set(["Do", "do", "Meu", "meu", "No", "no", "Nosso",
"nosso", "O", "o", "Outro", "outro", "Pelo", "pelo", "Seu", "seu", "Teu", "teu", "Todo",
"todo", "Vosso", "vosso"])

# aaSingular - Palavras no singular que possam ir para o plural com simples adicao de -s- para formacao do plural-singular com a palavra seguinte
aaSingular = set(["A", "Ao", "ao",
"Aquela", "aquela", "Aquele", "aquele", "Assunto", "assunto",
"Ato", "ato", "Baixa", "baixa", "Bela", "bela", "Belo",
"belo", "Bonita", "bonita", "Bonito", "bonito",
"Certa", "certa", "Certo", "certo", "Cuja", "cuja", "Cujo", "cujo",
"Da", "da", "Daquela", "daquela", "Daquele", "daquele",
"Derradeira", "derradeira", "Derradeiro", "derradeiro", "dessa",
"Dessa", "desta", "Desta", "deste", "Deste", "determinada",
"Determinada", "determinado", "Determinado", "Diversa",
"diversa", "Diverso", "diverso", "Do", "do",
"essa", "Essa", "esse", "Esse", "esta", "Esta", "este",
"Este", "Gasto", "gasto", "Guerreira", "guerreira", "Guerreiro", "guerreiro",
"Linda", "linda", "Lindo", "lindo", "Mesma",
"Meu", "meu", "Moderna", "moderna", "Moderno", "moderno", "Minha",
"minha", "Na", "na", "Naquela", "naquela", "Naquele", "naquele",
"nessa", "Nessa", "nesse", "Nesse", "nesta", "Nesta",
"neste", "Neste", "No", "no", "Nossa", "nossa", "Nosso", "nosso",
"O", "o", "Outra", "outra", "Outro", "outro", "Pela", "pela", "Pelo", "pelo", "Pequena",
"pequena", "Pequeno", "pequeno", "Primeira", "primeira", "Primeiro", "primeiro",
"profunda", "Profunda", "profundo", "Profundo",
"Seu", "seu", "Sua", "sua", "Tanta", "tanta", "Teu", "teu", "Tua", "tua",
"Vossa", "vossa", "Vosso",
"vosso"])

# abSingular - Palavras masculinas no singular  para formacao do plural-singular com adjetivos
abSingular = set(["Ao", "ao", "Aquele", "aquele",
"Ato", "ato", "Belo", "belo", "Bonito", "bonito",
"Certo", "certo", "Cujo", "cujo", "Daquele", "daquele", "Derradeiro", "derradeiro",
"Deste", "determinado", "Determinado", "Diverso", "diverso",
"Do", "do", "esse", "Esse", "este", "Este","Gasto", "gasto",
"Grupo", "grupo", "Guerreiro", "guerreiro", "Lindo", "lindo", "Meu",
"meu", "Moderno", "moderno", "Naquele", "naquele", "nesse",
"Nesse", "neste", "Neste", "No", "no", "Nosso", "nosso",
"O", "o", "Olho", "olho", "Organismo", "organismo", "Outro", "outro",
"Pelo", "pelo", "Pequeno", "pequeno", "Primeiro", "primeiro",
"profundo", "Profundo", "Seu",
"seu", "Teu", "teu", "Vosso", "vosso"])


# acSingular - Palavras femininas no singular  para formacao do plural-singular com a palavra seguinte
acSingular = set(["A", "Aquela", "aquela", "Baixa", "baixa", "Bela",
"bela", "Bonita", "bonita", "Carga", "carga",
"Certa", "certa", "Cuja", "cuja", "Da", "da", "Daquela", "daquela", "Derradeira", "derradeira",
"Dessa", "dessa", "Desta", "desta", "determinada", "Determinada",
"Diversa", "diversa", "essa", "Essa", "esta", "Esta", "Guerreira",
"guerreira", "Linda", "linda", "Mesma", "Minha",
"minha",  "Moderna", "moderna", "Na", "na", "Naquela", "naquela", "nessa", "Nessa",  "nesta",
"Nesta", "Nossa", "nossa", "Nova", "nova", "Outra", "outra", "Pela",
"pela", "Pequena", "pequena", "Primeira", "primeira", "profunda",
"Profunda", "Sua", "sua", "Tanta", "tanta",
"Tua", "tua", "Vossa", "vossa"])


# adSingular - Idem abSingular, mas sem o No - no
adSingular = set(["Ao",
"ao", "Aquele", "aquele", "Assunto", "assunto", "Ato", "ato",
"Belo", "belo", "Bonito", "bonito",
"Cargo", "cargo", "Certo", "certo", "Cujo", "cujo", "Daquele", "daquele", "Derradeiro", "derradeiro",
"Deste", "deste", "Determinado", "determinado",
"Diverso", "diverso", "Do", "do",
"Esse", "esse", "Este", "este", "Gasto", "gasto", "Grupo", "grupo", "Guerreiro", "guerreiro",
"Lindo", "lindo", "Meu", "meu", "Moderno", "moderno", "Naquele", "naquele",
"nesse", "Nesse", "neste", "Neste", "Nosso", "nosso", "Novo",
"novo", "O", "o", "Outro", "outro",
"Pelo", "pelo", "Pequeno", "pequeno", "Primeiro", "primeiro",
"profundo", "Profundo", "Seu", "seu", "Teu", "teu",
"Vosso", "vosso"])


# aeSingular - Palavras femininas no singular  para formacao do plural-singular  - terminadas em ado ido
aeSingular = set(["Aquela", "aquela", "Baixa",
"baixa", "Bela", "bela", "Bonita", "bonita", "Certa", "certa",
"Cuja", "cuja", "Da", "da", "Daquela", "daquela", "Derradeira", "derradeira", "determinada",
"Determinada", "Diversa", "diversa", "Guerreira", "guerreira", "Linda",
"linda", "Minha", "minha",
"Moderna", "moderna", "Na", "na","Naquela", "naquela", "Nossa", "nossa", "Outra",
"outra", "Pela", "pela", "Pequena", "pequena", "pouca", "Pouca", "Primeira", "primeira",
"profunda", "Profunda",
"Sua", "sua",  "Tanta", "tanta", "Toda", "toda",
"Tua", "tua", "Vossa", "vossa"])

# agSingular - Palavras femininas no singular  para formacao do plural-singular com a palavra seguinte SEM o A
agSingular = set(["Aquela", "aquela", "Baixa", "baixa", "Bela", "bela", "Bonita", "bonita", "Carga", "carga",
"Certa", "certa", "Cuja", "cuja", "Da", "da", "Daquela", "daquela", "Derradeira", "derradeira",
"Dessa", "dessa", "Desta", "desta", "determinada", "Determinada",
"Diversa", "diversa", "essa", "Essa", "esta", "Esta", "Guerreira",
"guerreira", "Linda", "linda", "Longa", "longa", "Mesma", "Minha",
"minha",  "Moderna", "moderna", "Na", "na", "Naquela", "naquela", "nessa", "Nessa",  "nesta",
"Nesta", "Nossa", "nossa", "Outra", "outra", "Pela",
"pela", "Pequena", "pequena", "Primeira", "primeira",
"profunda", "Profunda", "Sua", "sua", "Tanta", "tanta",
"Tua", "tua", "Vossa", "vossa"])


# ahSingular - Palavras masculinas no singular iniciadas com maiusculas
ahSingular = set(["Aquele", "Baixo", "Belo", "Bonito",
"Certo", "Daquele", "Determinado", "Diverso",
"Esse", "Este", "Grande", "Grupo", "Guerreiro", "Lindo", "Meu",
"Moderno", "Naquele", "Nosso",
"O", "Outro", "Pequeno", "Primeiro", "Profundo", "Seu",
"Teu", "Vosso"])


# ajSingular - Palavras femininas no singular iniciadas com maiusculas
ajSingular = set(["A", "Aquela", "Baixa", "Bela", "Bonita",
"Certa", "Daquela", "Diversa", "Essa", "Esta", "Guerreira",
"Linda", "Minha",
"Moderna", "Naquela", "Nossa", "Outra", "Pela",
"Pequena", "Primeira", "Profunda", "Sua", "Tanta",
"Tua", "Vossa"])



# varGenero - Variacao de genero -  feminino - masculino - - - - - masculino - masculino
varGenero = {"a": "o", "A": "O","À": "Ao","à": "ao",
"alta": "alto", "Alta": "Alto", "alto": "alto", "Alto": "Alto",
"amiga": "amigo", "Amiga": "Amigo", "amigo": "amigo", "Amigo": "Amigo",
"antiga": "antigo", "Antiga": "Antigo", "antigo": "antigo", "Antigo": "Antigo",
"ao": "ao", "Ao": "Ao",
"aquela": "aquele", "Aquela": "Aquele",
"aquele": "aquele", "Aquele": "Aquele",
"artigo": "artigo", "Artigo": "Artigo",
"assunto": "assunto", "Assunto": "Assunto",
"ato": "ato", "Ato": "Ato",
"baixa": "baixo", "Baixa": "Baixo", "baixo": "baixo", "Baixo": "Baixo",
"bela": "belo", "Bela": "Belo", "belo": "belo", "Belo": "Belo",
"bonita": "bonito", "Bonita": "Bonito", "bonito": "bonito", "Bonito": "Bonito",
"carga": "cargo", "Carga": "Cargo", "cargo": "cargo", "Cargo": "Cargo",
"certa": "certo", "Certa": "Certo", "certo": "certo", "Certo": "Certo",
"cuja": "cujo", "Cuja": "Cujo",
"cujo": "cujo", "Cujo": "Cujo", "da": "do", "Da": "Do",
"daquela": "daquele", "Daquela": "Daquele",
"daquele": "daquele", "Daquele": "Daquele",
"Derradeira": "Derradeiro", "derradeira": "derradeiro",
"Derradeiro": "Derradeiro", "derradeiro": "derradeiro",
"dessa": "desse",
"Dessa": "Desse", "desse": "desse",  "Desse": "Desse", "desta": "deste",
"Desta": "Deste", "deste": "deste",  "Deste": "Deste",
"determinada": "determinado", "Determinada": "Determinado",
"determinado": "determinado",  "Determinado": "Determinado",
"Direito": "Direito", "direito": "direito",
"Diversa": "Diverso", "diversa": "diverso",
"Diverso": "Diverso", "diverso": "diverso",
"Do": "Do", "do": "do", "doi": "doi", "Doi": "Doi", "dua": "doi",
"Dua": "Doi",
"Elemento": "Elemento", "elemento": "elemento",
"Erro": "Erro", "erro": "erro",
"essa": "esse", "Essa": "Esse", "esse": "esse",
"Esse": "Esse", "esta": "este", "Esta": "Este", "este": "este",
"Este": "Este",
"Gasto": "Gasto", "gasto": "gasto",
"Grupo": "Grupo", "grupo": "grupo",
"Guerreira": "Guerreiro", "guerreira": "guerreiro",
"Guerreiro": "Guerreiro", "guerreiro": "guerreiro",
"Linda": "Lindo", "linda": "lindo",
"Lindo": "Lindo", "lindo": "lindo",
"Longa": "Longo", "longa": "longo",
"Longo": "Longo", "longo": "longo",
"mesma": "mesmo", "Mesma": "Mesmo",
"mesmo": "mesmo", "Mesmo": "Mesmo", "meu": "meu",
"Meu": "Meu", "minha": "meu", "Minha": "Meu",
"moderna": "moderno", "Moderna": "Moderno",
"moderno": "moderno", "Moderno": "Moderno", "muita": "muito",
"Muita": "Muito", "muito": "muito", "Muito": "Muito", "na": "no",
"Na": "No",
"naquela": "naquele", "Naquela": "Naquele",
"naquele": "naquele", "Naquele": "Naquele",
"nessa": "nesse", "Nessa": "Nesse", "nesse": "nesse",
"Nesse": "Nesse", "nesta": "neste", "Nesta": "Neste", "neste": "neste",
"Neste": "Neste", "no": "no", "No": "No", "nossa": "nosso",
"Nossa": "Nosso", "nosso": "nosso", "Nosso": "Nosso",
"o": "o", "O": "O",
"Olho": "Olho", "olho": "olho",
"Organismo": "Organismo", "organismo": "organismo",
 "outra": "outro", "Outra": "Outro",
"outro": "outro", "Outro": "Outro",
"pela": "pelo", "Pela": "Pelo", "pelo": "pelo", "Pelo": "Pelo",
"pequena": "pequeno", "Pequena": "Pequeno", "pequeno": "pequeno", "Pequeno": "Pequeno",
"pouca": "pouco", "Pouca": "Pouco", "pouco": "pouco", "Pouco": "Pouco",
"primeira": "primeiro", "Primeira": "Primeiro", "primeiro": "primeiro",
"Primeiro": "Primeiro","profunda": "profundo", "Profunda": "Profundo",
"profundo": "profundo", "Profundo": "Profundo",
"seu": "seu", "Seu": "Seu", "sua": "seu", "Sua": "Seu",
"Terceira": "Terceiro", "terceira": "terceiro",
"Terceiro": "Terceiro", "terceiro": "terceiro",
"teu": "teu", "Teu": "Teu", "tanta": "tanto", "Tanta": "Tanto", "toda": "todo", "Toda": "Todo", "todo": "todo",
"Todo": "Todo", "tua": "teu", "Tua": "Teu", "un": "um", "Un": "Um",
"um": "un", "Um": "Un", "uma": "um", "Uma": "Um", "vossa": "vosso",
"Vossa": "Vosso", "vosso": "vosso", "Vosso": "Vosso"}


# varGenera - Variacao de genero -  masculino - feminino - - - - feminino - feminino
varGenera = {"a": "a", "A": "A","À":"À","à":"à",
"alta": "alta", "Alta": "Alta", "alto": "alta", "Alto": "Alta",
"amiga": "amiga", "Amiga": "Amiga", "amigo": "amiga", "Amigo": "Amiga",
"antiga": "antiga", "Antiga": "Antiga", "antigo": "antiga", "Antigo": "Antiga",
"ao":"à", "Ao":"À",
"aquela": "aquela", "Aquela": "Aquela",
"aquele": "aquela", "Aquele": "Aquela",
"baixa": "baixa", "Baixa": "Baixa", "baixo": "baixa", "Baixo": "Baixa",
"bela": "bela", "Bela": "Bela", "belo": "bela", "Belo": "Bela",
"bonita": "bonita", "Bonita": "Bonita", "bonito": "bonita", "Bonito": "Bonita",
"carga": "carga", "Carga": "Carga", "cargo": "carga", "Cargo": "Carga",
"certa": "certa", "Certa": "Certa", "certo": "certa", "Certo": "Certa",
"cuja": "cuja", "Cuja": "Cuja",
"cujo": "cuja", "Cujo": "Cuja", "da": "da", "Da": "Da",
"daquela": "daquela", "Daquela": "Daquela",
"daquele": "daquela", "Daquele": "Daquela",
"Derradeira": "Derradeira", "derradeira": "derradeira",
"Derradeiro": "Derradeira", "derradeiro": "derradeira",
"dessa": "dessa",
"Dessa": "Dessa", "desse": "dessa",  "Desse": "Dessa", "desta": "desta",
"Desta": "Desta", "deste": "desta",  "Deste": "Desta",
"determinada": "determinada", "Determinada": "Determinada",
"determinado": "determinada",  "Determinado": "Determinada",
"Diversa": "Diversa", "diversa": "diversa",
"Diverso": "Diversa", "diverso": "diversa",
"do": "da", "Do": "Da", "doi": "dua", "Doi": "Dua", "dua": "dua",
"Dua": "Dua", "essa": "essa", "Essa": "Essa", "esse": "essa",
"Esse": "Essa", "esta": "esta", "Esta": "Esta", "este": "esta",
"Este": "Esta",
"Guerreira": "Guerreira", "guerreira": "guerreira",
"Guerreiro": "Guerreira", "guerreiro": "guerreira",
"Linda": "Linda", "linda": "linda",
"Lindo": "Linda", "lindo": "linda",
"Longa": "Longa", "longa": "longa",
"Longo": "Longa", "longo": "longa",
"mesma": "mesma", "Mesma": "Mesma",
"mesmo": "mesma", "Mesmo": "Mesma", "meu": "minha",
"Meu": "Minha", "minha": "minha", "Minha": "Minha",
"moderna": "moderna", "Moderna": "Moderna", "moderno": "moderna",
"Moderno": "Moderna", "muita": "muita",
"Muita": "Muita", "muito": "muita", "Muito": "Muita", "na": "na",
"Na": "Na",
"naquela": "naquela", "Naquela": "Naquela",
"naquele": "naquela", "Naquele": "Naquela",
"nessa": "nessa", "Nessa": "Nessa", "nesse": "nessa",
"Nesse": "Nessa", "nesta": "nesta", "Nesta": "Nesta", "neste": "nesta",
"Neste": "Nesta", "no": "na", "No": "Na", "nossa": "nossa",
"Nossa": "Nossa", "nosso": "nossa", "Nosso": "Nossa",
"o": "a", "O": "A", "outra": "outra", "Outra": "Outra",
"outro": "outra", "pela": "pela",
"Pela": "Pela", "pelo": "pela", "Pelo": "Pela",
"pequena": "pequena", "Pequena": "Pequena", "pequeno": "pequena", "Pequeno": "Pequena",
"pouca": "pouca", "Pouca": "Pouca", "pouco": "pouca", "Pouco": "Pouca",
"primeira": "primeira", "Primeira": "Primeira", "primeiro": "primeira",
"Primeiro": "Primeira", "profunda": "profunda", "Profunda": "Profunda",
"profundo": "profunda", "Profundo": "Profunda",
"Própria": "Própria",
"própria": "própria",
"Próprio": "Própria",
"próprio": "própria",
"Próxima": "Próxima",
"próxima": "próxima",
"Próximo": "Próxima",
"próximo": "próxima",
"seu": "sua", "Seu": "Sua", "sua": "sua", "Sua": "Sua",
"tanta": "tanta", "Tanta": "Tanta",
"Terceira": "Terceira", "terceira": "terceira",
"Terceiro": "Terceira", "terceiro": "terceira",
"teu": "tua", "Teu": "Tua", "toda": "toda", "Toda": "Toda", "todo": "toda",
"Todo": "Toda", "tua": "tua", "Tua": "Tua", "un": "uma", "Un": "Uma",
"uma": "uma", "Uma": "Uma", "vossa": "vossa", "Vossa": "Vossa",
"vosso": "vossa", "Vosso": "Vossa"}


excExiste = set(["alfere", "Ante", "ante", "Ao", "ao", "apena", "Cai", "cai", "Cao", "cao", "Da", "da", "Dela",
"dela", "Dele", "dele", "Do", "do", "mai", "meno", "Na", "na", "naquela", "naquele", "No", "no", "noutra", "noutro", "simple"])

excDa = set(["carta", "curta", "década", "descoberta", "entrada",
"furada", "oferta", "porta", "tonelada"])

excEmFrente = set(["com", "de", "em", "minha", "nossa", "pela", "sua", "vossa"])

corPais = set(["algum", "antigo", "ao", "belo", "bom", "cada",
"centésimo", "certo", "décimo", "daquele",
"desse", "deste", "distante", "do", "doce", "doutro", "esse", "este",
"extenso", "grande", "heptagésimo", "hexagésimo",
"igual", "imenso", "jovem", "maior",  "melhor", "mesmo", "meu", "naquele",
"nenhum", "nesse", "neste", "no", "nonagésimo", "nono", "nosso",
"noutro", "num", "o", "octagésimo", "oitavo", "outro",
"pelo", "pequeno", "péssimo", "pior", "poderoso",
"próximo", "primeiro",
"quadragésimo", "quarto", "quinquagésimo",
"quinto", "respectivo", "sétimo", "segundo", "semelhante",
"seu", "sexto", "tanto", "terceiro", "teu", "todo", "trigésimo",
"um","único", "vasto", "velho", "vigésimo",
"vosso"])


excRenun = set(["a", "ao", "aos","à","às", "após"])

excDenun = set(["a", "à", "alguma", "antiga", "após",
"apresenta",  "apresentar", "apresentou", "apura", "apurando", "apurar",
"aquela", "as","às", "bela", "boa", "cada", "certa", "clara",
"conforme", "confirma", "cuja", "da", "daquela", "de", "desmente",
"desmentem", "dessa", "desta", "doutra", "em", "encaminha", "encaminhar",
"encaminhou", "essa", "esta", "existe", "existia", "existindo", "falsa",
"formaliza", "formalizada", "formula", "formulada", "grande", "grave",
"há", "houve", "houver", "igual", "investiga", "investigar",
"investigou", "julga", "julgar", "maior", "mais", "mediante", "meia",
"melhor", "mesma", "minha", "naquela", "nenhuma", "nessa", "nesta", "na",
"nossa", "noutra", "nova", "numa", "ofereceu", "outra", "pela", "pequena",
"pior", "por", "possível", "preciosa", "presente", "primeira",
"próxima", "qualquer", "recebe", "recebem", "receber",
"receberam", "receberem", "recebeu", "recebia", "recebiam", "respectiva",
"segunda", "segundo", "semelhante", "sobre", "sexta", "simples", "sua",
"tanta", "tem", "terceira", "tua", "toda", "tremenda", "uma",
"única", "vossa"])


mauMal = set(["cabid", "casad", "cheiros", "comid",
"comportad", "conceit", "conceituad", "conduzid", "conservad",
"cozinhad", "criad", "cuidad", "dormid", "dotad", "falad",
"feit", "figurad", "furad", "grad", "jeitos", "mandad",
"parecid", "parid", "passad", "pecad", "pesad", "planejad",
"post", "procedid", "pront", "pronunciad",
"propíci", "proporcionad", "própri",
"satisfeit", "servid", "sortead", "sucedid",
"trajad", "trapilh", "vestid", "vist"])

excVerbo = set(["estiv", "tiv"])

excGerundio = set(["deve", "devem", "pode", "podem", "vai", "Vamos", "vamos", "vão", "Vou", "vou"])

excGquando = set(["qua", "quando"])

excPosso = set(["quaisqu", "qualqu", "qu", "regul", "retangul", "tubul"])

# Palavras - terminados com a acento agudo - excecao para conjugacao de verbos Ex. babah
excvbah = set(["ag", "bab", "delir", "desacolher", "desencapar", "desvair",
"embeber", "encapar", "enfumar", "entrever", "equipar", "esfumar", "esgar",
"far", "feder", "filhar", "fumar", "ganz", "goi", "il", "ir", "jang", "jant",
"jat", "jurem", "lal", "lambar", "lap", "linguar", "macer", "mai", "man",
"mang", "mar", "marc", "marimb", "mascar", "mentir", "mir", "mur", "nan",
"oc", "Par", "paspalh", "pati", "pu", "reger", "remir", "revir",
"vir"])

# Palavras - terminados com a acento agudo - excecao para conjugacao de verbos Ex. bebeh
excvbeh = set(["anted", "beb", "desd", "trem"])

# Palavras - terminados com a acento agudo - excecao para conjugacao de verbos Ex. abeceh
excvbceh = set(["abe"])



# Substantivos - adjetivos confundidos com conjugacao verbal - Ex. abalo
excvbar = set(["abafáveis", "abafo", "abaixáveis", "abaixo", "abaláveis", "abalizáveis",
"abalo", "abano", "abarcáveis", "abarrotáveis", "abelhais", "abelhudo",
"abençoáveis", "abesto", "abjudicáveis", "abomináveis", "abonáveis", "abono",
"abortáveis", "aborto", "abraçáveis", "abraço", "abreviáveis", "abrigo",
"abundanciais", "abuso", "acabáveis", "acalentáveis", "acalento", "acarinháveis",
"acatáveis", "acato", "aceitáveis", "aceito", "aceno", "acentuais",
"acentuáveis", "acertáveis", "acerto", "acesso", "achatáveis", "acháveis",
"achego", "acidentais", "acidentáveis", "acidificáveis", "acionais", "acionáveis",
"aclaráveis", "aclimatáveis", "aclimatizáveis", "acomodáveis", "acompanháveis", "aconchego",
"acondicionáveis", "aconselháveis", "acopláveis", "acordais", "acordáveis", "acordo",
"acostáveis", "acosto", "acreditáveis", "acrescentáveis", "acrescento", "acumuláveis",
"acuo", "acupunturais", "acusáveis", "adaptáveis", "adega", "adegas",
"adensáveis", "adiáveis", "adicionais", "adicionáveis", "aditáveis", "adivinháveis",
"adivinho", "adjetivais", "adjetiváveis", "adjetivo", "adjudicáveis", "adjunto",
"administráveis", "admiráveis", "admoestáveis", "adnominais", "adnomináveis", "adorais",
"adoráveis", "adotáveis", "adubáveis", "adubo", "adulteráveis", "adverbiais",
"adversáveis", "adverso", "aeromodelo", "aeroplano", "aeroporto", "afagáveis",
"afago", "afastáveis", "afetais", "afetáveis", "afeto", "afiais",
"afiançáveis", "afiáveis", "afinais", "afinco", "afirmáveis", "afixais",
"afixáveis", "afixo", "afogo", "afronta", "afrontas", "afrontáveis",
"afugentáveis", "afundáveis", "agasalháveis", "agasalho", "agenciáveis", "agenda",
"agendas", "agilizáveis", "agitáveis", "aglomeráveis", "aglutináveis", "agraváveis",
"agravo", "agregáveis", "agrupáveis", "aguardo", "agulha", "agulhas",
"ajuntáveis", "alavanca", "alavancas", "albergáveis", "alcançáveis", "alcoolizáveis",
"alcunha", "alcunhas", "alegro", "aleijáveis", "alento", "alfabetizáveis",
"aliáveis", "alienáveis", "alimentais", "alimento", "alinhavo", "alinho",
"alisáveis", "alistáveis", "almejáveis", "almejo", "almoço", "almofada",
"almofadas", "alocáveis", "alteráveis", "alternáveis", "alterno", "altiva",
"altivas", "altivo", "alugáveis", "aluviais", "alvejáveis", "alvoroço",
"amaciáveis", "amansáveis", "amarela", "amarelas", "amarelo", "amarrotáveis",
"amasio", "amasso", "amáveis", "ambicionáveis", "ambientais", "ambientáveis",
"ameaçáveis", "ameaço", "amedrontáveis", "amiga", "amigas", "amigáveis",
"amigo", "amo", "amoldáveis", "amolgáveis", "amontoáveis", "amortizáveis",
"amostra", "amostrais", "amostras", "amostráveis", "amotináveis", "amparáveis",
"amparo", "ampliáveis", "amplificáveis", "amputáveis", "amuo", "anabolizáveis",
"analisáveis", "anarquizáveis", "andarilho", "andáveis", "andejo", "anelo",
"anexáveis", "anexo", "angelizáveis", "anglizáveis", "anilho", "animais",
"animalizáveis", "animáveis", "anodizáveis", "anojais", "anta", "antais",
"antas", "antecipáveis", "antedatáveis", "anteparo", "anteposto", "anteriorizáveis",
"anto", "anuláveis", "anunciáveis", "apagáveis", "apaixonáveis", "apalavráveis",
"apalpáveis", "apalpo", "apanho", "aparelháveis", "aparelho", "aparo",
"aparvalháveis", "apassiváveis", "apassivo", "apedrejo", "apegáveis", "apego",
"apeláveis", "apelido", "apelo", "apenáveis", "apenso", "aperfeiçoáveis",
"aperto", "apitáveis", "apito", "aplacáveis", "aplicáveis", "apoderáveis",
"apontáveis", "aportuguesáveis", "aposentáveis", "aposento", "aposta", "apostas",
"apostila", "apostilas", "aposto", "aprazáveis", "apreciáveis", "apresentáveis",
"aprimoráveis", "aprofundáveis", "aprontáveis", "apronto", "apropriáveis", "aproveitáveis",
"aproximáveis", "aprumo", "aramo", "aráveis", "arbitrais", "arbitráveis",
"arborizáveis", "arca", "arcas", "arco", "arena", "arenais",
"arenas", "arenga", "arengas", "areno", "argola", "argolas",
"argumentais", "argumento", "arisco", "arma", "armas", "armazenáveis",
"aro", "arquiteta", "arquitetas", "arquitetáveis", "arquiteto", "arquivais",
"arquivo", "arrancáveis", "arranco", "arranjáveis", "arranjo", "arrasáveis",
"arrasto", "arrebito", "arrecadáveis", "arregaço", "arreganho", "arregimentáveis",
"arregráveis", "arrematáveis", "arrepiáveis", "arrepio", "arrocho", "arrolo",
"arroto", "arrufo", "articuláveis", "asfalto", "asfixia", "asfixias",
"asneira", "asneiras", "asneiro", "aspiráveis", "assalariáveis", "assaltáveis",
"assalto", "assanháveis", "assediáveis", "asseguráveis", "assento", "asserto",
"assexualizáveis", "assimiláveis", "assinaláveis", "assináveis", "assoalho", "assobio",
"associais", "associáveis", "assombro", "assopro", "assunto", "atacáveis",
"atalho", "atenta", "atentas", "atento", "atenuáveis", "aterro",
"aterrorizáveis", "atestáveis", "atináveis", "atino", "atiráveis", "ativáveis",
"ativo", "atocho", "atomizáveis", "atonais", "atordoáveis", "atormentáveis",
"atraiçoáveis", "atraso", "atrito", "atrofia", "atrofias", "atropeláveis",
"atropelo", "atuais", "atualizáveis", "atuáveis", "aturáveis", "auditáveis",
"augurais", "aumentáveis", "aumento", "autenticáveis", "autentificáveis", "automotivo",
"autoprogramáveis", "autopropagáveis", "autopsiais", "autorizáveis", "autorrevogáveis", "avaliáveis",
"avalizáveis", "avanço", "avaria", "avarias", "avariáveis", "aventura",
"aventuras", "averbáveis", "avio", "avisáveis", "aviso", "avistáveis",
"azedo", "baba", "babais", "babas", "baga", "bagas",
"bago", "bailáveis", "bainha", "bainhas", "baixa", "baixas",
"baixo", "balança", "balanças", "balanço", "baliza", "balizas",
"balizáveis", "banca", "bancais", "bancas", "banco", "banha",
"banhas", "banháveis", "banho", "bardais", "bardo", "barganha",
"barganhas", "barra", "barrais", "barras", "barreira", "barreirais",
"barreiras", "barreiro", "barrica", "barricas", "barro", "basto",
"batalha", "batalhas", "batalháveis", "beijo", "beira", "beirais",
"beiras", "beneficiais", "beneficiáveis", "berro", "bestificáveis", "bexiga",
"bexigais", "bexigas", "bifurcáveis", "bimbais", "bingo", "biodegradáveis",
"bioensaio", "bisnaga", "bisnagas", "bitola", "bitolas", "bocejo",
"bochacro", "bochecha", "bochechas", "bochecho", "bochincho", "boda",
"bodas", "bofeto", "boga", "bogas", "bojo", "bola",
"bolacha", "bolachas", "bolas", "bolchevizáveis", "bolco", "boleta",
"boletas", "boleto", "bolha", "bolhas", "bolo", "bolsa",
"bolsas", "bolso", "bomba", "bombais", "bombarda", "bombardas",
"bombas", "bombo", "bondais", "boneca", "bonecas", "boneco",
"bonho", "bonificáveis", "boquejo", "borboleta", "borboletas", "borborinho",
"borboto", "borbulha", "borbulhas", "borcelo", "borco", "borda",
"bordas", "bordo", "borra", "borraçais", "borras", "borrega",
"borregas", "borrego", "borrifo", "bosquejo", "bossa", "bossas",
"bosta", "bostais", "bostas", "bota", "botais", "botas",
"boteco", "boto", "brabo", "bradais", "bradáveis", "brado",
"brama", "bramas", "branda", "brandais", "brandas", "brando",
"branquejáveis", "brasa", "brasas", "brasilianizáveis", "brasilificáveis", "brasilizáveis",
"brava", "bravais", "bravas", "bravo", "breca", "brecas",
"brecha", "brechas", "brejeira", "brejeirais", "brejeiras", "brejeiro",
"brevetáveis", "brida", "bridas", "briga", "brigais", "brigas",
"brilho", "brinco", "brisa", "brisas", "brita", "britas",
"broca", "brocais", "brocas", "brocha", "brochas", "broco",
"brolho", "bromais", "bromo", "broncografáveis", "broto", "broxa",
"broxas", "bruma", "brumais", "brumas", "brumo", "brutalizáveis",
"brutificáveis", "bruxa", "bruxas", "bruxo", "bufarinho", "bufo",
"bugia", "bugias", "bugio", "bula", "bulas", "bulha",
"bulhas", "bumbo", "bunda", "bundas", "bunho", "buraca",
"buracas", "buraco", "burburinho", "burga", "burgas", "burgo",
"burguesa", "burguesas", "burla", "burlas", "burláveis", "burocratizáveis",
"burra", "burras", "burricais", "burrico", "burro", "buscáveis",
"buto", "buzina", "buzinas", "cabedais", "cabotino", "cabrito",
"cacarejo", "cachimbo", "cacho", "cachoeira", "cachoeiras", "cachoeiro",
"cachucho", "cacicais", "cacifo", "cacimba", "cacimbas", "cacofonia",
"cacofonias", "cadastrais", "cadastráveis", "cadastro", "cadela", "cadelas",
"cadelo", "cadenciais", "cadinho", "caducáveis", "caduco", "caibrais",
"caibro", "caixeirais", "caixeiro", "calaceiro", "calafrio", "calça",
"calças", "calcináveis", "calco", "calço", "calculáveis", "calefrio",
"calejáveis", "calemburgo", "calha", "calhas", "calibráveis", "caligrafáveis",
"calma", "calmas", "calmáveis", "calmo", "calo", "calorificáveis",
"caluniáveis", "calva", "calvas", "calvo", "camaradais", "cambaio",
"cambais", "cambalacho", "cambalhota", "cambalhotas", "cambiais", "cambiáveis",
"cambito", "caminho", "campainha", "campainhas", "campais", "campanulais",
"campeiro", "campina", "campinas", "campino", "campo", "camufláveis",
"canalizáveis", "cancela", "cancelas", "canceláveis", "cancelo", "canceráveis",
"cancerizáveis", "candiais", "candidata", "candidatas", "candidato", "candonga",
"candongas", "canela", "canelas", "caneta", "canetas", "canga",
"cangas", "canonizáveis", "cansáveis", "cantais", "cantaridais", "cantaroláveis",
"cantáveis", "canto", "canudo", "canutilho", "capa", "capacitáveis",
"capas", "capinais", "capitais", "capitalizáveis", "capituláveis", "capoeira",
"capoeirais", "capoeiras", "capoeiro", "capota", "capotas", "capotáveis",
"capricho", "captáveis", "captura", "capturas", "capturáveis", "capucho",
"caracterizáveis", "carambola", "carambolas", "caramelizáveis", "caramelo", "carango",
"caranguejais", "caranguejo", "caranguejola", "caranguejolas", "carapetais", "caraterizáveis",
"carbonato", "carbonáveis", "carboneto", "carbonilo", "carbonizáveis", "carbono",
"carboxilo", "cardais", "cardanho", "cardo", "carenais", "carepo",
"carga", "cargas", "cargo", "caricatura", "caricaturais", "caricaturas",
"caricaturáveis", "cariciáveis", "carijo", "carimbáveis", "carimbo", "carmináveis",
"carnais", "carneiro", "carpa", "carpais", "carpas", "carpento",
"carpinteirais", "carpinteiro", "carpo", "carrais", "carrapatais", "carrapato",
"carregais", "carrego", "carreta", "carretas", "carreto", "carrinho",
"carro", "carruajáveis", "carta", "cartais", "cartas", "carujo",
"caruncho", "casais", "casamento", "casáveis", "casca", "cascais",
"cascalho", "cascas", "cascata", "cascatas", "cascavelo", "casco",
"cascunho", "casmurrais", "casmurro", "caso", "caspacho", "casquilho",
"castanhola", "castanholas", "castiçais", "castiço", "castificáveis", "castigáveis",
"castigo", "catalisáveis", "catalogais", "catalogáveis", "catano", "catapulta",
"catapultas", "catecumenizáveis", "categorizáveis", "catequizáveis", "catinga", "catingais",
"catingas", "catita", "catitas", "cativo", "catracego", "catraio",
"catucáveis", "caucionais", "caucionáveis", "caudilho", "causais", "causo",
"cautela", "cautelas", "cavaco", "cavalgáveis", "caverna", "cavernais",
"cavernas", "cefalizáveis", "cega", "cegas", "cego", "ceifa",
"ceifas", "celebráveis", "celebrizáveis", "celeuma", "celeumas", "censura",
"censuras", "censuráveis", "centelha", "centelhas", "centrais", "centralizáveis",
"centráveis", "centrificáveis", "centrifugáveis", "centro", "cepilháveis", "cercais",
"cerco", "cerebelo", "cerimoniais", "cerrais", "cerro", "certa",
"certas", "certificáveis", "certo", "cerveja", "cervejas", "cessáveis",
"chabouco", "chacina", "chacinas", "chafalho", "chaga", "chagas",
"chalaça", "chalaças", "chaleira", "chaleiras", "chaleiro", "chalo",
"chambrego", "chamego", "chamuscáveis", "chanasco", "chancela", "chancelas",
"chanco", "chanfro", "changueiro", "chantoais", "chapa", "chapas",
"chapinha", "chapinhas", "chapisco", "chapodo", "chapuço", "chaquiço",
"charango", "charaviscais", "charrua", "charruas", "charuto", "chasco",
"chavascais", "checo", "chefia", "chefias", "cheiro", "chibata",
"chibatas", "chicanáveis", "chichelo", "chichorrobio", "chilido", "china",
"chinas", "chincalho", "chinela", "chinelas", "chinelo", "chino",
"chiqueiro", "chisco", "chocalho", "chocho", "choco", "choito",
"chono", "choro", "chorro", "choto", "choupais", "choupo",
"chousais", "chouso", "chucháveis", "chuço", "chumaço", "chumbo",
"churdo", "churrio", "chusma", "chusmas", "chuvenisco", "chuvilho",
"chuvinha", "chuvinhas", "chuvisco", "cianeto", "cicatrizáveis", "ciclizáveis",
"cientificáveis", "cifo", "cifra", "cifras", "cifráveis", "cigana",
"ciganas", "cigano", "cigarra", "cigarras", "cigarro", "cilha",
"cilhas", "cilindro", "cimento", "cinta", "cintas", "cinto",
"cintura", "cinturas", "cio", "cipoais", "circinais", "circuito",
"circunavegáveis", "circuncentrais", "circuncentro", "circunstanciais", "circunstanciáveis", "circunvizinha",
"circunvizinhas", "circunvizinho", "cisco", "cisma", "cismas", "citáveis",
"civilizáveis", "clamáveis", "clarificáveis", "classificáveis", "claustrais", "claustro",
"clausura", "clausurais", "clausuras", "clava", "clavas", "clavelizáveis",
"clicáveis", "climatizáveis", "clinquerizáveis", "cliváveis", "clonais", "clorais",
"cloro", "cloroderiváveis", "clorofila", "clorofilas", "clorofilo", "coabitáveis",
"coactáveis", "coacto", "coadunáveis", "coaglutináveis", "coagmentáveis", "coaguláveis",
"coalháveis", "coalho", "coalizáveis", "coanháveis", "coanho", "coaptáveis",
"coarctáveis", "coarrendáveis", "coartáveis", "coassociáveis", "coáveis", "cobalto",
"cobiça", "cobiças", "cobiçáveis", "cobráveis", "cocainizáveis", "cocais",
"cocháveis", "cochicháveis", "cochicho", "cochilo", "cocho", "coco",
"cocontratáveis", "cocontrato", "coculo", "codejo", "codemandáveis", "codificáveis",
"codilháveis", "codinho", "codizáveis", "coeducáveis", "cofiáveis", "cofinho",
"cogitáveis", "cognominais", "cognomináveis", "cogumelo", "coifáveis", "coimáveis",
"coincho", "coindicáveis", "coiniciais", "coino", "coinquináveis", "cointegráveis",
"coiquinho", "coiraçáveis", "coisa", "coisas", "coisáveis", "coisificáveis",
"coiso", "coito", "coivarais", "colaboráveis", "colacionáveis", "colapso",
"colchetáveis", "colecionáveis", "colectáveis", "colecto", "coleta", "coletas",
"coletáveis", "coletivizáveis", "colgáveis", "coligáveis", "colimáveis", "colimitáveis",
"colmaço", "colmo", "colo", "colocáveis", "colonizáveis", "coloplizáveis",
"coloquiais", "coloquiáveis", "coloráveis", "colorizáveis", "coltarizáveis", "coluna",
"colunais", "colunas", "colunáveis", "comanda", "comandas", "comando",
"comarca", "comarcas", "comarco", "combináveis", "comborçáveis", "comborço",
"combusto", "começáveis", "começo", "comediáveis", "comemoráveis", "comenda",
"comendas", "comendáveis", "comensuráveis", "comentáveis", "comento", "comerciais",
"comercializáveis", "comerciáveis", "comináveis", "coministro", "comiscáveis", "comiseráveis",
"comissionáveis", "comisturáveis", "comocionais", "comoro", "compactáveis", "compacto",
"compactuáveis", "compagináveis", "comparáveis", "comparticipáveis", "compartilháveis", "compartimentais",
"compartimentalizáveis", "compartimentáveis", "compartimento", "compassáveis", "compasso", "compatibilizáveis",
"compeço", "compegáveis", "compendiáveis", "compenetráveis", "compensáveis", "compermutáveis",
"compiláveis", "complanáveis", "complementáveis", "complemento", "completa", "completas",
"completo", "complexa", "complexas", "complexáveis", "complexifico", "complexo",
"complicáveis", "comporta", "comportas", "comportáveis", "compra", "compras",
"compráveis", "compromissais", "compromisso", "comprováveis", "compugnáveis", "compulsáveis",
"computadorizáveis", "computatorizáveis", "computáveis", "comudáveis", "comungáveis", "comunicáveis",
"comunizáveis", "comutáveis", "conapo", "conaturalizáveis", "concatenáveis", "concaváveis",
"conceito", "conceituais", "conceitualizáveis", "conceituáveis", "concelebráveis", "concentralizáveis",
"concentráveis", "conceptualizáveis", "concertáveis", "concerto", "concha", "conchas",
"conchaváveis", "conchavo", "conchegáveis", "conciliáveis", "concionais", "concitáveis",
"conclamáveis", "concordáveis", "concrecionais", "concrecionáveis", "concretáveis", "concretizáveis",
"concreto", "concriáveis", "concubina", "concubinais", "concubinas", "concubináveis",
"concubino", "conculcáveis", "conculturais", "conculturáveis", "concursais", "concursáveis",
"concurso", "concurváveis", "condecoráveis", "condenáveis", "condensáveis", "condessa",
"condessas", "condicionais", "condicionáveis", "condigno", "condimentáveis", "condimento",
"conduplicáveis", "conduta", "condutas", "conduto", "conectáveis", "conexionáveis",
"confabuláveis", "confeccionáveis", "confederais", "confederáveis", "confeitáveis", "confeito",
"conferenciais", "conferenciáveis", "confessáveis", "confesso", "confiáveis", "confidenciais",
"confidenciáveis", "configuráveis", "confinais", "confináveis", "confirmáveis", "confiscáveis",
"confisco", "conflagráveis", "conflitáveis", "conflito", "conflituais", "conformais",
"conformalizáveis", "conformáveis", "confortáveis", "conforto", "confraternais", "confraternáveis",
"confraternizáveis", "confricáveis", "confrontáveis", "confronto", "confusionais", "confutáveis",
"congalardoáveis", "congeláveis", "congemináveis", "congeneráveis", "congestionáveis", "conglobais",
"conglobáveis", "conglomeráveis", "conglutináveis", "congonhais", "congraçáveis", "congraciáveis",
"congratuláveis", "congregáveis", "conjectura", "conjecturais", "conjecturas", "conjecturáveis",
"conjetura", "conjeturais", "conjeturas", "conjeturáveis", "conjugais", "conjugáveis",
"conjuntáveis", "conjuramentáveis", "conjuramento", "conjuráveis", "conluiáveis",
"conluio", "conotáveis", "conquistáveis", "consagráveis", "consciencializáveis", "consciencizáveis",
"conscientizáveis", "conselháveis", "conselheirais", "conselheiro", "conselho", "consertáveis",
"conserto", "conserváveis", "conservo", "consideráveis", "consignáveis", "consignificáveis",
"consoantáveis", "consoáveis", "consociáveis", "consogro", "consoláveis", "consolidáveis",
"consolo", "conspurcáveis", "constatáveis", "consubstanciais", "consulta", "consultas",
"consultáveis", "consumáveis", "consumo", "contabilizáveis", "contagiáveis", "contamináveis",
"contatáveis", "contato", "contáveis", "contempláveis", "contemporizáveis", "contentáveis",
"contento", "contentorizáveis", "contestáveis", "contextuais", "contextualizáveis", "contextuáveis",
"contingenciais", "conto", "contornáveis", "contorno", "contrabando", "contraditáveis",
"contradito", "contramarco", "contrapeso", "contraponto", "contraprotesto", "contraprova",
"contraprovas", "contrariáveis", "contrarraciocináveis", "contrarreformáveis", "contrarregráveis", "contrarreplicáveis",
"contrarreptáveis", "contrarrevolta", "contrarrevoltas", "contrarrevoltáveis", "contrasseguráveis", "contrasseguro",
"contrasseláveis", "contrasselo", "contrastáveis", "contratáveis", "contrato", "contravento",
"controláveis", "controlo", "contubernais", "conumeráveis", "conurbáveis", "convalidáveis",
"convencionais", "convencionalizáveis", "convencionáveis", "conveniais", "conversa", "conversalháveis",
"conversas", "conversáveis", "converso", "convibráveis", "convidáveis", "convizinháveis",
"convizinho", "convocáveis", "convoláveis", "convulsáveis", "convulsionáveis", "convulso",
"coobáveis", "coobrigáveis", "coocupáveis", "cooficiáveis", "coonestáveis", "cooperáveis",
"cooptáveis", "coordenáveis", "coossificáveis", "copa", "copais", "coparticipáveis",
"copas", "copeira", "copeiras", "copeiráveis", "copeiro", "copejáveis",
"copeláveis", "copiáveis", "copidescáveis", "copiláveis", "copilotáveis", "copiloto",
"copiografáveis", "coplana", "coplanas", "coplano", "copo", "copuláveis",
"coqueificáveis", "corais", "coráveis", "corcováveis", "corcovo", "corisco",
"cornais", "cornificáveis", "corno", "cornudáveis", "coro", "coroa",
"coroas", "coroáveis", "corporalizáveis", "corporificáveis", "corporizáveis", "corpusculizáveis",
"correlacionais", "correlacionáveis", "correlatáveis", "correlato", "corresponsabilizáveis", "correta",
"corretas", "corretáveis", "correto", "corricáveis", "corricho", "corripo",
"corroboráveis", "corrugáveis", "corrupiáveis", "cortáveis", "cortejáveis", "cortejo",
"cortina", "cortinas", "cortináveis", "coruja", "corujas", "coruscáveis",
"coscorejáveis", "coscoro", "cosipa", "cosipas", "cosmopolizáveis", "cosseguráveis",
"cosseguro", "costela", "costelas", "costeláveis", "costumáveis", "costura",
"costuras", "costuráveis", "cota", "cotanilho", "cotas", "cotáveis",
"cotejáveis", "cotia", "cotias", "cotizáveis", "coto", "cotonáveis",
"cotoveláveis", "cotovelo", "cotulo", "cotuteláveis", "couraça", "couraças",
"couraçáveis", "courela", "courelas", "cousa", "cousas", "cousáveis",
"cousificáveis", "couto", "couvais", "covacho", "cozinháveis", "craniais",
"craniáveis", "craticuláveis", "cravais", "craváveis", "cravejáveis", "cravo",
"credenciais", "credenciáveis", "creditáveis", "cremáveis", "cremo", "cremorizáveis",
"creosotais", "creosotáveis", "crepitáveis", "crespáveis", "crespo", "crestáveis",
"cretinizáveis", "criáveis", "criminais", "criminalizáveis", "crimináveis", "criocauterizáveis",
"crioconcentráveis", "criodessecáveis", "criptanalisáveis", "criptoanalisáveis", "criptografáveis", "crisalidáveis",
"crismais", "crismáveis", "criso", "crisoláveis", "crispáveis", "cristalizáveis",
"cristianizáveis", "criticáveis", "critiquizáveis", "criváveis", "crivo", "cromatizáveis",
"cromatolisáveis", "cromáveis", "cromo", "cromofotograváveis", "cromofotolito", "cronicáveis",
"croniquizáveis", "cronizáveis", "cronogoniometráveis", "cronometráveis", "cronoprojetáveis", "cronotaquimetráveis",
"cronotelemetráveis", "crotonizáveis", "cruciais", "cruciáveis", "crucificáveis", "crucifixáveis",
"crucifixo", "cruentáveis", "cruento", "crustificáveis", "cruzáveis", "cruzo",
"cuanho", "cuba", "cubais", "cubanizáveis", "cubas", "cubáveis",
"cubicáveis", "cubificáveis", "cubo", "cucais", "cuco", "cucurbitais",
"cuidáveis", "culatra", "culatrais", "culatras", "culmináveis", "culpabilizáveis",
"culpáveis", "cultiváveis", "cultivo", "cultuais", "cultuáveis", "culturalizáveis",
"cumpliciáveis", "cumprimentáveis", "cumprimento", "cumuláveis", "cunhais", "cunháveis",
"curarizáveis", "curateláveis", "curáveis", "curetáveis", "curiais", "curioso",
"currais", "curro", "cursáveis", "curso", "curva", "curvais",
"curvas", "curváveis", "curvejo", "curvo", "custo", "custodiais",
"customizáveis", "cutiláveis", "cutinizáveis", "cutisáveis", "cutucáveis", "dana",
"danas", "dançarina", "dançarinas", "dançarino", "dançáveis", "daninho",
"dano", "dardo", "data", "datais", "datas", "datáveis",
"debilitáveis", "debuxo", "decantáveis", "decapáveis", "decentralizáveis", "decepcionáveis",
"decerto", "decifráveis", "decilitro", "decimais", "decimáveis", "declaráveis",
"declináveis", "decodificáveis", "decoráveis", "decoro", "decremento", "decretais",
"decreto", "decupáveis", "decuplicáveis", "dedais", "dedicáveis", "dedilháveis",
"dedo", "defensáveis", "deflacionáveis", "deformáveis", "defraudáveis", "defunta",
"defuntas", "defunto", "degradáveis", "degredo", "degustáveis", "dejeto",
"delatáveis", "delegáveis", "deleitáveis", "demanda", "demandas", "demandáveis",
"demarcáveis", "demasia", "demasias", "demonstráveis", "demora", "demoras",
"denegáveis", "denodáveis", "denodo", "denominais", "densa", "densas",
"denso", "dentais", "dentelo", "denunciáveis", "deparáveis", "departamentalizáveis",
"deploráveis", "depopuláveis", "deportáveis", "depreciáveis", "depresso", "deriváveis",
"derogáveis", "derramáveis", "derrancáveis", "derrapáveis", "derratizáveis", "derrengáveis",
"derribáveis", "derriçáveis", "derriscáveis", "derrocáveis", "derrogáveis", "derrota",
"derrotas", "derrotáveis", "derrubáveis", "desabafáveis", "desabafo", "desabaláveis",
"desabalizáveis", "desabalroáveis", "desabáveis", "desabilitáveis", "desabitáveis", "desabituáveis",
"desaboçáveis", "desabonáveis", "desabordáveis", "desabrigáveis", "desabrocháveis", "desabrolháveis",
"desabusáveis", "desacamáveis", "desaçamáveis", "desacampáveis", "desacanháveis", "desacasaláveis",
"desacatáveis", "desacato", "desacavaláveis", "desaceitáveis", "desaceleráveis", "desacentuáveis",
"desaceráveis", "desacerbáveis", "desacertáveis", "desacerto", "desachegáveis", "desacidificáveis",
"desaciduláveis", "desaclimatáveis", "desaclimáveis", "desacobardáveis", "desacocháveis", "desacoimáveis",
"desacoitáveis", "desacolchetáveis", "desacolchoáveis", "desacomodáveis", "desacompanháveis", "desacompassáveis",
"desaconchegáveis", "desaconselháveis", "desacopláveis", "desacorçoáveis", "desacordáveis", "desacordo",
"desacoroçoáveis", "desacorrentáveis", "desacostáveis", "desacostumáveis", "desacoutáveis", "desacovardáveis",
"desacreditáveis", "desacuáveis", "desacumuláveis", "desadaptáveis", "desadensáveis", "desadjetiváveis",
"desadministráveis", "desadmoestáveis", "desadoráveis", "desadornáveis", "desaduláveis", "desadvogáveis",
"desafagáveis", "desafaimáveis", "desafamáveis", "desafeiçoáveis", "desafeitáveis", "desafeito",
"desaferráveis", "desaferroáveis", "desaferrolháveis", "desafervoráveis", "desafetáveis", "desafeto",
"desafiáveis", "desafináveis", "desafio", "desafiveláveis", "desafixáveis", "desafogáveis",
"desafogo", "desafolháveis", "desaforáveis", "desaforo", "desaforro", "desafortunáveis",
"desafrancesáveis", "desafreguesáveis", "desafrontáveis", "desafundáveis", "desagalonáveis", "desagarráveis",
"desagasalháveis", "desagastáveis", "desaglomeráveis", "desagoniáveis", "desagouráveis", "desagouro",
"desagradáveis", "desagrado", "desagraváveis", "desagravo", "desagregáveis", "desagrilhoáveis",
"desaguçáveis", "desaguisáveis", "desairáveis", "desajeitáveis", "desajudáveis", "desajustáveis",
"desalagáveis", "desalapáveis", "desalargáveis", "desalarmáveis", "desalcançáveis", "desalegráveis",
"desaleitáveis", "desalentáveis", "desalento", "desalergizáveis", "desalfandegáveis", "desalgemáveis",
"desaliáveis", "desalicerçáveis", "desalinhaváveis", "desalinháveis", "desalinho", "desalistáveis",
"desalmáveis", "desalmejo", "desalogenáveis", "desalteráveis", "desalugáveis", "desamalgamáveis",
"desamamentáveis", "desamarráveis", "desamarrotáveis", "desamassáveis", "desamáveis", "desambicionáveis",
"desambientáveis", "desamealháveis", "desamigáveis", "desamimáveis", "desamináveis", "desamistáveis",
"desamo", "desamoedáveis", "desamoestáveis", "desamolgáveis", "desamontoáveis", "desamortalháveis",
"desamortizáveis", "desamotináveis", "desamoucáveis", "desamparáveis", "desamparo", "desamuáveis",
"desanalfabetizáveis", "desancáveis", "desancoráveis", "desandáveis", "desaneláveis", "desanexáveis",
"desangustiáveis", "desanimalizáveis", "desanimáveis", "desanimizáveis", "desaninháveis", "desaninho",
"desanistiáveis", "desanojáveis", "desanuviáveis", "desapacientáveis", "desapadrinháveis", "desapaixonáveis",
"desaparafusáveis", "desaparelháveis", "desaparentáveis", "desapartáveis", "desapavoráveis", "desapeçonháveis",
"desapegáveis", "desapego", "desapertáveis", "desapiedáveis", "desaplicáveis", "desapoderáveis",
"desapontáveis", "desapoquentáveis", "desaportuguesáveis", "desapossáveis", "desapreçáveis", "desapreciáveis",
"desapreço", "desapressáveis", "desaprimoráveis", "desapropriáveis", "desaprováveis", "desaproveitáveis",
"desaprovisionáveis", "desaprumáveis", "desapuro", "desarborizáveis", "desarejáveis", "desarestáveis",
"desargentizáveis", "desarmáveis", "desarmo", "desarmonizáveis", "desaromatizáveis", "desaromáveis",
"desarraigáveis", "desarranjáveis", "desarranjo", "desarrazoáveis", "desarredondáveis", "desarrocháveis",
"desarrolháveis", "desarrumáveis", "desarterializáveis", "desarticuláveis", "desasnáveis", "desasselvajáveis",
"desassemelháveis", "desassimiláveis", "desassisáveis", "desassociáveis", "desassombráveis", "desassombro",
"desassossegáveis", "desassossego", "desassunto", "desatabafáveis", "desatacáveis", "desatapetáveis",
"desatarraxáveis", "desatáveis", "desataviáveis", "desatemoráveis", "desatemorizáveis", "desatentáveis",
"desatento", "desaterráveis", "desatestáveis", "desatilháveis", "desatináveis", "desatino",
"desativáveis", "desativo", "desatocho", "desatoláveis", "desatordoáveis", "desatracáveis",
"desatravancáveis", "desatreláveis", "desatribuláveis", "desautoráveis", "desautorizáveis", "desavacalháveis",
"desavantajáveis", "desaventuráveis", "desaverbáveis", "desavergonháveis", "desavinho", "desazo",
"desbabelizáveis", "desbagoáveis", "desbalanço", "desbalizáveis", "desbancáveis", "desbando",
"desbaratáveis", "desbarato", "desbastardáveis", "desbastáveis", "desbatizáveis", "desbeiçáveis",
"desbenzoiláveis", "desbenzoláveis", "desbocáveis", "desbolináveis", "desbordáveis", "desborráveis",
"desbotáveis", "desbragáveis", "desbrasileiráveis", "desbraváveis", "desbravejáveis", "desbrecáveis",
"desbriáveis", "desbridáveis", "desbritanizáveis", "desbrotáveis", "desbrutalizáveis", "desbundáveis",
"desburricáveis", "desburrificáveis", "descabaçáveis", "descabeláveis", "descadeiráveis", "descafeináveis",
"descafeinizáveis", "descafelo", "descalabro", "descalavro", "descalçáveis", "descalcificáveis",
"descalço", "descaliçáveis", "descalváveis", "descamáveis", "descambáveis", "descaminháveis",
"descaminho", "descamisáveis", "descampáveis", "descaneláveis", "descansáveis", "descanso",
"descantáveis", "descapacitáveis", "descapitalizáveis", "descapotáveis", "descapsuláveis", "descaracterizáveis",
"descaraterizáveis", "descaráveis", "descarbonatáveis", "descarbonáveis", "descarbonetáveis", "descarbonizáveis",
"descarboxiláveis", "descarburáveis", "descarburizáveis", "descarga", "descargas", "descargo",
"descarnais", "descarnalizáveis", "descarnáveis", "descarnizáveis", "descaroçáveis", "descaroláveis",
"descarráveis", "descarregáveis", "descarreiráveis", "descarretáveis", "descarreto", "descarriáveis",
"descarriláveis", "descartáveis", "descartelizáveis", "descasáveis", "descascáveis", "descaso",
"descasuláveis", "descativáveis", "descatolizáveis", "descavalgáveis", "descaváveis", "descaveiráveis",
"descegáveis", "descementáveis", "descentralizáveis", "descentráveis", "descepáveis", "descerebeláveis",
"descerebráveis", "descerebrizáveis", "descerráveis", "desciclizáveis", "descimbráveis", "descimentáveis",
"descimento", "descintáveis", "descintráveis", "descivilizáveis", "descoaguláveis", "descocáveis",
"descocháveis", "descoco", "descodificáveis", "descofráveis", "descoláveis", "descolmáveis",
"descolocáveis", "descolonizáveis", "descoloráveis", "descolorizáveis", "descompartimentais", "descompartimentáveis",
"descompassáveis", "descompasso", "descompatibilizáveis", "descompensáveis", "descomplicáveis", "descompromisso",
"desconceituáveis", "desconcentráveis", "desconcerto", "desconchavo", "desconfeito", "desconfiáveis",
"desconfiguráveis", "desconfortáveis", "desconforto", "descongeláveis", "desconsagráveis", "desconsoláveis",
"desconsolo", "descontamináveis", "descontáveis", "descontentáveis", "descontextualizáveis", "desconto",
"descontratáveis", "descontroláveis", "desconversáveis", "descortináveis", "descraváveis", "descreditáveis",
"descremo", "descriminalizáveis", "descrimináveis", "descristianizáveis", "descromáveis", "descruzáveis",
"descuidáveis", "descuido", "desculpa", "desculpas", "desculpáveis", "descultiváveis",
"descultuáveis", "descunháveis", "descuráveis", "descurváveis", "desdemonizáveis", "desdenháveis",
"desdenomináveis", "desdentáveis", "desdiferenciáveis", "desdobráveis", "desdobro", "desdoiráveis",
"desdoutrináveis", "desdramatizáveis", "deseclipsáveis", "deseconomizáveis", "desedificáveis", "deseducáveis",
"desefemináveis", "deseixáveis", "desejáveis", "desejo", "deseletrizáveis", "deselitizáveis",
"desemaláveis", "desembaçáveis", "desembaciáveis", "desembainháveis", "desembaláveis", "desembalsamáveis",
"desembalsamo", "desembalsáveis", "desembandeiráveis", "desembaraçáveis", "desembaraço", "desembaralháveis",
"desembarcáveis", "desembauláveis", "desembebedáveis", "desembestáveis", "desembocáveis", "desemboláveis",
"desembolsáveis", "desembolso", "desemborcáveis", "desembotáveis", "desembrecháveis", "desembrenháveis",
"desembrulháveis", "desembruxáveis", "desembuçáveis", "desembucháveis", "desemburráveis", "desemburricáveis",
"desemoglobinizáveis", "desemolduráveis", "desempacáveis", "desempacháveis", "desempacotáveis", "desempaláveis",
"desempalháveis", "desempapáveis", "desemparceiráveis", "desemparedáveis", "desemparelháveis", "desempastáveis",
"desempasteláveis", "desempatáveis", "desempedráveis", "desempenáveis", "desempenháveis", "desempenho",
"desempeno", "desemperráveis", "desempestáveis", "desemplasto", "desemplastráveis", "desemplumáveis",
"desempoáveis", "desempoçáveis", "desempoeiráveis", "desempoláveis", "desempolgáveis", "desempolháveis",
"desempossáveis", "desempregáveis", "desemprego", "desemprenháveis", "desemproáveis", "desempunháveis",
"desemulsificáveis", "desemulsionáveis", "desenamoráveis", "desencabeçáveis", "desencabrestáveis", "desencabuláveis",
"desencachaçáveis", "desencadernáveis", "desencaixáveis", "desencaixilháveis", "desencaixotáveis", "desencalacráveis",
"desencalháveis", "desencalmáveis", "desencambáveis", "desencaminháveis", "desencampáveis", "desencanalháveis",
"desencanáveis", "desencantáveis", "desencanto", "desencapáveis", "desencapotáveis", "desencaracoláveis",
"desencarapuçáveis", "desencarceráveis", "desencargo", "desencarnáveis", "desencarquilháveis", "desencarráveis",
"desencarregáveis", "desencarreiráveis", "desencartáveis", "desencascáveis", "desencasteláveis", "desencastoáveis",
"desencaváveis", "desencavilháveis", "desenceráveis", "desencerráveis", "desencoiraçáveis", "desencoiráveis",
"desencolerizáveis", "desencomendáveis", "desencontráveis", "desencontro", "desencorajáveis", "desencordoáveis",
"desencorpáveis", "desencorporáveis", "desencostáveis", "desencouraçáveis", "desencouráveis", "desencováveis",
"desencraváveis", "desencrespáveis", "desencruáveis", "desencurváveis", "desendeusáveis", "desendividáveis",
"desenerváveis", "desenevoáveis", "desenfadáveis", "desenfardáveis", "desenfarruscáveis", "desenfartáveis",
"desenfastiáveis", "desenfeitáveis", "desenfeitiçáveis", "desenfeixáveis", "desenfermáveis", "desenferrujáveis",
"desenfestáveis", "desenfeudáveis", "desenfezáveis", "desenfiáveis", "desenfileiráveis", "desenfloráveis",
"desenfrascáveis", "desenfrecháveis", "desenfronháveis", "desenfunáveis", "desenfurnáveis", "desengaçáveis",
"desengaioláveis", "desengajáveis", "desengalfinháveis", "desenganáveis", "desengancháveis", "desengano",
"desengarrafáveis", "desengasgáveis", "desengastalháveis", "desengastáveis", "desengatáveis", "desengatilháveis",
"desengavetáveis", "desengelháveis", "desenglobáveis", "desengodáveis", "desengomáveis", "desengonçáveis",
"desengorduráveis", "desengradáveis", "desengrenáveis", "desengriláveis", "desengrossáveis", "desenguiçáveis",
"desenháveis", "desenho", "desenjauláveis", "desenleváveis", "desenlodaçáveis", "desenlodáveis",
"desenodoáveis", "desenquadráveis", "desenrascáveis", "desenredáveis", "desenregeláveis", "desenriçáveis",
"desenrijáveis", "desenroláveis", "desenroscáveis", "desenrugáveis", "desensaboáveis", "desensacáveis",
"desensanguentáveis", "desensarilháveis", "desensináveis", "desentabuáveis", "desentabuláveis", "desentaipáveis",
"desentaláveis", "desentarameláveis", "desentediáveis", "desenterráveis", "desentesáveis", "desentesoiráveis",
"desentesouráveis", "desentibiáveis", "desentoáveis", "desentocáveis", "desentortáveis", "desentralháveis",
"desentrameláveis", "desentrançáveis", "desentranháveis", "desentraváveis", "desentrelaçáveis", "desentrevero",
"desentroncáveis", "desentronizáveis", "desentulháveis", "desenturváveis", "desentusiasmáveis", "desentusiasmo",
"desenubláveis", "desenvasáveis", "desenvasilháveis", "desenvencilháveis", "desenvenenáveis", "desenvergonháveis",
"desenvernizáveis", "desenviesáveis", "desenvincilháveis", "desenviscáveis", "desenxofráveis", "desenxovalháveis",
"desequilibráveis", "desequipáveis", "desequivocáveis", "deserdáveis", "deserta", "desertas",
"desertáveis", "desertificáveis", "deserto", "desescaláveis", "desescolarizáveis", "desescudáveis",
"desespantáveis", "desespanto", "desesperançáveis", "desesperáveis", "desespero", "desessenciáveis",
"desestabilizáveis", "desestagnáveis", "desestalinizáveis", "desestanháveis", "desestimáveis", "desestimuláveis",
"desestorváveis", "desestruturáveis", "desetanizáveis", "desevangelizáveis", "desexcomungáveis", "desfabuláveis",
"desfadigáveis", "desfalcaçáveis", "desfalcáveis", "desfanatizáveis", "desfardáveis", "desfareláveis",
"desfarinháveis", "desfasáveis", "desfatalizáveis", "desfebráveis", "desfecháveis", "desfecho",
"desfenestráveis", "desferráveis", "desferrolháveis", "desferrujáveis", "desfertilizáveis", "desfiáveis",
"desfibráveis", "desfibriláveis", "desfibrináveis", "desfiguráveis", "desfiláveis", "desfileiráveis",
"desfocáveis", "desfolgáveis", "desfolháveis", "desfonologizáveis", "desforçáveis", "desforço",
"desformato", "desformáveis", "desforra", "desforras", "desforráveis", "desfortificáveis",
"desfosforáveis", "desfraldáveis", "desfraternizáveis", "desfrecháveis", "desfrisáveis", "desfrouxo",
"desfrutáveis", "desfuncionais", "desgalháveis", "desgalvanizáveis", "desgarráveis", "desgarro",
"desgasconizáveis", "desgaseificáveis", "desgasificáveis", "desgastáveis", "desgeláveis", "desgenitalizáveis",
"desgentilizáveis", "desgeometrizáveis", "desgermanizáveis", "desglicerinizáveis", "desglobulizáveis", "desgoeláveis",
"desgorduráveis", "desgorjáveis", "desgostáveis", "desgosto", "desgovernáveis", "desgoverno",
"desgraça", "desgraças", "desgraçáveis", "desgraciáveis", "desgraduais", "desgraduáveis",
"desgravidáveis", "desgravizáveis", "desgraxáveis", "desgregáveis", "desgrilhoáveis", "desgrudáveis",
"desgrumáveis", "desguardáveis", "desidealizáveis", "desidratáveis", "desidrogenizáveis", "desigualáveis",
"desilusionáveis", "desimagináveis", "desimanizáveis", "desimantáveis", "desimpregnáveis", "desimpressionáveis",
"desincentivo", "desincháveis", "desinclináveis", "desincompatibilizáveis", "desincorporáveis", "desincrustáveis",
"desindicáveis", "desindiciáveis", "desinfamáveis", "desinfectáveis", "desinfernáveis", "desinfestáveis",
"desinfetáveis", "desinficionáveis", "desinflacionais", "desinflamáveis", "desinflamo", "desinfláveis",
"desinfluenciáveis", "desinglesáveis", "desingurgitáveis", "desinjuriáveis", "desinquietáveis", "desinsetáveis",
"desinsetizáveis", "desinstitucionalizáveis", "desintegráveis", "desinteiráveis", "desinteiriçáveis", "desinteressais",
"desinteressáveis", "desinternacionalizáveis", "desinternáveis", "desintimidáveis", "desintoxicáveis", "desintricáveis",
"desintrincáveis", "desinvagináveis", "desionizáveis", "desipotecáveis", "desirmanáveis", "desislamizáveis",
"desjuntáveis", "deslacráveis", "desladrilháveis", "deslancháveis", "deslapidáveis", "deslarváveis",
"deslaudáveis", "deslavráveis", "deslealdáveis", "deslegáveis", "deslegitimáveis", "desleitáveis",
"desleixáveis", "desleixo", "deslembráveis", "desliáveis", "desligáveis", "deslignificáveis",
"deslindáveis", "deslinhificáveis", "deslisáveis", "deslivráveis", "deslizáveis", "deslocáveis",
"deslustrais", "desmaculáveis", "desmagnetizáveis", "desmaio", "desmamáveis", "desmancháveis",
"desmancho", "desmandáveis", "desmando", "desmaranháveis", "desmarcáveis", "desmarchetáveis",
"desmarcializáveis", "desmargináveis", "desmascaráveis", "desmatáveis", "desmaterializáveis", "desmazelo",
"desmedráveis", "desmelhoráveis", "desmelindráveis", "desmembráveis", "desmensuráveis", "desmesmerizáveis",
"desmesuráveis", "desmetalizáveis", "desmetodizáveis", "desmielinizáveis", "desmilitarizáveis", "desmineralizáveis",
"desmitificáveis", "desmiudáveis", "desmobiláveis", "desmobilháveis", "desmobilizáveis", "desmoderáveis",
"desmoduláveis", "desmoedáveis", "desmoitáveis", "desmoldáveis", "desmonetizáveis", "desmonopolizáveis",
"desmonotonizáveis", "desmontáveis", "desmoralizáveis", "desmorfinizáveis", "desmoronáveis", "desmotiváveis",
"desmudo", "desmultiplicáveis", "desmunhecáveis", "desmusculáveis", "desnaturais", "desnudo",
"desonesto", "desonra", "desonras", "desordenáveis", "desova", "desovas",
"despacháveis", "despacho", "despacientáveis", "despagináveis", "despaisáveis", "despalatalizáveis",
"despalatizáveis", "desparafináveis", "desparafusáveis", "desparceláveis", "desparramáveis", "desparramo",
"desparráveis", "despedaçáveis", "despegáveis", "despeito", "despejáveis", "despejo",
"despenháveis", "despenho", "desperdiçáveis", "despersonalizáveis", "desperto", "despesa",
"despesas", "despeso", "despetaláveis", "despetrecháveis", "despigmentáveis", "despintáveis",
"despistáveis", "despojáveis", "despojo", "despolarizáveis", "despolimerizáveis", "despolpáveis",
"despontáveis", "despontuais", "despontuáveis", "despopularizáveis", "despopuláveis", "desportuguesáveis",
"desposáveis", "despovoáveis", "desprecatáveis", "despregáveis", "despreocupáveis", "despreparáveis",
"despreparo", "desprestigiáveis", "despreza", "desprezas", "desprezáveis", "desprezo",
"desprivilegiáveis", "desprogramáveis", "despronunciáveis", "despropanizáveis", "desproporcionais", "desproporcionáveis",
"despropositáveis", "despropriáveis", "desprumáveis", "desprussianizáveis", "despurificáveis", "desqualificáveis",
"desquitáveis", "desramáveis", "desratizáveis", "desrecalcáveis", "desregulamento", "desremediáveis",
"desrespeito", "desresponsabilizáveis", "desromantizáveis", "desrussificáveis", "dessa", "dessacralizáveis",
"dessacráveis", "dessagráveis", "dessalgáveis", "dessalificáveis", "dessalinificáveis", "dessalinizáveis",
"dessalitráveis", "dessangráveis", "dessas", "dessecáveis", "dessedentáveis", "dessegredáveis",
"dessegregáveis", "desseguráveis", "dessemelháveis", "dessensibilizáveis", "dessepultáveis", "dessexualizáveis",
"dessexuáveis", "dessilicáveis", "dessilicificáveis", "dessimpatizáveis", "dessinaláveis", "dessintonizáveis",
"dessocáveis", "dessolidarizáveis", "dessoráveis", "dessossegáveis", "dessoterráveis", "dessudáveis",
"dessulfuráveis", "dessulfurizáveis", "destacáveis", "destalho", "destanizáveis", "destapáveis",
"destelháveis", "destelo", "destemperáveis", "destempero", "desterraplenáveis", "desterráveis",
"desterro", "destiláveis", "destináveis", "destino", "destipificáveis", "destoáveis",
"destocáveis", "destrago", "destramáveis", "destrambelháveis", "destrancáveis", "destrançáveis",
"destratáveis", "destravancáveis", "destraváveis", "destreináveis", "destribalizáveis", "destrincáveis",
"destrinçáveis", "destrincháveis", "destripáveis", "destroçáveis", "destroço", "destronáveis",
"destroncáveis", "destronizáveis", "destrunfáveis", "destuteláveis", "desultrajáveis", "desumana",
"desumanais", "desumanas", "desumanáveis", "desumanizáveis", "desumano", "desumidificáveis",
"desunificáveis", "desurbanizáveis", "desusáveis", "desuso", "desvairáveis", "desvairo",
"desvalia", "desvalias", "desvalidáveis", "desvalido", "desvaloráveis", "desvalorizáveis",
"desvariáveis", "desvario", "desvascularizáveis", "desvassaláveis", "desveláveis", "desvelejáveis",
"desvelo", "desvencilháveis", "desvendáveis", "desveneráveis", "desventráveis", "desventura",
"desventuras", "desvergonháveis", "desviáveis", "desvigoráveis", "desvigorizáveis", "desvincáveis",
"desvinculáveis", "desvio", "desvirgináveis", "desvirginizáveis", "desvirilizáveis", "desvirtuáveis",
"desvisceráveis", "desvitalizáveis", "desvitamináveis", "desvitaminizáveis", "desvitrificáveis", "desvocalizáveis",
"desvulcanizáveis", "desvulgarizáveis", "detalháveis", "detardáveis", "detectáveis", "deterioráveis",
"determináveis", "detestáveis", "detetáveis", "detonáveis", "detratáveis", "deturpáveis",
"devassa", "devassas", "devassáveis", "devasso", "devastáveis", "develáveis",
"deventráveis", "devisáveis", "devoráveis", "devota", "devotas", "devotáveis",
"devoto", "devulcanizáveis", "dezincificáveis", "diabolizáveis", "diaclasáveis", "diafanizáveis",
"diagnosticáveis", "diagonalizáveis", "diagramáveis", "dialectais", "dialectáveis", "dialecto",
"dialetais", "dialetáveis", "dialeto", "dialisáveis", "dialogais", "dialogáveis",
"diamantizáveis", "diatermocoaguláveis", "diazotizáveis", "dica", "dicas", "dicionariáveis",
"dicionarizáveis", "dicotomizáveis", "diesáveis", "dieselizáveis", "difamáveis", "diferençais",
"diferençáveis", "diferenciais", "diferenciáveis", "dificultáveis", "difratáveis", "digitais",
"digitalinizáveis", "digitalizáveis", "digitáveis", "digladiáveis", "dignáveis", "dignificáveis",
"digno", "digressáveis", "digressionáveis", "digresso", "dilaceráveis", "dilapidáveis",
"dilatáveis", "diligenciáveis", "dilucidáveis", "diluviais", "dimanáveis", "dimensionais",
"dimensionáveis", "dimidiáveis", "dinamitáveis", "dinamitizáveis", "dinamizáveis", "dinda",
"dindas", "dindo", "dinumeráveis", "diplomacia", "diplomacias", "diplomatizáveis",
"diplomáveis", "direcionais", "direcionáveis", "discais", "disciplina", "disciplinas",
"disciplináveis", "disco", "discordáveis", "discrepáveis", "discriminais", "discrimináveis",
"discursais", "discursáveis", "discurso", "diselizáveis", "disfarçáveis", "disformáveis",
"disgregáveis", "disjuntáveis", "disjunto", "disparatáveis", "disparo", "dispensáveis",
"dispersáveis", "disperso", "disputais", "disputáveis", "dissecáveis", "dissemelháveis",
"dissemináveis", "dissertáveis", "dissidiáveis", "dissimetrizáveis", "dissimiláveis", "dissimuláveis",
"dissipáveis", "dissociais", "dissociáveis", "dissonáveis", "distais", "distanciáveis",
"distratáveis", "distrato", "distroficáveis", "disturbáveis", "dita", "ditas",
"ditáveis", "dito", "ditongais", "ditongáveis", "ditongo", "diuturnizáveis",
"divagáveis", "divaricáveis", "diversificáveis", "divinizáveis", "divisa", "divisas",
"divisáveis", "diviso", "divorciais", "divorciáveis", "divulgáveis", "dizimáveis",
"doáveis", "dobra", "dobrais", "dobras", "dobráveis", "dobro",
"doçais", "doçáveis", "docilizáveis", "documentais", "documentáveis", "documento",
"doestáveis", "dogmatizáveis", "doidejáveis", "doiráveis", "doirejáveis", "dolorizáveis",
"domáveis", "domesticáveis", "domiciliáveis", "domificáveis", "domináveis", "domingais",
"domingo", "domo", "dopáveis", "dorminhoco", "dormitáveis", "dorsalizáveis",
"dosáveis", "dotais", "dotáveis", "doudejáveis", "douráveis", "dourejáveis",
"doutora", "doutorais", "doutoras", "doutoráveis", "doutrina", "doutrinais",
"doutrinas", "doutrináveis", "draga", "dragas", "drenáveis", "dreno",
"droga", "drogas", "drogáveis", "dualizáveis", "dubláveis", "ducha",
"duchais", "duchas", "ducháveis", "ductilizáveis", "dueláveis", "duelo",
"dueto", "dulcificáveis", "dupla", "duplas", "dupláveis", "duplicáveis",
"duplo", "durais", "duráveis", "duro", "duvidáveis", "echacorvo",
"eclipsais", "eclipsáveis", "eco", "ecoáveis", "ecocardiografáveis", "ecoencefalografáveis",
"ecolocalizáveis", "ecolocáveis", "ecossistematizáveis", "ectomorfizáveis", "ecumenizáveis", "edafostratigrafáveis",
"edafostratografáveis", "edematizáveis", "edificáveis", "editais", "editáveis", "edito",
"editorais", "editoráveis", "editoria", "editoriais", "editorias", "educáveis",
"efabuláveis", "efeituáveis", "efemináveis", "efeminizáveis", "eferventáveis", "efetiváveis",
"efetivo", "efetuáveis", "eficaciais", "egipcianizáveis", "egocentralizáveis", "egocentráveis",
"egoidealizáveis", "eixo", "ejectáveis", "ejecto", "ejetáveis", "elaboráveis",
"electrificáveis", "electrizáveis", "electrocutáveis", "electrodiagnosticáveis", "electrolisáveis", "elenco",
"eletrificáveis", "eletrizáveis", "eletrolisáveis", "eleváveis", "elimináveis", "elitizáveis",
"elo", "elogiais", "elogiáveis", "elogio", "elongáveis", "elucidáveis",
"emancipáveis", "embaciáveis", "embalsamo", "embaraço", "embargáveis", "embargo",
"embasbacáveis", "embelecáveis", "embelezáveis", "embira", "embiras", "emboço",
"embolizáveis", "embolsáveis", "embolso", "embrionais", "embrulháveis", "embrulho",
"emenda", "emendas", "emendáveis", "ementa", "ementas", "eminenciais",
"emocionais", "emocionáveis", "empalamo", "empanado", "empanemo", "emparamento",
"empecilho", "empeço", "empelamo", "empelo", "empenháveis", "empenho",
"empeno", "empeso", "empilháveis", "emplastro", "emplumáveis", "empolamo",
"empolgáveis", "empostemo", "empregáveis", "emprego", "empresa", "empresariais",
"empresas", "emprestado", "empupo", "empuxo", "emulsificáveis", "emulsionáveis",
"enamoráveis", "enastro", "encabreimo", "encabuláveis", "encaçapáveis", "encagaçáveis",
"encaixáveis", "encalço", "encalho", "encanho", "encantáveis", "encanto",
"encarbonizáveis", "encarceráveis", "encargo", "encarnáveis", "encartáveis", "encastráveis",
"encenáveis", "encetáveis", "enchimento", "encinho", "enclausuráveis", "encoberto",
"encoito", "encolerizáveis", "encomenda", "encomendas", "encomiáveis", "encontráveis",
"encontro", "encosto", "encouto", "enculturais", "enculturáveis", "endereçáveis",
"endereço", "endeusáveis", "endireitáveis", "endocruzáveis", "endossáveis", "endosso",
"enduro", "enfado", "enfartáveis", "enfatizáveis", "enfeitiçáveis", "enferma",
"enfermas", "enfermo", "enfocáveis", "enforro", "enfumáveis", "enfunáveis",
"enfurnáveis", "engajáveis", "engaldrapo", "engano", "engaranho", "engasgo",
"engenho", "engo", "engobo", "enguiço", "enigmatizáveis", "enjeitáveis",
"enjoo", "enlaçáveis", "enlevo", "enodáveis", "enquadráveis", "enrabáveis",
"enredo", "enrugáveis", "ensaiáveis", "ensaio", "enseio", "ensejo",
"ensináveis", "ensino", "ensosso", "entabuláveis", "entalháveis", "enterráveis",
"enterro", "entestáveis", "entoáveis", "entolho", "entonáveis", "entorno",
"entramo", "entrançáveis", "entranháveis", "entravais", "entredevoráveis", "entrefolho",
"entrelinha", "entrelinhas", "entreoculto", "entreolho", "entrevero", "entrevistáveis",
"entroco", "entroço", "entronizáveis", "entropeço", "entrudais", "entrudo",
"entubáveis", "entulho", "enturmáveis", "entusiasmáveis", "entusiasmo", "enumeráveis",
"enunciáveis", "envenenáveis", "envergáveis", "envergonháveis", "envernizáveis", "enviáveis",
"envinhadalho", "envio", "envisgo", "enxabelho", "enxada", "enxadas",
"enxalço", "enxaqueca", "enxaquecas", "enxaugo", "enxeco", "enxerto",
"enxovais", "enxurro", "epidemia", "epidemiais", "epidemias", "epiláveis",
"epilogais", "equacionais", "equacionáveis", "equalizáveis", "equidimensionais", "equifinalizáveis",
"equilibráveis", "equiparáveis", "equipáveis", "ermais", "ermo", "ero",
"erosáveis", "eroso", "erradicáveis", "erro", "erudito", "erva",
"ervais", "ervas", "ervilha", "ervilhais", "ervilhas", "esbalho",
"esbarrunto", "esbelta", "esbeltas", "esbelto", "esbirro", "esboço",
"esboroo", "esbulho", "escabelo", "escabro", "escacilho", "escafandro",
"escafelo", "escalado", "escalambro", "escaláveis", "escalavro", "escaldo",
"escalonáveis", "escalpo", "escama", "escamas", "escambo", "escambro",
"escamosais", "escamoso", "escancaráveis", "escandalizáveis", "escanhoáveis", "escaninho",
"escapáveis", "escapo", "escarafuncho", "escarambo", "escaravelho", "escardaço",
"escarmento", "escarninho", "escarpa", "escarpas", "escarrinho", "escarro",
"escassilho", "esclerosáveis", "escocho", "escoiço", "escolarizáveis", "escoldrinháveis",
"escombro", "esconjuráveis", "esconjuro", "escopeta", "escopetas", "escoriais",
"escorregáveis", "escorricho", "escorva", "escorvas", "escoucinho", "escouço",
"escova", "escovalho", "escovas", "escovinha", "escovinhas", "escravelho",
"escravizáveis", "escritura", "escriturais", "escrituras", "escrucho", "escrutáveis",
"escrutináveis", "escudáveis", "escudeiro", "escudo", "escultura", "esculturais",
"esculturas", "escuma", "escumas", "escumilha", "escumilhas", "escura",
"escuras", "escurento", "escuro", "escusa", "escusas", "escusáveis",
"escuso", "escuta", "escutas", "escutáveis", "esfareláveis", "esfarinháveis",
"esfoiro", "esforço", "esfregáveis", "esfregulho", "esfriáveis", "esfumilho",
"esfuminho", "esgalho", "esgarçáveis", "esgotáveis", "esgoto", "esgrafito",
"esgrima", "esgrimas", "esguia", "esguias", "esguicho", "esguio",
"eslavizáveis", "esmaltáveis", "esmero", "esmicho", "esmola", "esmolas",
"esmorço", "esmordicáveis", "espaçáveis", "espacejáveis", "espaciais", "espacializáveis",
"espaço", "espada", "espadais", "espadana", "espadanais", "espadanas",
"espadas", "espadela", "espadelas", "espadilha", "espadilhas", "espalda",
"espaldas", "espalhafato", "espalháveis", "espalmáveis", "espanáveis", "espancáveis",
"espanco", "espanhola", "espanholas", "espanholizáveis", "espantáveis", "espanto",
"esparramáveis", "esparramo", "esparrimo", "esparrinho", "esparro", "esparsa",
"esparsas", "esparso", "espartilho", "espasmo", "espatifáveis", "espatuláveis",
"especiais", "especializáveis", "especiáveis", "especificáveis", "espectáveis", "espelháveis",
"espelho", "esperáveis", "esperta", "espertas", "esperto", "espessa",
"espessas", "espesso", "espetáveis", "espeto", "espiáveis", "espicaçáveis",
"espiga", "espigas", "espincho", "espingarda", "espingardas", "espinha",
"espinhais", "espinhas", "espinho", "espirais", "espirita", "espiritais",
"espiritas", "espiritualizáveis", "espirro", "espoliáveis", "esponja", "esponjas",
"espora", "esporas", "esporo", "esporuláveis", "esposáveis", "esposo",
"espritais", "espuma", "espumas", "esquadra", "esquadras", "esquadria",
"esquadrias", "esquadrinha", "esquadrinhas", "esquadrinháveis", "esquadrinho", "esquadro",
"esquematizáveis", "esquentáveis", "esquerda", "esquerdas", "esquerdo", "esquiço",
"esquilo", "esquina", "esquinais", "esquinas", "esquistificáveis", "esquiva",
"esquivas", "esquivo", "estabilizáveis", "estaca", "estacais", "estacas",
"estacionais", "estacionáveis", "estadia", "estadiais", "estadias", "estafeta",
"estafetas", "estagflacionais", "estagnáveis", "estaleiro", "estalido", "estalo",
"estaminais", "estampa", "estampas", "estampáveis", "estancáveis", "estanco",
"estandardizáveis", "estanháveis", "estardalhaço", "estatizáveis", "estatuais", "esteira",
"esteiras", "esteiro", "estenografáveis", "estenosáveis", "esterco", "estereofotografáveis",
"esterificáveis", "esterilizáveis", "esternutáveis", "estero", "estesia", "estesias",
"estevais", "esticáveis", "estigmatizáveis", "estilhaçáveis", "estilhaço", "estilizáveis",
"estilo", "estimáveis", "estimuláveis", "estinho", "estio", "estioláveis",
"estipendiáveis", "estipuláveis", "estivais", "estocáveis", "estofo", "estoirais",
"estojo", "estola", "estolas", "estopa", "estopas", "estornáveis",
"estorno", "estorvo", "estourais", "estourinho", "estouro", "estraçalháveis",
"estrada", "estradais", "estradas", "estrado", "estrago", "estrais",
"estralo", "estrambelho", "estrampalho", "estrangeira", "estrangeiras", "estrangeiro",
"estranguláveis", "estranha", "estranhas", "estranháveis", "estranho", "estratificáveis",
"estrebucháveis", "estreita", "estreitas", "estreitáveis", "estreito", "estrela",
"estrelas", "estrelico", "estrelouço", "estrema", "estremadura", "estremaduras",
"estremas", "estremáveis", "estribilho", "estribo", "estriga", "estrigas",
"estrimo", "estro", "estrobilizáveis", "estroixo", "estrombo", "estrondo",
"estrumais", "estrupido", "estrutura", "estruturais", "estruturas", "estruturáveis",
"estudáveis", "estudo", "estufa", "estufas", "estultificáveis", "estupida",
"estupidas", "estuprais", "estupráveis", "estupro", "esturrinho", "eterizáveis",
"eterna", "eternais", "eternas", "eternizáveis", "eterno", "eticizáveis",
"etilais", "etiláveis", "etilo", "etiqueta", "etiquetas", "eufemizáveis",
"eva", "evacuáveis", "evagináveis", "evais", "evangelizáveis", "evaporáveis",
"evaporizáveis", "evapotranspiráveis", "evas", "evidenciais", "evidenciáveis", "evitáveis",
"evo", "evocáveis", "evolucionais", "evolucionáveis", "exageráveis", "exagero",
"exalçáveis", "exalo", "examináveis", "exaráveis", "exaspero", "exausto",
"excecionais", "excecionáveis", "excelsa", "excelsas", "excelso", "excepcionais",
"excepcionáveis", "exceptuáveis", "excetuáveis", "excitáveis", "exclamáveis", "excogitáveis",
"excomungáveis", "excomunhais", "excomunháveis", "excremento", "exculpáveis", "excurso",
"execráveis", "executáveis", "exemplarizáveis", "exemplificáveis", "exemplo", "exercitáveis",
"exoneráveis", "exoráveis", "exorbitáveis", "exorcismáveis", "exorcismo", "exorcizáveis",
"exordiais", "exornáveis", "exortáveis", "expatriáveis", "expectáveis", "experienciais",
"experimentais", "experimentáveis", "experimento", "expetáveis", "expiáveis", "expiro",
"explanáveis", "explantáveis", "explicáveis", "explicitáveis", "exploráveis", "exponenciais",
"exportáveis", "expressáveis", "expresso", "exprobráveis", "expropriáveis", "expugnáveis",
"expulsáveis", "expulso", "expurgáveis", "expurgo", "exsicáveis", "exsudáveis",
"extenuáveis", "exteriorizáveis", "extermináveis", "externa", "externas", "externo",
"extirpáveis", "extracto", "extraditáveis", "extrapoláveis", "extrato", "extravaso",
"extravio", "extrema", "extremais", "extremas", "extremáveis", "extremo",
"extricáveis", "extrofiáveis", "exultáveis", "exurbanizáveis", "fabricáveis", "faccionais",
"faceira", "faceiras", "faceiro", "faceta", "facetas", "facha",
"fachas", "facho", "facilitáveis", "facultáveis", "fada", "fadas",
"fadiga", "fadigas", "fadistais", "fado", "fagais", "fago",
"fagulha", "fagulhas", "faia", "faiais", "faias", "fajuto",
"falaço", "falado", "falco", "falho", "falo", "falsa",
"falsas", "falsificáveis", "falto", "familiarizáveis", "fanais", "fanatizáveis",
"fandango", "fanfarra", "fanfarras", "fanfarronais", "fanicho", "fanico",
"fantasia", "fantasias", "fantasmais", "fanuco", "farda", "fardas",
"fardo", "farelo", "farfalha", "farfalhas", "farinha", "farinhas",
"faro", "farpa", "farpas", "farra", "farrapo", "farras",
"farta", "fartas", "fartáveis", "farto", "fasquia", "fasquias",
"fastigiais", "fasto", "fatigáveis", "fatura", "faturas", "faturáveis",
"fava", "favais", "favas", "favo", "favorita", "favoritas",
"favorito", "faxina", "faxinas", "fechais", "fecho", "fecundáveis",
"fecundo", "fedelha", "fedelhas", "fedelho", "federais", "fedonho",
"feira", "feirais", "feiras", "feita", "feitais", "feitas",
"feitio", "feito", "feitoria", "feitorias", "felicitáveis", "felpa",
"felpas", "feltro", "feminilizáveis", "feminizáveis", "fenais", "fendrelho",
"fenestrais", "feno", "fenomenalizáveis", "feriais", "feriáveis", "fermentais",
"fermentáveis", "fermento", "ferrais", "ferramenta", "ferramentais", "ferramentas",
"ferro", "ferrolho", "fertilizáveis", "festa", "festas", "festejáveis",
"festejo", "festina", "festinas", "festino", "fiáveis", "ficha",
"fichas", "ficháveis", "figa", "figas", "figo", "figura",
"figurais", "figuras", "figuráveis", "fila", "filas", "filha",
"filhas", "filhastro", "filho", "filiais", "filiáveis", "filmáveis",
"filo", "filosofais", "filtráveis", "filtro", "fina", "finais",
"finalizáveis", "financiais", "financiáveis", "finas", "fináveis", "findáveis",
"findo", "fino", "finta", "fintas", "fio", "firma",
"firmais", "firmas", "firmáveis", "firro", "fiscalizáveis", "fissionais",
"fissionáveis", "fissura", "fissurais", "fissuras", "fistulizáveis", "fita",
"fitas", "fito", "fixáveis", "fixo", "flageláveis", "flagelo",
"flaino", "flamejáveis", "flauta", "flautas", "flavortizáveis", "flecha",
"flechais", "flechas", "flexionais", "flexionáveis", "floco", "flora",
"florais", "floras", "floresta", "florestais", "florestas", "fluidais",
"fluidificáveis", "fluidizáveis", "fluido", "flutuáveis", "foca", "focais",
"focalizáveis", "focas", "focinho", "foco", "fofa", "fofas",
"fofo", "fofoca", "fofocas", "fogacho", "foicinho", "folclorizáveis",
"foleco", "folga", "folgas", "folha", "folhais", "folhas",
"folho", "folia", "folias", "fomento", "fonocardiografáveis", "forca",
"forcas", "forja", "forjas", "forjáveis", "forjicáveis", "formais",
"formalizáveis", "formato", "formáveis", "formiga", "formigais", "formigas",
"formo", "formoláveis", "formuláveis", "forquilha", "forquilhas", "forra",
"forras", "forro", "fortaço", "fortaleza", "fortalezas", "fortificáveis",
"fortuna", "fortunas", "fosca", "foscas", "fosco", "fosfato",
"fossa", "fossas", "fosso", "fotoativo", "fotocopiáveis", "fotodissociáveis",
"fotograváveis", "fotolito", "fotomultiplicáveis", "fotossensibilizáveis", "foucinho", "fraca",
"fracas", "fracasso", "fraccionáveis", "fracionáveis", "fraco", "fragmentáveis",
"fragmento", "fralda", "fraldas", "francesa", "francesas", "frangalho",
"franja", "franjais", "franjas", "franquia", "franquias", "frasquinho",
"fraternizáveis", "fratura", "fraturas", "fraturáveis", "fraudáveis", "frauta",
"frautas", "frecha", "frechais", "frechas", "freira", "freirais",
"freiras", "frenais", "frequentáveis", "frescais", "fresco", "freso",
"frevo", "friccionais", "friccionáveis", "frigorificáveis", "frisa", "frisas",
"friso", "frito", "froco", "froixo", "frontais", "fronteira",
"fronteiras", "fronteiro", "frouxa", "frouxas", "frouxo", "frulho",
"frustráveis", "fruta", "frutais", "frutas", "frutificáveis", "fruto",
"fuça", "fuças", "fueiro", "fuga", "fugas", "fulgurais",
"fulmináveis", "fumaça", "fumaças", "fumais", "fumáveis", "fumo",
"funcionais", "funcionalizáveis", "funda", "fundais", "fundalho", "fundamentais",
"fundamentáveis", "fundamento", "fundas", "fundáveis", "fundilho", "fundo",
"funesta", "funestas", "funesto", "funfo", "fungo", "furais",
"furáveis", "furo", "furto", "fusco", "fuso", "fustigáveis",
"futricáveis", "futura", "futuras", "futuráveis", "futurizáveis", "futuro",
"fuziláveis", "gabarito", "gadanho", "gadiço", "gaia", "gaiais",
"gaias", "gaiata", "gaiatas", "gaiato", "gaio", "gaita",
"gaitas", "gaja", "gajas", "gajo", "gala", "galas",
"galgais", "galgo", "galhofa", "galhofas", "galicismo", "galimo",
"galinha", "galinhas", "galinho", "galo", "galricho", "galvanizáveis",
"gamenho", "gancho", "ganço", "gandaia", "gandaias", "gangorra",
"gangorras", "gangrena", "gangrenas", "ganháveis", "ganho", "garatuja",
"garatujas", "garfo", "gargais", "gargalejo", "gargarejo", "garimpáveis",
"garimpo", "garino", "garota", "garotais", "garotas", "garoto",
"garra", "garramo", "garrancho", "garras", "garrido", "garro",
"garrocha", "garrochas", "garrota", "garrotas", "garrucha", "garruchas",
"garrula", "garrulas", "garrulejo", "gaseificáveis", "gasificáveis", "gastáveis",
"gasto", "gata", "gatafunho", "gatais", "gatas", "gatinho",
"gato", "gatuno", "gaudero", "geba", "gebas", "gelatina",
"gelatinas", "gelatinizáveis", "geleificáveis", "gelificáveis", "gelo", "gema",
"gemas", "gemelgo", "gemináveis", "generalizáveis", "gerais", "gerenciais",
"gerenciáveis", "germana", "germanais", "germanas", "germanizáveis", "germano",
"germinais", "germináveis", "gessais", "gesso", "gestionáveis", "gesto",
"gingro", "giro", "glicerina", "glicerinas", "globalizáveis", "gloriais",
"glorificáveis", "glosáveis", "goela", "goelas", "goio", "gola",
"golas", "golfo", "golo", "goma", "gomais", "gomas",
"gomáveis", "gomo", "gongo", "gorais", "gordura", "gordurais",
"gorduras", "gorgorejo", "gorra", "gorras", "gorro", "gostáveis",
"gosto", "governais", "governáveis", "governicho", "governo", "gozáveis",
"gozo", "gracejo", "graduais", "graduáveis", "grafo", "graio",
"gralha", "gralhas", "gramais", "gramaticais", "gramaticalizáveis", "gramaticáveis",
"gramofono", "grampo", "grana", "granais", "granas", "granido",
"granito", "granizo", "gratificáveis", "graváveis", "graveto", "gravitais",
"gravitáveis", "gravura", "gravuras", "grelha", "grelhas", "grelháveis",
"grelo", "greta", "gretas", "grifo", "grila", "grilas",
"grilheta", "grilhetas", "grilo", "grimpa", "grimpas", "grinalda",
"grinaldas", "grinfo", "gripais", "grisalha", "grisalhas", "grisalho",
"grito", "grogolejo", "gromo", "grumo", "grupais", "grupo",
"guardáveis", "guarita", "guaritas", "guasca", "guascas", "guedelha",
"guedelhas", "guedelho", "guerrilha", "guerrilhas", "guida", "guidas",
"guilhotina", "guilhotinas", "guinais", "guincho", "guisa", "guisas",
"guizo", "gulosa", "gulosas", "guloseima", "guloseimas", "guloso",
"guto", "habilitáveis", "habitáveis", "habituais", "halito", "halogenáveis",
"halogenoderiváveis", "harmonizáveis", "harpa", "harpas", "heliporto", "hemaglutináveis",
"hemossedimentais", "herdáveis", "heresia", "heresias", "herniais", "heteroaglutináveis",
"heterocarregáveis", "heteroimunizáveis", "heteropolimerizáveis", "heterossugestionais", "heterossugestionáveis", "heterotransplantáveis",
"hibernais", "hibridáveis", "hibridizáveis", "hidratáveis", "hidrato", "hidrocarbonáveis",
"hidrocarbono", "hidrolisáveis", "hidroportáveis", "hidroporto", "hidroxideriváveis", "hierarquizáveis",
"hiperajustáveis", "hiperaplicáveis", "hiperativo", "hiperbolizáveis", "hipercomplexo", "hiperestruturáveis",
"hiperexploráveis", "hiperimpregnáveis", "hiperimunizáveis", "hiperindustriais", "hiperindustrializáveis", "hiperirritáveis",
"hiperocupáveis", "hiperpigmentais", "hiperpigmento", "hipersensibilizáveis", "hipertrofia", "hipertrofiais",
"hipertrofias", "hiperurbanizáveis", "hipervascularizáveis", "hipnotizáveis", "hipossensibilizáveis", "hipoteca",
"hipotecas", "hipotecáveis", "hirta", "hirtas", "hirto", "histocompatibilizáveis",
"histodiferenciais", "historiais", "historicizáveis", "holocausto", "homeotransplantáveis", "homilia",
"homilias", "homologáveis", "homopolimerizáveis", "homotransplantáveis", "honesta", "honestas",
"honesto", "honoráveis", "honra", "honras", "honráveis", "hora",
"horais", "horas", "horizontais", "horripiláveis", "horrorizáveis", "horta",
"hortas", "hortejo", "horto", "hospedais", "hospedáveis", "hostilizáveis",
"hulhificáveis", "humana", "humanais", "humanas", "humanizáveis", "humano",
"humilháveis", "iberizáveis", "ichacorvo", "idealizáveis", "identificáveis", "ideologizáveis",
"idolatráveis", "idolatrizáveis", "ignificáveis", "ignoráveis", "igualáveis", "igualizáveis",
"ilegitimáveis", "ilha", "ilhais", "ilhas", "ilibáveis", "ilimitáveis",
"ilumináveis", "ilustráveis", "iluviais", "imaginais", "imagináveis", "imanentizáveis",
"imantáveis", "imaterializáveis", "imbondo", "imediata", "imediatas", "imediato",
"imerso", "imitáveis", "imobilizáveis", "imoláveis", "imortalizáveis", "impacto",
"impermeabilizáveis", "impetráveis", "implantáveis", "implementáveis", "implemento", "implicáveis",
"imploráveis", "importáveis", "importunáveis", "importuno", "imposto", "impostura",
"imposturas", "imprecáveis", "imprecisáveis", "impreciso", "impregnáveis", "impremeditáveis",
"imprensa", "imprensas", "imprensáveis", "impressionáveis", "impronto", "impronunciáveis",
"improporcionais", "improporcionáveis", "improváveis", "improvisáveis", "improviso", "impugnáveis",
"impulsionais", "impulsivo", "impulso", "imputáveis", "imunda", "imundas",
"imundo", "imunizáveis", "imutáveis", "inabitáveis", "inadaptáveis", "inaláveis",
"inalienáveis", "inalteráveis", "inanimo", "inativo", "inaugurais", "inauguráveis",
"incapacitáveis", "incapsuláveis", "incaracterizáveis", "incenso", "incentiváveis", "incentivo",
"incerta", "incertas", "incerto", "incesto", "incidentais", "incineráveis",
"inciso", "incitáveis", "inclináveis", "incoloro", "incomodáveis", "incompatibilizáveis",
"incomunicáveis", "inconformáveis", "inconfortáveis", "inconsolidáveis", "incontestáveis", "incorporais",
"incorporáveis", "incrementais", "incrementáveis", "incremento", "increpáveis", "incrimináveis",
"incrustáveis", "incubáveis", "inculcáveis", "inculpáveis", "indagáveis", "indelimitáveis",
"indemnizáveis", "indenizáveis", "indetectáveis", "indetermináveis", "indexáveis", "indigesta",
"indigestas", "indigesto", "indigitáveis", "indigna", "indignas", "indignáveis",
"indigno", "indisciplina", "indisciplinas", "indisciplináveis", "indiscrimináveis", "individuais",
"indocumentáveis", "indultáveis", "indulto", "indumento", "industriais", "industrializáveis",
"industriáveis", "inebriáveis", "inequacionais", "inequacionáveis", "inerciais", "inespoliáveis",
"infamáveis", "infartáveis", "infarto", "infecunda", "infecundas", "infecundo",
"inferiorizáveis", "infernais", "inferno", "infertilizáveis", "infiltráveis", "infirmáveis",
"infixáveis", "inflacionais", "inflacionáveis", "inflamáveis", "inflamo", "infláveis",
"inflexionais", "inflexionáveis", "influenciais", "influenciáveis", "informais", "informatizáveis",
"infundamentáveis", "infusa", "infusas", "infuso", "inglesa", "inglesas",
"ingresso", "ingurgitáveis", "iniciais", "inicializáveis", "iniciáveis", "inigualáveis",
"inimitáveis", "inimizáveis", "injetáveis", "injustiça", "injustiças", "injustificáveis",
"inobserváveis", "inocentáveis", "inoculáveis", "inováveis", "inquieta", "inquietas",
"inquietáveis", "inquieto", "inquilina", "inquilinas", "inquilino", "inquináveis",
"insaturáveis", "insemináveis", "insensibilizáveis", "inserto", "insinuáveis", "insosso",
"inspecionáveis", "inspiráveis", "instabilizáveis", "instaláveis", "instáveis", "instigáveis",
"instiláveis", "institucionalizáveis", "instrumentais", "instrumentalizáveis", "instrumentáveis", "instrumento",
"insubordináveis", "insufláveis", "insuláveis", "insultáveis", "insulto", "insumo",
"insurrecionais", "integrais", "integralizáveis", "integráveis", "inteira", "inteiras",
"inteiráveis", "inteiriça", "inteiriças", "inteiriço", "inteiro", "intelectualizáveis",
"inteligenciáveis", "intemperáveis", "intemperizáveis", "intencionais", "intencionáveis", "intensa",
"intensas", "intensificáveis", "intensivo", "intenso", "intentáveis", "intento",
"interatuáveis", "intercaláveis", "intercambiais", "intercambiáveis", "interceptáveis", "intercomunicáveis",
"intercruzáveis", "interdigitais", "interdigitáveis", "interdimensionais", "interditáveis", "interdito",
"interessais", "interesterificáveis", "interestratificáveis", "interinfluenciáveis", "interino", "interiorizáveis",
"interjeccionais", "interjecionais", "interjectivo", "interjetivo", "intermigráveis", "interna",
"internacionalizáveis", "internais", "internas", "interno", "interoperáveis", "interpeláveis",
"interpenetráveis", "interpretáveis", "interrogáveis", "interseccionais", "intersectáveis", "intertransformáveis",
"interturbáveis", "intervalo", "intestinais", "intestino", "intimáveis", "intimidáveis",
"intituláveis", "intoxicáveis", "intranquilizáveis", "intransitivo", "introjectáveis", "introjetáveis",
"intruido", "inumáveis", "inundáveis", "inutilizáveis", "invagináveis", "invalidáveis",
"invariáveis", "invectivo", "invejáveis", "inventariais", "inventariáveis", "inventáveis",
"invento", "invernais", "invernáveis", "inverno", "investigáveis", "invetivo",
"inviabilizáveis", "invido", "invitais", "invocáveis", "involucrais", "iodáveis",
"iodeto", "iodináveis", "iodo", "ionizáveis", "iriais", "irmanáveis",
"irracionalizáveis", "irradiais", "irrealizáveis", "irreconciliáveis", "irresponsabilizáveis", "irrigáveis",
"irritáveis", "irrogáveis", "isca", "iscas", "isentáveis", "isento",
"islamizáveis", "isola", "isolas", "isoláveis", "italianizáveis", "iteráveis",
"jana", "janais", "janas", "janela", "janelas", "janelo",
"jango", "jano", "janota", "janotas", "janta", "jantas",
"japonesa", "japonesas", "japonizáveis", "jardo", "jarreta", "jarretas",
"jarundo", "jato", "jeito", "jeremiais", "joelho", "jogáveis",
"jogo", "jorro", "judia", "judias", "judiciais", "jugado",
"jugais", "jugo", "juguláveis", "julgáveis", "juncais", "junco",
"junta", "juntas", "juntáveis", "junto", "jura", "juramentáveis",
"juramento", "juras", "jurema", "juremais", "juremas", "juremo",
"juridicizáveis", "jurisdicionais", "juro", "justa", "justas", "justiça",
"justiças", "justiçáveis", "justificáveis", "justo", "juvenilizáveis", "labanotáveis",
"labelo", "labirinto", "laborais", "laboráveis", "labrego", "labresto",
"labuta", "labutas", "laca", "lacaio", "lacas", "laceráveis",
"laço", "lacramo", "lacrimais", "lacrimáveis", "lacuna", "lacunas",
"ladeira", "ladeiras", "ladeiro", "ladra", "ladrais", "ladras",
"ladrilho", "ladripo", "ladroeira", "ladroeiras", "lagrimais", "lagrimáveis",
"laivo", "laja", "lajas", "lamba", "lambarisco", "lambas",
"lameirais", "lameiro", "lamentáveis", "lamento", "laminais", "lamináveis",
"lamino", "lampa", "lampas", "lampejo", "lampo", "lançáveis",
"lanceta", "lancetas", "lancetáveis", "lanchais", "lancissondáveis", "lanço",
"lanháveis", "lanterna", "lanternas", "lapa", "lapas", "lapidáveis",
"larga", "largas", "largo", "laringografáveis", "larva", "larvais",
"larvas", "lasca", "lascas", "lassa", "lassas", "lastimáveis",
"lastro", "latejo", "latina", "latinas", "latinizáveis", "latino",
"latitudinais", "lavajo", "laváveis", "lavego", "lavoráveis", "lavoura",
"lavouras", "lavra", "lavras", "lavráveis", "lavujo", "lazeira",
"lazeiras", "leccionais", "leccionáveis", "lecionais", "lecionáveis", "legais",
"legalizáveis", "legenda", "legendas", "legiferáveis", "legisláveis", "legitimáveis",
"leigais", "leigo", "leiloáveis", "leitais", "leito", "lembo",
"lembráveis", "lengalenga", "lengalengas", "lenha", "lenhais", "lenhas",
"lenho", "lenificáveis", "lenimento", "lenta", "lentas", "lento",
"lesáveis", "lesionais", "lesma", "lesmas", "letra", "letras",
"letro", "leváveis", "lexicalizáveis", "liáveis", "libáveis", "libelo",
"liberais", "liberalizáveis", "liberto", "libra", "libras", "libráveis",
"licenciais", "licenciáveis", "licitáveis", "lida", "lidas", "lideráveis",
"lidimáveis", "lido", "ligais", "ligáveis", "lima", "limas",
"limáveis", "limitáveis", "limo", "limpáveis", "limpo", "linco",
"linda", "lindas", "lindo", "linfa", "linfangiografáveis", "linfas",
"linfoangiografáveis", "linfolisáveis", "linguarais", "linimento", "linotipo", "liofilizáveis",
"liquenificáveis", "liquidáveis", "liquidificáveis", "lismo", "lisonja", "lisonjas",
"lista", "listas", "literarizáveis", "literatáveis", "literatizáveis", "literato",
"literatura", "literaturas", "litificáveis", "litigáveis", "livráveis", "livro",
"lixo", "loáveis", "loca", "locais", "localizáveis", "locas",
"locáveis", "locucionais", "locupletáveis", "logica", "logicais", "logicas",
"logo", "lográveis", "loira", "loiras", "loiro", "loisa",
"loisas", "lombilho", "lombriga", "lombrigas", "lorota", "lorotas",
"loto", "loura", "louras", "louro", "lousa", "lousas",
"louváveis", "lubrificáveis", "lucila", "lucilas", "lucro", "lucubráveis",
"ludibriáveis", "lupa", "lupas", "lupo", "lusco", "lusitanizáveis",
"lustrais", "lustráveis", "lustro", "lutáveis", "luteinizáveis", "luto",
"luxo", "maça", "macaca", "macacais", "macacas", "macaco",
"maçais", "maçaneta", "maçanetas", "maçaroca", "maçarocas", "maças",
"macavenco", "maceráveis", "machado", "machiais", "machucáveis", "macia",
"macias", "macio", "maço", "maconha", "maconhas", "macrossegmento",
"macuco", "maculáveis", "madeira", "madeirais", "madeiras", "madeiráveis",
"madeiro", "madeixa", "madeixas", "madraçais", "madraceiro", "madurais",
"maduráveis", "maduro", "magia", "magias", "magica", "magicas",
"magnetizáveis", "magnifica", "magnificas", "magnificáveis", "magoáveis", "magulho",
"maio", "majorais", "majoráveis", "mala", "malabarizáveis", "malandro",
"malas", "malbaratáveis", "maleabilizáveis", "malha", "malhais", "malhas",
"malho", "maliciáveis", "maligna", "malignas", "malignizáveis", "maligno",
"maloca", "malocas", "malográveis", "malogro", "malseguro", "malsináveis",
"malta", "maltas", "maltoso", "maluca", "malucas", "maluco",
"malversáveis", "mama", "mamais", "mamas", "mana", "manas",
"mancais", "manco", "mancomunáveis", "mandáveis", "mandinga", "mandingas",
"mando", "manduca", "manducas", "manducáveis", "maneira", "maneiras",
"maneiráveis", "manejáveis", "manejo", "manga", "mangaba", "mangabais",
"mangabas", "mangais", "mangas", "manhento", "manhoso", "manifestáveis",
"manifesto", "manilha", "manilhas", "maninho", "manipuláveis",
"manivela", "manivelas", "manjáveis", "manjerico", "mano", "manobráveis",
"manufacto", "manufacturáveis", "manufato", "manufaturáveis", "maquiáveis", "maquinais",
"maquináveis", "marafo", "marasmo", "maravilha", "maravilhas", "maravilháveis",
"marca", "marcas", "marceneiro", "marcha", "marchais", "marchas",
"marchetáveis", "marco", "marfolho", "marginais", "marginalizáveis", "maridais",
"marido", "marina", "marinas", "marinheiro", "marinho", "mariola",
"mariolas", "marisco", "marmelo", "marmo", "marmorizáveis", "marquesa",
"marquesais", "marquesas", "marrancho", "marreta", "marretas", "marroquino",
"marruco", "marteláveis", "martelo", "martirizáveis", "maruja", "marujais",
"marujas", "marujo", "masca", "mascabo", "mascaráveis", "mascas",
"mascavo", "masculinizáveis", "massa", "massacráveis", "massais", "massas",
"massificáveis", "mastigáveis", "masturbáveis", "matais", "materializáveis", "maticais",
"matilha", "matilhas", "matinais", "matizáveis", "mato", "matraca",
"matracas", "matreiro", "matriculáveis", "matrimoniais", "matrimoniáveis", "maturáveis",
"matuto", "maximizáveis", "maxixais", "mazurca", "mazurcas", "mealho",
"meandro", "mecanizáveis", "mecha", "mechas", "medalha", "medalhas",
"mediatizáveis", "medicais", "medicamento", "medicáveis", "medicina", "medicinais",
"medicinas", "medievalizáveis", "meditáveis", "meiga", "meigas", "meigo",
"meirinho", "melhoráveis", "melificáveis", "melindráveis", "melodia", "melodias",
"melodizáveis", "memoráveis", "memoriais", "memoriáveis", "memorizáveis", "mencionáveis",
"mendicáveis", "mendiga", "mendigas", "mendigáveis", "mendigo", "meno",
"menoscabáveis", "menoscabo", "menosprezáveis", "menosprezo", "menstruais", "mensurais",
"mensuráveis", "mentais", "mentalizáveis", "mentáveis", "mentira", "mentiras",
"mercadejáveis", "mercadizáveis", "mercantilizáveis", "mercáveis", "merenda", "merendais",
"merendas", "mergulháveis", "mergulho", "merujo", "mescais", "mesmerizáveis",
"mesquinho", "mestiçáveis", "mestiço", "metabolizáveis", "metacomunicáveis", "metaconglomeráveis",
"metaforizáveis", "metalificáveis", "metalizáveis", "metamorfizáveis", "metassedimentáveis", "metassedimento",
"metastáveis", "metilais", "metiláveis", "metodizáveis", "metralha", "metralhas",
"metralháveis", "metrificáveis", "metro", "metrocistografáveis", "metropolizáveis", "mexerico",
"mexilho", "micho", "microbiais", "microcirculáveis", "microcoaguláveis", "microcopiáveis",
"microfossilizáveis", "micromanipuláveis", "micrometráveis", "micropotencializáveis", "microprocesso", "mielinizáveis",
"migalha", "migalhas", "migmatizáveis", "migráveis", "mijo", "milimetrais",
"militarizáveis", "militáveis", "milonitizáveis", "mimáveis", "mimetizáveis", "mimo",
"mina", "minas", "mináveis", "minerais", "mineralizáveis", "mineráveis",
"mingo", "minhoca", "minhocais", "minhocas", "miniatura", "miniaturais",
"miniaturas", "miniaturizáveis", "minificáveis", "minimizáveis", "ministra", "ministrais",
"ministras", "ministráveis", "ministro", "minoráveis", "minotauro", "minuta",
"minutas", "minuto", "mira", "miras", "miráveis", "mirra",
"mirras", "miseráveis", "misgalho", "missa", "missais", "missas",
"mistificáveis", "mistura", "misturas", "misturáveis", "mitificáveis", "mitigáveis",
"mitizáveis", "mitra", "mitrais", "mitras", "mixáveis", "mixo",
"mnemonizáveis", "mobilizáveis", "moca", "moça", "mocas", "moças",
"mocho", "moço", "modalizáveis", "modeláveis", "modelo", "moderáveis",
"moderna", "modernas", "modernizáveis", "moderno", "modificáveis", "modorra",
"modorrais", "modorras", "moduláveis", "mofa", "mofas", "mofo",
"mofumbais", "moinho", "moira", "moirais", "moiras", "moiro",
"moita", "moitais", "moitas", "mojo", "moldáveis", "moldura",
"molduras", "moleca", "molecas", "molejo", "molengo", "molestáveis",
"molesto", "molháveis", "molho", "molificáveis", "molinha", "molinhas",
"molinho", "momo", "monarquia", "monarquias", "monco", "mondáveis",
"monetizáveis", "monidratáveis", "monidrato", "monitoráveis", "monitorizáveis", "monocloráveis",
"monocloro", "monogramo", "monopolizáveis", "monossilaba", "monossilabas", "monotongo",
"montáveis", "morais", "moralizáveis", "morangais", "morango", "morcegais",
"morcego", "mordaça", "mordaças", "mordomo", "morfina", "morfinas",
"morfino", "morfo", "morfodiferenciáveis", "morigeráveis", "morna", "mornais",
"mornas", "morno", "morraçais", "morrinho", "mortalha", "mortalhas",
"mortificáveis", "mosca", "moscas", "mosquito", "mostráveis", "motejo",
"moteto", "motiváveis", "motivo", "mouco", "moura", "mourais",
"mouras", "mourisco", "mouro", "movimentáveis", "movimento", "moxamo",
"muda", "mudas", "mudáveis", "mudo", "mufico", "mufumbo",
"mujimbo", "multiarticuláveis", "multiplicáveis", "multivariáveis", "multivibráveis", "mumificáveis",
"mundana", "mundanais", "mundanas", "mundano", "mundiais", "mundificáveis",
"mungango", "munheca", "munhecas", "murais", "muralha", "muralhas",
"murcho", "murmurinho", "muro", "musgo", "musicais", "musicalizáveis",
"musicáveis", "mutarrotáveis", "mutuais", "muxoxo", "nacionalizáveis", "nada", "nadais",
"nado", "namoráveis", "namoricho", "namorico", "namoro", "nana",
"nanais", "nanas", "narciso", "narcotizáveis", "narráveis", "nasalizáveis",
"nastro", "natura", "naturais", "naturalizáveis", "naturas", "naturáveis",
"naufragáveis", "navalha", "navalhas", "navegáveis", "neblina", "neblinas",
"nebrina", "nebrinas", "nebulizáveis", "necessitáveis", "necrosáveis", "necrotizáveis",
"negativa", "negativas", "negativizáveis", "negativo", "negáveis", "negligenciáveis",
"negociais", "negociáveis", "negrito", "neoformáveis", "neologismo", "nesga",
"nesgas", "neurosáveis", "neurotizáveis", "neutralizáveis", "nevoais", "nevoento",
"nicho", "nicotina", "nicotinas", "nicotináveis", "nicotino", "nicto",
"nimbo", "nina", "ninas", "ninhais", "ninho", "niponizáveis",
"nitrais", "nitrato", "nitreto", "nitroderiváveis", "nitrosáveis", "nitroso",
"nitruro", "niveláveis", "nobelizáveis", "nobilitáveis", "nobilizáveis", "noca",
"nocas", "nodais", "nodo", "noiva", "noivais", "noivas",
"noivo", "nojo", "nomenclatura", "nomenclaturais", "nomenclaturas", "nominais",
"nomináveis", "normalizáveis", "notabilizáveis", "notais", "notáveis", "noticiáveis",
"notificáveis", "nova", "novais", "novas", "novela", "novelas",
"novelo", "novo", "nuclearizáveis", "nulificáveis", "numerais", "numeráveis",
"nupciais", "obcecáveis", "objeccionais", "objeccionáveis", "objecionais", "objecionáveis",
"objectificáveis", "objectiváveis", "objetais", "objetáveis", "objetificáveis", "objetiváveis",
"objetivo", "objeto", "obliteráveis", "obnubiláveis", "obra", "obrada",
"obradas", "obrado", "obras", "obráveis", "obrigáveis", "obsedáveis",
"obsediáveis", "observáveis", "obsessionáveis", "obsoleta", "obsoletas", "obsoleto",
"obstáveis", "obtemperáveis", "obturáveis", "obviáveis", "oca", "ocas",
"ocasionais", "ocasionáveis", "ocáveis", "oco", "octanais", "octanáveis",
"octano", "oculta", "ocultas", "ocultáveis", "oculto", "ocupáveis",
"oferenda", "oferendas", "ofertáveis", "oficiais", "oficio", "ofuscáveis",
"ogiva", "ogivais", "ogivas", "ogivo", "oiro", "oitava",
"oitavas", "oitavo", "ojeriza", "ojerizas", "oleado", "olfacto",
"olfato", "olhais", "olháveis", "olho", "oliva", "olivais",
"olivas", "olvidáveis", "olvido", "onda", "ondas", "onduláveis",
"oneráveis", "onzeneiro", "opaca", "opacas", "opaco", "opais",
"opalino", "opáveis", "operacionalizáveis", "operáveis", "opináveis", "optimizáveis",
"opugnáveis", "opulenta", "opulentas", "opulento", "ora", "orais",
"orbitais", "orbitáveis", "orçamentais", "orçamento", "orçáveis", "ordenáveis",
"orelha", "orelhas", "organificáveis", "organizáveis", "organoformáveis", "organofosforáveis",
"orgulho", "orientais", "orientáveis", "originais", "originalizáveis", "origináveis",
"orla", "orlas", "ornamentais", "ornamentáveis", "ornamento", "orquestra",
"orquestrais", "orquestras", "ortiga", "ortigas", "ortopantomografáveis", "orvalho",
"osciláveis", "ostentáveis", "otimizáveis", "ourela", "ourelas", "ouriço",
"ouro", "ousáveis", "ousio", "outonais", "outono", "outorga",
"outorgas", "outorgáveis", "ova", "ovais", "ovas", "ovo",
"oxidáveis", "oxigenáveis", "oxigenizáveis", "ozonizáveis", "pacificáveis", "pacto",
"pactuais", "pactuáveis", "padrais", "padrinho", "padronizáveis", "paganizáveis",
"pagáveis", "pagináveis", "pago", "pairáveis", "paivo", "paladino",
"palanco", "palatalizáveis", "palatizáveis", "palavra", "palavras", "palestra",
"palestras", "paleta", "paletas", "palha", "palhais", "palhas",
"palheta", "palhetas", "palhiço", "paliais", "paliáveis", "palito",
"palma", "palmais", "palmas", "palmo", "paloma", "palomas",
"palpáveis", "palpo", "panaço", "panais", "pandais", "pando",
"panegiricais", "panfleto", "pango", "panificáveis", "pano", "pantografáveis",
"pantomima", "pantomimas", "pantomimáveis", "pantomimo", "papagaiais", "papagaio",
"papais", "paparico", "papáveis", "papo", "papoilais", "papouco",
"papoulais", "para", "paradoxais", "paradoxo", "parafuso", "parais",
"paralelizáveis", "paralisáveis", "paramentáveis", "paramento", "parametrizáveis", "paraninfo",
"parapeito", "parasito", "paráveis", "parcela", "parcelas", "parceláveis",
"pardais", "pardejo", "pardo", "parelha", "parelhas", "parelho",
"parenta", "parentais", "parentas", "parlamento", "parolo", "paroquiais",
"parro", "parrono", "participais", "participáveis", "particularizáveis", "partilháveis",
"pascoais", "pasmo", "paspalho", "passais", "passarinho", "passáveis",
"passinho", "passivo", "passo", "pasta", "pastas", "pasteurizáveis",
"pastilha", "pastilhas", "pastinha", "pastinhas", "pasto", "pastorais",
"patego", "patináveis", "patinho", "patoilo", "patroa", "patroas",
"patrocinais", "patrocináveis", "patrulháveis", "paula", "paulas", "paulificáveis",
"paurometabolizáveis", "pautais", "pautáveis", "pavimentáveis", "pavimento", "pecáveis",
"pecha", "pechas", "peçonhento", "peculiarizáveis", "pedicelo", "pedimentáveis",
"pedimento", "pediplanáveis", "pediplano", "pedra", "pedrais", "pedras",
"pega", "peganho", "pegas", "pego", "peido", "peito",
"pejo", "pela", "pelas", "peleja", "pelejas", "pelejáveis",
"peletizáveis", "pelo", "pelotizáveis", "pena", "penacho", "penais",
"penas", "penáveis", "pendorais", "pendurais", "penduráveis", "pendurico",
"peneira", "peneiras", "peneiráveis", "penetráveis", "penhoráveis", "penico",
"penitenciais", "pensamento", "pensáveis", "penso", "pentelho", "penumbra",
"penumbrais", "penumbras", "pepinais", "pepino", "pequinho", "peralvilho",
"perambuláveis", "percais", "percalço", "percepcionais", "perceptualizáveis", "percevejo",
"percoláveis", "perderiváveis", "perdigoto", "perdoáveis", "perduráveis", "peregrina",
"peregrinais", "peregrinas", "peregrináveis", "peregrino", "perereca", "pererecas",
"perfilháveis", "perfuráveis", "pergaminho", "pergunta", "perguntas", "perguntáveis",
"periciais", "perigo", "periodizáveis", "periquito", "perito", "perjuráveis",
"perjuro", "perlongáveis", "permineralizáveis", "permudáveis", "permuta", "permutas",
"permutáveis", "perorais", "peroráveis", "perpassáveis", "perpetráveis", "perpetuais",
"perpetuáveis", "perscrutáveis", "perseveráveis", "persignáveis", "personalizáveis", "personificáveis",
"perspectiva", "perspectivas", "perspectiváveis", "perspectivo", "perspetiváveis", "perspiráveis",
"pertucho", "perturba", "perturbas", "perturbáveis", "perua", "peruas",
"pesáveis", "pesca", "pescais", "pescas", "peso", "pesquisa",
"pesquisas", "pessoalizáveis", "pestilenciais", "peta", "petais", "petanisco",
"petas", "peteca", "petecas", "petisco", "petrechais", "petrecho",
"petrificáveis", "pezunho", "pia", "piacho", "piais", "pialo",
"pias", "picais", "picardia", "picardias", "piçarrais", "picirico",
"pico", "picuais", "pielografáveis", "pigais", "pigarro", "pigmento",
"pilha", "pilhas", "pilotáveis", "piloto", "pimpolho", "pincho",
"pinga", "pingalho", "pingas", "pingo", "pino", "pinoco",
"pinta", "pintainho", "pintas", "pintáveis", "pinto", "pio",
"piolho", "pioráveis", "pipa", "pipais", "pipas", "pirais",
"piramidais", "piramidáveis", "piramido", "pirilampo", "pirolisáveis", "piso",
"pista", "pistas", "pita", "pitas", "piteira", "piteiras",
"piteiro", "pito", "plaino", "planejáveis", "planetizáveis", "planificáveis",
"planilha", "planilhas", "plano", "planta", "plantais", "plantas",
"plasmáveis", "plasmolisáveis", "plasticizáveis", "plastificáveis", "platina", "platinas",
"plebiscito", "pleito", "plicáveis", "plissáveis", "plotáveis", "plugáveis",
"pluma", "plumas", "plumbaginais", "pluralizáveis", "pluriestratificáveis", "pluristratificáveis",
"poda", "podais", "podas", "poeira", "poeiras", "poento",
"poiso", "polarizáveis", "polca", "polcas", "polemica", "polemicas",
"policiais", "policiáveis", "policitáveis", "polimento", "polinizáveis", "polirramificáveis",
"politizáveis", "politonais", "polpa", "polpas", "polvilhais", "pombeiro",
"ponderais", "ponderáveis", "pontais", "pontificais", "pontinho", "ponto",
"pontuais", "pontuáveis", "popularizáveis", "pormenorizáveis", "pororoca", "pororocas",
"porta", "portais", "portas", "portáveis", "porto", "portucho",
"portuguesa", "portuguesas", "posicionais", "posicionáveis", "positiva", "positivas",
"positiváveis", "positivo", "possibilitáveis", "postais", "postáveis", "postemo",
"postergáveis", "postiço", "posto", "postuláveis", "postura", "posturais",
"posturas", "potenciais", "potencializáveis", "potenciáveis", "pousais", "pouso",
"povoáveis", "praga", "pragais", "pragas", "pragmatizáveis", "praguejáveis",
"prancha", "pranchas", "pranto", "praticáveis", "preamo", "prebendais",
"precalço", "precaucionais", "preceito", "preceituais", "precipitáveis", "precisa",
"precisas", "preciso", "preconceituais", "precondicionais", "precondicionáveis", "preconizáveis",
"predestináveis", "predetermináveis", "predicais", "predicamentais", "predicamento", "predicáveis",
"predito", "prefaciais", "preferenciais", "prefiguráveis", "prefiltro", "prefixais",
"prefixo", "prefoliáveis", "pregalho", "pregáveis", "pregnáveis", "prego",
"pregoáveis", "preguiça", "preguiças", "preito", "prejudicáveis", "prelaciais",
"prelaziais", "prelecionáveis", "prelibáveis", "preludiais", "prematuro", "premeditáveis",
"prenda", "prendas", "prenominais", "prensáveis", "prenunciáveis", "preparáveis",
"preparo", "preponderáveis", "preposicionais", "presa", "presas", "presáveis",
"prescruto", "presenciais", "presentificáveis", "preserváveis", "presidiais", "presigo",
"presilha", "presilhas", "preso", "pressionais", "pressionáveis", "pressurizáveis",
"prestacionais", "prestais", "prestáveis", "pretexto", "prevaricáveis", "prezáveis",
"prima", "primais", "primas", "primavera", "primaverais", "primaveras",
"primitivo", "primo", "principiáveis", "priorais", "prisco", "pristináveis",
"privatizáveis", "privilegiáveis", "proa", "proais", "proas", "problematizáveis",
"processais", "processáveis", "processionais", "processo", "proclamáveis", "procrastináveis",
"procriáveis", "procuráveis", "prodigalizáveis", "prodigáveis", "proemiais", "profaço",
"profanáveis", "profano", "professo", "professorais", "profetizáveis", "profilo",
"profissionalizáveis", "profligáveis", "profunda", "profundas", "profundáveis", "profundo",
"prognosticáveis", "programáveis", "progresso", "projetáveis", "projeto", "proletarizáveis",
"proliferáveis", "prologais", "prolongáveis", "promanáveis", "promulgáveis", "pronomiais",
"pronominais", "pronominalizáveis", "pronunciáveis", "propagáveis", "propaláveis", "propiciáveis",
"propina", "propinas", "propolisáveis", "proporcionais", "proporcionáveis", "propositais",
"propugnáveis", "propulsionais", "prorrogáveis", "prosa", "prosas", "prosificáveis",
"prospecto", "prosperáveis", "prosternais", "prosternáveis", "prostráveis", "protagonizáveis",
"protecionais", "proteláveis", "protestáveis", "protesto", "protocoláveis", "protocolo",
"prova", "provas", "prováveis", "provençalizáveis", "proverbiais", "providenciais",
"provisionais", "prudenciais", "prumo", "pseudojustificáveis", "psicanalisáveis", "psicologizáveis",
"publicáveis", "publicitáveis", "pugilo", "pugnáveis", "pulinho", "pulo",
"pulsáveis", "pulsionais", "pulso", "pululáveis", "pulverizáveis", "puntáveis",
"punto", "pupais", "pupáveis", "pupila", "pupilas", "pupilo",
"purgáveis", "purificáveis", "purpurais", "putrificáveis", "puxo", "quadra",
"quadras", "quadráveis", "quadriculáveis", "quadrilha", "quadrilhas", "quadrinizáveis",
"quadro", "quadrupedais", "quadruplicáveis", "qualificáveis", "quantificáveis", "quantizáveis",
"quarentena", "quarentenas", "quaresma", "quaresmais", "quaresmas", "quarta",
"quartais", "quartas", "quartilho", "quarto", "quebra", "quebrantáveis",
"quebranto", "quebras", "quebráveis", "queda", "quedas", "quedo",
"queijo", "queimáveis", "queixa", "queixais", "queixas", "queixo",
"queláveis", "querela", "querelas", "quereláveis", "questionáveis", "quico",
"quieta", "quietas", "quieto", "quilha", "quilhas", "quimbo",
"quimiossintetizáveis", "quina", "quinais", "quinas", "quincha", "quinchas",
"quinta", "quintais", "quintas", "quintessenciais", "quintessenciáveis", "quinto",
"quintuplicáveis", "quitanda", "quitandas", "quota", "quotas", "quotizáveis",
"rabanais", "rabano", "rabeira", "rabeiras", "rabeiro", "rabioso",
"rabiscáveis", "rabisco", "rabona", "rabonas", "rabusco", "racemizáveis",
"raciocinais", "raciocináveis", "racionais", "racionalizáveis", "racionáveis", "radiais",
"radiativáveis", "radiativo", "radicais", "radicalizáveis", "radioativáveis", "radioativo",
"radiocromatografáveis", "radiofonizáveis", "radiolocalizáveis", "radiomarcáveis", "radionavegáveis", "radiotransistorizáveis",
"raia", "raiais", "raias", "raio", "raiva", "raivais",
"raivas", "raláveis", "ralho", "ralo", "rama", "ramais",
"ramalha", "ramalhais", "ramalhas", "ramalho", "ramas", "ramificáveis",
"ramo", "ranço", "randomizáveis", "ranfo", "rango", "ranguinho",
"rapa", "rapas", "rapigo", "rapilho", "rapina", "rapinas",
"rapináveis", "rapinha", "rapinhas", "raposa", "raposas", "raposinha",
"raposinhas", "raposo", "rapto", "rara", "raras", "raro",
"rasa", "rasas", "rascunháveis", "rascunho", "rasgo", "raso",
"rasoira", "rasoiras", "raspanço", "rasteira", "rasteiras", "rasteiro",
"rastejáveis", "rastelo", "rastilho", "rastinho", "rastolho", "rasuráveis",
"rata", "ratas", "ratazana", "ratazanas", "ratificáveis", "ratinho",
"ratino", "rato", "ravina", "ravinas", "ravináveis", "razoáveis",
"razonáveis", "reabilitáveis", "reacionais", "reacopláveis", "readaptáveis", "reajuntáveis",
"reajustáveis", "realejo", "realimentáveis", "realizáveis", "realocáveis", "reambuláveis",
"reanalisáveis", "reaperto", "reapreciáveis", "reaproveitáveis", "rearranjáveis", "rearranjo",
"reaterro", "reativáveis", "reativo", "reatocho", "reaumento", "reavaliáveis",
"rebaixáveis", "rebaixo", "rebalanço", "rebanho", "rebarba", "rebarbas",
"rebeláveis", "rebentáveis", "rebento", "rebimbalho", "reboco", "rebojo",
"rebolais", "reboliço", "rebolo", "reboo", "rebordo", "rebuço",
"rebufo", "rebuliço", "rebumbo", "rebuscáveis", "reca", "recado",
"recalcáveis", "recalcificáveis", "recalcitráveis", "recalculáveis", "recambiáveis", "recamo",
"recandidato", "recanto", "recapacitáveis", "recapitalizáveis", "recapituláveis", "recarga",
"recargas", "recarregáveis", "recas", "recato", "recaucionáveis", "receita",
"receitas", "recepcionais", "recepcionáveis", "rechino", "recibo", "recicláveis",
"reciclo", "recidivo", "recinto", "reciprocáveis", "recitais", "reclamáveis",
"reclamo", "reclassificáveis", "reclináveis", "reclusa", "reclusas", "recluso",
"reco", "recobráveis", "recogitáveis", "recolo", "recolocáveis", "recolonizáveis",
"recombináveis", "recomeço", "recomendáveis", "recompartimentais", "recompartimentáveis", "recompensáveis",
"recompromisso", "reconcentráveis", "reconciliáveis", "recondicionáveis", "reconfessáveis", "reconfiguráveis",
"reconfortáveis", "reconforto", "recongraçáveis", "reconsagráveis", "reconsolidáveis", "reconstitucionalizáveis",
"reconsultáveis", "recontáveis", "recontratáveis", "recontrato", "reconvocáveis", "recordáveis",
"recortáveis", "recosto", "recredenciais", "recriáveis", "recrimináveis", "recristalizáveis",
"recristianizáveis", "recrutáveis", "recuo", "recuperáveis", "recursais", "recurso",
"recurváveis", "recurvo", "recusáveis", "redamo", "redatoriais", "redecoráveis",
"redemocratizáveis", "redemoinho", "redescontáveis", "redesconto", "redesenho", "redestináveis",
"redimensionáveis", "redobráveis", "reedificáveis", "reeditáveis", "reeditoráveis", "reeducáveis",
"reelaboráveis", "reembolsáveis", "reembolso", "reemendáveis", "reencontráveis", "reencontro",
"reendossáveis", "reenquadráveis", "reensino", "reenvernizáveis", "reenvio", "reenxerto",
"reestruturáveis", "reexageráveis", "reexagero", "reexpatriáveis", "reexpiáveis", "reexportáveis",
"refarto", "referenciais", "referenda", "referendas", "referendo", "referto",
"refluxo", "refolego", "refolho", "reforço", "reformato", "reformáveis",
"refotografáveis", "refracto", "refratáveis", "refresco", "refrigeráveis", "refrito",
"refrulho", "refugo", "refundáveis", "refunfo", "refutáveis", "regaço",
"regais", "regalo", "regata", "regatas", "regatinha", "regatinhas",
"regato", "regeláveis", "regeneráveis", "regimentais", "regimento", "regina",
"reginas", "regino", "registáveis", "registo", "registrais", "registráveis",
"registro", "rego", "regozijáveis", "regra", "regras", "regraváveis",
"regráveis", "regresso", "reguardo", "regueiro", "regulamento", "regularizáveis",
"reificáveis", "reimportáveis", "reinais", "reinerváveis", "reingresso", "reinjetáveis",
"reino", "reinoculáveis", "reintegráveis", "reinternacionalizáveis", "reinterpretáveis", "reiteráveis",
"reiuno", "reivindicáveis", "rejeitáveis", "rejeito", "rejubiláveis", "rejuntáveis",
"relacionais", "relacionáveis", "relampado", "relampo", "relançáveis", "relanço",
"relatáveis", "relativáveis", "relativizáveis", "relativo", "relato", "relaxáveis",
"relegáveis", "relembráveis", "relento", "releváveis", "relevo", "relho",
"religáveis", "relojo", "relvais", "remais", "remanejáveis", "remanso",
"remarcáveis", "rematerializáveis", "rememoráveis", "remendo", "remessa", "remessas",
"remimo", "remisturáveis", "remo", "remodeláveis", "remoinho", "remolgo",
"remontáveis", "remuneráveis", "renda", "rendas", "rendáveis", "renegociáveis",
"renormalizáveis", "renováveis", "renovo", "rentáveis", "renunciáveis", "reoficializáveis",
"reordenáveis", "reorientáveis", "reorquestráveis", "repairo", "reparametrizáveis", "reparáveis",
"reparo", "repasto", "repatriáveis", "repensáveis", "repesáveis", "repeso",
"repintáveis", "replaino", "repleto", "replicáveis", "repolhais", "repolho",
"reponderáveis", "reportáveis", "reposicionáveis", "reposta", "repostas", "reposto",
"repouso", "represa", "represas", "represáveis", "representáveis", "reprimenda",
"reprimendas", "reprocessáveis", "reprogramáveis", "reprojeto", "reprováveis", "repto",
"repudiáveis", "repulsa", "repulsas", "repuxo", "requadro", "requalificáveis",
"requebráveis", "requebro", "requeimáveis", "requentáveis", "requestionáveis", "requintáveis",
"requisito", "rescaldo", "resenha", "resenhas", "resenháveis", "reserva", "reservas", "reserváveis",
"resfriáveis", "resgatáveis", "resgo", "resguardáveis", "resguardo", "residenciais",
"resignáveis", "resina", "resinas", "resinificáveis", "resmungo", "respaldo",
"respeitáveis", "respeito", "respigáveis", "respingáveis", "respingo", "respirais",
"respiráveis", "respiro", "responsabilizáveis", "responsáveis", "responto", "resposta",
"respostas", "ressaca", "ressacas", "ressacáveis", "ressaibo", "ressalto",
"ressalva", "ressalvas", "resseguro", "ressintetizáveis", "ressolto", "ressonáveis",
"ressoo", "ressuscitáveis", "restauráveis", "restauro", "restáveis", "restito",
"restivo", "resto", "restolhais", "restolho", "restrelo", "resvalo",
"reta", "retaco", "retais", "retalho", "retaliáveis", "retardáveis",
"retardo", "retas", "retemperáveis", "retento", "retesáveis", "retificáveis",
"retipificáveis", "retiráveis", "retiro", "reto", "retocáveis", "retonho",
"retornáveis", "retorno", "retrabalháveis", "retrabalho", "retraço", "retranca",
"retrancas", "retratáveis", "retrato", "retroalimentáveis", "retroalimento", "retroativo",
"retrocarregáveis", "retrocruzáveis", "retrograda", "retrogradas", "reumanizáveis", "reunificáveis",
"reusáveis", "reuso", "reutilizáveis", "revacináveis", "revalidáveis", "revalorizáveis",
"revascularizáveis", "reveláveis", "reverberáveis", "reverenciais", "reverenciáveis", "reverificáveis",
"reversais", "reverso", "revezáveis", "revigoráveis", "revigorizáveis", "reviráveis",
"revisáveis", "revisitáveis", "revista", "revistas", "revistáveis", "revisto",
"revitalizáveis", "revivificáveis", "revocáveis", "revoltáveis", "revolto", "revoo",
"riba", "ribas", "ribeira", "ribeiras", "ribeiro", "ridicularizáveis",
"ridiculizáveis", "rifa", "rifas", "rifáveis", "rigo", "rija",
"rijais", "rijas", "rijo", "rilo", "rimáveis", "rimo",
"rincho", "ripais", "ripanço", "risco", "ritmáveis", "ritmo",
"rivalizáveis", "rixa", "rixas", "roboráveis", "robotizáveis", "rocais",
"rocinais", "roda", "rodais", "rodas", "rodáveis", "rodilha",
"rodilhas", "rodo", "rodopiáveis", "rodopio", "rodrigo", "rogais",
"rogáveis", "rogo", "rola", "rolas", "rolda", "roldas",
"roleta", "roletas", "rolha", "rolhas", "roliço", "rolo",
"romanesco", "romanizáveis", "romantizáveis", "ronca", "roncais", "roncas",
"roncáveis", "ronco", "roreja", "rorejas", "rosa", "rosais",
"rosas", "rosca", "roscas", "roscáveis", "roseta", "rosetas",
"rosmaninhais", "rosmaninho", "rota", "rotacionais", "rotas", "rotativo",
"rotinizáveis", "roto", "rotuláveis", "roubáveis", "roubo", "roufenho",
"roupa", "roupas", "rousso", "rua", "ruais", "ruas",
"rubificáveis", "ruborizáveis", "rubrica", "rubricas", "rubricáveis", "ruça",
"ruças", "rufo", "ruga", "rugas", "ruma", "rumas",
"rumba", "rumbas", "ruminais", "rumo", "rurbanizáveis", "rusga",
"rusgas", "russa", "russas", "russificáveis", "russo", "rutais",
"sabatina", "sabatinas", "sabotáveis", "sabugais", "sabugo", "sabujo",
"saburrais", "saca", "sacais", "sacarificáveis", "sacas", "sacha",
"sachas", "sacho", "sacholo", "saciáveis", "saco", "sacola",
"sacolas", "sacralizáveis", "sacramentais", "sacramentáveis", "sacramento", "sacrificais",
"sacrificáveis", "safano", "safirizáveis", "safo", "sagitais", "sagorro",
"sagráveis", "salariais", "saldo", "salgáveis", "salicilo", "salientáveis",
"salificáveis", "salináveis", "salino", "salitrais", "saliva", "salivais",
"salivas", "salmejo", "salmoeiro", "salmoira", "salmoiras", "salmoura",
"salmouras", "salpicáveis", "salpico", "salsa", "salsas", "salso",
"saltinho", "salto", "salváveis", "salvo", "sambenito", "sambláveis",
"sanáveis", "sancionáveis", "sanfona", "sanfonas", "sanfonáveis", "sangráveis",
"sanguento", "sanguinhais", "sanguinho", "sanitizáveis", "santificáveis", "sapa",
"sapais", "sapas", "sapato", "sapo", "sapoilo", "saponificáveis",
"sapoulo", "saraiva", "saraivas", "sarapico", "sarda", "sardas",
"sardo", "sarilho", "sarna", "sarnas", "saro", "sarrafaçais",
"sarrafo", "sarro", "sartais", "satelitizáveis", "satelizáveis", "saturáveis",
"sazonais", "sazonáveis", "seca", "secais", "secas", "secáveis",
"seccionais", "seccionáveis", "secessionais", "secionais", "secionáveis", "seco",
"secreta", "secretaria", "secretariais", "secretarias", "secretas", "secreto",
"secularizáveis", "secundáveis", "seda", "sedais", "sedas", "sedentarizáveis",
"sediáveis", "sedimento", "segmentais", "segmentáveis", "segmento", "segredáveis",
"segredo", "segregáveis", "segunda", "segundas", "segundo", "segura",
"seguras", "seguráveis", "seguro", "seita", "seitas", "seiva",
"seivais", "seivas", "seivo", "sela", "selas", "selecionais",
"selecionáveis", "seleto", "selo", "semantizáveis", "semelháveis", "sementais",
"semiajustáveis", "semiaplicáveis", "semicadastrais", "semicolo", "semidomesticáveis", "semiformato",
"semimanufaturáveis", "seminormais", "seminormáveis", "semitizáveis", "sensibilizáveis", "sentáveis",
"sentenciais", "sentenciáveis", "separáveis", "septais", "septáveis", "septo",
"sepulcrais", "sepulcro", "sepultáveis", "sepulto", "sequela", "sequelas",
"sequenciais", "sequestráveis", "sequestro", "serena", "serenas", "serenáveis",
"sereno", "seriais", "serigrafáveis", "seringa", "seringais", "seringas",
"seringonho", "seriografáveis", "serra", "serras", "serrino", "serro",
"sertaneja", "sertanejas", "sertanejo", "seta", "setas", "seteiro",
"setino", "seto", "setorizáveis", "seviciáveis", "sexo", "sextuplicáveis",
"sexuais", "siais", "sibila", "sibilas", "sibiláveis", "sibilo",
"siderais", "sideráveis", "sifilizáveis", "sifonais", "sigiláveis", "sigilo",
"sigla", "siglas", "signa", "signas", "significáveis", "signo",
"silabáveis", "silenciais", "silenciáveis", "silhueta", "silhuetas", "silicato",
"silicificáveis", "silogismo", "silva", "silvas", "silvo", "simbolizáveis",
"simetrizáveis", "simpatizáveis", "simplificáveis", "simuláveis", "sina", "sinais",
"sinalizáveis", "sinapizáveis", "sinas", "sincelo", "sincopais", "sincretizáveis",
"sincronizáveis", "sindicais", "sindicalizáveis", "sindicatais", "sindicato", "sindicáveis",
"sineta", "sinetas", "sinfonizáveis", "singularizáveis", "sinistra", "sinistras",
"sinistro", "sino", "sinonimizáveis", "sinterizáveis", "sintetizáveis", "sintonizáveis",
"sinuco", "sinuoso", "sirena", "sirenas", "sirigaita", "sirigaitas",
"sisa", "sisais", "sisas", "sismais", "sismo", "siso",
"sistematizáveis", "sitiais", "situáveis", "soalhais", "soalho", "sobejo",
"soberba", "soberbas", "soberbo", "sobrado", "sobrais", "sobráveis",
"sobreaviso", "sobrecabo", "sobrecasaca", "sobrecasacas", "sobrecasaco", "sobrecifráveis",
"sobreleito", "sobrenadáveis", "sobrepasso", "sobrepujáveis", "sobrerroldo", "sobrescrito",
"sobressalto", "sobresselo", "sobreuso", "sobrevoo", "socalco", "sociabilizáveis",
"socializáveis", "soco", "socresto", "sofismáveis", "sofisticáveis", "sogra",
"sogras", "sogro", "sola", "solas", "solavanco", "solda",
"soldas", "soldáveis", "soldo", "soletráveis", "solfejo", "solha",
"solhas", "solicitáveis", "solo", "solta", "soltas", "solto",
"solucionáveis", "soluço", "sombra", "sombrais", "sombras", "somo",
"sonda", "sondas", "sondáveis", "soneto", "sonháveis", "sonho",
"sono", "sopapo", "sopesáveis", "sopitáveis", "sopro", "sorais",
"soro", "soroneutralizáveis", "sorvais", "sorvo", "soslaio", "sospitais",
"sossegáveis", "sossego", "soterráveis", "soto", "soturna", "soturnas",
"soturno", "souto", "sova", "sovas", "sovela", "sovelas",
"sua", "suas", "suavizáveis", "subajustáveis", "subalterno", "subaplicáveis",
"subaproveitáveis", "subarrendáveis", "subcandidato", "subcontratáveis", "subenxertáveis", "subenxerto",
"subequipáveis", "suberizáveis", "subespecificáveis", "subestimáveis", "subfaturáveis", "subformato",
"subgrupo", "subjetivo", "subjugáveis", "sublimáveis", "subministráveis", "subniveláveis",
"suborbitais", "subordináveis", "subornáveis", "suborno", "subscrito", "subsequenciais",
"subsequenciáveis", "subsolo", "substanciais", "substancializáveis", "substanciáveis", "substantificáveis",
"substantivais", "substantiváveis", "substantivo", "suburbanizáveis", "subutilizáveis", "subvencionais",
"subvencionáveis", "sucata", "sucatas", "suco", "sudáveis", "sufixais",
"sufixo", "sufocáveis", "sufoco", "sugestionáveis", "suja", "sujas",
"sujáveis", "sujeita", "sujeitas", "sujeitáveis", "sujeito", "sujo",
"sulavento", "sulcáveis", "sulco", "sulfato", "sulfeto", "sulfito",
"sulfonais", "sulfonáveis", "sulfuráveis", "sulfureto", "sulfurizáveis", "sumagrais",
"sumariáveis", "sumarizáveis", "superajuntáveis", "superajustáveis", "superalimentáveis", "superalimento",
"superaplicáveis", "superativo", "superáveis", "supercandidato", "supercivilizáveis", "supercompacto",
"supercompensáveis", "supercomplexo", "superconcentráveis", "superconfortáveis", "superdimensionáveis", "superestruturais",
"superestruturáveis", "superexcitáveis", "superexploráveis", "superformato", "superimpregnáveis", "superindustriais",
"superindustrializáveis", "superinterpretáveis", "superlativo", "superocupáveis", "superselecionáveis", "suplantáveis",
"suplemento", "suportáveis", "suputáveis", "surdina", "surdinas", "surfáveis",
"surpresa", "surpresas", "surpreso", "surra", "surras", "surráveis",
"surrealizáveis", "surrupio", "surto", "suscitáveis", "suspeita", "suspeitas",
"suspeitáveis", "suspeito", "suspiráveis", "suspiro", "sussurráveis", "sussurro",
"sustáveis", "sustentáveis", "sustento", "susto", "suturais", "suturáveis",
"tabardilho", "tabela", "tabelas", "tabelionais", "tabica", "tabicas",
"tabuláveis", "tabuleta", "tabuletas", "taburno", "tacha", "tachas",
"tacháveis", "tacho", "taco", "taino", "taipais", "talha",
"talhas", "talho", "talo", "tamanco", "tampa", "tampas",
"tampáveis", "tampo", "tamponáveis", "tanais", "tanáveis", "tancamo",
"tanga", "tangas", "tangenciais", "tango", "tanino", "tapáveis",
"taramela", "taramelas", "tardo", "tarefa", "tarefas", "tarifa",
"tarifas", "tarja", "tarjas", "taroco", "tarola", "tarolas",
"tarolo", "tarouco", "tarrafa", "tarrafas", "tarrinco", "tarugo",
"tasca", "tascas", "tasco", "tasquinha", "tasquinhas", "tassalho",
"taxa", "taxais", "taxas", "taxáveis", "teca", "tecais",
"tecas", "tecla", "teclas", "tecnicizáveis", "teima", "teimas",
"teimosia", "teimosias", "tela", "telas", "telecomando", "telecomunicáveis",
"telecurso", "teledetectáveis", "telefonáveis", "telegrafáveis", "teleporto", "telerradiografáveis",
"teletrabalho", "televisa", "televisas", "televisáveis", "televisionais", "telha",
"telhais", "telhas", "telomerizáveis", "tematizáveis", "temperáveis", "tempero",
"templo", "temporizáveis", "tendenciais", "tensionais", "tentáveis", "tento",
"teorizáveis", "terça", "terças", "terceira", "terceiras", "terceirizáveis",
"terceiro", "terceto", "terço", "terebintina", "terebintinas", "tergiversáveis",
"terlado", "terminais", "termináveis", "termistorizáveis", "termorreguláveis", "terna",
"ternais", "ternas", "terno", "terra", "terrais", "terramoto",
"terras", "terrificáveis", "terrorizáveis", "teso", "tesoiro", "tesoura",
"tesouras", "tesouro", "testamentais", "testamento", "testáveis", "testemunha",
"testemunhais", "testemunhas", "testemunháveis", "testemunho", "testico", "testificáveis",
"teta", "tetas", "teto", "textualizáveis", "textura", "texturais",
"texturas", "texturizáveis", "tibiro", "ticais", "tijolo", "tilo",
"timbalejo", "timbráveis", "tingo", "tinta", "tintas", "tinto",
"tipificáveis", "tipografáveis", "tiranizáveis", "tiriricais", "tiro", "tisnáveis",
"titiláveis", "tituláveis", "tocaia", "tocaias", "tocáveis", "toco",
"toga", "togas", "toiço", "toirais", "toiro", "tolda",
"toldas", "toldáveis", "toldo", "toleráveis", "tombo", "tomo",
"tomografáveis", "tona", "tonais", "tonas", "tonificáveis", "tonsura",
"tonsuras", "tonsuráveis", "topo", "tora", "torais", "toras",
"torcicolo", "tormenta", "tormentas", "tormento", "torno", "toro",
"torpedo", "torráveis", "tortura", "torturas", "torturáveis", "torva",
"torvas", "torvelinho", "torvo", "tosca", "toscas", "tosco",
"tosquia", "tosquias", "tosquiáveis", "tostáveis", "totalizáveis", "touca",
"touça", "toucas", "touças", "touço", "tourais", "touro",
"toxica", "toxicas", "trabalháveis", "trabalho", "trabelho", "trabuco",
"trabulo", "traça", "traças", "traço", "tracolejo", "trado",
"trafegáveis", "traficáveis", "tragáveis", "tragedia", "tragedias", "trago",
"trajo", "tralha", "tralhas", "trama", "tramas", "tramáveis",
"trambolha", "trambolhas", "trambolho", "tramela", "tramelas", "tramelo",
"tramo", "tranca", "trancafiáveis", "trancas", "trancáveis", "trançáveis",
"tranco", "tranqueira", "tranqueiras", "tranquilizáveis", "transacionais", "transacionáveis",
"transbordáveis", "transcodificáveis", "transculturais", "transculturáveis", "transcuráveis", "transcurso",
"transfiguráveis", "transfinalizáveis", "transformáveis", "transfuso", "transignificáveis", "transistorizáveis",
"transitáveis", "transitivo", "transladáveis", "translado", "transliterais", "transliteráveis",
"translocais", "translocáveis", "transmigráveis", "transmudáveis", "transmutáveis", "transnacionalizáveis",
"transnadáveis", "transpiráveis", "transplantáveis", "transportáveis", "transtorno", "transubstanciais",
"transudáveis", "transumano", "transviáveis", "transvoo", "trapaça", "trapaças",
"traquejo", "traquitana", "traquitanas", "trasladáveis", "traslado", "traspassáveis",
"trastempo", "tratáveis", "trato", "traulito", "traumatizáveis", "trava",
"travais", "travas", "traváveis", "travejáveis", "travessa", "travessas",
"travesso", "travessura", "travessuras", "travo", "trefiláveis", "treináveis",
"treino", "treito", "trejeito", "treliça", "treliças", "tremelais",
"tremelejo", "tremoçais", "tremoço", "tremuláveis", "treno", "trepáveis",
"trepidáveis", "treplicáveis", "treslado", "trespassáveis", "trevisco", "triais",
"trianguláveis", "tributais", "tributáveis", "tributo", "tricalho", "tricotáveis",
"trifurcáveis", "trigais", "trigo", "trilaterais", "triláveis", "trilha",
"trilhas", "trilháveis", "trilho", "trinca", "trincais", "trincas",
"trincáveis", "trincháveis", "trincheira", "trincheiras", "trinco", "trino",
"trio", "tripa", "tripas", "triplicáveis", "tripo", "tripudiáveis",
"tripuláveis", "trissais", "trissecáveis", "trissecto", "trissulco", "trito",
"trituráveis", "triunfais", "triunfo", "trocadilho", "trocais", "troçais",
"trocáveis", "troçáveis", "troco", "troço", "tromba", "trombas",
"trombeta", "trombetas", "trombo", "trompa", "trompas", "troncais",
"troncho", "tronco", "trono", "tropa", "tropas", "tropeço",
"tropicais", "tropo", "trotejo", "trova", "trovas", "trováveis",
"troviscais", "trovisco", "trucidáveis", "truco", "trufa", "trufais",
"trufas", "truncáveis", "trunfa", "trunfas", "trunfo", "trustificáveis",
"tuba", "tubais", "tubas", "tuberculizáveis", "tubo", "tucano",
"tufáveis", "tufo", "tumba", "tumbais", "tumbas", "tumultuáveis",
"tuna", "tunais", "tunas", "tunda", "tundas", "turba",
"turbas", "turbáveis", "turbina", "turbinais", "turbinas", "turbo",
"turbolejo", "turiferais", "turificáveis", "turma", "turmas", "turra",
"turras", "turro", "turturejo", "turturinho", "turva", "turvas",
"turvo", "tutela", "tutelas", "uberais", "ufano", "ugalho",
"uiva", "uivas", "uivo", "ulceráveis", "ultimáveis", "ultracivilizáveis",
"ultracolapso", "ultraestruturais", "ultraestruturáveis", "ultrafiltráveis", "ultrafiltro", "ultrapassáveis",
"ultrapurificáveis", "ultrassonografáveis", "umidificáveis", "undo", "unguento", "unificáveis",
"uniformáveis", "uniformizáveis", "universalizáveis", "uno", "untáveis", "upa",
"upas", "uralitizáveis", "urano", "urbanizáveis", "urina", "urinas",
"urro", "urtiga", "urtigais", "urtigas", "usáveis", "usina",
"usinas", "usináveis", "uso", "ustuláveis", "usura", "usuras",
"usurpáveis", "utilizáveis", "vaca", "vacais", "vacas", "vaciláveis",
"vacilo", "vacinais", "vacináveis", "vacuolizáveis", "vadia", "vadias",
"vadio", "vagabunda", "vagabundas", "vagabundo", "vagais", "vagamundo",
"vago", "vaia", "vaias", "validáveis", "valido", "valo",
"valorizáveis", "valsa", "valsas", "vampira", "vampiras", "vampiro",
"vangloriáveis", "vaporáveis", "vaporizáveis", "vaqueiro", "varais", "varáveis",
"vareja", "varejas", "varejo", "variáveis", "varicela", "varicelas",
"vasca", "vascas", "vasco", "vasculháveis", "vasoativáveis", "vasoativo",
"vasodilatáveis", "vassala", "vassalas", "vassalo", "vassoira", "vassoirais",
"vassoiras", "vassoura", "vassouras", "vasta", "vastas", "vasto",
"vaticináveis", "vavuco", "vazia", "vazias", "vazio", "vecejo",
"vedais", "vedáveis", "vegetais", "vegetáveis", "veia", "veias",
"veio", "vela", "velas", "velejáveis", "velhaca", "velhacas",
"velhaco", "velo", "veludo", "venalizáveis", "vencelho", "vendáveis",
"vendo", "veneno", "venerais", "veneráveis", "veniaga", "veniagas",
"venta", "ventanejo", "ventas", "vento", "ventura", "venturas",
"verbalizáveis", "verberáveis", "verdasca", "verdascas", "verga", "vergais",
"vergas", "vergasta", "vergastas", "vergáveis", "vergonha", "vergonhas",
"verificáveis", "vermelha", "vermelhais", "vermelhas", "vermelho", "vermina",
"verminais", "verminas", "vernaculizáveis", "vernais", "verno", "verrina",
"verrinas", "verruga", "verrugais", "verrugas", "verruma", "verrumas",
"versais", "versáveis", "versejais", "versejáveis", "versificáveis", "verso",
"verticalizáveis", "vertiginais", "vesicais", "vestais", "vetáveis", "veto",
"vexáveis", "viabilizáveis", "viajáveis", "vianda", "viandas", "viaticáveis",
"vicejo", "viciáveis", "vida", "vidas", "vidrais", "vidro",
"viga", "vigas", "vigoráveis", "vigorizáveis", "vilegiatura", "vilegiaturas",
"vilico", "vimo", "vincilho", "vinco", "vinculáveis", "vindicáveis",
"vindima", "vindimais", "vindimas", "vingáveis", "vinhais", "vinho",
"vinificáveis", "violais", "violáveis", "violenta", "violentas", "violentáveis",
"violento", "virais", "virguláveis", "virilizáveis", "virtuais", "visco",
"visgo", "visibilizáveis", "visitáveis", "vislumbro", "viso", "visto",
"vistoria", "vistorias", "vistoriáveis", "visualizáveis", "vitalizáveis", "vitamina",
"vitaminas", "vitaminizáveis", "vitimáveis", "vitrificáveis", "vituperáveis", "vivenciais",
"vivenciáveis", "vivificáveis", "vivo", "vizinha", "vizinhais", "vizinhas",
"vizinho", "vocabularizáveis", "vocacionais", "vocalizáveis", "vociferáveis", "vogais",
"volatilizáveis", "volatizáveis", "voltejáveis", "voluto", "voo", "votáveis",
"voto", "vozeira", "vozeiras", "vulcanizáveis", "vulgarizáveis", "vulnerais",
"vulneráveis", "vulto", "vurmo", "xabouco", "xanto", "xaveco",
"xeno", "zabumba", "zabumbas", "zangarilho", "zaragata", "zaragatas",
"zaragatoa", "zaragatoas", "zebra", "zebrais", "zebras", "zelo",
"zero", "zicho", "zimbrais", "zimbro", "zinco", "zirro",
"zona", "zonais", "zonas", "zoncho", "zonza", "zonzas",
"zonzo", "zoo", "zornais", "zuco", "zulo", "zumbeiro",
"zurro"])


posMas = set(["afinal", "agora",
"aí", "amanhã", "antes", "assim",
"claro", "coitada", "coitado", "confesso", "depois", "enfim",
"entretanto", "hoje", "sim", "sobretudo", "talvez"])


excMas = set(["a", "as", "com", "da", "das", "de", "do", "dos",
"em", "não", "o", "os", "que", "sem", "um",
"uma"])

# excUgeral  - excecao Plural U Geral - para palavras terminadas em u
excUgeral = set(["adeu", "campu", "corpu", "ilheu", "juventu", "manau",
"marcu", "mateu", "onibu", "onu", "plu", "pu", "statu", "venu", "viniciu",
"viru", "zeu"])


# excAos = Execao para palavras que nao variam em determinadas situacoes - regra do plural para o singular
excAos = set(["a", "ao", "agora", "amba", "ambo", "ante", "ao", "as", "cada",
"catorze", "cerca", "cinco", "cinquenta", "como", "da", "daquela", "daquele",
"daquilo", "de", "dela", "dele", "dessa", "desse", "desta", "deste",
"dezanove", "dezassete", "dezenove", "dezessete", "dezoito", "do", "doze",
"e", "ela", "entre", "exclusivamente", "extra", "giga", "gravemente",
"infelizmente", "infinitamente", "inter", "injustamente", "me", "mega", "meia",
"meio", "meno", "mini", "muito", "na", "nada", "nessa", "nesse", "nesta",
"neste", "no", "nove", "noventa", "numa", "nunca", "oitenta", "oito", "onze",
"outrora", "perpetuamente", "para", "pouca", "pouco", "pura", "quanta",
"quarenta", "quase", "quatorze", "quatro", "que", "quinze", "sessenta",
"sete", "setenta", "sua", "suas", "treze", "todo", "trinta", "vinte", "zero"])

# aacCao = Concordancia Feminino - masculino - substantivos - adjetivos  terminados em -a-  e  -o-
aacCao = {"a": "o", "A": "O"}

# aacCrase = Concordancia Feminino - masculino - substantivos - adjetivos  terminados em -a-  e  -o-
aacCrase = {u"à": "a","À": "A"}



# aacCfm = Concordancia Feminino - masculino - substantivos - adjetivos  terminados em or
aacCfm = {"alguma": "algum",
"Amiga": "Amigo",
"amiga": "amigo",
"antiga": "antigo",
"aquela": "aquele",
"boa": "bom",
"Cara": "Caro",
"cara": "caro",
"Da": "Do",
"da": "do",
"digna": "digno",
"Distinta": "Distinto",
"distinta": "distinto",
"Essa": "Esse",
"essa": "esse",
"Esta": "Este",
"esta": "este",
"futura": "futuro",
"legítima": "legítimo",
"mà": "mau",
"mesma": "mesmo",
"minha": "meu",
"muita": "muito",
"Na": "No",
"na": "no",
"nenhuma": "nenhum",
"nona": "nono",
"nossa": "nosso",
"nova": "novo",
"oitava": "oitavo",
"outra": "outro",
"pela": "pelo",
"pouca": "pouco",
"Prezada": "Prezado",
"prezada": "prezado",
"primeira": "primeiro",
"própria": "próprio",
"próxima": "próximo",
"quarta": "quarto",
"quinta": "quinto",
"segunda": "segundo",
"Senhora": "Senhor",
"senhora": "senhor",
"sétima": "sétimo",
"sexta": "sexto",
"sua": "seu",
"terceira": "terceiro",
"Uma": "Um",
"uma": "um",
"velha": "velho",
"vossa": "vosso"}

# aacPfas = Concordancia Plural Feminino -  substantivos - adjetivos  terminados em or
aacPfas = {"alguma": "algumas",
"Amiga": "Amigas",
"amiga": "amigas",
"antiga": "antigas",
"aquela": "aquelas",
"boa": "boas",
"Cara": "Caras",
"cara": "caras",
"Da": "Das",
"da": "das",
"digna": "dignas",
"Distinta": "Distintas",
"distinta": "distintas",
"Essa": "Essas",
"essa": "essas",
"Esta": "Estas",
"esta": "estas",
"futura": "futuras",
"legítima": "legítimas",
"mà": "màs",
"mesma": "mesmas",
"minha": "minhas",
"muita": "muitas",
"Na": "Nas",
"na": "nas",
"nenhuma": "nenhumas",
"nona": "nonas",
"nossa": "nossas",
"nova": "novas",
"oitava": "oitavas",
"outra": "outras",
"pela": "pelas",
"pouca": "poucas",
"Prezada": "Prezadas",
"prezada": "prezadas",
"primeira": "primeiras",
"própria": "próprias",
"próxima": "próximas",
"quarta": "quartas",
"quinta": "quintas",
"segunda": "segundas",
"Senhora": "Senhoras",
"senhora": "senhoras",
"sétima": "sétimas",
"sexta": "sextas",
"sua": "suas",
"terceira": "terceiras",
"Uma": "Umas",
"uma": "umas",
"velha": "velhas",
"vossa": "vossas"}

# aacPfsa = Concordancia Plural Feminino -  substantivos - adjetivos  terminados em or
aacPfsa = {"alguma": "algumas",
"Amigas": "Amiga",
"amigas": "amiga",
"antigas": "antiga",
"aquelas": "aquela",
"boas": "boa",
"Caras": "Cara",
"caras": "cara",
"Das": "Da",
"das": "da",
"dignas": "digna",
"Distintas": "Distinta",
"distintas": "distinta",
"Essas": "Essa",
"essas": "essa",
"Estas": "Esta",
"estas": "esta",
"futuras": "futura",
"legítimas": "legítima",
"màs": "mà",
"mesmas": "mesma",
"minhas": "minha",
"muitas": "muita",
"Nas": "Na",
"nas": "na",
"nenhumas": "nenhuma",
"nonas": "nona",
"nossas": "nossa",
"novas": "nova",
"oitavas": "oitava",
"outras": "outra",
"pelas": "pela",
"poucas": "pouca",
"Prezadas": "Prezada",
"prezadas": "prezada",
"primeiras": "primeira",
"próprias": "própria",
"próximas": "próxima",
"quartas": "quarta",
"quintas": "quinta",
"segundas": "segunda",
"Senhoras": "Senhora",
"senhoras": "senhora",
"sétimas": "sétima",
"sextas": "sexta",
"suas": "sua",
"terceiras": "terceira",
"Umas": "Uma",
"umas": "uma",
"velhas": "velha",
"vossas": "vossa"}



# conCfm = Concordancia Feminino - masculino - substantivos - adjetivos
conCfm = set(["alguma", "Amiga",  "amiga", "antiga", "aquela", "boa", "Cara",
"cara", "Da", "da", "digna", "Essa", "essa", "Esta", "esta", "futura",
"legítima", "mà", "mesma", "minha", "muita", "Na", "na",
"nenhuma", "nona", "nossa", "nova", "oitava", "outra", "pela", "pouca",
"Prezada", "prezada", "primeira", "própria", "próxima",
"quarta", "quinta", "segunda", "sétima", "sexta", "sua",
"terceira", "Uma", "uma", "velha", "vossa"])

# conPf = Concordancia Plural Feminino - substantivos - adjetivos
conPf = set(["algumas", "Amigas",  "amigas", "antigas", "aquelas", "boas", "Caras",
"caras", "Das", "das", "dignas", "Essas", "essas", "Estas", "estas", "futuras",
"legítimas", "màs", "mesmas", "minhas", "muitas", "Nas",
"nas", "nenhumas", "nonas", "nossas", "novas", "oitavas", "outras", "pelas",
"poucas", "Prezadas", "prezadas", "primeiras", "próprias",
"próximas", "quartas", "quintas", "segundas",
"sétimas", "sextas", "suas",
"terceiras", "Umas", "umas", "velhas", "vossas"])


# aacCmf = Concordancia  Masculino - feminino substantivos - adjetivos
aacCmf = {"algum": "alguma",
"Amigo": "Amiga",
"amigo": "amiga",
"antigo": "antiga",
"aquele": "aquela",
"bom": "boa",
"Caro": "Cara",
"caro": "cara",
"Do": "Da",
"do": "da",
"digno": "digna",
"Distinto": "Distinta",
"distinto": "distinta",
"Esse": "Essa",
"esse": "essa",
"Este": "Esta",
"este": "esta",
"futuro": "futura",
"legítimo": "legítima",
"mau": "mà",
"meu": "minha",
"nenhum": "nenhuma",
"No": "Na",
"no": "na",
"nono": "nona",
"nosso": "nossa",
"novo": "nova",
"oitavo": "oitava",
"outro": "outra",
"pelo": "pela",
"Prezado": "Prezada",
"prezado": "prezada",
"próprio": "própria",
"próximo": "próxima",
"quarto": "quarta",
"quinto": "quinta",
"Senhor": "Senhora",
"senhor": "senhora",
"sétimo": "sétima",
"seu": "sua",
"sexto": "sexta",
"terceiro": "terceira",
"Um": "Uma",
"um": "uma",
"velho": "velha",
"vosso": "vossa"}

# aacPmos = Concordancia  Plural Masculino - substantivos - adjetivos
aacPmos = {"algum": "alguns",
"Amigo": "Amigos",
"amigo": "amigos",
"antigo": "antigos",
"aquele": "aqueles",
"bom": "bons",
"Caro": "Caros",
"caro": "caros",
"Do": "Dos",
"do": "dos",
"digno": "dignos",
"Distinto": "Distintos",
"distinto": "distintos",
"Esse": "Esses",
"esse": "esses",
"Este": "Estes",
"este": "estes",
"futuro": "futuros",
"legítimo": "legítimos",
"mau": "maus",
"meu": "meus",
"nenhum": "nenhuns",
"No": "Nos",
"no": "nos",
"nono": "nonos",
"nosso": "nossos",
"novo": "novos",
"oitavo": "oitavos",
"outro": "outros",
"pelo": "pelos",
"Prezado": "Prezados",
"prezado": "prezados",
"próprio": "próprios",
"próximo": "próximos",
"quarto": "quartos",
"quinto": "quintos",
"Senhor": "Senhores",
"senhor": "senhores",
"sétimo": "sétimos",
"seu": "seus",
"sexto": "sextos",
"terceiro": "terceiros",
"Um": "Uns",
"um": "uns",
"velho": "velhos",
"vosso": "vossos"}

# aacPmso = Concordancia  Plural Masculino - substantivos - adjetivos
aacPmso = {"algum": "alguns",
"Amigos": "Amigo",
"amigos": "amigo",
"antigos": "antigo",
"aqueles": "aquele",
"bons": "bom",
"Caros": "Caro",
"caros": "caro",
"Dos": "Do",
"dos": "do",
"dignos": "digno",
"Distintos": "Distinto",
"distintos": "distinto",
"Esses": "Esse",
"esses": "esse",
"Estes": "Este",
"estes": "este",
"futuros": "futuro",
"legítimos": "legítimo",
"maus": "mau",
"meus": "meu",
"nenhuns": "nenhum",
"Nos": "No",
"nos": "no",
"nonos": "nono",
"nossos": "nosso",
"novos": "novo",
"oitavos": "oitavo",
"outros": "outro",
"pelos": "pelo",
"Prezados": "Prezado",
"prezados": "prezado",
"próprios": "próprio",
"próximos": "próximo",
"quartos": "quarto",
"quintos": "quinto",
"Senhores": "Senhor",
"senhores": "senhor",
"sétimos": "sétimo",
"seus": "seu",
"sextos": "sexto",
"terceiros": "terceiro",
"Uns": "Um",
"uns": "um",
"velhos": "velho",
"vossos": "vosso"}


# conCmf = Concordancia Masculino - Feminino - substantivos - adjetivos
conCmf = set(["algum", "Amigo", "amigo", "antigo", "aquele", "bom", "Caro",
"caro", "Do", "do", "digno", "Distinto", "distinto", "Esse", "esse", "Este", "este",
"futuro", "legítimo", "mau", "meu", "nenhum", "No", "no", "nono",
"nosso", "novo", "oitavo", "outro", "pelo", "Prezado", "prezado",
"próprio", "próximo", "quarto", "quinto", "Senhor",
"senhor", "sétimo", "seu", "sexto", "terceiro", "Um", "um",
"velho", "vosso"])

# conPm = Concordancia Plural Masculino - substantivos - adjetivos
conPm = set(["alguns", "Amigos", "amigos", "antigos", "aqueles", "bons",
"Caros", "caros", "Dos", "dos", "dignos", "Distintos", "distintos", "Esses",
"esses", "Estes", "estes", "futuros", "legítimos", "maus",
"meus", "nenhuns", "Nos", "nos", "nonos", "nossos", "novos", "oitavos",
"outros", "pelos", "Prezados", "prezados", "próprios",
"próximos", "quartos", "quintos", "Senhores", "senhor",
"sétimo", "seu", "sexto", "terceiro", "Uns", "uns",
"velhos", "vossos"])


# excMentes = Execao para palavras com - mente - que admitem plural -s-
excMentes = set(["acampamentense", "alimenteira", "alimenteiro",
"casamenteira", "casamenteiro", "cimenteira", "cimenteiro", "clemente",
"demente", "deprimente", "dormente", "entremente", "fermentelense",
"fremente", "gemente", "inclemente", "juramenteira", "juramenteiro",
"livramentense", "mente", "mentecapta", "mentecapto", "moimentense",
"monumentense", "movimentense", "pimenteiralense", "pimenteira",
"pimenteirense", "pimenteiro", "pimentense",  "premente", "pseudodemente",
"semente", "sementeira", "sementeiro", "temente", "testamenteira",
"testamenteiro", "tremente", "veemente", "vestimenteira", "vestimenteiro"])


# excMento palavras terminadas em mento - excecao a regra
excMento = set(["a", "aci", "ador", "aju", "ajura", "ali", "ama", "apara",
"api", "ator", "autola", "bisseg", "ce", "co", "coag", "comparti",
"comple", "conjura", "cumpri", "de", "decre", "des", "desci",
"desma", "desorça", "despara", "despavi", "despig",
"desseg", "dor", "e", "empara", "ense", "escar", "escra",
"esto", "experi", "ferra", "fo", "frag", "funda", "hemossedi",
"imple", "incre", "la", "leni", "lini", "macrosseg", "metassedi",
"movi", "orna", "para", "pedi", "predica", "reali", "reimple",
"repavi", "retroali", "sacra", "salpi", "se", "seg", "suple",
"transcomple"])


# excIsta palavras terminadas em ista que nao admitem variacao de genero
excIsta = set(["Avista", "avista", "Boavista", "Consista", "consista",
"Desista", "desista", "Exista", "exista", "Entrevista", "entrevista",
"Insista", "insista", "Invista", "invista", "Lista", "lista", "Mista", "mista",
"Persista", "persista", "Pista", "pista", "Prevista", "prevista",
"Resista", "resista", "Revista", "revista", "Vista", "vista"])


# excAdo palavras terminadas em ado e ido  que nao admitem variacao de genero ou com variacao rara
excAdo = set(["almoxarif", "arcebisp", "atent", "c", "cade", "cág",
"caj", "calif", "cond", "consul", "côvado", "cup", "d", "duc",
"eleitor", "emir", "episcop", "euromerc", "fíg", "g",
"hipermerc", "hortomerc", "juiz", "mar", "merc", "microd",
"miniatent", "minigram", "minimerc", "minitecl",
"multialmoxarif", "multimerc", "n", "operari", "pr", "presbiter",
"princip", "proletari", "protetor",
"reitori", "sáb", "s", "sample", "sen", "servoacion",
"superfadig", "supermerc", "supersold", "t", "tecl", "xax"])

# excFeMfoa palavras femininas terminadas em a e masculinas mfoa
excFeMfoa = set(["maioria"])




# plMascovb - substantivos masculinos tambem encontrados em conjugacoes verbais. Ex. roubo, acerto, conserto
plMascovb = set(["ababalho", "abafo", "abalo", "abano", "abanico",
"abelhudo", "abono", "abordo", "aborto", "abraço",
"abrigo", "abrolho", "abrunho", "açaimo", "acalanto",
"acalento", "açalmo", "acato", "aceiro", "aceno",
"acerbo", "acerto", "acervo", "achego", "acolheito",
"aconchego", "acosto", "acrescento", "adejo", "adereço",
"adestro", "adjunto", "adunco", "afeito", "afixo",
"afogo", "afoito", "agasalho", "agoiro", "agoirento",
"agouro", "agourento", "agravio", "aio", "ajoujo",
"alanco", "aldrabo", "aldravo", "alegro", "alfabeto",
"algarvio", "algueiro", "alimento", "alinhavo", "almejo",
"alquino", "alterno", "alvacento", "alveiro", "alvoroço",
"amanho", "amarelento", "amasio", "amasso", "ameaço",
"amuo", "ancho", "andarilho", "andejo", "anelo",
"anexo", "anilho", "antolho", "anzolo", "apanho",
"aparato", "apedrejo", "apeiro", "aperto", "apoceiro", "apoio",
"aquaplano", "arcaboiço", "arcabouço", "argueiro", "armadilho",
"arneiro", "arpejo", "arquejo", "arranco", "arrasto",
"arrastro", "arrebito", "arregaço", "arreganho", "arreneiro",
"arrepio", "arrocho", "arrolho", "arrufo", "arrulho", "artefato",
"asneiro", "assento", "asserto", "assombro", "assopro",
"assovio", "atafego", "aterro", "atilho", "atino",
"atocho", "atraso", "atulho", "avanço", "avio",
"azulejo", "bacelo", "bacorejo", "bafejo", "bafordo",
"balbucio", "baldo", "bambino", "bamburro", "banzo",
"barafundo", "barbelo", "bardo", "barreno", "barulho",
"basto", "batouro", "bedelho", "beiço", "beijo",
"belisco", "beloiro", "belouro", "benquisto", "besoiro",
"bibico", "bicarpelo", "bichano", "bicromato", "bilro",
"bimbalho", "bingo", "birro", "biscato", "biscoito",
"bispo", "bloco", "boato", "bocanho", "bocejo",
"bochacro", "bochecho", "bochincho", "bofeto", "boquejo",
"borboto", "borco", "borcelo", "bordo", "borrego",
"borrifo", "bosquejo", "boteco", "brejeiro", "brilho",
"brinco", "broco", "brolho", "broto", "bufo", "bugalho",
"bumbo", "bunho", "burgo", "burrico", "cabotino",
"cabrito", "cacarejo", "cachimbo", "cacho", "cachoeiro", "cachucho",
"cacifo", "cadastro", "cadelo", "caduco", "caibro",
"calaceiro", "calo", "calco", "calço", "calefrio",
"calemburgo", "cambaio", "cambalacho", "cambito", "campeiro",
"campino", "cancelo", "canto", "canudo", "canutilho",
"capoeiro", "capricho", "capucho", "caramelo", "carango",
"carbono", "carbonato", "carboneto", "carbonilo", "carboxilo",
"cardanho", "carepo", "cargo", "carijo", "carneiro",
"carpento", "carpinteiro", "carrego", "carreto", "carujo",
"caruncho", "cascalho", "cascavelo", "cascunho", "casmurro",
"caspacho", "casquilho", "castiço", "castigo", "catano",
"catracego", "catraio", "caudilho", "causo", "cavaco",
"cerco", "cerebelo", "cerro", "chabouco", "chafalho",
"chaleiro", "chambrego", "chamego", "chanasco", "chanfro",
"changueiro", "chapisco", "chapodo", "chapuço", "chaquiço",
"charango", "charuto", "chasco", "checo", "chichelo",
"chichorrobio", "chino", "chincalho", "chiqueiro", "chisco",
"chocalho", "choco", "chocho", "choito", "chono",
"choro", "chorro", "choto", "chouso", "chuço",
"chumaço", "chumbo", "churdo", "churrio", "chuvenisco",
"chuvilho", "chuvisco", "cianeto", "circuito", "cisco",
"claustro", "coalho", "cochilo", "coculo", "codejo",
"cogumelo", "coino", "coiso", "colapso", "colo",
"colmaço", "colmo", "comarco", "combusto", "começo",
"comento", "comoro", "compacto", "compeço", "complemento",
"compromisso", "conapo", "conceito", "conchavo", "concreto",
"confeito", "conforto", "conserto", "conservo", "consogro", "consolo",
"consumo", "conto", "contorno", "contrasseguro", "contrasselo", "contrato",
"controlo", "convulso", "corcovo", "corisco", "corricho",
"corripo", "coscoro", "cotanilho", "coto", "couto",
"covacho", "cromo", "cromofotolito", "cuanho", "curro", "curso",
"curto-circuito", "curvo", "curvejo", "custo", "dardo",
"debuxo", "decerto", "decoro", "decremento", "decreto",
"dedo", "degredo", "dejeto", "denodo", "descalabro",
"descalavro", "desconto", "descoberto", "desenho", "desgosto", "desnudo", "desonesto", "despenho",
"desperto", "destino", "desvairo", "dialecto", "dialeto",
"disperso", "direto", "domo", "domingo", "dorminhoco", "dreno",
"echacorvo", "edito", "eixo", "elenco", "embaraço",
"emboço", "embolso", "embrulho", "emparamento", "empeço",
"empecilho", "empelo", "empeno", "empenho", "empeso", "emprego",
"empupo", "empuxo", "enastro", "encabreimo", "encalço",
"encalho", "encanho", "encargo", "encoberto", "encoito",
"encontro", "encosto", "encouto", "endereço", "enduro",
"enforro", "engaldrapo", "engo", "engaranho", "engasgo",
"engenho", "enguiço", "enlevo", "enredo", "enseio",
"ensejo", "ensino", "ensosso", "entolho", "entorno",
"entrefolho", "entreoculto", "entreolho", "entrevero", "entroço",
"entropeço", "entrudo", "entulho", "entusiasmo", "envio", "envinhadalho",
"envisgo", "enxabelho", "enxalço", "enxaugo", "enxeco",
"enxerto", "enxurro", "ero", "erro", "erudito",
"esbalho", "esbarrunto", "esbirro", "esboço", "esbulho",
"escabelo", "escacilho", "escafelo", "escalambro", "escalavro",
"escaldo", "escalpo", "escambo", "escambro", "escapo",
"escarafuncho", "escardaço", "escarmento", "escarro", "escassilho",
"escocho", "esconjuro", "escorricho", "escouço", "escovalho",
"escravelho", "escrucho", "escudo", "escurento", "escuso", "esfoiro",
"esforço", "esfregulho", "esfumilho", "esgalho", "esgrafito",
"esguicho", "esmero", "esmicho", "esmorço", "espalhafato",
"esparro", "esparrimo", "espartilho", "espasmo", "espincho",
"espirro", "esquadro", "esquiço", "estalo", "estaleiro",
"estanco", "estardalhaço", "esteiro", "estilhaço", "estofo",
"estojo", "estorvo", "estouro", "estrago", "estralo",
"estrambelho", "estrampalho", "estribo", "estribilho", "estroixo",
"estudo", "exagero", "exalo", "exaspero", "exausto",
"excurso", "experimento", "expiro", "extracto", "extravaso",
"extravio", "facho", "fajuto", "falho", "falto",
"fandango", "fanico", "fanicho", "fanuco", "fardo",
"farelo", "fasto", "fecho", "fecundo", "fedelho",
"fedonho", "feltro", "fendrelho", "ferrolho", "festino",
"filo", "filhastro", "findo", "firro", "flaino",
"floco", "fogacho", "foleco", "folho", "fomento",
"formato", "fortaço", "fracasso", "fragmento", "frangalho",
"freso", "frito", "froco", "froixo", "fronteiro",
"frulho", "fueiro", "fundalho", "fundilho", "fundo", "funfo",
"furto", "fuso", "gabarito", "gadanho", "gadiço",
"gaio", "gaiato", "galgo", "galimo", "galricho",
"gamenho", "ganço", "garfo", "gargalejo", "gargarejo",
"garrancho", "garro", "garrulejo", "gatafunho", "gatuno",
"gemelgo", "gesso", "gingro", "golfo", "gongo",
"gorgorejo", "gosto", "governicho", "gracejo", "grampo", "graveto",
"grelo", "grifo", "grinfo", "grisalho", "grito",
"grogolejo", "grumo", "guedelho", "guincho", "guizo",
"heliporto", "hidrocarbono", "hirto", "ichacorvo", "implemento",
"importuno", "improviso", "incesto", "incremento", "indulto",
"ingresso", "insosso", "insulto", "interdito", "interino",
"intestino", "inverno", "iodo", "isento", "janelo",
"jorro", "jugo", "junco", "labelo", "lacaio",
"ladrilho", "lamento", "lamino", "lampo", "lampejo",
"lanço", "largo", "lastro", "latejo", "leigo",
"leito", "lenho", "libelo", "linimento", "linotipo",
"lixo", "lombilho", "lucro", "lustro", "luxo", "macavenco",
"macio", "macrossegmento", "macuco", "madeiro", "madraceiro",
"maduro", "magulho", "malandro", "malho", "malogro",
"malseguro", "manco", "mando", "manejo", "manhento",
"manifesto", "marco", "marroquino", "mascavo", "matreiro",
"matuto", "menoscabo", "menosprezo", "mergulho", "mestiço",
"metassedimento", "mexerico", "mijo", "mixo", "mocho",
"mofo", "moiro", "molengo", "molesto", "molho",
"monco", "mordomo", "motejo", "muro", "murcho",
"namoro", "namorico", "namoricho", "narciso", "nastro",
"negrito", "nevoento", "nicho", "nicotino", "nicto",
"nimbo", "nitrato", "nitreto", "nitruro", "nodo",
"nojo", "novelo", "octano", "oficio", "olfacto",
"olfato", "onzeneiro", "opalino", "orgulho", "orvalho",
"ouriço", "ousio", "outono", "pacto", "pago",
"paladino", "palanco", "palhiço", "palito", "palmo",
"palpo", "panaço", "pano", "pando", "panfleto",
"pango", "pantomimo", "paparico", "papouco", "paradoxo",
"parafuso", "paraninfo", "parapeito", "parasito", "pardejo",
"parelho", "parolo", "parro", "parrono", "pasmo",
"paspalho", "passo", "patego", "patoilo", "peçonhento", "pedicelo",
"pediplano", "peganho", "pejo", "penacho", "pendurico",
"penico", "penso", "pentelho", "pepino", "peralvilho",
"percalço", "perdigoto", "perigo", "perito", "perjuro",
"pertucho", "peso", "petanisco", "petisco", "petrecho",
"pezunho", "piacho", "pialo", "pio", "picirico",
"pigarro", "piloto", "pimpolho", "pino", "pincho",
"pingalho", "pingo", "pinoco", "pirilampo", "piteiro",
"plaino", "plebiscito", "pleito", "poento", "poio",
"poiso", "pombeiro", "portucho", "postiço", "pouso",
"pranto", "preceito", "predicamento", "prefixo", "prego",
"preito", "preparo", "prescruto", "presigo", "pretexto",
"prisco", "profaço", "profano", "professo", "prospecto",
"protesto", "prumo", "pugilo", "pulo", "pulso",
"punto", "puxo", "quartilho", "quebranto", "quedo",
"queijo", "quimbo", "rabeiro", "rabisco", "rabusco",
"ralo", "ralho", "ranço", "ranfo", "rango",
"rapigo", "rapilho", "rapto", "rascunho", "rasgo",
"raspanço", "rastelo", "rastilho", "rastolho", "ratino",
"realejo", "rebento", "reboliço", "rebuço", "rebuliço",
"reco", "reciclo", "recosto", "recuo", "referto",
"refluxo", "refracto", "regaço", "rego", "regino",
"registo", "regresso", "regueiro", "reino", "reiuno",
"relato", "relho", "relojo", "remanso", "remo",
"remendo", "repleto", "repolho", "repto", "requisito",
"rescaldo", "resgo", "resguardo", "resmungo", "respaldo",
"respeito", "respingo", "respiro", "responto", "ressaibo",
"resto", "restauro", "restito", "restolho", "restrelo",
"resvalo", "retonho", "retorno", "retroalimento", "rigo", "rilo",
"rincho", "ripanço", "risco", "ritmo", "rodopio",
"rodrigo", "rogo", "roliço", "ronco", "roto",
"roubo", "roufenho", "rousso", "rufo", "rumo",
"sabugo", "sabujo", "sacho", "sacholo", "sacramento",
"safano", "safo", "sagorro", "saldo", "salicilo",
"salino", "salmejo", "salmoeiro", "salpico", "salso", "salto",
"salvo", "sambenito", "sanguento", "sapoilo", "sapoulo",
"sarapico", "saro", "sardo", "sarilho", "sarrafo",
"sarro", "segmento", "segredo", "seleto", "septo",
"sepulcro", "sepulto", "sequestro", "seringonho", "serrino",
"seto", "seteiro", "setino", "sexo", "sibilo",
"sigilo", "silicato", "silvo", "sincelo", "sinuco",
"soalho", "sobejo", "soberbo", "sobrecabo", "sobrecasaco",
"sobrerroldo", "sobressalto", "socalco", "socresto", "solavanco",
"solfejo", "soluço", "soneto", "sonho", "sopapo",
"sopro", "soro", "sorvo", "soslaio", "sossego",
"souto", "suco", "sufixo", "sufoco", "sulavento",
"sulco", "sulfato", "sulfeto", "sulfito", "sulfureto",
"suplemento", "surrupio", "surto", "suspiro", "sussurro",
"susto", "sustento", "tabardilho", "taburno", "taco",
"tacho", "taino", "talo", "tamanco", "tanino",
"tardo", "taroco", "tarolo", "tarouco", "tarrinco",
"tarugo", "tasco", "tassalho", "telecomando", "telecurso",
"teleporto", "teletrabalho", "tempero", "templo", "tento",
"terceto", "terramoto", "teso", "tesoiro", "testico",
"tibiro", "tijolo", "tilo", "timbalejo", "tingo",
"tiro", "toco", "toiço", "toiro", "tomo",
"tombo", "topo", "toro", "torcicolo", "torno",
"torpedo", "torvo", "touço", "touro", "trabalho",
"trabelho", "trabuco", "trabulo", "tracolejo", "trago",
"trajo", "tramo", "trambolho", "tramelo", "transumano",
"traquejo", "trastempo", "trato", "traulito", "travo",
"treino", "treito", "trejeito", "tremelejo", "tremoço",
"treno", "trevisco", "trio", "tributo", "tricalho",
"trino", "tripo", "trissecto", "trissulco", "trito",
"triunfo", "trocadilho", "troco", "troço", "trombo",
"trono", "tronco", "troncho", "tropo", "tropeço",
"trotejo", "truco", "tucano", "tufo", "turbolejo",
"turro", "turturejo", "ufano", "ugalho", "uno",
"undo", "unguento", "urano", "urro", "uso",
"vacilo", "vagamundo", "vago", "valo", "vaqueiro",
"varejo", "vavuco", "vecejo", "veio", "velo",
"vencelho", "vendo", "veneno", "ventanejo", "verno",
"verso", "veto", "vicejo", "vidro", "vilico",
"vinco", "vincilho", "viso", "visco", "visgo",
"vislumbro", "visto", "voo", "voluto", "vulto",
"vurmo", "xabouco", "xanto", "xaveco", "xeno",
"zangarilho", "zelo", "zero", "zicho", "zimbro",
"zinco", "zirro", "zoo", "zoncho", "zonzo",
"zuco", "zulo", "zumbeiro", "zurro"])



# excplMfoa - excecao preposicao A - palavras masculinas terminadas em -o- Ex.: A outros
excplMfoa = set(["cert","outr"])

# excddMfoa - excecao - palavras masculinas terminadas em -o- Ex.: A outros
excddMfoa = set(["mesm", "próxim", "junt"])

# excplMasco - excecao preposicao A - palavras masculinas terminadas em -o- que nao possui correspondente feminino- Ex.: A principio
excplMasco = set(["critério", "exemplo", "longo", "pedido", "ponto", "pretexto", "princípio", "propósito", "tempo", "título"])

# exMuiMasco - excecao plural MUITOS - palavras masculinas terminadas em -o- que nao possui correspondente feminino- Ex.: Muito longos
exMuiMasco = set(["longo"])

# plvrMasxo - ESPECIAL - palavras masculinas terminadas em -o- que nao possui correspondente feminino- ano e auto
plvrMasxo = set(["ano", "auto"])

# plvrMasco - palavras masculinas terminadas em -o- que nao possui correspondente feminino-
plvrMasco = set(["abuso", "acento", "acórdão", "acréscimo",
"actinomiceto", "adultério", "advérbio", "afídio",
"alabardeiro", "aldeão", "alelo", "alfarrábio",
"algoritmo", "alívio", "almoxarifado", "altímetro",
"amentilho", "amperímetro", "amuleto",
"ancião", "andaço", "andanho", "andrajo",
"anfiteatro","ângulo", "anho", "animalejo",
"anteprojeto", "anticorpo", "antigênio",
"antro", "anúncio", "anuro", "aparelho",
"apedido", "aplauso", "aposento", "aqueduto",
"arauto", "arbítrio", "arbusto", "arcebispado",
"arcebispo", "arenito", "arilho", "armário",
"arquegônio", "arquétipo", "arquipélago",
"arquivo", "arranjo", "arreio", "artelho", "artesão",
"artifício", "artigo", "arvoredo", "aspecto",
"assassinato", "assassínio", "assunto", "atentado","átomo",
"atributo", "auspício", "autocarro",
"autoconsumo", "autodomínio", "autogiro", "autogoverno", "autógrafo",
"autorretrato", "auxílio", "axônio", "babuíno",
"bacano", "bafômetro", "bairro", "balcedo", "bálsamo", "balsero",
"bambúrrio", "báratro", "barbitúrico", "bardino",
"baronato", "basalto", "basídio", "basidiocarpo",
"basidiomiceto", "batólito", "bebedouro", "benefício",
"benzopireno", "berçário", "bicampeonato", "bicoco",
"binóculo", "binômio", "biombo", "biorritmo",
"biótopo", "bióxido", "bloco", "bócio", "bolbilho",
"bolbo", "bolero", "borato", "bororo",
"borrelho", "braco", "brinquedo", "brometo",
"brônquio", "bruxedo", "buço", "bulbo",
"busto", "buxo", "búzio", "cabelo", "cabo",
"cabrestilho", "cabresto", "cacareco", "cacto",
"cadeado", "caderno", "cágado", "caixilho", "cajado",
"calabouço", "calafrio", "calangro", "cálculo", "calendário",
"calhamaço", "califado", "câmbio", "campeonato", "campo",
"camundongo", "candelabro", "cântico", "carboidrato",
"carbúnculo", "carro", "cartapácio", "cartucho",
"catálogo", "cavalheiro", "cedro", "celacanto", "cemitério",
"cenário", "censo", "centímetro", "cérebro",
"cetáceo", "chaparro", "cesto", "cestinho",
"chimango", "chuvisqueiro", "ciclamato", "cidadão",
"cílio", "cincerro", "circo", "círculo",
"círio", "cirro", "cladódio", "clorofluorcarboneto",
"coágulo", "código", "coentro", "colégio",
"colmilho", "colonato", "colono", "colóquio",
"colosso", "comando", "comício", "cômoro",
"compêndio",
"concelho", "condado", "condomínio", "congresso", "conjunto",
"conselho", "consenso", "consócio", "consórcio",
"consulado", "conteúdo", "contexto", "contrabando",
"contrassenso", "contratempo",
"contributo", "convênio", "convento", "coradouro",
"corimbo", "cormo", "corniso", "corpo", "corpúsculo",
"corsário", "cotovelo", "côvado", "crédito",
"crepúsculo", "cristão", "critério",
"crocodilo", "cronômetro", "cruzeiro",
"cubículo", "cúmulo", "cupido", "currículo", "damasco",
"decâmetro", "decênviro", "decímetro", "declínio",
"defeito", "defluxo", "defumadoiro", "delírio", "delito", "demérito",
"demiurgo", "demônio", "dentículo", "depósito",
"desafio", "deságio", "desânimo", "desequilíbrio", "desiderato",
"desígnio", "densitômetro", "desmando", "despenhadeiro", "desperdício", "desporto",
"despropósito", "desvão", "detrito", "diacho",
"diácono", "diálogo", "diâmetro", "dieno",
"diestro", "diferendo", "dígito", "dilúvio",
"dínamo", "dinheiro", "dinossáurio", "dinossauro",
"diodo", "dipeptídio", "dipolo", "díscolo",
"dispêndio", "dispensário", "dissídio", "distrito",
"distúrbio", "ditério", "ditirambo", "dividendo",
"divórcio", "dolo", "domicílio", "domínio",
"dorso", "ducado", "ducto", "duto",
"edifício", "educando", "efeito", "eflúvio",
"eleitorado","eletrodo", "eletrólito", "elmo", "eloendro",
"emirado", "emirato", "empréstimo", "encômio",
"endósporo", "enteropneusto", "entretanto", "eólito",
"epicentro", "episcopado", "episódio", "epitáfio", "epitalâmio",
"epíteto", "epóxido", "equilíbrio", "equinócio",
"erário", "erióforo", "eritrócito", "escândalo",
"escapulário", "escárnio", "esconderijo", "escopro",
"escrúpulo", "espaço", "espantalho", "espargo", "espectro",
"espectrômetro", "espetáculo", "espiráculo", "espirilo",
"espólio", "esporângio", "esporófilo", "essênio",
"estábulo", "estágio", "estatuto", "estearato",
"estegossauro", "estendedoiro", "estereótipo", "estímulo",
"estômago", "estômato", "estrato", "estrépito",
"estróbilo", "estrogênio", "estrondo", "estúdio", "eucalipto",
"eunuco", "euro", "euromercado", "evangelho", "evento",
"excesso","exemplo", "exercício", "exército", "externato",
"fagócito", "falhanço", "falucho", "faniquito",
"fanisco", "fascículo", "fastígio", "fastio",
"feitiço", "feldspato", "fenômeno", "fenótipo",
"feromônio", "feto", "fiapo", "fiasco",
"ficomiceto", "fideicomisso", "fígado", "fígaro",
"figueiredo", "filantropo", "filósofo", "filtro", "flagício",
"flamingo", "flanco", "flóculo", "flósculo",
"fluorcarboneto", "fluoreto", "fluxo", "fogo",
"folguedo", "folhelho", "folículo", "fólio",
"folíolo", "formulário", "fósforo", "fotodíodo",
"fotóforo", "fraguedo", "freixo", "frontispício",
"fruto", "fulcro", "fulereno", "funcho", "fundo", "furúnculo",
"fuzo", "gabro", "gado", "gafanhoto",
"gálago", "galfarro", "gambozino", "gametóforo",
"gancho", "gânglio", "garbo", "garraio", "gárrulo",
"gastrólito", "gatázio", "gatilho", "geco",
"gênero", "gênio", "genocídio", "genótipo",
"genro", "gentio", "gerânio", "gerbo",
"germanófono", "gibelino", "ginásio", "girino",
"girondino", "giroscópio", "gládio", "glasto",
"gliceraldeído", "glícido", "glicosídeo", "glóbulo",
"goliardo", "gonzo", "grandaço", "grão",
"gravanço", "gravitino", "grêmio", "gringo",
"grupo", "guardanapo", "guelfo", "haleto", "hálito", "halogeneto",
"hausto", "hecatonquiro", "hemíptero", "hemisfério",
"hemistíquio", "hepatócito", "heterocisto", "hexâmetro",
"hiato", "hibisco", "hidátodo", "hidreto",
"hidrocarboneto", "hidrofólio", "hidroperóxido", "hidróxido",
"hieroglifo", "hieróglifo", "hipermercado", "hipnósporo", "hipódromo",
"hipogrifo", "hirsuto", "holótipo", "homicídio",
"homopolímero", "horóscopo", "hospício",
"hortomercado", "hussardo",
"ictiossauro", "idioblasto", "imbróglio", "império",
"impropério", "imunocomplexo", "incêndio", "incensário",
"inchaço", "indígeno", "inferno", "influxo", "infólio",
"infortúnio", "início", "inquérito", "instinto", "insucesso",
"insurrecto", "intelecto", "intercâmbio", "interlúdio",
"internato", "intuito", "inventário", "invólucro",
"isótopo", "jambo", "janízaro", "jato",
"jazigo", "jeito", "joelho", "jornaleco",
"júbilo", "juizado", "lacólito", "lacticínio",
"lajedo", "lanifício", "lariço", "látego",
"laticínio", "latifúndio", "latrocínio", "lavabo",
"lenço", "lepidodendro", "letargo", "leucócito",
"leucotrieno", "libreto", "licopódio", "lídio",
"linfócito", "linfonodo", "lipídio", "lípido",
"litígio", "litro", "livreco", "livreto", "livro", "lobato",
"lóbulo", "lóculo", "lodo", "logaritmo",
"logogrifo", "logotipo", "logradouro", "lombelo", "longo",
"longuíssimo", "loranto", "lugarejo", "lusito",
"mabeco", "malacopterígio", "malefício", "manaixo",
"mandato", "manômetro", "maremoto", "maribondo", "marimbondo",
"marido", "martírio", "mastro", "mastruço",
"matadouro", "matrimônio", "medo", "medinho",
"megálito", "megaquiróptero", "melanócito", "membro",
"memorando", "meningococo", "mentastro", "mercado", "mercúrio",
"mérito", "metabólito", "meteorito", "meteoro",
"método", "metro", "micróbio", "microcircuito", "microdado",
"micrômetro", "microprotalo",
"microscópio", "microtúbulo", "mijolo", "míldio",
"milímetro", "miniatentado", "minério", "miniconcurso", "minigramado",
"minimercado", "ministério", "miniteclado", "minuto",
"miradouro", "mistério", "módulo", "molosso",
"molusco", "monólito", "monopólio", "monturo",
"morticínio", "moscardo", "mostruário", "motociclo",
"motreco", "multímetro", "mundo", "município", "munirmo",
"murmúrio", "musaranho", "músculo",
"multialmoxarifado", "multimercado", "nababo",
"nabuco", "nanômetro", "nasuto", "naufrágio", "náutilo",
"navio", "necrológio", "nefro", "negócio",
"nematocisto", "nervo", "neurônio", "neutrino",
"nexo", "níscaro", "nitrito", "nódulo",
"noticiário", "núcleo", "nucleótido", "número",
"nuno", "obelisco", "obstáculo", "ocaso",
"oceanário", "oceano", "ocelo", "ofício",
"oleoduto", "oligopólio", "olivedo",
"olmedo", "olmo", "ombro", "opérculo", "operariado",
"opiato", "opúsculo", "oráculo", "orago",
"orango", "orangotango", "orfanato", "organito",
"orifício", "ornato", "ossículo", "otólito",
"ouvido","ovário", "ovócito", "ovo", "oxiúro",
"paço", "pagão", "palácio", "palco", "pálio",
"palmito", "pandeiro", "panegírico", "pântano", "paquímetro",
"parágrafo", "paraíso", "paralelepípedo", "parâmetro",
"parátipo", "pároco", "passatempo", "patíbulo",
"pátio", "patrimônio", "pecadilho", "pecíolo",
"pedaço", "pedregulho", "pedroiço", "pedrouço",
"pedúnculo", "peitilho", "penduricalho", "penhasco",
"pequerrucho", "percurso", "perímetro", "período",
"pesadelo", "pescoço", "petróleo", "píncaro",
"pinheiro", "planalto", "plantio", "polígono", "polímero",
"polinômio", "pólipo", "pômulo", "ponto", "poro",
"portento", "postigo", "povo", "prado",
"prazo", "preâmbulo", "presbiterado", "precipício", "preciso", "preço",
"preconceito", "prédio", "prefácio", "prejuízo",
"prelo", "prelúdio", "prêmio", "prenúncio",
"presépio", "presídio", "presságio", "prestígio",
"préstimo", "presunto", "primórdio", "principado",
"princípio", "privilégio", "processo", "prodígio", "pródromo",
"produto", "prolegômeno", "proletariado", "propósito", "prosélito",
"prostíbulo", "protalo", "protetorado", "protótipo",
"proveito", "provento", "provérbio", "pseudópodo",
"púlpito", "quartzo", "quelato", "quesito",
"questionário", "quiabo", "quício", "quilo",
"quilombo", "quilômetro", "quinteto", "quisto",
"rabino", "rabo", "racimo", "rácio",
"raciocínio", "radiano", "radioisótopo",
"radiotelescópio", "rapazito", "rebotalho", "receptáculo",
"recesso", "redenho", "rédito", "reduto",
"refrigério", "refúgio", "régulo", "reixelo",
"reitoriado", "relâmpago", "relicário", "relógio",
"remédio", "reóstato",
"resfolgadouro", "resíduo", "respiradouro", "resquício",
"retábulo", "revérbero", "riacho", "ricto",
"robalo", "rochedo", "rododendro", "rofo",
"romancero", "rombo", "rosário", "rostro",
"rubicundo", "rublo", "sábado", "sacerdócio",
"sacrário", "sacrifício", "safio", "salmo",
"santiago", "saquito", "sarcodino",
"sarrabulho", "seculo", "século",
"seio", "semáforo", "semanário", "seminário",
"senado", "senso", "sequilho", "seresteiro",
"serviço", "shampoo", "sicário", "sicômoro",
"silêncio", "silfo", "siltito", "silvedo",
"símbolo", "simulacro", "Sindicato", "sindicato", "sinédrio", "síntipo",
"sipaio", "sítio", "sobretudo", "sobrolho",
"solilóquio", "solo", "sorriso", "sortilégio", "sovaco",
"sublanço", "submódulo", "subproduto", "subsídio",
"substrato", "subterfúgio", "subúrbio", "sucesso",
"sudário", "sufrágio", "suicídio", "sultanato",
"sumiço", "sumidouro", "supercílio", "supermercado",
"supersoldado", "supertalento", "suplício", "suspiradouro",
"sustentáculo", "tabuleiro", "tajasno", "talento", "tamarindo",
"tamoio", "taquígrafo", "tarro", "tartufo",
"taumaturgo", "taxímetro", "teatro", "teclado", "tédio", "teixo",
"telecentro", "teledisco", "telégrafo", "teleósteo",
"telescópio", "teliósporo", "telômero", "tempo",
"tentáculo", "teodolito", "termômetro", "termóstato",
"terraço", "terremoto", "testículo", "texto",
"texugo", "tilito", "tímpano", "tirso",
"título", "tojo", "tomógrafo", "topázio",
"tórculo", "tornozelo", "torresmo", "tortulho",
"toutiço", "trajeto", "trambelho", "trecho",
"triângulo", "triciclo", "tríduo", "triglicerídio",
"triglicérido", "triúnviro", "tubérculo", "túbulo",
"tugúrio", "túmulo", "tumulto", "turíbulo",
"ultimato", "universo", "utensílio", "vácuo",
"vacúolo", "vaticínio", "vedonho", "veículo",
"velocímetro", "veludilho", "vencilho", "ventrículo",
"ventríloquo", "verbo", "verdugo", "veredicto", "vestiário",
"vestígio", "vestuário", "vesúvio", "viaduto",
"vídeo", "videojogo", "vidrilho", "vilarejo",
"vínculo", "vinhedo", "violino", "vitupério",
"viveiro", "vocábulo", "vômito",
"voto", "xaxado", "xerimbabo",
"xifosuro", "xisto", "zagalejo", "zéfiro",
"zoilo", "zuavo"])

# plvrMasca - palavras masculinas terminadas em -a- que nao possui correspondente feminino-
plvrMasca = set(["anacronauta", "aneurisma", "angioma", "aroma", "ateroma", "axioma",
"bioma", "bumba", "carcinoma", "carisma", "celoma",
"cinema", "clima", "cólera", "cometa",
"condroma", "cromossoma", "cronograma", "delta", "dia",
"diadema", "diafragma", "dilema", "diploma",
"DNA", "dogma", "drama", "druida", "ecossistema",
"ectoplasma", "eczema", "edema", "endosperma",
"enfisema", "enigma", "epigrama", "escriba",
"esperma", "esquema", "estigma", "estoma", "estratagema",
"estroma", "exoplaneta", "facínora", "feocromocitoma", "fliperama", "floema",
"fluxograma", "fusca", "gameta", "gâmeta",
"genoma", "glaucoma", "harmonograma", "hematoma", "hemograma",
"histograma", "histoplasma", "holograma", "ideograma",
"idioma",  "inseticida", "jota",
"lema", "lexema", "linfogranuloma", "linfoma",
"macrocromossoma", "magma", "mantra", "mapa",
"mega", "melanoma", "melodrama", "mesotelioma", "microclima",
"microgameta", "mieloma", "mimodrama", "moma",
"monograma", "neuroblastoma", "neuroplasma", "organograma",
"pampa", "panorama", "paradigma",
"parricida", "patriarca", "pederasta", "periodograma", "perisperma",
"pesticida", "pijama", "planeta", "plasma",
"poema", "poeminha", "prisma", "problema",
"programa", "protoplasma", "psicodrama", "psicoplasma",
"puma", "quediva", "quilograma", "rapazola", "ritidoma", "rizoma", "samba",
"sarcoma", "sintoma", "sismograma", "sistema",
"sofisma", "subprograma", "subsistema", "superatleta",
"sutra", "tapa", "teledrama", "telefonema",
"telegrama", "teorema", "tracoma", "trauma", "trema",
"videograma", "zabumba"])

# plvrFemo - palavras femininas terminadas em -o-
plvrFemo = set(["fonofoto", "Fonofoto", "foto", "Foto", "libido", "Libido", "lipo", "Lipo", "moto", "Moto", "tribo", "Tribo"])

# excplFema - palavras femininas/masculinas que podem ser usadas com os artigos a / o
excplFema = set(["camisa"])

# excplVirg - evita o dois virgula cinco
excplVirg = set(["sua", "vírgula"])

# plFemavb -  - substantivos femininos tambem encontrados em conjugacoes verbais.
plFemavb = set(["abelha", "agrura", "ajuda", "alça", "alfaia", "algema", "amarra",
"ameaça", "anexa", "anilha", "arruaça", "avenca", "baixa", "bicha", "cata",
"causa", "chama", "cobra", "compra", "conserva", "conta", "cria", "culpa", "cunha",
"cura", "custa", "cava", "cozinha", "dança", "descoberta", "disputa", "dura", "eira",
"elegia", "empena", "encosta", "entranha", "entrega", "entrevista",
"era", "escala", "escolha", "escolta", "escora", "escusa", "espera", "estampa", "fala", "falha",
"falta", "fatia", "força", "forma", "funda", "marca", "pergunta", "prova", "quebra", "saia", "volta"])

# plvrFema - palavras femininas terminadas em -a- que nao possuem correspondente masculino- a-o
plvrFema = set(["aba", "abertura", "abcissa", "abóbada", "abóbora",
"abraçadeira", "abreviatura", "absolutinha", "acácia", "academia",
"academiazinha", "açailandesa", "acetilcolina", "achinesa",
"acrobacia", "actina", "açucena", "acupunctura", "acurácia",
"adaga", "adega", "adrenalina", "advocacia", "adufa",
"aeromoça", "africanesa", "afronta", "agenda",
"agendinha", "agonia", "agorafobia", "agricultura",
"agroindústria", "agrovila","água", "agulha", "agulhinha",
"alambresa", "alameda", "alavanca", "albanesa",
"alçada", "alcachofra", "alcateia", "alcova", "alcunha", "aldeia",
"aldeola", "aldeota", "aldeiazinha", "alegada", "alegoria", "alegria",
"alegriazinha", "aleluia", "alfândega", "alfazema",
"alforreca", "alforria", "alga", "algazarra",
"álgebra", "algibeira", "alguma", "alhambresa",
"alheira", "alínea", "alíquota", "alma", "almádena",
"almalaguesa", "almanjarra", "alminha", "almofada",
"almofadinha", "alopatia", "alpaca", "alparca", "alparcata", "alpendrada",
"alpercata", "alpercatinha", "alquimia", "alsácia",
"altiva", "altura", "alva", "alvenaria",
"alverca", "alvinha", "alvorada", "alvoradinha",
"alvura", "amarantesa", "amargura", "amazona", "ambrosia",
"ambrosina", "ameba", "amêijoa", "ameixa", "ameixinha",
"amêndoa", "amendoeira", "amesterdanesa", "ametista",
"amídala", "amígdala", "amilopectina", "amnistia", "amora",
"amoreira", "amostra", "amostrinha", "ampola",
"ampulheta", "amsterdamesa", "anacleta", "anaconda", "anágua",
"analandesa", "analgesia", "anamesa", "anarquia",
"anatomia", "anaurilandesa", "anca", "andadura", "andaluza",
"andamanesa", "andorinha", "anedota", "anemia", "anémona", "anêmona",
"anestesia", "anfetamina", "anfíbola", "angina", "angiosperma", "angolesa", "angústia",
"anilina", "anistia", "anita", "anomalia", "anorexia",
"anquinha", "anta", "antecâmara", "antena",
"antera", "antessala", "antipartícula", "antipatia",
"antiportuguesa", "antropofagia", "aorta", "apalpadela", "apalpadinha", "apatia",
"apoplexia", "aposentadoria", "aposta", "apostila", "apostinha",
"aquarela", "arábia", "aragoesa", "aragonesa",
"aramesa", "arandela", "aranha", "aranhinha", "arapuca", "arara",
"ararinha", "arca", "arcada", "arcádia",
"arcadinha", "ardenesa", "ardósia","área", "areia",
"areiinha", "arena", "arenga", "aresta",
"argamassa", "argila", "argola", "argolinha",
"argúcia", "arguesa", "aristocracia", "arma",
"armadilha", "armadura", "arminha", "arnesa",
"aroeira", "aroeirinha", "arouquesa", "arquibancada",
"arquitetura", "arrábida", "arraia", "arraiada",
"arrelia", "arritmia", "arroba", "arruda", "arrúdia",
"arrumadeira", "artilharia", "artimanha", "asa", "ascuma",
"asfixia", "asinha", "asma", "asneira", "aspa",
"aspereza", "aspirina", "assamesa", "assembleia",
"assessoria", "assimetria", "assinatura", "assinaturazinha",
"assuanesa", "astronomia", "astúcia", "ata",
"atmosfera", "atonia", "atrofia", "audácia",
"auditoria", "aula", "aulinha", "aura",
"auréola", "aurícula", "aurilandesa", "aurora",
"austera", "australesa", "autarquia", "autobiografiazinha", "autodefesa",
"autoescola", "autoestima", "autoestrada", "autofagia", "autoforma", "automaca",
"autonomia", "autópsia", "autoria", "avaitanhesa", "avareza", "avaria",
"aveia", "avença", "avenida", "avenidinha", "aventura",
"aventurazinha", "avezinha", "avicultura", "avilesa", "avioneta", "axila",
"aza", "azáfama", "azaleia", "azambuja", "azambujinha",
"azeitona", "azeitoninha", "azenha", "azerbaidjanesa",
"azia", "baamesa", "baba", "babaçulandesa",
"babosa", "baboseira", "babosinha", "baça",
"bacalhoa", "bacharela", "bacia", "baciazinha",
"badalada", "badaladinha", "baderna", "baeta",
"baforada", "baforadinha", "baga", "bagaceira",
"bagatela", "bagresa", "baia", "baía",
"baianinha", "baiona", "baionesa", "baioneta", "bainha",
"baixela", "baixeza", "baixinha", "bala", "balada",
"baladinha", "balaiada", "balaustrada",
"balaustradinha", "balbúrdia", "baleia", "balela",
"balinesa", "balinha", "baliza", "balsa",
"balsinha", "baltistanesa", "bambina", "bambinela",
"bananeirinha", "bananinha", "bancarrota", "bandalheira",
"bandanesa", "bandeira", "bandeirada", "bandeiradinha",
"bandeirola", "bandeja", "bandejinha", "bandurra",
"bandurrinha", "bangladesa", "banqueta", "banquinha",
"banquetinha", "baqueta", "barafunda", "barateza", "baratinha",
"baraúna", "barba", "barbaria", "barbatana", "barbearia",
"barbela", "barbicha", "barbinha", "barcaça",
"barcarola", "barcelonesa", "bardana", "baremesa",
"barganha", "baronesa", "baronesinha",
"barreira", "barrela",
"barretina", "barrica", "barricada", "barricadinha",
"barriga", "barrigueira", "barrinha", "barroca",
"barulhada", "barulheira", "basca", "basílica",
"bastilha", "batalha", "batalhinha", "batata", "batatinha",
"bátega", "bateia", "bateria", "batina",
"batota", "batotinha", "batuta", "baunilha",
"bauxita", "bazófia", "bazuca", "bearnesa",
"beataria", "bebedeira", "beiçola", "beijoca",
"beira", "beirada", "beiradinha", "beirinha",
"beladona", "belbutina", "belda", "beleza",
"belezinha", "belida", "bemposta",
"benemérita", "benfeitoria", "bengala", "bengalesa",
"bengalinha", "benguela", "benguelinha",
"benzedura", "beripoconesa", "berlinda", "berlinesa",
"berma", "bermuda", "bermudesa", "bernarda", "bernardina",
"bernardinha", "bernesa", "besteira", "besteirinha",
"bestiaga", "beterraba", "bétula",
"bexiga", "bezerrinha", "bia", "bíblia", "biblioteca",
"biboca", "bicharada", "bicicleta", "bifada",
"bifadinha", "bigamia", "bigorna", "bilha",
"bilharda", "bilheteria", "bilirrubina", "binda",
"bintanesa", "biofábrica", "biomassa", "biosfera",
"biqueira", "birmanesa", "birosca", "birra",
"birrinha", "bisarma", "bisca",
"biscaia", "bitola", "bichinha", "biópsia",
"bisnaga", "blasfêmia", "blusa", "blusinha",
"boa", "boataria", "bobina", "boca",
"bocada", "bocadinha", "bocarra", "boceta",
"bochecha", "bochechinha", "boda", "bodega",
"boemia", "bofetada", "bofetadinha", "boga",
"boiada", "boiadinha", "boicininga",
"boina", "boinha", "boininha", "bojuda",
"bojudinha", "bolacha", "bolachinha",
"bolada", "boladinha", "boleia", "boleta",
"bolha", "bolhinha", "bolonhesa",
"bolota", "bolsinha", "bomba",
"bombarda", "bombinha", "bonecra", "bonina",
"boniteza", "bonomia", "boquilha", "boquinha",
"borboleta", "borboletinha", "borbulha", "borda", "bordadura",
"bordalesa", "bordelesa", "bordinha", "bordoada",
"borga", "borgonha", "borgonhesa", "borla",
"borneira", "boroa", "borra", "borracha",
"borracheira", "borrachinha", "borradela", "borralheira",
"borrasca", "borrega", "bossa", "bosta", "bostela",
"bostinha", "bota",
"botelha", "botica", "botija", "botina",
"botinha", "botoeira", "botulina", "brabantesa",
"braçada", "bragancesa", "braguesa",
"braguilha", "brama", "brancura",
"brandeburguesa", "brandemburguesa", "brandura",
"braquesa", "brasa", "brasilandesa", "brasilesa",
"brasinha", "bravata", "braveza",
"bravura", "breca", "brecha", "breda",
"brejeira", "brenha", "bretanha", "brida",
"briga", "brigada", "briguinha", "brilhantina",
"brincadeira", "brincadeirinha", "briófita","brisa", "brita",
"broa", "broca", "brocha", "brochura",
"broinha", "bromocriptina", "bronca", "broncopneumonia",
"broxa", "bruaca", "bruaquinha", "bruma",
"brusca", "bruteza", "bruxaria", "bruxariazinha",
"bruxelesa", "bruzundanga", "bucha", "buchada",
"buchinha", "buda", "buenolandesa",
"bula", "bulgaresa", "bulha", "bunda",
"bundinha", "buraca", "burbonesa", "burga",
"burgalesa", "burguesa", "burguesia",
"burla", "burocracia", "burocratesa",
"burundinesa", "bússola",
"butanesa", "buzina", "caatinga", "cabaça",
"cabacinha", "cabana", "cabeçada",
"cabeceira", "cabecinha", "cabeleira", "caboclinha",
"cabrinha", "caçada",
"caçadinha", "caçarola", "cachaça", "cachacinha",
"cachoeira", "cachoeirinha", "cachopa", "caçoada", "cacimba", "cacimbinha",
"cacofonia", "cadeia", "cadeiazinha", "cadeira", "cadeirinha",
"cadela", "cadelinha", "caderneta", "cadernetinha",
"cafelandesa", "caiçara", "caimanesa", "calabresa",
"calça", "caixinha", "cajazeira", "calcesa", "calcinha",
"caldeira", "caldeirinha", "calha", "caliça",
"calmaria", "calota", "cama", "camada", "camadinha", "câmara",
"camaronesa", "cambada", "cambadinha", "cambalhota", "cambraia",
"câmera", "camila", "camioneta", "camisa",
"camiseta", "camisetinha", "camisinha", "camisola",
"camisolinha", "camoesa", "campainha", "campanha",
"campidanesa", "campina", "camponesa",
"canaresa", "canastra", "canavesa", "cancela",
"cancelinha", "candidatura", "candonga", "candura",
"canela", "canelinha", "caneta", "canetinha",
"canga", "cangalha", "cangalhinha", "caninha", "canoa",
"canoinha", "canseira", "cantata", "cantatinha",
"canja", "cantiga", "cantilena", "cantina", "cantonesa", "cantoria",
"canzoada", "capa", "capela", "capelinha",
"capinha", "capitania", "capivara", "capoeira",
"capoeirinha", "capota", "cápsula", "captura",
"carabina", "carambola", "caranguejola", "carapaça", "carapuça",
"carapucinha", "caravana", "caravela", "carcaça",
"carcacinha", "carestia", "carga", "caricatura",
"carícia", "carlota", "carlotinha", "carnaúba", "carnaubeira",
"carniça", "carnificina", "carranca", "carranquinha", "carreira",
"carreirinha", "carreta", "carriça", "carricinha",
"carroça", "carroceria", "carrocinha", "carta",
"cartada", "cartadinha", "cartaginesa", "carteirinha", "cartilha",
"cartinha", "cartola", "cartucheira", "carvoeira",
"casadinha", "cascata", "cascatinha", "caserna", "casimira", "casinha",
"caspa", "cassilandesa", "casta", "castanha", "castanhinha",
"castanhola", "castinha", "catacumba", "catalanesa", "catanguesa", "catapora",
"catapulta", "catarata", "catarina", "categoria", "catinga", "catingueira",
"catita", "cauda", "caudinha", "cautela", "cavalaria", "cavalgada",
"cavalgadinha", "cavalgadura", "cavalgata", "cavalhada",
"caveira", "caveirinha", "caverna", "caverninha",
"cazaquistanesa", "cebola", "cebolinha", "cédula",
"cegonha", "cegueira", "ceia", "ceifa", "cela",
"celeuma", "célula", "célulazinha", "cena",
"cenoura", "censura", "centelha", "centelhinha", "centena", "cepa",
"cera", "cerda", "cereja", "cerejeira", "cerejinha", "ceroula", "certeza",
"certezinha", "cerveja", "cervejaria", "cervejinha",
"cesta", "cevada", "chácara", "chacina",
"chacota", "chaga", "chalaça", "chaleira",
"chamadinha", "champanhesa", "chancela",
"chancelaria", "chapa", "chapada", "chapinha",
"charada", "charadinha", "charolesa", "charrua", "charutaria",
"charuteira", "chatinha", "chávena", "chefia", "chegada",
"chegadinha", "cheiinha", "chibata", "chibatinha", "china", "chinesa",
"chita", "choça", "choradeira", "choupana", "chula",
"chusma", "chuteira", "chuva", "chuvinha", "cidadania",
"cidadela", "cifra", "cigarreira", "cilada",
"cilha", "cimalha", "cimeira", "cimenteira", "cinemateca", "cingalesa",
"cintura", "cinturinha", "circunvizinha",
"cisterna", "clareira", "clareza", "cláusula",
"clausura", "clava", "clavícula", "clavina", "clavinha",
"clientela", "clorofila", "coalhada", "coalhadinha", "cobaia",
"cobardia", "cobertura", "cobiça", "cobrinha",
"cocaína", "cócega", "coelha",
"coelhinha", "cocheira", "cochinchinesa", "cochinesa",
"coimbresa", "coisa", "coisinha", "colcha",
"coleta", "coletânea", "coletinha", "colheita",
"cólica", "colina", "colinha", "colmeia",
"coluna", "comarca", "comédia", "comenda",
"comendadeira", "comidela", "comidinha", "comitiva", "companhia",
"comporta", "compostura", "compota", "compoteira", "compreendia",
"compressa", "comprinha", "comuna", "concha", "conchinha", "concórdia",
"concubina", "condessa", "condessinha", "conduta", "confeitaria", "confraria",
"congolesa", "conjectura", "conjetura", "conjuntura", "consulta", "consultoria",
"contenda", "contrafaixa", "contraforça", "contraoferta",
"contrapartida", "contraproposta", "contraporca",
"contraprova", "contrarrevolta", "contrassenha", "conversa", "conversinha",
"cooperativa", "coordenadoria", "cópia",
"cópiazinha", "cópula", "corça", "corda",
"cordilheira", "cordinha", "cordisburguesa", "cordislandesa",
"cordofanesa", "cordovesa", "corina", "corneta",
"cornetinha", "corninglesa", "cornualhesa", "cornucópia",
"coroa", "corôa", "coroinha", "corola",
"coronha", "corregedoria", "correia", "correnteza",
"correntezinha", "correria", "correta", "corrida",
"corridinha", "cortesia", "cortina", "cortininha",
"coruja", "corujinha", "corunhesa", "corveta",
"cosipa", "costa", "costela", "costeleta", "costinha", "costura",
"costureirinha", "cota", "cotanesa", "cotia",
"cotinha", "cotovia", "couraça", "courela",
"cousa", "cova", "covardia", "covinha",
"coxa", "coxia", "coxinha", "cozedura", "cracovesa",
"craonesa", "cratera", "criança", "criançada",
"criancinha", "criatura", "criaturinha", "crina", "cripta", "crisálida",
"cristalandesa", "crixalandesa", "crosta", "crua", "crucilandesa", "crueza",
"cruzadinha", "cruzelandesa", "cueca",
"cuequinha", "cuia", "culatra", "culpadinha", "cultura",
"cupromanganesa", "cúpula", "curadoria",
"curva", "curvatura", "curvinha", "custódia",
"dádiva", "dama", "danceteria", "dancinha",
"danesa", "daroesa", "data", "debandada",
"debulhadeira", "década", "defensoria", "defesa",
"delegacia", "delfina", "delfinesa", "delicadeza",
"delícia", "demanda", "demandinha", "demasia",
"democracia", "demora", "dentada",
"dentadinha", "dentadura", "derrocada", "derrota",
"derrotinha", "desarmonia", "descarga", "descida",
"descidinha", "descompostura", "descortesia", "desculpa",
"desdita", "desembocadura", "desenvoltura", "desfeita",
"desfilada", "desfiladinha", "desforra", "desgraça",
"desgraceira", "desonra",
"desova", "despensa", "despenseira", "despesa",
"despesinha", "despreza", "destilaria", "destreza",
"desvalia", "desventura", "deusa", "devassa",
"dezena", "diabrura", "diáfana", "diarreia",
"diáspora", "diatomácea", "dica", "dicotomia", "diepesa",
"dieta", "difteria", "dinamarquesa", "dinamarquesinha",
"dinastia", "dinheirama", "dioptria", "diplomacia", "direta",
"diretoria", "disciplina", "discórdia", "discoteca",
"disenteria", "dislexia", "dispneia",
"distrofia", "ditadura", "diva",
"dívida", "divinolandesa",
"divisa", "dobra", "doca", "doçaria", "doçura",
"doencinha", "dominiquesa", "donzelinha",
"dopamina", "dórica", "dormida", "dormidinha", "dorna",
"dosimetria", "douradura", "doutora", "doutrina",
"draga", "droga", "drogaria", "drupa",
"ducha", "duchinha", "duna", "duplicata",
"duquesa", "dureza", "dúvida", "dúzia",
"economesa", "economia", "editoria", "eficácia",
"elvira", "ema", "embaixada", "embaixadinha",
"embira", "embocadura", "emenda", "ementa",
"empada", "empadinha", "empáfia", "empatia",
"empresa", "encefalopatia", "enciclopédia", "encomenda",
"encruzilhada", "encruzilhadinha", "endorfina", "endoscopia", "enfermaria",
"engenharia", "engomadeira", "enguia", "enseada",
"enseadinha", "entrada", "entradinha", "entrelinha", "entremesa",
"entressafra", "envergadura", "enxada", "enxadinha",
"enxaqueca", "enxurrada", "enxurradinha", "enzima",
"epidemia", "epifania", "epístola", "epopeia",
"ergonomia", "eritreia", "errata", "erva", "ervilha",
"ervilheira", "ervinha", "esbelteza", "escada",
"escadaria", "escadinha", "escaleira", "escama",
"escapada", "escapadinha", "escápula", "escarcela",
"escarlatina", "escarpa", "escherichia", "escocesa",
"escócia", "escola", "escolinha", "escondia",
"escopeta", "escopeteira", "escorva", "escotilha",
"escova", "escovinha", "escravaria", "escravatura",
"escritura", "escuderia", "escultura", "esculturinha",
"escuma", "escumilha", "escuta", "esfera",
"esgrima", "esmeralda", "esmeraldinha", "esmola",
"esmolinha", "espada", "espadana", "espadela",
"espadilha", "espadinha", "espádua", "espalda",
"espanhola", "espátula", "especiaria", "espectrometria",
"espectroscopia", "espertalhona", "esperteza",
"espessura", "espiga", "espingarda", "espinhela",
"esplanada", "esplanadinha", "esponja", "esponjinha",
"espora", "esporinha", "espórtula", "espuma",
"esquadra", "esquadria", "esquadrinha", "esquina",
"esquiva", "esquizofrenia", "estaca", "estacada",
"estacadinha", "estadia", "estafeta", "estamenha",
"estampinha", "estanqueira", "estaquinha",
"estátua", "estatuazinha", "estatueta", "estatura", "estearina",
"esteira", "estesia", "estola", "estopa",
"estopinha", "estrada", "estradinha", "estrangeirada",
"estrangeiradinha", "estranheza", "estrasburguesa", "estratégia",
"estratosfera", "estrebaria", "estreia",
"estreiteza", "estreitinha", "estrela", "estrelinha",
"estrema", "estremadura", "estreptomicina", "estriga",
"estrutura", "estufa", "estultícia", "etapa",
"etapinha", "etiopesa", "etiqueta", "etnia",
"eucaristia", "euforia", "eugenia", "europeia",
"eva", "evasiva", "fábrica", "fábula",
"faca", "facada", "facadinha", "façanha",
"faceta", "facha", "fachada", "fachadinha",
"facúndia", "fada", "fadiga", "fadinha",
"fagulha", "fagulhinha", "faia", "faísca",
"falácia", "falcoaria", "falda",
"faltinha", "fama", "familiazinha", "familinha",
"fanfarra", "fanga", "fantasia", "fantasmagoria",
"faquinha", "farda", "fardeta", "fardetinha",
"fardinha", "farfalha", "faríngea", "farmácia",
"farinha", "farmacopeia", "farofa", "farofinha", "farpa", "farpela",
"farpinha", "farra", "farroupeira", "farroupilha",
"farrusca", "farsa", "fartura", "farturinha",
"fascíola", "fasquia", "fatídica", "fatiota",
"fatura", "faúlha", "fauna", "fausta",
"faustina", "fava", "faveira", "favela",
"favinha", "faxina", "fazenda", "fazendinha",
"fazendola", "fechadura", "fécula", "fedelha",
"fedentina", "fedra", "feijoada", "feira",
"feirinha", "feita", "feitiçaria", "feitoria", "feitura",
"feiura", "feiurinha", "felícia", "felpa",
"fêmea", "fenciclidina", "fenda",
"fenícia", "fera", "fereza", "feroesa",
"ferradura", "ferradurinha", "ferramenta", "ferrenha",
"ferroada", "ferroadinha", "ferromanganesa", "ferrovia",
"férula", "fervura", "fervurinha", "festa",
"festina", "festinha", "fiandeira", "fibra",
"fibrina", "fibromialgia", "fíbula", "ficha",
"fichinha", "fida", "fidalguia", "figueira",
"figueirinha", "figura", "figurinha", "fila",
"filantropia", "filatelia", "filáucia", "fileira",
"filharada", "filipinesa", "filogenia", "filomena",
"filosofia", "filoxera", "fímbria", "finesa",
"fineza", "finlandesa", "finlandesinha", "finta",
"finura", "firma", "firmeza", "firmezinha",
"firminha", "fisionomia", "fissura", "fístula",
"fita", "fiteira", "fitinha", "fivela",
"fivelinha", "flâmula", "flanela",
"flanelinha", "flauta", "flecha", "flechinha",
"fleuma", "flora", "flórea", "florentina",
"floresta", "floricultura", "florinha", "flotilha",
"fobia", "focinheira", "foda", "fodinha",
"fofoca", "fofoquinha", "fogaça", "fogueira",
"fogueirinha", "foiçada", "foiçadinha", "folga",
"folha", "folhinha", "folia", "forca",
"forcinha", "formanda", "formatura", "fórmica",
"formiga", "forminha", "formosura", "fórmula",
"fornada", "fornalha", "forneira", "forneirinha", "forquilha",
"forrinha", "fortaleza", "fortuna", "fotocópia",
"fotografiazinha", "fotogrametria", "fotonovela", "fotosfera",
"fóvea", "fraga", "fragata", "fralda", "fraldinha",
"frama", "francela", "francesa", "francesinha",
"franja", "franjinha", "franqueza",
"franquia", "fraqueza", "fratura", "frauta",
"frecha", "freguesa", "freguesia", "freguesinha",
"freiberguesa", "freima", "freira", "freirinha",
"frescura", "fresquinha", "fressura", "fresta",
"frestinha", "fria", "frieira", "frieza", "frígia",
"frigideira", "fritura", "fronha", "frontaria",
"fronteira", "frota", "fruta", "frutaria", "fruteira",
"fruticultura", "fuampa", "fuba", "fuça",
"fuga", "fugidia", "fujona", "fulva",
"fumaça", "fumaceira", "fumacinha", "fundura",
"funérea", "fúrcula", "furna", "fuzilaria",
"gabardina", "gabela", "gábia", "gabonesa",
"gadelha", "gaeta", "gaforina", "gagueira",
"gaia", "gaiata", "gaiola", "gaiolinha",
"gaita", "gaitinha", "gaivota", "gaivotinha",
"gajeira", "gala", "galantaria", "galanteria", "galáxia",
"galba", "galeota", "galera", "galeria",
"galerinha", "galesa", "galinha", "galhada",
"galharda", "galhardia", "galharia", "galheta",
"galhofa", "galícia", "galileia", "galinhola",
"galinicultura", "galocha", "galopada", "galopadinha",
"galuresa", "galvanoplastia", "gamaglobulina", "gamba",
"gâmbia", "gambiarra", "gamboa", "gamela", "gametófita",
"gana", "ganapada", "gandaia", "gândara",
"gandaresa", "gandra", "gandresa", "ganesa",
"ganga", "gangorra", "gangorrinha", "gangrena",
"garantia", "garapa", "garatuja",
"garça", "garcesa", "garcinha", "gardunha",
"garfada", "garfadinha", "gargalhada", "gargalhadinha",
"gargalheira", "garganta", "gargantilha", "gárgula", "garnacha",
"garoa", "garonesa", "garotada", "garotadinha", "garotinha",
"garoupa", "garoupinha", "garra", "garrafa",
"garrafeira", "garrafinha", "garraiada", "garrana",
"garrincha", "garrocha", "garrota", "garrucha", "garrula",
"garupa", "garvonesa", "gasista", "gasolina",
"gastromancia", "gastronomia", "gateira", "gatinesa",
"gauchada", "gaulesa", "gávea", "gavela",
"gaveta", "gavetinha", "gavota", "gaza",
"gazela", "gazeta", "gazetilha", "geada",
"geadinha", "geba", "geesa",
"geladeira", "gelara", "gelatina",
"geleia", "gelosia", "gema", "genebra",
"genebresa", "generala", "geneta", "gengiva",
"genoesa", "genovesa", "gentalha", "gentamicina",
"gentileza", "genuesa", "geometria", "geriatria",
"giba", "gibiteca", "giesta", "giganta",
"gigantinha", "gincana", "ginecomastia", "gineta",
"ginja", "gipsofila", "girafa", "girândola",
"gitana", "glândula", "glicerina", "glicina",
"gliptoteca", "gloríola", "goa", "goanesa",
"goela", "goesa", "goiaba", "goiabada", "gola",
"goleada", "goleadinha", "golfada", "golfadinha",
"golinha", "golpelha", "goma", "gonadotrofina", "gôndola",
"gonorreia", "gordinha", "gordura", "gordurinha", "górgona",
"gorja", "gorjeta", "gorjetinha", "gostosura",
"gota", "goteira", "gotinha", "governanta", "graça",
"gracinha", "gradezinha", "gradinha", "grafia", "grafiazinha", "gralha",
"gralhada", "gralhadinha", "gramínea", "grana", "granada",
"granadinha", "grandeza", "grandura", "graninha",
"granja", "granjinha", "granulometria", "grapa",
"graúna", "gravata", "gravatinha", "gravimetria",
"gravura", "gravurinha", "graxa", "graxinha",
"greda", "greguesa", "grelha", "grenha",
"greta", "grilheta", "grimpa", "grinalda",
"griolandesa", "grisalha", "griseta", "gritaria",
"groenlandesa", "gronelandesa", "groselha", "grosseria",
"grossura", "grota", "grua", "grulhada",
"grulhadinha", "gruta", "grutinha", "guabiroba", "guadramilesa",
"guaiaca", "guajaratuba", "guamesa", "guampa",
"guampada", "guanina", "guarita", "guarnecia",
"guasca", "guedelha", "gueira", "guelra", "guerra",
"guerrilha", "guerrinha", "guiana", "guianesa",
"guida", "guiga", "guilhotina", "guinada",
"guinadinha", "guitarra", "gula", "guloseima",
"gurguesa", "guria", "gurizada", "gurupema",
"gusa", "hamburguesa", "harmonia", "harpa",
"harpia", "hasta", "hástea", "hástia",
"havanesa", "hegemonia", "hégira", "helena",
"heliosfera", "hemeralopia", "hemiplegia", "hemocultura",
"hemofilia", "hemoglobina", "hemolinfa", "hemorragia",
"hena", "heparina", "hepatoscopia", "hera",
"herançazinha", "herbicida", "hercogamia", "herculandesa",
"hercúlea", "heresia", "heroína", "hetaira",
"heterodoxia", "heterogenia", "heteromorfia", "heterosfera",
"hialina", "hica", "hidra", "hidrazina",
"hidrocefalia", "hidrocortisona", "hidrofilia", "hidrofobia",
"hidrolandesa", "hidromancia", "hidropisia", "hidrosfera",
"hidrovia", "hidroxizina", "hiena", "hierarquia",
"hifa", "hinterlandesa", "hioscina", "hipercolesterolemia",
"hiperestesiada", "hipergamia", "hipermetropia", "hiperonímia",
"hipertermia", "hipertrofia", "hipervolemia", "hipocondria",
"hipocrisia", "hipoteca", "hipotenusa", "hipotermia",
"hipovalva", "hirta", "hirudina", "histamina",
"histerectomia", "histeria", "historieta", "hola",
"holanda", "holandesa", "holandesinha", "holandilha",
"homeopatia", "homeostasia", "homeotermia", "homilia",
"homofobia", "homogenia", "homonímia", "honconguesa",
"honra", "hora", "horda", "horinha",
"hormona", "horneblenda", "horrenda", "hórrida",
"horsa", "horta", "hortaliça", "hortalicinha",
"hortênsia", "horticultura", "hortinha", "hóspeda",
"hospedaria", "hóstia", "hua", "hula",
"hulha", "humílima", "hungaresa", "iara",
"icterícia", "ideia", "ideiazinha", "idiossincrasia",
"idiotia", "idolatria", "igreja",
"igrejinha", "iguaria", "ilha", "ilhota",
"ilíada", "iliteracia", "imbituba", "imodéstia",
"imperícia", "impostura", "imprensa", "imprimadura",
"impureza", "imunoglobulina", "incerteza",
"incultura", "indelicadeza", "indisciplina", "indochina",
"indochinesa", "indústria", "ineficácia", "inépcia",
"inércia", "infâmia", "infanta", "infantaria",
"influenza", "infraestrutura", "inglesa", "inglesinha",
"inhaca", "injustiça", "inópia", "insídia",
"inspetoria", "insulina", "inteireza", "intemerata",
"interferometria", "interlíngua", "internetesa", "inúbia",
"invernada", "inverneira", "investidura", "ionosfera",
"ipueira", "iraúna", "irlandesa", "irlandesinha",
"irma", "ironia", "iroquesa", "isca",
"islandesa", "isóbata", "isocórica",
"isoloma", "isomorfia", "isoniazida", "isonomia",
"isostasia", "isoterma", "isotropia",
"itacoatiara", "ivolandesa", "jabuticaba", "jabuticabeira",
"jaca", "jacobina", "jacuba", "jacutinga",
"jaguatirica", "jagunçada", "jaleca", "jamanta",
"jamba", "janaína", "jandaia", "janela",
"janelinha", "jangada", "jangadinha", "janianesa",
"janota", "janta", "jantarola", "japa",
"japana", "japonesa", "japonesinha", "jaqueta",
"jaquetinha", "jaquiranaboia", "jararaca", "jararaquinha",
"jarreta", "jarreteira", "jaula",
"javanesa", "jazida", "jazidinha", "jeira",
"jerarquia", "jeremiada", "jeropiga", "jetica",
"jia", "jiboia", "jinguba", "joalharia",
"joalheria", "joana", "joaninha", "joanita",
"joelhada", "joelhadinha", "jogatina",
"joia", "joiazinha", "jornada", "jornalesa",
"jota", "jotinha", "juba", "juçara",
"jucunda", "judeia", "judia",
"judicatura", "juíza", "juncinha", "jungia",
"juntura", "jura", "jurema", "jureminha",
"jurubeba", "jurujuba", "justeza", "justiça",
"juta", "jutlandesa", "labareda", "labuta",
"laca", "lacuna", "ladeira", "ladeirinha",
"ladina", "ladra", "ladrona", "lagartixa",
"lagoa", "lagoinha", "lagosta", "lagostinha",
"lágrima", "laguna", "laia", "laja",
"lájea", "lajinha", "lama", "lamba",
"lambada", "lambadinha", "lâmina", "lamivudina",
"lampa", "lâmpada", "lamparina", "lampinha",
"lampreia", "lampreiinha", "lançadeira", "lanceta",
"landesa", "langonha", "lanolina", "lanterna",
"lanterninha", "lapa", "lapela", "lapinha",
"lapiseira", "laqueadura", "laranjeira", "laranjeirinha",
"lareira", "lareirinha", "larga", "largada",
"largadinha", "largueza", "largura", "larva",
"lasanha", "lasca", "lascívia", "lasquinha",
"lassa", "lástima", "lata", "latada",
"lataria", "latinaria", "latinha",
"latrina", "lavandaria", "lavanderia", "lavareda",
"lavoura", "lavra", "lavratura", "lazarina",
"lazeira", "leandra", "lectina", "leda",
"legenda", "legislatura", "légua", "leitaria",
"leitura", "leiva", "lembrancinha", "lemna",
"lena", "lençaria", "lenda", "lengalenga",
"lenha", "lenhina", "lentilha", "leoa",
"leonesa", "leonina", "leopoldina", "lepra",
"lesma", "letra", "letrinha", "leucemia",
"leucena", "levadiça", "levedura", "leveza",
"leviana", "levítica", "lexia", "lhama",
"lhaneza", "lhanura", "liamba", "liana",
"libanesa", "libanesinha", "libânia", "libélula",
"libra", "liça",
"licenciatura", "licencinha", "lichia", "lida",
"lídima", "lidocaína", "ligadura", "ligaresa",
"ligeireza", "lignina", "lígula", "lila",
"lima", "limburguesa", "limeira", "limonada",
"limpeza", "limpezinha", "lina", "lindeza",
"linfa", "lingala", "língua", "lingueta",
"linguetinha", "linguiça", "linguicinha",
"linhaça", "lionesa", "liornesa", "lira",
"lisboesa", "lisbonesa", "lisinha",
"lisonja", "lisozima", "lista", "listeria",
"listinha", "lisura", "liteira", "literatura",
"litosfera", "lívia", "lixeirinha",
"lixívia", "lobotomia", "lôbrega", "loca",
"locanda", "locandeira", "logomarca",
"logudoresa", "loisa", "loja", "lojeca",
"lojinha", "lombada", "lombeira", "lombriga",
"lona", "lonca", "londresa", "londrina",
"longura", "lonjura", "lontra",
"lontrinha", "loquela", "loratadina", "lorena",
"lorenesa", "loriga", "lorota", "lotaríngia",
"loteria", "louça", "louçania", "louçaria",
"loucura", "loucurinha", "loureira", "lousa",
"Lua", "lua", "luanda", "luandesa", "lubina",
"lúbrica", "lucarna", "lucena", "lucila",
"lucrécia", "lúdica", "ludomania", "ludoteca",
"lufada", "luisita", "lula", "lulinha",
"luna", "lunda", "lundesa", "luneta",
"lupa", "lúrida", "lusa", "lusácia",
"lusíada", "lusofonia", "lutulenta", "luva",
"luveira", "luvinha", "luxemburguesa", "luzilandesa",
"maca", "macaçaresa", "maçada",
"maçadinha", "maçaneta", "maçaroca", "macaxeira",
"macega", "machadada", "machona", "machorra",
"macia", "macieira", "macieirinha", "maçonaria",
"maconha", "macroeconomia", "macroestrutura", "mácula",
"madalena", "madama", "madeira", "madeirinha",
"madeixa", "madona", "madrasta", "madrepérola",
"madressilva", "madrilesa", "madrugada", "madrugadinha",
"maduresa", "madureza", "máfia", "mafra",
"magda", "magia", "magistratura", "magnetosfera",
"mágoa",
"magreza", "maioria",
"mala", "malaguesa", "malaquesa", "maleta",
"malga", "malha", "malhinha", "malícia",
"malinesa", "malinha", "maloca", "malta",
"maltesa", "maltesia", "maltinha", "maluqueira",
"maluquesa", "malva", "malvina", "malvinesa",
"malvinha", "mama", "mamadeira", "maminha",
"manada", "manadinha", "manápula", "mancheia",
"mandala", "mandíbula", "mandinga", "mandioca",
"mandioquinha", "mandolinata", "mandrágora", "manduca",
"maneira", "maneirinha", "manesa", "manga",
"mangaba", "mangabinha", "manganesa", "mangueira",
"mangueirinha", "manha", "mania", "manica",
"manilha", "maniota", "maniotinha", "manivela",
"manjedoura", "manjerona", "manopla", "manquesa",
"mansarda", "manteiga", "mantilha", "máquina",
"maquinaria", "maragota", "maragotinha", "maranduba",
"maratona", "maravilha", "marcenaria", "marcha",
"marchinha", "maresia", "margarida", "margaridinha",
"margarina", "margarita", "marianesa", "marina",
"mariola", "marmelada", "marmeladinha", "marmórea",
"maroca", "maronesa", "marosca", "maroteira",
"marquesa", "marquesinha", "marreta", "marretinha",
"marselhesa", "marta", "maruja", "marujada",
"marujinha", "masca", "máscara", "máscula",
"masmorra", "massa", "masseira", "massinha",
"matadeira", "matilha", "matéria", "matinada", "matinadinha",
"matraca", "matriarca", "matrícula", "matrona",
"matula", "maura", "maurícia", "mazorca",
"mazurca", "meada", "meadinha", "mealhada",
"meca", "mecha", "medalha",
"medalhinha", "média", "mediania", "medicina",
"medula", "medusa", "megalomania", "megera",
"melancia", "melancolia", "melanina", "melatonina",
"melena", "melhoria", "melodia", "membrana",
"menesa", "meninada", "menopausa", "mentira",
"mentirinha", "mercadoria", "mercearia", "merda",
"merdinha", "merenda", "merendinha", "mesa",
"mesada", "mescalina", "meseta", "mesinha",
"mesnada", "mesquita", "messalina", "mestra",
"mestria", "metáfora", "metralha", "mica",
"microempresa", "microestrutura", "mídia", "migalha",
"migalhinha", "milanesa", "milícia", "mina",
"minerva", "minha", "minhoca", "miniatura",
"minoria", "minúcia", "minuta", "mioleira", "miopia",
"miosina", "miquelina", "mira", "mirada",
"miradinha", "mirandelesa", "mirandesa", "mirra",
"misantropia", "misericórdia", "missa",
"missinha", "mistura", "misturinha", "mitra",
"miuçalha", "moca",
"moçada", "mocetona", "mochila", "mochilinha",
"moçoila", "modéstia",
"modorra", "moeda", "moedinha",
"moega", "moela", "moelinha", "moenda",
"mofa", "mofina", "moira", "moirama",
"moita", "mola", "moldávia", "moldura",
"moleca", "molecada", "molecadinha", "molécula",
"moleira", "moleirinha", "molequinha", "moléstia",
"moleza", "molezinha", "molhadura", "molinha",
"mona", "monarquia", "monja", "monocultura",
"monotonia", "montanha", "montanhesa",
"montearia", "montesa", "montmartresa", "montra",
"montureira", "morada", "moradia", "moradinha",
"morávia", "morcela", "mordaça",
"mordedura", "moreia", "moreira", "morfina",
"morquesa", "mortalha", "morvandesa",
"morvanesa", "mosca", "mosquinha", "mosquitada",
"mostarda", "mostardinha", "mota",
"motocicleta", "motocicletinha", "movia",
"mucama", "mudancinha",
"mula", "muleta", "mulherada", "mulinha",
"multimídia", "múmia", "munheca", "muralha",
"murça", "murceira", "murta", "musa",
"musculatura", "nabiça", "naca", "nadadeira",
"nádega", "nafta", "naftalina", "naifa",
"naja", "namoradeira", "nana", "nantesa",
"napolesa", "narbonesa", "narina", "nata",
"natinha", "natura", "natureza", "naturopatia",
"náusea", "nauseabunda", "navalha", "navalhada",
"navalhinha", "navarra", "navarresa", "neandertalesa",
"neblina", "nebrina", "necromancia", "necrópsia", "nectarina",
"nédia", "neerlandesa", "nefelina", "nefelometria",
"negociata", "neoconcreta", "neoplasia", "neozelandesa", "nepalesa",
"neptunina", "nereida", "nervura", "néscia",
"nesga", "nespereira", "netiqueta", "neuma",
"neurastenia", "neuroanatomia", "nêustria",
"nevada", "nevadinha", "nevasca",
"névoa", "nevralgia", "nevropatia", "nexina",
"nhandirova", "niacina", "nicesa", "nicobaresa",
"nicotina", "nicotinamida", "nifedipina", "nigromancia",
"nímia", "nina", "ninfa", "ninfomania",
"ninhada", "ninhadinha", "ninharia", "nitreira",
"nitroglicerina", "nívea", "nivernesa", "niza",
"nobreza", "noca", "noctiluca", "nódoa",
"nogueira", "nogueirinha", "noitada", "nomenclatura",
"nômina", "nona", "nora", "noradrenalina",
"norinha", "norlandesa", "norma", "norminha",
"nortada", "norueguesa", "norueguesinha", "nostalgia",
"notícia", "notocorda", "novela", "novelinha",
"novena", "novilhada", "nua", "nuca",
"nuinha", "nunciatura", "oba",
"obra", "obrada", "obrinha", "ocarina",
"odalisca", "odisseia",
"odrada", "oferenda", "oficina",
"ogiva", "oitiva", "ojeriza", "olaia",
"olaria", "oldemburguesa", "oleogravura", "olga",
"oligarquia", "olímpia", "olimpíada", "olindesa",
"oliva", "oliveira", "oliveirinha", "olivina",
"ombreira", "omoplata", "onça", "oncinha",
"onda", "ondina", "ondinha", "ondurmanesa",
"onomatopeia", "ontogenia", "oogamia", "oosfera",
"opala", "opereta", "oranesa", "orca",
"orchata", "orelha", "orelhinha", "oréstia",
"organela", "orla", "orleanesa", "orlinha",
"orquestra", "orquídea", "ortiga",
"ortodoxia", "orvalhada", "orvalhadinha", "orvalheira",
"osga", "ossada", "ossadura", "ossamenta",
"ossatura", "osteossarcoma", "ostra", "otomana",
"ourama", "ourela", "ousadia", "outonada",
"outorga", "ouvidoria", "ova",
"ovelha", "ovelhinha", "ovinha", "oxitocina",
"paca", "pachorra", "paçoca", "paçoquinha",
"padaria", "página", "pala", "palavra",
"palavrinha", "palestra", "palestrinha", "paleta",
"palha", "palheta", "palhinha", "palhoça",
"palma", "palmeira", "palmeirinha", "palminha",
"palmira", "paloma", "pálpebra", "pampa",
"pamplonesa", "pancada", "pancadaria", "pândega",
"panela", "panelinha", "pantera", "pantomima",
"papelada", "papelaria", "paquistanesa", "parábola",
"paralisia", "parcela", "parceria", "pardela",
"pardesa", "parelha", "parenta", "parentela",
"parnaíba", "paródia", "paróquia", "parreira",
"parreirinha", "parta", "partenesa", "partícula",
"partitura", "parvonesa", "páscoa", "pasmaceira",
"passarada", "passarela", "passeata", "pasta",
"pastilha", "pastilhinha", "pastinha", "pastorícia",
"pataca", "pataquinha", "pateada", "patifaria",
"patilha", "patroa", "patroinha", "patuscada",
"paula", "paulada", "paulina", "pavana",
"paveia", "peça", "pecha", "pecinha",
"peçonha", "pediatria", "pedofilia", "pedra",
"pedrada", "pedrenesa", "pedresa",
"pedrinha", "pega", "pegada", "pegadinha",
"peia", "peleja", "peliça", "película",
"pena", "penedia", "peneira", "peneirinha",
"penha", "penicilina", "peninha", "península",
"penteadeira", "penumbra", "peonada", "pepsina",
"pequenada", "pequinesa", "pera", "perda",
"pereira", "pereirinha", "perereca", "perfídia",
"perfumaria", "perguntinha", "perícia",
"periferia", "perinha", "peripécia", "permuta",
"perna", "perninha", "pérola", "perpétua",
"perspectiva", "perspicácia", "pertinácia",
"perua", "peruana", "peruca",
"pesca", "pescada", "pescadinha", "pescaria",
"peseta", "pesquisa", "pessoa", "pessoinha",
"peta", "pétala", "peteca", "petersburguesa",
"petinga", "petinha", "petiza", "petizinha",
"pia", "piada", "piadinha", "picardia",
"piemontesa", "pilha", "pílula", "pimenta",
"pimentinha", "pinacoteca", "pinga", "pintura",
"piovesa", "pipa", "piranha",
"pirataria", "piroga", "piscina", "piscininha",
"pista", "pistola", "pita", "pitada",
"pitadinha", "piteira", "pitinha", "pitombeira",
"pizza", "placa", "placenta", "placentinha",
"planilha", "planta", "plantinha", "planura",
"plaquinha", "plataforma", "plateia", "platina",
"pleura", "pluma", "pneumonia", "pobreza",
"pocema", "pocilga", "poda",
"poeira", "poeirada", "poeirinha", "poesia",
"poetisa", "poia", "poiaresa", "polca",
"poliandria", "polícia", "policromia", "polifonia",
"poligamia", "polissemia", "polonesa", "polpa",
"poltrona", "pólvora", "pomada", "pompa",
"pontada", "pontadinha", "pontaria", "popa",
"popinha", "populaça", "porcaria", "porcelana",
"pororoca", "portaria", "portela",
"portinha", "portinhola", "portugalesa", "portuguesa",
"portuguesinha", "postura", "pota", "pouca", "poupancinha",
"pousada", "pousadinha", "póvoa", "praça",
"pracinha", "praga", "praia", "prancha",
"pranchada", "pranchinha", "prataria", "prateleira",
"prateleirinha", "precisa", "prédica", "prefeitura", "preguiça",
"preguicinha", "preia", "preitesa", "prelada",
"premissa", "prenda", "prendinha",
"presilha", "pressa", "pressinha", "presteza",
"pretensa", "pretória", "primavera", "primazia",
"princesa", "princesinha", "prioresa", "proa",
"procuradoria", "proeza", "profecia", "profilaxia",
"profundeza", "promessa", "promotoria", "propaganda",
"propagandinha", "propina", "prosa", "prosápia",
"prosinha", "prosódia", "próstata", "proteína",
"provedoria", "psicologuesa", "publicitesa",
"pudicícia", "pulga", "pulseira", "pulseirinha",
"pureza", "quadraria", "quadratura", "quadrela",
"quadricomia", "quadriga", "quadrilha", "quadrilhinha",
"quadrissemana", "quantia", "quaresma",
"quebradura", "quebrantada", "quebreira", "queda",
"quedinha", "queimadura", "queiroga", "quelha",
"quenga", "quentura", "queratina", "queratomalácia",
"quercinesa", "querela", "quietura", "quilha",
"quimbanda", "quimera", "quimioprofilaxia", "quimiotaxia",
"quimiotripsina", "quina", "quincha", "quinina",
"quininha", "quinquilharia", "quintana", "quinzena",
"quirguistanesa", "quiromancia", "quitanda", "quitina",
"quixaba", "quota", "quotinha", "rabeca",
"rabeira", "rabona", "raça",
"racinha", "radioastronomia", "rádula", "rainha",
"rainhazinha", "raiva", "raivinha", "rajada",
"rajadinha", "ramaria", "rampa", "rampinha",
"ranchada", "rapa", "rapadura", "rapariga",
"rapaziada", "rapina", "rapinha", "raposa",
"raposinha", "rasoira", "raspadura", "ratazana",
"ratoeira", "ravina", "reabertura", "realeza",
"rebarba", "rebarbativa", "rebeca", "rebeldia",
"reboleira", "rebuçada", "reca", "recandidatura",
"recarga", "recebedoria", "receita", "receitinha",
"recoveira", "rédea", "redoma", "redondeza",
"reengenharia", "reentrada", "reentradinha", "refinaria",
"regalia", "regata", "regatinha", "régia",
"regina", "regra", "regrinha", "régua",
"reitoria", "relíquia", "relojoaria", "remessa",
"rena", "renda", "rendinha",
"reninha", "rentesa", "réplica", "reposta",
"represa", "reprimenda", "república", "resenha",
"resina", "resinga", "resposta",
"respostinha", "ressaca", "ressalva", "réstia",
"restinga", "retaguarda", "retina", "retinopatia",
"retorta", "retranca", "revelia", "revenda",
"reviravolta", "revista", "revistinha", "revoada",
"revoadinha", "reza", "rezina", "rezinha",
"riba", "ribaldeira", "ribalta", "ricina",
"rifa", "rifampicina", "rijeza", "riodonoresa",
"rionoresa", "riqueza", "risada", "risadinha",
"risota", "rixa", "rizicultura", "rocha",
"rochinha", "rociada", "rodela", "rodelinha",
"rodilha", "rodopsina", "rodovia", "rolada",
"roladinha", "rolda", "roldana", "roleta",
"rolha", "romanhesa", "romaria",
"romena", "ronca", "ronceira",
"rondinha", "ronha", "roreja", "rosa",
"rosácea", "rosalina", "rosaria", "rosca",
"rósea", "roseira", "roseta", "rosilha",
"rosina", "rosinha", "rosita", "rosquilha",
"rosquinha", "rota", "roterdamesa", "rotina",
"rotunda", "rotura", "roubalheira", "roupa",
"rouparia", "roupeta", "roupinha", "rua",
"ruana", "ruanda", "ruandesa", "rubéola",
"rúbida", "rubra", "rubrica", "ruça",
"rudeza", "ruela", "ruelinha", "ruga",
"ruína", "ruinha", "ruma", "rumba",
"rupia", "ruptura", "rusga", "rútila",
"sabatina", "sabedoria", "sabinada", "saboneteira",
"sacha", "sacola", "sacolinha", "sacrista",
"sacristia", "sadia", "safadeza", "safena",
"safira", "safra", "saga", "sageza",
"saiaguesa", "saiinha", "sala",
"salada", "saladinha", "salamandra", "salema",
"saleta", "salinha", "saliva", "salmoira",
"salmonela", "salmoura", "saloia", "salsa",
"salsicha", "salsinha", "samarinesa", "samarra",
"sanfona", "sanfoninha", "sanga", "sangalhesa",
"sangesa", "sangria", "sangueira", "sanhuda",
"sanita", "santana", "santiaguesa", "sanzala",
"sapataria", "sapucaia", "sarabanda", "saracura",
"saraiva", "sarça", "sarda", "sardenha",
"sardinha", "sarjeta", "sarna", "sassaresa",
"sátira", "sauna", "saúva", "savana",
"savelha", "seara", "sebastiana", "sebenta",
"secretaria", "secretina", "secura", "seda",
"segadeira",  "segunda", "segundona", "segurelha",
"seira", "seita", "seiva", "selaginela", "selha",
"selva", "selvageria", "selvajaria", "semana",
"sêmea", "semeadura", "sena", "senda",
"sendinesa", "senegalesa", "senha",
"senhorita", "seninha", "senzala", "sépia",
"septicemia", "sepultura", "sequanesa", "sequeira",
"sequela", "serapilheira", "sereia", "serenata",
"seringa", "serotonina", "serpa", "serpentina",
"serra", "serrania", "serrinha",
"serventia", "sérvia", "sesmaria", "sesta",
"seta", "setemesa", "siamesa", "sibila",
"sienesa", "sigla", "signa", "silhueta",
"silicomanganesa", "silva", "silveira", "sílvia",
"silvicultura", "silvinha", "simetria", "simonia",
"simpatia", "simpleza", "sina", "sinagoga",
"sincronia", "síndroma", "sineta", "sinfonia",
"singalesa", "singeleza", "sintonia", "sirena",
"sirigaita", "siroquesa", "sirventesa", "soalheira",
"soberania", "soberba", "sobrancelha",
"sobrancelhinha", "sobranceria", "sobrecarga", "sobrecarta",
"sobrecasaca", "sobremesa", "sobrevida", "sobrevidinha",
"sobrinha", "sociologuesa", "sodomia", "sofia",
"soja", "sola", "soldadura", "soleira",
"solércia", "solha", "solteirona", "soltura",
"sombra", "sombria", "sombrinha", "sonata",
"sonda", "soneca", "sonoplastia", "sopa",
"sopinha", "sorveteria", "sotaina", "sousa",
"souza", "sova", "sovela",
"sovinha", "sua", "suazilandesa", "subclávia",
"subdelegacia", "subida", "subidinha", "subtileza",
"sucata", "súcia", "suçuarana", "sudanesa",
"sudanesinha", "sujeira", "sujeirinha", "sultana",
"súmula", "sundanesa", "supercopa", "superempresa",
"superfina", "superfininha", "superliga", "supertaça",
"súplica", "supremacia", "surdina", "surinamesa",
"surra", "surriada", "surriadinha", "susana",
"suserania", "sutileza", "taba", "tabacaria",
"tabaqueira", "tabatinga", "tabela", "tabelinha",
"taberna", "taberninha", "tabica", "tablatura",
"tablita", "tábua", "tabuada", "tabuleta",
"tabuletinha", "taça", "tacada", "tacadinha",
"tacha", "tachinha", "tacinha", "taenia",
"tailandesa", "taiwanesa", "tajiquistanesa", "talagarça",
"taleira", "talidomida", "talisca", "tamareira",
"tamaresa", "tangapema", "tangerina", "tapeçaria",
"tapera", "taperinha", "tapioca", "taprobana",
"tapuia", "taquara", "taquarinha", "tarambola",
"taramela", "tarantela", "tarasconesa", "tarbesa",
"tardeza", "tarefa", "tarentesa", "tarifa",
"tarja", "tarjinha", "tarola", "tarouquesa",
"tarrafa", "tarragonesa", "tartaruga", "tasca",
"tascinha", "tasquinha", "tauromaquia", "tavanesa",
"taverna", "távola", "taxa", "taxonomia",
"teada", "tebaida", "teca", "tecla",
"tecnocracia", "teia", "teima", "teimosia",
"tela", "telefonia", "telemetria", "telenovela",
"telepatia", "televisa", "telha", "telhinha",
"telinha", "telitoquia", "têmpera",
"temperatura", "têmpora",
"tempura", "tenda", "tendinha", "tenra",
"teocracia", "teoria", "teoriazinha",
"tequila", "terapiazinha", "terebintina", "teresa",
"teresinha", "terna", "ternatesa", "ternura",
"ternurinha", "Terra", "terra", "terracota", "terrantesa",
"terrina", "terrinha", "tesouraria", "tessitura",
"testosterona", "teutoburguesa", "textura", "tiara",
"tiazinha", "tíbia", "tibieza", "tigela",
"tigelinha", "tigresa", "tijuca", "tintura",
"tipoia", "tirada", "tirania", "tireóidea",
"tirolesa", "tiroxina", "tisana", "toada",
"toalha", "toalhinha", "tocaia", "tocha",
"toga", "togolesa", "toguesa", "toleima",
"toleteira", "tona", "tonelada", "tonganesa",
"tonina", "tonquinesa", "tonsila", "tonsura",
"tonteira", "tontura", "tora", "tordilha",
"torga", "torinha", "tornesa", "torpeza",
"torquesa", "torrada", "torradinha", "torrantesa",
"torreira", "tortura", "torva", "tosquia",
"touca", "touça", "toupeira", "toupeirinha",
"tourada", "touradinha", "toxina", "toxinazinha",
"trabalheira", "trácia", "tragédia", "tragicomédia",
"traineira", "tralha", "trama",
"trambolha", "tramela", "traminha", "tramoia",
"trancinha", "tranqueta", "trapaça", "trapalhada",
"traparia", "trapassaria", "trapeira", "traqueia",
"traquitana", "trava", "travadoira", "travessia",
"travessura", "travinha", "trégua", "trela",
"treliça", "tremedeira", "tremelga", "tremesa",
"tremoceira", "tréplica", "tribuna", "trigonometria",
"trincadeira", "trincheira", "tripa", "tripeça",
"tripecinha", "tripinha", "tripla", "tripsina",
"tristeza", "tristezinha", "tristura", "trofa",
"troia", "trolha", "tromba", "trombeta",
"trompa", "tronqueira", "tropa", "tropeada",
"tropilha", "tropomiosina", "troponina", "tropopausa",
"troposfera", "trova", "trovoada", "trovoadinha",
"trufa", "truncadura", "truncatura", "truta",
"tua", "tuatara", "túbara", "tugandesa",
"tulha", "tulipa", "tulonesa", "tumba",
"tuna", "tunda", "tundra", "túnica",
"tuquianesa", "turbamulta", "turbina", "turca",
"turfa", "turinesa", "turma", "turmalina",
"turminha", "turquesa", "turquia", "turra",
"turrinha", "tutela", "tuvalesa",
"uberlandesa", "ucha", "ufania", "ugandesa",
"uiva", "ulna", "umbanda",
"umbela", "umbuzada", "unha", "unhada",
"unhagata", "unhinha", "upa", "urca",
"urdideira", "urdidura", "ureia", "uremia",
"uretra", "urgueira", "urina",
"urtiga", "urubuzada", "usbequistanesa",
"usina", "usura", "utopia",
"utopiazinha", "utópica", "uva", "uvinha",
"uzbequistanesa", "vaca", "vacaria", "vacatura",
"vagareza", "vagina", "vagoneta", "vagueira",
"vaia", "vaiamontesa", "valáquia", "valeira",
"valentia", "valentona", "valeta", "valetinha",
"valsa", "valsinha", "válvula", "vanessa",
"vanguarda", "vaqueirama", "vaquejada", "vaquejadinha",
"vaquinha", "varanda", "varandinha", "vareja",
"vareta", "varga", "varicela", "variegada",
"varíola", "varredura", "várzea", "vasectomia",
"vasilha", "vasilhinha", "vasopressina", "vassoira",
"vassoirinha", "vassoura", "vassourada", "vassourinha",
"vedeta", "veia", "veiazinha", "veiga",
"vela", "velada", "velhacaria", "velhada",
"velharia", "velhota", "velhusca", "velinha",
"veneta", "veniaga", "ventania", "ventura",
"verbena", "verbinha", "verborragia", "verdasca",
"verdelha", "verdura", "verdurinha", "vereda",
"veredinha", "verga", "vergasta", "vergonha",
"vergôntea", "veria",
"vermina", "veronesa", "verrina", "verruga",
"verruma", "versalhada", "versalhesa", "vértebra",
"vesga", "vesícula", "vespa", "véspera",
"vespinha", "véstia", "vestidura", "vestimenta",
"vetusta", "vexada", "vexadinha", "vezeira",
"viana", "vianda", "vianesa", "vianinha",
"viatura", "víbora", "vicunha", "vida",
"videira", "videirinha", "videoendoscopia", "videoteca",
"vidinha", "vidraça", "vidraçaria", "vieira",
"viela", "vienesa", "vietnamesa", "viga",
"viguesa", "vila", "vilafrancada", "vilania",
"vilariça", "vilegiatura", "vileza", "vilinha",
"vilta", "vina", "vincelha", "vindicta",
"vindima", "vindimadeira", "vindita", "vínea",
"vingancinha", "vinhaça", "vinhada",
"vinheta", "vinhetinha", "vinícola", "vintena",
"violácea", "violinha", "virada",
"viradeira", "viradinha", "virginiana", "virgolandesa",
"vírgula", "virilha", "visada", "visadinha",
"víscera", "viscondessa", "viseira", "víspora",
"vistoria", "vita", "vitamina", "vitaminazinha",
"vitela", "vitelinha", "viticultura", "vítima",
"vitinha", "vitivinicultura", "vitrina", "vitrola",
"vivenda", "vivendinha", "viveza", "vívida",
"vizeira", "vodca",
"voltinha", "volumetria", "volúpia", "vossenhoria",
"vozearia", "vozeira", "vozeria", "vulgata",
"vulva", "vurtemberguesa", "xangaiesa", "xaputa",
"xenofobia", "xícara", "xilogravura", "zabumba",
"zabumbada", "zaga", "zaida", "zalcitabina",
"zambujeira", "zana", "zaragata", "zaragatoa",
"zarolha", "zarzuela", "zebra", "zelandesa",
"zeugma", "zibelina", "zidovudina", "zina",
"zoada", "zoadinha", "zoeira", "zoidiogamia",
"zoina", "zombaria", "zona", "zonza",
"zoolatria", "zostera", "zura", "zurrapa",
"zurzia"])


# plvrMfoa - palavras masculinas - femininas apenas com alteracao do o-a menin[o] menin[a]
plvrMfoa = set(["abecedarian", "abelan", "abelian", "abelonian",
"abert", "abevilian", "abian", "abirian",
"abiscan", "ablutôman", "abraamian", "abraquian",
"abretan", "abrótan", "absolut", "absurd",
"abulôman", "abziritan", "acacian", "acadian",
"acaiacan", "acaian", "acanian", "acantodian",
"acantopterigian", "acaridian", "acarnan", "acarnanian",
"acarnean", "acarnian", "accian", "accipitrian",
"acefalian", "acerran", "acheulian", "acian",
"acidalian","ácid", "acipitrian", "acistan", "acleidian",
"aclidian", "acolalan", "açorian", "acorizan",
"acormian", "acrian", "acridian", "acromiocoracoidian",
"actéan", "adamastorian", "adamian", "addisonian",
"adept", "adiáfan", "adisonian", "adlerian",
"admirôman", "adonian", "adotian", "adrastian",
"adrianopolitan", "adrumetan", "adúlter", "aerian",
"aerófan", "aerôman", "afifan", "african",
"agalisian", "agamian", "agarenian", "aginian",
"agonian", "agorâman", "agostinian", "agrarian",
"agrian", "agripinian", "aguaian", "agud",
"aladian", "alagoan", "alaman", "alan",
"alantoidian", "alarician", "alasquian", "alban",
"albarran", "albertogalvan", "albian", "albigensian",
"albionian", "albitan", "albuquerquian", "alcan",
"alcmanian", "alcochetan", "alcoolôman", "aldean",
"alectorian", "alectoromancian", "alencarian", "alentejan",
"aleuromancian", "alexandrin", "alexian", "alfierian",
"algaravian", "algerian", "algôman", "algomanian",
"algonquian", "alhei", "aliban", "alifan",
"alofan", "alogian", "alpinopolitan", "alsacian",
"alt", "althusserian", "altinopolitan", "alun",
"aluvian", "alv", "alvesian", "alvinopolitan",
"alvitan", "amalfitan", "amarel", "amarelinh",
"amarg", "amastrian", "amazonian", "ambian",
"ambilian", "ambrosian", "amebian", "ameliopolitan",
"american", "americanôman", "amibian", "amidalian",
"amig", "amigdalian", "amonian", "amorian",
"amorinopolitan", "amorosian", "amorrean", "ampl",
"amsterdamian", "anacoretôman", "anafrodisian", "analantoidian",
"anamnian", "anan", "anarian", "anastacian",
"anastasian", "anatolian", "anaxagorian", "anchietan",
"ancian", "anciran", "anconitan", "andaman",
"andorian", "andorran", "andorrian", "anedotôman",
"anencefalian", "anfibian", "anficelian", "angarian",
"angatuban", "angelian", "angican", "angimacurian",
"anglican", "anglôman", "angolan", "anguilan",
"angustíman", "anibalian", "animiquian", "anomian",
"anônim", "anopenian", "ansarian", "anselmian", "antarian",
"antelian", "antelicotragian", "antelitragian", "anterian",
"antessignan", "anticôman", "antífon", "antig",
"antigenidian", "antilian", "antilopian", "antinomian",
"antioquian", "antipodian", "antipolitan", "antirrepublican",
"antirrobertian", "antiultramontan", "antocian", "antôman",
"antonian", "antoninian", "antuerpian", "apalachian",
"apelician", "apexian", "apician", "apófan",
"apóstol", "aprustan", "apuan", "apulean",
"apulian", "aquarian", "aquenian", "aquilean",
"aquilian", "aquilitan", "aquitan", "aquitanian",
"aracajuan", "araçatiban", "aracnean", "aracnian",
"aracnoidian", "aradian", "aragan", "araguaian",
"aralian", "aranean", "araucan","árbitr", "arc",
"arcadian", "arcan", "ardian", "arduinian",
"areiopolitan", "arenapolitan", "arenopolitan", "argan",
"argelian", "argentin", "argian", "argiletan",
"arguesian", "arian", "arissantorinian", "aristofanian",
"aristoxenian", "aritenoidian", "aritmôman", "arlesian",
"armêni", "armenian", "arminian", "armorican",
"arnopolitan", "arocelitan", "arquean", "arquiloquian",
"arquitet", "arsenomelan", "artejan", "artemonian",
"artesian", "arubian", "asbolan", "ascanian",
"ascaridian", "ascetan", "ascidian", "asclepitan",
"ascófan", "ascrian", "asculan", "asgardian",
"asian", "asmonian","ásper", "aspidian", "assassin",
"assidian", "assídu", "assirian", "assisian", "assotan",
"astecan", "astenôman", "astian", "astigitan",
"astorgan", "astragalian", "astromancian", "asturian",
"astut", "ataman", "atelan", "atelorraquidian",
"atent", "aterian", "atérman", "atibaian",
"atilian", "atlasian", "atletican", "atloidian",
"atran", "aturian", "auburnian", "audian", "august",
"augustan", "augustinian", "aulican", "aurelian",
"aurignacian", "ausentan", "ausetan", "ausonian",
"australasian", "australian", "austrasian", "austríac",
"autárquic", "autocaricat", "autodian", "autopian", "autoplan",
"autunian", "auxitan", "avan", "avançad",
"avelinopolitan", "averan", "avidian", "axinian",
"azerbaidjan", "azoian", "baamian", "babiruss",
"bacalaurean", "bachelardian", "bachian",
"bacian", "baconian", "bacterian", "bacteridian",
"bactrian", "bactromancian", "badarian", "baenossaurian",
"bagageir", "bahamian", "baian", "baianopolitan",
"bailadeir", "bailarin", "bajan",
"bajocian", "bálan", "balanç", "balancinh",
"balantidian", "baleeir", "baletôman", "balseir", "baltasarian",
"balzacian", "balzaquian", "bananeir",
"banc", "band", "bandoleir", "banezian", "bangladechian",
"bangladeshian", "banh", "banheir", "banian",
"banqueir", "baquean", "baquian", "barat", "barateir",
"barbadian", "bárbar", "barbian", "barbican",
"barbudan", "barbuzan", "barc", "barcan",
"bardesan", "bardesanian", "bardisan", "barician",
"barqueir", "barquinh", "barr", "barrac",
"barran", "barranc", "barraquinh", "barremian",
"barsanian", "bartonian", "basanomelan", "basanomélan",
"bashkirian", "basilian", "basilidian", "basoquian",
"bastard", "batan", "batesian", "baudelairian",
"baudrillardian", "bauhausian", "baxterian", "beat",
"bêbad", "bebian", "bechuan", "becquerian",
"beethovenian", "bel", "belasian", "belitan",
"bellinian", "belomancian", "belostêman", "beltian",
"beltran", "bempostan", "bendit", "benedit",
"beneventan", "bengasian", "benign", "benossaurian",
"bent", "benthamian", "bentopolitan", "benzedeir",
"bercian", "beregran", "bergistan", "bergitan",
"bergmanian", "bergsonian", "berkelian", "berman",
"bernardian", "bertopolitan", "bessdian", "betabacterian",
"betican", "betinian", "bezerr", "biafran",
"biamerican", "bibian", "bibliocleptôman", "bibliôman",
"bibliomancian", "bic", "bicheir", "bicotidian",
"bigodeir", "bigodud", "biguaçuan", "biguan", "bilaquian",
"bilheteir", "bíman", "biplan", "biquotidian",
"birman", "birmanian", "bisaian", "bisbilhoteir",
"biscargitan", "bisgargitan", "bismarckian", "bismarquian",
"bisonh", "bisonheri", "bizarr", "bob",
"bobinh", "bocacian", "bocagian", "boccaccian",
"bodian", "bodonian", "boêmi", "bogomilian",
"bogotan", "boituvan", "bol", "bolan",
"bolborraquidian", "bolet", "bolinh", "bolivarian",
"bolivian", "bols", "bonec", "bonfinopolitan",
"bonifacian", "bonit", "bononian", "bonosian",
"boolean", "boolian", "bopian", "boppian",
"boran", "borbonian", "borborian", "borcan",
"bornean", "boschiman", "bosforan", "bósni",
"bosnian", "bosporan", "bosquiman", "bostonian",
"botafogan", "boticelian", "botnian", "botsuan",
"boximan", "boxíman", "bozonian", "bracaraugustan",
"bragan", "bragançan", "branc", "brancaran",
"brand", "brandesian", "branquinh", "braquistéfan",
"braseir", "brasileir", "brasilian", "brasitan",
"brasopolitan", "braunian", "brav", "brechtian",
"brentanian", "bricôman", "brigadian", "brigian",
"britan", "brixian", "brodosquian", "bromoetan",
"bromopentan", "brownian", "brucian", "brunean",
"brut", "brutian", "brux", "bruxôman",
"buan", "bubetan", "buenopolitan", "bugeran",
"bugi", "bulboclitoridian", "bulbopenian", "burdigalian",
"burguesinh", "burr", "burrinh", "burundian",
"busan", "butan", "byronian", "cabaçuan",
"caban", "cabocl", "cabriuvan", "cabuçuan",
"cachoeiran", "cachorr", "cacodemonôman",
"cacofatôman", "cacofonôman",
"cadmian", "caetan", "caetanopolitan", "cafezopolitan",
"caiaponian", "caicavian", "caietan", "cainian",
"cajan", "calaarian", "calabrian", "calan",
"calçad", "calçadinh", "calcanean", "caledonian",
"californian", "calm", "calomelan", "calomélan",
"calovian", "calv", "camapuan", "camarajiban",
"camaran", "cambaratiban", "camberran", "cambojan",
"cambojian", "cambrian", "cameleonian", "camelian",
"cameronian", "camilian", "camonian", "campanian",
"campian", "campichan", "can", "canadian",
"canapolitan", "cancerian", "cândid", "candidat",
"canec", "canfan", "canisian", "canossian",
"canovian", "cantaridian", "cantorian", "canzianopolitan",
"confeiteir", "contígu", "caparican", "caparitan", "capetian", "capinopolitan",
"capivarian", "capixaban", "capoeiran",
"caprelian", "caprian", "capricornian", "capsitan",
"capuan", "caralitan", "caramban", "caramulan",
"caran", "carapicuiban", "caravaggian", "carboran",
"carcaman", "cardan", "carelian", "carian",
"caribian", "cariguan", "carlopolitan", "carlostadian",
"carlovingian", "carlylian", "carman", "carmanian",
"carmelitan", "carmelopolitan", "carmopolitan", "carnacian",
"carnaiban", "carolingian", "carolstadian", "carontian",
"carotidian", "carp", "carpetan", "carpian",
"carpocracian", "carpometacarpian", "carpopterigian", "carquesian",
"carranchan", "carrapitan", "carrilan", "carroceir",
"carseolan", "carsitan", "carteir", "cartesian",
"cartimitan", "cartorian", "cartusian", "carvalhopolitan",
"cas", "casac", "casc", "caseir",
"casimirian", "caspian", "casquivan", "cassinian",
"castelhan", "castelperronian", "casterlian", "castilian",
"castrinopolitan", "cataban", "cataclismian", "catalan",
"catapan", "catarian", "catarquean", "catarrinian",
"cátedr", "catian", "catolician", "catonian",
"catruman", "caucasian", "caudíman", "cauranan",
"caval", "cavalarian", "cavalariç", "cavalerian", "cavian",
"caxemirian", "cebian", "cebuan", "cefaleditan",
"cefalodian", "cefalôman", "cefalorraquian", "cefalorraquidian",
"ceg", "ceifeir", "celestian", "celestin", "celossomian", "celsian",
"celtôman", "cenoman", "cenomanian", "centíman",
"centôman", "centrafrican", "cerafan", "cerban",
"cercan", "cercinitan", "cerdonian", "cerebrorraquidian",
"ceresopolitan", "cerretan", "cert", "cervicoacronian",
"cesarian", "cestóplan", "cetan", "cevian",
"cézannian", "chabacan", "chabian", "chadian",
"chamusan", "chandlerian", "chaplinian", "charabilhan",
"charadôman", "charian", "charmutian", "charran",
"chassidian", "chat", "chatelperronian", "chatian", "chavelhan",
"chavian", "chei", "chelean", "chic", "chilen",
"chiman", "chinel", "chinelinh", "chiquitan", "chiraquian",
"chiriguan", "chopinian", "churchillian", "cian",
"cían", "cianocetan", "cianomelan", "cianomélan",
"cibilitan", "cicendelian", "ciceronian", "cicindelian",
"ciclan", "cicloalcan", "ciclobutan", "ciclocefalian",
"cicloexan", "ciclopentan", "ciclopian", "ciconian",
"cigan", "cigarr", "cilician", "cimófan",
"cimolian", "cinéfil", "cinófil",
"cingaporean", "cingapurian", "cint", "cinzent",
"circadian", "circameridian", "circassian", "circatejan",
"circunfaringian", "cisfretan", "cisjuran", "cisleitan",
"cispadan", "cistarian", "cistebutan", "cistidian",
"cistigritan", "cisuranian", "citopolitan", "clactonian",
"clar", "clarian", "clastôman", "claudelian",
"claudiopolitan", "clazomenian", "cleófan", "cleptôman",
"cletôman", "clidoepitroclian", "clinôman", "clismian",
"clitôman", "clitorian", "clitoridian", "clitrôman",
"clopêman", "clopôman", "cloralôman", "cloraluretan",
"cloretan", "clorófan", "clorofilian", "clorometan",
"clunisian", "cóan", "cobert", "coblencian",
"cocainôman", "coccigian", "cocigean", "códan",
"coestaduan", "colapian", "colarian", "colcozian",
"colectan", "colerôman", "colestan", "coletan",
"colheitan", "colhetan", "colombian", "colonian",
"coman", "comanchian", "comblanchian", "combonian",
"cômod", "comorian", "companheir", "comparoquian",
"complex", "compósit", "compostelan", "compost", "comprovincian",
"comtean", "comtian", "côncan", "concis", "concordian",
"condilian", "condilotroclian", "condistritan", "condrocoracoidian",
"condroepitroclian", "condrostibian", "confucian", "coniacian",
"conidian", "conquilian", "consistorian", "constantinian",
"constantinopolitan", "contan", "contínu", "conurban",
"convertiplan", "cop", "copan", "copeir",
"copernican", "copernician", "coplan", "coprôman",
"coracoidian", "coralian", "coran", "cordeiropolitan",
"cordilheiran", "cordíman", "cordofanian", "corean",
"coreôman", "corguinhan", "corimbacterian", "corinebacterian",
"corintian", "coriorretinian", "coritan", "cormarian",
"córne", "cornean", "cornelian", "cornian",
"coroidean", "coronarian", "coronoidean", "correggian",
"correntezan", "corrigan", "cortiç", "cortonian",
"corumbaiban", "cosan", "cosmopolitan", "cossetan",
"costepitroclian", "costocoracoidian", "costopubian", "costotraquelian",
"costureir", "cotejipan", "cotian", "cotidian",
"cotiloidian", "coulombian", "couperinian", "courbetian",
"coveitian", "cozinheir", "crabilhan", "cracoian",
"cracovian", "cramulhan", "cranachian", "cranian",
"cravilhan", "crédul", "cremasterian", "cremonian", "cricaritenoidian",
"cricoaritenoidian", "crioul", "criptomelan", "crisamebian",
"crisobálan", "crisopolitan", "cristalinian", "cristalofilian",
"cristianapolitan", "cristianopolitan", "cristinapolitan", "cristopolitan",
"crivellian", "croacian", "crocian", "crocodilian",
"cromagnonian", "cromerian", "cromobacterian", "crotopodôman",
"crucífer", "cruôman", "ctonian", "cuaitian", "cub",
"cuban", "cubitocarpian", "cubitofalangian", "cuboidocunian",
"cudan", "cuiaban", "cuiauan", "cuisian",
"culicidian", "culteran", "cultôman", "cuman",
"cumberlandian", "cunean", "cuproasbolan", "curilian",
"curitiban", "curituban", "curvelan", "curvian",
"cutchian", "cutian", "cuuaitian", "cuvaitian",
"d'alembertian", "dacnôman", "dacnomonôman", "dactilógraf",
"dalinian", "daltonian", "dan", "dançarin",
"dantian", "danubian", "daomean", "darwinian", "datilógraf",
"decan", "décim", "decrépit", "decuman",
"decúman", "defunt", "deleuzian", "dens",
"dentuç", "derradeir", "descobert", "desembaraçad",
"desert", "desgraçad", "desgraçadinh", "desídi",
"deslandesian", "deslandian", "desmobacterian", "destr",
"destríman", "desuman", "detent", "deucalian", "deuterostomian",
"devonian", "devot", "deximontan", "dextran",
"dextríman", "dextrôman", "dextrorfan", "dezembran",
"diab", "diaconian", "diacranian", "diacrian",
"diáfan", "dianopolitan", "dianteir", "diatérman",
"diazometan", "dibenziletan", "dibenzilometan", "diboran",
"dibrometan", "dibromopropan", "dicerian", "dicloretan",
"diclorometan", "dicondilian", "dicran", "dicresiletan",
"dicresilofenilometan", "dicresilometan", "diderotian", "diditan",
"didonian", "difenan", "difeniletan", "difenilmetan",
"difenilobutan", "difenilodicloretan", "difenilodiclorometan", "difenilodimetiletan",
"difenilometan", "difenilopropan", "difenoletan", "diiodetan",
"diiodometan", "diluvian", "diminut", "dinaftilometan",
"dinancian", "dinarian", "dind", "dinitropropan",
"diocesan", "dionisian", "dioscorian", "diospolitan",
"dioxan", "dipirrilmetan", "diplantidian", "diplectan",
"diplobacterian", "dipsôman", "direit", "dirian",
"discípul", "discret", "discurseir", "dispost", "dissan",
"dissilan", "dissilican", "dissilicoetan", "dissolut", "distint",
"distomian", "dit", "ditérman", "divers", "divin",
"divinopolitan", "divisopolitan", "dízim", "djibutian",
"dobradiç", "doceir", "dodecan", "dodecapolitan",
"domazan", "domerian", "dominican", "don",
"donatian", "doninh", "donzel", "dordonian",
"dorisopolitan", "dormideir", "dorsiepitroclian", "dorsimedian",
"dorsoepitroclian", "dorsomedian", "doxôman", "draconian",
"dragoman", "dramaturg", "drogoman", "drummondian",
"dúbi", "dulcinopolitan", "dulian", "duodecan",
"duodécim", "dupl", "durkheimian",
"eberthian", "ebúrne", "ebusitan", "eçanian",
"ecavian", "ecetran", "ecian", "eclesian",
"eclus", "ectocranian", "ectorraquidian", "eculan",
"edetan", "edipian", "efedrin", "efêmer",
"egban", "egelestan", "egestan", "egipan",
"egípci", "egipcian", "egitan", "egnacian",
"eicosan", "eifelian", "einstanian", "einsteinian",
"einstenian", "eisensteinian", "elafian", "elan",
"elean", "eleit", "eletrotrépan", "eleuterôman",
"eleuteropolitan", "eliotian", "elisabetan", "elisabetian",
"elisian", "elizabethan", "elpinian", "elusan",
"elzevirian", "emeritan", "emilian", "emilianopolitan",
"empedoclian", "emporitan", "empreiteir", "enan",
"encefalorraquian", "encefalorraquidian", "encúman", "endaspidian",
"enderodan", "endocranian", "endorraquian", "endorraquidian",
"eneaboran", "enferm", "enfermeir", "engraçad",
"enoseôman", "enteôman", "enterobacterian", "enterocelomian",
"entremontan", "entrepan", "entrerrian", "enxabregan",
"eocambrian", "eodevonian", "eolian", "eperlan",
"epicondilian", "epicranian", "epictonian", "epidauritan",
"epífan", "episcopalian", "epitacian", "epitauritan",
"epitetôman", "epitroclean", "epitrocleoolecranian", "epoxietan",
"equatorian", "equiculan", "equinean", "equinian",
"equinocian", "eranian", "erasmian", "erastian",
"ergasiôman", "ergôman", "erídan", "erotôman",
"ervan", "esbelt", "escafoiducunean", "escafoldoastragalian",
"escalabitan", "escapulotroquiterian", "escardonitan", "escass",
"esclerocornean", "escorpian", "escrav", "escribôman",
"escrit", "escur", "esfenovomerian",
"esferorradian", "esfincterian", "esgui", "eslováqui",
"esmitian", "esofagian", "esopian", "esparnacian",
"espars", "espartan", "espencerian", "espenglerian",
"esperantinopolitan", "espert", "espess", "espinh",
"espintrian", "espiran", "espiritan", "espiropentan",
"esplêndid", "espoletan", "espontâne", "esquerd",
"esquilian", "esquisit", "esquizocelomian", "esquizotripan",
"estabian", "estalian", "estalinian", "estampian",
"estancarian", "estancian", "estapedian", "estatan",
"estatuderian", "estéfan", "estefanian", "estégan",
"estendalian", "estenostéfan", "estereorradian", "esternomasseterian",
"esternotroquinian", "esterradian", "estesicorian", "estesiôman",
"estilocoplan", "estilpnocloran", "estilpnomelan", "estonian",
"estrabonian", "estramontan", "estrangeir", "estranh",
"estreit", "estricnôman", "estrit", "estrôman",
"estrombolian", "estrumeir", "estupend", "esubian",
"etan", "etére", "etern", "eterôman",
"etiluretan", "etiopian", "etolian", "euclidean",
"euclidian", "eudoxian", "eugenopolitan", "eulerian",
"eunapolitan", "eunomian", "eurafrican", "euralian",
"euramerican", "eurasian", "euriapolitan", "euripidian",
"euritan", "euscarian", "eustaquian", "eustéfan",
"eutiquian", "eutireoidian", "evangelistan", "exaspidian",
"exat", "excels", "exígu", "exocranian",
"extens", "extern", "extint", "extrem",
"exurban", "fabian", "fabrician", "faceir",
"fagifulan", "faisqueir", "falangean", "falangian", "falanginian",
"falansterian", "faloidian", "falopian", "fals",
"famenian", "familisterian", "faquirian", "faringian",
"farinheir", "farmacôman", "fart", "fasian",
"fatícan", "faunian", "faustian", "favorit",
"faxineir", "fazendeir", "feacian", "febronian",
"fedrian", "fei", "feiticeir",
"felin", "felizard", "feminin", "fenectan",
"fenian", "fenilcroman", "fenilmetan", "feniluretan",
"feretrian", "fernandopolitan", "férre", "ferrialofan",
"ferroestilpnomelan", "ferromagnesian", "fichtian", "ficocian",
"fidalg", "fidian", "fidiculan", "fidjian",
"fieldingian", "fig", "fijian", "filadelfian",
"filantropôman", "filh", "filintian", "filipin", "filipopolitan",
"filolusitan", "filôman", "filonian", "filósof",
"fin", "financeir", "fingid", "fioropolitan",
"firman", "firminopolitan", "físic", "flacian", "flacourtian",
"flacurtian", "flagelôman", "flameng", "flandrian",
"flaubertian", "flaviopolitan", "fleminguian", "flenusian",
"florestopolitan", "florian", "florianopolitan", "fluoran",
"fluormetan", "fluvian", "foc", "fof",
"fogueteir", "fonôman", "fontan", "forasteir",
"forçad", "forentan", "foretan", "formian",
"formosan", "forr", "forrageir", "forulan", "fosc",
"fosforan", "foss", "fotinian", "fotógraf",
"fotoplan", "frac", "franc", "francan",
"francian", "francinopolitan", "franciscan", "franciscopolitan",
"francôman", "franconian", "frang", "frangipan",
"franklinian", "fraseôman", "frasnian", "franzin", "frebelian",
"fregelan", "freirean", "freirian", "frentan",
"freudian", "fri", "fris", "frisian",
"friulan", "frívol", "froebelian",
"froux", "fuchsian", "fucsian",
"fulan", "fúlgid", "fulian",
"fundan", "funest", "furan", "furfuran",
"furian", "fusan", "fuscíman", "futur",
"gabalitan", "gabian", "gaditan", "gaian",
"gaj", "gálban", "galecian", "galeg",
"galen", "galerian", "galh", "galhan", "galheir",
"galhoban", "galhofeir", "galican", "galician",
"galilean", "galinheir", "galizian", "galoisian",
"galôman", "galopolitan", "galvan", "gambian",
"gameleir", "gandhian", "gans", "garaban",
"garapeir", "garavan", "gardonian", "gargantuan",
"gargasian", "garot", "garran", "garrettian",
"garumnian", "gastresofagian", "gastrodiáfan", "gat",
"gaúch", "gaulitan", "gaurian", "gaussian",
"gavelan", "gavieir", "gazan", "gedinian",
"gedrosian", "gefirian", "geiserian", "geladinh",
"geleir", "gême", "geminian", "genetlian",
"genian", "genovevan", "genuín", "geoplan",
"georgian", "georgin", "geran", "gercinopolitan",
"geresan", "geresian", "german", "gessnerian",
"getulian", "gidian", "ginasian", "ginecôman",
"giroplan", "giscardian", "gitan", "givetian",
"givfitan", "gladstonian", "glaucófan", "gliceran",
"gliciltriptofan", "glicosaminoglican", "glossian", "glozelian",
"goan", "gobinian", "goelan", "goethian",
"goiabeir", "goian", "goianapolitan", "goianian",
"goisian", "goldonian", "gomarian", "gomesian",
"gongan", "gonzaguian", "gord", "gordian",
"gorduch", "gorgian", "gorr", "gorutuban",
"gotlandian", "gouvean", "graban", "gracianopolitan",
"grafôman", "gramadeir", "grat", "gravan",
"gravetian", "graviscan", "grecian", "grecôman",
"greg", "gregorian", "gril", "gritadeir",
"grocian", "groselheir", "gross", "grosseir",
"guaçuan", "guaiuban", "gualterian", "guan",
"guanaban", "guarapuavan", "guaraqueçaban", "guaratuban",
"guaxupean", "guerreir", "guerrilheir",
"guinean", "guipuscoan", "guiratingan", "guis",
"guitan", "gunzian", "gurutuban", "gusan",
"gutembergian", "gutemberguian", "gutenbergian", "hadranitan",
"hahnemannian", "haitian", "halafian", "hallstattian",
"halomancian", "halstatian", "hamidian", "hamiltonian",
"hamletian", "handelian", "hanoverian", "hanovrian",
"hansean", "hansenian", "harmófan", "hartmanian",
"hassidian", "hastensian", "hatorian", "hauterivian",
"havaian", "havan", "haxixôman", "hebraic",
"hediond", "hegelian", "heideggerian", "héli",
"heliconian", "heliófan", "helióman", "heliopolitan",
"helopian", "helvecian", "helvidian", "hemasian",
"hematotímpan", "hemimontan", "hendecan", "henrican",
"hentriacontan", "hequelian", "heraclitian", "herban",
"herbartian", "herciliopolitan", "hercinian", "herculan",
"herculanian", "herdeir", "herderian", "heredian",
"heriban", "hermesian", "hermian", "hermitian",
"hermogenian", "herodian", "herodotian", "heroic",
"heroinôman", "hertzian", "hesbaian", "hesperófan",
"hessian", "heteralian", "heteroplan", "heterosoounian",
"heterostéfan", "heterousian", "hexaclorocicloexan", "hexacloroetan",
"híbrid", "hidraeroplan", "hidrelitan", "hidroaeroplan",
"hidrobiplan", "hidroboran", "hidrobrasileir", "hidrodipsôman",
"hidrófan", "hidrôman", "hidromonoplan", "hidrotímpan",
"hierapolitan", "hierosolimitan", "higrófan", "hilbertian",
"hilean", "hileian", "hindustan", "hipolitan",
"hipôman", "hipovarian", "hipsistarian", "hircan",
"hirtíman", "hispan", "histerôman", "hitlerian",
"hjelmslevian", "hobbesian", "hoffmannian", "holaspidian",
"holbachian", "holbaquian", "hollywoodian", "holocranian",
"homeousian", "hominian", "homoplan", "homousian",
"honest", "horacian", "hospedeir", "hospitaleir",
"hudsonian", "hugoan", "hullian",
"human", "humaniformian", "humanitarian", "humboldtian",
"humian", "húngar", "hunterian", "huronian",
"husserlian", "hutchinsonian", "huterian", "huttonian",
"iacrian", "iamnotan", "iauácan", "ibaran",
"ibaretan", "ibegean", "ibgean", "ibitiguaçuan",
"ibsenian", "iconôman", "icosan", "icositan",
"idiociclófan", "ienissean", "ienisseian", "igacian",
"igaraçuan", "igeditan", "iguaçuan", "iguan",
"iguanian", "iliberitan", "iliopretrocanterian", "iliopretrocantinian",
"iliorrotulian", "iliotrocanterian", "iliotrocantinian", "iliturgitan",
"illinoisian", "ilocan", "ilopolitan", "ilorcitan",
"imediat", "imens", "imorredour", "imperfeit",
"impolut", "imprevist", "imund", "inat",
"inaudit", "incaut", "incert", "inclus",
"incógnit", "incômod", "incrédul",
"inculcadeir", "incult", "indaiatuban", "indecis", "indian",
"indigest", "indign", "indiret", "indiscret",
"indisfarçad", "indômit", "indonési", "indostan",
"industan", "inédit", "inequívoc", "inesian",
"infecund", "infinit", "infravermelh", "infus",
"ingênu", "ingrat", "ingrian", "inimig",
"iniodimian", "injust", "inócu", "inquiet",
"inquilin", "insan", "insensat", "insólit",
"instantâne",
"insulindian", "inteir", "inteiriç", "intens",
"intercuspidian", "interioran", "intern", "internetian",
"interplan", "intrarraquidian", "intrus", "inuman",
"inúmer", "invers", "invict", "invitatorian",
"iodometan", "iolopolitan", "ioruban", "ipojucan",
"ipresian", "ipuan", "iracemapolitan", "iranian",
"iraquian", "irian", "iridian", "iridocornean",
"irineopolitan", "irinotecan", "irituian", "irrequiet",
"irriquiet", "irvingian", "isidorian", "ismaelian",
"ismailian", "isobritan", "isobutan", "isoctan",
"isófan", "isofluoran", "isopentan", "isquiointertrocanterian",
"isquiopenian", "isquiopubian", "isquiopubitrocanterian", "isquiotrocanterian",
"israelian", "istrian", "itabiran", "itainopolitan",
"itaiopolitan", "italian", "italianôman", "italician",
"italôman", "itamarandiban", "itamontan", "itan",
"itaocan", "itaparican", "itapecerican", "itapetingan",
"itapolitan", "itaquaquecetuban", "itatiban", "itibiran",
"itiquiran", "ituan", "ituiutaban", "iucatan",
"jabutian", "jacksonian", "jacobian", "jacsonian",
"jaguariban", "jaguaruan", "jalesian", "jamaican",
"jambolan", "jan", "jandaian", "jangan",
"jardineir", "jardinopolitan", "jarr", "jassidian",
"jauácan", "jeceaban", "jeffersonian", "jenerian",
"jennerian", "jerezan", "jerezian", "jernean",
"jeronimian", "jerosolimitan", "jeruzan", "jesenopolitan",
"jessean", "jibutian", "joalheir", "joanesian",
"joanopolitan", "jobinian", "jordan", "jordanian",
"jovian", "jovinian", "judaic", "judean",
"julian", "juliopolitan", "jument", "jundiaian",
"junguian", "junqueir", "junqueiropolitan", "junquerian",
"junt", "jupiterian", "juquerinan", "juquerionan",
"juquirionan", "jurassian", "juruban", "jusan",
"just", "justiceir", "justinian", "justinopolitan",
"juvenalian", "kandorian", "kelvinian", "kemalian",
"keynesian", "kiribatian", "kryptonian", "kuwaitian",
"lábdan", "lábi", "labican", "labortan",
"laçad", "lacanian", "lacerdopolitan", "lacetan",
"lactametan", "lactovegetarian", "ladan", "ládan",
"ladroeir", "lágan", "lagart", "lagean",
"lagiran", "lagoan", "lagrangian", "laian",
"lajean", "lajian", "laletan", "lalôman",
"lamarckian", "lamartinian", "lambran", "lamean",
"lamian", "laminitan", "lancasterian", "lancastrian",
"landenian", "landrinopolitan", "languedocian", "lânguid",
"lantan", "lântan", "laocian", "laodicean",
"laosian", "laostan", "laotian", "lapachan",
"lapian", "laplacian", "laramian", "larendan",
"laringian", "lasciv", "lassalian", "lassallian",
"laterculian", "latíman", "latin", "laud",
"láudan", "lauran", "laurencian", "laurentian",
"lauretan", "lausanian", "lausannian", "lausian",
"lavadeir", "lavandeir", "lavaterian", "lecontian",
"legendrian", "legerian", "legítim", "leibnitzian",
"leibnizian", "leiteir", "leman", "lemurian",
"lenan", "leninian", "lent", "leodoman",
"leoflorestan", "leonian", "leopardian", "leopolitan",
"lepidomelan", "leptitan", "leptocardian", "lesbian",
"lesotian", "letôman", "leucófan", "leucopetrian",
"levalloistan", "levaloisian", "levian", "levinopolitan",
"lhan", "liberian", "libertin", "líbi",
"libisosan", "librian", "liburnian", "licean",
"licopolitan", "liechtensteinian", "ligeir", "ligerian",
"ligitan", "ligorian", "liguorian", "ligurian",
"liliputean", "liliputian", "lilliputian", "liman",
"limian", "límpid", "lind", "linean",
"linguedocian", "linh", "linnean", "liócran",
"liparitan", "líquid", "lis", "lisboan",
"lisonjeir", "listenstainian", "listentainian", "lisztian",
"literôman", "litrean", "littrean", "lituan",
"lituâni", "liuran", "livonian", "livramentan",
"lixeir", "lob", "lobatian", "lobeir",
"locarnian", "locustian", "loflucarban", "logaritmôman",
"logôman", "loir", "loiraç", "lomb",
"lombocostotraquelian", "lombodorsotraquelian", "lombotrocantinian", "lombrosian",
"londinian", "longev", "longíman",
"longinian", "longínqu", "longulan", "lopan",
"loretan", "lorman", "lorquian", "lotaringian",
"lotoman", "louc", "louletan", "lour",
"lourencian", "louretan", "lovelacian", "luan",
"lucaian", "lucan", "lucanian", "lucasian",
"lucianopolitan", "lucífer", "luciferian", "lucilian",
"lucopetrian", "lucrecian", "luculian", "ludian",
"ludovician", "luhmannian", "luisian", "lulian",
"lumberitan", "lupinan", "lupinopolitan", "lusacian",
"lusitan", "lusitâni", "lusitanian", "lusófon",
"lutecian", "luteran", "lutevan", "lutjan",
"maç", "macac", "macacuan", "macatuban",
"macedonian", "machadian", "macian", "maciç",
"madeireir", "madimamuan", "maestrichtian", "mafometan",
"magan", "magdalean", "magdalenian", "magdalian",
"maglemosian", "magn", "magnesian", "magoadinh",
"magosian", "magr", "magrebian", "magrinh",
"maian", "maimonidian", "maioran", "malacitan",
"malacopterigian", "malasian", "malauian", "malavian",
"malawian", "maldan", "maldichan", "maldit",
"maldivan", "maldivian", "malian", "malign",
"mallarmean", "malpighian", "malthusian", "maluc",
"malucan", "malvernian", "mambucan", "mameluc",
"man", "manchesterian", "manchurian", "mancian",
"mancunian", "mandchurian", "mandean", "mandian",
"mandibulomentonian", "mandruan", "manducan", "mangan",
"manganomelan", "mangaratiban", "manitan", "mans",
"mansfieldian", "mant", "mantenopolitan", "mantovan",
"mantuan", "maometan", "maracajuan", "maracian",
"maragogipan", "maragojipan", "maran", "marangatuan",
"marçan", "marcelian", "marceneir", "marchan",
"marcian", "marcianopolitan", "marcoman", "marcosian",
"marcovian", "marculian", "mardan", "marfinian",
"mariamitan", "marian", "marianopolitan", "mariapolitan",
"marijuan", "marinettian", "marinopolitan", "mariopolitan",
"marisopolitan", "marlian", "marlowian", "marmontelian",
"marmulan", "marran", "marrec", "marsilian",
"martaban", "martinican", "martinopolitan", "marxian",
"máscul", "masculin", "massangan", "massilian",
"massilitan", "matarruan", "mateolan", "matutin",
"maupassantian", "maurian", "maurician", "mauritan",
"mauritâni", "mauritanian", "maurrasian", "máxim",
"maximilian", "maxwellian", "mazovian", "mazurian",
"mean", "meclenburgian", "médan", "median",
"mediocarpian", "mediotarsian", "mediterran", "mediterranian",
"medonh", "megalôman", "megalopolitan", "megapolitan",
"megarian", "meig", "melan",
"melanchthonian", "melanostibian", "melcoman", "melecian",
"melfalan", "melgacian", "melian", "melindan",
"melitan", "melitonian", "melôman", "melquisedecian",
"menalian", "menandrian", "mendelian", "mendelssohnian",
"mendig", "menian", "menin", "mensageir",
"mentan", "mentesan", "mentonian", "menuccian",
"meonian", "mer", "mercaptan", "mercesan",
"mercurian", "merfian", "merian", "meridian",
"meringian", "meroitan", "merovingian", "meruocan",
"mesm", "mesmerian", "mesmerôman", "mesoamerican",
"mesocarpian", "mesocordilheiran", "mesodevonian", "mesometatarsian",
"mesopotâmi", "mesotarsian", "mesovarian", "messalian",
"messianopolitan", "mesvinian", "metacarpian", "metacarpofalangian",
"metafisicôman", "metamelfalan", "metan", "metastasian",
"metatarsian", "metatarsofalangian", "metiluretan", "metonian",
"metoxietan", "metoxifluoran", "metrôman", "metropolitan",
"mexican", "micenian", "michan", "micoquian",
"micran", "microbian", "mididitan", "migecian",
"miguelopolitan", "milenarian", "milesian", "miletan",
"milevetan", "milevitan", "milhan", "milian",
"milician", "miliradian", "millian", "miltonian",
"mimosean", "mindelian", "minean", "mineir",
"minian", "mínim", "ministr", "minoan",
"minuan", "minúscul", "mirabálan",
"mirabólan", "mirandolan", "mirandopolitan", "mirianopolitan",
"mirmeleonian", "mirobálan", "míser", "missianopolitan",
"mississipian", "mississippian", "mistralian", "misulan",
"mitanian", "mitchurinian", "mitôman", "miúd",
"moametan", "moç", "moçambican", "mocan",
"mocinh", "mococan", "mod", "modern",
"modest", "modinh", "moican", "mojian",
"molibduran", "molucan", "momentâne", "monasterian",
"mongolian", "monjolan", "monocarpian", "monocefalian",
"monôman", "mononfalian", "monoplan", "monossilan",
"monóton", "monovarian", "monozonian", "monroean",
"montan", "montemuran", "montessorian", "monteverdian",
"montevidean", "montian", "montijan", "mórbid",
"mordovian", "moren", "morfinôman", "morgean",
"morgian", "moribund", "morn", "morretian",
"mort", "mosan", "mosassaurian", "motan",
"mour", "mousterian", "mozartian", "mucan",
"muçulman", "mud", "mudinh", "muédan",
"muensterian", "muentzerian", "muggletonian", "mugletonian",
"mulat", "multan", "multiplan", "muncausenian", "munchhausenian",
"mundan", "muratorian", "murcian", "murexan", "muribecan",
"murilhan", "muritiban", "museôman", "musican",
"musicôman", "mussetian", "mussolinian", "musterian",
"mutan", "mútu", "mutunopolitan", "mutustratan",
"naalian", "nabantian", "nabocovian", "nabuquian",
"naftalan", "naftôman", "namibian", "namurian",
"nantuan", "napolitan", "narandiban", "narcoguerrilh",
"narcôman", "nasocian", "nasofaringian", "nasogenian",
"nasserian", "nassovian", "natalin", "natercian",
"natinian", "natinopolitan", "nativan", "nativitan",
"náufrag", "nauran", "nauruan", "naxian",
"nazarean", "neandertalian", "neapolitan", "necisian",
"necissian", "neditan", "nefand", "nefast",
"nefridian", "nefrocelian", "negr", "negreir",
"nemean", "nemertian", "nemetan", "nemian",
"neocomanian", "neocomian", "neonomian", "neopolitan",
"neorrepublican", "neorroman", "neperian", "nepotian",
"neptunian", "neronian", "neropolitan", "nerulan",
"nestorian", "net", "netunian", "neur",
"neustrian", "neutr", "newtonian", "nhanguaçuan",
"nhunguaçuan", "nicaraguan", "nicerocian", "nicerotian",
"nicomelan", "nicopolitan", "nietzschian", "nigerian",
"nilopolitan", "ninfôman", "nitrofuran", "nitrometan",
"noecian", "noemian", "noetan", "noetian",
"noiv", "nojent", "nolan", "nomentan",
"nonan", "norban", "norcaran", "nordestan",
"norian", "norman", "normand", "norpinan", "nosôman",
"noss", "nostôman", "noterian", "notídan",
"noturn", "nov", "novacian", "novat",
"novgorodian", "noviç", "novilh", "núbi",
"nubian", "nuculan", "numestran", "numidian",
"numistran", "nunalvarian", "oasian", "obes",
"obreir", "obscen", "obscur", "obsidian",
"obsolet", "obstetrican", "obstetríci", "oc",
"ocan", "ocauçuan", "occipitatloidian", "occipitomentonian",
"oceanógraf", "ocipitatloidian", "ocipitomentonian", "ocitan",
"ocran", "ocriculan", "octacosan", "octadecan",
"octavian", "octazan", "ocult", "offenbachian",
"ofidian", "ogan", "oitav", "oiticican",
"olbian", "olécran", "olecranian", "oleir",
"olíban", "oliculan", "oligôman", "olimpian",
"olisiponian", "olivençan", "olmedan", "olulan",
"omacefalian", "omalian", "oman", "omanian",
"omocefalian", "omotraquelian", "omotroquinian", "omotroquiterian",
"oneôman", "oniôman", "onomatôman", "onquestian",
"ontarian", "onusian", "opac", "opidan",
"opimian", "opiôman", "oportun", "opost",
"opsôman", "opulent", "oran", "oratorian",
"orbibarian", "orcian", "orcistan", "ordovician",
"orelhan", "orelheir", "oretan", "orexíman",
"organomagnesian", "orígan", "oritan", "oriund", "ormuzian",
"ornitisquian", "ornitôman", "orquestôman", "orquidian",
"ortódan", "ortodox", "osirian", "oslerian",
"ossean", "ossenian", "ossiandrian", "ostran",
"otaian", "otavian", "otoman", "otomian",
"outr", "ovarian", "ovidian", "ovolactovegetarian",
"oxfordian", "oxican", "oxonian", "pacatuban",
"padovan", "padroeir", "paduan", "paisan",
"palacian", "palauan", "paleoamerican", "paleochadian",
"paleodevonian", "paleossiberian", "paleoterian", "palepolitan",
"palermitan", "palestin", "palestinian", "palestrinian",
"palican", "palm", "palmean", "palmeiropolitan", "palmian",
"palmifalangian", "palminopolitan", "palmiplan", "palmirian",
"palmitopolitan", "palmopolitan", "palonzan", "palouzan",
"palpíman", "paludan", "paludian", "pambotan",
"pâmpan", "pampangan", "pampean", "pampian",
"panamian", "pandan", "pândan", "panfilian",
"panglossian", "panonian", "panopolitan", "panormitan",
"pantopelagian", "panurgian", "papeir", "papinian",
"papuan", "paquilemurian", "parabolan", "paracalan",
"paracefalian", "paracelsian", "paracian", "paradoxian",
"paradoxidian", "paragolcian", "paraguaian", "paraiban",
"paraisopolitan", "paralel", "paralian", "paran",
"paranaiban", "paranapolitan", "parapian", "paratireoidian",
"paratucan", "parentan", "parian", "parican",
"parisian", "parkinsonian", "parman", "parmigian",
"parnaiban", "parnasian", "paroquian", "parotian",
"parotidean", "parotidian", "parran", "parrascan",
"parsan", "parsian", "parteir", "partenian",
"partenopolitan", "parvatan", "pascalian", "pascoan",
"passagian", "past", "pasteuranian", "pasteurian",
"pastorean", "pastorian", "pastran", "pat",
"patalitan", "pataran", "paternian", "patôman",
"patrasan", "pátri", "patríci", "patrician",
"patripassian", "patron", "pauletian", "paulian",
"paulician", "paulistan", "paulopolitan", "pausulan",
"pavan", "pavlovian", "pechuran", "peciolean",
"peciolian", "pecuran", "pedan", "pedestrian",
"pedíman", "pedreir", "pedrinopolitan", "pedunculian",
"pégan", "pegasian", "peguan", "peircian",
"peixeir", "pelagian", "pelasgian", "pelean",
"pelican", "pelusian", "pelvian", "pelvitrocanterian",
"peman", "penapolitan", "pendencian", "pêndul",
"penean", "penian", "pensilvanian", "pentapolitan",
"penúltim", "pepucian", "pepuzian", "pequen",
"pequenin", "peregrin", "perfeit", "pérfid",
"periamigdalian", "periecian", "periesofagian", "perineoclitoridian",
"perioecian", "peripenian", "periurban", "permian",
"pernambucan", "perosian", "perrian", "persepolitan",
"persian", "peruan", "peruiban", "perusian",
"peruvian", "pervers", "péssim", "pessoan",
"pestalozzian", "pestan", "petrarquian", "petrobrucian",
"petrobrusian", "petrolífer", "petropolitan", "peucédan",
"pfaffian", "piagetian", "pian", "picassian",
"pífan", "pilan", "pinan", "pint",
"pioneir", "pioran", "piracicaban", "piracurucan",
"piran", "pirandellian", "piratiningan", "piratuban",
"pirauban", "pirenopolitan", "piritiban", "pirófan",
"pirôman", "piromelan", "piromêlan", "pironeptunian",
"pisan", "piscian", "pítan", "pitecian",
"pitopolitan", "pitulan", "placencian", "plantinian",
"plátan", "platician", "platicran", "platonian",
"plautian", "pleban", "plen", "plesiometacarpian",
"plinian", "plumban", "plutarquian", "plutonian",
"pluvian", "pneumotímpan", "pobrezinh", "pojucan",
"polandian", "poliactinian", "polian", "poliglican",
"polinesian", "polinopolitan", "poliplan", "politicôman",
"poliuretan", "polonian", "polovtsian", "pomb",
"pomeran", "pomeranian", "pompadourian", "pompean",
"pompeian", "poncian", "ponteir",
"pontian", "pontifíci", "pontocaspian", "popelican",
"poplican", "popperian", "porc", "poriôman",
"port", "porteir", "portlandian", "portulan",
"potician", "potirendaban", "potôman", "potsdamian",
"poundian", "poxorean", "pracean",
"pracian", "pradopolitan", "praian", "prasian",
"prastran", "prat", "pratapolitan", "pratian",
"praxean", "praxilian", "praxitelian",
"prefeit", "preparatorian", "pres", "presbiteran",
"presbiterian", "prestilian", "prestíman", "pretan",
"pretorian", "pretucan", "prévi", "previst",
"priabonian", "priamôman", "prim", "primeir",
"principan", "priscilian", "proboscidian", "procelian",
"proclian", "proculian", "procustian", "profund",
"prolan", "pront", "propan", "propost",
"própri", "proran", "prossifonian", "prossimian",
"prostitut", "proteinoglican", "proteoglican", "prototerian",
"proudhonian", "proustian", "provincian", "próxim",
"prudentopolitan", "prudhommian", "prussian", "prussôman",
"psatirian", "psilomelan", "pterigian", "pterocerian",
"pubian", "publican", "puccinian", "pudian",
"pupil", "pur", "puritan", "púrpur",
"pursarian", "put", "puteolan", "quacrian",
"quadr", "quadríman", "quadrúman", "quádrupl",
"quadrurban", "quaquerian", "quarenten", "quarenton",
"quart", "quartadeciman", "quartan", "quarteir",
"quartodeciman", "quatriduan", "quatruman", "quebradeir",
"quebradiç", "quedan", "queirosian", "queix",
"quelonian", "quenian", "queplerian", "querencian",
"querôman", "querquetulan", "quevedian", "quiban",
"quiet", "quiluran", "quiman", "quimeridgian",
"quin", "quinan", "quinquegencian", "quint",
"quintadeciman", "quintan", "quintilian", "quinzopolitan",
"quiribatian", "quirinopolitan", "quirófan", "quironian",
"quitandeir", "quotidian", "rában", "rabdófan",
"rabelaisian", "rabelian", "rabican", "rabichan",
"rabosan", "rabud", "racinian", "radícul",
"radiocarpian", "radiôman", "rafaelian", "ráfan",
"rafardan", "ragusan", "rai", "raian",
"rajaputan", "rajputan", "ram", "ramalh",
"rameir", "rapan", "raquidian", "rar",
"ras", "raspadeir", "rastafarian", "rasteir",
"rat", "ratzelian", "rauracian", "ravelian",
"rechan", "rechonchud", "recíproc", "reclus",
"recobert", "redavian", "redescobert", "redonian", "reduvian",
"reescrit", "referend", "regadeir", "regalian",
"regan", "regateir", "regatian", "regian",
"reginopolitan", "reichian", "religiôman", "remot",
"renan", "renanian", "repentin", "repolhud",
"reptilian", "republican", "repuls", "ret",
"retilian", "retinian", "retorian", "retroestilian",
"retrogasserian", "retroiridian", "retrossifonian", "reuchlinian",
"ri", "rianapolitan", "ribacudan", "ribagorzan",
"ribanceir", "ribatejan", "ribeir", "ribeiropolitan",
"ric", "ricaç", "ricardian", "ricardin",
"ricinian", "ridícul", "riemannian", "rifan",
"rij", "rinopolitan", "riojan", "riquetsian",
"risonh", "ríspid", "rissian", "ritterian",
"rizoplan", "robenhausian", "robercian", "robertian",
"robervalian", "robespierrian", "robinsonian", "robust",
"roçad", "roceir", "rod", "rodanian",
"rodarsenian", "rodelan", "rodendran", "rodesian",
"rodoarsenian", "rodófan", "rodriguian", "roemerian",
"rogacian", "rol", "rolandian", "roldonian",
"roman", "romeir", "rondonian", "rondonopolitan",
"ronsardian", "rooseveltian", "roqueir", "roquian",
"rosapolitan", "rosian", "rosman", "rosminian",
"rossinian", "rotarian", "roteiran", "rotineir",
"rotoplan", "rótul", "rotulian", "rouc",
"rousseaunian", "rox", "roxolan", "ruan",
"ruban", "rubelitan", "rubiacean", "rubican",
"rudolfian", "ruian", "ruivan", "rupelian",
"rurban", "ruselan", "rusquinian", "russ",
"russan", "russelian", "russellian", "russian",
"rustican", "rutilian", "saadian", "saarian",
"sabacian", "sában", "sabatarian", "sabatian",
"sabean", "sabelian", "sábi", "sabin",
"sabinan", "sabinian", "sabinopolitan", "saboian",
"sac", "sacassan", "sacoman", "sacr",
"sacramentan", "sacran", "sacríleg", "sacrococcigian",
"sacroprofan", "sacrorraquian", "sacrorraquidian", "sacrossant",
"sacrotrocanterian", "saelian", "saferian", "sagitarian",
"sahelian", "saian", "salacian", "salapitan",
"salernitan", "salesian", "salesopolitan", "salgadeir",
"saliferian", "salinopolitan", "salman", "salonitan",
"salpesan", "salpingian", "salsetan", "saltimbanc", "salvaterrian",
"salvatorian", "salveterran", "samanian", "samaritan",
"sambaetiban", "sambaquian", "samian", "samoan",
"samoran", "samosatian", "sampsean", "sanchesian",
"sandemanian", "sanfranciscan", "sanoisian", "sansimonian",
"sant", "santanopolitan", "santonian", "santopolitan",
"santorian", "santorinian", "sap", "sarafan",
"saragoçan", "sarapuian", "sarcocarpian", "sarcofagian",
"sardian", "sarian", "sarmacian", "sartrian",
"satanian", "saticulan", "satrican", "saturnian",
"saumurian", "saurisquian", "saussurian", "saxonian",
"schillerian", "schlegelian", "schoenbergian",
"schopenhauerian", "schubertian", "scottian", "seabranan",
"sebastianopolitan", "sebusian", "sec", "secan",
"secret", "secundan", "secundian", "sedetan",
"segarelian", "segestan", "segovian",
"segur", "sel", "selepitan", "seleucian",
"selimbrian", "semelitan", "sementeir", "semiprofan",
"semisserran", "semisshakespearian", "semissoberan", "senegalian",
"senegambian", "senhori", "senonian", "septempedan",
"septíman", "sequacian", "séquan", "sequanian",
"sequardian", "serapidian", "seren", "sergipan",
"seringueir", "sernancelhan", "serran", "serranopolitan",
"serravalian", "sertanej", "sertanopolitan", "sertolian",
"sertorian", "serv", "servian", "sesquiplan",
"setabitan", "setempedan", "setian", "setican",
"sétim", "setíman", "sevatian", "sever",
"severian", "sevilhan", "sext", "sextadeciman",
"shakespearian", "siablesian", "siagitan", "sibaritan",
"siberian", "sican", "sicelitan", "sicilian",
"sicran", "sideromelan", "sideromélan", "sideropolitan",
"sidonian", "siegenian", "sifilidôman", "sifilíman",
"sifilôman", "sifonian", "siglôman", "siguitan",
"sílab", "silan", "silbian", "silesian",
"silican", "silicobutan", "silicoetan", "silicometan",
"silicopentan", "silicopropan", "siloxan", "silurian",
"silvan", "silvian", "silvianopolitan", "simedian",
"simelian", "simian", "simonian", "simonsenian",
"simplician", "sincefalian", "sincer", "sincranian",
"sinemurian", "sínfan", "sinfisian", "singel",
"sinian", "sinistr", "sinistrôman", "sinocarotidian",
"siracusan", "sirenian", "sis", "siscian",
"sissomian", "sisud", "sitian", "sitiôman",
"smithian", "smithsonian", "soberan", "sobrean",
"sobredit", "sobreir", "sobressan", "sóbri",
"sóci", "socinian", "socioamerican", "sofoclian",
"sofôman", "sogaban", "sogdian", "sogr",
"solan", "solarian", "sold",
"soletan", "solt", "solteir", "somalian",
"sonâmbul", "sonor", "sons", "sopeir",
"soran", "sórdid", "sorelian", "sorian",
"sorocaban", "sorraian", "sortud", "sorveteir",
"sosan", "sosian", "soteropolitan", "soturn",
"sousean", "sousian", "sovcozian", "spencerian",
"spenglerian", "spinozian", "stahlian", "stancarian",
"stendhalian", "stenonian", "sternian", "straussian",
"strindbergian", "suan", "suarezian", "suasan",
"subacetan", "subcoracodian", "suberan", "subertan",
"súbit", "sublapsarian", "subpericranian", "subsaharian",
"subsignan", "substitut", "subuman", "suburitan",
"suculent", "sudertan", "suec", "suelitan",
"suessan", "suessetan", "suessonian", "suessulan",
"sufetulan", "sufolquian", "suíç", "suicidôman",
"suj", "sujeit", "sulan", "sulcitan",
"sulfatarian", "sulfonetilmetan", "sumatriptan", "sumenian",
"sumerian", "supost", "supramundan", "surd",
"surian", "surpres", "susan", "susanopolitan",
"suseran", "susian", "suspeit", "tabacôman",
"taban", "tabatingan", "tabelian", "taberneir",
"tabordian", "tacian", "tácit", "tafian",
"tagan", "tagorian", "tainian", "tairetan",
"taitian", "talaveran", "talh", "talipôman",
"tamacan", "tamanh", "tamarineir", "tamian",
"tamp", "tanatôman", "tanetan", "tang",
"tângan", "tangitan", "tanzanian", "tapetorretinian",
"tapiocan", "tapiratiban", "taproban", "tapsitan",
"taquarituban", "tardenosian", "tarituban", "tarsian",
"tarsofalangian", "tarsometatarsian", "tartessian", "tarvisan",
"tasian", "tasmanian", "tatuian", "taubatean",
"taunusian", "taurian", "taurisan", "tauromiritan",
"tavastian", "taxaspidian", "taylorian", "tchacavian",
"tchadian", "tean", "teban", "tebasan", "técnic",
"tegeócran", "tejan", "telemetacarpian", "telian",
"telurian", "temesan", "temnospondilian", "temperadinh",
"tenitan", "teodosian", "teôman", "terç",
"terceir", "terencian", "teresian", "teresopolitan",
"tergilan", "termidorian", "termopolitan", "tern",
"tertulian", "tesour", "tessalian", "testeir",
"testemunh", "tet", "tetraboran", "tetrabromoetan",
"tetracloralcan", "tetracloretan", "tetraclorometan", "tetracontan",
"tetracosan", "tetradecan", "tetraetilplumban", "tetrafenilfuran",
"tetraidrofuran", "tetraidropiran", "tetraiodoetan", "tetrametilometan",
"tetrametilosilan", "tetranitrometan", "tetrassilan", "teutôman",
"teverian", "texan", "texian", "ti",
"tian", "tibaran", "tiberian", "tibetan",
"tibiocalcanean", "tibiotarsian", "tiburcian", "tietean",
"tijucan", "tilestonian", "timboteuan", "timbrian",
"timelician", "timicitan", "tin", "tindaritan",
"tingitan", "tint", "tintureir", "tiofan",
"tiofuran", "tiran", "tiranôman", "tireoidian",
"tiroidian", "tirolean", "tironian", "tirrenian",
"tísan", "tisdritan", "titan", "titi",
"titilôman", "titonian", "tlandrian", "toan",
"toarcian", "tobaguian", "tobian",
"tocantinopolitan", "tocarian", "tol",
"tolan", "told", "toledan", "tolentinian",
"toletan", "tolosan", "tolstoian", "tomôman",
"tomosvarian", "tongan", "tongrian", "tont",
"torcularian", "torguian", "torment", "torneir",
"torrean", "torrejan", "torrian", "tort",
"tortonian", "tosc", "toscan", "totan",
"traç", "tractarian", "traducian", "tragan",
"tragian", "traianapolitan", "traiçoeir", "tralian",
"tramontan", "tramp", "tranc", "trancosan",
"tranqueir", "tranquil", "transamerican", "transarian",
"transbaicalian", "transbrasilian", "transcranian", "transerran",
"transfretan", "transiberian", "transilvan", "transilvanian",
"transjuran", "transleitan", "transmundan", "transpadan",
"transtigritan", "transuranian", "transvaalian", "transvalian",
"trapezometacarpian", "traquean", "traquelian", "traqueoesofagian",
"traseir", "trasmontan", "trasserran", "trastagan",
"travess", "travesseir", "treban", "trebian",
"trebulan", "trecorian", "trêfeg", "tremend",
"trêmul", "trentonian", "trepadeir", "trépan",
"trespan", "trev", "trevisan", "triacontan",
"trian", "tribrometan", "tribromoetan", "tribromometan",
"tribromopropan", "tribun", "trican", "tricloroetan",
"triclorometan", "tricloronitrometan", "tricloropropan", "tricôman",
"tricordian", "tricosan", "tricotonan", "tricuspidian",
"tridecan", "triduan", "trífan", "trifenilalquilidenofosforan",
"trifenilmetan", "trifluorometan", "trigueir", "trilh",
"trimarian", "trimetafan", "trimetilmetan", "trinc",
"trinitarian", "trinitrometan", "trioctubrian", "trioxan",
"triplan", "tripolitan", "triptofan", "triptófan",
"trisormian", "trissilan", "tristíman", "tritetracontan",
"tritonian", "triunfan", "troçad", "trocan",
"trocanterian", "trocantinian", "troian", "tromboxan",
"tromôman", "tropan", "troquinian", "troquiscan",
"troquiterian", "trunf", "tub", "tubian",
"tucitan", "tucuman", "tufican", "tujan",
"tulian", "tunapolitan", "tunetan", "tungomelan",
"tunisian", "tupian", "turanian", "turb",
"turbulent", "turcoman", "turdetan", "turduletan",
"turingian", "turnesian", "turonian", "turquestan",
"turv", "tuscan", "tusculan", "tuvaluan",
"tzigan", "uadaian", "ubanguian", "ubatuban",
"ucecian", "ucranian", "ufôman", "ugrian",
"ulan", "ulfilan", "ulissiponian", "ulsterian","últim",
"ultrarrepublican", "umbilican", "umbrian", "umbríman",
"umbrôman", "umerolecranian", "umeroolecranian", "unamunian",
"undecan", "undécim", "undeciman", "undecuman",
"undevicesiman","únic", "uníman", "uquedan",
"uralian", "uranófan", "urartuan", "urban", "urs",
"ursinh", "uruguaian", "usalitan", "usidican",
"usineir", "usuan", "uterôman", "uterovarian",
"uxian", "uxorian", "uzalitan", "vacan",
"vacarian", "vacean", "vadi",
"vagabund", "valangian", "valanginian", "valencian",
"valentin", "valentinian", "valerian", "valesian",
"valinginian", "valisoletan", "valombrosian", "valquirian",
"valsorcian", "valtonian", "valverdian", "vampir",
"vandean", "varan", "varangian", "varatojan",
"varcian", "varejeir", "vargian", "variscan",
"variscanian", "varronian", "varsovian", "varuan",
"varzean", "varzian", "vasc", "vassal",
"vast", "vatelian", "vaterian", "vaterladian",
"vatican", "vauclusian", "vazi", "vealdian",
"vegetalian", "vegetalin", "vegetarian", "veientan",
"vejentan", "velazquian", "velh", "velhac",
"velocíman", "vemelian", "venafran", "vendean",
"venecian", "venerand", "venetulan", "venezian",
"venezolan", "venezuelan", "venloquian", "vent",
"ventaneir", "venusian", "veragran", "veranopolitan",
"verbôman", "verdadeir", "verdian", "vergestan",
"vergilian", "verlainian", "vermelh", "vermelhinh",
"vermian", "vermidian", "verneian", "verrocchian",
"verulan", "verumontan", "vesalian", "vesan",
"vescelan", "vesgueir", "vespian", "vespuccian",
"vespucian", "vestinian", "vesulian", "vesuvian",
"veteran", "vianopolitan", "vibrionian", "vicencian",
"vicentopolitan", "vidian", "vieirian", "vietnamian",
"vigilancian", "vilaçoran", "vilanovian", "vilpian",
"vincian", "vind", "vindian", "vindobonian",
"violeir", "violent", "virchovian", "virgilian",
"virginian", "virginopolitan", "virglorian", "virgulian",
"virsungian", "virtonian", "visaian", "visanian",
"visean", "vistulian", "vitaminôman", "vitelian",
"vitian", "vitizan", "vitorian", "vitorin",
"vítre", "viúv", "viventan", "vivian",
"vizinh", "volan", "volapucôman", "volapuquian",
"volaterran", "volcentan", "volgian", "volscian",
"voltairian", "vomerian", "voprinian", "vormian",
"voss", "vulcanian", "vulfilan", "vulfilian",
"vulpian", "wagnerian", "washingtonian", "weberian",
"wildian", "xabregan", "xantitan", "xifoidian",
"xinguan", "zabian", "zairian", "zambian",
"zamoran", "zângan", "zeugitan", "zimbabuan",
"zombeteir", "zoroastrian", "zorr", "zuinglian",
"zwinglian"])

# plInvavb -  - substantivos de dois generos tambem encontrados em conjugacoes verbais.
plInvavb = set(["recruta", "reserva"])

# plvrInva - palavras de dois generos terminadas em -a-
plvrInva = set(["acrobata", "ala", "alfa", "aristocrata", "artista",
"atalaia", "atleta", "autarca", "autocrata", "bacana", "banana", "bandeirinha", "belga",
"besta", "beta", "burocrata", "cabeça", "cabra",
"cabrinha", "caça", "caixa", "camarada", "canalha", "capixaba",
"cara", "careca", "carioca", "carinha", "carmelita", "cestinha", "cineasta", "cinza", "cisma", "colega",
"crista", "democrata", "desavergonhada", "déspota",
"digna", "diplomata", "dourada", "extra",
"faixa", "fantasma", "ginasta", "gorila", "grama",
"grama", "guarda", "hipócrita",
"idiota",  "idólatra",
"impala", "indígena", "infanticida", "israelita", "japona", "java",
"jeca", "judoca", "laranja", "laranja",
"lhama", "longa", "magnata", "maia",
"maiorca", "maneta", "meia", "metropolita",
"monarca", "musicoterapeuta", "obstetra", "oligarca",
"ordenança", "papa", "palerma", "paquera", "paraíba",
"parasita", "parasita", "pateta", "patola",
"patriota", "pediatra", "penetra", "peralta",
"perneta", "picareta", "pilantra",
"pindoba", "pirata", "plutocrata", "poliglota",
"ponta", "porra", "porta", "prata", "primata",
"psicopata", "psiquiatra", "quilombola", "rábula",
"sacana", "salvaguarda", "salvaterra", "sanguessuga", "santista",
"sarcasta", "segurança", "sentinela",
"sentinela", "silvícola", "sósia", "sovina",
"suicida", "sulamita", "suprema",
"tagarela", "tanta", "taramela", "tecnocrata",
"terapeuta", "terapeuta", "triatleta", "troglodita",
"trouxa", "ultravioleta", "vigia", "violeta"])


# plvrMfr - palavras de dois generos terminadas em -r-
plvrMfr = set(["agricolar", "agroalimentar", "algar", "anterior", "antirradar",
"antissolar", "arquissecular", "arquivulgar", "atriovascular",
"atrioventricular", "auricular", "auriculoventricular", "autossimilar",
"auxiliar", "axilar", "bacelar", "bicolor", "biliar", "bilinear",
"bipolar", "bissecular", "capilar", "capitular", "cardiopulmonar",
"cardiovascular", "cavalar", "caviar", "cedular",
"cefaloauricular", "celular", "chanceler", "chofer", "circular", "colíder",
"complementar", "consular", "curricular", "designer", "dipolar", "disciplinar",
"discipular", "diverticular", "elementar", "epistolar", "escolar",
"espetacular", "estelar", "exemplar", "extracurricular", "exterior",
"familiar", "halter","ímpar", "incolor", "inferior", "interior", "irregular",
"líder", "lunar", "maior", "Major", "major", "malmequer",
"mártir", "medular", "melhor", "menor", "milhar",
"militar", "Mor", "mor", "muladar", "multicapsular",
"multicarpelar", "multidisciplinar", "multifamiliar", "multifilar",
"multilinear", "multilobular", "multilocular", "multimilenar",
"multinodular", "multinucelar", "multinuclear", "multiocular",
"muscular","nuclear", "ocular", "par", "paramilitar", "parlamentar",
"particular", "pelicular", "peninsular", "perpendicular", "pior", "polar",
"pomar", "popular", "posterior", "preliminar", "prostibular",
"protocolar","protossolar", "pulmonar", "quadrangular", "quatrissecular",
"regular", "repórter", "retangular", "reticular",
"retilinear", "salutar", "secular", "semiangular",
"semianular", "semicircular", "semipopular", "semirregular",
"semissecular", "similar", "singular", "solar",
"subcircular", "subintervalar", "subliminar", "submandibular",
"superior", "testicular", "titular",
"transcelular", "transecular", "tricolor", "tripolar", "trissecular",
"tubercular", "tubular", "tutelar", "ultrassolar",
"uniangular", "unianular", "uniarticular", "unicapsular",
"unicarpelar", "unicelular", "uniciliar", "unifamiliar",
"unifilar", "uniglandular", "unilaminar", "unilinear",
"unilobar", "unilobular", "unimolecular", "unimuscular",
"uninuclear", "uninucleolar", "uniocular", "uniovular",
"unipolar", "unitegumentar", "univalvular", "univesicular",
"univocabular", "urceolar", "uterolombar", "utricular",
"uvular", "vaginovulvar", "valecular", "valspodar",
"valvar", "valvular", "variolar", "vascular",
"ventricular", "verticilar", "vesicular", "vestibular",
"vocabular", "volar", "vulgar", "vulvar",
"zanzibar"])

# plvrFemr - palavras femininas terminadas em -r-
plvrFemr = set(["colher", "liminar", "mulher", "multimulher",
"supermulher"])

# excPlvrmr - excecoes palavras terminadas em -r- Exemplo: rigor
excPlvrmr = set(["favor","rigor"])

# plvrMascr - palavras masculinas terminadas em -r-
plvrMascr = set(["abajur", "açor", "açúcar", "afazer",
"africânder", "alamar", "albor", "algar",
"alguidar", "altar", "alter", "alvor",
"amanhecer", "amargor", "amor", "andor",
"ar", "ardor", "ator", "avatar",
"azar", "bar", "bastidor", "bazar",
"bilhar", "biopoder", "bolor", "cadáver",
"calcanhar", "calor", "câncer", "capacitor",
"capitular", "carácter", "caráter", "castor",
"casuar", "cateter", "celular", "césar",
"clamor", "clangor", "cobertor",
"cocar", "colar", "contêiner", "contrapoder",
"cultor", "czar", "desamor",
"despertar", "desprazer", "dissabor", "dólar",
"emir", "equador", "escâner", "esfíncter",
"esgar", "espaldar", "esplendor", "estupor",
"eurodólar", "familiar", "faquir", "fator",
"favor", "fedor", "fémur", "fêmur",
"fervor", "flexor", "flúor", "folar",
"folgar", "fragor", "freezer", "fulgor",
"fulmar", "funicular", "furor", "glaciar",
"hangar", "horror", "hospitalar", "humor",
"insular", "jaguar", "jantar", "labor",
"lagar", "langor", "lar", "lavor",
"lazer", "lêmur", "licor", "limiar",
"logar", "louvor", "luar", "lugar",
"luminar", "lupanar", "magiar", "malar",
"manjar", "mar", "mártir", "maxilar",
"menir", "mester", "microcomputador", "minicomputador",
"mister", "molar", "motor", "monsenhor", "muar",
"néctar", "nenúfar", "odor", "olhar",
"ômer","óscar", "paladar", "palmar",
"palor", "par", "patamar",
"pavor", "pendor", "pendular", "penhor",
"pilar", "placar", "polegar",
"pomar", "pormenor", "pôster", "prazer",
"primor", "prior", "pudor", "pundonor",
"quasar", "quefazer", "radar", "rancor",
"reator", "revólver", "rigor", "rubor",
"rumor", "sabor", "setor", "silhar",
"souvenir", "subpoder", "suéter", "suor",
"supercomputador", "superpoder", "superser", "suvenir",
"talher", "tambor", "tear", "teleprompter",
"televisor", "temor", "tenor", "teor",
"terror", "timor", "tôner", "torpor",
"transistor", "transpônder", "trator", "travor",
"tremor", "tuíter", "tumor", "ureter",
"valor", "vapor", "vedor",
"vetiver", "viajor", "vigor", "vizir",
"zíper"])

# plvrMfal - palavras masculinas/femininas terminadas em -al-
plvrMfal = set(["acidenta", "adiciona", "adverbia", "agricultura",
"algodoa", "amara", "ambulatoria", "amia",
"ancestra", "angelica", "anticoncepciona", "antimonia",
"anua", "area", "arena", "argumenta",
"armoria", "aroma", "arquitetura", "astra",
"atona", "atua", "audiovisua", "avena",
"azinha", "bagua", "bambua", "bamburra",
"bana", "barroca", "basa", "bestia", "bica", "Biena", "biena",
"boa", "boça", "borea", "braça",
"braga", "braquia", "breja", "bria",
"bruma", "bruta", "buca", "buça",
"caba", "cadeira", "cagaça", "cama",
"camaraza", "caniba", "Capita", "capita",
"carna", "carrasca", "carrega", "casua",
"catarra", "catua", "caucha", "cauda",
"celestia", "cenda", "centea", "centra",
"cerimonia", "cipoa", "circunstancia", "clora",
"colegia", "colonia", "coloquia", "colossa",
"comensa", "comercia", "comportamenta", "conceitua",
"condiciona", "confessiona", "conjuga", "continenta",
"convenciona", "cora", "cordia", "corna",
"corpora", "couva", "cova", "credencia", "crimina",
"crucia", "cua", "cuba", "cultura", "descomuna",
"desigua", "deslea", "diferencia", "digita",
"distrita", "diagona", "ditatoria", "divina", "documenta",
"dominica", "dorsa", "editoria", "emociona",
"empresaria", "ervaça", "esparga", "especia",
"espinha", "espiritua", "essencia", "estadua", "estata",
"estenda", "esteva", "etana", "eterna",
"eucalipta", "eventua", "excepciona", "existencia",
"experimenta", "factoria", "faia", "fana",
"fata", "fatoria", "fava", "fecha",
"federa", "fenomena", "fera", "ferragia",
"filia", "fina", "financia", "firma", "fisca",
"flora", "fluvia", "fora", "forma",
"frasca", "frecha", "freixia", "fronta",
"fruga", "funcha", "fundamenta", "gaia",
"gavia", "genia", "genita", "gera",
"gestua", "glioxa", "globa", "graa",
"gradua", "grama", "gramatica", "gutura", "habitua",
"hexagona", "hipertextua", "historia", "homossexua",
"horizonta", "idea", "igua", "ilega",
"imateria", "imora", "imorta", "imparcia",
"imperia", "impessoa", "inaugura", "incidenta", "individua",
"industria", "inferna", "informa", "inicia",
"instrumenta", "integra", "intelectua", "intempora",
"intenciona", "internaciona", "interna", "irraciona",
"irrea", "jovia", "judicia", "juvena",
"latera", "lea", "lega", "lexica", "libera",
"liga", "litera", "loca", "lodaça", "logica",
"lustra", "macambira", "macega", "madriga", "magistra",
"maiora", "manua", "marcia", "margina", "marita",
"mataga", "materna", "materia", "matina", "matorra",
"matricia", "medica", "medieva", "medronha", "mensa",
"menstrua", "menta", "mercuria", "meridiona", "mesca",
"metana", "milha", "minera", "missa",
"mistra", "moira", "monumenta", "mora",
"morta", "mundia", "municipa", "mura", "multinaciona",
"musica", "naciona", "nata", "natura",
"neanderta", "neolibera", "nepa", "nomina",
"norma", "numera", "nupcia", "occipita",
"ocidenta", "oficia", "ogiva", "oliva", "orienta",
"ora", "orbita", "origina", "ova", "ovença", "pacova",
"pactua", "pastiça", "pastora", "paterna", "patriarca",
"peda", "pedesta", "peitora", "pentagona",
"pentobarbita", "pessoa", "pinha", "plura",
"poia", "policia", "pontua", "porcentua",
"porta", "posta", "potencia", "prejudicia", "presidencia",
"primordia", "principa", "profissiona", "proporciona",
"provença", "proverbia", "providencia", "provincia",
"provisiona", "qua", "raciona", "radica",
"rea", "recita", "referencia", "rega",
"regiona", "representaciona", "residencia", "reviva",
"ria", "ritua", "riva", "rosa",
"roseira", "rotaciona", "rura", "sabuga",
"sacramenta", "sapa", "sarça", "sarrafaça",
"saturna", "semana", "semestra", "semifina", "senda",
"senega", "sensaciona", "sensua", "sentimenta",
"sercia", "seria", "seringa", "serviça",
"sexua", "sia", "sidera", "sindica",
"sisa", "situaciona", "sobrenatura", "socia",
"substancia", "sucursa", "superpolicia", "supraespinha",
"surrea", "taboca", "taipa", "ta", "teatra",
"temperamenta", "tempora", "tenda", "termina",
"terra", "tetragona", "tiopenta", "torrencia",
"tota", "tradiciona", "tramaga", "transnaciona", "transexua",
"transversa", "tremeda", "triba", "triga",
"trigona", "triunfa", "trivia", "tropica",
"tutoria", "umbra", "universa", "ura",
"urgueira", "urogenita", "usua", "va",
"vegeta", "vena", "verba", "verdea",
"vertica", "virgina", "virtua", "viscera",
"visua", "vita", "vitra", "voca", "voga",
"vorta"])

# plvrMascal - palavras masculinas terminadas em -al-
plvrMascal = set(["anima", "arraia", "arroza", "arsena",
"ava", "aventa", "banana", "beira",
"boca", "borna", "cabeda", "cacaua",
"cadava", "cafeza", "cajueira", "camba",
"cana", "canavia", "candea", "capinza",
"carda", "cardea", "cardia", "Carnava",
"carnava", "carvalha", "casa", "castanha",
"castiça", "cerea", "cereja", "chaca",
"chaparra", "chavasca", "choupa", "colmea",
"coqueira", "crista", "curra", "deda",
"Edita", "edita", "enxova", "escoria",
"feijoa", "fenobarbita", "festiva", "feuda",
"figueira", "funera", "genera", "hospita", "Hospita",
"jogra", "jorna", "lamaça", "laranja",
"litora", "loca", "manancia", "manantia",
"manca", "mandioca", "mangua", "mangueza",
"marecha", "materia", "memoria",
"meta", "milhara", "milheira", "palha",
"palmita", "pana", "pantana", "parda",
"parieta", "pascoa", "percentua", "pinheira",
"pomba", "ponta", "pontifica", "punha",
"quinta", "rabaça", "rama", "sa",
"salgueira", "seixa", "sina", "sobra", "subtota", "telejorna",
"Tribuna", "tribuna", "vara", "vendava"])

# excPlvral - excecoes palavras terminadas em -ais- Exemplo: cais, demais, jamais, mais
excPlvral = set(["c", "dem", "jam", "m"])


# plvrMascel - palavras masculinas terminadas em -el-
plvrMascel = set(["bachar", "broqu", "bur", "capit", "carret",
"carross", "cart", "cinz", "coquet",
"corc", "cord", "coron", "coud",
"decib", "dies", "donz",
"doss", "espinh", "farn", "f",
"fun", "furri", "g", "hidrom",
"hot", "lamb", "laur", "man",
"manu", "m", "menestr", "migu",
"moscat", "mot", "neb", "ourop",
"pain", "pap", "parc", "past",
"penafi", "pich", "pinc", "pinh",
"pix", "plant", "quart", "rap",
"rang", "redond", "rodoan", "saquit",
"sarapat", "ton", "trop", "verg",
"xair", "xar"])

# plvrAmascel - palavras Acentuadas masculinas terminadas em -el-
plvrAmascel = set(["arráte", "béte", "coromânde",
"cromoníque", "cúme", "cuproníque",
"díse", "esnórque", "fárte", "ferroníque",
"fúse", "galvanoníque", "guíme", "línge",
"mispíque", "ríge", "ríme", "rodôme",
"saíme", "síme", "túne"])

# excPlvrel - excecoes palavras terminadas em -eis- Exemplo: anegalheis, reis (com acenteo no e)
excPlvrel = set(["anegalh", "el", "fart", "r"])


# plvrMfel - palavras de dois generos terminadas em -el-
plvrMfel = set(["cru", "fi", "infi", "isab", "isra", "nob", "no"])


# plvrFemel - palavras femininas terminadas em -el-
plvrFemel = set(["cascav"])

# excplMascu - excecao preposicao A - palavras masculinas terminadas em -u- - Ex.: A meu, A seu, A teu
excplMascu = set(["me", "se", "te"])

# plvrMascu - palavras masculinas terminadas em -u-
plvrMascu = set(["abane", "abante", "abapor", "ab",
"acap", "acar", "aceste", "ac",
"aganipe", "agele", "agre", "ain",
"alifire", "alise", "amade", "amate",
"amebe", "amicle", "amorre", "anamburuc",
"anance", "ance", "androce", "ang",
"Ante", "antisse", "an", "apame",
"apoge", "aque", "aquile", "aracnoide",
"arame", "ara", "ardie", "arinfe",
"Ariste", "aritenoide", "armoze", "arque",
"arvade", "asaque", "asclepiade", "ascre",
"asdode", "asmode", "asmone", "asquelone",
"asside", "atamante", "atene", "ate",
"atiade", "atiroide", "at", "aule",
"babaç", "baba", "bab", "bacalha",
"bacura", "bacurub", "baguaç", "baiac",
"bailé", "baj", "balandra", "bali",
"bamb", "bang", "barce", "barde",
"bartolome", "bar", "barzab", "Basile",
"ba", "baur", "beij", "bela",
"belelé", "belope", "berimba", "betsile",
"bé", "bij", "bisna", "biz",
"bla", "boiguaç", "bolé", "Borrome",
"botaré", "botie", "brace", "braquicore",
"bresla", "bre", "briare", "briaré",
"bronte", "br", "bub", "burea",
"bur", "caaguaç", "cabuç", "cacaré",
"caca", "cadme", "cadmone", "caduce",
"caduve", "cafarne", "cafuç", "caguaç",
"caitit", "caj", "cajur", "cala",
"calde", "calha", "cal", "camafe",
"camar", "camba", "canaane", "canane",
"cancle", "cande", "candir", "canguç",
"cangur", "canoide", "canope", "carac",
"caramur", "carapa", "carapic", "carib",
"carne", "car", "caruar", "carur",
"cassiope", "cataguaç", "catarre", "catata",
"catech", "cauch", "ceratoioide", "cé",
"chabé", "chande", "chapé", "charé",
"charte", "cha", "chibe", "chichisbé",
"chim", "chir", "ch", "chuch",
"churre", "ciadopitioide", "cifioide", "cilace",
"cilene", "cile", "circe", "cirene",
"Cirine", "cirsoide", "citere", "cite",
"citofaringe", "cla", "clidocoracoide", "clidomastoide",
"codie", "coip", "colipe", "colise",
"colone", "colora", "conope", "corcire",
"core", "corife", "coroide", "corone",
"coronoide", "cor", "coruché", "cossie",
"costocoracoide", "costocoracoroide", "costoxifoide", "cotiloide",
"crajur", "cretche", "cricaritenoide", "cricoaritenoide",
"cricoide", "cricotireoide", "cricotiroide", "criptojude",
"cr", "c", "cuc", "culde",
"cunauar", "cupuaç", "curaça", "cura",
"curur", "curuz", "dacriope", "degra",
"dia", "ec", "embiruç", "em",
"enx", "eorde", "epicure", "epidafne",
"epidaure", "epige", "epistrofe", "eque",
"erimante", "erine", "erite", "eritre",
"eritrofle", "er", "escarabe", "escarcé",
"esminte", "esmirne", "espale", "espele",
"espermatope", "esponde", "estilioide", "estiloide",
"estomode", "estratonice", "estubere", "ete",
"etope", "eube", "euripide", "europe",
"eve", "faetonte", "falane", "falante",
"falasarne", "fane", "fare", "farise",
"farrumpe", "febe", "feloide", "fere",
"fereze", "fich", "filace", "filiste",
"filobasile", "firmide", "flie", "floresce",
"foce", "fogaré", "frise", "frixe",
"gabir", "gaibé", "galile", "gate",
"ga", "gaze", "gebade", "gergese",
"gete", "gigante", "gimnobelide", "ginece",
"gire", "girgase", "girre", "gite",
"glenoide", "gogiare", "goiam", "gomorre",
"gordie", "grabe", "grage", "gra",
"gregue", "grisé", "gris", "guabir",
"guaicur", "guarib", "guatamb", "gua",
"guigrateute", "gur", "hamate", "hamire",
"hebre", "hecate", "hecatombe", "hemoperitone",
"heracle", "heraclite", "herie", "herme",
"hermione", "hesiode", "hete", "he",
"heve", "hialoide", "hible", "hierope",
"hilocere", "himene", "hioide", "hiotiroide",
"hipate", "hipofle", "hipoge", "hore",
"hut", "iatre", "ic", "idume",
"igaraç", "iguaç", "ilhé", "il",
"imb", "imp", "inamb", "incré",
"ingranzé", "inhamb", "iola", "iridocoroide",
"isse", "ita", "itone", "iture",
"jabir", "jabur", "jacobe", "jac",
"jacurut", "jaguaruç", "jamb", "jap",
"ja", "jaze", "jebuse", "jenisse",
"jes", "jinze", "jira", "jube",
"jubile", "jude", "juj", "jungi",
"jurur", "justiniane", "labé", "lacra",
"lala", "lamb", "landa", "landr",
"lapite", "lara", "larisse", "lebré",
"lene", "lepidoste", "lerne", "lete",
"lé", "lice", "licre", "lige",
"liofloe", "lipare", "lorditome",
"lua", "lul", "lumaré", "macabe",
"macac", "macaré", "maca", "mac",
"madiane", "majaque", "maj", "malacoste",
"mal", "mana", "manch", "mandacar",
"mande", "mang", "manib",
"manique", "mani", "manté", "man",
"manumiti", "marab", "maracat", "mara",
"marimba", "marip", "marone", "maruce",
"masde", "mase", "masore", "massile",
"mastaré", "mastoide", "matate", "mat",
"ma", "mausolé", "maze", "meduse",
"melibe", "melite", "menandre", "mene",
"menipe", "men", "mesenquitre", "mesofle",
"metimne", "me", "meze", "mia",
"micene", "mide", "miiofone", "milase",
"mile", "milhare", "milhere", "milhe",
"milre", "mindana", "mine", "minga",
"ming", "mitilene", "mitre", "mit",
"mize", "mocama", "molung", "monte",
"mor", "m", "muie", "mule",
"mulung", "mundé", "munduruc", "mung",
"murna", "mur", "murucutut", "murur",
"muse", "mutusce", "nabate", "nadaba",
"nal", "namb", "nape", "nasore",
"natabe", "natine", "naumaque", "nazare",
"nazire", "nazore", "nda", "nemee",
"neme", "nese", "nhamb", "nicola",
"ninfe", "nise", "n", "nube",
"odontoide", "omb", "omoioide", "omumbe",
"ore", "orse", "otrione", "ot",
"oxe", "ozue", "pacat", "paca",
"pac", "pade", "page", "palene",
"panac", "panage", "pange", "panque",
"pante", "panticape", "papanicola", "pap",
"paquifle", "parafle", "paraguaç", "parale",
"parda", "parore", "partenope", "par",
"patare", "pata", "pa", "peg",
"pelebre", "pelene", "pele", "pelope",
"pelorne", "pera", "pergame", "perge",
"peridere", "perige", "perine", "peritone",
"perize", "perone", "Perse", "per",
"petre", "peucale", "peucane", "pia",
"pidne", "pige", "pigme", "pimple",
"piné", "pinté", "piracat", "piramb",
"piraruc", "pirene", "pire", "pise",
"pissele", "pitane", "pité", "pit",
"pituaç", "platea", "plebe", "plesiopene",
"pne", "pneumoperitone", "policne", "polidesmoide",
"Pompe", "pontine", "porne", "pot",
"povaré", "povilé", "povolé", "prestiloide",
"priape", "pritane", "procoracoide", "propile",
"proscarabe", "proteocefaloide", "prote", "pr",
"psamoscafe", "pseudofle", "psi", "pterigoide",
"pure", "purupur", "putumuj", "quante",
"quarta", "quatipur", "quene", "queneze",
"quiloperitone", "quina", "quincaj", "quinguing",
"quixexe", "radame", "ragoide", "raguse",
"rame", "ra", "reb", "recr",
"retroestiloide", "retromastoide", "ré", "rife",
"ripe", "rodope", "rome", "sabe",
"saduce", "sag", "sal", "same",
"sande", "sanhaç", "sapule", "saquité",
"sara", "sasse", "satarque", "semin",
"sens", "septogle", "serape", "sesamoide",
"se", "sicurij", "sige", "sigmoide",
"siloge", "sine", "sinimb", "sit",
"sobe", "sobrecé", "sobrec", "sola",
"solidé", "sotade", "sové", "subatloide",
"subaxoide", "sucurij", "sucuruj", "surucuc",
"surur", "suss", "tabaré", "tab",
"tacur", "Tade", "tajaç", "tange",
"tapir", "tapur", "taquaraç", "taquaruç",
"tata", "tat", "ta", "taumatope",
"tcha", "tebe", "tegee", "terapne",
"tere", "terine", "tespie", "tetartope",
"teté", "te", "tiane", "tife",
"tifoide", "timane", "timb", "timele",
"tinfe", "tip", "tireoide", "tirisse",
"tiroide", "tof", "tol", "tomabe",
"torone", "tor", "tre", "trocoide",
"trofé", "trogofle", "troque", "troussea",
"tucanaç", "tur", "turuç", "tut",
"ua", "ub", "ubuç", "uiraç",
"uirapur", "ulisse", "umb", "upene",
"ur", "urub", "uruc", "uruta",
"urut", "uscarde", "vace", "vade",
"vagana", "varapa", "varde", "vasse",
"va", "vé", "vile", "vit",
"xamp", "xelelé", "xere", "xexé",
"xilre", "zaibats", "zancle", "za",
"zemare"])



# plvrMfeu - palavras de dois generos terminadas em -u-
plvrMfeu = set(["amad", "bid", "calund", "coand", "cuand", "cud", "hind",
"land", "lund", "mandch", "mand", "murund", "nand", "nhand", "urd",
"vod", "vud", "zeb", "zul"])


# plvrFemu - palavras femininas terminadas em -u-
plvrFemu = set(["bant", "na"])


# plvrMasci - palavras masculinas terminadas em -i-
plvrMasci = set(["abacax", "bacur", "bâmb", "beribér", "biquín", "bir", "biribir", "birigu",
"bistur", "bo", "bonsa", "canicân", "carir", "caxix", "chass", "cobó",
"confre", "Dav", "dicur", "epóx", "esqu", "fre", "futevôle", "gib",
"guaran", "gur", "heró", "hóque", "jabut", "jacúz", "javal", "jóque",
"júr", "jurit", "missúr", "mitir", "muzumb", "oitch", "oit", "pa","papa",
"parat", "pênalt", "pip", "piqu", "pit", "pône", "pot", "raviól",
"re", "rub", "sac", "sagu", "samura", "sanse", "sapot", "sir", "sirir",
"tambaqu", "tambur", "táx", "tsunam", "tucup", "vôle", "xix"])


# plvrMfei - palavras de dois generos terminadas em -i-
plvrMfei = set(["dodó", "faqu", "gar", "ianomâm", "naz", "náz", "nisse", "paparazz", "travest", "zumb"])


# plvrFemi - palavras femininas terminadas em -i-
plvrFemi = set(["le", "sucur"])



# plvrMfcao - palavras dois generos terminadas em -cao- cedilha til a o
plvrMfcao = set(["lotaç"])

# plvrMascao - palavras masculinas terminadas em -cao- cedilha til a o
plvrMascao = set(["arç", "balç",
"cabeç", "calç",
"carroç", "Coraç",  "coraç",
"crianç", "escanç",
"infanç", "tiç",
"tropeç"])

# excPlvrao - palavras terminadas em -ao- cujo plural eh diferente de -oes-, nao tem plural ou admite outras formacoes
excPlvrao = set(["afeg", "alde", "alem", "amont", "anci", "bang", "bordeg",
"borguinh", "c", "capel", "capit", "catal", "ch", "charlat", "cibercidad", "cidad", "coirm", "concidad",
"concrist", "crist", "de", "engatat", "ent", "escriv", "est", "estrab", "foç", "fugi",
"garnach", "geres", "goz", "gr", "guardi", "h",  "hortel", "ignorant",
"irm","órf","órg","Õrg", "p", "pag", "pardal", "repont",
"s", "sen", "solim", "t", "tabeli", "tintur", "usucapi", "zombeir", "zorn", "zorr"])

# excPlvrnasce - excecoes palavras masculinas terminadas em -E- Exemplo: convite
excPlvrnasce = set(["convit", "toqu"])


# plvrNfemcao - palavras nao femininas terminadas em -cao- cedilha til a o
plvrNfemcao = set(["almoç", "arç", "balç",
"cabeç", "cadarç", "calç",
"carroç", "Coraç",  "coraç",
"crianç", "escanç",
"infanç", "lotaç",
"pedaç", "pescoç", "tiç",
"tropeç"])

#plvrFemao  - Palavras femininas terminadas em -ao-
plvrFemao = set(["acess", "Acess",
"acupress", "Acupress", "ades", "Ades",
"admiss", "Admiss", "adust", "Adust",
"agress", "Agress", "alus", "Alus",
"amarelid", "Amarelid", "amplid", "Amplid",
"animadvers", "Animadvers", "antevers", "Antevers",
"antevis", "Antevis", "apreens", "Apreens", "aptid", "Aptid",
"ascens", "Ascens", "aspers", "Aspers",
"autogest", "Autogest", "avers", "Avers", "avuls", "Avuls"
"bernard", "Bernard", "bioimpress", "Bioimpress",
"branquid", "Branquid",
"brusquid", "Brusquid", "certid", "Certid",
"cess", "Cess", "circuncis", "Circuncis",
"cis", "Cis", "coaliz", "Coaliz",
"coes", "Coes", "colis", "Colis",
"combust", "Combust", "comich", "Comich",
"comiss", "Comiss", "compaix", "Compaix",
"compreens", "Compreens", "compress", "Compress",
"compuls", "Compuls", "comunh", "Comunh",
"concess", "Concess", "concis", "Concis",
"conclus", "Conclus", "concuss", "Concuss",
"conex", "Conex", "confiss", "Confiss",
"confus", "Confus", "congest", "Congest",
"contram", "contrarraz", "Contrarraz",
"contus", "Contus", "convers", "Convers",
"convex", "Convex", "convuls", "Convuls",
"corras", "Corras",
"corros", "Corros", "cosmovis", "Cosmovis",
"crucifix", "Crucifix", "Cubat", "d",
"D", "decis", "Decis", "defens",
"Defens", "deflex", "Deflex", "dem", "Dem", "demiss", "Demiss",
"densid", "Densid", "depress",
"Depress", "deris", "Deris", "descens",
"Descens", "descompress", "Descompress", "desconex",
"Desconex", "desilus", "Desilus",
"despretens", "Despretens", "desuni",
"Desuni", "devassid",
"Devassid", "difus", "Difus", "digest",
"Digest", "digress", "Digress", "dimens",
"Dimens", "discuss", "Discuss", "dispers",
"Dispers", "dissens", "Dissens", "dissuas",
"Dissuas", "distens", "Distens", "divers",
"Divers", "divis", "Divis",
"eclos", "Eclos", "efus",
"Efus", "egest", "Egest", "electrocuss",
"Electrocuss", "eletrofus", "Eletrofus", "emers",
"Emers", "emiss", "Emiss", "emuls",
"Emuls", "entrevis", "Entrevis", "eros",
"Eros", "escans", "Escans",
"escravid", "Escravid", "escrupulid",
"Escrupulid", "escurid", "Escurid", "espessid",
"Espessid", "evas", "Evas", "Ex-pris",
"exatid", "Exatid", "exaust", "Exaust",
"excis", "Excis", "exclus", "Exclus",
"excomunh", "Excomunh", "excurs", "Excurs",
"expans", "Expans", "explos", "Explos",
"express", "Express", "expuls", "Expuls",
"extens", "Extens", "extors", "Extors",
"extrovers", "Extrovers", "extrus", "Extrus",
"f", "F", "fiss", "Fiss",
"flex", "Flex",
"fotoemiss", "Fotoemiss", "fotofiss", "Fotofiss",
"Fraz", "fresquid",
"Fresquid", "frouxid", "Frouxid", "fus",
"Fus", "gest", "Gest", "gratid",
"Gratid", "hipertens", "Hipertens", "hipotens",
"Hipotens", "ilus", "Ilus", "imensid",
"Imensid", "imers", "Imers", "imiss",
"Imiss", "implos", "Implos", "imprecis",
"Imprecis", "impress", "Impress", "impuls",
"Impuls", "imunodepress", "Imunodepress", "inaptid",
"Inaptid", "incis", "Incis", "inclus",
"Inclus", "incompreens", "Incompreens", "incurs",
"Incurs", "indecis", "Indecis", "indigest",
"Indigest", "indivis", "Indivis", "inexatid",
"Inexatid", "inexpress", "Inexpress", "inflex",
"Inflex", "infus", "Infus", "ingest",
"Ingest", "ingratid", "Ingratid", "insubmiss",
"Insubmiss", "intens", "Intens", "intercess",
"Intercess", "interconex", "Interconex", "intromiss",
"Intromiss", "introvers", "Introvers", "intrus",
"Intrus", "invas", "Invas", "invers",
"Invers", "irreflex", "Irreflex", "irreligi",
"Irreligi", "irris", "Irris", "lassid",
"Lassid", "legi", "Legi", "lentid",
"Lentid", "les", "Les", "maloclus",
"Maloclus", "macrogest", "Macrogest", "macrorregi", "Macrorregi",
"mans", "Mans", "mansid",
"Mansid", "m", "M", "mãos",
"Mãos", "microgest", "Microgest", "microrregi", "Microrregi",
"miss", "Miss", "mornid", "Mornid",
"Motos", "multid", "Multid", "negrid",
"Negrid", "obsess", "Obsess", "ocasi",
"Ocasi", "oclus", "Oclus", "omiss",
"Omiss", "opini", "Opini", "opress",
"Opress", "paix", "Paix", "pens",
"Pens", "percuss", "Percuss", "perdig",
"Perdig", "perfus", "Perfus", "permiss",
"Permiss", "perplex", "Perplex", "persuas",
"Persuas", "pervers", "Pervers", "podrid",
"Podrid", "possess", "Possess", "precess",
"Precess", "precis", "Precis",
"Preclus", "preclus", "preens",
"Preens", "press", "Press", "pretens",
"Pretens", "pretid", "Pretid", "previs",
"Previs", "pris", "Pris", "prociss",
"Prociss", "profiss", "Profiss", "profus",
"Profus", "progress", "Progress", "prontid",
"Prontid", "propens", "Propens", "propuls",
"Propuls", "provis", "Provis", "puls",
"Puls", "quest", "Quest", "radiodifus",
"Radiodifus", "radiotelevis", "Radiotelevis", "ras",
"Ras", "raz", "Raz", "readmiss",
"Readmiss", "rebeli", "Rebeli", "recens",
"Recens", "recess", "Recess", "reclus",
"Reclus", "reconvers", "Reconvers", "reflex",
"Reflex", "regi", "Regi", "regress",
"Regress", "reimpress", "Reimpress", "reinvers",
"Reinvers", "religi", "Religi", "remiss",
"Remiss", "repercuss", "Repercuss", "repreens",
"Repreens", "repress", "Repress", "repuls",
"Repuls", "rescis", "Rescis", "retid",
"Retid", "retransmiss", "Retransmiss", "retrovers",
"Retrovers", "reuni", "Reuni", "revers",
"Revers", "revis", "Revis", "rez",
"Rez", "rouquid", "Rouquid", "sans", "saz",
"Saz", "secess", "Secess", "sequid",
"Sequid", "servid", "Servid", "sess",
"Sess", "sobrefus", "Sobrefus", "sofreguid",
"Sofreguid", "soid", "Soid", "solid",
"Solid", "soliflux", "Soliflux", "su",
"Su", "suas", "Suas", "subcomiss",
"Subcomiss", "subdivis", "Subdivis", "submers",
"Submers", "submiss", "Submiss", "subpress", "subvers",
"Subvers", "sucess", "Sucess", "sugest",
"Sugest",  "superpress", "supervis", "Supervis", "supress",
"Supress", "suspens", "Suspens", "televis",
"Televis", "tens", "Tens", "transfus",
"Transfus", "transgress", "Transgress", "transmiss",
"Transmiss", "uni", "Uni", "vaguid",
"Vaguid", "vastid", "Vastid", "vaz",
"Vaz", "vermelhid", "Vermelhid", "vers",
"Vers", "vis", "Vis"])

#excPlvrio  - Excecao palavras masculinas terminadas em ario - erio - irio - orio.
excPlvrio = set(["crité"])


#plvrFmenta  - Palavras femininas terminadas em -menta-
plvrFmenta = set(["ciu", "curti", "e", "escara",
"farda", "ferra", "frege", "frigi", "hipono",
"impedi", "ju", "multiferra", "nanoferra",
"ossa", "pala", "pi", "superferra", "superpi",
"tor", "travesti", "urdi", "varri", "vesti"])


# plvrFemal - palavras femininas terminadas em -al-
plvrFemal = set(["bacana", "catedra", "Catedra", "espira", "vesta"])

# plvrMasces - palavras masculinas terminadas em -es- frances, ingles
plvrMasces = set(["açailand", "african", "alambr", "alban", "alhambr", "almalagu", "amarant",
"amesterdan", "amsterdam", "analand", "anam", "anauriland", "andaman",
"angol", "antiportugu", "arago", "aragon", "aram", "arden", "argu", "arn",
"arouqu", "assam", "assuan", "auriland", "austral", "avaitanh", "avil",
"azerbaidjan", "baam", "babaçuland", "bagr", "baion", "balin", "baltistan",
"bandan", "banglad", "barcelon", "barem", "baron", "bearn", "bengal",
"beripocon", "berlin", "bermud", "bern", "bintan", "birman", "bolonh",
"bordal", "bordel", "borgonh", "brabant", "braganc", "bragu", "brandeburgu",
"brandemburgu", "braqu", "brasiland", "brasil", "bruxel", "buenoland", "bulgar",
"burbon", "burgal", "burgu", "burocrat", "burundin", "butan", "cafeland",
"caiman", "calabr", "calc", "camaron", "camo", "campidan", "campon", "canar",
"canav", "canton", "cartagin", "cassiland", "catalan", "catangu", "cazaquistan",
"cercoschampanh", "charol", "chin", "cingal", "cochinchin", "cochin", "coimbr",
"ongol", "cordisburgu", "cordisland", "cordofan", "cordov", "corningl",
"cornualh", "corunh", "cort", "cotan", "cracov", "craon", "cristaland",
"crixaland", "cruciland", "cruzeland", "cupromangan", "dan", "daro", "delfin",
"descort", "diep", "dinamarqu", "divinoland", "dominiqu", "econom", "entrem",
"escoc", "estrasburgu", "etiop", "fero", "ferromangan", "filipin", "fin",
"finland", "franc", "fregu", "freibergu", "gabon", "gal", "galur", "gandar",
"gandr", "gan", "garc", "garon", "garvon", "gatin", "gaul", "ge", "genebr",
"geno", "genov", "genu", "goan", "go", "gregu", "grioland", "groenland",
"groneland", "guadramil", "guam", "guian", "gurgu", "hamburgu", "havan",
"herculand", "hidroland", "hinterland", "holand", "honcongu", "hungar",
"indochin", "ingl", "internet", "irland", "iroqu", "island", "ivoland",
"janian", "japon", "javan", "jornal", "jutland", "land", "leon", "liban",
"ligar", "limburgu", "lion", "liorn", "lisbo", "lisbon", "logudor", "londr",
"loren", "luand", "lund", "luxemburgu", "luziland", "macaçar", "madril",
"madur", "malagu", "malaqu", "malin", "malt", "maluqu", "malvin", "man",
"mangan", "manqu", "marian", "maron", "marqu", "marselh", "m", "men",
"milan", "mirandel", "mirand", "m", "montanh", "mont", "montmartr",
"morqu", "morvand", "morvan", "nant", "napol", "narbon", "navarr",
"neandertal", "neerland", "neoportugu", "neozeland", "nepal", "nic",
"nicobar", "nivern", "norland", "noruegu", "oldemburgu", "olind", "ondurman",
"oran", "orlean", "pamplon", "paquistan", "pard", "parten", "parvon",
"pedren", "pedr", "pequin", "petersburgu", "piemont", "piov", "poiar",
"polon", "portugal", "preit", "psicologu", "publicit", "quercin",
"quirguistan", "rent", "r", "riodonor", "rionor", "romanh", "roterdam",
"ruand", "saiagu", "samarin", "sangalh", "sang", "santiagu", "sassar",
"sendin", "senegal", "sequan", "setem", "siam", "sien", "silicomangan",
"singal", "siroqu", "sirvent", "sociologu", "suaziland", "sudan", "sundan",
"surinam", "tailand", "taiwan", "tajiquistan", "tamar", "tarascon", "tarb",
"tarent", "tarouqu", "tarragon", "tavan", "ternat", "terrant", "teutoburgu",
"tirol", "togol", "togu", "tongan", "tonquin", "torn", "torqu", "torrant",
"trem", "tugand", "tulon", "tuquian", "turin", "turqu", "tuval", "uberland",
"ugand", "usbequistan", "uzbequistan", "vaiamont", "veron", "versalh", "vian",
"vien", "vietnam", "vigu", "virgoland", "vurtembergu", "xangai", "zeland"])


# aacpNos = Concordancia  com o pronome Nos (com acento)
aacpNos = {"abandona": "abandonamos",
"abandonou": "abandonamos",
"abdica": "abdicamos",
"abdicasse": "abdicássemos",
"abdicou": "abdicamos",
"abraça": "abraçamos",
"abraçou": "abraçamos",
"abre": "abrimos",
"abriu": "abrimos",
"acaba": "acabamos",
"acabou": "acabamos",
"aceita": "aceitamos",
"aceitaria": "aceitaríamos",
"aceitou": "aceitamos",
"acha": "achamos",
"achava": "achávamos",
"achou": "achamos",
"acolhe": "acolhemos",
"acompanha": "acompanhamos",
"acompanhou": "acompanhamos",
"acorda": "acordamos",
"acordou": "acordamos",
"acredita": "acreditamos",
"acreditava": "acreditávamos",
"acreditou": "acreditamos",
"acrescenta": "acrescentamos",
"acrescentou": "acrescentamos",
"adivinha": "adivinhamos",
"adivinhou": "adivinhamos",
"admira": "admiramos",
"admirou": "admiramos",
"admitir": "admitirmos",
"adora": "adoramos",
"adorou": "adoramos",
"adota": "adotamos",
"adote": "adotemos",
"adotou": "adotamos",
"adquire": "adquirimos",
"adquiriu": "adquirimos",
"adulterar": "adulterarmos",
"adverte": "advertimos",
"advertiu": "advertimos",
"advoga": "advogamos",
"advogou": "advogamos",
"age": "agimos",
"agiu": "agimos",
"agradece": "agradecemos",
"agradeceu": "agradecemos",
"ajuda": "ajudamos",
"ajudará": "ajudaremos",
"ajudou": "ajudamos",
"ajunta": "ajuntamos",
"ajuntou": "ajuntamos",
"ajusta": "ajustamos",
"ajustou": "ajustamos",
"alcançava": "alcançávamos",
"alimenta": "alimentamos",
"alimentou": "alimentamos",
"aliviava": "aliviávamos",
"almoça": "almoçamos",
"almoçou": "almoçamos",
"alugasse": "alugássemos",
"ama": "amamos",
"amarra": "amarramos",
"amarrou": "amarramos",
"amava": "amávamos",
"amou": "amamos",
"analisa": "analisamos",
"analisasse": "analisássemos",
"analisou": "analisamos",
"anda": "andamos",
"andava": "andávamos",
"andou": "andamos",
"anuncia": "anunciamos",
"apanha": "apanhamos",
"apanhou": "apanhamos",
"apareça": "apareçamos",
"aparece": "aparecemos",
"apareceu": "aparecemos",
"aplaude": "aplaudimos",
"aplaudiu": "aplaudimos",
"aplicaria": "aplicaríamos",
"apodrece": "apodrecemos",
"apodreceu": "apodrecemos",
"apoia": "apoiamos",
"apoiou": "apoiamos",
"aprende": "aprendemos",
"aprendeu": "aprendemos",
"apresenta": "apresentamos",
"apresentará": "apresentaremos",
"apresentou": "apresentamos",
"apresta": "aprestamos",
"aprestou": "aprestamos",
"aprova": "aprovamos",
"aprovasse": "aprovássemos",
"aproveita": "aproveitamos",
"aprovou": "aprovamos",
"arranjará": "arranjaremos",
"arranjasse": "arranjássemos",
"arrasta": "arrastamos",
"arrastou": "arrastamos",
"arromba": "arrombamos",
"arrombou": "arrombamos",
"aspira": "aspiramos",
"aspirou": "aspiramos",
"assina": "assinamos",
"assinar": "assinarmos",
"assinou": "assinamos",
"assiste": "assistimos",
"assistir": "assistirmos",
"assistiu": "assistimos",
"associa": "associamos",
"assume": "assumimos",
"assumiu": "assumimos",
"ataca": "atacamos",
"atacou": "atacamos",
"atiraria": "atiraríamos",
"atravessa": "atravessamos",
"atravessava": "atravessávamos",
"atravessou": "atravessamos",
"aumentar": "aumentarmos",
"avalia": "avaliamos",
"avaliar": "avaliarmos",
"avaliou": "avaliamos",
"avança": "avançamos",
"avançou": "avançamos",
"bate": "batemos",
"bateu": "batemos",
"batia": "batíamos",
"batiza": "batizamos",
"batizou": "batizamos",
"bebe": "bebemos",
"beneficia": "beneficiamos",
"beneficiou": "beneficiamos",
"bota": "botamos",
"botou": "botamos",
"brada": "bradamos",
"bradou": "bradamos",
"briga": "brigamos",
"brigou": "brigamos",
"brinca": "brincamos",
"brincava": "brincávamos",
"brincou": "brincamos",
"busca": "buscamos",
"buscou": "buscamos",
"cabe": "cabemos",
"caçoa": "caçoamos",
"caçoou": "caçoamos",
"cai": "caímos",
"cairia": "cairíamos",
"caísse": "caíssemos",
"caiu": "caímos",
"cala": "calamos",
"calcula": "calculamos",
"calculou": "calculamos",
"calou": "calamos",
"caminha": "caminhamos",
"caminhou": "caminhamos",
"cantará": "cantaremos",
"carece": "carecemos",
"careceu": "carecemos",
"carrega": "carregamos",
"carregava": "carregávamos",
"carregou": "carregamos",
"casa": "casamos",
"casou": "casamos",
"cavalga": "cavalgamos",
"cavalgou": "cavalgamos",
"celebra": "celebramos",
"celebrou": "celebramos",
"chama": "chamamos",
"chamava": "chamávamos",
"chamou": "chamamos",
"checa": "checamos",
"checou": "checamos",
"chega": "chegamos",
"chegar": "chegarmos",
"chegará": "chegaremos",
"chegava": "chegávamos",
"chegou": "chegamos",
"chegue": "cheguemos",
"chora": "choramos",
"chorava": "chorávamos",
"chore": "choremos",
"chorou": "choramos",
"cita": "citamos",
"citou": "citamos",
"cobra": "cobramos",
"cobrou": "cobramos",
"colabora": "colaboramos",
"colaborou": "colaboramos",
"coloca": "colocamos",
"colocou": "colocamos",
"combaterá": "combateremos",
"combina": "combinamos",
"combinava": "combinávamos",
"combinou": "combinamos",
"come": "comemos",
"começa": "começamos",
"começar": "começarmos",
"começou": "começamos",
"comenta": "comentamos",
"comentou": "comentamos",
"comer": "comermos",
"comete": "cometemos",
"cometeu": "cometemos",
"comeu": "comemos",
"comia": "comíamos",
"compara": "comparamos",
"comparou": "comparamos",
"completa": "completamos",
"completou": "completamos",
"compra": "compramos",
"compraria": "compraríamos",
"comprava": "comprávamos",
"compreende": "compreendemos",
"compreendeu": "compreendemos",
"comprou": "compramos",
"comunga": "comungamos",
"comungou": "comungamos",
"concorda": "concordamos",
"concordará": "concordaremos",
"concordou": "concordamos",
"concorria": "concorríamos",
"condena": "condenamos",
"condenou": "condenamos",
"confeccionaria": "confeccionaríamos",
"confiava": "confiávamos",
"confirmar": "confirmarmos",
"congelaria": "congelaríamos",
"conhece": "conhecemos",
"conheceu": "conhecemos",
"conhecia": "conhecíamos",
"conjetura": "conjeturamos",
"conjeturou": "conjeturamos",
"conquista": "conquistamos",
"conquistou": "conquistamos",
"consegue": "conseguimos",
"conseguia": "conseguíamos",
"conseguir": "conseguirmos",
"conseguirá": "conseguiremos",
"conseguiria": "conseguiríamos",
"conseguiu": "conseguimos",
"conservar": "conservarmos",
"conserve": "conservemos",
"considera": "consideramos",
"considerar": "considerarmos",
"considerava": "considerávamos",
"considerou": "consideramos",
"consiga": "consigamos",
"constata": "constatamos",
"constatou": "constatamos",
"constrói": "construímos",
"construiu": "construímos",
"conta": "contamos",
"continua": "continuamos",
"continuará": "continuaremos",
"continuava": "continuávamos",
"continuou": "continuamos",
"contou": "contamos",
"contratasse": "contratássemos",
"contratava": "contratávamos",
"controla": "controlamos",
"controlou": "controlamos",
"conversa": "conversamos",
"conversará": "conversaremos",
"conversava": "conversávamos",
"conversou": "conversamos",
"convida": "convidamos",
"convidou": "convidamos",
"coopera": "cooperamos",
"cooperou": "cooperamos",
"corre": "corremos",
"correrá": "correremos",
"correu": "corremos",
"costuma": "costumamos",
"costumava": "costumávamos",
"costumou": "costumamos",
"cota": "cotamos",
"cotou": "cotamos",
"crê": "cremos",
"credencia": "credenciamos",
"credenciou": "credenciamos",
"cresce": "crescemos",
"cresceu": "crescemos",
"crescia": "crescíamos",
"creu": "cremos",
"cria": "criamos",
"criar": "criarmos",
"criava": "criávamos",
"criou": "criamos",
"critica": "criticamos",
"criticou": "criticamos",
"cruza": "cruzamos",
"cuida": "cuidamos",
"cuidou": "cuidamos",
"culpa": "culpamos",
"culpou": "culpamos",
"cultiva": "cultivamos",
"cultivou": "cultivamos",
"cumpre": "cumprimos",
"cumpriu": "cumprimos",
"dá": "damos",
"dança": "dançamos",
"dançasse": "dançássemos",
"dançava": "dançávamos",
"dava": "dávamos",
"debitará": "debitaremos",
"decide": "decidimos",
"decidirá": "decidiremos",
"decidiu": "decidimos",
"defende": "defendemos",
"defendeu": "defendemos",
"defendia": "defendíamos",
"define": "definimos",
"definiu": "definimos",
"deita": "deitamos",
"deitou": "deitamos",
"deixa": "deixamos",
"deixaria": "deixaríamos",
"deixou": "deixamos",
"demonstra": "demonstramos",
"demonstrou": "demonstramos",
"denomina": "denominamos",
"denominou": "denominamos",
"denuncia": "denunciamos",
"denunciou": "denunciamos",
"deposita": "depositamos",
"depositou": "depositamos",
"derrubar": "derrubarmos",
"desce": "descemos",
"descende": "descendemos",
"desceu": "descemos",
"descobre": "descobrimos",
"descobrir": "descobrirmos",
"descobrirá": "descobriremos",
"descobriu": "descobrimos",
"desconfiava": "desconfiávamos",
"descoroa": "descoroamos",
"deseja": "desejamos",
"desejaria": "desejaríamos",
"desejava": "desejávamos",
"desejou": "desejamos",
"desempenha": "desempenhamos",
"desempenhou": "desempenhamos",
"desentranha": "desentranhamos",
"desentranhou": "desentranhamos",
"desenvolve": "desenvolvemos",
"desenvolveu": "desenvolvemos",
"desenvolvia": "desenvolvíamos",
"designa": "designamos",
"designou": "designamos",
"despachava": "despachávamos",
"despreza": "desprezamos",
"desprezou": "desprezamos",
"desrespeitar": "desrespeitarmos",
"desviava": "desviávamos",
"detesta": "detestamos",
"detestou": "detestamos",
"deu": "demos",
"deva": "devamos",
"deve": "devemos",
"deveria": "deveríamos",
"devia": "devíamos",
"diagnosticará": "diagnosticaremos",
"diga": "digamos",
"dirá": "diremos",
"diria": "diríamos",
"discutia": "discutíamos",
"dispõe": "dispomos",
"dispor": "dispormos",
"disputa": "disputamos",
"disputou": "disputamos",
"disse": "dissemos",
"dissesse": "disséssemos",
"divaga": "divagamos",
"divagou": "divagamos",
"diverge": "divergimos",
"divergiu": "divergimos",
"divide": "dividimos",
"dividia": "dividíamos",
"dividiu": "dividimos",
"diz": "dizemos",
"dizia": "dizíamos",
"dobra": "dobramos",
"dobrou": "dobramos",
"dormia": "dormíamos",
"dormirá": "dormiremos",
"dormiu": "dormimos",
"dubla": "dublamos",
"duplica": "duplicamos",
"duplicou": "duplicamos",
"é": "somos",
"elabora": "elaboramos",
"elaborou": "elaboramos",
"elege": "elegemos",
"elegeu": "elegemos",
"embarga": "embargamos",
"embargou": "embargamos",
"embirra": "embirramos",
"embirrou": "embirramos",
"emprega": "empregamos",
"empregou": "empregamos",
"encara": "encaramos",
"encarou": "encaramos",
"enche": "enchemos",
"encheu": "enchemos",
"encolhia": "encolhíamos",
"encontra": "encontramos",
"encontrar": "encontrarmos",
"encontrará": "encontraremos",
"encontrou": "encontramos",
"enjeita": "enjeitamos",
"entende": "entendemos",
"entendesse": "entendêssemos",
"entra": "entramos",
"entrar": "entrarmos",
"entrava": "entrávamos",
"entrega": "entregamos",
"entregou": "entregamos",
"entrevista": "entrevistamos",
"entrevistou": "entrevistamos",
"entrou": "entramos",
"envia": "enviamos",
"enviaria": "enviaríamos",
"enviou": "enviamos",
"enxuga": "enxugamos",
"enxugará": "enxugaremos",
"enxugou": "enxugamos",
"era":"éramos",
"ergue": "erguemos",
"ergueu": "erguemos",
"erra": "erramos",
"errou": "erramos",
"escapa": "escapamos",
"escapou": "escapamos",
"escolhe": "escolhemos",
"escolheu": "escolhemos",
"escolhia": "escolhíamos",
"escondia": "escondíamos",
"escorrega": "escorregamos",
"escorregou": "escorregamos",
"escrevia": "escrevíamos",
"escutava": "escutávamos",
"esgota": "esgotamos",
"esgotou": "esgotamos",
"especifica": "especificamos",
"especificou": "especificamos",
"espera": "esperamos",
"esperava": "esperávamos",
"esperou": "esperamos",
"esquece": "esquecemos",
"está": "estamos",
"estabelece": "estabelecemos",
"estabelecer": "estabelecermos",
"estabeleceu": "estabelecemos",
"estar": "estarmos",
"estará": "estaremos",
"estaria": "estaríamos",
"estava": "estávamos",
"esteja": "estejamos",
"estima": "estimamos",
"estimou": "estimamos",
"estive": "estivemos",
"estiver": "estivermos",
"estivesse": "estivéssemos",
"estornará": "estornaremos",
"estoura": "estouramos",
"estourou": "estouramos",
"estraga": "estragamos",
"estragou": "estragamos",
"estranha": "estranhamos",
"estranhasse": "estranhássemos",
"estranhe": "estranhemos",
"estranhou": "estranhamos",
"estudará": "estudaremos",
"estudava": "estudávamos",
"exerce": "exercemos",
"exerceu": "exercemos",
"exige": "exigimos",
"exigiu": "exigimos",
"existe": "existimos",
"existiu": "existimos",
"expõe": "expomos",
"exportava": "exportávamos",
"expressa": "expressamos",
"expressou": "expressamos",
"faça": "façamos",
"fala": "falamos",
"falar": "falarmos",
"falará": "falaremos",
"falava": "falávamos",
"falou": "falamos",
"falta": "faltamos",
"faltou": "faltamos",
"fará": "faremos",
"faria": "faríamos",
"faz": "fazemos",
"fazer": "fazermos",
"fazia": "fazíamos",
"fechar": "fecharmos",
"fechava": "fechávamos",
"fez": "fizemos",
"fica": "ficamos",
"ficar": "ficarmos",
"ficará": "ficaremos",
"ficasse": "ficássemos",
"ficava": "ficávamos",
"ficou": "ficamos",
"filma": "filmamos",
"filmou": "filmamos",
"finge": "fingimos",
"fingiu": "fingimos",
"fiscaliza": "fiscalizamos",
"fiscalizou": "fiscalizamos",
"fizer": "fizermos",
"fizesse": "fizéssemos",
"flartava": "flartávamos",
"foge": "fugimos",
"foi": "fomos",
"folgava": "folgávamos",
"for": "formos",
"forma": "formamos",
"formava": "formávamos",
"formou": "formamos",
"fornece": "fornecemos",
"forneceu": "fornecemos",
"fosse": "fossemos",
"frequentava": "frequentávamos",
"fugisse": "fugíssemos",
"fugiu": "fugimos",
"fumava": "fumávamos",
"funda": "fundamos",
"fundava": "fundávamos",
"fundou": "fundamos",
"furta": "furtamos",
"furtou": "furtamos",
"ganha": "ganhamos",
"ganhará": "ganharemos",
"ganhava": "ganhávamos",
"ganhou": "ganhamos",
"garantirá": "garantiremos",
"gasta": "gastamos",
"gastou": "gastamos",
"gosta": "gostamos",
"gostaria": "gostaríamos",
"gostava": "gostávamos",
"gostou": "gostamos",
"goza": "gozamos",
"gozou": "gozamos",
"gradua": "graduamos",
"graduou": "graduamos",
"guarda": "guardamos",
"guardou": "guardamos",
"há": "havemos",
"habita": "habitamos",
"habitou": "habitamos",
"havia": "havíamos",
"ia":"íamos",
"ignorava": "ignorávamos",
"imagina": "imaginamos",
"imaginava": "imaginávamos",
"imita": "imitamos",
"imitaria": "imitaríamos",
"imitou": "imitamos",
"implanta": "implantamos",
"implantou": "implantamos",
"impõe": "impomos",
"importa": "importamos",
"importou": "importamos",
"impôs": "impusemos",
"impregna": "impregnamos",
"impregnou": "impregnamos",
"indica": "indicamos",
"insistia": "insistíamos",
"insistiu": "insistimos",
"instala": "instalamos",
"instalou": "instalamos",
"interfere": "interferimos",
"interferiu": "interferimos",
"interpreta": "interpretamos",
"interpretou": "interpretamos",
"interviesse": "interviéssemos",
"invejava": "invejávamos",
"inventa": "inventamos",
"inventou": "inventamos",
"investiu": "investimos",
"invocará": "invocaremos",
"ir": "irmos",
"irá": "iremos",
"iria": "iríamos",
"janta": "jantamos",
"jantasse": "jantássemos",
"jantava": "jantávamos",
"jantou": "jantamos",
"joga": "jogamos",
"jogou": "jogamos",
"julga": "julgamos",
"julgou": "julgamos",
"jura": "juramos",
"jurou": "juramos",
"labora": "laboramos",
"lamenta": "lamentamos",
"lançar": "lançarmos",
"lava": "lavamos",
"lavar": "lavarmos",
"lavou": "lavamos",
"lê": "lemos",
"legalizaria": "legalizaríamos",
"lembra": "lembramos",
"lembrou": "lembramos",
"lesse": "lêssemos",
"leva": "levamos",
"levanta": "levantamos",
"levantou": "levantamos",
"levará": "levaremos",
"levava": "levávamos",
"levou": "levamos",
"lia": "líamos",
"liberte": "libertemos",
"ligava": "ligávamos",
"localiza": "localizamos",
"localizou": "localizamos",
"lustrava": "lustrávamos",
"luta": "lutamos",
"lutou": "lutamos",
"machuca": "machucamos",
"machucou": "machucamos",
"manda": "mandamos",
"mandou": "mandamos",
"mantém": "mantemos",
"marca": "marcamos",
"marcha": "marchamos",
"marchará": "marcharemos",
"marchou": "marchamos",
"marcou": "marcamos",
"mata": "matamos",
"matara": "matáramos",
"matava": "matávamos",
"matou": "matamos",
"mente": "mentimos",
"mentiu": "mentimos",
"merece": "merecemos",
"mereceu": "merecemos",
"merecia": "merecíamos",
"meter": "metermos",
"mistura": "misturamos",
"misturou": "misturamos",
"molda": "moldamos",
"moldou": "moldamos",
"montaria": "montaríamos",
"mora": "moramos",
"morará": "moraremos",
"morava": "morávamos",
"more": "moremos",
"morou": "moramos",
"morrer": "morrermos",
"morresse": "morrêssemos",
"mostra": "mostramos",
"mostrou": "mostramos",
"muda": "mudamos",
"mudou": "mudamos",
"nasce": "nascemos",
"nascer": "nascermos",
"nasceu": "nascemos",
"necessita": "necessitamos",
"necessitasse": "necessitássemos",
"necessitou": "necessitamos",
"nota": "notamos",
"notar": "notarmos",
"notou": "notamos",
"observa": "observamos",
"observar": "observarmos",
"observou": "observamos",
"obteve": "obtivemos",
"oferece": "oferecemos",
"ofereceu": "oferecemos",
"olha": "olhamos",
"olhar": "olharmos",
"olhava": "olhávamos",
"olhou": "olhamos",
"opta": "optamos",
"optou": "optamos",
"organizava": "organizávamos",
"ouve": "ouvimos",
"ouvia": "ouvíamos",
"ouviria": "ouviríamos",
"ouviu": "ouvimos",
"paga": "pagamos",
"pagará": "pagaremos",
"pagava": "pagávamos",
"pagou": "pagamos",
"parar": "pararmos",
"parava": "parávamos",
"parece": "parecemos",
"pareceu": "parecemos",
"parte": "partimos",
"participa": "participamos",
"participou": "participamos",
"partirá": "partiremos",
"partiu": "partimos",
"passa": "passamos",
"passar": "passarmos",
"passasse": "passássemos",
"passava": "passávamos",
"passear": "passearmos",
"passou": "passamos",
"pede": "pedimos",
"pediu": "pedimos",
"pega": "pegamos",
"pegar": "pegarmos",
"pegou": "pegamos",
"pensa": "pensamos",
"pensar": "pensarmos",
"pensava": "pensávamos",
"percebe": "percebemos",
"percebeu": "percebemos",
"percebia": "percebíamos",
"percorre": "percorremos",
"percorreu": "percorremos",
"perde": "perdemos",
"perder": "perdermos",
"perderá": "perderemos",
"perdeu": "perdemos",
"perdoa": "perdoamos",
"perdoou": "perdoamos",
"pergunta": "perguntamos",
"perguntou": "perguntamos",
"pertence": "pertencemos",
"pertenceu": "pertencemos",
"pode": "podemos",
"poderá": "poderemos",
"poderia": "poderíamos",
"podia": "podíamos",
"põe": "pomos",
"poria": "poríamos",
"possa": "possamos",
"possui": "possuímos",
"possuiu": "possuímos",
"poupa": "poupamos",
"pouparia": "pouparíamos",
"poupou": "poupamos",
"precisa": "precisamos",
"precisar": "precisarmos",
"precisaria": "precisaríamos",
"precisava": "precisávamos",
"predisse": "predissemos",
"prefere": "preferimos",
"preferia": "preferíamos",
"preferiu": "preferimos",
"prende": "prendemos",
"prepara": "preparamos",
"prepararia": "prepararíamos",
"preparava": "preparávamos",
"preparou": "preparamos",
"presencia": "presenciamos",
"presenciou": "presenciamos",
"presta": "prestamos",
"prestou": "prestamos",
"pretende": "pretendemos",
"pretendeu": "pretendemos",
"pretendia": "pretendíamos",
"processa": "processamos",
"processou": "processamos",
"procura": "procuramos",
"procurou": "procuramos",
"produz": "produzimos",
"produziu": "produzimos",
"profere": "proferimos",
"proferiu": "proferimos",
"projetava": "projetávamos",
"promete": "prometemos",
"propõe": "propomos",
"provará": "provaremos",
"providencia": "providenciamos",
"publicaria": "publicaríamos",
"puder": "pudermos",
"punha": "púnhamos",
"pus": "pusemos",
"puxa": "puxamos",
"puxou": "puxamos",
"quebrava": "quebrávamos",
"quer": "queremos",
"querer": "querermos",
"queria": "queríamos",
"quis": "quisemos",
"quiser": "quisermos",
"quisera": "quiséramos",
"quisesse": "quiséssemos",
"rebolava": "rebolávamos",
"receava": "receávamos",
"recebe": "recebemos",
"receberá": "receberemos",
"recebeu": "recebemos",
"recebia": "recebíamos",
"reconhece": "reconhecemos",
"reconheceu": "reconhecemos",
"recorde": "recordemos",
"reduziu": "reduzimos",
"registra": "registramos",
"registrou": "registramos",
"regrava": "regravamos",
"reivindica": "reivindicamos",
"reivindicou": "reivindicamos",
"rema": "remamos",
"reparar": "repararmos",
"repete": "repetimos",
"repetiu": "repetimos",
"representa": "representamos",
"representou": "representamos",
"requere": "requeremos",
"resiste": "resistimos",
"resistiu": "resistimos",
"resolve": "resolvemos",
"resolveria": "resolveríamos",
"resolveu": "resolvemos",
"resolvia": "resolvíamos",
"respeita": "respeitamos",
"respeitou": "respeitamos",
"respira": "respiramos",
"respirava": "respirávamos",
"respirou": "respiramos",
"responder": "respondermos",
"respondia": "respondíamos",
"responsabiliza": "responsabilizamos",
"responsabilizou": "responsabilizamos",
"retesou": "retesamos",
"retoma": "retomamos",
"retomou": "retomamos",
"retribuiu": "retribuímos",
"retroceder": "retrocedermos",
"reúne": "reunimos",
"reuniu": "reunimos",
"revivia": "revivíamos",
"reza": "rezamos",
"reze": "rezemos",
"rezou": "rezamos",
"roeu": "roemos",
"rói": "roemos",
"rola": "rolamos",
"rolava": "rolávamos",
"rolou": "rolamos",
"rouba": "roubamos",
"roubou": "roubamos",
"sabe": "sabemos",
"saber": "sabermos",
"saberá": "saberemos",
"saberia": "saberíamos",
"sabia": "sabíamos",
"saboreia": "saboreamos",
"sacrifica": "sacrificamos",
"sai": "saímos",
"saía": "saíamos",
"saiba": "saibamos",
"sair": "sairmos",
"sairá": "sairemos",
"saísse": "saíssemos",
"sanciona": "sancionamos",
"saudava": "saudávamos",
"segue": "seguimos",
"seguiu": "seguimos",
"segura": "seguramos",
"segurará": "seguraremos",
"segurou": "seguramos",
"seja": "sejamos",
"selecionará": "selecionaremos",
"sente": "sentimos",
"sentia": "sentíamos",
"sentir": "sentirmos",
"ser": "sermos",
"será": "seremos",
"seria": "seríamos",
"serve": "servimos",
"serviu": "servimos",
"sinta": "sintamos",
"sofre": "sofremos",
"sofreu": "sofremos",
"sonha": "sonhamos",
"sonhava": "sonhávamos",
"sonhou": "sonhamos",
"sorri": "sorrimos",
"sorria": "sorríamos",
"sorriu": "sorrimos",
"soube": "soubemos",
"souber": "soubermos",
"soubesse": "soubéssemos",
"supera": "superamos",
"superou": "superamos",
"suplica": "suplicamos",
"supomos": "supomos",
"suprirá": "supriremos",
"surpreende": "surpreendemos",
"suspeitava": "suspeitávamos",
"suspira": "suspiramos",
"suspirava": "suspirávamos",
"sustenta": "sustentamos",
"ta": "tamos",
"teimava": "teimávamos",
"tem": "temos",
"teme": "tememos",
"temia": "temíamos",
"tenciona": "tencionamos",
"tenha": "tenhamos",
"tenta": "tentamos",
"tentaria": "tentaríamos",
"tentasse": "tentássemos",
"tentou": "tentamos",
"ter": "termos",
"terá": "teremos",
"teria": "teríamos",
"termina": "terminamos",
"terminava": "terminávamos",
"testar": "testarmos",
"tinha": "tínhamos",
"tira": "tiramos",
"tirava": "tirávamos",
"tire": "tiremos",
"tive": "tivemos",
"tiver": "tivermos",
"tivesse": "tivéssemos",
"toca": "tocamos",
"tocou": "tocamos",
"toma": "tomamos",
"tomar": "tomarmos",
"tomará": "tomaremos",
"tomava": "tomávamos",
"tomou": "tomamos",
"topa": "topamos",
"torna": "tornamos",
"trabalha": "trabalhamos",
"trabalhava": "trabalhávamos",
"trabalhou": "trabalhamos",
"traja": "trajamos",
"transpirava": "transpirávamos",
"trata": "tratamos",
"tratará": "trataremos",
"tratava": "tratávamos",
"traz": "trazemos",
"trazia": "trazíamos",
"treina": "treinamos",
"tremia": "tremíamos",
"trepasse": "trepássemos",
"trilha": "trilhamos",
"trouxe": "trouxemos",
"usa": "usamos",
"usava": "usávamos",
"usou": "usamos",
"usufruía": "usufruíamos",
"utiliza": "utilizamos",
"utilizasse": "utilizássemos",
"utilizou": "utilizamos",
"vai": "vamos",
"vale": "valemos",
"valorizava": "valorizávamos",
"vaticina": "vaticinamos",
"vê": "vemos",
"veio": "viemos",
"vem": "vimos",
"vence": "vencemos",
"venceu": "vencemos",
"vende": "vendemos",
"vendeu": "vendemos",
"ver": "vermos",
"verá": "veremos",
"verifica": "verificamos",
"verificou": "verificamos",
"vestia": "vestíamos",
"via": "víamos",
"viaja": "viajamos",
"viajou": "viajamos",
"vinha": "vínhamos",
"vir": "virmos",
"vira": "viramos",
"vire": "viremos",
"visite": "visitemos",
"visse": "víssemos",
"visualiza": "visualizamos",
"visualizar": "visualizarmos",
"vive": "vivemos",
"vivencia": "vivenciamos",
"viver": "vivermos",
"viverá": "viveremos",
"vivesse": "vivêssemos",
"vivia": "vivíamos",
"voa": "voamos",
"volta": "voltamos",
"voltar": "voltarmos",
"voltará": "voltaremos",
"voltaria": "voltaríamos",
"volte": "voltemos",
"voltou": "voltamos",
"vota": "votamos",
"votou": "votamos"}


# concNos - concordancia com o pronome Nos (com acento) - Nos vai
concNos = set(["abandona", "abandonou", "abdica", "abdicasse",
"abdicou", "abraça", "abraçou", "abre",
"abriu", "acaba", "acabou", "aceita",
"aceitaria", "aceitou", "acha", "achava",
"achou", "acolhe", "acompanha", "acompanhou",
"acorda", "acordou", "acredita", "acreditava",
"acreditou", "acrescenta", "acrescentou", "adivinha",
"adivinhou", "admira", "admirou", "admitir",
"adora", "adorou", "adota", "adote",
"adotou", "adquire", "adquiriu", "adulterar",
"adverte", "advertiu", "advoga", "advogou",
"age", "agiu", "agradece", "agradeceu",
"ajuda", "ajudará", "ajudou", "ajunta",
"ajuntou", "ajusta", "ajustou", "alcançava",
"alimenta", "alimentou", "aliviava", "almoça",
"almoçou", "alugasse", "ama", "amarra",
"amarrou", "amava", "amou", "analisa",
"analisasse", "analisou", "anda", "andava",
"andou", "anuncia", "apanha", "apanhou",
"apareça", "aparece", "apareceu", "aplaude",
"aplaudiu", "aplicaria", "apodrece", "apodreceu",
"apoia", "apoiou", "aprende", "aprendeu",
"apresenta", "apresentará", "apresentou", "apresta",
"aprestou", "aprova", "aprovasse", "aproveita",
"aprovou", "arranjará", "arranjasse", "arrasta",
"arrastou", "arromba", "arrombou", "aspira",
"aspirou", "assina", "assinar", "assinou",
"assiste", "assistir", "assistiu", "associa",
"assume", "assumiu", "ataca", "atacou",
"atiraria", "atravessa", "atravessava", "atravessou",
"aumentar", "avalia", "avaliar", "avaliou",
"avança", "avançou", "bate", "bateu",
"batia", "batiza", "batizou", "bebe",
"beneficia", "beneficiou", "bota", "botou",
"brada", "bradou", "briga", "brigou",
"brinca", "brincava", "brincou", "busca",
"buscou", "cabe", "caçoa", "caçoou",
"cai", "cairia", "caísse", "caiu",
"cala", "calcula", "calculou", "calou",
"caminha", "caminhou", "cantará", "carece",
"careceu", "carrega", "carregava", "carregou",
"casa", "casou", "cavalga", "cavalgou",
"celebra", "celebrou", "chama", "chamava",
"chamou", "checa", "checou", "chega",
"chegar", "chegará", "chegava", "chegou",
"chegue", "chora", "chorava", "chore",
"chorou", "cita", "citou", "cobra",
"cobrou", "colabora", "colaborou", "coloca",
"colocou", "combaterá", "combina", "combinava",
"combinou", "come", "começa", "começar",
"começou", "comenta", "comentou", "comer",
"comete", "cometeu", "comeu", "comia",
"compara", "comparou", "completa", "completou",
"compra", "compraria", "comprava", "compreende",
"compreendeu", "comprou", "comunga", "comungou",
"concorda", "concordará", "concordou", "concorria",
"condena", "condenou", "confeccionaria", "confiava",
"confirmar", "congelaria", "conhece", "conheceu",
"conhecia", "conjetura", "conjeturou", "conquista",
"conquistou", "consegue", "conseguia", "conseguir",
"conseguirá", "conseguiria", "conseguiu", "conservar",
"conserve", "considera", "considerar", "considerava",
"considerou", "consiga", "constata", "constatou",
"constrói", "construiu", "conta", "continua",
"continuará", "continuava", "continuou", "contou",
"contratasse", "contratava", "controla", "controlou",
"conversa", "conversará", "conversava", "conversou",
"convida", "convidou", "coopera", "cooperou",
"corre", "correrá", "correu", "costuma",
"costumava", "costumou", "cota", "cotou",
"crê", "credencia", "credenciou", "cresce",
"cresceu", "crescia", "creu", "cria",
"criar", "criava", "criou", "critica",
"criticou", "cruza", "cuida", "cuidou",
"culpa", "culpou", "cultiva", "cultivou",
"cumpre", "cumpriu", "dá", "dança",
"dançasse", "dançava", "dava", "debitará",
"decide", "decidirá", "decidiu", "defende",
"defendeu", "defendia", "define", "definiu",
"deita", "deitou", "deixa", "deixaria",
"deixou", "demonstra", "demonstrou", "denomina",
"denominou", "denuncia", "denunciou", "deposita",
"depositou", "derrubar", "desce", "descende",
"desceu", "descobre", "descobrir", "descobrirá",
"descobriu", "desconfiava", "descoroa", "deseja",
"desejaria", "desejava", "desejou", "desempenha",
"desempenhou", "desentranha", "desentranhou", "desenvolve",
"desenvolveu", "desenvolvia", "designa", "designou",
"despachava", "despreza", "desprezou", "desrespeitar",
"desviava", "detesta", "detestou", "deu",
"deva", "deve", "deveria", "devia",
"diagnosticará", "diga", "dirá", "diria",
"discutia", "dispõe", "dispor", "disputa",
"disputou", "disse", "dissesse", "divaga",
"divagou", "diverge", "divergiu", "divide",
"dividia", "dividiu", "diz", "dizia",
"dobra", "dobrou", "dormia", "dormirá",
"dormiu", "dubla", "duplica", "duplicou",
"é", "elabora", "elaborou", "elege",
"elegeu", "embarga", "embargou", "embirra",
"embirrou", "emprega", "empregou", "encara",
"encarou", "enche", "encheu", "encolhia",
"encontra", "encontrar", "encontrará", "encontrou",
"enjeita", "entende", "entendesse", "entra",
"entrar", "entrava", "entrega", "entregou",
"entrevista", "entrevistou", "entrou", "envia",
"enviaria", "enviou", "enxuga", "enxugará",
"enxugou", "era", "ergue", "ergueu",
"erra", "errou", "escapa", "escapou",
"escolhe", "escolheu", "escolhia", "escondia",
"escorrega", "escorregou", "escrevia", "escutava",
"esgota", "esgotou", "especifica", "especificou",
"espera", "esperava", "esperou", "esquece",
"está", "estabelece", "estabelecer", "estabeleceu",
"estar", "estará", "estaria", "estava",
"esteja", "estima", "estimou", "estive",
"estiver", "estivesse", "estornará", "estoura",
"estourou", "estraga", "estragou", "estranha",
"estranhasse", "estranhe", "estranhou", "estudará",
"estudava", "exerce", "exerceu", "exige",
"exigiu", "existe", "existiu", "expõe",
"exportava", "expressa", "expressou", "faça",
"fala", "falar", "falará", "falava",
"falou", "falta", "faltou", "fará",
"faria", "faz", "fazer", "fazia",
"fechar", "fechava", "fez", "fica",
"ficar", "ficará", "ficasse", "ficava",
"ficou", "filma", "filmou", "finge",
"fingiu", "fiscaliza", "fiscalizou", "fizer",
"fizesse", "flartava", "foge", "foi",
"folgava", "for", "forma", "formava",
"formou", "fornece", "forneceu", "fosse",
"frequentava", "fugisse", "fugiu", "fumava",
"funda", "fundava", "fundou", "furta",
"furtou", "ganha", "ganhará", "ganhava",
"ganhou", "garantirá", "gasta", "gastou",
"gosta", "gostaria", "gostava", "gostou",
"goza", "gozou", "gradua", "graduou",
"guarda", "guardou", "há", "habita",
"habitou", "havia", "ia", "ignorava",
"imagina", "imaginava", "imita", "imitaria",
"imitou", "implanta", "implantou", "impõe",
"importa", "importou", "impôs", "impregna",
"impregnou", "indica", "insistia", "insistiu",
"instala", "instalou", "interfere", "interferiu",
"interpreta", "interpretou", "interviesse", "invejava",
"inventa", "inventou", "investiu", "invocará",
"ir", "irá", "iria", "janta",
"jantasse", "jantava", "jantou", "joga",
"jogou", "julga", "julgou", "jura",
"jurou", "labora", "lamenta", "lançar",
"lava", "lavar", "lavou", "lê",
"legalizaria", "lembra", "lembrou", "lesse",
"leva", "levanta", "levantou", "levará",
"levava", "levou", "lia", "liberte",
"ligava", "localiza", "localizou", "lustrava",
"luta", "lutou", "machuca", "machucou",
"manda", "mandou", "mantém", "marca",
"marcha", "marchará", "marchou", "marcou",
"mata", "matara", "matava", "matou",
"mente", "mentiu", "merece", "mereceu",
"merecia", "meter", "mistura", "misturou",
"molda", "moldou", "montaria", "mora",
"morará", "morava", "more", "morou",
"morrer", "morresse", "mostra", "mostrou",
"muda", "mudou", "nasce", "nascer",
"nasceu", "necessita", "necessitasse", "necessitou",
"nota", "notar", "notou", "observa",
"observar", "observou", "obteve", "oferece",
"ofereceu", "olha", "olhar", "olhava",
"olhou", "opta", "optou", "organizava",
"ouve", "ouvia", "ouviria", "ouviu",
"paga", "pagará", "pagava", "pagou",
"parar", "parava", "parece", "pareceu",
"parte", "participa", "participou", "partirá",
"partiu", "passa", "passar", "passasse",
"passava", "passear", "passou", "pede",
"pediu", "pega", "pegar", "pegou",
"pensa", "pensar", "pensava", "percebe",
"percebeu", "percebia", "percorre", "percorreu",
"perde", "perder", "perderá", "perdeu",
"perdoa", "perdoou", "pergunta", "perguntou",
"pertence", "pertenceu", "pode", "poderá",
"poderia", "podia", "põe", "poria",
"possa", "possui", "possuiu", "poupa",
"pouparia", "poupou", "precisa", "precisar",
"precisaria", "precisava", "predisse", "prefere",
"preferia", "preferiu", "prende", "prepara",
"prepararia", "preparava", "preparou", "presencia",
"presenciou", "presta", "prestou", "pretende",
"pretendeu", "pretendia", "processa", "processou",
"procura", "procurou", "produz", "produziu",
"profere", "proferiu", "projetava", "promete",
"propõe", "provará", "providencia", "publicaria",
"puder", "punha", "pus", "puxa",
"puxou", "quebrava", "quer", "querer",
"queria", "quis", "quiser", "quisera",
"quisesse", "rebolava", "receava", "recebe",
"receberá", "recebeu", "recebia", "reconhece",
"reconheceu", "recorde", "reduziu", "registra",
"registrou", "regrava", "reivindica", "reivindicou",
"rema", "reparar", "repete", "repetiu",
"representa", "representou", "requere", "resiste",
"resistiu", "resolve", "resolveria", "resolveu",
"resolvia", "respeita", "respeitou", "respira",
"respirava", "respirou", "responder", "respondia",
"responsabiliza", "responsabilizou", "retesou", "retoma",
"retomou", "retribuiu", "retroceder", "reúne",
"reuniu", "revivia", "reza", "reze",
"rezou", "roeu", "rói", "rola",
"rolava", "rolou", "rouba", "roubou",
"sabe", "saber", "saberá", "saberia",
"sabia", "saboreia", "sacrifica", "sai",
"saía", "saiba", "sair", "sairá",
"saísse", "sanciona", "saudava", "segue",
"seguiu", "segura", "segurará", "segurou",
"seja", "selecionará", "sente", "sentia",
"sentir", "ser", "será", "seria",
"serve", "serviu", "sinta", "sofre",
"sofreu", "sonha", "sonhava", "sonhou",
"sorri", "sorria", "sorriu", "soube",
"souber", "soubesse", "supera",
"superou", "suplica", "supomos", "suprirá",
"surpreende", "suspeitava", "suspira", "suspirava",
"sustenta", "ta", "teimava", "tem",
"teme", "temia", "tenciona", "tenha",
"tenta", "tentaria", "tentasse", "tentou",
"ter", "terá", "teria", "termina",
"terminava", "testar", "tinha", "tira",
"tirava", "tire", "tive", "tiver",
"tivesse", "toca", "tocou", "toma",
"tomar", "tomará", "tomava", "tomou",
"topa", "torna", "trabalha", "trabalhava",
"trabalhou", "traja", "transpirava", "trata",
"tratará", "tratava", "traz", "trazia",
"treina", "tremia", "trepasse", "trilha",
"trouxe", "usa", "usava", "usou",
"usufruía", "utiliza", "utilizasse", "utilizou",
"vai", "vale", "valorizava", "vaticina",
"vê", "veio", "vem", "vence",
"venceu", "vende", "vendeu", "ver",
"verá", "verifica", "verificou", "vestia",
"via", "viaja", "viajou", "vinha",
"vir", "vira", "vire", "visite",
"visse", "visualiza", "visualizar", "vive",
"vivencia", "viver", "viverá", "vivesse",
"vivia", "voa", "volta", "voltar",
"voltará", "voltaria", "volte", "voltou",
"vota", "votou"])


# aacpEle = Concordancia  com o pronome Ele
aacpEle = {"abandonamos": "abandonou",
"abdicássemos": "abdicasse",
"abdicamos": "abdicou",
"abraçamos": "abraçou",
"abrimos": "abriu",
"acabamos": "acabou",
"aceitamos": "aceitou",
"aceitaríamos": "aceitaria",
"achávamos": "achava",
"achamos": "achou",
"acolhemos": "acolhe",
"acompanhamos": "acompanhou",
"acordamos": "acordou",
"acreditávamos": "acreditava",
"acreditamos": "acreditou",
"acrescentamos": "acrescentou",
"adivinhamos": "adivinhou",
"admiramos": "admirou",
"admitirmos": "admitir",
"adoramos": "adorou",
"adotamos": "adotou",
"adotemos": "adote",
"adquirimos": "adquiriu",
"adulterarmos": "adulterar",
"advertimos": "advertiu",
"advogamos": "advogou",
"agimos": "agiu",
"agradecemos": "agradeceu",
"ajudamos": "ajudou",
"ajudaremos": "ajudará",
"ajuntamos": "ajuntou",
"ajustamos": "ajustou",
"alcançávamos": "alcançava",
"alimentamos": "alimentou",
"aliviávamos": "aliviava",
"almoçamos": "almoçou",
"alugássemos": "alugasse",
"amávamos": "amava",
"amamos": "amou",
"amarramos": "amarrou",
"analisássemos": "analisasse",
"analisamos": "analisou",
"andávamos": "andava",
"andamos": "andou",
"anunciamos": "anuncia",
"apanhamos": "apanhou",
"apareçamos": "apareça",
"aparecemos": "apareceu",
"aplaudimos": "aplaudiu",
"aplicaríamos": "aplicaria",
"apodrecemos": "apodreceu",
"apoiamos": "apoiou",
"aprendemos": "aprendeu",
"apresentamos": "apresentou",
"apresentaremos": "apresentará",
"aprestamos": "aprestou",
"aprovássemos": "aprovasse",
"aprovamos": "aprovou",
"aproveitamos": "aproveita",
"arranjássemos": "arranjasse",
"arranjaremos": "arranjará",
"arrastamos": "arrastou",
"arrombamos": "arrombou",
"aspiramos": "aspirou",
"assinamos": "assinou",
"assinarmos": "assinar",
"assistimos": "assistiu",
"assistirmos": "assistir",
"associamos": "associa",
"assumimos": "assumiu",
"atacamos": "atacou",
"atiraríamos": "atiraria",
"atravessávamos": "atravessava",
"atravessamos": "atravessou",
"aumentarmos": "aumentar",
"avaliamos": "avaliou",
"avaliarmos": "avaliar",
"avançamos": "avançou",
"batíamos": "batia",
"batemos": "bateu",
"batizamos": "batizou",
"bebemos": "bebe",
"beneficiamos": "beneficiou",
"botamos": "botou",
"bradamos": "bradou",
"brigamos": "brigou",
"brincávamos": "brincava",
"brincamos": "brincou",
"buscamos": "buscou",
"caçoamos": "caçoou",
"caímos": "caiu",
"caíssemos": "caísse",
"cabemos": "cabe",
"cairíamos": "cairia",
"calamos": "calou",
"calculamos": "calculou",
"caminhamos": "caminhou",
"cantaremos": "cantará",
"carecemos": "careceu",
"carregávamos": "carregava",
"carregamos": "carregou",
"casamos": "casou",
"cavalgamos": "cavalgou",
"celebramos": "celebrou",
"chamávamos": "chamava",
"chamamos": "chamou",
"checamos": "checou",
"chegávamos": "chegava",
"chegamos": "chegou",
"chegaremos": "chegará",
"chegarmos": "chegar",
"cheguemos": "chegue",
"chorávamos": "chorava",
"choramos": "chorou",
"choremos": "chore",
"citamos": "citou",
"cobramos": "cobrou",
"colaboramos": "colaborou",
"colocamos": "colocou",
"comíamos": "comia",
"combateremos": "combaterá",
"combinávamos": "combinava",
"combinamos": "combinou",
"começamos": "começou",
"começarmos": "começar",
"comemos": "comeu",
"comentamos": "comentou",
"comermos": "comer",
"cometemos": "cometeu",
"comparamos": "comparou",
"completamos": "completou",
"comprávamos": "comprava",
"compramos": "comprou",
"compraríamos": "compraria",
"compreendemos": "compreendeu",
"comungamos": "comungou",
"concordamos": "concordou",
"concordaremos": "concordará",
"concorríamos": "concorria",
"condenamos": "condenou",
"confeccionaríamos": "confeccionaria",
"confiávamos": "confiava",
"confirmarmos": "confirmar",
"congelaríamos": "congelaria",
"conhecíamos": "conhecia",
"conhecemos": "conheceu",
"conjeturamos": "conjeturou",
"conquistamos": "conquistou",
"conseguíamos": "conseguia",
"conseguimos": "conseguiu",
"conseguiríamos": "conseguiria",
"conseguiremos": "conseguirá",
"conseguirmos": "conseguir",
"conservarmos": "conservar",
"conservemos": "conserve",
"considerávamos": "considerava",
"consideramos": "considerou",
"considerarmos": "considerar",
"consigamos": "consiga",
"constatamos": "constatou",
"construímos": "construiu",
"contamos": "contou",
"continuávamos": "continuava",
"continuamos": "continuou",
"continuaremos": "continuará",
"contratássemos": "contratasse",
"contratávamos": "contratava",
"controlamos": "controlou",
"conversávamos": "conversava",
"conversamos": "conversou",
"conversaremos": "conversará",
"convidamos": "convidou",
"cooperamos": "cooperou",
"corremos": "correu",
"correremos": "correrá",
"costumávamos": "costumava",
"costumamos": "costumou",
"cotamos": "cotou",
"credenciamos": "credenciou",
"cremos": "creu",
"crescíamos": "crescia",
"crescemos": "cresceu",
"criávamos": "criava",
"criamos": "criou",
"criarmos": "criar",
"criticamos": "criticou",
"cruzamos": "cruza",
"cuidamos": "cuidou",
"culpamos": "culpou",
"cultivamos": "cultivou",
"cumprimos": "cumpriu",
"dávamos": "dava",
"damos": "dá",
"dançamos": "dança",
"dançássemos": "dançasse",
"dançávamos": "dançava",
"debitaremos": "debitará",
"decidimos": "decidiu",
"decidiremos": "decidirá",
"defendíamos": "defendia",
"defendemos": "defendeu",
"definimos": "definiu",
"deitamos": "deitou",
"deixamos": "deixou",
"deixaríamos": "deixaria",
"demonstramos": "demonstrou",
"demos": "deu",
"denominamos": "denominou",
"denunciamos": "denunciou",
"depositamos": "depositou",
"derrubarmos": "derrubar",
"descemos": "desceu",
"descendemos": "descende",
"descobrimos": "descobriu",
"descobriremos": "descobrirá",
"descobrirmos": "descobrir",
"desconfiávamos": "desconfiava",
"descoroamos": "descoroa",
"desejávamos": "desejava",
"desejamos": "desejou",
"desejaríamos": "desejaria",
"desempenhamos": "desempenhou",
"desentranhamos": "desentranhou",
"desenvolvíamos": "desenvolvia",
"desenvolvemos": "desenvolveu",
"designamos": "designou",
"despachávamos": "despachava",
"desprezamos": "desprezou",
"desrespeitarmos": "desrespeitar",
"desviávamos": "desviava",
"detestamos": "detestou",
"devíamos": "devia",
"devamos": "deva",
"devemos": "deve",
"deveríamos": "deveria",
"diagnosticaremos": "diagnosticará",
"digamos": "diga",
"diríamos": "diria",
"diremos": "dirá",
"discutíamos": "discutia",
"dispomos": "dispõe",
"dispormos": "dispor",
"disputamos": "disputou",
"disséssemos": "dissesse",
"dissemos": "disse",
"divagamos": "divagou",
"divergimos": "divergiu",
"dividíamos": "dividia",
"dividimos": "dividiu",
"dizíamos": "dizia",
"dizemos": "diz",
"dobramos": "dobrou",
"dormíamos": "dormia",
"dormimos": "dormiu",
"dormiremos": "dormirá",
"dublamos": "dubla",
"duplicamos": "duplicou",
"elaboramos": "elaborou",
"elegemos": "elegeu",
"embargamos": "embargou",
"embirramos": "embirrou",
"empregamos": "empregou",
"encaramos": "encarou",
"enchemos": "encheu",
"encolhíamos": "encolhia",
"encontramos": "encontrou",
"encontraremos": "encontrará",
"encontrarmos": "encontrar",
"enjeitamos": "enjeita",
"entendêssemos": "entendesse",
"entendemos": "entende",
"entrávamos": "entrava",
"entramos": "entrou",
"entrarmos": "entrar",
"entregamos": "entregou",
"entrevistamos": "entrevistou",
"enviamos": "enviou",
"enviaríamos": "enviaria",
"enxugamos": "enxugou",
"enxugaremos": "enxugará",
"éramos": "era",
"erguemos": "ergueu",
"erramos": "errou",
"escapamos": "escapou",
"escolhíamos": "escolhia",
"escolhemos": "escolheu",
"escondíamos": "escondia",
"escorregamos": "escorregou",
"escrevíamos": "escrevia",
"escutávamos": "escutava",
"esgotamos": "esgotou",
"especificamos": "especificou",
"esperávamos": "esperava",
"esperamos": "esperou",
"esquecemos": "esquece",
"estávamos": "estava",
"estabelecemos": "estabeleceu",
"estabelecermos": "estabelecer",
"estamos": "está",
"estaríamos": "estaria",
"estaremos": "estará",
"estarmos": "estar",
"estejamos": "esteja",
"estimamos": "estimou",
"estivéssemos": "estivesse",
"estivemos": "estive",
"estivermos": "estiver",
"estornaremos": "estornará",
"estouramos": "estourou",
"estragamos": "estragou",
"estranhássemos": "estranhasse",
"estranhamos": "estranhou",
"estranhemos": "estranhe",
"estudávamos": "estudava",
"estudaremos": "estudará",
"exercemos": "exerceu",
"exigimos": "exigiu",
"existimos": "existiu",
"expomos": "expõe",
"exportávamos": "exportava",
"expressamos": "expressou",
"façamos": "faça",
"falávamos": "falava",
"falamos": "falou",
"falaremos": "falará",
"falarmos": "falar",
"faltamos": "faltou",
"faríamos": "faria",
"faremos": "fará",
"fazíamos": "fazia",
"fazemos": "faz",
"fazermos": "fazer",
"fechávamos": "fechava",
"fecharmos": "fechar",
"ficássemos": "ficasse",
"ficávamos": "ficava",
"ficamos": "ficou",
"ficaremos": "ficará",
"ficarmos": "ficar",
"filmamos": "filmou",
"fingimos": "fingiu",
"fiscalizamos": "fiscalizou",
"fizéssemos": "fizesse",
"fizemos": "fez",
"fizermos": "fizer",
"flartávamos": "flartava",
"folgávamos": "folgava",
"fomos": "foi",
"formávamos": "formava",
"formamos": "formou",
"formos": "for",
"fornecemos": "forneceu",
"fossemos": "fosse",
"frequentávamos": "frequentava",
"fugíssemos": "fugisse",
"fugimos": "fugiu",
"fumávamos": "fumava",
"fundávamos": "fundava",
"fundamos": "fundou",
"furtamos": "furtou",
"ganhávamos": "ganhava",
"ganhamos": "ganhou",
"ganharemos": "ganhará",
"garantiremos": "garantirá",
"gastamos": "gastou",
"gostávamos": "gostava",
"gostamos": "gostou",
"gostaríamos": "gostaria",
"gozamos": "gozou",
"graduamos": "graduou",
"guardamos": "guardou",
"habitamos": "habitou",
"havíamos": "havia",
"havemos": "há",
"íamos": "ia",
"ignorávamos": "ignorava",
"imaginávamos": "imaginava",
"imaginamos": "imagina",
"imitamos": "imitou",
"imitaríamos": "imitaria",
"implantamos": "implantou",
"impomos": "impõe",
"importamos": "importou",
"impregnamos": "impregnou",
"impusemos": "impôs",
"indicamos": "indica",
"insistíamos": "insistia",
"insistimos": "insistiu",
"instalamos": "instalou",
"interferimos": "interferiu",
"interpretamos": "interpretou",
"interviéssemos": "interviesse",
"invejávamos": "invejava",
"inventamos": "inventou",
"investimos": "investiu",
"invocaremos": "invocará",
"iríamos": "iria",
"iremos": "irá",
"irmos": "ir",
"jantássemos": "jantasse",
"jantávamos": "jantava",
"jantamos": "jantou",
"jogamos": "jogou",
"julgamos": "julgou",
"juramos": "jurou",
"lêssemos": "lesse",
"líamos": "lia",
"laboramos": "labora",
"lamentamos": "lamenta",
"lançarmos": "lançar",
"lavamos": "lavou",
"lavarmos": "lavar",
"legalizaríamos": "legalizaria",
"lembramos": "lembrou",
"lemos": "lê",
"levávamos": "levava",
"levamos": "levou",
"levantamos": "levantou",
"levaremos": "levará",
"libertemos": "liberte",
"ligávamos": "ligava",
"localizamos": "localizou",
"lustrávamos": "lustrava",
"lutamos": "lutou",
"machucamos": "machucou",
"mandamos": "mandou",
"mantemos": "mantém",
"marcamos": "marcou",
"marchamos": "marchou",
"marcharemos": "marchará",
"matáramos": "matara",
"matávamos": "matava",
"matamos": "matou",
"mentimos": "mentiu",
"merecíamos": "merecia",
"merecemos": "mereceu",
"metermos": "meter",
"misturamos": "misturou",
"moldamos": "moldou",
"montaríamos": "montaria",
"morávamos": "morava",
"moramos": "morou",
"moraremos": "morará",
"moremos": "more",
"morrêssemos": "morresse",
"morrermos": "morrer",
"mostramos": "mostrou",
"mudamos": "mudou",
"nascemos": "nasceu",
"nascermos": "nascer",
"necessitássemos": "necessitasse",
"necessitamos": "necessitou",
"notamos": "notou",
"notarmos": "notar",
"observamos": "observou",
"observarmos": "observar",
"obtivemos": "obteve",
"oferecemos": "ofereceu",
"olhávamos": "olhava",
"olhamos": "olhou",
"olharmos": "olhar",
"optamos": "optou",
"organizávamos": "organizava",
"ouvíamos": "ouvia",
"ouvimos": "ouviu",
"ouviríamos": "ouviria",
"púnhamos": "punha",
"pagávamos": "pagava",
"pagamos": "pagou",
"pagaremos": "pagará",
"parávamos": "parava",
"pararmos": "parar",
"parecemos": "pareceu",
"participamos": "participou",
"partimos": "partiu",
"partiremos": "partirá",
"passássemos": "passasse",
"passávamos": "passava",
"passamos": "passou",
"passarmos": "passar",
"passearmos": "passear",
"pedimos": "pediu",
"pegamos": "pegou",
"pegarmos": "pegar",
"pensávamos": "pensava",
"pensamos": "pensa",
"pensarmos": "pensar",
"percebíamos": "percebia",
"percebemos": "percebeu",
"percorremos": "percorreu",
"perdemos": "perdeu",
"perderemos": "perderá",
"perdermos": "perder",
"perdoamos": "perdoou",
"perguntamos": "perguntou",
"pertencemos": "pertenceu",
"podíamos": "podia",
"podemos": "pode",
"poderíamos": "poderia",
"poderemos": "poderá",
"pomos": "põe",
"poríamos": "poria",
"possamos": "possa",
"possuímos": "possuiu",
"poupamos": "poupou",
"pouparíamos": "pouparia",
"precisávamos": "precisava",
"precisamos": "precisa",
"precisaríamos": "precisaria",
"precisarmos": "precisar",
"predissemos": "predisse",
"preferíamos": "preferia",
"preferimos": "preferiu",
"prendemos": "prende",
"preparávamos": "preparava",
"preparamos": "preparou",
"prepararíamos": "prepararia",
"presenciamos": "presenciou",
"prestamos": "prestou",
"pretendíamos": "pretendia",
"pretendemos": "pretendeu",
"processamos": "processou",
"procuramos": "procurou",
"produzimos": "produziu",
"proferimos": "proferiu",
"projetávamos": "projetava",
"prometemos": "promete",
"propomos": "propõe",
"provaremos": "provará",
"providenciamos": "providencia",
"publicaríamos": "publicaria",
"pudermos": "puder",
"pusemos": "pus",
"puxamos": "puxou",
"quebrávamos": "quebrava",
"queríamos": "queria",
"queremos": "quer",
"querermos": "querer",
"quiséramos": "quisera",
"quiséssemos": "quisesse",
"quisemos": "quis",
"quisermos": "quiser",
"rebolávamos": "rebolava",
"receávamos": "receava",
"recebíamos": "recebia",
"recebemos": "recebeu",
"receberemos": "receberá",
"reconhecemos": "reconheceu",
"recordemos": "recorde",
"reduzimos": "reduziu",
"registramos": "registrou",
"regravamos": "regrava",
"reivindicamos": "reivindicou",
"remamos": "rema",
"repararmos": "reparar",
"repetimos": "repetiu",
"representamos": "representou",
"requeremos": "requere",
"resistimos": "resistiu",
"resolvíamos": "resolvia",
"resolvemos": "resolveu",
"resolveríamos": "resolveria",
"respeitamos": "respeitou",
"respirávamos": "respirava",
"respiramos": "respirou",
"respondíamos": "respondia",
"respondermos": "responder",
"responsabilizamos": "responsabilizou",
"retesamos": "retesou",
"retomamos": "retomou",
"retribuímos": "retribuiu",
"retrocedermos": "retroceder",
"reunimos": "reuniu",
"revivíamos": "revivia",
"rezamos": "rezou",
"rezemos": "reze",
"roemos": "rói",
"rolávamos": "rolava",
"rolamos": "rolou",
"roubamos": "roubou",
"saíamos": "saía",
"saímos": "saiu",
"saíssemos": "saísse",
"sabíamos": "sabia",
"sabemos": "sabe",
"saberíamos": "saberia",
"saberemos": "saberá",
"sabermos": "saber",
"saboreamos": "saboreia",
"sacrificamos": "sacrifica",
"saibamos": "saiba",
"sairemos": "sairá",
"sairmos": "sair",
"sancionamos": "sanciona",
"saudávamos": "saudava",
"seguimos": "seguiu",
"seguramos": "segurou",
"seguraremos": "segurará",
"sejamos": "seja",
"selecionaremos": "selecionará",
"sentíamos": "sentia",
"sentimos": "sentiu",
"sentirmos": "sentir",
"seríamos": "seria",
"seremos": "será",
"sermos": "ser",
"servimos": "serviu",
"sintamos": "sinta",
"sofremos": "sofreu",
"somos":"é",
"sonhávamos": "sonhava",
"sonhamos": "sonhou",
"sorríamos": "sorria",
"sorrimos": "sorriu",
"soubéssemos": "soubesse",
"soubemos": "soube",
"soubermos": "souber",
"superamos": "superou",
"suplicamos": "suplica",
"supomos": "supomos",
"supriremos": "suprirá",
"surpreendemos": "surpreendeu",
"suspeitávamos": "suspeitava",
"suspirávamos": "suspirava",
"suspiramos": "suspira",
"sustentamos": "sustenta",
"tínhamos": "tinha",
"tamos": "ta",
"teimávamos": "teimava",
"temíamos": "temia",
"tememos": "teme",
"temos": "tem",
"tencionamos": "tenciona",
"tenhamos": "tenha",
"tentássemos": "tentasse",
"tentamos": "tentou",
"tentaríamos": "tentaria",
"teríamos": "teria",
"teremos": "terá",
"terminávamos": "terminava",
"terminamos": "termina",
"termos": "ter",
"testarmos": "testar",
"tirávamos": "tirava",
"tiramos": "tira",
"tiremos": "tire",
"tivéssemos": "tivesse",
"tivemos": "tive",
"tivermos": "tiver",
"tocamos": "tocou",
"tomávamos": "tomava",
"tomamos": "tomou",
"tomaremos": "tomará",
"tomarmos": "tomar",
"topamos": "topa",
"tornamos": "torna",
"trabalhávamos": "trabalhava",
"trabalhamos": "trabalhou",
"trajamos": "traja",
"transpirávamos": "transpirava",
"tratávamos": "tratava",
"tratamos": "trata",
"trataremos": "tratará",
"trazíamos": "trazia",
"trazemos": "traz",
"treinamos": "treina",
"tremíamos": "tremia",
"trepássemos": "trepasse",
"trilhamos": "trilha",
"trouxemos": "trouxe",
"usávamos": "usava",
"usamos": "usou",
"usufruíamos": "usufruía",
"utilizássemos": "utilizasse",
"utilizamos": "utilizou",
"víamos": "via",
"vínhamos": "vinha",
"víssemos": "visse",
"valemos": "vale",
"valorizávamos": "valorizava",
"vamos": "vai",
"vaticinamos": "vaticina",
"vemos": "vê",
"vencemos": "venceu",
"vendemos": "vendeu",
"veremos": "verá",
"verificamos": "verificou",
"vermos": "ver",
"vestíamos": "vestia",
"viajamos": "viajou",
"viemos": "veio",
"vimos": "vem",
"viramos": "vira",
"viremos": "vire",
"virmos": "vir",
"visitemos": "visite",
"visualizamos": "visualiza",
"visualizarmos": "visualizar",
"vivêssemos": "vivesse",
"vivíamos": "vivia",
"vivemos": "vive",
"vivenciamos": "vivencia",
"viveremos": "viverá",
"vivermos": "viver",
"voamos": "voou",
"voltamos": "voltou",
"voltaríamos": "voltaria",
"voltaremos": "voltará",
"voltarmos": "voltar",
"voltemos": "volte",
"votamos": "votou"}

# concEle - concordancia com o pronome Ele  - Nenhum de nos vamos
concEle = set(["abandonamos", "abdicássemos", "abdicamos", "abraçamos", "abrimos",
"acabamos", "aceitamos", "aceitaríamos", "achávamos", "achamos",
"acolhemos", "acompanhamos", "acordamos", "acreditávamos", "acreditamos",
"acrescentamos", "adivinhamos", "admiramos", "admitirmos", "adoramos",
"adotamos", "adotemos", "adquirimos", "adulterarmos", "advertimos",
"advogamos", "agimos", "agradecemos", "ajudamos", "ajudaremos",
"ajuntamos", "ajustamos", "alcançávamos", "alimentamos", "aliviávamos",
"almoçamos", "alugássemos", "amávamos", "amamos", "amarramos",
"analisássemos", "analisamos", "andávamos", "andamos", "anunciamos",
"apanhamos", "apareçamos", "aparecemos", "aplaudimos", "aplicaríamos",
"apodrecemos", "apoiamos", "aprendemos", "apresentamos", "apresentaremos",
"aprestamos", "aprovássemos", "aprovamos", "aproveitamos", "arranjássemos",
"arranjaremos", "arrastamos", "arrombamos", "aspiramos", "assinamos",
"assinarmos", "assistimos", "assistirmos", "associamos", "assumimos",
"atacamos", "atiraríamos", "atravessávamos", "atravessamos", "aumentarmos",
"avaliamos", "avaliarmos", "avançamos", "batíamos", "batemos",
"batizamos", "bebemos", "beneficiamos", "botamos", "bradamos",
"brigamos", "brincávamos", "brincamos", "buscamos", "caçoamos",
"caímos", "caíssemos", "cabemos", "cairíamos", "calamos",
"calculamos", "caminhamos", "cantaremos", "carecemos", "carregávamos",
"carregamos", "casamos", "cavalgamos", "celebramos", "chamávamos",
"chamamos", "checamos", "chegávamos", "chegamos", "chegaremos",
"chegarmos", "cheguemos", "chorávamos", "choramos", "choremos",
"citamos", "cobramos", "colaboramos", "colocamos", "comíamos",
"combateremos", "combinávamos", "combinamos", "começamos", "começarmos",
"comemos", "comentamos", "comermos", "cometemos", "comparamos",
"completamos", "comprávamos", "compramos", "compraríamos", "compreendemos",
"comungamos", "concordamos", "concordaremos", "concorríamos", "condenamos",
"confeccionaríamos", "confiávamos", "confirmarmos", "congelaríamos", "conhecíamos",
"conhecemos", "conjeturamos", "conquistamos", "conseguíamos", "conseguimos",
"conseguiríamos", "conseguiremos", "conseguirmos", "conservarmos", "conservemos",
"considerávamos", "consideramos", "considerarmos", "consigamos", "constatamos",
"construímos", "contamos", "continuávamos", "continuamos", "continuaremos",
"contratássemos", "contratávamos", "controlamos", "conversávamos", "conversamos",
"conversaremos", "convidamos", "cooperamos", "corremos", "correremos",
"costumávamos", "costumamos", "cotamos", "credenciamos", "cremos",
"crescíamos", "crescemos", "criávamos", "criamos", "criarmos",
"criticamos", "cruzamos", "cuidamos", "culpamos", "cultivamos",
"cumprimos", "dávamos", "damos", "dançamos", "dançássemos",
"dançávamos", "debitaremos", "decidimos", "decidiremos", "defendíamos",
"defendemos", "definimos", "deitamos", "deixamos", "deixaríamos",
"demonstramos", "demos", "denominamos", "denunciamos", "depositamos",
"derrubarmos", "descemos", "descendemos", "descobrimos", "descobriremos",
"descobrirmos", "desconfiávamos", "descoroamos", "desejávamos", "desejamos",
"desejaríamos", "desempenhamos", "desentranhamos", "desenvolvíamos", "desenvolvemos",
"designamos", "despachávamos", "desprezamos", "desrespeitarmos", "desviávamos",
"detestamos", "devíamos", "devamos", "devemos", "deveríamos",
"diagnosticaremos", "digamos", "diríamos", "diremos", "discutíamos",
"dispomos", "dispormos", "disputamos", "disséssemos", "dissemos",
"divagamos", "divergimos", "dividíamos", "dividimos", "dizíamos",
"dizemos", "dobramos", "dormíamos", "dormimos", "dormiremos",
"dublamos", "duplicamos", "elaboramos", "elegemos", "embargamos",
"embirramos", "empregamos", "encaramos", "enchemos", "encolhíamos",
"encontramos", "encontraremos", "encontrarmos", "enjeitamos", "entendêssemos",
"entendemos", "entrávamos", "entramos", "entrarmos", "entregamos",
"entrevistamos", "enviamos", "enviaríamos", "enxugamos", "enxugaremos",
"éramos", "erguemos", "erramos", "escapamos", "escolhíamos",
"escolhemos", "escondíamos", "escorregamos", "escrevíamos", "escutávamos",
"esgotamos", "especificamos", "esperávamos", "esperamos", "esquecemos",
"estávamos", "estabelecemos", "estabelecermos", "estamos", "estaríamos",
"estaremos", "estarmos", "estejamos", "estimamos", "estivéssemos",
"estivemos", "estivermos", "estornaremos", "estouramos", "estragamos",
"estranhássemos", "estranhamos", "estranhemos", "estudávamos", "estudaremos",
"exercemos", "exigimos", "existimos", "expomos", "exportávamos",
"expressamos", "façamos", "falávamos", "falamos", "falaremos",
"falarmos", "faltamos", "faríamos", "faremos", "fazíamos",
"fazemos", "fazermos", "fechávamos", "fecharmos", "ficássemos",
"ficávamos", "ficamos", "ficaremos", "ficarmos", "filmamos",
"fingimos", "fiscalizamos", "fizéssemos", "fizemos", "fizermos",
"flartávamos", "folgávamos", "fomos", "formávamos", "formamos",
"formos", "fornecemos", "fossemos", "frequentávamos", "fugíssemos",
"fugimos", "fumávamos", "fundávamos", "fundamos", "furtamos",
"ganhávamos", "ganhamos", "ganharemos", "garantiremos", "gastamos",
"gostávamos", "gostamos", "gostaríamos", "gozamos", "graduamos",
"guardamos", "habitamos", "havíamos", "havemos","íamos",
"ignorávamos", "imaginávamos", "imaginamos", "imitamos", "imitaríamos",
"implantamos", "impomos", "importamos", "impregnamos", "impusemos",
"indicamos", "insistíamos", "insistimos", "instalamos", "interferimos",
"interpretamos", "interviéssemos", "invejávamos", "inventamos", "investimos",
"invocaremos", "iríamos", "iremos", "irmos", "jantássemos",
"jantávamos", "jantamos", "jogamos", "julgamos", "juramos",
"lêssemos", "líamos", "laboramos", "lamentamos", "lançarmos",
"lavamos", "lavarmos", "legalizaríamos", "lembramos", "lemos",
"levávamos", "levamos", "levantamos", "levaremos", "libertemos",
"ligávamos", "localizamos", "lustrávamos", "lutamos", "machucamos",
"mandamos", "mantemos", "marcamos", "marchamos", "marcharemos",
"matáramos", "matávamos", "matamos", "mentimos", "merecíamos",
"merecemos", "metermos", "misturamos", "moldamos", "montaríamos",
"morávamos", "moramos", "moraremos", "moremos", "morrêssemos",
"morrermos", "mostramos", "mudamos", "nascemos", "nascermos",
"necessitássemos", "necessitamos", "notamos", "notarmos", "observamos",
"observarmos", "obtivemos", "oferecemos", "olhávamos", "olhamos",
"olharmos", "optamos", "organizávamos", "ouvíamos", "ouvimos",
"ouviríamos", "púnhamos", "pagávamos", "pagamos", "pagaremos",
"parávamos", "pararmos", "parecemos", "participamos", "partimos",
"partiremos", "passássemos", "passávamos", "passamos", "passarmos",
"passearmos", "pedimos", "pegamos", "pegarmos", "pensávamos",
"pensamos", "pensarmos", "percebíamos", "percebemos", "percorremos",
"perdemos", "perderemos", "perdermos", "perdoamos", "perguntamos",
"pertencemos", "podíamos", "podemos", "poderíamos", "poderemos",
"pomos", "poríamos", "possamos", "possuímos", "poupamos",
"pouparíamos", "precisávamos", "precisamos", "precisaríamos", "precisarmos",
"predissemos", "preferíamos", "preferimos", "prendemos", "preparávamos",
"preparamos", "prepararíamos", "presenciamos", "prestamos", "pretendíamos",
"pretendemos", "processamos", "procuramos", "produzimos", "proferimos",
"projetávamos", "prometemos", "propomos", "provaremos", "providenciamos",
"publicaríamos", "pudermos", "pusemos", "puxamos", "quebrávamos",
"queríamos", "queremos", "querermos", "quiséramos", "quiséssemos",
"quisemos", "quisermos", "rebolávamos", "receávamos", "recebíamos",
"recebemos", "receberemos", "reconhecemos", "recordemos", "reduzimos",
"registramos", "regravamos", "reivindicamos", "remamos", "repararmos",
"repetimos", "representamos", "requeremos", "resistimos", "resolvíamos",
"resolvemos", "resolveríamos", "respeitamos", "respirávamos", "respiramos",
"respondíamos", "respondermos", "responsabilizamos", "retesamos", "retomamos",
"retribuímos", "retrocedermos", "reunimos", "revivíamos", "rezamos",
"rezemos", "roemos", "rolávamos", "rolamos", "roubamos",
"saíamos", "saímos", "saíssemos", "sabíamos", "sabemos",
"saberíamos", "saberemos", "sabermos", "saboreamos", "sacrificamos",
"saibamos", "sairemos", "sairmos", "sancionamos", "saudávamos",
"seguimos", "seguramos", "seguraremos", "sejamos", "selecionaremos",
"sentíamos", "sentimos", "sentirmos", "seríamos", "seremos",
"sermos", "servimos", "sintamos", "sofremos", "somos",
"sonhávamos", "sonhamos", "sorríamos", "sorrimos", "soubéssemos",
"soubemos", "soubermos", "superamos", "suplicamos", "supomos",
"supriremos", "surpreendemos", "suspeitávamos", "suspirávamos", "suspiramos",
"sustentamos", "tínhamos", "tamos", "teimávamos", "temíamos",
"tememos", "temos", "tencionamos", "tenhamos", "tentássemos",
"tentamos", "tentaríamos", "teríamos", "teremos", "terminávamos",
"terminamos", "termos", "testarmos", "tirávamos", "tiramos",
"tiremos", "tivéssemos", "tivemos", "tivermos", "tocamos",
"tomávamos", "tomamos", "tomaremos", "tomarmos", "topamos",
"tornamos", "trabalhávamos", "trabalhamos", "trajamos", "transpirávamos",
"tratávamos", "tratamos", "trataremos", "trazíamos", "trazemos",
"treinamos", "tremíamos", "trepássemos", "trilhamos", "trouxemos",
"usávamos", "usamos", "usufruíamos", "utilizássemos", "utilizamos",
"víamos", "vínhamos", "víssemos", "valemos", "valorizávamos",
"vamos", "vaticinamos", "vemos", "vencemos", "vendemos",
"veremos", "verificamos", "vermos", "vestíamos", "viajamos",
"viemos", "vimos", "viramos", "viremos", "virmos",
"visitemos", "visualizamos", "visualizarmos", "vivêssemos", "vivíamos",
"vivemos", "vivenciamos", "viveremos", "vivermos", "voamos",
"voltamos", "voltaríamos", "voltaremos", "voltarmos", "voltemos",
"votamos"])

# NUMERAIS DOIS GENEROS PARA COMPLEMENTAR COM == E UM ===
numEmfoa  = set(["Bilhão", "bilhão",
"Bilhões", "bilhões", "Cento", "cento", "Cinquenta", "cinquenta",
"Duzentas", "duzentas", "Duzentos", "duzentos", "Mil", "mil", "Milhão", "milhão",
"Milhões", "milhões", "Novecentas", "novecentas", "Novecentos", "novecentos",
"Noventa", "noventa", "Oitenta", "oitenta", "Oitocentas", "oitocentas", "Oitocentos", "oitocentos", "Quarenta", "quarenta",
"Quatrilhão", "quatrilhão", "Quatrilhões", "quatrilhões",
"Quatrocentas", "quatrocentas", "Quatrocentos", "quatrocentos", "Quinhentas", "quinhentas",
"Quinhentos", "quinhentos", "Quintilhão",
"quintilhão", "Quintilhões", "quintilhões",
"Seiscentas", "seiscentas", "Seiscentos", "seiscentos",
"Sessenta", "sessenta", "Setecentas", "setecentas", "Setecentos", "setecentos",
"Setenta", "setenta", "Trezentas", "trezentas", "Trezentos", "trezentos",
"Trilhão", "trilhão", "Trilhões", "trilhões",
"Trinta", "trinta", "Vinte", "vinte"])



# NUMERAIS MASCULINO PARA COMPLEMENTAR COM == E UM ===
numEmasc  = set(["Bilhão", "bilhão",
"Bilhões", "bilhões", "Cento", "cento", "Cinquenta", "cinquenta",
"Duzentos", "duzentos", "Mil", "mil", "Milhão", "milhão",
"Milhões", "milhões", "Novecentos", "novecentos",
"Noventa", "noventa", "Oitenta", "oitenta", "Oitocentos", "oitocentos", "Quarenta", "quarenta",
"Quatrilhão", "quatrilhão", "Quatrilhões", "quatrilhões",
"Quatrocentos", "quatrocentos", "Quinhentos", "quinhentos", "Quintilhão",
"quintilhão", "Quintilhões", "quintilhões", "Seiscentos", "seiscentos",
"Sessenta", "sessenta", "Setecentos", "setecentos", "Setenta", "setenta", "Trezentos", "trezentos",
"Trilhão", "trilhão", "Trilhões", "trilhões",
"Trinta", "trinta", "Vinte", "vinte"])



# NUMERAIS FEMININO PARA COMPLEMENTAR COM == E UMA ====
numEfem  = set(["Bilhão", "bilhão",
"Bilhões", "bilhões", "Cento", "cento", "Cinquenta", "cinquenta",
"Duzentas", "duzentas", "Mil", "mil", "Milhão", "milhão",
"Milhões", "milhões", "Novecentas", "novecentas",
"Noventa", "noventa", "Oitenta", "oitenta", "Oitocentas", "oitocentas", "Quarenta", "quarenta",
"Quatrilhão", "quatrilhão", "Quatrilhões", "quatrilhões",
"Quatrocentas", "quatrocentas", "Quinhentas", "quinhentas", "Quintilhão",
"quintilhão", "Quintilhões", "quintilhões", "Seiscentas", "seiscentas",
"Sessenta", "sessenta", "Setecentas", "setecentas", "Setenta", "setenta", "Trezentas", "trezentas",
"Trilhão", "trilhão", "Trilhões", "trilhões",
"Trinta", "trinta", "Vinte", "vinte"])


# NUMERAIS FEMININO == duzentas canetas ==============
numFem  = set(["Catorze", "catorze", "Cem", "cem", "Cinco", "cinco",
"Cinquenta", "cinquenta", "Dez", "dez", "Dezenove", "dezenove", "Dezesseis", "dezesseis",
"Dezessete", "dezessete", "Dezoito", "dezoito", "Doze", "doze",  "Duas", "duas",
"Duzentas","duzentas", "Mil", "mil", "Nove", "nove", "Novecentas", "novecentas",
"Noventa", "noventa", "Oitenta", "oitenta", "Oito", "oito", "Oitocentas", "oitocentas",
"Onze", "onze", "Quarenta", "quarenta", "Quatorze", "quatorze", "Quatro", "quatro",
"Quatrocentas", "quatrocentas", "Quinhentas", "quinhentas", "Quinze", "quinze",
"Seis", "seis", "Seiscentas", "seiscentas", "Sessenta",  "sessenta",
"Sete", "sete", "Setecentas", "setecentas", "Setenta", "setenta",
"Três", "três", "Treze", "treze",
"Trezentas", "trezentas", "Trinta", "trinta",
"Vinte", "vinte"])



# NUMERAIS MASCULINO == duzentas canetas ==============
numMasc  = set(["Catorze", "catorze", "Cem", "cem", "Cinco", "cinco",
"Cinquenta", "cinquenta", "Dez", "dez", "Dezenove", "dezenove", "Dezesseis", "dezesseis",
"Dezessete", "dezessete", "Dezoito", "dezoito", "Dois", "dois", "Doze", "doze",
"Duzentos","duzentos", "Mil", "mil", "Nove", "nove", "Novecentos", "novecentos",
"Noventa", "noventa", "Oitenta", "oitenta", "Oito", "oito", "Oitocentos", "oitocentos",
"Onze", "onze", "Quarenta", "quarenta", "Quatorze", "quatorze", "Quatro", "quatro",
"Quatrocentos", "quatrocentos", "Quinhentos", "quinhentos", "Quinze", "quinze",
"Seis", "seis", "Seiscentos", "seiscentos", "Sessenta",  "sessenta",
"Sete", "sete", "Setecentos", "setecentos", "Setenta", "setenta",
"Três", "três", "Treze", "treze",
"Trezentos", "trezentos", "Trinta", "trinta",
"Vinte", "vinte"])


# NUMERAIS FEMININO-MASCULINO == duzentos canetas == trezentas carros =======
nuMfmoas  = set(["Duzent","duzent", "Novecent", "novecent",
"Oitocentas", "oitocent", "Quatrocent", "quatrocent", "Quinhent",
"quinhentas", "Seiscent", "seiscent", "Setecent",
"setecent", "Trezent", "trezent"])

# NUMERAIS FEMININO-MASCULINO == sem os ENTOS e ENTAS = duzentos trezentas
numMfmoa  = set(["Catorze", "catorze", "Cem", "cem", "Cento", "cento", "Cinco", "cinco",
"Cinquenta", "cinquenta","Dez", "dez", "Dezenove", "dezenove", "Dezesseis", "dezesseis",
"Dezessete", "dezessete", "Dezoito", "dezoito", "Dois", "dois", "Doze", "doze",
"Mil", "mil", "Nove", "nove", "Noventa", "noventa", "Oitenta", "oitenta", "Oito", "oito",
"Onze", "onze", "Quarenta", "quarenta",  "Quatorze", "quatorze", "Quatro", "quatro",
"Quinze", "quinze", "Seis", "seis","Sessenta", "sessenta", "Sete", "sete", "Setenta", "setenta",
"Três", "três", "Treze", "treze",
"Trinta", "trinta", "Vinte", "vinte"])

# PALAVRAS MASCULINAS == terminadas em [aeiou]rao =til ao= mulherao, caldeirao, = calorao ==
plvrMrao  = set(["abeir", "ailer", "air", "albacor",
"albardeir", "alcatr", "alcor","alfarrobeir",
"algir", "almeir", "alpalheir", "alqueir", "alvador",
"alvadur", "amir", "angular", "ar", "arur", "asneir", "azar","bafor",
"bair", "baleeir", "bandeir", "bar", "beir", "beltr", "besour", "betatr",
"biar", "biber", "biqueir", "bonacheir", "boqueir", "botir", "brancar",
"Brasileir", "brasileir", "brejeir", "bur", "butir", "cadeir", "Caldeir",
"caldeir", "calmeir", "calor", "camar", "cambur", "canastr", "cangueir",
"canjir", "cantimar", "capeir", "capoeir", "caracanir", "car", "carreir",
"casar", "cavaleir", "caveir", "cavir", "centr", "chamborreir",
"champorreir", "chapeir", "chapeleir", "chaporreir", "char", "charqueir",
"chaveir", "cheir", "chor", "choror", "chuveir", "ciclotr", "ciger",
"cintur", "cir", "cisir", "citeropter", "cizir", "clar", "coentr", "coer",
"coir", "companheir", "congueir", "contempor", "convier", "cor", "cormor",
"coscor", "costur", "cour", "cur", "deuter", "diamor", "diatessar", "dinheir",
"doutor", "dur", "ebur", "electr", "eler", "encontr", "eriger", "escadeir",
"escaleir", "escor", "escur", "esgueir", "espadeir", "espor", "esteir",
"estir", "estupor", "estur", "faceir", "fachar", "falastr", "faveir", "feir",
"fever", "figur", "flaqueir", "flor", "fortalheir", "fraqueir", "frieir",
"fur", "galeir", "galer", "gar", "gaspar", "gir", "gizir", "gor", "gorgor",
"goteir", "grossalheir", "grosseir", "har", "hiper", "hor", "interfer",
"jacatir", "jaguar", "jaguatir", "jangueir", "jeir", "jur", "labor", "ladeir",
"lafar", "lagueir", "lamar", "lambeir", "lameir", "lampar", "langueir",
"largueir", "lazar", "leir", "lezir", "ligueir", "linguar", "lingueir",
"linhar", "lir", "liseir", "listr", "longeir", "longer", "longueir", "lumar",
"maceir", "macor", "madraceir", "madur", "magreir", "malheir", "manguar",
"mangueir", "manjer", "mar", "martir", "masseir", "matir", "mazorreir",
"megafeir", "milhar", "milheir", "milher", "Mineir", "mir", "moir",
"molangueir", "molanqueir", "moleir", "mor", "mour", "mulher", "muquir",
"mutir", "muxir", "napeir", "naper", "neutr", "olheir", "palheir", "palmeir",
"pandeir", "par", "parreir", "parvoeir", "patr", "pecador", "peir", "peneir",
"pequir", "percher", "perchur", "piapar", "piquir", "pir", "plasteir",
"plastr", "plastreir", "poceir", "poeir", "polar", "poleir", "poltr", "por",
"positr", "potir", "punxir", "putir", "puxir", "quarteir", "quarter",
"quinteir", "rafeir", "regueir", "ribeir", "salmoir", "salmour", "sanhar",
"sapir", "sar", "satir", "sear", "seir", "sendeir", "sensabor", "simplacheir",
"simplalheir", "simpleir", "sincrotr", "solteir", "sopeir", "superver",
"taleir", "tambolar", "tambur", "taper", "tar", "tempor", "tesoir", "tesour",
"tinhor", "tintur", "tir", "toir", "toleir", "tor", "toupeir", "tour",
"trigueir", "tronqueir", "tubar", "tucur", "vair", "valdur", "valeir",
"vaner", "var", "vaseir", "vassoir", "vazeir", "ver", "verber", "vibor",
"votur", "vozeir", "zar", "zombeir"])


# PALAVRAS MASCULINAS == terminadas em IDA = inseticida, pesticida
excplvrIda  = set(["alme", "almoráv", "Antárt", "apátr",
"aquemên","ár", "arsác", "Atlânt",
"aven","áv", "bacteric", "bel", "bioc", "br", "bríg",
"cantár", "carót", "cefe", "cianam", "ciclofosfam",
"coinc", "col", "consol", "contrapart", "conv", "crisál", "cu",
"dec", "div", "dív", "dru", "duv", "dúv", "el", "eluc",
"endiv", "endo", "engrav", "enegrec", "env", "erm", "espermic",
"estup", "filic", "fóc", "formam", "formic", "fratric", "fungic",
"genoc", "germic", "grav", "gu", "herbic", "homic", "inc", "infantic",
"insectic", "insetic", "intim", "inval", "isoniaz", "lap", "láp",
"liqu", "liv", "luc", "maldorm", "margar", "m", "nere", "n", "ox",
"parót", "parric", "pestic", "plac", "poliacrilam", "pres",
"progr", "quir", "ratic", "regic", "reinc", "res", "sassân",
"selêuc", "seljúc", "sol", "subdiv", "suic",
"sulfam", "talidom", "táur", "teba", "tiranic", "transgr",
"transluc", "trep", "tropicam", "truc", "uxoric", "ven", "v", "za"])


# PALAVRAS REPETIDAS == EXCECAO
excpLvrep  = set(["PARA", "Para", "para", "PELA", "Pela", "pela", "PELAS", "Pelas", "pelas",
"PELO", "Pelo", "pelo", "PELOS", "Pelos", "pelos"])


# PALAVRAS MASCULINAS TERMINADAS EM  IVA == EXCECAO
excpLvriva  = set(["adject", "adjet", "alque", "arqu", "at", "av", "cat",
"conv", "cr", "cult", "der", "desarqu", "desat", "desmot", "efet", "e",
"enra", "esqu", "incent", "invect", "mot", "Ne", "no", "object", "objet",
"pa", "pr", "qued", "reat", "reav", "rev", "sara", "Sara", "u", "v"])

# PALAVRAS TERMINADAS EM  ADA == EXCECAO
excpLvrada  = set(["Andr", "c", "Camar", "camar", "Can", "Lusí",
"lusí", "n", "nôm"])

# PALAVRAS FEMININAS TERMINADAS EM  NC,A == EXCECAO = venc,a - presenc,a - ananc,a
excpLvrnca  = set(["acarr", "acri", "ader", "afi", "agrav", "alc", "am",
"ape", "aquer", "arrom", "av", "av", "bag", "bal", "bon", "b", "ca",
"c", "conv", "d", "desconv", "destr", "det", "detr", "el", "embal",
"eng", "enj", "enr", "entr", "entr", "esbal", "esbr", "esc", "escarv",
"escarv", "esfr", "esfreg", "esmi", "esp", "estr", "estr",
"estr", "ev", "fem", "furd", "g", "gar", "inj", "l", "nu", "parr",
"pert", "perv", "pi", "p", "p", "p", "r", "rasp", "reconv", "resp",
"r", "rip", "segur", "sobr", "tr", "v"])

# plMfevb - substantivos dois generos terminados em -e- tambem encontrados em conjugacoes verbais.
plMfevb = set(["corte"])

# excplMfe - excecao ao artigo A em palavras de dois generos terminadas em - e - exemplo: A requintes de crueldade
excplMfe = set(["requint"])

# PALAVRAS DE DOIS GENEROS TERMINADAS EM  E == paranaense - elegante - galante
plvrMfe = set(["aaleniens", "aalens", "aaquenens", "abacatens",
"abacaxiens", "abacens", "abadens", "abadianens",
"abadiens", "abadinens", "abaeteens", "abaetetubens",
"abaetezinhens", "abaiarens", "abaibens", "abairens",
"abaitinguens", "abambresens", "abapanens", "abareens",
"abatiaens", "abderens", "abdonens", "abecedens",
"abelardens", "abelens", "abelhens", "aberrant",
"abertanens", "abevilens", "abicens", "abismens",
"abitinens", "abitureirens", "abiulens", "ablitens",
"aboborens", "aborígen", "aborígin", "abraamens",
"abraãozens", "abranchens", "abrangent", "abrantens",
"abrãoens", "abrasant", "abreuens", "absorvent",
"abstinent", "abufariens", "abulens", "abunãens",
"abunanens", "abundanciens", "abundant", "abuniens",
"acaãens", "acaciens", "acadiens", "acaiacens",
"acaiaquens", "açaiens", "açailandens", "açailandiens",
"açaitubens", "açaizalens", "acajutibens", "acampamentens",
"acanãens", "acananens", "acaraens", "acarapens",
"acarauens", "açareens", "acariens", "aceguaens",
"acesitens", "acheulens", "aciolens", "acmonens",
"acompanhant", "aconchegant", "açorens", "açoriens",
"acrens", "acreunens", "açucenens", "açuens",
"adamantinens", "adelaidens", "adelandens", "aderent",
"adjacent", "adolescent", "adotant", "adrianopolens",
"adstringent", "adustinens", "afifens", "afogadens",
"afraniens", "afrodescendent", "afrodisiens", "afuaens",
"agaunens", "agent", "aginens", "agirinens",
"agitant", "agonens", "agonizant", "agravant",
"agrest", "agrestinens", "agricolandens", "agridoc",
"agripens", "agripinens", "agrolandens", "aguanilens",
"agudens", "aguiarens", "aimoreens", "aiquarens",
"airãoens", "aiuabens", "aiuruoquens", "ajudant",
"ajuricabens", "alabandens", "alabanens", "alanceant",
"alandroalens", "alantens", "alarmant", "alasquens",
"alavens", "albacetens", "albanens", "albens",
"albergariens", "albertinens", "albicastrens", "albicens",
"albigens", "albingaunens", "albinitens", "albintimiliens",
"albufeirens", "albuquerquens", "alcacerens", "alcalarens",
"alcanenens", "alcantarens", "alcantilens", "alceciens",
"alcetiens", "alcobacens", "alcoutenens", "alcoutinens",
"aldeens", "aldegalens", "aldematens", "alecrinens",
"alegrens", "alegret", "alegretens", "alegriens",
"alenquerens", "aletriniens", "alexandriens", "alexanens",
"alfamens", "alfandeguens", "alfenens", "alfredens",
"algarbiens", "algarviens", "algidens", "algonquiens",
"algonquinens", "alhadens", "alhandrens", "aliancens",
"aliciant", "alijoens", "aliterant", "aljezurens",
"aljubarrotens", "aljustrelens", "almadens", "almeidens",
"almeiriens", "almeirinens", "almenarens", "almens",
"almescarens", "almesquinhens", "almocens", "almodovarens",
"almofadens", "aloandiens", "alpalhoens", "alpens",
"alpercatens", "alpestrens", "alpiarcens", "alpinopolens",
"alsiens", "altairens", "altamirens", "altaneirens",
"alteadens", "altens", "alterens", "alterosens",
"altinens", "altinopolens", "altissonant", "altoens",
"altoniens", "altuens", "alucinant", "alunens",
"alurens", "alutrens", "alutriens", "alvaçãozens",
"alvaiazerens", "alvaranens", "alvarelhens", "alvarelitens",
"alvarenguens", "alvarens", "alvarinens", "alvejant",
"alvens", "alverquens", "alvesens", "alvilandiens",
"alvinitent", "alvinlandens", "alvinlandiens", "alvinopolens",
"alvinopolitens", "alvitens", "alvoradens", "amacianens",
"amadorens", "amananens", "amandabens", "amandens",
"amanhecens", "amaniuens", "amant", "amanuens",
"amapaens", "amaporanens", "amaporens", "amarajiens",
"amaralens", "amarantens", "amarantinens", "amarelens",
"amarens", "amargosens", "amatariens", "amatauraens",
"amaturaens", "amazonens", "amazoniens", "ambianens",
"ambivalent", "amblíop", "ambraciens", "ambrosiens",
"ambulant", "amedarens", "ameixens", "ameliens",
"ameliopolens", "americanens", "amienens", "amiens",
"amiternens", "amitinens", "amizadens", "amocens",
"amolarens", "amoniens", "amontadens", "amorescens",
"amorinopolens", "amorosens", "amparens", "amperens",
"amuamuaipens", "anabolizant", "anacletens", "anadiens",
"anajaens", "anajasens", "anajatenens", "anajatubens",
"anajens", "analandens", "analandiens", "anamanens",
"anamariens", "ananasens", "ananatubens", "ananidens",
"ananindeuens", "anapuens", "anapuruens", "anarquizant",
"anastaciens", "anatiens", "anauaens", "anaurilandens",
"ancarens", "anchietens", "ancianens", "ancoriens",
"andant", "andaraiens", "andaricens", "andesens",
"andiraens", "andirens", "andirobalens", "andirobens",
"andologens", "andorrens", "andradens", "andradinens",
"andreens", "andrelandens", "anelant", "aneliens",
"anemuriens", "anemutuens", "angaiarens", "angaiens",
"angatubens", "angaturamens", "angejens", "angelens",
"angelicens", "angelinens", "angicalens", "angicanens",
"angicãozens", "angicens", "angiquens", "angolens",
"angrens", "anguerataens", "anguerens", "angusturens",
"anhanduiens", "anhanduizinhens", "anhangaens", "anhangaiens",
"anhanguens", "anhanguerens", "anhembiens", "anhumens",
"anicunens", "aninhens", "anisiens", "anitapolens",
"aniversariant", "anoriaçuens", "anoriens", "anseriform",
"ansianens", "antecedent", "antens", "anticariens",
"anticirens", "anticoagulant", "anticongelant", "antigen",
"antigonens", "antilhens", "antinhens", "antioquens",
"antioquiens", "antioxidant", "antofagastens", "antoninens",
"antuerpiens", "anunciant", "anutibens", "apach",
"apaixonant", "apajatubens", "aparecidens", "apavorant",
"aperibens", "apeuens", "apiaiens", "apinajeens",
"apodiens", "apoiant", "apolinariens", "apolonidens",
"apolonidiens", "aporaens", "aporeens", "aporemens",
"aprazibilens", "aprazivelens", "aproagens", "aproaguens",
"apuaens", "apuaremens", "apucaranens", "apucaraninhens",
"apuiens", "apuliens", "apurimaens", "aquabonens",
"aquaplatens", "aquens", "aquicaldens", "aquidabanens",
"aquidauanens", "aquileens", "aquinens", "aquirabens",
"aquirasens", "aquitanens", "arabelens", "arabricens",
"arabrigens", "arabutanens", "araçaçuens", "araçaens",
"araçaibens", "araçaiens", "araçajiens", "aracajuens",
"araçariguamens", "aracatiaçuens", "aracatiarens", "aracatiens",
"araçatubens", "aracatuens", "araciens", "aracitabens",
"aracoiabens", "aracrucens", "aracruzens", "aragarcens",
"aragoianens", "aragonens", "araguacemens", "araguaciens",
"araguaçuens", "araguaiarens", "araguaiens", "araguainens",
"araguainhens", "araguananens", "araguapacens", "araguariens",
"araguatinens", "arainens", "araiosens", "arajarens",
"aramariens", "arambareens", "araminens", "arananens",
"aranauens", "aranduens", "aranhens", "arant",
"arantinens", "arapaens", "arapariens", "arapeiens",
"arapiracens", "arapiraquens", "arapirens", "arapoemens",
"araponguens", "araporanens", "arapotiens", "arapuaens",
"araquaens", "araquariens", "araquenens", "araraiens",
"araranguaens", "ararapirens", "araraquarens", "ararendaens",
"ararens", "araricaens", "arariens", "araripens",
"araripinens", "arariusens", "araruamens", "araruens",
"ararunaquarens", "ararunens", "araruvens", "aratacens",
"aratamens", "aratibens", "araticuens", "aratingaubens",
"aratubens", "aratuipens", "arauaens", "araucariens",
"arauens", "araujens", "araunens", "araxaens",
"arborescent", "arcadens", "arceburguens", "arcelinens",
"arcens", "arcobotant", "arcobrigens", "arcoens",
"arcoverdens", "arcozelens", "arcuens", "ardenens",
"ardent", "areadens", "arealvens", "areanens",
"areãozens", "areens", "areinhens", "areiopolens",
"arembepens", "arenalens", "arenapolens", "arequembauens",
"arerens", "aresens", "arfant", "arganilens",
"argenitens", "argentens", "argiritens", "argoimens",
"argoinens", "argolens", "argoncilhens", "ariauens",
"aricanduvens", "aricanguens", "ariciens", "aricorens",
"arimanens", "ariminens", "arinens", "aripibriens",
"aripuanens", "ariquemens", "ariranhens", "aririens",
"aristeuzens", "aristidens", "arisuelens", "aritagnaens",
"arizonens", "armaçãozens", "armamarens", "armazenens",
"arneirosens", "arnoniens", "aroasens", "aroeirens",
"aromatizant", "arouquens", "arpoarens", "arquejant",
"arquens", "arraialens", "arraiolens", "arreens",
"arrematant", "arrependidens", "arrepiant", "arrogant",
"arrojadens", "arronchens", "arrozalens", "arrozeirens",
"arrozens", "arrudens", "arrulhant", "artemisens",
"artífic", "aruajaens", "aruananens", "aruaruens",
"aruasens", "aruauens", "aruens", "arujaens",
"arumandubens", "arumanens", "aruminens", "aruntens",
"arutiens", "arvens", "arvernens", "arvoredens",
"arvorezinhens", "ascendent", "asclepíad", "asconens",
"ascurrens", "asdrubalens", "asfixiant", "asidonens",
"aspasiens", "aspirant", "assaltant", "assiens",
"assinant", "assisens", "assistenciens", "assistent",
"assistuens", "assoant", "assuncionens", "astapens",
"astartiens", "astens", "astiens", "astipalens",
"astolfens", "astorguens", "astrapeens", "asturicens",
"atacant", "atafonens", "atalaiens", "atalantens",
"ataleens", "atalens", "atalhens", "ataquens",
"ataubens", "atendent", "ateniens", "atenuant",
"aternens", "aterradens", "aterradinhens", "atiaçuens",
"atibaiens", "atiliens", "atimirinens", "atimirinzens",
"atinent", "atiradens", "atlant", "atoladeirens",
"atoleirens", "atorment", "atraent", "atriens",
"atuant", "atubens", "aturiaiens", "augustobrigens",
"augustodunens", "aurelianens", "aurens", "auriens",
"auriflamens", "aurignacens", "aurilandens", "aurilandiens",
"aurinhacens", "auriverdens", "aurizonens", "auroens",
"aurorens", "ausent", "ausentens", "ausonens",
"autasasens", "autazens", "autocolant", "autócton",
"autoimun", "automedont", "autossuficient", "avaiens",
"avanhandavens", "avareens", "avaricens", "avatinquarens",
"avelarens", "avencens", "avens", "aviacionens",
"aviltant", "avisens", "avoant", "axininens",
"axininzens", "axixaens", "azambujens", "azauriens",
"azeitonens", "azevedens", "azilens", "azuritens",
"baamens", "babaçuens", "babaçulandiens", "babaçuzens",
"babanens", "babiens", "babiloniens", "bacabalens",
"bacabens", "bacadens", "bacaetavens", "bacainens",
"bacatubens", "bacatuens", "bacaubens", "bacaxaens",
"bacelaens", "bacelarens", "bacuriens", "bacuriteuens",
"bacuritiens", "bacuritubens", "bacurizinhens", "badajozens",
"badenens", "baependiens", "baetens", "bageens",
"bagrens", "baguaçuens", "baguariens", "bahamens",
"baianens", "baiãozens", "baiens", "baiepanens",
"bailens", "bailiquens", "baionens", "bajeens",
"bajulant", "balaiadens", "balaieirens", "balançant",
"balbuciant", "baleens", "baleiens", "balens",
"baliniens", "balizens", "baliziens", "balneariens",
"baloiçant", "balouçant", "balsamens", "balsens",
"baltasarens", "baluartens", "bambeant", "bamboant",
"bamboleant", "bambuiens", "banabuinens", "bananalens",
"bananeirens", "bananens", "bandarrens", "bandeirant",
"bandeirantens", "bandeirens", "bangladens", "banguens",
"bannaquens", "banquetens", "banzaens", "baracajaens",
"baraunens", "barbacenens", "barbadens", "barbalhens",
"barbarens", "barbosens", "barcadens", "barcarenens",
"barcelens", "barcelonens", "barcinonens", "bardariens",
"baremens", "bargant", "baririens", "baritens",
"baronens", "barquens", "barquinhens", "barracãozens",
"barracens", "barraconens", "barradens", "barragenens",
"barranceirens", "barranquens", "barraquens", "barraquinhens",
"barreirãozens", "barreirens", "barreirinhens", "barrens",
"barrentens", "barrestivens", "barretens", "barretinens",
"barrigadens", "barrigudens", "barrinhens", "barroçãozens",
"barrocassalens", "barrolandens", "barroquens", "barroquinhens",
"barrosens", "baruens", "barueriens", "baruriens",
"basculant", "basiliens", "bassanens", "bastens",
"bastianens", "bastiãozens", "bastosens", "bataguaçuens",
"bataiporanens", "batalhens", "batataens", "batataiens",
"batatalens", "batateirens", "batatubens", "bateens",
"bateiens", "batelãozens", "bateriens", "batinguens",
"batoquens", "batoviens", "bauens", "bauinens",
"baunilhens", "bauruens", "bautens", "bauxiens",
"bearnens", "beatificant", "bebedourens", "beberibens",
"beculonens", "beirens", "bejens", "bejuens",
"belchiorens", "belemens", "belenens", "belezens",
"beligerant", "belisariens", "belizens", "belmontens",
"belterrens", "belunens", "belvederens", "bempostens",
"benacens", "benaventens", "bendizent", "beneditens",
"beneditinens", "beneficent", "beneleitens", "benemerent",
"benevenutens", "benevolent", "benfazent", "benfiquens",
"bengalens", "benguelens", "beninens", "bentabreuens",
"bentens", "bentinhens", "bequimoens", "bequimonens",
"berilens", "beriquiens", "beritens", "berlinens",
"bermudens", "bernardens", "bernardinens", "bernens",
"bertinhens", "bertioguens", "bertolinens", "beruriens",
"besourens", "bestificant", "betabloqueant", "betaniens",
"beterrens", "betimens", "betinens", "betuliens",
"betulonens", "beverniens", "bexiguens", "bezerrens",
"biafrens", "bibarrens", "biboquens", "bicanquens",
"bicaquens", "bicasens", "bicudens", "bicuibens",
"bidens", "bien", "biform", "bifront",
"bifurqu", "biguaçuens", "biguazinhens", "bilacens",
"bilaquens", "bilhostr", "bilidens", "biliens",
"bilíngu", "biltr", "binguens", "biocenos",
"bioluminescent", "bipen", "biquens", "biquinhens",
"birbant", "biribeirens", "biribiriens", "biriguiens",
"biririquens", "biritibens", "biritinguens", "birmanens",
"biscaiens", "bissanens", "bissauens", "bitupitaens",
"bituranens", "bituriens", "biturunens", "bivalent",
"bixopaens", "bizarrens", "bloqueant", "boaciquens",
"boaçuens", "boapabens", "boavianens", "bocainens",
"bocaiuvens", "bocejant", "boçoroquens", "bodincomagens",
"bodocoens", "bodocongoens", "bodoens", "bodoquenens",
"bodotens", "boeguaçuens", "bofetens", "boguaçuens",
"boiaçuens", "boiant", "boiarucens", "boiaruquens",
"boinens", "boiteuxburguens", "boituvens", "bojuruens",
"bolachens", "bolamens", "bolcheviqu", "bolichens",
"bombarralens", "bonaerens", "bonens", "bonfinens",
"bonhuens", "bonifaciens", "boninalens", "bonitens",
"bonjesuens", "bononiens", "bonretirens", "bonsucessens",
"boqueirãozens", "boqueirens", "boquiens", "boquinens",
"boquirens", "boraceens", "boraens", "borbens",
"borboletens", "borboremens", "borbulhant", "bordalens",
"bordelens", "born", "bosens", "bosnens",
"bosniens", "bosquens", "bosquíman", "bostonens",
"botafoguens", "botelhens", "botens", "botiquens",
"botocudens", "botucatuens", "bovilens", "boximan",
"bracarens", "bracejant", "bracens", "bragancens",
"bragantinens", "braguens", "brâman", "bram",
"branacens", "brancens", "brandonens", "brandurens",
"branquens", "branquinhens", "brasabrantens", "brasens",
"brasilandens", "brasilandiens", "brasileens", "brasileiens",
"brasileirens", "brasilens", "brasiliens", "brasinens",
"brasiolens", "brasitanens", "brasopolens", "braunens",
"bravens", "brejãozens", "brejatubens", "brejaubens",
"brejauvens", "brejens", "brejetubens", "brejinhens",
"brejoens", "brejonens", "bresciens", "bressanens",
"bretens", "brev", "brevens", "brigadens",
"brigant", "brigidens", "briguens", "brilhant",
"brilhantens", "brilhantinens", "brincant", "britanens",
"britaniens", "britens", "brixiens", "brococoens",
"brodosquiens", "brotens", "brumadens", "brumadiens",
"brumadinhens", "brumalens", "brundueens", "brusquens",
"brutamont", "bruxelens", "bruxuleant", "buarens",
"bucarestens", "bucuitubens", "budapestens", "buenairens",
"buenens", "buenolandens", "buenopolens", "bueraremens",
"bufadeirens", "bufant", "bugrens", "buiquens",
"bujaruens", "bulhonens", "bulidens", "bupevens",
"buquinens", "buracicens", "buraciquens", "buranhenens",
"buraramens", "burdegalens", "burdigalens", "burichens",
"buriens", "burietaens", "buritamens", "buritiens",
"buritiramens", "buritiranens", "buritizalens", "buritizeirens",
"buritizinhens", "burizeirens", "burquinens", "bursaonens",
"burundiens", "butanens", "butiaens", "caaguaçuens",
"caapiranguens", "caaporanens", "caatibens", "caatinguens",
"cabaceirens", "cabacens", "cabalianens", "cabanens",
"cabeçanens", "cabeceirens", "cabecens", "cabeçudens",
"cabedelens", "cabeleirens", "cabens", "cabilonens",
"cabiunens", "cabixiens", "caboclens", "cabralens",
"cabraliens", "cabreuvens", "cabriuvens", "cabroboens",
"cabuçuens", "caburanens", "cabureens", "cacaiens",
"cacatuens", "cacebuens", "cacequiens", "cacerens",
"cacetens", "cachimbens", "cachoeirens", "cachoeirinhens",
"cachorrens", "cacimbens", "cacimbinhens", "cacoalens",
"cacondens", "caculeens", "cadavalens", "cadeens",
"cadent", "cadet", "cadoriens", "cadozens",
"cadurcens", "caenens", "caenzens", "caetanens",
"caeteens", "caetiteens", "cafarnaunens", "cafearens",
"cafeeirens", "cafeens", "cafelandens", "cafezalens",
"cafezalzinhens", "cafezens", "cafufaens", "cafundoens",
"cagadens", "caguaçuens", "caiabuens", "caiacuens",
"caiadens", "caianens", "caiaponiens", "caibateens",
"caibiens", "caiçarens", "caicoens", "caieirens",
"caienens", "caiens", "caiguataens", "caipirens",
"caiporens", "caipuens", "cairariens", "caireens",
"cairiens", "cairuens", "caitaens", "caitauens",
"caitauzens", "caiteens", "caititeens", "caitituens",
"caiuaens", "caiubens", "caiubiens", "caiuens",
"cajaens", "cajaibens", "cajamarens", "cajapioens",
"cajariens", "cajatiens", "cajazeirens", "cajobiens",
"cajubiens", "cajubinens", "cajueirens", "cajuiens",
"cajuriens", "cajuruens", "cajuzeirens", "calabacens",
"calaboucens", "calambauens", "calamens", "calanaquens",
"calatravens", "calcedonens", "calcutaens", "caldens",
"calens", "calheirens", "calhetens", "caliciform",
"californiens", "calipolens", "calmant", "calmonens",
"calogerens", "calojiens", "calpens", "calubrigens",
"calumbiens", "camaçajiens", "camaçandiens", "camacanens",
"camacaocens", "camaçariens", "camacãzens", "camachens",
"camacupens", "camaipiens", "camalauens", "camaldulens",
"camaleonens", "camamuens", "camanducaiens", "camapuanens",
"camapuens", "camaquanens", "camaquens", "camaradens",
"camaraens", "camarajibens", "camaratubens", "camarens",
"camarguens", "camaronens", "cambaiens", "cambaleant",
"cambaquarens", "cambaraens", "cambaratibens", "cambarazinhens",
"cambaubens", "cambeens", "camberrens", "cambiant",
"cambiascens", "cambiasquens", "cambirens", "cambiuens",
"camboapinens", "camboataens", "camboinens", "camboriuens",
"cambrens", "cambucaens", "cambuciens", "cambuçuens",
"cambuiens", "cambuizens", "cambuquirens", "camburiens",
"camelens", "cametaens", "caminhant", "caminhens",
"camobiens", "camocinens", "camolenguens", "camopiens",
"campanariens", "campanens", "campanhens", "campaniens",
"campechens", "campeirens", "campelens", "campens",
"campestr", "campestrens", "campinaçuens", "campinens",
"campomaiorens", "camuciens", "camucinens", "camurujiens",
"camurupinens", "camuscadens", "camutanguens", "camutiens",
"canaanens", "canabarrens", "canabravens", "canaçuens",
"canacuriens", "canadens", "canadiens", "canaens",
"canafistulens", "canalens", "canamanens", "canamariens",
"cananeens", "canaranens", "canarianens", "canariens",
"canastrãozens", "canastrens", "canatibens", "canauaniens",
"canavialens", "canavieirens", "cancanens", "cancelãozens",
"cancelens", "cancheens", "candealens", "candeens",
"candelariens", "candent", "candialens", "candibens",
"candoiens", "candombens", "candumbens", "caneapens",
"canecens", "canedens", "caneleirens", "canelens",
"canelinhens", "canens", "cangalhens", "cangatens",
"cangatiens", "cangotens", "canguaretamens", "canguçuens",
"canguens", "canguerens", "canhemborens", "canhobens",
"canhotinhens", "canhumanens", "canindeens", "canindezinhens",
"canitarens", "canivetens", "canjaranens", "canjiquens",
"canoananens", "canoanens", "canoeirens", "canoens",
"canoinhens", "cansançãozens", "cantagalens", "cantanhedens",
"cantant", "cantãozens", "cantens", "cantinhens",
"cantonens", "cantoriens", "cantuariens", "canudens",
"canumanens", "canutamens", "canzens", "capacetens",
"capadens", "capaneens", "capanemens", "capanens",
"capãozens", "caparaozens", "capatariens", "capeçuens",
"capelens", "capelinens", "capelinhens", "caperens",
"capetinguens", "capiaens", "capibaribens", "capinalens",
"capinens", "capintubens", "capinzalens", "capistranens",
"capitanens", "capitariens", "capitoliens", "capitulens",
"capituvens", "capivarens", "capivariens", "capivaritens",
"capixabens", "capoeiranens", "capoeirens", "caponguens",
"caporanguens", "caprariens", "caprichens", "capsens",
"capuavens", "capubinens", "capuens", "capumbens",
"caputirens", "caquemonens", "caquendens", "caraaens",
"carabuçuens", "caracaraens", "caracaraiens", "caracolens",
"caraguataiens", "caraguatatubens", "caraibalens", "caraibeirens",
"caraibens", "caraibunens", "caraiens", "caraipens",
"caraivens", "carajaens", "carajazinhens", "carajuvens",
"carambeiens", "caramuruens", "caranaibens", "caranandinens",
"caranandubens", "carananens", "caranapatubens", "carandaiens",
"carandazalens", "caranens", "carangolens", "carapajoens",
"caraparuens", "carapebens", "carapebuens", "carapevens",
"carapicuibens", "carapiens", "carapinens", "carapoens",
"carapotoens", "caratacens", "caratatenens", "caratatinens",
"caratinguens", "caratuvens", "caravelens", "carazinhens",
"carbonitens", "carburant", "carcanhens", "carcassonens",
"carcavelens", "cardealens", "cardosens", "careaçuens",
"careirens", "carent", "cariacens", "cariacicens",
"cariaciquens", "carib", "cariceens", "caridadens",
"carimanens", "carimataens", "carinhanhens", "carmens",
"carmesiens", "carmonens", "carmopoliens", "carnaibens",
"carnaubaens", "carnaubalens", "carnaubeirens", "carnaubens",
"carnaubinhens", "carneirens", "carneirinhens", "carnicens",
"carnobiens", "carnoioens", "caroarens", "carolinens",
"caroviens", "carpinens", "carquejens", "carranquens",
"carrapateirens", "carrapatens", "carrapatinhens", "carrapichelens",
"carrarens", "carrazedens", "carregadorens", "carregalens",
"carreirens", "cartaginens", "cartaxens", "cartomant",
"caruaens", "carualinens", "caruarens", "caruaruens",
"caruataiens", "carubinhens", "carumbeens", "carutaperens",
"caruzuens", "carvalhens", "carvoalens", "carvoalzinhens",
"carvoeirens", "casablanquens", "cascalens", "cascalheirens",
"cascalhens", "cascantens", "cascatens", "cascatinhens",
"cascavelens", "cascudens", "caseirens", "casevelens",
"casimirens", "casinhens", "casquens", "cassens",
"cassianens", "cassiens", "cassilandens", "cassilandiens",
"cassinens", "cassiteritens", "cassuens", "castanhalens",
"castanheirens", "castanhens", "castelandens", "castelandiens",
"castelens", "castelhanens", "castelinhens", "castilhens",
"castraleucens", "castrens", "casualidadens", "catabanens",
"cataguarinens", "cataguasens", "catalanens", "catanduvens",
"catanhedens", "catarens", "catarinens", "catauarens",
"catendens", "catequesens", "cateriangonguens", "catetens",
"catiarens", "catiboabens", "catiguaens", "catiguens",
"catimbauens", "catingalens", "catingueirens", "catinguens",
"catitens", "cativant", "catolandens", "catolandiens",
"catoleens", "catolezinhens", "catrimaniens", "catuanens",
"catuçabens", "catuens", "catuipens", "catuleens",
"catulezinhens", "catumbiens", "catundens", "catuneens",
"catuniens", "catupiens", "caturaiens", "caturambens",
"caturamens", "caturiaens", "caturiteens", "caubiens",
"caucaiens", "cauipens", "caurens", "cavalcantens",
"cavaleirens", "cavalheirens", "cavalinens", "cavalinhens",
"cavaquens", "caveirens", "cavens", "caxambuens",
"caxanguens", "caxeens", "caxemirens", "caxiens",
"caxitoreens", "cazaqu", "cearens", "cebedens",
"ceboleirens", "ceciens", "cedralens", "cedreirens",
"cedrens", "cedrolandens", "cedrolandiens", "cefalópod",
"ceilonens", "celebrant", "célebr", "celejens",
"celens", "celest", "celinens", "celoricens",
"celsens", "cemoabens", "centenariens", "centralens",
"centralinens", "centroavant", "ceraimens", "ceramicens",
"cercadens", "cercalens", "cerens", "cernachens",
"cerqueirens", "cerquilhens", "cerquinhens", "cerradinhens",
"cerrens", "cerritens", "cerveirens", "cervens",
"cervinhens", "cesariens", "cessant", "ceutens",
"chacarens", "chadiens", "chafarizens", "chaguens",
"chaleens", "chamejant", "champanhens", "chamusquens",
"chancens", "chapadãozens", "chapadens", "chapadinhens",
"chapecoens", "charqueadens", "charruens", "chavalens",
"chavens", "chaviens", "chef", "chegant",
"chelens", "chiadorens", "chiapetens", "chibant",
"chibens", "chilens", "chimarronens", "chiqu",
"chiqueirens", "chiquitens", "chocant", "choninens",
"chopinzinhens", "chorãozens", "choroens", "chorozinhens",
"chorrochoens", "chumbens", "chunens", "cianortens",
"cibaumens", "cicatrizant", "cicladens", "cidreirens",
"cient", "ciganens", "ciliciens", "cilopolandiens",
"cimbrens", "cinaniens", "cinfanens", "cintilant",
"cipoalens", "cipoens", "cipolandens", "circeens",
"circeiens", "circens", "circulant", "circundant",
"circunjacent", "circunstant", "cirtens", "cisneirens",
"cisteciens", "cisterciens", "citaniens", "cizicens",
"claraibens", "clarananens", "claravalens", "clarens",
"clariniens", "classicizant", "claudicant", "claudiens",
"clazomenens", "clementinens", "clevelandens", "client",
"cluniacens", "cluniens", "coadjuvant", "coariens",
"cobard", "cobrens", "cocaens", "cocalinhens",
"cocauens", "cochinchinens", "cochinens", "cocociens",
"codajacens", "codajaens", "codoens", "coeficient",
"coerent", "coexistent", "coimbrens", "coincident",
"coirens", "coiteens", "coivarens", "colant",
"colarens", "colatinens", "coleant", "colegiens",
"colident", "colimbriens", "colinens", "coliponens",
"colombens", "colombiens", "coloniens", "coloradens",
"colorant", "colossens", "colunens", "comanch",
"comandant", "comandatubens", "comanens", "combatent",
"combinadens", "comediant", "comendens", "comerciant",
"comerciens", "comercinhens", "cometens", "comodorens",
"comorens", "comovent", "comparecent", "competent",
"complacent", "complutens", "component", "comprovant",
"comunens", "comunicant", "comurens", "conceiçãonens",
"conceiçãozens", "conceicionens", "conceiçoens", "concepcionens",
"concernent", "concertant", "conchalens", "conchens",
"concludent", "concomitant", "concordant", "concordiens",
"concorrent", "concupiscent", "condadens", "condeixens",
"condescendent", "condeubens", "condicionant", "condizent",
"condorens", "conducent", "conduruens", "conferent",
"confiant", "confident", "confinant", "confinens",
"conflitant", "confluent", "congêner", "congolens",
"congonhalens", "congonhens", "congonhinhens", "congruent",
"conguens", "conimbricens", "conimbrigens", "conivent",
"conquistens", "conradens", "conscient", "consent",
"consequent", "conservant", "consistent", "consoant",
"consort", "constanciens", "constant", "constituint",
"contagens", "contagiant", "contaminant", "contendens",
"content", "contestadens", "continent", "contingent",
"contraent", "contrafort", "contramestr", "contraproducent",
"contrastant", "contratant", "contribuint", "contundent",
"convalescent", "convenient", "conventens", "convergent",
"convincent", "convivent", "cooperant", "copacabanens",
"copatanens", "copenhaguens", "copinalens", "coqueiralens",
"coqueirens", "coquens", "coquinhens", "corant",
"corcirens", "corcovadens", "cordeirens", "cordeiropolens",
"cordens", "cordiform", "cordijesuens", "cordilheirens",
"cordimariens", "cordisburguens", "cordislandens", "cordobens",
"cordofon", "coreauens", "coreens", "coremens",
"corfinens", "corfiniens", "corguinhens", "coribens",
"corinens", "coriniens", "corintens", "coriposens",
"coririens", "coriuvens", "corixens", "corjesuens",
"cormariens", "coroaciens", "coroadens", "coroataens",
"coroens", "coromandelens", "coronelens", "corredeirens",
"corregourens", "correguens", "corrent", "correntens",
"correntinens", "correntinhens", "correspondent", "cortant", "Cort",
"cortesens", "cortonens", "coruchens", "corumbaens",
"corumbaibens", "corumbataiens", "corupaens", "coruripens",
"coruscant", "corvens", "cosmopolens", "cosmoramens",
"cossurens", "costarricens", "costarriquens", "costens",
"cotaxeens", "cotejipens", "cotiens", "cotiporanens",
"courens", "coutens", "coutinhens", "covalent",
"covard", "covilhanens", "covilhens", "covoens",
"coxilhens", "coxinens", "craibens", "craolandens",
"craqu", "crastumiens", "cratens", "crateusens",
"cravinhens", "cravolandens", "cremens", "cremerens",
"cremonens", "crent", "crepitant", "crescent",
"cretens", "criciumalens", "criciumens", "crioulens",
"crisoliens", "crisolitens", "crisopolens", "cristalandens",
"cristalandiens", "cristalens", "cristaliens", "cristalinens",
"cristianopolens", "cristinens", "cristinopolens", "cristovens",
"critens", "criuvens", "crixaens", "crixalandens",
"crixalandiens", "crixasens", "croataens", "crocant",
"croiatens", "crominiens", "crotoniens", "cruanjiens",
"crubixaens", "cruçaiens", "cruçalmens", "cruciant",
"crucilandens", "crucilandiens", "crustuminens", "cruxatiens",
"cruzaliens", "cruzaltens", "cruzeirens", "cruzeirinhens",
"cruzelandens", "cruzelandiens", "cruzens", "cruzetens",
"cruziliens", "cubatanens", "cubatãozens", "cubatens",
"cubatiens", "cubens", "cucuiens", "cuiabaens",
"cuiabens", "cuiambuquens", "cuiariens", "cuieirens",
"cuipiranguens", "cuiranens", "cuiteens", "cuitejiens",
"cuitezeirens", "culminant", "cumariens", "cumaruens",
"cumbens", "cúmplic", "cumuruxatibens", "cunaniens",
"cundinamarquens", "cunduruens", "cunhambebens", "cunhanjiens",
"cunhens", "cunquens", "cuparaquens", "cupinens",
"cupirens", "curaçaens", "curariaens", "curariens",
"curarizinhens", "curatiens", "curens", "curet",
"curicuriariens", "curimataens", "curimataiens", "curionopolens",
"curitibanens", "curitibens", "curitubens", "curiuvens",
"curralens", "curralinhens", "cursant", "curuaens",
"curuaiens", "curuataiens", "curucaens", "curuçaens",
"curuçambabens", "curuens", "curumuens", "curupaens",
"curupaitiens", "curupirens", "cururipens", "cururupuens",
"curuzuens", "curvelens", "custodiens", "cutiens",
"cutilant", "cutiporanens", "dacarens", "damanens",
"dançant", "debilitant", "debutant", "decadent",
"decápod", "decent", "decepcionant", "declarant",
"declinant", "decliv", "decorrent", "decrescent",
"deferent", "deficient", "degradant", "deliberant",
"delinquent", "deliquescent", "delirant", "deltoid",
"demandant", "dement", "democratizant", "denunciant",
"dependent", "depoent", "depositant", "depriment",
"derribadinhens", "derrubadens", "desamparadens", "desarmant",
"desbordant", "descalvadens", "descampadens", "descansens",
"descarretens", "descendent", "descobertens", "desconcertant",
"descongestionant", "descontent", "descrent", "deselegant",
"desemboquens", "desencorajant", "desequilibrant", "desertens",
"desfigurant", "desfilant", "desfolhant", "desgastant",
"desideriens", "desidratant", "designiens", "desinfectant",
"desinfetant", "desinteressant", "deslizant", "deslumbrant",
"desobedient", "desodorant", "desolant", "desoxidant",
"despachant", "desparasitant", "despenhadens", "dessemelhant",
"destoant", "desviant", "desviens", "detectiv",
"detergent", "determinant", "detetiv", "detonant",
"devassant", "devorant", "diademens",
"dialogant", "diamantens", "diamantinens", "dianens",
"diasens", "dibionens", "diceriens", "diens",
"diferent", "dignificant", "dilacerant", "diletant",
"diligent", "diluent", "dimanant", "diniens",
"dinisiens", "diocleciens", "diodens", "diodorens",
"dioens", "diolandens", "dionisiens", "dioramens",
"diplostêmon", "dirceuens", "dirigent", "diriment",
"discent", "discernent", "discordant", "discrepant",
"discriminant", "discursant", "disform", "displicent",
"disputant", "dissemelhant", "dissident", "dissolvent",
"dissonant", "distant", "diuens", "divagant",
"divalent", "divergent", "divinens", "divinesiens",
"divinolandens", "divionens", "divisens", "divisopolens",
"divitens", "djibutiens", "doberiens", "dobradens",
"dobradinhens", "doc", "docent", "doent", "doidejant",
"doiradens", "dolcinopolens", "dolearinens", "dolent",
"domeliens", "dominant", "domingadens", "dominguens",
"dominiquens", "dopant", "dorandiens", "dorens",
"dorilens", "dorizonens", "dorment", "dormidens",
"doudejant", "douradinhens", "douradoquarens", "dresdens",
"duartens", "duartinens", "dublinens", "duereens",
"dumiens", "dumontens", "dumontinens", "duobarrens",
"duocorreguens", "duplavilens", "dúplic", "durandeens",
"duriens", "dutrens", "eandens", "ebaubens",
"eboracens", "eborens", "eburodunens", "echaporens",
"ecoporanguens", "edealinens", "edeens", "edeiens",
"edenens", "edificant", "edulcorant", "eferent",
"efervescent", "eficient", "efluent", "egesteniens",
"egidiens", "egiens", "eginens", "eginiens",
"egiptanens", "egiptens", "egitanens", "egitaniens",
"eirens", "eirunepeens", "elatens", "elbens",
"eldoradens", "eleens", "elegant", "elesbanens",
"elesbonens", "eletrens", "eleuteriens", "elidens",
"eliens", "elisiariens", "eloiens", "eloquent",
"elusens", "elvens", "ematubens", "emazens",
"embaraiens", "embiralens", "embiruçuens", "emboabens",
"embriagant", "embuens", "embuguaçuens", "emburradens",
"emens", "emergent", "emeritens", "emidiens",
"emigrant", "emiliens", "eminent", "emitent",
"emocionant", "emolient", "emparedadens", "empestant",
"empoçadens", "empolgant", "emulsionant", "encaixant",
"encantadens", "encantens", "encarnacionens", "enchentens",
"encruzilhadens", "endossant", "eneidens", "enemutuens",
"enens", "enervant", "enganens", "engenhens",
"engraxat", "eniens", "enorm", "enseadens",
"ensiform", "entebens", "entorpecent", "entradens",
"entredent", "entrefolhens", "entrement", "entrepeladens",
"entrerriens", "entroncamentens", "entusiasmant", "envirens",
"envolvent", "enxuens", "eordens", "epidamniens",
"epifanens", "epirens", "epitaciens", "eporens",
"equadorens", "equânim", "equestr", "equetliens",
"equidistant", "equivalent", "erbessens", "ercavicens",
"erebanguens", "eremitens", "erereens", "eretriens",
"erexinens", "ergavicens", "ericeirens", "ermens",
"ermensidens", "ermesindens", "ermidens", "ernaginens",
"ernestinens", "errant", "ervalens", "ervaliens",
"erveirens", "esbravejant", "escadens", "escaldant",
"escalvadens", "escamiform", "escapant", "escapciens",
"escodrens", "escondidens", "escorchant", "escouralens",
"escrevent", "escurialens", "esfuziant", "esgotant",
"esmeraldens", "esmerilens", "esminens", "esnob",
"espectant", "espelhant", "esperancens", "esperantinens",
"esperens", "espessant", "espinescent", "espinhens",
"espinilhens", "espinosens", "espinozens", "esplanadens",
"esplendent", "espongitiens", "esponjitiens", "esposendens",
"espumant", "espumosens", "estabilizant", "estacadens",
"estacielens", "estadunidens", "estafant", "estalant",
"estanciens", "estandartens", "estarrejens", "estatizant",
"esteens", "estefaniens", "esteiens", "esteirens", "estep",
"esterilizant", "estilhaçant", "estimulant", "estivens",
"estobens", "estonteant", "estorilens", "estradens",
"estratonicens", "estreant", "estrebuchant", "estreitens",
"estrelens", "estrem", "estremens", "estremocens",
"estremozens", "estrepitant", "estressant", "estrident",
"estrombiens", "estruturant", "estuant", "estudant",
"estupefacient", "esvoaçant", "etíop", "eucariont",
"euclidens", "euromens", "eusebiens", "euxenitens",
"evanescent", "evangelistens", "evident", "exalant",
"exaltant", "exaporanens", "exasperant", "excedent",
"excelent", "excitant", "excludent", "excrescent",
"excruciant", "executant", "exercent", "exigent",
"existent", "exorbitant", "expectorant", "expedient",
"expeditens", "experient", "expirant",
"expoent", "exponent", "extasiant", "extenuant",
"extracionens", "extraterrestr", "extravagant", "extremens",
"exuberant", "exuens", "exultant", "ezequielens",
"fabiens", "fabricant", "fabricens", "fabricianens",
"facheant", "fadistens", "fafens", "fagulhant",
"fagundens", "faialens", "fainens", "faiscant",
"falant", "falciform", "falconiform", "faleriens",
"falerionens", "faltant", "famalicens", "famens",
"fãozens", "farejant", "farens", "farfalhant",
"fariens", "farinhens", "fariscant", "farnient",
"faroens", "farolens", "farpant", "farrapens",
"farroupilhens", "farsant", "farturens", "fascinant",
"fatigant", "fatimens", "faustianens", "faustiniens",
"favalens", "favaniens", "faveirens", "favoniens",
"faxinalens", "fazendens", "febricitant", "fecundant",
"feijoens", "feirant", "feirens", "feitoriens",
"feitosens", "felgarens", "felgueirens", "felicens",
"felipens", "felisburguens", "felixlandens", "felizens",
"feltriens", "felup", "fenaens", "fenicens",
"ferent", "ferentinens", "fermedens", "fermentelens",
"fernandopolens", "feroc", "ferrazens", "ferreirens",
"ferrens", "ferruginhens", "fertilizant", "fervedourens",
"fervent", "fervilhant", "ficant", "ficobiont",
"ficulens", "fidelandens", "fidelens", "figueirens",
"figueiroens", "figurant", "filadelfens", "filadelfiens",
"filant", "filipens", "filomeliens", "filomenens",
"filtrant", "finciens", "finens", "firmens",
"firminens", "fissíped", "fixant", "flagelant",
"flageliform", "flagrant", "flamant", "flamejant",
"flamonens", "flamoniens", "flandrens", "flaviens",
"flechalens", "flecheirens", "flexeirens", "floraiens",
"floraliens", "floranens", "floraniens", "florarriquens",
"florealens", "florejant", "florencens", "florens",
"florent", "florentinens", "florescent", "florestalens",
"florestens", "florestinens", "florianens", "floridens",
"florineens", "florinens", "floriniens", "fluent",
"fluminens", "fluorescent", "flutuant", "foceens",
"focens", "folquens", "fondu", "font",
"fontourens", "foralens", "forens", "formigant",
"formigueirens", "formiguens", "formosens", "fornens",
"forquilhens", "forquilinhens", "fortalexiens", "fortalezens",
"fort", "fortens", "fortificant", "fortunens",
"foscoens", "fosfolipas", "fosfolípas", "fosfolípid",
"fosforescent", "fotossintetizant", "fradens", "fragosens",
"fragrant", "fraiburguens", "francens", "francisquens",
"franquens", "fraquejant", "fravasonens", "frecheirens",
"frecheirinhens", "freitens", "freixiens", "frement",
"frequent", "friburguens", "frisant", "frondent",
"fronteant", "fronteirens", "frustrant", "frutalens",
"fucens", "fugent", "fuldens", "fulgent",
"fulgurant", "fulminant", "fumant", "fumegant",
"funchalens", "funcionant", "fundanens", "fundãozens",
"fundens", "fundent", "fundonens", "fúnebr",
"fungiform", "furant", "furent", "furfonens",
"furnens", "fusiform", "fustigant", "fuzilant",
"fuzilens", "gabiarrens", "gabiens", "gabirobens",
"gabonens", "gabrielens", "gadens", "gaguejant",
"gaiabalens", "gaiens", "gaivotens", "galant",
"galantens", "galdinens", "galeanens", "galenens",
"galens", "galheirens", "galiens", "galileens",
"galinens", "galinhens", "galopant", "galvanens",
"gamaleirens", "gambiens", "gamboens", "gameleirens",
"gamelens", "gamens", "ganchens", "gandariens",
"gandhiens", "gandrens", "ganduens", "ganens",
"gangorrens", "ganguens", "gantens", "garajubens",
"garanhuens", "garanhunens", "garapuavens",
"garapuens", "gararuens", "garcens", "garcezens",
"garciens", "gardeniens", "gardoniens", "gargantens",
"gargasiens", "gargauens", "garibaldens", "garibaldinens",
"garimpens", "garopabens", "garrafaens", "garrotens",
"garruchens", "garuvens", "gasparens", "gaspariens",
"gastrópod", "gatens", "gauchens", "gauerens",
"gaupoens", "gauramens", "gavianens", "gavionens",
"gazeens", "gazens", "gefirens", "gelens",
"gemelens", "gement", "genabens", "genebrens",
"generalizant", "generosens", "genovens", "genuens",
"genuinens", "geolandens", "geometrizant", "georginens",
"gerent", "germaniens", "geront", "gertrudens",
"gestant", "getulandens", "getuliens", "getulinens",
"gezehens", "gigant", "gigantens", "gilbueens",
"gilens", "girant", "girvasens", "glaucilandens",
"glaucilandiens", "glaurens", "glicerens", "gliceriens",
"globiform", "gloriens", "glorinhens", "gluciniens",
"godinens", "godoens", "godoiens", "goens",
"goiabalens", "goiabeirens", "goianaens", "goianasens",
"goiandirens", "goianens", "goianesiens", "goianiens",
"goianinhens", "goianirens", "goianortens", "goiasens",
"goiatinens", "goiatubens", "goiens", "goipabens",
"goitacasens", "goitaens", "goleganens", "golpeant",
"gonçalens", "gonçalvens", "gondidens", "gondirens",
"gondomarens", "gonfens", "gongojiens", "gonzaguens",
"goroboens", "gorutubens", "gotardens", "gotejant",
"goticizant", "goulartens", "gouveens", "governant",
"gracens", "graciens", "graciosens", "graçuiens",
"gradauens", "gradient", "grajauens", "gramachens",
"gramadens", "gramadinhens", "gramalens", "gramamens",
"gramens", "graminens", "graminhens", "granadens",
"granatens", "grand", "grandens", "grandiloquent",
"grandolens", "granitens", "granjeirens", "granjens",
"granzatens", "gratificant", "gravataens", "gravataiens",
"gravatalens", "grav", "gregoriens", "gremiens",
"grijoens", "grilant", "grilens", "grinincens",
"gritant", "groairens", "grogu", "grossens",
"grotanens", "grotens", "gruiform", "guabijuens",
"guabirabens", "guabirubens", "guachens", "guaciens",
"guacoceens", "guaçuboiens", "guaçuens", "guaçuiens",
"guadalupens", "guaiacens", "guaianaens", "guaianasens",
"guaibens", "guaibinens", "guaiçarens", "guaicuiens",
"guaiens", "guaimbeens", "guaipavens", "guaiquicens",
"guairacaens", "guairaçaens", "guairens", "guaiubens",
"guajaaens", "guajaraens", "guajaratubens", "guajaratubinhens",
"guajeruens", "guajuvirens", "guamaens", "guamaranguens",
"guamareens", "guampadens", "guanabarens", "guanaceens",
"guanambiens", "guananasens", "guanandiens", "guanduens",
"guanhanens", "guanumbiens", "guapeens", "guapiaçuens",
"guapiarens", "guapimiriens", "guapiramens", "guapiranguens",
"guapirens", "guapoens", "guaporanguens", "guaporeens",
"guaporemens", "guaporens", "guapuanens", "guarabirens",
"guaraçaiens", "guaraciabens", "guaraciamens", "guaraciens",
"guaraens", "guaraiens", "guaraipens", "guaraitubens",
"guarajiens", "guaramiranguens", "guaramiriens", "guaramirinens",
"guaranesiens", "guaraniaçuens", "guaraniens", "guarantãens",
"guarantanens", "guarapariens", "guarapiranguens", "guarapuanens",
"guarapuavens", "guarapuavinhens", "guaraqueçabens", "guararaens",
"guararapens", "guararemens", "guararens", "guararuens",
"guarataiens", "guaratinguens", "guaratinguetaens", "guaratubens",
"guarauens", "guaraunens", "guaraverens", "guaraxainens",
"guardens", "guardinhens", "guareiens", "guarens",
"guaribens", "guaricanguens", "guariguariens", "guarinens",
"guariramanens", "guarirobens", "guaritens", "guarizinhens",
"guartelens", "guaruens", "guarujaens", "guarulhens",
"guataens", "guatambuens", "guataparaens", "guatemalens",
"guavirens", "guaxens", "guaxiens", "guaximens",
"guaxindibens", "guaxupeens", "guedens", "guianens",
"guidens", "guidovalens", "guiens", "guimaranens",
"guimaraniens", "guindant", "guineens", "guiomaens",
"guirapaens", "guiratinguens", "guiricemens", "guizalhant",
"gulosens", "gundens", "gundianens", "guraitubens",
"gurinhatãens", "gurinhatanens", "gurinheens", "gurinhenens",
"guriuens", "gurjanens", "gurjãoens", "gurupaens",
"gurupiens", "guzolandens", "habitant", "halicarnassens",
"haliciens", "halidiens", "hamaniens", "hamburguens",
"hamondens", "haplostêmon", "harmoniens", "havaiens",
"hebridens", "heimatens", "heitoraiens", "helenens",
"heliodorens", "heliopoliens", "helmodens", "helveciens",
"hematitens", "hemonens", "heracleens", "heracliens",
"herbitens", "herbulens", "herciniens", "herculandens",
"herculandiens", "herculanens", "herdoniens", "hereg",
"herviens", "hesitant", "hespérid", "hexacord",
"hiant", "hibernant", "hiblens", "hidissens",
"hidratant", "hidrolandens", "hidrolinens", "hienipens",
"hierapolens", "hierolofiens", "hilariant", "hinterlandiens",
"hipoglicemiant", "hiponens", "hippi", "hipsodont",
"hispalens", "hispaliens", "hispaniens", "histoniens",
"holandens", "hollywoodens", "holmiens", "holofotens",
"homenageant", "homogeneizant", "homonadens", "homotransplant",
"hondurens", "horaciens", "horens", "horizontens",
"horizontinens", "hormens", "horripilant", "horrorens",
"horrorizant", "hortens", "hortolandens", "hortolandiens",
"hósped", "hospiciens", "host", "hostiens", "hostiliens",
"hotentot", "hugonapolens", "huguenot", "huitanaanens",
"humaitaens", "humbertuens", "humild", "humildens",
"iacaiaens", "iacanguens", "iaciarens", "iacriens",
"iaçuens", "ianatamens", "iandareens", "ianqu",
"iapiens", "iapuens", "iaqu", "iaquiranens",
"iaramontens", "iarens", "iaripens", "iassens",
"iatecaens", "iatiens", "iauacanaens", "iauareteens",
"iaunariens", "iavaritens", "ibaitiens", "ibaramens",
"ibareens", "ibaretamens", "ibateens", "ibateguarens",
"ibatibens", "ibatuiens", "ibemens", "ibertioguens",
"ibiaçaens", "ibiaciens", "ibiaçuceens", "ibiaçuens",
"ibiaens", "ibiaguiens", "ibiaiens", "ibiajarens",
"ibianens", "ibiapabens", "ibiapinens", "ibiaporanens",
"ibiarens", "ibicabens", "ibicaraiens", "ibicareens",
"ibicatuens", "ibicuanens", "ibicuiens", "ibicuitabens",
"ibicuitinguens", "ibiens", "ibimirinens", "ibipebens",
"ibipetubens", "ibipetunens", "ibipirens", "ibipitanguens",
"ibiporanens", "ibiporanguens", "ibiquarens", "ibiquerens",
"ibirabens", "ibiracatuens", "ibiraciens", "ibiraçuens",
"ibiraens", "ibiraiarens", "ibirajaens", "ibirajubens",
"ibiramens", "ibiranguens", "ibiranhenens", "ibirapaens",
"ibirapitanguens", "ibirapuãens", "ibirapuitanens", "ibiraremens",
"ibirataiens", "ibiratinguens", "ibiriteens", "ibirocaiens",
"ibirubaens", "ibirubens", "ibitiarens", "ibitiguaçuens",
"ibitiguaiens", "ibitiguirens", "ibitinguens", "ibitiocens",
"ibitioquens", "ibitipocens", "ibitipoquens", "ibitiramens",
"ibitiranguens", "ibitirens", "ibitirunens", "ibititaens",
"ibitiurens", "ibitiuvens", "ibitubens", "ibituens",
"ibitunanens", "ibitupanens", "ibituporanguens", "ibiturunens",
"ibiunens", "iboens", "iborepiens", "ibotiramens",
"ibuaçuens", "ibuguaçuens", "icaiçarens", "icaiçarinhens",
"icapoaens", "icapuiens", "icaraiens", "icaraimens",
"icariens", "icatuens", "icemens", "icenens",
"icens", "icenzens", "icoens", "iconhens",
"iconiens", "icoraciens", "icozeirens", "icozinhens",
"idanhens", "idealens", "ideologizant", "idiolandens",
"idomenens", "iemenens", "ienisseens", "ienisseiens",
"iepeens", "iepens", "iersonens", "ietens",
"igaçabens", "igaciens", "igapoens", "igaporanens",
"igaraçuens", "igaraiens", "igarapavens", "igarapebens",
"igarapeens", "igarassuens", "igarataens", "igaratens",
"igaratinguens", "igarauens", "igarens", "igariteens",
"igaroiens", "igatiens", "igatiquirens", "igatuens",
"ignorant", "igrapiunens", "igrejens", "igrejinhens",
"iguabens", "iguaçuens", "iguaens", "iguaibens",
"iguaiens", "igualdadens", "iguapens", "iguaraciens",
"iguaraçuens", "iguatamens", "iguatemiens", "iguatiens",
"iguatuens", "iguirens", "iguituens", "ijaciens",
"ijucapiramens", "ijuiens", "ildefonsens", "ilercaonens",
"ilerdens", "ilergavonens", "ilhabelens", "ilhavens",
"ilheens", "ilhenens", "ilheuens", "ilhotens",
"iliberitens", "ilicens", "iliciniens", "iliens",
"ilipens", "ilipulens", "iliturgens", "ilurconens",
"ilurgavonens", "iluronens", "ilustr", "imacarens",
"imaculadens", "imanent", "imaruiens", "imbarieens",
"imbaubens", "imbauvalens", "imbauzinhens", "imbeens",
"imbel", "imberb", "imbiruçuens", "imbitubens",
"imbituvens", "imboiarens", "imbuens", "imbuguaçuens",
"imbuiens", "imbuirens", "imburanens", "imbuzalens",
"imigrant", "iminent", "imitant", "impacient",
"impactant", "impant", "impenitent", "imperant",
"imperatricens", "imperatrizens", "imperiens", "impermanent",
"impermeabilizant", "impersistent", "impertinent", "impetrant",
"implant", "implicant", "implorant", "imponent",
"important", "impotent", "impressionant", "imprevident",
"improcedent", "imprudent", "impúber", "impudent",
"impun", "imun", "imunizant", "imunocompetent",
"imunodeficient", "inaciens", "inadimplent", "inajaens",
"inalant", "inambuens", "inânim", "inaparent",
"incandescent", "incapacitant", "incessant", "incident",
"incipient", "inclement", "incoerent", "incólum",
"incombent", "incompetent", "inconcludent", "inconfidenciens",
"inconfident", "inconfidentens", "incongruent", "inconscient",
"inconsequent", "inconsistent", "inconstant", "incontinent",
"inconvenient", "indagant", "indaiabirens", "indaiaens",
"indaialens", "indaiatubens", "indaiens", "indecent",
"indeiscent", "independenciens", "independent", "indianopolens",
"indiaporanens", "indiarobens", "indiferent", "indigent",
"indignificant", "individualizant", "indolent", "indulgent",
"industrialens", "inebriant", "ineficient", "inerent",
"inerm", "inert", "inesens", "inexistent",
"inexperient", "infamant", "infam", "infant",
"infectant", "infestant", "influent", "informant",
"infren", "infringent", "ingaens", "ingazeirens",
"ingent", "ingratens", "ingredient", "ingressant",
"inhacoraens", "inhaiens", "inhambupens", "inhamunens",
"inhanduiens", "inhangapiens", "inhapiens", "inhapinens",
"inhapinhens", "inhataens", "inhatiunens", "inhaumens",
"inhobinens", "inhomirinens", "inhuçuens", "inhumens",
"inhuporanguens", "iniciant", "inimatubens", "inimutabens",
"inoanens", "inobservant", "inocenciens", "inocent",
"inoens", "inoperant", "inquietant", "insalubr",
"inscient", "insign", "insignificant", "insinuant",
"insipient", "insistent", "insolent", "insolvent",
"inson", "instant", "instigant", "insubsistent",
"insuficient", "insultant", "insurgent", "intanens",
"integrant", "inteligent", "intemperant", "intendenciens",
"intendent", "interagent", "interamnens", "intercaciens",
"intercadent", "intercedent", "intercomunicant", "intercruzant",
"interdependent", "interessant", "interferent", "interfoliens",
"interlandens", "interlandiens", "intermitent", "intérpret",
"intervenient", "intolerant", "intoxicant", "intransigent",
"intrigant", "intumescent", "inubiens", "invaginant",
"invariant", "invernadens", "iogu", "iomereens",
"ionizant", "ipabens", "ipaguaçuens", "ipameriens",
"ipamorotiens", "ipandreens", "ipaneens", "ipanemens",
"ipanguaçuens", "ipaporanguens", "ipargens", "ipatinguens",
"ipauçuens", "ipaumirinens", "ipaunens", "ipecaetaens",
"ipeçuens", "ipeens", "ipeguens", "iperabens",
"iperoens", "ipeuatens", "ipeunens", "ipiabens",
"ipiauens", "ipiauiens", "ipiguaens", "ipiibens",
"ipiracens", "ipiraens", "ipiranguens", "ipiraquens",
"ipitanguens", "ipiunens", "ipixunens", "ipoemens",
"ipoiens", "ipojucens", "ipojuquens", "ipomeens",
"ipomeiens", "iporaens", "iporanens", "iporanguens",
"iporcens", "iprens", "ipuaçuens", "ipuanens",
"ipubiens", "ipuçabens", "ipucens", "ipueirens",
"ipuens", "ipuiunens", "ipumirinens", "ipupiarens",
"ipuquens", "iracemapolens", "iracemens", "iraceminhens",
"iraciens", "iraguaçuens", "iraiens", "iraitiens",
"irajaens", "irajaiens", "irajubens", "iramaiens",
"iraniens", "irapeens", "iraporanguens", "irapuaens",
"irapuanens", "irapuens", "irapuiens", "irapuruens",
"iraputãzens", "iraquarens", "iraquitanens", "iraraens",
"irarens", "iratamens", "iratiens", "iratinguens",
"irauçubens", "ireceens", "irereens", "iretamens",
"iriant", "iridescent", "iriens", "iriranens",
"iririens", "iririteuens", "iriritibens", "iritreens",
"iritreiens", "irituiens", "iroiens", "irradiant",
"irrelevant", "irreverent", "irritant", "irundiarens",
"irundiens", "irupiens", "isabelens", "isaiens",
"isidorens", "isindens", "isiodens", "isiondens",
"islandens", "isolant", "isópod", "isóscel",
"isostêmon", "isotransplant", "israelandens", "israelens",
"issens", "itaarens", "itabaianens", "itabaianinhens",
"itabapoanens", "itaberabens", "itaberaens", "itaberaiens",
"itaberens", "itabiens", "itabirens", "itabirinhens",
"itabiritens", "itabocens", "itaboquens", "itaboraiens",
"itabunens", "itacajaens", "itacambirens", "itacarambiens",
"itacareens", "itacatuens", "itacavens", "itacens",
"itaciens", "itaciramens", "itacoatiarens", "itacolomiens",
"itaçuceens", "itaçuens", "itacurubens", "itacuruçaens",
"itaens", "itaeteens", "itaguaçuens", "itaguaens",
"itaguaiens", "itaguajeens", "itaguandibens", "itaguarens",
"itaguaruens", "itaguatinens", "itaguirens", "itaiacoquens",
"itaiaens", "itaianopolens", "itaibens", "itaiboens",
"itaiçabens", "itaiciens", "itaiens", "itaimbeens",
"itaimbezinhens", "itainens", "itainopolens", "itaiococens",
"itaioens", "itaiopolens", "itaipabens", "itaipavens",
"itaipeens", "itaipuens", "itaitubens", "itaituens",
"itaiubens", "itaiuens", "itajaçuens", "itajaens",
"itajaiens", "itajarens", "itajibens", "itajiens",
"itajimirinens", "itajobiens", "itajubaens", "itajubaquarens",
"itajubatibens", "itajubens", "itajuens", "itajuipens",
"itajuruens", "itajutibens", "italegrens", "italicens",
"italvens", "itamanariens", "itamaracaens", "itamarajuens",
"itamarandibens", "itamarateens", "itamaratiens", "itamariens",
"itamataiens", "itambacuriens", "itambaracaens", "itambeens",
"itambiens", "itamirens", "itamirinens", "itamojiens",
"itamontens", "itamotiaguens", "itamuriens", "itanagrens",
"itanajeens", "itanens", "itanguaens", "itanhaenens",
"itanhaens", "itanhanduens", "itanheens", "itanhenens",
"itanhiens", "itanhomiens", "itaobinens", "itaocarens",
"itaocens", "itaoquens", "itapaçarocens", "itapaçaroquens",
"itapaciens", "itapacuraens", "itapajeens", "itapajipens",
"itapanhoacanguens", "itaparaens", "itaparicens", "itapariquens",
"itapebiens", "itapebuçuens", "itapecericens", "itapeceriquens",
"itapecuruens", "itapecurumirinens", "itapeens", "itapeinens",
"itapeipuens", "itapejarens", "itapemens", "itapemirinens",
"itapequaens", "itaperens", "itaperiuens", "itaperunens",
"itapeteiuens", "itapetinens", "itapetinguens", "itapetinguiens",
"itapetininguens", "itapeuaens", "itapeunens", "itapevens",
"itapeviens", "itapiçumens", "itapicuruens", "itapicurumirinens",
"itapinens", "itapinimens", "itapipocens", "itapipoquens",
"itapiranguens", "itapirapuanens", "itapiraremens", "itapirens",
"itapiruçuens", "itapiruens", "itapirunens", "itapitocaiens",
"itapiunens", "itapixeens", "itapixunens", "itapoaens",
"itapoens", "itapolens", "itaporanens", "itaporanguens",
"itaporeens", "itapororocens", "itapororoquens", "itapuanens",
"itapucens", "itapucuens", "itapulens", "itapuquens",
"itapuranguens", "itapurens", "itaquaens", "itaquaquecetubens",
"itaquaraiens", "itaquarens", "itaquariens", "itaquatiarens",
"itaquerens", "itaquiens", "itaquirens", "itaquitinguens",
"itaranens", "itarantinens", "itarareens", "itaremens",
"itaririens", "itaroqueens", "itaroquenens", "itarumanens",
"itarumens", "itassuceens", "itataens", "itatiaiens",
"itatiaiuçuens", "itatibens", "itatiens", "itatinens",
"itatinguens", "itatinguiens", "itatirens", "itatubens",
"itatupanens", "itaubelens", "itaubens", "itauçuens",
"itaueirens", "itauens", "itaunens", "itauninhens",
"itauzens", "itaveraens", "itaveravens", "itaxamens",
"iterereens", "itibirens", "itinerant", "itinguens",
"itiquirens", "itirapinens", "itirapuanens", "itirens",
"itiruçuens", "itiubens", "itiucens", "itiuquens",
"itobiens", "itororoens", "itoupavens", "ituaçuens",
"ituberaens", "ituens", "ituetens", "ituguaçuens",
"ituiens", "ituinens", "ituiutabens", "itumbiarens",
"itumirinens", "itupararanguens", "itupavens", "itupevens",
"itupiranguens", "ituporanguens", "ituquiens", "ituramens",
"itutinguens", "ituveravens", "iuaens", "iubatinguens",
"iucaiens", "iucatanens", "iuiraens", "iuiteporanens",
"iuiuens", "iunens", "iuniens", "iunzens",
"iupebens", "iuruens", "iuticens", "iutiquens",
"ivaiens", "ivaiporanens", "ivaitinguens", "ivatens",
"ivatubens", "ivinheimens", "ivitiens", "ivoiriens",
"ivolandens", "ivoraens", "ivotiens", "ixuens",
"ixuzinhens", "izarens", "jabaeteens", "jabaquarens",
"jabitacaens", "jaboatanens", "jaboatãozens", "jaboraens",
"jaborandiens", "jaboticabalens", "jabotiens", "jaburizinhens",
"jaburuens", "jaburunens", "jabuticabalens", "jabuticabeirens",
"jabuticabens", "jabuticatubens", "jabutiens", "jacamaens",
"jacampariens", "jaçanens", "jacaquaens", "jacaraciens",
"jacaraipens", "jacarandaens", "jacarandirens", "jacarauens",
"jacareacanguens", "jacarecicens", "jacareciens", "jacareciquens",
"jacareens", "jacareiens", "jacarequarens", "jacaretinguens",
"jacarezinhens", "jacatanens", "jacaunens", "jaceabens",
"jacent", "jacerendiens", "jacerubens", "jaciabens",
"jaciarens", "jaciens", "jaciguaens", "jacintens",
"jaciobens", "jaciparanaens", "jaciporanens", "jaciremens",
"jacitarens", "jacksoniens", "jacobinens", "jaconeens",
"jacruaruens", "jacsoniens", "jacubens", "jacuecanguens",
"jacuens", "jacuiens", "jacuipens", "jacuizinhens",
"jacumanens", "jacundaens", "jacundazinhens", "jacupembens",
"jacupiranguens", "jacuriciens", "jacuriens", "jacurunens",
"jacururuens", "jacurutuens", "jacutinguens", "jadibaruens",
"jaeteuens", "jafens", "jagod", "jaguapitanens",
"jaguaquarens", "jaguaraciens", "jaguaraçuens", "jaguaraens",
"jaguaranens", "jaguarariens", "jaguareens", "jaguarembeens",
"jaguarens", "jaguaretamens", "jaguariaivens", "jaguaribarens",
"jaguaribens", "jaguaricatuens", "jaguariens", "jaguarinhens",
"jaguaripens", "jaguaritens", "jaguaritirens", "jaguariunens",
"jaguaronens", "jaguaruanens", "jaguarunens", "jaguatiricens",
"jaguatiriquens", "jaibarens", "jaibens", "jaicoens",
"jald", "jalens", "jalesens", "jamacaruens",
"jamaicens", "jamaparaens", "jamaparens", "jamariens",
"jamaxinzinhens", "jambeirens", "jambruquens", "jambuaçuens",
"jamprucens", "jampruquens", "janaubens", "jandaiens",
"jandairens", "jandirens", "jandiritiuens", "janduiens",
"janduissens", "jangadens", "janguens", "jansenens",
"jantant", "januarens", "januariens", "japaraibens",
"japaratinguens", "japaratubens", "japecanguens", "japeinens",
"japericens", "japeriens", "japeriquens", "japiens",
"japiinens", "japiinzens", "japirens", "japiubens",
"japoatanens", "japonens", "japuatanens", "japueens",
"japuens", "japuibens", "japumirinens", "japuraens",
"jaqueirens", "jaquiranens", "jaracatiaens", "jaraguaens",
"jaraguariens", "jaraguens", "jaramataiens", "jaraquiens",
"jararacens", "jararaquens", "jardimirinens", "jardineirens",
"jardinens", "jardinesiens", "jardinopolens", "jariens",
"jarinuens", "jarmelens", "jarudorens", "jaruens",
"jassens", "jataiens", "jataizinhens", "jataubens",
"jatiens", "jatimanens", "jatiucens", "jatiuquens",
"jatobaens", "jatobalens", "jatobazinhens", "jatuaranens",
"jauarauens", "jauariens", "jauatoens", "jaudilandens",
"jaudilandiens", "jauens", "jaupaciens", "jauquarens",
"jauruens", "jauzinhens", "javariens", "javens",
"jazent", "jeceabens", "jeitosens", "jejuiens",
"jenipabuens", "jenipapeirens", "jenipapens", "jenipaubens",
"jequeriens", "jequieens", "jequiens", "jequiezinhens",
"jequiriçaens", "jequiricens", "jequitaiens", "jequitibaens",
"jequitinhonhens", "jeremiens", "jeremoabens", "jeremunens",
"jeremunzens", "jererauens", "jeriaçuens", "jeribaens",
"jeribatubens", "jericoaquarens", "jericoens", "jerimatalhens",
"jeriquaquarens", "jeriquarens", "jeritacoens", "jerivaens",
"jeroaquarens", "jeromenhens", "jeronimens", "jersonens",
"jeruens", "jerumenhens", "jerumirinens", "jesuanens",
"jesuaniens", "jesuens", "jesuitens", "jetens",
"jetibaens", "jiboiens", "jijuens", "jiloens",
"jinimbuens", "jiparanaens", "jipubens", "jiquiriçaens",
"jiquitaiens", "jirauens", "jiribatubens", "jiruaens",
"jiruens", "jitaunens", "jitiranens", "joaçabens",
"joaçubens", "joaimens", "joan", "joanens",
"joanesens", "joanesiens", "joanetens", "joanicens",
"joaninens", "joanopolens", "joaquimens", "joaquinens",
"joarizinhens", "joatubens", "joeiranens", "joiens",
"joinvilens", "jojocens", "jojoquens", "joniens",
"jordanens", "jordaniens", "jordãozinhens", "jorgens",
"joseens", "josefens", "joselandens", "joselandiens",
"jovianiens", "juabens", "juaçabens", "juacemens",
"juaciens", "juaens", "juatamens", "juatubens",
"juazeirens", "juazeirinhens", "juazens", "jubaiens",
"jubilant", "jubirens", "jucaens", "juçaralens",
"juçarens", "juçariens", "jucatiens", "juciapens",
"jucuens", "jucumanens", "jucumpemens", "jucuruçuens",
"jucururiens", "jucururuens", "jucurutuens", "judaizant",
"jueranens", "juiraçuens", "julianens", "julianiens",
"juliens", "juliobrigens", "jumens", "jumirinens",
"juncens", "juncionens", "jundiaens", "jundiaiens",
"jundiapebens", "junens", "junqueirens", "junqueiropolens",
"junquens", "jupaguaens", "juparens", "juparitenens",
"jupiaens", "jupiens", "jupiranguens", "juquiaens",
"juquiquerens", "juquiratibens", "juquitibens", "juraciens",
"juramentens", "jurandens", "jurapaitens", "juraraens",
"jureens", "juremalens", "juremens", "juriciens",
"juripiranguens", "juritianhens", "juritiens", "juruaens",
"juruaiens", "jurubebens", "juruceens", "juruenens",
"juruens", "jurumiriens", "jurupariens", "jurupebens",
"jurupemens", "jurutiens", "juruvalhens", "juscelandens",
"juscelandiens", "juscelinens", "jutaiens", "juticens",
"jutiens", "jutiquens", "jutlandens", "jutlandiens",
"juturnaibens", "juvenciens", "juventinens", "labreens",
"lacens", "lacerant", "lacerdens", "laciens",
"laciniens", "lacobricens", "lacobrigens", "lacrimejant",
"lactant", "lactent", "lacustr", "lacustrens",
"ladainhens", "ladarens", "ladariens", "ladeirens",
"laetiens", "lafaietens", "lafonens", "lagamarens",
"lagartens", "lagoadens", "lagoanens", "lagoens",
"lagoinhens", "lagonens", "lagópod", "laguens",
"laguinhens", "lagunens", "lagunitens", "lajeadens",
"lajeadinhens", "lajedãoens", "lajedens", "lajedinhens",
"lajedonens", "lajeirens", "lajens", "lajiens",
"lajinhens", "lalimens", "lamaronens", "lambariens",
"lambasens", "lambedorens", "lambedourens", "lambesens",
"lamecens", "lameirens", "laminens", "lamounierens",
"lampejant", "lâmur", "lançant", "lanciens",
"lancinant", "landrinopolens", "langu", "languipnens",
"lanhadens", "lanhelens", "lanhonens", "lanhosens",
"lantejoulant", "lantunens", "lanuens", "laodicens",
"laosens", "laozens", "lapelens", "lapens",
"lapinhens", "laplatens", "laponenens", "lapurdens",
"laranjaiens", "laranjalens", "laranjeirens", "larens",
"larguens", "larissens", "larnens", "lassancens",
"lastrens", "latejant", "latent", "lateranens",
"latoratens", "laurbanens", "laurentinens", "lausanens",
"lausperen", "lavagant", "lavinens", "laviniens",
"lavourens", "lavrant", "lavrens", "lavrinhens",
"laxant", "leandrens", "leblonens", "legent",
"legorn", "leiriens", "leitens", "leixonens",
"lemens", "lemniens", "lemovicens", "lenciens",
"lençoens", "lençoiens", "lençolens", "lentiens",
"leodiens", "leonicens", "leopoldens", "leopoldinens",
"leopolens", "leptimagnens", "lesotens", "levalloisens",
"levaloisens", "levergerens", "levinopolens", "libarnens",
"liberalizant", "liberatens", "liberdanens", "libertens",
"libriform", "licitant", "licuriens", "lidianopolens",
"lidiens", "liechtensteinens", "liechtensteiniens", "ligant",
"lígur", "limarens", "limeirens", "limiens",
"limitant", "limítrof", "limoeirens", "lind",
"lindoestens", "lindoiens", "linens", "linharens",
"liparens", "lipsiens", "liquidant", "lirinens",
"lisbonens", "lisimaquiens", "litigant", "litisconsort",
"lituaniens", "livramentens", "lixiviant", "lizardens",
"loandens", "lobatens", "lobelens", "lobens",
"locrens", "loirejant", "lombardiens", "londinens",
"londoniens", "londrinens", "longaens", "lontrens",
"lordelens", "lorenens", "loretens", "loronens",
"lourdens", "lourencens", "lourens", "lourinhanens",
"lourinhanhens", "lousadens", "lousanens", "lousonens",
"louveirens", "lovaniens", "luandens", "lubiens",
"lubrificant", "lucaiens", "lucelburgens", "luceliens",
"lucenens", "lucens", "lucerens", "lucianopolens",
"lucilant", "lucoferonens", "lucrinens", "ludovicens",
"lufens", "lugdunens", "lúgubr", "luisburguens",
"luisens", "luisianens", "luisianiens", "luizens",
"lumbricoid", "lumiarens", "luminariens", "luminescent",
"luminosens", "lundens", "lupercens", "luperciens",
"lupiens", "lupinopolens", "luquens", "lursens",
"lusitaniens", "lustosens", "luteciens", "luxemburguens",
"luxoviens", "luxuriant", "luzens", "luzent",
"luzernens", "luzianens", "luzianiens", "luziens",
"luzilandens", "luzilandiens", "mabens", "mabideriens",
"macabuens", "macabuzinhens", "maçacaraens", "macaeens",
"macaens", "maçaens", "macaibens", "macaiens",
"macajatubens", "macajubens", "macambirens", "macanareens",
"maçant", "macaocens", "macaoquens", "macapaens",
"macaparanens", "macapuanens", "macaquens", "macaraniens",
"macarauens", "macarens", "macataens", "macatubens",
"macauariens", "macaubalens", "macaubens", "macauens",
"macedens", "macedinens", "macedoniens", "maceioens",
"machacaliens", "machacalizens", "machadens", "machadiens",
"machadinhens", "machiquens", "machorrens", "macieirens",
"macielens", "macond", "macronutrient", "macucens",
"macuens", "macuinens", "macujeens", "macupinens",
"macuquens", "macuripariens", "macurureens", "macuxixiriens",
"madagascarens", "madailens", "madalenens", "madeirens",
"madiens", "madragoens", "madrilens", "madureirens",
"maetinguens", "mafrens", "mafrensens", "magalhens",
"magdalenens", "magdens", "mageens", "magnat",
"magnet", "magnificent", "maguariens", "maiauataens",
"maiens", "maiobens", "maiquiniquens", "mairiens",
"mairiporanens", "mairipotabens", "majeens", "majorens",
"majorlandens", "majorlandiens", "majurianens", "malabarens",
"malacachetens", "malaquiens", "malaripens", "malauiens",
"maldivens", "maldizent", "maledicent", "maletens",
"malgax", "malhadens", "malhadorens", "maliciens",
"maliens", "malinens", "malletens", "malocens",
"maloquens", "maltens", "maluens", "malvinens",
"mamanacaraiens", "mamanguapens", "mambaiens", "mambemb",
"mamboreens", "mambucabens", "mambuquens", "mamedens",
"mamens", "mamiaens", "mamoeirens", "mamonens",
"mamoriaens", "mamparrens", "manacaens", "manacapuruens",
"managuens", "manairens", "manaquiriens", "manariens",
"manauens", "mandaçaiens", "mandacaruens", "mandaguaçuens",
"mandaguariens", "mandant", "mandembens", "mandiquiens",
"mandiritubens", "mandirobens", "mandiuens", "mandivens",
"manducens", "manduquens", "manduriens", "mangabeirens",
"mangabens", "mangalens", "mangaraiens", "mangarataiens",
"mangaratibens", "mangaratuens", "mangualdens", "manguariens",
"mangueirens", "mangueirinhens", "manguens", "manguinhens",
"manguncens", "manhanens", "manhuaçuens", "manhumirinens",
"maniaçuens", "manicens", "maniçobens", "manicoreens",
"manifestant", "manimbuens", "manitubens", "manivens",
"mansens", "mansidãoens", "mansinhens", "mansuetudinens",
"manteiguens", "mantenens", "manteniens", "mantenopolens",
"mapanens", "mapangapens", "mapiaens", "mapirenens",
"mapixiens", "mapuaens", "maquineens", "maraãens",
"maraanens", "maraãzens", "marabaens", "maracaçumeens",
"maracaens", "maracaiens", "maracajaens", "maracajauens",
"maracajuens", "maracananens", "maracanauens", "maracanens",
"maracasens", "maracatubens", "maracoeirens", "maracujaens",
"maragogiens", "maragojiens", "maragojipens", "maragojipinhens",
"maraguaens", "maraialens", "marajaens", "marajoarens",
"marajoens", "marambainhens", "marangatuens", "maranguapens",
"maranhens", "marapaniens", "marapaninens", "marapeens",
"marapuamens", "marariens", "mararupaens", "marataens",
"marataizens", "marauaens", "marauens", "marav",
"maravilhens", "marcant", "marcelandens", "marcelens",
"marcelinens", "marcens", "marcescent", "marchant",
"marcoens", "marcolinens", "marcondens", "marcondesiens",
"marcosiens", "marcuens", "mareant", "maremens",
"maresiens", "marfinens", "margaridens", "marialvens",
"marianens", "mariantens", "mariapolens", "maribondens",
"maricaens", "maricoabens", "mariens", "marientalens",
"marilacens", "marilandens", "marilandiens", "marilenens",
"marilens", "mariliens", "mariluzens", "marimbens",
"marimbondens", "marinens", "maringaens", "marinheirens",
"marinhens", "marinopolens", "maripaens", "mariquitens",
"mariricuens", "maristelens", "maritubens", "marizens",
"marlieriens", "marmelandens", "marmelandiens", "marmelarens",
"marmeleirens", "marmelens", "marojalens", "marombens",
"marotens", "marovens", "marquens", "marquinhens",
"marrequens", "marroquens", "marruaens", "martinens",
"martinesiens", "martinhens", "martinopolens", "martiriens",
"maruanunens", "marudaens", "maruinens", "marumbiens",
"marupiarens", "marvanens", "marzaganens", "marzaganiens",
"marzagonens", "mascotens", "massapeens", "massarelens",
"massiliens", "massilitanens", "mastaurens", "matabel",
"matanens", "matapajipens", "matapiens", "matapiquarens",
"mataracens", "matarandibens", "mataraquens", "mataripens",
"mataripuanens", "mateens", "mateirens", "matens",
"materlandens", "matiens", "matildens", "matinens",
"matinhens", "matintinens", "matipoens", "matoens",
"matonens", "matoqueimadens", "matosinhens", "matotuiens",
"matozinhens", "matrinxaens", "matritens", "matuens",
"matupaens", "maturucens", "maturuquens", "matutinens",
"mauaens", "maueens", "mauens", "mauesens",
"mauriens", "mauritiens", "maustrarens", "maxacalens",
"maxacaliens", "maxaranguapens", "maximens", "mazaganens",
"mazagonens", "mealhadens", "mearinens", "mecejanens",
"medeirens", "medens", "medianeirens", "medicilandens",
"medievalizant", "medinens", "medíocr", "mediolanens",
"mediquens", "meditacionens", "medubrigens", "meens",
"megarens", "meladens", "meladinhens", "melanciens",
"melariens", "meldens", "meleirens", "melens",
"melgacens", "meliant", "melibeens", "melissens",
"melitens", "melosens", "melquiadens", "memoriens",
"mencheviqu", "mendac", "mendanhens", "mendens",
"mendesens", "mendicant", "mendoncens", "mercant",
"mercedens", "mercesens", "mergulhant", "meridens",
"meridianens", "merirumanens", "meritiens", "merobrigens",
"mertolens", "meruocens", "meruoquens", "mesquitens",
"messanens", "messiens", "messinens", "metalinens",
"metaurens", "metelandens", "metens", "metuapirens",
"mevanionens", "mianmarens", "micaelens", "micenens",
"micuiens", "migrant", "miguelens", "miguelopens",
"milagrens", "milanens", "milasens", "milfontens",
"milharens", "milíped", "militant", "milliarens",
"mimosens", "mindens", "minduriens", "mineirens",
"mineirolandens", "minens", "minervens", "minodunens",
"minoens", "minorens", "minturnens", "míop",
"mipibuens", "mirabelens", "mirabolant", "miracatuens",
"miracemens", "miracicens", "miraciquens", "miradorens",
"miradourens", "miragaiens", "miraginens", "miraguaiens",
"miraiens", "miraimens", "miraltens", "miralucens",
"miraluzens", "miramantens", "mirambeens", "mirandelens",
"mirandens", "mirandibens", "mirandopolens", "mirangabens",
"miranguens", "mirantens", "mirantonens", "mirapinimens",
"miraporanguens", "mirariens", "mirasselvens", "mirassolandens",
"mirassolens", "mirens", "miriápod", "mirinens",
"mirinzalens", "miritiens", "miritizalens", "miritubens",
"mirizalens", "mirobrigens", "miroroens", "misenens",
"misericordiens", "missionariens", "missionens", "mit",
"mitilenens", "mitilens", "miudezens", "moacirens",
"móbil", "mobralens", "mocajubens", "mocambeirens",
"mocambens", "mocambinhens", "moçambiquens", "moçâmed",
"moçamedens", "moçárab", "mococens", "mocoens",
"mocoquens", "mocotoens", "modelens", "modenens",
"modernens", "modernizant", "modestens", "moedens",
"moemens", "moendens", "mogadourens", "mogeirens",
"mogonciacens", "moibandens", "moimentens", "moinant",
"moinhens", "moiporaens", "moirabens", "moitens",
"moitinhens", "mojeirens", "mojiguaçuens", "mojiguiçabens",
"mojimirinens", "mojuens", "mojutaperens", "mol",
"molequens", "molheirens", "molhens", "moloquens",
"molumbens", "momanaens", "momatens", "mombacens",
"mombejens", "mombucens", "mombuquens", "monçanens",
"monchiquens", "moncionens", "monçonens", "moncorvens",
"mondaiens", "mond", "mondeguens", "mondinens",
"mondubinens", "moneraens", "monerens", "monetens",
"monfortens", "mongaguaens", "mongubens", "monicens",
"monjolandens", "monjolens", "monjoliens", "monsanens",
"montalegrens", "montalvanens", "montalvaniens", "montanhens",
"montant", "montargilens", "montauriens", "monteirens",
"montemorens", "montenegrens", "montens", "montesens",
"montevideuens", "montezumens", "montijens", "montmartrens",
"montoitens", "montrealens", "monumentens", "moquenzens",
"moradens", "moralens", "morangabens", "moraujens",
"morceguens", "mordent", "moreirens", "morenens",
"morens", "morgantens", "moriruens", "mormacens",
"mororoens", "morparaens", "mortaguens", "mortugabens",
"morumbiens", "morungabens", "morungavens", "mosqueirens",
"mossoroens", "mostardens", "mosteirens", "motens",
"motiens", "motivant", "mouranens", "mourens",
"mouronens", "mousteriens", "movent", "movimentens",
"moxotoens", "mozarlandens", "mozarlandiens", "mozondoens",
"muanaens", "muanens", "mucajaens", "mucajaiens",
"mucambens", "mucuens", "mucujeens", "mucuratens",
"mucuriciens", "mucuriens", "mucuripens", "mudancens",
"mudanciens", "muiraneuens", "muladens", "mulatens",
"muliternens", "multiform", "multiplex", "multíplic",
"multirresistent", "mulunguens", "mumbabens", "mumbucens",
"mumbuquens", "mumificant", "mundauens", "munduriens",
"mundurucuens", "mungubens", "munhozens", "municipalens",
"munícip", "muniens", "muniquens", "muquiçabens",
"muquiens", "murajaens", "murcens", "murens",
"muriaeens", "muribecens", "muribequens", "muriciens",
"muricizalens", "muriçoquens", "muriens", "muritibens",
"muritiens", "muriuens", "murmurant", "murmurejant",
"murtinhens", "murtosens", "murucupiens", "murunduens",
"murupeens", "mururuens", "murutinguens", "musticens",
"mustierens", "mutaens", "mutambeirens", "mutambens",
"mutanens", "mutant", "mutãzens", "muticens",
"mutinens", "mutucens", "mutucumens", "mutuens",
"mutuipens", "mutumparanaens", "mutunens", "mutunzens",
"mutuquens", "muzambinhens", "myanmens", "nabuquens",
"nacamurens", "nacipens", "nadant", "nadens",
"nagadiens", "najeens", "nantens", "nanuquens",
"napocens", "naquens", "narandibens", "naraniuens",
"narbonens", "narcotraficant", "narniens", "narrant",
"naruens", "nascenciens", "nascent", "nascentens",
"nassadens", "natalens", "natant", "naterciens",
"natinguiens", "nativens", "natividadens", "natividens",
"nativitatens", "natubens", "nauariens", "nauruens",
"nauseant", "navarrens", "navegant", "navegantens",
"naviens", "naviraiens", "nazareens", "nazarenens",
"nazarezinhens", "nazariens", "neandertalens", "neblinant",
"necromant", "neerlandens", "negerens", "negligent",
"negociant", "negrãoens", "negrejant", "negrens",
"negromant", "nelens", "nemausens", "nemausiens",
"nemetacens", "nemetens", "nemorens", "neocomiens",
"neocrucens", "neoerens", "neolimens", "neorresendens",
"neorrussens", "neossantarenens", "neossourens", "neozelandens",
"neozelandiens", "nepalens", "nepanens", "nepomucenens",
"neriens", "neropolens", "nertobrigens", "nerulonens",
"nescaniens", "netinens", "neustriens", "nevens",
"nhambiquarens", "nhamundaens", "nhandearens", "nhandutibens",
"nhecolandens", "nhecolandiens", "nhonhepens", "nhunguaçuens",
"nicaraguens", "niceens", "nicens", "niciens",
"nicolandens", "nicolandiens", "nicolauens", "nicomedens",
"nidificant", "nigerens", "nigromant", "nilens",
"nioaquens", "nipoanens", "nipoens", "niponens",
"niquelandens", "niquelandiens", "nisens", "nitent",
"niteroiens", "nitrificant", "nivernens", "nizens",
"nobilitant", "nobr", "nogueirens", "nolens",
"nômad", "nonatens", "nonoaiens", "nonsens",
"norbanens", "norbens", "nordestens", "nordestinens",
"norens", "noronhens", "nortelandens", "nortelandiens",
"nortens", "northamptonens", "norueguens", "novalens",
"novariens", "novens", "novilhonens", "novocomens",
"nubent", "nubiens", "nuguaçuens", "numiniens",
"nupebens", "nuporanguens", "nutrient", "oacariens",
"oasiens", "obdiplostêmon", "obedient", "obidens",
"objetivant", "obrant", "obsedant", "obsediant",
"observant", "obstant", "obulconens", "ocarens",
"ocauçuens", "oceaniens", "ocelens", "ocidentalizant",
"ocidentens", "ocorrent", "ocr", "octodurens",
"ocultens", "ocunens", "ocupant", "odemirens",
"odivelens", "odorant", "oduciens", "oeirens",
"oestens", "ofegant", "oferent", "ofertant",
"oficiant", "oficinens", "ofuscant", "oiapoquens",
"oiteirens", "oitiuens", "oitizeirens", "olariens",
"olarionens", "olbiens", "olbonens", "oleastrens",
"oleens", "olegariens", "oleirens", "olent",
"olhanens", "olifant", "oligostêmon", "olimpens",
"olimpiens", "olindens", "olindinens", "olisiponens",
"olissiponens", "olivalens", "olivedens", "oliveirens",
"olivencens", "olivenciens", "olivens", "oliviens",
"omanens", "omelet", "omnipotent", "omniscient",
"oncens", "ondeant", "ondinens", "ondulant",
"onens", "onhens", "onipotent", "onipresent",
"oniscient", "onofrens", "onubens", "operant",
"opinant", "oponent", "optant", "oradens",
"oratoriens", "orçant", "ordinarot", "ordoviciens",
"oricoens", "orientalens", "orient", "orientens",
"orindiuvens", "oriolens", "orionens", "oriximinaens",
"orizaniens", "orizonens", "orlandiens", "orleanens",
"oroandens", "oroboens", "orocoens", "orosens",
"ortigueirens", "ortonens", "ortroniens", "osasquens",
"oscens", "oscilant", "osoriens", "ossonobens",
"ostiens", "ostrens", "otroniens", "ouguelens",
"ouraniens", "oureens", "ouremens", "ourenens",
"ourens", "ourenzens", "ouricanens", "ouricuriens",
"ouriens", "ourinhens", "ouriquens", "ourivens",
"ourizonens", "outeirens", "outorgant", "ouvidorens",
"ouvint", "ovant", "ovarens", "ovetens",
"ovitens", "oxidant", "pacaembuens", "pacajaens",
"pacajuens", "pacatubens", "pacatuens", "pacens",
"pachequens", "pacienciens", "paciens", "pacient",
"paçoquens", "pacotiens", "pacotubens", "pacovalens",
"pacovalzinhens", "pacuens", "pacuiens", "pacujaens",
"pacuriens", "pacutiens", "pacutubens", "padecent",
"padilhens", "paduens", "pagant", "paiaguaens",
"paiaiaens", "paialens", "paineirens", "painelens",
"painens", "paiolens", "paiolinhens", "paiquereens",
"paivens", "pajauens", "pajeens", "pajeuens",
"pajuçarens", "palamens", "palapoutens", "palauens",
"palenens", "palens", "paleomanens", "palermens",
"palestinens", "palestrant", "palhalens", "palhanenens",
"palhens", "palhetens", "palhocens", "palhocinhens",
"palionens", "palmarens", "palmatoriens", "palmeiralens",
"palmeirandens", "palmeirens", "palmeirinens", "palmeirinhens",
"palmelens", "palmens", "palmíped", "palmirens",
"palmitalens", "palmitens", "palmitinhens", "palmopolens",
"palomens", "palotinens", "palpitant", "paltonens",
"paluens", "pamonhens", "pampanens", "pampeirens",
"pampilhosens", "panacuiens", "panamaens", "panambiens",
"panamens", "panaripens", "pancens", "panelens",
"panfiliens", "pangareens", "panoiens", "panoramens",
"panquens", "pantanalens", "pantanalzinhens", "pantanens",
"panteens", "panticapens", "papagaiens", "papanduvens",
"papariens", "papoulens", "papuriens", "paquens",
"paquequerens", "paquetaens", "paquet", "paquevirens",
"paquicamens", "paquiderm", "paquistanens", "paracambiens",
"paracambuquens", "paracatuens", "paracuaens", "paracuruens",
"paradens", "paradisens", "paraens", "parafusens",
"paraguaçuens", "paraguaiens", "paraguariens", "paraibanens",
"paraibens", "paraibunens", "paraiens", "paraipabens",
"paraisens", "paraisolandens", "paraisolandiens", "paraisopolens",
"paraitinguens", "paraiuvens", "parajuens", "parajuruens",
"paralisant", "paralissens", "paramanens", "parambuens",
"paramirinens", "paramirinhens", "paramotiens", "paranabiens",
"paranacitens", "paranacityens", "paranaens", "paranãens",
"paranaguaçuens", "paranaguaens", "paranaibens", "paranaiens",
"paranajiens", "parananens", "paranapanemens", "paranapoemens",
"paranapuanens", "paranariens", "paranatamens", "paranavaiens",
"paranãzens", "parangabens", "paranhens", "paraopebens",
"parapeumens", "parapuanens", "parapuens", "paraquemens",
"parariens", "paratariens", "parateiens", "paratibens",
"paratiens", "paratijiens", "paratinguens", "paratudalens",
"parauaraens", "parauens", "paraunens", "parazinhens",
"parceirens", "pardalens", "pardilhoens", "pardinhens",
"pareciens", "paredens", "paredonens", "parelheirens",
"parelhens", "parent", "paricaens", "paricatubens",
"pariconhens", "parijoens", "parintinens", "paripiens",
"paripiranguens", "paripueirens", "pariquerens", "paririens",
"parisiens", "parmens", "parnaguaens", "parnaguens",
"parnaibens", "parnaiuicavens", "parnamirinens", "parnaramens",
"parnasens", "parobeens", "parodiant", "pars",
"participant", "partícip", "parturient", "paruruens",
"pascoalens", "pascoens", "passabenens", "passabenzens",
"passagenens", "passagens", "passagenzens", "passant",
"passarinhens", "passaunens", "passavant", "passeant",
"passeens", "passens", "passeriform", "passirens",
"pastens", "pastilhens", "pastinhens", "pastorens",
"pataibens", "patamuteens", "patavicens", "pateant",
"pateironens", "patens", "patiens", "patif",
"patizalens", "patracens", "patrens", "patriarquens",
"patriciens", "patrimoniens", "patrocinens", "patrociniens",
"patrulhens", "patuens", "pauapixunens", "pauiniens",
"paulens", "pauliceens", "pauliceiens", "paulinens",
"pauliniens", "paulistanens", "paulistaniens", "paulistens",
"paulivens", "pausulens", "pauxiens", "paveramens",
"paviens", "pavoens", "pavonens", "pavuçuens",
"pavunens", "paxiuens", "paxuens", "pazens",
"peabiruens", "pebens", "peçanhens", "pecenens",
"pecenzens", "pedant", "pederneirens", "pedestr",
"pedint", "pedralvens", "pedrariens", "pedregulhens",
"pedreirens", "pedrens", "pedrinhens", "pedritens",
"pedroguens", "pedronens", "pedrosens", "peixens",
"peixinhens", "peixotens", "pejuçarens", "peladens",
"pelenens", "pelens", "peloponens", "pelotens",
"penacovens", "penafidelens", "penafortens", "penaguiens",
"penalvens", "penamacorens", "penamarens", "penant",
"penapolens", "pendanguens", "pendenciens", "pendent",
"penedens", "penediens", "penedonens", "penelens",
"penetrant", "penichens", "penitent", "pensant",
"pepitalens", "pequeriens", "pequiaens", "pequiens",
"pequiriens", "pequizeirens", "perapeunens", "percevejens",
"perciform", "percucient", "perdicens", "perdicionens",
"perdigonens", "perdiguens", "perdizens", "perdonens",
"peregrinens", "pereirens", "pereirinhens", "peren",
"perequeens", "perfurant", "pergens", "pergunt",
"periclitant", "pericoens", "perigarens", "perigosens",
"perimboens", "perimiriens", "perimirinens", "periquarens",
"periquitens", "peritoroens", "permanent", "pernambuquens",
"pernambuquinhens", "perobalens", "perobens", "perolens",
"perpetuens", "perrengu", "perrexilens", "perscrutant",
"perseveranciens", "perseverant", "persistent", "pertencent",
"pertilens", "pertinent", "perturbant", "peruanens",
"peruens", "peruibens", "pervagant", "pescadorens",
"pesqueirens", "pesqueirinhens", "pessanhens", "pessegueirens",
"pessoens", "pest", "pestilent", "petavionens",
"petavoniens", "petinens", "petragoricens", "petrens",
"petrogradens", "petrolandens", "petrolandiens", "petrolinens",
"petrovinens", "petulant", "petuniens", "piabanhens",
"piabens", "piacabuçuens", "piaçabuçuens", "piacaens",
"piacatubens", "piacatuens", "piaçuens", "piancoens",
"piassabuçuens", "piatãens", "piatanens", "piatãzens",
"piauens", "piauiens", "piaunens", "piavorens",
"picadens", "picadinhens", "picant", "pichens",
"piciform", "picinguabens", "picoens", "pictavens",
"picuens", "picuiens", "piedadens", "piedens",
"pienens", "piens", "pietatens", "pilarens",
"pilens", "piliens", "pilonens", "pimenteiralens",
"pimenteirens", "pimentelens", "pimentens", "pinareens",
"pincens", "pindaibens", "pindaiens", "pindaivens",
"pindamonhangabens", "pindareens", "pindens", "pindobaçuens",
"pindobalens", "pindobens", "pindoguabens", "pindoramens",
"pindoretamens", "pindotibens", "pindovalens", "pinens",
"pingadorens", "pingant", "pingent", "pinhalens",
"pinhalitens", "pinhalonens", "pinhalzinhens", "pinhãoens",
"pinhãozens", "pinhãozinhens", "pinheiralens", "pinheirens",
"pinheirinhens", "pinhelens", "pinhonens", "pinhotibens",
"piníped", "pintadens", "pintens", "pinturilant",
"pioneirens", "pionens", "pipeirens", "pipens",
"pipilant", "pipinuquens", "piquens", "piquerobiens",
"piquetens", "piquiriens", "pirabeirabens", "pirabens",
"pirabibuens", "piracaibens", "piracaiens", "piracanjubens",
"piracauaens", "piracemens", "piracicabens", "piraçununguens",
"piracurucens", "piracuruquens", "piraguarens", "piraiens",
"pirainhens", "piraiuarens", "pirajaens", "pirajibens",
"pirajubens", "pirajuens", "pirajuiens", "piramboiens",
"pirambuens", "pirangaiens", "piranguçuens", "piranguens",
"piranguinhens", "piranguitens", "piranhens", "piranjiens",
"piranjinens", "pirapanemens", "pirapemens", "pirapetinguens",
"pirapitinguiens", "pirapoens", "piraporens", "pirapozens",
"pirapozinhens", "piraputanguens", "piraquarens", "piratabens",
"piratiniens", "piratininguens", "piratubens", "pirauaens",
"piraubens", "pirens", "pirgens", "piriaens",
"piriens", "piriform", "piripaens", "piripiriens",
"piritibens", "piritubens", "pirizalens", "pirizeirens",
"pirpiritubens", "pisaurens", "pisauriens", "piscambens",
"pistoriens", "pitangueirens", "pitanguens", "pitanguiens",
"pitimbuens", "pititinguens", "pitombeirens", "pitubens",
"pitunens", "piuens", "piuiens", "piumens",
"piunens", "piunzens", "piuvens", "pixainens",
"pixainzens", "planaltens", "planaltinens", "planetens",
"plangens", "plangent", "planinens", "planurens",
"plastificant", "plateens", "platelmint", "platens",
"platinens", "pleninens", "pleuronectiform", "poaens",
"poaiens", "pobr", "poçanens", "poçãoens",
"poçãozens", "pocens", "pochiquens", "pocinhens",
"pocionens", "poçoens", "poconeens", "poçoneens",
"pocranens", "podimirinens", "podr", "poeirens",
"poent", "poiarens", "poiriens", "pojuquens",
"polens", "policarpens", "politizant", "polivalent",
"poloniens", "poluent", "polvorens", "pombalens",
"pombens", "pombinhens", "pomeranens", "pomerodens",
"pometinens", "pompeens", "pompelonens", "pompeuens",
"poncianens", "pongaiens", "pontalens", "pontaletens",
"pontalindens", "pontalinens", "pontalzinhens", "pontens",
"pontevilens", "pontezinhens", "pontiens", "pontinens",
"pontinhens", "pontonens", "populinens", "populoniens",
"porangabens", "porangatuens", "poranguens", "porciunculens",
"porecatuens", "porfiriens", "poronguens", "pororoquens",
"portaginens", "portalegrens", "porteirens", "porteirinhens",
"portelens", "portens", "portimonens", "portinhens",
"portlandiens", "portonens", "portucalens", "portuens",
"portugalens", "porvirens", "possant", "possens",
"possident", "postulant", "poteens", "potenjiens",
"potent", "potiens", "potiraguaens", "potiraguarens",
"potiraguens", "potirendabens", "potirens", "potiretamens",
"potreirens", "potreiritens", "potunduvens", "pousens",
"povens", "povoacens", "povoens", "poxorens",
"poxoreuens", "pracinhens", "pradens", "pradinhens",
"pradosens", "praguens", "praiens", "prainhens",
"pranchadens", "pranchitens", "prataniens", "pratapolens",
"pratens", "praticant", "pratinhens", "prazerens",
"precedent", "precípit", "precoc", "predisponent",
"predominant", "preeminent", "preexcelent", "preexistent",
"preguens", "preguicens", "prement", "premonstratens",
"prensens", "preocupant", "preponderant", "preponent",
"prepotent", "present", "president", "presidiens",
"presidutrens", "pressionant", "prestant", "prestens",
"pretendent", "pretoriens", "prevalecent", "prevalent",
"prevenidens", "prevident", "prienens", "primaiens",
"primaverens", "primorens", "princesens", "principiant",
"privatizant", "probidadens", "procariont", "procedent",
"procopiens", "proeminent", "proencens", "profetens",
"profissionalizant", "progressens", "promissens", "propagant",
"propelent", "proponent", "propriaens", "prosperidadens",
"protecionens", "proteiform", "protestant", "provenient",
"providenciens", "provident", "provocant", "prudent",
"prudentens", "prusens", "prusiens", "puanens",
"puãzens", "pubent", "pubescent", "puçaens",
"pucalpens", "puguens", "puiraçuens", "pujant",
"puladorens", "pulsant", "puluens", "punens",
"pungent", "puntiform", "pupiniens", "purezens",
"purgant", "purilandens", "purilandiens", "pusilânim",
"putinguens", "putiriens", "puxavant", "puxinanãens",
"puxinanãzens", "quadrant", "quadrens", "quadriburgens",
"quadrúped", "quajaraens", "quaraçuens", "quaraiens",
"quarentens", "quariens", "quartelens", "quataens",
"quatiçabens", "quatiens", "quatiguabens", "quatiguaens",
"quatiguens", "quatipuruens", "quatiquanens", "quatitubens",
"quebequens", "queboens", "quebrachens", "quebrangulens",
"quebrant", "queçabens", "quedens", "queimadens",
"queirosens", "queixadens", "queluzens", "queluzitens",
"quent", "queremens", "querenciens", "querent",
"querronens", "quersonens", "quevedens", "quiçamaens",
"quiçamãens", "quiçamanens", "quiceens", "quiescent",
"quietens", "quijinguens", "quilombens", "quilópod",
"quimamiens", "quimiossintetizant", "quincoeens", "quincuncaens",
"quintanens", "quintens", "quintinens", "quipaens",
"quipapaens", "quiririens", "quiririnens", "quiriritiuens",
"quiromant", "quitandinhens", "quiterianopolens", "quiteriens",
"quitimbuens", "quitundens", "quitunguens", "quixabeirens",
"quixabens", "quixadaens", "quixariuens", "quixeloens",
"quixeramobiens", "quixeramobinens", "quixereens", "quixoaens",
"quizotens", "rabdomant", "rabecens", "rabequens",
"rabichens", "rabugic", "rabulic", "raciarens",
"raciocinant", "radiadorens", "radiant", "radiouvint",
"rafaelens", "rafardens", "raimundens", "raizalens",
"raizamens", "raizens", "rajadens", "ramadens",
"ramalhens", "ramalhetens", "ramens", "ramirens",
"ramnens", "ramonens", "ranchariens", "ranchens",
"ranchinhens", "rangent", "rapac", "rapant",
"rapinant", "raposens", "rareant", "rarescent",
"rasant", "rascant", "rasens", "raspadorens",
"rastejant", "rateens", "ratens", "ratic",
"ratit", "ratonens", "ravenatens", "ravenens",
"reafirmant", "reagent", "realens", "realezens",
"rebeld", "rebent", "reboant", "rebojens",
"rebordosens", "reboucens", "rebrilhant", "recalcitrant",
"recendent", "recent", "reciens", "recifens",
"recipient", "reclamant", "reconfortant", "recorrent",
"recreens", "recuperadens", "recursens", "recusant",
"recusens", "redencens", "redencionens", "redençonens",
"redent", "redentorens", "redinhens", "redolent",
"redondens", "redundant", "redutens", "referent",
"refervent", "reformens", "refrigerant", "refugiadens",
"refulgent", "regant", "regenciens", "regeneracionens",
"regeneraçonens", "regenerens", "regent", "regentens",
"regiens", "reginens", "reginopolens", "registrens",
"regrant", "reguenguens", "reguens", "regurgitant",
"reinant", "reincident", "relampejant", "relaxant",
"relevant", "reliquiens", "relumbrant", "relutant",
"reluzent", "relvadens", "remanescent", "remansonens",
"remediens", "remens", "rêmig", "remigens",
"remigiens", "reminiscent", "remont", "renascencens",
"renascent", "renitent", "renqu", "rent",
"repartimenens", "repelent", "repent", "repetent",
"replicant", "repont", "repousant", "representant",
"reptant", "repugnant", "requerent", "requint",
"reriutabens", "rescendent", "resendens", "reservens",
"resfolegant", "residenciens", "resident", "resistent",
"respeitant", "resplandecent", "resplandens", "resplandescent",
"resplendent", "resplendorens", "respondent", "ressaltant",
"ressaquens", "ressoant", "ressonant", "restant",
"restauracionens", "restaurant", "restinguens", "restringent",
"resultant", "resvalant", "retardant", "reticent",
"retiens", "retirant", "retirolandens", "retirolandiens",
"retumbant", "reumatoid", "reverberant", "reverent",
"revoltant", "riachãoens", "riachens", "riachinhens",
"riachonens", "riachuelens", "rialmens", "rialtens",
"riazulens", "ribacudens", "ribaduriens", "ribamarens",
"ribatejens", "ribeirãoens", "ribeirens", "ribeirinhens",
"ribeironens", "ribeiropolens", "ricinens", "rident",
"riens", "rifainens", "rinareens", "rinconens",
"rinoceront", "rinopolens", "rioclarens", "riograndens",
"riolandens", "riozinhens", "riquezens", "ritapolens",
"ritens", "rivens", "riversulens", "robenhausiens",
"robertens", "roçadens", "roçagant", "rochedens",
"rochedinhens", "rocinant", "rodaniens", "rodant",
"rodeens", "rodeiens", "rodeirens", "rodens",
"rodiens", "rodopiant", "roladorens", "rolandens",
"rolandiens", "rolant", "rolantens", "rolimorens",
"romãoens", "romarens", "romariens", "romelandens",
"romelandiens", "rompant", "rompent", "roncadens",
"roncadorens", "rondant", "rondinhens", "rondonens",
"rondoniens", "ronflant", "roquens", "roraimens",
"rorejant", "rosalandens", "rosalens", "rosaliens",
"rosalindens", "rosanens", "rosarens", "rosariens",
"roseiralens", "roseirens", "roteirens", "rotomagens",
"rotundens", "ruandens", "rubelitens", "rubescent",
"rubiaceens", "rubiacens", "rubiatabens", "rubiens",
"rubineens", "rubinens", "rud", "rugent",
"ruilandens", "ruinens", "ruminant", "rumorejant",
"rupestr", "russens", "rutenens", "rutilant",
"saarens", "sabaens", "sabaraens", "sabarens",
"sabaudiens", "sabaunens", "sabiaens", "sabiaguabens",
"sabinens", "sabinopolens", "saboeirens", "saboguens",
"saboiens", "sabonetens", "saborens", "sabratens",
"sabrosens", "sabugalens", "sabugueirens", "sabujiens",
"sacadens", "sacambuens", "sacavenens", "sacerdot",
"sacoiens", "sacramentens", "sacripant", "sacrocordiens",
"sadens", "saens", "safarens", "safirens",
"sagradens", "sagrens", "saibrens", "saicãzens",
"saiens", "sainaumens", "saireens", "sajens",
"salaciens", "salamanquens", "salamanticens", "salariens",
"salavessens", "saldanhens", "salens", "salesens",
"salesopolens", "saletens", "salgadaliens", "salgadens",
"salgadinhens", "salgueirens", "salient", "salinadens",
"salinens", "saliniens", "salininhens", "salitrens",
"salmanticens", "salmouranens", "saloaens", "salobinhens",
"salobrens", "salomeens", "salonens", "salonicens",
"salsens", "saltant", "saltens", "saltinhens",
"saltitant", "salubr", "salutariens", "salvacionens",
"salvadorens", "salvaterrens", "salvens", "salzedens",
"samambaiens", "samaumens", "sambaetibens", "sambaibatubens",
"sambaibens", "sambaquiens", "sambarnabens", "sambeens",
"sambeneditens", "sambernardens", "sambertens", "sambonaventurens",
"samoens", "samorens", "samosatens", "samuelens",
"sanagens", "sananduvens", "sancandidens", "sanderens",
"sandesideriens", "sandominicens", "sandovalens", "sandovalinens",
"sandumonens", "sanens", "sanestefanens", "sanfelicens",
"sanfirminens", "sanfranciscens", "sangalhens", "sanganens",
"sanganhanens", "sangonens", "sanguessuguens", "sanharoens",
"sanjoanens", "sanjosefens", "sanludovicens", "sansimoniens",
"santanderiens", "santanens", "santanesiens", "santareens",
"santarenens", "santarens", "santelmens", "santens",
"santiaguens", "santomens", "santuens", "sãomarinhens",
"sapatariens", "sapeaçuens", "sapecadens", "sapeens",
"sapens", "sapezalens", "sapient", "sapiranguens",
"sapopemens", "sapucaiens", "sapucaranens", "sapuparens",
"saquaremens", "saquinhens", "saracurens", "saracuruens",
"saraivens", "sarandiens", "sarandirens", "sarapuiens",
"sararaens", "sardiens", "sardinhens", "sardoaens",
"sardoalens", "sarens", "sargentens", "sarutaiaens",
"sarzedens", "satélit", "satanens", "satens", "satubens",
"satubinens", "satubinzens", "saturnens", "saubarens",
"saudadens", "saudavelens", "saudens", "saueens",
"sauiaens", "sauraurens", "saurópod", "sauvens",
"seabrens", "searens", "sebalens", "sebastianens",
"sebastopolens", "seberiens", "secadorens", "secanens",
"secant", "sedens", "segestens", "segiens",
"segisamajuliens", "segisamonens", "segobrigens", "segontinens",
"segredens", "seguint", "segundens", "segurens",
"segusiens", "seichelens", "seivalens", "seixalens",
"selbachens", "sel", "seleucens", "seleuciens",
"selmens", "selvens", "semelhant", "semovent",
"senadorens", "senegalens", "senegambiens", "senens",
"senhoritens", "sentinelens", "separacionens", "sepetibens",
"sepotubens", "sepulturens", "sequens", "sequent",
"serenens", "sergiens", "sergipens", "seribinhens",
"sericitens", "seridoens", "seriens", "seritinguens",
"serjiens", "sernachens", "sernancelhens", "serpeant",
"serpens", "serpenteant", "serralitrens", "serranens",
"serraniens", "serrariens", "serrazens", "serrens",
"serrinhens", "serritens", "serrolandens", "serrolandiens",
"serrotens", "sertaginens", "sertanejens", "sertanens",
"sertaniens", "sertanopolens", "sertãozinhens", "servens",
"servent", "sesimbrens", "setubalens",
"setubinhens", "severens", "severinens", "severiniens",
"seychellens", "siamens", "siariens", "sibilant",
"siboens", "sicens", "sicilens", "siciliens",
"sidens", "sidrolandens", "sidrolandiens", "sienens",
"significant", "silenciens", "silens", "silent",
"silvadens", "silvaldens", "silvanectens", "silvanens",
"silvaniens", "silvant", "silveiraniens", "silveirens",
"silvens", "silveriens", "silvestr", "silvestrens",
"silvianopolens", "silviens", "simbiont", "símil",
"simonens", "simonesiens", "simonsenens", "simpatiens",
"simpatizant", "simpliciens", "sinadens", "sincoraens",
"sindens", "sineens", "sinfanens", "singapurens",
"singidunens", "singlet", "singrant", "siniens",
"sinimbuens", "sinopens", "sintrens", "sipicueens",
"siqu", "siqueirens", "siquimens", "siriens",
"sirigadens", "sirijiens", "sirinhaenens", "siririens",
"sirmiens", "sisaponens", "sisolens", "sitiaens",
"sitiant", "sitiens", "sitionovens", "siupeens",
"soajens", "soant", "soarens", "sobejant",
"soberanens", "sobradens", "sobradinhens", "sobralens",
"sobraliens", "sobrant", "sobreirens", "sobrejacent",
"sobrelevant", "sobressalent", "sobrevivent", "socamirinens",
"socavanens", "sociedadens", "socorrens", "sodrelandens",
"sodrelandiens", "sodreliens", "sofalens", "solaneens",
"solanens", "soledadens", "solemarens", "solen",
"solert", "solicitant", "solimariacens", "soluçant",
"solvens", "solvent", "sonant", "sopens",
"soquens", "sorocabanens", "sorocabens", "sororoens",
"sorrident", "sosseguens", "soteiens", "sourens",
"sousaniens", "souselens", "sousens", "soutens",
"stevensonens", "suaçuiens", "suaçuranens", "suanens",
"suav", "suazilandens", "suazilandiens", "subaiens",
"subaumens", "subchef", "subcomponent", "subconscient",
"subidens", "subjacent", "sublim", "sublimizant",
"subsequent", "subservient", "subsistent", "substituint",
"subtenent", "suburanens", "sucatinguens", "sucessens",
"suciant", "sucubens", "sucupirens", "sucureens",
"sucuriens", "sucurijuens", "sucuriuens", "sucurizinhens",
"sucuruens", "sucurujuens", "sudanens", "sudoest",
"suessiens", "suezens", "sufetulens", "suficient",
"sufocant", "sugant", "suicens", "sulcens",
"sulinens", "sulmonens", "sumareens", "sumaumens",
"sumeens", "sumidourens", "sumuriens", "superaguiens",
"supergigant", "superimportant", "superintendent", "superpotent",
"supervenient", "suplent", "suplicant", "súplic",
"supurant", "surfactant", "surinamens", "surpreendent",
"surpresens", "surrealizant", "surrinens", "surubinens",
"suruiens", "susanens", "suspens", "suspirant",
"suspirens", "sussurrant", "sustinens", "tabacens",
"tabaiens", "tabainhens", "tabajarens", "tabapuanens",
"tabaquens", "tabatinguens", "tabaunens", "tabenens",
"taberoeens", "tabirens", "taboadens", "tabocalens",
"taboquens", "taboquinhens", "tabosens", "tabracens",
"tabuacens", "tabuadens", "tabualens", "tabuanens",
"tabudens", "tabuens", "tabuinhens", "tabuleirens",
"tabuleironens", "tacaimboens", "tacaleens", "tacaratuens",
"tachens", "taciateuens", "tacibens", "tacimens",
"tacimirinens", "taciuanens", "tacteant", "tacuiquenens",
"tacuruens", "tadeuens", "tagant", "tagastens",
"taguaiens", "taguatinguens", "taiaçuens", "taiaçupebens",
"taiamanens", "taibens", "tainens", "tainhens",
"taiobeirens", "taipens", "taipinhens", "taipuens",
"tairetaens", "taiuvens", "tajatubens", "tajiens",
"tajuabens", "talabricens", "talabrigens", "talarens",
"talhadens", "talhant", "talimanens", "tamandareens",
"tamanduaens", "tamaniquaens", "tamanqueirens", "tamaquariens",
"tamaranens", "tamarineirens", "tambaquiens", "tambauens",
"tambeens", "tamboarens", "tamboataens", "tamborilens",
"tamborilzinhens", "tamburiens", "tameguens", "tamequens",
"tamoiens", "tamulens", "tanabiens", "tanariens",
"tancredinhens", "tanetiens", "tangaraens", "tangent",
"tangilens", "tanguaens", "tanhaçuens", "tanquens",
"tanquinhens", "tapaiunens", "taparaens", "taparubens",
"tapauaens", "tapejarens", "tapens", "taperens",
"taperiens", "taperinhens", "taperoaens", "taperuabens",
"tapinens", "tapiracanguens", "tapiraiens", "tapirainens",
"tapiraipens", "tapiramens", "tapiramutaens", "tapiranguens",
"tapirapuanens", "tapiratibens", "tapirens", "taporens",
"tapuiens", "tapuiramens", "tapuruens", "tapuruquarens",
"taquapiriens", "taquaraçuens", "taquaralens", "taquarandiens",
"taquaremboens", "taquarens", "taquariens", "taquarilens",
"taquarinhens", "taquaritinguens", "taquaritubens", "taquarivaiens",
"taquarixinens", "taquaruçuens", "tarabaiens", "taraens",
"taraguaens", "taramens", "taraquaens", "tarauacaens",
"tardenens", "tardenoisens", "tardenoisiens", "targinens",
"tarianens", "taritubens", "tarouquens", "tarquiniens",
"tarracens", "tarracinens", "tarraconens", "tarrafens",
"tarragens", "tarsens", "tartamudeant", "tartaruguens",
"taruaçuens", "tarumanens", "tarumirinens", "tarumirinhens",
"tarusconiens", "tasmaniens", "tatairens", "tatajubens",
"tate/ment", "tateant", "tatuamunhens", "tatuens",
"tatuiens", "tatuquarens", "tauaens", "tauapeçaçuens",
"tauapens", "tauapiranguens", "tauarianens", "tauariãzinhens",
"tauariens", "tauaruanens", "taubateens", "taumataens",
"taumaturgens", "taumaturguens", "taunayens", "taunens",
"tauoaens", "tauriniens", "taurunens", "tavarens",
"tavirens", "tavorens", "teanens", "tebaidens",
"tebens", "teçaindaens", "tefeens", "tegianens",
"teixeirens", "tejuçuoquens", "tejucupapens", "tejupaens",
"telheirens", "telhens", "telmessens", "telonens",
"tement", "temerantens", "temperens", "tendent",
"tenent", "tens", "tentugalens", "tênu",
"tenuipens", "teodorens", "teotonens", "teotoniens",
"terceirens", "terebrant", "terenens", "terereens",
"teresens", "teresinens", "teresinhens", "termens",
"termessens", "terminant", "terópod", "terracinens",
"terratenent", "terreirens", "terrens", "terrestr",
"terrinhens", "tesourens", "tesourinhens", "tespiens",
"tessalonicens", "tetaniform", "tetens", "tetrápod",
"tetravalent", "teudalens", "teuiniens", "teutoburgiens",
"teutoniens", "tiaguens", "tianguaens", "tiarajuens",
"tibajiens", "tibauens", "tibiens", "tibiriçaens",
"tibiriens", "tibunens", "ticiens", "ticinens",
"tieteens", "tigr", "tigrens", "tigrinhens",
"tijipioens", "tijocens", "tijoquens", "tijuaçuens",
"tijucalens", "tijucens", "tijucuçuens", "tijuquens",
"tijuquinhens", "tilintant", "timacens", "timbaubens",
"timbeens", "timbirens", "timboens", "timboraens",
"timboreens", "timboteuens", "timbozalens", "timbozinhens",
"timbuens", "timbuiens", "timburiens", "timiadens",
"timorens", "tinguaens", "tinguatibens", "tipiens",
"tipitiens", "tipl", "tiquaruçuens", "tiraciens",
"tiracinens", "tiradentens", "tirant", "tirens",
"tiriens", "tiriricalens", "tiririquens", "tiritant",
"tirocinant", "tirsens", "tissanens", "tissens",
"titeens", "titiaens", "titiarrens", "titilant",
"titubeant", "tiuiens", "toant", "tobatiens",
"tobiarens", "tobiens", "tocaiens", "tocandirens",
"tocanquirens", "tocant", "tocantinens", "tocantiniens",
"togiens", "togolens", "toguens", "toldens",
"toledens", "tolerant", "toleriens", "toloiriens",
"tolosens", "tomarens", "tomarutemens", "tomasiens",
"tomasinens", "tomazinens", "tombadorens", "tombant",
"tombens", "tomens", "tonant", "tonantinens",
"tondelens", "tondens", "tonificant", "toniolens",
"tonitroant", "tonitruant", "tonteant", "tontens",
"topaziens", "topens", "torcidens", "toriparuens",
"toritamens", "torixoreuens", "tormentinhens", "tornacens",
"torneirens", "tornejant", "tornens", "toropiens",
"toroquaens", "torp", "torranens", "torreant", "torreens",
"torrens", "torrent", "torreonens", "torriens",
"torrinhens", "torronens", "tortens", "torturant",
"totalizant", "tourens", "tourinhens", "toxicodependent",
"trabijuens", "tracens", "tracuaens", "tracuateuens",
"tracunhaenens", "tracupaens", "traficant", "traianens",
"traipuens", "trairens", "trairiens", "traironens",
"traituens", "trajanens", "tral", "tralens",
"tramandaiens", "trancosens", "tranqueirens", "tranquilizant",
"transbordant", "transcendent", "transeunt", "transformant",
"transient", "transigent", "transparent", "transplant",
"transudant", "trapiaens", "trapichens", "trapipens",
"trasbordant", "trast", "tratant", "traumatizant",
"travesseirens", "travessens", "travessiens", "travessonens",
"treiens", "trejens", "trelens", "tremant",
"tremecens", "tremedalens", "tremeluzent", "tremembeens",
"trement", "tremulant", "trepidant", "trescalant",
"tressuant", "trevisens", "triacens", "triangulens",
"triarroiens", "tribarrens", "tribiquens", "tribiuens",
"triboboens", "triboquens", "tribracens", "tributant",
"tricachoeirens", "tricasens", "triciens", "tricoronens",
"tricorreguens", "tricrucens", "trident", "tridivisens",
"trifigueirens", "triforquilhens", "trifraternens", "trifronteirens",
"triladeirens", "trilagoens", "trilagunens", "trimaiens",
"trimariens", "trimontens", "trimorrens", "trinchant",
"trindadens", "tripassens", "tripetrens", "tripinens",
"tripinheirens", "tríplic", "tripocens", "tripontens",
"tripudiant", "tripulant", "trirranchens", "trirriens",
"trirrivens", "trist", "tristradens", "tritiens",
"triturant", "triunfant", "triunfens", "trivalent",
"trivendens", "troant", "trofens", "troiens",
"trombens", "trombudens", "trombuquens", "tronquens",
"trovejant", "truçuens", "tuaregu", "tubaronens",
"tubendubens", "tubuliform", "tubunens", "tucanens",
"tuciens", "tucumadeuens", "tucumaens", "tucunareens",
"tucundubens", "tucunduvens", "tucunens", "tucunzens",
"tucupareens", "tucuribens", "tucuruiens", "tucuruviens",
"tudens", "tuguriens", "tuinens", "tuiueens",
"tuiuiuens", "tuiutiens", "tuiutinguens", "tularens",
"tumbens", "tumiritinguens", "tunant", "tuneirens",
"tunens", "tungus", "tuntuens", "tuntumens",
"tuntunens", "tupaciguarens", "tupãens", "tupanaciens",
"tupanatinguens", "tupanciretanens", "tupandiens", "tupanens",
"tupantubens", "tupaocens", "tupaoquens", "tuparaceens",
"tuparaiens", "tuparandiens", "tuparendiens", "tuparetamens",
"tupãzens", "tupeens", "tupiens", "tupinambaens",
"tupinens", "tupiraçabens", "tupiramens", "tupiratinens",
"tupitinguens", "turbant", "turbilhonant", "turiaçuens",
"turiassonens", "turielens", "turinens", "turiubens",
"turmalinens", "turonens", "turonicens", "turtuveant",
"turumanens", "tururuens", "turvaniens", "turvelandens",
"turvens", "turviniens", "turvolandens", "tusculandens",
"tutiens", "tutoiens", "uacananens", "uacariens",
"uacociens", "uacurizalens", "uanaracens", "uapeens",
"uaraens", "uariens", "uarinens", "uarirambaens",
"uarumens", "uatumãens", "uauaens", "uaupeens",
"ubaens", "ubaiens", "ubairens", "ubaitabens",
"ubajarens", "ubaporanguens", "ubaranens", "ubariens",
"ubatanens", "ubatens", "ubatibens", "ubatubens",
"ubaunens", "ubazinhens", "uberabens", "uberlandens",
"uberlandiens", "ubiraçabens", "ubiraçuens", "ubiraitaens",
"ubirajarens", "ubirataiens", "ubiratanens", "ubiretamens",
"ubiretens", "uburanens", "uchoens", "ucraniens",
"ugandens", "uibaiens", "uiraponguens", "uiraunens",
"uivant", "ulbiens", "ulianopolens", "uliens",
"ulissiponens", "ulsteriens", "ultrajant", "ultralev",
"ultravident", "ulubrens", "ululant", "umanens",
"umariens", "umaritubens", "umarizalens", "umbaraens",
"umbaubens", "umbeliform", "umbelinens", "umbuens",
"umburanens", "umburaninhens", "umburatibens", "umburetamens",
"umbuzeirens", "umiriens", "umirinens", "umuaramens",
"unaiens", "unânim", "unciform", "undant",
"unens", "unidens", "uniflorens", "uniform",
"unionens", "univalent", "univalv", "universalizant",
"universens", "uomaloliens", "upacaraiens", "upamirinens",
"upanemens", "upatininguens", "upgrad", "uraiens",
"urandiens", "uraniens", "urariquerens", "urent",
"urgent", "urticant", "usbequ", "uscanens",
"uscens", "usinens", "uspens", "utariens",
"utent", "utiaritiens", "uticens", "utinguens",
"uvaens", "uvaiens", "uvalens", "uxamens",
"uxitubens", "uzalens", "uzbequ", "uzitens",
"vacacaiens", "vacaricens", "vacariens", "vacens",
"vacilant", "vacinant", "vadens", "vagant",
"vagens", "vaguens", "vaiamontens", "valadarens",
"valanens", "valanginens", "valboens", "valcovens",
"valdelandens", "valdens", "valdigens", "valencens",
"valenciens", "valens", "valent", "valentens",
"valentinens", "valerianens", "valeriens", "valinhens",
"valonguens", "valpacens", "valparaisens", "valsant",
"valverdens", "vandeens", "vanderleiens", "vanens",
"vangloriens", "vanuatens", "vanuatuens", "vapicens",
"vapincens", "vaquejadorens", "vardagatens", "varganens",
"vargeanens", "vargedens", "varginhens", "vargitens",
"varguens", "variant", "varjanens", "varjotens",
"varpens", "varsoviens", "vartens", "varzeanens",
"varzeariens", "varzedens", "varzeens", "varzelandens",
"varzelandiens", "varzinhens", "vasconcelens", "vascuens",
"vasiens", "vasquesens", "vassourens", "vastens",
"vaudens", "vauens", "vauthierens", "vauzinhens",
"vazamorens", "vazant", "vazantens", "vazlandens",
"vazlandiens", "veadeirens", "veadens", "veadinhens",
"vedant", "veement", "veiguens", "veimarens",
"veirens", "velabrens", "velanens", "velaniens",
"velens", "veliens", "venanciens", "venceslauens",
"venceslauzens", "vendeens", "vendens", "venenens",
"venens", "venetens", "venezens", "ventaniens",
"ventripotent", "veraneant", "veranens", "veravens",
"vercelens", "verd", "verdejant", "verdejantens",
"verdens", "verdunens", "veredens", "veredinhens",
"vereens", "verens", "vergiliens", "verissimens",
"vermelhens", "vermiform", "vermoilens", "verodunens",
"veronens", "verruguens", "vertent", "vertentens",
"vesicant", "vesonciens", "vespasianens", "vestiens",
"veterocretens", "vetonens", "vetuloniens", "viaciens",
"viadutens", "viajant", "viamanens", "viamonens",
"viandant", "vianens", "vianopolens", "viatens",
"viboniens", "vibrant", "vic", "vicejant", "vicelens",
"vicenciens", "vicentadens", "vicent", "vicentens",
"vicentinens", "viciant", "vidalens", "videirens",
"vident", "vidigalens", "vidigueirens", "vidinhens",
"vieguens", "vieirens", "vienens", "vietcongu",
"vietnamens", "vigariens", "vigent", "vigiens",
"vigilant", "vilacondens", "vilafranquens", "vilanovens",
"vilhenens", "vimaranens", "vimeirens", "vimieirens",
"vimiosens", "vinciens", "vinculant", "vindobonens",
"vindonissens", "vinhaens", "vinhalens", "vinhaticens",
"vinhedens", "violetens", "viracionens", "viradourens",
"virent", "virgiliens", "virginens", "virginiens",
"virgolandens", "virmondens", "virodunens", "virunens",
"viscens", "viseens", "viseuens", "visiens",
"visitant", "vistant", "vitelens", "vitodurens",
"vitorianens", "vitoriens", "vitorinens", "vitudurens",
"vivac", "vivariens", "vivent", "vivificant",
"vizelens", "voant", "vociferant", "volant",
"volateant", "volcens", "volitant", "volsiniens",
"voltens", "voltiniens", "vorac", "voraens",
"vordens", "votant", "votorantinens", "votourens",
"votuporanguens", "voturunens", "votuveravens", "vouzelens",
"vozeant", "vulgiens", "vurtemberguens", "wagneriens",
"xambioaens", "xambioazens", "xambioens", "xambreens",
"xambrens", "xangaiens", "xanxerens", "xapuriens",
"xavant", "xavantens", "xavantinens", "xavierens",
"xaxiens", "xaxinens", "xinguarens", "xinguens",
"xiquexiquens", "zacariens", "zairens", "zambiens",
"zamens", "zanatens", "zandavalens", "zargunchant",
"zelot", "zeolit", "zigurat", "zimbabuens",
"zot", "zuriquens"])


# PALAVRAS MASCULINAS TERMINADAS EM  E == abacate - chocolate
plvrMasce = set(["abacat", "abat", "acident", "açougu",
"açud", "afluent", "alambiqu", "alard",
"alarm", "albergu", "alfaiat", "alfinet",
"alicat", "alicerc", "almanaqu", "almirant",
"alp", "altifalant", "ambient", "andaim",
"anticiclon", "antílop", "apart", "apêndic",
"apetit", "aram", "arrabald", "arrebit",
"arrecif", "arremat", "artrópod", "atabaqu",
"ataúd", "aug", "azedum", "azeit",
"azimut", "bacamart", "badulaqu", "bagarot",
"bagr", "bail", "balancet", "balaústr",
"bald", "baldram", "baleot", "balot",
"baluart", "banquet", "baqu", "barbant",
"barit", "baronet", "barret", "barrot",
"bastonet", "bataqu", "batent", "batuqu",
"bazulaqu", "belich", "bequ", "berloqu",
"berrant", "besant", "betum", "bib",
"bicam", "bich", "bicorn", "bident",
"bif", "bigod", "bilhet", "bíped",
"biscat", "bisont", "bispot", "bitat",
"bit", "bivaqu", "blagu", "blecaut",
"bobech", "bocet", "bod", "bof",
"bofet", "boicot", "boiot",
"bond", "bonifrat", "bosqu", "bosquet",
"bot", "botoqu", "box", "bracelet",
"brand", "breloqu", "brequ", "bricabraqu",
"brilharet", "brioch", "bris", "broch",
"bronz", "brulot", "bucr", "bufet",
"bugr", "bul", "buqu", "byt",
"cabid", "cacet", "cachalot", "caciqu",
"cacoet", "cad", "cafajest", "caid",
"caingangu", "caixot", "calafat", "calhambequ",
"cálic", "calot", "camalot", "camarot",
"candomb", "canivet", "cânon", "capacet",
"capot", "caracter", "carbon", "cardum",
"carm", "carpet", "cart", "casebr",
"cassetet", "caul", "cavalet", "certam",
"chal", "chantr", "chequ", "chern",
"cheviot", "chicot", "chifr", "chisp",
"chist", "chocolat", "chop", "choqu",
"chut", "ciceron", "ciclon", "ciclop",
"cineclub", "cisn", "ciúm", "clarinet",
"clavinot", "clip", "cliqu", "clon",
"club", "coch", "cocot", "códic",
"cofr", "cognom", "coic", "coiot",
"colchet", "coldr", "colet", "coliform",
"combat", "compadr", "conclav", "cond",
"con", "confet", "confrad", "conhaqu",
"cônjug", "contrachequ", "contrast", "control",
"convit", "coqu", "corpet",
"corticoesteroid", "corticosteroid", "cortum", "costum",
"cotilédon", "couc", "covilhet",
"coxot", "crem", "crep", "creton",
"crim", "croqu", "croquet", "cual",
"cum", "cuneiform", "cunhet", "curtum",
"debat", "debiqu", "deboch", "decalqu",
"decot", "défic", "deleit", "dent",
"dervix", "desair", "desastr", "desplant", "deuteromicet",
"diabret", "diamant", "dichot", "dicinodont",
"dinamit", "diplont", "diqu", "disparat",
"disquet", "ditam", "dog", "dogu",
"dolomit", "drinqu", "driv", "duend",
"duqu", "eclips", "eland",
"elefant", "eletrochoqu", "embat", "enclav",
"enfart", "enfeit", "enlac", "entalh",
"ent", "envelop", "enxam", "escadot",
"escalop", "escalp", "escaparat", "escarlat",
"escroqu", "esmalt", "espadart", "espécim",
"espermatozoid", "espinafr", "espongiform", "esport",
"esquif", "estam", "estandart", "estand",
"esteroid", "estilet", "estip",
"estípit", "estress", "estrum", "estuqu",
"eunect", "exam", "factoid", "fad",
"fagot", "falset", "fantoch",
"fanzin", "faroest", "farolet", "fat",
"fedelhot", "feix", "ferret", "fetich",
"fiacr", "fiambr", "fidalgot", "filet",
"filhot", "filiform", "filip", "film",
"filmet", "fiord", "fland", "flandr",
"flet", "floret", "fogon", "foguet",
"fol", "fon", "frad", "franchinot",
"frand", "frangot", "franquisqu", "fraqu",
"freg", "freir", "frem", "frescalhot",
"fret", "front", "fust", "fut",
"gabinet", "galeot", "galhardet", "galop",
"garrot", "gendarm", "gen", "geod",
"gerifalt", "germ", "gibanet", "gigabyt",
"ginet", "glid", "gneiss", "god",
"godem", "gol", "golp", "grafit",
"gramofon", "gravam", "grud", "grumet",
"guach", "guant", "gud", "gued",
"gum", "halter", "haplobiont", "hectar",
"heteroterm", "heterotransplant", "hidrant", "hierofant",
"holofot", "horizont", "iat",
"icebergu", "ien", "ilhot", "infart",
"ingr", "inguch", "inham", "interclub",
"interfon", "invit", "iogurt", "ipr",
"jab", "jardinet", "jarret", "jegu",
"jip", "joanet", "joguet", "joul",
"jul", "labrost", "laçarot", "lacet",
"lacr", "lanch", "larot", "laud",
"lazaron", "legum", "leit", "lembret",
"lêmur", "lequ", "liam", "licorn",
"liern", "limonet", "linc", "linet",
"lingot", "lipot", "livret", "lord",
"lot", "lovelac", "lugr", "lum",
"lustr", "macadam", "mac", "macet",
"machet", "magaref", "magazin", "magot",
"mainfram", "malamut", "malot", "mamut",
"mangot", "mantelet", "mapuch", "margrav",
"mascat", "massacr", "massam",
"mastodont", "matalot", "mavort", "maxilíped",
"maxix", "megabyt", "meiot", "melcatref",
"membranofon", "mendengu", "meninot", "mentastr",
"mequetref", "mestr", "microfilm", "microfon",
"micronutrient", "milagr", "mil", "milhafr",
"minaret", "minuet", "mirant", "miron",
"mixomicet", "molecot", "molequ", "monandengu",
"mong", "monovolum", "moringu", "morrot",
"mosquet", "mot", "motet", "mouch",
"mous", "mouss", "mucequ", "mujiqu",
"mussequ", "naip", "nair", "negrum",
"nepent", "nom", "nort", "novilhot",
"num", "nun", "odr", "ogr",
"optimat", "orat", "origon", "pacot",
"padr", "pagod", "palacet", "palanqu",
"palet", "palpit", "palud", "palustr",
"papelot", "parch", "parêntes", "parqu",
"passaman", "passaport", "pechisbequ", "peix",
"pelam", "pelot", "pen", "pent",
"pep", "pequenot", "perendengu", "perfum",
"pern", "pernoit", "pesadum", "pêsam",
"petimetr", "petit", "pexot", "picl",
"pilot", "pinot", "pintalegr", "pintalegret",
"piparot", "pipelin", "pipot", "piqueniqu",
"piquet", "pistolet", "pivet", "ponch",
"pontífic", "porr", "porret", "post",
"pot", "potich", "prenom", "príncip",
"pronom", "pseudópod", "puf", "quacr",
"queixum", "quep", "quequ", "quib",
"quilat", "quilobit", "quiosqu", "quir",
"quíri", "quirit", "quitut", "quixot",
"quocient", "rabanet", "radiotelefon", "raid",
"ralhet", "ramalhet", "ramilhet", "ran",
"rapazot", "rebenqu", "rebit", "reboqu",
"rebot", "recalqu", "recif", "reclam",
"record", "recort", "redingot", "regim",
"remoqu", "repeniqu", "repiquet", "reproch",
"requif", "requifif", "retoqu", "retrofoguet",
"reviret", "ricochet", "rifl", "ringu",
"rizópod", "rob", "robl", "rodet",
"rolet", "romanc", "romancet", "roqu",
"roquet", "rosbif", "roug", "sabonet",
"sabr", "sainet", "saiot", "salamalequ",
"salmonet", "salmoniform", "sanduích", "sangu",
"saqu", "sax", "saxofon",
"semblant", "semestr", "serrot", "sil",
"silicon", "sinet", "sit", "skat",
"slid", "sobrenom", "soquet", "sorvet",
"sotaqu", "subíndic", "tabef", "tablet",
"tacap", "tal", "talh", "talip", "talud",
"talvegu", "tamboret", "tanqu", "tapet",
"tapum", "tarambot", "tatam", "teatrofon",
"telefilm", "telefon", "tentam", "test",
"timbal", "timbr", "tim", "tiqu",
"tíquet", "tiranet", "toalet", "toalhet",
"toilet", "told", "tolet", "tomat",
"topet", "torniquet", "tótem", "trâmit",
"transport", "traqu", "traquet", "trimestr",
"trombon", "trompet", "truqu", "trust",
"tul", "ubr", "uísqu", "upanixad",
"vagonet", "valet", "vasilham", "velam",
"velhot", "velocíped", "ventr", "verbet",
"verdet", "verm", "vernissag", "versalet",
"vexam", "videocasset", "videoclip", "videofon",
"videogam", "vilancet", "vim", "vinagr",
"vindim", "vinhet", "virot", "viscond",
"voltaret", "volum", "vombat", "vórtic",
"xail", "xal", "xarop", "xenotransplant",
"xerif", "xilofon", "xiquexiqu", "zagalot",
"ziguezagu"])



# PALAVRAS FEMININAS TERMINADAS EM  E == alface - comadre
plvrFeme = set(["aeronav", "aguardent", "alfac", "altitud",
"amiloidos", "amizad", "amplitud", "anális",
"antítes", "apólic", "apóstrof", "apoteos",
"art", "artrit", "artros","árvor",
"atitud", "autoclav", "avalanch", "av",
"bacant", "bacilos", "bacinet", "bacoric",
"bacterios", "barbári", "bas", "basidiomicet",
"beatic", "bebedic", "bérber", "bernardic",
"bipatent", "bipirâmid", "bisbilhotic", "bizarric",
"boat", "bólid", "borboletic", "boutiqu",
"brejeiric", "bronquit", "bront", "burric",
"bursit", "butiqu", "cabin", "calech",
"caminhonet", "camionet", "cariátid", "cári",
"carn", "carolic", "casset", "catástrof",
"caturric", "celulas", "celulos", "chanc",
"charg", "charret", "chatic", "chav",
"chican", "chochic", "cíclad", "clâmid",
"claqu", "class", "clav", "comadr",
"commoditi", "coort", "coqueluch", "côrt",
"cortesanic", "couv", "crech", "crendic", "cretinic",
"criancic", "cris", "cúspid", "danaid",
"debêntur", "démarch", "dendrit", "dengu",
"denguic", "dermatit", "dermatos", "diaclas",
"diális", "diástol", "diátes", "diatrib",
"dies", "dioces", "doidic", "dos",
"doutoric", "dríad", "echarp", "efígi", "efemérid",
"elips", "elit", "enchent","ênfas", "enfiteus",
"enquet", "enquêt", "epiderm", "epífis",
"epígraf", "epistax", "equimos", "equinoderm",
"equip", "esfing", "esofagit", "espaçonav",
"espádic", "espéci", "esquisitic", "estalactit",
"estalagmit", "estant", "estenos", "estirp",
"estouvic", "estrangeiric", "estrig", "estrof",
"estroinic", "estultic", "eumênid", "exeges",
"exopeptidas", "fac", "faceiric", "falang", "fanfarronic",
"fanfreluch", "farfalhic", "fas", "fauc",
"febr", "fidalguet", "filel", "finitud",
"foic", "fom", "fosfatas", "fouc",
"francesic", "fras", "fraud", "frent",
"frol", "frond", "furunculos", "gabardin",
"gabolic", "gad", "gaf", "gaiatic",
"galanic", "galeguic", "gangu", "garag",
"garçonet", "garotic", "garridic", "garrulic",
"gastrit", "gastroenterit", "gatunic", "gênes",
"gengivit", "gent", "gland", "glicogenos",
"glomerulonefrit", "glot", "gnaiss", "górgon",
"grad", "grev", "grif", "grip",
"gulodic", "gulosic", "habitud", "hast",
"hecatomb", "hélic", "hemapófis", "hematos",
"hemodiális", "hemoptis", "hepatit", "hexos",
"hidrocel", "hidrolas", "hidrólis", "hipérbol",
"hipótes", "hipotipos", "idiotic", "imposturic",
"imundic", "imundíci", "incompletud", "infinitud",
"inimizad", "inquietud", "intempéri", "interfac",
"interfas", "intrujic", "invencionic", "jambacoss",
"janotic", "jogralic", "jubart", "judit",
"juventud", "ladroíc", "lag", "laj",
"lambaric", "lamechic", "lanchonet", "land",
"lápid", "laring", "lassitud", "latitud",
"lebr", "leishmanios", "lent", "leptomital",
"limousin", "limusin", "linfatit", "lingeri",
"lipas", "literatic", "litóclas", "longitud",
"louvaminhic", "macaquic", "madam", "madracic",
"madr", "magnitud", "maiones", "majestad",
"majoret", "malandret", "malandric", "malaquit",
"malcriadic", "maluquet", "maluquic", "manchet",
"mandalet", "mandaret", "manhosic", "mapl",
"maquet", "maquiavelic", "maquillag", "marinet",
"marionet", "mariquic", "maritorn", "marotic",
"marquis", "martinet", "mascot", "mastit", "matrafic",
"megalópol", "meiguic", "ménad", "mênad",
"mening", "meningit", "meninic", "merc",
"mesmic", "mesóclis", "metad", "metamorfos",
"metástas", "metempsicos", "métop", "metrópol",
"mexeriquic", "micos", "minisséri", "miríad",
"miten", "modernic", "molenguic", "momic",
"mort", "multitud", "muss", "náiad",
"nav", "necrópol", "necros", "nefelibatic",
"nereid", "neuros", "nev", "nevros",
"niquic", "noit", "nout", "nuanc",
"nucleas", "nucleossíntes", "oceanit", "olimpíad",
"onzenic", "ordinaric", "oréad", "osteoartros",
"overdos", "palermic", "pan", "pantomimic",
"paráfras", "pared", "parlapatic", "part",
"parvoíc", "patent", "patetic", "penc",
"penélop", "pentos", "peptidas", "peraltic",
"performanc", "perífras", "perliquitet", "perric",
"pevid", "picap", "pieguic", "pios",
"pirâmid", "pirit", "planíci", "pleb",
"plêiad", "pleópod", "pol", "politiquic",
"pont", "popelin", "potestad", "prax",
"prec", "proglótid", "prol", "proteas",
"prótes", "psicos", "quermess", "quietud",
"quimas", "quitinet", "raqu", "raquet",
"red", "repris", "retret", "revanch",
"rosal", "sabujic", "sandic", "saúd",
"sed", "seg", "sement", "séri",
"serp", "serpent", "sílfid", "simbios",
"similitud", "símplic", "sinaps", "síncop",
"síndrom", "sinops", "sintax", "síntes",
"siren", "sirt", "sobrevest", "socialit",
"solicitud", "sort", "soviet", "sovinic",
"subclass", "subenfiteus", "subespéci", "subpart",
"suít", "superclass", "superfíci", "sutach",
"tab", "tagarelic", "talmud",
"tard", "tart", "tempestad", "tendinit",
"tes", "teslinic", "tétrad", "tirit",
"toilett", "tolic", "tontic", "torpitud",
"toss", "trafulhic", "trampolinic", "transaminas",
"trapalhic", "traquinic", "tratantic", "tremeliqu",
"trios", "tripartit", "trombos", "truanic",
"trud", "trup", "tub", "tuberal",
"tuberculos", "turboélic", "turpitud", "urb",
"valis", "vedet", "velhic",
"vicissitud", "vigaric", "viros",
"virtud", "vitrin", "voltair", "vontad",
"zoonos"])

# PALAVRAS MASCULINAS TERMINADAS EM  Z == CARTAZ
plvrMascz = set(["albatro", "alborno", "alcatra", "alcatru", "alfo",
"algero", "algo", "almofari", "andalu", "antra",
"arcabu", "arca", "arroa", "bissetri", "caba",
"capata", "capu", "carta", "chafari", "chamari",
"entreme", "gilva", "gora", "jae", "ja",
"machaca", "mati", "ma", "nari", "pratarra",
"rapa", "reve", "ri", "verni", "xadre"])



# PALAVRAS FEMININAS TERMINADAS EM  Z == cruz - gravidez
plvrFemz = set(["atri", "avestru", "cicatri", "codorni", "cru",
"diretri", "eletromotri", "gravide", "imperatri", "languide",
"lu", "matri", "meretri", "motri", "no",
"nude", "nutri", "operatri", "pa", "pequene",
"perdi", "polide", "rectri", "sordide", "tamari",
"tectri", "timide", "ve", "vo"])



# PALAVRAS DE DOIS GENEROS TERMINADAS EM  Z == CAPAZ
plvrInvaz = set(["alfara", "aprendi", "atro", "auda", "bre",
"camba", "capa", "contuma", "efica", "fala",
"famana", "feli", "fero", "fuga", "incapa",
"inefica", "infeli", "lamba", "loqua", "menda",
"mina", "morda", "ori", "perspica", "pertina",
"peti", "prima", "pugna", "roa", "saga",
"sequa", "soe", "suspica", "tena", "torqua",
"velo", "vera", "viva", "vora"])





# EXCECAO PARA PALAVRAS TERMINADAS EM  inho == Carlinhos - Marquinhos
excInho = set(["Carl", "carl", "Marqu", "marqu"])

# EXCECAO PARA PALAVRAS MASCULINAS TERMINADAS  o == MESMO
excMist = set(["Cas", "cas", "Junt", "junt", "Mesm", "mesm"])

# EXCECAO PARA PALAVRAS MASCULINAS TERMINADAS  a == MESMA
excMista = set(["Casa", "casa", "Larga", "larga", "Mesma", "mesma", "Minha", "minha", "Sua", "sua", "Tua", "tua"])

# PALAVRAS QUE ATRAEM A PROCLISE
aProclise = set(["abaixo", "acaso", "acima", "acinte", "acolá", "adentro", "adiante",
"afinal", "afora", "agora", "aí", "além", "algo",
"alguém", "algum", "ali", "amanhã", "amiúde",
"antes", "aquela", "aquelas", "aquele", "aqueles", "aquém", "aqui",
"aquilo", "assaz", "assim", "atrás", "bastante", "bem", "breve",
"cá", "cedo", "certa", "certo", "como", "conforme", "debalde", "decerto",
"demais", "demasiado", "dentro", "depois", "depressa", "devagar",
"diversas", "diversos", "embaixo", "enfim", "essa", "essas", "esse",
"esses", "esta", "estas", "este", "estes", "hoje", "imenso", "isso",
"isto", "lá", "longe", "mais", "mal", "melhor", "menos", "muita",
"muitas", "muito", "muitos", "ninguém", "ontem", "ora", "outra", "outras",
"outro", "outrora", "outros",
"perto", "pior", "porque", "porventura", "pouca", "poucas", "pouco", "pouco",
"poucos", "quando", "quanto", "quão", "quase", "quer", "quiçá", "sempre",
"sim", "só", "talvez", "tampouco", "tanto", "tarde", "tem", "têm", "temos",
"tenha", "tenham",  "tenho", "teria", "teremos", "teriam", "teve", "tinha", "tinham", "tiveram", "toda",
"todas", "todo", "todos", "tudo", "várias", "vários"])

# ATRAI PROCLISE
mProclise = {"abaixo": "Abaixo",
"acaso": "Acaso",
"acima": "Acima",
"acinte": "Acinte",
"acolá": "Acolá",
"adentro": "Adentro",
"adiante":"Adiante",
"afinal": "Afinal",
"afora": "Afora",
"agora": "Agora",
"aí": "Aí",
"além": "Além",
"algo": "Algo",
"alguém": "Alguém",
"algum": "Algum",
"ali": "Ali",
"amanhã": "Amanhã",
"amiúde": "Amiúde",
"antes": "Antes",
"aquela": "Aquela",
"aquelas": "Aquelas",
"aquele": "Aquele",
"aqueles": "Aqueles",
"aquém": "Aquém",
"aqui": "Aqui",
"aquilo": "Aquilo",
"assaz": "Assaz",
"assim": "Assim",
"atrás": "Atrás",
"bastante": "Bastante",
"bem": "Bem",
"breve": "Breve",
"cá": "Cá",
"cedo": "Cedo",
"certa": "Certa",
"certo": "Certo",
"como": "Como",
"conforme": "Conforme",
"debalde": "Debalde",
"decerto": "Decerto",
"demais": "Demais",
"demasiado": "Demasiado",
"dentro": "Dentro",
"depois": "Depois",
"depressa": "Depressa",
"devagar": "Devagar",
"diversas": "Diversas",
"diversos": "Diversos",
"embaixo": "Embaixo",
"enfim": "Enfim",
"essa": "Essa",
"essas": "Essas",
"esse": "Esse",
"esses": "Esses",
"esta": "Esta",
"estas": "Estas",
"este": "Este",
"estes": "Estes",
"hoje": "Hoje",
"imenso": "Imenso",
"isso": "Isso",
"isto": "Isto",
"lá": "Lá",
"longe": "Longe",
"mais": "Mais",
"mal": "Mal",
"melhor": "Melhor",
"menos": "Menos",
"muita": "Muita",
"muitas": "Muitas",
"muito": "Muito",
"muitos": "Muitos",
"ninguém": "Ninguém",
"ontem": "Ontem",
"ora": "Ora",
"ou": "Ou",
"outra": "Outra",
"outras": "Outras",
"outro": "Outro",
"outrora": "Outrora",
"outros": "Outros",
"perto": "Perto",
"pior": "Pior",
"porque": "Porque",
"porventura": "Porventura",
"pouca": "Pouca",
"poucas": "Poucas",
"pouco": "Pouco",
"poucos": "Poucos",
"quando": "Quando",
"quanto": "Quanto",
"quão": "Quão",
"quase": "Quase",
"quer": "Quer",
"quiçá": "Quiçá",
"sempre": "Sempre",
"sim": "Sim",
"só": "Só",
"talvez": "Talvez",
"tampouco": "Tampouco",
"tanto": "Tanto",
"tarde": "Tarde",
"tem": "Tem",
"têm": "Têm",
"temos": "Temos",
"tenha": "Tenha",
"tenham": "Tenham",
"tenho": "Tenho",
"teria": "Teria",
"teremos": "Teremos",
"teriam": "Teriam",
"teve": "Teve",
"tinha": "Tinha",
"tinham":"Tinham",
"tiveram": "Tiveram",
"toda": "Toda",
"todas": "Todas",
"todo": "Todo",
"todos": "Todos",
"tudo": "Tudo",
"várias": "Várias",
"vários": "Vários"}



# VERBOS REGULARES TERMINADOS EM AR - RADICAL
plvrVbrar = set(["abaf", "abaix", "abal", "abaliz", "abalo", "aban",
"abarc", "abarrot", "abasm", "abast", "abelg", "abelh",
"abelhud", "abenço", "abest", "abgreg", "abilol", "abjudic",
"abjug", "abjurg", "ableg", "abneg", "aboc", "abodeg",
"abomin", "abon", "aborbot", "abort", "aboug", "abraç",
"abrand", "abrasileir", "abrevi", "abrig", "abrilhant", "abund",
"abundanci", "abus", "acab", "acalent", "acalm", "acam",
"acamp", "acanel", "acarici", "acarinh", "acarret", "acartol",
"acasal", "acastel", "acat", "acaval", "acebol", "aceit",
"acen", "acentu", "acerc", "acert", "acess", "acetin",
"ach", "achancel", "achat", "acheg", "achocalh", "achocolat",
"acident", "acidific", "aciment", "acinzent", "acion", "acirolog",
"acirr", "aclam", "aclar", "aclimat", "aclimatiz", "aco",
"acobert", "acocor", "acoit", "açoit", "acomod", "acompanh",
"aconch", "aconchav", "aconcheg", "acondicion", "acondiment", "aconselh",
"aconsonant", "acopl", "acorcov", "acorcund", "acord", "acorrent",
"acortin", "acoss", "acost", "acostum", "acotovel", "acovard",
"acredit", "acrescent", "acrianç", "acristan", "acu", "açucar",
"acumul", "acunh", "acupuntur", "acur", "acus", "acutel",
"adapt", "adarg", "adeg", "adelg", "adeng", "adens",
"adentr", "aderg", "adi", "adiant", "adicion", "adinheir",
"adit", "adivinh", "adjetiv", "adjetiviz", "adjudic", "adjunt",
"adjur", "adjutor", "adjuv", "adlig", "administr", "admir",
"admoest", "admonest", "adnomin", "adnot", "adnumer", "adoç",
"adocic", "adoent", "ador", "adot", "adreg", "adub",
"adul", "adulter", "adun", "adverbi", "advers", "advincul",
"advoc", "advog", "advult", "aerific", "aeriz", "aerofotograf",
"aeromodel", "aeroplan", "aeroport", "aerossoliz", "aerossond", "aerotransport",
"afag", "afam", "afan", "afarinh", "afast", "afeiço",
"afeijo", "afemin", "afet", "afezo", "afi", "afianç",
"afidalg", "afili", "afin", "afinc", "afirm", "afivel",
"afix", "aflameng", "afob", "afof", "afog", "afrescalh",
"afront", "afroux", "afugent", "afund", "afunil", "ag",
"agach", "agaleg", "agarr", "agasalh", "agenci", "agend",
"agigant", "agiliz", "agit", "aglomer", "aglutin", "agoniz",
"agrav", "agreg", "agrup", "aguard", "aguç", "aguent",
"agulh", "ajeit", "ajindung", "ajoelh", "ajud", "ajudeng",
"ajunt", "alag", "alambr", "alaranj", "alarg", "alastr",
"alaud", "alavanc", "alberg", "alç", "alcanç", "alcooliz",
"alcovit", "alcunh", "aldeag", "alegr", "aleij", "aleit",
"alent", "alert", "aletarg", "alevant", "alfabetiz", "alfandeg",
"algem", "alguerg", "ali", "alianç", "alicerç", "alici",
"alien", "alij", "aliment", "alinh", "alinhav", "alis",
"alist", "alivi", "almeceg", "almej", "almoç", "almofad",
"almondeg", "aloc", "aloj", "along", "alter", "altern",
"altiv", "alu", "alucin", "alug", "alum", "alumbr",
"alumi", "aluvi", "alvej", "alverg", "alvor", "alvorat",
"alvorej", "alvoriç", "alvoroç", "alvorot", "alvot", "am",
"amaci", "amag", "amaldiço", "amans", "amanteig", "amarel",
"amarr", "amarrot", "amartel", "amasi", "amass", "amatung",
"amatut", "ambicion", "ambient", "ameaç", "amealh", "amedront",
"amenin", "amig", "amim", "amineir", "amofin", "amoit",
"amol", "amold", "amoldur", "amolec", "amoleg", "amolent",
"amolest", "amolg", "amonto", "amoquec", "amoraliz", "amordaç",
"amoren", "amorfanh", "amorfiz", "amorisc", "amorn", "amortiz",
"amoss", "amostr", "amotin", "amour", "ampar", "ampli",
"amplific", "amput", "amu", "amulat", "amulher", "amulhereng",
"amumi", "anaboliz", "anafrag", "analgiz", "analis", "anarcis",
"anarquiz", "ancor", "and", "andaim", "andarilh", "andej",
"anedotiz", "anegr", "anegralh", "anel", "anex", "angari",
"angeliz", "angliz", "angul", "angusti", "anilh", "anim",
"animaliz", "aniversari", "anodiz", "anoit", "anoj", "anot",
"anov", "anovel", "anoxi", "ant", "antecip", "antedat",
"antepar", "antepass", "antepost", "anterioriz", "anualiz", "anul",
"anumer", "anunci", "anuvi", "apacific", "apadrinh", "apadro",
"apag", "apaixon", "apalavr", "apalm", "apalp", "apanh",
"apar", "aparament", "aparcel", "aparelh", "aparent", "aparentel",
"aparoqui", "aparreir", "apart", "aparv", "aparvalh", "aparvej",
"aparvo", "apascent", "apasco", "apassiv", "apastor", "apatron",
"apatrulh", "apaulist", "apaut", "apavor", "apavoriz", "apedr",
"apedrej", "apeg", "apej", "apel", "apelid",
"apenh", "apenhor", "apens", "apension", "apequen", "aperfeiço",
"aperr", "apert", "aperu", "apesent", "apespont", "apesso",
"apincel", "apinh", "apit", "aplac", "aplain", "aplan",
"aplic", "aplum", "apoder", "apont", "apoquent", "aport",
"aportinh", "aportugues", "aposent", "aposs", "apost", "apostil",
"apostilh", "apostol", "apostrof", "apostur", "apotec", "apotent",
"apraz", "apreci", "apresent", "apresilh", "apress", "aprimor",
"aprocheg", "aprofund", "apront", "apropri", "aproveit", "aproxim",
"aprum", "aquartel", "aquebrant", "aqued", "aquiet", "ar",
"aram", "arbitr", "arboriz", "arc", "ardig", "arej",
"aren", "areng", "argol", "argument", "arisc", "arm",
"armazen", "armeng", "arquitet", "arquiv", "arraig", "arranc",
"arranç", "arranch", "arranh", "arranj", "arras", "arrast",
"arrebanh", "arrebat", "arrebent", "arrebic", "arrebit", "arrecad",
"arredr", "arregaç", "arregal", "arreganh", "arregiment", "arregr",
"arregueir", "arreig", "arremang", "arremat", "arreneg", "arreng",
"arrepi", "arrepic", "arrig", "arrisc", "arroch", "arrog",
"arrol", "arromb", "arrot", "arrox", "arru", "arruaç",
"arruf", "arrug", "arrum", "arrupi", "articul", "arvor",
"arvorej", "asfalt", "asfixi", "asneir", "aspir", "ass",
"assalari", "assalt", "assanh", "assedi", "assegur", "asseit",
"assent", "assert", "assexualiz", "assimil", "assin", "assinal",
"asso", "assoalh", "assobi", "associ", "assol", "assombr",
"assopr", "assosseg", "assunt", "assust", "atabalho", "atabern",
"atabu", "atac", "atalh", "atanaz", "ataref", "atartarug",
"atenaz", "atent", "atenu", "aterr", "aterriss", "aterriz",
"aterroriz", "atest", "atiç", "atin", "atir", "ativ",
"ato", "atoalh", "atocai", "atoch", "atoiç", "atol",
"atomiz", "aton", "atordo", "atorment", "atrac", "atraiço",
"atras", "atrel", "atribul", "atricul", "atrit", "atrofi",
"atropel", "atu", "atualiz", "atur", "aturg", "audit",
"auditor", "aug", "augur", "aument", "ausent", "auspici",
"autentic", "autentific", "autodeslig", "autograf", "autolig", "automedic",
"automotiv", "autoproclam", "autoprogram", "autopropag", "autopsi", "autoriz",
"autorrevog", "auxili", "avacalh", "avali", "avaliz", "avanç",
"avancarg", "avarg", "avari", "avassal", "aventur", "averb",
"averdug", "averdung", "avergonh", "avermelh", "avex", "avi",
"avis", "avist", "aviv", "avo", "avult", "azar",
"azed", "azeit", "azoug", "bab", "bafor", "bag",
"bagunç", "bail", "bainh", "baix", "bajog", "bajul",
"balanç", "balang", "baliz", "banc", "banh", "bard",
"barganh", "barr", "barreg", "barreir", "barric", "bast",
"batalh", "batiz", "batuc", "beberic", "beij", "beir",
"benefici", "berr", "berreg", "bestific", "betum", "bexig",
"bibliograf", "bifurc", "bimb", "bin", "bing", "biodegrad",
"biodiversific", "bioensai", "biofilosof", "biofortific", "biograf", "bisbilhot",
"bisnag", "bitol", "blef", "blog", "bocej", "bocel",
"boch", "bochacr", "bochech", "bochinch", "bod", "bodej",
"bof", "bofej", "bofet", "bog", "boiç", "boicot",
"boir", "boj", "bol", "bolach", "bolc", "bolç",
"bolcheviz", "bolet", "bolh", "bolich", "bolin",
"bomb", "bombard", "bombej", "bombi", "bombic", "bonanç",
"bonç", "bond", "bonderiz", "bonec", "bong", "bonh",
"bonific", "boquej", "boquiz", "borbolet", "borborej", "borborinh",
"borbot", "borboto", "borboton", "borbulh", "borc", "borcel",
"bord", "bordej", "borr", "borraç", "borrat", "borreg",
"borriç", "borrif", "borrisc", "bosquej", "bosquerej", "boss",
"bost", "bostej", "bot", "botec", "boto", "botoc",
"bouç", "bouch", "bour", "bovariz", "box", "brab",
"bracej", "bracteol", "brad", "bradej", "bralh", "bram",
"bramaniz", "brand", "branquej", "braquicefaliz", "bras", "brasilianiz",
"brasilific", "brasiliz", "braso", "brason", "brav", "bravej",
"brec", "brech", "breganh", "brejeir", "bretanhiz", "bretech",
"bretes", "bretess", "brevet", "brevi", "brid", "brig",
"brilh", "brilhant", "brinc", "brincalh", "brincalhot", "brind",
"brioc", "briquet", "briquit", "bris", "brit", "britaniz",
"broc", "broch", "brol", "brolh", "brom", "broncograf",
"broquel", "brosl", "bross", "brot", "brotoej", "brox",
"brum", "brumaliz", "brutaliz", "brutific", "brux", "bubui",
"bubuluc", "buçal", "bucoliz", "budo", "buf", "bufarinh",
"bugi", "bugigang", "buibil", "bul", "bulat", "bulev",
"bulh", "bumb", "bund", "bung", "bungul", "bunh",
"buquin", "burac", "burburej", "burburij", "burburinh", "burg",
"burgues", "buril", "burl", "burocratiz", "burr", "burric",
"burrific", "busc", "buscul", "bussol", "but", "butuc",
"buzeg", "buzi", "buzin", "buzul", "cabal", "cabed",
"cabl", "cabograf", "caboroc", "cabort", "cabot", "cabotin",
"cabouc", "cabraz", "cabrej", "cabriol", "cabrit", "cabroc",
"cabruc", "cabuet", "cabul", "caç", "caçamb", "cacarej",
"cacet", "cach", "cachet", "cachimb", "cachin", "cacho",
"cachoeir", "cachuch", "cacic", "cacif", "cacimb", "caço",
"cacofoni", "cacofoniz", "cacograf", "cacuminaliz", "cadastr", "cadaveriz",
"cadel", "cadenci", "cadinh", "cadmi", "cadmium", "caduc",
"cafang", "cafein", "cafel", "cafif", "caftin", "caftiniz",
"cafulet", "cafumb", "cafun", "cafung", "cag", "cagaliz",
"caguet", "cai", "caibr", "cainç", "cairel", "caititu",
"caixeir", "Cajam", "cal", "calabariz", "calabrot", "calaceir",
"calafet", "calafri", "calamistr", "calamoc", "calandr", "calapult",
"calc", "calç", "calcet", "calcific", "calcin", "calcitr",
"calcograf", "calcul", "calefri", "calej", "calembur", "calemburg",
"calemburiz", "calendariz", "calh", "calhanc", "calibr", "calicul",
"caligin", "caligraf", "calm", "calorific", "caloriz", "caluni",
"calv", "calvej", "camarad", "camarinh", "camb", "cambai",
"cambalach", "cambalhot", "cambi", "cambit", "cambo", "cambr",
"cambul", "camburil", "caminh", "camoniz", "campainh",
"campan", "campanul", "campean", "campeir", "campin", "camufl",
"canaliz", "cancaniz", "cancel", "cancer", "canceriz", "candi",
"candidat", "candidiz", "candil", "candol", "candong", "canel",
"canet", "canfor", "cang", "canguerej", "canguinh", "canhon",
"canibaliz", "canoniz", "canoriz", "cans", "cant", "cantacant",
"cantarej", "cantaric", "cantarid", "cantarol", "cantorinh", "canud",
"canutilh", "caotiz", "cap", "capacit", "capang", "caparazon",
"capataz", "capeng", "capiang", "capin", "capisc", "capit",
"capitaliz", "capitul", "capoeir", "capot", "caprich", "caprific",
"capsul", "capt", "captur", "capuch", "caquerej", "caquin",
"caracol", "caracolej", "caracteriz", "carambol", "caramel", "carameliz",
"caraminhol", "caramunh", "carang", "caranguej", "caranguejol", "carantonh",
"carapet", "carateriz", "carbon", "carbonat", "carbonet", "carbonific",
"carbonil", "carbonitrogen", "carboniz", "carboret", "carboxil", "carbur",
"carburiz", "carcach", "carcav", "carcer", "card", "cardanh",
"cardiovasculariz", "cardiovasculiz", "carduç", "caren", "carep", "carg",
"carguej", "cari", "caric", "caricatur", "carici", "carij",
"carimb", "carmesin", "carmin", "carn", "carnaliz", "carnavaliz",
"carneir", "carnific", "caroc", "carop", "carp", "carpent",
"carpet", "carpinteir", "carpintej", "carr", "carrameliz", "carrapat",
"carreg", "carrej", "carret", "carril", "carrilh", "carrilhon",
"carrinh", "carruaj", "cart", "cartabux", "carteliz", "cartilaginific",
"cartograf", "carton", "cartorializ", "caruj", "carunch", "carvej",
"carvoej", "cas", "casal", "casamat", "casament", "casbi",
"casc", "cascabulh", "cascalh", "cascat", "cascavel", "cascavilh",
"cascunh", "caseific", "casmurr", "caspach", "casquej", "casquilh",
"casquin", "cass", "cassumbul", "castanhol", "castelhaniz", "castiç",
"castific", "castig", "castr", "castramet", "castrin", "cat",
"cataboliz", "catadupej", "catalis", "cataliz", "catalog", "catalogiz",
"catan", "cataplasm", "catapult", "catecumeniz", "catedratiz", "categoriz",
"catequiz", "cateteriz", "catimb", "cating", "catit", "cativ",
"catoliciz", "catoliz", "catoniz", "catraceg", "catrafi", "catrafil",
"catrai", "catrapisc", "catuc", "caturr", "caturrit", "cauchut",
"caucion", "caudilh", "caulific", "caulin", "cauliniz", "caurin",
"caus", "causaliz", "caustic", "caustific", "cautel", "cauteri",
"cauteriz", "cav", "cavac", "cavalg", "cavalo", "cavatin",
"cavern", "cavid", "cavil", "cavilh", "cavit", "cavoc",
"cavouc", "cavuc", "cavumbul", "caxing", "cedilh", "cefaliz",
"ceg", "ceif", "ceiv", "celebr", "celebriz", "celeum",
"celific", "celticiz", "celtiz", "celulariz", "celv", "cement",
"cendr", "censoriz", "censur", "censuriz", "centelh", "centissecul",
"centoniz", "centr", "centraliz", "centrific", "centrifug", "centupl",
"centuplic", "ceot", "cepilh", "cer", "cerabel", "cerabol",
"ceramoplasm", "ceratiniz", "cerc", "cercilh", "cerebel", "cerebraliz",
"ceresit", "cerific", "cerimoni", "cerimonializ", "ceringonh", "cern",
"cerr", "cert", "certific", "cervej", "cesariz", "cespit",
"cess", "cesur", "cetoderiv", "cetoniz", "cev", "chabaç",
"chabouc", "chaç", "chacin", "chacoalh", "chacolh", "chafalh",
"chafund", "chafurd", "chag", "chair", "chal", "chalaç",
"chaldr", "chaleir", "chalr", "cham", "chamalot", "chamboc",
"chambreg", "chameg", "chamej", "champ", "champanhiz", "chamusc",
"chanasc", "chanc", "chancel", "chanel", "chanfalh", "chanfr",
"chang", "changueir", "chanquelh", "chant", "chantaj", "chanto",
"chap", "chapej", "chapiç", "chapilh", "chapin", "chapinh",
"chapisc", "chapod", "chapolet", "chapot", "chapuç", "chapuj",
"chapulh", "chapuz", "chaquiç", "charabald", "charamel", "charang",
"charavisc", "charing", "charivariz", "charl", "charo", "charol",
"charond", "charrisc", "charru", "charut", "chasc", "chasquin",
"chatin", "chauviniz", "chavanisc", "chavasc", "chavet", "chec",
"chefi", "cheg", "cheir", "cheiret", "cheiric", "cheirin",
"cheld", "chelindr", "cheravasc", "chi", "chiang", "chiat",
"chib", "chibat", "chiç", "chicalh", "chican", "chich",
"chichel", "chichi", "chichil", "chichorrobi", "chicot", "chifr",
"chileniz", "chilid", "chilr", "chimbi", "chimp", "chin",
"chinc", "chincalh", "chinch", "chinel", "chinfr", "chinfrin",
"chiniz", "chiqueir", "chisc", "chisn", "chisp", "choc",
"chocalej", "chocalh", "choch", "chocolat", "chofer", "chofr",
"choin", "choit", "chon", "chor", "choramig", "choraming",
"chorinc", "chorococ", "chorr", "chorrilh", "chorrisc", "chot",
"choup", "chous", "chouss", "chout", "chuç", "chuch",
"chuchu", "chuet", "chuf", "chumaç", "chumb", "chumbreg",
"chumbut", "chup", "chupinh", "chupist", "chupit", "churd",
"churin", "churri", "chusm", "chut", "chuvenisc", "chuverisc",
"chuvilh", "chuvinh", "chuvisc", "ci", "cianet", "cianiz",
"cianogen", "cianos", "cianur", "cib", "cibernetiz", "cicatriz",
"cicer", "ciceronianiz", "ciceroniz", "cici", "cicliz", "cientific",
"cif", "cifr", "cigan", "cigarr", "cilh", "cilici",
"cilindr", "cimbr", "ciment", "cinc", "cinch", "cinchoniz",
"cincunval", "cineangiocardiograf", "cineangiocoronariograf", "cinecoronariograf", "cinefic", "cinegraf",
"cinematiz", "cinematograf", "cinemiz", "ciner", "cingul", "cint",
"cintil", "cintur", "cinz", "cinzel", "cipo", "cirand",
"circi", "circin", "circuit", "circunaveg", "circuncentr", "circuncid",
"circuncircul", "circund", "circundut", "circungir", "circunrod", "circunson",
"circunspeccion", "circunspecion", "circunst", "circunstanci", "circunstancializ", "circunstancion",
"circuntorn", "circunvag", "circunval", "circunvizinh", "circunvo", "ciriliz",
"cis", "cisalh", "cisc", "cisel", "cism", "cisp",
"cistorradiograf", "cit", "citariz", "citograf", "citol", "citolis",
"cium", "civiliz", "cizan", "cizani", "cizel", "clam",
"clangor", "clangorej", "claraboi", "clarej", "clarific", "clarimostr",
"clarin", "classiciz", "classific", "claudic", "claustr", "clausul",
"clausur", "clav", "claveliz", "clemenci", "cleopatriz", "clericaliz",
"clic", "climatiz", "clinic", "clinqueriz", "clip", "clis",
"clisteriz", "cliv", "cloc", "clon", "clor", "cloraliz",
"cloretiliz", "cloritiz", "cloriz", "cloroderiv", "clorofil", "cloroform",
"cloroformiz", "clorossulfon", "clorovaporiz", "co", "coabit", "coacerv",
"coact", "coadapt", "coadat", "coadjuv", "coadministr", "coadun",
"coagit", "coaglutin", "coagment", "coagul", "coalh", "coaliz",
"coangust", "coanh", "coapostol", "coapt", "coarct", "coarrend",
"coart", "coassin", "coassoci", "coat", "coax", "cobalt",
"cobaltiz", "cobiç", "cobr", "cobrej", "coc", "coç",
"cocainiz", "coch", "cochich", "cochil", "cochin", "cocient",
"cocontrat", "cocor", "cocori", "cocoric", "cocri", "cocul",
"codej", "codemand", "codific", "codilh", "codinh", "codiz",
"codomin", "coedit", "coeduc", "coelabor", "coerd", "coestrel",
"cofi", "cofinanci", "cofinh", "cofund", "cogi", "cogit",
"cognomin", "cogul", "cogumel", "coicinh", "coif", "coigual",
"coim", "coimper", "coin", "coinch", "coinciner", "coindic",
"coinfect", "coinici", "coinquin", "cointegr", "cointeress", "coiquinh",
"coiraç", "coirman", "cois", "coisific", "coit", "coivar",
"col", "colabor", "colacion", "colapis", "colaps", "colchet",
"colcho", "colecion", "colecistograf", "colect", "coleg", "colet",
"coletiviz", "colg", "colider", "colig", "colim", "colimit",
"colitig", "colm", "colmaç", "colmat", "coloc", "colodion",
"coloniz", "colopliz", "coloqui", "color", "coloriz", "colostomiz",
"colquiciniz", "colt", "coltariz", "colubrej", "colui", "colun",
"comand", "comandit", "comarc", "combin", "comborç", "combust",
"começ", "comedi", "comemor", "comend", "comensur", "coment",
"comerci", "comercializ", "comich", "comil", "comin", "coministr",
"comir", "comisc", "comiser", "comissari", "comission", "comistur",
"comocion", "comod", "comonomeriz", "comor", "compact", "compactu",
"compadr", "compagin", "companh", "compar", "comparticip", "compartilh",
"compartiment", "compartimentaliz", "compass", "compatibiliz", "compeç", "compeg",
"compendi", "compenetr", "compens", "compermut", "compil", "complain",
"complan", "compleiço", "complement", "complet", "complex", "complexific",
"complic", "complici", "compliz", "complot", "comport", "compr",
"compromiss", "comprov", "compugn", "compuls", "compurg", "comput",
"computadoriz", "computatoriz", "computoriz", "comud", "comung", "comunic",
"comunitariz", "comuniz", "comut", "conap", "conaturaliz", "conç",
"concaniz", "concaten", "concav", "conceitu", "conceitualiz",
"concelebr", "concentr", "concentraliz", "conceptualiz", "concert", "concession",
"conch", "conchav", "concheg", "concili", "concion", "concit",
"conclam", "concomit", "concord", "concrecion", "concret", "concretiz",
"concri", "concubin", "conculc", "concultur", "concurs", "concurv",
"condecor", "conden", "condens", "condess", "condici", "condicion",
"condign", "condiment", "condo", "condrific", "conduplic", "condut",
"conect", "conectoriz", "conet", "conexion", "confabul", "confeccion",
"confeder", "confeiço", "confeit", "conferenci", "confess", "confi",
"conficion", "confidenci", "configur", "confin", "confirm", "confisc",
"conflagr", "conflit", "conflitu", "conform", "conformaliz", "confort",
"confratern", "confraterniz", "confric", "confront", "confusion", "confut",
"congalardo", "congel", "congemin", "congener", "congestion", "conglob",
"conglomer", "conglutin", "congonh", "congraç", "congraci", "congratul",
"congreg", "conhaquiz", "conjectur", "conjeitur", "conjetur", "conjug",
"conjumin", "conjunt", "conjur", "conjurament", "conlui", "conomin",
"conot", "conquist", "consagr", "consarcin", "consciencializ", "conscienciz",
"conscientiz", "consecr", "conselh", "conselheir", "consensualiz", "consert",
"conserv", "consider", "consign", "consignific", "conso", "consoant",
"consoci", "consogr", "consol", "consolid", "conson", "consonantiz",
"consorci", "conspir", "conspurc", "const", "constat", "constel",
"constern", "constip", "constitucionaliz", "consubstanci", "consult", "consum",
"consumeriz", "cont", "contabiliz", "contact", "contagi", "contamin",
"contat", "conteir", "contempl", "contemporiz", "content", "contentoriz",
"contest", "contextu", "contextualiz", "contingenci", "continu", "contorcion",
"contorn", "contorv", "contrabalanç", "contraband", "contracambi", "contracen",
"contracol", "contracunh", "contradanç", "contradeclar", "contradit", "contraestimul",
"contrafort", "contraindic", "contrainform", "contramalh", "contramand", "contramanifest",
"contramanobr", "contramarc", "contramarch", "contramin", "contraminut", "contramold",
"contramur", "contranivel", "contraorden", "contrapedal", "contrapes", "contrapont",
"contraprotest", "contraprov", "contrapunço", "contrari", "contrarraciocin", "contrarreform",
"contrarregr", "contrarreplic", "contrarrept", "contrarrevolt", "contrarrevolucion", "contrassalv",
"contrassegur", "contrassel", "contrast", "contrat", "contratel", "contratestemunh",
"contratir", "contratualiz", "contraval", "contravent", "contrist", "control",
"contubern", "conturb", "conumer", "conurb", "convalid", "convencion",
"convencionaliz", "conveni", "conventicul", "convers", "conversalh", "convibr",
"convid", "convizinh", "convoc", "convol", "convuls", "convulsion",
"coob", "coobrig", "coocup", "coofici", "coonest", "cooper",
"coopt", "coorden", "coorient", "coossific", "cop", "coparticip",
"copatrocin", "copeir", "copej", "copel", "copi", "copidesc",
"copil", "copilot", "copiograf", "coplan", "copod", "copolimeriz",
"coprocess", "coptiz", "copul", "coqueific", "cor", "coraliz",
"corandel", "corc", "corcov", "cordo", "coreograf", "corisc",
"corn", "cornach", "cornej", "cornific", "cornitromb", "cornud",
"coro", "corograf", "coronariograf", "corporaliz", "corporific", "corporiz",
"corpusculiz", "correlacion", "correlat", "corresponsabiliz", "corret", "correx",
"corric", "corrich", "corrip", "corrobor", "corroteiriz", "corruchi",
"corrug", "corrupi", "cort", "cortej", "cortilh", "cortin",
"coruj", "corunh", "corusc", "corvej", "coscor", "coscorej",
"coscuvilh", "cosic", "cosip", "cosmin", "cosmopolitiz", "cosmopoliz",
"cossegur", "costel", "costum", "costur", "cot", "cotanilh",
"cotej", "coti", "cotidianiz", "cotiz", "coton", "cotovel",
"cotul", "cotutel", "couç", "coucinh", "coudel", "couraç",
"courel", "cous", "cousific", "cout", "couv", "covach",
"covend", "cozinh", "craion", "crani", "craquej", "craticul",
"crav", "cravej", "cravin", "credenci", "credibiliz", "credit",
"crel", "crem", "cremoriz", "cren", "creolin", "creosot",
"crepit", "crepusculej", "crepusculiz", "cresp", "crest", "cretific",
"cretiniz", "cri", "cricril", "crimin", "criminaliz", "crimp",
"criocauteriz", "crioconcentr", "criodessec", "criptanalis", "criptoanalis", "criptograf",
"cris", "crisalid", "crism", "crisol", "crisp", "cristaliz",
"cristianiz", "cristologiz", "critic", "critiquiz", "criv", "crocidiz",
"crocit", "crom", "cromatiz", "cromatolis", "cromiz", "cromofotograv",
"cromofotolit", "cromofototipograv", "cromofotozincograv", "cromograv", "cromolitograv", "cromotipi",
"cromotipograv", "cromoxilograf", "cromoxilograv", "cromozincograv", "cromozincolitograv", "cromozincotipograv",
"cronic", "croniquiz", "croniz", "cronogoniometr", "cronometr", "cronoprojet",
"cronotaquimetr", "cronotelemetr", "crotoniz", "cruci", "crucific", "crucifix",
"crucigi", "cruent", "crustific", "cruz", "cuanh", "cuarl",
"cub", "cubaniz", "cubic", "cubific", "cuc", "cucul",
"cucurbit", "cucuric", "cucurit", "cuf", "cugi", "cuid",
"cuil", "cuinch", "cuinh", "cuji", "culap", "culatr",
"culc", "culim", "culmin", "culp", "culpabiliz", "cultiv",
"cultu", "culturaliz", "cumplici", "cumpriment", "cumul", "cungugut",
"cunh", "cuoz", "cur", "curariz", "curatel", "curet",
"curi", "curiband", "curibot", "curios", "curr", "curs",
"cursometr", "curuc", "curv", "curvej", "curvet", "curvigraf",
"cuspilh", "cuspinh", "cust", "custodi", "customiz", "cutil",
"cutiniz", "cutis", "cutivacin", "cutuc", "dactilograf", "dadiv",
"daguerreotip", "daltoniz", "damasquin", "damej", "dan", "danç",
"dançaric", "dançarin", "dandin", "danific", "daninh", "dard",
"dardej", "dat", "datilograf", "dealb", "dealv", "deambul",
"dearticul", "deaur", "deb", "debac", "debag", "deband",
"debang", "debel", "debentur", "debic", "debilit", "debit",
"deblater", "deboch", "deboiç", "deborc", "debord", "debot",
"debouç", "debras", "debrid", "debru", "debruç", "debulh",
"debut", "debux", "decalc", "decalv", "decamp", "decant",
"decap", "decapit", "decein", "decemplic", "decentraliz", "decep",
"decepcion", "decert", "decifr", "decilitr", "decim", "decimaliz",
"declam", "declar", "declin", "decliv", "deco", "decodific",
"decol", "decor", "decortic", "decot", "decrement", "decrepit",
"decret", "decript", "decru", "decup", "decupl", "decuplic",
"decuri", "decurs", "decuss", "ded", "dedecor", "dedetiz",
"dedic", "dedign", "dedilh", "dedur", "defas", "defec",
"defenestr", "defens", "defibril", "definh", "deflacion", "deflag",
"deflagr", "deflegm", "deflexion", "deflor", "deform", "defraud",
"defront", "defum", "defunt", "degas", "degast", "degel",
"degener", "deglobuliz", "degol", "degot", "degrad", "degran",
"degred", "degringol", "degust", "deific", "deioniz", "deit",
"deix", "dejarret", "dejeju", "dejet", "delagrimej", "delapid",
"delat", "deleg", "deleit", "delet", "delgaç", "delib",
"deliber", "delici", "delig", "delimit", "deling", "delir",
"delivr", "delong", "demand", "demarc", "demasi", "dement",
"demitiz", "democratiz", "demodul", "demograf", "demolh", "demonetiz",
"demoniz", "demonstr", "demor", "demostr", "demud", "deneg",
"denerv", "denod", "denomin", "denot", "dens", "densific",
"dent", "dentel", "denticul", "denud", "denunci", "depar",
"departamentaliz", "depauper", "depen", "dependur", "depenic", "depil",
"deplor", "deplum", "depolariz", "depolm", "depopul", "deport",
"deposit", "deprav", "deprec", "depreci", "depred", "depress",
"depur", "deput", "dequit", "deriv", "derog", "derrab",
"derraig", "derram", "derranc", "derrap", "derratiz", "derreg",
"derreig", "derrelh", "derreng", "derrepast", "derrib", "derriç",
"derrisc", "derroc", "derroç", "derroch", "derrog", "derronch",
"derrot", "derrub", "dervirg", "des", "desab", "desabaf",
"desabal", "desabaliz", "desabalro", "desabanc", "desabast", "desabeir",
"desabelh", "desabenço", "desabez", "desabilit", "desabit", "desabitu",
"desaboc", "desaboç", "desabon", "desaborc", "desabord", "desabotin",
"desaboto", "desabraç", "desabragal", "desabrasileir", "desabrig", "desabroch",
"desabrolh", "desabus", "desacadel", "desaçaim", "desacam", "desaçam",
"desacamarad", "desacamp", "desacanalh", "desacanh", "desacaravelh", "desacasal",
"desacat", "desacaudel", "desacaudilh", "desacautel", "desacaval", "desaceit",
"desaceler", "desacentu", "desacer", "desacerb", "desacert", "desacheg",
"desachorum", "desacident", "desacidific", "desacidul", "desacism", "desaclim",
"desaclimat", "desacobard", "desacoch", "desacoim", "desacoit", "desacolchet",
"desacolcho", "desacolher", "desacomod", "desacompanh", "desacompass", "desaconcheg",
"desaconselh", "desacopl", "desacoraço", "desacorço", "desacord", "desacoroço",
"desacorrent", "desacost", "desacostum", "desacoto", "desacout", "desacovard",
"desacrav", "desacravelh", "desacredit", "desacu", "desacumul", "desacunh",
"desacur", "desadapt", "desadens", "desadereç", "desadestr", "desadjetiv",
"desadministr", "desadmoest", "desador", "desadorment", "desadorn", "desaduch",
"desadul", "desadun", "desadvog", "desaer", "desafag", "desafaim",
"desafam", "desafast", "desafeiço", "desafeit", "desaferr", "desaferro",
"desaferrolh", "desaferrolho", "desafervor", "desafet", "desafi", "desafigur",
"desafilh", "desafin", "desafiuz", "desafivel", "desafix", "desafog",
"desafolh", "desafor", "desaforr", "desafortun", "desafrances", "desafregues",
"desafreim", "desafront", "desafu", "desafueir", "desafum", "desafund",
"desafusc", "desagalardo", "desagalinh", "desagalo", "desagalon", "desagarr",
"desagasalh", "desagast", "desagend", "desaglomer", "desagoir", "desagoni",
"desagour", "desagrad", "desagrav", "desagreg", "desagrilho", "desagrup",
"desaguach", "desaguax", "desaguç", "desaguis", "desain", "desair",
"desajeit", "desajouj", "desajud", "desajunt", "desajust",
"desalag", "desalap", "desalarg", "desalarm", "desalastr", "desalbard",
"desalcaliz", "desalcanç", "desalcatro", "desalcoil", "desaleald", "desalegr",
"desaleit", "desalent", "desalergiz", "desalfai", "desalfandeg", "desalforj",
"desalgem", "desali", "desalicerç", "desalij", "desalinh", "desalinhav",
"desalist", "desalivi", "desalm", "desalmej", "desaloc", "desalogen",
"desaloj", "desalquil", "desalter", "desalug", "desalumbr", "desalvor",
"desam", "desamachuc", "desamain", "desamalgam", "desamament", "desamanceb",
"desamaneir", "desamanh", "desamant", "desamantilh", "desamarinh", "desamarinheir",
"desamarr", "desamarrot", "desamartel", "desamass", "desambicion", "desambient",
"desambigu", "desambiguiz", "desamealh", "desamedront", "desamid", "desamig",
"desamim", "desamin", "desaminiz", "desamist", "desamnisti", "desamodorr",
"desamoed", "desamoest", "desamofin", "desamolg", "desamont", "desamonto",
"desamortalh", "desamortiz", "desamotin", "desamouc", "desampar", "desamu",
"desamur", "desamuriz", "desanacroniz", "desanalfabetiz", "desanc", "desancor",
"desand", "desanel", "desanex", "desangeliz", "desangusti", "desanich",
"desanil", "desanim", "desanimaliz", "desanimiz", "desaninh", "desanisti",
"desanoj", "desanul", "desanuvi", "desapacient", "desapadrinh", "desapadro",
"desapag", "desapaixon", "desaparafus", "desaparelh", "desaparent", "desapart",
"desapassion", "desapavor", "desapeçonh", "desapeçonhent", "desapeg", "desapeir",
"desapen", "desapendo", "desapens", "desaperr", "desapert", "desapiad",
"desapied", "desaplic", "desapoder", "desapolvilh", "desapont", "desapoquent",
"desaportugues", "desaposent", "desaposs", "desapreç", "desapreci", "desapresilh",
"desapress", "desaprest", "desaprimor", "desaproposit", "desapropri", "desaprov",
"desaproveit", "desaprovision", "desaproxim", "desaprum", "desapur", "desaquartel",
"desaquerenç", "desaquerenci", "desaquinho", "desar", "desaranh", "desarboriz",
"desarc", "desarej", "desarest", "desargent", "desargentiz", "desarist",
"desarm", "desarmadilh", "desarmoniz", "desarom", "desaromaliz", "desaromatiz",
"desarquitet", "desarquiv", "desarraig", "desarranc", "desarranch", "desarranh",
"desarranj", "desarrazo", "desarreat", "desarrebit", "desarred", "desarredond",
"desarregaç", "desarreig", "desarremedi", "desarreneg", "desarrest", "desarreves",
"desarrig", "desarrim", "desarrinc", "desarrisc", "desarroch", "desarrolh",
"desarrot", "desarroup", "desarru", "desarruf", "desarrug", "desarrulh",
"desarrum", "desarterializ", "desarticul", "desartilh", "desarvor", "desas",
"desasn", "desaspir", "desassanh", "desassazo", "desassazon", "desassel",
"desasselvaj", "desassemelh", "desassent", "desassest", "desassimil", "desassimilh",
"desassinal", "desassis", "desassoci", "desassolv", "desassom", "desassombr",
"desassosseg", "desassunt", "desassust", "desastr", "desat", "desatabaf",
"desatac", "desatafulh", "desatapet", "desatapulh", "desatarrax", "desatasc",
"desataud", "desatavi", "desatedi", "desatemor", "desatemoriz", "desatent",
"desaterr", "desatest", "desatilh", "desatin", "desativ", "desatoch",
"desatol", "desatordo", "desatrac", "desatranc", "desatravanc", "desatravess",
"desatrel", "desatrem", "desatribul", "desatualiz", "desaug", "desautor",
"desautoriz", "desauxili", "desavacalh", "desavag", "desavantaj", "desaventur",
"desaverb", "desavergonh", "desavex", "desavez", "desavi", "desavigor",
"desavilt", "desavinc", "desavinh", "desavis", "desavist", "desaviv",
"desavolum", "desavult", "desaz", "desazinhavr", "desazot", "desbab",
"desbabeliz", "desbadal", "desbag", "desbago", "desbagulh", "desbalanç",
"desbalç", "desbalis", "desbaliz", "desbals", "desbamb", "desbanaliz",
"desbanc", "desband", "desbandalh", "desbandeir", "desbar", "desbarat",
"desbarb", "desbarbariz", "desbarr", "desbarranc", "desbarret", "desbarrig",
"desbast", "desbastard", "desbatiz", "desbatoc", "desbeiç", "desbenzoil",
"desbenzol", "desbestializ", "desbic", "desbitol", "desbobin", "desboc",
"desbolin", "desbolot", "desboqueir", "desborc", "desborcel", "desborcin",
"desbord", "desboro", "desborr", "desbot", "desboto", "desbrag",
"desbrasileir", "desbrav", "desbravej", "desbrec", "desbri", "desbrid",
"desbrinc", "desbritaniz", "desbro", "desbroch", "desbrot", "desbrum",
"desbrutaliz", "desbuch", "desbulh", "desbund", "desburocratiz", "desburric",
"desburrific", "desbutaniz", "descabaç", "descabel", "descaç", "descachaç",
"descachel", "descachol", "descadastr", "descadeir", "descafein", "descafeiniz",
"descafel", "descal", "descalabr", "descalavr", "descalç", "descalcet",
"descalcific", "descalcin", "descalho", "descalibr", "descaliç", "descalv",
"descam", "descamb", "descambimb", "descambulh", "descaminh", "descamis",
"descamp", "descan", "descancar", "descancel", "descanch", "descandaliz",
"descanel", "descang", "descangalh", "descangot", "descanhot", "descanjic",
"descans", "descant", "descap", "descapacit", "descapel", "descapitaliz",
"descapot", "descapsul", "descar", "descaracteriz", "descarapuç", "descarateriz",
"descarbon", "descarbonat", "descarbonet", "descarboniz", "descarboxil", "descarbur",
"descarburiz", "descarg", "descarn", "descarnaliz", "descarnardesenterr", "descarniz",
"descaroç", "descarol", "descarr", "descarreg", "descarreir", "descarret",
"descarri", "descarril", "descarrilh", "descart", "descarteliz", "descartorializ",
"descas", "descasal", "descasc", "descasp", "descasquej", "descasul",
"descatequiz", "descativ", "descatoliz", "descaud", "descautel", "descav",
"descavalg", "descaveir", "descaxel", "descefaliz", "desceg", "descelulariz",
"descement", "descentr", "descentraliz", "descep", "descerc", "descerebel",
"descerebr", "descerebriz", "descerr", "deschancel", "deschapel", "deschumb",
"descicliz", "descifr", "descimbr", "desciment", "desciminaliz", "descint",
"descintr", "desciviliz", "desclassific", "desclor", "descloret", "descloroformiz",
"descoagul", "descoalh", "descoc", "descoch", "descodific", "descofr",
"descogot", "descoif", "descoim", "descoit", "descoivar", "descol",
"descolg", "descolm", "descoloc", "descoloniz", "descolor", "descoloriz",
"descomand", "descombin", "descomission", "descompact", "descompadr", "descompagin",
"descompanh", "descomparticip", "descompartiment", "descompartimentaliz", "descompass", "descompatibiliz",
"descompens", "descompil", "descomplic", "descompromiss", "descomung", "desconceitu",
"desconcentr", "desconcert", "desconchav", "desconcheg", "desconcili", "desconcord",
"descondens", "descondicion", "desconect", "desconfeit", "desconfess", "desconfi",
"desconfigur", "desconform", "desconfort", "desconfront", "descongel", "descongestion",
"desconjug", "desconjunt", "desconjur", "desconsagr", "desconsert", "desconsider",
"desconsol", "desconstitucionaliz", "descont", "descontagi", "descontamin", "descontent",
"descontentoriz", "descontextualiz", "descontingenci", "descontinu", "descontrat", "descontrol",
"desconturb", "desconvers", "desconvid", "desconvoc", "descoorden", "descop",
"descor", "descorch", "descorço", "descord", "descorn", "descoro",
"descoroço", "descorporiz", "descorrelacion", "descorrent", "descortej", "descortic",
"descortiç", "descortin", "descoruch", "descost", "descostum", "descostur",
"descoto", "descout", "descrav", "descraveir", "descravej", "descraviz",
"descredenci", "descredibiliz", "descredit", "descrem", "descrimin", "descriminaliz",
"descriptograf", "descrist", "descristianiz", "descrom", "descruz", "descuid",
"desculatr", "desculp", "desculpabiliz", "descultiv", "descultu", "descultur",
"descunh", "descupiniz", "descur", "descurv", "descuscut", "desdeix",
"desdemoniz", "desdenh", "desdenomin", "desdent", "desdiagram", "desdiferenci",
"desdob", "desdobr", "desdoir", "desdour", "desdoutrin", "desdramatiz",
"deseclips", "deseconomiz", "desedific", "deseduc", "desefemin", "deseiv",
"deseix", "desej", "deseletriz", "deselimin", "deselitiz", "desemaç",
"desemadeir", "desemal", "desemalh", "desemalhet", "desemaranh", "desematilh",
"desembaç", "desembaci", "desembainh", "desembal", "desembals", "desembalsam",
"desembandeir", "desembaraç", "desembaralh", "desembarc", "desembarg", "desembarr",
"desembarranc", "desembarrig", "desembarril", "desembati", "desembaul", "desembebed",
"desembeiç", "desembest", "desembezerr", "desembirr", "desemblinh", "desemboc",
"desembol", "desembols", "desemborc", "desemborr", "desemborrach", "desemborrasc",
"desembosc", "desembot", "desembraç", "desembrai", "desembram", "desembrech",
"desembrenh", "desembriag", "desembrid", "desembrion", "desembrom", "desembrulh",
"desembrum", "desembrusc", "desembrux", "desembuç", "desembuch", "desemburilh",
"desemburr", "desemburric", "desemed", "desemend", "desemetiniz", "desemoç",
"desemoglobiniz", "desemoinh", "desemoldur", "desemp", "desempac", "desempacavir",
"desempach", "desempacot", "desempadralh", "desempal", "desempalh", "desempalm",
"desempan", "desempandeir", "desempanturr", "desempap", "desempapel", "desempar",
"desemparceir", "desempared", "desemparelh", "desempast", "desempastel", "desempat",
"desempaves", "desempeç", "desempecilh", "desempeçonh", "desempedr", "desempeflig",
"desempeg", "desempen", "desempenc", "desempenh", "desempern", "desemperr",
"desempertig", "desempest", "desempilh", "desempin", "desemplast", "desemplastr",
"desemplum", "desempo", "desempoç", "desempoeir", "desempol", "desempoleir",
"desempolg", "desempolh", "desemposs", "desempreg", "desemprenh", "desempro",
"desempunh", "desemulsific", "desemulsion", "desenamor", "desenastr", "desenat",
"desencab", "desencabeç", "desencabel", "desencabrest", "desencabrit", "desencabul",
"desencach", "desencachaç", "desencadern", "desencadilh", "desencafif", "desencafu",
"desencaipor", "desencaix", "desencaixilh", "desencaixot", "desencalacr", "desencalh",
"desencalm", "desencamarad", "desencamb", "desencaminh", "desencamis", "desencamp",
"desencan", "desencanalh", "desencanastr", "desencang", "desencangalh", "desencant",
"desencanto", "desencap", "desencapar", "desencapel", "desencapoeir", "desencapot",
"desencapsul", "desencaracol", "desencaralh", "desencarang", "desencarapel", "desencarapinh",
"desencarapuç", "desencarcer", "desencarg", "desencarn", "desencarquilh", "desencarr",
"desencarranc", "desencarrapit", "desencarreg", "desencarreir", "desencarret", "desencarril",
"desencarrilh", "desencart", "desencas", "desencasac", "desencasc", "desencasquet",
"desencastel", "desencasto", "desencatarr", "desencatarro", "desencav", "desencavac",
"desencavalg", "desencavern", "desencavilh", "desencep", "desencer", "desencerr",
"desencharc", "desencilh", "desenclaustr", "desenclavinh", "desencobr", "desencoc",
"desencofr", "desencoif", "desencoir", "desencoiraç", "desencoiv", "desencoivar",
"desencol", "desencoleriz", "desencomend", "desenconch", "desencontr", "desencor",
"desencoraj", "desencordo", "desencorp", "desencorpor", "desencortiç", "desencoscor",
"desencost", "desencostal", "desencour", "desencouraç", "desencout", "desencov",
"desencovil", "desencovilh", "desencrav", "desencravelh", "desencravilh", "desencrenc",
"desencresp", "desencrost", "desencru", "desencruz", "desencub", "desencuc",
"desenculatr", "desencurral", "desencurv", "desendemoninh", "desendeus", "desendireit",
"desendivid", "desenergiz", "desenerv", "desenevo", "desenfad", "desenfaix",
"desenfard", "desenfardel", "desenfarpel", "desenfarrusc", "desenfart", "desenfasti",
"desenfeit", "desenfeitiç", "desenfeix", "desenferm", "desenferruj", "desenfest",
"desenfeud", "desenfez", "desenfi", "desenfileir", "desenflech", "desenflor",
"desenfoc", "desenforc", "desenforj", "desenform", "desenforn", "desenfrasc",
"desenfrech", "desenfren", "desenfronh", "desenfueir", "desenfun", "desenfurn",
"desenfusc", "desengaç", "desengaiol", "desengaj", "desengalap", "desengalfinh",
"desengami", "desengan", "desenganch", "desengaranh", "desengarraf", "desengarranch",
"desengasg", "desengast", "desengastalh", "desengat", "desengatilh", "desengavet",
"desengelh", "desenglob", "desengod", "desengolf", "desengom", "desengonç",
"desengord", "desengordur", "desengorgit", "desengraç", "desengrad", "desengrainh",
"desengralh", "desengranz", "desengravat", "desengravit", "desengrax", "desengraz",
"desengren", "desengrenh", "desengril", "desengrimp", "desengrinald", "desengrip",
"desengross", "desengrum", "desengrun", "desenguaranch", "desenguiç", "desengulh",
"desengurgit", "desengurunh", "desenh", "desenjaul", "desenjo", "desenjorc",
"desenlaç", "desenlambuz", "desenlap", "desenlev", "desenliç", "desenlod",
"desenlodaç", "desenlut", "desenluv", "desenodo", "desenoj", "desenovel",
"desenquadr", "desenqueix", "desenrai", "desenraiv", "desenram", "desenrasc",
"desenred", "desenregel", "desenriç", "desenrij", "desenrist", "desenriz",
"desenrodilh", "desenrol", "desenrolh", "desenrosc", "desenroup", "desenrug",
"desensabo", "desensaburr", "desensac", "desensanguent", "desensarilh", "desensart",
"desenseb", "desensil", "desensin", "desensoc", "desensolv", "desensombr",
"desensop", "desensurr", "desentabu", "desentabul", "desentaip", "desental",
"desentalisc", "desentalo", "desentaramel", "desentarrax", "desentedi", "desenterr",
"desenterro", "desentes", "desentesoir", "desentesour", "desentibi", "desento",
"desentoc", "desenton", "desentorro", "desentort", "desentralh", "desentramel",
"desentranç", "desentranh", "desentrap", "desentrav", "desentrelaç", "desentrelinh",
"desentressolh", "desentrev", "desentrever", "desentrincheir", "desentroix", "desentronc",
"desentroniz", "desentropilh", "desentros", "desentroux", "desentub", "desentulh",
"desenturm", "desenturv", "desentusiasm", "desenubl", "desenvas", "desenvasilh",
"desenvencilh", "desenvenen", "desenvered", "desenverg", "desenvergonh", "desenverniz",
"desenvies", "desenvincilh", "desenviol", "desenvisc", "desenxarci", "desenxerg",
"desenxofr", "desenxov", "desenxovalh", "desequilibr", "desequip", "desequivoc",
"deserd", "deserotiz", "desert", "desertific", "desescal", "desescolariz",
"desescud", "desespant", "desespartilh", "desesper", "desesperanç", "desespinh",
"desespiraliz", "desespiritualiz", "desesquip", "desessenci", "desestabiliz", "desestagn",
"desestaliniz", "desestanh", "desestatiz", "desesteir", "desestim", "desestimul",
"desestiv", "desestorv", "desestr", "desestrelej", "desestrib", "desestrutur",
"desestud", "desetaniz", "desevangeliz", "desexcit", "desexcomung", "desfabric",
"desfabul", "desfaç", "desfadig", "desfai", "desfalc", "desfalcaç",
"desfalco", "desfanatiz", "desfantasi", "desfard", "desfarel", "desfarinh",
"desfas", "desfascistiz", "desfataliz", "desfatan", "desfavel", "desfaz",
"desfebr", "desfech", "desfeminiz", "desfenestr", "desferr", "desferrolh",
"desferruj", "desfertiliz", "desfi", "desfibr", "desfibril", "desfibrilh",
"desfibrin", "desfigur", "desfil", "desfileir", "desfilh", "desfili",
"desfilipiz", "desfit", "desfivel", "desflegm", "desflor", "desflorest",
"desfoc", "desfocaliz", "desfoleg", "desfolg", "desfolh", "desfoli",
"desfonologiz", "desforç", "desform", "desformat", "desforn", "desforr",
"desfortific", "desfosfor", "desfosforil", "desfrad", "desfragment", "desfrald",
"desfranj", "desfraterniz", "desfrech", "desfrequent", "desfris", "desfroux",
"desfrunch", "desfrut", "desfunch", "desfuncion", "desfund", "desfurfur",
"desgab", "desgadelh", "desgaf", "desgaiv", "desgalg", "desgalh",
"desgalvaniz", "desgargal", "desgarr", "desgarron", "desgas", "desgasconiz",
"desgaseific", "desgasific", "desgast", "desgavel", "desgel", "desgenitaliz",
"desgentiliz", "desgeometriz", "desgermaniz", "desgermin", "desglabr", "desglamouriz",
"desgliceriniz", "desglobuliz", "desglos", "desgoel", "desgordur", "desgorgomil",
"desgorj", "desgost", "desgovern", "desgovernamentaliz", "desgraç", "desgraci",
"desgradu", "desgram", "desgramaticaliz", "desgran", "desgrav", "desgravid",
"desgravit", "desgraviz", "desgrax", "desgreg", "desgrenh", "desgret",
"desgrilho", "desgrinald", "desgrud", "desgrum", "desguamp", "desguard",
"desguarit", "desguedelh", "desgui", "desidealiz", "desidentific", "desideologiz",
"desidrat", "desidrogen", "desidrogeniz", "design", "desigual", "desilumin",
"desilusion", "desilustr", "desimagin", "desiman", "desimaniz", "desimant",
"desimobiliz", "desimplant", "desimplic", "desimpossibilit", "desimpossibiliz", "desimpregn",
"desimprens", "desimpression", "desinç", "desincentiv", "desinch", "desinclin",
"desincompatibiliz", "desincorpor", "desincrimin", "desincrust", "desincub", "desindex",
"desindic", "desindici", "desindividualiz", "desindustrializ", "desinfam", "desinfeccion",
"desinfecion", "desinfect", "desinfern", "desinfest", "desinfet", "desinficion",
"desinfl", "desinflacion", "desinflam", "desinfluenci", "desinform", "desingles",
"desingurgit", "desinjuri", "desinquiet", "desinset", "desinsetiz", "desinstal",
"desinstitucionaliz", "desinsufl", "desintegr", "desinteir", "desinteiriç", "desinterdit",
"desinteress", "desinterlig", "desintern", "desinternacionaliz", "desinterpret", "desintestin",
"desintimid", "desintoxic", "desintric", "desintrinc", "desinvagin", "desinvern",
"desinviol", "desioniz", "desipnotiz", "desipotec", "desirman", "desisc",
"desislamiz", "desjarret", "desjeju", "desjunt", "deslabializ", "deslaç",
"deslacr", "desladrilh", "deslan", "deslanch", "deslap", "deslapid",
"deslarg", "deslarv", "deslass", "deslastr", "deslat", "deslaud",
"deslav", "deslavr", "desleald", "desleg", "deslegitim", "desleig",
"desleit", "desleix", "deslembr", "desli", "deslig", "deslignific",
"deslim", "deslind", "deslinhific", "deslint", "deslis", "deslivr",
"desliz", "desloc", "deslocaliz", "deslod", "deslogr", "deslomb",
"deslouc", "deslouv", "deslumbr", "deslustr", "desmacul", "desmadeir",
"desmagin", "desmagnetiz", "desmai", "desmain", "desmalencoliz", "desmalh",
"desmalhet", "desmalici", "desmam", "desmament", "desman", "desmanch",
"desmand", "desmang", "desmangalh", "desmanganes", "desmanilh", "desmaninh",
"desmaniv", "desmantel", "desmantilh", "desmaquil", "desmaquilh", "desmaranh",
"desmarc", "desmarchet", "desmarcializ", "desmargariniz", "desmargin", "desmarisc",
"desmascar", "desmassaj", "desmastr", "desmat", "desmaterializ", "desmatricul",
"desmazel", "desmazorr", "desmedr", "desmedul", "desmelancoliz", "desmelen",
"desmelhor", "desmelindr", "desmembr", "desmemori", "desmensur", "desment",
"desmergulh", "desmescl", "desmesmeriz", "desmesur", "desmetaliz", "desmetodiz",
"desmielin", "desmieliniz", "desmilitariz", "desmin", "desmineraliz", "desmiol",
"desmistific", "desmitific", "desmiud", "desmobil", "desmobilh", "desmobiliz",
"desmoç", "desmoch", "desmoder", "desmodul", "desmoed", "desmoir",
"desmoit", "desmold", "desmond", "desmonetiz", "desmonopoliz", "desmonotoniz",
"desmont", "desmonto", "desmoraliz", "desmorfiniz", "desmoron", "desmortalh",
"desmortific", "desmotiv", "desmour", "desmout", "desmud", "desmultiplic",
"desmunhec", "desmunici", "desmunicion", "desmur", "desmurch", "desmuscul",
"desmusg", "desnacionaliz", "desnalg", "desnarig", "desnasal", "desnasaliz",
"desnastr", "desnat", "desnatur", "desnaturaliz", "desnazific", "desnecessit",
"desneg", "desnegoci", "desnerv", "desnev", "desninh", "desniquel",
"desnivel", "desnoc", "desnodo", "desnoit", "desnoiv", "desnot",
"desnovel", "desnu", "desnubl", "desnuc", "desnucleariz", "desnud",
"desobrig", "desobumbr", "desocult", "desocup", "desodor", "desodoriz",
"desoficializ", "desofusc", "desol", "desoleific", "desolh", "desoner",
"desonest", "desonr", "desopil", "desopress", "desorbit", "desorçament",
"desorden", "desorelh", "desorganiz", "desorient", "desorn", "desoss",
"desoug", "desov", "desoxid", "desoxigen", "despach", "despacient",
"despadr", "despaganiz", "despagin", "despais", "despalataliz", "despalatiz",
"despalet", "despalh", "despalhet", "despalm", "despalmilh", "despamp",
"despampan", "despancreatiz", "despap", "desparafin", "desparafus", "desparament",
"desparasit", "desparceir", "desparcel", "desparelh", "desparr", "desparram",
"despartidariz", "desparv", "despass", "despast", "despatri", "despatrocin",
"despaulific", "despavil", "despaviment", "despeç", "despecuni", "despedaç",
"despedr", "despeg", "despeit", "despeitor", "despej", "despel",
"despelh", "despen", "despenaliz", "despenc", "despendur", "despenh",
"despenitenci", "despentaniz", "desperdiç", "desperfil", "despersonaliz", "despersonific",
"despert", "despes", "despesc", "despestan", "despetal", "despetrech",
"despic", "despied", "despigment", "despilch", "despinç", "desping",
"despinhaliz", "despinic", "despint", "despiolh", "despiroc", "despist",
"desplant", "desplum", "despoeir", "despoetiz", "despoj", "despolariz",
"despolet", "despoli", "despolimeriz", "despoliniz", "despolitiz", "despoloniz",
"despolp", "despolvilh", "desponder", "despong", "despont", "despontu",
"despopul", "despopulariz", "desportilh", "desportugues", "despos", "desposs",
"despostiç", "despostig", "despotiz", "despovo", "despraguej", "desprecat",
"despreci", "despreg", "despreguiç", "despremi", "desprenh", "despreocup",
"desprepar", "despresilh", "despressuriz", "desprestigi", "desprez", "desprimor",
"desprior", "despriv", "desprivatiz", "desprivilegi", "desprocess", "desprofan",
"desprogram", "desproletariz", "despronunci", "despropaniz", "desproporcion", "desproposit",
"despropri", "desproteiniz", "desprotoniz", "desprum", "desprussianiz", "despucel",
"despudor", "despueriliz", "despum", "despurific", "desquadr", "desquadrilh",
"desquadrinh", "desqualific", "desquam", "desquart", "desquartin", "desquebr",
"desqueix", "desqueixol", "desqui", "desquici", "desquiet", "desquit",
"desrab", "desraig", "desram", "desrat", "desratiz", "desrecalc",
"desrefolh", "desregionaliz", "desregr", "desregulament", "desrelv", "desremedi",
"desrend", "desrepublicaniz", "desresin", "desrespeit", "desresponsabiliz", "desretrat",
"desriç", "desrisc", "desrisonh", "desriz", "desroch", "desrolh",
"desromaniz", "desromantiz", "desrosc", "desroup", "desrug", "desrum",
"desrussific", "dessabor", "dessaburr", "dessacr", "dessacraliz",
"dessagr", "dessaibr", "dessain", "dessalg", "dessalific", "dessalinific",
"dessaliniz", "dessalitr", "dessamouc", "dessangr", "dessarr", "dessarro",
"dessazon", "dessazonaliz", "dessec", "dessedent", "dessegment", "dessegred",
"dessegreg", "dessegur", "dessein", "desseiv", "dessel", "dessemelh",
"dessensibiliz", "dessepult", "dessesm", "dessexu", "dessexualiz", "dessilic",
"dessilicat", "dessilicific", "dessimpatiz", "dessinal", "dessincroniz", "dessintoniz",
"dessiti", "desso", "dessoalh", "dessobraç", "dessoc", "dessoçobr",
"dessol", "dessold", "dessolh", "dessolidariz", "dessolvat", "dessor",
"dessosseg", "dessoterr", "dessu", "dessubjug", "dessubstanci", "dessud",
"dessuj", "dessulfon", "dessulfur", "dessulfuriz", "dessurr", "destabiliz",
"destaboc", "destac", "destal", "destalh", "destaling", "destamp",
"destanch", "destaniniz", "destaniz", "destap", "destar", "destaramel",
"destarrach", "destel", "destelh", "destemper", "desterneir", "desterr",
"desterraplen", "desterritorializ", "desterro", "destet", "destil", "destin",
"destipific", "destiraniz", "destisn", "desto", "destoc", "destoitiç",
"destold", "destomb", "deston", "destor", "destornilh", "destorro",
"destorv", "destouc", "destoutiç", "destoutuç", "destoxific", "destraç",
"destrag", "destraj", "destram", "destrambelh", "destramel", "destranc",
"destranç", "destrat", "destrav", "destravanc", "destravess", "destrein",
"destrel", "destrep", "destrib", "destribaliz", "destriç", "destrig",
"destrinc", "destrinç", "destrinch", "destrip", "destripul", "destroc",
"destroç", "destron", "destronc", "destroniz", "destropilh", "destrunf",
"desturv", "destutel", "desug", "desultraj", "desuman", "desumaniz",
"desumect", "desumidific", "desunh", "desunific", "desuniform", "desurbaniz",
"desus", "desvac", "desvair", "desvali", "desvalid", "desvalij",
"desvalis", "desvalor", "desvaloriz", "desvalvul", "desvar", "desvari",
"desvasculariz", "desvassal", "desvel", "desvelej", "desvencelh", "desvencilh",
"desvend", "desvener", "desventr", "desventur", "desvergonh", "desvermin",
"desvertebr", "desverticaliz", "desvez", "desvi", "desvidr", "desvidraç",
"desvig", "desvigi", "desvigor", "desvigoriz", "desvinc", "desvincilh",
"desvincul", "desvir", "desvirg", "desvirgin", "desvirginiz", "desvirgul",
"desviriliz", "desvirtu", "desviscer", "desvisg", "desvitaliz", "desvitamin",
"desvitaminiz", "desvitrific", "desvizinh", "desvocaliz", "desvolum", "desvoluntari",
"desvulcaniz", "desvulgariz", "desxadrez", "desxaret", "deszel", "deszinc",
"deszip", "detalh", "detard", "detect", "detenç", "deterior",
"determin", "detest", "detet", "deton", "detor", "detram",
"detrat", "detrect", "detrinç", "detron", "detrunc", "deturb",
"deturp", "devass", "devast", "devel", "deventr", "devis",
"devitrific", "devor", "devot", "devulcaniz", "devulg", "dezinc",
"dezincific", "diaboliz", "diaclas", "diadem", "diafaniz", "diafragm",
"diagnostic", "diagonaliz", "diagram", "dialect", "dialet", "dialis",
"dialog", "diamantiz", "diariz", "diatermocoagul", "diazot", "diazotiz",
"dic", "dicionari", "dicionariz", "dicotomiz", "dies", "dieseliz",
"difam", "diferenç", "diferenci", "dificult", "difrat", "digit",
"digitaliniz", "digitaliz", "digladi", "dign", "dignific", "digress",
"digression", "dilacer", "dilani", "dilapid", "dilarg", "dilat",
"diligenci", "dilucid", "diluvi", "diman", "dimension", "dimeriz",
"dimidi", "dinamit", "dinamitiz", "dinamiz", "dind", "dinumer",
"diploidiz", "diplom", "diplomaci", "diplomatiz", "direcion", "dirradic",
"disc", "discant", "disciplin", "discord", "discrep", "discretiz",
"discrimin", "discurs", "discus", "discutinh", "diseliz", "disfarç",
"disform", "disgram", "disgreg", "disjunt", "dispar", "disparat",
"dispens", "dispers", "disponibiliz", "disput", "dissec", "dissemelh",
"dissemin", "dissert", "dissidi", "dissimetriz", "dissimil", "dissimul",
"dissip", "dissoci", "disson", "dist", "distanci", "distension",
"distil", "distrat", "distrofic", "disturb", "dit", "ditong",
"diuturniz", "divag", "divaric", "diversific", "diviniz", "divis",
"divorci", "divulg", "dizim", "do", "dob", "dobr",
"doç", "dociliz", "document", "doest", "dogmatiz", "doidej",
"doidivan", "doir", "doirej", "dolariz", "dolomitiz", "dolomiz",
"doloriz", "dom", "domestic", "domicili", "domific", "domin",
"doming", "dop", "dorment", "dorminh", "dorminhoc", "dormit",
"dorsaliz", "dos", "dosific", "dossel", "dot", "dotaliz",
"doudej", "doudivan", "dour", "dourej", "douti", "doutor",
"doutrin", "drag", "drain", "dramatiz", "drapej", "dren",
"dribl", "drog", "dualiz", "dubl", "duch", "ductiliz",
"duel", "dueliz", "duet", "dulcific", "dulçor", "dupl",
"duplic", "dur", "durb", "duvid", "ealm", "earc",
"ebaniz", "eburn", "eburnific", "ec", "echacorv", "eclips",
"eco", "ecocardiograf", "ecoencefalograf", "ecograf", "ecoloc", "ecolocaliz",
"economiz", "ecossistematiz", "ectomorfiz", "ecumeniz", "edafostratigraf", "edafostratograf",
"edemaci", "edematiz", "edeniz", "edent", "edicion", "edicul",
"edific", "edit", "editor", "editori", "educ", "educomunic",
"edulç", "edulcor", "edulçor", "eequacion", "efabul", "efeitu",
"efemeriz", "efemin", "efeminiz", "efervent", "efetiv", "efetu",
"eficaci", "efigi", "egipcianiz", "egocentr", "egocentraliz", "egoidealiz",
"egoism", "eiç", "eim", "eiv", "eix", "ejacul",
"eject", "ejet", "el", "elabor", "elanç", "elastific",
"elastiz", "eleboriz", "electrific", "electriz", "electrocortigraf", "electrocut",
"electrodeposit", "electrodiagnostic", "electrolis", "electromedic", "electron", "electrotip",
"elegantiz", "elenc", "eletrific", "eletriz", "eletrocortigraf", "eletrocut",
"eletrodeposit", "eletrolis", "eletromedic", "eletron", "eletroniz", "eletrotip",
"elev", "elici", "elicit", "elimin", "elips", "elitiz",
"elogi", "elong", "elucid", "elucubr", "elutri", "emaç",
"emaçaroc", "emaci", "emacul", "emadeir", "emadeix", "emagazin",
"emagot", "emagr", "emagrent", "emal", "emalh", "emalhet",
"eman", "emanch", "emancip", "emanding", "emangueir", "emanhuç",
"emanilh", "emanjeric", "emanoc", "emant", "emantilh", "emar",
"emaranh", "emarj", "emarlot", "emartilh", "emascar", "emascul",
"emass", "emassilh", "emastr", "ematilh", "embabac", "embabad",
"embaç", "embacel", "embaci", "embag", "embagaç", "embainh",
"embal", "embalanç", "embalç", "embaloiç", "embalouç", "embals",
"embalsam", "emban", "embanan", "emband", "embandeir", "embaraç",
"embarafust", "embaralh", "embarb", "embarbasc", "embarbel", "embarbilh",
"embarc", "embard", "embarg", "embarr", "embarrac", "embarranc",
"embarreir", "embarrel", "embarret", "embarric", "embarrig", "embarril",
"embarulh", "embas", "embasbac", "embast", "embasti", "embastilh",
"embatin", "embatoc", "embatuc", "embatum", "embauc", "embaul",
"embebec", "embebed", "embeber", "embebor", "embeiç", "embelec",
"embelen", "embelez", "embelg", "embelinh", "embeloir", "embelour",
"embelout", "embern", "emberrinch", "embesilh", "embesoir", "embesour",
"embespinh", "embest", "embetesg", "embetum", "embezerr", "embiboc",
"embic", "embicheir", "embigot", "embilh", "embiloc", "embing",
"embioc", "embiot", "embir", "embirich", "embiroc", "embirr",
"embirrinch", "embirut", "embisc", "embitesg", "embizug", "emblem",
"emblematiz", "embob", "embobin", "emboc", "emboç", "emboçal",
"embocet", "embochech", "embodalh", "embodeg", "embodoc", "emboit",
"emboj", "embol", "embolat", "emboldreg", "emboldri", "embolet",
"embolig", "embolinh", "emboliz", "emboloir", "embolor", "embolot",
"embolour", "embols", "embon", "embonec", "embonecr", "embonic",
"emboquilh", "emborbet", "emborc", "embornal", "emborquilh", "emborr",
"emborrach", "emborralh", "emborrasc", "embosc", "embosn", "embost",
"embostel", "embot", "embotelh", "embotij", "embotilh", "emboug",
"embout", "embrab", "embraç", "embraceir", "embrag", "embrai",
"embram", "embranc", "embrech", "embrenh", "embret", "embriag",
"embric", "embrid", "embrigad", "embrinc", "embrion", "embroc",
"embroch", "embrom", "embruac", "embrulh", "embrum", "embrusc",
"embrut", "embrux", "embuç", "embuçal", "embuch", "embud",
"emburac", "emburel", "emburgues", "emburilh", "emburr", "emburric",
"emburrisc", "emburulh", "embuzi", "embuzin", "emech", "emed",
"emedoiç", "emedoich", "emedouç", "emedouch", "emel", "emend",
"emendic", "emenin", "ement", "emeroiç", "emerouç", "emeruj",
"emetiz", "emezi", "emigr", "eminenci", "eminhoc", "emocion",
"emoir", "emoit", "emold", "emoldur", "emoleir", "emolh",
"emon", "emordaç", "emoroiç", "emorouç", "emorrinh", "emosc",
"emosqueir", "emost", "emour", "emp", "empac", "empacavir",
"empach", "empaçoc", "empacot", "empades", "empadilh", "empadro",
"empaiol", "empal", "empalam", "empalet", "empalh", "empalhasc",
"empalheir", "empali", "empalit", "empalm", "empalud", "empampan",
"empan", "empanad", "empanc", "empandeir", "empandilh", "empandin",
"empando", "empaneir", "empanem", "empang", "empangin", "empanh",
"empantan", "empantuf", "empanturr", "empanzin", "empap", "empapel",
"empapoil", "empapoul", "empapuç", "empaquet", "empar", "emparament",
"emparc", "emparceir", "emparcel", "empard", "empared", "emparelh",
"emparent", "emparr", "emparreir", "emparv", "emparvat", "emparvo",
"empasm", "empast", "empastel", "empat", "empaturr", "empaupic",
"empaves", "empavo", "empavon", "empeç", "empecad", "empecilh",
"empeçonh", "empeçonhent", "empedern", "empedr", "empeg", "empeir",
"empeireir", "empeitic", "empej", "empel", "empelam", "empelic",
"empelicul", "empelot", "empen", "empenach", "empenc", "empendo",
"empenetr", "empenh", "empenhor", "empenuj", "empeol", "empepin",
"empequenit", "empereir", "emperiquit", "emperl", "empern", "empernic",
"emperol", "emperr", "empertig", "empes", "empesg", "empesso",
"empest", "empetec", "empetelec", "empez", "empezinh", "empic",
"empicot", "empilch", "empilec", "empilh", "empin", "empinhoc",
"empinoc", "empiol", "empiolh", "empior", "empip", "empipoc",
"empisc", "empistol", "empiteir", "emplac", "emplanch", "emplasm",
"emplast", "emplastr", "emplum", "empo", "empobr", "empoç",
"empocilg", "empoder", "empoeir", "empoit", "empol", "empolam",
"empolasm", "empoleir", "empolg", "empolh", "empolm", "empolv",
"empolvilh", "empomad", "empomb", "emponch", "emponder", "empont",
"empop", "emporc", "emporcalh", "emposs", "empost", "empostem",
"empostig", "emprateleir", "empraz", "empreg", "empreguiç", "empreit",
"emprenh", "empres", "empresari", "emprest", "emprestad", "emprision",
"empro", "emprum", "empulg", "empulh", "empunh", "empup",
"empurpur", "empurr", "empux", "emul", "emulsific", "emulsion",
"emund", "emuralh", "emurch", "enagen", "enaif", "enaip",
"enalt", "enamor", "enarr", "enastr", "enat", "enateir",
"encab", "encabeç", "encabeir", "encabel", "encabeliz", "encabreim",
"encabrest", "encabrit", "encabrun", "encabul", "encabulh", "encaçap",
"encach", "encachaç", "encachapuç", "encachi", "encachoeir", "encachol",
"encadarro", "encadeir", "encadern", "encafif", "encafu", "encafurn",
"encagaç", "encaibr", "encaieir", "encaipor", "encairel", "encaix",
"encaixilh", "encaixot", "encal", "encalacr", "encalamech", "encalamistr",
"encalamouc", "encalburd", "encalc", "encalç", "encald", "encaldeir",
"encaleir", "encalg", "encalh", "encalic", "encaliç", "encalist",
"encalistr", "encalm", "encalmurr", "encalomb", "encalor", "encalung",
"encam", "encamaç", "encamarad", "encamarinh", "encamarot", "encamb",
"encambeir", "encambich", "encambit", "encambo", "encambon", "encambuc",
"encambulh", "encamel", "encamer", "encaminh", "encamis", "encamoroiç",
"encamorouç", "encamp", "ençamp", "encampan", "encampesin", "encamurç",
"encan", "encanalh", "encanastr", "encancel", "encancer", "encandil",
"encandol", "encanel", "encanfinf", "encang", "encangalh", "encangot",
"encanh", "encanic", "encaniç", "encanit", "encaniz", "encano",
"encant", "encanteir", "encantin", "encantinh", "encanto", "encanton",
"encanut", "encanzin", "encanzo", "encap", "encapach", "encapar",
"encapel", "encapet", "encapoeir", "encapot", "encaprich", "encapsul",
"encapuch", "encapuz", "encar", "encaracol", "encaramanch", "encaramel",
"encaramon", "encaramuj", "encarang", "encaranguej", "encarant", "encarantonh",
"encarapel", "encarapinh", "encarapit", "encarapuç", "encaravelh", "encarboniz",
"encarcad", "encarcer", "encarch", "encardum", "encarent", "encaret",
"encarg", "encarij", "encarioc", "encarn", "encarneir", "encarniç",
"encaroç", "encaroch", "encarol", "encarquilh", "encarr", "encarramon",
"encarranc", "encarrapiç", "encarrapit", "encarrasc", "encarraspan", "encarreg",
"encarreir", "encarret", "encarril", "encarrilh", "encarrisc", "encart",
"encartalh", "encartaz", "encarteir", "encartol", "encartuch", "encarvo",
"encarvoej", "encarvoiç", "encas", "encasac", "encasal", "encasc",
"encascalh", "encascor", "encasmurr", "encasquet", "encasquilh", "encastalh",
"encastel", "encasto", "encastr", "encasul", "encataplasm", "encatarr",
"encatarro", "encatrafi", "encatramon", "encatravilh", "encatrin", "encauch",
"encaud", "encaustic", "encav", "encavac", "encaval", "encavaleir",
"encavalg", "encavalit", "encavern", "encavilh", "encazuc", "encebol",
"encedr", "encegueir", "enceir", "encel", "enceleir", "encen",
"encendr", "encensori", "encentr", "encep", "encer", "encerebr",
"enceroil", "enceroul", "encerr", "encert", "encervej", "encest",
"encet", "enceved", "enchacot", "enchafurd", "enchamarr", "enchambo",
"enchambr", "enchamej", "enchampanh", "enchapel", "enchapot", "enchapour",
"enchapuç", "encharc", "encharol", "enchavet", "enchiçar", "enchicharr",
"enchimarr", "enchiment", "enchiqueir", "enchoç", "enchocalh", "enchoiriç",
"enchombr", "enchouriç", "enchous", "enchumaç", "enchumarr", "enchumb",
"enchumbeir", "enchusm", "encieir", "encilh", "encim", "encimalh",
"encinch", "encinh", "encint", "encinz", "encinzeir", "encirr",
"encisc", "encist", "encium", "enclaustr", "enclausur", "enclavinh",
"encloac", "encobard", "encobert", "encobilh", "encobr", "encoch",
"encocurut", "encod", "encodo", "encof", "encofr", "encoif",
"encoim", "encoir", "encoiraç", "encoit", "encoivar", "encol",
"encoleir", "encoleriz", "encoloc", "encomend", "encomi", "encomiss",
"encomoroç", "encomoroiç", "encomorouç", "encomprid", "encomunh", "encon",
"enconap", "enconc", "enconch", "encond", "encondiz", "enconformiz",
"enconhac", "enconic", "enconteir", "enconto", "encontr", "encontro",
"encop", "encoquin", "encoquinh", "encor", "encoraj", "encorc",
"encorcund", "encordel", "encordo", "encorn", "encornet", "encornich",
"encoronh", "encorp", "encorpor", "encorquilh", "encorrelh", "encorrent",
"encorri", "encorric", "encorrilh", "encorrovilh", "encortelh", "encortiç",
"encortin", "encoruj", "encoscor", "encospi", "encost", "encostal",
"encostel", "encostr", "encotinh", "encouch", "encour", "encouraç",
"encout", "encov", "encovard", "encovil", "encovilh", "encovit",
"encramoioç", "encramouç", "encrass", "encrav", "encraveir", "encravelh",
"encravilh", "encrenc", "encresp", "encript", "encris", "encrist",
"encristin", "encroc", "encrost", "encru", "encruent", "encrust",
"encruz", "encruzilh", "encub", "encuc", "encucharr", "encueir",
"enculp", "encultur", "encunh", "encurral", "encurrelh", "encurric",
"encurrich", "encurrilh", "encurt", "encurv", "encutinh", "endech",
"endeflux", "endej", "endemiz", "endemoninh", "endent", "endereç",
"endeus", "endiabr", "endilg", "endinh", "endinheir", "endireit",
"endivid", "endocardiograf", "endocit", "endocruz", "endoid", "endoming",
"endors", "endoss", "endoud", "endoutrin", "endrag", "endromin",
"endur", "endurent", "enebri", "energiz", "enerv", "enesg",
"enev", "enevo", "enfaceir", "enfachoc", "enfad", "enfaix",
"enfang", "enfanic", "enfar", "enfard", "enfardel", "enfardo",
"enfardon", "enfarel", "enfarfalh", "enfarinh", "enfarn", "enfaro",
"enfarp", "enfarpel", "enfarrap", "enfarrusc", "enfart", "enfas",
"enfascal", "enfasti", "enfatiot", "enfatiz", "enfatu", "enfebr",
"enfechel", "enfedilh", "enfei", "enfeir", "enfeit", "enfeitiç",
"enfeix", "enfelp", "enfeltr", "enfeluj", "enfen", "enferm",
"enferr", "enferret", "enferrolh", "enferruj", "enfest", "enfesto",
"enfeston", "enfeud", "enfez", "enfezin", "enfi", "enfieir",
"enfileir", "enfinc", "enfirm", "enfistul", "enfit", "enfiteutic",
"enfivel", "enfix", "enflait", "enflanel", "enfleim", "enflor",
"enfobi", "enfoc", "enfog", "enfoiç", "enfol", "enfolech",
"enfolh", "enfolip", "enforc", "enforj", "enform", "enformos",
"enforn", "enforquilh", "enforr", "enfort", "enfouç", "enfrac",
"enfrag", "enfraldic", "enfranj", "enfraquent", "enfrasc", "enfrech",
"enfren", "enfrenesi", "enfrenis", "enfrent", "enfrest", "enfri",
"enfrond", "enfronh", "enfront", "enfrut", "enfu", "enfueir",
"enfuleim", "enfulij", "enfum", "enfumaç", "enfumar", "enfun",
"enfuner", "enfunil", "enfunisc", "enfurd", "enfuri", "enfurn",
"enfus", "enfusc", "enfust", "enfuzil", "eng", "engabel",
"engabrich", "engaç", "engadanh", "engadelh", "engai", "engaifon",
"engaiol", "engaj", "engajat", "engal", "engalan", "engalap",
"engaldrap", "engaldrip", "engalfinh", "engalg", "engalh", "engalhardet",
"engalhop", "engalhot", "engali", "engalic", "engalicism", "engalinh",
"engalisp", "engalop", "engalriç", "engambel", "engambit", "engamel",
"engan", "enganch", "engangorr", "engangren", "engaramp", "engaranh",
"engarant", "engarap", "engaravit", "engarbon", "engarf", "engargant",
"engarguel", "engarnach", "engarr", "engarraf", "engarrafon", "engarup",
"engasg", "engasgalh", "engasgat", "engast", "engastalh", "engasto",
"engat", "engatalh", "engatanh", "engatiç", "engatilh", "engatinh",
"engavel", "engavet", "engavinh", "engazofil", "engazop", "engazup",
"engelh", "engemin", "engendr", "engenh", "engeroc", "engess",
"engig", "englob", "englobul", "engob", "engod", "engodilh",
"engolf", "engolip", "engom", "engonç", "engonh", "engoni",
"engord", "engordur", "engorgit", "engorr", "engraç", "engrad",
"engrambel", "engramp", "engrampon", "engranz", "engravat", "engravid",
"engravit", "engrax", "engraz", "engrazul", "engrel", "engren",
"engrenh", "engrif", "engril", "engrim", "engrimp", "engrimpin",
"engrimpon", "engrinald", "engrip", "engriquilh", "engrol", "engross",
"engrum", "engrunh", "enguedelh", "enguiç", "enguirland", "enguizalh",
"engulh", "engulip", "engulosin", "engunh", "enigm", "enigmatiz",
"enjaez", "enjalg", "enjambr", "enjang", "enjangr", "enjaul",
"enjeir", "enjeit", "enjeriz", "enjic", "enjo", "enjoi",
"enjoin", "enjon", "enjorc", "enjorg", "enjouj", "enjuan",
"enjug", "enjunç", "enlabared", "enlabi", "enlabirint", "enlabrusc",
"enlabuz", "enlaç", "enlaçarot", "enlacr", "enladeir", "enlag",
"enlaiv", "enlam", "enlambuj", "enlambuz", "enlamin", "enlanz",
"enlap", "enlaruç", "enlat", "enleiv", "enlerd", "enlev",
"enliç", "enloc", "enlod", "enlodaç", "enloiç", "enloir",
"enlois", "enloj", "enlomb", "enlouc", "enlouç", "enlour",
"enlous", "enluar", "enludr", "enlur", "enlut", "enluv",
"enod", "enodo", "enog", "enoiriç", "enoit", "enoj",
"enoliz", "enosilh", "enouriç", "enout", "enov", "enovel",
"enquadr", "enquadrilh", "enquart", "enqueij", "enqueix", "enquezil",
"enquilh", "enquist", "enquit", "enquizil", "enrab", "enrabeir",
"enrabich", "enrabit", "enraç", "enrai", "enraiv", "enram",
"enramalh", "enramalhet", "enramel", "enramilhet", "enranç", "enranch",
"enrapos", "enrasc", "enrastilh", "enred", "enredoiç", "enredom",
"enredouç", "enreg", "enregel", "enregist", "enregistr", "enregueir",
"enreix", "enrelh", "enrelheir", "enrelv", "enremel", "enremin",
"enremiss", "enrend", "enrepolh", "enresin", "enresm", "enrest",
"enresti", "enreves", "enri", "enric", "enriç", "enricon",
"enrif", "enrij", "enrilh", "enrilheir", "enrim", "enrincon",
"enrip", "enriquent", "enrist", "enristr", "enrix", "enriz",
"enroc", "enroch", "enrod", "enrodel", "enrodilh", "enrodrig",
"enrol", "enrolh", "enrosc", "enroup", "enrox", "enruç",
"enruf", "enrug", "enrumin", "ensabo", "ensaburr", "ensac",
"ensacol", "ensai", "ensaibr", "ensainh", "ensalç", "ensaliv",
"ensalm", "ensalmoir", "ensalmour", "ensalsich", "ensamarr", "ensambenit",
"ensambl", "ensamp", "ensanch", "ensandal", "ensandol", "ensanduich",
"ensanef", "ensangu", "ensanguent", "ensanguinh", "ensanh", "ensanzor",
"ensapez", "ensarapulh", "ensardinh", "ensarilh", "ensarjet", "ensarraf",
"ensarranh", "ensart", "ensartilh", "enseb", "ensec", "ensed",
"ensei", "enseir", "enseiv", "ensej", "enselv", "ensement",
"enserralh", "ensesg", "ensev", "ensil", "ensilv", "ensilveir",
"ensimesm", "ensin", "enso", "ensoalh", "ensoalheir", "ensoberb",
"ensobrad", "ensoc", "ensod", "ensofreg", "ensog", "ensoiss",
"ensoleir", "ensolv", "ensombr", "enson", "ensop", "ensopi",
"ensorn", "ensoss", "ensovac", "ensuj", "ensujent", "ensumagr",
"ensurr", "ensurraip", "ensurro", "entabic", "entaboc", "entabu",
"entabul", "entaburr", "entach", "entafulh", "entaip", "ental",
"entaleig", "entaleir", "entalh", "entalic", "entaling", "entalisc",
"entalo", "entam", "entanc", "entang", "entaniç", "entap",
"entapet", "entapiç", "entapig", "entapiz", "entaramel", "entarrax",
"entartalh", "entavol", "entec", "entedi", "enteix", "entej",
"entejol", "entejuc", "entel", "entelh", "entelheir", "entendi",
"entenebr", "entenx", "enterr", "enterreir", "enterro", "enterturb",
"entes", "enteso", "entesoir", "entesoiris", "entesour", "entesouriz",
"entest", "entesteferr", "entibi", "entic", "entijol", "entijuc",
"entint", "entirit", "entisic", "entisn", "entiv", "ento",
"entoalh", "entoc", "entocai", "entog", "entoiç", "entoiceir",
"entoir", "entoj", "entol", "entold", "entolh", "enton",
"entont", "entorn", "entorp", "entorro", "entort", "entouc",
"entouç", "entouceir", "entoupeir", "entour", "entourinh", "entr",
"entraj", "entralh", "entram", "entrambelic", "entrambic", "entranç",
"entranh", "entranqueir", "entrap", "entrast", "entravess",
"entravisc", "entreadivinh", "entreajud", "entream", "entrearremess", "entrebeij",
"entrecerr", "entrech", "entrechoc", "entrecort", "entrecruz", "entredevor",
"entredilacer", "entredisput", "entrefal", "entrefech", "entrefestej", "entrefi",
"entrefigur", "entrefolh", "entreg", "entrelaç", "entrelembr", "entrelh",
"entrelig", "entrelinh", "entreloiç", "entrelouç", "entremescl", "entremistur",
"entremostr", "entremurmur", "entrenubl", "entreocult", "entreolh", "entrepar",
"entrepass", "entrepelej", "entrepens", "entrepern", "entrepic", "entrepigait",
"entrepil", "entreplic", "entrerreclam", "entrerreg", "entrescut", "entresilh",
"entressac", "entressach", "entressolh", "entressonh", "entressufoc", "entretalh",
"entretel", "entretoc", "entretrucid", "entreturb", "entrev", "entrevaz",
"entrevel", "entrever", "entrevist", "entrez", "entrezilh", "entrilh",
"entrinç", "entrincheir", "entriz", "entroc", "entroç", "entroix",
"entromb", "entron", "entronc", "entronch", "entroniz", "entropeç",
"entropic", "entropigait", "entropilh", "entros", "entroux", "entrovisc",
"entrud", "entruj", "entrunf", "entrunfinh", "entrupic", "entrupigait",
"entub", "entuch", "entuf", "entujuc", "entulh", "entupigait",
"enturb", "enturm", "enturv", "enturvisc", "entusiasm", "enubl",
"enumer", "enunci", "enuvi", "envagin", "envaid", "enval",
"envalenton", "envar", "envaret", "envarot", "envas", "envasilh",
"envaz", "envelhac", "envelhent", "envelop", "envencelh", "envencilh",
"envenen", "enventan", "enver", "enverdej", "envered", "enverg",
"envergonh", "envermelh", "enverniz", "enverrug", "enves", "envesg",
"envess", "envi", "enviç", "envid", "envidilh", "envidraç",
"envieir", "envies", "envig", "envigor", "envigot", "envinagr",
"envincilh", "envinh", "envinhadalh", "enviper", "envisc", "envisg",
"enviveir", "envol", "envolucr", "envolut", "envult", "enxabelh",
"enxad", "enxadrez", "enxalavi", "enxalç", "enxalm", "enxambl",
"enxambr", "enxamelh", "enxampo", "enxampu", "enxaquec", "enxaquet",
"enxarci", "enxarop", "enxaug", "enxec", "enxelh", "enxequet",
"enxerc", "enxerg", "enxert", "enxi", "enxilh", "enxofr",
"enxog", "enxogalh", "enxombr", "enxor", "enxot", "enxov",
"enxovalh", "enxudr", "enxug", "enxumbr", "enxundi", "enxurd",
"enxurr", "enxurri", "enzamp", "enzamparin", "enzampon", "enzin",
"enzon", "enzonz", "eoliz", "epicuriz", "epidemi", "epigeniz",
"epigraf", "epigram", "epigramatic", "epigramatiz", "epil", "epilepsi",
"epileptiz", "epilog", "episodi", "epist", "epistoliz", "epitet",
"epitom", "equacion", "equaliz", "equidimension", "equidist", "equifinaliz",
"equilibr", "equimos", "equip", "equipar", "equiponder", "equivoc",
"ergastul", "ergotiz", "eriç", "erisipel", "erm",
"eros", "erotiz", "err", "erradic", "erriç", "eruct",
"erudit", "erv", "ervilh", "ervo", "esamp", "esbabac",
"esbagaç", "esbagach", "esbaganh", "esbago", "esbagulh", "esbalanç",
"esbald", "esbalh", "esbaliz", "esbalouc", "esbamb", "esbambo",
"esbandalh", "esbandeir", "esbandulh", "esbang", "esbanj", "esbanzalh",
"esbar", "esbarafust", "esbaralh", "esbarb", "esbarbot", "esbard",
"esbardalh", "esbarr", "esbarranc", "esbarreg", "esbarrel", "esbarrig",
"esbarro", "esbarroc", "esbarrond", "esbarrot", "esbarru", "esbarrunt",
"esbeat", "esbeiç", "esbeij", "esbeir", "esbelt", "esbeltific",
"esbeltiz", "esbenair", "esbenic", "esberraç", "esberri", "esbic",
"esbich", "esbij", "esbilhot", "esbilit", "esbirr", "esboç",
"esbocel", "esbodeg", "esbodel", "esbof", "esbofet", "esboiç",
"esboicel", "esbol", "esbombard", "esborcel", "esborcin", "esbord",
"esbordo", "esborni", "esboro", "esborol", "esborr", "esborraç",
"esborrach", "esborralh", "esborrat", "esborreg", "esborret", "esborrif",
"esborrot", "esboten", "esbouç", "esboucel", "esbrabej", "esbraç",
"esbracej", "esbram", "esbranquiç", "esbravej", "esbrech", "esbrinç",
"esbrit", "esbriz", "esbro", "esbronc", "esbrucin", "esbrug",
"esbrum", "esbugalh", "esbulh", "esburac", "esburg", "esbux",
"escabeç", "escabech", "escabel", "escabeleir", "escabeliz", "escabich",
"escabr", "escabuf", "escabuj", "escabulh", "escac", "escach",
"escacho", "escachol", "escachouç", "escacilh", "escaço", "escadabulh",
"escadeir", "escadraç", "escadrilh", "escafandr", "escafel", "escafon",
"escaful", "escafun", "escagaç", "escaganit", "escagarrinh", "escaiol",
"escal", "escalad", "escalamanc", "escalambr", "escalamouc", "escalavr",
"escald", "escaldeir", "escalf", "escaliç", "escalmorr", "escalon",
"escalp", "escalpel", "escalpeliz", "escalpirr", "escalrach", "escalv",
"escalvej", "escam", "escamalh", "escamalho", "escamb", "escambi",
"escambich", "escambr", "escamech", "escamel", "escamis", "escamond",
"escamos", "escamot", "escamp", "escamuj", "escamurç", "escamurreng",
"escan", "escanç", "escancar", "escancel", "escanch", "escand",
"escandaliz", "escanec", "escaneriz", "escang", "escangalh", "escanganh",
"escangot", "escanho", "escanhot", "escanifr", "escaninh", "escantilh",
"escanto", "escanzel", "escanzi", "escanzurr", "escap", "escapel",
"escapolitiz", "escaqueir", "escaraban", "escarafol", "escarafunch", "escaramb",
"escaramel", "escaramuç", "escarapel", "escarapuç", "escaravelh", "escarç",
"escarcalh", "escarcavel", "escarch", "escarchol", "escard", "escardaç",
"escardiç", "escardich", "escardilh", "escarduç", "escarfunch", "escarg",
"escari", "escarific", "escarlat", "escarment", "escarmouç", "escarn",
"escarnefuch", "escarni", "escarnic", "escarnific", "escarninh", "escarnitiz",
"escaroç", "escarol", "escarouç", "escarp", "escarpel", "escarpi",
"escarpilh", "escarquej", "escarr", "escarrach", "escarraman", "escarranch",
"escarrapach", "escarrapanch", "escarrapat", "escarrapiç", "escarrapich", "escarrinh",
"escarum", "escarv", "escarvanç", "escarvo", "escarvunç", "escasal",
"escasc", "escassilh", "escasul", "escatel", "escatim", "escatin",
"escatoliz", "escav", "escavac", "escavaç", "escavach", "escavalo",
"escaveir", "escavin", "escavinh", "escaxel", "eschamej", "esclamouc",
"esclerific", "escleros", "esco", "escobilh", "escoc", "escoch",
"escochin", "escod", "escogi", "escoiç", "escoicinh", "escoim",
"escolariz", "escoldrinh", "escoli", "escolm", "escolt", "escombr",
"escongemin", "esconjunt", "esconjur", "escons", "escopet", "escor",
"escorç", "escorcemel", "escorch", "escorçomel", "escord", "escori",
"escorific", "escorj", "escorn", "escornich", "escoro", "escorraç",
"escorrach", "escorreg", "escorrich", "escorrimaç", "escorripich", "escorropich",
"escortic", "escortin", "escortinh", "escoruj", "escorv", "escosip",
"escoti", "escotomiz", "escouç", "escoucemel", "escoucepinh", "escoucinh",
"escov", "escovalh", "escovilh", "escovinh", "escozic", "escrach",
"escrachet", "escrachin", "escrafunch", "escramalh", "escramel", "escrament",
"escramouc", "escramuç", "escravelh", "escraviz", "escresp", "escrevinh",
"escritur", "escriv", "escrofulariz", "escrofuliz", "escruch", "escruci",
"escrunch", "escrupuliz", "escrut", "escrutin", "escud", "escudeir",
"escudel", "escudrinh", "esculach", "esculc", "esculhamb", "escult",
"escultur", "escum", "escumilh", "escur", "escurej", "escurent",
"escus", "escut", "esdruxul", "esdruxuliz", "esfacel", "esfagunt",
"esfaim", "esfalc", "esfalcaç", "esfalf", "esfalrip", "esfandeg",
"esfango", "esfanic", "esfard", "esfarel", "esfarfalh", "esfarinh",
"esfarp", "esfarpel", "esfarrap", "esfarrip", "esfarrunc", "esfatach",
"esfatan", "esfatech", "esfateix", "esfati", "esfavel", "esfelg",
"esfergulh", "esfervilh", "esfi", "esfiamp", "esfiap", "esfibr",
"esfloc", "esflor", "esfocinh", "esfoeir", "esfogueir", "esfoiç",
"esfoir", "esfol", "esfoleg", "esfolh", "esfolhaç", "esfoli",
"esforç", "esforfalh", "esforgulh", "esformig", "esforneir", "esfornic",
"esfornig", "esforquilh", "esforric", "esfoss", "esfossiliz", "esfossinh",
"esfouç", "esfrald", "esfranç", "esfrandelh", "esfrangalh", "esfreg",
"esfregonç", "esfregulh", "esfri", "esfrol", "esfrunh", "esfueir",
"esfugant", "esfugent", "esfulinh", "esfum", "esfumaç", "esfumar",
"esfumilh", "esfuminh", "esfundilh", "esfurac", "esfuranc", "esfurgalh",
"esfuri", "esfurinh", "esfuro", "esfutric", "esfuzi", "esfuzil",
"esg", "esgaç", "esgadanh", "esgadelh", "esgadunh", "esgaiv",
"esgaldrip", "esgalg", "esgalh", "esgalinh", "esgalrich", "esgamo",
"esgamp", "esgan", "esganch", "esganiç", "esganif", "esganifr",
"esganit", "esgar", "esgarabanh", "esgarabulh", "esgarafunch", "esgarafunh",
"esgaratuj", "esgaravan", "esgaravat", "esgaravunch", "esgaravunh", "esgarç",
"esgardunh", "esgarg", "esgargal", "esgargalh", "esgarn", "esgarnach",
"esgarr", "esgarrabunh", "esgarrafunch", "esgarranch", "esgarranh", "esgasgalh",
"esgatanh", "esgatiç", "esgod", "esgoel", "esgoldrej", "esgorj",
"esgot", "esgrabulh", "esgrafi", "esgrafit", "esgraminh", "esgravan",
"esgravat", "esgravelh", "esgravulh", "esgrelh", "esgril", "esgrim",
"esgrouvi", "esgrouvinh", "esgrovinh", "esguard", "esguarit", "esguedelh",
"esgueir", "esguelh", "esgui", "esguich", "esguilh", "esguit",
"esgurej", "esladro", "eslag", "eslagart", "eslaviz", "eslavoir",
"eslavour", "esling", "esm", "esmadrig", "esmag", "esmagach",
"esmagalh", "esmagunch", "esmai", "esmaleit", "esmalh", "esmalm",
"esmalt", "esmamon", "esmani", "esmanj", "esmantel", "esmaranh",
"esmaravalh", "esmarc", "esmarmoir", "esmarmour", "esmarnec", "esmarnoc",
"esmarro", "esmarrot", "esmartuç", "esmatruç", "esmech", "esmelm",
"esmelodr", "esmen", "esmer", "esmeril", "esmerilh", "esmerm",
"esmeti", "esmich", "esmifr", "esmigalh", "esmijaç", "esmilh",
"esmiol", "esmirr", "esmiud", "esmiunç", "esmoc", "esmoch",
"esmoic", "esmoit", "esmol", "esmolamb", "esmon", "esmonc",
"esmond", "esmorç", "esmordaç", "esmordel", "esmordic", "esmordiç",
"esmorm", "esmoron", "esmorr", "esmorraç", "esmorrat", "esmorz",
"esmoss", "esmouc", "esmouquel", "esmout", "esmur", "esmurr",
"esmurraç", "esmurreg", "esmurreng", "esnac", "esnob", "esnoc",
"esnu", "espaç", "espacej", "espaci", "espacializ", "espad",
"espadachin", "espadag", "espadan", "espadeir", "espadel", "espadi",
"espadilh", "espadu", "espajeg", "espald", "espaldeir", "espalh",
"espalhafat", "espalhag", "espalit", "espalm", "espalomb", "espampan",
"espampar", "espan", "espanasc", "espanc", "espandong", "espanej",
"espanhol", "espanholiz", "espant", "espantarol", "espap", "espapaç",
"espaparr", "esparcel", "espardalh", "espargel", "esparr", "esparrach",
"esparralh", "esparram", "esparranh", "esparreg", "esparrel", "esparrim",
"esparrinh", "esparrod", "espars", "espartej", "espartilh", "esparv",
"esparvo", "espasm", "espassaric", "espassarot", "espatarr", "espatif",
"espatilh", "espatul", "espavent", "espavil", "espavit", "espavoriz",
"espec", "espeç", "especi", "especializ", "especific", "espect",
"espectraliz", "espectrific", "especul", "espedaç", "espedarç", "espedreg",
"espeit", "espeitor", "espelh", "espendur", "espenej", "espenic",
"espenifr", "espenuj", "esper", "esperanç", "esperanci", "esperdiç",
"esperdigot", "esperjur", "esperlux", "espermatiz", "espermeg", "esperneg",
"esperrinch", "espert", "espertin", "espescoç", "espess", "espet",
"espetaculiz", "espetal", "espetraliz", "espetrific", "espevit", "espezinh",
"espi", "espiant", "espicaç", "espicaço", "espich", "espicoiç",
"espicouç", "espicul", "espif", "espig", "espigait", "espiguilh",
"espildr", "espilr", "espinafr", "espinç", "espinch", "espingalh",
"espingard", "espinh", "espinic", "espinot", "espiolh", "espion",
"espip", "espipoc", "espir", "espiral", "espiraliz", "espirantiz",
"espirit", "espiritiz", "espiritualiz", "espirr", "espirrich", "esplain",
"esplendor", "esplenectomiz", "espo", "espoc", "espoj", "espojinh",
"espoldr", "espoldrinh", "espolet", "espoli", "espolin", "espolinh",
"espolp", "esponj", "espont", "esponteir", "esponto", "espor",
"esporific", "esporo", "esporr", "esporteir", "esportul", "esporul",
"espos", "espostej", "espot", "espotrej", "espotric", "espouc",
"espragat", "esprai", "espreguiç", "espreit", "espremeg", "esprit",
"espulg", "espum", "espumaç", "espumarej", "espumej", "espupin",
"esput", "esquadr", "esquadraç", "esquadrej", "esquadri", "esquadrilh",
"esquadrinh", "esquadron", "esquartej", "esquartel", "esquartilh", "esquematiz",
"esquent", "esquerd", "esquerdiz", "esqui", "esquiç", "esquif",
"esquil", "esquin", "esquip", "esquirol", "esquiss", "esquistific",
"esquit", "esquiv", "estaban", "estabilit", "estabiliz", "estaboug",
"estabul", "estac", "estaç", "estacion", "estadi", "estadiz",
"estadualiz", "estaf", "estafeg", "estaferm", "estafet", "estafon",
"estagflacion", "estagi", "estagn", "estai", "estal", "estaleir",
"estalej", "estalic", "estalid", "estalon", "estambr", "estamin",
"estamp", "estampilh", "estanc", "estanci", "estandardiz",
"estanh", "estard", "estardalh", "estardalhaç", "estarel", "estarrific",
"estarrinc", "estarro", "estartal", "estasi", "estassalh", "estatel",
"estatistific", "estatiz", "estatu", "estaz", "esteanaz", "esteatiz",
"esteir", "estelic", "estenaz", "estenodactilograf", "estenodatilograf", "estenofotolitograf",
"estenograf", "estenos", "estenotip", "ester", "esterc", "esterç",
"estereofotograf", "estereograf", "estereotip", "estereotomiz", "esterific", "esteriliz",
"esterlic", "esternic", "esternud", "esternut", "esterotomiz", "esterr",
"esterro", "estertor", "estertoriz", "estesi", "estesilh", "estesoir",
"estesour", "esteticiz", "estetiz", "estev", "esti", "estic",
"estigm", "estigmatiz", "estilet", "estiletiz", "estilh",
"estilhaç", "estilicid", "estiliz", "estim", "estimul", "esting",
"estinh", "estiol", "estiomen", "estipendi", "estipul", "estir",
"estiraç", "estiv", "estoc", "estoç", "estof", "estoir",
"estoirinh", "estoj", "estol", "estomag", "estomeg", "estoment",
"eston", "estonh", "estont", "estop", "estopet", "estoqueir",
"estorceg", "estorcic", "estordeg", "estorn", "estorneg", "estornic",
"estorrej", "estorric", "estorro", "estorteg", "estortur", "estorv",
"estorved", "estour", "estourinh", "estoutin", "estouv", "estov",
"estr", "estrab", "estrabouc", "estrabouch", "estraçalh", "estracinh",
"estraço", "estrad", "estradul", "estrafeg", "estrag", "estral",
"estralej", "estramalh", "estrambalh", "estrambelh", "estramont", "estramp",
"estrampalh", "estramunh", "estranç", "estrancilh", "estrancinh", "estranfeni",
"estranfoli", "estranfolinh", "estranfoni", "estrangalh", "estrangul",
"estranh", "estrapaç", "estrapilh", "estraquej", "estrategi", "estratific",
"estrav", "estravir", "estrebang", "estrebord", "estrebuch", "estrefeg",
"estreg", "estreit", "estrel", "estrelej", "estrelic", "estrelouç",
"estrem", "estremadur", "estremalh", "estremud", "estremulh", "estremunh",
"estrenoit", "estrenout", "estrep", "estrepass", "estrepit", "estress",
"estressalh", "estretal", "estri", "estrib", "estribilh", "estriç",
"estrich", "estricniniz", "estricniz", "estridul", "estrig", "estril",
"estrim", "estrinc", "estrinç", "estrinch", "estrip", "estrobiliz",
"estroç", "estrocic", "estroin", "estroix", "estromb", "estromp",
"estronc", "estronç", "estrond", "estrondej", "estronf", "estrongalh",
"estropeli", "estropi", "estrotej", "estrouvinh", "estroux", "estrovinh",
"estrufeg", "estrum", "estrumbic", "estrupid", "estrutur", "estruturaliz",
"estu", "estuc", "estuch", "estud", "estuf", "estug",
"estuliz", "estultific", "estum", "estupefic", "estupid", "estupidific",
"estupigait", "estupor", "estupr", "esturdi", "esturr", "esturric",
"esturrinh", "esvali", "esvalij", "esvar", "esvaz", "esvazi",
"esvent", "esventr", "esverç", "esverdinh", "esvid", "esvidig",
"esvinh", "esviscer", "esvoaç", "esvoej", "esvurm", "eterific",
"eteriz", "etern", "eterniz", "eticiz", "etil", "etiliz",
"etimologiz", "etiquet", "eufemiz", "eufoniz", "euforiz", "eugeniz",
"eutrofiz", "ev", "evacu", "evagin", "evangeliz", "evapor",
"evaporiz", "evapotranspir", "evaz", "eventr", "evermin", "evidenci",
"evir", "eviscer", "evit", "evoc", "evol", "evolucion",
"exabund", "exacerb", "exager", "exagit", "exal", "exalç",
"exalt", "examin", "exar", "exarticul", "exasper", "exatific",
"exaust", "exautor", "excarcer", "excardin", "excarnific", "excecion",
"excels", "excepcion", "exceptu", "excetu", "excis", "excit",
"exclam", "exclaustr", "exclausur", "excogit", "excomung", "excomunh",
"excorpor", "excrement", "excret", "excruci", "exculp", "excurs",
"excursion", "excurv", "execr", "execut", "exempl", "exemplariz",
"exemplific", "exempt", "exercit", "exerd", "exgreg", "exiguific",
"exil", "exit", "exoner", "exor", "exorbit", "exorcism",
"exorciz", "exordi", "exorn", "exort", "expatri", "expect",
"expector", "expedr", "expeitor", "experienci", "experiment", "expet",
"expetor", "expi", "expil", "expir", "explan", "explant",
"explic", "explicit", "explor", "explot", "exponenci", "export",
"expostul", "express", "exprob", "exprobr", "expropri", "expugn",
"expuls", "expurg", "exsic", "exst", "exsu", "exsud",
"extasi", "extenu", "exterioriz", "extermin", "extern", "externaliz",
"externiz", "extirp", "extrabord", "extract", "extradicion", "extradit",
"extrafoli", "extrapass", "extrapol", "extrat", "extravaganci", "extravas",
"extravi", "extrem", "extric", "extrofi", "extub", "exuber",
"exul", "exulcer", "exult", "exum", "exund", "exurbaniz",
"exventr", "fabric", "fabul", "fabuliz", "faccion", "faceci",
"faceir", "facej", "facet", "fach", "fachol", "facilit",
"facion", "factor", "facult", "facundi", "fad", "fadej",
"fadig", "fadist", "fadistocratiz", "fag", "fagulh", "fai",
"fain", "fair", "faix", "fajut", "fal", "falaç",
"falaci", "faladr", "falaz", "falc", "falcaç",
"falcatru", "falco", "falej", "falh", "falhoc", "faloc",
"falporri", "falquej", "fals", "falsific", "falt", "faluc",
"fambiss", "familiariz", "familot", "famul", "fan", "fanatiz",
"fandang", "fanding", "fanf", "fanfarr", "fanfarron", "fanfurri",
"fangan", "fangarel", "fanic", "fanich", "fantasi", "fantasm",
"fantasmagoriz", "fantasmiz", "fantoch", "fantochin", "fanuc", "faquiriz",
"far", "faradiz", "farandol", "fard", "farej", "farel",
"faretr", "farfalh", "farfalhej", "farfalin", "farin", "faringaliz",
"farinh", "farisc", "farofi", "farol", "faroliz", "farp",
"farpo", "farr", "farrap", "farrasc", "farronc", "farsol",
"fart", "farung", "fascin", "fascistiz", "fasqui", "fast",
"fastigi", "fataliz", "fatan", "fatej", "fati", "fatig",
"fator", "fatoriz", "fatuej", "fatur", "faul", "faulh",
"fautoriz", "fav", "faveliz", "favoni", "favorit", "favoriz",
"faxin", "febricit", "febriliz", "fech", "fecund", "fecundiz",
"fedanh", "fedelh", "feder", "federaliz", "fedinch", "fedonh",
"fefet", "feitori",
"feitoriz", "fel", "feldspatiz", "felicit", "feloni", "felp",
"feltr", "femb", "femenç", "feminiliz", "feminiz", "fen",
"fendilh", "fendrelh", "fendrilh", "fenestr", "fenoliz", "fenomenaliz",
"feri", "ferment", "ferr", "ferrament", "ferrej", "ferret",
"ferreto", "ferro", "ferrob", "ferrolh", "fertiliz", "fervelh",
"fervent", "fervilh", "fervor", "fescoliz", "fest", "festej",
"festin", "festo", "feston", "feudaliz", "fi", "fiamp",
"fibril", "fic", "fich", "fideliz", "figur",
"fil", "filagran", "filet", "filhar", "filhastr",
"fili", "filigran", "filip", "filipiz", "filistri", "film",
"filmograf", "filoidin", "filosof", "filtr", "fimbri", "fin",
"finaliz", "financi", "finc", "find", "findaliz", "finf",
"fint", "firm", "firr", "fiscaliz", "fiscel", "fisg",
"fisiologiz", "fissiliz", "fission", "fissur", "fistul", "fistuliz",
"fit", "fituc", "fix", "flabel", "flafl", "flagel",
"flagici", "flagr", "flain", "flam", "flamb", "flamej",
"flamispir", "flamul", "flan", "flart", "flaut", "flavortiz",
"flebotomiz", "flech", "flegmasi", "flert", "flexibiliz", "flexion",
"flexu", "flim", "flip", "flirt", "flit", "floc",
"flocon", "flocul", "flogistic", "flogos", "flor", "florej",
"florest", "flostri", "flot", "fluid", "fluidific", "fluidiz",
"fluor", "flutu", "fob", "fobit", "foc", "foç",
"focaliz", "focinh", "fof", "fofoc", "fofuenh", "fogach",
"foiç", "foicinh", "folcloriz", "folec", "foleg", "folerp",
"folestri", "folg", "folgaz", "folh", "folhec", "folhetiniz",
"foli", "foment", "fonetiz", "fonfon", "fonocardiograf", "fonocinematograf",
"fonofilm", "fonograf", "fonologiz", "forc", "forç", "forcej",
"forj", "forjic", "form", "formaliz", "format", "formic",
"formicid", "formig", "formiguej", "formilh", "formol", "formoliz",
"formosent", "formul", "formulariz", "fornej", "fornic", "fornig",
"fornizi", "forquilh", "forr", "forraj", "forrej", "forric",
"fortaç", "fortaleg", "fortalez", "fortific", "fortun", "fosc",
"fosfat", "fosfor", "fosforej", "fosforil", "fosforit", "fosforiz",
"foss", "fossang", "fossilific", "fossiliz", "fotoativ", "fotocopi",
"fotodissoci", "fotofosforiliz", "fotograf", "fotograv", "fotointerpret", "fotolit",
"fotolitograf", "fotometr", "fotominiaturiz", "fotomont", "fotomultiplic", "fotosforil",
"fotossensibiliz", "fotossintetiz", "fototip", "fototipi", "fouç", "foucinh",
"frac", "fracass", "fraccion", "fracion", "fractur", "frad",
"fradej", "fragiliz", "fragment", "fragor", "frald", "fraldej",
"fraldic", "framaçoniz", "frances", "francesi", "franciuz", "francofoniz",
"frangalh", "franj", "franqui", "fraquej", "fraquent", "frasquej",
"frasquinh", "fraterniz", "fratur", "fraud", "fraut", "frech",
"freg", "freir", "frejuc", "fren", "frenesi", "frequent",
"fres", "fresc", "fresquej", "fret", "fretej", "frev",
"fricatiz", "friccion", "frigorific", "frinch", "fris", "frit",
"froc", "froix", "frol", "frond", "frondej", "front",
"fronteir", "froux", "frufrulh", "frufrut", "frulh", "frustr",
"fruteg", "frutific", "fubec", "fuç", "fueir",
"fug", "fujic", "fulaniz", "fulec", "fulgent", "fulgur",
"fulguraresfuzi", "fulmin", "fulo", "fum", "fumaç", "fumar",
"fumeg", "fumig", "fun", "funambul", "funcion", "funcionaliz",
"fund", "fundalh", "fundament", "fundilh", "funest", "funf",
"funfi", "fung", "fur", "furac", "furdunç", "furfunh",
"furifun", "furo", "furri", "furt", "furtic", "furunf",
"furung", "fus", "fusc", "fusg", "fusion", "fustig",
"futic", "futiliz", "futret", "futric", "futuc",
"futuriz", "fuxic", "fuzil", "gab", "gabarit", "gabarol",
"gabi", "gabion", "gadanh", "gadiç", "gadunh", "gaf",
"gafeir", "gafej", "gaguej", "gai", "gaiat", "gaifon",
"gaimo", "gait", "gaiv", "gaj", "gal", "galan",
"galantiz", "galardi", "galardo", "galderi", "galdrop", "galeguiz",
"galg", "galhof", "galic", "galiciparl", "galicism", "galiciz",
"galim", "galimatiz", "galinh", "galiv", "galo", "galomaniz",
"galon", "galop", "galopin", "galr", "galrej", "galrich",
"galroch", "galvaniz", "galvanotip", "gam", "gambel", "gambi",
"gamenh", "gananci", "ganç", "ganch", "gandai", "gandul",
"ganf", "gangorr", "gangren", "ganguiss", "ganh", "ganiç",
"ganiz", "ganz", "gapin", "gapui", "garabulh", "garalh",
"garanç", "garatuj", "garavet", "garf", "garfej", "garfi",
"garg", "gargadaç", "gargalaç", "gargalej", "gargalh", "gargantu",
"gargarej", "gargolej", "gargorej", "garguel", "garimb", "garimp",
"garin", "garn", "garo", "garot", "garr", "garram",
"garranch", "garrid", "garroch", "garrot", "garrotearestrangul", "garruch",
"garrul", "garrulej", "garu", "gasalh", "gasconiz", "gaseific",
"gasguit", "gasific", "gasm", "gast", "gastrofotograf", "gastrul",
"gat", "gatafunh", "gatanh", "gatinh", "gatun", "gatunh",
"gatur", "gaturr", "gaud", "gauder", "gauderi", "gaudi",
"gaudin", "gav", "gavion", "gazil", "gazilh", "gazin",
"gazofil", "gazol", "gazu", "gazul", "gazup", "geb",
"gebr", "gegadelh", "gel", "gelatin", "gelatinific", "gelatiniz",
"geleific", "gelific", "gem", "gemelg", "gemelhic", "gemi",
"gemic", "gemin", "gemul", "generaliz", "gentiliz", "geocentr",
"geograf", "geometriz", "georreferenci", "ger", "gerenci", "german",
"germaniz", "germin", "gess", "gest", "gesticul", "gestion",
"gestualiz", "gigantiz", "ginastic", "ging", "gingr", "gir",
"giraldin", "girandol", "girassol", "girogir", "girovag", "giz",
"glaç", "gladi", "glamoriz", "glanduliz", "glicerin", "glicuroniz",
"gliss", "globaliz", "globuliz", "glomer", "glori", "glorific",
"glos", "gloss", "gloter", "glotiz", "glotor", "glugluej",
"gluglurej", "gluglut", "glutin", "godemiz", "goder", "godi",
"goel", "gofr", "goi", "goit", "goiv", "gol",
"golej", "golelh", "golf", "golfej", "golip", "golp",
"gom", "gombel", "gomific", "gomit", "gomo", "gon",
"gondol", "gong", "gongor", "gongoriz", "gor", "gordur",
"gorgol", "gorgolej", "gorgolh", "gorgorej", "gorr", "gosm",
"gosmilh", "gost", "gotej", "goticiz", "goug", "govern",
"governamentaliz", "governich", "goz", "gracej", "gracit", "graçol",
"grad", "gradearesterro", "gradej", "gradin", "gradu", "graf",
"grafil", "grafit", "grafitiz", "grai", "gralh", "gram",
"gramaticaliz", "gramaticiz", "graminh", "gramofon", "gramp",
"gran", "granid", "granit", "granitific", "graniz", "granul",
"granzin", "grasn", "grass", "grassit", "gratific", "gratin",
"gratul", "grav", "gravet", "gravid", "gravit", "gravunh",
"gravur", "grazin", "greciz", "greguej", "grel", "grelh",
"gret", "grif", "gril", "grilhet", "grimp", "grinald",
"grinf", "grip", "gris", "grisalh", "grit", "griv",
"grogolej", "grol", "grom", "grong", "gros", "grosm",
"grossul", "grud", "grugrulej", "grugrulh", "grugulej", "grugunz",
"grulh", "grum", "gruzi", "guaguax", "guai",
"gualdrip", "guard", "guarit", "guasc", "guaz", "guedelh",
"gueng", "guerram", "guerrilh", "gueviss", "gugluej", "gui",
"guid", "guilhoch", "guilhotin", "guiloch", "guin", "guinch",
"guind", "guinh", "guirland", "guis", "guiz", "guizalh",
"gulaim", "gulap", "gulaz", "guleim", "gulos", "guloseim",
"gulosin", "gunf", "gungun", "gurgit", "gust", "gut",
"guturaliz", "habilit", "habit", "habitu", "hachur", "halit",
"halogen", "halogenoderiv", "haragan", "hariol", "harmoniz", "harp",
"harpej", "harpeji", "hastilh", "hausmaniz", "haussemaniz", "haussmanniz",
"hebet", "hebetiz", "hectometr", "hegemoniz", "heleboriz", "heleniz",
"heliport", "helitransport", "hemaglutin", "hematiz", "hematos", "hemoaglutin",
"hemoconcentr", "hemodialis", "hemolis", "hemossediment", "hepatectomiz", "hepatiz",
"herboriz", "herd", "heresi", "hermetific", "herni", "heroiciz",
"heroific", "heroiz", "hesit", "het", "heteroaglutin", "heterocarreg",
"heterocromatiz", "heterogeneiz", "heteroimuniz", "heteropolimeriz", "heterossugestion", "heterotransplant",
"hialiniz", "hiatiz", "hibern", "hibrid", "hibridiz", "hidrat",
"hidrobor", "hidrocarbon", "hidrocut", "hidrodessulfuriz", "hidroextrat", "hidrofobiz",
"hidrogen", "hidrolis", "hidrometr", "hidroport", "hidrotomiz", "hidroxideriv",
"hiemaliz", "hierarquiz", "hieratiz", "hifeniz", "higieniz", "hilari",
"hilariz", "hiperadapt", "hiperafirm", "hiperajust", "hiperaliment", "hiperaplic",
"hiperaprimor", "hiperativ", "hiperbenefici", "hiperboliz", "hipercapacit", "hipercaricatur",
"hipercomplex", "hiperdimension", "hiperdos", "hiperemi", "hiperequip", "hiperestesi",
"hiperestrutur", "hiperexager", "hiperexplor", "hiperglicemi", "hiperimpregn", "hiperimuniz",
"hiperindustri", "hiperindustrializ", "hiperinfeccion", "hiperinflacion", "hiperirrit", "hiperliterariz",
"hiperocup", "hiperofert", "hiperorden", "hiperorganiz", "hiperotimiz", "hiperoxid",
"hiperoxigen", "hiperpigment", "hiperpolariz", "hipersemantiz", "hipersensibiliz", "hipertrofi",
"hiperurbaniz", "hipervaloriz", "hipervasculariz", "hiperventil", "hipnotiz", "hipocratiz",
"hipofisectomiz", "hipoglicemi", "hipossensibiliz", "hipostasi", "hiposteniz", "hipotec",
"hipotrofi", "hipoxi", "hirt", "hirudiniz", "hisp", "hispanific",
"hispaniz", "hispid", "hissop", "histeriz", "histocompatibiliz", "histodiferenci",
"histori", "historializ", "historiciz", "historiz", "histrion", "holandiz",
"holocaust", "holofotiz", "holograf", "homeotransplant", "homicid", "homili",
"hominiz", "homizi", "homogeniz", "homolog", "homopolimeriz", "homostimul",
"homotransplant", "honest", "honestiz", "honor", "honorific", "honr",
"honrific", "hor", "horej", "horizont", "horizontaliz", "hormoniz",
"horoscop", "horoscopiz", "horrific", "horripil", "horroriz", "hort",
"hortej", "horticult", "hosped", "hospitaliz", "hostiliz", "hulhific",
"human", "humaniz", "humet", "humific", "humild", "humilh",
"humoriz", "ianquiz", "iarovis", "iberiz", "iç", "ichacorv",
"iconiz", "icterici", "idealiz", "identific", "ideograf", "ideologiz",
"idiliz", "idiot", "idiotific", "idiotiz", "idolatr", "idolatriz",
"ignific", "ignifug", "igniz", "ignobiliz", "ignomini", "ignor",
"igual", "iguald", "igualh", "igualiz", "il", "ilegaliz",
"ilegitim", "ilh", "ilib", "iliç", "ilig", "ilimit",
"ilumin", "ilusion", "ilustr", "ilut", "iluvi", "imagin",
"iman", "imanentiz", "imaniz", "imant", "imantiz", "imaterializ",
"imbecilit", "imbeciliz", "imbelic", "imbelinh", "imbic",
"imbond", "imbram", "imbric", "imedi", "imediat", "imers",
"imigr", "imit", "imobiliz", "imoir", "imol", "imortaliz",
"imot", "imour", "imp", "impacient", "impact", "impalud",
"impan", "imparcializ", "impassibiliz", "impelg", "imper", "imperfeiço",
"imperializ", "imperl", "impermeabiliz", "impersonaliz", "impersonific", "impertinenci",
"impessoaliz", "impet", "impetilh", "impetr", "impinim", "implant",
"implement", "implic", "implor", "impont", "impopulariz", "import",
"importun", "impossibilit", "impost", "impostur", "imprec", "imprecat",
"imprecis", "impregn", "impremedit", "imprens", "impression", "imprestabiliz",
"imprim", "improb", "impront", "impronunci", "improper", "improporcion",
"impropri", "improv", "improvis", "impugn", "impuls", "impulsion",
"impulsiv", "impurific", "imput", "imund", "imuniz", "imut",
"imutabiliz", "inabilit", "inabit", "inadapt", "inadat", "inadestr",
"inadun", "inal", "inalien", "inalter", "inambul", "inanim",
"inativ", "inaugur", "inç", "incamer", "incapacit", "incapsul",
"incaracteriz", "incardin", "incarn", "incens", "incentiv", "incert",
"incest", "incestu", "inch", "inci", "incident", "inciner",
"incis", "incit", "inclin", "inco", "incolor", "incomod",
"incompatibiliz", "incomunic", "inconform", "inconfort", "inconsolid", "inconst",
"inconstitucionaliz", "incontest", "incorpor", "incrass", "increment", "increp",
"incrimin", "incris", "incruent", "incrust", "incub", "inculc",
"inculp", "incurric", "incursion", "indag", "indegr", "indelimit",
"indemniz", "indeniz", "independentiz", "independiz", "indetect", "indetermin",
"index", "indianiz", "indic", "indici", "indicific", "indigeniz",
"indigest", "indigit", "indign", "indignific", "indilg", "indinh",
"indisciplin", "indiscrimin", "individu", "individualiz", "indociliz", "indocument",
"indolent", "indoutrin", "indromin", "indulgenci", "indult", "indument",
"industri", "industrializ", "indut", "inebri", "ineptiz", "inequacion",
"inerci", "inertiz", "inerv", "inespoli", "infam", "infantiliz",
"infart", "infeccion", "infecion", "infect", "infecund", "infelicit",
"inferioriz", "infern", "inferniz", "infertiliz", "infest", "infet",
"infibul", "inficion", "infiltr", "infinc", "infirm", "infix",
"infl", "inflacion", "inflam", "inflexion", "inflor", "influenci",
"infon", "inform", "informaliz", "informatiz", "infortun", "infrond",
"infuleim", "infundament", "infunic", "infus", "ingenu", "ingles",
"ingress", "ingund", "ingurgit", "inici", "inicializ", "inigual",
"inimist", "inimit", "inimiz", "injeriz", "injet", "injunç",
"injuri", "injustiç", "injustific", "inlernaliz", "inliç", "inobserv",
"inocent", "inocul", "inor", "inoscul", "inov", "inquart",
"inquezil", "inquiet", "inquilin", "inquin", "inquizil", "insacul",
"insaliv", "insatur", "insemin", "insensibiliz", "insert", "insidi",
"insigni", "insimul", "insinu", "insipid", "insol", "insolubiliz",
"insoni", "insonoriz", "insoss", "inspecion", "inspet", "inspir",
"inspiss", "inst", "instabiliz", "instal", "instanci", "instantaniz",
"instaur", "instig", "instil", "institucionaliz", "instrument", "instrumentaliz",
"insubordin", "insufl", "insul", "insult", "insum", "insurreccion",
"insurrecion", "integr", "integraliz", "inteig", "inteir", "inteiriç",
"inteleccion", "intelecion", "intelectualiz", "inteligenci", "intemper", "intemperiz",
"intencion", "intens", "intensific", "intensiv", "intent", "interatu",
"intercal", "intercambi", "intercept", "intercomunic", "interconect", "intercruz",
"interdigit", "interdimension", "interdit", "interespacej", "interess", "interesterific",
"interestratific", "interfoli", "interfon", "interiç", "interinfluenci",
"interioriz", "interjeccion", "interjecion", "interjectiv", "interjeicion", "interjetiv",
"interlaç", "interlig", "interm", "intermigr", "intermistur", "intermostr",
"intern", "internacionaliz", "internaliz", "interoper", "interpel", "interpenetr",
"interpic", "interpol", "interpolimeriz", "interpres", "interpret", "interrog",
"intersach", "intersec", "interseccion", "intersect", "interset", "intertransform",
"intertrav", "interturb", "interval", "intervencion", "intestel", "intestin",
"intic", "intim", "intimid", "intitul", "inton", "intoxic",
"intranquiliz", "intransitiv", "intrassel", "intrech", "intric", "intrigalh",
"intrinc", "introject", "introjet", "intrors", "intruid", "intruj",
"intub", "intuicion", "inum", "inund", "inutiliz", "invagin",
"invalid", "invari", "invas", "invect", "invectiv", "invej",
"invencion", "invent", "inventari", "invern", "invespr", "investig",
"invet", "inveter", "invetiv", "inviabiliz", "inviccion", "invid",
"invidi", "inviolabiliz", "inviper", "inviscer", "invisibiliz", "invit",
"invoc", "involucr", "invulg", "inzon", "iob", "iod",
"iodet", "iodin", "ioniz", "iontiz", "iotiz", "ir",
"iri", "iridi", "iris", "iriz", "irman", "ironiz",
"irracionaliz", "irradi", "irrealiz", "irreconcili", "irresponsabiliz", "irreverenci",
"irrig", "irrit", "irrog", "irror", "isc", "isent",
"islamiz", "isoacentu", "isoaglutin", "isocroniz", "isol", "isomeriz",
"israeliz", "italianiz", "italiciz", "itemiz", "iter", "jacobiniz",
"jact", "jactit", "jacul", "jaez", "jan", "janel",
"jang", "janot", "jant", "jantaric", "jantarinh", "japones",
"japoniz", "Japonv", "jard", "jardin", "jaroviz", "jarret",
"jarund", "jat", "jatanci", "jatit", "javeliz", "javr",
"jazzific", "jeit", "jeju", "jerarquiz", "jeremi", "jeremin",
"jesuitiz", "jimbr", "job", "joeir", "joelh", "jog",
"jogalh", "jogat", "jogatilh", "jogral", "jograliz", "joguet",
"jol", "jong", "jonj", "jornalistic", "jornaliz", "jorr",
"jovializ", "jubil", "judi", "judic", "judici", "judicializ",
"jug", "jugad", "jugul", "julg", "junc", "junt",
"jur", "jurament", "jurem", "jurg", "juridiciz", "jurisdicion",
"jurisdicionaliz", "just", "justiç", "justific", "juveniliz", "labanot",
"labefact", "label", "labializ", "labiovelariz", "labirint", "labor",
"labreg", "labrest", "labut", "labuz", "lac", "laç",
"lacai", "lacer", "laconiz", "lacr", "lacram", "lacran",
"lacrim", "lacrimej", "lact", "lactoniz", "lacun", "ladeir",
"ladr", "ladrej", "ladrilh", "ladrinch", "ladrinh", "ladrip",
"ladro", "ladroeir", "lagrim", "lagrimej", "laicific", "laiciz",
"laiv", "laj", "lal", "lamb", "lambar", "lambarej",
"lambarisc", "lambaz", "lambic", "lambisc", "lambit", "lambrec",
"lambreg", "lambrequinej", "lambris", "lambriz", "lambuç", "lambuj",
"lambuz", "lamech", "lameir", "lamel", "lament", "lamin",
"lamp", "lampadej", "lamparej", "lampej", "lampi", "lamuri",
"lamurinh", "lan", "lanç", "lanceol", "lancet", "lanch",
"lancin", "lancissond", "languç", "lanh", "lantejoil", "lantejoul",
"lantern", "lanzo", "lap", "laparotomiz", "lapias", "lapid",
"lapidific", "lapij", "lapis", "lapislazul", "larach", "larapi",
"larapin", "larg", "laringograf", "larp", "larunj", "larv",
"larvej", "lasc", "lass", "lastim", "lastr", "latanh",
"lateg", "latej", "lateraliz", "latic", "latin", "lating",
"latiniz", "latitudin", "latrocin", "laudaniz", "lav", "lavaj",
"lavarej", "lavarisc", "lavasc", "laveg", "lavoir", "lavor",
"lavour", "lavr", "lavuj", "lax", "lazar", "lazeir",
"lazer", "leald", "leccion", "lecion", "legaliz",
"legend", "legifer", "legisl", "legitim", "legr", "leiaut",
"leig", "leilo", "leir", "leit", "leix", "lematiz",
"lemb", "lembr", "leng", "lengaleng", "lenh", "lenhific",
"lenific", "leniment", "lent", "lentej", "lentejoil", "lentejoul",
"leri", "les", "lesion", "lesm", "lest", "letargi",
"letific", "letr", "letriz", "lev", "levant", "leved",
"levi", "levig", "levit", "lexicaliz", "lexicograf", "li",
"lib", "libel", "liber", "liberaliz", "libert", "libr",
"librin", "licenci", "licit", "lid", "lidair", "lider",
"lidim", "lig", "lignific", "ligul", "lilliputianiz", "lim",
"limit", "limp", "linc", "linch", "lind", "lineariz",
"linf", "linfangiograf", "linfatiz", "linfoangiograf", "linfolis", "ling",
"lingot", "linguaj", "linguar", "linguarej", "linific", "liniment",
"linotip", "liofiliz", "lipoaspir", "lipolis", "liquenific", "liquid",
"liquidific", "liric", "liriciz", "lisboetiz", "lisc", "lism",
"lisogeniz", "lisonj", "list", "listr", "lit", "literariz",
"literat", "literatej", "literatiz", "literatur", "litic", "litific",
"litig", "litigi", "litograf", "livel", "livid", "livr",
"lix", "lixivi", "liz", "lo", "lobol", "lobotomiz",
"lobreg", "lobreguej", "lobrig", "loc", "localiz", "locion",
"locucion", "locuplet", "logic", "logiciz", "logorr",
"logr", "loir", "loirej", "lois", "lombaliz", "lombariz",
"lombi", "lombilh", "lombrig", "lorc", "lord", "lorg",
"lorot", "losangol", "lot", "lotej", "louçainh", "louquej",
"lour", "lourej", "lous", "louv", "louvaminh", "luademel",
"luarej", "luariz", "lubric", "lubrific", "lucej", "lucern",
"luch", "lucid", "lucifer", "lucil", "luciol", "lucr",
"lucubr", "ludibri", "luf", "lug", "lumi", "lumpesin",
"lungul", "lunul", "lup", "lupan", "lupuliz", "lur",
"lusc", "luscofusc", "lusific", "lusitaniz", "lustr", "lut",
"luteiniz", "lux", "luxuri", "maç", "macac", "macadamiz",
"macambuzi", "maçanet", "maçaroc", "macarron", "macavenc", "macer",
"macet", "machad", "machi", "machic", "machin", "machuc",
"maci", "maconh", "maçoniz", "macrodimension", "macroequip", "macrograf",
"macrossegment", "macuc", "macul", "macumb", "madefic", "madeir",
"madeiriz", "madeix", "madgi", "madoniz", "madorn", "madorr",
"madraç", "madraceir", "madrigaliz", "madrinh", "madrug", "madur",
"mafi", "magi", "magic", "magin", "magnetiz", "magnific",
"mago", "magulh", "mai", "maiusculiz", "major", "mal",
"malabariz", "malacachet", "malandr", "malax", "malbarat", "malconfi",
"mald", "maldiço", "maleabiliz", "malefici", "maleiniz", "malfad",
"malform", "malgast", "malgovern", "malh", "malhet", "malici",
"malign", "maligniz", "malin", "malingu", "maloc", "malogr",
"malpar", "malquist", "malsegur", "malsin", "malt", "maltos",
"maltrabalh", "maltrat", "maluc", "malvers", "malvez", "mam",
"mamaen", "mambemb", "mamp", "mamparr", "mamuj", "man",
"manc", "manch", "mancip", "mancolit", "mancomun", "mancorn",
"mand", "manding", "mandjol", "mandri", "mandrian", "mandril",
"mandrion", "manduc", "manduqui", "maneir", "manej", "mang",
"mangab", "mangol", "mangon", "mangonh", "mangr", "manguinh",
"manheir", "manhent", "manhos", "maniat", "maniet", "manifest",
"maniganci", "manilh", "maninh", "manipul", "manisfest",
"manivel", "manj", "manjeric", "manobr", "manoc", "manquej",
"manquit", "manquitol", "manualiz", "manufact", "manufactur", "manufat",
"manufatur", "manzanz", "maometaniz", "mapi", "maqui", "maquiaveliz",
"maquil", "maquilh", "maquin", "maquiniz", "mar", "maraf",
"marambai", "marangol", "maranh", "marasm", "maratiz", "maravilh",
"marc", "marcej", "marceneir", "march", "marchet", "marcializ",
"marechaliz", "marej", "marf", "marfiniz", "marfolh", "marg",
"margin", "marginaliz", "marid", "marimb", "marin", "marinh",
"marinheir", "mariol", "maripos", "marisc", "marlot", "marm",
"marmel", "marmoriz", "maromb", "marques", "marr", "marralh",
"marranch", "marret", "marric", "marro", "marroquin", "marruc",
"marteir", "martel", "martelej", "martilh", "martiriz", "maruj",
"marulh", "masc", "mascab", "mascar", "mascarr", "mascav",
"mascot", "masculiniz", "mass", "massacr", "massaj", "massific",
"masteriz", "mastig", "mastuc", "masturb", "mat", "matabich",
"matalot", "matej", "matematiz", "materializ", "matic", "matilh",
"matin", "matiz", "matrac", "matracolej", "matraquej", "matreir",
"matri", "matricul", "matrimoni", "matriz", "matruc", "matur",
"maturrang", "matut", "maximaliz", "maximiz", "maxix", "mazanz",
"mazel", "mazurc", "mealh", "meandr", "mecaniz", "mecanograf",
"mech", "medalh", "medianiz", "mediatiz", "medic", "medicament",
"medicin", "medievaliz", "mediocriz", "medit", "medr", "mefitiz",
"meig", "meirinh", "mel", "melancoliz", "melaniz", "melhor",
"melific", "melifluent", "melindr", "melingr", "melodi", "melodiz",
"melodram", "melodramatiz", "melotipograf", "memor", "memori", "memoriz",
"men", "mencion", "mendic", "mendig", "meng", "menoriz",
"menoscab", "menospreç", "menospreci", "menosprez", "mensaliz", "menstru",
"mensur", "ment", "mentaliz", "mentir", "mentirol", "merc",
"mercadej", "mercadiz", "mercanci", "mercantiliz", "merceriz", "mercurializ",
"mercurific", "merej", "merend", "merenduc", "meretrici", "merg",
"mergulh", "merm", "meruj", "merunh", "mesc", "mescab",
"mescl", "mesmeriz", "mesquinh", "mess", "messianiz", "mestiç",
"mesticiz", "mesur", "metaboliz", "metacomunic", "metaconglomer", "metafisic",
"metaforiz", "metagraf", "metalific", "metaliz", "metameriz", "metamorfiz",
"metassediment", "metast", "metatetiz", "metatiz", "meteoriz", "metil",
"metodific", "metodiz", "metr", "metralh", "metrific", "metrocistograf",
"metronomiz", "metropoliz", "mexelh", "mexeric", "mexeruc", "mexilh",
"mezinh", "mi", "miasm", "mich", "microbi", "microcinematograf",
"microcircul", "microcoagul", "microcopi", "microdimension", "microequip", "microfilm",
"microfossiliz", "microfotograf", "micrograf", "microgranul", "micrograv", "micromanipul",
"micrometr", "micropotencializ", "microprocess", "microscop", "microsserrilh", "mict",
"midiatiz", "mieliniz", "mig", "migalh", "migmatiz", "migr",
"mij", "milici", "milimetr", "milionariz", "milit", "militariz",
"milonitiz", "mim", "mimeograf", "mimetiz", "mimic", "min",
"miner", "mineraliz", "ming", "minhoc", "mini", "miniatur",
"miniaturiz", "minific", "minimaliz", "miniminiz", "minimiz", "ministr",
"ministrific", "minor", "minoriz", "minotaur", "minotauriz", "minuci",
"minud", "minudenci", "minuscul", "minusculiz", "minut", "mir",
"miracul", "mirific", "miron", "mirr", "misc", "miscigen",
"miscr", "miser", "miserabiliz", "misgalh", "miss", "mission",
"misticiz", "mistific", "mistur", "mitific", "mitig", "mitiz",
"mitr", "mitridatiz", "mix", "mixorof", "mnemoniz", "mobil",
"mobilh", "mobiliz", "moc", "moç", "moch", "modaliz",
"model", "modeliz", "moder", "moderniz", "modic",
"modific", "modilh", "modorn", "modorr", "modul", "modulariz",
"moest", "mof", "mofumb", "mofund", "mogang", "mogic",
"moin", "moinh", "moir", "moirej", "moiriz", "moit",
"moj", "mojic", "mold", "moldur", "molec", "moleculariz",
"molej", "moleng", "molest", "molg", "molh", "molific",
"molin", "molinh", "molism", "mom", "monaquiz", "monarqui",
"monarquiz", "monc", "mond", "monetiz", "mongoliz", "monidrat",
"monilh", "monitor", "monitoriz", "monoclor", "monoculiz", "monodi",
"monograf", "monogram", "monolocul", "monolog", "monopoliz", "monossilab",
"monotip", "monotiz", "monotong", "monotoniz", "mont", "monumentaliz",
"moquec", "mor", "moraliz", "morang", "morbidiz", "morceg",
"morchet", "mordaç", "mordic", "mordisc", "mordom", "mordomiz",
"morej", "morf", "morfil", "morfin", "morfiniz", "morfodiferenci",
"morg", "moriger", "morn", "morraç", "morrinh", "morseg",
"mortalh", "mortaliz", "mortific", "mosc", "mosquit", "moss",
"mossec", "mosseg", "mostr", "mostreng", "motej", "motet",
"motin", "motiv", "motomecaniz", "motoriz", "mouc", "mouf",
"moum", "mour", "mourej", "mourisc", "mouriz", "moviment",
"mox", "moxam", "muchic", "muçulmaniz", "mud", "muem",
"muen", "mufic", "mufumb", "mugang", "mujimb", "mujin",
"mulad", "mult", "multangul", "multiangul", "multiarticul", "multicelul",
"multicit", "multigraf", "multinund", "multiplex", "multiplic", "multiprocess",
"multivari", "multivibr", "mumific", "mumiz", "mundan", "mundaniz",
"mundi", "mundializ", "mundific", "mungang", "munhec", "munici",
"municion", "municipaliz", "mupic", "mupuc", "muquinh", "mur",
"muralh", "murch", "murjin", "murmulh", "murmur", "murmurej",
"murmuri", "murmurinh", "muruj", "musealiz", "musg", "music",
"musicaliz", "mussit", "mutarrot", "mutil", "mutr", "mutreit",
"mutret", "mutu", "muxic", "muxim", "muxox", "muxurund",
"nacar", "nacariz", "nacionaliz", "nad", "nadific", "nag",
"naip", "namor", "namoric", "namorich", "namorisc", "nan",
"narcis", "narcotiz", "narr", "nasal", "nasaliz", "nastr",
"natur", "naturaliz", "naufrag", "navalh", "naveg", "navel",
"nazific", "neblin", "nebrin", "nebuliz", "necessit", "necrologi",
"necropsi", "necros", "necrotiz", "nectariz", "nefanalis", "nefeliniz",
"neg", "negaç", "negativ", "negativiz", "neglicenci", "negligenci",
"negoci", "negraliz", "negrej", "negrit", "nenec", "neocoloniz",
"neoform", "neologism", "neologiz", "nervosiz", "nesg", "neurasteniz",
"neuros", "neurotiz", "neutraliz", "nev", "nevisc", "nevo",
"nevoaç", "nevoent", "nic", "nich", "nicotin", "nicotiz",
"nict", "nid", "nidific", "nigel", "niilific", "nili",
"nimb", "nin", "ninain", "ninh", "niponiz", "niquel",
"niqueliz", "niquil", "nirvan", "nirvaniz", "nitidific", "nitidiz",
"nitr", "nitragin", "nitrat", "nitret", "nitrific", "nitroderiv",
"nitros", "nitrur", "nivel", "nobeliz", "nobilific", "nobilit",
"nobiliz", "noc", "noctambul", "noctivag", "nod", "nodo",
"noiv", "noj", "nomadiz", "nomencl", "nomenclatur", "nomin",
"nominaliz", "normaliz", "normatiz", "nosograf", "nostalgiz", "not",
"notabiliz", "notambul", "notici", "notific", "noturniz", "nov",
"novel", "novici", "nuanç", "nubel", "nubl", "nucleariz",
"nulific", "numer", "nunci", "nupci", "nut", "nutrific",
"nuverinh", "obcec", "obdur", "ober", "obfirm", "objeccion",
"objecion", "objectific", "objectiv", "objetific", "objetiv",
"objurg", "oblat", "obliter", "obnubil", "obnunci", "obr",
"obrad", "obrej", "obrig", "obsceniz", "obscurantiz", "obsecr",
"obsed", "obsedi", "obsequi", "observ", "obsession", "obsidi",
"obsolet", "obst", "obstacul", "obstaculiz", "obstin", "obstinarporfi",
"obstip", "obtemper", "obtest", "obtur", "obumbr", "obvi",
"oc", "ocasion", "ocidentaliz", "octan", "octangul", "octuplic",
"ocult", "ocup", "ocurs", "odor", "odoriz", "ofeg",
"oferend", "ofert", "ofici", "oficializ", "ofusc", "og",
"ogiv", "oig", "oir", "oirej", "oiriç", "oitav",
"ojeriz", "olead", "oleagin", "olfact", "olfat", "olh",
"oligarquiz", "oliv", "olivel", "oloriz", "olvid", "omin",
"onaniz", "ond", "ondej", "ondo", "ondul", "oner",
"onzen", "onzeneir", "op", "opac", "opacific", "opalin",
"opaliniz", "opaliz", "oper", "operacionaliz", "opi", "opil",
"opin", "oportuniz", "opt", "optimiz", "opugn", "opulent",
"opuscul", "or", "oracul", "oraculiz", "oraliz", "orbit",
"orç", "orçament", "orden", "ordenh", "ordinh", "orelh",
"orfan", "organific", "organiz", "organoform", "organofosfor", "organsin",
"orgi", "orgulh", "orient", "orientaliz", "origin", "originaliz",
"orl", "orladur", "orn", "ornament", "ornej", "orograf",
"orquestr", "orquiectomiz", "ortig", "ortocromatiz", "ortogonaliz", "ortograf",
"ortonormaliz", "ortopantomograf", "ortorretific", "orvalh", "oscil", "oscit",
"oscul", "osleriz", "osseaniz", "ossianiz", "ossific", "ostent",
"ostraciz", "oteeniz", "oteniz", "otimiz", "oug", "our",
"ourej", "ourel", "ouriç", "ourin", "ous", "ousi",
"out", "outav", "outon", "outorg", "ov", "ovacion",
"oval", "ovaliz", "ovul", "oxid", "oxigen", "oxigeniz",
"oxitoniz", "ozon", "ozonific", "ozoniz", "pabul", "pacej",
"pachol", "pacient", "pacific", "pact", "pactu", "padej",
"padr", "padrinh", "padroniz", "paduc", "pag", "pagai",
"paganiz", "pagin", "pair", "paiv", "paladin", "palanc",
"palataliz", "palatiz", "palavr", "palavrej", "palecion", "palej",
"paleograf", "palerm", "palestr", "palet", "palh", "palhet",
"palhiç", "pali", "palidej", "palific", "palit", "palm",
"palmato", "palmatori", "palmej", "palmilh", "palom", "palomb",
"palor", "palorej", "palp", "palpabiliz", "palpebrej", "palpit",
"palr", "pan", "panaç", "pand", "pandeg", "pandemiz",
"pandicul", "pandilh", "pandulh", "panegiric", "panegiriz", "panej",
"panflet", "pang", "pangai", "panific", "panoramiz", "panri",
"pansianiz", "pantaniz", "pantograf", "pantomim", "pantomin", "panzu",
"pap", "papagai", "papaguç", "papaguej", "paparic", "paparrot",
"papej", "papelon", "papi", "papoc", "papoil", "papouc",
"papoul", "papuj", "paqueir", "paquer", "par", "parabeniz",
"paraboliz", "paradigmatiz", "paradox", "parafin", "parafiniz", "parafus",
"paragon", "paragraf", "paraleliz", "paralis", "paralitic", "paralogiz",
"parament", "parametriz", "paramilitariz", "paramuscul", "parangon", "paraninf",
"parapeit", "parasit", "parcel", "parcelariz", "parcialid", "parcializ",
"pard", "pardal", "pardalit", "pardej", "parej", "parelh",
"parent", "parentes", "parentesi", "paresi", "parific", "parisianiz",
"parl", "parlament", "parlamentariz", "parlamentiz", "parlend", "parleng",
"parodi", "parol", "paroqui", "parouvel", "paroxitoniz", "parr",
"parraf", "parranç", "parron", "partej", "particion", "particip",
"particulariz", "partidariz", "partilh", "parturej", "parvoeir", "parvoej",
"pascent", "pasco", "pasigraf", "pasm", "paspalh", "pasquin",
"pass", "passaj", "passaman", "passarinh", "passarit", "passej",
"passinh", "passiv", "past", "pastej", "pasteuriz", "pastich",
"pastilh", "pastinh", "pastor", "pastoraliz", "pastorej", "pastoriz",
"patanh", "patarat", "pateg", "patej", "patent", "patentiz",
"patet", "pati", "patin", "patinh", "patoil", "patol",
"patranh", "patriz", "patro", "patrocin", "patrol", "patroniz",
"patrulh", "patusc", "paul", "paulific", "paulistaniz", "pauperiz",
"paurometaboliz", "paus", "paut", "pavej", "paves", "paviment",
"paxaliz", "peal", "pec", "pech", "pechinch", "pechiring",
"peçonhent", "peculiariz", "pedagogiz", "pedal", "pedalej", "pedantiz",
"pedestaliz", "pedic", "pedicel", "pediculiz", "pediment", "pedinch",
"pedinh", "pedint", "pediplan", "pedr", "peg", "peganh",
"peguilh", "peguinh", "pegunt", "peid", "peidorr", "pein",
"peit", "peitalh", "peix", "pej", "pejor", "pel",
"pelech", "pelej", "peletiz", "pelich", "pelid", "pelintr",
"pelletiz", "peloriz", "pelotiz", "pember", "pembr", "pen",
"penach", "penaliz", "pendenci", "pendo", "pendor", "pendul",
"pendur", "penduric", "peneir", "penej", "peneplaniz", "penetr",
"penhor", "penic", "penific", "penisc", "penitenci", "pens",
"pension", "pentelh", "penuj", "penumbr", "pepin",
"peptiz", "peptoniz", "pequilh", "pequinh", "peragr", "peralt",
"peralvilh", "perambul", "perc", "percalç", "percepcion", "perceptualiz",
"percevej", "perch", "percint", "percol", "percorug", "perderiv",
"perdig", "perdigot", "perdo", "perdulari", "perdur", "peregrin",
"pereniz", "pererec", "perfectibiliz", "perfeicion", "perfeiço", "perfi",
"perfil", "perfilh", "perfoli", "perform", "perfum", "perfur",
"pergaminh", "pergunt", "perici", "periclit", "perig", "periodiz",
"peripatetiz", "periquit", "perit", "peritoniz", "perjur", "perl",
"perlabor", "perlav", "perlej", "perleng", "perlong", "perlustr",
"permanganiz", "permeabiliz", "permineraliz", "permud", "permut", "pernej",
"pernoit", "pernosticiz", "pernout", "perol", "peroliz", "peror",
"peroxid", "perpass", "perpetr", "perpetu", "perreng", "perscrut",
"persever", "persign", "persilh", "personaliz", "personific", "persoterr",
"perspectiv", "perspetiv", "perspir", "pertent", "pertuch", "perturb",
"peru", "pervag", "pervoeir", "pes", "pesc", "pesg",
"pespeg", "pespont", "pesquis", "pessoaliz", "pestanej", "pestifer",
"pestilenci", "pestinh", "pet", "petanisc", "petard", "petarol",
"petec", "peteg", "peticion", "petilh", "petisc", "petit",
"petrarquiz", "petrech", "petrific", "petroliz", "pexot", "pezunh",
"pfuc", "pi", "piab", "piach", "piaf", "pial",
"pialh", "pianç", "pianiz", "pianol", "pic", "picardi",
"picaret", "piçarr", "pich", "piciric", "picot", "picu",
"picuinh", "pielograf", "pif", "pig", "pigarr", "pigment",
"pil", "pilch", "pild", "pildr", "pilh", "pilheri",
"pilot", "pilul", "pimb", "pimp", "pimpl", "pimpolh",
"pimpon", "pin", "pinç", "pincel", "pinch", "pincharol",
"pindariz", "pindoc", "pindong", "pindr", "pinel", "ping",
"pingalh", "pingent", "pingol", "pinic", "pinimb", "pinoc",
"pinot", "pint", "pintainh", "pintal", "pintalg", "pintalh",
"pintarol", "pintipar", "pinturil", "piolh", "pior", "pip",
"piparot", "pipet", "pipi", "pipil", "pipin", "pipiric",
"pipit", "pipoc", "piquet", "pir", "piramb", "piramid",
"piramidiz", "pirang", "pirat", "pirilamp", "pirilampej", "piriric",
"piroc", "pirograf", "pirograv", "pirolis", "pirotecniz", "pirraç",
"pirroniz", "piruet", "pis", "pisc", "piscainh", "piscalh",
"pisg", "piso", "pissit", "pist", "pit", "piteir",
"pitosg", "piuc", "pivot", "pixer", "placid", "placit",
"plagi", "plain", "plan", "planch", "planej", "planetiz",
"planific", "planilh", "planimetr", "planiz", "planopolariz", "plant",
"plasm", "plasmolis", "plastic", "plasticiz", "plastific", "platin",
"platoniz", "plebiscit", "plebiz", "pleit", "plenific", "pletor",
"pletoriz", "plic", "plid", "pliss", "plot", "plug",
"plum", "plumbagin", "plumej", "pluraliz", "pluricelul", "pluriestratific",
"plurific", "plurissecul", "pluristratific", "plutarquiz", "plutocratiz", "pneumencefalograf",
"pneumoencefalograf", "poc", "pod", "podometr", "poeir", "poematiz",
"poent", "poet", "poetific", "poetiz", "pois", "poit",
"poj", "polariz", "polc", "poleg", "polemic", "polemiz",
"polici", "policit", "policopi", "policrom", "policromiz", "poligraf",
"poliment", "polimeriz", "poliniz", "polirramific", "politic", "politip",
"politiz", "politon", "poloniz", "polp", "polvariz", "polvilh",
"polvoriz", "pombeir", "pomej", "pomific", "pomiz", "pompoariz",
"ponç", "ponder", "pong", "ponh", "pontalet",
"ponticul", "pontific", "pontilh", "pontinh", "ponto", "pontu",
"pontualiz", "popi", "popoc", "popul", "populariz", "porej",
"porfi", "porfiriz", "pormenoriz", "pornograf", "pororoc", "port",
"portaj", "portuch", "portugaliz", "portugues", "porvisc", "pos",
"posfaci", "posicion", "positiv", "pospont", "poss", "possession",
"possibilit", "possibiliz", "post", "posteg", "postej", "postem",
"posterg", "posteriz", "postiç", "postil", "postul", "postur",
"potenci", "potencializ", "potoc", "poup", "pous", "pousent",
"pout", "povo", "pracat", "pracej", "pradej", "prag",
"pragalh", "pragmatiz", "praguej", "pranch", "prant", "prasm",
"pratej", "pratic", "prax", "preadapt", "preadivinh", "prealeg",
"pream", "preambul", "preanunci", "prebend", "precalç", "precant",
"precariz", "precat", "precaucion", "precautel", "preceit", "preceitu",
"precific", "precint", "precipit", "precis", "precogit", "preconceitu",
"precondicion", "preconiz", "predat", "predemarc", "predestin", "predetermin",
"predic", "predicament", "predilecion", "predit", "predomin", "preemin",
"preestipul", "prefabric", "prefaci", "preferenci", "prefigur", "prefiltr",
"prefix", "prefoli", "prefulgur", "preg", "pregalh", "pregn",
"prego", "pregost", "preguiç", "pregunt", "pregust", "preit",
"preiteg", "preitej", "prejudic", "prelaci", "prelazi", "prelecion",
"prelev", "preli", "prelib", "preludi", "prem", "premasteriz",
"prematur", "premedit", "premi", "premud", "prend", "prenh",
"prenomin", "prenot", "prens", "prenunci", "prenunic", "preocup",
"preopin", "preorden", "prepar", "preponder", "preposicion", "preposter",
"preprocess", "prequestion", "pres", "presagi", "prescrut", "presenci",
"present", "presentific", "preserv", "presidi", "presig", "presilh",
"pressagi", "presselecion", "pression", "pressuriz", "prest", "prestacion",
"prestidigit", "prestigi", "presum", "pretej", "preternaturaliz", "pretext",
"pretorianiz", "pretraç", "prevaric", "prez", "prim", "primaver",
"primaveriz", "primitiv", "primor", "principi", "prior", "prioriz",
"prisc", "prism", "prismatiz", "pristin", "priv", "privatiz",
"privilegi", "pro", "probabiliz", "problematiz", "procedimentaliz", "process",
"procession", "proclam", "proclin", "procrastin", "procri", "procur",
"procurv", "prodig", "prodigaliz", "proej", "proemi", "proemin",
"profaç", "profalç", "profan", "profaniz", "profess", "professor",
"profet", "profetiz", "profil", "profissionaliz", "proflig", "profund",
"progener", "prognostic", "program", "programatiz", "progress", "projeit",
"projet", "prolat", "proletariz", "prolifer", "prolific", "prolog",
"prolong", "proman", "promand", "promulg", "pronomi", "pronomin",
"pronominaliz", "pronostic", "prontific", "prontuari", "pronunci", "propag",
"propal", "proparoxitoniz", "propici", "propin", "propolis", "propoliz",
"proporcion", "proporcionaliz", "proposit", "propugn", "propuls", "propulsion",
"prorrog", "pros", "prosific", "prospect", "prosper", "prospet",
"prostern", "prostr", "protagoniz", "protecion", "protel", "protest",
"protestantiz", "protocol", "protocoliz", "proton", "prototific", "protuber",
"prov", "provençaliz", "proverbi", "proverbializ", "providenci", "provincializ",
"provincianiz", "provisc", "provision", "provoc", "proz", "prudenci",
"prum", "prussianiz", "pseudojustific", "pseudotrabalh", "psicanalis", "psicoanalis",
"psicograf", "psicolog", "psicologiz", "pu", "pub", "public",
"publicit", "publiciz", "pudl", "pueriliz", "pugil", "pugn",
"puit", "puj", "pul", "pulh", "pulic", "pulinh",
"puls", "pulsion", "pult", "pulul", "pulveriz", "pulvin",
"punç", "puncej", "puncion", "punço", "punt", "pup",
"pupil", "pupiniz", "purg", "purific", "purpur", "purpurej",
"purpuriz", "putrific", "pux", "quadr", "quadricul", "quadrilh",
"quadrilob", "quadriniz", "quadruped", "quadrupl", "quadruplic", "qualific",
"quantific", "quantiz", "quar", "quarenten", "quaresm", "quart",
"quartej", "quartilh", "quass", "quaterniz", "quebr", "quebrant",
"qued", "queij", "queim", "quein", "queix", "quel",
"quenh", "quent", "queratiniz", "querel", "queren", "querenci",
"querend", "querosen", "question", "questiuncul", "quevind", "quezil",
"quiband", "quic", "quiet", "quilat", "quilh", "quilific",
"quilometr", "quilot", "quilunz", "quimb", "quimeriz", "quimific",
"quimiossintetiz", "quin", "quinch", "quing", "quinho",
"quintav", "quintessenci", "quintuplic", "quirograf", "quit", "quitand",
"quitiniz", "quixotiz", "quizil", "quocient", "quot", "quotidianiz",
"quotiz", "raban", "rabeir", "rabej", "rabi", "rabid",
"rabiniz", "rabios", "rabisc", "rabon", "rabot", "rabuj",
"rabul", "rabulej", "rabun", "rabunh", "rabusc", "racemiz",
"rach", "racin", "raciocin", "racion", "racionaliz", "raço",
"racont", "radi", "radiativ", "radic", "radicaliz", "radici",
"radioativ", "radiobaliz", "radiocromatograf", "radiodetect", "radiodiagnostic", "radiofoniz",
"radiograf", "radiolocaliz", "radiomarc", "radionaveg", "radiopreg", "radiossond",
"radiotelegraf", "radiotelevis", "radiotransistoriz", "radob", "radub", "raf",
"raff", "rafi", "rafin", "rai", "raiv", "raivej",
"raj", "ral", "ralent", "ralh", "ram", "ramalh",
"ramalhet", "ramang", "ramel", "ramific", "ramilhet", "ramoc",
"rampan", "ranç", "ranceon", "rancidific", "rancific", "rancorej",
"rancur", "randomiz", "ranf", "rang", "ranguinh", "ranh",
"ranhur", "ranhuz", "ranilh", "ranzinz", "rap", "rapel",
"rapi", "rapig", "rapilh", "rapin", "rapinh", "rapos",
"raposinh", "rapt", "raquitiz", "rar", "rarej", "ras",
"rasc", "rascanh", "rascunh", "rasg", "rasmolg", "rasoir",
"rasour", "rasp", "raspanç", "raspinh", "rastaquer", "rasteir",
"rastej", "rastel", "rasteriz", "rastilh", "rastinh", "rastolh",
"rastrej", "rasur", "rat", "ratazan", "ratific", "ratin",
"ratinh", "raton", "raus", "ravenh", "ravin", "ravinh",
"razi", "razo", "razon", "reabilit", "reabit", "reabitu",
"reabjur", "reaboto", "reabraç", "reabrasileir", "reabrolh", "reabrot",
"reaceit", "reach", "reacion", "reacomod", "reacompanh", "reacopl",
"reacord", "reacostum", "reacumul", "reacus", "readapt", "readjudic",
"readmir", "readot", "reafet", "reafirm", "reafix", "reafret",
"reagend", "reaglomer", "reagrav", "reagrup", "reajunt", "reajust",
"realarg", "realç", "realegr", "realej", "realent", "realiment",
"realinh", "realist", "realiz", "realoc", "realoj", "realug",
"realumi", "ream", "reambul", "reanalis", "reand", "reanex",
"reanim", "reanimaliz", "reanod", "reapaixon", "reaparelh", "reapert",
"reaplic", "reapoder", "reapont", "reaportugues", "reaposs", "reapreci",
"reapresent", "reaprision", "reapropri", "reaprov", "reaproveit", "reaprovision",
"reaproxim", "reaquist", "rearboriz", "rearm", "rearmazen", "rearquitet",
"rearranj", "rearrum", "rearticul", "reasfalt", "reassegur", "reassent",
"reassimil", "reassin", "reassoci", "reassoldad", "reassom", "reassopr",
"reat", "reatent", "reaterr", "reatest", "reatiç", "reativ",
"reatoch", "reatoriz", "reatravess", "reatualiz", "reatulh", "reaument",
"reautentic", "reautoriz", "reautu", "reavali", "reavi", "reavigor",
"reavis", "reavist", "reaviv", "reavult", "reb", "rebaix",
"rebalanç", "rebals", "reban", "rebanh", "rebarb", "rebarbariz",
"rebat", "rebatinh", "rebatiz", "rebeij", "rebel", "rebelic",
"rebelion", "rebent", "reberr", "rebesbelh", "rebic", "rebimb",
"rebimbalh", "rebiopsi", "rebisc", "rebit", "rebitol", "rebo",
"rebobin", "reboc", "reboj", "rebol", "rebolc", "reboliç",
"rebomb", "rebor", "reborbulh", "rebord", "reborqui", "rebot",
"rebouç", "rebrad", "rebram", "rebrasil", "rebrilh", "rebrot",
"rebuç", "rebuf", "rebulh", "rebuliç", "rebumb", "rebusc",
"rebusn", "rec", "recabed", "recach", "recad", "recadastr",
"recadej", "recal", "recalc", "recalç", "recalcific", "recalcitr",
"recalcul", "recalendariz", "recalibr", "recam", "recambi", "recanaliz",
"recandidat", "recant", "recap", "recapacit", "recapitaliz", "recapitul",
"recapt", "recaptur", "recarbonat", "recarboniz", "recarbur", "recard",
"recarg", "recarimb", "recarreg", "recartilh", "recartorializ", "recas",
"recat", "recategoriz", "recativ", "recauchut", "recaucion", "recautel",
"recav", "recavalg", "receit", "recelulariz", "recen", "recentr",
"recepcion", "recept", "recerr", "rechaç", "rechap", "rechec",
"recheg", "rechi", "rechin", "rechoch", "rechonch", "rechup",
"recib", "recicl", "recidiv", "recint", "reciproc", "recit",
"reciviliz", "reclacitr", "reclam", "reclamiz", "reclassific", "reclin",
"reclus", "recobr", "recoc", "recogit", "recoit", "recol",
"recolig", "recoloc", "recoloniz", "recolt", "recombin", "recomeç",
"recomend", "recoment", "recomod", "recompact", "recompartiment", "recompartimentaliz",
"recompens", "recompil", "recomplet", "recompr", "recompromiss", "reconceitu",
"reconcentr", "reconcert", "reconcili", "recondicion", "reconect", "reconfess",
"reconfigur", "reconfirm", "reconfort", "recongraç", "reconquist", "reconsagr",
"reconsert", "reconsider", "reconsolid", "reconsorci", "reconstitucionaliz", "reconsult",
"recont", "recontat", "recontextualiz", "recontrat", "reconvoc", "recopi",
"recopil", "recorç", "record", "recoro", "recort", "recost",
"recout", "recov", "recrav", "recredenci", "recrest", "recri",
"recrimin", "recriptograf", "recristaliz", "recristianiz", "recrucific", "recrut",
"recruz", "recu", "recud", "recuid", "recuinh", "recuit",
"recultiv", "recunh", "recuper", "recur", "recurs", "recurv",
"recus", "redactilograf", "redam", "redambal", "redardej", "redatilograf",
"redatori", "redeclar", "redecor", "redecret", "rededetiz", "rededic",
"redemocratiz", "redemoinh", "redemonstr", "redeng", "redeposit", "redescont",
"redesenh", "redesgrenh", "redesign", "redespert", "redestin", "redigit",
"redimension", "redintegr", "redirecion", "redispar", "rediviniz", "redizim",
"redob", "redobr", "redoiç", "redoir", "redomoinh", "redopi",
"redouç", "redour", "redr", "redubl", "redund", "reduplic",
"reeco", "reedific", "reedit", "reeditor", "reeduc", "reelabor",
"reembaralh", "reembarc", "reembas", "reembols", "reembrulh", "reemend",
"reemigr", "reempacot", "reempalh", "reempolg", "reemporc", "reemposs",
"reempreg", "reencadern", "reencafu", "reencaix", "reencaminh", "reencan",
"reencap", "reencarcer", "reencarn", "reencen", "reencet", "reencontr",
"reencuc", "reendereç", "reendireit", "reendoss", "reenergiz", "reenfi",
"reenfolh", "reengaj", "reengendr", "reengolip", "reenlaç", "reenovel",
"reenquadr", "reensai", "reensin", "reentel", "reenterr", "reentr",
"reentroniz", "reenvas", "reenverniz", "reenvi", "reenvid", "reenxert",
"reequilibr", "reequip", "reescal", "reescalon", "reescrach", "reescrachet",
"reescrachin", "reescrafunch", "reescramalh", "reescramel", "reescrament", "reescramouc",
"reescramuç", "reescraviz", "reescrevinh", "reescritur", "reespecific", "reesper",
"reespos", "reespum", "reestamp", "reestiliz", "reestimul", "reestipul",
"reestrutur", "reestud", "reetiquet", "reexager", "reexamin", "reexecut",
"reexit", "reexpatri", "reexpi", "reexplic", "reexplor", "reexport",
"reextradit", "refal", "refals", "refart", "refastel", "refator",
"refatur", "refech", "refecund", "refeg", "refent", "referenci",
"referend", "referment", "referr", "refert", "refestel", "refez",
"refi", "refigur", "refil", "refilh", "refili", "refilm",
"refiltr", "refin", "refinanci", "refinc", "refint", "refirm",
"refit", "refl", "reflacion", "reflexion", "reflorest", "reflutu",
"reflux", "refoc", "refocaliz", "refocil", "refog", "refoleg",
"refolg", "refolh", "refoment", "refonologiz", "reforç", "reforj",
"reform", "reformat", "reformul", "refortific", "refotograf", "refract",
"refractur", "refranj", "refrat", "refratur", "refreg", "refrequent",
"refresc", "refret", "refric", "refriger", "refris", "refrit",
"refrond", "refrulh", "refug", "refugi", "refulgur", "refum",
"refund", "refunf", "refunfumeg", "refung", "refunh", "refus",
"refust", "refut", "reg", "regaç", "regal", "regalardo",
"regalvaniz", "regambi", "reganh", "regaseific", "regat", "regatinh",
"regel", "regener", "regent", "reger", "regermin", "regiment",
"regin", "regionaliz", "regir", "regist", "registr", "rego",
"regojiz", "regolf", "regom", "regoug", "regozij", "regr",
"regraci", "regrav", "regrax", "regress", "reguard", "regueif",
"regueir", "reguinch", "reguing", "regulament", "regulariz", "regurgit",
"reidrat", "reific", "reimagin", "reimplant", "reimplement", "reimport",
"reimput", "rein", "reinaugur", "reincarn", "reincit", "reincorpor",
"reincrust", "reindex", "reindustrializ", "reinerv", "reinfecion", "reinfect",
"reinfet", "reinflam", "reingress", "reinici", "reinicializ", "reinjet",
"reinocul", "reinstal", "reinstaur", "reintegr", "reintern", "reinternacionaliz",
"reinternaliz", "reinterpret", "reinterrog", "reinum", "reinvent", "reinvid",
"reinvoc", "reioniz", "reislamiz", "reiter", "reitoriz", "reiun",
"reivindic", "reix", "rejeit", "rejubil", "rejunc", "rejunt",
"rejur", "rejustific", "rel", "relacion", "relacr", "relamp",
"relampad", "relampadej", "relampaguej", "relampej", "relampr", "relanç",
"relancet", "relarg", "relat", "relativ", "relativiz", "relav",
"relax", "releg", "relegitim", "releix", "relembr", "relent",
"relep", "relev", "relevant", "relh", "relicit", "relig",
"relim", "relinch", "reling", "reloc", "relocaliz", "reloj",
"relouc", "relum", "relumbr", "relustr", "relut", "relv",
"relvej", "rem", "remadaleniz", "remagnetiz", "remalh", "remalinh",
"remanch", "remand", "remanej", "remang", "remanipul", "remanisc",
"remans", "remanufatur", "remarc", "remarid", "remartel", "remasc",
"remasteriz", "remastig", "remat", "rematerializ", "rematricul", "rembrandtiz",
"rembraniz", "remed", "remel", "remelg", "remembr", "rememor",
"rememori", "remend", "remenic", "remergulh", "remess", "remestr",
"remigi", "remigr", "remilitariz", "remim", "remin", "remineraliz",
"reminiscenci", "remir", "remistur", "remix", "remobil", "remobiliz",
"remoc", "remoç", "remodel", "remodul", "remoel", "remoidanh",
"remoinh", "remolc", "remolg", "remolh", "remonarquiz", "remond",
"remonotoniz", "remont", "remor", "remorc", "remordic", "remordisc",
"remost", "remostr", "remouc", "remud", "remug", "remunc",
"remuner", "remunhet", "remurmur", "remurmurej", "remusg", "renacionaliz",
"renatur", "renaturaliz", "renaveg", "rend", "renderiz", "rendibiliz",
"rendilh", "renegoci", "renegrej", "renembr", "renet", "renifl",
"renivel", "renormaliz", "renov", "rent", "rentabiliz", "rentilh",
"renumer", "renunci", "renzilh", "reobrig", "reobserv", "reocup",
"reoficializ", "reolis", "reoper", "reopin", "reorden", "reorganiz",
"reorient", "reorquestr", "reoxid", "reoxigen", "reoxigeniz", "repactu",
"repag", "repaganiz", "repagin", "repair", "repanh", "repapoil",
"repapoul", "repar", "reparafus", "reparametriz", "reparcel", "reparticion",
"repartilh", "repass", "repassarit", "repast", "repatan", "repatri",
"repatrocin", "repaviment", "repeç", "repech", "reped", "repeg",
"repeit", "repel", "repenetr", "repenic", "repens", "reperfil",
"repergunt", "repertori", "repes", "repesc", "repeten", "repic",
"repilg", "repimp", "repimpin", "repimpon", "repinch", "repinhaliz",
"repinic", "repint", "repiquet", "repis", "repisc", "repivatiz",
"replain", "replanej", "replanilh", "replant", "replen", "replet",
"replic", "repois", "repolariz", "repoleg", "repolg", "repolh",
"repoliniz", "reponder", "repont", "repontu", "repopul", "repopulariz",
"report", "reposicion", "repost", "repotron", "repous", "repovo",
"repreg", "repres", "represent", "reprimend", "reprincipi", "repris",
"repristin", "repristiniz", "reprivatiz", "reprocess", "reproch", "reprofund",
"reprogram", "reprojet", "reprov", "rept", "republic", "republicaniz",
"repudi", "repugn", "repul", "repuls", "repulul", "repurg",
"repurific", "reput", "repux", "requadr", "requadruped", "requalific",
"requebr", "requeim", "requent", "request", "requestion", "requint",
"requisit", "rerradi", "rerratific", "rerrespir", "resbun", "rescald",
"rescum", "resenh", "reserv", "resfolg", "resfri", "resg",
"resgalg", "resgat", "resguard", "residenci", "resign", "resin",
"resinific", "reslumbr", "resm", "resmalh", "resmeleng", "resmolg",
"resmoninh", "resmun", "resmung", "resmunh", "resmuninh", "resmusg",
"respald", "respanç", "respeit", "respig", "resping", "respinh",
"respir", "respons", "responsabiliz", "respont", "respost", "respumej",
"resqui", "ressabi", "ressac", "ressaci", "ressaib", "ressaibi",
"ressalt", "ressaltit", "ressalv", "ressanfonin", "ressangr", "ressec",
"ressegund", "ressegur", "ressel", "resseren", "resserr", "ressic",
"ressimboliz", "ressincroniz", "ressintetiz", "resso", "ressobr", "ressocializ",
"ressold", "ressolh", "ressolt", "resson", "ressonh", "ressopr",
"ressu", "ressubtiliz", "ressud", "ressulc", "ressum", "ressumbr",
"ressupin", "ressuscit", "rest", "restamp", "restart", "restaur",
"restauric", "restel", "restil", "restir", "restit", "restiv",
"restolh", "restral", "restrel", "restrib", "restrical", "restridul",
"restrug", "restrutur", "restuc", "result", "resval", "ret",
"retabul", "retac", "retalh", "retali", "retanch", "retard",
"reteim", "retelefon", "retelegraf", "retelh", "retemper", "retent",
"reterritorializ", "retes", "retesi", "retest", "reticenci", "retific",
"retilint", "retiliz", "retintin", "retipific", "retir", "retisn",
"retoc", "retoiç", "retom", "retonh", "retoric", "retorn",
"retos", "retost", "retouc", "retouç", "retov", "retrabalh",
"retraç", "retram", "retranc", "retranç", "retranslad", "retransplant",
"retransport", "retras", "retraslad", "retrat", "retrav", "retrein",
"retrilh", "retrin", "retrinc", "retrinch", "retro", "retroaliment",
"retroativ", "retrocarreg", "retrocit", "retrocruz", "retrodat", "retrograd",
"retroilumin", "retropedal", "retrovacin", "retrovej", "retruc", "retub",
"retumb", "reumaniz", "reumatiz", "reunific", "reurbaniz", "reus",
"reutiliz", "revacin", "revalid", "revaloriz", "revasculariz", "reveget",
"revel", "revener", "reverber", "reverdej", "reverenci", "reverg",
"reverific", "revermelh", "revers", "reves", "revess", "revez",
"revibr", "reviç", "revid", "revigor", "revigorent", "revigoriz",
"revindic", "reving", "revir", "revirginiz", "revis", "revisit",
"revist", "revitaliz", "revivenci", "revivific", "revo", "revoc",
"revoej", "revolc", "revolit", "revolt", "revolucion", "revor",
"revuls", "rez", "rezing", "rezip", "rib", "ribald",
"ribeir", "ribomb", "riç", "ricerc", "ricochet", "ridic",
"ridiculariz", "ridiculiz", "rif", "rifl", "rig", "rij",
"ril", "rilh", "rim", "rimbomb", "rinç", "rinch",
"rinchal", "rinchavelh", "rincon", "rinf", "rinh", "rip",
"ripanç", "ripost", "risc", "riscanh", "ritm", "ritmiz",
"ritornel", "ritualiz", "rivaliz", "rix", "riz", "robech",
"robor", "roboriz", "robotiz", "roc", "roç", "roçag",
"rocalh", "roceg", "roci", "rocin", "rocl", "rod",
"rodej", "rodilh", "rodizi", "rodomoinh", "rodopi", "rodrig",
"rog", "roj", "rol", "rold", "rolet", "rolh",
"roliç", "rom", "romanesc", "romaniz", "romantiz", "romesent",
"ronc", "ronch", "rond", "ronfl", "ronquej", "ronron",
"ror", "rorej", "ros", "rosc", "roset", "rosmaninh",
"rosn", "rosnor", "rostilh", "rot", "rotacion", "rotativ",
"roteiriz", "rotej", "rotiniz", "rotograv", "rotul", "roub",
"rouç", "roufenh", "roup", "rouquej", "rouss", "rouxinol",
"rouxinoliz", "ru", "rubej", "rubific", "ruborej", "ruboriz",
"rubric", "rubriz", "ruç", "rudimentariz", "ruf", "rufi",
"rufl", "rug", "rugin", "rugit", "ruid", "rul",
"rulh", "rum", "rumb", "rumi", "rumin", "rumor",
"rumorej", "runf", "runh", "rup", "ruraliz", "rurbaniz",
"rusg", "russ", "russianiz", "russific", "russiz", "rustic",
"rustific", "rut", "rutil", "sabadej", "sabatin", "sabatiz",
"saberec", "sabererec", "sabich", "sabot", "sabrec", "sabuc",
"sabug", "sabuj", "sabul", "saburr", "sac", "saçaric",
"sacarific", "sacarrolh", "sach", "sachol", "saci", "sacol",
"sacolej", "sacraliz", "sacrament", "sacrific", "sacul", "saf",
"safan", "safiriz", "safrej", "sagin", "sagit", "sagorr",
"sagot", "sagr", "saibr", "salamord", "salari", "sald",
"salg", "salh", "salicil", "salient", "salific", "salin",
"saliniz", "salitr", "salitriz", "saliv", "salmej", "salmiz",
"salmodej", "salmodi", "salmoeir", "salmoir", "salmour", "salpic",
"salpiment", "salpint", "salpoeir", "salpres", "sals", "salt",
"saltarel", "saltaric", "saltarilh", "saltarinh", "saltinh", "saltit",
"salubrific", "salubriz", "saluç", "salud", "salv", "salvaguard",
"samang", "samb", "sambarc", "sambenit", "sambl", "samboc",
"sambr", "samp", "san", "sanatoriz", "sancion", "sand",
"sandej", "sanef", "sanfon", "sanfonin", "sanforiz", "sangr",
"sangrentej", "sanguej", "sanguel", "sanguent", "sanguessug", "sanguific",
"sanguinh", "sanic", "sanific", "sanitariz", "sanitiz", "sanj",
"sanquit", "sanscritiz", "santific", "santig", "sap", "sapat",
"sapec", "sapej", "saperec", "sapoil", "saponific", "sapori",
"sapoul", "saprem", "saquel", "sar", "sarafusc", "saraiv",
"sarandilh", "sarangonh", "sarapant", "sarapic", "sarapint", "saraqueim",
"saraquit", "sard", "sardanapaliz", "sardoniz", "sarilh", "sarj",
"sarn", "sarp", "sarr", "sarrabisc", "sarraf", "sarrafaç",
"sarrazin", "sart", "sass", "satani", "sataniz", "satelitiz",
"sateliz", "satiriz", "satisd", "satur", "saussuritiz", "saxific",
"sazo", "sazon", "schilleriz", "sebenisc", "sec", "seccion",
"secession", "seci", "secion", "seculariz",
"secund", "secundariz", "securitiz", "sed", "sedentariz", "sedi",
"sediment", "seg", "segment", "segreg",
"segur", "seit", "seiv", "sel", "selecion", "selet",
"selvatic", "semantiz", "semb", "sembr", "semelh", "sement",
"semiadapt", "semiafirm", "semiajust", "semianimaliz", "semiaplic", "semiaprimor",
"semiaprov", "semibenefici", "semicadastr", "semicapacit", "semicarboniz", "semicaricatur",
"semicerr", "semicol", "semiconect", "semidespert", "semidevor", "semidiscretiz",
"semidiviniz", "semidobr", "semidomestic", "semienergiz", "semiequip", "semifech",
"semiformat", "semilevant", "semimanufatur", "seminorm", "semiofert", "semiograf",
"semipatrocin", "semipint", "semipoliniz", "semitiz", "semito", "semiton",
"semivo", "semostr", "seng", "senhoriz", "seniliz", "sensaboriz",
"sensibiliz", "sensific", "sensualiz", "sent", "sentenci", "sentimentaliz",
"sentinel", "separ", "sept", "septuplic", "sepulcr", "sepult",
"sequel", "sequenci", "sequestr", "sequi", "serafiz", "seramang",
"serand", "seranz", "seren", "sereniz", "seri", "serigait",
"serigraf", "sering", "seringonh", "seriograf", "sermo", "sermon",
"sero", "serpej", "serpent", "serpentiniz", "serr", "serralh",
"serramanc", "serrazin", "serrilh", "serrin", "serrot", "sertanej",
"sertaniz", "serviliz", "sesm", "sess", "session", "sestr",
"set", "seteir", "setin", "setoriz", "setuplic", "sev",
"sevandij", "severiz", "sevici", "sex", "sextav", "sextuplic",
"sexu", "sexualiz", "sezo", "si", "sibarit", "sibil",
"sibilantiz", "sibit", "sider", "sideriz", "sifiliz", "sifl",
"sifon", "sigil", "sigl", "sigliz", "siglonimiz", "sign",
"signific", "silab", "silabific", "silenci", "silhuet", "silicat",
"silicatiz", "silicific", "silicon", "silogism", "silogistic", "silogiz",
"silv", "simbol", "simboliz", "simetriz", "simorfil", "simpatiz",
"simplific", "simul", "sin", "sinal", "sinalej", "sinaliz",
"sinapiz", "sincel", "sinceriz", "sincop", "sincopiz", "sincretiz",
"sincroniz", "sindic", "sindicaliz", "sindicat", "sinet", "sinfoniz",
"singr", "singulariz", "singult", "sinistr", "sinistriz", "siniz",
"sinonimiz", "sinoptiz", "sintec", "sinteriz", "sintetiz", "sintomatiz",
"sintoniz", "sinu", "sinuc", "sinuos", "siren", "sirg",
"sirigait", "siriric", "sis", "sism", "sistem", "sistematiz",
"sistol", "siti", "situ", "snif", "so", "soalh",
"soass", "sobalç", "sobej", "soberaniz", "soberb", "sobescav",
"sobest", "sobi", "sobneg", "soborralh", "sobr", "sobraç",
"sobrad", "sobranç", "sobras", "sobreabund", "sobreagit", "sobrealcooliz",
"sobrealcunh", "sobrealiment", "sobreavali", "sobreavis", "sobreazed", "sobrecab",
"sobrecapitaliz", "sobrecasac", "sobrecifr", "sobrecompens", "sobrecostur", "sobredoir",
"sobredour", "sobrefatur", "sobreger", "sobregiz", "sobreirrit", "sobreleit",
"sobrelev", "sobrelot", "sobrelustr", "sobremaravilh", "sobremont", "sobrenad",
"sobrenaturaliz", "sobreolh", "sobrepair", "sobrepartilh", "sobrepass", "sobrepens",
"sobrepes", "sobrepovo", "sobrepuj", "sobrerrest", "sobrerrold", "sobrerrond",
"sobrescrit", "sobresper", "sobressalt", "sobressar", "sobressatur", "sobressel",
"sobressol", "sobrestim", "sobretax", "sobretro", "sobreus", "sobrevaloriz",
"sobrevigi", "sobrevo", "soc", "socalc", "socav", "sochi",
"sociabiliz", "socializ", "socioculturaliz", "soçobr", "soçoc", "socorn",
"socratiz", "socrest", "sod", "sodomiz", "sofism", "sofistic",
"sofor", "sofrald", "sofreg", "sofren", "sog", "sogr",
"sojig", "sojorn", "sojug", "sol", "solaç", "solanc",
"solap", "solariz", "solavanc", "solaz", "sold", "solec",
"soleciz", "soleniz", "soletr", "solev", "solevant", "solf",
"solfej", "solh", "solicit", "solid", "solidariz", "solidific",
"soliloqui", "solinh", "solmiz", "solt", "solubiliz", "soluç",
"solucion", "som", "somatiz", "sombr", "sombrej", "somergulh",
"somitic", "somp", "son", "sonambul", "sond", "soneg",
"sonet", "sonh", "sonoriz", "sopap", "sopeg", "sopes",
"sopi", "sopit", "sopont", "soporiz", "sopr", "sopraniz",
"sopres", "soprez", "sopux", "soqueix", "sor", "sorbic",
"sorn", "soroneutraliz", "sororoc", "sorovacin", "sorrab", "sorrasc",
"sorren", "sorv", "soslai", "sospit", "sosseg", "sossobr",
"sosteniz", "sostrej", "sot", "sotalh", "sotavent", "soterr",
"sotranc", "soturn", "soturniz", "soupic", "sout", "sov",
"sovel", "sovietiz", "sovin", "sprint", "su", "suaviz",
"subadapt", "subafirm", "subaglomer", "subajust", "subaliment", "subaltern",
"subalterniz", "subaplic", "subaprimor", "subaprov", "subaproveit", "subarrend",
"subassin", "subast", "subavali", "subcandidat", "subcapacit", "subcaricatur",
"subclassific", "subconcession", "subconscientiz", "subcontrat", "subdelir", "subderiv",
"subdimension", "subelisc", "subemend", "subempraz", "subempreit", "subenfiteutic",
"subenxert", "subequip", "suberific", "suberiniz", "suberiz", "subespecific",
"subestim", "subexamin", "subfatur", "subfinanci", "subformat", "subfret",
"subgrup", "subinstal", "subintitul", "subjectiv", "subjeit", "subjetiv",
"subjetiviz", "subjug", "sublamel", "sublev", "sublevant", "sublicenci",
"sublim", "sublimiz", "sublinh", "subloc", "subminiaturiz", "subministr",
"subnivel", "subofert", "suborbit", "subordin", "suborn", "subparcel",
"subpatrocin", "subplanilh", "subpuj", "subscrit", "subsecretari", "subsequenci",
"subsidi", "subsol", "substanci", "substancializ", "substantific", "substantiv",
"subsult", "subterr", "subtiliz", "subtitul", "subtotaliz", "subtrat",
"suburbaniz", "subutiliz", "subvaloriz", "subvencion", "subverbet", "suc",
"sucat", "suci", "sud", "sudoest", "sudorific", "suest",
"sufix", "sufl", "sufoc", "sufrag", "sufumig", "sug",
"sugestion", "sugil", "suicid", "suj", "sujeit", "sujic",
"sujig", "sujug", "sul", "sulanc", "sulavent", "sulc",
"sulfat", "sulfatiz", "sulfet", "sulficarbonat", "sulfit", "sulfon",
"sulfur", "sulfuret", "sulfuriz", "sumagr", "sumari", "sumariz",
"sumbr", "sumptualiz", "sumul", "sung", "sunguil", "suntualiz",
"supedit", "supenhor", "super", "superabund", "superadapt", "superafirm",
"superaglomer", "superajunt", "superajust", "superaliment", "superaplic", "superaprimor",
"superaprov", "superativ", "superauxili", "superavali", "superbenefici", "supercalandr",
"supercalcin", "supercandidat", "supercapacit", "supercapitaliz", "supercaricatur", "superciviliz",
"supercompact", "supercompens", "supercomplex", "superconcentr", "superconfort", "superdiagnostic",
"superdimension", "superendivid", "superenergiz", "superequip", "superescal", "superestim",
"superestrutur", "superexager", "superexalt", "superexamin", "superexcit", "superexplor",
"superfatur", "superfet", "superformat", "superimpregn", "superindustri", "superindustrializ",
"superinterpret", "superioriz", "superlativ", "superlot", "superocup", "superofert",
"superorden", "superorganiz", "superotimiz", "superoxid", "superoxigen", "superpatrocin",
"superplanilh", "superpopul", "superpovo", "supersatur", "superselecion", "superst",
"supertrein", "supervaloriz", "supervis", "supervision", "suplant", "suplement",
"suplic", "suplici", "supont", "suport", "supracit", "supraexcit",
"supralimin", "supralist", "supur", "suput", "surc", "surdin",
"surf", "surmont", "surpass", "surpres", "surr", "surrealiz",
"surri", "surrib", "surripi", "surripilh", "surrupi", "surt",
"suruc", "sururuc", "susceptibiliz", "suscetibiliz", "suscit", "suspeit",
"suspir", "sussurr", "sussurrej", "sust", "susteniz", "sustent",
"sut", "sutiliz", "sutur", "sux", "tabardilh", "tabel",
"tabeli", "tabelion", "tabeliz", "tabic", "tabiz", "tabuc",
"tabul", "tabulet", "taburn", "tac", "tach", "tachon",
"tactiliz", "taful", "tafulh", "tagant", "tagarel", "tain",
"taip", "tairoc", "tal", "taler", "talh", "talim",
"taling", "talion", "talon", "talud", "tamanc", "tamborej",
"tamboril", "tamborin", "tamis", "tamp", "tampon", "tamusc",
"tan", "tanaz", "tanc", "tancam", "tanch", "tang",
"tangenci", "tanific", "tanin", "taniniz", "tano", "tantaliz",
"tap", "tapeç", "tapet", "tapetiz", "tapiç", "tapiz",
"tapulh", "taquigraf", "tar", "taralh", "taramel", "tarantul",
"tard", "tardinh", "taref", "tarel", "tarif", "tarific",
"tarimb", "tarj", "taroc", "tarol", "tarouc", "tarraf",
"tarrax", "tarrinc", "tarrut", "tartamel", "tartariz", "tartufic",
"tartufiz", "tarug", "tasc", "tascanh", "tasco", "tasg",
"tasquin", "tasquinh", "tassalh", "tastavel", "tastavelh", "tatal",
"tataranh", "tatiliz", "tatu", "tauroboliz", "tauxi", "tax",
"taxi", "taxidemiz", "taxidermiz", "tayloriz", "teatin", "teatraliz",
"tec", "tecl", "tecniciz", "tecnocratiz", "tecnologiz", "teim",
"teimosi", "tel", "telecomand", "telecomunic", "telecurs", "teledetect",
"telefon", "telefoniz", "telefotograf", "telegraf", "telegui", "teleport",
"telerradiograf", "telerregul", "telescop", "telessinaliz", "telessond", "telessupervision",
"teletrabalh", "teletransport", "televis", "television", "telex", "telh",
"telint", "telomeriz", "teluriz", "tematiz", "tembl", "temon",
"temoriz", "temper", "tempestu", "templ", "temporaliz", "temporej",
"temporiz", "tenalh", "tenaz", "tencion", "tenço", "tendenci",
"tenebriz", "tenoriz", "tenrific", "tension", "tent", "tenten",
"teocratiz", "teofiliz", "teologiz", "teoriz", "tequet", "terç",
"terceiriz", "tercet", "terci", "terebintin", "terebr",
"tergivers", "terjur", "terlad", "terlint", "termin", "termistoriz",
"termonucleariz", "termorregul", "tern", "terr", "terraform", "terramot",
"terraplan", "terraplen", "terrej", "terrific", "terrinc", "territorializ",
"terror", "terroriz", "tes", "tesauriz", "tesmin", "tesoir",
"tesour", "tessel", "test", "testament", "testavilh", "testemunh",
"testic", "testific", "testilh", "testug", "tet", "tetaniz",
"tetoniz", "textiliz", "textualiz", "textur", "texturiz", "tib",
"tibir", "tibung", "tic", "tiet", "tijol", "til",
"tild", "tilint", "timbalej", "timbr", "timbu", "timon",
"timpaniz", "tindaliz", "tindalliz", "ting", "tinguij", "tint",
"tintalh", "tintil", "tintin", "tintinabul", "tintl", "tipic",
"tipific", "tipiz", "tipograf", "tir", "tiramol", "tiraniz",
"tirilint", "tiriric", "tirit", "tirlint", "tirocin", "tisic",
"tisn", "titic", "titil", "titiric", "titub", "titul",
"titulariz", "tlint", "to", "toc", "tocai", "toes",
"tog", "toiç", "toir", "toirej", "told", "tolej",
"toler", "tolin", "tom", "tomb", "tombol", "tomograf",
"ton", "tonaliz", "tonific", "tonitro", "tonitru", "toniz",
"tons", "tonsur", "tontej", "top", "topej", "topet",
"topi", "topicaliz", "topograf", "tor", "torceg", "torcicol",
"torcion", "torcul", "torment", "torn", "tornej", "toron",
"torped", "torpilh", "torporiz", "torr", "torrej", "torric",
"torrific", "torrisc", "tortuej", "tortur", "torv", "torvelinh",
"tos", "tosc", "toscanej", "tosquenej", "tosqui", "tosseg",
"tossic", "tossig", "tost", "tosteg", "totaliz", "totemiz",
"touc", "touç", "tour", "tourej", "toxic", "trabalh",
"trabalhisc", "trabalhuc", "trabelh", "trabuc", "trabul", "traç",
"tracej", "tracion", "tracolej", "trad", "tradicionaliz", "trafeg",
"trafic", "trafulh", "trag", "tragedi", "tragin", "traicion",
"train", "traj", "tralad", "tralh", "tram", "tramalh",
"trambalh", "trambec", "trambic", "trambolh", "tramel", "tramit",
"tramont", "trampej", "trampolin", "tran", "tranc", "tranç",
"trancafi", "trancafil", "tranch", "tranqueir", "tranquelej", "tranquibern",
"tranquiberni", "tranquiliz", "tranquin", "trans", "transacion", "transbord",
"transbordej", "transcendentaliz", "transco", "transcodific", "transcol", "transcomplement",
"transcultur", "transcur", "transcurs", "transdisciplin", "transeccion", "transecion",
"transexualiz", "transfect", "transfeg", "transfigur", "transfiltr", "transfinaliz",
"transfix", "transfon", "transfor", "transform", "transfret", "transfur",
"transfus", "transignific", "transistoriz", "transit", "transitiv", "translad",
"transliter", "transloc", "translouc", "translucid", "translumbr", "transmalh",
"transmetil", "transmigr", "transmont", "transmud", "transmut", "transnacionaliz",
"transnad", "transnoit", "transnomin", "transnout", "transparent", "transpass",
"transpir", "transplant", "transport", "transtorn", "transtroc", "transubstanci",
"transud", "transum", "transuman", "transvas", "transvaz", "transverber",
"transvi", "transvo", "transvor", "trapaç", "trapach", "trapass",
"trapej", "traquej", "traquin", "traquitan", "trasand", "trasbord",
"trascal", "trascur", "trasfeg", "trasfoli", "traslad", "trasmalh",
"trasmont", "trasmud", "trasnoit", "trasnout", "traspass", "trasplant",
"trasport", "trass", "trastej", "trastemp", "trastorn", "trastroc",
"trasvas", "trasvaz", "trasvi", "trat", "trator", "traulit",
"traumatiz", "traut", "trav", "travej", "traves", "travess",
"travessur", "trebelh", "trebo", "trefeg", "trefil", "trefin",
"trein", "treit", "trejeit", "trejug", "trejur", "trelad",
"treliç", "trem", "tremebrilh", "tremel", "tremelej", "tremelhic",
"tremelic", "tremoç", "tremud", "tremul", "tremulej", "tremulin",
"tren", "trep", "trepan", "trepic", "trepid", "treplic",
"tresand", "tresbord", "trescal", "tresdobr", "tresfeg", "tresfi",
"tresfigur", "tresfil", "tresfoleg", "tresfolg", "tresfoli", "tresgast",
"tresjur", "treslad", "treslouc", "tresmalh", "tresmont", "tresmud",
"tresnoit", "tresnout", "trespass", "tressalt", "tressu", "trestamp",
"trestorn", "tresvari", "tresvi", "trevisc", "tri", "triangul",
"triangulariz", "tribaliz", "tribof", "tribul", "tribut", "tricalh",
"tricot", "trifurc", "trig", "trijunt", "tril", "trilater",
"trilh", "trimbol", "trin", "trinc", "trincafi", "trincafil",
"trinch", "trincheir", "trincolej", "trinf", "trint", "trinul",
"trip", "triplic", "tripsiniz", "tripudi", "tripul", "tris",
"trisc", "triss", "trissec", "trissect", "trissulc", "tristific",
"trit", "tritin", "tritrin", "tritur", "triunf", "trivializ",
"tro", "troc", "troç", "trocadilh", "trocafi", "troch",
"trocisc", "tromb", "trombej", "trombet", "trombic", "trombocintilograf",
"trombos", "tromp", "trompej", "trompic", "trompilh", "tron",
"tronc", "tronch", "tronej", "trop", "tropeç", "tropeg",
"tropej", "tropeli", "tropic", "tropicaliz", "troquel", "troquilh",
"trosqui", "trot", "trotej", "trotin", "trouç", "trov",
"trovej", "trovisc", "trovo", "truc", "trucid", "trucil",
"truf", "trumbic", "trunc", "trunf", "trup", "trupit",
"trusqui", "trustific", "trutin", "tub", "tuberculiniz", "tuberculiz",
"tuberific", "tuberiz", "tubuliz", "tucan", "tuf", "tumb",
"tumefic", "tumidific", "tumific", "tumul", "tumuliz", "tumultu",
"tun", "tund", "tung", "tunisific", "tur", "turb",
"turbilho", "turbilhon", "turbin", "turbolej", "turbulej", "turciz",
"turibul", "turibuliz", "turifer", "turific", "turm", "turquiz",
"turr", "turturej", "turturin", "turturinh", "turtuvi", "turv",
"turvej", "tut", "tutej", "tutel", "tuti", "tutor",
"tutuc", "tutuvi", "uber", "ufan", "ug", "ugalh",
"uiv", "ulcer", "ultim", "ultraciviliz", "ultracolaps", "ultradimension",
"ultraequip", "ultraestrutur", "ultrafiltr", "ultraj", "ultram", "ultramontaniz",
"ultraofert", "ultraotimiz", "ultrapass", "ultrapasteuriz", "ultraprov", "ultrapurific",
"ultrarromantiz", "ultrassonograf", "ulul", "umect", "umidific", "un",
"unanim", "unanimific", "unanimiz", "und", "undul", "unguent",
"unh", "unific", "uniform", "uniformiz", "unilocul", "unitiz",
"universaliz", "univoc", "unt", "untit", "up", "uralitiz",
"uraliz", "uran", "uraniz", "urbaniz", "urin", "urr",
"urtic", "urtig", "urubuz", "us", "usin", "usit",
"usm", "ustul", "usufructu", "usufrutu", "usur", "usurp",
"ut", "utensili", "utiliz", "utopiz", "vac", "vacil",
"vacin", "vacuoliz", "vadi", "vag", "vagabund", "vagamund",
"vaguej", "vai", "val", "vali", "valid", "valor",
"valoriz", "vals", "valsej", "vampir", "vampiriz", "vand",
"vandaliz", "vanglori", "vanguej", "vantaj", "vapor", "vaporiz",
"vapul", "vaquean", "vaqueir", "vaquej", "var", "vardasc",
"varej", "vari", "varicel", "varieg", "varioliz", "varrenh",
"varrisc", "varrusc", "vasc", "vascolej", "vasculariz", "vasculh",
"vasculiz", "vaselin", "vasoativ", "vasodilat", "vasquej", "vassal",
"vassoir", "vassour", "vast", "vaticin", "vavuc", "vaz",
"vazi", "vecej", "ved", "veementiz", "veget", "vegetaliz",
"vei", "veicul", "vel", "velariz", "velej", "velhac",
"velic", "velud", "velut", "venaliz", "vencelh", "vend",
"venefic", "venefici", "venen", "vener", "veniag", "venializ",
"vent", "ventanej", "ventil", "ventoinh", "ventur", "verbaliz",
"verbej", "verber", "verbet", "verdasc", "verdeg", "verdej",
"verduniz", "verg", "vergalh", "vergast", "vergonh", "verific",
"vermelhej", "vermicul", "vermifug", "vermin", "vermit",
"vern", "vernaculiz", "vernaliz", "veroduniz", "verrin", "verrug",
"verrum", "vers", "versej", "versific", "version", "vertebraliz",
"verticaliz", "vertigin", "vertiginiz", "vesic", "vess", "vest",
"vestirenxurr", "vet", "vetoriz", "vex", "vez", "viabiliz",
"viaj", "viand", "viatic", "vibr", "vibratiliz", "viç",
"vicej", "vici", "vid", "vidr", "vig", "vigariz",
"vigi", "vigil", "vigor", "vigoriz", "vilegiatur", "vili",
"vilic", "vilific", "vilipendi", "vim", "vinagr", "vinc",
"vincilh", "vincul", "vindic", "vindim", "ving", "vinh",
"vinific", "vintiplic", "viol", "violent", "vir", "viravir",
"viravolt", "virginaliz", "virginiz", "virgul", "viriliz", "virtu",
"virtualiz", "vis", "visc", "viscondiz", "visg", "visibiliz",
"vision", "visit", "vislumbr", "visp", "visper", "vispor",
"vist", "vistori", "vistoriz", "visualiz", "vitalici", "vitaliz",
"vitamin", "vitaminiz", "vitim", "vitimiz", "vitori", "vitraliz",
"vitrific", "vitriol", "vitrioliz", "vitualh", "vituper", "viv",
"vivalh", "vivenci", "vivent", "vivific", "vivissect", "vizinh",
"vo", "vocabulariz", "vocabuliz", "vocacion", "vocaliz", "vocifer",
"voej", "vog", "volatiliz", "volatiz", "volc", "volit",
"volt", "voltej", "voltij", "voltiz", "volubiliz", "volum",
"voluntari", "volut", "volutu", "vomit", "vortilhon", "vot",
"vozeir", "vulcaniz", "vulgariz", "vulnerabiliz", "vult",
"vun", "vung", "vunz", "vurm", "vuvuc", "xabouc",
"xacat", "xadrez", "xant", "xaquet", "xarop", "xavec",
"xaxat", "xen", "xerardiz", "xeret", "xeroc", "xerocopi",
"xerograf", "xerox", "xetr", "xilograf", "xilograv", "ximbic",
"xin", "xing", "xinguil", "xoc", "xon", "xot",
"xumberg", "xumbreg", "zabumb", "zagai", "zagalot", "zagunch",
"zalum", "zamp", "zamparin", "zang", "zangarilh", "zangul",
"zangurr", "zanz", "zanzarilh", "zaragat", "zaragato", "zaranz",
"zarelh", "zargunch", "zarp", "zav", "zebr", "zel",
"zer", "zi", "zich", "zimbr", "zinc", "zincograf",
"zincograv", "zing", "zingr", "zinzilul", "zip", "zirr",
"zizani", "zizi", "zo", "zomb", "zon", "zonch",
"zongol", "zonz", "zonzon", "zoograf", "zorn", "zorrag",
"zort", "zuc", "zuel", "zul", "zumb", "zumbai",
"zumbeir", "zumbr", "zun", "zung", "zunzun", "zup",
"zurr"])


# VERBOS REGULARES TERMINADOS EM CAR, GAR E CCEDILHA AR - RADICAL
plvrCVbrar = set(["a", "abar", "abel", "abgre", "abju", "abjudi",
"abjur", "able", "abne", "abo", "abode", "abou",
"abra", "abri", "acer", "ache", "acidifi", "acirolo",
"aconche", "acrian", "adar", "ade", "adel", "aden",
"ader", "adjudi", "adli", "ado", "adoci", "adre",
"advo", "aerifi", "afa", "afian", "afidal", "afin",
"aflamen", "afo", "agale", "agre", "agu", "ajindun",
"ajuden", "al", "ala", "alar", "alavan", "alber",
"alcan", "aldea", "aletar", "alfande", "alguer", "alian",
"alicer", "almece", "almo", "almonde", "alo", "alon",
"alu", "alver", "alvori", "alvoro", "ama", "amantei",
"amatun", "amea", "ami", "amol", "amole", "amoque",
"amorda", "amoris", "amplifi", "amulheren", "anafra", "apa",
"apacifi", "ape", "apla", "apli", "apote", "aproche",
"ar", "ardi", "aren", "aris", "armen", "arrai",
"arran", "arrebi", "arrega", "arrei", "arreman", "arren",
"arrene", "arrepi", "arri", "arris", "arro", "arru",
"arrua", "assosse", "ata", "atartaru", "ati", "atoi",
"atra", "atur", "au", "autenti", "autentifi", "autodesli",
"autoli", "automedi", "autopropa", "autorrevo", "avan", "avancar",
"avar", "averdu", "averdun", "azou", "ba", "bagun",
"bajo", "balan", "ban", "barre", "barri", "batu",
"beberi", "berre", "bestifi", "bexi", "bifur", "bin",
"biodiversifi", "biofortifi", "bisna", "blo", "bo", "boi",
"bol", "bombi", "bon", "bonan", "bone", "bonifi",
"bor", "borra", "borre", "borri", "borris", "bote",
"boto", "bou", "brasilifi", "bre", "bri", "brin",
"brio", "bro", "brutifi", "bubulu", "bun", "bur",
"bura", "burri", "burrifi", "bus", "butu", "buze",
"ca", "caboro", "cabou", "cabro", "cabru", "caci",
"cadu", "cafan", "cafun", "cain", "cal", "calamo",
"calcifi", "calembur", "calhan", "calorifi", "can", "candon",
"cantari", "capan", "capen", "capian", "capis", "caprifi",
"car", "caran", "carbonifi", "cardu", "cari", "carnifi",
"caro", "carre", "cartilaginifi", "cas", "caseifi", "casti",
"castifi", "catalo", "catin", "catrace", "catrapis", "catu",
"caulifi", "causti", "caustifi", "cava", "caval", "cavo",
"cavou", "cavu", "caxin", "ce", "celifi", "centrifi",
"centrifu", "centupli", "cer", "cerifi", "certifi", "cha",
"chaba", "chabou", "chala", "chambo", "chambre", "chame",
"chamus", "chan", "chanas", "chapi", "chapis", "chapu",
"chaqui", "charan", "charavis", "charin", "charris", "chas",
"chavanis", "chavas", "che", "cheiri", "cheravas", "chi",
"chian", "chin", "chis", "cho", "chorami", "choramin",
"chorin", "choroco", "chorris", "chu", "chuma", "chumbre",
"chuvenis", "chuveris", "chuvis", "cientifi", "cin", "cinefi",
"circunave", "circunva", "cis", "clarifi", "classifi", "claudi",
"cli", "clini", "clo", "co", "cobi", "cocori",
"codifi", "coedu", "coindi", "coira", "coisifi", "col",
"cole", "coli", "coliti", "colma", "colo", "comar",
"combor", "come", "comis", "compe", "complexifi", "compli",
"compur", "comun", "comuni", "con", "conche", "concul",
"condrifi", "condupli", "confis", "confri", "congra", "congre",
"conju", "consignifi", "conspur", "contrabalan", "contradan",
"contraindi", "contramar", "contrarrepli", "convo", "coobri", "coossifi",
"copides", "coqueifi", "cor", "coris", "cornifi", "corporifi",
"corri", "corru", "corus", "cosi", "cou", "coura",
"cousifi", "cretifi", "criodesse", "criti", "croni", "crucifi",
"crustifi", "cu", "cubi", "cubifi", "cucuri", "cul",
"curu", "cutu", "dan", "dançari", "danifi", "deba",
"deban", "debi", "deboi", "debor", "debou", "debru",
"decal", "decempli", "decodifi", "decorti", "decupli", "dedi",
"defe", "defla", "deifi", "dele", "delga", "deli",
"delin", "delon", "demar", "dene", "densifi", "depeni",
"depre", "dero", "derrai", "derran", "derre", "derrei",
"derren", "derri", "derris", "derro", "dervir", "desaban",
"desabo", "desabor", "desabra", "desabri", "desache", "desacidifi",
"desaconche", "desadere", "desadvo", "desafa", "desafo", "desafus",
"desagre", "desagu", "desala", "desalar", "desalcan", "desalfande",
"desalicer", "desalo", "desalu", "desamachu", "desami", "desamol",
"desamou", "desan", "desapa", "desape", "desapli", "desapre",
"desaqueren", "desar", "desarrai", "desarran", "desarrega", "desarrei",
"desarrene", "desarri", "desarrin", "desarris", "desarru", "desassosse",
"desata", "desatas", "desatra", "desatran", "desatravan", "desau",
"desava", "desavin", "desba", "desbal", "desbalan", "desban",
"desbarran", "desbarri", "desbato", "desbei", "desbi", "desbo",
"desbor", "desbra", "desbre", "desbrin", "desburri", "desburrifi",
"desca", "descaba", "descacha", "descal", "descalcifi", "descali",
"descan", "descanji", "descar", "descarapu", "descaro", "descarre",
"descas", "descaval", "desce", "descer", "desclassifi", "desco",
"descodifi", "descol", "descolo", "descompli", "descomun", "desconche",
"desconju", "desconvo", "descorti", "desedifi", "desedu", "desema",
"desemba", "desembar", "desembara", "desembarran", "desembarri", "desembei",
"desembo", "desembor", "desemborras", "desembos", "desembra", "desembria",
"desembrus", "desembu", "desemburri", "desemo", "desempa", "desempe",
"desempefli", "desempen", "desemperti", "desempo", "desempol", "desempre",
"desemulsifi", "desencabe", "desencacha", "desencan", "desencar", "desencaran",
"desencarapu", "desencarran", "desencarre", "desencas", "desencasa", "desencava",
"desencaval", "desenchar", "desenco", "desencoira", "desencorti", "desencoura",
"desencren", "desencu", "desenfarrus", "desenfeiti", "desenfo", "desenfor",
"desenfras", "desenfus", "desenga", "desengas", "desengon", "desengra",
"desengui", "desenjor", "desenla", "desenli", "desenloda", "desenras",
"desenri", "desenros", "desenru", "desensa", "desenso", "desentalis",
"desento", "desentran", "desentrela", "desentron", "desenver", "desenvis",
"desenxer", "desequivo", "desertifi", "desesperan", "desexcomun", "desfa",
"desfabri", "desfadi", "desfal", "desfalca", "desfo", "desfol",
"desfole", "desfor", "desfortifi", "desgal", "desgaseifi", "desgasifi",
"desgra", "desgre", "desidentifi", "desimpli", "desin", "desindi",
"desinteiri", "desinterli", "desintoxi", "desintri", "desintrin", "desipote",
"desis", "desla", "deslar", "desle", "deslei", "desli",
"deslignifi", "deslinhifi", "deslo", "deslou", "desman", "desmar",
"desmaris", "desmistifi", "desmitifi", "desmo", "desmortifi", "desmultipli",
"desmunhe", "desmus", "desnal", "desnari", "desnazifi", "desne",
"desno", "desnu", "desobri", "desofus", "desoleifi", "desou",
"despaulifi", "despe", "despeda", "despen", "desperdi", "despersonifi",
"despes", "despi", "despin", "despini", "despiro", "despon",
"desposti", "despre", "despregui", "despurifi", "desqualifi", "desrai",
"desrecal", "desri", "desris", "desros", "desru", "desrussifi",
"dessal", "dessalifi", "dessalinifi", "dessamou", "desse", "dessegre",
"dessili", "dessilicifi", "desso", "dessobra", "dessosse", "dessubju",
"desta", "destabo", "destalin", "destipifi", "desto", "destoiti",
"destou", "destouti", "destoutu", "destoxifi", "destra", "destran",
"destravan", "destri", "destrin", "destro", "destron", "desu",
"desumidifi", "desunifi", "desva", "desvi", "desvidra", "desvin",
"desvir", "desvis", "desvitrifi", "deszin", "deten", "detrin",
"detrun", "devitrifi", "devul", "dezin", "dezincifi", "di",
"diagnosti", "dialo", "diferen", "dignifi", "dilar", "dirradi",
"dis", "disfar", "disgre", "disse", "distrofi", "diton",
"diva", "divari", "diversifi", "divul", "do", "domesti",
"domifi", "domin", "dorminho", "dosifi", "dra", "dro",
"dulcifi", "dupli", "e", "ear", "eburnifi", "ecolo",
"edifi", "edu", "educomuni", "edul", "ei", "elan",
"elastifi", "electrifi", "electrodiagnosti", "electromedi", "elen", "eletrifi",
"eletromedi", "elon", "ema", "emaçaro", "emandin", "emanhu",
"emanjeri", "emano", "emba", "embaba", "embaga", "embal",
"embalan", "embaloi", "embalou", "embar", "embara", "embarbas",
"embarra", "embarran", "embarri", "embasba", "embato", "embatu",
"embau", "embebe", "embei", "embel", "embele", "embetes",
"embi", "embibo", "embilo", "embin", "embio", "embiro",
"embis", "embites", "embizu", "embo", "embode", "embodo",
"emboldre", "emboli", "embone", "emboni", "embor", "emborras",
"embos", "embou", "embra", "embran", "embri", "embria",
"embrin", "embro", "embrua", "embrus", "embu", "embura",
"emburri", "emburris", "emedoi", "emedou", "emendi", "emeroi",
"emerou", "eminho", "emorda", "emoroi", "emorou", "emos",
"empa", "empaço", "empalhas", "empan", "empapu", "empar",
"empaupi", "empe", "empeiti", "empeli", "empen", "emperni",
"emperti", "empes", "empete", "empetele", "empi", "empile",
"empinho", "empino", "empipo", "empis", "empla", "empo",
"empocil", "empol", "empor", "emposti", "empre", "empregui",
"empul", "emulsifi", "en", "encabe", "encacha", "encachapu",
"encaga", "encal", "encalamou", "encali", "encalun", "encama",
"encambu", "encamoroi", "encamorou", "encamur", "encan", "encani",
"encar", "encaran", "encarapu", "encario", "encarni", "encaro",
"encarran", "encarrapi", "encarras", "encarre", "encarris", "encarvoi",
"encas", "encasa", "encausti", "encava", "encaval", "encazu",
"enchapu", "enchar", "encho", "enchoiri", "enchouri", "enchuma",
"encis", "encloa", "encoira", "encolo", "encomoro", "encomoroi",
"encomorou", "encon", "enconha", "enconi", "encor", "encorri",
"encorti", "encoura", "encramoio", "encramou", "encren", "encro",
"encu", "encurri", "endere", "endil", "endomin", "endra",
"enes", "enfacho", "enfan", "enfani", "enfarrus", "enfeiti",
"enfin", "enfiteuti", "enfo", "enfoi", "enfor", "enfou",
"enfra", "enfraldi", "enfras", "enfuma", "enfunis", "enfus",
"enga", "engal", "engali", "engalri", "engas", "engati",
"engero", "engi", "engon", "engra", "engui", "enjal",
"enjan", "enji", "enjor", "enju", "enjun", "enla",
"enlabrus", "enlaru", "enli", "enlo", "enloda", "enloi",
"enlou", "eno", "enoiri", "enouri", "enra", "enran",
"enras", "enre", "enredoi", "enredou", "enri", "enro",
"enrodri", "enros", "enru", "ensa", "ensal", "ense",
"enses", "enso", "ensofre", "ensova", "entabi", "entabo",
"entalei", "entali", "entalin", "entalis", "entan", "entani",
"entapi", "ente", "enteju", "enti", "entiju", "entisi",
"ento", "entoi", "entou", "entrambeli", "entrambi", "entran",
"entravis", "entre", "entrecho", "entrela", "entreli", "entreloi",
"entrelou", "entrepi", "entrepli", "entrerre", "entressa", "entressufo",
"entreto", "entrin", "entro", "entron", "entrope", "entropi",
"entrovis", "entrupi", "entuju", "enturvis", "envelha", "enver",
"enverru", "enves", "envi", "envidra", "envis", "enxal",
"enxaque", "enxau", "enxe", "enxer", "enxo", "enxu",
"epigramati", "epilo", "equivo", "eri", "erradi", "erri",
"es", "esbaba", "esbaga", "esbalan", "esbalou", "esban",
"esbarran", "esbarre", "esbarri", "esbarro", "esbei", "esbeltifi",
"esbeni", "esberra", "esbi", "esbo", "esbode", "esboi",
"esborra", "esborre", "esbou", "esbra", "esbranqui", "esbrin",
"esbron", "esbru", "esbur", "esbura", "esca", "escabe",
"escachou", "escadra", "escaga", "escalaman", "escalamou", "escali",
"escamur", "escamurren", "escan", "escane", "escar", "escaramu",
"escarapu", "escarda", "escardi", "escardu", "escarifi", "escarmou",
"escarni", "escarnifi", "escaro", "escarou", "escarrapi", "escarvan",
"escarvun", "escas", "escava", "esclamou", "esclerifi", "esco",
"escoi", "escor", "escorifi", "escorra", "escorre", "escorrima",
"escorti", "escou", "escozi", "escramou", "escramu", "escul",
"esfal", "esfalca", "esfande", "esfani", "esfarrun", "esfel",
"esflo", "esfoi", "esfole", "esfolha", "esfor", "esformi",
"esforni", "esforri", "esfou", "esfran", "esfre", "esfregon",
"esfuma", "esfura", "esfuran", "esfutri", "esga", "esgal",
"esgani", "esgar", "esgati", "esla", "eslin", "esma",
"esmadri", "esmar", "esmarne", "esmarno", "esmartu", "esmatru",
"esmija", "esmiun", "esmo", "esmoi", "esmon", "esmor",
"esmorda", "esmordi", "esmorra", "esmou", "esmurra", "esmurre",
"esmurren", "esna", "esno", "espa", "espada", "espaje",
"espalha", "espan", "espanas", "espandon", "espapa", "esparre",
"espassari", "espe", "especifi", "espectrifi", "espeda", "espedar",
"espedre", "espeni", "esperan", "esperdi", "esperme", "esperne",
"espesco", "espetrifi", "espi", "espica", "espicoi", "espicou",
"espin", "espini", "espipo", "espo", "esporifi", "espotri",
"espou", "espregui", "espreme", "espul", "espuma", "esquadra",
"esqui", "esquistifi", "esta", "estabou", "estafe", "estali",
"estan", "estardalha", "estarrifi", "estarrin", "estatistifi", "esteli",
"ester", "esterifi", "esterli", "esterni", "esti", "estilha",
"estin", "estira", "esto", "estoma", "estome", "estorce",
"estorci", "estorde", "estorne", "estorni", "estorri", "estorte",
"estra", "estrabou", "estrafe", "estran", "estrapa", "estratifi",
"estre", "estreban", "estrefe", "estreli", "estrelou", "estri",
"estrin", "estro", "estroci", "estron", "estrufe", "estrumbi",
"estu", "estultifi", "estupefi", "estupidifi", "esturri", "esver",
"esvidi", "esvoa", "eterifi", "evo", "exal", "exatifi",
"excarnifi", "excomun", "exemplifi", "exgre", "exiguifi", "expli",
"expur", "exsi", "extri", "fa", "fabri", "fadi",
"fal", "fala", "falca", "falho", "falo", "falsifi",
"falu", "fandan", "fandin", "fani", "fanu", "faris",
"farras", "farron", "farun", "fati", "femen", "fi",
"fin", "fis", "fitu", "flo", "flogisti", "fluidifi",
"fo", "fofo", "foi", "fol", "fole", "folhe",
"for", "forji", "formi", "forni", "forri", "forta",
"fortale", "fortifi", "fos", "fossan", "fossilifi", "fotomultipli",
"fou", "fra", "fraldi", "fre", "freju", "fres",
"frigorifi", "fro", "frute", "frutifi", "fu", "fube",
"fuji", "fule", "fuma", "fume", "fumi", "fun",
"fura", "furdun", "furti", "furun", "fus", "fusti",
"futi", "futri", "futu", "fuxi", "gadi", "gal",
"gali", "gan", "gani", "gar", "garan", "gargada",
"gargala", "gaseifi", "gasifi", "gelatinifi", "geleifi", "gelifi",
"gemel", "gemelhi", "gemi", "gin", "ginasti", "girova",
"gla", "glorifi", "gomifi", "gon", "gou", "gramati",
"granitifi", "gratifi", "gron", "guas", "guen", "hermetifi",
"heroifi", "heterocarre", "hiperapli", "hipote", "hispanifi", "homolo",
"honorifi", "honrifi", "horrifi", "hulhifi", "humifi", "i",
"identifi", "idiotifi", "ignifi", "ignifu", "ili", "imbeli",
"imbi", "imbri", "impel", "impersonifi", "impli", "impre",
"impurifi", "in", "incomuni", "incul", "incurri", "inda",
"indi", "indicifi", "indignifi", "indil", "infin", "infuni",
"injun", "injusti", "injustifi", "inli", "insti", "intei",
"inteiri", "intensifi", "intercomuni", "interesterifi", "interestratifi", "interi",
"interla", "interli", "interpi", "interro", "interse", "inti",
"intoxi", "intri", "intrin", "investi", "invo", "invul",
"irri", "irro", "is", "jan", "jantari", "jazzifi",
"jo", "jon", "jornalisti", "ju", "judi", "jul",
"jun", "jur", "justi", "justifi", "la", "labre",
"laicifi", "lambaris", "lambi", "lambis", "lambre", "lambu",
"lan", "langu", "lapidifi", "lar", "las", "late",
"lati", "latin", "lavaris", "lavas", "lave", "le",
"lei", "len", "lengalen", "lenhifi", "lenifi", "letifi",
"levi", "li", "lignifi", "lin", "linifi", "liquenifi",
"liquidifi", "liri", "lis", "liti", "litifi", "lo",
"lobre", "lobri", "logi", "lombri", "lor", "lu",
"lubri", "lubrifi", "lus", "luscofus", "lusifi", "ma",
"maca", "maçaro", "macaven", "machi", "machu", "macu",
"madefi", "madra", "madru", "magi", "magnifi", "malo",
"malu", "man", "mandin", "mandu", "manjeri", "mano",
"mar", "maris", "marri", "marru", "mas", "massifi",
"masti", "mastu", "mati", "matra", "matru", "maturran",
"mazur", "medi", "mei", "melifi", "men", "mendi",
"menospre", "mer", "mercurifi", "merendu", "mes", "mesti",
"metacomuni", "metafisi", "metalifi", "metodifi", "metrifi", "mexeri",
"mexeru", "mi", "mimi", "min", "minho", "minifi",
"ministrifi", "mirifi", "mis", "mistifi", "miti", "mitifi",
"mo", "modi", "modifi", "mogan", "mogi", "moji",
"mol", "mole", "molen", "molifi", "mon", "monolo",
"monoton", "moque", "mor", "moran", "morce", "morda",
"mordi", "mordis", "morra", "morse", "mortifi", "mos",
"mosse", "mostren", "mou", "mouris", "muchi", "mufi",
"mugan", "multipli", "mumifi", "mundifi", "mungan", "munhe",
"mupi", "mupu", "mus", "musi", "muxi", "na",
"nadifi", "namori", "namoris", "naufra", "nave", "nazifi",
"ne", "nega", "nene", "nes", "nevis", "nevoa",
"ni", "nidifi", "niilifi", "nitidifi", "nitrifi", "no",
"nobilifi", "noctiva", "notifi", "nuan", "nulifi", "nutrifi",
"o", "obce", "objectifi", "objetifi", "objur", "obri",
"octupli", "ofe", "ofus", "oi", "oiri", "opa",
"opacifi", "or", "organifi", "orti", "ortorretifi", "ossifi",
"ou", "ouri", "outor", "ozonifi", "pa", "pacifi",
"padu", "palan", "palhi", "palifi", "pan", "pana",
"pande", "panegiri", "panifi", "papagu", "papari", "papo",
"papou", "paraliti", "parifi", "parlen", "parran", "pate",
"patus", "paulifi", "pe", "pechirin", "pedi", "penduri",
"peni", "penifi", "penis", "per", "percal", "percoru",
"perdi", "perere", "peri", "perlen", "perlon", "perren",
"personifi", "perva", "pes", "pespe", "petanis", "pete",
"petis", "petrifi", "pfu", "pi", "pian", "piciri",
"pin", "pindo", "pindon", "pini", "pino", "pintal",
"pipiri", "pipo", "piran", "piriri", "piro", "pirra",
"pis", "pitos", "piu", "planifi", "plasti",
"plastifi", "plenifi", "pli", "plu", "pluriestratifi", "plurifi",
"pluristratifi", "po", "poetifi", "pol", "pole", "polemi",
"polirramifi", "politi", "pomifi", "pon", "pontifi", "popo",
"pororo", "porvis", "poste", "poster", "posti", "poto",
"pra", "prati", "pre", "preale", "precal", "precifi",
"predemar", "predi", "prefabri", "pregui", "preite", "prejudi",
"prenuni", "presentifi", "presi", "pretra", "prevari", "pris",
"prodi", "profa", "profal", "profli", "prognosti", "prolifi",
"prolo", "prolon", "promul", "pronosti", "prontifi", "propa",
"prorro", "prosifi", "prototifi", "provis", "provo", "pseudojustifi",
"psicolo", "publi", "puli", "pun", "pur", "purifi",
"putrifi", "quadrupli", "qualifi", "quantifi", "qui", "quilifi",
"quimifi", "quin", "quintupli", "rabis", "rabus", "radi",
"radiodiagnosti", "radiomar", "radionave", "radiopre", "raman", "ramifi",
"ramo", "ran", "rancidifi", "rancifi", "rapi", "ras",
"rasmol", "raspan", "ratifi", "re", "reabra", "readjudi",
"real", "realar", "realo", "realu", "reapli", "reati",
"reautenti", "rebalan", "rebeli", "rebi", "rebis", "rebo",
"rebol", "reboli", "rebou", "rebu", "rebuli", "rebus",
"recal", "recalcifi", "recar", "recarre", "recaval", "recha",
"reche", "recipro", "reclassifi", "reco", "recoli", "recolo",
"recome", "recongra", "reconvo", "recor", "recrucifi", "rededi",
"reden", "redoi", "redou", "redupli", "reedifi", "reedu",
"reembar", "reempol", "reempor", "reempre", "reencu", "reendere",
"reenla", "reescramou", "reescramu", "reespecifi", "reexpli", "refe",
"refin", "refo", "refol", "refole", "refor", "refortifi",
"refre", "refres", "refri", "refu", "refun", "refunfume",
"rega", "regaseifi", "regou", "reguin", "reifi", "reinterro",
"reinvo", "reivindi", "rejun", "rejustifi", "relan", "relar",
"rele", "reli", "relin", "relo", "relou", "reman",
"remanis", "remar", "remas", "remasti", "remel", "remeni",
"remo", "remol", "remor", "remordi", "remordis", "remou",
"remu", "remun", "remus", "renave", "reobri", "repa",
"repe", "repeni", "repes", "repi", "repil", "repini",
"repis", "repli", "repol", "repole", "repre", "republi",
"repur", "repurifi", "requalifi", "rerratifi", "res", "resfol",
"resgal", "resinifi", "resmelen", "resmol", "resmun", "resmus",
"respan", "respi", "respin", "ressa", "resse", "ressi",
"ressul", "restauri", "restru", "restu", "reta", "retifi",
"retipifi", "reto", "retoi", "retori", "retou", "retra",
"retran", "retrin", "retrocarre", "retru", "reunifi", "rever",
"reverifi", "revi", "revin", "revindi", "revivifi", "revo",
"revol", "rezin", "ri", "ricer", "ridi", "rin",
"ripan", "ris", "ro", "roça", "roce", "rodri",
"roli", "romanes", "ron", "ros", "rou", "ru",
"rubifi", "rubri", "rus", "russifi", "rusti", "rustifi",
"sa", "sabere", "saberere", "sabre", "sabu", "saçari",
"sacarifi", "sacrifi", "sal", "salifi", "salpi", "saltari",
"salu", "salubrifi", "saman", "sambar", "sambo", "sanguessu",
"sanguifi", "sani", "sanifi", "santi", "santifi", "sape",
"sapere", "saponifi", "sarafus", "sarapi", "sarrabis", "sarrafa",
"saxifi", "se", "sebenis", "segre", "selvati", "semiapli",
"semidomesti", "sen", "sensifi", "septupli", "seraman", "serin",
"serraman", "setupli", "sextupli", "signifi", "silabifi", "silicifi",
"silogisti", "simplifi", "sindi", "sinte", "sinu", "sir",
"siriri", "so", "sobal", "sobne", "sobra", "sobran",
"sobrecasa", "socal", "soço", "sofisti", "sofre",
"soji", "soju", "sola", "solan", "solavan", "sole",
"solidifi", "solu", "somiti", "sone", "sope", "sorbi",
"sororo", "sorras", "sosse", "sotran", "soupi", "su",
"subapli", "subclassifi", "subelis", "subenfiteuti", "suberifi", "subespecifi",
"subju", "sublo", "substantifi", "sudorifi", "sufo", "sufra",
"sufumi", "suji", "suju", "sul", "sulan", "sun",
"superapli", "superdiagnosti", "supli", "sur", "suru", "sururu",
"ta", "tabi", "tabu", "tairo", "talin", "taman",
"tamus", "tan", "tanifi", "tape", "tapi", "tarifi",
"taro", "tarou", "tarrin", "tartufi", "taru", "tas",
"te", "telecomuni", "tenrifi", "ter", "terrifi", "terrin",
"testi", "testifi", "testu", "ti", "tibun", "tin",
"tipi", "tipifi", "tiriri", "tisi", "titi", "titiri",
"to", "toi", "tonifi", "torce", "torri", "torrifi",
"torris", "tos", "tosse", "tossi", "toste", "tou",
"toxi", "tra", "trabalhis", "trabalhu", "trabu", "trafe",
"trafi", "trambe", "trambi", "tran", "transcodifi", "transfe",
"transignifi", "translo", "translou", "transtro", "trapa", "trasfe",
"trastro", "trefe", "treju", "treli", "tremelhi", "tremeli",
"tremo", "trepi", "trepli", "tresfe", "tresfol", "tresfole",
"treslou", "trevis", "tri", "trifur", "trin", "tripli",
"tris", "trisse", "trissul", "tristifi", "tro", "trocis",
"trombi", "trompi", "tron", "trope", "tropi", "trou",
"trovis", "tru", "trumbi", "trun", "trustifi", "tuberifi",
"tumefi", "tumidifi", "tumifi", "tun", "tunisifi", "turifi",
"tutu", "u", "ultrapurifi", "umidifi", "unanimifi", "unifi",
"univo", "urti", "va", "vardas", "varie", "varris",
"varrus", "vas", "vavu", "velha", "veli", "venefi",
"venia", "ver", "verdas", "verde", "verifi", "vermifu",
"verru", "versifi", "vesi", "vi", "viati", "vili",
"vilifi", "vin", "vindi", "vinifi", "vintipli", "vis",
"vitrifi", "vivifi", "vo", "vol", "vun", "vuvu",
"xabou", "xave", "xero", "ximbi", "xin", "xo",
"xumber", "xumbre", "zan", "zin", "zorra", "zu",
"zun"])

# EXCECAO VERBOS REGULARES TERMINADOS EM ER - RADICAL + bdhilmnoprstuvxz + er
exvrVbrxer = set(["como", "vivo"])

# VERBOS REGULARES TERMINADOS EM ER - RADICAL + bdhilmnoprstuvxz + er
plvrVbrxer = set(["abat", "abced", "absced", "abscond", "absolv",
"absorb", "absorv", "aced", "acend", "acolh",
"acomet", "acorr", "adjaz", "adscrev", "ambitend",
"ambivert", "amov", "anteced", "anted", "antemov",
"antevert", "apend", "aperceb", "apreend", "aprem",
"aprend", "ard", "arremet", "arrepend", "ascend",
"astrev", "atend", "atrev", "aurisplend", "autoabsorv",
"autoatend", "autocompreend", "autoconced", "autoconvert", "autodefend",
"autodesenvolv", "autopreench", "autopromov", "autossuspend", "avert",
"barr", "bat", "beb", "benz", "bival",
"carcom", "catafed", "ced", "chov", "cincunscrev",
"circumpercorr", "circunjaz", "circunscrev", "circunv", "circunvolv",
"codev", "coescrev", "coestend", "colh", "combat",
"com", "comet", "comov", "compon", "compreend",
"compromet", "conceb", "conced", "concorr", "condescend",
"contend", "contrabat", "contrarrespond", "contrarromp", "contravert",
"controvert", "convert", "conviv", "convolv", "coocorr",
"corr", "correspond", "corromp", "cos", "coz",
"debat", "decorr", "defend", "delamb", "demov",
"depend", "deperd", "depreend", "derret", "desacolh",
"desacorr", "desaperceb", "desaprend", "desarrepend", "desatend",
"desbeb", "descend", "descom", "descomet", "descomov",
"descompreend", "descompromet", "desconvert", "descos", "descrev",
"desd", "desembeb", "desench", "desencolh", "desentend",
"desentolh", "desenvolv", "desescond", "deslamb", "desmord",
"desmorr", "despend", "desperceb", "desprecav", "despreconceb",
"desprend", "despromov", "dessocorr", "dessorv", "dessoub",
"destem", "destolh", "desviv", "dev", "devolv",
"discorr", "dispend", "disperd", "dissolv", "distend",
"embat", "embeb", "empreend", "encend", "ench",
"encolh", "encoz", "enfond", "entend", "entolh",
"entrebat", "entrecolh", "entrecorr", "entrecoz", "entremet",
"entremord", "entreperceb", "entreprend", "entrescolh", "entromet",
"envolv", "erod", "esbat", "escafed", "escolh",
"escond", "escorr", "escoz", "escrev", "esferv",
"espland", "esplend", "esprem", "essolv", "estend",
"estrel", "estrem", "esvarr", "evolv", "exced",
"excel", "expend", "exsolv", "extravert", "fed",
"fend", "ferv", "fod", "frem", "frend",
"gem", "hiperabsorv", "hiperconced", "hiperconcorr", "hiperdesenvolv",
"hiperestend", "hipodesenvolv", "imov", "impend", "improced",
"incend", "incompreend", "incorr", "independ", "inscrev",
"intend", "interced", "intercorr", "intercorrespond", "interdepend",
"intermet", "interprend", "interromp", "intervert", "intromet",
"introvert", "invert", "irromp", "jaz", "juncarench",
"lamb", "locomov", "lucitrem", "malcoz", "malsofr",
"manuscrev", "megadesenvolv", "met", "mex", "mord",
"morremorr", "morr", "mov", "ocorr", "ofend",
"pendepend", "pend", "perceb", "percorr", "persolv",
"persorv", "perv", "pervert", "porfend", "precav",
"preced", "preconceb", "precorr", "preench", "prem",
"premorr", "prend", "prerrebat", "prerromp", "prescrev",
"pretend", "previv", "proced", "promet", "promov",
"propend", "prorromp", "proscrev", "protend", "reabsorv",
"reacend", "reaprend", "rearrepend", "reascend", "rebat",
"rebeb", "rebenz", "receb", "reced", "recend",
"recolh", "recom", "recompreend", "recompromet", "reconcorr",
"reconvert", "recorr", "recos", "recoz", "redescend",
"redissolv", "reempreend", "reench", "reentend", "reescond",
"reescrev", "refend", "referv", "reinscrev", "relamb",
"remet", "remex", "remord", "remorr", "remov",
"rend", "reouv", "repercorr", "repervert", "repreend",
"reprend", "repromet", "rescend", "rescrev", "resmo",
"resmord", "resolv", "resplend", "respond", "ressorv",
"ressubmet", "retranscrev", "retrem", "retroced", "retrovend",
"retrovert", "revend", "revert", "reviv", "revolv",
"romp", "semiabsorv", "semiconced", "semiviv", "sobrecos",
"sobredistend", "sobreintend", "sobrejaz", "sobrepend", "sobrescrev",
"sobressorv", "sobressorv", "sobrestiv", "sobreviv", "sobrexced",
"socorr", "sofr", "solv", "somet", "sorv",
"sovert", "subabsorv", "subconced", "subdesenvolv", "subentend",
"subentend", "subintend", "subjaz", "submet", "submilh",
"subscrev", "subvert", "suced", "superabsorv", "superconced",
"superconcorr", "superdesenvolv", "superientend", "superintend", "surpreend",
"suspend", "telev", "tem", "tend", "tol",
"tolh", "tramet", "transcend", "transcorr", "transcrev",
"transmov", "transvert", "trel", "trem", "tremet",
"tremetrem", "tresentend", "tresv", "ultraconcorr", "varr",
"vasomov", "vend", "vert", "viv", "volv"])

# VERBOS REGULARES TERMINADOS EM ER - RADICAL + cer
plvrVbrcer = set(["abalore", "abaloro", "abarele", "abarole", "abaste",
"abe", "ablaque", "abolere", "abolore", "aborele",
"aborre", "abrande", "abranque", "abrute", "acae",
"acaules", "aclare", "aconhe", "aconhes", "aconte",
"acre", "acres", "adoe", "adoles", "adorme",
"aferves", "afortale", "agude", "alente", "alparde",
"aluze", "alves", "alvore", "amadure", "amanhe",
"amare", "amarele", "amares", "amodorre", "amole",
"amorne", "amorte", "amurche", "anoite", "anoute",
"antenas", "apara", "apare", "apete", "apodre",
"aque", "aquies", "arbores", "arrefe", "arvore",
"atere", "aterre", "atre", "aurores", "autoabaste",
"autoaque", "autoconhe", "autoenrique", "automere", "azule",
"baste", "bolore", "brande", "cancres", "care",
"chore", "coales", "cognes", "cognos", "compade",
"compare", "compes", "concres", "confortale", "conhe",
"contabes", "contor", "convales", "conven", "coude",
"cres", "decres", "deliques", "denegre", "depere",
"desabaste", "desaborre", "desadorme", "desagrade", "desapare",
"desapare", "desaque", "desbaste", "desbrave", "descandele",
"descare", "des", "descompade", "desconhe", "desconven",
"desembrave", "desembrute", "desempe", "desempederne", "desempobre",
"desemude", "desencare", "desenegre", "desenfure", "desengrande",
"desengrave", "desengraves", "desenlouque", "desenobre", "desenraive",
"desenrique", "desenrouque", "desenrubes", "desensande", "desensoberbe",
"desensurde", "desentenebre", "desenterne", "desentorpe", "desentrete",
"desentriste", "desentume", "desentumes", "desescure", "desfale",
"desfavore", "desflores", "desfortale", "desfrondes", "desfrutes",
"desguarne", "desintumes", "desmae", "desmarele", "desmere",
"desmore", "desnobre", "desobede", "desobscure", "despere",
"deste", "destor", "destorpe", "destriste", "desumede",
"desvae", "desvane", "desverde", "detumes", "dispare",
"distor", "dorme", "eferves", "eflores", "elangues",
"elaste", "emadure", "emagre", "emanque", "emarele",
"embambe", "embarate", "embarbe", "embaste", "embebe",
"embele", "embeve", "embolore", "emborne", "embrabe",
"embrande", "embranque", "embrave", "embrune", "embrute",
"emburre", "emenine", "emorne", "emorte", "emouque",
"empale", "empalide", "emparde", "emparve", "emparvoe",
"empe", "empederne", "empequene", "emplumes", "empobre",
"empodere", "emprete", "empubes", "empure", "empurpure",
"empute", "emude", "emugre", "emurche", "enalte",
"enarde", "encale", "encalve", "encande", "encane",
"encare", "encegue", "encrudele", "encrudes", "encrue",
"encruele", "endente", "endoide", "endolore", "endonde",
"endoude", "endure", "enegre", "enfebre", "enfene",
"enferve", "enflores", "enforte", "enfraque", "enfroixe",
"enfrouxe", "enfrute", "enfure", "engafe", "engalharde",
"engrade", "engrae", "engrande", "engraves", "engravide",
"engrume", "enlangues", "enlivide", "enloire", "enlorpe",
"enlouque", "enloure", "enlute", "enobre", "enoite",
"enorgulhe", "enoute", "enraive", "enrare", "enrije",
"enrique", "enrobuste", "enrouque", "enrubes", "enrude",
"enrudes", "ensande", "ensarne", "ensaude", "ensoberbe",
"ensombre", "ensurde", "entale", "entangue", "entarde",
"ente", "entenebre", "entenre", "entenue", "entepide",
"enterne", "entibe", "entole", "entonte", "entorpe",
"entreconhe", "entrete", "entretor", "entreve", "entriste",
"entronque", "entume", "entumes", "entupe", "enturge",
"enturve", "envaide", "envale", "envane", "envelhe",
"enverde", "envermelhe", "envile", "erubes", "erve",
"esbranque", "esbrave", "escadele", "escandale", "escande",
"escandes", "escarde", "escarne", "esclare", "escure",
"esfale", "esferve", "esmae", "esmagre", "esmore",
"espaire", "espare", "espavore", "espere", "esplande",
"esplende", "esplendes", "esque", "esquerde", "estabele",
"estarre", "esterile", "esterre", "estonte", "estor",
"estre", "estreite", "estrele", "estreme", "estrope",
"estupide", "esvae", "esvane", "etegue", "evanes",
"even", "excres", "exer", "fale", "farte",
"favore", "fene", "flaves", "flores", "fluides",
"fluore", "fluores", "forne", "fortale", "fosfore",
"fosfores", "fronde", "frondes", "frutes", "gere",
"grade", "grume", "guare", "guarne", "hetegue",
"hiperabaste", "imere", "imunofluores", "incandes", "infrutes",
"interte", "intimide", "intumes", "inturges", "invales",
"juvenes", "lactes", "langues", "languides", "lare",
"lates", "lentes", "liques", "livides", "loire",
"loure", "madure", "manque", "mere", "murche",
"nas", "nites", "nobre", "noite", "noute",
"obede", "obscure", "obsoles", "ofere", "opales",
"orgulhe", "oures", "pade", "palide", "parde",
"pare", "pas", "pere", "permane", "perten",
"perven", "podre", "preaque", "preestabele", "prenas",
"prevale", "pubes", "raive", "rances", "rares",
"reabaste", "readorme", "reagrade", "realvore", "reamanhe",
"reapare", "reapare", "reaque", "rebrande", "reconhe",
"reconvales", "reconven", "recres", "recrudes", "redes",
"reendure", "reenrique", "reestabele", "refe", "reflores",
"reforne", "refortale", "refrondes", "rejuvenes", "remaes",
"remanes", "remere", "renas", "repas", "resplande",
"resplende", "restabele", "retor", "reverde", "revives",
"revivis", "robuste", "rubes", "rubores", "semiabaste",
"sobreaque", "sobrecres", "sobrete", "subabaste", "subestabele",
"substabele", "superabaste", "superaque", "te", "tenebres",
"tor", "torpe", "transpare", "trasportale", "tume",
"tumes", "turges", "umede", "vane", "vanes",
"vemelhe", "ven", "verde", "vermelhe", "viles"])

# VERBOS REGULARES TERMINADOS EM ER - RADICAL + ger
plvrVbrger = set(["abran", "abster", "amer", "asper", "atan",
"coele", "co", "confran", "constran", "coprote",
"corre", "desconfran", "desconstran", "desele", "desprote",
"desre", "deter", "ele", "esbran", "estran",
"fran", "hiperprote", "malre", "mon", "mun",
"plan", "porre", "preele", "prote", "quinquele",
"ran", "reele", "refran", "re", "rerran",
"rin", "superprote", "tan", "treele", "vi"])

# VERBOS REGULARES TERMINADOS EM ER - RADICAL + uer
plvrVbruer = set(["erg", "falg", "reerg", "semierg", "soerg"])

# VERBO TER - [Tt]em
vbTer = set(["em","êm", "emos", "ende", "endes",
"endo", "enha", "enhais", "enham", "enhamos",
"enhas", "enho", "ens", "er", "erá",
"erão", "erás", "erdes", "erei", "ereis",
"erem", "eremos", "eres", "eria", "eriam",
"eríamos", "erias", "eríeis", "ermos", "eve",
"ido", "inha", "inham","ínhamos", "inhas",
"ínheis", "ive", "ivemos", "iver", "ivera",
"iveram", "ivéramos", "iveras", "iverdes", "ivéreis",
"iverem", "iveres", "ivermos", "ivesse", "ivésseis",
"ivessem", "ivéssemos", "ivesses", "iveste", "ivestes"])

# VERBO DAR - [Dd]ou
vbDar = set([u"á", "ado", "ai", "ais", "amos", "ando","ão",
"ar", "ará", "arão", "arás", "ardes",
"arei", "areis", "arem", "aremos", "ares",
"aria", "ariam", "aríamos", "arias", "aríeis",
"armos","ás", "ava", "avam","ávamos",
"avas","áveis","ê", "eem", "ei",
"eis", "emos", "er", "era", "eram",
"éramos", "eras", "erdes","éreis", "erem",
"eres", "ermos","ês", "esse","ésseis",
"essem","éssemos", "esses", "este", "estes",
"eu", "ou"])

# VERBO HAVER - [Hh]ei
vbHaver = set([u"á", "aja", "ajais", "ajam", "ajamos",
"ajas","ão","ás", "avei", "aveis",
"avemos", "avendo", "aver", "averá", "averão",
"averás", "averdes", "averei", "avereis", "averem",
"averemos", "averes", "averia", "averiam", "averíamos",
"averias", "averíeis", "avermos", "avia", "aviam",
"avíamos", "avias", "avido", "avíeis", "i",
"ouve", "ouvemos", "ouver", "ouvera", "ouveram",
"ouvéramos", "ouveras", "ouverdes", "ouvéreis", "ouverem",
"ouveres", "ouvermos", "ouvesse", "ouvésseis", "ouvessem",
"ouvéssemos", "ouvesses", "ouveste", "ouvestes"])

# VERBO FICAR - [Ff]ico
vbFicar = set(["ica", "icado", "icai", "icais", "icam",
"icamos", "icando", "icar", "icara", "icará",
"icaram", "icáramos", "icarão", "icaras", "icarás",
"icardes", "icarei", "icareis", "icáreis", "icarem",
"icaremos", "icares", "icaria", "icariam", "icaríamos",
"icarias", "icaríeis", "icarmos", "icas", "icasse",
"icásseis", "icassem", "icássemos", "icasses", "icaste",
"icastes", "icava", "icavam", "icávamos", "icavas",
"icáveis", "ico", "icou", "ique", "iquei",
"iqueis", "iquem", "iquemos", "iques"])

# VERBO ESTAR - [Ee]stou
vbEstar = set(["stá", "stado", "stai", "stais", "stamos",
"stando", "stão", "star", "stará", "starão",
"starás", "stardes", "starei", "stareis", "starem",
"staremos", "stares", "staria", "stariam", "staríamos",
"starias", "staríeis", "starmos", "stás", "stava",
"stavam", "stávamos", "stavas", "stáveis", "steja",
"stejais", "stejam", "stejamos", "stejas", "steve",
"stive", "stivemos", "stiver", "stivera", "stiveram",
"stivéramos", "stiveras", "stiverdes", "stivéreis", "stiverem",
"stiveres", "stivermos", "stivesse", "stivésseis", "stivessem",
"stivéssemos", "stivesses", "stiveste", "stivestes", "stou"])

# VERBO SER
vbSer = set([u"é", "era", "eram","éramos", "eras",
"éreis","és", "foi", "fomos", "for",
"fora", "foram", "fôramos", "foras", "fordes",
"fôreis", "forem", "fores", "formos", "fosse",
"fôsseis", "fossem", "fôssemos", "fosses", "foste",
"fostes", "fui", "são", "sê", "sede",
"seja", "sejais", "sejam", "sejamos", "sejas",
"sendo", "ser", "será", "serão", "serás",
"serdes", "serei", "sereis", "serem", "seremos",
"seres", "seria", "seriam", "seríamos", "serias",
"seríeis", "sermos", "sido", "sois", "somos",
"sou"])


# vbFICARsing = verbo ficar conjugacao plural
vbFICARplur = {"ica": "icam",
"icado": "icado",
"icai": "icai",
"icais": "icais",
"icam": "icam",
"icamos": "icamos",
"icando": "icando",
"icar": "icarem",
"icara": "icaram",
"icará": "icarão",
"icaram": "icaram",
"icáramos": "icáramos",
"icarão": "icarão",
"icaras": "icáreis",
"icarás": "icareis",
"icardes": "icardes",
"icarei": "icaremos",
"icareis": "icareis",
"icáreis": "icáreis",
"icarem": "icarem",
"icaremos": "icaremos",
"icares": "icardes",
"icaria": "icariam",
"icariam": "icariam",
"icaríamos": "icaríamos",
"icarias": "icaríeis",
"icaríeis": "icaríeis",
"icarmos": "icarmos",
"icas": "icais",
"icasse": "icassem",
"icásseis": "icásseis",
"icassem": "icassem",
"icássemos": "icássemos",
"icasses": "icásseis",
"icaste": "icastes",
"icastes": "icastes",
"icava": "icavam",
"icavam": "icavam",
"icávamos": "icávamos",
"icavas": "icáveis",
"icáveis": "icáveis",
"ico": "icamos",
"icou": "icaram",
"ique": "iquem",
"iquei": "icamos",
"iqueis": "iqueis",
"iquem": "iquem",
"iquemos": "iquemos",
"iques": "iqueis"}


# vbFICARsing = verbo ficar conjugacao singular
vbFICARsing = {"ica": "ica",
"icado": "icado",
"icai": "ica",
"icais": "icas",
"icam": "ica",
"icamos": "iquei",
"icando": "icando",
"icar": "icar",
"icara": "icara",
"icará": "icará",
"icaram": "icou",
"icáramos": "icara",
"icarão": "icará",
"icaras": "icaras",
"icarás": "icarás",
"icardes": "icares",
"icarei": "icarei",
"icareis": "icarás",
"icáreis": "icaras",
"icarem": "icar",
"icaremos": "icarei",
"icares": "icares",
"icaria": "icaria",
"icariam": "icaria",
"icaríamos": "icaria",
"icarias": "icarias",
"icaríeis": "icarias",
"icarmos": "icar",
"icas": "icas",
"icasse": "icasse",
"icásseis": "icasses",
"icassem": "icasse",
"icássemos": "icasse",
"icasses": "icasses",
"icaste": "icaste",
"icastes": "icaste",
"icava": "icava",
"icavam": "icava",
"icávamos": "icava",
"icavas": "icavas",
"icáveis": "icavas",
"ico": "ico",
"icou": "icou",
"ique": "ique",
"iquei": "iquei",
"iqueis": "iques",
"iquem": "ique",
"iquemos": "ique",
"iques": "iques"}



# vbESTARsing = verbo estar conjugacao plural
vbESTARplur = {"stá": "stão",
"stado": "stado",
"stai": "stai",
"stais": "stais",
"stamos": "stamos",
"stando": "stando",
"stão": "stão",
"star": "starem",
"stará": "starão",
"starão": "starão",
"starás": "stareis",
"stardes": "stardes",
"starei": "staremos",
"stareis": "stareis",
"starem": "starem",
"staremos": "staremos",
"stares": "stardes",
"staria": "stariam",
"stariam": "stariam",
"staríamos": "staríamos",
"starias": "staríis",
"staríis": "staríis",
"starmos": "starmos",
"stás": "stais",
"stava": "stavam",
"stavam": "stavam",
"stávamos": "stávamos",
"stavas": "stáveis",
"stáveis": "stáveis",
"steja": "stejam",
"stejais": "stejais",
"stejam": "stejam",
"stejamos": "stejamos",
"stejas": "stejais",
"steve": "stiveram",
"stive": "stivemos",
"stivemos": "stivemos",
"stiver": "stiverem",
"stivera": "stiveram",
"stiveram": "stiveram",
"stivéramos": "stivéramos",
"stiveras": "stivéreis",
"stiverdes": "stiverdes",
"stivéreis": "stivéreis",
"stiverem": "stiverem",
"stiveres": "stiverdes",
"stivermos": "stivermos",
"stivesse": "stivessem",
"stivésseis": "stivésseis",
"stivessem": "stivessem",
"stivéssemos": "stivéssemos",
"stivesses": "stivésseis",
"stiveste": "stivestes",
"stivestes": "stivestes",
"stou": "stamos"}


# vbESTARsing = verbo estar conjugacao singular
vbESTARsing = {"stá": "stá",
"stado": "stado",
"stai": "stá",
"stais": "stás",
"stamos": "stou",
"stando": "stando",
"stão": "stá",
"star": "star",
"stará": "stará",
"starão": "stará",
"starás": "starás",
"stardes": "stares",
"starei": "starei",
"stareis": "starás",
"starem": "star",
"staremos": "starei",
"stares": "stares",
"staria": "staria",
"stariam": "staria",
"staríamos": "staria",
"starias": "starias",
"staríis": "starias",
"starmos": "star",
"stás": "stás",
"stava": "stava",
"stavam": "stava",
"stávamos": "stava",
"stavas": "stavas",
"stáveis": "stavas",
"steja": "steja",
"stejais": "stejas",
"stejam": "steja",
"stejamos": "steja",
"stejas": "stejas",
"steve": "steve",
"stive": "stive",
"stivemos": "stive",
"stiver": "stiver",
"stivera": "stivera",
"stiveram": "steve",
"stivéramos": "stivera",
"stiveras": "stiveras",
"stiverdes": "stiveres",
"stivéreis": "stiveras",
"stiverem": "stiver",
"stiveres": "stiveres",
"stivermos": "stiver",
"stivesse": "stivesse",
"stivésseis": "stivesses",
"stivessem": "stivesse",
"stivéssemos": "stivesse",
"stivesses": "stivesses",
"stiveste": "stiveste",
"stivestes": "stiveste",
"stou": "stou"}



# vbSERplur = verbo ser conjugacao plural
vbSERplur = {u"é": "são",
"era": "eram",
"eram": "eram",
"éramos":"éramos",
"eras":"éreis",
"éreis":"éreis",
"és": "sois",
"foi": "foram",
"fomos": "fomos",
"for": "forem",
"fora": "foram",
"foram": "foram",
"fôramos": "fôramos",
"foras": "fôreis",
"fordes": "fordes",
"fôreis": "fôreis",
"forem": "forem",
"fores": "fordes",
"formos": "formos",
"fosse": "fossem",
"fôsseis": "fôsseis",
"fossem": "fossem",
"fôssemos": "fôssemos",
"fosses": "fôsseis",
"foste": "fostes",
"fostes": "fostes",
"fui": "fomos",
"são": "são",
"sê": "sede",
"sede": "sede",
"seja": "sejam",
"sejais": "sejais",
"sejam": "sejam",
"sejamos": "sejamos",
"sejas": "sejais",
"sendo": "sendo",
"ser": "serem",
"será": "serão",
"serão": "serão",
"serás": "sereis",
"serdes": "serdes",
"serei": "seremos",
"sereis": "sereis",
"serem": "serem",
"seremos": "seremos",
"seres": "serdes",
"seria": "seriam",
"seriam": "seriam",
"seríamos": "seríamos",
"serias": "seríeis",
"seríeis": "seríeis",
"sermos": "sermos",
"sido": "sido",
"sois": "sois",
"somos": "somos",
"sou": "somos"}

# vbSERsing = verbo ser conjugacao singular
vbSERsing = {u"é":"é",
"era": "era",
"eram": "era",
"éramos": "era",
"eras": "eras",
"éreis": "eras",
"és":"és",
"foi": "foi",
"fomos": "fui",
"for": "for",
"fora": "fora",
"foram": "foi",
"fôramos": "fora",
"foras": "foras",
"fordes": "fores",
"fôreis": "foras",
"forem": "for",
"fores": "fores",
"formos": "for",
"fosse": "fosse",
"fôsseis": "fosses",
"fossem": "fosse",
"fôssemos": "fosse",
"fosses": "fosses",
"foste": "foste",
"fostes": "foste",
"fui": "fui",
"são":"é",
"sê": "sê",
"sede": "sê",
"seja": "seja",
"sejais": "sejas",
"sejam": "seja",
"sejamos": "seja",
"sejas": "sejas",
"sendo": "sendo",
"ser": "ser",
"será": "será",
"serão": "será",
"serás": "serás",
"serdes": "seres",
"serei": "serei",
"sereis": "serás",
"serem": "ser",
"seremos": "serei",
"seres": "seres",
"seria": "seria",
"seriam": "seria",
"seríamos": "seria",
"serias": "serias",
"seríeis": "serias",
"sermos": "ser",
"sido": "sido",
"sois":"és",
"somos": "sou",
"sou": "sou"}



# aavbSEF = Participio de verbos abundantes precedidos de SER ESTAR e FICAR
aavbSEF = {"aceitad":"aceit",
"acendid":"aces",
"benzid":"bent",
"elegid":"eleit",
"expressad":"express",
"exprimid":"express",
"expulsad":"expuls",
"extinguid":"extint",
"fal":"falad",
"ganhad":"ganh",
"gastad":"gast",
"imergid":"imers",
"imprimid":"impress",
"limpad":"limp",
"matad":"mort",
"morrid":"mort",
"pagad":"pag",
"pegad":"peg",
"prendid":"pres",
"salvad":"salv",
"soltad":"solt",
"submergid":"submers",
"suspendid":"suspens"}


# aavbTH = Participio de verbos abundantes precedidos de TER e HAVER
aavbTH = {"aceit":"aceitad",
"aces":"acendid",
"bent":"benzid",
"eleit":"elegid",
"express":"expressad",
"express":"exprimid",
"expuls":"expulsad",
"extint":"extinguid",
"fal":"falad",
"ganh":"ganhad",
"gast":"gastad",
"imers":"imergid",
"impress":"imprimid",
"limp":"limpad",
"mort":"matad",
"mort":"morrid",
"pag":"pagad",
"peg":"pegad",
"pres":"prendid",
"salv":"salvad",
"solt":"soltad",
"submers":"submergid",
"suspens":"suspendid"}


# acvbSEF = Participio de verbos abundantes precedidos de SER ESTAR e FICAR
acvbSEF = set(["aceitad", "acendid", "benzid", "elegid",
"expressad", "exprimid", "expulsad", "extinguid", "ganhad", "gastad", "fal", "imergid",
"imprimid", "limpad", "matad", "morrid", "pagad", "pegad", "prendid", "salvad",
"soltad", "submergid", "suspendid"])


# acvbTH = Participio de verbos abundantes precedidos de TER e HAVER
acvbTH = set(["aceit", "aces", "bent", "eleit",
"express", "express", "expuls", "extint", "fal", "ganh", "gast", "imers", "impress",
"limp", "mort", "mort", "pag", "peg", "pres", "salv", "solt", "submers",
"suspens"])


# Palavras compostas, plural - singular - publicos-alvo ,  arquivos-fonte
tCompostops = set(["abarés-tucura", "abatis-timbaí", "abelhas-cachorro", "abelhas-criança",
"abelhas-europa", "abelhas-flor", "abelhas-irapuá", "abelhas-irapuã",
"abelhas-limão", "abelhas-mãe", "abelhas-mosquito", "abelhas-ouro",
"abelhas-rainha", "abelhas-uruçu", "abioranas-guta", "abiuranas-guta",
"acácias-angico", "acácias-parassol", "açafrões-palhinha", "açaís-catinga",
"açaís-chumbo", "acajus-catinga", "acampamentos-base", "acarás-aiá",
"acarás-apuã", "acarás-bandeira", "acarás-bararuá", "acarás-bererê",
"acarás-bobo", "acarás-camaleão", "acarás-diadema", "acarás-disco",
"acarás-fantasma", "acarás-ferro", "acarás-fumaça", "acarás-fuso",
"acarás-manaçaravé", "acarás-mocó", "acarás-papagaio", "acarás-pataquira",
"acarás-pinima", "acarás-prata", "acarás-salema", "acarás-saveiro",
"acarás-tilápia", "acarás-topete", "acaris-cacimba", "acaris-chicote",
"acaris-espada", "acaris-laranja", "acaris-lima", "aços-carbono",
"aços-liga", "açordas-mona", "açúcares-cande", "açúcares-cândi",
"açúcares-pedra", "acutis-tapuia","ágnus-dei","águas-belense",
"águas-formosense","águas-mato","águas-mel","águas-morna",
"águas-mornense","águas-tinta","águas-vermelhense", "agulhas-mãe",
"agulhas-padrão", "agulhões-bandeira", "agulhões-trombeta", "aipos-rábano",
"alas-pivô", "albacoras-bandolim", "algodões-colódio", "algodões-cravo",
"algodões-macaco", "algodões-pólvora", "alhos-rocambole", "almoços-debate",
"aluna-modelo", "alunas-modelo", "aluno-modelo", "alunos-modelo",
"amarelos-canário", "amarelos-cidrão", "amarelos-cobalto", "amarelos-enxofre",
"amarelos-laranja", "amarelos-limão", "amarelos-oca", "amarelos-ocre",
"amarelos-oiro", "amarelos-ouro", "amarelos-palha", "amarelos-topázio",
"amendoeiras-coca", "amoreiras-cipó", "amoreiras-tataíba", "amostras-tipo",
"amperes-espira", "amperes-giro", "amperes-hora", "amperes-segundo",
"amperes-volta", "ampérios-espira", "ampérios-giro", "ampérios-hora",
"ampérios-segundo", "ampérios-volta", "anambés-pitiú", "anambés-pombo",
"anambés-sol", "ananás-selvagem", "ananás-silvestre", "andins-palmeira",
"andirobas-jareuá", "andores-império", "angelins-araroba", "angelins-coco",
"angelins-pedra", "angelins-rosa", "angicos-barbatimão", "angicos-cangalha",
"angicos-cedro", "angicos-rosa", "angicos-surucucu", "angonas-puíta",
"anis-doce", "anis-estrelado", "anis-verde", "animais-flor",
"animais-máquina", "animais-planta", "anjos-viola", "anos-base",
"anos-calendário", "anos-luz", "anúncios-sanduíche", "araçás-congonha",
"araçás-cotão", "araçás-lima",
"araçás-pedra", "araçás-rosa",
"araçaris-banana", "araçaris-minhoca", "araiás-garapa", "aranhas-bode",
"aranhas-caranguejeira", "aranhas-caranguejo", "araras-canindé", "araras-chauanauá",
"araras-macau", "araras-pitiú", "ararutas-palmeira", "araticuns-apê",
"araticuns-caca", "araticuns-catinga", "araticuns-cortiça", "araticuns-pitaiá",
"áreas-alvo","áreas-cliente", "arecas-bambu", "arecas-banguá",
"areias-manteiga", "aroeiras-mucuri", "aroeiras-salso", "Arquivos-fonte", "arquivos-fonte",
"arquivos-texto", "arraias-acauã", "arraias-boró", "arraias-cocal",
"arraias-papagaio", "arraias-santo", "arrozes-agulha", "arrozes-cateto",
"arrudas-parietária","árvores-mãe","árvores-vaca", "asas-delta",
"asilos-escola", "assis-brasiliense", "assuntos-chave", "aterros-barragem",
"atividades-fim", "atividades-meio", "atos-show","átomos-grama",
"atuns-bandolim", "auxílios-alimentação", "Auxílios-Alimentação", "auxílios-creche",
"auxílios-doença", "auxílios-enfermidade", "auxílios-funeral", "auxílios-maternidade",
"auxílios-moradia", "auxílios-natalidade", "auxílios-paletó", "auxílios-reclusão",
"auxílios-saúde", "auxílios-transporte", "aves-elefante", "aves-lira",
"aviões-correio", "aviões-espião", "aviões-foguete", "aviões-laboratório",
"aviões-tanque", "azeitonas-rei", "azereiros-pado", "azuis-cobalto",
"azuis-faiança", "azuis-ferrete", "azuis-pavão", "azuis-piscina",
"azuis-pombinho", "azuis-seda", "azuis-turquesa", "azulões-boia",
"bacus-pedra", "bacuparis-cipó", "bacuraus-pituí", "bacuraus-tesoura",
"bacurubus-ficheira", "badejos-ferro", "badejos-fogo", "badejos-mira",
"badejos-padre", "badejos-sabão", "badejos-saltão", "badejos-sangue",
"bagres-ariaçu", "bagres-bandeira", "bagres-cachola", "bagres-cacumo",
"bagres-caiacoco", "bagres-cambeja", "bagres-cangatá", "bagres-catinga",
"bagres-fita", "bagres-gaivota", "bagres-gonguito", "bagres-guri",
"bagres-leilão", "bagres-mandi", "bagres-mandim", "bagres-morcego",
"bagres-papai", "bagres-sapipoca", "bagres-sapo", "bagres-sari",
"bagres-surubim", "bagres-urutu", "bagres-veludo", "baiacus-areia",
"baiacus-caixão", "baiacus-cofre", "baiacus-dondom", "baiacus-espinho",
"baiacus-franguinho", "baiacus-gorajuba", "baiacus-graviola", "baiacus-guaiama",
"baiacus-guaiamá", "baiacus-guarajuba", "baiacus-panela", "baiacus-pinima",
"baiacus-pininga", "baiacus-pintado", "baleias-dente", "baleias-espadarte",
"baleias-minke", "baleias-pamonha", "baleias-piloto", "baleias-sei",
"balões-piloto", "balões-sonda", "bálsamos-caboriba", "bálsamos-coral",
"bambus-balde", "bambus-bengala", "bambus-cana", "bambus-taquara",
"banás-muela", "bananas-figo", "bananas-gabu", "bananas-inajá",
"bananas-macã", "bananas-maçã", "bananas-mãe", "bananas-mânea",
"bananas-menina", "bananas-najá", "bananas-ouro", "bananas-pacova",
"bananas-pacová", "bananas-pão", "bananas-pedra", "bananas-prata",
"bananas-timbó", "bananeirinhas-zebra", "banhos-maria", "bárbus-sumatrano",
"bárbus-ticto", "barcas-farol", "barcaças-cisterna", "barcos-armadilha",
"barcos-automóvel", "barcos-bomba", "barcos-dragão", "barcos-farol",
"barcos-luz", "barreiras-chave", "batatas-aipo", "batatas-atum",
"batatas-ceará", "batatas-cenoura", "batatas-cogumelo", "batatas-ovo",
"batatas-salsa", "baúnas-fogo", "begônias-bengala", "begônias-sangue",
"beijos-pirá", "beijus-moqueca", "beijus-poqueca", "berdelhos-feijão",
"berimbaus-viola", "besourinhos-ametista", "besouros-bola", "besouros-mangangá",
"besouros-rinoceronte", "bias-fortense", "bichas-alfinete", "bichos-bolo",
"bichos-cadela", "bichos-engenheiro", "bichos-folha", "bichos-lixeiro",
"bichos-palha", "bichos-pau", "bichos-preguiça", "bichos-trem",
"bichos-tutu", "bicos-pimenta", "bicos-será", "bicudas-mangalô",
"bicudos-maquiné", "bicudos-pimenta", "Biocódigos-fonte", "biocódigos-fonte",
"bisnagas-bengala",
"boais-bonifácio", "boais-natura", "bodiões-batata", "bodiões-lamego",
"bodiões-papagaio", "bodiões-sabonete", "bodiões-tucano", "bois-bumbá",
"bois-calemba", "bois-calumba", "bois-cavalo", "bois-culemba",
"bois-espácio", "bois-espaço", "bombas-foguete", "bombas-granada",
"bombas-parafuso", "bombas-relógio", "bombas-turbina", "bonecos-palito",
"bonitos-cachorro", "bonitos-canário", "bonitos-cavala", "borás-boi",
"borás-cavalo", "borboletas-azuis-seda", "borboletas-carijó", "borboletas-coruja",
"borboletas-espelho", "borboletas-folha", "borboletas-imperador", "borboletas-jandaia",
"borboletas-oitenta", "borboletas-rubi", "botos-cinza", "botos-tucuxi",
"botões-polaina", "brancos-parreira", "bredos-caruru", "bredos-malho",
"bredos-mama", "bredos-rabaça", "breus-jauaricica", "breus-sucuruba",
"brigues-barca", "brigues-escuna", "bugios-labareda", "buritis-palito",
"buritis-palmito", "cabas-beiju", "cabas-camaleão", "cabas-mutuca",
"cabaças-purunga", "cabaços-grogojó", "cabaços-marimba", "cabeças-duras-prego",
"cabeças-duras-relógio", "cabeças-pitanga", "cabos-guia", "cabras-cabriola",
"cacaus-jacaré", "cacaus-maracujá", "cacaueiros-laranja", "cacaueiros-suriname",
"cacaueiros-trindade", "cacaueiros-venezuela", "cachões-porango", "cachorros-rabeca",
"cactos-melão", "cactos-ouriço", "cactos-rosa", "Cades-Barneia",
"cafés-bucha", "cafés-cabeça", "cafés-caneca", "cafés-chifre",
"cafés-concerto", "cafés-concha", "cafés-conosco", "cafés-escolha",
"cafés-moca", "cafés-teatro", "caixas-bomba", "cajás-manga",
"calorias-grama", "camas-beliche", "camadas-limite", "camaleões-caba",
"camaleões-ferro", "camarões-ferro", "camarões-lagosta", "camarões-lixo",
"camarões-pistola", "camarões-rosa", "camarões-sossego", "cambarás-pitanga",
"cambarás-rosa", "cambuís-bala", "caminhões-baú", "caminhões-betoneira",
"caminhões-bomba", "caminhões-fossa", "caminhões-pipa", "caminhões-tanque",
"camomilas-catinga", "camorins-sovela", "campos-altense", "campos-geraiense",
"campos-novense", "camurins-corcunda", "camurins-flecha", "camurins-sovela",
"camurins-tapa", "camurins-ticopá", "canas-capim", "canas-manteiga",
"cananas-capeta", "canários-assobio", "canários-baeta", "candirus-caju",
"canelas-abacate", "canelas-anhuíba", "canelas-bagre", "canelas-baraúna",
"canelas-batalha", "canelas-batata", "canelas-bezerro", "canelas-bibiru",
"canelas-braúna", "canelas-caixeta", "canelas-catiguá", "canelas-cedro",
"canelas-cravo", "canelas-ferrugem", "canelas-fogo", "canelas-funcho",
"canelas-goiaba", "canelas-goiacá", "canelas-gosma", "canelas-guacá",
"canelas-guaiacá", "canelas-guaicá", "canelas-imbuia", "canelas-inhaíba",
"canelas-inhaúba", "canelas-jacu", "canelas-japu", "canelas-limão",
"canelas-louro", "canelas-marmelada", "canelas-massapê", "canelas-mate",
"canelas-merda", "canelas-mescla", "canelas-muriá", "canelas-murici",
"canelas-oiti", "canelas-pimenta", "canelas-poca", "canelas-pororoca",
"canelas-prego", "canelas-rapadura", "canelas-rosa", "canelas-samambaia",
"canelas-seiva", "canelas-tapinha", "canelas-tapinhoã", "canelas-tatu",
"caneleiras-cravo", "canelinhas-ubá", "Canetas-fonte", "canetas-fonte", "canetas-tinteiro",
"cangas-rosa", "cangulos-baié", "cangulos-rei", "cangurus-rato",
"canhões-arpão", "canhões-revólver", "canjicas-piruruca", "canoas-rosa",
"capas-capote", "capas-gorja", "capas-palhiça", "capas-palhoça",
"capas-pele", "capas-saia", "capas-zero", "capacetinhos-cinza",
"capelas-carapuça", "capins-alpiste", "capins-amonjeaba", "capins-andacaá",
"capins-andrequicé", "capins-angola", "capins-angolão", "capins-apé",
"capins-araguaí", "capins-arroz", "capins-atana", "capins-aveia",
"capins-balça", "capins-bambu", "capins-bambuzinho", "capins-bandeira",
"capins-barata", "capins-batatal", "capins-bengala", "capins-bobó",
"capins-bobô", "capins-bolota", "capins-burrão", "capins-cabaiú",
"capins-caiana", "capins-calandrínia", "capins-camalote", "capins-camelão",
"capins-canarana", "capins-caninha", "capins-canudinho", "capins-canutão",
"capins-capivara", "capins-cati", "capins-catinga", "capins-cebola",
"capins-cevada", "capins-chuvisco", "capins-cidreira", "capins-cidrilho",
"capins-cidró", "capins-cocorobó", "capins-colchão", "capins-cololó",
"capins-colônia", "capins-colonião", "capins-coqueirinho", "capins-cortesia",
"capins-elefante", "capins-elimo", "capins-espartilho", "capins-esteira",
"capins-estrela", "capins-fartura", "capins-ferro", "capins-flabelo",
"capins-flecha", "capins-flor", "capins-foice", "capins-frecha",
"capins-gafanhoto", "capins-gengibre", "capins-gordura", "capins-grama",
"capins-guaiamum", "capins-guatemala", "capins-guiné", "capins-jaçapé",
"capins-jaguaré", "capins-jaraguá", "capins-jasmim", "capins-jaú",
"capins-jerivá", "capins-junco", "capins-lanceta", "capins-leque",
"capins-limão", "capins-lixa", "capins-maçambará", "capins-malota",
"capins-manga", "capins-marajó", "capins-marmelada", "capins-marreca",
"capins-massapé", "capins-membeca", "capins-meruquiá", "capins-miçanga",
"capins-milhã", "capins-mina", "capins-mium", "capins-mori",
"capins-morotó", "capins-mourão", "capins-murim", "capins-mururu",
"capins-napier", "capins-natal", "capins-navalha", "capins-navalheira",
"capins-onça", "capins-orvalho", "capins-palma", "capins-palmeira",
"capins-panasco", "capins-pancuã", "capins-pangola", "capins-papuã",
"capins-paraturá", "capins-penacho", "capins-peripomongo", "capins-piqui",
"capins-pirimembeca", "capins-pororó", "capins-quicé", "capins-quicuio",
"capins-quicuiú", "capins-uamá", "capins-vassoura", "capins-veludo",
"capins-vetiver", "capins-zaranza", "capitães-chico", "capras-capela",
"carás-coco", "carás-cova", "carás-doce", "carás-inhame",
"caraguatás-acanga", "caraguatás-piteira", "caramujos-boi", "caramujos-cascudo",
"caramujos-maçã", "caramujos-pião", "caramujos-tigela", "carandás-muriti",
"carandás-piranga", "caranguejos-aranha", "caranguejos-baioneta", "caranguejos-uçá",
"carapanãs-ora", "carapanãs-pinima", "cardeais-bispo", "cardeais-diácono",
"cardeais-violeta", "cardos-acanto", "cardos-ananá", "cardos-bosta",
"cardos-limão", "cardos-mandacaru", "cardos-melão", "cardos-palmatória",
"cardos-sanguinho", "cargas-chumbo", "cargas-limite", "carlos-alvense",
"carlos-barbosense", "carlos-chaguense", "carlos-gomense", "carlos-quinto",
"carmins-fibrina", "carmins-nacará", "carmins-nata", "carmópolis-mineiro",
"carrapatos-estrela", "carrapatos-fogo", "carrapatos-pólvora", "carros-bomba",
"carros-chefe", "carros-conceito", "carros-dormitório", "carros-guincho",
"carros-leito", "carros-madrinha", "carros-pipa", "carros-restaurante",
"carros-salão", "carros-suporte", "carros-tanque", "carruagens-salão",
"cartas-bilhete", "cartas-bomba", "cartas-compromisso", "cartas-contrato",
"cartas-fiança", "cartas-piloto", "cartas-telegrama", "cartões-couro",
"cartões-fibra", "cartões-ficha", "cartões-janela", "cartões-palha",
"cartões-pedra", "cartões-resposta", "carumas-cacueme", "carvalhos-anão",
"carvalhos-cortiça", "carvalhos-roble", "carvalhos-rosa", "carvões-tição",
"casas-abrigo", "casas-piano", "casas-torre", "cascas-açucena",
"cascas-pereira", "cascudos-espada", "cascudos-espinho", "cascudos-lima",
"cascudos-viola", "casos-limite", "castanhas-caiaté", "castanhas-jurera",
"castanhas-sapucaia", "castanhos-barril", "castanhos-pipa", "catas-altense",
"catálogos-dicionário", "catanhões-tesoura", "cateteres-balão", "causas-chave",
"cavalas-aipim", "cavalas-canga", "cavalas-canguçu", "cavalinhos-mosca",
"cavalos-boi", "cavalos-fantasma", "cavalos-hora", "cavalos-mosca",
"cavalos-vapor", "cebolas-barrão", "cebolas-cecém", "cedros-batata",
"cedros-canela", "cedros-canjerana", "cedros-faia", "cedros-mogno",
"cedros-rosa", "células-alvo", "células-filha", "células-flama",
"células-guarda", "células-mãe", "células-mestre", "células-ovo",
"células-semente", "células-tronco", "Células-Tronco", "centáureas-calcítrapa",
"centigramas-força", "cerigados-sabão", "cerigados-tapoã", "chás-biriba",
"chás-mate", "chás-pucá", "chapas-base", "chapéus-boneca",
"chapéus-chile", "chapéus-coco", "chapéus-cone", "chapéus-panamá",
"charadas-anagrama", "charadas-metagrama", "chefes-banda", "chefes-barra",
"chefes-pala", "cheques-borracha", "cheques-calção", "chicharros-calabar",
"chicharros-cavala", "chicos-angu", "chimarritas-balão", "chorões-salgueiro",
"chorrós-boi", "ciclídeos-borboleta", "cidades-cogumelo", "cidades-dormitório",
"cidades-estado", "cidades-Estado", "cidades-fantasma", "cidades-jardim",
"cidades-laboratório", "cidades-museu", "cidades-satélite", "cidades-sede",
"Cidades-Sede", "cigarras-cobra", "cigarras-rainha", "cigarrinhas-empoasca",
"cilindros-eixo", "cimentos-amianto", "cinamis-cocaína", "cinemas-olho",
"cipós-abacate", "cipós-alho", "cipós-almécega", "cipós-azougue",
"cipós-barroca", "cipós-brasil", "cipós-caatinga", "cipós-caboclo",
"cipós-caçaú", "cipós-café", "cipós-camarão", "cipós-caniana",
"cipós-caravuí", "cipós-carijó", "cipós-carneiro", "cipós-catiguá",
"cipós-catinga", "cipós-chumbo", "cipós-cobra", "cipós-cola",
"cipós-corimbó", "cipós-correlha", "cipós-cravo", "cipós-cruz",
"cipós-cumaruapé", "cipós-curapé", "cipós-curimbó", "cipós-curura",
"cipós-cururu", "cipós-escada", "cipós-escova", "cipós-florão",
"cipós-guaçu", "cipós-guibé", "cipós-guira", "cipós-icica",
"cipós-imbé", "cipós-jabutá", "cipós-jagube", "cipós-jarrinha",
"cipós-jatobá", "cipós-jiboia", "cipós-língua", "cipós-mata",
"cipós-morcego", "cipós-paratudo", "cipós-paré", "cipós-pau",
"cipós-prata", "cirurgiãs-dentista", "cirurgiões-dentista", "claros-escuro",
"clavárias-pilão", "clientes-aluno", "clientes-alvo", "cobertas-abrigo",
"cobertas-salão", "cobras-alcatifa", "cobras-capelo", "cobras-capim",
"cobras-cascavel", "cobras-chicote", "cobras-cipó", "cobras-coral",
"cobras-escada", "cobras-escorpião", "cobras-espada", "cobras-facão",
"cobras-furtacor", "cobras-jabuti", "cobras-madeira", "cobras-norato",
"cobras-papagaio", "cobras-pilão", "cobras-tigre", "cobras-topete",
"cobras-veado", "cobras-vidro", "cocadas-puxa", "cochonilhas-farinha",
"cochonilhas-pimenta", "cochonilhas-vírgula", "cocos-bingolê", "cocos-catulé",
"cocos-desafio", "cocos-feijão", "cocos-gavião", "cocos-guariroba",
"cocos-indaiá", "cocos-lanho", "cocos-macaúba", "cocos-naiá",
"cocos-peneruê", "cocos-pindoba", "Códigos-fonte", "códigos-fonte",
"códigos-objeto",
"coelhos-borboleta", "colas-esmalte", "coletores-tronco", "Coletores-Tronco",
"comédias-bailado", "comédias-balé", "comédias-drama", "comédias-farsa",
"comédias-pastelão", "conceitos-chave", "congonhas-cachimbo", "congonhas-caixeta",
"congros-rosa", "contas-salário", "copais-cauri", "copaíbas-angelim",
"copaíbas-jutaí", "copaíbas-marimari", "copos-balão", "coqueiros-açaí",
"coqueiros-acunã", "coqueiros-airi", "coqueiros-anaiá", "coqueiros-anão",
"coqueiros-aracuri", "coqueiros-ariri", "coqueiros-babunha", "coqueiros-bacaba",
"coqueiros-bocaiuva", "coqueiros-brejaúba", "coqueiros-buçu", "coqueiros-buriti",
"coqueiros-caiaué", "coqueiros-carandá", "coqueiros-carandaí", "coqueiros-catulé",
"coqueiros-cauí", "coqueiros-dicuri", "coqueiros-guariroba", "coqueiros-guriri",
"coqueiros-iri", "coqueiros-jataí", "coqueiros-javari", "coqueiros-jeribá",
"coqueiros-macho", "coqueiros-marajaíba", "coqueiros-mumbaca", "coqueiros-naiá",
"coqueiros-piaçaba", "coqueiros-pissandó", "coqueiros-tarampaba", "coqueiros-tucum",
"coqueiros-tucumã", "coquinhos-babá", "corais-boia", "corais-cérebro",
"corcorocas-jurumirim", "corcorocas-sargo", "cornos-godinho", "coroas-crísti",
"corvinas-linha", "costas-abaixo", "costas-acima", "costas-arriba",
"couros-cromo", "couves-caraíba", "couves-cavaleiro", "couves-coco",
"couves-cravinho", "couves-flor", "couves-manteiga", "couves-nabiça",
"couves-nabo", "couves-penca", "couves-rábano", "couves-rábão",
"couves-repolho", "couves-rosa", "couves-saboia", "crateras-lago",
"craterelos-cornucópia", "cravos-cabecinha", "cravos-giroflé", "cravos-minhardise",
"cravos-renda", "crepes-cetim", "crepes-georgete", "crepes-marroquim",
"crianças-problema", "criciúmas-cipó", "cristas-ervilha", "cristais-paulistense",
"critérios-base", "crocorocas-jurumim", "cruzadores-escola", "cruzes-diabo",
"cuacos-malagueta", "cuias-maracá", "cumarus-ferro", "cúmulos-cirro",
"cúmulos-estrato", "cúmulos-nimbo", "cúmulos-vulcão", "curiangos-tesoura",
"curiangos-tesourão", "curicas-bacabal", "curicas-urubu", "curimatãs-pacu",
"currais-novense", "curuás-iuquira", "curuás-pixuna", "Dados-fonte", "dados-fonte",
"datas-base", "datas-marco", "decretos-lei", "deus-dará",
"Deus-Homem", "deus-mercado", "dias-coelhense", "dias-multa",
"diabos-alma", "diamantes-alençon", "diamantes-benefício", "diamantes-bristol",
"diamantes-chapa", "diamantes-fazenda", "diamantes-matura", "diamantes-mesa",
"diamantes-rosa", "diamantes-tabla", "dianteiros-centro", "dinas-centímetro",
"dinamarquês-espanhol", "dinheiros-papel", "discos-madre", "discos-mãe",
"discos-mestre", "dispositivos-alvo", "documentos-base", "dois-abrilense",
"dois-correguense", "dois-irmanense", "dois-irmãosense", "dois-lajeadense",
"dois-riachense", "dois-riense", "dois-vizinhense", "domingos-costense",
"domingos-martinense", "domingos-paisense", "domingos-pratense", "donzelas-fogo",
"dores-campense", "dores-guanhãesense", "dores-indaiaense", "dores-macabuense",
"dores-marmeladense", "dores-paraibunense", "dores-turvense", "dores-vitoriense",
"doses-padrão", "dourados-carapau","és-nordeste","és-noroeste",
"és-sudeste","és-sudoeste","és-sueste", "edifícios-garagem",
"elementos-chave", "elias-faustense", "embaúbas-manga", "embiras-sebo",
"empregos-alvo", "empresas-alvo", "empresas-mãe", "endossos-caução",
"endossos-mandato","ênulas-campana", "episódios-piloto", "equivalentes-grama",
"ervas-abelha", "ervas-agrimônia", "ervas-aleluia", "ervas-alheira",
"ervas-almíscar", "ervas-almiscareira", "ervas-ambrósia", "ervas-andorinha",
"ervas-aranha", "ervas-arroz", "ervas-azebre", "ervas-baleeira",
"ervas-belida", "ervas-besteira", "ervas-bezerra", "ervas-bicha",
"ervas-borboleta", "ervas-botão", "ervas-brico", "ervas-bruxa",
"ervas-bugre", "ervas-canária", "ervas-candeia", "ervas-capitão",
"ervas-carocha", "ervas-carpinteiro", "ervas-carvalhinha", "ervas-caúna",
"ervas-cavalinha", "ervas-chumbo", "ervas-coentrinha", "ervas-coivinha",
"ervas-colégio", "ervas-contraveneno", "ervas-couvinha", "ervas-crina",
"ervas-cruz", "ervas-dedal", "ervas-sal", "escamas-farinha",
"escamas-peixe", "escamas-prego", "escaravelhos-bosteiro", "escolas-cantina",
"escolas-modelo", "escorpiões-vinagre", "escovinhas-margarida", "espaços-imagem",
"espaços-objeto", "espaços-tempo", "espadas-sabre", "espadartas-maca",
"espadartes-meca", "espécies-alvo", "espécies-tipo", "espinelas-rubi",
"espinélios-almandina", "espinheiros-cachorro", "espinheiros-cambra", "espinheiros-carneiro",
"espinhos-guabiru", "espinhos-rosa", "espodumênios-esmeralda", "estacas-prancha",
"estacas-testemunha", "Estados-ilha", "Estados-nação", "Estados-Nação",
"estados-providência", "estados-tampão", "estratégias-chave", "estratos-cirro",
"estratos-cúmulo", "estratos-nimbo", "estrelas-guia", "estrelinhas-ametista",
"estudos-piloto", "eucaliptos-limão", "eucaliptos-pimenta", "euclides-cunhense",
"exibições-teste", "experiências-piloto", "facheiros-cardeiro", "facheiros-jamacaru",
"fadas-madrinha", "fadigas-corrosão", "falcões-caburé", "falcões-gerifalte",
"falcões-tagarote", "falsas-ervas-mate", "falsos-paratudo", "falsos-paus-brasil",
"farias-britense", "farinhas-farelo", "fatos-macaco", "fatores-chave",
"favas-assaria", "favas-barbatimão", "favas-belém", "favas-café",
"favas-caranguejo", "favas-chicote", "favas-coceira", "favas-contra",
"favas-cumandália", "favas-oliá", "favas-oró", "favas-pixurim",
"favas-terra", "favas-tonca", "fazendas-modelo", "fechos-relâmpago",
"feijões-adzuki", "feijões-alfanje", "feijões-amendoim", "feijões-aspargo",
"feijões-azeite", "feijões-azuqui", "feijões-bacamarte", "feijões-baetão",
"feijões-baru", "feijões-batata", "feijões-besugo", "feijões-cabacinha",
"feijões-café", "feijões-canadá", "feijões-capão", "feijões-careta",
"feijões-carito", "feijões-carrapatinho", "feijões-carrapato", "feijões-carumbé",
"feijões-cavalinho", "feijões-chicote", "feijões-china", "feijões-chocolate",
"feijões-chumbinho", "feijões-coco", "feijões-congo", "feijões-corda",
"feijões-cutelinho", "feijões-enxofre", "feijões-espada", "feijões-espadinho",
"feijões-farinha", "feijões-favona", "feijões-fidalgo", "feijões-figueira",
"feijões-flor", "feijões-frade", "feijões-fradinho", "feijões-glória",
"feijões-guando", "feijões-gurutuba", "feijões-imperador", "feijões-jiritana",
"feijões-lablabe", "feijões-lagartixa", "feijões-macáçar", "feijões-macúndi",
"feijões-mãezinha", "feijões-mancanha", "feijões-manteiga", "feijões-mucuna",
"feijões-mula", "feijões-mungo", "feijões-oró", "feijões-papa",
"feijões-pical", "feijões-pinheirinho", "feijões-pombinha", "feijões-sapota",
"feijões-selim", "feijões-soja", "feijões-telefone", "feijões-tepari",
"feijões-tonca", "feijões-tropeiro", "feijões-vassoura", "feijões-veludo",
"feijões-vinha", "feijoeiros-lagartixa", "feixas-fradinho", "ferrais-maçã",
"ferrais-tâmara", "ferros-ferrita", "ferros-gusa", "ferros-ilmenita",
"ferros-pau", "festas-homenagem", "fetos-cipó", "fícus-benjamim",
"figos-flor", "figos-porco", "figueiras-babosa", "figueiras-benjamim",
"figueiras-bispo", "figueirinhas-hera", "figuras-chave", "filhos-família",
"filmes-catástrofe", "filmes-evento", "filmes-família", "filmes-piloto",
"filodendros-imbé", "filtros-prensa", "fios-sardinha", "firmas-membro",
"Firmas-Membro", "fitas-crepe", "fitas-demo", "flos-santório",
"flores-camarão", "flores-cunhense", "focas-harpa", "focas-leopardo",
"focas-monge", "foguetes-alvo", "foguetes-sonda", "folhas-flor",
"folhas-formiga", "folhas-galo", "folhas-lixa", "forças-tarefa",
"formigas-brasa", "formigas-cabaça", "formigas-caiapó", "formigas-cascavel",
"formigas-cobra", "formigas-conga", "formigas-correição", "formigas-ferro",
"formigas-leão", "formigas-mandioca", "formigas-maravilha", "formigas-oncinha",
"formigas-pixixica", "formigas-quenquém", "formigas-rainha", "formigas-raspa",
"formigas-saúva", "formigas-vespa", "fórmulas-grama", "fósseis-guia",
"fósseis-índice", "fossos-diamante", "frases-título", "freis-bode",
"freis-jorge", "frevos-abafo", "frevos-canção", "frevos-coqueiro",
"frevos-ventania", "fronteiras-faixa", "fronteiras-linha", "fronteiras-zona",
"frutas-barrete", "frutas-bolsa", "frutas-caurim", "frutas-gargantilha",
"frutas-gundra", "frutas-manga", "frutas-manila", "frutas-pão",
"frutas-tirilha", "frutas-trilha", "fruteiras-pão", "fulanos-dos-anzóis-carapuça",
"fungadores-onça", "furriéis-rosa", "gás-estufa", "gás-mil",
"gafanhotos-bandeira", "gafanhotos-cobra", "galos-bandeira", "galos-brejo",
"gansos-cisne", "gansos-patola", "garças-cinza", "garças-socoí",
"gargantas-rubi", "garotas-propaganda", "garotos-propaganda", "garoupas-bexiga",
"garoupas-chita", "garoupas-gato", "garrinchas-trovão", "gases-estufa",
"gatos-lagária", "gatos-maracajá", "gatos-paul", "gatos-sapato",
"gatos-tigre", "gaviões-caboclo", "gaviões-caburé", "gaviões-caripira",
"gaviões-cavã", "gaviões-covã", "gaviões-formiga", "gaviões-fumaça",
"gaviões-inajé", "gaviões-indaié", "gaviões-lavadeira", "gaviões-mateiro",
"gaviões-padre", "gaviões-pato", "gaviões-peneira", "gaviões-pinhé",
"gaviões-pombo", "gaviões-puva", "gaviões-quiriquiri", "gaviões-rapina",
"gaviões-relógio", "gaviões-sauveiro", "gaviões-tanató", "gaviões-tesoira",
"gaviões-tesoura", "gaviões-tinga", "gaviões-vaqueiro", "gerânios-hera",
"gerânios-rosa", "gessos-cré", "gessos-estuque", "gestos-chave",
"gilas-caiota", "globulárias-turbito", "goitacás-jacurito", "goitacás-mopi",
"golfinhos-piloto", "gomas-laca", "gomas-resina", "goranás-timbó",
"gotas-serena", "gralhas-cancã", "gramas-cravo", "gramas-forquilha",
"gramas-lanceta", "gramas-orvalho", "gramas-sapo", "gramas-seda",
"graminhas-seda", "granadas-foguete", "gravatas-borboleta", "gravatás-borboleta",
"gravatás-canga", "gravatás-zebra", "grilos-cantor", "grilos-toupeira",
"groçaís-azeite", "grupos-base", "guajanás-timbó", "guajarás-abio",
"guajarás-cinza", "guajarás-curumim", "guajarás-poca", "guajarás-preto",
"guapuís-cipó", "guarás-tereba", "guarabus-amarelo", "guarabus-batata",
"guarabus-cebola", "guaranás-timbó", "guardas-espelho", "guardas-marinha",
"Guarulhos-Nazaré", "guatambus-marfim", "guatambus-peroba", "guatambus-rosa",
"guerras-relâmpago", "guirás-acangatara", "herás-herahum",
"herpes-virótico", "herpes-zoster", "herpes-zóster", "homens-aranha",
"homens-bomba", "homens-chave", "homens-gol", "homens-máquina",
"Homens-Máquina", "homens-massa", "homens-morcego", "homens-mosca",
"homens-orquestra", "homens-padrão", "homens-pássaro", "homens-peixe",
"homens-placa", "homens-sanduíche", "homens-tipo", "horas-atividade",
"horas-aula", "horas-homem", "horas-luz", "horas-máquina",
"hortelãs-boi", "hortelãs-pimenta", "hospitais-escola","íbis-branco",
"íbis-sagrado", "icacorés-catinga", "idades-limite", "ideias-chave",
"ideias-mãe", "ideias-número", "ilhas-Estado", "Ilhas-estado",
"Ilhas-Estado", "inajaranas-envira", "inambus-galinha", "inambus-relógio",
"inambus-saracuíra", "inamus-galinha", "inamus-relógio", "inamus-saracuíra",
"indivíduos-alvo", "informações-chave", "ingás-caixão", "ingás-cipó",
"ingás-cururu", "ingás-feijão", "ingás-ferradura", "ingás-macaco",
"ingás-veludo", "inglês-russo", "inhambus-galinha", "inhambus-peba",
"inhambus-relógio", "inhambus-saracuíra", "inhambus-tona", "inhames-cigarra",
"inhames-farinha", "inhames-gudu", "inhames-monstro", "inhames-nambu",
"inhames-taro", "inhames-tívoli", "inhames-zambuco", "instrumentos-chave",
"interdições-relâmpago","íons-grama", "ipês-batata", "ipês-boia",
"ipês-mandioca", "ipês-tabaco", "ipecacuanhas-poaia", "ituís-cavalo",
"ituís-pinima", "jãs-vaz", "jaburus-moleque", "jabutis-aperema",
"jabutis-araconha", "jabutis-carumbé", "jabutis-jurema", "jabutis-machado",
"jabutis-tucumã", "jacamins-copejuba", "jacamins-copetinga", "jacamins-cunhã",
"jacamins-cupejuba", "jacamins-cupijuba", "jacamins-cupitinga", "jacamins-una",
"jacarandás-banana", "jacarandás-cabiúna", "jacarandás-caroba", "jacarandás-caviúna",
"jacarandás-ferro", "jacarandás-pitanga", "jacarandás-rosa", "jacarés-cacau",
"jacarés-catinga", "jacarés-copaíba", "jacarés-coroa", "jacarés-curua",
"jacarés-curulana", "jacarés-paguá", "jacus-estalo", "jacus-molambo",
"jacus-porco", "jacus-queixada", "jacus-taquara", "jacundás-coroa",
"jacundás-pinima", "jacundás-piranga", "jades-califórnia", "jaguás-arroz",
"jaguaretés-pixuna", "jaguaretês-pixuna", "jambeiros-rosa", "jambos-chá",
"jambos-rosa", "japiins-xexéu", "japins-soldado", "japus-gamela",
"jararacas-cruzeira", "jararacas-taiá", "jararacuçus-tapete", "jararacuçus-tipiti",
"jararacuçus-topete", "jardins-escola", "jarrinhas-arraia", "jarrinhas-monstro",
"jasmins-bogari", "jasmins-cambraia", "jasmins-catavento", "jasmins-estrela",
"jasmins-junquilho", "jasmins-lacre", "jasmins-laranja", "jasmins-manacá",
"jasmins-manga", "jasmins-mangueira", "jasmins-manteiga", "jasmins-murta",
"jasmins-pipoca", "jasmins-porcelana", "jasmins-sombra", "jasmins-trombeta",
"jasmins-vapor", "jataís-mondé", "jataís-mosquito", "jataís-pororó",
"jataís-pororoca", "jatobás-capão", "jenipapos-rosa", "jequitibás-cedro",
"jequitibás-rosa", "jesus-amparense", "jesus-galhense", "jesus-itabapoanense",
"jesus-lapense", "jimbas-muxi", "joanas-guenza", "joaninhas-guenza",
"jogos-treino", "joões-balão", "joões-minhoca", "joões-porca",
"jóqueis-clube", "juás-uva", "juritis-pupu", "jutaís-café",
"jutaís-catinga", "jutaís-pororoca", "lagartas-aranha", "lagartas-compasso",
"lagartas-elasmo", "lagartas-rosca", "lagos-mar", "lambaris-prata",
"lambdas-zero", "lápis-lazúli", "lápis-tinta", "laranjas-aperu",
"laranjas-baía", "laranjas-cravo", "laranjas-lima", "laranjas-melancia",
"laranjas-natal", "laranjas-pera", "laranjas-toranja", "laranjeiras-melancia",
"laranjeiras-pera", "laranjeiras-toranja", "larvas-alfinete", "larvas-arame",
"lés-nordeste", "lés-noroeste", "lés-oeste", "lés-sudeste",
"lés-sudoeste", "lés-sueste", "legaços-cozinho", "legações-cabecinho",
"lenços-papel", "leos-metro", "letras-guia", "libras-massa",
"libras-peso", "licenças-maternidade", "licenças-paternidade", "licenças-prêmio",
"limas-faca", "limas-serra", "limões-canudo", "limões-cravo",
"línguas-alvo", "Línguas-fonte", "línguas-fonte",
"línguas-mãe", "línguas-padrão",
"linguados-lixa", "linguagens-fonte", "linguagens-objeto", "linhas-base",
"linhas-bloco", "linhos-cânhamo", "lírios-ferro", "lírios-tigre",
"lírios-tuicara", "livros-caixa", "livros-cassete", "livros-razão",
"livros-texto", "lobos-guará", "lobos-tigre", "loés-nordeste",
"loés-noroeste", "loés-sudoeste", "loés-sueste", "loireiros-rosa",
"loiros-abacate", "loiros-batata", "loiros-bosta", "loiros-cânfora",
"loiros-cerejo", "loiros-cravo", "loiros-faia", "loiros-inhamuí",
"loiros-mamorim", "loiros-nhamuí", "loiros-pimenta", "loiros-puxuri",
"loiros-rosa", "loiros-tamancão", "loiros-tamanco", "lótus-amarelo",
"lótus-americano", "lótus-azul", "lótus-índico", "lótus-vermelho",
"loureiros-cerejo", "loureiros-rosa", "louros-abacate", "louros-aritu",
"louros-batata", "louros-bosta", "louros-cânfora", "louros-cerejo",
"louros-cravo", "louros-inhamuí", "louros-mamorim", "louros-mamuí",
"louros-nhamuí", "louros-pimenta", "louros-puxuri", "louros-rosa",
"louros-salgueiro", "louros-tamancão", "louros-tim", "lúcias-lima",
"lugres-barca", "lugres-escuna", "lugres-iate", "lugres-patacho",
"luís-alvense", "luís-antoniense", "luís-cacheiro", "luís-caixeiro",
"luís-correense", "luís-correiense", "luís-dominguense", "luís-gomense",
"luís-gonzaguense", "luís-quatorze", "luís-quinze", "luís-teixeira",
"luís-treze", "luís-vianense", "lumens-bilirrubina", "lumens-hora",
"lúmenes-bilirrubina", "lúmenes-hora", "luzes-cuco", "maçãs-cuca",
"macacos-aranha", "macacos-castanha", "macacos-cipó", "macacos-esquilo",
"macacos-itapoá", "macacos-leão", "macacos-patrona", "macacos-prego",
"madeiras-seringa", "madres-cravo", "mães-tia", "magalhães-almeidense",
"magalhães-baratense", "magnólias-tulipa", "maguaris-pistola", "maguaris-pistolão",
"maitacas-curica", "malas-arte", "malvas-algodão", "malvas-cajuçara", "malvas-carrapicho",
"malvas-flor", "malvas-piranga", "malvas-reloginho", "malvas-relógio",
"malvas-rosa", "malvas-sardinha", "malvas-taquari", "mamas-cachorra",
"mamas-cadela", "mamães-dengue", "mamatos-cúmulo", "mamoeiros-melão",
"manas-chica", "mandatos-tampão", "mandis-bagre", "mandis-bandeira",
"mandis-boi", "mandis-cachorro", "mandis-cambaia", "mandis-casaca",
"mandis-fidalgo", "mandis-irajá", "mandis-moela", "mandis-palha",
"mandis-urutu", "mandiocas-tucumã", "mandioquinhas-salsa", "manés-jacá",
"mangas-rosa", "mangabas-ovo", "manganês-preto", "mangarás-pena",
"manguaris-pistola", "manguaris-pistolão", "mangues-canaponga", "mangues-canoé",
"mangues-ciriúba", "mangues-guapirá", "mangues-saraíba", "mangues-seriúba",
"manjares-grude", "mantos-xale", "mapas-múndi", "máquinas-caixão",
"máquinas-ferramenta", "maracujás-mamão", "maracujás-melão", "maracujás-periquito",
"maracujás-peroba", "maracujás-sururuca", "maracujás-suspiro", "marajás-piranga",
"marcas-símbolo", "marcelas-margaça", "marchas-rancho", "marchas-regresso",
"marcos-parentense", "marias-cadeira", "marias-candelária", "marias-farinha",
"marias-fede", "marias-ferrugem", "marias-fia", "marias-fumaça",
"marias-irré", "marias-isabel", "marias-lecre", "marias-leite",
"marias-mucanguê", "marias-rapé", "marias-rita", "marias-rosa",
"marias-teresa", "maribondos-cavalo", "maribondos-chapéu", "maribondos-tatu",
"marimbondos-beju", "marimbondos-cachorro", "marimbondos-cavalo", "marimbondos-chapéu",
"marimbondos-mangangá", "marimbondos-tatu", "mariposas-espelito", "mariposas-imperador",
"mariscos-faca", "mariscos-pedra", "mariscos-tatu", "marmeladas-cachorro",
"marques-loureiro", "marquês-valencense", "marrecas-ananaí", "marrecas-apaí",
"marrecas-arrebio", "marrecas-cabocla", "marrecas-carijó", "marrecas-cricri",
"marrecas-espelho", "marrecas-toicinho", "marrecas-toucinho", "marrecas-tururu",
"marrecos-mandarim", "martins-gil", "martins-gravata", "matérias-espelho",
"mateus-lemense", "mateus-sulino", "matias-barbosense", "matias-cardosense",
"matias-olimpiense", "matos-capim", "matos-salema", "matrizes-gaveta",
"mês-luz", "melodias-tenor", "melões-caboclo", "melões-cantalupo",
"melros-cachoeiro", "mendes-pimentelense", "meninas-aranha", "meninos-aranha",
"meninos-prodígio", "menos-mal", "menos-mau", "menos-valia",
"mercados-alvo", "merendibas-bagre", "meridianos-origem", "mesas-ninho",
"meses-luz", "mestres-empada", "mestres-escama", "mestres-escola",
"mestres-terreiro", "mestres-vinagre", "metais-tipo", "metros-esteno",
"metros-hipercinese", "metros-hiperestesia", "metros-newton", "miaos-iao",
"micholes-quati", "micos-estrela", "micos-leão", "microgramas-força",
"milandos-homem", "milhãs-garça", "milhos-alho", "milhos-alpiste",
"milhos-arroz", "milhos-dente", "milhos-pipoca", "milhos-rei",
"milhos-sorgo", "miligramas-força", "miligramas-minuto", "miligramas-segundo",
"minas-bocainense", "minas-butiaense", "minas-entrerriense", "minas-generalense",
"minas-geralense", "minas-jardinense", "minas-madrense", "minas-mimosense",
"minas-montalegrense", "minas-novense", "minas-vicentino", "mineiros-tieteense",
"minerais-gema", "minerais-índice", "minutas-padrão", "minutos-luz",
"mirins-preguiça", "mirindibas-bagre", "mirindibas-rosa", "mirorós-caramuru",
"mirras-mofina", "mochos-carijó", "mochos-diabo", "mocoins-caca",
"modelos-base", "moedas-ouro", "moedas-papel", "moldes-caldeira",
"moléculas-grama", "molhes-molhe", "monjolos-ferro", "montes-altense",
"montes-clarense", "montes-clarino", "morcegos-buldogue", "morcegos-pescador",
"morcegos-vampa", "morcegos-vampiro", "mormólices-folha", "mororós-cipó",
"morosos-cigano", "moscas-aranha", "moscas-bicheira", "moscas-oura",
"moscas-piroga", "moscas-vareja", "mosquitos-berne", "mosquitos-palha",
"mosquitos-pernilongo", "mosquitos-pólvora", "mosquitos-prego", "mosquitos-remela",
"mostardeiras-violeta", "motores-canhão", "motores-foguete", "muçambés-catinga",
"muçandeiras-sangue", "mucutas-veado", "mudanças-chave", "mulheres-aranha",
"mulheres-gato", "mulheres-homem", "mulheres-objeto", "municípios-sede",
"muricis-pitanga", "murtas-índia", "murtas-pimenta", "mururés-carrapatinho",
"mururés-pajé", "musgos-piraná", "mutuns-cavalo", "nambus-relógio",
"nambus-saracuíra", "narcisos-ramilhete", "navios-aríete", "navios-cisterna",
"navios-correio", "navios-escola", "navios-escolta", "navios-farol",
"navios-hospital", "navios-laboratório", "navios-oficina", "navios-sonda",
"navios-tanque", "navios-tênder", "navios-transporte", "negras-mina",
"negros-aça", "negros-mina", "neurônios-chave", "neurônios-espelho",
"neves-paulistense", "nhambus-guçu", "nhambus-relógio", "nhambus-saracuíra",
"nhôs-chico", "nogueiras-pecã", "normas-padrão", "nozes-macadâmia",
"nozes-pecã", "números-índice", "obaranas-rato", "objetivos-chave",
"objetos-alvo", "objetos-partícula", "objetos-símbolo", "oés-nordeste",
"oés-noroeste", "oés-sudoeste", "oés-sueste", "ogãs-ilu",
"ohms-centímetro", "oiros-fio", "oiros-pigmento", "oiros-pimenta",
"oiticicas-cica", "olhos-cia", "onças-boi", "ondas-maré",
"ondas-piloto","ônibus-camarote", "opalas-arlequim", "opalas-fígado",
"opalas-musgo","óperas-bufa","óperas-rock", "orçamentos-programa",
"ostraceiros-pirupiru", "otis-xavante", "ouriços-escudo", "ouros-pigmento",
"ouros-pimenta", "pacamões-cuíca", "pacamões-niquim", "pacos-caatinga",
"pacos-catinga", "pacovas-catinga", "pacovas-sororoca", "pacus-apó",
"pacus-chidão", "pacus-oerudá", "pacuás-sororoca", "padinas-pavão",
"pães-bengala", "pães-canoa", "paes-landinense", "país-alvo",
"país-chave", "país-irmão", "pais-lemense", "país-membro",
"país-parceiro", "país-sede", "painas-cipó", "painas-ganga",
"países-alvo", "países-chave", "países-ilha", "Países-Ilha",
"países-sede", "pajés-sacaca", "palavras-chave", "palavras-fantasma",
"palavras-filtro", "palavras-testemunho", "palavras-valise", "paletós-saco",
"palhas-fede", "pálios-cirro", "pálios-cúmulo", "palmas-christi",
"palmeiras-açaí", "palmeiras-andim", "palmeiras-areca", "palmeiras-bambu",
"palmeiras-buri", "palmeiras-camedória", "palmeiras-catolé", "palmeiras-catulé",
"palmeiras-chifre", "palmeiras-cipó", "palmeiras-dendê", "palmeiras-iará",
"palmeiras-jaraúva", "palmeiras-jiçara", "palmeiras-juçara", "palmeiras-laca",
"palmeiras-leque", "palmeiras-palmira", "palmeiras-piaçaba", "palmeiras-princesa",
"palmeiras-pupunha", "palmitos-juçara", "pâmpanos-pacu", "pampos-arabebéu",
"pampos-aracanguira", "pampos-tibilo", "pampos-tibiro", "panapanás-piranga",
"panapanás-tauá", "panos-baju", "panos-família", "panos-piloto",
"papagaios-curau", "papagaios-goiaba", "papagaios-poaeiro", "papagaios-urubu",
"papéis-alumínio", "papéis-arroz", "papéis-bíblia", "papéis-bond",
"papéis-bule", "papéis-carbono", "papéis-cartão", "papéis-carvão",
"papéis-celofane", "papéis-chupão", "papéis-filtro", "papéis-ilustração",
"papéis-jornal", "papéis-ladrão", "papéis-manilha", "papéis-manteiga",
"papéis-moeda", "papéis-ofício", "papéis-oxford", "papéis-ozalide",
"papéis-palha", "papéis-paquete", "papéis-pergaminho", "papéis-pigmento",
"papéis-pluma", "papéis-porcelana", "papéis-registro", "papéis-relâmpago",
"papéis-tabaco", "papéis-tela", "papéis-título", "paraparás-peúva",
"pardais-boi", "pardos-mato", "pardos-mosto", "pardos-violeta",
"pargos-rosa", "pargos-tereso", "Paris-Dakar", "Paris-Tóquio",
"parreiras-caapeba", "partículas-espelho", "parus-frade", "parus-jandaia",
"parus-papagaio", "parus-soldado", "parus-tucano", "passageiros-quilômetro",
"passarinhos-moscardo", "pássaros-alfaiate", "pássaros-angu", "pássaros-boi",
"pássaros-frade", "pássaros-martelo", "pássaros-maú", "pássaros-mosca",
"pássaros-pêndulo", "pássaros-pretos-soldado", "pássaros-sino", "pássaros-sol",
"pássaros-soldado", "pássaros-trombeta", "pássaros-voa", "passes-passe",
"pastos-aranha", "pastos-borla", "patos-arminho", "patos-coral",
"patos-ferrão", "patos-ganso", "patos-lavanco", "patos-mandarim",
"patos-minense", "patos-pataca", "patos-pomerânia", "patuás-balaio",
"paus-água", "paus-alazão", "paus-alecrim", "paus-alho",
"paus-ama", "paus-amendoim", "paus-andrade", "paus-arara",
"paus-azeite", "paus-azeitona", "paus-azevim", "paus-bala",
"paus-bálsamo", "paus-barro", "paus-bicho", "paus-boia",
"paus-bosta", "paus-brasil", "paus-breu", "paus-cabra",
"paus-cachorro", "paus-cadeira", "paus-cadela", "paus-caixão",
"paus-caixeta", "paus-candeia", "paus-canela", "paus-caninana",
"paus-canoa", "paus-canudo", "paus-capitão", "paus-cardoso",
"paus-carga", "paus-carne", "paus-carrapato", "paus-carvão",
"paus-caseco", "paus-catinga", "paus-cavalo", "paus-caxique",
"paus-cera", "paus-cetim", "paus-cobra", "paus-colher",
"paus-concha", "paus-conta", "paus-coral", "paus-costa",
"paus-cravo", "paus-crioulo", "paus-cruz", "paus-cumaru",
"paus-danejo", "paus-dumo", "paus-duno", "paus-espeto",
"paus-espinha", "paus-esteira", "paus-facho", "paus-farinha",
"paus-fava", "paus-fede", "paus-feitiço", "paus-ferro",
"paus-fígado", "paus-formiga", "paus-forquilha", "paus-fumaça",
"paus-funcho", "paus-gamela", "paus-gasolina", "paus-gonçalo",
"paus-guiné", "paus-homem", "paus-jangada", "paus-jantar",
"paus-jerimu", "paus-jerimum", "paus-jiboia", "paus-jiçara",
"paus-josé", "paus-lacre", "paus-lágrima", "paus-laranja",
"paus-leite", "paus-lepra", "paus-limão", "paus-lírio",
"paus-lixa", "paus-maçã", "paus-macaco", "paus-macucu",
"paus-mamão", "paus-manteiga", "paus-marfim", "paus-milho",
"paus-mocó", "paus-moeda", "paus-mondé", "paus-monjolo",
"paus-morcego", "paus-mucumbi", "paus-mungongue", "paus-obá",
"paus-osso", "paus-ová", "paus-pacobala", "paus-papel",
"paus-paraíba", "paus-pedra", "paus-pente", "paus-pepu",
"paus-perdiz", "paus-pernambuco", "paus-pérola", "paus-pêssego",
"paus-piaçaba", "paus-piaçava", "paus-pimenta", "paus-pombo",
"paus-ponte", "paus-porco", "paus-purga", "paus-quássia",
"paus-quiabo", "paus-quizemba", "paus-rainha", "paus-rei",
"paus-ripa", "paus-rosa", "paus-sabão", "paus-sangue",
"paus-santos-macaco", "paus-seringa", "paus-serrote", "paus-tartaruga",
"paus-tatu", "paus-tenente", "paus-terra", "paus-terras-jundiaí",
"paus-trombeta", "paus-tucano", "paus-urubu", "paus-vaca",
"paus-vala", "paus-vintém", "paus-viola", "paus-violeta",
"paus-visgo", "paus-zebra", "pavios-china", "pavões-bode",
"paxiúbas-manjerona", "pés-terra", "pés-vela", "peças-chave",
"peças-revólver", "pedras-bazar", "pedras-bronze", "pedras-canta",
"pedras-curuba", "pedras-escrófula", "pedras-ferro", "pedras-fita",
"pedras-fogo", "pedras-foguense", "pedras-gema", "pedras-grão",
"pedras-ímã", "pedras-jade", "pedras-judeu", "pedras-lar",
"pedras-lipse", "pedras-lispe", "pedras-louça", "pedras-mármore",
"pedras-mendobi", "pedras-olar", "pedras-pombo", "pedras-rosa",
"pedras-sabão", "pedras-salamandra", "pedras-soda", "pedras-ume",
"pedras-umes-caá", "pegas-boi", "pegas-flor", "peixes-agulha",
"peixes-aipim", "peixes-alecrim", "peixes-anjo", "peixes-aranha",
"peixes-azeite", "peixes-balgado", "peixes-beta", "peixes-boi",
"peixes-borboleta", "peixes-boto", "peixes-briga", "peixes-bruxa",
"peixes-cabra", "peixes-cachimbo", "peixes-cachorro", "peixes-cadela",
"peixes-cana", "peixes-canga", "peixes-cão", "peixes-capim",
"peixes-carago", "peixes-carlim", "peixes-carta", "peixes-cavaleiro",
"peixes-cavalo", "peixes-charuto", "peixes-cigarra", "peixes-cobra",
"peixes-cobrelo", "peixes-coelho", "peixes-coiro", "peixes-congo",
"peixes-corneta", "peixes-corno", "peixes-correio", "peixes-couro",
"peixes-cravo", "peixes-diabo", "peixes-disco", "peixes-doutor",
"peixes-drácula", "peixes-elefante", "peixes-escama", "peixes-escorpião",
"peixes-esmargal", "peixes-espada", "peixes-espinho", "peixes-fila",
"peixes-flor", "peixes-folha", "peixes-frade", "peixes-galo",
"peixes-gato", "peixes-homem", "peixes-imperador", "peixes-japoneses-corneta",
"peixes-lagarto", "peixes-leão", "peixes-lenha", "peixes-leque",
"peixes-lima", "peixes-lixa", "peixes-lobo", "peixes-lua",
"peixes-macaco", "peixes-machado", "peixes-madama", "peixes-martelo",
"peixes-moela", "peixes-morcego", "peixes-mulher", "peixes-naire",
"peixes-palmito", "peixes-papagaio", "peixes-paraíso", "peixes-pau",
"peixes-pedra", "peixes-pena", "peixes-pérsico", "peixes-pescador",
"peixes-pica", "peixes-piça", "peixes-piloto", "peixes-pimenta",
"peixes-piolho", "peixes-pombo", "peixes-porco", "peixes-prata",
"peixes-prego", "peixes-rabo", "peixes-raposo", "peixes-rato",
"peixes-rei", "peixes-roda", "peixes-sabão", "peixes-sanguessuga",
"peixes-sapo", "peixes-serra", "peixes-sol", "peixes-soldado",
"peixes-sono", "peixes-tamarindo", "peixes-tigre", "peixes-tordo",
"peixes-trombeta", "peixes-vaca", "peixes-vampiro", "peixes-vela",
"peixes-ventosa", "peixes-zebra", "peixes-zorro", "penas-base",
"penas-sargo", "pepinos-castanha", "pequiás-café", "pequiás-cetim",
"pequiás-mamão", "pequiás-mamona", "pequiás-marfim", "pequiás-pedra",
"peras-água", "peras-amerim", "peras-amorim", "peras-figo",
"peras-fita", "peras-manteiga", "peras-marmelo", "peras-marquesa",
"peras-noiva", "peras-pão", "peras-pérola", "peras-prata",
"peras-rosa", "peramelas-narigudo", "percevejos-pernilongo", "pereiras-bergamota",
"pereiras-vaqueta", "perfis-diagrama", "períodos-chave", "periquitos-estrela",
"periquitos-gangarra", "periquitos-rei", "periquitos-tapuia", "periquitos-testinha",
"periquitos-tuim", "periquitos-urubu", "periquitos-vassoura", "perobas-café",
"perobas-cetim", "perobas-poca", "perobas-rosa", "perobas-tambu",
"perobas-tigre", "perpétuas-rosa", "personagens-título", "pescadas-aratanha",
"pescadas-banana", "pescadas-cachorro", "pescadas-cambuci", "pescadas-cambucu",
"pescadas-carvoa", "pescadas-corvina", "pescadas-cutinga", "pescadas-dentão",
"pescadas-enchova", "pescadas-foguete", "pescadas-goiva", "pescadas-jaguara",
"pescadas-manteiga", "pescadas-marmota", "pescadas-rosa", "pescadas-sofia",
"pescadinhas-goete", "pescadinhas-marmota", "pesos-fórmula", "pesos-galo",
"pesos-mosca", "pesos-pena", "pesos-pesado", "pesos-pluma",
"pessoas-chave", "petos-cavalinho", "petos-formigueiro", "pianos-bar",
"picais-polho", "picões-uçu", "pictogramas-signo", "pictogramas-sinal",
"picuís-caboclo", "pimentas-albarrã", "pimentas-apuã", "pimentas-cereja",
"pimentas-cumari", "pimentas-cumarim", "pimentas-lambari", "pimentas-malagueta",
"pimentas-pitanga", "pimentas-sino", "pimentões-maçã", "pindás-siririca",
"pingos-pingo", "pinguins-imperador", "pinhas-ata", "pinheiros-aiacauite",
"pinheiros-casquinha", "pinheiros-larício", "pinhões-paraguai", "pinos-osso",
"pintas-caldeira", "pintas-cardeira", "pintas-cardim", "pintas-cuia",
"pintas-ferreiro", "pintos-cordeiro", "piões-madeira", "pirás-andirá",
"pirás-bandeira", "pirás-batata", "pirás-inambu", "pirás-tamanduá",
"pirás-tapioca", "pirás-tatu", "pirajás-pupunha", "pirangas-saíra",
"piranhas-cachorro", "piranhas-caju", "piranhas-mapará", "pires-albuquerquense",
"pires-ferreirense", "pires-riense", "píretros-partênio", "pirucaias-ubeba",
"piscos-ribeiro", "pistões-coleira", "pitangas-açu", "pitangas-pimenta",
"pitangas-traíra", "pitanguás-açu", "pitiás-café", "pitiús-açu",
"pitos-cobra", "pitos-formigueiro", "pitombas-açu", "placas-mãe",
"planos-sequência", "plantas-alimento", "plantas-camarão", "plantas-mãe",
"plantas-milho", "plantas-rainha", "plantas-telégrafo", "pneus-balão",
"pós-abdome", "pós-abdômen", "pós-abdominal", "pós-aceleração",
"pós-acordo", "pós-adolescência", "pós-albiano", "pós-almoço",
"pós-alveolar", "pós-anestesia", "pós-anestésico", "pós-apocalíptico",
"pós-apocalítico", "pós-apostólico", "pós-aptiano", "pós-articulado",
"pós-auricular", "pós-barba", "pós-bélico", "pós-bíblico",
"pós-Big", "pós-boca", "pós-braquial", "pós-cabralino",
"pós-cartilagíneo", "pós-cirurgia", "pós-cirúrgico", "pós-classicismo",
"pós-clássico", "pós-cloração", "pós-coito", "pós-colonial",
"pós-combustão", "pós-comicial", "pós-complexo", "pós-comunhão",
"pós-comunismo", "pós-conciliar", "pós-concurso", "pós-condicionar",
"pós-conflito", "pós-consulado", "pós-Copenhague", "pós-crédito",
"pós-crise", "pós-datar", "pós-decisional", "pós-dental",
"pós-dilatação", "pós-diluviano", "pós-dorsal", "pós-dorso",
"pós-doutor", "pós-doutorado", "pós-doutoramento", "pós-dublagem",
"pós-edêmico", "pós-eleitoral", "pós-embrionário", "pós-emissão",
"pós-escolar", "pós-escrito", "pós-espermatogênese", "pós-espermatogenético",
"pós-esqui", "pós-estenótico", "pós-estruturalismo", "pós-evento",
"pós-exercício", "pós-exílico", "pós-exílio", "pós-fato",
"pós-fenício", "pós-fixado", "pós-flandriano", "pós-floração",
"pós-formação", "pós-franquista", "pós-frontal", "pós-ganglionar",
"pós-gênito", "pós-genômico", "pós-glacial", "pós-glaciário",
"pós-graduação", "Pós-Graduação", "pós-graduado", "Pós-Graduado",
"pós-graduando", "pós-graduar", "pós-guerra", "pós-Guerra",
"pós-hipnótico", "pós-hipofisário", "pós-hipófise", "pós-homérico",
"pós-ictal", "pós-impressionismo", "pós-impressionístico", "pós-industrial",
"pós-instalação", "pós-internação", "pós-jogo", "pós-kantianismo",
"pós-kantianístico", "pós-kantiano", "pós-kantismo", "pós-kantístico",
"pós-Kioto", "pós-laboral", "pós-larva", "pós-letal",
"pós-maestrichtiano", "pós-magmático", "pós-mão",
"pós-maturação", "pós-medieval", "pós-médio", "pós-membranoso",
"pós-menopausa", "pós-menopausado", "pós-menopáusico", "pós-mensalão",
"pós-mento", "pós-meridiano", "pós-milagre", "pós-milenarismo",
"pós-militar", "pós-minimalismo", "pós-mitótico", "pós-modernismo",
"Pós-Modernismo", "pós-modernístico", "pós-moderno", "pós-mortal",
"pós-morte", "pós-natal", "pós-nominal", "pós-nuclear",
"pós-nupcial", "pós-occipúcio", "pós-ocular", "pós-operatório",
"pós-opercular", "pós-oral", "pós-orbital", "pós-pago",
"pós-palatal", "pós-palato", "pós-parametrizado", "pós-parnasiano",
"pós-parto", "pós-pasteuriano", "pós-pastoriano", "pós-peitoral",
"pós-perna", "pós-perturbação", "pós-pilocênico", "pós-plioceno",
"pós-positivismo", "pós-prandial", "pós-premiação", "pós-processamento",
"pós-produção", "pós-produtor", "pós-puberal", "pós-puerperal",
"pós-punk", "pós-qualificação", "pós-Quioto", "pós-radical",
"pós-reforma", "pós-revolução", "pós-Revolução", "pós-romano",
"pós-romântico", "pós-romantismo", "pós-seco", "pós-secundário",
"pós-separação", "pós-sequelamento", "pós-siluriano", "pós-simbolismo",
"pós-simbolístico", "pós-sináptico", "pós-sincronia", "pós-sincrônico",
"pós-sincronização", "pós-sincronizar", "pós-sincronizável", "pós-socrático",
"pós-subversivo", "pós-tectônico", "pós-temporada", "pós-temporal",
"pós-teodosiano", "pós-terciário", "pós-textual", "pós-tônico",
"pós-traducional", "pós-transcricional", "pós-transfusional", "pós-translacional",
"pós-tratamento", "pós-trauma", "pós-traumático", "Pós-Traumático",
"pós-vacinação", "pós-vacinal", "pós-védico", "pós-velar",
"pós-venda", "pós-verbal", "pós-vida", "pós-vocálico",
"pós-werberniano", "pós-zonatoso", "poços-caldense", "podridões-castanha",
"poilões-forro", "polcas-mazurca", "polegadas-libra", "pólvoras-calhau",
"pólvoras-dinamite", "pomas-candil", "pombas-cabocla", "pombas-carijó",
"pombas-cascavel", "pombas-espelho", "pombas-juriti", "pombas-pararu",
"pombas-pariri", "pombas-picuí", "pombas-pucaçu", "pombas-rola",
"pombas-trocal", "pombinhas-cascavel", "pombos-anambé", "pombos-andorinha",
"pombos-cambalhota", "pombos-carneau", "pombos-carrier", "pombos-cetim",
"pombos-cetineta", "pombos-correio", "pombos-dragão", "pombos-gravatinha",
"pombos-lua", "pombos-nicobar", "pombos-pega", "pombos-tambor",
"pombos-torcaz", "pombos-trocal", "pombos-trocaz", "pombos-tumbler",
"pontas-cabeça", "pontas-dentro", "pontes-aqueduto", "pontes-canal",
"pontes-gestalense", "pontes-safena", "pontos-chave", "pontos-limite",
"pontos-vogal", "populações-alvo", "porcos-espim", "porcos-espinho",
"porcos-pisco", "porcos-ribeiro", "porcos-veado", "portos-abrigo",
"português-espanhol", "posições-chave", "postos-chave", "potós-pimenta",
"poupas-boubela", "preços-alvo", "pregos-cachorro", "pretos-aça",
"pretos-amarelo", "pretos-azeviche", "pretos-carvão", "priores-vigário",
"prisões-albergue", "processos-chave", "produções-família", "profissionais-padrão",
"Programas-fonte","programas-fonte", "programas-objeto", "projéteis-foguete", "projetos-piloto",
"projetos-tipo", "proteínas-chave", "públicos-alvo", "puçás-coca",
"puçás-poca", "pulas-pula", "pulgões-carmim", "pulseiras-relógio",
"punhais-baioneta", "punhos-punhete", "pupunhas-piranga", "pupunhas-pitanga",
"puracis-iara", "putumuiús-iriribá", "putumujus-iriribá", "quadros-alvo",
"quartzos-fantasma", "quartzos-mórion", "quássias-simaruba", "queijos-cavalo",
"queixas-crime", "quenquéns-caiapó", "questões-chave", "quicês-acica",
"quilogramas-força", "quilogramas-metro", "quilogramas-peso", "quilômetros-passageiro",
"quilovares-hora", "quilovátios-hora", "quilovolts-ampere",
"quilowats-hora", "quinas-calisaia", "quinas-caribé", "quinas-cruzeiro",
"quinas-laranjeira", "quinas-pereira", "quissandas-muxito", "quotas-parte",
"rãs-cachorro", "rãs-manteiga", "rabelos-coelha", "rádios-cassete",
"rádios-gravador", "rádios-vitrola", "raias-arara", "raias-boi",
"raias-borboleta", "raias-chita", "raias-gererera", "raias-jamanta",
"raias-leopardo", "raias-lixa", "raias-maçã", "raias-manteiga",
"raias-prego", "raias-sapo", "raias-ticonha", "raias-viola",
"rainhas-mãe", "raízes-escora", "ratos-calunga", "ratos-canguru",
"ratos-capivara", "ratos-catita", "ratos-chinchila", "ratos-coró",
"ratos-cuandu", "ratos-espinho", "ratos-musgo", "ratos-toró",
"redes-fole", "regiões-continente", "relacionamentos-relâmpago", "relógios-calendário",
"relógios-pulseira", "relógios-vassoura", "reservas-provisão", "reuniões-almoço",
"robalos-aratanha", "robalos-estoque", "robalos-flecha", "robalos-miraguaia",
"roceiros-planta", "rochas-reservatório", "rodas-coroa", "rodas-pagode",
"rodas-vinho", "rolas-bagaço", "rolas-capa", "rolas-cascavel",
"rolas-pajeú", "rolas-pau", "rolhas-espécie", "rolinhas-cascavel",
"romances-relâmpago", "romances-rio", "roncadores-taboca", "rosas-amélia",
"rosas-bilanca", "rosas-chá", "rosas-madeira", "rosas-maravilha",
"rosas-mesquita", "rosas-mosqueta", "rosas-palmeirão", "roxos-terra",
"rubis-almandina", "rubis-balaio", "rubis-califórnia", "rubis-espinela",
"rubis-estrela", "rubis-topázio", "sabiás-anu", "sabiás-barranco",
"sabiás-cachorro", "sabiás-cavalo", "sabiás-coleira", "sabiás-gongá",
"sabiás-gute", "sabiás-laranja", "sabiás-laranjeira", "sabiás-pimenta",
"sabões-dentifrício", "sabres-baioneta", "sabres-punhal", "sachas-cebola",
"sachas-manga", "sachas-rosa", "safiras-estrela", "saganhos-oiro",
"saganhos-ouro", "saguis-carantinga", "saguis-imperador", "saguins-caratinga",
"saguins-imperador", "saís-andorinha", "saís-arara", "saís-papagaio",
"saís-sapucaia", "saias-balão", "saias-calça", "saias-calção",
"salas-cofre", "salários-base", "salários-família", "salários-hora",
"salários-maternidade", "sales-oliveirense", "salmões-rei", "salmões-truta",
"salsas-limão", "salsas-proa", "samambaias-cumaru", "samambaias-mosquito",
"sambas-canção", "sambas-choro", "sambas-enredo", "sambas-exaltação",
"sambas-lenço", "sambas-roda", "sanhaços-coqueiro", "sanhaços-frade",
"sanhaços-pardo", "sanhaçus-coqueiro", "sanhaçus-frade", "santos-dumontense",
"Santos-Guarujá", "sapajus-aurora", "sapos-aru", "sapos-boi",
"sapos-cachorro", "sapos-folha", "sapos-intanha", "sapos-peixe",
"sapos-pipa", "sapos-pulga", "saquis-monge", "saracuras-muirá",
"sarapós-tuvira", "sardas-ágata", "sardinhas-bandeira", "sardinhas-bandeirada",
"sardinhas-facão", "sardinhas-gato", "sardinhas-laje", "sardinhas-maromba",
"sardinhas-prata", "sargos-veado", "sassafrás-amarelo", "sauís-caratinga",
"sauiás-bandeira", "saúnas-rolha", "savanas-parque", "sebos-confeito",
"segundos-luz", "seguros-democracia", "seguros-desemprego", "seguros-garantia",
"seguros-maternidade", "seguros-saúde", "semens-contra", "sêmenes-contra",
"senátus-consulto", "sequestros-relâmpago", "serigados-badejo", "serigados-cherne",
"serigados-sabão", "serigados-tapuã", "serigados-xerre", "seringueiras-chicote",
"seringueiras-itaúba", "seringueiras-tambaqui", "serpentinas-jararaca", "serras-abaixo",
"serras-acima", "serviços-chave", "setores-alvo", "setores-chave",
"sicômoros-figueira", "sigmas-zero", "signos-salomão", "silenes-aurora",
"sílicas-gel", "silvas-framboesa", "simarubas-copaia", "símiles-península",
"sinhás-carandá", "sintomas-chave", "siris-baú", "siris-candeia",
"siris-chita", "siris-patola", "sirigados-badejo", "sirigados-cherne",
"sirigados-sabão", "sirigados-tapuã", "sirigados-xerre", "situações-limite",
"situações-problema", "socós-boi", "sofás-bicama", "sofás-cama",
"solas-sapato", "sorgos-vassoura", "substâncias-traço", "surubins-caparari",
"surubins-chicote", "surubins-lima", "surubins-lira", "surubins-mena",
"surubins-pirambubu", "surucucus-patioba", "surucucus-pindoba", "surucucus-tapete",
"taiás-jararaca", "taiás-jiboia", "taiás-mangarito", "taiaçus-caiaguira",
"taiaçus-carapiá", "tajás-jiboia", "tajás-mangarito", "tajaçus-caiaguira",
"tajaçus-carapiá", "tamanduás-bandeira", "tamanduás-cavalo", "tamanduás-colete",
"tamanduás-jaleco", "tamanduás-macambira", "tambores-onça", "tangerinas-cravo",
"tanques-rede", "taperebás-cedro", "taperibás-cedro", "tapiras-caapora",
"taquaras-bambu", "tarifas-prêmio", "tartaranhões-pigargo", "tartarugas-aligator",
"tartarugas-caixão", "tartarugas-lira", "tatus-bola", "tatus-canastra",
"tatus-cascudo", "tatus-folha", "tatus-galinha", "tatus-mulita",
"tatus-veado", "tatus-vespa", "teatros-revista", "telegramas-vale",
"telhas-canal", "temas-chave", "tendas-abrigo", "tentos-carolina",
"teodolitos-bússola", "terras-mãe", "testes-diagnóstico", "textos-base",
"textos-foguete", "Textos-fonte", "textos-fonte", "textos-legenda", "textos-piloto",
"tias-rosa", "tiés-caburé", "tiés-fogo", "tiés-galo",
"tiés-guaçus-paroara", "tiés-piranga", "tiés-sangue", "tiês-sangue",
"tiés-veludo", "timbós-boticário", "timbós-caá", "timbós-carajuru",
"timbós-catinga", "timbós-cipó", "timbós-macaquinho", "timbós-pau",
"timbós-taturuaia", "timbós-titica", "timbós-urucu", "tinguis-botó",
"tinguis-capeta", "tintas-cão", "tintas-francisca", "tipos-altura",
"tíquetes-alimentação", "tíquetes-refeição", "tíquetes-restaurante", "títulos-chave",
"tobias-barretense", "todos-santense", "tojos-chamusco", "tojos-gatão",
"tojos-gatum", "tojos-gatunha", "tomás-aquinense", "tomás-gonzaguense",
"tomadas-banana", "tomates-cabacinha", "tomates-cajá", "tomates-cereja",
"tomates-chimango", "tomates-groselha", "tomates-pêra", "toneladas-força",
"toneladas-milha", "toneladas-quilômetro", "topázios-baía", "topázios-ouro",
"topázios-palmira", "tordos-pisco", "tornos-revólver", "trás-anteontem",
"traíras-pedra", "traíras-pixuna", "traíras-pixúria", "travessas-calço",
"três-agostense", "três-arroiense", "três-barrense", "três-biquense",
"três-boquense", "três-bracense", "três-cachoeirense", "três-casense",
"três-coroense", "três-coronense", "três-correguense", "três-cruzense",
"três-divisense", "três-estradense", "três-figueirense", "três-forquilhense",
"três-fronteirense", "três-ilhense", "três-irmanense", "três-ladeirense",
"três-lagoano", "três-lagoense", "três-maiense", "três-mariense",
"três-morrense", "três-outubrense", "três-passense", "três-pedrense",
"três-pinheirense", "três-pocense", "três-pontano", "três-pontense",
"três-ranchense", "três-riense", "três-vendense", "tremês-arroxeado",
"tremês-branco", "tremês-preto", "tremês-ribeirinho", "tremês-rijo",
"tremês-ruivo", "trens-bala", "trens-hospital", "trens-tipo",
"trevos-maçaroca", "trigos-arroz", "trigos-barbela", "trincheiras-abrigo",
"tubarões-anequim", "tubarões-baleia", "tubarões-elefante", "tubarões-grelha",
"tubarões-jaguara", "tubarões-limão", "tubarões-martelo", "tubarões-raposa",
"tubarões-tigre", "tubarões-tintureira", "tucanos-boi", "tucanos-cachorrinho",
"tucanos-pacova", "tucumãs-arara", "tucumãs-piranga", "tucumãs-piririca",
"tucumãs-purupuru", "tucuns-marajá", "tucunarés-cipó", "tucunarés-mereçá",
"tuiparas-estrelinha", "tuiués-tipi", "tumbas-satélite", "turbilhos-escadinha",
"turbilhos-roseta", "turistas-padrão", "turmas-piloto", "uaçaís-chumbo",
"ubaias-muxama", "ubaranas-camorim", "ubaranas-jiruma", "ubaranas-rato",
"ubins-rana", "uirás-angu", "umiris-bálsamo", "unguentos-basilicão",
"unguentos-populeão", "unhas-gata", "unidades-base", "universos-bolha",
"universos-ilha", "unués-bolina", "upas-antiar", "upas-tieuté",
"usinas-piloto", "usinas-protótipo", "usuários-chave", "uvas-cão",
"uvas-espim", "uvas-espinha", "uvas-isabel", "uvas-maçã",
"uvas-rei", "uvas-tinta", "vagões-cavalariça", "vagões-cisterna",
"vagões-leito", "vagões-pipa", "vagões-restaurante", "vagões-salão",
"vagões-tanque", "vagões-tremonha", "vales-alimentação", "vales-brinde",
"vales-cultura", "Vales-Cultura", "vales-refeição", "vales-transporte",
"valores-oiro", "valores-ouro", "valores-papel",
"vamos-peneirar", "vars-hora", "vars-horímetro", "variáveis-chave",
"varicelas-zóster", "vátios-hora", "veados-bororó", "veados-camocica",
"veados-canela", "veados-caracu", "veados-cariacu", "veados-foboca",
"veados-garapu", "veados-guatapará", "veados-suaçutinga", "veados-virá",
"velas-luz", "velames-metro", "verbos-suporte", "verdes-abacate",
"verdes-água", "verdes-alface", "verdes-alga", "verdes-alvo",
"verdes-azeitona", "verdes-bandeira", "verdes-bexiga", "verdes-bronze",
"verdes-cinza", "verdes-claro", "verdes-cré", "verdes-escuro",
"verdes-esmeralda", "verdes-flavo", "verdes-gai", "verdes-gaio",
"verdes-garrafa", "verdes-imperial", "verdes-jade", "verdes-limo",
"verdes-limoso", "verdes-malva", "verdes-mar", "verdes-mineral",
"verdes-montanha", "verdes-musgo", "verdes-negro", "verdes-oliva",
"verdes-pavão", "verdes-piscina", "verdes-salsa", "verdes-verdete",
"verdes-viena", "verdes-virgo", "verdeais-branco", "verdeais-cobrançosa",
"verdeais-quebrançosa", "verdeais-rija", "verdeais-tinta", "verdelhos-feijão",
"vermes-arame", "vermes-cabelo", "vermes-chicote", "vermes-leão",
"vermelhos-aricó", "vermelhos-cachorro", "vermelhos-cereja", "vermelhos-cioba",
"vermelhos-henrique", "vermelhos-paramirim", "vermelhos-púrpura", "vermelhos-salmão",
"vermelhos-siriúba", "vernizes-emulsão", "vespas-tatu", "viaturas-munição",
"víboras-alcatifa", "vidoeiros-cerejeira", "vilas-boense", "vimeiros-brózio",
"vinháticos-cabeleira", "vírus-base", "vogas-avante", "volumes-minuto",
"vorás-boi", "votos-vista", "watts-hora", "watts-minuto",
"watts-segundo", "zés-ninguém", "zonas-alvo"])


# Termos compostos com duas palavras iguais
igualTc = set(["agar","ágar", "agora", "assim",
"baco", "baga", "bang", "bango",
"bangu", "bangue", "bate", "bau",
"béu", "bilo", "bique", "bira",
"biri", "bite", "bole", "boles",
"buia", "bula", "bule", "bules",
"bum", "cê", "chã", "chape",
"chás", "chau", "cheira", "chia",
"chica", "chico", "chora", "chuca",
"chupa", "chupas", "cina", "cloque",
"coió", "coiós", "côncavo", "convexo",
"coró", "corre", "corres", "cospe",
"cospes", "cotó", "crás", "cubo",
"cuiú", "curá", "curo", "curre",
"curro", "cuspe", "dão", "dói",
"dois", "dorme", "dormes", "duplo",
"êh", "empurra", "empurras", "esconde",
"escondes", "fecha", "fechas", "ferve",
"fiau", "fila", "fim", "foge",
"foges", "fuco", "fula", "fusca",
"fusco", "ganha", "gris", "guaju",
"gue", "guira", "gumba", "hula",
"ido", "jaga", "jile", "jiles",
"jorro", "lambe", "lemba", "lenga",
"lengue", "lero", "liga", "lio",
"lufa", "luze", "luzes", "mal",
"marche", "mari", "mas", "mata",
"mato", "mela", "menga", "mexe",
"mexes", "miá", "mija", "mijas",
"mio", "mocha", "mochas", "mole",
"molhe", "moxa", "muana", "mutá",
"ningres", "oba", "ogá", "paco",
"pala", "panga", "passa", "passe",
"pega", "pegas", "pia", "pica",
"pico", "pilha", "pinga", "pingas",
"pingo", "pio", "pire", "puco",
"pula", "puxa", "puxas", "quebra",
"queima", "quente", "querê", "quero",
"quete", "quiê", "quinti", "quiri",
"range", "ranges", "raspa", "reco",
"rema", "remas", "reque", "réu",
"ró", "rói", "rola", "rolas",
"ronca", "roque", "rufe", "rufes",
"ruge", "ruges", "safa", "safas",
"sape", "sede", "selá", "serra",
"serras", "sina", "sinas", "siro",
"soque", "sui", "tange", "tanges",
"tape", "teco", "tefe", "tem",
"tepe", "teque", "terém", "tero",
"terra", "téu", "tiã", "tico",
"Tico", "tio", "tique", "toca",
"tole", "toque", "torna", "treme",
"tremes", "troca", "trocas", "truz",
"tuco", "vai", "vais", "vem",
"verga", "vergas", "vim", "vira",
"viras", "xaque", "Xique", "zirra",
"zum"])




# Termos compostos com duas palavras diferentes
tcmpsto = set(["abacaxi-branco", "abacaxi-silvestre", "abacaxis-brancos",
"abacaxis-silvestres", "abaixo-assinado", "abaixo-assinados",
"abajá-mirim", "abelha-africana", "abelha-alemã",
"abelha-amarela", "abelha-brava", "abelha-cachorro",
"abelha-campeadora", "abelha-caucasiana", "abelha-comum",
"abelha-criança", "abelha-doméstica", "abelha-escura",
"abelha-europa", "abelha-europeia", "abelha-flor",
"abelha-indígena", "abelha-italiana", "abelha-limão",
"abelha-macho", "abelha-mãe", "abelha-mestra",
"abelha-mirim", "abelha-mosquito", "abelha-mosquitos",
"abelha-mulata", "abelha-neutra", "abelha-obreira",
"abelha-operária", "abelha-ouro", "abelha-preta",
"abelha-sanharó", "abelha-uruçu", "abelhas-africanas",
"abelhas-alemãs", "abelhas-amarelas", "abelhas-bravas",
"abelhas-campeadoras", "abelhas-caucasianas", "abelhas-comuns",
"abelhas-domésticas", "abelhas-escuras", "abelhas-europeias",
"abelhas-indígenas", "abelhas-italianas", "abelhas-machas",
"abelhas-machos", "abelhas-mestras", "abelhas-mirins",
"abelhas-mulatas", "abelhas-neutras", "abelhas-obreiras",
"abelhas-operárias", "abelhas-pretas", "abelhas-sanharós",
"abelhas-uruçu", "abóbora-cabaça", "abóbora-carneira",
"abóbora-chila", "abóbora-gila", "abóbora-menina",
"abóbora-porqueira", "abutre-barbudo", "abutre-cinéreo",
"abutre-comum", "abutre-fouveiro", "abutre-fusco",
"abutres-barbudos", "abutres-cinéreos", "abutres-comuns",
"abutres-fouveiros", "abutres-fuscos", "acácia-amarela",
"acácia-angico", "acácia-arábica", "acácia-argentina",
"acácia-asiática", "acácia-bastarda", "acácia-branca",
"acácia-dealbada", "acácia-falsa", "acácia-meleira",
"acácia-mimosa", "acácia-negra", "acácia-verdadeira",
"acácia-vermelha", "acácias-amarelas", "acácias-arábicas",
"acácias-argentinas", "acácias-asiáticas", "acácias-bastardas",
"acácias-brancas", "acácias-dealbadas", "acácias-falsas",
"acácias-meleiras", "acácias-mimosas", "acácias-negras",
"acácias-verdadeiras", "acácias-vermelhas", "acadêmico-científico",
"acadêmico-profissional", "açafrão-agreste", "açafrão-bastardo",
"açafrão-bravo", "açafrão-palhinha", "açafrão-vermelho",
"açafroeira-indígena", "açafroeiras-indígenas", "açafrões-agrestes",
"açafrões-bastardos", "açafrões-bravos", "açafrões-vermelhos",
"açaí-branco", "açaí-catinga", "açaí-chumbo",
"açaí-mirim", "açaís-brancos", "açaís-mirins",
"acaju-catinga", "acalenta-menino", "acampamento-base",
"acanto-bastardo", "acapurana-vermelha", "acapuranas-vermelhas",
"aço-carbono", "aço-liga", "açoita-cavalo",
"acre-doce", "açúcar-pedra", "adiposo-genital",
"adiposos-genitais", "administrador-delegado", "administrador-depositário",
"administrador-geral", "administrador-judicial", "administradora-delegada",
"administradora-judicial", "administradores-gerais", "administrativo-financeiro",
"Administrativo-Financeiro", "Advocacia-Geral", "advogada-geral",
"Advogada-Geral", "advogado-geral", "Advogado-Geral",
"afegão-americano", "afonso-arinense", "afonso-bezerrense",
"afonso-claudiense", "afonso-cunhense", "afro-americana",
"afro-americanas", "afro-americano", "afro-americanos",
"afro-ameríndio", "afro-árabe", "afro-asiático",
"afro-baiana", "afro-baianas", "afro-baiano", "afro-baianos", "afro-brasileirismo", "afro-brasileirístico",
"afro-brasileira", "afro-brasileiras", "afro-brasileiro", "Afro-Brasileiro", "afro-brasileiros",
"afro-britânica", "afro-britânicas", "afro-britânico",
"afro-britânicos", "afro-caribenho", "afro-cubano",
"afro-francês", "afro-indígena", "afro-inglês",
"afro-lusitanismo", "afro-lusitanístico", "afro-lusitana",
"afro-lusitanas", "afro-lusitano", "afro-lustanos",
"afro-negrismo", "afro-negrístico", "afro-negro",
"agente-motorista", "agentes-motoristas", "agrião-bravo",
"agrião-falso", "agrião-menor","ágrico-indústria",
"ágrico-industrial","ágrico-pecusário", "agrimônia-bastarda",
"agrimônias-bastardas", "agriões-bravos", "agriões-falsos",
"agriões-menores", "agro-doce", "agro-hidrologia",
"agro-hórtico","água-viva","águas-vivas",
"águia-belicosa","águia-branquense","águia-brasileira",
"águia-calva","águia-chilena","águia-cinzenta",
"águia-coroada","águia-destruidora","águia-equatorial",
"águia-gritadeira","águia-pescadora","águia-pesqueira",
"águia-rabalva","águia-real","águias-belicosas",
"águias-brasileiras","águias-calvas","águias-chilenas",
"águias-cinzentas","águias-coroadas","águias-destruidoras",
"águias-equatoriais","águias-gritadeiras","águias-pescadoras",
"águias-pesqueiras","águias-rabalvas","águias-reais",
"agulha-branca", "agulha-crioula", "agulha-ferrugenta",
"agulha-mãe", "agulha-padrão", "agulha-preta",
"agulhão-azul", "agulhão-bandeira", "agulhão-branco",
"agulhão-lambaio", "agulhão-negro", "agulhão-roliço",
"agulhão-trombeta", "agulhas-brancas", "agulhas-crioulas",
"agulhas-ferrugentas", "agulhas-pretas", "agulheira-menor",
"agulheira-moscada", "agulheiras-menores", "agulheiras-moscadas",
"agulhões-azuis", "agulhões-brancos", "agulhões-lambaios",
"agulhões-negros", "agulhões-roliços", "aipo-bravo",
"aipo-chimarrão", "aipo-falso", "aipo-hortense",
"aipo-rábano", "aipo-salsão", "aipos-bravos",
"aipos-chimarrões", "aipos-falsos", "aipos-hortenses",
"aipos-salsões", "air-bag", "ajudante-general",
"ajunta-pedra", "Al-Manámah", "Al-Qaeda",
"ala-armador", "ala-pivô", "alabastro-calcário",
"alabastro-oriental", "alabastros-calcários", "alabastros-orientais",
"alamanda-amarela", "alamanda-cheirosa", "alamandas-amarelas",
"alamandas-cheirosas","álamo-branco","álamo-negro",
"álamo-preto","álamos-brancos","álamos-negros",
"álamos-pretos", "alas-armadoras", "alas-armadores",
"albacora-azul", "albacora-bandolim", "albacora-branca",
"albacora-cachorra", "albacora-lajeira", "albacora-verdadeira",
"albacoras-azuis", "albacoras-brancas", "albacoras-cachorras",
"albacoras-lajeiras", "albacoras-verdadeiras", "albatroz-errante",
"albatroz-gigante", "albatroz-real", "albatroz-viageiro",
"albatrozes-errantes", "albatrozes-gigantes", "albatrozes-reais",
"albatrozes-viageiros", "alça-perna", "alcachofra-brava",
"alcachofra-hortense", "alcachofras-bravas", "alcachofras-hortenses",
"alcaçuz-bravo", "alcaçuz-falso", "alcaçuz-indiano",
"alcaçuz-silvestre", "alcaçuzes-bravos", "alcaçuzes-falsos",
"alcaçuzes-indianos", "alcaçuzes-silvestres", "alcaidaria-mor",
"alcaidarias-mores", "alcaide-menor", "alcaide-mor",
"alcaide-pequeno", "alcaides-menores", "alcaides-mores",
"alcaides-pequenos", "alcalino-terroso", "alcaparreira-cheirosa",
"alcaparreiras-cheirosas", "alecrim-bravo", "alecrins-bravos",
"alegra-campo", "além-Atlântico", "além-Brasil",
"além-fronteira", "além-mar", "além-mundo",
"além-Pirenéus", "além-Reno", "além-túmulo",
"alemães-ocidentais", "alemães-orientais", "alemão-ocidental",
"alemão-oriental", "alfa-amilase", "alfa-hélice",
"alfa-sinucleína", "alfa-Sinucleína", "alface-americana",
"alface-brava", "alface-crespa", "alface-frisada",
"alface-paulista", "alface-repolhuda", "alface-romana",
"alfaces-americanas", "alfaces-bravas", "alfaces-crespas",
"alfaces-frisadas", "alfaces-paulistas", "alfaces-repolhudas",
"alfaces-romanas", "alfafa-arbórea", "alfafa-brasileira",
"alfafa-espinhosa", "alfafa-gigante", "alfafa-lupulina",
"alfafa-mineira", "alfafa-nacional", "alfafa-rústica",
"alfafa-verdadeira", "alfafas-arbóreas", "alfafas-brasileiras",
"alfafas-espinhosas", "alfafas-gigantes", "alfafas-lupulinas",
"alfafas-mineiras", "alfafas-nacionais", "alfafas-rústicas",
"alfafas-verdadeiras", "alfavaca-carpunha", "alfavaca-cheirosa",
"alfavaca-silvestre", "alfavacas-carpunhas", "alfavacas-cheirosas",
"alfavacas-silvestres", "alfazema-brava", "alfazemas-bravas",
"alforva-brava", "alforvas-bravas", "alga-perlada",
"algarobeira-preta", "algarobeiras-pretas", "algas-perladas",
"algodão-bravo", "algodão-colódio", "algodão-cravo",
"algodão-doce", "algodão-macaco", "algodão-pólvora",
"algodoeiro-americano", "algodoeiro-bravo", "algodoeiro-crioulo",
"algodoeiro-gigante", "algodoeiros-americanos", "algodoeiros-bravos",
"algodoeiros-crioulos", "algodoeiros-gigantes", "algodões-bravos",
"algodões-doces", "alho-espanhol", "alho-francês",
"alho-grosso", "alho-íngreme", "alho-macho",
"alho-mágico", "alho-mourisco", "alho-ordinário",
"alho-poró", "alho-porro", "alho-porró",
"alho-porrô", "alho-rocambole", "alho-silvestre",
"alhos-espanhóis", "alhos-franceses", "alhos-grossos",
"alhos-íngremes", "alhos-machos", "alhos-mágicos",
"alhos-mouriscos", "alhos-ordinários", "alhos-porós",
"alhos-porros", "alhos-porrós", "alhos-porrôs",
"alhos-silvestres", "aliba-caçô", "alisso-cheiroso",
"alissos-cheirosos", "alma-aflita", "alma-danada",
"alma-perdida", "almas-aflitas", "almas-danadas",
"almas-perdidas", "almécega-verdadeira", "almécegas-verdadeiras",
"almecegueira-cheirosa", "almecegueira-mansa", "almecegueira-verdadeira",
"almecegueira-vermelha", "almecegueiras-cheirosas", "almecegueiras-mansas",
"almecegueiras-verdadeiras", "almecegueiras-vermelhas", "almecegueiro-bravo",
"almecegueiros-bravos", "almino-afonsense", "almirante-mor",
"almirante-vermelho", "almirantes-mores", "almirantes-vermelhos",
"almoço-ajantarado", "almoço-ceia", "almoço-debate",
"almoços-ajantarados", "almotacé-mor", "almotacés-mores",
"alpiste-miúdo", "alpistes-miúdos", "alqueire-mineiro",
"alqueire-paulista", "alqueires-mineiros", "alqueires-paulistas",
"alquequenje-amarelo", "alquequenje-bastardo", "alquequenjes-amarelos",
"alquequenjes-bastardos", "alsácio-loreno", "alta-costura",
"alta-fidelidade", "alta-florense", "alta-luz",
"alta-roda", "alta-tensão", "altar-mor",
"altares-mores", "altas-costuras", "altas-fidelidades",
"altas-luzes", "altas-rodas", "altas-tensões",
"alto-alecrinense", "alto-alegrense", "alto-alemão",
"alto-alentejano", "alto-araguaiense", "alto-astral",
"alto-brilho", "alto-comando", "alto-comissariado",
"alto-comissariato", "alto-comissário", "alto-contraste",
"alto-cúmulo", "alto-duriense", "alto-estrato",
"alto-explosivo", "alto-falante", "alto-falantes", "alto-forno", "alto-fornos",
"alto-fuste", "alto-garcense", "alto-grandense", "alto-grandenses",
"alto-horário", "alto-itaunense", "alto-lajeadense",
"alto-longaense", "alto-madeirense", "alto-mar",
"alto-maranhense", "alto-minhoto", "alto-navarro",
"alto-paraguaiense", "alto-paraguaio", "alto-paraisense",
"alto-paranaense", "alto-paranaibense", "alto-parnaibano",
"alto-piquirinense", "alto-relevo", "alto-rodriguense",
"alto-santense", "alto-saxão", "alto-serrano",
"alto-vácuo", "alto-voltaico", "alto-voltense",
"altos-alemães", "altos-astrais", "altos-comandos",
"altos-comissariados", "altos-comissários", "altos-contrastes",
"altos-cúmulos", "altos-estratos", "altos-explosivos",
"altos-fornos", "altos-fundos", "altos-fustes",
"altos-horários", "altos-lissos", "altos-mares",
"altos-relevos", "altos-vácuos", "aluno-aprendiz",
"aluno-mestre", "aluno-professor", "alunos-mestres",
"alvéloa-amarela", "alvéola-branca", "alvéolas-brancas",
"alvéolo-condiliano", "alvéolo-dental", "alvéolo-dentário",
"alvéolo-labial", "alvéolo-nasal", "alvéolo-palatal",
"alvéolo-subnasal", "ama-seca", "amansa-besta",
"amansa-senhor", "amaranto-verde", "amarantos-verdes",
"amarelo-canário", "amarelo-cidrão", "amarelo-cinzento",
"amarelo-claro", "amarelo-cobalto", "amarelo-enxofre",
"amarelo-escuro", "amarelo-esverdeado", "amarelo-fosco",
"amarelo-gualdo", "amarelo-laranja", "amarelo-limão",
"amarelo-oca", "amarelo-ocre", "amarelo-oiro",
"amarelo-ouro", "amarelo-palha", "amarelo-topázio",
"amarelo-torrado", "amarelos-cinzentos", "amarelos-claros",
"amarelos-escuros", "amarelos-esverdeados", "amarelos-foscos",
"amarelos-torrados", "amaro-leitense", "amarra-pinto",
"amas-secas", "amassa-barro","âmbar-amarelo",
"âmbar-cinzento","âmbares-amarelos","âmbares-cinzentos",
"ambaúba-mansa", "ambaúbas-mansas", "ambaúva-mansa",
"ambaúvas-mansas", "ambrósia-americana", "ambrósias-americanas",
"amêijoa-branca", "amêijoa-redonda", "amêijoas-brancas",
"amêijoas-redondas", "ameixa-amarela", "ameixa-americana",
"ameixa-japonesa", "ameixa-preta", "ameixas-amarelas",
"ameixas-americanas", "ameixas-japonesas", "ameixas-pretas",
"ameixeira-brava", "ameixeiras-bravas", "amêndoa-brava",
"amêndoa-durázia", "amêndoa-molar", "amêndoas-bravas",
"amêndoas-durázias", "amêndoas-molares", "amendoeira-coca",
"amendoeira-durázia", "amendoeira-molar", "amendoeiras-durázias",
"amendoeiras-molares", "amendoim-bravo", "amendoim-falso",
"amendoim-rajado", "amendoim-rasteiro", "amendoim-roxo",
"amendoins-bravos", "amendoins-falsos", "amendoins-rajados",
"amendoins-rasteiros", "amendoins-roxos", "americana-brasileira",
"americana-canadense", "americana-venezuelana", "americanas-canadenses",
"americano-brasileiro", "americano-canadense", "americano-soviético",
"americano-venezuelano", "americanos-canadenses", "ametista-oriental",
"ametistas-orientais", "amido-mosqueado", "amido-negro",
"amido-preto", "amidos-mosqueados", "amidos-negros",
"amidos-pretos", "amieiro-negro", "amieiros-negros",
"amigo-oculto", "amigo-secreto", "amigo-urso",
"amigos-ocultos", "amigos-secretos", "amigos-ursos",
"âmio-maior","âmio-vulgar", "amizade-colorida",
"amizades-coloridas", "amor-agarradinho", "amor-agarrado",
"amor-crescido", "amor-entrelaçado", "amor-perfeito",
"amor-próprio", "amor-seco", "amora-branca",
"amora-brava", "amora-preta", "amora-verde",
"amora-vermelha", "amoras-brancas", "amoras-bravas",
"amoras-pretas", "amoras-verdes", "amoras-vermelhas",
"amoré-preto", "amoreira-branca", "amoreira-cipó",
"amoreira-negra", "amoreira-preta", "amoreira-tataíba",
"amoreiras-brancas", "amoreiras-negras", "amoreiras-pretas",
"amores-agarradinhos", "amores-agarrados", "amores-crescidos",
"amores-entrelaçados", "amores-perfeitos", "amores-próprios",
"amores-secos", "amorés-pretos", "amostra-tipo",
"amovo-inamovível", "ampere-espira", "ampere-giro",
"ampere-hora", "ampere-segundo", "ampere-volta",
"ampério-espira", "ampério-giro", "ampério-hora",
"ampério-segundo", "ampério-volta", "anacã-mirim",
"anacãs-mirins", "anadéis-mores", "anadel-mor",
"anafa-cheirosa", "anafa-maior", "anafas-cheirosas",
"anafas-maiores", "anafe-menor", "anafe-ordinário",
"anafes-menores", "anafes-ordinários", "anajá-brava",
"anajá-mirim", "anajás-bravas", "anajás-mirins",
"analista-chefe", "analista-programador", "analistas-chefes",
"analítico-expositivo", "anambé-azul", "anambé-branco",
"anambé-grande", "anambé-militar", "anambé-pitiú",
"anambé-pombo", "anambé-preto", "anambé-roxo",
"anambé-sol", "anambé-vermelho", "anambés-azuis",
"anambés-brancos", "anambés-grandes", "anambés-militares",
"anambés-pitiú", "anambés-pretos", "anambés-roxos",
"anambés-vermelhos", "ananases-selvagens", "ananases-silvestres",
"anarco-comunismo", "anarco-comunista", "anarco-sindicalismo",
"anarco-sindicalista", "anchova-preta", "anchovas-pretas",
"andiroba-branca", "andiroba-jareuá", "andiroba-suruba",
"andiroba-vermelha", "andirobas-brancas", "andirobas-surubas",
"andirobas-vermelhas", "ando-boliviano", "ando-chileno",
"ando-colombiano", "ando-equatoriano", "ando-peruano",
"andor-império", "andorinha-azul", "andorinha-cavadeira",
"andorinha-coleira", "andorinha-grande", "andorinhas-azuis",
"andorinhas-cavadeiras", "andorinhas-coleiras", "angélica-branca",
"angélica-mansa", "angélica-silvestre", "angélicas-brancas",
"angélicas-mansas", "angelim-amarelo", "angelim-amargo",
"angelim-amargoso", "angelim-araroba", "angelim-coco",
"angelim-doce", "angelim-pedra", "angelim-pintado",
"angelim-preto", "angelim-rajado", "angelim-rosa",
"angelins-amarelos", "angelins-amargos", "angelins-amargosos",
"angelins-doces", "angelins-pintados", "angelins-pretos",
"angelins-rajados", "angico-amarelo", "angico-barbatimão",
"angico-branco", "angico-cangalha", "angico-cedro",
"angico-preto", "angico-rajado", "angico-rosa",
"angico-roxo", "angico-surucucu", "angico-verdadeiro",
"angico-vermelho", "angicos-amarelos", "angicos-brancos",
"angicos-pretos", "angicos-rajados", "angicos-roxos",
"angicos-surucucu", "angicos-verdadeiros", "angicos-vermelhos",
"anglo-africano", "anglo-alemão", "anglo-americano",
"anglo-árabe", "anglo-asiático", "anglo-australiano",
"anglo-brasileiro", "anglo-bretão", "anglo-canadense",
"anglo-canadiano", "anglo-catolicismo", "anglo-católico",
"anglo-francês", "anglo-galo", "anglo-gaulês",
"anglo-germânico", "anglo-holandês", "anglo-indiano",
"anglo-irlandês", "anglo-israelismo", "anglo-italiano",
"anglo-luso", "anglo-normando", "anglo-nubiano",
"anglo-português", "anglo-russo", "anglo-saxã",
"anglo-saxão", "anglo-saxônico", "anglo-saxônio",
"anglo-saxonismo", "angona-puíta", "anil-bastardo",
"anil-bravo", "anil-miúdo", "anil-trepador",
"anileira-verdadeira", "anileiras-verdadeiras", "anima-membeca",
"anis-bastardos", "anis-bravos", "anis-miúdos",
"anis-trepadores", "anises-doces", "anises-estrelados",
"anises-verdes", "aniso-hiperleucocitose", "aniso-hipocitose",
"aniso-hipoleucocitose", "anjo-bento", "anjo-mau",
"anjo-papudo", "anjo-viola", "anjos-bentos",
"anjos-maus", "anjos-papudos", "ano-base",
"ano-bom", "ano-calendário", "ano-exercício",
"ano-luz",
"ano-opsia", "ano-óptico", "anos-bons",
"anos-exercícios", "anos-luz",
"ansarinha-branca", "ansarinha-malhada",
"ansarinha-vermífuga", "ansarinhas-brancas", "ansarinhas-malhadas",
"ansarinhas-vermífugas", "anta-gameleira", "anta-sapateira",
"antas-gameleiras", "antas-sapateiras", "ante-epígrafe",
"ante-hélice", "ante-hipófise", "ante-histórico",
"ântero-dorsal","ântero-inferior","ântero-lateral",
"ântero-posterior","ântero-superior", "anti-Bush",
"anti-China", "anti-halo", "anti-hélio",
"anti-HIV", "anti-inato", "anti-íneo",
"anti-Marx", "anti-Rho", "anti-Saddam",
"anti-spam", "antílope-real", "antílopes-reais",
"antiômega-mais", "antônio-almeidense", "antônio-carlense",
"antônio-diense", "antônio-joanense", "antônio-martinense",
"antro-nasal", "antropo-histografia", "antropo-histográfico",
"anu-branco", "anu-chorado", "anu-coroca",
"anu-coroia", "anu-corrido", "anu-dourado",
"anu-galego", "anu-grande", "anu-guaicuru",
"anu-peixe", "anu-pequeno", "anu-preto",
"anu-velho", "anum-branco", "anum-chorado",
"anum-coroca", "anum-coroia", "anum-corrido",
"anum-dourado", "anum-galego", "anum-grande",
"anum-guaicuru", "anum-peixe", "anum-pequeno",
"anum-preto", "anum-velho", "anuns-brancos",
"anuns-chorados", "anuns-corocas", "anuns-coroias",
"anuns-corridos", "anuns-dourados", "anuns-galegos",
"anuns-grandes", "anuns-guaicurus", "anuns-peixes",
"anuns-pequenos", "anuns-pretos", "anuns-velhos",
"anus-brancos", "anus-chorados", "anus-corocas",
"anus-coroias", "anus-corridos", "anus-dourados",
"anus-galegos", "anus-grandes", "anus-guaicurus",
"anus-peixes", "anus-pequenos", "anus-pretos",
"anus-velhos", "apa-bico", "apacanim-branco",
"apacanins-brancos", "apaga-fanóis", "apaga-gás",
"apaga-gases", "apaga-penol", "apaga-tristeza",
"apanha-bolas", "apanha-migalhas", "apanha-moscas",
"apanha-saia", "apapá-verdadeiro", "apapás-verdadeiros",
"apara-lápis", "apara-livros", "apara-mangaba",
"apart-hotel", "aperta-chico", "aperta-galha",
"aperta-goela", "aperta-livros", "aperta-luvas",
"aperta-nervos", "aperta-papéis", "aperta-ruão",
"Apicum-Açu", "apoia-braços", "apoia-mãos",
"apoia-nuca", "aposentador-mor", "aposentadores-mores",
"apostólico-cristão", "aprendiz-marinheiro", "aprendizes-marinheiros",
"aquém-fronteiras", "aquém-mar", "aquém-oceano",
"aquém-pireneus","árabe-brasileiro","árabe-Brasileiro",
"árabe-israelense","árabe-saudita", "arábico-britânico",
"arábico-espanhol", "arábico-português", "arábigo-britânico",
"arábigo-espanhol", "arábigo-português", "araçá-branco",
"araçá-bravo", "araçá-cagão", "araçá-cinzento",
"araçá-congonha", "araçá-cotão", "araçá-felpudo",
"araçá-lima", "araçá-mirim", "araçá-pedra",
"araçá-rosa", "araçá-roxo", "araçá-verde",
"araçá-vermelho", "Aracaju-Barra", "araçarana-mirim",
"araçaranas-mirins", "aranha-bode", "aranha-caranguejeira",
"aranha-caranguejo", "arara-amarela", "arara-azul",
"arara-canindé", "arara-caro", "arara-celeste",
"arara-chauanauá", "arara-cipó", "arara-hiacinta",
"arara-jacinto", "arara-macau", "arara-pequena",
"arara-pitiú", "arara-preta", "arara-verde",
"arara-vermelha", "araras-amarelas", "araras-azuis",
"araras-caros", "araras-celestes", "araras-cipós",
"araras-hiacintas", "araras-jacintos", "araras-macau",
"araras-pequenas", "araras-pitiú", "araras-pretas",
"araras-verdes", "araras-vermelhas", "araticum-abareno",
"araticum-alvadio", "araticum-apê", "araticum-bravo",
"araticum-caca", "araticum-cagão", "araticum-catinga",
"araticum-cortiça", "araticum-fedorento", "araticum-grande",
"araticum-mirim", "araticum-peludo", "araticum-pitaiá",
"araticum-ponhê", "araticuns-abarenos", "araticuns-alvadios",
"araticuns-bravos", "araticuns-cagões", "araticuns-fedorentos",
"araticuns-grandes", "araticuns-mirins", "araticuns-peludos",
"araticuns-ponhês", "aratu-marinheiro", "aratu-vermelho",
"aratus-marinheiros", "aratus-vermelhos", "arbusto-milagroso",
"arbustos-milagrosos", "arco-celeste", "arco-íris",
"Arco-íris", "arco-irisado", "arco-irisar",
"arco-irisense", "arco-verdense", "arcos-celestes",
"arcos-íris", "arcos-irisados","área-alvo",
"área-chave","área-cliente","área-fim",
"áreas-fins", "areia-branquense", "areia-engolideira",
"areia-gulosa", "areia-manteiga", "areias-engolideiras",
"areias-gordas", "areias-gulosas", "areno-argiloso",
"areno-humoso", "areno-siltoso", "ares-condicionados",
"ares-refrigerados", "argentino-brasileiro", "argento-vivo",
"argentos-vivos", "argilo-arenoso",
"armador-gerente", "armador-locatário", "armador-proprietário",
"armadores-gerentes", "armadores-locatários", "armadores-proprietários",
"armados-comuns", "armazém-geral", "armazéns-gerais",
"armeiro-mor", "armeiros-mores", "arquipélago-estado",
"arquivo-fonte", "Arquivo-Fonte", "arquivo-texto",
"arraia-acauã", "arraia-boró", "arraia-branca",
"arraia-listrada", "arraia-mijona", "arraia-miúda",
"arraia-papagaio", "arraia-preta", "arraia-santo",
"arraia-tremedeira", "arraias-brancas", "arraias-listradas",
"arraias-mijonas", "arraias-miúdas", "arraias-pintadinhas",
"arraias-pretas", "arraias-tremedeiras", "arranha-céu",
"arroio-grandense", "arroio-meense", "arromba-peito",
"arroz-agulha", "arroz-amarelão", "arroz-bravo",
"arroz-cateto", "arroz-crioulo", "arroz-doce",
"arroz-pequeno", "arroz-silvestre", "arrozes-amarelões",
"arrozes-bravos", "arrozes-crioulos", "arrozes-doces",
"arrozes-pequenos", "arrozes-silvestres", "arruda-amarela",
"arruda-brava", "arruda-comum", "arruda-fedorenta",
"arruda-parietária", "arruda-preta", "arruda-rajada",
"arruda-silvestre", "arruda-vermelha", "arrudas-amarelas",
"arrudas-bravas", "arrudas-comuns", "arrudas-fedorentas",
"arrudas-pretas", "arrudas-rajadas", "arrudas-silvestres",
"arrudas-vermelhas", "arte-educador", "arte-educadora", "arte-educadoras",
"arte-educadores", "arte-finalizar", "artemísia-brava",
"artemísia-verdadeira", "artemísias-bravas", "artemísias-verdadeiras",
"artístico-cultural", "artístico-literário","árvore-branca",
"árvore-copal","árvore-mãe","árvore-santa",
"árvore-triste","árvores-brancas","árvores-copais",
"árvores-santas","árvores-tristes", "asfalto-vanadífero",
"asfaltos-vanadíferos", "asilo-colônia", "asilo-escola",
"asilos-colônias","ásio-americano","ásio-australiano",
"ásio-brasileiro","ásio-francês","ásio-inglês",
"ásio-lusitanismo","ásio-lusitanístico","ásio-lusitano",
"ásio-português","ásio-saxão", "assessor-chefe", "assessora-chefe",
"assessoras-chefes", "assessores-chefes", "astro-rei",
"astros-reis", "atividade-fim", "atividade-meio",
"ativo-assistido", "ato-show","átomo-grama",
"ator-empresário", "atores-empresários", "atos-show",
"atriz-empresária", "atrizes-empresárias", "atum-azul",
"atum-bandolim", "atum-branco", "atum-preto",
"atum-rabilho", "atum-verdadeiro", "atuns-azuis",
"atuns-brancos", "atuns-pretos", "atuns-verdadeiros",
"audio-olfatório", "audio-oral", "auditor-fiscal",
"Auditor-Fiscal", "auditores-fiscais", "Auditores-Fiscais",
"augusto-correense", "augusto-limense", "augusto-pestanense",
"augusto-severense", "australo-africano", "austro-africano",
"austro-áfrico", "austro-alemão", "austro-asiático",
"austro-bávaro", "austro-brasileiro", "austro-central",
"austro-flamengo", "austro-holandês", "austro-húngaro",
"austro-siberiano", "autor-cientista", "autora-cientista",
"autoras-cientistas", "autores-cientistas",
"auxílio-acidentário", "auxílio-acidente",
"Auxílio-Acidente", "auxílio-alimentação",
"Auxílio-Alimentação", "auxílio-creche",
"auxílio-desemprego", "Auxílio-Desemprego",
"auxílio-doença", "auxílio-educação",
"auxílio-enfermidade", "auxílio-funeral",
"auxílio-maternidade", "auxílio-moradia",
"auxílio-natalidade", "auxílio-paletó",
"auxílio-reclusão", "auxílio-refeição",
"Auxílio-Refeição", "auxílio-saúde",
"auxílio-transporte",
"avant-garde", "ave-maria",
"Ave-Maria", "ave-marias", "Ave-Marias",
"aveia-amarela", "aveia-comum", "aveia-estéril",
"aveias-amarelas", "aveias-comuns", "aveias-estéreis",
"avenca-branca", "avenca-brasileira", "avenca-cuneiforme",
"avenca-estrelada", "avenca-grande", "avenca-miúda",
"avenca-negra", "avenca-paulista", "avenca-trapeziforme",
"avencas-brasileiras", "avencas-cuneiformes", "avencas-estreladas",
"avencas-grandes", "avencas-miúdas", "avencas-negras",
"avencas-paulistas", "avencas-trapeziformes", "avião-cegonha",
"avião-correio", "avião-espião", "avião-foguete",
"avião-laboratório", "avião-suicida", "avião-tanque",
"aviões-suicidas",
"avó-rainha", "azeitona-rei", "azuis-celestes",
"azuis-claras", "azuis-claros", "azuis-esbranquiçadas",
"azuis-esbranquiçados", "azuis-escuras", "azuis-escuros",
"azuis-marinhos", "azul-celeste", "azul-cinzento",
"azul-claro", "azul-cobalto", "azul-esbranquiçado",
"azul-escuro", "azul-esverdeado", "azul-faiança",
"azul-ferrete", "azul-marinho", "azul-piscina",
"azul-seda", "azul-turquesa", "azulão-bicudo",
"azulões-bicudos", "baba-ovo", "bailarino-escarlate",
"bailarinos-escarlates", "baixa-grandense", "baixa-mar",
"baixa-verdense", "baixo-alemão", "baixo-astral",
"baixo-bretão", "baixo-espiritismo", "baixo-império",
"baixo-latim", "baixo-relevo", "baixo-ventre",
"baixos-relevos", "baleia-anã", "baleia-assassina",
"baleia-azul", "baleia-bicuda", "baleia-branca",
"baleia-corcunda", "baleia-dente", "baleia-espadarte",
"baleia-franca", "baleia-lisa", "baleia-mirim",
"baleia-pamonha", "baleia-piloto", "baleia-pintada",
"baleia-preta", "baleia-verdadeira", "baleias-anãs",
"baleias-assassinas", "baleias-azuis", "baleias-bicudas",
"baleias-brancas", "baleias-corcundas", "baleias-francas",
"baleias-lisas", "baleias-mirins", "baleias-pintadas",
"baleias-pretas", "baleias-verdadeiras", "banana-anã",
"banana-branca", "banana-caturra", "banana-comprida",
"banana-figo", "banana-francesa", "banana-grande",
"banana-inajá", "banana-maçã", "banana-mãe",
"banana-menina", "banana-mole", "banana-najá",
"banana-nanica", "banana-ouro", "banana-pacova",
"banana-pacová", "banana-pão", "banana-passa",
"banana-pedra", "banana-prata", "banana-rajada",
"banana-real", "banana-roxa", "banana-timbó",
"banana-verde", "bananas-anãs", "bananas-brancas",
"bananas-caturras", "bananas-compridas", "bananas-francesas",
"bananas-grandes", "bananas-moles", "bananas-nanicas",
"bananas-passas", "bananas-rajadas", "bananas-reais",
"bananas-roxas", "bananas-split", "bananas-verdes",
"bananeira-anã", "bananeira-branca", "bananeira-brava",
"bananeira-chinesa", "bananeira-nanica", "bananeira-petiça",
"bananeira-rainha", "bananeira-vermelha", "bananeiras-anãs",
"bananeiras-brancas", "bananeiras-bravas", "bananeiras-chinesas",
"bananeiras-nanicas", "bananeiras-petiças", "bananeiras-rainhas",
"bananeiras-vermelhas", "batata-baroa", "batata-baronesa",
"batata-branca", "batata-brava", "batata-ceará",
"batata-chinesa", "batata-cogumelo", "batata-doce",
"batata-inglesa", "batata-miúda", "batata-portuguesa",
"batata-roxa", "batata-salsa", "batata-semente",
"batata-silvestre", "batata-suíça", "batatas-baroas",
"batatas-baronesas", "batatas-brancas", "batatas-bravas",
"batatas-chinesas", "batatas-doces", "batatas-inglesas",
"batatas-miúdas", "batatas-portuguesas", "batatas-roxas",
"batatas-silvestres", "batatas-suíças", "bate-estaca",
"bate-estacas", "bebê-proveta", "bebês-provetas",
"belo-horizontina", "belo-horizontinas", "belo-horizontino", "belo-horizontinos",
"bem-acabado", "bem-acabada", "bem-acabadas", "bem-acabados",
"bem-aceita", "bem-aceitas", "bem-aceito", "bem-aceitos",
"bem-acondicionada", "bem-acondicionadas", "bem-acondicionado", "bem-acondicionados",
"bem-acondiçoada", "bem-acondiçoadas", "bem-acondiçoado", "bem-acondiçoados",
"bem-aconselhada", "bem-aconselhadas", "bem-aconselhado", "bem-aconselhados",
"bem-acostumada", "bem-acostumadas", "bem-acostumado", "bem-acostumados",
"bem-adaptada", "bem-adaptadas", "bem-adaptado", "bem-adaptados",
"bem-afamada", "bem-afamadas", "bem-afamado", "bem-afamados",
"bem-afeiçoada", "bem-afeiçoadas", "bem-afeiçoado", "bem-afeiçoados",
"bem-afortunar", "bem-agradecida", "bem-agradecidas", "bem-agradecido",
"bem-agradecidos", "bem-ajambrada", "bem-ajambradas", "bem-ajambrado",
"bem-ajambrados", "bem-amada", "bem-amadas", "bem-amado",
"bem-amados", "bem-andança", "bem-andanças", "bem-andante",
"bem-andantes", "bem-apanhada", "bem-apanhadas", "bem-apanhado",
"bem-apanhados", "bem-apessoada", "bem-apessoadas", "bem-apessoado",
"bem-apessoados", "bem-apresentada", "bem-apresentadas", "bem-apresentado",
"bem-apresentados", "bem-arranjada", "bem-arranjadas", "bem-arranjado",
"bem-arranjados", "bem-arrumada", "bem-arrumadas", "bem-arrumado",
"bem-arrumados", "Bem-aventurada", "bem-aventurada", "Bem-aventuradas",
"bem-aventuradas", "Bem-aventurado", "Bem-aventurados", "bem-aventurados",
"bem-aventurança", "bem-aventuranças", "bem-aventurar", "bem-avinda",
"bem-avindas", "bem-avindo", "bem-avindos", "bem-avisada",
"bem-avisadas", "bem-avisado", "bem-avisados", "bem-bom",
"bem-bons", "bem-caçada", "bem-caçadas", "bem-caçado",
"bem-caçados", "bem-casada", "bem-casadas", "bem-casadinho",
"bem-casadinhos", "bem-casado", "bem-casados", "bem-comportada",
"bem-comportadas", "bem-comportado", "bem-comportados", "bem-composta",
"bem-compostas", "bem-composto", "bem-compostos", "bem-conceituada",
"bem-conceituadas", "bem-conceituado", "bem-conceituados", "bem-conformada",
"bem-conformadas", "bem-conformado", "bem-conformados", "bem-convidada",
"bem-convidadas", "bem-convidado", "bem-convidados", "bem-criada",
"bem-criadas", "bem-criado", "bem-criados", "bem-curada",
"bem-curadas", "bem-curado", "bem-curados", "bem-dada",
"bem-dadas", "bem-dado", "bem-dados", "bem-disposta",
"bem-dispostas", "bem-disposto", "bem-dispostos", "bem-ditosa",
"bem-ditosas", "bem-ditoso", "bem-ditosos", "bem-dizente",
"bem-dizentes", "bem-dizer", "bem-dormida", "bem-dormidas",
"bem-dormido", "bem-dormidos", "bem-dotada", "bem-dotadas",
"bem-dotado", "bem-dotados", "bem-educada", "bem-educadas",
"bem-educado", "bem-educados", "bem-encarada", "bem-encaradas",
"bem-encarado", "bem-encarados", "bem-ensinada", "bem-ensinadas",
"bem-ensinado", "bem-ensinados", "bem-estar", "bem-fadada",
"bem-fadadas", "bem-fadado", "bem-fadados", "bem-fadar",
"bem-falante", "bem-falantes", "bem-fazente", "bem-fazentes",
"bem-fazer", "bem-feita", "bem-feitas", "bem-feito",
"bem-feitos", "bem-haja", "bem-humorada", "bem-humoradas",
"bem-humorado", "bem-humorados", "bem-ida", "bem-idas",
"bem-ido", "bem-idos", "bem-intencionada", "bem-intencionadas",
"bem-intencionado", "bem-intencionados", "bem-lançada", "bem-lançadas",
"bem-lançado", "bem-lançados", "bem-mandada", "bem-mandadas",
"bem-mandado", "bem-mandados", "bem-merecer", "bem-merecida",
"bem-merecidas", "bem-merecido", "bem-merecidos", "bem-nado",
"bem-nados", "bem-nascida", "bem-nascidas", "bem-nascido",
"bem-nascidos", "bem-ordenada", "bem-ordenadas", "bem-ordenado",
"bem-ordenados", "bem-ouvida", "bem-ouvidas", "bem-ouvido",
"bem-ouvidos", "bem-parada", "bem-paradas", "bem-parado",
"bem-parados", "bem-pensante", "bem-pensantes", "bem-posta",
"bem-postas", "bem-posto", "bem-postos", "bem-procedida",
"bem-procedidas", "bem-procedido", "bem-procedidos", "bem-proporcionada",
"bem-proporcionadas", "bem-proporcionado", "bem-proporcionados", "bem-querença",
"bem-querenças", "bem-querente", "bem-querentes", "bem-querer",
"bem-sabida", "bem-sabidas", "bem-sabido", "bem-sabidos",
"bem-soante", "bem-soantes", "bem-sonância", "bem-sonâncias",
"bem-sonante", "bem-sonantes", "bem-sucedida", "bem-sucedidas",
"bem-sucedido", "bem-sucedidos", "bem-talhada", "bem-talhadas",
"bem-talhado", "bem-talhados", "bem-temente", "bem-tementes",
"bem-vestida", "bem-vestidas", "bem-vestido", "bem-vestidos",
"bem-vestir", "bem-vinda", "Bem-vinda", "bem-vindas",
"Bem-vindas", "bem-vindo", "Bem-vindo", "bem-vindos",
"Bem-vindos", "bem-vista", "bem-vistas", "bem-visto",
"bem-vistos", "best-seller", "best-sellers", "besta-fera",
"besta-quadrada", "bestas-feras", "Blu-ray",
"Blu-Ray", "blue-jeans", "bossa-nova",
"boia-fria", "boias-frias", "bota-fora", "bota-foras",
"boto-amarelo", "boto-branco", "boto-cinza",
"boto-preto", "boto-tucuxi", "boto-vermelho",
"botos-amarelos", "botos-brancos", "botos-pretos",
"botos-vermelhos", "brejo-cruzense", "brejo-grandense",
"brejo-santense", "cabo-grandense", "cabo-guia",
"cabo-joanense", "cabo-verdense", "cabo-verdianismo",
"cabo-verdiano", "cabra-cega", "cabras-cegas",
"caça-minas", "caça-moscas", "caça-níquel",
"caça-palavra", "Caça-Palavra", "caça-palavras",
"caga-fogo", "caixa-forte", "caixa-prego",
"caixa-preta", "caixas-fortes", "caixas-pregos",
"caixas-pretas", "cálcio-fosfato", "caloria-grama",
"cama-beliche", "caminhão-baú", "caminhão-betoneira",
"caminhão-bomba", "caminhão-fossa", "caminhão-pipa",
"caminhão-tanque", "caminhões-baú", "canário-assobio",
"canário-assoviador", "canário-baeta", "canário-belga",
"canário-pardinho", "canário-pardo", "canários-assoviadores",
"canários-belgas", "canários-pardinhos", "canários-pardos",
"carro-bomba", "carro-chefe", "carro-conceito",
"carro-dormitório", "carro-forte", "carro-guincho",
"carro-leito", "carro-madrinha", "carro-patrulha",
"carro-pipa", "carro-restaurante", "carro-salão",
"carro-suporte", "carro-tanque", "carro-vassoura",
"carros-fortes", "carta-bilhete", "carta-bomba",  "carta-limpa",
"carta-partida", "carta-piloto", "carta-postal",
"carta-telegrama", "cartão-alimentação",
"Cartão-Alimentação", "cartão-ficha", "cartão-postal",
"cartão-resposta", "Cartão-Resposta", "cartas-postais",
"cartões-postais", "Cartões-Respostas", "cavalo-branco",
"cavalo-fantasma", "cavalo-hora", "cavalo-judeu",
"cavalo-marinho", "cavalo-mosca", "cavalo-reiuno",
"cavalo-rinc