//
// MonoTests.Mono.Math.ArithmeticBig.cs
//
// Authors:
//	Ben Maurer
//
// Copyright (c) 2003 Ben Maurer. All rights reserved
//

using System;
using Mono.Math;
using NUnit.Framework;

namespace MonoTests.Mono.Math.Arithmetic.Big {

	public abstract class ArithmeticBig_Base : BigIntegerTestSet {
		
		BigInteger A, B;
		
		public ArithmeticBig_Base () 
		{
			A = new BigInteger (a);
			B = new BigInteger (b);
		}

		public abstract uint[] a {
			get;
		}

		public abstract uint[] b {
			get;
		}

		public abstract uint[] ExpectedAplusB {
			get;
		}

		public abstract uint[] ExpectedAminusB {
			get;
		}

		public abstract uint[] ExpectedAtimesB {
			get;
		}

		public abstract uint[] ExpectedAdivB {
			get;
		}

		public abstract uint[] ExpectedAmodB {
			get;
		}

		public abstract uint[] ExpectedASquared {
			get;
		}

		public abstract uint[] ExpectedBSquared {
			get;
		}

		#region Addition
		
		[Test]
		public void AplusB () 
		{
			BigInteger Expected = new BigInteger (ExpectedAplusB);
			Expect (A + B, Expected);
		}

		[Test]
		public void BplusA  () 
		{
			BigInteger Expected = new BigInteger (ExpectedAplusB);
			Expect (B + A, Expected);
		}

		#endregion

		#region Subtraction
		
		[Test]
		public void AminusB () 
		{
			BigInteger Expected = new BigInteger (ExpectedAminusB);
			Expect (A - B, Expected);
		}
			
		#endregion

		#region Multiplication
		
		[Test]
		public void AtimesB () 
		{
			BigInteger Expected = new BigInteger (ExpectedAtimesB);
			Expect (A * B, Expected);
		}

		[Test]
		public void BtimesA () 
		{
			BigInteger Expected = new BigInteger (ExpectedAtimesB);
			Expect (B * A, Expected);
		}

		[Test]
		public void Asquared () 
		{
			BigInteger Expected = new BigInteger (ExpectedASquared);
			Expect (A * A, Expected);
		}

		[Test]
		public void Bsquared () 
		{
			BigInteger Expected = new BigInteger (ExpectedBSquared);
			Expect (B * B, Expected);
		}

		#endregion

		#region Division
		
		[Test]
		public void ABdivB () 
		{
			BigInteger AB = new BigInteger (ExpectedAtimesB);
			Expect (AB / B, A);
		}

		[Test]
		public void ABdivA () 
		{
			BigInteger AB = new BigInteger (ExpectedAtimesB);
			Expect (AB / A, B);
		}

		[Test]
		public void AdivB () 
		{
			BigInteger Expected = new BigInteger (ExpectedAdivB);
			Expect (A / B, Expected);
		}
		#endregion

		#region Mod
		[Test]
		public void AmodB () 
		{
			BigInteger Expected = new BigInteger (ExpectedAmodB);
			Expect (A % B, Expected);
		}

		[Test]
		public void BmodA () 
		{
			Expect (B % A, B);
		}

		[Test]
		public void ABmodB () 
		{
			BigInteger AB = new BigInteger (ExpectedAtimesB);
			Expect (AB % B, 0);
		}

		[Test]
		public void ABmodA () 
		{
			BigInteger AB = new BigInteger (ExpectedAtimesB);
			Expect (AB % A, 0);
		}
		
		#endregion
		
		#region Compare
		
		[Test]
		public void AgtB () 
		{
			Assertion.Assert(A > B);
		}
		
		[Test]
		public void ANotLtB () 
		{
			Assertion.Assert(!(A < B));
		}
		
		[Test]
		public void BNotGtA () 
		{
			Assertion.Assert(!(B > A));
		}
		
		[Test]
		public void AltB () 
		{
			Assertion.Assert(B < A);
		}
		
		[Test]
		public void AeqA () 
		{
			Assertion.Assert(A == A);
		}
		[Test]
		public void BeqB () 
		{
			Assertion.Assert(B == B);
		}
		
		[Test]
		public void AneqB () 
		{
			Assertion.Assert(A != B);
		}
		
		[Test]
		public void BneqA () 
		{
			Assertion.Assert(B != A);
		}

		#endregion
	}

	public class Random2048a1024b: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0xdbbda64c, 0x1a4e4dbd, 0x26847ccc, 0xc9897ecc, 0x215759b7, 0x37055866,
					0x8eb4e2ed, 0xa6c9c7a9, 0x36a087c7, 0x2b1d0192, 0xba08ef99, 0x77f570a1,
					0x0ab67993, 0xe4064de4, 0xaa860c0d, 0xdbace82b, 0x696df85a, 0x1d22f1ed,
					0x80d60685, 0xc360f73e, 0x4b8ca13d, 0x1ee5ca6f, 0x42490a0c, 0x00b5e783,
					0x1b803e1e, 0x3cffe9b9, 0xe17b071e, 0x6ce95cd5, 0x429fdb88, 0xe524a3c7,
					0x051e452a, 0x834bbf27, 0xe25211f0, 0xc4171b01, 0x4ef237ab, 0xdc8f9efd,
					0x46efffe7, 0xe162931d, 0x8576eb30, 0x9ac33969, 0x026d2252, 0x41758665,
					0x9aa43456, 0xa8e4a6f8, 0xe9785649, 0x76c8b136, 0x38a40d78, 0xfa3bacdc,
					0xcf446c35, 0x8c22cfee, 0x76a47fb3, 0x5cc25cde, 0xade755f5, 0xa31e6711,
					0xe53ef4c8, 0xe6312f00, 0x215f3377, 0xe66d0dcd, 0xf0563395, 0x130d5918,
					0x97e5c4bc, 0x3cfc3053, 0xcdbc19ad, 0x87e4011a
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0x92d3b581, 0x2710dd85, 0x59864d28, 0x7952c636, 0x4dbf0478, 0x360ed50e,
					0xd84ce175, 0xd321f743, 0x0b2434c7, 0xbe833d3a, 0xeec1b441, 0x08c98e67,
					0xd611fc13, 0x0803415d, 0x415d0cfc, 0xdf006733, 0x54379f5c, 0x8fab3be7,
					0x3aa840af, 0xb11453f2, 0x8bafb87b, 0x584bdcaf, 0x80a18f6e, 0x1c7bac68,
					0xe1ac8b8c, 0x4ac52f82, 0x6b510c1d, 0x198fbb6a, 0x5f2ab9d4, 0x40f86f80,
					0x4114f868, 0xf1c2448e
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
					0xdbbda64c, 0x1a4e4dbd, 0x26847ccc, 0xc9897ecc, 0x215759b7, 0x37055866,
					0x8eb4e2ed, 0xa6c9c7a9, 0x36a087c7, 0x2b1d0192, 0xba08ef99, 0x77f570a1,
					0x0ab67993, 0xe4064de4, 0xaa860c0d, 0xdbace82b, 0x696df85a, 0x1d22f1ed,
					0x80d60685, 0xc360f73e, 0x4b8ca13d, 0x1ee5ca6f, 0x42490a0c, 0x00b5e783,
					0x1b803e1e, 0x3cffe9b9, 0xe17b071e, 0x6ce95cd5, 0x429fdb88, 0xe524a3c7,
					0x051e452a, 0x834bbf28, 0x7525c771, 0xeb27f886, 0xa87884d4, 0x55e26533,
					0x94af0460, 0x1771682c, 0x5dc3cca6, 0x6de530ac, 0x0d915719, 0xfff8c3a0,
					0x8965e897, 0xb1ae3560, 0xbf8a525c, 0x7ecbf293, 0x7a011a75, 0xd93c1410,
					0x237c0b92, 0x1bce0bd5, 0xb14cc063, 0x0dd6b0d1, 0x39970e70, 0xfb6a43c1,
					0x65e08437, 0x02acdb69, 0x030bbf04, 0x31323d50, 0x5ba73fb2, 0x2c9d1482,
					0xf7107e90, 0x7df49fd4, 0x0ed11216, 0x79a645a8
				};
			}
		}
		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0xdbbda64c, 0x1a4e4dbd, 0x26847ccc, 0xc9897ecc, 0x215759b7, 0x37055866,
					0x8eb4e2ed, 0xa6c9c7a9, 0x36a087c7, 0x2b1d0192, 0xba08ef99, 0x77f570a1,
					0x0ab67993, 0xe4064de4, 0xaa860c0d, 0xdbace82b, 0x696df85a, 0x1d22f1ed,
					0x80d60685, 0xc360f73e, 0x4b8ca13d, 0x1ee5ca6f, 0x42490a0c, 0x00b5e783,
					0x1b803e1e, 0x3cffe9b9, 0xe17b071e, 0x6ce95cd5, 0x429fdb88, 0xe524a3c7,
					0x051e452a, 0x834bbf27, 0x4f7e5c6f, 0x9d063d7b, 0xf56bea83, 0x633cd8c6,
					0xf930fb6f, 0xab53be0e, 0xad2a09ba, 0xc7a14225, 0xf748ed8a, 0x82f2492a,
					0xabe28015, 0xa01b1891, 0x13665a36, 0x6ec56fd8, 0xf747007c, 0x1b3b45a9,
					0x7b0cccd8, 0xfc779407, 0x3bfc3f03, 0xabae08ec, 0x22379d7a, 0x4ad28a62,
					0x649d655a, 0xc9b58297, 0x3fb2a7eb, 0x9ba7de4b, 0x85052777, 0xf97d9dae,
					0x38bb0ae7, 0xfc03c0d3, 0x8ca72144, 0x9621bc8c
				};
			}
		}
		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0x7e07e1f3, 0x6a675027, 0xbcfd1837, 0xf6fb0771, 0xb9745456, 0x5617bb02,
					0xc8e784fa, 0xcb769578, 0xb74c305f, 0x98833dec, 0x3c1160a9, 0x38e53c88,
					0x0a6aaeda, 0x30dd3e16, 0x1c42b026, 0x94170e75, 0x3ad09fa6, 0x71fcd114,
					0xe9bd23a6, 0x31254d38, 0xec9a6527, 0x3b035cbf, 0xb7513d4e, 0xb0cbbb90,
					0x94af7307, 0x4acccec8, 0x973f936e, 0x5c5400e4, 0x022d2092, 0x84910b57,
					0xdca61ea7, 0xde181481, 0xd8cde3cc, 0x033496d3, 0x1b0ddd43, 0x221d6f93,
					0xd745729d, 0xc6da4bbe, 0x9372df96, 0xc3997aa8, 0xbe4738d5, 0x95f5b824,
					0x8b077974, 0x91ff2acd, 0x3e3c8970, 0xb0cccc25, 0xdf4a41b5, 0x1aaa83cc,
					0xab802b5b, 0x9346786a, 0xda7c9172, 0x11c0e5ef, 0x4febcd66, 0x45f31d90,
					0xbaf46606, 0x21a4ebf4, 0x10d5662e, 0x1b5a9f0e, 0xb4563323, 0x85f627b8,
					0x67baea50, 0x8c2db69a, 0x3177443d, 0xda1f8cce, 0xeaf7d034, 0x047b8892,
					0xec507176, 0x594663f0, 0x9d95d110, 0x3813268f, 0x2cadcad2, 0x09515190,
					0x6a5094ec, 0x6e516b15, 0x08ea1439, 0x90a19c00, 0x7a226467, 0xd1c7c67b,
					0x883f53db, 0xc6e5d69e, 0x0cc1705a, 0x00d7e17d, 0x8649e8ae, 0x9db9953d,
					0x2bb6f9d0, 0x3e1050c0, 0xdbb25609, 0xc70d5253, 0xf2a44b0b, 0x88da6d3a,
					0xb04099bb, 0xbd744c14, 0x89f6d528, 0x92eded15, 0xad5098bc, 0x4077846c
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x1, 0x7f20f9d3, 0x70c59f6f, 0x09a99f2b, 0xc29c063c, 0xd314e8f3, 0xa24af40b,
					0x3192b797, 0xbad33e25, 0x84cdd287, 0x0a9c0809, 0xed99dcd0, 0xde1ddca0,
					0x7e456f97, 0x90795f58, 0x8ff42f88, 0x46c6ebda, 0x210b4381, 0x1584beab,
					0xea01b7ef, 0x855b3e95, 0x575654df, 0x297bbafb, 0x77d1b139, 0x479506ae,
					0x71dff636, 0xab09c773, 0xb28a10b0, 0xdf69dfa9, 0x6aba0e65, 0x8bae9086,
					0xb54f14bd, 0xdc75f322
				};
			}
		}

		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0x1152b33a, 0x34a63caa, 0x14bd4558, 0xd1b57542, 0xc32e9a1b, 0x6c78bbdc,
					0x65292b48, 0x33307699, 0x9e49510f, 0x6465df13, 0x534d249a, 0x3b3ae2cd,
					0x5274a150, 0xfeef03ba, 0xe1df6e14, 0xd5ed3135, 0x152dadd4, 0x25086007,
					0xc94664c4, 0xf3183ada, 0x90e4eb88, 0xafbc13b4, 0xeb316da5, 0x8f822eba,
					0x868727c9, 0x691d7fea, 0xc0ff07f7, 0x967ae15b, 0x4d667f6c, 0x8ca28f57,
					0x51d458b8, 0xa81e1c3e
				};
			}
		}

		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0xbc9e0705, 0x2d67995d, 0x97350327, 0x3a6839ef, 0xb2c281b1, 0xc73306b1,
					0xa4385776, 0x55919b94, 0x790356a6, 0xc3c8cf2f, 0xf42563d3, 0xd03ae4ab,
					0x3adc5543, 0xdf1c3a93, 0xa89e1577, 0x6b58c8d7, 0x58ad7974, 0x21e33d28,
					0x5bd00a59, 0x0bf9d5ce, 0xff7cf725, 0xe818fb34, 0xdd2f3d2f, 0x22d5cfc5,
					0xd68bd585, 0x314fce82, 0x91a0c880, 0x29fc0c7b, 0x3b24edcc, 0x36e7657d,
					0x7f4ec9f3, 0xb5090067, 0x2d85195f, 0xfd217a73, 0x0ed43997, 0xef24805b,
					0xf8f01d79, 0x5015e762, 0xaa5dfa5b, 0x30714d09, 0xbd94a537, 0x8d53d6d4,
					0x7e0ee0ed, 0xeb5e8afe, 0xefaff6a9, 0xfb9e8b68, 0xc9dac138, 0xcd107bcd,
					0x29511770, 0xc72dc999, 0xcce3cf3e, 0x4aa1a315, 0xcd358309, 0x3eaa51ca,
					0xce23c07b, 0x4bc7b741, 0xa072f3d1, 0xbc847af0, 0x128391b7, 0x43a1a1ab,
					0x099c1a8f, 0x03b6fd1a, 0x35346a49, 0xfbf3f4b0, 0xc94e120f, 0xca2c3130,
					0x8eddcea5, 0x92a27f50, 0xd0dabd5d, 0x72cbfbbb, 0xe244dc44, 0x71b36732,
					0x512d949a, 0x9b07ff27, 0x9ae278b3, 0x5c67f5eb, 0x9bac1c9b, 0xba315d10,
					0x9e9e2f78, 0xedd50319, 0xc72aba34, 0x4591a520, 0x47c8e2b8, 0xd2581e44,
					0x915c8265, 0xa0f3298b, 0x16e3f98d, 0x1d3255eb, 0x67ad4451, 0xb8c6848d,
					0x5e363aa9, 0xb6d6b34c, 0x0f3d3c57, 0x3186c476, 0x2e3303ef, 0xa1f0b7e2,
					0xade9a761, 0x9c51030d, 0x1d6a8ab6, 0x1ee7c80e, 0x12d283ff, 0xec6621d7,
					0x342101aa, 0x175c3957, 0x68de3812, 0x4ccd898d, 0x64e48788, 0xdf95cd4b,
					0xb474ee50, 0x67c643dc, 0x70e61f88, 0x78870803, 0x1e08a43b, 0x2ad45c61,
					0x5309867a, 0x7b300960, 0x7393fb7a, 0xf2cf6514, 0x3b92e17b, 0x35dcc866,
					0x92925ee1, 0x43d5c8b5, 0x5dcd38a1, 0x4d108dfc, 0xd95c5c67, 0x9e8891ac,
					0xbcb2d55f, 0x625136a4
				};
			}
		}

		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0x54362a1c, 0x042584df, 0x8222b8fc, 0x8f310535, 0xffffaf04, 0x415816d0,
					0x909d9ab0, 0x11f120a4, 0x81467638, 0xad441f5f, 0x31e3bb49, 0x8c9b4fa7,
					0xca48b1b9, 0x438e460d, 0x8863c56a, 0xbd84cf0b, 0xdfc6627b, 0x12446a15,
					0xb2770965, 0x923a5ce7, 0x24e67a2b, 0x277ad7f5, 0xe324d6e5, 0x15bfdb9a,
					0x58f48720, 0x3006c299, 0xef162f92, 0xa7bf36ec, 0x98cc398a, 0x014cf447,
					0xa9c069d2, 0xec9c5268, 0x99293259, 0x5e98903f, 0x21418228, 0x095b2128,
					0xcdcc1254, 0x27aa19eb, 0x72794988, 0x35f28cc5, 0xe6402a7b, 0x0f630eb5,
					0xed484279, 0x311df827, 0x81288145, 0xee5b4502, 0x1cdd44bc, 0xf68dacff,
					0x2e389561, 0x9da07b0c, 0x0a19f8ec, 0xdfb82fe9, 0x1c05e30d, 0xc4ee5182,
					0xb7aed093, 0xe3bb2e26, 0x1de63a71, 0x82f3f24c, 0x228a0746, 0x30e5e4fa,
					0x2c936451, 0x190d9e98, 0x892a2bdf, 0x5593bec4
				};
			}
		}
	}

	public class Random1024a1024b: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0xe8c8bd9b, 0x0f2d1975, 0x8eaf4c7f, 0xa513ca34, 0x0539c22d, 0xbc167aec,
					0x92a2cf7d, 0xaf10cb82, 0xc27b9bdd, 0x93c1e4cd, 0x686bd61c, 0xe23f8486,
					0x99245663, 0xfb924c03, 0x99801568, 0x6fccc5e1, 0x57d4d4ec, 0x93a8267f,
					0x846ba741, 0x445b28b1, 0x5e898372, 0x670a590d, 0x2897594e, 0xcd480b48,
					0xa6b8711c, 0xc5046cb6, 0xcc79b3b2, 0x445cd544, 0xefcfbe5d, 0x6b63c7f6,
					0x51117990, 0x7c80efbe
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0xbee17847, 0xe569d2e5, 0xac236b39, 0x0c489f84, 0xb6696bb6, 0x15f7fe55,
					0xfb77131c, 0x873fffe9, 0xb257e171, 0x2c828f55, 0xfc928959, 0x47a85c04,
					0xebf46c67, 0x5abb6902, 0x5be13139, 0x5a511864, 0xf57c0143, 0x13dcc6de,
					0xcc136221, 0x15f4f853, 0x58f52105, 0x4548a1d8, 0x9c0800ea, 0x3dba87e8,
					0x9cfa44f3, 0x59122da2, 0x17fe648c, 0x0a5e6c8d, 0x9706505e, 0x21d8d4ad,
					0xe8d92591, 0xf5e91c22
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
					0x1, 0xa7aa35e2, 0xf496ec5b, 0x3ad2b7b8, 0xb15c69b8, 0xbba32de3, 0xd20e7942,
					0x8e19e29a, 0x3650cb6c, 0x74d37d4e, 0xc0447423, 0x64fe5f76, 0x29e7e08b,
					0x8518c2cb, 0x564db505, 0xf56146a1, 0xca1dde46, 0x4d50d62f, 0xa784ed5e,
					0x507f0962, 0x5a502104, 0xb77ea477, 0xac52fae5, 0xc49f5a39, 0x0b029331,
					0x43b2b610, 0x1e169a58, 0xe478183e, 0x4ebb41d2, 0x86d60ebb, 0x8d3c9ca4,
					0x39ea9f22, 0x726a0be0
				};
			}
		}

		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0x29e74553, 0x29c3468f, 0xe28be146, 0x98cb2aaf, 0x4ed05677, 0xa61e7c96,
					0x972bbc61, 0x27d0cb99, 0x1023ba6c, 0x673f5577, 0x6bd94cc3, 0x9a972881,
					0xad2fe9fc, 0xa0d6e301, 0x3d9ee42f, 0x157bad7c, 0x6258d3a9, 0x7fcb5fa0,
					0xb8584520, 0x2e66305e, 0x0594626d, 0x21c1b734, 0x8c8f5864, 0x8f8d8360,
					0x09be2c29, 0x6bf23f14, 0xb47b4f26, 0x39fe68b7, 0x58c96dff, 0x498af348,
					0x683853fe, 0x8697d39c
				};
			}
		}

		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0xad920287, 0x33b4e05d, 0xf6782118, 0xc4c63752, 0x068997db, 0xbfee40d7,
					0x2d4f2993, 0x7eebbea1, 0x1457805b, 0xff8fff22, 0x91057d22, 0x591de9d5,
					0x0891bc5a, 0xa972a76b, 0xf28a29dc, 0xc391c2a0, 0x54973886, 0x624b2e32,
					0x7b606c23, 0x2d55978e, 0x5b88b68d, 0xbdddbad5, 0x8d86ac43, 0xef99c973,
					0x4442b4fd, 0xd56672d2, 0xefbbb6ac, 0x9103ca5f, 0xca7bd8f9, 0x407c45fd,
					0x6bb9557d, 0x7dfac4b7, 0x0054bc14, 0x218c222e, 0x3483ef14, 0x8e737763,
					0x1b831e5d, 0x17d28c9f, 0x3a0b544e, 0x49aa11a2, 0x7a244f22, 0xa0397691,
					0xa7df289f, 0x75c8db6e, 0x2b66ebd9, 0xe5f81ef6, 0x3d1482e4, 0x088605c4,
					0xf93723e2, 0x414aca69, 0x4902937e, 0x645bc986, 0xd9a76e10, 0x21c21a0d,
					0x349c40ba, 0xc63fe272, 0x2d3b29c6, 0xadd913bd, 0x1095cfb6, 0x980d74ed,
					0x5c0c07c2, 0x05a67c9e, 0x2902ea37, 0xad469f3c
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x1
				};
			}
		}

		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0x29e74553, 0x29c3468f, 0xe28be146, 0x98cb2aaf, 0x4ed05677, 0xa61e7c96,
					0x972bbc61, 0x27d0cb99, 0x1023ba6c, 0x673f5577, 0x6bd94cc3, 0x9a972881,
					0xad2fe9fc, 0xa0d6e301, 0x3d9ee42f, 0x157bad7c, 0x6258d3a9, 0x7fcb5fa0,
					0xb8584520, 0x2e66305e, 0x0594626d, 0x21c1b734, 0x8c8f5864, 0x8f8d8360,
					0x09be2c29, 0x6bf23f14, 0xb47b4f26, 0x39fe68b7, 0x58c96dff, 0x498af348,
					0x683853fe, 0x8697d39c
				};
			}
		}

		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0xd3ac7511, 0xda37c6f0, 0xd3676686, 0x3e75ca2d, 0x2c34ab61, 0x220493b2,
					0x9c0ccf97, 0x9e154a16, 0xd059f25d, 0x88c203f7, 0xb4e8b3b2, 0x1e087384,
					0x14d55512, 0x44a30dcc, 0x17d7153a, 0xd7f52e45, 0x02274060, 0x76e0336d,
					0xa8608fbd, 0xf42fca56, 0xe7b95e9d, 0x787c27aa, 0xafc11f13, 0x1b66a178,
					0x97710f23, 0xa4bdcf90, 0xcf4cafd9, 0x288554ea, 0x294b0a79, 0xc8d02fd9,
					0x48810d2d, 0x2ea649eb, 0x00fd8df6, 0x6c2c0e3b, 0x09e97167, 0xf78c7c9b,
					0x8287c66b, 0xaf26b8c1, 0x1b0a7c69, 0x715a86c8, 0x416d1c3f, 0xa74de987,
					0x119bc8c0, 0x1d5cf66a, 0xbd9e98d4, 0x5c1f72f0, 0x461f0311, 0x3fe6c57b,
					0xbd827dfe, 0x464e17b0, 0x6746a287, 0xfedf1131, 0x2e4a9384, 0x68b74835,
					0x7f1890bf, 0x3c8f861b, 0x2ef49f89, 0x9be1f04b, 0xc7d14cc9, 0x57c09b0e,
					0x32e49f8c, 0xcb2e2d3c, 0x43787af0, 0xae845104
				};
			}
		}

		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0x8e53751f, 0x5f6dd455, 0xe73df53f, 0x6cca88f3, 0xd7af11fa, 0x32003a1f,
					0xf0140015, 0xbbaf870f, 0x202242ed, 0xf0515cc7, 0x2cd9d3cc, 0x55c27930,
					0x58517126, 0x90ac5a65, 0xdf1c572f, 0xa734dfea, 0x1178b713, 0x1463153d,
					0x04e3f702, 0x25194cf2, 0xf9dcb3db, 0xe213b47b, 0x4666029a, 0x51374275,
					0xe9b85f80, 0x8d4aaa6e, 0x8ddd7735, 0xa6dda66c, 0x9594a530, 0x5a7f352c,
					0x3b48c122, 0x50efb29a, 0xc79bd6b4, 0x2440b7ce, 0xe4715871, 0x690421da,
					0xb3d24cd4, 0x20bc0fd3, 0x22200cd2, 0x0b06278e, 0x4ec52207, 0xace37b7a,
					0xe512d8bf, 0x9320fae5, 0xe1221b0d, 0x29ee05b6, 0xda382119, 0xe0458dbb,
					0xc79fc5ad, 0x073db465, 0x9ebc66f3, 0x3a00f27f, 0xb5f59316, 0xd84c2156,
					0xd4797e82, 0xb280e61b, 0xff54c2ea, 0xce29dfb1, 0x3f304113, 0xf9926e5e,
					0xed93ce71, 0xf5b54271, 0x320fb6a1, 0x4cfb7484
				};
			}
		}
	}

	public class Random648a592b: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0x80, 0xe25d5da8, 0x21955bf0, 0x03a657c4, 0xfc958503, 0x439d9b34,
					0x751c1ca6, 0xc0efa543, 0x6908a832, 0x5db264fa, 0xb089fca4, 0x629b76a5,
					0xf156b58e, 0xc18706ad, 0x61bfc9b7, 0x5be1f7ce, 0xdee2ef8f, 0xb0f949d9,
					0x5a118b6b, 0x5266ffdd, 0x090019ff
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0xf433, 0xdad65473, 0x1679f349, 0x7c8fbada, 0x2fa98df4, 0x3a4f47c9,
					0x39b128eb, 0xa7ff55ab, 0x8ea2596b, 0x283f32a8, 0xcb1e7271, 0x9b3b77ae,
					0x371adf91, 0xbc6814b5, 0xd75016f2, 0x22624a3e, 0x2281cda3, 0xd80e824d,
					0x8ab69459
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
					0x80, 0xe25d5da8, 0x21965023, 0xde7cac38, 0x130f784c, 0xc02d560e,
					0xa4c5aa9a, 0xfb3eed0c, 0xa2b9d11e, 0x05b1baa6, 0x3f2c560f, 0x8adaa94e,
					0xbc752800, 0x5cc27e5b, 0x98daa949, 0x184a0c84, 0xb6330681, 0xd35b9417,
					0x7c93590f, 0x2a75822a, 0x93b6ae58
				};
			}
		}
		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0x80, 0xe25d5da8, 0x219467bc, 0x28d00351, 0xe61b91b9, 0xc70de05a,
					0x45728eb2, 0x86a05d7a, 0x2f577f46, 0xb5b30f4f, 0x21e7a339, 0x3a5c43fd,
					0x2638431d, 0x264b8eff, 0x2aa4ea25, 0x9f79e319, 0x0792d89d, 0x8e96ff9b,
					0x378fbdc7, 0x7a587d8f, 0x7e4985a6
				};
			}
		}
		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0x7af1dc, 0x42892886, 0x270d1508, 0x35afbef2, 0xf32044cc, 0x7ec39e7c,
					0xd5be5d85, 0xe3e4d670, 0x8f00e646, 0x610c85b9, 0xf49d42aa, 0xd58c679d,
					0xc518cbd4, 0x810d256b, 0xcf4bf1e6, 0x3d950962, 0xa45fed9c, 0x9961651f,
					0xf7ee8ff4, 0x69726f4e, 0x66c4e1bc, 0xd254aaa1, 0xc8fb80db, 0xb7186aa6,
					0xd0a3485b, 0x1d56955c, 0x711f41b6, 0xd8235f7a, 0xf4bc1659, 0xbfd046f3,
					0xa83904d6, 0x85d2437d, 0xa29ef4be, 0x54bf7242, 0x72e297e5, 0xc11d231c,
					0xc9642775, 0x95e680d5, 0x215a75a7
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x871c53, 0x1f53c91f
				};
			}
		}
		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0x5909, 0xfe7418f1, 0x18cafdb8, 0x78b332d4, 0x8734f3ec, 0x248f2ac6,
					0xf31487c2, 0xd6b17b47, 0x4e035271, 0x7d8eddd3, 0xa1ec3476, 0xaa2ed8c9,
					0x15fd7425, 0x6bd31fe4, 0x1abfee03, 0xee7a13d7, 0x64c250dc, 0xf50b6c8c,
					0x028f4238
				};
			}
		}
		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0x40e3, 0x2586a38b, 0x9f13a3ae, 0xd6690553, 0xd60e7eb2, 0x3b040fd1,
					0xe23ecfa7, 0x46ab3047, 0x16419030, 0xb81c68be, 0x033a0147, 0xf62d84a7,
					0xbb0dc6a0, 0x9d817e08, 0x04604d2a, 0x87a4bc14, 0xdfd159bc, 0x2d00e161,
					0x0d089cc2, 0x7b1f4599, 0xa3b67334, 0xa3ac3a3b, 0x26d08e66, 0xfe40ac1b,
					0x0b51cd6d, 0xe0cdb969, 0x15053d95, 0x09105000, 0x13848bc3, 0x43ac0edf,
					0x5de02efc, 0xff030324, 0x0153a0ad, 0x21348f68, 0x9e351771, 0xc4485dbf,
					0xc7e7bcbd, 0xbf36d47e, 0xdfcff8ac, 0xd17be619, 0xf0a3cc01
				};
			}
		}
		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0xe8f2e3a9, 0x7d72c3f8, 0xbae80572, 0xbfa765fe, 0xf1f2ae8f, 0x014d09f4,
					0xbb00e418, 0x84108f06, 0xb17d9cab, 0x1e7fbb68, 0x96495444, 0x71e3b073,
					0x11b725ec, 0xb2412093, 0x99e3013d, 0x6146b623, 0x50a06090, 0xefaa710b,
					0x73a273bc, 0x10cb33ce, 0x0788144b, 0xc2db435c, 0x23446289, 0x313b9d18,
					0x596e6c25, 0xab12758e, 0xc00bf8bd, 0x18800070, 0x29117981, 0xc413df55,
					0xdb4d943f, 0x54331203, 0xa895d355, 0xf6d83621, 0x90ab9202, 0x9564fbb1,
					0x388306f1
				};
			}
		}
	}

	public class Carry10_1589a612b: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0x192483, 0xedfc2a5a, 0xaf6aa3ad, 0xb3c27ee0, 0x1dbb97c0, 0x5a89358e,
					0x0bc99cc9, 0xee284efd, 0x17d9ca4e, 0x5cb4db86, 0x7f0ddc13, 0xc7711ad4,
					0x06f375e9, 0x71ec7ca6, 0xbb2eb849, 0x6914bec9, 0x7196db80, 0x5e38d3a6,
					0xb33d1976, 0x26e7681b, 0x5a189160, 0xa0c2a1c3, 0xdbe797bb, 0xbe2b21ad,
					0x1014dd56, 0x5a438a13, 0xc48ca63c, 0x022002e1, 0x4a0989d1, 0xbd3f2d76,
					0x38f7059a, 0xe428965c, 0x02a3f023, 0x2cf84c51, 0xee3c2278, 0x1406d5a1,
					0xe2e48a81, 0x4f66b880, 0x25b58362, 0x0733403c, 0x9ae2115c, 0x6c9e6310,
					0xf6e2b83b, 0x88b1229e, 0xf2e7acc6, 0x0e16ec9a, 0xf2cacba4, 0xf5466060,
					0x8a80d714, 0x6a2e6004
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0x10, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
					0x192483, 0xedfc2a5a, 0xaf6aa3ad, 0xb3c27ee0, 0x1dbb97c0, 0x5a89358e,
					0x0bc99cc9, 0xee284efd, 0x17d9ca4e, 0x5cb4db86, 0x7f0ddc13, 0xc7711ad4,
					0x06f375e9, 0x71ec7ca6, 0xbb2eb849, 0x6914bec9, 0x7196db80, 0x5e38d3a6,
					0xb33d1976, 0x26e7681b, 0x5a189160, 0xa0c2a1c3, 0xdbe797bb, 0xbe2b21ad,
					0x1014dd56, 0x5a438a13, 0xc48ca63c, 0x022002e1, 0x4a0989d1, 0xbd3f2d76,
					0x38f705aa, 0xe428965c, 0x02a3f023, 0x2cf84c51, 0xee3c2278, 0x1406d5a1,
					0xe2e48a81, 0x4f66b880, 0x25b58362, 0x0733403c, 0x9ae2115c, 0x6c9e6310,
					0xf6e2b83b, 0x88b1229e, 0xf2e7acc6, 0x0e16ec9a, 0xf2cacba4, 0xf5466060,
					0x8a80d714, 0x6a2e6004
				};
			}
		}

		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0x192483, 0xedfc2a5a, 0xaf6aa3ad, 0xb3c27ee0, 0x1dbb97c0, 0x5a89358e,
					0x0bc99cc9, 0xee284efd, 0x17d9ca4e, 0x5cb4db86, 0x7f0ddc13, 0xc7711ad4,
					0x06f375e9, 0x71ec7ca6, 0xbb2eb849, 0x6914bec9, 0x7196db80, 0x5e38d3a6,
					0xb33d1976, 0x26e7681b, 0x5a189160, 0xa0c2a1c3, 0xdbe797bb, 0xbe2b21ad,
					0x1014dd56, 0x5a438a13, 0xc48ca63c, 0x022002e1, 0x4a0989d1, 0xbd3f2d76,
					0x38f7058a, 0xe428965c, 0x02a3f023, 0x2cf84c51, 0xee3c2278, 0x1406d5a1,
					0xe2e48a81, 0x4f66b880, 0x25b58362, 0x0733403c, 0x9ae2115c, 0x6c9e6310,
					0xf6e2b83b, 0x88b1229e, 0xf2e7acc6, 0x0e16ec9a, 0xf2cacba4, 0xf5466060,
					0x8a80d714, 0x6a2e6004
				};
			}
		}

		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0x192483e, 0xdfc2a5aa, 0xf6aa3adb, 0x3c27ee01, 0xdbb97c05, 0xa89358e0,
					0xbc99cc9e, 0xe284efd1, 0x7d9ca4e5, 0xcb4db867, 0xf0ddc13c, 0x7711ad40,
					0x6f375e97, 0x1ec7ca6b, 0xb2eb8496, 0x914bec97, 0x196db805, 0xe38d3a6b,
					0x33d19762, 0x6e7681b5, 0xa189160a, 0x0c2a1c3d, 0xbe797bbb, 0xe2b21ad1,
					0x014dd565, 0xa438a13c, 0x48ca63c0, 0x22002e14, 0xa0989d1b, 0xd3f2d763,
					0x8f7059ae, 0x428965c0, 0x2a3f0232, 0xcf84c51e, 0xe3c22781, 0x406d5a1e,
					0x2e48a814, 0xf66b8802, 0x5b583620, 0x733403c9, 0xae2115c6, 0xc9e6310f,
					0x6e2b83b8, 0x8b1229ef, 0x2e7acc60, 0xe16ec9af, 0x2cacba4f, 0x54660608,
					0xa80d7146, 0xa2e60040, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x19248, 0x3edfc2a5, 0xaaf6aa3a, 0xdb3c27ee, 0x01dbb97c, 0x05a89358,
					0xe0bc99cc, 0x9ee284ef, 0xd17d9ca4, 0xe5cb4db8, 0x67f0ddc1, 0x3c7711ad,
					0x406f375e, 0x971ec7ca, 0x6bb2eb84, 0x96914bec, 0x97196db8, 0x05e38d3a,
					0x6b33d197, 0x626e7681, 0xb5a18916, 0x0a0c2a1c, 0x3dbe797b, 0xbbe2b21a,
					0xd1014dd5, 0x65a438a1, 0x3c48ca63, 0xc022002e, 0x14a0989d, 0x1bd3f2d7,
					0x638f7059
				};
			}
		}

		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0xa, 0xe428965c, 0x02a3f023, 0x2cf84c51, 0xee3c2278, 0x1406d5a1, 0xe2e48a81,
					0x4f66b880, 0x25b58362, 0x0733403c, 0x9ae2115c, 0x6c9e6310, 0xf6e2b83b,
					0x88b1229e, 0xf2e7acc6, 0x0e16ec9a, 0xf2cacba4, 0xf5466060, 0x8a80d714,
					0x6a2e6004
				};
			}
		}

		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0x278, 0x26f9da2c, 0x9eeffbc8, 0xcf702902, 0x1e35c385, 0x3b2b2610,
					0xb02ab71a, 0x3165f09c, 0xdb4768c2, 0x84de913e, 0x37474521, 0x95ea5714,
					0x722b78fd, 0x703600e4, 0x1e883c87, 0x286772c1, 0xa94e590f, 0xb93d67fd,
					0xd0194699, 0x4ceb1741, 0x5e4f3d86, 0x30365af8, 0xb22b3d2a, 0x4ea8ffaf,
					0xe046488c, 0x5e8b4c26, 0xfbc177e6, 0x9c00d7b1, 0x1c597e95, 0xe3c08605,
					0xe652147c, 0x5ed54f4e, 0x643d3b19, 0xec7e7f4f, 0xc6e1ef37, 0x32ead8f3,
					0xdcc21123, 0x6b61703f, 0x8e9f8ada, 0x83537069, 0xb7d3bf76, 0x245e39db,
					0x25d07352, 0xf844cd2a, 0x46dd334c, 0xd71fc2f1, 0x0709a11f, 0x553907da,
					0x748d17f2, 0x065a05de, 0x07363794, 0xc4790752, 0x6c30b993, 0x0e0d67ff,
					0xefbd1f94, 0xa07463dd, 0x5ba54f91, 0xe7e21018, 0x4f37e81b, 0x89f6434f,
					0x3cc714be, 0x95c8588c, 0x73b72300, 0x552f80dc, 0xfa6b1ec6, 0x383f4843,
					0xcebe7df7, 0x09a867a1, 0xbe736fb7, 0x4570ab4e, 0xd024b70b, 0x95f97b71,
					0x00db547e, 0x9b6430c8, 0xf3981995, 0xfa8bf51b, 0x49ff9a93, 0xa9d1ee7b,
					0xd22e4ed7, 0xaccd79e9, 0x4ecbe913, 0x60d98c62, 0xb2963be9, 0xbd288c77,
					0xb048ce48, 0x13af506a, 0xfab605c1, 0xd2c0b9c8, 0x2b47137e, 0xf0933545,
					0x67fb359a, 0x4d6f8c23, 0x81086afc, 0x6456062a, 0x32e42a1f, 0xa2d6fc80,
					0x2d47f9f0, 0x60927cde, 0xfc902889, 0xf5730010
				};
			}
		}

		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0x100, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000
				};
			}
		}
	}

	public class Carry10_2048a512b: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0x1, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0xa4e3d73d, 0x75eac131, 0x302dc7c5, 0xa47bcd7e, 0x716087de, 0xc58e1aff,
					0xb2a66ae9, 0x6c00067b, 0x5d8a1d8b, 0x0a424a97, 0xf02a5516, 0x5ce13f11,
					0x9734eddc, 0xb26f0a2d, 0xdc2eea99, 0xac218b97
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
					0x1, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0xa4e3d73d, 0x75eac131, 0x302dc7c5, 0xa47bcd7e, 0x716087de, 0xc58e1aff,
					0xb2a66ae9, 0x6c00067b, 0x5d8a1d8b, 0x0a424a97, 0xf02a5516, 0x5ce13f11,
					0x9734eddc, 0xb26f0a2d, 0xdc2eea99, 0xac218b97
				};
			}
		}

		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0x5b1c28c2, 0x8a153ece, 0xcfd2383a, 0x5b843281, 0x8e9f7821, 0x3a71e500,
					0x4d599516, 0x93fff984, 0xa275e274, 0xf5bdb568, 0x0fd5aae9, 0xa31ec0ee,
					0x68cb1223, 0x4d90f5d2, 0x23d11566, 0x53de7469
				};
			}
		}

		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0xa4e3d73d, 0x75eac131, 0x302dc7c5, 0xa47bcd7e, 0x716087de, 0xc58e1aff,
					0xb2a66ae9, 0x6c00067b, 0x5d8a1d8b, 0x0a424a97, 0xf02a5516, 0x5ce13f11,
					0x9734eddc, 0xb26f0a2d, 0xdc2eea99, 0xac218b97, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x1, 0x8d73ed52, 0x63ccf8dd, 0xbdf7d56b, 0x5745e24e, 0x33c99f37, 0x50504831,
					0xbbc7006e, 0xff3c56a3, 0xc6588019, 0x455441a7, 0x2e587257, 0x74865b5d,
					0x72d47e06, 0xf0087241, 0xa83b2419, 0x6822bd94, 0x8414d0e8, 0xf829760f,
					0x3f26b3f7, 0x2e593e61, 0xed63d275, 0x47733c04, 0x5cb283d8, 0xa390d3e9,
					0xae81f5ef, 0xeeaee18c, 0xf4f7204c, 0xdafdfe69, 0xc5589266, 0xca6cf579,
					0x51bf0855, 0x06c615c0, 0x98b8c458, 0xd35a666f, 0x5b703ba2, 0xecfc2df3,
					0x125547e8, 0x548104fc, 0xd0d9417e, 0x763bdbc0, 0x5de3e15b, 0x531c9668,
					0x2724724e, 0x425710f6, 0x484386f9, 0x416a7d95, 0x27ee2792, 0x5dc52c9e
				};
			}
		}

		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0x9b7fd926, 0x0a7e23f3, 0x81fd3d29, 0x515de74b, 0xead6979c, 0x0a528677,
					0xe3fed5f9, 0xea231c08, 0x7af3b842, 0x2552eb1d, 0xccfa1564, 0xd60e4d46,
					0x0754ca39, 0x5aa50851, 0xa02788ed, 0xb91ae4ce
				};
			}
		}

		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0x1, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000
				};
			}
		}

		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0x6a34b68e, 0x2b0c30ab, 0x5ba73560, 0x1460a1b5, 0x25249ae7, 0x657f4d83,
					0xffc12095, 0x4effc129, 0xf17a97ee, 0x5e353b56, 0x7031fd7f, 0x362f9cc2,
					0xc710f9b6, 0xf0cf83a1, 0xdcd3d217, 0x4a753e4c, 0x8302911c, 0x7c70914e,
					0xaf2372d5, 0x8f133c24, 0x8ee893ea, 0x95c92a17, 0x0161ecf9, 0xc884260e,
					0x87da5987, 0x64f242e6, 0xf5ff8d04, 0xc775e988, 0xc24a9407, 0x71b5adfc,
					0x3949fe76, 0x310b5311
				};
			}
		}
	}

	public class Carry10_612a612b: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0x10, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0x8, 0x73ca4b23, 0x1b754264, 0x22aaac98, 0x0abfe205, 0xf2fd1c98, 0x76a78fb8,
					0x7398b632, 0x306eacc9, 0xee3a8700, 0xab1bd9e4, 0xf5f53e01, 0x0b6ebccf,
					0x000e5f13, 0xdd29a13f, 0x49ed6d71, 0x497d8e79, 0x3f8d15b9, 0x3ab921b3,
					0xd32f5a18
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
					0x18, 0x73ca4b23, 0x1b754264, 0x22aaac98, 0x0abfe205, 0xf2fd1c98,
					0x76a78fb8, 0x7398b632, 0x306eacc9, 0xee3a8700, 0xab1bd9e4, 0xf5f53e01,
					0x0b6ebccf, 0x000e5f13, 0xdd29a13f, 0x49ed6d71, 0x497d8e79, 0x3f8d15b9,
					0x3ab921b3, 0xd32f5a18
				};
			}
		}

		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0x7, 0x8c35b4dc, 0xe48abd9b, 0xdd555367, 0xf5401dfa, 0x0d02e367, 0x89587047,
					0x8c6749cd, 0xcf915336, 0x11c578ff, 0x54e4261b, 0x0a0ac1fe, 0xf4914330,
					0xfff1a0ec, 0x22d65ec0, 0xb612928e, 0xb6827186, 0xc072ea46, 0xc546de4c,
					0x2cd0a5e8
				};
			}
		}

		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0x87, 0x3ca4b231, 0xb7542642, 0x2aaac980, 0xabfe205f, 0x2fd1c987,
					0x6a78fb87, 0x398b6323, 0x06eacc9e, 0xe3a8700a, 0xb1bd9e4f, 0x5f53e010,
					0xb6ebccf0, 0x00e5f13d, 0xd29a13f4, 0x9ed6d714, 0x97d8e793, 0xf8d15b93,
					0xab921b3d, 0x32f5a180, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x1
				};
			}
		}

		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0x7, 0x8c35b4dc, 0xe48abd9b, 0xdd555367, 0xf5401dfa, 0x0d02e367, 0x89587047,
					0x8c6749cd, 0xcf915336, 0x11c578ff, 0x54e4261b, 0x0a0ac1fe, 0xf4914330,
					0xfff1a0ec, 0x22d65ec0, 0xb612928e, 0xb6827186, 0xc072ea46, 0xc546de4c,
					0x2cd0a5e8
				};
			}
		}

		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0x100, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000
				};
			}
		}

		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0x47, 0x7104118d, 0xeb746bdc, 0x21bf9b52, 0x01733d95, 0x88b143b1,
					0x87f0fb41, 0x65df0c96, 0xc490f2a6, 0x12c8f6ea, 0x252973b1, 0xc0ea701c,
					0x71f6701c, 0x69eb7698, 0x2cde7627, 0x137fdad2, 0xc59510bb, 0x64483a25,
					0xa16d5cb5, 0xea6d7512, 0x84c280bf, 0xa3b9bf2a, 0x1e289f48, 0x20d752ac,
					0x8981bc2d, 0xbf6d17e3, 0xdb46db1e, 0x55564d03, 0xbdba86b5, 0x5f4efcf0,
					0x4adf8db1, 0xb32ea58d, 0x15b3d80c, 0xc1792429, 0xba324943, 0xc4c05ae5,
					0x43b0b0b6, 0x4e5944d5, 0xc484e240
				};
			}
		}
	}

	public class Carry10_924a592b: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0x10000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0xeec8, 0xa90762a8, 0xdc8618f1, 0x4f8d783d, 0x0748ca77, 0x44fdeff4,
					0xb34f7ca0, 0x81b11736, 0x3ad01b51, 0xba49b2d5, 0x12d298be, 0x730bb558,
					0xb7abee99, 0xcb510110, 0x089d9c00, 0xbbade215, 0x7906beaa, 0xe96adc81,
					0xb888d515
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
					0x10000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x0000eec8, 0xa90762a8,
					0xdc8618f1, 0x4f8d783d, 0x0748ca77, 0x44fdeff4, 0xb34f7ca0, 0x81b11736,
					0x3ad01b51, 0xba49b2d5, 0x12d298be, 0x730bb558, 0xb7abee99, 0xcb510110,
					0x089d9c00, 0xbbade215, 0x7906beaa, 0xe96adc81, 0xb888d515
				};
			}
		}

		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0xfffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffff1137, 0x56f89d57,
					0x2379e70e, 0xb07287c2, 0xf8b73588, 0xbb02100b, 0x4cb0835f, 0x7e4ee8c9,
					0xc52fe4ae, 0x45b64d2a, 0xed2d6741, 0x8cf44aa7, 0x48541166, 0x34aefeef,
					0xf76263ff, 0x44521dea, 0x86f94155, 0x1695237e, 0x47772aeb
				};
			}
		}

		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0xeec, 0x8a90762a, 0x8dc8618f, 0x14f8d783, 0xd0748ca7, 0x744fdeff,
					0x4b34f7ca, 0x081b1173, 0x63ad01b5, 0x1ba49b2d, 0x512d298b, 0xe730bb55,
					0x8b7abee9, 0x9cb51011, 0x0089d9c0, 0x0bbade21, 0x57906bea, 0xae96adc8,
					0x1b888d51, 0x50000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x1127, 0x51b35432, 0xce7b7c2d, 0x97083ec4, 0x26a0fd6f, 0x49534242,
					0x190ffd84, 0x0a3cf05b, 0x3d4666fa, 0xa8a72cf5, 0x74b047c8
				};
			}
		}

		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0x5a34, 0xc1bc59cf, 0x14981300, 0xa6aa3b5b, 0x2d60df09, 0x0d02e128,
					0x0340c427, 0xc6069082, 0x8ec3ccdf, 0xe93e223f, 0x5e346e10, 0xec7f1b4c,
					0x19777e61, 0x6d9d1012, 0x56c0dc81, 0x17f763a9, 0x526bc8e2, 0xb84858c0,
					0xdf90b498
				};
			}
		}

		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0x1000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000
				};
			}
		}

		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0xdeb9b792, 0x4696e25b, 0x66527d74, 0x29e8ba21, 0x7aabbac1, 0xf92534aa,
					0xcbb6af91, 0x81c29141, 0xb3eefb25, 0x902b08c1, 0xe5e48df5, 0x539a9458,
					0x586e65a8, 0x61cf19a4, 0xb7087d2e, 0xa86df134, 0xf7c037e2, 0x358db628,
					0xb91e6f64, 0x1603d551, 0x8ccbb468, 0x9510d3b9, 0x0b3b4413, 0x3e4d261a,
					0x4bf7be1a, 0xf71c88ac, 0xbf509806, 0x0b25bb31, 0x022b093f, 0x47c200aa,
					0x73bcbbc5, 0x786bcfd7, 0x01e83960, 0xf7a5257d, 0x844747fb, 0x5c8bd29b,
					0x47abf3b9
				};
			}
		}
	}
	
	public class CarryFF_2048a512b: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0x1, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0xd3d9d10e, 0xb2f0b0d3, 0x1f13609b, 0x39b039c5, 0x35de9114, 0x8f4e7bc6,
					0xf22166c4, 0x385ef08f, 0xcee7b659, 0xef7b9e0d, 0xa8db00d2, 0x2b078e0d,
					0x11591b10, 0xd6b0c621, 0x6ff27375, 0xf5e4aa7a
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
					0x2, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0xd3d9d10e, 0xb2f0b0d3, 0x1f13609b, 0x39b039c5, 0x35de9114, 0x8f4e7bc6,
					0xf22166c4, 0x385ef08f, 0xcee7b659, 0xef7b9e0d, 0xa8db00d2, 0x2b078e0d,
					0x11591b10, 0xd6b0c621, 0x6ff27375, 0xf5e4aa79
				};
			}
		}

		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0x1, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0x2c262ef1, 0x4d0f4f2c, 0xe0ec9f64, 0xc64fc63a, 0xca216eeb, 0x70b18439,
					0x0dde993b, 0xc7a10f70, 0x311849a6, 0x108461f2, 0x5724ff2d, 0xd4f871f2,
					0xeea6e4ef, 0x294f39de, 0x900d8c8a, 0x0a1b5585
				};
			}
		}

		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0x1, 0xa7b3a21d, 0x65e161a6, 0x3e26c136, 0x7360738a, 0x6bbd2229, 0x1e9cf78d,
					0xe442cd88, 0x70bde11f, 0x9dcf6cb3, 0xdef73c1b, 0x51b601a4, 0x560f1c1a,
					0x22b23621, 0xad618c42, 0xdfe4e6eb, 0xebc954f3, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0x2c262ef1, 0x4d0f4f2c,
					0xe0ec9f64, 0xc64fc63a, 0xca216eeb, 0x70b18439, 0x0dde993b, 0xc7a10f70,
					0x311849a6, 0x108461f2, 0x5724ff2d, 0xd4f871f2, 0xeea6e4ef, 0x294f39de,
					0x900d8c8a, 0x0a1b5586
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x2, 0x6ab30e86, 0xa5eab10c, 0xf0b110fb, 0x4bbdab3e, 0x7077f470, 0xd425b304,
					0xd27d54c9, 0xf9dad24b, 0xf3b6936d, 0x829f2349, 0x5a49aac1, 0x10c451c4,
					0x2e2ce79d, 0xbb2876b1, 0xfd6c2716, 0xa03d330e, 0x95e99dc8, 0x66bba462,
					0x9f896b2b, 0x75282af8, 0xb3d317ff, 0x978d1249, 0x34b9dd27, 0x47e9d339,
					0x52900cd0, 0xc916641c, 0xcdddd3b5, 0xf8cf823f, 0xd9a48128, 0x9104dab4,
					0x1113dc0d, 0x537298d1, 0x5295c52d, 0xf1583e39, 0x38697c80, 0xaed83a88,
					0x50b49c28, 0xd7639120, 0x7ce405c9, 0xd1ff44c7, 0xf3ec6b98, 0x660fe843,
					0xc1dead4a, 0x7ff025af, 0x6055a2e6, 0x9070248a, 0xd3d0e8bd, 0x510b3167
				};
			}
		}

		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0x128ca249, 0x61b38561, 0x5865396f, 0xda82264b, 0xf7650aca, 0xc618f1d1,
					0x58a82223, 0x31f0e3d2, 0x5ca39e46, 0xb9e5e9d4, 0xc980b4bb, 0x6e6c387a,
					0xf220f6d7, 0x5e55e72c, 0x6f0aa0a3, 0x5f200ee9
				};
			}
		}

		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0x3, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xfffffffc, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000001
				};
			}
		}

		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0xaf50c7f2, 0x509d208c, 0xdb444820, 0x4f5080ac, 0xf68e26d4, 0x3266b318,
					0x795ae7c9, 0x909bb5f9, 0x595442ea, 0xa7c63be2, 0x0860a3cc, 0x18bfe855,
					0x1343dd40, 0x49fb6aca, 0x09698366, 0xa5a58482, 0x4ce53840, 0x85332cd5,
					0x24eb28fc, 0x4027d419, 0xb81f6430, 0xac6e4e0b, 0x6e67b5d7, 0xfbe3c71f,
					0xd0a4e850, 0xed7e4428, 0x87b66bb8, 0xf8d65494, 0xc6bbd534, 0x46bd0ce2,
					0x3b04ea11, 0x9ed64224
				};
			}
		}
	}

	public class CarryFF_2048a2048b: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0x1, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0xb731d57c, 0x8e2f779e, 0x7d1fab2a, 0x364776a2, 0x2cc5c154, 0xa74a02e3,
					0x4743c390, 0xc08a2586, 0x5456059e, 0xd343063f, 0x091b1db7, 0x6d58e725,
					0xdb66c9ce, 0x6af4c80f, 0x274553cd, 0xde2fa475, 0xb0b1f796, 0x02c09afc,
					0xea7a64be, 0x2c95dc03, 0x989777c9, 0xea40893e, 0x9a9c1d65, 0xf096fa51,
					0xc4f33acf, 0x5350c4e7, 0x1815cc9b, 0x9821cb7b, 0x63122665, 0x09294d5f,
					0xb00c1de2, 0x87b5a827, 0xcb57546b, 0x49edabfb, 0x20e281e0, 0x00de2802,
					0xc2e1d451, 0xbdef52d1, 0xac71dc7a, 0x67f8a302, 0x73f2ddad, 0x7a216bd0,
					0xbd3016da, 0xdc80da93, 0x5b24f481, 0xb2f6e9e9, 0x58878f22, 0x808910a3,
					0xcb67c1e9, 0x6ff38125, 0xe244f915, 0xa5752720, 0xf1e4c7e3, 0x81c30d54,
					0x0a1e215a, 0xac58c2bf, 0xba5bf3cf, 0x5cad50d3, 0xaeb62831, 0xcda95ef5,
					0x787a1755, 0x484f87c0, 0xf284a112, 0xfba4cc92
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
					0x2, 0xb731d57c, 0x8e2f779e, 0x7d1fab2a, 0x364776a2, 0x2cc5c154, 0xa74a02e3,
					0x4743c390, 0xc08a2586, 0x5456059e, 0xd343063f, 0x091b1db7, 0x6d58e725,
					0xdb66c9ce, 0x6af4c80f, 0x274553cd, 0xde2fa475, 0xb0b1f796, 0x02c09afc,
					0xea7a64be, 0x2c95dc03, 0x989777c9, 0xea40893e, 0x9a9c1d65, 0xf096fa51,
					0xc4f33acf, 0x5350c4e7, 0x1815cc9b, 0x9821cb7b, 0x63122665, 0x09294d5f,
					0xb00c1de2, 0x87b5a827, 0xcb57546b, 0x49edabfb, 0x20e281e0, 0x00de2802,
					0xc2e1d451, 0xbdef52d1, 0xac71dc7a, 0x67f8a302, 0x73f2ddad, 0x7a216bd0,
					0xbd3016da, 0xdc80da93, 0x5b24f481, 0xb2f6e9e9, 0x58878f22, 0x808910a3,
					0xcb67c1e9, 0x6ff38125, 0xe244f915, 0xa5752720, 0xf1e4c7e3, 0x81c30d54,
					0x0a1e215a, 0xac58c2bf, 0xba5bf3cf, 0x5cad50d3, 0xaeb62831, 0xcda95ef5,
					0x787a1755, 0x484f87c0, 0xf284a112, 0xfba4cc91
				};
			}
		}

		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0x1, 0x48ce2a83, 0x71d08861, 0x82e054d5, 0xc9b8895d, 0xd33a3eab, 0x58b5fd1c,
					0xb8bc3c6f, 0x3f75da79, 0xaba9fa61, 0x2cbcf9c0, 0xf6e4e248, 0x92a718da,
					0x24993631, 0x950b37f0, 0xd8baac32, 0x21d05b8a, 0x4f4e0869, 0xfd3f6503,
					0x15859b41, 0xd36a23fc, 0x67688836, 0x15bf76c1, 0x6563e29a, 0x0f6905ae,
					0x3b0cc530, 0xacaf3b18, 0xe7ea3364, 0x67de3484, 0x9cedd99a, 0xf6d6b2a0,
					0x4ff3e21d, 0x784a57d8, 0x34a8ab94, 0xb6125404, 0xdf1d7e1f, 0xff21d7fd,
					0x3d1e2bae, 0x4210ad2e, 0x538e2385, 0x98075cfd, 0x8c0d2252, 0x85de942f,
					0x42cfe925, 0x237f256c, 0xa4db0b7e, 0x4d091616, 0xa77870dd, 0x7f76ef5c,
					0x34983e16, 0x900c7eda, 0x1dbb06ea, 0x5a8ad8df, 0x0e1b381c, 0x7e3cf2ab,
					0xf5e1dea5, 0x53a73d40, 0x45a40c30, 0xa352af2c, 0x5149d7ce, 0x3256a10a,
					0x8785e8aa, 0xb7b0783f, 0x0d7b5eed, 0x045b336d
				};
			}
		}

		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0x1, 0x6e63aaf9, 0x1c5eef3c, 0xfa3f5654, 0x6c8eed44, 0x598b82a9, 0x4e9405c6,
					0x8e878721, 0x81144b0c, 0xa8ac0b3d, 0xa6860c7e, 0x12363b6e, 0xdab1ce4b,
					0xb6cd939c, 0xd5e9901e, 0x4e8aa79b, 0xbc5f48eb, 0x6163ef2c, 0x058135f9,
					0xd4f4c97c, 0x592bb807, 0x312eef93, 0xd481127d, 0x35383acb, 0xe12df4a3,
					0x89e6759e, 0xa6a189ce, 0x302b9937, 0x304396f6, 0xc6244cca, 0x12529abf,
					0x60183bc5, 0x0f6b504f, 0x96aea8d6, 0x93db57f6, 0x41c503c0, 0x01bc5005,
					0x85c3a8a3, 0x7bdea5a3, 0x58e3b8f4, 0xcff14604, 0xe7e5bb5a, 0xf442d7a1,
					0x7a602db5, 0xb901b526, 0xb649e903, 0x65edd3d2, 0xb10f1e45, 0x01122147,
					0x96cf83d2, 0xdfe7024b, 0xc489f22b, 0x4aea4e41, 0xe3c98fc7, 0x03861aa8,
					0x143c42b5, 0x58b1857f, 0x74b7e79e, 0xb95aa1a7, 0x5d6c5063, 0x9b52bdea,
					0xf0f42eaa, 0x909f0f81, 0xe5094225, 0xf7499923, 0x48ce2a83, 0x71d08861,
					0x82e054d5, 0xc9b8895d, 0xd33a3eab, 0x58b5fd1c, 0xb8bc3c6f, 0x3f75da79,
					0xaba9fa61, 0x2cbcf9c0, 0xf6e4e248, 0x92a718da, 0x24993631, 0x950b37f0,
					0xd8baac32, 0x21d05b8a, 0x4f4e0869, 0xfd3f6503, 0x15859b41, 0xd36a23fc,
					0x67688836, 0x15bf76c1, 0x6563e29a, 0x0f6905ae, 0x3b0cc530, 0xacaf3b18,
					0xe7ea3364, 0x67de3484, 0x9cedd99a, 0xf6d6b2a0, 0x4ff3e21d, 0x784a57d8,
					0x34a8ab94, 0xb6125404, 0xdf1d7e1f, 0xff21d7fd, 0x3d1e2bae, 0x4210ad2e,
					0x538e2385, 0x98075cfd, 0x8c0d2252, 0x85de942f, 0x42cfe925, 0x237f256c,
					0xa4db0b7e, 0x4d091616, 0xa77870dd, 0x7f76ef5c, 0x34983e16, 0x900c7eda,
					0x1dbb06ea, 0x5a8ad8df, 0x0e1b381c, 0x7e3cf2ab, 0xf5e1dea5, 0x53a73d40,
					0x45a40c30, 0xa352af2c, 0x5149d7ce, 0x3256a10a, 0x8785e8aa, 0xb7b0783f,
					0x0d7b5eed, 0x045b336e
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x2
				};
			}
		}
		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0x919c5506, 0xe3a110c3, 0x05c0a9ab, 0x937112bb, 0xa6747d56, 0xb16bfa39,
					0x717878de, 0x7eebb4f3, 0x5753f4c2, 0x5979f381, 0xedc9c491, 0x254e31b4,
					0x49326c63, 0x2a166fe1, 0xb1755864, 0x43a0b714, 0x9e9c10d3, 0xfa7eca06,
					0x2b0b3683, 0xa6d447f8, 0xced1106c, 0x2b7eed82, 0xcac7c534, 0x1ed20b5c,
					0x76198a61, 0x595e7631, 0xcfd466c8, 0xcfbc6909, 0x39dbb335, 0xedad6540,
					0x9fe7c43a, 0xf094afb0, 0x69515729, 0x6c24a809, 0xbe3afc3f, 0xfe43affa,
					0x7a3c575c, 0x84215a5c, 0xa71c470b, 0x300eb9fb, 0x181a44a5, 0x0bbd285e,
					0x859fd24a, 0x46fe4ad9, 0x49b616fc, 0x9a122c2d, 0x4ef0e1ba, 0xfeeddeb8,
					0x69307c2d, 0x2018fdb4, 0x3b760dd4, 0xb515b1be, 0x1c367038, 0xfc79e557,
					0xebc3bd4a, 0xa74e7a80, 0x8b481861, 0x46a55e58, 0xa293af9c, 0x64ad4215,
					0x0f0bd155, 0x6f60f07e, 0x1af6bdda, 0x08b666db
				};
			}
		}

		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0x3, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xfffffffc, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000001
				};
			}
		}

		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0x831848eb, 0x7efecc65, 0x6b917c16, 0xc02ad024, 0x95d45734, 0x86436661,
					0xb659d06b, 0xed2d3d12, 0x8d22697f, 0x42aa156f, 0xa1072a98, 0x56d021ec,
					0x71659f68, 0x53afda42, 0xce0c8d36, 0x218e78c9, 0xdac9ff16, 0xd691c60c,
					0xf35dcca8, 0xedd6ce19, 0x61c6778e, 0x95b3dbd5, 0x7ac4b8e9, 0xf7219bea,
					0xb1ffc29b, 0x3f29f023, 0x9abc7824, 0x8854e481, 0x03ecd704, 0x6545fd15,
					0x1e789d7e, 0x64f0d0f2, 0xed7f5f3d, 0xd38a98d1, 0x9f143e4a, 0x74405b96,
					0xdd7299a0, 0xe6a03999, 0x962736af, 0x475c8990, 0x6b222718, 0xb2000344,
					0x198a8807, 0x59447387, 0x8198a9b0, 0x0e34d28b, 0x2acf7959, 0x4791c51a,
					0x545fb0b5, 0x2b95bebb, 0x52c4cfd6, 0x5fa6a819, 0xd87c20a5, 0xe9cd81b4,
					0xdc620add, 0x9f236459, 0x1c8058f6, 0xea57d9a4, 0xee9a5f32, 0xe2e6d0d9,
					0xd3bb1614, 0x5992c7d9, 0xe62d876e, 0x7d18ab9c, 0x31160464, 0x1d4b15ab,
					0xef2178d0, 0x5aebb512, 0x307345ca, 0x6be7e04b, 0xe554a3c3, 0xa271b8e8,
					0xb316ef66, 0x84e2cc4c, 0xd3a37fe4, 0x27867429, 0xe0297925, 0xf66c846c,
					0xc571306f, 0xa5983076, 0x9694454c, 0xb04f6438, 0x81288598, 0x40dfafca,
					0x4f645897, 0xed3a7565, 0x02c41610, 0x321d723d, 0xde46dea7, 0x6aa6de00,
					0x294e52eb, 0x49d22518, 0x4b119084, 0xbc07be46, 0xf520d1cc, 0xf8822758,
					0x240fd2ba, 0x1b2f9ad9, 0x39bbe880, 0xfe9df2c8, 0x2ee4c826, 0x1f92d91e,
					0xf5473b5a, 0x77351fd6, 0x73b2508a, 0xb2aef47d, 0x91945bfc, 0x7340f38d,
					0x0a295bda, 0xe59bb199, 0x83b1e7cb, 0x3bd75db1, 0x33c8e678, 0x4b9febfd,
					0x27b5953e, 0xe3b53c78, 0x623fc46d, 0x52407458, 0x7b99b946, 0x48ad38aa,
					0x25522881, 0xa14df5f5, 0x81bd8270, 0x68f2d9cd, 0x3930bedc, 0x0dc0cda4,
					0xa268fbc5, 0x0a890344
				};
			}
		}
	}

	public class CarryFF_569a489b: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0x3ffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0x143, 0xb6570190, 0xc15616df, 0x770d129f, 0x12ad442c, 0x61dd4434,
					0x26741e85, 0xe754aaff, 0x05f32773, 0x662eee13, 0x033e9626, 0xf996291a,
					0x2d47a501, 0xe84fbb77, 0xbb164332, 0x9b5cc677
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
					0x4000000, 0x00000000, 0x00000143, 0xb6570190, 0xc15616df, 0x770d129f,
					0x12ad442c, 0x61dd4434, 0x26741e85, 0xe754aaff, 0x05f32773, 0x662eee13,
					0x033e9626, 0xf996291a, 0x2d47a501, 0xe84fbb77, 0xbb164332, 0x9b5cc676
				};
			}
		}
		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0x3ffffff, 0xffffffff, 0xfffffebc, 0x49a8fe6f, 0x3ea9e920, 0x88f2ed60,
					0xed52bbd3, 0x9e22bbcb, 0xd98be17a, 0x18ab5500, 0xfa0cd88c, 0x99d111ec,
					0xfcc169d9, 0x0669d6e5, 0xd2b85afe, 0x17b04488, 0x44e9bccd, 0x64a33988
				};
			}
		}
		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0x5, 0x0ed95c06, 0x4305585b, 0x7ddc344a, 0x7c4ab510, 0xb1877510, 0xd099d07a,
					0x179d52ab, 0xfc17cc9d, 0xcd98bbb8, 0x4c0cfa58, 0x9be658a4, 0x68b51e94,
					0x07a13eed, 0xdeec590c, 0xca6d7319, 0xdbffffff, 0xffffffff, 0xfffffebc,
					0x49a8fe6f, 0x3ea9e920, 0x88f2ed60, 0xed52bbd3, 0x9e22bbcb, 0xd98be17a,
					0x18ab5500, 0xfa0cd88c, 0x99d111ec, 0xfcc169d9, 0x0669d6e5, 0xd2b85afe,
					0x17b04488, 0x44e9bccd, 0x64a33989
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x329ce, 0x3ac64866, 0x804dd21c
				};
			}
		}
		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0x135, 0x2915345d, 0x7d81de96, 0x2cd111cc, 0x708e7519, 0x8f9ab2e8,
					0x3290ee90, 0x5a4a3bdb, 0x9246ef64, 0xfba67306, 0x6978a7a4, 0xebb25821,
					0xaf52890c, 0x7cfeeccd, 0x4db77977, 0xb541acfb
				};
			}
		}
		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0xfffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xf8000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001
				};
			}
		}
		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0x19955, 0xa16dc713, 0xc289fc4c, 0x7330e48a, 0x5272262f, 0xff810531,
					0x326eeb48, 0xcc5c9bcd, 0xa9ade1f6, 0xde046e24, 0x15c751fb, 0xffa8341c,
					0x78e84ab6, 0xfd3eddc3, 0x1b8de164, 0xc3afd26c, 0x679660ca, 0x0042fe9c,
					0xfe4e8f34, 0x618b171e, 0x4bd287d5, 0xf262890a, 0x2602a8e4, 0x2f493eec,
					0x06a6bd02, 0x969ecadc, 0xd1a0f0ed, 0xeafcee09, 0xb4f421d9, 0x94dd4a6f,
					0x59644b51
				};
			}
		}
	}
	
	public class CarryFF_569a489b2: ArithmeticBig_Base {

		public override uint[] a {
			get {
				return new uint[] {
					0x3ffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff
				};
			}
		}

		public override uint[] b {
			get {
				return new uint[] {
					0x1f9, 0x505733ec, 0x18619439, 0xf8faa678, 0x08f98e39, 0x74a6aa2a,
					0xff120277, 0xdcbc33c5, 0xd99a52c2, 0x0469e5ed, 0x565db652, 0x265e7f3a,
					0x287dcd49, 0x27bca834, 0x4a3d43bd, 0xea286963
				};
			}
		}

		public override uint[] ExpectedAplusB {
			get {
				return new uint[] {
									  0x4000000, 0x00000000, 0x000001f9, 0x505733ec, 0x18619439, 0xf8faa678,
									  0x08f98e39, 0x74a6aa2a, 0xff120277, 0xdcbc33c5, 0xd99a52c2, 0x0469e5ed,
									  0x565db652, 0x265e7f3a, 0x287dcd49, 0x27bca834, 0x4a3d43bd, 0xea286962
								  };
			}
		}
		public override uint[] ExpectedAminusB {
			get {
				return new uint[] {
					0x3ffffff, 0xffffffff, 0xfffffe06, 0xafa8cc13, 0xe79e6bc6, 0x07055987,
					0xf70671c6, 0x8b5955d5, 0x00edfd88, 0x2343cc3a, 0x2665ad3d, 0xfb961a12,
					0xa9a249ad, 0xd9a180c5, 0xd78232b6, 0xd84357cb, 0xb5c2bc42, 0x15d7969c
				};
			}
		}

		public override uint[] ExpectedAtimesB {
			get {
				return new uint[] {
					0x7, 0xe5415ccf, 0xb0618650, 0xe7e3ea99, 0xe023e638, 0xe5d29aa8, 0xabfc4809,
					0xdf72f0cf, 0x1766694b, 0x0811a797, 0xb55976d9, 0x489979fc, 0xe8a1f735,
					0x249ef2a0, 0xd128f50e, 0xf7a8a1a5, 0x8bffffff, 0xffffffff, 0xfffffe06,
					0xafa8cc13, 0xe79e6bc6, 0x07055987, 0xf70671c6, 0x8b5955d5, 0x00edfd88,
					0x2343cc3a, 0x2665ad3d, 0xfb961a12, 0xa9a249ad, 0xd9a180c5, 0xd78232b6,
					0xd84357cb, 0xb5c2bc42, 0x15d7969d
				};
			}
		}

		public override uint[] ExpectedAdivB {
			get {
				return new uint[] {
					0x206c6, 0x4ebbf625, 0x27bb6f89
				};
			}
		}

		public override uint[] ExpectedAmodB {
			get {
				return new uint[] {
					0x182, 0x533228c6, 0x9dd3397e, 0x3b27eb8c, 0x3562366d, 0x6ba6b9d8,
					0x3484a126, 0x4e1d156a, 0x89ce0d65, 0xfb247cdc, 0x9386673f, 0x3e5a93ac,
					0x78008635, 0x46144618, 0x7803d5f4, 0x1a5cad04
				};
			}
		}

		public override uint[] ExpectedASquared {
			get {
				return new uint[] {
					0xfffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
					0xf8000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
					0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001
				};
			}
		}

		public override uint[] ExpectedBSquared {
			get {
				return new uint[] {
					0x3e56e, 0x114177a0, 0x15384723, 0x1f337d0f, 0x08acb440, 0x132f0b3d,
					0x91db9725, 0xc0778882, 0xd13ce6b5, 0x27f1a641, 0x8984133e, 0xa72a0330,
					0xed9590e6, 0xa57b346f, 0x68b348ed, 0x8b78ef0f, 0x80196826, 0xc48d6c46,
					0x8432300c, 0x91430fc6, 0x6b082e12, 0x42603ed9, 0x6f0d01ac, 0xb21d3f2a,
					0x1abfc5e1, 0x8b09b4c4, 0x5b967354, 0x221f263c, 0x43f2a9dc, 0x32315538,
					0x16525c49
				};
			}
		}
	}
}
