//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleDesigner/ScriptPanel.cpp
//! @brief     Implements class ScriptPanel
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/SampleDesigner/ScriptPanel.h"
#include "GUI/Model/Sample/SampleItem.h"
#include "GUI/Model/ToCore/SampleToCore.h"
#include "GUI/Support/Util/DesignerHelper.h"
#include "GUI/View/Info/CautionSign.h"
#include "GUI/View/Info/PythonSyntaxHighlighter.h"
#include "GUI/View/Tool/UpdateTimer.h"
#include "Sample/Multilayer/MultiLayer.h"
#include "Sim/Export/ExportToPython.h"

#include <QScrollBar>
#include <QStackedWidget>
#include <QTextEdit>
#include <QVBoxLayout>

namespace {

const int accumulateUpdatesDuringMsec = 20.;
}

ScriptPanel::ScriptPanel(QWidget* parent)
    : QWidget(parent)
    , m_textEdit(new QTextEdit)
    , m_highlighter(nullptr)
    , m_updateTimer(new UpdateTimer(accumulateUpdatesDuringMsec, this))
    , m_cautionSign(new CautionSign(m_textEdit))
    , m_currentSample(nullptr)
{
    setWindowTitle("Python Script");
    setObjectName("ScriptPanel");

    m_textEdit->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

    auto* mainLayout = new QVBoxLayout(this);
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addWidget(m_textEdit);

    m_textEdit->setReadOnly(true);
    QFont textFont("Monospace");
    m_textEdit->setFont(textFont);
    m_textEdit->setFontPointSize(DesignerHelper::getPythonEditorFontSize());

    connect(m_updateTimer, &UpdateTimer::timeToUpdate, this, &ScriptPanel::updateEditor,
            Qt::UniqueConnection);
}

void ScriptPanel::setCurrentSample(SampleItem* sampleItem)
{
    m_currentSample = sampleItem;
    updateEditor();
}

void ScriptPanel::onSampleModified()
{
    if (isVisible())
        m_updateTimer->scheduleUpdate();
}

void ScriptPanel::updateEditor()
{
    if (!m_highlighter) {
        m_highlighter = new PythonSyntaxHighlighter(m_textEdit->document());
        m_textEdit->setLineWrapMode(QTextEdit::NoWrap);
    }

    const int oldScrollbarValue = m_textEdit->verticalScrollBar()->value();

    const QString codeSnippet = generateCodeSnippet();
    if (!codeSnippet.isEmpty())
        m_textEdit->setText(codeSnippet);
    else
        m_textEdit->clear();

    m_textEdit->verticalScrollBar()->setValue(oldScrollbarValue);
}

void ScriptPanel::showEvent(QShowEvent*)
{
    m_updateTimer->scheduleUpdate();
}

void ScriptPanel::hideEvent(QHideEvent*)
{
    m_updateTimer->reset();
}

QString ScriptPanel::generateCodeSnippet()
{
    m_cautionSign->clear();

    if (m_currentSample == nullptr)
        return {};

    QString result;
    try {
        auto sample = GUI::ToCore::itemToSample(*m_currentSample);
        result.append(QString::fromStdString(Py::Export::sampleCode(*sample)));
    } catch (const std::exception& ex) {
        QString message =
            QString("Generation of Python Script failed. Code is not complete.\n\n"
                    "It can happen if sample requires further assembling or some of sample "
                    "parameters are not valid. See details below.\n\n%1")
                .arg(QString::fromStdString(ex.what()));

        m_cautionSign->setCautionMessage(message);
    }

    return result;
}
