/***************************************************************************
*   Copyright (C) 2007-2010 by Thomas Thelliez aka jblud                  *
*   Contact : <admin.kontrol@gmail.com>                                   *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as               *
*   published by the Free Software Foundation; either version 2.0 of      *
*   the License, or (at your option) any later version.                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
*   General Public License for more details.                              *
*                                                                         *
*   You should have received a copy of the GNU General Public             *
*   License along with this program; if not, write to                     *
*   the Free Software Foundation, Inc.,                                   *
*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          *
***************************************************************************/

#include "client_central_window.h"

ClientCentralWindow::ClientCentralWindow(QToolButton *menuButton)
    : menu(menuButton)
{	
    settings = new QSettings("OpenNetwork", "KontrolPack", this);
    user_mode = settings->value("user/mode", "NONE").toString();
    database = new Database;
    client_map = new QMap<int, RemoteComputer*>();
    remote_info_list = database->getRemoteClients();
    tabDialog = new TabDialog(remote_info_list);
    client_nbr = 0;
    QGridLayout *layout = new QGridLayout(this);

    tab = new TranslucentTabWidget(this);
#ifdef WIN32
    QFont font("Arial", 8, true);
    tab->setDocumentMode(true);
#elif defined __APPLE__
    QFont font = QFont();
    tab->setDocumentMode(false);
#else
    QFont font("Arial", 10, true);
    tab->setDocumentMode(true);
#endif
    textEdit = new QTextBrowser(this);
    textEdit->setFont(font);
    textEdit->setReadOnly(true);
    textEdit->setLineWrapMode(QTextEdit::FixedColumnWidth);
    textEdit->setLineWrapColumnOrWidth(1000);
    textEdit->textCursor().insertHtml(
            QString(tr("<font color=\"#000000\">")
                    + tr("KontrolPack improves interoperability between systems by connecting remote computers regardless of their operating system.<ul><li><img src=\":/images/security.png\"/>&nbsp;&nbsp;&nbsp;Secure <b>encrypted tunnel</b> over TCP/IP.</li><li><img src=\":/images/explorer.png\"/>&nbsp;&nbsp;&nbsp;Navigate through remote file systems.</li><li><img src=\":/images/terminal.png\"/>&nbsp;&nbsp;&nbsp;Handle <b>remote shell command executions</b>.</li><li><img src=\":/images/document.png\"/>&nbsp;&nbsp;&nbsp;<b>Share files between remote computers</b>.</li>")
                    + tr("</ul>")));
    clientHome = new ClientHome(textEdit, remote_info_list);
    clientHome->setAttribute(Qt::WA_TranslucentBackground, false);
    clientHome->setAttribute(Qt::WA_NoSystemBackground, false);

#ifdef WIN32
    this->setStyleSheet("ClientHome {background : white}");
#endif

    tab->setFont(font);
    tab->setTabsClosable(true);
    tab->setMovable(true);    
    tab->addTab(clientHome, QIcon(":/images/connection.png"), tr("Information"));

    QObject::connect(clientHome->addComputerButton, SIGNAL(clicked()), this, SLOT(show_connect_dialog()));
    QObject::connect(tabDialog, SIGNAL(add_tab_signal(QString,QString,QString,QString)), this, SLOT(add_tab(QString,QString,QString,QString)));
    QObject::connect(tab, SIGNAL(tabCloseRequested(int)), this, SLOT(remove_tab_by_index(int)));
    QObject::connect(clientHome, SIGNAL(connect_remote_clients(QString,QString,QString)), this, SLOT(show_connect_dialog(QString,QString,QString)));

    tab->setCornerWidget(menu, Qt::TopLeftCorner);
    layout->addWidget(tab, 0, 0);
    setLayout(layout);
    this->setMinimumSize(800,450);
}

void ClientCentralWindow::show_connect_dialog()
{
    tabDialog->ipAddress->setText("");
    tabDialog->password->setText("");
    tabDialog->exec();
}

void ClientCentralWindow::show_connect_dialog(QString ip, QString port, QString port_file)
{
    tabDialog->ipAddress->setText(ip.trimmed());
    tabDialog->port->setText(port.trimmed());
    tabDialog->password->setText("");
    tabDialog->portFile->setText(port_file.trimmed());
    tabDialog->exec();
}

void ClientCentralWindow::add_tab(QString ip, QString port, QString portFile, QString password)
{
    client_nbr++;
    RemoteComputer *remoteComputer = new RemoteComputer(ip, port, portFile, password, client_nbr);

#ifdef WIN32
    this->setStyleSheet("ClientHome {background : white} RemoteComputer {background : white}");
#endif
    tab->addTab(remoteComputer, QIcon(":/images/connection.png"), tr("Loading..."));
    tab->setCurrentWidget(remoteComputer);
    QObject::connect(remoteComputer, SIGNAL(remove_tab_signal(int)), this, SLOT(remove_tab_by_client_nbr(int)));
    QObject::connect(remoteComputer, SIGNAL(showMessage(QString,QString,int)), this, SLOT(showMessageSlot(QString,QString,int)));
    QObject::connect(remoteComputer, SIGNAL(update_os_information(QString,QString,int)), this, SLOT(update_os_information(QString,QString,int)));
    QObject::connect(remoteComputer, SIGNAL(add_command_to_database(Command*)), this, SLOT(add_command_to_database_slot(Command*)));
    client_map->insert(client_nbr, remoteComputer);
}

void ClientCentralWindow::remove_tab_by_index(int index)
{
    RemoteComputer *remoteComputer;
    if (remoteComputer = qobject_cast<RemoteComputer *> (tab->widget(index))) {
        remoteComputer->controller->master_tcp_client->stop_client();
        remoteComputer->controller->file_tcp_client->stop_client();
        tab->removeTab(index);
        client_nbr--;
        client_map->remove(remoteComputer->client_nbr);
        delete remoteComputer;
    }
}

void ClientCentralWindow::remove_tab_by_client_nbr(int client_nbr)
{
    if (client_map->contains(client_nbr)) {
        RemoteComputer *remoteComputer = client_map->take(client_nbr);
        remoteComputer->controller->master_tcp_client->stop_client();
        remoteComputer->controller->file_tcp_client->stop_client();
        tab->removeTab(tab->indexOf(remoteComputer));
        client_nbr--;
        client_map->remove(client_nbr);
    }
}

void ClientCentralWindow::showMessageSlot(QString title, QString content, int duration)
{
    emit showMessage(title, content, duration);
}

void  ClientCentralWindow::update_os_information(QString os, QString title, int client_nbr)
{
    if (client_map->contains(client_nbr)) {
        RemoteComputer *remoteComputer = client_map->value(client_nbr);
        RemoteComputerInfo *remoteInfo = new RemoteComputerInfo(this);
        remoteInfo->system = os;
        remoteInfo->hostname = title;
        remoteInfo->ip = remoteComputer->ip;
        remoteInfo->port = remoteComputer->port.toInt();
        remoteInfo->port_file = remoteComputer->portFile.toInt();
        QString info_string = QString(remoteInfo->hostname + DELIMITER_DATABASE + remoteInfo->ip + DELIMITER_DATABASE + QString::number(remoteInfo->port) + DELIMITER_DATABASE + QString::number(remoteInfo->port_file));
        // Save remote clients
        bool saved = false;
        if (!remote_info_list.isEmpty()) {
            for (int k = 0; k < remote_info_list.size(); k++) {
                RemoteComputerInfo *remoteInfoFromList = remote_info_list.at(k);
                QString string = QString(remoteInfoFromList->hostname + DELIMITER_DATABASE + remoteInfoFromList->ip + DELIMITER_DATABASE + QString::number(remoteInfoFromList->port) + DELIMITER_DATABASE + QString::number(remoteInfoFromList->port_file));
                if (QString::compare(string, info_string, Qt::CaseInsensitive) == 0) {
                    remoteInfo->id = remoteInfoFromList->id;
                    remoteComputer->id_server = remoteInfoFromList->id;
                    if (!user_mode.startsWith("END", Qt::CaseInsensitive)) {
                        remoteComputer->execShell->id_server = remoteInfoFromList->id;
                        remoteComputer->execShell->set_commands_list(database->getCommandsByIdServer(remoteInfoFromList->id));
                    }
                    saved = true;
                    break;
                }
            }
        }
        if (!saved) {
            remote_info_list.append(remoteInfo);
            int id = remote_info_list.size() + 1;
            remoteInfo->id = id;
            remoteComputer->id_server = id;
            if (!user_mode.startsWith("END", Qt::CaseInsensitive)) {
                remoteComputer->execShell->id_server = id;
                remoteComputer->execShell->set_commands_list(database->getCommandsByIdServer(id));
            }
            database->saveRemoteClients(remoteInfo);
            clientHome->refreshCLientsList(remote_info_list);
        }
        tab->setTabText(tab->indexOf(remoteComputer), title);
        if (os.startsWith(QString(LINUX), Qt::CaseSensitive)) {
            tab->setTabIcon(tab->indexOf(remoteComputer), QIcon(":/images/linux.png"));
        } else if (os.startsWith(WINDOWS, Qt::CaseSensitive)) {
            tab->setTabIcon(tab->indexOf(remoteComputer), QIcon(":/images/windows.png"));
        } else if (os.startsWith(MACOSX, Qt::CaseSensitive)) {
            tab->setTabIcon(tab->indexOf(remoteComputer), QIcon(":/images/finder.png"));
        } else {
            tab->setTabIcon(tab->indexOf(remoteComputer), QIcon(":/images/linux.png"));
        }
    }
}

void ClientCentralWindow::drop_database()
{
    database->dropRemoteClients();
    database->dropCommands();
    for (int k = 0; k < remote_info_list.size(); k++) {
        remote_info_list.removeAt(k);
    }
    database->dropRemoteClients();
    database->dropCommands();
    clientHome->refreshCLientsList(remote_info_list);
}

void ClientCentralWindow::add_command_to_database_slot(Command *command)
{
    QList<Command*> *list = database->getCommandsByIdServer(command->id_server);
    if (list->size() == 75)
        for (int i = 0; i < 30; i++) {
            Command *cmd = list->at(i);
            database->delete_command(cmd->id_server, cmd->id_command, cmd->command);
        }
    command->id_command = list->size() + 1;
    list = database->getCommandsByIdServer(command->id_server);
    database->saveCommand(command);
}
