package latexDraw.generators.svg;

import static java.lang.Math.min;

import java.awt.geom.RectangularShape;

import latexDraw.figures.LaTeXDrawRectangle;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.SVGElement;
import latexDraw.parsers.svg.elements.SVGGElement;
import latexDraw.parsers.svg.elements.SVGRectElement;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for a rectangle.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LRectangleSVGGenerator extends LPolygonSVGGenerator
{

	public LRectangleSVGGenerator(LaTeXDrawRectangle f)
	{
		super(f);
	}
	
	
	
	/**
	 * Creates a rectangle from an SVG rect element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LRectangleSVGGenerator(SVGRectElement elt)
	{
		this(new LaTeXDrawRectangle(true));
		
		setSVGRectParameters(elt);
		applyTransformations(elt);
	}
	
	
	
	
	public LRectangleSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates a rectangle from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LRectangleSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new LaTeXDrawRectangle(false));

		setNumber(elt);
		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGRectElement))
			throw new IllegalArgumentException();
		
		setSVGLatexdrawParameters(elt);
		setSVGRectParameters((SVGRectElement)elt2);
		setSVGShadowParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHADOW));
		setSVGDbleBordersParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_DBLE_BORDERS));
		
		if(withTransformation)
			applyTransformations(elt);
	}
	
	
	
	private void setSVGRectParameters(SVGRectElement elt)
	{
		if(elt==null)
			return ;
		
		setSVGParameters(elt);
		
		double rx = elt.getRx();
		LaTeXDrawRectangle r = (LaTeXDrawRectangle)getShape();
		double gap = getPositionGap()/2.;
		
		r.setFirstPoint(elt.getX()+gap, elt.getY()+gap);
		r.setLastPoint(elt.getX()+elt.getWidth()-gap, elt.getY()+elt.getHeight()-gap);
		
		if(rx>0)
			r.setIsRound(true);
		
		RectangularShape rs = (RectangularShape)r.getShape();
		r.setFrameArc((2*rx)/(min(rs.getHeight(), rs.getWidth())- (r.hasDoubleBoundary() ? r.getDoubleSep()+r.getThickness() : 0)));
	}

	
	
	@Override
	public SVGElement toSVG(SVGDocument document)
	{
		if(document==null || document.getFirstChild().getDefs()==null)
			throw new IllegalArgumentException();

		double gap = getPositionGap();
		LaTeXDrawPoint2D nw = shape.getTheNWPoint();
		LaTeXDrawPoint2D se = shape.getTheSEPoint();
		SVGElement elt;
        SVGElement root = new SVGGElement(document);
        root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_RECT);
        root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
        
        if(shape.hasShadow())
        {
        	elt = new SVGRectElement(nw.x-gap/2., nw.y-gap/2., Math.max(1, (se.x-nw.x)+gap), Math.max(1, (se.y-nw.y)+gap), document);
        	setSVGShadowAttributes(elt, true);
        	setSVGRoundCorner(elt);
        	root.appendChild(elt);
        }
        
        if(shape.hasShadow() && !shape.getLineStyle().equals(PSTricksConstants.LINE_NONE_STYLE))
        {// The background of the borders must be filled is there is a shadow.
	        elt = new SVGRectElement(nw.x-gap/2., nw.y-gap/2., Math.max(1, (se.x-nw.x)+gap), Math.max(1, (se.y-nw.y)+gap), document);
	        setSVGBorderBackground(elt, root);
	        setSVGRoundCorner(elt);
        }
        
        elt = new SVGRectElement(nw.x-gap/2., nw.y-gap/2., Math.max(1, (se.x-nw.x)+gap), Math.max(1, (se.y-nw.y)+gap), document);
        root.appendChild(elt);
        setSVGAttributes(document, elt, true);
        setSVGRoundCorner(elt);
        
        if(shape.hasDoubleBoundary())
        {
        	elt = new SVGRectElement(nw.x-gap/2., nw.y-gap/2., Math.max(1, (se.x-nw.x)+gap), Math.max(1, (se.y-nw.y)+gap), document);
        	setSVGDoubleBordersAttributes(elt);
        	setSVGRoundCorner(elt);
        	root.appendChild(elt);
        }

        setSVGRotationAttribute(root);
        
		return root;
	}
	
	
	
	/**
	 * Sets the roundness of the SVG shape.
	 * @param elt The SVG element into which the roundness must be set.
	 * @since 2.0.0
	 */
	protected void setSVGRoundCorner(SVGElement elt)
	{
		if(elt==null)
			return ;
		
		if(((LaTeXDrawRectangle)shape).isRound())
		{
	        RectangularShape r = (RectangularShape)shape.getShape();
			double add = shape.hasDoubleBoundary() ? shape.getDoubleSep() + shape.getThickness() : 0;
			double value = 0.5*(min(r.getWidth(), r.getHeight())-add)*((LaTeXDrawRectangle)shape).getFrameArc();
			
			elt.setAttribute(SVGAttributes.SVG_RX, String.valueOf(value));
		}
	}
}
