package latexDraw.ui.components;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowFocusListener;
import java.awt.image.BufferedImage;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import latexDraw.ui.dialog.ExceptionFrameDialog;


/** 
 * This class defines a button which displays a frame containing
 * others buttons; a kind of menu with buttons.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 05/02/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class ListJToggleButton extends JToggleButton implements ActionListener, ChangeListener, WindowFocusListener
{
	private static final long serialVersionUID = 1L;

	/** The frame which displays the toolbar */
	JWindow buttonsFrame;
	
	/** The toolbar which contains the buttons */
	JToolBar toolbar;
	
	/** The width of the triangle drawn in the button */
	public static final int WIDTH_TRIANGLE = 8;
	
	/** The height of the triangle drawn in the button */
	public static final int HEIGHT_TRIANGLE = 6;
	
	/** Define the location of the panel of buttons. */
	protected int location;
	
	/** The frame is at the north of the button. */
	public static final short LOCATION_NORTH = 1;
	
	/** The frame is at the south of the button. */
	public static final short LOCATION_SOUTH = 2;
	
	
	
	/**
	 * The constructor using a text.
	 * @param frame The main frame containing the ListJToggleButton.
	 * @param txt The text of the ListJToggleButton.
	 * @param locate The position of the mini toolbar.
	 * @since 1.9.1
	 */
	public ListJToggleButton(JFrame frame, String txt, int locate)
	{
		super(txt);
		
		setIcon();
		intializing(frame, locate);
	}
	
	
	
	/**
	 * The constructor using an icon.
	 * @param icon The icon of the ListJToggleButton.
	 * @param locate The position of the mini toolbar.
	 * @param frame The main frame containing the ListJToggleButton.
	 */
	public ListJToggleButton(JFrame frame, Icon icon, int locate)
	{
		super(icon);

		intializing(frame, locate);
	}

	
	
	/**
	 * Initialises the ListJToggleButton.
	 * @param frame The main frame containing the ListJToggleButton.
	 * @param locate The position of the mini toolbar.
	 * @since 1.9.1
	 */
	protected void intializing(JFrame frame, int locate)
	{
		try
		{
			try{setLocation(locate);}
			catch(Exception e) {location = LOCATION_SOUTH; }
		
			toolbar = new JToolBar();
			JPanel buttonsPanel = new JPanel();
			
			toolbar.setFloatable(false);
			
			buttonsFrame = new JWindow(frame);
			buttonsFrame.setVisible(false);
			buttonsFrame.setAlwaysOnTop(true);
			
			buttonsPanel.add(toolbar);
			buttonsPanel.setBorder(BorderFactory.createEtchedBorder());
			buttonsFrame.add(buttonsPanel);
			buttonsFrame.pack();
			buttonsFrame.addWindowFocusListener(this);
			
			addActionListener(this);
			setMargin(new Insets(1,1,1,1));
			JButton closeB = new CloseButton(null);
			addComponent(closeB);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	
	/**
	 * @return Creates an imageIcon containing the triangle icon.
	 * @since 1.9.1
	 */
	protected ImageIcon createTriangleIcon()
	{
		int width = 10, height = 20;
		BufferedImage bufferImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
		Graphics2D graphic = bufferImage.createGraphics();
		paintTriangle(graphic, width, height);
		
		return new ImageIcon(bufferImage);
	}
	
	
	
	/**
	 * Sets the location of the mini toolbar.
	 * @param locat The position of the mini toolbar.
	 * @throws IllegalArgumentException If the location is neither <code>LOCATION_NORTH</code> nor <code>LOCATION_SOUTH</code>.
	 */
	public void setLocation(int locat)
	{
		if(locat==LOCATION_NORTH || locat==LOCATION_SOUTH)
			location = locat;
		else 
			throw new IllegalArgumentException();
	}
	
	
	
	@Override
	public void setSelected(boolean sel)
	{
		updateSelected();
	}
	
	
	
	/**
	 * Adds a separator to the toolbar.
	 */
	public void addSeparator()
	{
		Component c = toolbar.getComponentAtIndex(toolbar.getComponentCount()-1);
		toolbar.remove(c);
		toolbar.addSeparator();
		toolbar.add(c);
		buttonsFrame.pack();
	}
	
	
	
	/**
	 * Adds a button to the toolbar of the listJToggleButton.
	 * @param b The new button to add.
	 */
	public void addComponent(Component b)
	{
		toolbar.add(b, null, toolbar.getComponentCount()-1);
		buttonsFrame.pack();
		
		if(b instanceof AbstractButton)
		{
			((AbstractButton)b).addActionListener(this);
			((AbstractButton)b).addChangeListener(this);
		}
	}
	
	
	
	
	/**
	 * Paints the triangle icon in a graphic.
	 * @param g The graphic.
	 * @param width The width of the object.
	 * @param height The height of the object.
	 * @since 1.9.1
	 */
	public void paintTriangle(Graphics g, int width, int height)
	{
		if(g==null || width<1 || height<1)
			throw new IllegalArgumentException();
		
		Graphics2D g2 = (Graphics2D) g;
    	
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		g2.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
		g2.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY);
		
		int nbPts = 3;
		int gap = 4;
		int xs[] = new int[nbPts];
		int ys[] = new int[nbPts];
		
		if(location==LOCATION_SOUTH)
		{
			xs[0] = (width-WIDTH_TRIANGLE)>>1;
			xs[1] = (width+WIDTH_TRIANGLE)>>1;
			xs[2] = width>>1;
			
			ys[0] = height-HEIGHT_TRIANGLE-gap;
			ys[1] = height-HEIGHT_TRIANGLE-gap;
			ys[2] = height-gap;
		}
		else
		{
			xs[0] = (width-WIDTH_TRIANGLE)>>1;
			xs[1] = (width+WIDTH_TRIANGLE)>>1;
			xs[2] = width>>1;
			
			ys[0] = HEIGHT_TRIANGLE+gap;
			ys[1] = HEIGHT_TRIANGLE+gap;
			ys[2] = gap;
		}
		
		g.setColor(Color.red);
		g.fillPolygon(xs, ys, nbPts);
		g.setColor(Color.black);
		g.drawPolygon(xs, ys, nbPts);
	}
	
	
	
	@Override
	protected void paintComponent(Graphics g)
	{
		super.paintComponent(g);
		
		if(getText().equals(""))//$NON-NLS-1$
			paintTriangle(g,getSize().width, getSize().height);
	}



	/**
	 * Defines the position of the mini-toolbar.
	 * @since 1.9.1
	 */
	public void defineToolbarLocation()
	{
		buttonsFrame.setLocation(getLocationOnScreen().x, getLocationOnScreen().y + 
								(location==LOCATION_SOUTH ? getHeight() : -buttonsFrame.getHeight()));
	}
	
	

	public void actionPerformed(ActionEvent e)
	{
		Object src = e.getSource();
	
		if(src instanceof ListJToggleButton)
		{
			defineToolbarLocation();
			updateSelected();
			buttonsFrame.setVisible(!buttonsFrame.isVisible());
			return ;
		}
		
		if(src instanceof JCheckBox || src instanceof JRadioButton)
			return ;
		
		if(src instanceof CloseButton)
		{
			buttonsFrame.setVisible(false);
			return;
		}
		
		if(src instanceof JToggleButton || src instanceof JButton)
		{
			buttonsFrame.setVisible(false);
			
			updateSelected();
			return ;
		}
	}





	public void stateChanged(ChangeEvent e)
	{
		Object src = e.getSource();
		
		if(src instanceof JToggleButton && !(src instanceof JCheckBox || src instanceof JRadioButton))
			setSelected(((JToggleButton)src).isSelected());
	}




	/**
	 * @return The location of the panel of buttons.
	 */
	public int getLocationButtonPanel()
	{
		return location;
	}
	
	
	
	
	/**
	 * Sets the main button to selected or not following if a button is selected or not.
	 */
	public void updateSelected()
	{
		int i, size = toolbar.getComponentCount();
		boolean selected = false;
		Component src;
		
		for(i=0; i<size && !selected; i++)
		{
			src = toolbar.getComponent(i);
			
			if(src instanceof JToggleButton && !(src instanceof JCheckBox || src instanceof JRadioButton) && 
				((JToggleButton)toolbar.getComponent(i)).isSelected())
				selected = true;
		}
		
		super.setSelected(selected);
	}



	public void windowGainedFocus(WindowEvent e)
	{
		toolbar.requestFocusInWindow();
	}



	public void windowLostFocus(WindowEvent e)
	{
		if(e.getOppositeWindow()!=null)
			buttonsFrame.setVisible(false);
	}



	/**
	 * The icon of the button will be the red triangle.
	 * @since 1.9.1
	 */
	public void setIcon()
	{
		super.setIcon(createTriangleIcon());
	}

	
	
	@Override
	public void setIcon(Icon defaultIcon)
	{
		if(!getText().equals(""))//$NON-NLS-1$
			setIcon();
		else 
			super.setIcon(defaultIcon);
	}
}
