/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "testlib/environments.h"

// Boost C++
#include <boost/filesystem.hpp>

// C++ Standard Library
#include <fstream>
#include <map>
#include <string>
#include <sstream>

// Media Scanner Library
#include "mediascanner/dbusservice.h"
#include "mediascanner/logging.h"
#include "mediascanner/glibutils.h"
#include "mediascanner/mediaroot.h"
#include "mediascanner/writablemediaindex.h"
#include "mediascanner/utilities.h"

// Media Scanner Plugin for Grilo
#include "grlmediascanner/mediasource.h"

namespace mediascanner {

static const logging::Domain kInfo("info", logging::info());

////////////////////////////////////////////////////////////////////////////////

DBusEnvironment::DBusEnvironment(const FileSystemPath &data_path,
                                 const FileSystemPath &index_path)
    : data_path_(data_path)
    , index_path_(index_path)
    , dbus_daemon_pid_(0)
    , mediascanner_pid_(0) {
}

void DBusEnvironment::SetUp() {
    const char *const dbus_daemon_argv[] = {
        "dbus-daemon", "--session", "--fork",
        "--print-address=1", "--print-pid=2", nullptr
    };

    Wrapper<char> standard_output;
    Wrapper<char> standard_error;
    Wrapper<GError> error;
    int exit_status = 0;

    const bool dbus_started =
            g_spawn_sync(data_path_.string().c_str(),
                         const_cast<char **>(dbus_daemon_argv),
                         nullptr, G_SPAWN_SEARCH_PATH, nullptr, this,
                         standard_output.out_param(),
                         standard_error.out_param(),
                         &exit_status, error.out_param());

    ASSERT_FALSE(error) << to_string(error);
    ASSERT_TRUE(WIFEXITED(exit_status))
            << "exit status: " << exit_status << std::endl
            << "standard output: " << standard_output.get() << std::endl
            << "standard error: " << standard_error.get();
    ASSERT_EQ(0, WEXITSTATUS(exit_status))
            << "exit status: " << exit_status << std::endl
            << "standard output: " << standard_output.get() << std::endl
            << "standard error: " << standard_error.get();
    ASSERT_TRUE(dbus_started);

    const size_t n = strcspn(standard_output.get(), "\r\n");
    standard_output.get()[n] = '\0';

    const std::string session_address = standard_output.get();
    ASSERT_FALSE(session_address.empty());

    std::istringstream(standard_error.get()) >> dbus_daemon_pid_;
    ASSERT_NE(0, dbus_daemon_pid_);

    g_setenv("DBUS_SESSION_BUS_ADDRESS", session_address.c_str(), true);

    kInfo("Running D-Bus daemon with pid={1} at=<{2}>")
            % dbus_daemon_pid_ % session_address;

    const std::string media_index_arg =
            "--media-index-path=" + index_path_.string();
    const char *const mediascanner_argv[] = {
        MEDIASCANNER_SERVICE, media_index_arg.c_str(),
        "--disable-scanning", "--disable-volume-monitor",
        MEDIA_DIR, nullptr
    };

    const bool scanner_started =
            g_spawn_async(data_path_.string().c_str(),
                          const_cast<char **>(mediascanner_argv), nullptr,
                          G_SPAWN_DO_NOT_REAP_CHILD, nullptr, this,
                          &mediascanner_pid_, error.out_param());

    ASSERT_FALSE(error) << to_string(error);
    ASSERT_TRUE(scanner_started);

    // Start media scanner service.
    kInfo("Running mediascanner with pid={1}") % mediascanner_pid_;

    const Wrapper<GDBusConnection> session_bus =
            take(g_dbus_connection_new_for_address_sync
                        (session_address.c_str(),
                         G_DBUS_CONNECTION_FLAGS_MESSAGE_BUS_CONNECTION |
                         G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_CLIENT,
                         nullptr, nullptr, error.out_param()));

    EXPECT_FALSE(error) << to_string(error) << std::endl << session_address;
    ASSERT_TRUE(session_bus);

    dbus::MediaScannerProxy service;
    const bool connected = service.ConnectAndWait(session_bus,
                                                  Wrapper<GCancellable>(),
                                                  error.out_param());
    EXPECT_FALSE(error) << to_string(error);
    ASSERT_TRUE(connected);

    // Wait up to one minute for the media scanner becoming available.
    Wrapper<char> media_scanner_address;

    for (int i = 0; i < 60; ++i) {
        if (media_scanner_address =
                take(g_dbus_proxy_get_name_owner(service.handle().get())))
            break;

        const std::string connection_name =
            dbus::ConnectionName(service.connection());
        kInfo("Waiting for mediascanner to become available at <{1}>...")
                % connection_name;

        Wrapper<GMainLoop> loop = take(g_main_loop_new(nullptr, true));
        Timeout::AddOnce(boost::posix_time::milliseconds(1000),
                         std::bind(&g_main_loop_quit, loop.get()));
        g_main_loop_run(loop.get());
    }

    ASSERT_TRUE(media_scanner_address);

    // Check if the media scanner we found operates on the same index.
    std::string index_path;

    const bool has_index_path_property =
            service.index_path.ReadValue(&service, &index_path);

    ASSERT_TRUE(has_index_path_property);
    ASSERT_EQ(index_path_, index_path);
}

static void terminate_child(pid_t child) {
    int status = 0;

    switch (waitpid(child, &status, WNOHANG)) {
    case 0:
        std::cout << "Terminating process " << child << std::endl;
        kill(child, SIGTERM);
        break;

    case -1:
        std::cout << "Failed to wait for process " << child
                  << ": " << g_strerror(errno) << std::endl;
        break;

    default:
        if (WIFEXITED(status)) {
            std::cout << "Process " << child << " has exited already with an "
                         "exit status of " << WEXITSTATUS(status) << std::endl;
        } else if (WIFSIGNALED(status)) {
            std::cout << "Process " << child << " was terminated already "
                         "with signal " << WTERMSIG(status) << std::endl;
        } else {
            std::cout << "Process " << child << " has unknown status."
                      << std::endl;
        }
        break;
    }
}

void DBusEnvironment::TearDown() {
    if (mediascanner_pid_) {
        terminate_child(mediascanner_pid_);
        g_spawn_close_pid(mediascanner_pid_);
        mediascanner_pid_ = 0;
    }

    if (dbus_daemon_pid_) {
        kill(dbus_daemon_pid_, SIGTERM);
        dbus_daemon_pid_ = 0;
    }
}

////////////////////////////////////////////////////////////////////////////////

LuceneEnvironment::LuceneEnvironment(const FileSystemPath &index_path)
    : index_path_(index_path) {
}

void LuceneEnvironment::SetUp() {
    boost::filesystem::remove_all(index_path_);
    boost::filesystem::create_directories(index_path_.parent_path());

    const MediaRootManagerPtr root_manager(new MediaRootManager);
    root_manager->AddManualRoot(MEDIA_DIR);

    WritableMediaIndex writer(root_manager);

    const bool writer_opened = writer.Open(index_path_);
    EXPECT_EQ(std::string(), writer.error_message());
    EXPECT_EQ(index_path_, writer.path());
    ASSERT_TRUE(writer_opened);

    FillMediaIndex(&writer);
}

void LuceneEnvironment::FillMediaIndex(WritableMediaIndex */*writer*/) {
}

////////////////////////////////////////////////////////////////////////////////

GriloPluginEnvironment::GriloPluginEnvironment(const std::string &plugin_id,
                                               const std::string &source_id,
                                               const ConfigMap &config)
    : plugin_id_(plugin_id)
    , source_id_(source_id)
    , config_params_(config)
    , destroy_count_(0)
    , object_(nullptr) {
}

void GriloPluginEnvironment::SetUp() {
    Wrapper<GrlRegistry> registry = wrap(grl_registry_get_default());
    Wrapper<GError> error;

    Wrapper<GrlConfig> config = take(grl_config_new(plugin_id_.c_str(),
                                                    source_id_.c_str()));

    for (const auto &p: config_params_) {
        grl_config_set_string(config.get(), p.first.c_str(), p.second.c_str());
    }

    const bool config_added = grl_registry_add_config
            (registry.get(), config.get(), error.out_param());

    EXPECT_FALSE(error);
    ASSERT_TRUE(config_added);

    const bool plugin_loaded = grl_registry_load_plugin_by_id
            (registry.get(), plugin_id_.c_str(), error.out_param());

    EXPECT_FALSE(error);
    ASSERT_TRUE(plugin_loaded);

    const Wrapper<GrlPlugin> plugin =
            wrap(grl_registry_lookup_plugin(registry.get(),
                                            plugin_id_.c_str()));

    ASSERT_TRUE(plugin);

    const FileSystemPath plugin_path = grl_plugin_get_filename(plugin.get());
    ASSERT_EQ(FileSystemPath(GRILO_PLUGIN_DIR), plugin_path.parent_path());

    const Wrapper<GrlSource> source =
            wrap(grl_registry_lookup_source(registry.get(),
                                            plugin_id_.c_str()));

    ASSERT_TRUE(source);

    object_ = source.get<GObject>();
    g_object_weak_ref(object_, &GriloPluginEnvironment::OnSourceDestroy, this);
}

void GriloPluginEnvironment::TearDown() {
    Wrapper<GrlRegistry> registry = wrap(grl_registry_get_default());
    Wrapper<GError> error;

    EXPECT_EQ(0, destroy_count_);

    const bool plugin_unloaded = grl_registry_unload_plugin
            (registry.get(), plugin_id_.c_str(), error.out_param());

    EXPECT_FALSE(error);
    ASSERT_TRUE(plugin_unloaded);

    EXPECT_EQ(1, destroy_count_)
            << "  Source: " << source_id_ << std::endl
            << "   #refs: " << (object_ ? object_->ref_count : 0);
}

void GriloPluginEnvironment::OnSourceDestroy(gpointer data, GObject *) {
    GriloPluginEnvironment *const env =
            static_cast<GriloPluginEnvironment *>(data);

    env->object_ = nullptr;
    ++env->destroy_count_;
}

////////////////////////////////////////////////////////////////////////////////

MediaScannerEnvironment::MediaScannerEnvironment
                                        (const FileSystemPath &index_path)
    : GriloPluginEnvironment(GRL_MEDIA_SCANNER_PLUGIN_ID,
                             GRL_MEDIA_SCANNER_PLUGIN_ID,
                             MakeConfig(index_path)) {
}

MediaScannerEnvironment::ConfigMap MediaScannerEnvironment::MakeConfig
                                        (const FileSystemPath &index_path) {
    ConfigMap config;

    config.insert(ConfigParam(GRL_MEDIA_SCANNER_CONFIG_INDEX_PATH,
                              index_path.string()));

    return config;
}

////////////////////////////////////////////////////////////////////////////////

TheMovieDbEnvironment::TheMovieDbEnvironment(const std::string &mock_file)
    : GriloPluginEnvironment("grl-tmdb", "grl-tmdb", MakeConfig())
    , mock_file_(mock_file) {
}

void TheMovieDbEnvironment::SetUp() {
    if (not mock_file_.empty())
        g_setenv("GRL_NET_MOCKED", mock_file_.c_str(), true);

    GriloPluginEnvironment::SetUp();
}

TheMovieDbEnvironment::ConfigMap TheMovieDbEnvironment::MakeConfig() {
    ConfigMap config;
    config.insert(ConfigParam(GRL_CONFIG_KEY_APIKEY, "fakekey"));
    return config;
}

////////////////////////////////////////////////////////////////////////////////

LastFmEnvironment::LastFmEnvironment(const std::string &mock_file)
    : GriloPluginEnvironment("grl-lastfm-albumart", "grl-lastfm-albumart",
                             ConfigMap())
    , mock_file_(mock_file) {
}

void LastFmEnvironment::SetUp() {
    if (not mock_file_.empty())
        g_setenv("GRL_NET_MOCKED", mock_file_.c_str(), true);

    GriloPluginEnvironment::SetUp();
}


////////////////////////////////////////////////////////////////////////////////

#if GUDEV_VERSION >= 175

void FakeUDevEnvironment::SetUp() {
    using boost::filesystem::current_path;
    using boost::filesystem::create_directories;
    using boost::filesystem::create_symlink;

    boost::system::error_code ec;

    struct stat media_dir_info;
    ASSERT_EQ(0, stat(MEDIA_DIR, &media_dir_info)) << g_strerror(errno);

    const std::string device_name = "dm-1";
    const std::string device_uuid = "25baa005-8898-4255-a173-b47a72720f7a";
    const dev_t device_number = media_dir_info.st_dev;

    std::ostringstream device_node;
    device_node << major(device_number) << ':' << minor(device_number);

    const FileSystemPath rundir = current_path() / "run";
    const FileSystemPath sysfsdir = rundir / "sysfs";
    const FileSystemPath udevdir = rundir / "udev";

    // Write mock version of the udev configuration file

    const FileSystemPath udev_conf_path = current_path() / "udev.conf";
    std::ofstream udev_conf_stream(udev_conf_path.string().c_str());
    udev_conf_stream << "udev_run = " << udevdir.string() << std::endl;
    ASSERT_TRUE(udev_conf_stream);

    // Create mock version of the sysfs filesystem

    boost::filesystem::remove_all(sysfsdir);

    const FileSystemPath device_node_dir =
            sysfsdir / "dev" / "block" / device_node.str();
    const FileSystemPath block_device_dir =
            sysfsdir / "block" / device_name;
    const std::string real_block_device_axis =
            "devices/virtual/block/" + device_name;
    const FileSystemPath real_block_device_dir =
            sysfsdir / real_block_device_axis;
    const std::string subsystem_axis = "class/block";
    const FileSystemPath subsystem_dir = sysfsdir / subsystem_axis;

    ASSERT_TRUE(create_directories(real_block_device_dir));
    ASSERT_TRUE(create_directories(device_node_dir.parent_path()));
    ASSERT_TRUE(create_directories(block_device_dir.parent_path()));
    ASSERT_TRUE(create_directories(subsystem_dir));

    // Must use relative path as udev-175's symlink resolving is broken.
    create_symlink("../" + real_block_device_axis,
                                block_device_dir, ec);
    create_symlink("../../" + real_block_device_axis,
                                device_node_dir, ec);
    create_symlink("../../../../" + subsystem_axis,
                                block_device_dir / "subsystem", ec);
    create_symlink("../../" + real_block_device_axis,
                                subsystem_dir / device_name, ec);

    const FileSystemPath uevent_filename = real_block_device_dir / "uevent";
    std::ofstream uevent_stream(uevent_filename.string().c_str());

    uevent_stream
            << "MAJOR=" << major(device_number) << std::endl
            << "MINOR=" << minor(device_number) << std::endl
            << "DEVNAME=" << device_name << std::endl
            << "DEVTYPE=disk" << std::endl;

    ASSERT_TRUE(uevent_stream);

    // Create mock version of the udev data directory

    boost::filesystem::remove_all(udevdir);

    const FileSystemPath udev_data_dir = udevdir / "data";
    ASSERT_TRUE(create_directories(udev_data_dir));

    std::ostringstream entry_name;
    const FileSystemPath entry_path = udev_data_dir / ("b" + device_node.str());
    std::ofstream entry_stream(entry_path.string().c_str());

    entry_stream
            << "N:" << device_name << std::endl
            << "S:disk/by-id/dm-name-ubuntu-root" << std::endl
            << "S:disk/by-uuid/" << device_uuid << std::endl
            << "S:mapper/ubuntu-root" << std::endl
            << "S:ubuntu/root" << std::endl
            << "E:ID_FS_TYPE=ext4" << std::endl
            << "E:ID_FS_USAGE=filesystem" << std::endl
            << "E:ID_FS_UUID=" << device_uuid << std::endl
            << "E:ID_FS_UUID_ENC=" << device_uuid << std::endl
            << "E:ID_FS_VERSION=1.0" << std::endl;

    ASSERT_TRUE(entry_stream);

    // Patch environment variables

    g_setenv("UDEV_CONFIG_FILE", udev_conf_path.string().c_str(), true);
    g_setenv("SYSFS_PATH", sysfsdir.string().c_str(), true);
}

#else // GUDEV_VERSION

#error Unsupported UDev version

#endif // GUDEV_VERSION

////////////////////////////////////////////////////////////////////////////////

} // namespace mediascanner
