///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ABSTRACT_FILE_COLUMN_WRITER_H
#define __ABSTRACT_FILE_COLUMN_WRITER_H

#include <core/Core.h>
#include "AtomsFileWriter.h"
#include "ChannelColumnMapping.h"

namespace AtomViz {

/**
 * \brief Abstract base class for atom file format writers that can
 *        handle a variable number of data columns with a customizable
 *        mapping from data channels to data columns.
 *
 * This abstract base class is meant as basis for specialized file format
 * writers that write atomistic data from a AtomsObject to a data file with tabular structure 
 * (like the simple XYZ format).
 * 
 * This base class stores the mapping between the DataChannel in the AtomsObject 
 * and the columns in the output file.
 * 
 * \sa DataChannel, AtomsObject, ChannelColumnMapping, DataRecordWriterHelper
 * \author Alexander Stukowski 
 */ 
class ATOMVIZ_DLLEXPORT AbstractFileColumnWriter : public AtomsFileWriter
{
protected:

	/// \brief Constructs a new instance of this class. 
	/// \param isLoading Specifies whether the object's data fields will be initialized from the 
	///                  data stored in a scene file after the instance has been created.
	AbstractFileColumnWriter(bool isLoading = false) : AtomsFileWriter(isLoading) {}

public:

	/// \brief Returns the mapping between data columns in the output file and
	///        the DataChannel objects in the source AtomsObject. 
	const ChannelColumnMapping& channelMapping() const { return _channelMapping; }
	
	/// \brief Sets the mapping between data columns in the output file and
	///        the DataChannel objects in the source AtomsObject.
	void setChannelMapping(const ChannelColumnMapping& mapping);

	// From AtomsFileWriter:

	/// \brief Prepares the writer based on the scene to be exported.
	virtual bool prepareScene(DataSet* dataset, bool suppressDialogs = false);

protected:

	/// \brief Saves the class' contents to the given stream.
	/// \sa RefTarget::saveToStream() 
	virtual void saveToStream(ObjectSaveStream& stream);
	
	/// \brief Loads the class' contents from the given stream.
	/// \sa RefTarget::loadFromStream() 
	virtual void loadFromStream(ObjectLoadStream& stream);		

	/// \brief Creates a copy of this object.
	/// \sa RefTarget::Clone() 
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);	
	
private:

	/// Stores the mapping between DataChannels in the AtomsObject and
	/// the columns in the output file. 
	ChannelColumnMapping _channelMapping;
	
private:
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(AbstractFileColumnWriter)	
};


};	// End of namespace AtomViz

#endif // __ABSTRACT_FILE_COLUMN_WRITER_H
