///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/objects/shapes/SimpleShapeObject.h>
#include <core/scene/objects/shapes/ShapeObject.h>
#include <core/scene/ObjectNode.h>
#include <core/viewport/Viewport.h>

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(SimpleShapeObject, SceneObject)

/******************************************************************************
* Makes the object render itself into the viewport.
* The viewport transformation is already set up, when this method is called by the
* system. The object has to be rendered in the local object coordinate system.
******************************************************************************/
void SimpleShapeObject::renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp)
{
	if(!shapeValidity.contains(time))
		buildShape(time, shape, shapeValidity);
	vp->renderNodeBezierShape(shape, contextNode);
}

/******************************************************************************
* This asks the object if it supports the conversion to another
******************************************************************************/
bool SimpleShapeObject::canConvertTo(PluginClassDescriptor* objectClass)
{
	// Can convert to a ShapeObject.
	if(PLUGINCLASSINFO(ShapeObject)->isKindOf(objectClass)) return true;

	return SceneObject::canConvertTo(objectClass);
}

/******************************************************************************
* Lets the object convert itself to another SceneObject derived type.
******************************************************************************/
SceneObject::SmartPtr SimpleShapeObject::convertTo(PluginClassDescriptor* objectClass, TimeTicks time)
{
	if(PLUGINCLASSINFO(ShapeObject)->isKindOf(objectClass)) {
		if(!shapeValidity.contains(time))
			buildShape(time, shape, shapeValidity);
        ShapeObject* shapeObj = new ShapeObject();
		shapeObj->setShape(shape);
		shapeObj->setObjectValidity(shapeValidity);
		return shapeObj;
	}
	return SceneObject::convertTo(objectClass, time);
}

};
