/*	URL_Dialog

PIRL CVS ID: URL_Dialog.java,v 1.5 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import java.net.URL;
import java.net.MalformedURLException;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;

/**	<I>URL_Dialog</I> provides a dialog used to interactively obtain the
	information needed to construct a URL.
<p>
	URL_Dialog is a reusable dialog. Once constructed it only needs to
	be setVisible to be activated. If it had been used previously the
	previous field values that were accepted will remain. The current
	URL definition is renewed when the Accept button is pressed and the
	information is valid. To be valid, the Protocol and Hostname fields
	must not be empty and the Port number field must either be empty or
	an integer number. The default protocol is http, but may be
	overridden.
<p>
	@author	Bradford Castalia, UA/PIRL
	@version 1.5
*/
public class URL_Dialog
	extends JDialog
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Viewers.URL_Dialog (1.5 2012/04/16 06:22:59)";

/**	Default protocol.
*/
public static final String
	DEFAULT_PROTOCOL		= "http";
private static String
	Default_Protocol		= DEFAULT_PROTOCOL;

//	Information fields.
private	JTextField
	Protocol_Field			= new JTextField (5),
	Hostname_Field			= new JTextField (30),
	Port_Number_Field		= new JTextField (5),
	Pathname_Field			= new JTextField (40);

//	The Current URL constructed from the last Accept action.
private URL
	Current_URL				= null;

//	Icon displayed on the dialog box.
private Icon
	Decorative_Icon			= null;


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTORS	= 1 << 0,
	DEBUG_GET			= 1 << 1,
	DEBUG_UI			= 1 << 3,
	DEBUG_ALL			= -1,

	DEBUG       		= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a URL_Dialog.
<p>
	@param	title	The title String for the dialog window.
	@param	icon	The Icon to be displayed on the dialog. If null no
		Icon is displayed.
	@param	owner	The Frame with which the dialog is associated,
		which may be null.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public URL_Dialog
	(
	String			title,
	Icon			icon,
	Frame			owner,
	boolean			modal
	)
{
super (owner, (title == null ? "URL" : title), modal);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println (">>> URL_Dialog");

Decorative_Icon = icon;

//	Build the panels.
getContentPane ().add (Panels (), BorderLayout.CENTER);
pack ();
getRootPane ().setMinimumSize (getContentPane ().getSize ());
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println ("<<< URL_Dialog");
}

/**	Constructs a URL_Dialog.
<p>
	@param	title	The title String for the dialog window.
	@param	icon	The Icon to be displayed on the dialog. If null no
		Icon is displayed.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public URL_Dialog
	(
	String			title,
	Icon			icon,
	boolean			modal
	)
{this (title, icon, null, modal);}

/**	Constructs a URL_Dialog.
<p>
	@param	title	The title String for the dialog window.
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public URL_Dialog
	(
	String			title,
	boolean			modal
	)
{this (title, null, null, modal);}

/**	Constructs a URL_Dialog.
<p>
	@param	modal	If true, the dialog will be modal; if false, the
		dialog will not be modal.
*/
public URL_Dialog
	(
	boolean			modal
	)
{this (null, null, null, modal);}

/**	Constructs a default URL_Dialog.
*/
public URL_Dialog ()
{this (null, null, null, false);}

/*------------------------------------------------------------------------------
	PUBLIC INTERFACE
*/
/**	Get a URL.
<p>
	A modal URL_Dialog is constructed from which to obtain a URL.
<p>
	@param	title	The title String for the dialog window.
	@param	icon	The Icon to be displayed on the dialog. If null no
		Icon is displayed.
	@param	parent	The parent window that dialog is associated with,
		which may be null.
	@return	A URL constructed from the information obtained. This will
		be null if the user Canceled the dialog.
*/
public static URL Get_URL
	(
	String			title,
	Icon			icon,
	JFrame			parent
	)
{
if ((DEBUG & DEBUG_GET) != 0)
	System.out.println
		(">>> URL_Dialog.Get_URL");
URL_Dialog
	URL_dialog = new URL_Dialog
		(title, icon, parent, true);
URL_dialog.setVisible (true);
if ((DEBUG & DEBUG_GET) != 0)
	System.out.println
		("<<< URL_Dialog.Get_URL");
return URL_dialog.Current_URL;
}

/**	Get a URL.
<p>
	A default URL_Dialog is used to obtain a URL.
<p>
	@return	A URL constructed from the information obtained. This will
		be null if the user Canceled the dialog.
	@see	#Get_URL(String, Icon, JFrame)
*/
public static URL Get_URL ()
{return Get_URL (null, null, null);}

/**	Get the current URL.
<p>
	The current URL is produced after each successful accept action.
<p>
	@return	The current URL. This wil be null if no URL has yet been
		constructed.
*/
public URL Current_URL ()
{return Current_URL;}

/**	Get the default protocol.
<p>
	@return	The default protocol String.
*/
public static String Default_Protocol ()
{return Default_Protocol;}

/**	Set the default protocol.
<p>
	Trailing "://" characters are stripped off.
<p>
	@param	protocol	The default protocol String. A null or empty
		value will set the {@link #DEFAULT_PROTOCOL}.
*/
public static void Default_Protocol
	(
	String	protocol
	)
{Default_Protocol = Check_Protocol (protocol);}

/**	Set the Protocol field entry.
<p>
	Trailing "://" characters are stripped off.
<p>
	@param	protocol	The default protocol String. A null or empty
		value will set the {@link #DEFAULT_PROTOCOL}.
	@return	This URL_Dialog.
*/
public URL_Dialog Protocol
	(
	String	protocol
	)
{
Protocol_Field.setText (Check_Protocol (protocol));
return this;
}

/**	Set the Hostname field entry.
<p>
	@param	hostname	The hostname String.
	@return	This URL_Dialog.
*/
public URL_Dialog Hostname
	(
	String	hostname
	)
{
Hostname_Field.setText (hostname);
return this;
}

/**	Set the Port number field entry.
<p>
	@param	port	The port number. A negative value clears the field.
	@return	This URL_Dialog.
*/
public URL_Dialog Port_Number
	(
	int		port
	)
{
if (port < 0)
	Port_Number_Field.setText (null);
else
	Port_Number_Field.setText (Integer.toString (port));
return this;
}

/**	Set the Pathname field entry.
<p>
	@param	pathname	The pathname String.
	@return	This URL_Dialog.
*/
public URL_Dialog Pathname
	(
	String	pathname
	)
{
Pathname_Field.setText (pathname);
return this;
}

/*==============================================================================
	Panels
*/
private JPanel Panels ()
{
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println (">>> URL_Dialog.Panels");
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

//	Icon or empty space.
location.gridx		= GridBagConstraints.RELATIVE;
location.gridy		= GridBagConstraints.RELATIVE;
location.gridwidth	= 1;
location.gridheight	= 3;
location.weightx	= 0.0;
location.weighty	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.anchor		= GridBagConstraints.NORTH;
location.insets		= new Insets (10, 10, 10, 5);
if (Decorative_Icon == null)
	//	Place holder.
	panel.add (Box.createHorizontalGlue (), location);
else
	//	Decorative icon.
	panel.add (new JLabel (Decorative_Icon), location);

//	Protocol.
Protocol_Field.setText (Default_Protocol);
location.gridheight	= 1;
location.gridx		= 1;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (10, 0, 5, 5);
panel.add (new JLabel("Protocol:"), location);

Hostname_Field.setToolTipText ("Protocol for the server");
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 0.0;
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (10, 0, 5, 10);
panel.add (Protocol_Field, location);

//	Hostname.
location.gridheight	= 1;
location.gridx		= 1;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 5, 5);
panel.add (new JLabel("Hostname:"), location);

Hostname_Field.setToolTipText ("Host name of the server");
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 5, 10);
panel.add (Hostname_Field, location);

//	Port number.
location.gridx		= 1;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 5, 5);
panel.add (new JLabel("Port number:"), location);

Port_Number_Field.setToolTipText ("Port number for the server");
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 0.0;
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 5, 10);
panel.add (Port_Number_Field, location);

//	Pathname.
location.gridx		= 1;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 10, 5);
panel.add (new JLabel("Pathname:"), location);

Pathname_Field.setToolTipText ("Path to the file to be accessed");
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 0, 10, 10);
panel.add (Pathname_Field, location);

//	Buttons panel:

JPanel
	button_panel = new JPanel (new GridBagLayout ());
JButton
	button;
location.gridx		= GridBagConstraints.RELATIVE;
location.gridwidth	= 1;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.insets		= new Insets (0, 0, 0, 0);

//	Accept.
button				= new JButton ("Accept");
button.setMnemonic ('A');
button.setToolTipText ("Use the current URL");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Accept (true);}});
button.setDefaultCapable (true);
getRootPane ().setDefaultButton (button);
location.anchor		= GridBagConstraints.WEST;
button_panel.add (button, location);

location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
button_panel.add (Box.createHorizontalGlue (), location);

//	Clear.
button				= new JButton ("Clear");
button.setMnemonic ('C');
button.setToolTipText ("Clear all fields");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Clear ();}});
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.anchor		= GridBagConstraints.CENTER;
button_panel.add (button, location);

location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
button_panel.add (Box.createHorizontalGlue (), location);

//	Cancel.
button		= new JButton ("Cancel");
button.setMnemonic ('n');
button.setToolTipText ("Cancel the operation");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Accept (false);}});
location.gridwidth	= GridBagConstraints.REMAINDER;
location.weightx	= 0.0;
location.fill		= GridBagConstraints.NONE;
location.anchor		= GridBagConstraints.EAST;
button_panel.add (button, location);

location.weightx	= 1.0;
location.fill		= GridBagConstraints.HORIZONTAL;
location.insets		= new Insets (0, 10, 10, 10);
panel.add (button_panel, location);

if ((DEBUG & DEBUG_UI) != 0)
	System.out.println ("<<< URL_Dialog.Panels");
return panel;
}

/*==============================================================================
	Actions
*/
private void Accept
	(
	boolean		accept
	)
{
if (accept)
	{
	String
		protocol;
	if ((protocol = Protocol_Field.getText ()).length () == 0)
		{
		Dialog_Box.Notice
			("The server protocol must be specified.", this);
		Protocol_Field.setText (Default_Protocol);
		return;
		}

	String
		host;
	if ((host = Hostname_Field.getText ()).length () == 0)
		{
		Dialog_Box.Notice
			("The name of the server host must be specified.", this);
		return;
		}

	int
		port = -1;
	String
		value;
	if ((value = Port_Number_Field.getText ()).length () != 0)
		{
		try {port = Integer.parseInt (value);}
		catch (NumberFormatException exception)
			{
			Dialog_Box.Error
				("The Port value must be an integer number.", this);
			return;
			}
		}

	String
		pathname = Pathname_Field.getText ();
	if (pathname.length () != 0 &&
		pathname.charAt (0) != '/')
		pathname = "/" + pathname;

	URL
		url = null;
	try {url = new URL (protocol, host, port, pathname);}
	catch (MalformedURLException exception)
		{
		//	This shouldn't happen as long as the protocol is acceptable.
		Dialog_Box.Error
			("Unable to construct a URL from the information provided.\n"
			+ exception.getMessage (),
			this);
		return;
		}
	Current_URL = url;
	}
else
	{
	if (Current_URL != null)
		{
		//	Restore the previous values.
		Hostname_Field.setText (Current_URL.getHost ());
		int
			port = Current_URL.getPort ();
		if (port < 0)
			Port_Number_Field.setText (null);
		else
			Port_Number_Field.setText (Integer.toString (port));
		Pathname_Field.setText (Current_URL.getFile ());
		}
	else
		Clear ();
	Current_URL = null;
	}
setVisible (false);
}

/**	Clears the fields.
<p>
	The Protocol field will be reset to the {@link #Default_Protocol()
	default} value.
*/
public void Clear ()
{
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println
		(">-< URL_Dialog.Clear");
Protocol_Field.setText (Default_Protocol);
Hostname_Field.setText (null);
Port_Number_Field.setText (null);
Pathname_Field.setText (null);
}

/*==============================================================================
	Helpers
*/
private static String Check_Protocol
	(
	String	protocol
	)
{
if (protocol != null)
	{
	//	Remove trailing "://" if present.
	int
		index = protocol.length () - 1;
	while (index >= 0 &&
			protocol.charAt (index) == '/')
		 --index;
	while (index >= 0 &&
			protocol.charAt (index) == ':')
		--index;
	if (++index < protocol.length ())
		protocol = protocol.substring (0, index);
	if (protocol.length () != 0)
		return protocol;
	}
return DEFAULT_PROTOCOL;
}

/*==============================================================================
	Application test stub
*/
public static void main (String[] arguments) 
{
if (DEBUG != DEBUG_OFF)
	System.out.println ("*** " + ID);
try
	{
	URL
		url = null;
	url = Get_URL ();
	System.out.println ("URL - " + url);
	}
catch (Exception exception)
	{
	System.out.println (exception.getMessage ());
	System.exit (1);
	}
System.exit (0);
}

}	//	End of URL_Dialog class.

