/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2009 Mathias Panzenböck <grosser.meister.morti@gmx.net>
* Copyright (C) 2009-2010 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/ 
#ifndef SMOOTHTASKS_TASKBARLAYOUT_H
#define SMOOTHTASKS_TASKBARLAYOUT_H

#include <QGraphicsLayout>
#include <QList>
#include <QPointer>
#include <QObject>
#include <QTime>

#include "SmoothTasks/ExpansionDirection.h"
#include "SmoothTasks/TaskItem.h"

// C++
#include <limits>

namespace SmoothTasks {

class TaskItem;
class TaskbarLayout;
class TaskbarItem;
class RowInfo;

class TaskbarLayout : public QObject, public QGraphicsLayout {
	Q_OBJECT
	Q_INTERFACES(QGraphicsLayout)
	Q_PROPERTY(Qt::Orientation orientation READ orientation WRITE setOrientation)
	Q_PROPERTY(qreal spacing READ spacing WRITE setSpacing)
	Q_PROPERTY(int fps READ fps WRITE setFps)
	Q_PROPERTY(bool animationsEnabled READ animationsEnabled WRITE setAnimationsEnabled)
	Q_PROPERTY(QGraphicsLayoutItem* draggedItem READ draggedItem)
	Q_PROPERTY(int minimumRows READ minimumRows WRITE setMinimumRows)
	Q_PROPERTY(int maximumRows READ maximumRows WRITE setMaximumRows)
	Q_PROPERTY(int rows READ rows)
	Q_PROPERTY(qreal cellHeight READ cellHeight)
	Q_PROPERTY(qreal expandedWidth READ expandedWidth WRITE setExpandedWidth)
	Q_PROPERTY(qreal expandDuration READ expandDuration WRITE setExpandDuration)
	
	friend class TaskbarItem;
	
	public:
		enum TaskbarLayoutType {
			ByShape        = 0,
			MaxSqueeze     = 1,
			FixedItemCount = 2,
			FixedSize      = 3,
			LimitSqueeze   = 4
		};

		TaskbarLayout(
			Qt::Orientation orientation = Qt::Horizontal,
			QGraphicsLayoutItem *parent = NULL);
		~TaskbarLayout();

		void takeFrom(TaskbarLayout *other);
		int  dragItem(TaskItem *item, QDrag *drag, const QPointF& pos);
		void moveDraggedItem(const QPointF& pos);
		void dragLeave();

		QSizeF sizeHint(Qt::SizeHint which, const QSizeF& constraint = QSizeF()) const;
		void   setGeometry(const QRectF& rect);

		void      clear(bool forceDeleteItems = false);
		void      startAnimation();
		void      stopAnimation();
		void      skipAnimation();
		int       count() const { return m_items.size(); }
		TaskItem *itemAt(int index) const;
		TaskItem *itemAt(const QPointF& pos) const;
		int       addItem(SmoothTasks::TaskItem* item);
		void      insertItem(int index, SmoothTasks::TaskItem* item);
		void      move(int fromIndex, int toIndex);
		void      removeAt(int index);
		void      removeItem(TaskItem *item);
		int       indexOf(TaskItem *item) const;
		int       rowOf(TaskItem *item) const;
		int       rowOf(int index) const;
		int       currentDragIndex() const { return m_currentIndex; }
		bool      isDragging() const { return m_draggedItem != NULL; }

		virtual int optimumCapacity() const = 0;

		virtual TaskbarLayoutType type() const = 0;

		TaskItem *draggedItem() const;

		Qt::Orientation orientation() const { return m_orientation; }
		void            setOrientation(Qt::Orientation orientation);

		qreal spacing() const { return m_spacing; }
		void  setSpacing(qreal spacing);

		int  fps() const { return m_fps; }
		void setFps(int fps);

		qreal animationSpeed() const { return m_animationSpeed; }
		void setAnimationSpeed(qreal animationSpeed);

		bool animationsEnabled() const { return m_animationsEnabled; }
		void setAnimationsEnabled(bool animationsEnabled);

		int  maximumRows() const { return m_maximumRows; }
		void setMaximumRows(int maximumRows);

		int  minimumRows() const { return m_minimumRows; }
		void setMinimumRows(int minimumRows);

		void setRowBounds(int minimumRows, int maximumRows);

		int rows() const { return m_rows; }

		qreal  cellHeight() const { return m_cellHeight; }
		qreal  comulativePreferredItemRdWidthStatic(const qreal maxRdHeight = std::numeric_limits<qreal>::infinity(), const bool decomposeGroups = false, int *const count = 0, qreal* maxItemRdHeight = 0) const;
		qreal  averagePreferredItemRdWidthStatic(const qreal maxRdHeight = std::numeric_limits<qreal>::infinity(), const bool decomposeGroups = false, int *const count = 0, qreal* maxItemRdHeight = 0) const;

		qreal expandedWidth() const { return m_expandedWidth; }
		void  setExpandedWidth(qreal expandedWidth);

		int  expandDuration() const { return m_expandDuration; }
		void setExpandDuration(int expandDuration);

		void rdToWorld(const QRectF&  src, QRectF&  dst, const QRectF& effectiveRect, bool rtl, bool isVertical) const;
		void rdToWorld(const QPointF& src, QPointF& dst, const QRectF& effectiveRect, bool rtl, bool isVertical) const;
		void worldToRd(const QRectF&  src, QRectF&  dst, const QRectF& effectiveRect, bool rtl, bool isVertical) const;
		void worldToRd(const QPointF& src, QPointF& dst, const QRectF& effectiveRect, bool rtl, bool isVertical) const;
		void swapRdAndWorldSize(const QSizeF& src, QSizeF& dst, bool isVertical) const;
	
	signals:
		void sizeHintChanged(Qt::SizeHint which);

	public slots:
		void preferredItemSizeStaticChange(TaskItem* item);
		void preferredItemSizeDynamicChange(TaskItem* item);

	protected:

		const QList<TaskbarItem*>& items() const { return m_items; }
		QRectF       effectiveGeometry()   const;
		TaskbarItem *draggedTaskbarItem()  const { return m_draggedItem; }

		virtual int  rowOf(const QPointF& pos) const;

		void buildRows(const int itemsPerRow, QList<RowInfo>& rowInfos, int& rows) const;
		void updateItemsRowCache();

		/**
		 * Distribute the items over the rows.
		 *
		 * This pure virtual function must be reimplemented in a subclass
		 * and has to set m_cellHeight, m_rows, m_rowInfos and the row
		 * of the items (e.g. by using updateItemsRowCache).
		 * It can use TaskItem::preferredRdSizeStatic for layouting but
		 * it must not use TaskItem::preferredRdWidthDynamic.
		 * It must not change sizeHint because this could result
		 * in endless cycles with changing geometry.
		 */
		virtual void updateLayoutStatic() = 0;
		
		/**
		 * Layout the rows.
		 *
		 * This method uses m_cellHeight, m_rowInfos and
		 * TaskItem::preferredRdWidthDynamic to set geometry of the items.
		 * It starts the animation.
		 */
		virtual void updateLayoutDynamic();

	private slots:
		void animate();

	private:
		static const qreal PIXELS_PER_SECOND;

		int indexOf(const QPointF& pos, int *row = NULL, bool *overLeftPart = NULL) const;
		void connectItem(TaskItem *item);
		void disconnectItem(TaskItem *item);

		TaskbarItem         *m_draggedItem;
		int                  m_currentIndex;
		bool                 m_mouseIn;
		QList<TaskbarItem*>  m_items;
		Qt::Orientation      m_orientation;
		qreal                m_spacing;
		QTimer              *m_animationTimer;
		QPointF              m_grabPos;
		int                  m_fps;
		qreal                m_animationSpeed;
		bool                 m_animationsEnabled;
		int                  m_minimumRows;
		int                  m_maximumRows; // use INT_MAX for "no" maximum
		qreal                m_expandedWidth;
		int                  m_expandDuration;
		QTime                m_animationFrameTimer;
		bool                 m_preferredSizeChanged;

	protected:
		qreal                m_cellHeight;
		int                  m_rows;
		QList<RowInfo>       m_rowInfos;
};

class TaskbarItem {
	
	public:
		
		TaskbarItem(TaskItem *item)
			: item(item),
			  row(0) {}
		~TaskbarItem();

		TaskItem          *item;
		int                row;
};

class RowInfo {

	public:
		RowInfo(int startIndex, int endIndex)
			: startIndex(startIndex),
			  endIndex(endIndex) {}

		int   startIndex;
		int   endIndex;
};

} // namespace SmoothTasks
#endif
