// Copyright (c) 2006, Google Inc.
// All rights reserved.
//
// Author: Li Liu
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#include <fcntl.h>
#include <pthread.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#include <unistd.h>
#include <sys/utsname.h>
#include <sys/wait.h>

#include <cstdio>
#include <cstdlib>
#include <ctime>

#include "common/linux/file_id.h"
#include "client/linux/handler/gettid.h"
#include "client/linux/handler/linux_thread.h"
#include "client/minidump_file_writer.h"
#include "client/minidump_file_writer-inl.h"
#include "google_breakpad/common/minidump_format.h"
#include "client/linux/handler/minidump_generator.h"
#include "client/linux/handler/cpu_context.h"
#include "client/linux/handler/sigcontext.h"

#ifndef CLONE_UNTRACED
#define CLONE_UNTRACED 0x00800000
#endif

#ifdef __x86_64__
// For x86_64, there is a good chance the libs used are compiled without
// frame pointer, and breakpad will failed to get the stack information
// for those threads running in libs without frame pointer perserved. So
// we just ignore those threads, and only reports the crashing thread. The
// crashing thread mostly crashes in our code which should be compiled with
// frame pointer, so we can get the stack information.
#define ONLY_CRAHSING_THREAD
#endif

// This unnamed namespace contains helper functions.
namespace {

using namespace google_breakpad;

// Argument for the writer function.
struct WriterArgument {
  MinidumpFileWriter *minidump_writer;

  // Context for the callback.
  void *version_context;

  // Pid of the thread who called WriteMinidumpToFile
  pid_t requester_pid;

  // Tid of the crashing thread.
  pid_t crashed_tid;

  // Signal number when crash happed. Can be 0 if this is a requested dump.
  int signo;

  // The frame pointer of the signal handler frame.  Can be zero if this
  // is a requested dump.
  uintptr_t sighandler_frame_pointer;

  // Signal context when crash happed. Can be empty if this is a requested dump.
  // This is actually an out parameter, but it will be filled in at the start
  // of the writer thread.
  SignalContext sig_ctx;

  // Used to get information about the threads.
  LinuxThread *thread_lister;
};

// Write the thread stack info minidump.
bool WriteThreadStack(uintptr_t frame_pointer,
                      uintptr_t stack_pointer,
                      const LinuxThread *thread_lister,
                      UntypedMDRVA *memory,
                      MDMemoryDescriptor *loc) {
  // Maximum stack size for a thread.
  uintptr_t stack_bottom = thread_lister->GetThreadStackBottom(frame_pointer);
  if (stack_bottom > stack_pointer) {
    size_t size = stack_bottom - stack_pointer;
    if (size > 0) {
      if (!memory->Allocate(size))
        return false;
      memory->Copy(reinterpret_cast<void*>(stack_pointer), size);
      loc->start_of_memory_range = stack_pointer;
      loc->memory = memory->location();
    }
    return true;
  }
  return false;
}

// Write CPU context based on signal context.
bool WriteContext(RawContext *context,
                  const SignalContext &sig_ctx,
                  const DebugRegs *debug_regs) {
  assert(!sig_ctx.IsEmpty());
  if (!sig_ctx.CopyTo(context))
    return false;

  if (debug_regs != NULL) {
    CPUContext reg(NULL, NULL, debug_regs);
    return reg.CopyTo(context);
  }
  return true;
}

// Write CPU context based on provided registers.
bool WriteContext(RawContext *context,
                  const CPUContext *cpu_context) {
  if (!context || !cpu_context)
    return false;

  return cpu_context->CopyTo(context);
}

// Write information about a crashed thread.
// When a thread crash, kernel will write something on the stack for processing
// signal. This makes the current stack not reliable, and our stack walker
// won't figure out the whole call stack for this. So we write the stack at the
// time of the crash into the minidump file, not the current stack.
bool WriteCrashedThreadStream(MinidumpFileWriter *minidump_writer,
                              const WriterArgument *writer_args,
                              const ThreadInfo &thread_info,
                              MDRawThread *thread) {
  assert(!writer_args->sig_ctx.IsEmpty());

  thread->thread_id = thread_info.pid;

  UntypedMDRVA memory(minidump_writer);
  if (!WriteThreadStack(writer_args->sig_ctx.GetFramePointer(),
                        writer_args->sig_ctx.GetStackPointer(),
                        writer_args->thread_lister,
                        &memory,
                        &thread->stack))
    return false;

  TypedMDRVA<RawContext> context(minidump_writer);
  if (!context.Allocate())
    return false;
  thread->thread_context = context.location();
  memset(context.get(), 0, sizeof(RawContext));
  return WriteContext(context.get(), writer_args->sig_ctx, NULL);
}

// Write information about a thread.
// This function only processes thread running normally at the crash.
bool WriteThreadStream(MinidumpFileWriter *minidump_writer,
                       const LinuxThread *thread_lister,
                       const ThreadInfo &thread_info,
                       MDRawThread *thread) {
  thread->thread_id = thread_info.pid;

  struct user_regs_struct user_regs;
  memset(&user_regs, 0, sizeof(user_regs));
  if (!thread_lister->GetRegisters(thread_info.pid, &user_regs)) {
    return false;
  }

  struct user_fpregs_struct fp_regs;
  DebugRegs debug_regs;
  memset(&fp_regs, 0, sizeof(fp_regs));
  // Get all the registers.
  thread_lister->GetFPRegisters(thread_info.pid, &fp_regs);
  thread_lister->GetDebugRegisters(thread_info.pid, &debug_regs);

  CPUContext cpu_context(&user_regs, &fp_regs, &debug_regs);
    // Write context
  TypedMDRVA<RawContext> context(minidump_writer);
  if (!context.Allocate())
    return false;
  thread->thread_context = context.location();
  memset(context.get(), 0, sizeof(RawContext));
  if (!WriteContext(context.get(), &cpu_context))
    return false;

  UntypedMDRVA memory(minidump_writer);
  return WriteThreadStack(cpu_context.GetFramePointer(),
                          cpu_context.GetStackPointer(),
                          thread_lister,
                          &memory,
                          &thread->stack);
}

bool WriteCPUInformation(MDRawSystemInfo *sys_info) {
  const char *proc_cpu_path = "/proc/cpuinfo";
  char line[128];
  char vendor_id[13];
  const char vendor_id_name[] = "vendor_id";
  const size_t vendor_id_name_length = sizeof(vendor_id_name) - 1;

  struct CpuInfoEntry {
    const char *info_name;
    int value;
  } cpu_info_table[] = {
    { "processor", -1 },
    { "model", 0 },
    { "stepping",  0 },
    { "cpuid level", 0 },
    { NULL, -1 },
  };

  memset(vendor_id, 0, sizeof(vendor_id));

  FILE *fp = fopen(proc_cpu_path, "r");
  if (fp != NULL) {
    while (fgets(line, sizeof(line), fp)) {
      CpuInfoEntry *entry = &cpu_info_table[0];
      while (entry->info_name != NULL) {
        if (!strncmp(line, entry->info_name, strlen(entry->info_name))) {
          char *value = strchr(line, ':');
          value++;
          if (value != NULL)
            sscanf(value, " %d", &(entry->value));
        }
        entry++;
      }

      // special case for vendor_id
      if (!strncmp(line, vendor_id_name, vendor_id_name_length)) {
        char *value = strchr(line, ':');
        if (value == NULL)
          continue;

        value++;
        while (*value && isspace(*value))
          value++;
        if (*value) {
          size_t length = strlen(value);
          // we don't want the trailing newline
          if (value[length - 1] == '\n')
            length--;
          // ensure we have space for the value
          if (length < sizeof(vendor_id))
            strncpy(vendor_id, value, length);
        }
      }
    }
    fclose(fp);
  }

  // /proc/cpuinfo contains cpu id, change it into number by adding one.
  cpu_info_table[0].value++;

  sys_info->number_of_processors = cpu_info_table[0].value;
  sys_info->processor_level      = cpu_info_table[3].value;
  sys_info->processor_revision   = cpu_info_table[1].value << 8 |
                                   cpu_info_table[2].value;

  sys_info->processor_architecture = MD_CPU_ARCHITECTURE_UNKNOWN;
  struct utsname uts;
  if (uname(&uts) == 0) {
    // Match x86_64 architecture.
    if (strstr(uts.machine, "x86_64") == uts.machine) {
      sys_info->processor_architecture = MD_CPU_ARCHITECTURE_AMD64;
    } else if ((strstr(uts.machine, "x86") == uts.machine) ||
               (strlen(uts.machine) == 4 &&
                uts.machine[0] == 'i' &&
                uts.machine[2] == '8' &&
                uts.machine[3] == '6')) {
      // Match i*86 and x86* as X86 architecture.
      sys_info->processor_architecture = MD_CPU_ARCHITECTURE_X86;
      if (vendor_id[0] != '\0')
        memcpy(sys_info->cpu.x86_cpu_info.vendor_id, vendor_id,
               sizeof(sys_info->cpu.x86_cpu_info.vendor_id));
    }
  }
  return true;
}

bool WriteOSInformation(MinidumpFileWriter *minidump_writer,
                        MDRawSystemInfo *sys_info) {
  sys_info->platform_id = MD_OS_LINUX;

  struct utsname uts;
  if (uname(&uts) == 0) {
    char os_version[512];
    size_t space_left = sizeof(os_version);
    memset(os_version, 0, space_left);
    const char *os_info_table[] = {
      uts.sysname,
      uts.release,
      uts.version,
      uts.machine,
      "GNU/Linux",
      NULL
    };
    for (const char **cur_os_info = os_info_table;
         *cur_os_info != NULL;
         cur_os_info++) {
      if (cur_os_info != os_info_table && space_left > 1) {
        strcat(os_version, " ");
        space_left--;
      }
      if (space_left > strlen(*cur_os_info)) {
        strcat(os_version, *cur_os_info);
        space_left -= strlen(*cur_os_info);
      } else {
        break;
      }
    }

    MDLocationDescriptor location;
    if (!minidump_writer->WriteString(os_version, 0, &location))
      return false;
    sys_info->csd_version_rva = location.rva;
  }
  return true;
}

// Callback context for get writting thread information.
struct ThreadInfoCallbackCtx {
  MinidumpFileWriter *minidump_writer;
  const WriterArgument *writer_args;
  TypedMDRVA<MDRawThreadList> *list;
  int thread_index;
};

// Callback run for writing threads information in the process.
bool ThreadInfomationCallback(const ThreadInfo &thread_info,
                              void *context) {
  ThreadInfoCallbackCtx *callback_context =
    static_cast<ThreadInfoCallbackCtx *>(context);
  bool success = true;
  MDRawThread thread;
  memset(&thread, 0, sizeof(MDRawThread));
  if (thread_info.pid != callback_context->writer_args->crashed_tid ||
      callback_context->writer_args->sig_ctx.IsEmpty()) {
#ifdef ONLY_CRAHSING_THREAD
    // Ignore those threads that works normally.
    return true;
#else
    success = WriteThreadStream(callback_context->minidump_writer,
                           callback_context->writer_args->thread_lister,
                           thread_info, &thread);
#endif
  } else {
    success = WriteCrashedThreadStream(callback_context->minidump_writer,
                                       callback_context->writer_args,
                                       thread_info, &thread);
  }
  if (success) {
    callback_context->list->CopyIndexAfterObject(
        callback_context->thread_index++,
        &thread, sizeof(MDRawThread));
  }
  return success;
}

// Stream writers
bool WriteThreadListStream(MinidumpFileWriter *minidump_writer,
                           const WriterArgument *writer_args,
                           MDRawDirectory *dir) {
  // Get the thread information.
  const LinuxThread *thread_lister = writer_args->thread_lister;
  int thread_count = thread_lister->GetThreadCount();
  if (thread_count < 0)
    return false;

#ifdef ONLY_CRAHSING_THREAD
  // Only count the crashing thread.
  thread_count = 1;
#endif

  TypedMDRVA<MDRawThreadList> list(minidump_writer);
  if (!list.AllocateObjectAndArray(thread_count, sizeof(MDRawThread)))
    return false;
  dir->stream_type = MD_THREAD_LIST_STREAM;
  dir->location = list.location();
  list.get()->number_of_threads = thread_count;

  ThreadInfoCallbackCtx context;
  context.minidump_writer = minidump_writer;
  context.writer_args = writer_args;
  context.list = &list;
  context.thread_index = 0;
  CallbackParam<ThreadCallback> callback_param(ThreadInfomationCallback,
                                               &context);
  thread_lister->ListThreads(&callback_param);
  return true;
}

bool WriteCVRecord(MinidumpFileWriter *minidump_writer,
                   MDRawModule *module,
                   const char *module_path) {
  TypedMDRVA<MDCVInfoPDB70> cv(minidump_writer);

  // Only return the last path component of the full module path
  const char *module_name = strrchr(module_path, '/');
  // Increment past the slash
  if (module_name)
    ++module_name;
  else
    module_name = "<Unknown>";

  size_t module_name_length = strlen(module_name);
  if (!cv.AllocateObjectAndArray(module_name_length + 1, sizeof(u_int8_t)))
    return false;
  if (!cv.CopyIndexAfterObject(0, const_cast<char *>(module_name),
                               module_name_length))
    return false;

  module->cv_record = cv.location();
  MDCVInfoPDB70 *cv_ptr = cv.get();
  memset(cv_ptr, 0, sizeof(MDCVInfoPDB70));
  cv_ptr->cv_signature = MD_CVINFOPDB70_SIGNATURE;
  cv_ptr->age = 0;

  // Get the module identifier
  FileID file_id(module_path);
  unsigned char identifier[16];

  if (file_id.ElfFileIdentifier(identifier)) {
    cv_ptr->signature.data1 = (uint32_t)identifier[0] << 24 |
      (uint32_t)identifier[1] << 16 | (uint32_t)identifier[2] << 8 |
      (uint32_t)identifier[3];
    cv_ptr->signature.data2 = (uint32_t)identifier[4] << 8 | identifier[5];
    cv_ptr->signature.data3 = (uint32_t)identifier[6] << 8 | identifier[7];
    cv_ptr->signature.data4[0] = identifier[8];
    cv_ptr->signature.data4[1] = identifier[9];
    cv_ptr->signature.data4[2] = identifier[10];
    cv_ptr->signature.data4[3] = identifier[11];
    cv_ptr->signature.data4[4] = identifier[12];
    cv_ptr->signature.data4[5] = identifier[13];
    cv_ptr->signature.data4[6] = identifier[14];
    cv_ptr->signature.data4[7] = identifier[15];
  }
  return true;
}

struct ModuleInfoCallbackCtx {
  MinidumpFileWriter *minidump_writer;
  const WriterArgument *writer_args;
  TypedMDRVA<MDRawModuleList> *list;
  int module_index;
};

bool ModuleInfoCallback(const ModuleInfo &module_info,
                        void *context) {
  ModuleInfoCallbackCtx *callback_context =
    static_cast<ModuleInfoCallbackCtx *>(context);
  // Skip those modules without name, or those that are not modules.
  if (strlen(module_info.name) == 0 ||
      !strchr(module_info.name, '/'))
    return true;

  MDRawModule module;
  memset(&module, 0, sizeof(module));
  MDLocationDescriptor loc;
  if (!callback_context->minidump_writer->WriteString(module_info.name, 0,
                                                      &loc))
    return false;
  module.base_of_image = (u_int64_t)module_info.start_addr;
  module.size_of_image = module_info.size;
  module.module_name_rva = loc.rva;

  if (!WriteCVRecord(callback_context->minidump_writer, &module,
                     module_info.name))
    return false;
  callback_context->list->CopyIndexAfterObject(
      callback_context->module_index++, &module, MD_MODULE_SIZE);
  return true;
}

bool WriteModuleListStream(MinidumpFileWriter *minidump_writer,
                           const WriterArgument *writer_args,
                           MDRawDirectory *dir) {
  TypedMDRVA<MDRawModuleList> list(minidump_writer);
  int module_count  = writer_args->thread_lister->GetModuleCount();
  if (module_count <= 0 ||
      !list.AllocateObjectAndArray(module_count, MD_MODULE_SIZE))
    return false;
  dir->stream_type = MD_MODULE_LIST_STREAM;
  dir->location = list.location();
  list.get()->number_of_modules = module_count;
  ModuleInfoCallbackCtx context;
  context.minidump_writer = minidump_writer;
  context.writer_args = writer_args;
  context.list = &list;
  context.module_index = 0;
  CallbackParam<ModuleCallback> callback(ModuleInfoCallback, &context);
  return writer_args->thread_lister->ListModules(&callback) == module_count;
}

bool WriteSystemInfoStream(MinidumpFileWriter *minidump_writer,
                           const WriterArgument *writer_args,
                           MDRawDirectory *dir) {
  TypedMDRVA<MDRawSystemInfo> sys_info(minidump_writer);
  if (!sys_info.Allocate())
    return false;
  dir->stream_type = MD_SYSTEM_INFO_STREAM;
  dir->location = sys_info.location();

  return WriteCPUInformation(sys_info.get()) &&
         WriteOSInformation(minidump_writer, sys_info.get());
}

bool WriteExceptionStream(MinidumpFileWriter *minidump_writer,
                          const WriterArgument *writer_args,
                          MDRawDirectory *dir) {
  // This happenes when this is not a crash, but a requested dump.
  if (writer_args->sig_ctx.IsEmpty())
    return false;

  TypedMDRVA<MDRawExceptionStream> exception(minidump_writer);
  if (!exception.Allocate())
    return false;

  dir->stream_type = MD_EXCEPTION_STREAM;
  dir->location = exception.location();
  exception.get()->thread_id = writer_args->crashed_tid;
  exception.get()->exception_record.exception_code = writer_args->signo;
  exception.get()->exception_record.exception_flags = 0;
  if (!writer_args->sig_ctx.IsEmpty()) {
    exception.get()->exception_record.exception_address =
      writer_args->sig_ctx.GetIP();
  } else {
    return true;
  }

  // Write context of the exception.
  TypedMDRVA<RawContext> context(minidump_writer);
  if (!context.Allocate())
    return false;
  exception.get()->thread_context = context.location();
  memset(context.get(), 0, sizeof(RawContext));
  return WriteContext(context.get(), writer_args->sig_ctx, NULL);
}

bool WriteMiscInfoStream(MinidumpFileWriter *minidump_writer,
                         const WriterArgument *writer_args,
                         MDRawDirectory *dir) {
  TypedMDRVA<MDRawMiscInfo> info(minidump_writer);
  if (!info.Allocate())
    return false;

  dir->stream_type = MD_MISC_INFO_STREAM;
  dir->location = info.location();
  info.get()->size_of_info = sizeof(MDRawMiscInfo);
  info.get()->flags1 = MD_MISCINFO_FLAGS1_PROCESS_ID;
  info.get()->process_id = writer_args->requester_pid;

  return true;
}

bool WriteBreakpadInfoStream(MinidumpFileWriter *minidump_writer,
                             const WriterArgument *writer_args,
                             MDRawDirectory *dir) {
  TypedMDRVA<MDRawBreakpadInfo> info(minidump_writer);
  if (!info.Allocate())
    return false;

  dir->stream_type = MD_BREAKPAD_INFO_STREAM;
  dir->location = info.location();

  info.get()->validity = MD_BREAKPAD_INFO_VALID_DUMP_THREAD_ID |
                        MD_BREAKPAD_INFO_VALID_REQUESTING_THREAD_ID;
  info.get()->dump_thread_id = getpid();
  info.get()->requesting_thread_id = writer_args->requester_pid;
  return true;
}

// Prototype of writer functions.
typedef bool (*WriteStringFN)(MinidumpFileWriter *,
                              const WriterArgument *,
                              MDRawDirectory *);

// Function table to writer a full minidump.
WriteStringFN writers[] = {
  WriteThreadListStream,
  WriteModuleListStream,
  WriteSystemInfoStream,
  WriteExceptionStream,
  WriteMiscInfoStream,
  WriteBreakpadInfoStream,
};

// Will call each writer function in the writers table.
// It runs in a different process from the crashing process, but sharing
// the same address space. This enables it to use ptrace functions.
int Write(void *argument) {
  WriterArgument *writer_args =
    static_cast<WriterArgument *>(argument);

  if (!writer_args->thread_lister->SuspendAllThreads())
    return -1;

  if (writer_args->sighandler_frame_pointer != 0 &&
      !writer_args->thread_lister->FindSigContext(
          writer_args->sighandler_frame_pointer,
          &writer_args->sig_ctx))
    return -1;

  MinidumpFileWriter *minidump_writer = writer_args->minidump_writer;
  TypedMDRVA<MDRawHeader> header(minidump_writer);
  TypedMDRVA<MDRawDirectory> dir(minidump_writer);
  if (!header.Allocate())
    return 0;

  int writer_count = sizeof(writers) / sizeof(writers[0]);
  // Need directory space for all writers.
  if (!dir.AllocateArray(writer_count))
    return 0;
  header.get()->signature = MD_HEADER_SIGNATURE;
  header.get()->version = MD_HEADER_VERSION;
  header.get()->time_date_stamp = time(NULL);
  header.get()->stream_count = writer_count;
  header.get()->stream_directory_rva = dir.position();

  int dir_index = 0;
  MDRawDirectory local_dir;
  for (int i = 0; i < writer_count; ++i) {
    if (writers[i](minidump_writer, writer_args, &local_dir))
      dir.CopyIndex(dir_index++, &local_dir);
  }

  writer_args->thread_lister->ResumeAllThreads();
  return 0;
}

}  // namespace

namespace google_breakpad {

MinidumpGenerator::MinidumpGenerator() {
  AllocateStack();
}

MinidumpGenerator::~MinidumpGenerator() {
}

void MinidumpGenerator::AllocateStack() {
  stack_.reset(new char[kStackSize]);
}

bool MinidumpGenerator::WriteMinidumpToFile(
    const char *file_pathname,
    int signo,
    uintptr_t sighandler_frame_pointer,
    SignalContext *raw_sig_ctx) const {

  assert(file_pathname != NULL);
  assert(stack_.get() != NULL);

  if (file_pathname == NULL)
    return false;

  MinidumpFileWriter minidump_writer;
  if (minidump_writer.Open(file_pathname)) {
    WriterArgument argument;
    memset(&argument, 0, sizeof(argument));
    LinuxThread thread_lister(getpid());
    argument.thread_lister = &thread_lister;
    argument.minidump_writer = &minidump_writer;
    argument.requester_pid = getpid();
    // Signal we are catching are synchronous signals, and the execution
    // of signal handlers should be on the thread which is causing the signal.
    argument.crashed_tid = gettid();
    argument.signo = signo;
    argument.sighandler_frame_pointer = sighandler_frame_pointer;
#ifdef __x86_64__
    if (raw_sig_ctx)
      argument.sig_ctx.SetSignalContext(raw_sig_ctx->GetRawContext());
#endif

    pid_t cloned_pid = clone(Write, stack_.get() + kStackSize,
                           CLONE_VM | CLONE_FILES | CLONE_FS | CLONE_UNTRACED,
                           (void*)&argument);
    waitpid(cloned_pid, NULL, __WALL);
    if (raw_sig_ctx != NULL)
        raw_sig_ctx->SetSignalContext(argument.sig_ctx.GetRawContext());
    return true;
  }

  return false;
}

}  // namespace google_breakpad
