-- Topal: GPG/GnuPG and Alpine/Pine integration
-- Copyright (C) 2001--2017  Phillip J. Brooke
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License version 3 as
-- published by the Free Software Foundation.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

with Ada.Characters.Latin_1;
with Ada.Containers.Vectors;
with Ada.Strings.Unbounded;
with Ada.Text_IO;

package Globals is

   -- Initial number of keys to be associated.
   Initial_KL : constant Positive := 300;

   Not_Implemented : exception;

   -- Some renamings of unbounded strings.
   subtype UBS is Ada.Strings.Unbounded.Unbounded_String;

   NullUBS : UBS renames Ada.Strings.Unbounded.Null_Unbounded_String;

   function ToUBS(S : String) return UBS
     renames Ada.Strings.Unbounded.To_Unbounded_String;

   function ToStr(U : UBS) return String
     renames Ada.Strings.Unbounded.To_String;

   -- We will want to play with arrays of UBS.
   type UBS_Array is array (Integer range <>) of UBS;

   -- Pointers to UBS arrays.
   type UBS_Array_Pointer is access UBS_Array;

   -- We also want expanding arrays....
   package UVP
   is new Ada.Containers.Vectors (Index_Type => Positive,
                                  Element_Type => UBS,
                                  "=" => Ada.Strings.Unbounded."=");
   subtype UVV is UVP.Vector;

   -- We also want key-value pairs of UBS.
   type UP is record
      Key, Value : UBS;
   end record;
   package UPP
   is new Ada.Containers.Vectors (Index_Type => Positive,
				  Element_Type => UP);
   subtype UPV is UPP.Vector;

   -- Want to insert line feeds in some strings.
   NL : constant String    := Ada.Characters.Latin_1.LF & "";

   -- Some terminal twiddling.
   ANSI_CSI : constant String := Ada.Characters.Latin_1.ESC & "[";
   ANSI_SGR : constant String := "m";
   ANSI_SGR_Reset : constant String := ANSI_CSI & ANSI_SGR;

   -- The result file stuff.
   Result_File : Ada.Text_IO.File_Type;

   -- What is our process ID?  (Needed for tempfiles.)
   Our_PID : Integer;

   -- Our home directory.
   Topal_Directory : UBS;

   type Binaries is
     (Chmod, Clear, Cut, Date, Diff, File,
      Formail, GPGOP, GPGSM, Grep, Iconv, Less, Locale,
      Md5sum, Metamail, Mimeconstruct, Mimetool,
      Mkdir, Mkfifo, Mv, Openssl, Rm, Runmailcap,
      Scp, Sed, Ssh, Stty, Tee, Test);

   type Binaries_UBS is array (Binaries) of UBS;

   type UBS_Opts_Keys is (My_Key, General_Options, 
			  GPG_Options, GPGSM_Options,
                          Receiving_Options, Sending_Options,
                          Colour_Menu_Title, Colour_Menu_Key,
                          Colour_Menu_Choice, Colour_Important,
                          Colour_Banner, Colour_Info, Decrypt_Prereq);

   type Boolean_Opts_Keys is (Decrypt_Cached_Fast_Continue,
                              Verify_Cached_Fast_Continue,
                              Verify_Not_Cached_Fast_Continue,
                              FE_Simple,
                              Inline_Separate_Output,
                              Omit_Inline_Disposition_Name,
                              Omit_Inline_Disposition_Header,
                              Save_On_Send, -- If true, write to .topal/lastmail
                              ANSI_Terminal,
                              Wait_If_Missing_Keys,
			      Attachment_Trap,
			      Fix_Fcc,
			      Fix_Bcc,
                              Debug,
                              -- The following are not in the config file.
                              No_Clean, -- Not preserved in config file.
                              All_Headers, -- Not config file.
                              Read_From, -- Not config file.
                              Ask_Charset -- Not config file.
                             );
   subtype Boolean_Opts_Keys_Save is Boolean_Opts_Keys
     range Decrypt_Cached_Fast_Continue .. Debug;

   type Positive_Opts_Keys is (Decrypt_Not_Cached,
                               Decrypt_Not_Cached_Use_Cache,
                               Decrypt_Cached,
                               Decrypt_Cached_Use_Cache,
                               Verify_Not_Cached,
                               Verify_Not_Cached_Use_Cache,
                               Verify_Cached,
                               Verify_Cached_Use_Cache,
                               Mime_Viewer_Decrypt,
                               Mime_Viewer_Verify,
                               Use_Agent,
			       Replace_IDs,
			       Include_Send_Token);
   type UBS_Opts_Array is array (UBS_Opts_Keys) of UBS;

   type Boolean_Opts_Array is array (Boolean_Opts_Keys) of Boolean;

   type Positive_Opts_Array is array (Positive_Opts_Keys) of Positive;


   -- Configuration data types.
   type Config_Record is
      record
         Binary : Binaries_UBS;
         UBS_Opts : UBS_Opts_Array;
         Boolean_Opts : Boolean_Opts_Array;
         Positive_Opts : Positive_Opts_Array;
         --  decrypt-not-cached: 1 always, 2 ask, 3 never
         --  decrypt-not-cached-use-cache: 1 always, 2 ask, 3 never
         --  decrypt-cached: 1 always, 2 ask, 3 never
         --  decrypt-cached-use-cache: 1 use, 2 again-replace,
         --                            3 again-ask-replace,
         --                            4 again-never, 5 offer

         --  Ditto for verify-not-cached, verify-not-cached-use-cache,
         --  verify-cached, verify-cached-use-Cache

         -- MIME_Viewer: 1 ask, 2 use metamail, 3 use run-mailcap,
         --              4 save dummy message to ~/mail, 5 skip attachments
         -- There is one for _Decrypt and one of _Verify.

         -- Use_Agent: asks about using gpg-agent.
         -- 1 never, 2 only for decrypting (not signing!), 3 always
	 
	 -- Replace_IDs: only effective in sendmail-path mode.
	 -- 0 do nothing, 1 replace only Message-ID, 2 replace both Message-ID and Content-ID.
	 -- 1 & 2 use this pattern replacement:
	 --   <alpine.DEB.2.00.1103061044360.3250@catbert.scm.tees.ac.uk>
	 -- to
	 --   <1103061044360.3250%pjb@scm.tees.ac.uk>
	 -- where the address used is Alt_Sign (from the From line).
	 
	 -- Include_Send_Token: only effective in sendmail-path mode.
	 -- 1 never, 2 ask, 3 always. Defaults to 3, but you may find 2 more useful.

         -- A list of keys to be associated with email addresses.
	 -- Key: the crypto key; Value: the email.
	 AKE : UPV;
         -- A list of keys to be excluded from the key list.
         XK : UVV;
         -- And the same again, this time for the secret key lists.
         SAKE : UPV;
         -- A list of keys to be excluded from the key list.
         SXK : UVV;
         -- A list of keys and the default send mode.
         SD : UPV;
	 -- A list of email addresses (Key) and the sendmail command to use (value).
	 SC : UPV;
	 -- Sending tokens.  A list of email addresses (from) and the
	 --  token to use.  @ANY@ isn't valid.
	 ST : UPV;
      end record;

   Config : Config_Record;

end Globals;
