/*
 * Copyright (C) 2016-2025 by the Widelands Development Team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <https://www.gnu.org/licenses/>.
 *
 */

#include "ui_fsmenu/crash_report.h"

#include "graphic/text_layout.h"
#include "io/fileread.h"
#include "logic/filesystem_constants.h"
#include "ui_basic/box.h"
#include "ui_basic/button.h"
#include "ui_basic/multilinetextarea.h"
#include "ui_basic/textarea.h"
#include "ui_fsmenu/tech_info.h"

namespace FsMenu {

constexpr int kButtonSize = 160;

class CrashReportDetails : public UI::Window {
public:
	CrashReportDetails(FsMenu::MainMenu& menu, const std::string& report)
	   : UI::Window(&menu,
	                UI::WindowStyle::kFsMenu,
	                "crash_details",
	                0,
	                0,
	                0,
	                0,
	                _("Crash Report Details")) {
		UI::Box* box = new UI::Box(
		   this, UI::PanelStyle::kFsMenu, "main_box", 0, 0, UI::Box::Vertical, 0, 0, kPadding);

		std::string detailed_text_rt = richtext_escape(report);
		newlines_to_richtext(detailed_text_rt);
		detailed_text_rt =
		   as_richtext_paragraph(detailed_text_rt, UI::FontStyle::kFsMenuInfoPanelParagraph);

		UI::MultilineTextarea* details = new UI::MultilineTextarea(
		   box, "details", 0, 0, 700, 400, UI::PanelStyle::kFsMenu, detailed_text_rt);

		UI::Box* buttons_box = new UI::Box(
		   box, UI::PanelStyle::kFsMenu, "buttons_box", 0, 0, UI::Box::Horizontal, 0, 0, kPadding);

		UI::Button* b_open_link = new UI::Button(
		   buttons_box, "open_link", 0, 0, kButtonSize, 0, UI::ButtonStyle::kFsMenuSecondary,
#if CAN_OPEN_HYPERLINK
		   _("Open link")
#else
		   _("Copy link")
#endif
		);
		UI::Button* b_copy = new UI::Button(buttons_box, "copy", 0, 0, kButtonSize, 0,
		                                    UI::ButtonStyle::kFsMenuSecondary, _("Copy report"),
		                                    _("Copy the full report to the clipboard"));
		UI::Button* b_close = new UI::Button(
		   buttons_box, "close", 0, 0, kButtonSize, 0, UI::ButtonStyle::kFsMenuPrimary, _("Close"));

		b_open_link->sigclicked.connect([]() {
#if CAN_OPEN_HYPERLINK
			SDL_OpenURL(CrashReportWindow::kReportBugsURL);
#else
			SDL_SetClipboardText(CrashReportWindow::kReportBugsURL);
#endif
		});

		b_copy->sigclicked.connect([report]() { SDL_SetClipboardText(report.c_str()); });

		b_close->sigclicked.connect([this]() { die(); });

		buttons_box->add(b_open_link, UI::Box::Resizing::kExpandBoth);
		buttons_box->add(b_copy, UI::Box::Resizing::kExpandBoth);
		buttons_box->add(b_close, UI::Box::Resizing::kExpandBoth);
		box->add(details, UI::Box::Resizing::kExpandBoth);
		box->add(buttons_box, UI::Box::Resizing::kExpandBoth);

		set_center_panel(box);
		center_to_parent();
		initialization_complete();
	}
};

CrashReportWindow::CrashReportWindow(FsMenu::MainMenu& menu, const FilenameSet& crash_files)
   : UI::Window(&menu, UI::WindowStyle::kFsMenu, "crash", 0, 0, 0, 0, _("Crash Report")),
     clear_on_close_(crash_files) {

	UI::Box* box = new UI::Box(
	   this, UI::PanelStyle::kFsMenu, "main_box", 0, 0, UI::Box::Vertical, 0, 0, kPadding);

	// not localized
	std::string detailed_text =
	   format("# Autogenerated crash report for %u crash log(s):\n\n", crash_files.size());
	for (const std::string& filename : crash_files) {
		detailed_text += "\n- ";
		detailed_text += FileSystem::fs_filename(filename.c_str());
	}
	detailed_text += "\n\n## Technical report:\n";
	detailed_text += TechInfo(TechInfo::Type::kAbout).get_markdown();
	for (const std::string& filename : crash_files) {
		detailed_text += "\n\n## ";
		detailed_text += filename;
		detailed_text += "\n\n```\n";

		FileRead fr;
		try {
			fr.open(*g_fs, filename);
			detailed_text += fr.data(fr.get_size());
		} catch (const std::exception& e) {
			detailed_text += "Error opening crash report file:\n";
			detailed_text += e.what();
		}
		detailed_text += "```\n";
	}

	UI::MultilineTextarea* header = new UI::MultilineTextarea(
	   box, "header", 0, 0, 200, 200, UI::PanelStyle::kFsMenu,
	   as_richtext_paragraph(
	      format(_("It looks like Widelands crashed the last time you ran it, but a backtrace for "
	               "debugging was automatically generated. Please provide the report along with a "
	               "description of what was going on in the game and what you did last before the "
	               "crash to the Widelands Development Team under %s so we can fix the problem."),
	             as_url_hyperlink(kReportBugsURL)),
	      UI::FontStyle::kFsMenuLabel),
	   UI::Align::kCenter, UI::MultilineTextarea::ScrollMode::kNoScrolling);

	UI::Button* b_details =
	   new UI::Button(box, "details", 0, 0, 2 * kButtonSize, 0, UI::ButtonStyle::kFsMenuSecondary,
	                  _("Show details…"), _("View the detailed crash report"));

	UI::Box* buttons_box = new UI::Box(
	   box, UI::PanelStyle::kFsMenu, "buttons_box", 0, 0, UI::Box::Horizontal, 0, 0, kPadding);
	UI::Button* b_open_link = new UI::Button(
	   buttons_box, "open_link", 0, 0, kButtonSize, 0, UI::ButtonStyle::kFsMenuSecondary,
#if CAN_OPEN_HYPERLINK
	   _("Open link")
#else
	   _("Copy link")
#endif
	);
	UI::Button* b_copy =
	   new UI::Button(buttons_box, "copy", 0, 0, kButtonSize, 0, UI::ButtonStyle::kFsMenuSecondary,
	                  _("Copy report"), _("Copy the full report to the clipboard"));
	UI::Button* b_remind_me_later =
	   new UI::Button(buttons_box, "remind_me_later", 0, 0, kButtonSize, 0,
	                  UI::ButtonStyle::kFsMenuSecondary, _("Remind me later"));
	UI::Button* b_close = new UI::Button(
	   buttons_box, "close", 0, 0, kButtonSize, 0, UI::ButtonStyle::kFsMenuPrimary, _("Close"));

	buttons_box->add(b_open_link, UI::Box::Resizing::kExpandBoth);
	buttons_box->add(b_copy, UI::Box::Resizing::kExpandBoth);
	buttons_box->add(b_remind_me_later, UI::Box::Resizing::kExpandBoth);
	buttons_box->add(b_close, UI::Box::Resizing::kExpandBoth);

	b_open_link->sigclicked.connect([]() {
#if CAN_OPEN_HYPERLINK
		SDL_OpenURL(kReportBugsURL);
#else
		SDL_SetClipboardText(kReportBugsURL);
#endif
	});

	b_copy->sigclicked.connect([detailed_text]() { SDL_SetClipboardText(detailed_text.c_str()); });

	b_remind_me_later->sigclicked.connect([this]() {
		clear_on_close_.clear();
		die();
	});

	b_close->sigclicked.connect([this]() { die(); });

	b_details->sigclicked.connect([&menu, detailed_text]() {
		CrashReportDetails d(menu, detailed_text);
		d.run<UI::Panel::Returncodes>();
	});

	box->add(header, UI::Box::Resizing::kFullSize);
	box->add_space(kPadding);
	box->add(b_details, UI::Box::Resizing::kAlign, UI::Align::kCenter);
	box->add_space(kPadding);
	box->add(buttons_box, UI::Box::Resizing::kFullSize);

	set_center_panel(box);
	center_to_parent();
	initialization_complete();
}

void CrashReportWindow::die() {
	for (const std::string& filename : clear_on_close_) {
		try {
			g_fs->fs_rename(filename, filename + kOldCrashExtension);
			verb_log_info("Marked crash report as old: %s", filename.c_str());
		} catch (const std::exception& e) {
			log_err("Could not mark crash report '%s' as old: %s", filename.c_str(), e.what());
		}
	}

	UI::Window::die();
}

}  // namespace FsMenu
