/*
 * colormap.i --
 *
 * Yorick functions to deal with colormaps.
 *
 * ---------------------------------------------------------------------------
 *
 * This file is part of YLib available at <https://github.com/emmt/ylib> and
 * licensed under the MIT "Expat" License.
 *
 * Copyright (C) 2010, Éric Thiébaut.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 *
 * ----------------------------------------------------------------------------
 */

func extract_image_colors(img, quiet=, full=)
/* DOCUMENT cmap = extract_image_colors(img)

     Extract the unique colors of image IMG, a 3-by-any array of char.
     IMG(1,..), IMG(2,..) and IMG(3,..) are assumed to be the channels of the
     image (presumably the red, green and blue ones).

     The result is a 3-by-N array of char with N the number of unique colors.

   SEE ALSO: heapsort.
 */
{
  if (! is_func(heapsort)) include, "yeti.i", 1;
  local q;
  dims = dimsof(img);
  rank = numberof(dims) - 1;
  if (structof(img) != char || rank < 2 || dims(2) != 3) {
    error, "expecting a 3-by-ANY array of char";
  }
  q = ((int(img(1,*)) << 16) | (int(img(2,*)) << 8) | int(img(3,*)));
  heapsort, q;
  k = where(q(dif));
  n = numberof(k) + 1;
  if (! quiet) {
    write, format="found %d unique color(s)\n", n;
  }
  if (n > 1) {
    q = grow(q(1), q(k+1));
  } else {
    q = q(1);
  }
  cmap = array(char,3,n);
  cmap(1,) = char(q >> 16);
  cmap(2,) = char(q >>  8);
  cmap(3,) = char(q      );
  return cmap;
}

func make_indexed_image(img, quiet=)
/* DOCUMENT ptr = make_indexed_image(img)

     Extract the unique colors of RGB image IMG, a 3-by-any array of char and
     return an indexed version of the image with the unique colors.  IMG(1,..),
     IMG(2,..) and IMG(3,..) are assumed to be the channels of the image
     (presumably the red, green and blue ones).

     The result is 2 component pointer:

         *ptr(1) = the indexed image IDX: an array of 1-based integer index in
                   the colormap;

         *ptr(2) = the color table CMAP: a 3-by-N array of char with N the
                   number of unique colors.

     such that CMAP(,IDX) yields the original image.

   SEE ALSO: heapsort.
 */
{
  if (! is_func(heapsort)) include, "yeti.i", 1;
  local q;
  dims = dimsof(img);
  rank = numberof(dims) - 1;
  if (structof(img) != char || rank < 2 || dims(2) != 3) {
    error, "expecting a 3-by-ANY array of char";
  }
  c = ((long(img(1,..)) << 16) | (long(img(2,..)) << 8) | long(img(3,..)));
  j = heapsort(c);
  q = c(j);
  r = (q(dif) > 0);
  c(j) = r(cum) + 1;
  k = where(r);
  n = numberof(k) + 1;
  if (! quiet) {
    write, format="found %d unique color(s)\n", n;
  }
  if (n > 1) {
    q = grow(q(1), q(k+1));
  } else {
    q = q(1);
  }
  cmap = array(char,3,n);
  cmap(1,) = char(q >> 16);
  cmap(2,) = char(q >>  8);
  cmap(3,) = char(q      );
  return [&c, &cmap];
}

func compare_colors(a, b, f)
/* DOCUMENT [cab, cba] = compare_colors(a, b);
         or [cab, cba] = compare_colors(a, b, f);

     Compare the two color tables A and B (respectively 3-by-NA and 3-by-NB
     arrays of char) and return a two component vector [CAB, CBA] with CAB the
     average cost of re-encoding the colors of A using the ones in B and
     conversely for CBA.

     If provided, F is a function which will be called with a single argument C
     which is a 3-by-NA-by-NB array of color differences.  It should return a
     NA-by-NB array of color distances.  The default is use the sum of squared
     differences for the color channels.

   SEE ALSO: compare_colors_matrix, reindex_colors.
 */
{
  c = compare_colors_matrix(a, b, f);
  cab = avg(c(,min)); // average the minimal cost if A is repainted with B
  cba = avg(c(min,)); // average the minimal cost if B is repainted with A
  return [cab, cba];
}

func reindex_colors(a, b, f)
/* DOCUMENT idx = reindex_colors(a, b);
         or idx = reindex_colors(a, b, f);

     Return an indirection table IDX, such that j = IDX(i) is the index of the
     entry in color table B that best match the j-th entry in color table A.
     Color tables A and B are respectively 3-by-NA and 3-by-NB arrays of char..

     If provided, F is a function which will be called with a single argument C
     which is a 3-by-NA-by-NB array of color differences.  It should return a
     NA-by-NB array of color distances.  The default is use the sum of squared
     differences for the color channels.

   SEE ALSO: compare_colors, compare_colors_matrix.
 */
{
  c = compare_colors_matrix(a, b, f);
  return c(,mnx);
}

func compare_colors_matrix(a, b, f)
/* DOCUMENT c = compare_colors_matrix(a, b);
         or c = compare_colors_matrix(a, b, f);

     Return the matrix C of color distances between color tables A and B:

         c(i,j) = distance between color a(i,) and color b(j,)

     Color tables A and B are respectively 3-by-NA and 3-by-NB arrays of char.

     If provided, F is a function which will be called with a single argument C
     which is a 3-by-NA-by-NB array of color differences.  It should return a
     NA-by-NB array of color distances.  The default is use the sum of squared
     differences for the color channels.

   SEE ALSO: compare_colors, reindex_colors.
*/
{
  /* Check color table A. */
  dims = dimsof(a);
  rank = numberof(dims) - 1;
  if (structof(a) != char || rank != 2 || dims(2) != 3) {
    error, "expecting a 3-by-N array of char for color table A";
  }
  na = dims(3);

  /* Check color table B. */
  dims = dimsof(b);
  rank = numberof(dims) - 1;
  if (structof(b) != char || rank != 2 || dims(2) != 3) {
    error, "expecting a 3-by-N array of char for color table B";
  }
  nb = dims(3);

  /* Color distances. */
  c = int(a) - int(b(,-,));
  if (is_void(f)) {
    c = (c*c)(sum,..);
  } else {
    c = f(c);
  }
  return c;
}

func get_colormap(name)
{
  return symbol_def("CMAP_" + strcase(1, name) + "_TABLE");
}

func best_colormap(a, f, quiet=)
{
  local b, cbest;
  jbest = -1;
  for (j = 1; j <= numberof(CMAP_NAMES); ++j) {
    name = CMAP_NAMES(j);
    c = compare_colors(a, get_colormap(name), f)(1);
    if (jbest == -1 || c < cbest) {
      jbest = j;
      cbest = c;
    }
  }
  if (! quiet) {
    write, format="best colormap = \"%s\"\n", CMAP_NAMES(jbest);
  }
  return CMAP_NAMES(jbest);
}

func uncolor_image(img, cmap, cost, quiet=)
/* DOCUMENT z = uncolor_image(img);
         or z = uncolor_image(img, cmap);
         or z = uncolor_image(img, cmap, cf);

     Given RGB image IMG, this function attempts to convert it to a gray scale
     image.  In the first case, the gray levels are computed form the red,
     green and blue levels using the NTSC formula:

         GRAY = 0.2126*RED + 0.7152*GREEN + 0.0722*BLUE;

     If a color table CMAP is provided (as a name or as a 3-by-N arrays of
     char) it is used as the color map with which the RGB image was produced.
     CMAP can be "auto" to automatically find the best color map among the
     known ones.  If FC is provided it is used as a color cost function.

   SEE ALSO:
 */
{
  dims = dimsof(img);
  rank = numberof(dims) - 1;
  if (structof(img) != char || rank != 3 || dims(2) != 3) {
    error, "expecting a RGB image";
  }
  if (is_void(cmap)) {
    return char(lround(0.2126*img(1,..) + 0.7152*img(2,..) + 0.0722*img(3,..)));
  }
  local colors, index;
  ptr = make_indexed_image(img, quiet=quiet);
  eq_nocopy, index, *ptr(1);
  eq_nocopy, colors, *ptr(2);
  if (is_scalar(cmap) && structof(cmap) == string) {
    if (cmap == "auto") {
      cmap = best_colormap(colors, fc, quiet=quiet);
    }
    eq_nocopy, cmap, get_colormap(cmap);
  }
  level = reindex_colors(colors, cmap, fc) - 1;
  lmax = max(level);
  if (lmax <= 0xff) {
    level = char(level);
  } else if (lmax <= 0x7fff) {
    level = short(level);
  }
  return level(index);
}

/*--------------------------------------------------------------------------*/

local CMAP_NAMES;
func cmap_codger(name, output)
/* DOCUMENT cmap_codger, name;
         or cmap_codger, name, ouput;
     Generate code for colormap NAME, e.g. "rainbow".  Optional argument
     OUTPUT is a filename to append the code or a text stream.  If NAME is "*"
     code for all color maps is generated.

   SEE ALSO: palette.
 */
{
  local red, green, blue;
  if (is_string(output)) output = open(output, "a");
  if (name == "*") {
    for (i = 1; i <= numberof(CMAP_NAMES); ++i) {
      cmap_codger, CMAP_NAMES(i), output;
      write, output, format="%s", "\n";
    }
    return;
  }
  window, 0;
  fma;
  palette, name+".gp";
  palette, red, green, blue, query=1;
  n = numberof(red);
  NAME = strcase(1, name);

  local separator;
  newcol = ", ";
  newline = ",\n  ";
  triplet = "[%3d,%3d,%3d]%s";
  write, output, format="CMAP_%s_TABLE = char(\n [", NAME;
  for (i = 1; i <= n; ++i) {
    if (i == n) {
      separator = "]);\n";
    } else if (i%5 == 0) {
      eq_nocopy, separator, newline;
    } else {
      eq_nocopy, separator, newcol;
    }
    write, output, format=triplet, red(i), green(i), blue(i), separator;
  }
  write, output, format="CMAP_%s_NUMBER = %d;\n", NAME, n;
  write, output, format="CMAP_%s_LEVEL = (1.0/%.1f)*indgen(0:%d);\n",
    NAME, n - 1.0, n - 1;
  write, output, format="func %s(i)\n\
{\n\
  if (am_subroutine()) {\n\
    if (! is_void(i)) window, i;\n\
    palette, \"%s.gp\";\n\
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {\n\
    if (id <= Y_LONG) i /= 255.0;\n\
    return char(interp(CMAP_%s_TABLE, CMAP_%s_LEVEL, i, 2) + 0.5);\n\
  } else if (is_void(i)) {\n\
    return CMAP_%s_TABLE;\n\
  } else {\n\
    error, \"expecting scalar real or integer\";\n\
  }\n\
}\n", name, name, NAME, NAME, NAME;
}
CMAP_NAMES = [/*"blue_brown",*/ "bone", "copper", "earth", "gray", "heat",
              "hsv", "jet", "ncar", "rainbow", "stern", "yarg"];

local bone, copper, earth, gray, heat, hsv, jet, ncar;
local rainbow, stern, yarg;
/* DOCUMENT  bone, win;          or  rgb = bone(i);
         or  copper, win;        or  rgb = copper(i);
         or  earth, win;         or  rgb = earth(i);
         or  gray, win;          or  rgb = gray(i);
         or  heat, win;          or  rgb = heat(i);
         or  hsv, win;           or  rgb = hsv(i);
         or  jet, win;           or  rgb = jet(i);
         or  ncar, win;          or  rgb = ncar(i);
         or  rainbow, win;       or  rgb = rainbow(i);
         or  stern, win;         or  rgb = stern(i);
         or  yarg, win;          or  rgb = yarg(i);

     When called as a subroutine, these functions load a specific color
     palette for window WIN.  If WIN is not specified, the curent graphics
     window is used.  When called as a function, the returned value is the RGB
     triplet at relative position i in the color table.  If i is an integer
     the relative position is i/255.0; otherwise, i is a real in the range
     [0,1]. If i is nil, the complete color table is returned.

     Note that all the code for these functions has been automatically
     produced by: cmap_codger, "*";

    SEE ALSO palette, window, cmap_codger.
 */
CMAP_BONE_TABLE = char(
 [[  0,  0,  1], [  1,  1,  3], [  2,  2,  4], [  3,  3,  5], [  4,  4,  6],
  [  5,  5,  8], [  6,  6,  9], [  7,  7, 10], [  7,  7, 12], [  8,  8, 13],
  [  9,  9, 14], [ 10, 10, 15], [ 11, 11, 17], [ 12, 12, 18], [ 13, 13, 19],
  [ 14, 14, 21], [ 15, 15, 22], [ 16, 16, 23], [ 17, 17, 24], [ 18, 18, 26],
  [ 19, 19, 27], [ 20, 20, 28], [ 21, 21, 30], [ 21, 21, 31], [ 22, 22, 32],
  [ 23, 23, 33], [ 24, 24, 35], [ 25, 25, 36], [ 26, 26, 37], [ 27, 27, 39],
  [ 28, 28, 40], [ 29, 29, 41], [ 30, 30, 42], [ 31, 31, 44], [ 32, 32, 45],
  [ 33, 33, 46], [ 34, 34, 48], [ 35, 35, 49], [ 35, 35, 50], [ 36, 36, 51],
  [ 37, 37, 53], [ 38, 38, 54], [ 39, 39, 55], [ 40, 40, 57], [ 41, 41, 58],
  [ 42, 42, 59], [ 43, 43, 60], [ 44, 44, 62], [ 45, 45, 63], [ 46, 46, 64],
  [ 47, 47, 66], [ 48, 48, 67], [ 49, 49, 68], [ 49, 49, 69], [ 50, 50, 71],
  [ 51, 51, 72], [ 52, 52, 73], [ 53, 53, 74], [ 54, 54, 76], [ 55, 55, 77],
  [ 56, 56, 78], [ 57, 57, 80], [ 58, 58, 81], [ 59, 59, 82], [ 60, 60, 83],
  [ 61, 61, 85], [ 62, 62, 86], [ 63, 63, 87], [ 63, 63, 89], [ 64, 64, 90],
  [ 65, 65, 91], [ 66, 66, 92], [ 67, 67, 94], [ 68, 68, 95], [ 69, 69, 96],
  [ 70, 70, 98], [ 71, 71, 99], [ 72, 72,100], [ 73, 73,101], [ 74, 74,103],
  [ 75, 75,104], [ 76, 76,105], [ 77, 77,107], [ 77, 77,108], [ 78, 78,109],
  [ 79, 79,110], [ 80, 80,112], [ 81, 81,113], [ 82, 82,114], [ 83, 84,115],
  [ 84, 85,116], [ 85, 86,117], [ 86, 88,118], [ 87, 89,119], [ 88, 90,120],
  [ 89, 91,121], [ 90, 93,121], [ 91, 94,122], [ 91, 95,123], [ 92, 97,124],
  [ 93, 98,125], [ 94, 99,126], [ 95,100,127], [ 96,102,128], [ 97,103,129],
  [ 98,104,130], [ 99,106,131], [100,107,132], [101,108,133], [102,109,134],
  [103,111,135], [104,112,136], [105,113,136], [105,115,137], [106,116,138],
  [107,117,139], [108,118,140], [109,120,141], [110,121,142], [111,122,143],
  [112,123,144], [113,125,145], [114,126,146], [115,127,147], [116,129,148],
  [117,130,149], [118,131,150], [119,132,150], [119,134,151], [120,135,152],
  [121,136,153], [122,138,154], [123,139,155], [124,140,156], [125,141,157],
  [126,143,158], [127,144,159], [128,145,160], [129,147,161], [130,148,162],
  [131,149,163], [132,150,164], [133,152,164], [134,153,165], [134,154,166],
  [135,156,167], [136,157,168], [137,158,169], [138,159,170], [139,161,171],
  [140,162,172], [141,163,173], [142,165,174], [143,166,175], [144,167,176],
  [145,168,177], [146,170,178], [147,171,178], [148,172,179], [148,174,180],
  [149,175,181], [150,176,182], [151,177,183], [152,179,184], [153,180,185],
  [154,181,186], [155,183,187], [156,184,188], [157,185,189], [158,186,190],
  [159,188,191], [160,189,192], [161,190,192], [162,192,193], [162,193,194],
  [163,194,195], [164,195,196], [165,197,197], [166,198,198], [167,199,199],
  [169,200,200], [170,201,201], [172,202,202], [173,203,203], [175,204,204],
  [176,205,205], [178,206,206], [179,206,206], [181,207,207], [182,208,208],
  [184,209,209], [185,210,210], [186,211,211], [188,212,212], [189,213,213],
  [191,214,214], [192,215,215], [194,216,216], [195,217,217], [197,218,218],
  [198,219,219], [200,220,220], [201,220,220], [202,221,221], [204,222,222],
  [205,223,223], [207,224,224], [208,225,225], [210,226,226], [211,227,227],
  [213,228,228], [214,229,229], [216,230,230], [217,231,231], [219,232,232],
  [220,233,233], [221,234,234], [223,234,234], [224,235,235], [226,236,236],
  [227,237,237], [229,238,238], [230,239,239], [232,240,240], [233,241,241],
  [235,242,242], [236,243,243], [237,244,244], [239,245,245], [240,246,246],
  [242,247,247], [243,248,248], [245,248,248], [246,249,249], [248,250,250],
  [249,251,251], [251,252,252], [252,253,253], [254,254,254], [255,255,255]]);
CMAP_BONE_NUMBER = 240;
CMAP_BONE_LEVEL = (1.0/239.0)*indgen(0:239);
func bone(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "bone.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_BONE_TABLE, CMAP_BONE_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_BONE_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}

CMAP_COPPER_TABLE = char(
 [[  0,  0,  0], [  1,  1,  1], [  3,  2,  1], [  4,  3,  2], [  5,  3,  2],
  [  7,  4,  3], [  8,  5,  3], [  9,  6,  4], [ 11,  7,  4], [ 12,  8,  5],
  [ 13,  8,  5], [ 15,  9,  6], [ 16, 10,  6], [ 17, 11,  7], [ 19, 12,  7],
  [ 20, 13,  8], [ 21, 13,  8], [ 23, 14,  9], [ 24, 15, 10], [ 25, 16, 10],
  [ 27, 17, 11], [ 28, 18, 11], [ 29, 18, 12], [ 31, 19, 12], [ 32, 20, 13],
  [ 33, 21, 13], [ 35, 22, 14], [ 36, 23, 14], [ 37, 23, 15], [ 39, 24, 15],
  [ 40, 25, 16], [ 41, 26, 16], [ 43, 27, 17], [ 44, 28, 18], [ 45, 28, 18],
  [ 47, 29, 19], [ 48, 30, 19], [ 49, 31, 20], [ 51, 32, 20], [ 52, 33, 21],
  [ 53, 33, 21], [ 55, 34, 22], [ 56, 35, 22], [ 57, 36, 23], [ 59, 37, 23],
  [ 60, 38, 24], [ 61, 38, 24], [ 63, 39, 25], [ 64, 40, 25], [ 65, 41, 26],
  [ 67, 42, 27], [ 68, 43, 27], [ 69, 43, 28], [ 71, 44, 28], [ 72, 45, 29],
  [ 73, 46, 29], [ 75, 47, 30], [ 76, 48, 30], [ 77, 48, 31], [ 79, 49, 31],
  [ 80, 50, 32], [ 81, 51, 32], [ 83, 52, 33], [ 84, 53, 33], [ 85, 53, 34],
  [ 87, 54, 35], [ 88, 55, 35], [ 89, 56, 36], [ 91, 57, 36], [ 92, 58, 37],
  [ 93, 58, 37], [ 95, 59, 38], [ 96, 60, 38], [ 97, 61, 39], [ 99, 62, 39],
  [100, 63, 40], [101, 63, 40], [103, 64, 41], [104, 65, 41], [105, 66, 42],
  [107, 67, 42], [108, 68, 43], [109, 68, 44], [111, 69, 44], [112, 70, 45],
  [113, 71, 45], [115, 72, 46], [116, 73, 46], [117, 73, 47], [119, 74, 47],
  [120, 75, 48], [121, 76, 48], [123, 77, 49], [124, 78, 49], [125, 78, 50],
  [127, 79, 50], [128, 80, 51], [129, 81, 51], [131, 82, 52], [132, 83, 53],
  [133, 83, 53], [135, 84, 54], [136, 85, 54], [137, 86, 55], [139, 87, 55],
  [140, 88, 56], [141, 88, 56], [143, 89, 57], [144, 90, 57], [145, 91, 58],
  [147, 92, 58], [148, 93, 59], [149, 93, 59], [151, 94, 60], [152, 95, 61],
  [153, 96, 61], [155, 97, 62], [156, 98, 62], [157, 98, 63], [159, 99, 63],
  [160,100, 64], [161,101, 64], [163,102, 65], [164,103, 65], [165,103, 66],
  [167,104, 66], [168,105, 67], [169,106, 67], [171,107, 68], [172,108, 68],
  [173,108, 69], [175,109, 70], [176,110, 70], [177,111, 71], [179,112, 71],
  [180,113, 72], [181,113, 72], [183,114, 73], [184,115, 73], [185,116, 74],
  [187,117, 74], [188,118, 75], [189,118, 75], [191,119, 76], [192,120, 76],
  [193,121, 77], [195,122, 77], [196,123, 78], [197,123, 79], [199,124, 79],
  [200,125, 80], [201,126, 80], [203,127, 81], [204,128, 81], [205,128, 82],
  [207,129, 82], [208,130, 83], [209,131, 83], [211,132, 84], [212,133, 84],
  [213,133, 85], [215,134, 85], [216,135, 86], [217,136, 87], [219,137, 87],
  [220,138, 88], [221,138, 88], [223,139, 89], [224,140, 89], [225,141, 90],
  [227,142, 90], [228,143, 91], [229,143, 91], [231,144, 92], [232,145, 92],
  [233,146, 93], [235,147, 93], [236,148, 94], [237,148, 94], [239,149, 95],
  [240,150, 96], [241,151, 96], [243,152, 97], [244,153, 97], [245,153, 98],
  [247,154, 98], [248,155, 99], [249,156, 99], [251,157,100], [252,158,100],
  [253,158,101], [254,159,101], [254,160,102], [255,161,102], [255,162,103],
  [255,163,104], [255,163,104], [255,164,105], [255,165,105], [255,166,106],
  [255,167,106], [255,168,107], [255,168,107], [255,169,108], [255,170,108],
  [255,171,109], [255,172,109], [255,173,110], [255,173,110], [255,174,111],
  [255,175,111], [255,176,112], [255,177,113], [255,178,113], [255,178,114],
  [255,179,114], [255,180,115], [255,181,115], [255,182,116], [255,183,116],
  [255,183,117], [255,184,117], [255,185,118], [255,186,118], [255,187,119],
  [255,188,119], [255,188,120], [255,189,120], [255,190,121], [255,191,122],
  [255,192,122], [255,193,123], [255,193,123], [255,194,124], [255,195,124],
  [255,196,125], [255,197,125], [255,198,126], [255,198,126], [255,199,127]]);
CMAP_COPPER_NUMBER = 240;
CMAP_COPPER_LEVEL = (1.0/239.0)*indgen(0:239);
func copper(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "copper.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_COPPER_TABLE, CMAP_COPPER_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_COPPER_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}

CMAP_EARTH_TABLE = char(
 [[  0,  0,  0], [  0,  0, 46], [  0,  0, 58], [  0,  0, 69], [  0,  0, 81],
  [  0,  0, 92], [  0,  0,104], [  0,  0,116], [  0,  3,116], [  1,  6,116],
  [  2,  8,116], [  2, 11,116], [  3, 13,117], [  4, 16,117], [  5, 18,117],
  [  5, 21,117], [  6, 23,117], [  7, 26,118], [  8, 28,118], [  8, 31,118],
  [  9, 33,118], [ 10, 36,118], [ 11, 38,119], [ 11, 41,119], [ 12, 43,119],
  [ 13, 45,119], [ 14, 48,119], [ 15, 50,120], [ 15, 52,120], [ 16, 55,120],
  [ 17, 57,120], [ 18, 59,120], [ 18, 61,121], [ 19, 64,121], [ 20, 66,121],
  [ 21, 68,121], [ 22, 70,121], [ 22, 72,122], [ 23, 74,122], [ 24, 77,122],
  [ 25, 79,122], [ 26, 81,122], [ 26, 83,123], [ 27, 85,123], [ 28, 87,123],
  [ 29, 89,123], [ 30, 91,123], [ 31, 93,124], [ 31, 95,124], [ 32, 97,124],
  [ 33, 99,124], [ 34,100,124], [ 35,102,125], [ 36,104,125], [ 36,106,125],
  [ 37,108,125], [ 38,109,125], [ 39,111,126], [ 40,113,126], [ 41,115,126],
  [ 41,116,126], [ 42,118,126], [ 43,120,127], [ 44,121,127], [ 45,123,127],
  [ 46,125,127], [ 47,126,127], [ 48,128,128], [ 48,128,126], [ 48,129,125],
  [ 49,129,124], [ 49,130,123], [ 50,131,122], [ 50,131,120], [ 51,132,119],
  [ 51,133,118], [ 52,133,117], [ 52,134,115], [ 53,134,114], [ 53,135,113],
  [ 54,136,111], [ 54,136,110], [ 55,137,109], [ 55,138,108], [ 56,138,106],
  [ 56,139,105], [ 57,140,104], [ 57,140,102], [ 58,141,101], [ 58,141,100],
  [ 59,142, 98], [ 59,143, 97], [ 60,143, 96], [ 61,144, 94], [ 61,145, 93],
  [ 62,145, 92], [ 62,146, 90], [ 63,146, 89], [ 63,147, 88], [ 64,148, 86],
  [ 64,148, 85], [ 65,149, 84], [ 65,150, 82], [ 66,150, 81], [ 67,151, 80],
  [ 67,151, 78], [ 68,152, 77], [ 68,153, 76], [ 69,153, 74], [ 69,154, 73],
  [ 70,155, 71], [ 71,155, 70], [ 73,156, 71], [ 76,156, 72], [ 78,157, 72],
  [ 81,158, 73], [ 83,158, 73], [ 86,159, 74], [ 88,160, 75], [ 91,160, 75],
  [ 94,161, 76], [ 96,161, 76], [ 99,162, 77], [101,163, 77], [104,163, 78],
  [106,164, 79], [109,165, 79], [111,165, 80], [114,166, 80], [117,166, 81],
  [119,167, 82], [121,168, 82], [122,168, 82], [124,168, 83], [126,169, 83],
  [128,169, 83], [129,170, 84], [131,170, 84], [133,171, 84], [135,171, 85],
  [136,172, 85], [138,172, 85], [140,172, 86], [141,173, 86], [143,173, 86],
  [145,174, 87], [147,174, 87], [149,175, 87], [150,175, 88], [152,175, 88],
  [154,176, 88], [156,176, 89], [157,177, 89], [159,177, 89], [161,178, 90],
  [163,178, 90], [165,179, 90], [166,179, 91], [168,179, 91], [170,180, 91],
  [172,180, 92], [174,181, 92], [175,181, 92], [177,182, 93], [179,182, 93],
  [181,183, 93], [183,183, 94], [183,182, 94], [184,181, 94], [184,181, 95],
  [185,180, 95], [185,179, 95], [186,178, 96], [186,177, 96], [187,176, 97],
  [187,175, 97], [187,174, 97], [188,173, 98], [188,172, 98], [189,171, 98],
  [189,170, 99], [190,169, 99], [190,168, 99], [190,167,100], [191,166,100],
  [191,165,100], [192,164,101], [192,163,101], [193,163,104], [195,164,106],
  [196,164,108], [197,165,111], [198,165,113], [199,166,116], [201,167,118],
  [202,167,121], [203,168,123], [204,169,126], [205,170,129], [207,171,131],
  [208,172,134], [209,173,137], [210,174,139], [211,175,142], [213,176,145],
  [214,177,148], [215,178,150], [216,179,153], [217,181,156], [219,182,159],
  [220,184,162], [221,185,165], [222,187,168], [223,188,170], [225,190,173],
  [226,192,176], [227,194,179], [228,196,182], [229,198,185], [231,200,189],
  [232,202,192], [233,204,195], [234,206,198], [235,208,201], [237,211,204],
  [238,213,207], [239,215,211], [240,218,214], [241,221,217], [243,223,220],
  [244,226,224], [245,229,227], [246,232,230], [247,235,234], [249,238,237],
  [250,241,241], [251,244,244], [252,248,248], [253,251,251], [255,255,255]]);
CMAP_EARTH_NUMBER = 240;
CMAP_EARTH_LEVEL = (1.0/239.0)*indgen(0:239);
func earth(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "earth.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_EARTH_TABLE, CMAP_EARTH_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_EARTH_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}

CMAP_GRAY_TABLE = char(
 [[  0,  0,  0], [  1,  1,  1], [  2,  2,  2], [  3,  3,  3], [  4,  4,  4],
  [  5,  5,  5], [  6,  6,  6], [  7,  7,  7], [  9,  9,  9], [ 10, 10, 10],
  [ 11, 11, 11], [ 12, 12, 12], [ 13, 13, 13], [ 14, 14, 14], [ 15, 15, 15],
  [ 16, 16, 16], [ 17, 17, 17], [ 18, 18, 18], [ 19, 19, 19], [ 20, 20, 20],
  [ 21, 21, 21], [ 22, 22, 22], [ 23, 23, 23], [ 25, 25, 25], [ 26, 26, 26],
  [ 27, 27, 27], [ 28, 28, 28], [ 29, 29, 29], [ 30, 30, 30], [ 31, 31, 31],
  [ 32, 32, 32], [ 33, 33, 33], [ 34, 34, 34], [ 35, 35, 35], [ 36, 36, 36],
  [ 37, 37, 37], [ 38, 38, 38], [ 39, 39, 39], [ 41, 41, 41], [ 42, 42, 42],
  [ 43, 43, 43], [ 44, 44, 44], [ 45, 45, 45], [ 46, 46, 46], [ 47, 47, 47],
  [ 48, 48, 48], [ 49, 49, 49], [ 50, 50, 50], [ 51, 51, 51], [ 52, 52, 52],
  [ 53, 53, 53], [ 54, 54, 54], [ 55, 55, 55], [ 57, 57, 57], [ 58, 58, 58],
  [ 59, 59, 59], [ 60, 60, 60], [ 61, 61, 61], [ 62, 62, 62], [ 63, 63, 63],
  [ 64, 64, 64], [ 65, 65, 65], [ 66, 66, 66], [ 67, 67, 67], [ 68, 68, 68],
  [ 69, 69, 69], [ 70, 70, 70], [ 71, 71, 71], [ 73, 73, 73], [ 74, 74, 74],
  [ 75, 75, 75], [ 76, 76, 76], [ 77, 77, 77], [ 78, 78, 78], [ 79, 79, 79],
  [ 80, 80, 80], [ 81, 81, 81], [ 82, 82, 82], [ 83, 83, 83], [ 84, 84, 84],
  [ 85, 85, 85], [ 86, 86, 86], [ 87, 87, 87], [ 89, 89, 89], [ 90, 90, 90],
  [ 91, 91, 91], [ 92, 92, 92], [ 93, 93, 93], [ 94, 94, 94], [ 95, 95, 95],
  [ 96, 96, 96], [ 97, 97, 97], [ 98, 98, 98], [ 99, 99, 99], [100,100,100],
  [101,101,101], [102,102,102], [103,103,103], [105,105,105], [106,106,106],
  [107,107,107], [108,108,108], [109,109,109], [110,110,110], [111,111,111],
  [112,112,112], [113,113,113], [114,114,114], [115,115,115], [116,116,116],
  [117,117,117], [118,118,118], [119,119,119], [121,121,121], [122,122,122],
  [123,123,123], [124,124,124], [125,125,125], [126,126,126], [127,127,127],
  [128,128,128], [129,129,129], [130,130,130], [131,131,131], [132,132,132],
  [133,133,133], [134,134,134], [135,135,135], [137,137,137], [138,138,138],
  [139,139,139], [140,140,140], [141,141,141], [142,142,142], [143,143,143],
  [144,144,144], [145,145,145], [146,146,146], [147,147,147], [148,148,148],
  [149,149,149], [150,150,150], [151,151,151], [153,153,153], [154,154,154],
  [155,155,155], [156,156,156], [157,157,157], [158,158,158], [159,159,159],
  [160,160,160], [161,161,161], [162,162,162], [163,163,163], [164,164,164],
  [165,165,165], [166,166,166], [167,167,167], [169,169,169], [170,170,170],
  [171,171,171], [172,172,172], [173,173,173], [174,174,174], [175,175,175],
  [176,176,176], [177,177,177], [178,178,178], [179,179,179], [180,180,180],
  [181,181,181], [182,182,182], [183,183,183], [185,185,185], [186,186,186],
  [187,187,187], [188,188,188], [189,189,189], [190,190,190], [191,191,191],
  [192,192,192], [193,193,193], [194,194,194], [195,195,195], [196,196,196],
  [197,197,197], [198,198,198], [199,199,199], [201,201,201], [202,202,202],
  [203,203,203], [204,204,204], [205,205,205], [206,206,206], [207,207,207],
  [208,208,208], [209,209,209], [210,210,210], [211,211,211], [212,212,212],
  [213,213,213], [214,214,214], [215,215,215], [217,217,217], [218,218,218],
  [219,219,219], [220,220,220], [221,221,221], [222,222,222], [223,223,223],
  [224,224,224], [225,225,225], [226,226,226], [227,227,227], [228,228,228],
  [229,229,229], [230,230,230], [231,231,231], [233,233,233], [234,234,234],
  [235,235,235], [236,236,236], [237,237,237], [238,238,238], [239,239,239],
  [240,240,240], [241,241,241], [242,242,242], [243,243,243], [244,244,244],
  [245,245,245], [246,246,246], [247,247,247], [249,249,249], [250,250,250],
  [251,251,251], [252,252,252], [253,253,253], [254,254,254], [255,255,255]]);
CMAP_GRAY_NUMBER = 240;
CMAP_GRAY_LEVEL = (1.0/239.0)*indgen(0:239);
func gray(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "gray.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_GRAY_TABLE, CMAP_GRAY_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_GRAY_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}

CMAP_HEAT_TABLE = char(
 [[  0,  0,  0], [  1,  0,  0], [  2,  0,  0], [  4,  0,  0], [  5,  0,  0],
  [  7,  0,  0], [  8,  0,  0], [ 10,  0,  0], [ 11,  0,  0], [ 13,  0,  0],
  [ 15,  0,  0], [ 17,  0,  0], [ 18,  0,  0], [ 20,  0,  0], [ 21,  0,  0],
  [ 23,  0,  0], [ 24,  0,  0], [ 26,  0,  0], [ 27,  0,  0], [ 28,  0,  0],
  [ 30,  0,  0], [ 31,  0,  0], [ 33,  0,  0], [ 34,  0,  0], [ 36,  0,  0],
  [ 37,  0,  0], [ 39,  0,  0], [ 40,  0,  0], [ 42,  0,  0], [ 43,  0,  0],
  [ 46,  0,  0], [ 47,  0,  0], [ 49,  0,  0], [ 50,  0,  0], [ 52,  0,  0],
  [ 53,  0,  0], [ 55,  0,  0], [ 56,  0,  0], [ 57,  0,  0], [ 59,  0,  0],
  [ 60,  0,  0], [ 62,  0,  0], [ 63,  0,  0], [ 65,  0,  0], [ 66,  0,  0],
  [ 68,  0,  0], [ 69,  0,  0], [ 70,  0,  0], [ 72,  0,  0], [ 73,  0,  0],
  [ 76,  0,  0], [ 78,  0,  0], [ 79,  0,  0], [ 81,  0,  0], [ 82,  0,  0],
  [ 84,  0,  0], [ 85,  0,  0], [ 86,  0,  0], [ 88,  0,  0], [ 89,  0,  0],
  [ 92,  0,  0], [ 94,  0,  0], [ 95,  0,  0], [ 97,  0,  0], [ 98,  0,  0],
  [ 99,  0,  0], [101,  0,  0], [102,  0,  0], [104,  0,  0], [105,  0,  0],
  [108,  0,  0], [110,  0,  0], [111,  0,  0], [113,  0,  0], [114,  0,  0],
  [115,  0,  0], [117,  0,  0], [118,  0,  0], [120,  0,  0], [121,  0,  0],
  [123,  0,  0], [124,  0,  0], [126,  0,  0], [127,  0,  0], [128,  0,  0],
  [130,  0,  0], [131,  0,  0], [133,  0,  0], [134,  0,  0], [136,  0,  0],
  [139,  0,  0], [140,  0,  0], [141,  0,  0], [143,  0,  0], [144,  0,  0],
  [146,  0,  0], [147,  0,  0], [149,  0,  0], [150,  0,  0], [152,  0,  0],
  [153,  0,  0], [155,  0,  0], [156,  0,  0], [157,  0,  0], [159,  0,  0],
  [160,  0,  0], [162,  0,  0], [163,  0,  0], [165,  0,  0], [166,  0,  0],
  [169,  0,  0], [170,  0,  0], [172,  0,  0], [173,  0,  0], [175,  1,  0],
  [176,  3,  0], [178,  5,  0], [179,  7,  0], [181,  9,  0], [182, 11,  0],
  [185, 15,  0], [186, 17,  0], [188, 18,  0], [189, 20,  0], [191, 22,  0],
  [192, 24,  0], [194, 26,  0], [195, 28,  0], [197, 30,  0], [198, 32,  0],
  [201, 35,  0], [202, 37,  0], [204, 39,  0], [205, 41,  0], [207, 43,  0],
  [208, 45,  0], [210, 47,  0], [211, 49,  0], [212, 51,  0], [214, 52,  0],
  [215, 54,  0], [217, 56,  0], [218, 58,  0], [220, 60,  0], [221, 62,  0],
  [223, 64,  0], [224, 66,  0], [226, 68,  0], [227, 69,  0], [228, 71,  0],
  [231, 75,  0], [233, 77,  0], [234, 79,  0], [236, 81,  0], [237, 83,  0],
  [239, 85,  0], [240, 86,  0], [241, 88,  0], [243, 90,  0], [244, 92,  0],
  [246, 94,  0], [247, 96,  0], [249, 98,  0], [250,100,  0], [252,102,  0],
  [253,103,  0], [255,105,  0], [255,107,  0], [255,109,  0], [255,111,  0],
  [255,115,  0], [255,117,  0], [255,119,  0], [255,120,  0], [255,122,  0],
  [255,124,  0], [255,126,  0], [255,128,  0], [255,130,  0], [255,132,  0],
  [255,136,  7], [255,137, 11], [255,139, 15], [255,141, 19], [255,143, 23],
  [255,145, 27], [255,147, 31], [255,149, 35], [255,151, 39], [255,153, 43],
  [255,156, 51], [255,158, 54], [255,160, 58], [255,162, 62], [255,164, 66],
  [255,166, 70], [255,168, 74], [255,170, 78], [255,171, 82], [255,173, 86],
  [255,175, 90], [255,177, 94], [255,179, 98], [255,181,102], [255,183,105],
  [255,185,109], [255,187,113], [255,188,117], [255,190,121], [255,192,125],
  [255,196,133], [255,198,137], [255,200,141], [255,202,145], [255,204,149],
  [255,205,153], [255,207,156], [255,209,160], [255,211,164], [255,213,168],
  [255,215,172], [255,217,176], [255,219,180], [255,221,184], [255,222,188],
  [255,224,192], [255,226,196], [255,228,200], [255,230,204], [255,232,207],
  [255,236,215], [255,238,219], [255,239,223], [255,241,227], [255,243,231],
  [255,245,235], [255,247,239], [255,249,243], [255,251,247], [255,253,251]]);
CMAP_HEAT_NUMBER = 240;
CMAP_HEAT_LEVEL = (1.0/239.0)*indgen(0:239);
func heat(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "heat.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_HEAT_TABLE, CMAP_HEAT_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_HEAT_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}

CMAP_HSV_TABLE = char(
 [[255,  0,  0], [255,  6,  0], [255, 13,  0], [255, 19,  0], [255, 25,  0],
  [255, 32,  0], [255, 38,  0], [255, 44,  0], [255, 50,  0], [255, 57,  0],
  [255, 63,  0], [255, 69,  0], [255, 76,  0], [255, 82,  0], [255, 88,  0],
  [255, 95,  0], [255,101,  0], [255,107,  0], [255,113,  0], [255,120,  0],
  [255,126,  0], [255,132,  0], [255,139,  0], [255,145,  0], [255,151,  0],
  [255,158,  0], [255,164,  0], [255,170,  0], [255,176,  0], [255,183,  0],
  [255,189,  0], [255,195,  0], [255,202,  0], [255,208,  0], [255,214,  0],
  [255,221,  0], [255,227,  0], [255,233,  0], [255,239,  0], [253,244,  0],
  [251,248,  0], [249,252,  0], [245,255,  0], [239,255,  0], [233,255,  0],
  [226,255,  0], [220,255,  0], [214,255,  0], [208,255,  0], [201,255,  0],
  [195,255,  0], [189,255,  0], [182,255,  0], [176,255,  0], [170,255,  0],
  [163,255,  0], [157,255,  0], [151,255,  0], [145,255,  0], [138,255,  0],
  [132,255,  0], [126,255,  0], [119,255,  0], [113,255,  0], [107,255,  0],
  [100,255,  0], [ 94,255,  0], [ 88,255,  0], [ 81,255,  0], [ 75,255,  0],
  [ 69,255,  0], [ 63,255,  0], [ 56,255,  0], [ 50,255,  0], [ 44,255,  0],
  [ 37,255,  0], [ 31,255,  0], [ 25,255,  0], [ 18,255,  0], [ 12,255,  0],
  [  7,255,  1], [  5,255,  6], [  3,255, 10], [  1,255, 14], [  0,255, 19],
  [  0,255, 26], [  0,255, 32], [  0,255, 38], [  0,255, 45], [  0,255, 51],
  [  0,255, 57], [  0,255, 63], [  0,255, 70], [  0,255, 76], [  0,255, 82],
  [  0,255, 89], [  0,255, 95], [  0,255,101], [  0,255,108], [  0,255,114],
  [  0,255,120], [  0,255,126], [  0,255,133], [  0,255,139], [  0,255,145],
  [  0,255,152], [  0,255,158], [  0,255,164], [  0,255,171], [  0,255,177],
  [  0,255,183], [  0,255,189], [  0,255,196], [  0,255,202], [  0,255,208],
  [  0,255,215], [  0,255,221], [  0,255,227], [  0,255,234], [  0,255,240],
  [  0,255,246], [  0,255,252], [  0,251,255], [  0,245,255], [  0,239,255],
  [  0,232,255], [  0,226,255], [  0,220,255], [  0,213,255], [  0,207,255],
  [  0,201,255], [  0,194,255], [  0,188,255], [  0,182,255], [  0,176,255],
  [  0,169,255], [  0,163,255], [  0,157,255], [  0,150,255], [  0,144,255],
  [  0,138,255], [  0,131,255], [  0,125,255], [  0,119,255], [  0,113,255],
  [  0,106,255], [  0,100,255], [  0, 94,255], [  0, 87,255], [  0, 81,255],
  [  0, 75,255], [  0, 68,255], [  0, 62,255], [  0, 56,255], [  0, 50,255],
  [  0, 43,255], [  0, 37,255], [  0, 31,255], [  0, 24,255], [  0, 18,255],
  [  1, 13,255], [  4,  9,255], [  6,  5,255], [  8,  1,255], [ 13,  0,255],
  [ 20,  0,255], [ 26,  0,255], [ 32,  0,255], [ 39,  0,255], [ 45,  0,255],
  [ 51,  0,255], [ 58,  0,255], [ 64,  0,255], [ 70,  0,255], [ 76,  0,255],
  [ 83,  0,255], [ 89,  0,255], [ 95,  0,255], [102,  0,255], [108,  0,255],
  [114,  0,255], [121,  0,255], [127,  0,255], [133,  0,255], [140,  0,255],
  [146,  0,255], [152,  0,255], [158,  0,255], [165,  0,255], [171,  0,255],
  [177,  0,255], [184,  0,255], [190,  0,255], [196,  0,255], [203,  0,255],
  [209,  0,255], [215,  0,255], [221,  0,255], [228,  0,255], [234,  0,255],
  [240,  0,255], [247,  0,255], [249,  0,251], [251,  0,247], [253,  0,243],
  [255,  0,238], [255,  0,232], [255,  0,226], [255,  0,219], [255,  0,213],
  [255,  0,207], [255,  0,200], [255,  0,194], [255,  0,188], [255,  0,181],
  [255,  0,175], [255,  0,169], [255,  0,163], [255,  0,156], [255,  0,150],
  [255,  0,144], [255,  0,137], [255,  0,131], [255,  0,125], [255,  0,118],
  [255,  0,112], [255,  0,106], [255,  0,100], [255,  0, 93], [255,  0, 87],
  [255,  0, 81], [255,  0, 74], [255,  0, 68], [255,  0, 62], [255,  0, 55],
  [255,  0, 49], [255,  0, 43], [255,  0, 37], [255,  0, 30], [255,  0, 24]]);
CMAP_HSV_NUMBER = 240;
CMAP_HSV_LEVEL = (1.0/239.0)*indgen(0:239);
func hsv(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "hsv.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_HSV_TABLE, CMAP_HSV_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_HSV_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}

CMAP_JET_TABLE = char(
 [[  0,  0,143], [  0,  0,148], [  0,  0,152], [  0,  0,156], [  0,  0,160],
  [  0,  0,164], [  0,  0,169], [  0,  0,173], [  0,  0,177], [  0,  0,181],
  [  0,  0,185], [  0,  0,190], [  0,  0,194], [  0,  0,198], [  0,  0,202],
  [  0,  0,206], [  0,  0,211], [  0,  0,215], [  0,  0,219], [  0,  0,223],
  [  0,  0,227], [  0,  0,232], [  0,  0,236], [  0,  0,240], [  0,  0,244],
  [  0,  0,248], [  0,  0,253], [  0,  2,255], [  0,  6,255], [  0, 10,255],
  [  0, 14,255], [  0, 19,255], [  0, 23,255], [  0, 27,255], [  0, 31,255],
  [  0, 35,255], [  0, 40,255], [  0, 44,255], [  0, 48,255], [  0, 52,255],
  [  0, 56,255], [  0, 61,255], [  0, 65,255], [  0, 69,255], [  0, 73,255],
  [  0, 77,255], [  0, 82,255], [  0, 86,255], [  0, 90,255], [  0, 94,255],
  [  0, 98,255], [  0,103,255], [  0,107,255], [  0,111,255], [  0,115,255],
  [  0,119,255], [  0,124,255], [  0,128,255], [  0,132,255], [  0,136,255],
  [  0,141,255], [  0,145,255], [  0,149,255], [  0,153,255], [  0,157,255],
  [  0,162,255], [  0,166,255], [  0,170,255], [  0,174,255], [  0,178,255],
  [  0,183,255], [  0,187,255], [  0,191,255], [  0,195,255], [  0,199,255],
  [  0,204,255], [  0,208,255], [  0,212,255], [  0,216,255], [  0,220,255],
  [  0,225,255], [  0,229,255], [  0,233,255], [  0,237,255], [  0,241,255],
  [  0,246,255], [  0,250,255], [  0,254,255], [  3,255,255], [  7,255,255],
  [ 12,255,255], [ 16,255,255], [ 20,255,251], [ 24,255,247], [ 28,255,243],
  [ 33,255,238], [ 37,255,234], [ 41,255,230], [ 45,255,226], [ 49,255,222],
  [ 54,255,217], [ 58,255,213], [ 62,255,209], [ 66,255,205], [ 70,255,201],
  [ 75,255,196], [ 79,255,192], [ 83,255,188], [ 87,255,184], [ 91,255,180],
  [ 96,255,175], [100,255,171], [104,255,167], [108,255,163], [112,255,159],
  [117,255,154], [121,255,150], [125,255,146], [129,255,142], [133,255,138],
  [138,255,133], [142,255,129], [146,255,125], [150,255,121], [154,255,117],
  [159,255,112], [163,255,108], [167,255,104], [171,255,100], [175,255, 96],
  [180,255, 91], [184,255, 87], [188,255, 83], [192,255, 79], [196,255, 75],
  [201,255, 70], [205,255, 66], [209,255, 62], [213,255, 58], [217,255, 54],
  [222,255, 49], [226,255, 45], [230,255, 41], [234,255, 37], [238,255, 33],
  [243,255, 28], [247,255, 24], [251,255, 20], [255,255, 16], [255,255, 12],
  [255,255,  7], [255,255,  3], [255,254,  0], [255,250,  0], [255,246,  0],
  [255,241,  0], [255,237,  0], [255,233,  0], [255,229,  0], [255,225,  0],
  [255,220,  0], [255,216,  0], [255,212,  0], [255,208,  0], [255,204,  0],
  [255,199,  0], [255,195,  0], [255,191,  0], [255,187,  0], [255,183,  0],
  [255,178,  0], [255,174,  0], [255,170,  0], [255,166,  0], [255,162,  0],
  [255,157,  0], [255,153,  0], [255,149,  0], [255,145,  0], [255,141,  0],
  [255,136,  0], [255,132,  0], [255,128,  0], [255,124,  0], [255,119,  0],
  [255,115,  0], [255,111,  0], [255,107,  0], [255,103,  0], [255, 98,  0],
  [255, 94,  0], [255, 90,  0], [255, 86,  0], [255, 82,  0], [255, 77,  0],
  [255, 73,  0], [255, 69,  0], [255, 65,  0], [255, 61,  0], [255, 56,  0],
  [255, 52,  0], [255, 48,  0], [255, 44,  0], [255, 40,  0], [255, 35,  0],
  [255, 31,  0], [255, 27,  0], [255, 23,  0], [255, 19,  0], [255, 14,  0],
  [255, 10,  0], [255,  6,  0], [255,  2,  0], [253,  0,  0], [248,  0,  0],
  [244,  0,  0], [240,  0,  0], [236,  0,  0], [232,  0,  0], [227,  0,  0],
  [223,  0,  0], [219,  0,  0], [215,  0,  0], [211,  0,  0], [206,  0,  0],
  [202,  0,  0], [198,  0,  0], [194,  0,  0], [190,  0,  0], [185,  0,  0],
  [181,  0,  0], [177,  0,  0], [173,  0,  0], [169,  0,  0], [164,  0,  0],
  [160,  0,  0], [156,  0,  0], [152,  0,  0], [148,  0,  0], [143,  0,  0]]);
CMAP_JET_NUMBER = 240;
CMAP_JET_LEVEL = (1.0/239.0)*indgen(0:239);
func jet(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "jet.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_JET_TABLE, CMAP_JET_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_JET_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}

CMAP_NCAR_TABLE = char(
 [[  0,  0,128], [  0,  9,115], [  0, 19,103], [  0, 28, 91], [  0, 38, 79],
  [  0, 47, 66], [  0, 57, 54], [  0, 66, 42], [  0, 76, 30], [  0, 85, 18],
  [  0, 95,  5], [  0, 94, 12], [  0, 85, 37], [  0, 75, 61], [  0, 66, 85],
  [  0, 56,110], [  0, 47,134], [  0, 37,158], [  0, 28,183], [  0, 18,207],
  [  0,  9,231], [  0,  0,255], [  0, 19,255], [  0, 37,255], [  0, 55,255],
  [  0, 73,255], [  0, 92,255], [  0,110,255], [  0,128,255], [  0,146,255],
  [  0,165,255], [  0,183,255], [  0,194,255], [  0,200,255], [  0,206,255],
  [  0,212,255], [  0,218,255], [  0,225,255], [  0,231,255], [  0,237,255],
  [  0,243,255], [  0,249,255], [  0,254,253], [  0,254,244], [  0,253,234],
  [  0,253,225], [  0,253,215], [  0,252,205], [  0,252,196], [  0,251,186],
  [  0,251,176], [  0,250,167], [  0,250,157], [  0,250,144], [  0,250,130],
  [  0,251,115], [  0,251,100], [  0,252, 85], [  0,252, 71], [  0,253, 56],
  [  0,253, 41], [  0,254, 27], [  0,254, 12], [  1,254,  0], [ 11,249,  0],
  [ 21,244,  0], [ 30,239,  0], [ 40,235,  0], [ 50,230,  0], [ 59,225,  0],
  [ 69,220,  0], [ 79,216,  0], [ 89,211,  0], [ 98,206,  0], [103,208,  0],
  [106,213,  0], [108,217,  0], [110,222,  0], [113,227,  0], [115,232,  0],
  [117,236,  0], [120,241,  0], [122,246,  0], [125,251,  0], [128,255,  1],
  [134,255,  7], [140,255, 13], [146,255, 19], [153,255, 24], [159,255, 30],
  [165,255, 36], [171,255, 42], [177,255, 48], [184,255, 54], [190,255, 60],
  [196,255, 57], [202,255, 51], [208,255, 45], [214,255, 39], [220,255, 33],
  [226,255, 28], [232,255, 22], [238,255, 16], [244,255, 10], [250,255,  4],
  [255,253,  0], [255,250,  0], [255,246,  0], [255,242,  0], [255,238,  0],
  [255,234,  0], [255,231,  0], [255,227,  0], [255,223,  0], [255,219,  0],
  [255,215,  0], [255,212,  1], [255,209,  2], [255,207,  3], [255,204,  5],
  [255,201,  6], [255,198,  8], [255,195,  9], [255,192, 11], [255,189, 12],
  [255,186, 14], [255,181, 14], [255,170, 13], [255,159, 11], [255,148, 10],
  [255,137,  8], [255,126,  7], [255,115,  5], [255,103,  4], [255, 92,  3],
  [255, 81,  1], [255, 70,  0], [255, 63,  0], [255, 56,  0], [255, 50,  0],
  [255, 43,  0], [255, 37,  0], [255, 30,  0], [255, 23,  0], [255, 17,  0],
  [255, 10,  0], [255,  4,  0], [255,  0,  8], [255,  0, 33], [255,  0, 57],
  [255,  0, 82], [255,  0,106], [255,  0,130], [255,  0,155], [255,  0,179],
  [255,  0,203], [255,  0,228], [255,  0,252], [246,  4,255], [236,  8,255],
  [227, 13,255], [217, 17,255], [208, 22,255], [198, 27,255], [189, 31,255],
  [179, 36,255], [170, 40,255], [160, 45,255], [158, 51,254], [166, 59,252],
  [174, 66,251], [182, 74,249], [190, 82,247], [197, 90,246], [205, 98,244],
  [213,106,242], [221,113,241], [229,121,239], [237,129,238], [238,135,238],
  [239,141,239], [239,146,239], [240,152,240], [241,158,241], [241,164,241],
  [242,169,242], [243,175,243], [243,181,243], [244,186,244], [245,192,245],
  [246,199,246], [247,205,247], [248,211,248], [249,217,249], [250,223,250],
  [251,230,251], [252,236,252], [253,242,253], [254,248,254], [255,255,255]]);
CMAP_NCAR_NUMBER = 200;
CMAP_NCAR_LEVEL = (1.0/199.0)*indgen(0:199);
func ncar(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "ncar.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_NCAR_TABLE, CMAP_NCAR_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_NCAR_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}

CMAP_RAINBOW_TABLE = char(
 [[255,  0, 42], [255,  0, 36], [255,  0, 31], [255,  0, 26], [255,  0, 20],
  [255,  0, 15], [255,  0, 10], [255,  0,  4], [255,  5,  0], [255, 11,  0],
  [255, 16,  0], [255, 22,  0], [255, 27,  0], [255, 32,  0], [255, 38,  0],
  [255, 43,  0], [255, 48,  0], [255, 54,  0], [255, 59,  0], [255, 65,  0],
  [255, 70,  0], [255, 75,  0], [255, 81,  0], [255, 91,  0], [255, 97,  0],
  [255,102,  0], [255,108,  0], [255,113,  0], [255,118,  0], [255,124,  0],
  [255,129,  0], [255,135,  0], [255,140,  0], [255,145,  0], [255,151,  0],
  [255,156,  0], [255,161,  0], [255,167,  0], [255,178,  0], [255,183,  0],
  [255,188,  0], [255,194,  0], [255,199,  0], [255,204,  0], [255,210,  0],
  [255,215,  0], [255,221,  0], [255,226,  0], [255,231,  0], [255,237,  0],
  [255,242,  0], [255,247,  0], [255,253,  0], [245,255,  0], [240,255,  0],
  [235,255,  0], [229,255,  0], [224,255,  0], [219,255,  0], [213,255,  0],
  [208,255,  0], [202,255,  0], [197,255,  0], [192,255,  0], [186,255,  0],
  [181,255,  0], [175,255,  0], [170,255,  0], [159,255,  0], [154,255,  0],
  [149,255,  0], [143,255,  0], [138,255,  0], [132,255,  0], [127,255,  0],
  [122,255,  0], [116,255,  0], [111,255,  0], [106,255,  0], [100,255,  0],
  [ 95,255,  0], [ 89,255,  0], [ 84,255,  0], [ 73,255,  0], [ 68,255,  0],
  [ 63,255,  0], [ 57,255,  0], [ 52,255,  0], [ 46,255,  0], [ 41,255,  0],
  [ 36,255,  0], [ 30,255,  0], [ 25,255,  0], [ 19,255,  0], [ 14,255,  0],
  [  9,255,  0], [  3,255,  0], [  0,255,  1], [  0,255, 12], [  0,255, 17],
  [  0,255, 23], [  0,255, 28], [  0,255, 33], [  0,255, 39], [  0,255, 44],
  [  0,255, 49], [  0,255, 55], [  0,255, 60], [  0,255, 66], [  0,255, 71],
  [  0,255, 76], [  0,255, 82], [  0,255, 87], [  0,255, 98], [  0,255,103],
  [  0,255,109], [  0,255,114], [  0,255,119], [  0,255,125], [  0,255,130],
  [  0,255,135], [  0,255,141], [  0,255,146], [  0,255,152], [  0,255,157],
  [  0,255,162], [  0,255,168], [  0,255,173], [  0,255,184], [  0,255,189],
  [  0,255,195], [  0,255,200], [  0,255,205], [  0,255,211], [  0,255,216],
  [  0,255,222], [  0,255,227], [  0,255,232], [  0,255,238], [  0,255,243],
  [  0,255,248], [  0,255,254], [  0,250,255], [  0,239,255], [  0,234,255],
  [  0,228,255], [  0,223,255], [  0,218,255], [  0,212,255], [  0,207,255],
  [  0,201,255], [  0,196,255], [  0,191,255], [  0,185,255], [  0,180,255],
  [  0,174,255], [  0,169,255], [  0,164,255], [  0,153,255], [  0,148,255],
  [  0,142,255], [  0,137,255], [  0,131,255], [  0,126,255], [  0,121,255],
  [  0,115,255], [  0,110,255], [  0,105,255], [  0, 99,255], [  0, 94,255],
  [  0, 88,255], [  0, 83,255], [  0, 78,255], [  0, 67,255], [  0, 62,255],
  [  0, 56,255], [  0, 51,255], [  0, 45,255], [  0, 40,255], [  0, 35,255],
  [  0, 29,255], [  0, 24,255], [  0, 18,255], [  0, 13,255], [  0,  8,255],
  [  0,  2,255], [  2,  0,255], [  7,  0,255], [ 18,  0,255], [ 24,  0,255],
  [ 29,  0,255], [ 34,  0,255], [ 40,  0,255], [ 45,  0,255], [ 50,  0,255],
  [ 56,  0,255], [ 61,  0,255], [ 67,  0,255], [ 72,  0,255], [ 77,  0,255],
  [ 83,  0,255], [ 88,  0,255], [ 93,  0,255], [104,  0,255], [110,  0,255],
  [115,  0,255], [120,  0,255], [126,  0,255], [131,  0,255], [136,  0,255],
  [142,  0,255], [147,  0,255], [153,  0,255], [158,  0,255], [163,  0,255],
  [169,  0,255], [174,  0,255], [180,  0,255], [190,  0,255], [196,  0,255],
  [201,  0,255], [206,  0,255], [212,  0,255], [217,  0,255], [223,  0,255],
  [228,  0,255], [233,  0,255], [239,  0,255], [244,  0,255], [249,  0,255],
  [255,  0,254], [255,  0,249], [255,  0,243], [255,  0,233], [255,  0,227],
  [255,  0,222], [255,  0,217], [255,  0,211], [255,  0,206], [255,  0,201]]);
CMAP_RAINBOW_NUMBER = 240;
CMAP_RAINBOW_LEVEL = (1.0/239.0)*indgen(0:239);
func rainbow(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "rainbow.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_RAINBOW_TABLE, CMAP_RAINBOW_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_RAINBOW_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}

CMAP_STERN_TABLE = char(
 [[  0,  0,  0], [ 18,  1,  1], [ 36,  2,  3], [ 54,  3,  5], [ 72,  4,  7],
  [ 90,  5,  9], [108,  6, 11], [127,  7, 13], [145,  8, 15], [163,  9, 17],
  [199, 11, 21], [217, 12, 23], [235, 13, 25], [254, 14, 27], [249, 15, 29],
  [244, 16, 31], [239, 17, 33], [234, 18, 35], [229, 19, 37], [223, 20, 39],
  [218, 21, 41], [213, 22, 43], [208, 23, 45], [203, 24, 47], [197, 25, 49],
  [192, 26, 51], [187, 27, 53], [182, 28, 55], [177, 29, 57], [172, 30, 59],
  [161, 32, 63], [156, 33, 65], [151, 34, 67], [146, 35, 69], [140, 36, 71],
  [135, 37, 73], [130, 38, 75], [125, 39, 77], [120, 40, 79], [115, 41, 81],
  [109, 42, 83], [104, 43, 85], [ 99, 44, 87], [ 94, 45, 89], [ 89, 46, 91],
  [ 83, 47, 93], [ 78, 48, 95], [ 73, 49, 97], [ 68, 50, 99], [ 63, 51,101],
  [ 52, 53,105], [ 47, 54,107], [ 42, 55,109], [ 37, 56,111], [ 32, 57,113],
  [ 26, 58,115], [ 21, 59,117], [ 16, 60,119], [ 11, 61,121], [  6, 62,123],
  [ 64, 64,127], [ 65, 65,129], [ 66, 66,131], [ 67, 67,133], [ 68, 68,135],
  [ 69, 69,137], [ 70, 70,139], [ 71, 71,141], [ 72, 72,143], [ 73, 73,145],
  [ 75, 75,149], [ 76, 76,151], [ 77, 77,153], [ 78, 78,155], [ 79, 79,157],
  [ 80, 80,159], [ 81, 81,161], [ 82, 82,163], [ 83, 83,165], [ 84, 84,167],
  [ 85, 85,169], [ 86, 86,171], [ 87, 87,173], [ 88, 88,175], [ 89, 89,177],
  [ 90, 90,179], [ 91, 91,181], [ 92, 92,183], [ 93, 93,185], [ 94, 94,187],
  [ 96, 96,191], [ 97, 97,193], [ 98, 98,195], [ 99, 99,197], [100,100,199],
  [101,101,201], [102,102,203], [103,103,205], [104,104,207], [105,105,209],
  [106,106,211], [107,107,213], [108,108,215], [109,109,217], [110,110,219],
  [111,111,221], [112,112,223], [113,113,225], [114,114,227], [115,115,229],
  [117,117,233], [118,118,235], [119,119,237], [120,120,239], [121,121,241],
  [122,122,243], [123,123,245], [124,124,247], [125,125,249], [126,126,251],
  [128,128,255], [129,129,251], [130,130,247], [131,131,243], [132,132,238],
  [133,133,234], [134,134,230], [135,135,226], [136,136,221], [137,137,217],
  [139,139,209], [140,140,204], [141,141,200], [142,142,196], [143,143,192],
  [144,144,187], [145,145,183], [146,146,179], [147,147,175], [148,148,170],
  [149,149,166], [150,150,162], [151,151,158], [152,152,153], [153,153,149],
  [154,154,145], [155,155,141], [156,156,136], [157,157,132], [158,158,128],
  [160,160,119], [161,161,115], [162,162,111], [163,163,107], [164,164,102],
  [165,165, 98], [166,166, 94], [167,167, 90], [168,168, 85], [169,169, 81],
  [170,170, 77], [171,171, 73], [172,172, 68], [173,173, 64], [174,174, 60],
  [175,175, 56], [176,176, 51], [177,177, 47], [178,178, 43], [179,179, 39],
  [181,181, 30], [182,182, 26], [183,183, 22], [184,184, 17], [185,185, 13],
  [186,186,  9], [187,187,  5], [188,188,  0], [189,189,  3], [190,190,  7],
  [192,192, 15], [193,193, 19], [194,194, 22], [195,195, 26], [196,196, 30],
  [197,197, 34], [198,198, 38], [199,199, 41], [200,200, 45], [201,201, 49],
  [203,203, 57], [204,204, 60], [205,205, 64], [206,206, 68], [207,207, 72],
  [208,208, 76], [209,209, 79], [210,210, 83], [211,211, 87], [212,212, 91],
  [213,213, 95], [214,214, 98], [215,215,102], [216,216,106], [217,217,110],
  [218,218,114], [219,219,117], [220,220,121], [221,221,125], [222,222,129],
  [224,224,137], [225,225,140], [226,226,144], [227,227,148], [228,228,152],
  [229,229,156], [230,230,159], [231,231,163], [232,232,167], [233,233,171],
  [234,234,175], [235,235,178], [236,236,182], [237,237,186], [238,238,190],
  [239,239,194], [240,240,197], [241,241,201], [242,242,205], [243,243,209],
  [245,245,216], [246,246,220], [247,247,224], [248,248,228], [249,249,232],
  [250,250,235], [251,251,239], [252,252,243], [253,253,247], [254,254,251]]);
CMAP_STERN_NUMBER = 240;
CMAP_STERN_LEVEL = (1.0/239.0)*indgen(0:239);
func stern(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "stern.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_STERN_TABLE, CMAP_STERN_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_STERN_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}

CMAP_YARG_TABLE = char(
 [[255,255,255], [254,254,254], [253,253,253], [252,252,252], [251,251,251],
  [250,250,250], [249,249,249], [248,248,248], [246,246,246], [245,245,245],
  [244,244,244], [243,243,243], [242,242,242], [241,241,241], [240,240,240],
  [239,239,239], [238,238,238], [237,237,237], [236,236,236], [235,235,235],
  [234,234,234], [233,233,233], [232,232,232], [230,230,230], [229,229,229],
  [228,228,228], [227,227,227], [226,226,226], [225,225,225], [224,224,224],
  [223,223,223], [222,222,222], [221,221,221], [220,220,220], [219,219,219],
  [218,218,218], [217,217,217], [216,216,216], [214,214,214], [213,213,213],
  [212,212,212], [211,211,211], [210,210,210], [209,209,209], [208,208,208],
  [207,207,207], [206,206,206], [205,205,205], [204,204,204], [203,203,203],
  [202,202,202], [201,201,201], [200,200,200], [198,198,198], [197,197,197],
  [196,196,196], [195,195,195], [194,194,194], [193,193,193], [192,192,192],
  [191,191,191], [190,190,190], [189,189,189], [188,188,188], [187,187,187],
  [186,186,186], [185,185,185], [184,184,184], [182,182,182], [181,181,181],
  [180,180,180], [179,179,179], [178,178,178], [177,177,177], [176,176,176],
  [175,175,175], [174,174,174], [173,173,173], [172,172,172], [171,171,171],
  [170,170,170], [169,169,169], [168,168,168], [166,166,166], [165,165,165],
  [164,164,164], [163,163,163], [162,162,162], [161,161,161], [160,160,160],
  [159,159,159], [158,158,158], [157,157,157], [156,156,156], [155,155,155],
  [154,154,154], [153,153,153], [152,152,152], [150,150,150], [149,149,149],
  [148,148,148], [147,147,147], [146,146,146], [145,145,145], [144,144,144],
  [143,143,143], [142,142,142], [141,141,141], [140,140,140], [139,139,139],
  [138,138,138], [137,137,137], [136,136,136], [134,134,134], [133,133,133],
  [132,132,132], [131,131,131], [130,130,130], [129,129,129], [128,128,128],
  [127,127,127], [126,126,126], [125,125,125], [124,124,124], [123,123,123],
  [122,122,122], [121,121,121], [120,120,120], [118,118,118], [117,117,117],
  [116,116,116], [115,115,115], [114,114,114], [113,113,113], [112,112,112],
  [111,111,111], [110,110,110], [109,109,109], [108,108,108], [107,107,107],
  [106,106,106], [105,105,105], [104,104,104], [102,102,102], [101,101,101],
  [100,100,100], [ 99, 99, 99], [ 98, 98, 98], [ 97, 97, 97], [ 96, 96, 96],
  [ 95, 95, 95], [ 94, 94, 94], [ 93, 93, 93], [ 92, 92, 92], [ 91, 91, 91],
  [ 90, 90, 90], [ 89, 89, 89], [ 88, 88, 88], [ 86, 86, 86], [ 85, 85, 85],
  [ 84, 84, 84], [ 83, 83, 83], [ 82, 82, 82], [ 81, 81, 81], [ 80, 80, 80],
  [ 79, 79, 79], [ 78, 78, 78], [ 77, 77, 77], [ 76, 76, 76], [ 75, 75, 75],
  [ 74, 74, 74], [ 73, 73, 73], [ 72, 72, 72], [ 70, 70, 70], [ 69, 69, 69],
  [ 68, 68, 68], [ 67, 67, 67], [ 66, 66, 66], [ 65, 65, 65], [ 64, 64, 64],
  [ 63, 63, 63], [ 62, 62, 62], [ 61, 61, 61], [ 60, 60, 60], [ 59, 59, 59],
  [ 58, 58, 58], [ 57, 57, 57], [ 56, 56, 56], [ 54, 54, 54], [ 53, 53, 53],
  [ 52, 52, 52], [ 51, 51, 51], [ 50, 50, 50], [ 49, 49, 49], [ 48, 48, 48],
  [ 47, 47, 47], [ 46, 46, 46], [ 45, 45, 45], [ 44, 44, 44], [ 43, 43, 43],
  [ 42, 42, 42], [ 41, 41, 41], [ 40, 40, 40], [ 38, 38, 38], [ 37, 37, 37],
  [ 36, 36, 36], [ 35, 35, 35], [ 34, 34, 34], [ 33, 33, 33], [ 32, 32, 32],
  [ 31, 31, 31], [ 30, 30, 30], [ 29, 29, 29], [ 28, 28, 28], [ 27, 27, 27],
  [ 26, 26, 26], [ 25, 25, 25], [ 24, 24, 24], [ 22, 22, 22], [ 21, 21, 21],
  [ 20, 20, 20], [ 19, 19, 19], [ 18, 18, 18], [ 17, 17, 17], [ 16, 16, 16],
  [ 15, 15, 15], [ 14, 14, 14], [ 13, 13, 13], [ 12, 12, 12], [ 11, 11, 11],
  [ 10, 10, 10], [  9,  9,  9], [  8,  8,  8], [  6,  6,  6], [  5,  5,  5],
  [  4,  4,  4], [  3,  3,  3], [  2,  2,  2], [  1,  1,  1], [  0,  0,  0]]);
CMAP_YARG_NUMBER = 240;
CMAP_YARG_LEVEL = (1.0/239.0)*indgen(0:239);
func yarg(i)
{
  if (am_subroutine()) {
    if (! is_void(i)) window, i;
    palette, "yarg.gp";
  } else if (is_scalar(i) && (id = identof(i)) <= Y_DOUBLE) {
    if (id <= Y_LONG) i /= 255.0;
    return char(interp(CMAP_YARG_TABLE, CMAP_YARG_LEVEL, i, 2) + 0.5);
  } else if (is_void(i)) {
    return CMAP_YARG_TABLE;
  } else {
    error, "expecting scalar real or integer";
  }
}
