/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       Options.cpp
//
//  contents:   This file contains the implementation of a generic class for command line options
//              parsing.
//
//----------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "Options.h"

#include <string>
#include <set>
using namespace std;

#ifndef WIN32
#include <strings.h>
#define stricmp  strcasecmp
#endif

bool Options::StrLess::operator()(const char* left, const char* right) const
{
	return (stricmp(left, right) < 0);
}

Options::Format::Format(const Options::FormatNode rawFormat[], int rawFormatSize)
{
	for (int i=0; i<rawFormatSize; i++)
		addOption(rawFormat[i].option, rawFormat[i].bHasParam);
}

Options::Format::Format()
{
}

void Options::Format::addOption(const char* option, bool bHasParam)
{
	m_formatMap[option] = bHasParam;
}

bool Options::Format::optionExists(const char *option) const
{
	return (m_formatMap.find(option) != m_formatMap.end());
}

bool Options::Format::optionHasParam(const char* option) const
{
	FormatMap::const_iterator it = m_formatMap.find(option);
	if (it == m_formatMap.end())
		return false; //Option not found - so it doesn't have a parameter.
	return it->second;
}

bool Options::optionExists(const char* option) const
{
	return m_optionsMap.find(option) != m_optionsMap.end();
}

/*
This parsing function goes over the command line, looking for words that begin with '-'.
When '-' is found, we check if this is a valid option. If it is, and it has a parameter, the
next word in the command line is assumed to be its value.
*/
bool Options::parse(int argc, const char* argv[],
					const Format &format)
{
	const char* optionName = NULL;
	const char* optionVal = NULL;

	if ((argv == NULL) || (argc < 0))
		return false;

	//We start from the second parameter, since the first is the program name.
	for (int i=1; i<argc; i++)
	{
		if (argv[i][0] == '-')
		{
			optionName = argv[i] + 1;
			if (optionExists(optionName))
			{
				/*The option was already specified in this command line -
				this is considered as an error.*/
				return false;
			}

			if (!format.optionExists(optionName))
			{
				/*This option was not specified in the format.*/
				return false;
			}

			if (format.optionHasParam(optionName))
			{
				//If this option has a parameter, read it and skip it.
				if (++i >= argc)
					return false; //error - this option should have a parameter.
				optionVal = argv[i];
			}
			else
			{
				optionVal = "";
			}
			m_optionsMap[optionName] = optionVal;
		}
	}

	m_parsed = true;
	return true;
}

bool Options::parse(int argc, const char* argv[],
					const FormatNode rawFormat[], int rawFormatSize)
{
	return parse(argc, argv, Format(rawFormat, rawFormatSize));
}

Options::Options(int argc, const char* argv[],
				 const Format &format)
{
	parse(argc, argv, format);
}

Options::Options(int argc, const char* argv[],
				 const FormatNode rawFormat[], int rawFormatSize)
{
	parse(argc, argv, rawFormat, rawFormatSize);
}

Options::Options()
{
	m_parsed = false;
}

const char* Options::getOption(const char* option) const
{
	OptionsMap::const_iterator it = m_optionsMap.find(option);
	return (it == m_optionsMap.end()) ? NULL : it->second;
}

